#ifndef HASHINCLUDED
#define HASHINCLUDED
//#define HASHTABLE_NOASSERTIFCLASH			// Prevent the hashtable from asserting if we have duplicates
#pragma warning( push )
#pragma warning( disable : 4002 )
#include <core/hashtable.h>
#pragma warning( pop )
#endif

#include "../misc/gencrc.h"
#include "CamExporter.h"
#include <Export/ScriptExport.h>
#include <direct.h>			// for _mkdir
#include <io.h>				// for _access, _chmod
#include <process.h>
#include <sys/stat.h>		// for access flags
#include "Link/LinkUI.h"
#include "Trigger/Trigger.h"
#include "../PropEdit/ConfigData.h"
#include "../PropEdit/ParseFuncs.h"
#include "Material/NextMat.h"
#include "path.h"
#include "../PropEdit/ScriptIniParser.h"
#include "../PropEdit/PropFlags.h"
#include "../PropEdit/PreProcessor.h"
#include "../UI/OKtoAll.h"
#include "ModelExportOptions.h"
#include "misc/mdir.h"
#include "misc/util.h"
#include "ObjAnimExport.h"
#include "../Material/NExtMat.h"
#include "../Texture/NExtTexture.h"
#include "sfp.h"
#include "../Material/TerrainTypes.h"
#include "../misc/gencrc.h"
#include "nodecrc.h"
#include "appdata.h"
#include "../Particle/NExtParticle.h"
#include "MemDebug.h"

#define  LINK_DELIM   "link_"
#define  WRITE_ACCESS  0x02
#define  DEBUGLINE(x) { FILE* fp = fopen("c:\\debug.log","a"); fprintf(fp, "%s\n", x); fclose(fp); }

#include "Material/NExtMultiMat.h"

#define TERRAIN_WARN_MIN 40		// Reserved min terrain ID for terrain that should be warned on
#define TERRAIN_WARN_MAX 50		// Reserved max terrain ID for terrain that should be warned on

// For ProcessSFPNodes (uses templates, must be external)
char blend_modes[vNUM_BLEND_MODES][32] = 
{
	{ "diffuse" },
	{ "add" },
	{ "fixadd" },
	{ "sub" },
	{ "fixsub" },
	{ "blend" },
	{ "fixblend" },
	{ "modulate" },
	{ "fixmodulate (Fixed Alpha)" },
	{ "brighten" },
	{ "fixbrighten" }
};

struct MtlEntry
{
	Mtl*     mtl;			// The material containing the texture to use
	CStr     type;			// Particle effect type
	INodeTab nodes;			// All the nodes containing this material and particle effect type

	bool operator== (MtlEntry& right)
	{
		if (mtl  == right.mtl &&
			type == right.type)
			return true;

		return false;
	}
};
/////////////////////////////////////////////////////////////

class TrigDepEnum: public DependentEnumProc
{
public:
	Tab<ReferenceMaker*> reflist;
	
	int proc(ReferenceMaker *rmaker)
	{
		reflist.Append(1,&rmaker);
		return DEP_ENUM_CONTINUE;
	}
};

struct AltScript
{
	CStr filename;
	CStr buffer;

	bool operator== (AltScript& right)
	{
		if (filename == right.filename)
			return true;

		return false;
	}
};

class ScriptExporter : public IScriptExporter, public ScriptIniParser, public DynScriptPreprocessor
{
	FILE*   fp;							// File being exported
	CStr    scriptBuf;					// The script buffer for the current node
	CStr    strCmds;					// The command buffer for the current node
	DWORD   flags;						// The flags for the current node
	
	bool    tlist[vNUM_TERRAIN_TYPES];	// Terrain list for auto-generated LoadTerrain script
	LinkList<CStr> particle_tex_list;

	LinkList<ConfigProp>		props;		// List of properties for the current node
	LinkList<LONG>				links;		// List of all the link indicies for the current node
	LinkList<CStr>	            scripts;	// List of all the trigger scripts
	LinkList<MtlEntry>          mtlList;	// Material tracking list for SFP types

	LinkList<AltScript>			altscripts;	// Alternate script buffers (cumulates as nodes are processed)

	Tab<INode*>					nodelist;	// Temporary list of nodes to store the selection set
	INodeTab                    SFPNodes;

	TrigDepEnum					trigDepEnum;// Trigger dependency enumerator

	Lst::HashTable<unsigned long>	triggerList;// List of all trigger objects to check for duplicates
	
	void    CreateNodeIDList(Tab <INode *> &m_exportable_nodes);
	int     GetNodeIndex(INode* node);
	bool    OutputNode(INode* node);
	bool    GetLinks(INode* node);
	bool    GetRefLinks(INode* node);
	bool    GetManualLinks(INode* node);
	void    DumpLinks();
	void	DumpLinks(CStr& str);
	void    GetRotation(INode* node,float *rot);
	void    GetTerrainList(Interface* ip,bool* terrain_list,bool bClear = true);
	void	UpdatePropTerrains(Tab <INode *>& node_list, bool* terrain_list);
	void	ClearTerrainList(bool* terrain_list);
	bool    ScanTerrainTypes(LinkList<ConfigProp>* props, bool* terlist);
	void    DumpLoadAnims();
	void    DumpExtScripts();						// Output the external ,q script templates generated via MAXScript
	void    PreProcessSFPNodes(INodeTab& nodes);	// Processes Screen Facing Polys during node enum (Sets up mtl DB)
	void    PostProcessSFPNodes(INodeTab& nodes);	// Processes Screen Facing Polys after node enum (Writes generation scripts)

	// Nodelist generation functions
	void    GetAll(INode* root);
	void    GetSelected();
	void    GetSelectionSet(CStr name);
	void    GetVisibleOnly(INode* root);

	bool    IsExportable(INode* node);

	bool    IsCheckbox(ConfigProp* cprop);
	bool    IsExportable(ConfigProp* cprop);
	bool    IsOptExportable(ConfigProp* cprop, CStr value);

	bool    DumpAltScripts();

	Point3  ComputePosition(INode* node);
	Point3  ComputePosition(INode* node, CStr strCluster);

	bool	ShouldExportNode(INode* node);

	void    FindSFPNodes(Tab<INode*>& SFPNodes,INode* root=NULL);
	void	ClearTriggerList();

	void	PostParticleMax(INode* node);
	CStr    GetEmitScript(INode* node);

	void	SaveTerrainList(bool* tlist);
	bool	LoadTerrainList(bool* tlist);

	void	AddParticleTexture( CStr particle_tex );
	void	OutputLoadParticleTexturesScript( void );

	bool    VerifyNodeNames(INode* node);

public:
	ScriptExporter();
	~ScriptExporter();

	bool    ExportScriptsFromSet( char* scene_name, Tab <INode *> &m_exportable_nodes, bool bModel=false, char* modelName=NULL );
	bool	ExportScripts( SceneExportOptions& export_options, bool bModel=false, char* modelName=NULL );
	bool	ExportableNodesExist( SceneExportOptions& export_options );

	void	        ClearParticleTextureList( void );
	LinkList<CStr>* GetParticleTextureList() { return &particle_tex_list; }

	bool	OutputStrNode(INode* node, CStr& outBuf, CStr& outBuf2, LinkList<CStr>* scripts, bool bForceOrigin = false);

	// Required for VC to compile
	inline bool	ParseScriptIni(char* filename = NULL) { return ScriptIniParser::ParseScriptIni(filename); }
};

static ScriptExporter s_script_exporter;

ScriptExporter::ScriptExporter() : triggerList(4)
{
}

ScriptExporter::~ScriptExporter()
{
	ClearTriggerList();
}

DWORD WINAPI ScriptExportProgressFunc(LPVOID arg) 
{
    return(0);
}

// Scan through the level and find any SFP Nodes
void ScriptExporter::FindSFPNodes(Tab<INode*>& SFPNodes, INode* root)
{
	if (!root)
	{
		root = gInterface->GetRootNode();
		SFPNodes.ZeroCount();
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
		FindSFPNodes(SFPNodes,root->GetChildNode(i));

	CStr value;
	if (GetSFPEffect(root,value) &&
		value != CStr("None"))
	{
		SFPNodes.Append(1,&root);
	}
}

// Dumps out all the alternate scripts that accumulated during the node enumeration
/*
bool ScriptExporter::DumpAltScripts()
{
	Link<AltScript>* curscript = altscripts.GetHead();
	FILE* fp;


	CStr strPath = CStr(getenv(APP_ENV)) + CStr(QN_PATH) + ;

	while(curscript)
	{
		CStr FileName = strPath + curscript->data.filename;

		fp = fopen(FileName,"w");

		while (!fp)
		{
			// Check the file attributes to see if the .qn is read-only
			if (_access(FileName,WRITE_ACCESS)!=0)
			{
				char ErrMsg[256];
				int  rVal;
				sprintf(ErrMsg,"The file '%s' does not have write access enabled.\nWould you like to manually remove the Read-Only flag?",(char*)FileName);
				rVal=MessageBoxAll(NULL,ErrMsg,"Export Scripts",MB_YESNO|MB_ICONWARNING);

				if (rVal==IDYES)
				{
					if (_chmod(FileName, _S_IREAD | _S_IWRITE)==-1)
					{
						MessageBox(NULL,"An error occurred while changing the file's permission settings!","Export Scripts",MB_ICONSTOP|MB_OK);
						return false;
					}

					break;
				}
				
				return false;
			}

			char ErrorMsg[256];
			sprintf(ErrorMsg,"Unable to save file.  Check file attributes.\nPath: %s",FileName);
			MessageBox(NULL,ErrorMsg,"ScriptExport Error",MB_ICONSTOP|MB_OK);
			return false;
		}

		fputs((char*)curscript->data.buffer,fp);
		fclose(fp);

		curscript = curscript->next;
	}
}
*/

bool ScriptExporter::IsExportable(INode* node)
{
	Object* obj=node->EvalWorldState(0).obj;

	Class_ID cid=obj->ClassID();
	SClass_ID scid=obj->SuperClassID();

	CStr propBuffer;
	node->GetUserPropBuffer(propBuffer);

	// Don't export the node if it's been flagged to be omitted
	if (strstr(propBuffer,"// CMD:OMIT") ||
		strstr(propBuffer,"// CMD:FORCEOMIT"))
		return false;

	if ( obj->ClassID() == vTRIGGER_CLASS_ID )
		return true;

	if ( obj->SuperClassID() == HELPER_CLASS_ID ||
		 obj->SuperClassID() == SHAPE_CLASS_ID  ||
		 obj->SuperClassID() == CAMERA_CLASS_ID ||
		 obj->SuperClassID() == LIGHT_CLASS_ID  ||
		 obj->ClassID() == Class_ID(TARGET_CLASS_ID,0) ||
		 obj->ClassID() == BONE_OBJ_CLASSID)
	{
		return false;
	}

	if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
	{
		// Only export the parent
		if( node->GetParentNode() != gInterface->GetRootNode())
		{
			return false;
		}
	}

	return true;
}

void ScriptExporter::GetAll(INode* root)
{
	INode* child;
	int i;

	for(i=0;i<root->NumberOfChildren();i++)
	{
		child=root->GetChildNode(i);

		if (IsExportable(child))
			nodelist.Append(1,&child);

		if (child->NumberOfChildren()>0)
			GetAll(child);
	}
}

void ScriptExporter::GetSelected()
{
	INode* node;

	int numNodes=gInterface->GetSelNodeCount();

	for(int i=0;i<numNodes;i++)
	{
		node=gInterface->GetSelNode(i);

		if (node && IsExportable(node))
			nodelist.Append(1,&node);
	}
}

void ScriptExporter::GetSelectionSet(CStr name)
{
	if (name==CStr(""))
		return;

	// Find set
	int numSets=gInterface->GetNumNamedSelSets();

	for(int i=0;i<numSets;i++)
	{
		CStr setName=gInterface->GetNamedSelSetName(i);

		// We've found the selection set, get all of the nodes it contains
		if (setName==name)
		{
			INode* node;
			int    numNodes=gInterface->GetNamedSelSetItemCount(i);

			for(int j=0;j<numNodes;j++)
			{
				node=gInterface->GetNamedSelSetItem(i,j);

				if (IsExportable(node))
					nodelist.Append(1,&node);
			}

			return;
		}
	}
}

void ScriptExporter::GetVisibleOnly(INode* root)
{
	INode* child;
	int i;

	for(i=0;i<root->NumberOfChildren();i++)
	{
		child=root->GetChildNode(i);

		// Only add the node if it's not hidden or frozen
		if (!child->IsHidden() &&
			!child->IsFrozen())
		{
			if (IsExportable(child))
				nodelist.Append(1,&child);
		}

		if (child->NumberOfChildren()>0)
			GetAll(child);
	}
}

bool ScriptExporter::ExportableNodesExist( SceneExportOptions& export_options )
{
	nodelist.ZeroCount();

	// Read in the Scripts.ini to configDB
	if (!ParseScriptIni())
		return false;

	if (export_options.m_ExportAll)
	{
		GetAll(gInterface->GetRootNode());
	}
	else if (export_options.m_ExportSelected)
	{
		GetSelected();
	}
	else if (export_options.m_ExportSelectionSet)
	{
		GetSelectionSet(export_options.m_ExportSet);
	}
	else if (export_options.m_ExportVisibleOnly)
	{
		GetVisibleOnly(gInterface->GetRootNode());
	}
	
	return ( nodelist.Count() > 0 );
}

bool ScriptExporter::ExportScripts( SceneExportOptions& export_options, bool bModel, char* modelName )
{
	MessageBoxResetAll();

	AssignSharedData();
	nodelist.ZeroCount();
	ClearParticleTextureList();

	// Read in the Scripts.ini to configDB
	if (!ParseScriptIni())
		return false;

	if (export_options.m_ExportAll)
	{
		GetAll(gInterface->GetRootNode());
	}
	else if (export_options.m_ExportSelected)
	{
		GetSelected();
	}
	else if (export_options.m_ExportSelectionSet)
	{
		GetSelectionSet(export_options.m_ExportSet);
	}
	else if (export_options.m_ExportVisibleOnly)
	{
		GetVisibleOnly(gInterface->GetRootNode());
	}

	ExportOptions* options = GetExportOptions();

	if (options->m_ExportType == ExportOptions::vEXPORT_MODEL)
	{
		ModelExportOptions meo;
		GetModelExportOptions( &meo );

		if (modelName)
			return ExportScriptsFromSet(modelName, nodelist, true, modelName);

		return ExportScriptsFromSet(meo.m_ModelName, nodelist, true, modelName);
	}

	return ExportScriptsFromSet(export_options.m_SceneName, nodelist, bModel, modelName);
}

void ScriptExporter::CreateNodeIDList(Tab <INode *> &m_exportable_nodes)
{
	int      numNodes=m_exportable_nodes.Count();

	// Number all the nodes
	for(int i=0;i<numNodes;i++)
		m_exportable_nodes[i]->SetNodeLong(i);
}

void ScriptExporter::GetRotation(INode* node,float *rot)
{
// From NExtScriptMan::GetRotation (skate3)
#if 0
	Interval valid = FOREVER;
	Matrix3 tmat( 1 );

	Control *c = node->GetTMController();
	tmat = n->GetParentTM( 0 );
	c->GetValue( 0, &tmat, valid, CTRL_ABSOLUTE);

	Quat q(tmat);
	QuatToEuler( q, rot );	
#else
	//Interval valid = FOREVER;
	//
	//Quat q(node->GetNodeTM( 0 ));
	//QuatToEuler( q, rot );	

	Matrix3 tm = node->GetNodeTM(0);
	Quat q(tm);
	QuatToEuler( q, rot, EULERTYPE_XYZ );	

	// Switch rotations to game coord system
	// Max:   X: right Y: depth  Z: up
	// Game:  X: right Y: up     Z: -depth

	// Ordering: XYZ in MAX is XZY in game:
	// X is same, so same order
	// Y, Z is swaped so, order is swapped XZY
	Matrix3 tm2;
	tm2.IdentityMatrix();
	tm2.RotateX(rot[0]);
	tm2.RotateZ(-rot[1]);
	tm2.RotateY(rot[2]);
	
	Quat q2(tm2);
	QuatToEuler(q2, rot, EULERTYPE_XYZ);
#endif
}

bool ScriptExporter::GetRefLinks(INode* node)
{
	RefList& refList=node->GetRefList();

	bool bChildLinks=false;
	RefListItem* refItem=refList.FirstItem();
	
	links.Clear();

	while(refItem!=NULL)
	{
		if (refItem->maker)
		{
			if (refItem->maker->ClassID()==vLINK_OBJ_CLASS_ID)
			{
				LinkObject* link=(LinkObject*)refItem->maker;

				// Don't allow links to link back to themselves
				if (link->to_node &&
					link->to_node!=node)
				{
					LONG nodeIndex=link->to_node->GetNodeLong();
					links.AddUnique(&nodeIndex);
					bChildLinks=true;
				}
			}
		}

		refItem=refItem->next;
	}

	return bChildLinks;
}

bool ScriptExporter::GetManualLinks(INode* node)
{
	// Add any manual links that the user may have entered in the properties
	int numProps=props.GetSize();
	bool bAdded=false;

	for(int i=0;i<numProps;i++)
	{
		// Check if there's any extended link information
		CStr propName=props[i].name;
		CStr propValue=props[i].value;

		// Convert the property name to lower case so we're case insensitive
		propName.toLower();

		if (Instr(propName,LINK_DELIM)==0)
		{
			// Get link index
			if (HasAlpha(propValue))
			{
				INode* linkNode=gInterface->GetINodeByName(propValue);

				if (!linkNode)
				{
					char strErr[256];
					sprintf(strErr,"WARNING!  A manual link in node '%s' links to a non-existent node '%s'.\nThe link is being ignored.",(char*)node->GetName(),(char*)propValue);
					MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Manual Link System",MB_ICONWARNING|MB_OK);
					ReportWarning( strErr );
				}
				else
				{
					LONG   linkIndex=linkNode->GetNodeLong();
					links.Add(&linkIndex);
					bAdded=true;
				}
			}
			else
			{
				LONG   linkIndex=atoi(props[i].value);
				links.AddUnique(&linkIndex);
				bAdded=true;
			}
		}
	}

	return bAdded;
}

bool ScriptExporter::GetLinks(INode* node)
{
	Object* obj;
	obj=node->EvalWorldState(0).obj;

	links.Clear();

	// Determine if this is a link object
	// (if so add it's link to the list)
	if ( obj->ClassID() == vLINK_OBJ_CLASS_ID )
	{
		LinkObject* link=(LinkObject*)obj;

		if (link->to_node)
		{
			LONG nodeIndex=link->to_node->GetNodeLong();
			links.AddUnique(&nodeIndex);
		}
	}

	// Determine if this is a trigger object
	// (also capable of having links)
	if ( obj->ClassID() == vTRIGGER_CLASS_ID )
	{
		Trigger* trigger=(Trigger*)obj;
		
		// Scan through the dependents for LinkObjects
		node->EnumDependents(&trigDepEnum);

		int numRefs=trigDepEnum.reflist.Count();

		for(int i=0;i<numRefs;i++)
		{
			ReferenceMaker* depref=trigDepEnum.reflist[i];

			if (depref->ClassID()==vLINK_OBJ_CLASS_ID)
			{
				LinkObject* lobj=(LinkObject*)depref;

				// There will likely be 2 linkobjects connected as dependents
				// we want to find the next link in the chain that the trigger is connecting
				// to, so we find the to of the link obj that we are from
				if (lobj->from_node==node)
				{
					// The to is our next link
					if (lobj->to_node)
					{
						LONG nodeIndex=lobj->to_node->GetNodeLong();
						links.AddUnique(&nodeIndex);
					}
				}
			}
		}
	}

	if (links.GetSize()>0)
		return true;

	return false;
}

void ScriptExporter::DumpLinks(CStr& str)
{
	int numLinks=links.GetSize();

	if (numLinks==0)
		return;

	char buf[256];
	char eol[3];
	eol[0] = 10;
	eol[1] = 13;
	eol[2] = 0;

	str += "\tLinks=[";

	for(int i=0;i<numLinks;i++)
	{
		sprintf(buf, "%d", links[i]);
		str += buf;

		if (i+1<numLinks)
			str += ",";
	}

	str += CStr("]") + eol;
}

void ScriptExporter::DumpLinks()
{
	int numLinks=links.GetSize();

	if (numLinks==0)
		return;

	fprintf(fp,"\tLinks=[");

	for(int i=0;i<numLinks;i++)
	{
		fprintf(fp,"%d",links[i]);

		if (i+1<numLinks)
			fprintf(fp,",");
	}

	fprintf(fp,"]\n");
}

// Can't be local if we're using it with a templatized class
struct AnimEntry
{
	CStr name;
	bool bNetOnly;

	bool operator==(AnimEntry& right)
	{
		if (name==right.name)
			return true;

		return false;
	}
};

void ScriptExporter::DumpExtScripts()
{
	RetrieveExtMAXScriptData();
	
	Link<ScriptBuf>* link = scriptBuffers.GetHead();
	CStr buf;

	while(link)
	{
		//buf = GetTemplateScript(link,&props);
		buf = ReplaceStr(link->data.buffer,"\r\n","\n");	

		fprintf(fp, "script %s\n",(char*)link->data.scriptName);
		fprintf(fp, "%s",(char*)buf);
		fprintf(fp, "endscript\n\n");
		link = link->next;
	}
}

void ScriptExporter::DumpLoadAnims()
{
	LinkList<AnimEntry>  anims;
	LinkList<ConfigProp> cplist;

	int count=nodelist.Count();

	for(int i=0;i<count;i++)
	{
		CStr  strName1;
		CStr  strName2;
		CStr  propBuf;
		DWORD flags;
		int   pos=0;
		int   lstpos=0;

		INode* node=nodelist[i];

		node->GetUserPropBuffer(propBuf);

		// Acquire the class and type names
		CStr className,typeName;

		if (!node->GetUserPropString("Class",className))
			node->GetUserPropString("class",className);
		
		if (!node->GetUserPropString("Type",typeName))
			node->GetUserPropString("type",typeName);

		// Acquire the flags from the property buffer so we can tell if
		// absent in net games is selected
		//AddGlobalDefaults(propBuf);
		ParseNodeConfigProps(&cplist,&flags,node,NULL,NULL);
		//ParseConfigProps(&cplist,&flags,propBuf,NULL,NULL);

		AddGlobalDefaults(propBuf,&cplist,&flags,NULL,className,typeName);

		// Convert the parameters to their default values if necessary
		ConvertToDefaults(className,typeName,&cplist);

		// Find the properties we're looking for in the list
		Link<ConfigProp>* curLink=cplist.GetHead();

		while(curLink)
		{
			CStr name=curLink->data.name;
			name.toLower();

			char* rname = strrchr(name, '/');

			if (rname)
				name = CStr((rname + 1));

			if (name==CStr("animname"))
			{
				if (curLink->data.value==CStr("!"))
				{
					char strErr[256];
					sprintf(strErr,"The node '%s' of class '%s' and type '%s' contains an unresolved default for property 'AnimName'.\nIt is being ignored!",(char*)node->GetName(),(char*)className,(char*)typeName);

					MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"AnimLoad Unresolved Default",MB_ICONWARNING|MB_OK);
					ReportWarning( strErr );
				}
				else
					strName1=curLink->data.value+" ";
			}

			if (name==CStr("extra_anims"))
			{
				if (curLink->data.value==CStr("!"))
				{
					char strErr[256];
					sprintf(strErr,"The node '%s' of class '%s' and type '%s' contains an unresolved default for property 'Extra_Anims'.\nIt is being ignored!",(char*)node->GetName(),(char*)className,(char*)typeName);

					MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"AnimLoad Unresolved Default",MB_ICONWARNING|MB_OK);
					ReportWarning( strErr );
				}
				else
					strName2=curLink->data.value+" ";
			}

			curLink=curLink->next;
		}

		// Parse list of animations in the AnimName field
		while((pos=Instr(strName1," ",pos))!=-1)
		{
			char buf[256];
			strcpy(buf,((char*)strName1)+lstpos);
			buf[pos-lstpos]='\0';
			StripToken(buf);

			if (strlen(buf)>0)
			{
				AnimEntry newEntry;
				newEntry.name=buf;
				
				if (flags & CCLASS_ABSENTINNETGAMES)
					newEntry.bNetOnly=true;
				else
					newEntry.bNetOnly=false;

				// Add the name to the list
				Link<AnimEntry>* foundLink=anims.AddToTailFindUnique(&newEntry);
				
				// If we weren't able to add a new link this will ensure that if
				// it wasn't added (another one was found) that that one is assigned
				// netonly false if the new one would have been
				if (!newEntry.bNetOnly)
					foundLink->data.bNetOnly=false;
			}

			pos++;
			lstpos=pos;
		}

		pos=0;
		lstpos=0;

		// Parse list of animations in the Extra_Anims field
		while((pos=Instr(strName2," ",pos))!=-1)
		{
			char buf[256];
			strcpy(buf,(char*)strName2+lstpos);
			buf[pos-lstpos]='\0';
			StripToken(buf);

			if (strlen(buf)>0)
			{
				AnimEntry newEntry;
				newEntry.name=buf;
				
				if (flags & CCLASS_ABSENTINNETGAMES)
					newEntry.bNetOnly=true;
				else
					newEntry.bNetOnly=false;

				// Add the name to the list
				Link<AnimEntry>* foundLink=anims.AddToTailFindUnique(&newEntry);
				
				// If we weren't able to add a new link this will ensure that if
				// it wasn't added (another one was found) that that one is assigned
				// netonly false if the new one would have been
				if (!newEntry.bNetOnly)
					foundLink->data.bNetOnly=false;
			}

			pos++;
			lstpos=pos;
		}
	}

	// Now that we have the list constructed dump it out
	fprintf(fp,"script load_level_anims\n");

	Link<AnimEntry>* curNode=anims.GetHead();

	while(curNode)
	{
		if (curNode->data.bNetOnly)
		{
			fprintf(fp,"\tif not InNetGame\n");
			//fprintf(fp,"\t\tanimload_%s\n",(char*)curNode->data.name);
			fprintf(fp,"\t\t%s\n",(char*)curNode->data.name);
			fprintf(fp,"\tendif\n");
		}
		else
		{
			//fprintf(fp,"\tanimload_%s\n",(char*)curNode->data.name);
			fprintf(fp,"\t%s\n",(char*)curNode->data.name);
		}

		curNode=curNode->next;
	}

	fprintf(fp,"endscript\n\n");
}

bool ScriptExporter::IsCheckbox(ConfigProp* cprop)
{
	if (strstr(cprop->extdata,CStr("// @nextparm | ") + cprop->name + " | check"))
		return true;

	return false;
}

bool ScriptExporter::IsExportable(ConfigProp* cprop)
{
	if (strstr(cprop->extdata,CStr("// @nextint | ") + cprop->name))
		return false;

	return true;
}

bool ScriptExporter::IsOptExportable(ConfigProp* cprop, CStr value)
{
	CStr srch     = CStr("// @nextintdef | ") + cprop->name;
	int  pos      = Instr(cprop->extdata,srch);
	int  rpos     = 0;

	if (pos != -1)
	{
		CStr defvalue = GetRemainLinePartial(cprop->extdata.Substr(pos+srch.Length(),cprop->extdata.Length()),&rpos);

		if (IsInstr(defvalue,"|",0))
		{
			defvalue = ReplaceStr(defvalue,"|","");

			char buf[512];
			strcpy(buf,defvalue);
			StripToken(buf);

			if (cprop->value == CStr(buf))
				return true;
			else
				return false;
		}
		else
		{
			if (cprop->value == value)
				return true;
			else
				return false;
		}
	}

	return false;
}

Point3 ScriptExporter::ComputePosition(INode* node)
{
	Matrix3 mat;
	Point3  pos;
	float   tmp;

	mat = node->GetNodeTM(0);		// Get node's transform matrix at time 0

	pos=mat.GetTrans();
	tmp=pos.z;
	//pos.z=pos.y;
	pos.z=-pos.y;
	pos.y=tmp;

	return pos;
}

bool ScriptExporter::VerifyNodeNames(INode* node)
{
	if (HasInvalidCharacters(node->GetName()))
	{
		char buf[256];
		sprintf(buf, "The node '%s' contains invalid characters in its name.  Export Aborted.", (char*)node->GetName());
		MessageBox(gInterface->GetMAXHWnd(), buf, "Invalid Characters", MB_ICONWARNING|MB_OK);
		return false;
	}

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);

		if (!VerifyNodeNames(child))
			return false;
	}

	return true;
}

Point3 ScriptExporter::ComputePosition(INode* node,CStr strObject)
{
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);
	
	if (seo.m_ExportFromOrigin)
	{
		INode* linkNode = GetCOREInterface()->GetINodeByName(strObject);

		if (!linkNode)
			return Point3(0.0f,0.0f,0.0f);

		// The coordinates for this object should be relative to the linked node
		return ComputePosition(node) - ComputePosition(linkNode);
	}

	return ComputePosition(node);
}

#define BUFOUT(x)  fputs(x, fp); outBuf += x;
#define BUFOUT2(x) outBufScript += x;

static int nCountON = 0;

/////////////////////////////////////////////////////////////////////////////////////////////
bool ScriptExporter::OutputStrNode(INode* node, CStr& outBuf, CStr& outBuf2, LinkList<CStr>* scripts, bool bForceOrigin)
{
	outBuf = outBuf2 = "";

	//////////////////////////////////////////////////////////////////////////////////////
	// Check node CRC data to determine if we need to re-output node data
	// if the CRCs match we can use the stored data from last output
	char    tmpBuf[5000];
	NodeCRC ncrcNew(node);	
	Object* obj = node->EvalWorldState(0).obj;

	/////////////////////////////////////////////////////////////////////////////////////

	Matrix3 mat;				// Node's TM
	Point3  pos;				// Node's position
	float   rot[3];				// Euler rotation of node
	CStr    propBuf;			// Node's property buffer
	bool    bUseDefaults=false;	// Use defaults looked up from scripts.ini
	bool    bExportLocalSpace;

	CStr    nodeName=node->GetName();
	CStr    className;
	CStr    typeName;
	CStr    strCluster;

	// Determine if we should be exporting in local coordinates as opposed to world
	if (GetEvalProp(node, "ExportLocalSpace") == CStr("TRUE"))
		bExportLocalSpace = true;
	else
		bExportLocalSpace = false;

	int     i;

	// Warn if the name has a space in it
	if (strstr(nodeName," "))
	{
		CStr nospaces;
		char strErr[256];
		int  result;

		nospaces=ReplaceStr(nodeName," ","");
		sprintf(strErr,"The node '%s' has a space in it's name.  This is unsupported by the game engine.  Shall I rename it to '%s'?",(char*)nodeName,(char*)nospaces);
		result = MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Invalid Name",MB_YESNO|MB_ICONQUESTION);
		ReportWarning( strErr );

		if (result==IDYES)
		{
			node->SetName(nospaces);
			nodeName = nospaces;
		}
		else
			return false;
	}

	node->GetUserPropBuffer(propBuf);

	// Replace all '*'s in the buffer with the name of the node
	propBuf=ReplaceStr(propBuf, "*", nodeName);

	// Parse all the config property data
	//AddGlobalDefaults(propBuf);

	className = GetClassName(propBuf);

	// Ensure that the class exists
	if (className!=CStr("") && !ClassExists(className))
	{
		char strErr[256];
		sprintf(strErr,"In the node '%s' you used Class '%s' which does not exist in the scripts.ini file.\nWould you like to continue exporting anyway?",(char*)nodeName,(char*)className);
		int rVal=MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Node using Unknown Class",MB_ICONWARNING|MB_YESNO|MB_OK);
		ReportWarning( strErr );

		if (rVal==IDNO)
			return false;
	}

	// If there is no class, default it to the default class, created at start
	if (className.Length()==0)
	{
		bUseDefaults=true;

		// This should only be done depending on the node type
		if (obj->ClassID()==vTRIGGER_CLASS_ID)
		{
			char strErr[256];
			int  rVal;
			sprintf(strErr,"WARNING! Trigger Object '%s' hasn't been assigned any property information.\nIt will be treated as a '%s' for now.\nDo you want to continue the export?",(char*)nodeName,TRIGGER_CLASS);

			rVal=MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Trigger Object Unassigned",MB_ICONWARNING|MB_OKCANCEL);
			ReportWarning( strErr );

			if (rVal==IDCANCEL)
				return false;

			className=TRIGGER_CLASS;
		}
		else if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			     obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			className=CAMERA_CLASS;
		}
		else
		{
			className=DEFAULT_CLASS;
		}
	}

	// Switch the class name if it is equal to the old type
	if (className==CStr(OLD_DEFAULT_CLASS))
	{
		className=DEFAULT_CLASS;
	}

	//if (!node->GetUserPropString("Type",typeName))
	//	node->GetUserPropString("type",typeName);

	typeName = GetTypeName(propBuf);

	// Ensure that the type exists
	if (className!=CStr("LevelGeometry") && typeName!=CStr("") && !TypeExists(className,typeName))
	{
		char strErr[256];
		sprintf(strErr,"In the node '%s' you used Type '%s' of Class '%s' which does not exist in the scripts.ini file.\nWould you like to continue exporting anyway?",(char*)nodeName,(char*)typeName,(char*)className);
		int rVal=MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Node using Unknown Type",MB_ICONWARNING|MB_YESNO|MB_OK);
		ReportWarning( strErr );

		if (rVal==IDNO)
			return false;
	}

	// Parse property data
	LinkList<ConfigProgram> programs;
	LinkList<ConfigScript>  cscripts;

	// If the class name was defaulted, look up the defaults from the config file
	if (bUseDefaults)
	{
		// Acquire default info
		//CStr defaultBuf=GetClassTypeRecord(className,typeName);
		//scriptBuf=ParseConfigProps(&props,&flags,defaultBuf,&strCmds,&strCluster);
		propBuf   = GetClassTypeRecord(className,typeName);
		scriptBuf = ParseConfigProps(&props,&flags,propBuf,&strCmds,&strCluster,&programs);
	}
	else
		scriptBuf = ParseNodeConfigProps(&props,&flags,node,&strCmds,&strCluster,&programs,&cscripts);

	AddGlobalDefaults(propBuf,&props,&flags,&strCmds,className,typeName);
	CombineDynUIProps(&props,className,typeName);

	char* classNameRD = strrchr(className,'/');
	if (classNameRD)
		classNameRD++;
	else
		classNameRD = className;

	char* typeNameRD = strrchr(typeName,'/');
	if (typeNameRD)
		typeNameRD++;
	else
		typeNameRD = typeName;

	// Get node position in game coordinate system
	if (bForceOrigin)
	{
		pos.x = 0.0f;
		pos.y = 0.0f;
		pos.z = 0.0f;
	}
	else
		pos = ComputePosition(node,strCluster);

	// Get node's Euler rotation in game coordinate system
	GetRotation(node, rot);	

	//sprintf(tmpBuf,"\tPosition = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
	//BUFOUT(tmpBuf);

	if (bExportLocalSpace)
		sprintf(tmpBuf,"\tPos = (%f,%f,%f)\n", 0.0f, 0.0f, 0.0f);
	else
		sprintf(tmpBuf,"\tPos = (%f,%f,%f)\n", pos.x,pos.y,pos.z);

	outBuf += tmpBuf;
	//BUFOUT(tmpBuf);

	sprintf(tmpBuf,"\tAngles = (%f,%f,%f)\n", rot[0],rot[1],rot[2]);
	outBuf += tmpBuf;
	//BUFOUT(tmpBuf);
	
	sprintf(tmpBuf,"\tName = %s\n",nodeName);
	outBuf += tmpBuf;
	//BUFOUT(tmpBuf);

	sprintf(tmpBuf,"\tClass = %s\n",classNameRD);
	outBuf += tmpBuf;
	//BUFOUT(tmpBuf);

	if (typeName.Length()>0)
	{
		sprintf(tmpBuf,"\tType = %s\n",typeNameRD);		// Types are optional
		outBuf += tmpBuf;
		//BUFOUT(tmpBuf);
	}

	if (flags & CCLASS_TRICKOBJECT)
	{
		sprintf(tmpBuf,"\tTrickObject");
		outBuf += tmpBuf;
		//BUFOUT(tmpBuf);

		if (strCluster.Length()>0)
		{
			sprintf(tmpBuf," Cluster = %s",(char*)strCluster);
			outBuf += tmpBuf;
			//BUFOUT(tmpBuf);
		}
		
		sprintf(tmpBuf,"\n");
		outBuf += tmpBuf;
		//BUFOUT(tmpBuf);
	}

	//VerifyMemory();

	if( className== CStr( "ParticleObject" ))
	{
		if (obj->ClassID() == PARTICLE_BOX_CLASS_ID )
		{
			IParticleBox* p_box, *p_mid, *p_end;
			INode* mid_node, *end_node;
			Control* scale_ctrl;
			ScaleValue scale;

			p_box = dynamic_cast< IParticleBox* > ( obj );

			if (p_box)
			{
				p_mid = p_box->GetMidBox();

				if (p_mid)
				{
					mid_node = p_mid->GetObjectNode();

					scale_ctrl = node->GetTMController()->GetScaleController();
					scale_ctrl->GetValue( 0, &scale, FOREVER );

					p_end = p_box->GetEndBox();

					if (p_end)
					{
						end_node = p_end->GetObjectNode();
				
						sprintf(tmpBuf,"\tBoxDimsStart = (%f,%f,%f)\n",	p_box->GetWidth() * scale.s.x,
																		p_box->GetHeight() * scale.s.z,
																		p_box->GetLength() * scale.s.y );

						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);

						//VerifyMemory();

						scale_ctrl = mid_node->GetTMController()->GetScaleController();
						scale_ctrl->GetValue( 0, &scale, FOREVER );

						// Get node position in game coordinate system
						pos = ComputePosition(mid_node);
						if (bExportLocalSpace)
							pos -= ComputePosition(node);

						sprintf(tmpBuf,"\tMidPosition = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);

						sprintf(tmpBuf,"\tBoxDimsMid = (%f,%f,%f)\n",	p_mid->GetWidth() * scale.s.x,
																		p_mid->GetHeight() * scale.s.z,
																		p_mid->GetLength() * scale.s.y );

						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);

						//VerifyMemory();

						scale_ctrl = end_node->GetTMController()->GetScaleController();
						scale_ctrl->GetValue( 0, &scale, FOREVER );

						// Get node position in game coordinate system
						pos = ComputePosition(end_node);
						if (bExportLocalSpace)
							pos -= ComputePosition(node);

						sprintf(tmpBuf,"\tEndPosition = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);

						sprintf(tmpBuf,"\tBoxDimsEnd = (%f,%f,%f)\n",	p_end->GetWidth() * scale.s.x,
																		p_end->GetHeight() * scale.s.z,
																		p_end->GetLength() * scale.s.y );

						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);

						CStr particle_tex;

						//VerifyMemory();

						particle_tex = "";
						if( node->GetUserPropString( "Core_Values/Texture", particle_tex ))
						{	
							int i;
							CStr out_tex;

							if( particle_tex == CStr(vRESTORE_DEFAULT_SEQ))
							{
								particle_tex = GetDefault( className, typeName, "Core_Values/Texture" );
							}			

							out_tex = particle_tex;
							for( i = 0; i < particle_tex.Length(); i++ )
							{
								if( particle_tex[i] == '.' )
								{
									out_tex = particle_tex.remove( i );
								}
							}

							// Ensure no subdirectories are included
							if (strstr(out_tex, "\\"))
							{
								char  buf[1024];
								char* pos = strrchr((char*)out_tex, '\\') + 1;
								assert(strlen(pos) < 1024);
								strcpy(buf, pos);
								out_tex = buf;
							}

							sprintf( tmpBuf, "\tTexture = %s\n", out_tex );
							outBuf += tmpBuf;
							//BUFOUT( tmpBuf );

							AddParticleTexture( particle_tex );
							ncrcNew.nodeFlags |= NODEFLAG_HAS_PTEX;
						}

					}
				}
			}

			//VerifyMemory();
		}
	}

	//VerifyMemory();

	////////////////////////// START EVIL BLOCK
	// Convert the parameters to their default values if necessary
	ConvertToDefaults(className,typeName,&props);

	// We'll now build a property list from this so we can analyze nextreq rulesets
	PropList propList(hInstance, 1);
	BuildPropList(&propList, &props);

	// Dump out check box properties as flags
	int numProps=props.GetSize();
	
	for(i = 0; i < numProps; i++)
	{
		// Only properties that have their @nextreq requirements met should be output
		if (propList.RulesMet(i, node))
		{
			if (IsCheckbox(&props[i]))
			{
				if (props[i].value == CStr("TRUE"))
				{
					// Ensure we don't print out any property categorization data
					char* name = strrchr(props[i].name, '/');

					if (name)
					{
						sprintf(tmpBuf,"\t%s\n", ++name);
						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);
					}
					else
					{
						sprintf(tmpBuf,"\t%s\n",props[i].name);
						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);
					}
				}
			}
		}
	}
	//////////////////////// END EVIL BLOCK

	// Dump out the default parameters (as listed in the properties)
	for(i = 0; i < numProps; i++)
	{
		// Replace any '*'s with the object name
		props[i].value=ReplaceStr(props[i].value,"*",nodeName);

		// Don't add the property if it's a manual link entry
		// or a checkbox (aka flag) or it hasn't met its @nextreq requirements
		if (Instr(props[i].name,LINK_DELIM)!=0 &&
			!IsCheckbox(&props[i]) &&
			IsExportable(&props[i]) &&
			propList.RulesMet(i, node))
		{
			CStr name = props[i].name;
			
			// Strip any grouping info
			if (IsInstr(name, "/"))
				name = CStr(strrchr((char*)name, '/') + 1);

			CStr value;
			value = GetDefault(className,typeName,name);

			if (!IsOptExportable(&props[i],value))
			{
				// TrickObject's are a special case and should be converted
				// to the format that the engine expects them in as opposed to standard props
				if (name == CStr("TrickObject"))
				{
					if (props[i].value != CStr("[none]") &&
						props[i].value != CStr(""))
					{
						sprintf(tmpBuf, "\tTrickObject Cluster = %s\n", (char*)props[i].value);
						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);
					}
				}
				else
				{
					// Handle missing properties, if a property is missing it should use the default
					if (props[i].value.Length() == 0 ||
						props[i].value == CStr("[none]"))
					{
						if (value == CStr(""))
						{
							sprintf(tmpBuf,"//\t%s = ?\t// Undefined (Not in scripts.ini)\n",(char*)name);
							outBuf += tmpBuf;
							//BUFOUT(tmpBuf);
						}
						else
							if (value != CStr("[none]"))
							{
								sprintf(tmpBuf,"\t%s = %s\t// Undefined (Defaulted)\n",(char*)name,(char*)value);
								outBuf += tmpBuf;
								//BUFOUT(tmpBuf);
							}
					}
					else
					{
						sprintf(tmpBuf,"\t%s = %s\n",(char*)name,(char*)props[i].value);
						outBuf += tmpBuf;
						//BUFOUT(tmpBuf);
					}
				}
			}
		}
	}

	// Dump out the default commands
	if (strCmds.Length() > 0)
	{
		strCmds=ReplaceStr(strCmds,"\n","\n\t");
		
		if (strCmds.Length()>1)
			strCmds=strCmds.Substr(0,strCmds.Length()-1);

		strCmds = ReplaceStr(strCmds,"\r\n","\n");

		sprintf(tmpBuf,"\t%s",(char*)strCmds);
		outBuf += tmpBuf;
		//BUFOUT(tmpBuf);
	}

	// Attach TriggerScript if necessary
	
	if (programs.GetSize() > 0 || !IsBlank(scriptBuf))
	{
			CStr scriptName=CStr(node->GetName())+CStr("Script");

			sprintf(tmpBuf,"\tTriggerScript = %s\n",(char*)scriptName);
			outBuf += tmpBuf;
			//BUFOUT(tmpBuf);

			if (scripts)
				(*scripts).Add(&scriptName);
			
			ncrcNew.nodeFlags |= NODEFLAG_HAS_SCRIPT;
	}

	////////////// Perform parsing on script buffer for later use (Formerly Exporting Node Scripts) ////////////////
	CStr outBufScript;

	if (programs.GetSize()>0)
	{
		PostMAXScriptNode(node);
		PostParticleMax(node);
		AssignEmitScript(GetEmitScript(node));

		Link<ConfigProgram>* link = programs.GetHead();
		scriptBuf = "";

		// Execute all attached MaxScript programs
		while(link)
		{
			UpdateDynUIScripts(&props,flags,link->data.filename);
			scriptBuf += GetTemplate();

			// Set node flag indicating type of stored programs
			if (link->data.type == ConfigProgram::CFGPROG_VOLATILE)
				ncrcNew.nodeFlags |= NODEFLAG_HAS_VOLATILE_PROG;
			else
				if (link->data.type == ConfigProgram::CFGPROG_STATIC)
					ncrcNew.nodeFlags |= NODEFLAG_HAS_STATIC_PROG;

			link = link->next;
		}
	}

	//if (HasData(scriptBuf))
	if (!IsBlank(scriptBuf))
	{
		// Replace '*' with node name for the script buffer
		scriptBuf = ReplaceStr(scriptBuf,"*", nodeName);

		sprintf(tmpBuf,"script %sScript\n",(char*)nodeName);
		outBuf2 += tmpBuf;
		//BUFOUT2(tmpBuf);
		
		// Combine the commands from each alternate script buffer into the script
		Link<ConfigScript>* curscript = cscripts.GetHead();

		while(curscript)
		{
			scriptBuf = curscript->data.buffer + scriptBuf + CStr("\n");
			curscript = curscript->next;
		}

		scriptBuf = ReplaceStr(scriptBuf,"\r\n","\n");	

		sprintf(tmpBuf,"%s",scriptBuf);
		outBuf2 += tmpBuf;
		//BUFOUT2(tmpBuf);

		sprintf(tmpBuf,"endscript\n\n\n");
		outBuf2 += tmpBuf;
		//BUFOUT2(tmpBuf);

		ncrcNew.nodeFlags |= NODEFLAG_HAS_SCRIPT;
	}

	// Store content of any external scripts /////////////////////////////////////
	/*
	RetrieveExtMAXScriptData();
	
	Link<ScriptBuf>* link = scriptBuffers.GetHead();
	CStr buf;

	while(link)
	{
		//buf = GetTemplateScript(link,&props);
		buf = ReplaceStr(link->data.buffer,"\r\n","\n");	

		sprintf(tmpBuf, "script %s\n",(char*)link->data.scriptName);
		BUFOUT2(tmpBuf);

		sprintf(tmpBuf, "%s",(char*)buf);
		BUFOUT2(tmpBuf);

		sprintf(tmpBuf, "endscript\n\n");
		BUFOUT2(tmpBuf);

		link = link->next;
	}
	*/
	//////////////////////////////////////////////////////////////////////////////

	// Get link information (if this is a link)
	if (obj->ClassID()==vLINK_OBJ_CLASS_ID)
		GetLinks(node);
	else
		GetRefLinks(node);

	GetManualLinks(node);
	DumpLinks();

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	// Scan for terrain props (will restore, move to OutputNode)
	
	// ScanTerrainTypes now stores its data in scene root
	// since it will not always get called here if the node hasn't changed
	// ref. LoadTerrainList/SaveTerrainList
	//ScanTerrainTypes(&props, tlist);

	// Store the buffer within the node
	AssignNodeOutput(node, &ncrcNew, outBuf, outBufScript);
	return true;
}
//////////////////////////////////////////////////////////////////////////////////////////////

bool ScriptExporter::OutputNode(INode* node)
{
	//VerifyMemory();

	//////////////////////////////////////////////////////////////////////////////////////
	// Check node CRC data to determine if we need to re-output node data
	// if the CRCs match we can use the stored data from last output
	CStr    outBuf;
	char    tmpBuf[5000];
	NodeCRC ncrcNew(node);	
	Object* obj = node->EvalWorldState(0).obj;

	if(	( obj->ClassID() != PARTICLE_BOX_CLASS_ID ) &&
		( GetNodeOutput(node, &ncrcNew, outBuf )))
	{
		fputs(outBuf, fp);

		if (obj->ClassID()==vLINK_OBJ_CLASS_ID)
			GetLinks(node);
		else
			GetRefLinks(node);

		GetManualLinks(node);
		DumpLinks();

		NodeCRC ncrc;
		ncrc.Retrieve(node);

		// We must also add a triggerscript to the scripts list if necessary
		if (ncrc.nodeFlags & NODEFLAG_HAS_SCRIPT)
		{
			CStr scriptName = CStr(node->GetName()) + CStr("Script");
			scripts.Add(&scriptName);
		}
		
		return true;
	}

	/////////////////////////////////////////////////////////////////////////////////////

	Matrix3 mat;				// Node's TM
	Point3  pos;				// Node's position
	float   rot[3];				// Euler rotation of node
	CStr    propBuf;			// Node's property buffer
	bool    bUseDefaults=false;	// Use defaults looked up from scripts.ini
	bool    bExportLocalSpace;

	CStr    nodeName=node->GetName();
	CStr    className;
	CStr    typeName;
	CStr    strCluster;

	int     i;

	// Determine if we should be exporting in local coordinates as opposed to world
	if (GetEvalProp(node, "ExportLocalSpace") == CStr("TRUE"))
		bExportLocalSpace = true;
	else
		bExportLocalSpace = false;


	// Warn if the name has a space in it
	if (strstr(nodeName," "))
	{
		CStr nospaces;
		char strErr[256];
		int  result;

		nospaces=ReplaceStr(nodeName," ","");
		sprintf(strErr,"The node '%s' has a space in it's name.  This is unsupported by the game engine.  Shall I rename it to '%s'?",(char*)nodeName,(char*)nospaces);
		result = MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Invalid Name",MB_YESNO|MB_ICONQUESTION);
		ReportWarning( strErr );

		if (result==IDYES)
		{
			node->SetName(nospaces);
			nodeName = nospaces;
		}
		else
			return false;
	}

	node->GetUserPropBuffer(propBuf);

	// Replace all '*'s in the buffer with the name of the node
	propBuf=ReplaceStr(propBuf, "*", nodeName);

	// Parse all the config property data
	//AddGlobalDefaults(propBuf);

	className = GetClassName(propBuf);

	// Ensure that the class exists
	if (className!=CStr("") && !ClassExists(className))
	{
		char strErr[256];
		sprintf(strErr,"In the node '%s' you used Class '%s' which does not exist in the scripts.ini file.\nWould you like to continue exporting anyway?",(char*)nodeName,(char*)className);
		int rVal=MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Node using Unknown Class",MB_ICONWARNING|MB_YESNO|MB_OK);
		ReportWarning( strErr );

		if (rVal==IDNO)
			return false;
	}

	// If there is no class, default it to the default class, created at start
	if (className.Length()==0)
	{
		bUseDefaults=true;

		// This should only be done depending on the node type
		if (obj->ClassID()==vTRIGGER_CLASS_ID)
		{
			char strErr[256];
			int  rVal;
			sprintf(strErr,"WARNING! Trigger Object '%s' hasn't been assigned any property information.\nIt will be treated as a '%s' for now.\nDo you want to continue the export?",(char*)nodeName,TRIGGER_CLASS);

			rVal=MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Trigger Object Unassigned",MB_ICONWARNING|MB_OKCANCEL);
			ReportWarning( strErr );

			if (rVal==IDCANCEL)
				return false;

			className=TRIGGER_CLASS;
		}
		else if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			     obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			className=CAMERA_CLASS;
		}
		else
		{
			className=DEFAULT_CLASS;
		}
	}

	// Switch the class name if it is equal to the old type
	if (className==CStr(OLD_DEFAULT_CLASS))
	{
		className=DEFAULT_CLASS;
	}

	//if (!node->GetUserPropString("Type",typeName))
	//	node->GetUserPropString("type",typeName);

	typeName = GetTypeName(propBuf);

	// Ensure that the type exists
	if (className!=CStr("LevelGeometry") && typeName!=CStr("") && !TypeExists(className,typeName))
	{
		char strErr[256];
		sprintf(strErr,"In the node '%s' you used Type '%s' of Class '%s' which does not exist in the scripts.ini file.\nWould you like to continue exporting anyway?",(char*)nodeName,(char*)typeName,(char*)className);
		int rVal=MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Node using Unknown Type",MB_ICONWARNING|MB_YESNO|MB_OK);
		ReportWarning( strErr );

		if (rVal==IDNO)
			return false;
	}

	// Parse property data
	LinkList<ConfigProgram> programs;
	LinkList<ConfigScript>  cscripts;

	// If the class name was defaulted, look up the defaults from the config file
	if (bUseDefaults)
	{
		// Acquire default info
		//CStr defaultBuf=GetClassTypeRecord(className,typeName);
		//scriptBuf=ParseConfigProps(&props,&flags,defaultBuf,&strCmds,&strCluster);
		propBuf   = GetClassTypeRecord(className,typeName);
		scriptBuf = ParseConfigProps(&props,&flags,propBuf,&strCmds,&strCluster,&programs);
	}
	else
		scriptBuf = ParseNodeConfigProps(&props,&flags,node,&strCmds,&strCluster,&programs,&cscripts);

	AddGlobalDefaults(propBuf,&props,&flags,&strCmds,className,typeName);
	CombineDynUIProps(&props,className,typeName);

	char* classNameRD = strrchr(className,'/');
	if (classNameRD)
		classNameRD++;
	else
		classNameRD = className;

	char* typeNameRD = strrchr(typeName,'/');
	if (typeNameRD)
		typeNameRD++;
	else
		typeNameRD = typeName;

	// Get node position in game coordinate system
	pos = ComputePosition(node,strCluster);

	// Get node's Euler rotation in game coordinate system
	GetRotation(node, rot);	

	//sprintf(tmpBuf,"\tPosition = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
	//BUFOUT(tmpBuf);

	if (bExportLocalSpace)
		sprintf(tmpBuf, "\tPos = (%f,%f,%f)\n", 0.0f, 0.0f, 0.0f);
	else
		sprintf(tmpBuf,"\tPos = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
	BUFOUT(tmpBuf);

	sprintf(tmpBuf,"\tAngles = (%f,%f,%f)\n", rot[0],rot[1],rot[2]);
	BUFOUT(tmpBuf);
	
	sprintf(tmpBuf,"\tName = %s\n",nodeName);
	BUFOUT(tmpBuf);

	sprintf(tmpBuf,"\tClass = %s\n",classNameRD);
	BUFOUT(tmpBuf);

	if (typeName.Length()>0)
	{
		sprintf(tmpBuf,"\tType = %s\n",typeNameRD);		// Types are optional
		BUFOUT(tmpBuf);
	}

	if (flags & CCLASS_TRICKOBJECT)
	{
		sprintf(tmpBuf,"\tTrickObject");
		BUFOUT(tmpBuf);

		if (strCluster.Length()>0)
		{
			sprintf(tmpBuf," Cluster = %s",(char*)strCluster);
			BUFOUT(tmpBuf);
		}
		
		sprintf(tmpBuf,"\n");
		BUFOUT(tmpBuf);
	}

	//VerifyMemory();

	if( className== CStr( "ParticleObject" ))
	{
		if (obj->ClassID() == PARTICLE_BOX_CLASS_ID )
		{
			IParticleBox* p_box, *p_mid, *p_end;
			INode* mid_node, *end_node;
			Control* scale_ctrl;
			ScaleValue scale;
			
			p_box = dynamic_cast< IParticleBox* > ( obj );

			if (p_box)
			{
				p_mid = p_box->GetMidBox();

				if (p_mid)
				{
					mid_node = p_mid->GetObjectNode();

					scale_ctrl = node->GetTMController()->GetScaleController();
					scale_ctrl->GetValue( 0, &scale, FOREVER );

					p_end = p_box->GetEndBox();

					if (p_end)
					{
						end_node = p_end->GetObjectNode();
				
						sprintf(tmpBuf,"\tBoxDimsStart = (%f,%f,%f)\n",	p_box->GetWidth() * scale.s.x,
																		p_box->GetHeight() * scale.s.z,
																		p_box->GetLength() * scale.s.y );

						BUFOUT(tmpBuf);

						//VerifyMemory();

						scale_ctrl = mid_node->GetTMController()->GetScaleController();
						scale_ctrl->GetValue( 0, &scale, FOREVER );

						// Get node position in game coordinate system
						pos = ComputePosition(mid_node);
						if (bExportLocalSpace)
							pos -= ComputePosition(node);

						sprintf(tmpBuf,"\tMidPosition = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
						BUFOUT(tmpBuf);

						sprintf(tmpBuf,"\tBoxDimsMid = (%f,%f,%f)\n",	p_mid->GetWidth() * scale.s.x,
																		p_mid->GetHeight() * scale.s.z,
																		p_mid->GetLength() * scale.s.y );

						BUFOUT(tmpBuf);

						//VerifyMemory();

						scale_ctrl = end_node->GetTMController()->GetScaleController();
						scale_ctrl->GetValue( 0, &scale, FOREVER );

						// Get node position in game coordinate system
						pos = ComputePosition(end_node);
						if (bExportLocalSpace)
							pos -= ComputePosition(node);

						sprintf(tmpBuf,"\tEndPosition = (%f,%f,%f)\n", pos.x,pos.y,pos.z);
						BUFOUT(tmpBuf);

						sprintf(tmpBuf,"\tBoxDimsEnd = (%f,%f,%f)\n",	p_end->GetWidth() * scale.s.x,
																		p_end->GetHeight() * scale.s.z,
																		p_end->GetLength() * scale.s.y );

						BUFOUT(tmpBuf);

						CStr particle_tex;

						//VerifyMemory();

						particle_tex = "";
						if( node->GetUserPropString( "Core_Values/Texture", particle_tex ))
						{	
							int i;
							CStr out_tex;

							if( particle_tex == CStr(vRESTORE_DEFAULT_SEQ))
							{
								particle_tex = GetDefault( className, typeName, "Core_Values/Texture" );
							}			

							out_tex = particle_tex;
							for( i = 0; i < particle_tex.Length(); i++ )
							{
								if( particle_tex[i] == '.' )
								{
									out_tex = particle_tex.remove( i );
								}
							}

							// Ensure no subdirectories are included
							if (strstr(out_tex, "\\"))
							{
								char  buf[1024];
								char* pos = strrchr((char*)out_tex, '\\') + 1;
								assert(strlen(pos) < 1024);
								strcpy(buf, pos);
								out_tex = buf;
							}

							sprintf( tmpBuf, "\tTexture = %s\n", out_tex );
							BUFOUT( tmpBuf );

							AddParticleTexture( particle_tex );
							ncrcNew.nodeFlags |= NODEFLAG_HAS_PTEX;
						}

					}
				}
			}
		}
	}


	//VerifyMemory();

	////////////////////////// START EVIL BLOCK
	// Convert the parameters to their default values if necessary
	ConvertToDefaults(className,typeName,&props);

	// We'll now build a property list from this so we can analyze nextreq rulesets
	PropList propList(hInstance, 1);
	BuildPropList(&propList, &props);

	// Dump out check box properties as flags
	int numProps=props.GetSize();
	
	for(i = 0; i < numProps; i++)
	{
		// Only properties that have their @nextreq requirements met should be output
		if (propList.RulesMet(i, node))
		{
			if (IsCheckbox(&props[i]))
			{
				if (props[i].value == CStr("TRUE"))
				{
					// Ensure we don't print out any property categorization data
					char* name = strrchr(props[i].name, '/');

					if (name)
					{
						sprintf(tmpBuf,"\t%s\n", ++name);
						BUFOUT(tmpBuf);
					}
					else
					{
						sprintf(tmpBuf,"\t%s\n",props[i].name);
						BUFOUT(tmpBuf);
					}
				}
			}
		}
	}
	//////////////////////// END EVIL BLOCK

	// Dump out the default parameters (as listed in the properties)
	for(i = 0; i < numProps; i++)
	{
		// Replace any '*'s with the object name
		props[i].value=ReplaceStr(props[i].value,"*",nodeName);

		// Don't add the property if it's a manual link entry
		// or a checkbox (aka flag) or it hasn't met its @nextreq requirements
		if (Instr(props[i].name,LINK_DELIM)!=0 &&
			!IsCheckbox(&props[i]) &&
			IsExportable(&props[i]) &&
			propList.RulesMet(i, node))
		{
			CStr name = props[i].name;
			
			// Strip any grouping info
			if (IsInstr(name, "/"))
				name = CStr(strrchr((char*)name, '/') + 1);

			CStr value;
			value = GetDefault(className,typeName,name);

			if (!IsOptExportable(&props[i],value))
			{
				// TrickObject's are a special case and should be converted
				// to the format that the engine expects them in as opposed to standard props
				if (name == CStr("TrickObject"))
				{
					if (props[i].value != CStr("[none]") &&
						props[i].value != CStr(""))
					{
						sprintf(tmpBuf, "\tTrickObject Cluster = %s\n", (char*)props[i].value);
						BUFOUT(tmpBuf);
					}
				}
				else
				{
					// Handle missing properties, if a property is missing it should use the default
					if (props[i].value.Length() == 0 ||
						props[i].value == CStr("[none]"))
					{
						if (value == CStr(""))
						{
							sprintf(tmpBuf,"//\t%s = ?\t// Undefined (Not in scripts.ini)\n",(char*)name);
							BUFOUT(tmpBuf);
						}
						else
							if (value != CStr("[none]"))
							{
								sprintf(tmpBuf,"\t%s = %s\t// Undefined (Defaulted)\n",(char*)name,(char*)value);
								BUFOUT(tmpBuf);
							}
					}
					else
					{
						sprintf(tmpBuf,"\t%s = %s\n",(char*)name,(char*)props[i].value);
						BUFOUT(tmpBuf);
					}
				}
			}
		}
	}

	// Dump out the default commands
	if (strCmds.Length() > 0)
	{
		strCmds=ReplaceStr(strCmds,"\n","\n\t");
		
		if (strCmds.Length()>1)
			strCmds=strCmds.Substr(0,strCmds.Length()-1);

		strCmds = ReplaceStr(strCmds,"\r\n","\n");

		sprintf(tmpBuf,"\t%s",(char*)strCmds);
		BUFOUT(tmpBuf);
	}

	// Attach TriggerScript if necessary
	
	if (programs.GetSize() > 0 || !IsBlank(scriptBuf))
	{
			CStr scriptName=CStr(node->GetName())+CStr("Script");

			sprintf(tmpBuf,"\tTriggerScript = %s\n",(char*)scriptName);
			BUFOUT(tmpBuf);

			scripts.Add(&scriptName);
			ncrcNew.nodeFlags |= NODEFLAG_HAS_SCRIPT;
	}

	////////////// Perform parsing on script buffer for later use (Formerly Exporting Node Scripts) ////////////////
	CStr outBufScript;

	if (programs.GetSize()>0)
	{
		PostMAXScriptNode(node);
		PostParticleMax(node);
		AssignEmitScript(GetEmitScript(node));

		Link<ConfigProgram>* link = programs.GetHead();
		scriptBuf = "";

		// Execute all attached MaxScript programs
		while(link)
		{
			UpdateDynUIScripts(&props,flags,link->data.filename);
			scriptBuf += GetTemplate();

			// Set node flag indicating type of stored programs
			if (link->data.type == ConfigProgram::CFGPROG_VOLATILE)
				ncrcNew.nodeFlags |= NODEFLAG_HAS_VOLATILE_PROG;
			else
				if (link->data.type == ConfigProgram::CFGPROG_STATIC)
					ncrcNew.nodeFlags |= NODEFLAG_HAS_STATIC_PROG;

			link = link->next;
		}
	}

	//if (HasData(scriptBuf))
	if (!IsBlank(scriptBuf))
	{
		// Replace '*' with node name for the script buffer
		scriptBuf = ReplaceStr(scriptBuf,"*", nodeName);

		sprintf(tmpBuf,"script %sScript\n",(char*)nodeName);
		BUFOUT2(tmpBuf);
		
		// Combine the commands from each alternate script buffer into the script
		Link<ConfigScript>* curscript = cscripts.GetHead();

		while(curscript)
		{
			scriptBuf = curscript->data.buffer + scriptBuf + CStr("\n");
			curscript = curscript->next;
		}

		scriptBuf = ReplaceStr(scriptBuf,"\r\n","\n");	

		sprintf(tmpBuf,"%s",scriptBuf);
		BUFOUT2(tmpBuf);

		sprintf(tmpBuf,"endscript\n\n\n");
		BUFOUT2(tmpBuf);

		ncrcNew.nodeFlags |= NODEFLAG_HAS_SCRIPT;
	}

	// Store content of any external scripts /////////////////////////////////////
	/*
	RetrieveExtMAXScriptData();
	
	Link<ScriptBuf>* link = scriptBuffers.GetHead();
	CStr buf;

	while(link)
	{
		//buf = GetTemplateScript(link,&props);
		buf = ReplaceStr(link->data.buffer,"\r\n","\n");	

		sprintf(tmpBuf, "script %s\n",(char*)link->data.scriptName);
		BUFOUT2(tmpBuf);

		sprintf(tmpBuf, "%s",(char*)buf);
		BUFOUT2(tmpBuf);

		sprintf(tmpBuf, "endscript\n\n");
		BUFOUT2(tmpBuf);

		link = link->next;
	}
	*/
	//////////////////////////////////////////////////////////////////////////////

	// Get link information (if this is a link)
	if (obj->ClassID()==vLINK_OBJ_CLASS_ID)
		GetLinks(node);
	else
		GetRefLinks(node);

	GetManualLinks(node);
	DumpLinks();

	////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	// Scan for terrain props (will restore, move to OutputNode)
	
	// ScanTerrainTypes now stores its data in scene root
	// since it will not always get called here if the node hasn't changed
	// ref. LoadTerrainList/SaveTerrainList
	//ScanTerrainTypes(&props, tlist);

	// Store the buffer within the node
	AssignNodeOutput(node, &ncrcNew, outBuf, outBufScript);
	return true;
}

void ScriptExporter::ClearTerrainList(bool* terrain_list)
{
	if (terrain_list)
	{
		for( int i = 0; i < vNUM_TERRAIN_TYPES; i++ )
		{
			terrain_list[i] = false;
		}
	}
}

void ScriptExporter::GetTerrainList(Interface* ip,bool* terrain_list, bool bClear)
{
	// From NExtExportMan::GetTerrainList  (skate3)

	MtlBaseLib* mtl_lib;
	Mtl* mtl; 
	Mtl* sub_mtl;
	//MtlBase* mat;
	int i, j;
	
	assert( terrain_list );

	if (bClear)
	{
		for( i = 0; i < vNUM_TERRAIN_TYPES; i++ )
		{
			terrain_list[i] = false;
		}
	}

	mtl_lib = ip->GetSceneMtls();
	int mlibCount = mtl_lib->Count();
	for( i = 0; i < mlibCount; i++ )
	{
		mtl = (Mtl*) (*mtl_lib)[i];
		
		if( mtl->IsMultiMtl())
		{
			for( j = 0; j < mtl->NumSubMtls(); j++ )
			{
				sub_mtl = mtl->GetSubMtl(j);
				if( sub_mtl )
				{
					if( sub_mtl->ClassID() == NEXT_MATERIAL_CLASS_ID )
					{
						INExtMaterial* next_mat;

						next_mat = dynamic_cast< INExtMaterial* > ( sub_mtl );
						if( next_mat )
						{
							// Warn if this terrain type is not supposed to be used
							if (next_mat->GetTerrainType() > TERRAIN_WARN_MIN && next_mat->GetTerrainType() < TERRAIN_WARN_MAX)
							{
								char bufError[256];
								sprintf(bufError,"The material '%s' is using an invalid terrain type '%s'.\nThis type will be ignored.", (char*)next_mat->GetName(), (char*)next_mat->GetTerrainName( next_mat->GetTerrainType() ) );

								MessageBoxAll(gInterface->GetMAXHWnd(),bufError,"Invalid Terrain Type",MB_ICONWARNING|MB_OK);
								ReportWarning( bufError );
							}

							terrain_list[next_mat->GetTerrainType()] = true;
						}
					}
				}
			}
		}
		else
		{
			if( mtl->ClassID() == NEXT_MATERIAL_CLASS_ID )
			{
				INExtMaterial* next_mat;

				next_mat = dynamic_cast< INExtMaterial* > ( mtl );
				if( next_mat )
				{
					if (next_mat->GetTerrainType() > TERRAIN_WARN_MIN && next_mat->GetTerrainType() < TERRAIN_WARN_MAX)
					{
						char bufError[256];
						sprintf(bufError,"The material '%s' is using an invalid terrain type '%s'.\nThis type will be ignored.", (char*)next_mat->GetName(), (char*)next_mat->GetTerrainName( next_mat->GetTerrainType() ) );

						MessageBoxAll(gInterface->GetMAXHWnd(),bufError,"Invalid Terrain Type",MB_ICONWARNING|MB_OK);
						ReportWarning( bufError );
					}

					terrain_list[next_mat->GetTerrainType()] = true;
				}
			}
		}	
	}
}

void ScriptExporter::UpdatePropTerrains(Tab <INode *>& node_list, bool* terrain_list)
{
	int nNodes = node_list.Count();
	AppDataChunk* appdata;

	for(int i = 0; i < nNodes; i++)
	{
		appdata = node_list[i]->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_TERRAIN_TYPE);

		if (appdata && appdata->data)
		{
			int idx = *((int*)appdata->data);

			if (idx != -1)
				terrain_list[idx] = true;
		}
	}
}

bool ScriptExporter::ShouldExportNode(INode* node)
{
	CStr propBuffer;
	CStr name;
	node->GetUserPropBuffer(propBuffer);

	name = node->GetName();

	// CMD:OMIT can be removed through PE UI,  CMD:FORCEOMIT cannot

	// Don't export items that explicitly have object omission instructions
	if (IsInstr(propBuffer,"// CMD:OMIT") ||
		IsInstr(propBuffer,"// CMD:FORCEOMIT"))
		return false;

	return true;
}

void ScriptExporter::ClearTriggerList()
{
	/*
	triggerList.IterateStart();
	CStr* str;

	while(str = triggerList.IterateNext())
	{
		delete str;
	}
	*/

	triggerList.FlushAllItems();
}

bool ScriptExporter::ScanTerrainTypes(LinkList<ConfigProp>* props, bool* terlist)
{
	Link<ConfigProp>* tcurprop = props->GetHead();
	char cmpTerrain[256];
	bool bResult = false;

	while(tcurprop)
	{
		if (strstr(tcurprop->data.name, "TerrainType"))
		{
			for(int i = 0; i < vNUM_TERRAIN_TYPES; i++)
			{
				strcpy(cmpTerrain, "TERRAIN_");
				strcat(cmpTerrain, terrain_types[i]);
				ucase(cmpTerrain);

				if (strstr(tcurprop->data.value, cmpTerrain))
				{
					terlist[i] = true;
					bResult = true;
				}
			}
		}

		tcurprop = tcurprop->next;
	}

	return bResult;
}

void ScriptExporter::SaveTerrainList(bool* tlist)
{
	ReferenceMaker* scene = gInterface->GetScenePointer();
	AppDataChunk* appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_TERRAINLIST);

	if (appdata)
		scene->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_TERRAINLIST);

	unsigned char* terlist = (unsigned char*)malloc(sizeof(unsigned int) + sizeof(bool) * vNUM_TERRAIN_TYPES);
	unsigned int* listsize = (unsigned int*)terlist;

	scene->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_TERRAINLIST, sizeof(unsigned int) + sizeof(bool) * vNUM_TERRAIN_TYPES, terlist);

	*listsize = vNUM_TERRAIN_TYPES;
	terlist += sizeof(unsigned int);
	memcpy(terlist, tlist, sizeof(bool) * vNUM_TERRAIN_TYPES);
}

bool ScriptExporter::LoadTerrainList(bool* tlist)
{
	ReferenceMaker* scene = gInterface->GetScenePointer();
	AppDataChunk* appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_TERRAINLIST);

	if (appdata && appdata->data)
	{
		unsigned char* pos = (unsigned char*)appdata->data;
		unsigned int size = *((unsigned int*)pos);
		pos += sizeof(unsigned int);

		memcpy(tlist, pos, sizeof(bool) * size);
		return true;
	}

	return false;
}

bool ScriptExporter::ExportScriptsFromSet( char* scene_name, Tab <INode *> &m_exportable_nodes, bool bModel, char* modelName )
{
	CStr FileName=getenv(APP_ENV);
	char qcomp_path[_MAX_PATH];
	char* project_root;
	char *args[3];

	// Verify export names
	int nNodes = m_exportable_nodes.Count();
	int i;

	for(i = 0; i < nNodes; i++)
	{
		if (!VerifyNodeNames(m_exportable_nodes[i]))
			return false;
	}

	ClearTerrainList(tlist);
	//LoadTerrainList(tlist);

	AssignSharedData();

	FindSFPNodes(SFPNodes);
	PreProcessSFPNodes(SFPNodes);

	ClearTriggerList();
	altscripts.Clear();

	MessageBoxResetAll();

	if (FileName==CStr(""))
	{
		char ErrorMsg[256];
		sprintf(ErrorMsg,"Couldn't export script the environment variable '%s' is undefined.",APP_ENV);
		MessageBoxAll(NULL,ErrorMsg,"ExportScripts",MB_ICONSTOP|MB_OK);
		return false;
	}

	// Make the target dir
	CStr MkDir;
	
	if (bModel)
	{
		ModelExportOptions meo;
		GetModelExportOptions(&meo);
		
		if (meo.m_ExportToSceneDir)
		{
			SceneExportOptions seo;
			GetSceneExportOptions(&seo);

			if (modelName)
				MkDir=CStr(getenv(APP_ENV))+CStr(MODEL_QN_PATH)+ seo.m_SceneName + CStr("\\") + meo.m_ModelDir + CStr("\\") + modelName;
			else
				MkDir=CStr(getenv(APP_ENV))+CStr(MODEL_QN_PATH)+ seo.m_SceneName + CStr("\\") + meo.m_ModelDir + CStr("\\") + meo.m_ModelName;
		}
		else
		{
			if (modelName)
				MkDir=CStr(getenv(APP_ENV))+CStr(MODEL_QN_PATH)+ meo.m_ModelDir + CStr("\\") + modelName;
			else
				MkDir=CStr(getenv(APP_ENV))+CStr(MODEL_QN_PATH)+ meo.m_ModelDir + CStr("\\") + meo.m_ModelName;
		}
	}
	else
		MkDir=CStr(getenv(APP_ENV))+CStr(QN_PATH)+CStr(scene_name);

	MDir(MkDir);

	// The filename should be in the format "c:\skate4\data\levels\la\la.qn"
	if (bModel)
	{
		ModelExportOptions meo;
		GetModelExportOptions(&meo);
		
		if (meo.m_ExportToSceneDir)
		{
			SceneExportOptions seo;
			GetSceneExportOptions(&seo);

			if (modelName)
				FileName+=CStr(MODEL_QN_PATH)+ seo.m_SceneName + CStr("\\") + meo.m_ModelDir + CStr("\\") + CStr(modelName) + CStr("\\") + scene_name + CStr(".qn");
			else
				FileName+=CStr(MODEL_QN_PATH)+ seo.m_SceneName + CStr("\\") + meo.m_ModelDir + CStr("\\") + meo.m_ModelName + CStr("\\") + scene_name + CStr(".qn");
		}
		else
		{
			if (meo.m_ModelDir.Length()==0)
			{
				if (modelName)
					FileName+=CStr(MODEL_QN_PATH)+ CStr(modelName) + CStr("\\") + scene_name + CStr(".qn");
				else
					FileName+=CStr(MODEL_QN_PATH)+ meo.m_ModelName + CStr("\\") + scene_name + CStr(".qn");
			}
			else
			{
				if (modelName)
					FileName+=CStr(MODEL_QN_PATH)+ CStr(modelName) + CStr("\\") + meo.m_ModelName + CStr("\\") + scene_name + CStr(".qn");
				else
					FileName+=CStr(MODEL_QN_PATH)+ meo.m_ModelDir + CStr("\\") + meo.m_ModelName + CStr("\\") + scene_name + CStr(".qn");
			}
		}
	}
	else
		FileName+=CStr(QN_PATH)+CStr(scene_name)+CStr("\\")+CStr(scene_name)+CStr(".qn");

	CStr strProps;

	// Generate the .qn file for the level
	fp=fopen(FileName,"w");

	if (!fp)
	{
		// Check the file attributes to see if the .qn is read-only
		if (_access(FileName,WRITE_ACCESS)!=0)
		{
			char message[256];
			int result;
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", FileName );
			result = MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_CHECKOUT_UNLOCK );
			if( result == IDWRITABLE )
			{
				// break the lock, if necessary
				SetFileAttributes( FileName, FILE_ATTRIBUTE_NORMAL );
			}
			else if( result == IDCHECKOUT )
			{
				char *args[4];
				int process_result;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = FileName;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );
				process_result = ExecuteCommand( args );
				if (_access(FileName,WRITE_ACCESS)!=0)
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Script",MB_ICONSTOP|MB_OK);
					return false;
				}
			}
			else
			{
				return false;
			}
		}
		else
		{
			char ErrorMsg[256];
			sprintf(ErrorMsg,"Unable to save file.  Check file attributes.\nPath: %s",FileName);
			MessageBox(NULL,ErrorMsg,"ScriptExport Error",MB_ICONSTOP|MB_OK);
			return false;
		}

		fp=fopen(FileName,"w");
		if (!fp)
		{
			char strErr[256];
			sprintf(strErr, "Couldn't save script file '%s'. Operation Aborted.", (char*)FileName);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "ScriptExporter Problem", MB_ICONWARNING|MB_OK);
			return false;
		}
	}

	// Number all the nodes for later link processing
	CreateNodeIDList(m_exportable_nodes);

	scripts.Clear();

	//ClearTerrainList(tlist);

	// Export node array
	gInterface->ProgressStart(_T("Exporting NodeArray"), TRUE, 
					ScriptExportProgressFunc, NULL );

//////////////////////////////////////////  Export NodeArray  /////////////////////////////////////////////////////

	if (bModel)
		fprintf(fp, "%s_NodeArray = \n[\n",scene_name);
	else
		fprintf(fp,"NodeArray = \n[\n");

	int numNodes=m_exportable_nodes.Count();

	for(i=0;i<numNodes;i++)
	{
		if (ShouldExportNode(m_exportable_nodes[i]))
		{
			// Determine if this is a duplicate node and add it to the trigger list
			Object* obj = m_exportable_nodes[i]->EvalWorldState(0).obj;

			if ( obj && obj->ClassID() == vTRIGGER_CLASS_ID )
			{
				unsigned long val = GenerateCRC(m_exportable_nodes[i]->GetName());

				if (triggerList.GetItem(val))
				{
					char sbuf[256];
					sprintf(sbuf,"There are multiple trigger nodes with the name '%s'.",(char*)m_exportable_nodes[i]->GetName());
					MessageBoxAll(gInterface->GetMAXHWnd(),sbuf,"Duplicate Trigger",MB_ICONWARNING|MB_OK);
					ReportWarning( sbuf );
				}
				else
				{
					triggerList.PutItem(val, &val);
				}
			}

			fprintf(fp,"{\n");
			fprintf(fp,"\t// Node %d\n",i);
			
			if (!OutputNode(m_exportable_nodes[i]))
			{
				fclose(fp);
				gInterface->ProgressEnd();
				//SaveTerrainList(tlist);
				return false;
			}

			fprintf(fp,"}\n");
		}

		// Process nodes that contain screen facing poly effects
		/*
		CStr value;
		if (m_exportable_nodes[i]->GetUserPropString("SFP_Effect",value) &&
			value != CStr("None"))
		{
			SFPNodes.Append(1,&m_exportable_nodes[i]);
		}
		*/

		if (gInterface->GetCancel())
		{
			fclose(fp);
			gInterface->ProgressEnd();
			//SaveTerrainList(tlist);
			return false;
		}

		gInterface->ProgressUpdate( i * 100 / numNodes );
	}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

	gInterface->ProgressEnd();

	fprintf(fp,"]\n\n");

	// Dump the trigger scripts
	Link<CStr>* curNode=scripts.GetHead();

	if (bModel)
		fprintf(fp,"%sTriggerScripts = \n[\n",scene_name);
	else
		fprintf(fp,"TriggerScripts = \n[\n");
	
	if (!bModel)
	{
		// Force auto-generated LoadTerrain script
		fprintf(fp,"\tLoadTerrain\n");
	}

	while(curNode)
	{
		fprintf(fp,"\t%s\n",(char*)curNode->data);
		curNode=curNode->next;
	}

	fprintf(fp,"]\n\n");

	// The load particle texture script should not be dumped out when
	// exporting models or skins, only scenes
	ExportOptions* options = GetExportOptions();

	if (options->m_ExportType == ExportOptions::vEXPORT_SCENE)
	        OutputLoadParticleTexturesScript();

	// Go back through and dump the attached scripts
	gInterface->ProgressStart(_T("Exporting node scripts"), TRUE, 
					ScriptExportProgressFunc, NULL );

	// Reset any alternate script data that may exist from previous
	// script updates.  The buffers will be built up from this point
	ClearAltScripts();

//////////////////////////////////// Export Node scripts ///////////////////////////////

	for(i=0;i<numNodes;i++)
	{
		//////////////////////////////////////////////////////////////////////////////////////
		// We should be garunteed at this point that all nodes with scripts/programs in the scene have
		// script output buffer data since OutputNode has been already called on everything
		CStr    outBuf;
		NodeCRC ncrc;

		ncrc.Retrieve(m_exportable_nodes[i]);

		if (ncrc.nodeFlags & NODEFLAG_HAS_SCRIPT)
		{
			if (ncrc.nodeFlags & NODEFLAG_HAS_VOLATILE_PROG)
			{
				CStr                    scriptBuf, propBuf, className, typeName;
				DWORD                   flags;
				LinkList<ConfigProp>    props;
				LinkList<ConfigProgram> programs;
				LinkList<ConfigScript>  cscripts;

				m_exportable_nodes[i]->GetUserPropBuffer(propBuf);

				className = GetClassName(propBuf);
				typeName  = GetTypeName(propBuf);

				scriptBuf = ParseNodeConfigProps(&props,&flags,m_exportable_nodes[i],NULL,NULL,&programs,&cscripts);
				AddGlobalDefaults(propBuf,&props,&flags,NULL,className,typeName);

				CombineDynUIProps(&props,className,typeName);
				ConvertToDefaults(className,typeName,&props);

				// The node contains a volatile program so we must reevaluate it now
				if (programs.GetSize()>0)
				{
					PostMAXScriptNode(m_exportable_nodes[i]);
					PostParticleMax(m_exportable_nodes[i]);
					AssignEmitScript(GetEmitScript(m_exportable_nodes[i]));

					Link<ConfigProgram>* link = programs.GetHead();
					scriptBuf = "";

					// Execute all attached MaxScript programs
					while(link)
					{
						UpdateDynUIScripts(&props,flags,link->data.filename);
						scriptBuf += GetTemplate();

						link = link->next;
					}
				}

				if (HasData(scriptBuf))
				{
					CStr nodeName = m_exportable_nodes[i]->GetName();

					// Replace '*' with node name for the script buffer
					scriptBuf = ReplaceStr(scriptBuf,"*", nodeName);

					fprintf(fp,"script %sScript\n",(char*)nodeName);
					
					// Combine the commands from each alternate script buffer into the script
					Link<ConfigScript>* curscript = cscripts.GetHead();

					while(curscript)
					{
						scriptBuf = curscript->data.buffer + scriptBuf + CStr("\n");
						curscript = curscript->next;
					}

					scriptBuf = ReplaceStr(scriptBuf,"\r\n","\n");	
					fprintf(fp,"%s",scriptBuf);
					fprintf(fp,"endscript\n\n\n");
				}
			}
			else
			{
				// No volatile programs attached to this node we can safely use the data
				// cached within the nodes buffer
				if (GetNodeOutputScript(m_exportable_nodes[i], NULL, outBuf))
				{
					fputs(outBuf, fp);
				}
				else
				{
					char strErr[512];
					sprintf(strErr, "The node '%s' does not contain script output buffer data despite being flagged as such!\nContact Adam\n", (char*)m_exportable_nodes[i]->GetName());
					MessageBoxAll(gInterface->GetMAXHWnd(), strErr, "Internal Next.gup Plugin Problem!", MB_ICONWARNING|MB_OK);
					ReportWarning( strErr );
				}
			}
		}

		/////////////////////////////////////////////////////////////////////////////////////

		/*
		CStr  propBuf;
		CStr  scriptBuf;
		LinkList<ConfigProgram> programs;
		CStr  className, typeName;
		DWORD flags;

		LinkList<ConfigScript> cscripts;

		m_exportable_nodes[i]->GetUserPropBuffer(propBuf);

		// Parse all the config property data
		//AddGlobalDefaults(propBuf);

		//if (!m_exportable_nodes[i]->GetUserPropString("Class",className))
		//	m_exportable_nodes[i]->GetUserPropString("class",className);
		
		className = GetClassName(m_exportable_nodes[i]);

		//if (!m_exportable_nodes[i]->GetUserPropString("Type",typeName))
		//	m_exportable_nodes[i]->GetUserPropString("type",typeName);

		typeName = GetTypeName(m_exportable_nodes[i]);

		scriptBuf=ParseNodeConfigProps(&props,&flags,m_exportable_nodes[i],NULL,NULL,&programs,&cscripts);
		//scriptBuf=ParseConfigProps(&props,&flags,propBuf,NULL,NULL,&programs,&cscripts);
		AddGlobalDefaults(propBuf,&props,&flags,NULL,className,typeName);

		CombineDynUIProps(&props,className,typeName);
		ConvertToDefaults(className,typeName,&props);
		*/

// <snip>


		if (gInterface->GetCancel())
		{
			fclose(fp);
			gInterface->ProgressEnd();
			//SaveTerrainList(tlist);
			return false;
		}

		/*
		// Go through the alternate scripts and accumulate them
		Link<ConfigScript>* curscript = cscripts.GetHead();
		char eol[3];
		eol[0] = 10;
		eol[1] = 13;
		eol[2] = 0;

		while(curscript)
		{
			AltScript srch;
			srch.filename = curscript->data.filename;

			Link<AltScript>* link = altscripts.Find(&srch);

			if (!link)
				link = altscripts.Add(&srch);

			// Add the script
			link->data.buffer += CStr("script ") + curscript->data.name + eol;
			link->data.buffer += curscript->data.buffer + eol;
			link->data.buffer += CStr("endscript") + eol + eol;

			curscript = curscript->next;
		}
		*/

		gInterface->ProgressUpdate( i * 100 / numNodes );
	}
////////////////////////////////////////////////////////////////////////////////////////////////////////////

	gInterface->ProgressEnd();

	if (!bModel)
	{
		// Dump Terrain script
		//GetTerrainList(gInterface,tlist);
		GetTerrainList(gInterface,tlist,false);
		UpdatePropTerrains(m_exportable_nodes, tlist);

		fprintf(fp,"script LoadTerrain\n");

		for(i=0;i<vNUM_TERRAIN_TYPES;i++)
		{
			if (tlist[i])
			{
				// This would normally be bad, but GetTerrainName simply returns a const value
				INExtMaterial* nextMat=NULL;

				// Comment out if this is a terrain that was removed
				//if (i > 40 && i < 50)
				//	fprintf(fp,"// ");

				fprintf(fp,"SetTerrain%s\n",nextMat->GetTerrainName(i));
			}
		}
	
		fprintf(fp,"endscript\n\n");

		// Dump out the animation load script
		DumpLoadAnims();

		// Dump out the LoadCameras script
		CameraExporter* camExporter=GetCamExporter();
		assert(camExporter);

		camExporter->ExportQN(fp);
		fprintf(fp,"\n");

		// Dump out the LoadObjectAnims script
		ObjectExporter* objExporter=GetObjExporter();
		assert(objExporter);

		objExporter->ExportQN(fp);
		fprintf(fp,"\n");
	}

	DumpExtScripts();
	ClearAltScripts();

	PostProcessSFPNodes(SFPNodes);

	fclose(fp);
	//SaveTerrainList(tlist);

	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	if (seo.m_QNCompressionLevel != OPTLEVEL_NONE)
	{
		ScriptOptimizer* pScriptOptimizer = new ScriptOptimizer;
		pScriptOptimizer->SetOptimizeLevel(seo.m_QNCompressionLevel);
		pScriptOptimizer->OptimizeScripts(FileName);
	}

	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( GetQuickviewPlatform() == vQUICKVIEW_PS2 )
	{
		sprintf( qcomp_path, "%s\\bin\\win32\\qcomp.exe", project_root );
	}
	else if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		sprintf( qcomp_path, "%s\\bin\\win32\\qbrx.bat", project_root );
	}	
	else // NGC
	{
		sprintf( qcomp_path, "%s\\bin\\win32\\qbrg.bat", project_root );
	}

	args[0] = qcomp_path;
	args[1] = FileName;	
	args[2] = NULL;	

	//int result = ( spawnv( _P_WAIT, args[0], args ) == 0 );					
	int result = ExecuteCommand( args );

// This recompiles the script, but doesn't add the files to the PRE file
//	CString cExec;	
//	cExec.Format("qcomp %s -dir %s\\data\\%s", filename_q, skateRoot, PS2_SCRIPT_DIR );
	
// A better solution is to run the "qcompall" batch file, which should
// theoretically update all of your files before running the game
	//int result = (int)ShellExecute(NULL, "open", "qcompall",
              //NULL, NULL, SW_HIDE);

	//DumpAltScripts();

	return true;
}

bool GetExtents(INode* node, Point3& min, Point3& max)
{
	// Convert to a tri object and scan the mesh to determine it's XYZ extents  aml
	Object* obj=node->EvalWorldState(0).obj;

	if (obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID,0)))
	{
		TriObject* triobj=(TriObject*)obj->ConvertToType(0,Class_ID(TRIOBJ_CLASS_ID,0));

		Mesh& mesh=triobj->GetMesh();

		if (mesh.numVerts==0)
			return false;

		// Assign min/max to start
		min=max=mesh.verts[0];

		min=min;
		max=max;

		for(int i=0;i<mesh.numVerts;i++)
		{
			Point3 pt=mesh.verts[i];

			if (pt.x<min.x)
				min.x=pt.x;

			if (pt.y<min.y)
				min.y=pt.y;

			if (pt.z<min.z)
				min.z=pt.z;

			if (pt.x>max.x)
				max.x=pt.x;

			if (pt.y>max.y)
				max.y=pt.y;

			if (pt.z>max.z)
				max.z=pt.z;
		}

		if (obj!=triobj)
			triobj->DeleteThis();

		return true;
	}

	return false;
}

void ScriptExporter::PostParticleMax(INode* node)
{
	// Scan through the current node list and find this node
	// then post the maximum count for the SFP system type
	MtlEntry srch;
	srch.mtl = node->GetMtl();
	CStr className, typeName;

	GetSFPEffect(node,srch.type);
	
	// If the value is defaulted, look up the default value
	if (srch.type == CStr("!"))
	{
		if (!node->GetUserPropString("Class",className))
			node->GetUserPropString("class",className);

		if (!node->GetUserPropString("Type",typeName))
			node->GetUserPropString("type",typeName);

		srch.type = GetDefault(className,typeName,"SFP_Effect");
	}

	Link<MtlEntry>* link = mtlList.Find(&srch);

	// This node is not an SFP node or the SFP material list was not built
	if (!link)
		return;

	AssignSharedParticleMax(link->data.nodes.Count());
}

void	ScriptExporter::AddParticleTexture( CStr particle_tex )
{
	int i;
	CStr tex;

	tex = particle_tex;
	for( i = 0; i < particle_tex.Length(); i++ )
	{
		if( particle_tex[i] == '.' )
		{
			tex = particle_tex.remove( i );
		}
	}
	
	for( i = 0; i < particle_tex_list.GetSize(); i++ )
	{
		if( stricmp( particle_tex_list[i], tex ) == 0 )
		{
			return;
		}
	}

	particle_tex_list.Add( &tex );
}

void	ScriptExporter::ClearParticleTextureList( void )
{
	particle_tex_list.Clear();
}

void	ScriptExporter::OutputLoadParticleTexturesScript( void )
{
	int i;

	fprintf(fp,"script LoadAllParticleTextures\n");

	for( i = 0; i < particle_tex_list.GetSize(); i++ )
	{
		fprintf(fp,"\tLoadParticleTexture \"particles\\%s\"\n", particle_tex_list[i] );
	}
	fprintf(fp,"endscript\n\n");
}

CStr ScriptExporter::GetEmitScript(INode* node)
{
	Mtl* mtl = node->GetMtl();

	if (!mtl)
		return CStr("");

	if (mtl->ClassID() != NEXT_MATERIAL_CLASS_ID)
		return CStr("");

	CStr type;

	if (!GetSFPEffect(node,type))
		return CStr("");

	if (type == CStr("!"))
	{
		CStr className, typeName;

		if (!node->GetUserPropString("Class",className))
			node->GetUserPropString("class",className);

		if (!node->GetUserPropString("Type",typeName))
			node->GetUserPropString("type",typeName);

		type = GetDefault(className,typeName,"SFP_Effect");
	}

	INExtMaterial* nxMtl = dynamic_cast<INExtMaterial*>(mtl);

	// Get first pass texture name
	Texmap* tmap = nxMtl->GetTexmap(0);

	// Only bitmap textures, abort if proceedural
	if (tmap->ClassID() != Class_ID(BMTEX_CLASS_ID,0) &&
		tmap->ClassID() != NEXT_TEXTURE_CLASS_ID)
		return CStr("");
	
	BitmapTex* bmtmap = (BitmapTex*)tmap;
	CStr texName = ExtractFile(bmtmap->GetMapName());

	return ReplaceStr(CStr("SFPInit_")+texName+CStr("_")+type," ","_");
}

void ScriptExporter::PostProcessSFPNodes(INodeTab& nodes)
{
	CStr propBuf;
	LinkList<ConfigProp> cprops;
	DWORD flags;
	CStr  className, typeName;

	Link<MtlEntry>* link = mtlList.GetHead();

	if (link)
		fprintf(fp,"//// Generated via sfpemit.ms\n");

	while(link)
	{
		CStr name;
		CStr scriptBuf;
		CStr fullname;

		if (link->data.mtl)
		{
			Mtl* mtl = link->data.mtl;

			if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
			{
				INExtMaterial* nxMat = dynamic_cast< INExtMaterial* > ( mtl );
				Texmap* tmap = nxMat->GetTexmap(0);

				// Only bitmap textures, abort if proceedural
				if (tmap->ClassID() == Class_ID(BMTEX_CLASS_ID,0) ||
					tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
				{
					BitmapTex* bmtmap = (BitmapTex*)tmap;
					fullname = bmtmap->GetMapName();
					name = ExtractFile(bmtmap->GetMapName());
				}			
			}
		}

		name = ReplaceStr(name," ","_");

		fprintf(fp,"script SFPInit_%s_%s\n",(char*)name,(char*)link->data.type);

		for(int i = 0; i < link->data.nodes.Count(); i++)
		{
			INode* node = link->data.nodes[i];

			// Get class and type
			if (!nodes[i]->GetUserPropString("Class",className))
				nodes[i]->GetUserPropString("class",className);

			if (!nodes[i]->GetUserPropString("Type",typeName))
				nodes[i]->GetUserPropString("type",typeName);

			// Get properties and add/convert data
			nodes[i]->GetUserPropBuffer(propBuf);

			ParseConfigProps(&cprops,&flags,propBuf,NULL,NULL,NULL,NULL);
			AddGlobalDefaults(propBuf,&cprops,&flags,NULL,className,typeName);
			CombineDynUIProps(&cprops,className,typeName);
			ConvertToDefaults(className,typeName,&cprops);

			// Post data to be available through MAXScript
			PostMAXScriptNode(nodes[i]);
			PostParticleMax(nodes[i]);
			AssignEmitScript(GetEmitScript(nodes[i]));

			UpdateDynUIScripts(&cprops,flags,"sfpemit.ms");
			scriptBuf += GetTemplate();
		}

		scriptBuf = ReplaceStr(scriptBuf,"\r\n","\n");
		fprintf(fp,"%s\n",scriptBuf);
		fprintf(fp,"endscript\n");

		// Copy this material to the appropriate dir
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		CStr path = CStr(getenv(APP_ENV)) + CStr(QN_PATH) + CStr(seo.m_SceneName) + CStr("\\");
		CStr destname = path + CStr("sfp\\") + name + CStr(".png");
		mkdir(path + "sfp");
		CopyFile(fullname,destname,false);

		link = link->next;
	}
}

Mtl* GetNExtActualMtl(INode* node)
{
	Mtl* mtl = node->GetMtl();

	if (!mtl)
		return NULL;

	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		return mtl;
	}

	if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID)
	{
		// We'll choose the sub material by the mtlID of the first face in the mesh
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh* mesh = &triObj->mesh;
			
			int mtlID = mesh->getFaceMtlIndex(0);
			Mtl* submtl = mtl->GetSubMtl(mtlID % mtl->NumSubMtls());

			if (submtl && submtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
			{
				if (triObj != obj)
					triObj->DeleteThis();

				return mtl;
			}

			if (triObj != obj)
				triObj->DeleteThis();

			return NULL;
		}
		else
			return NULL;
	}

	return NULL;
}

void ScriptExporter::PreProcessSFPNodes(INodeTab& nodes)
{
	if (!fp)
		return;

	mtlList.Clear();

	int count = nodes.Count();

	// Run through all the SFP nodes and group them by material
	for(int i=0;i<count;i++)
	{
		MtlEntry srch;
		CStr     className, typeName;
		//srch.mtl = nodes[i]->GetMtl();
		srch.mtl = GetNExtActualMtl(nodes[i]);
		
		GetSFPEffect(nodes[i],srch.type);

		if (srch.type == CStr("!"))
		{
			if (!nodes[i]->GetUserPropString("Class",className))
				nodes[i]->GetUserPropString("Class",className);

			if (!nodes[i]->GetUserPropString("Type",typeName))
				nodes[i]->GetUserPropString("type",typeName);

			srch.type = GetDefault(className,typeName,"SFP_Effect");
		}

		Link<MtlEntry>* entry = mtlList.Find(&srch);
		
		if (entry)
			entry->data.nodes.Append(1, &nodes[i]);
		else
		{
			srch.nodes.Append(1, &nodes[i]);
			mtlList.Add(&srch);
		}
	}

	Link<MtlEntry>* curlink = mtlList.GetHead();

	// Ensure that there's at least one valid material in the list
	/*
	bool bMtlExists = false;

	while(curlink)
	{
		if (curlink->data.mtl && curlink->data.mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
		{
			bMtlExists = true;
			break;
		}

		curlink = curlink->next;
	}
	*/

/*	// Don't generate the script unless data will be added to it
	if (curlink)// && bMtlExists)
	{
		fprintf(fp,"script SFPInit\n");
		fprintf(fp,"// Autogenerated script to activate all screen facing poly particle systems\n");

		while(curlink)
		{
			if (!curlink->data.mtl || curlink->data.mtl->ClassID() != NEXT_MATERIAL_CLASS_ID)
			{
				curlink = curlink->next;
				continue;
			}

			INExtMaterial* nxMtl = dynamic_cast<INExtMaterial*>(curlink->data.mtl);

			// Get first pass texture name
			Texmap* tmap = nxMtl->GetTexmap(0);

			// Only bitmap textures, abort if proceedural
			if (tmap->ClassID() != Class_ID(BMTEX_CLASS_ID,0) &&
				tmap->ClassID() != NEXT_TEXTURE_CLASS_ID)
			{
				curlink = curlink->next;
				continue;
			}
			
			BitmapTex* bmtmap = (BitmapTex*)tmap;
			CStr texName = ExtractFile(bmtmap->GetMapName());

			CStr objName = ReplaceStr(CStr("Effect_")+texName+CStr("_")+curlink->data.type," ","_");
			int  max = curlink->data.nodes.Count();
			int  blendMode = nxMtl->GetBlendMode(0);
			int  fixedAlpha = nxMtl->GetFixedValue(0);
			CStr emitName = ReplaceStr(CStr("Emit_")+texName+CStr("_")+curlink->data.type," ","_");

			fprintf(fp,
					"CreateParticleSystem name=%s max=%i blendmode=%s fix=%i type=%s texture=%s emitscript=SFPInit_%s\n",
					(char*)objName,
					max,
					blend_modes[blendMode],
					fixedAlpha,
					(char*)curlink->data.type,
					(char*)texName,
					(char*)emitName
					);

			curlink = curlink->next;
		}
//		fprintf(fp,"endscript\n\n");
/*
		curlink = mtlList.GetHead();
		while(curlink)
		{
			if (!curlink->data.mtl || curlink->data.mtl->ClassID() != NEXT_MATERIAL_CLASS_ID)
			{
				curlink = curlink->next;
				continue;
			}

			INExtMaterial* nxMtl = dynamic_cast<INExtMaterial*>(curlink->data.mtl);

			// Get first pass texture name
			Texmap* tmap = nxMtl->GetTexmap(0);

			// Only bitmap textures, abort if proceedural
			if (tmap->ClassID() != Class_ID(BMTEX_CLASS_ID,0) &&
				tmap->ClassID() != NEXT_TEXTURE_CLASS_ID)
			{
				curlink = curlink->next;
				continue;
			}
			
			BitmapTex* bmtmap = (BitmapTex*)tmap;
			CStr texName = ExtractFile(bmtmap->GetMapName());

			CStr emitName = ReplaceStr(CStr("Emit_")+texName+CStr("_")+curlink->data.type," ","_");

			int cnt = curlink->data.nodes.Count();

			if (cnt>0)
			{
				fprintf(fp,"script SFPInit_%s\n", (char*)emitName);

				for(int i=0;i<cnt;i++)
				{
					INode* node = curlink->data.nodes[i];
					Box3   box;
					float  width, height;
					Point3 min,max;

					Point3 pos = ComputePosition(node);		
					GetExtents(node,min,max);

					width  = max.x - min.x;
					height = max.y - min.y;

					// If either of these are 0 then most likely the wrong axis was used
					// choose the Z axis instead to get the intended values
					if (width == 0)
						width = max.z - min.z;

					if (height == 0)
						height = max.z - min.z;

					Color color = nxMtl->GetColor(0);
					int r,g,b,a;

					r = (int)(color.r * 255.0f);
					g = (int)(color.g * 255.0f);
					b = (int)(color.b * 255.0f);
					a = 255;

					fprintf(fp,"setpos pos=(%f,%f,%f)\n", pos.x, pos.y, -pos.z);
					fprintf(fp,"setparticlesize sw=%f sh=%f\n",width,height);
					fprintf(fp,"setcolor corner=0 er=%i eg=%i eb=%i ea=%i\n",r,g,b,a);
					fprintf(fp,"emit num=1\n");
				}

				fprintf(fp,"endscript\n\n");
			}

			curlink = curlink->next;
		}
	}		// end if (curlink)
*/
}

IScriptExporter* GetScriptExporter( void )
{
	return &s_script_exporter;
}

