// Check.cpp :
//

#include "stdafx.h"
#include <stdlib.h>
#include <stdio.h>
#include "\skate4\code\gel\scripting\tokens.h"

using namespace Script;

typedef unsigned long uint32;
typedef unsigned char uint8;

// This is so that skiptoken.cpp can be #included.
#define Dbg_MsgAssert( _c, _params )							\
																\
if( !( _c ))													\
{																\
	MsgAssert_PrintfIntoBuffer _params;							\
	Assert( 0, pMsgAssert_Buffer );								\
}

char *pFormats=NULL;
uint8 *pQBFile=NULL;

#define WORD_BUFFER_SIZE 100
char WordBuffer[WORD_BUFFER_SIZE+1];

#define MAX_CHECKSUM_NAME_CHARS 50
struct SChecksumName
{
	uint32 Checksum;
	char pName[MAX_CHECKSUM_NAME_CHARS+1];
};

#define MAX_CHECKSUM_NAMES 1000
SChecksumName ChecksumNames[MAX_CHECKSUM_NAMES];
int NumChecksumNames=0;

bool ErrorFound=false;

#define MSGASSERT_BUFFER_SIZE 1000
static char pMsgAssert_Buffer[MSGASSERT_BUFFER_SIZE];
void MsgAssert_PrintfIntoBuffer(const char* p_text, ...)
{
	pMsgAssert_Buffer[0]=0;
	if (p_text)
	{
		va_list args;
		va_start( args, p_text );

		sprintf( pMsgAssert_Buffer, p_text, args);
		
		va_end( args );
	}
}

void Assert(int Condition, char *pMessage)
{
	if (!Condition)
	{
		MessageBox(NULL,pMessage,"Check.exe assertion failed.",MB_OK);
		exit(EXIT_FAILURE);
	}
}

void Error(const char *pMessage)
{
	MessageBox(NULL,pMessage,"Check.exe",MB_OK);
	ErrorFound=true;
}

const char *FindChecksumName(uint32 Checksum)
{
	for (int i=0; i<NumChecksumNames; ++i)
	{
		if (ChecksumNames[i].Checksum==Checksum)
		{
			return ChecksumNames[i].pName;
		}
	}
	return NULL;
}

void AddChecksumName(uint32 Checksum, const char *pName)
{
	Assert(NumChecksumNames<MAX_CHECKSUM_NAMES,"Too many checksums");
	const char *pFoundName=FindChecksumName(Checksum);
	if (pFoundName)
	{
		Assert(stricmp(pFoundName,pName)==0,"Checksum clash");
		return;
	}
	ChecksumNames[NumChecksumNames].Checksum=Checksum;
	strncpy(ChecksumNames[NumChecksumNames].pName,pName,MAX_CHECKSUM_NAME_CHARS);
	ChecksumNames[NumChecksumNames].pName[MAX_CHECKSUM_NAME_CHARS]=0;
	++NumChecksumNames;
}

// -------------------- Checksum stuff ------------------------

unsigned long CRC_TABLE[256] = // CRC polynomial 0xedb88320
{
      0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
      0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
      0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
      0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
      0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
      0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
      0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
      0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
      0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
      0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
      0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
      0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
      0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
      0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
      0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
      0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
      0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
      0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
      0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
      0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
      0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
      0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
      0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
      0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
      0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
      0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
      0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
      0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
      0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
      0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
      0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
      0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
      0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
      0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
      0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
      0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
      0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
      0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
      0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
      0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
      0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
      0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
      0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
      0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
      0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
      0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
      0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
      0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
      0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
      0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
      0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
      0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
      0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
      0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
      0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
      0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
      0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
      0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
      0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
      0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
      0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
      0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
      0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
      0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

unsigned long GenerateChecksum(const char *pString)
{
	// Initializing the CRC to all one bits avoids failure of detection
	// should entire data stream get cyclically bit-shifted by one position.
	// The calculation of the probability of this happening is left as
	// an exercise for the reader.
	unsigned long rc = 0xffffffff;
	int Len=strlen(pString);
	for (int i=0; i<Len; i++)
	{
		char ch=pString[i];
        // Convert to lower case.
        if (ch>='A' && ch<='Z') ch='a'+ch-'A';
		rc = CRC_TABLE[(rc^ch) & 0xff] ^ ((rc>>8) & 0x00ffffff);
	}
	
	Assert(rc,"Checksum is zero !!!");
	AddChecksumName(rc,pString);
	return rc;
}

const char *GetTypeName(uint8 Type)
{
	switch (Type)
	{
	case ESCRIPTTOKEN_NAME:					return "Name"; break;
	case ESCRIPTTOKEN_INTEGER:				return "Integer"; break;
	case ESCRIPTTOKEN_HEXINTEGER:			return "Hex integer"; break;
	case ESCRIPTTOKEN_FLOAT:				return "Float"; break;
	case ESCRIPTTOKEN_STRING:				return "String"; break;
	case ESCRIPTTOKEN_LOCALSTRING:			return "Local string"; break;
	case ESCRIPTTOKEN_STARTARRAY:			return "Array"; break;
	case ESCRIPTTOKEN_VECTOR:				return "Vector"; break;
	case ESCRIPTTOKEN_PAIR:					return "Pair"; break;
	default: Assert(0,"Unknown type in call to GetTypeName"); break;
	}

	return "";
}

void ErrorRequiredComponent(const char *pCh)
{
	Assert(*pCh=='{',"Expected *pCh=='{'");
	++pCh;
	char Buf[1000+1];
	sprintf(Buf,"Missing required component:\n\n");
	int c=strlen(Buf);
	while (*pCh!='}')
	{
		Assert(c<1000,"Too many chars in {...}");
		Buf[c++]=*pCh++;
	}
	Buf[c]=0;
	Error(Buf);
}

void ErrorWrongType(uint32 Checksum, uint8 CorrectType, uint8 WrongType)
{
	char Buf[1000];
	sprintf(Buf,"'%s' is required to be of type %s, not %s",FindChecksumName(Checksum),GetTypeName(CorrectType),GetTypeName(WrongType));
	Error(Buf);
}

void ErrorTwoComponentsWithSameName(uint32 Checksum)
{
	char Buf[1000];
	sprintf(Buf,"Duplicate component name '%s'",FindChecksumName(Checksum));
	Error(Buf);
}

const char *SkipWhiteSpaceAndComments(const char *pCh)
{
	while (true)
	{
		// Skip white space
		while (*pCh==' ' || *pCh=='\t' || *pCh==0x0d || *pCh==0x0a)
		{
			++pCh;
		}
		// Skip any comment
		if (*pCh==';')
		{
			++pCh;
			// Skip to end of line.
			while (*pCh && *pCh!=0x0a)
			{
				++pCh;
			}
			if (*pCh)
			{
				// Skip over the 0x0a
				++pCh;
			}
			// There may be more white space and comments to come, so repeat.
			continue;
		}
		break;
	}
	return pCh;
}

const char *GetNextWord(const char *pCh)
{
	pCh=SkipWhiteSpaceAndComments(pCh);

	if (!((*pCh>='0' && *pCh<='9') || 
		  (*pCh>='a' && *pCh<='z') ||
		  (*pCh>='A' && *pCh<='Z') ||
		  *pCh=='.' ||
		  *pCh=='-' ||
		  *pCh=='_'))
	{
		WordBuffer[0]=*pCh++;
		WordBuffer[1]=0;
		return pCh;
	}

	int c=0;
	while (*pCh && 
			((*pCh>='0' && *pCh<='9') || 
			 (*pCh>='a' && *pCh<='z') ||
			 (*pCh>='A' && *pCh<='Z') ||
			 *pCh=='.' ||
			 *pCh=='-' ||
			 *pCh=='_'))
	{
		Assert(c<WORD_BUFFER_SIZE,"Word too long");
		WordBuffer[c++]=*pCh++;
	}
	WordBuffer[c]=0;
	return pCh;
}

const char *GetNameAndType(const char *pCh, uint32 *pNameChecksum, uint8 *pType)
{
	Assert(*pCh=='{',"Expected '{'");
	++pCh;

	pCh=GetNextWord(pCh);
	Assert(stricmp(WordBuffer,"ComponentName")==0,"Expected \'ComponentName\'");
	pCh=GetNextWord(pCh);
	Assert(stricmp(WordBuffer,"=")==0,"Expected \'=\'");
	pCh=GetNextWord(pCh);
	*pNameChecksum=GenerateChecksum(WordBuffer);

	*pType=ESCRIPTTOKEN_UNDEFINED;
	pCh=GetNextWord(pCh);
	Assert(stricmp(WordBuffer,"DataType")==0,"Expected \'DataType\'");
	pCh=GetNextWord(pCh);
	Assert(stricmp(WordBuffer,"=")==0,"Expected \'=\'");
	pCh=GetNextWord(pCh);
	if (stricmp(WordBuffer,"Checksum")==0)
	{
		*pType=ESCRIPTTOKEN_NAME;
	}
	else if (stricmp(WordBuffer,"Integer")==0)
	{
		*pType=ESCRIPTTOKEN_INTEGER;
	}
	else if (stricmp(WordBuffer,"Float")==0)
	{
		*pType=ESCRIPTTOKEN_FLOAT;
	}
	else if (stricmp(WordBuffer,"Vector")==0)
	{
		*pType=ESCRIPTTOKEN_VECTOR;
	}
	else if (stricmp(WordBuffer,"Pair")==0)
	{
		*pType=ESCRIPTTOKEN_PAIR;
	}
	else if (stricmp(WordBuffer,"String")==0)
	{
		*pType=ESCRIPTTOKEN_STRING;
	}
	else if (stricmp(WordBuffer,"LocalString")==0)
	{
		*pType=ESCRIPTTOKEN_LOCALSTRING;
	}
	else if (stricmp(WordBuffer,"Array")==0)
	{
		*pType=ESCRIPTTOKEN_STARTARRAY;
	}
	else
	{
		Assert(0,"Unrecognized data type");
	}
	return pCh;
}

#include <c:/skate4/code/gel/scripting/skiptoken.cpp>

const uint8 *SkipEndOfLines(const uint8 *pToken)
{
	while (*pToken==ESCRIPTTOKEN_ENDOFLINE || *pToken==ESCRIPTTOKEN_ENDOFLINENUMBER)
	{
		pToken=SkipToken(pToken);
	}
	return pToken;
}

// Searches through the QB looking for a component with name Checksum and of type Type.
const uint8 *FindComponent(uint32 Checksum, uint8 RequiredType)
{
	Assert(pQBFile!=NULL,"NULL pQBFile");

	bool FoundNameEquals=false;
	const uint8 *pToken=pQBFile;
	const uint8 *pFoundComponent=NULL;

	while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)
	{
		switch (*pToken)
		{
        case ESCRIPTTOKEN_NAME:
			if (*(uint32*)(pToken+1)==Checksum)
			{
				pToken=SkipToken(pToken);
				pToken=SkipEndOfLines(pToken);
				if (*pToken==ESCRIPTTOKEN_EQUALS)
				{
					if (FoundNameEquals)
					{
						ErrorTwoComponentsWithSameName(Checksum);
					}
					FoundNameEquals=true;

					pToken=SkipToken(pToken);
					pToken=SkipEndOfLines(pToken);
					switch (*pToken)
					{
					case ESCRIPTTOKEN_NAME:
						if (RequiredType==ESCRIPTTOKEN_NAME)
						{
							pFoundComponent=pToken;
						}
						else
						{
							ErrorWrongType(Checksum,RequiredType,*pToken);
						}
						break;

					case ESCRIPTTOKEN_INTEGER:
					case ESCRIPTTOKEN_HEXINTEGER:
						if (RequiredType==ESCRIPTTOKEN_FLOAT || RequiredType==ESCRIPTTOKEN_INTEGER)
						{
							pFoundComponent=pToken;
						}
						else
						{
							ErrorWrongType(Checksum,RequiredType,*pToken);
						}
			            break;

			        case ESCRIPTTOKEN_FLOAT:
					case ESCRIPTTOKEN_ENUM:
					case ESCRIPTTOKEN_VECTOR:
					case ESCRIPTTOKEN_PAIR:
					case ESCRIPTTOKEN_STRING:
					case ESCRIPTTOKEN_LOCALSTRING:
					case ESCRIPTTOKEN_STARTARRAY:
						if (*pToken==RequiredType)
						{
							pFoundComponent=pToken;
						}
						else
						{
							ErrorWrongType(Checksum,RequiredType,*pToken);
						}
			            break;

					default:
						Assert(0,"Unexpected token following =");
						break;
					} // switch (*pToken)
				} // if (*pToken==ESCRIPTTOKEN_EQUALS)
			} // if (*(uint32*)(pToken+1)==Checksum)
			break;

		case ESCRIPTTOKEN_KEYWORD_SCRIPT:
			// Level qb files now may contain scripts, so skip over these.
			pToken=SkipToken(pToken);
			while (*pToken!=ESCRIPTTOKEN_KEYWORD_ENDSCRIPT)
			{
				Assert(*pToken!=ESCRIPTTOKEN_ENDOFFILE,"Got an end-of-file before endscript ?");
				pToken=SkipToken(pToken);
			}
			// No need to do another SkipToken, cos there's one after this case statement.
			break;

		default:
			break;
		} // switch (*pToken)

		pToken=SkipToken(pToken);
	} // while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)

	return pFoundComponent;
}

float GetFloat()
{
	float f=0;
	bool Frac=false;
	float d=10;
	char *pCh=WordBuffer;
	bool Negative;
	if (*pCh=='-')
	{
		Negative=true;
		++pCh;
	}
	else
	{
		Negative=false;
	}
	while (*pCh)
	{
		if (Frac)
		{
			Assert(*pCh>='0' && *pCh<='9',"Bad char in fractional part of float");
			f+=(*pCh-'0')/d;
			Assert(d<100000000,"Too many digits in fractional part of float");
			d*=10;
		}
		else
		{
			if (*pCh=='.')
			{
				Frac=true;
			}
			else
			{
				Assert(*pCh>='0' && *pCh<='9',"Bad char in integer part of float");
				f=f*10+*pCh-'0';
			}
		}
		++pCh;
	}
	if (Negative)
	{
		f=-f;
	}
	return f;
}

long GetInteger()
{
	long i=0;
	char *pCh=WordBuffer;
	bool Negative;
	if (*pCh=='-')
	{
		Negative=true;
		++pCh;
	}
	else
	{
		Negative=false;
	}
	while (*pCh)
	{
		Assert(*pCh>='0' && *pCh<='9',"Bad char in integer");
		i=i*10+*pCh-'0';
		++pCh;
	}
	if (Negative)
	{
		i=-i;
	}
	return i;
}

const char *CheckFurtherConditions(const char *pCh, const uint8 *pComponent, bool *pMatch, 
								   bool ErrorIfBadValue, uint32 NameChecksum, uint8 ExpectedType)
{
	Assert(pComponent!=NULL,"NULL pComponent");
	*pMatch=true;

	while (true)
	{
		pCh=GetNextWord(pCh);
		Assert(WordBuffer[0],"End-of-file reached before '}'");

		if (stricmp(WordBuffer,"Range")==0)
		{
			pCh=GetNextWord(pCh);
			Assert(strcmp(WordBuffer,"(")==0,"Expected a '(' after word 'Range'");

			float a;
			pCh=GetNextWord(pCh);
			a=GetFloat();

			pCh=GetNextWord(pCh);
			Assert(strcmp(WordBuffer,",")==0,"Expected a ',' between range values.");

			float b;
			pCh=GetNextWord(pCh);
			b=GetFloat();

			pCh=GetNextWord(pCh);
			Assert(strcmp(WordBuffer,")")==0,"Expected a ')' after word range values.");

			switch (ExpectedType)
			{
			case ESCRIPTTOKEN_INTEGER:
				{
					int Val;
					Assert(*pComponent==ESCRIPTTOKEN_INTEGER ||
						   *pComponent==ESCRIPTTOKEN_HEXINTEGER ||
						   *pComponent==ESCRIPTTOKEN_FLOAT,"Bad *pComponent");
					if (*pComponent==ESCRIPTTOKEN_FLOAT)
					{
						Val=(int)*(float*)(pComponent+1);
					}
					else
					{
						Val=*(int*)(pComponent+1);
					}

					if (!(Val>=a && Val<=b))
					{
						*pMatch=false;
						if (ErrorIfBadValue)
						{
							char Buf[1000];
							sprintf(Buf,"Error in \"%s=%d\"\n\n%s must be in the range %d to %d",
									FindChecksumName(NameChecksum),Val,
									FindChecksumName(NameChecksum),(int)a,(int)b);
							Error(Buf);
						}
					}
					break;
				}
			case ESCRIPTTOKEN_FLOAT:
				{
					float Val;
					Assert(*pComponent==ESCRIPTTOKEN_INTEGER ||
						   *pComponent==ESCRIPTTOKEN_HEXINTEGER ||
						   *pComponent==ESCRIPTTOKEN_FLOAT,"Bad *pComponent");
					if (*pComponent==ESCRIPTTOKEN_FLOAT)
					{
						Val=*(float*)(pComponent+1);
					}
					else
					{
						Val=(float)*(int*)(pComponent+1);
					}

					if (!(Val>=a && Val<=b))
					{
						*pMatch=false;
						if (ErrorIfBadValue)
						{
							char Buf[1000];
							sprintf(Buf,"Error in \"%s=%f\"\n\n%s must be in the range %.2f to %.2f",
									FindChecksumName(NameChecksum),Val,
									FindChecksumName(NameChecksum),a,b);
							Error(Buf);
						}
					}
					break;
				}
			default:
				Assert(0,"'Range' condition currently only works for integers and floats.");
				break;
			}
		}
		else if (stricmp(WordBuffer,"Equals")==0)
		{
			pCh=GetNextWord(pCh);
			Assert(strcmp(WordBuffer,"(")==0,"Expected a '(' after word 'Equals'");

			switch (ExpectedType)
			{
			case ESCRIPTTOKEN_NAME:
				{
					Assert(*pComponent==ESCRIPTTOKEN_NAME,"Bad *pComponent");

					uint32 ChecksumVal=*(uint32*)(pComponent+1);

					const char *pListStart=pCh;
					*pMatch=false;
					while (true)
					{
						pCh=GetNextWord(pCh);
						if (strcmp(WordBuffer,")")==0)
						{
							break;
						}
						if (strcmp(WordBuffer,","))
						{
							if (GenerateChecksum(WordBuffer)==ChecksumVal)
							{
								*pMatch=true;
							}
						}
					}

					if (*pMatch==false && ErrorIfBadValue)
					{
						char Buf[1000];
						sprintf(Buf,"Error in \"%s=%s\"\n\n%s must be one of:\n",
								FindChecksumName(NameChecksum),FindChecksumName(ChecksumVal),
								FindChecksumName(NameChecksum));

						const char *pFoo=pListStart;
						while (true)
						{
							pFoo=GetNextWord(pFoo);
							if (strcmp(WordBuffer,")")==0)
							{
								break;
							}
							if (strcmp(WordBuffer,","))
							{
								strcat(Buf,WordBuffer);
								strcat(Buf,"\n");
							}
						}

						Error(Buf);
					}
					break;
				}
			case ESCRIPTTOKEN_STRING:
				{
					Assert(*pComponent==ESCRIPTTOKEN_STRING,"Bad *pComponent");

					const char *pString=(const char*)(pComponent+5); // Skip over the string length.

					const char *pListStart=pCh;
					*pMatch=false;
					while (true)
					{
						pCh=GetNextWord(pCh);
						if (strcmp(WordBuffer,")")==0)
						{
							break;
						}
						if (strcmp(WordBuffer,",") && strcmp(WordBuffer,"\""))
						{
							if (strcmp(WordBuffer,pString)==0)
							{
								*pMatch=true;
							}
						}
					}

					if (*pMatch==false && ErrorIfBadValue)
					{
						char Buf[1000];
						sprintf(Buf,"Error in \"%s=\"%s\"\"\n\n%s must be one of:\n",
								FindChecksumName(NameChecksum),pString,
								FindChecksumName(NameChecksum));

						const char *pFoo=pListStart;
						while (true)
						{
							pFoo=GetNextWord(pFoo);
							if (strcmp(WordBuffer,")")==0)
							{
								break;
							}
							if (strcmp(WordBuffer,",") && strcmp(WordBuffer,"\""))
							{
								strcat(Buf,"\"");
								strcat(Buf,WordBuffer);
								strcat(Buf,"\"");
								strcat(Buf,"\n");
							}
						}

						Error(Buf);
					}
					break;
				}
			case ESCRIPTTOKEN_LOCALSTRING:
				{
					Assert(*pComponent==ESCRIPTTOKEN_LOCALSTRING,"Bad *pComponent");

					const char *pString=(const char*)(pComponent+5); // Skip over the string length.

					const char *pListStart=pCh;
					*pMatch=false;
					while (true)
					{
						pCh=GetNextWord(pCh);
						if (strcmp(WordBuffer,")")==0)
						{
							break;
						}
						if (strcmp(WordBuffer,",") && strcmp(WordBuffer,"'"))
						{
							if (strcmp(WordBuffer,pString)==0)
							{
								*pMatch=true;
							}
						}
					}

					if (*pMatch==false && ErrorIfBadValue)
					{
						char Buf[1000];
						sprintf(Buf,"Error in \"%s='%s'\"\n\n%s must be one of:\n",
								FindChecksumName(NameChecksum),pString,
								FindChecksumName(NameChecksum));

						const char *pFoo=pListStart;
						while (true)
						{
							pFoo=GetNextWord(pFoo);
							if (strcmp(WordBuffer,")")==0)
							{
								break;
							}
							if (strcmp(WordBuffer,",") && strcmp(WordBuffer,"'"))
							{
								strcat(Buf,"'");
								strcat(Buf,WordBuffer);
								strcat(Buf,"'");
								strcat(Buf,"\n");
							}
						}

						Error(Buf);
					}
					break;
				}
			case ESCRIPTTOKEN_INTEGER:
				{
					Assert(*pComponent==ESCRIPTTOKEN_INTEGER,"Bad *pComponent");

					long IntegerVal=*(long*)(pComponent+1);

					const char *pListStart=pCh;
					*pMatch=false;
					while (true)
					{
						pCh=GetNextWord(pCh);
						if (strcmp(WordBuffer,")")==0)
						{
							break;
						}
						if (strcmp(WordBuffer,","))
						{
							if (GetInteger()==IntegerVal)
							{
								*pMatch=true;
							}
						}
					}

					if (*pMatch==false && ErrorIfBadValue)
					{
						char Buf[1000];
						sprintf(Buf,"Error in \"%s=%d\"\n\n%s must be one of:\n",
								FindChecksumName(NameChecksum),IntegerVal,
								FindChecksumName(NameChecksum));

						const char *pFoo=pListStart;
						while (true)
						{
							pFoo=GetNextWord(pFoo);
							if (strcmp(WordBuffer,")")==0)
							{
								break;
							}
							if (strcmp(WordBuffer,","))
							{
								strcat(Buf,WordBuffer);
								strcat(Buf,"\n");
							}
						}

						Error(Buf);
					}
					break;
				}
			default:
				Assert(0,"'Equals' condition currently only works for names.");
				break;
			}
		}
		else if (WordBuffer[0]=='}' && WordBuffer[1]==0)
		{
			break;
		}
	}
	return pCh;
}

const char *SkipPastEndBrace(const char *pCh)
{
	while (true)
	{
		Assert(*pCh,"End-of-file reached before '}'");
		if (*pCh=='}')
		{
			break;
		}
		++pCh;
	}
	++pCh;
	return pCh;
}

const char *SkipTest(const char *pCh)
{
	Assert(*pCh=='{',"Expected '{'");
	while (*pCh!='}')
	{
		Assert(*pCh,"Unexpected end of file");
		++pCh;
	}
	++pCh;
	return pCh;
}

// Does the test pointed to by pCh. pCh must point to the open curly brace. Returns a pointer to just
// after the closing curly brace.
// Sets *pMatch according to whether the test was successful or not.
// If ErrorIfNoComponent is set, it will bring up an error message box if the specified component name
// and type does not exist.
// If ErrorIfBadValue is set it will bring up an error message box if the component does exist, but
// fails the extra tests, such as ranges, allowed values, etc.
const char *DoSingleTest(const char *pCh, bool *pMatch, bool ErrorIfNoComponent, bool ErrorIfBadValue)
{
	Assert(*pCh=='{',"*pCh is not '{'");
	const char *pTestStart=pCh;

	*pMatch=true;

	uint32 Checksum;
	uint8 Type;
	pCh=GetNameAndType(pCh,&Checksum,&Type);
	const uint8 *pComponent=FindComponent(Checksum,Type);
	if (pComponent)
	{
		pCh=CheckFurtherConditions(pCh,pComponent,pMatch,ErrorIfBadValue,Checksum,Type);
	}
	else
	{
		*pMatch=false;
		pCh=SkipPastEndBrace(pCh);

		if (ErrorIfNoComponent)
		{
			ErrorRequiredComponent(pTestStart);
		}
	}
	return pCh;
}

const char *CheckFormat(const char *pCh, bool *pMatch)
{
	*pMatch=false;

	pCh=SkipWhiteSpaceAndComments(pCh);
	Assert(strnicmp(pCh,"<StartFormat>",strlen("<StartFormat>"))==0,"Expected <StartFormat>");
	pCh+=strlen("<StartFormat>");

	pCh=SkipWhiteSpaceAndComments(pCh);
	Assert(strnicmp(pCh,"<Conditions>",strlen("<Conditions>"))==0,"Expected <Conditions>");
	pCh+=strlen("<Conditions>");

	bool Match=true;
	while (true)
	{
		pCh=SkipWhiteSpaceAndComments(pCh);
		if (*pCh=='{')
		{
			if (Match)
			{
				pCh=DoSingleTest(pCh,&Match,false,false);
			}
			else
			{
				pCh=SkipTest(pCh);
			}
		}
		else
		{
			break;
		}
	}

	if (Match)
	{
		*pMatch=true;

		pCh=SkipWhiteSpaceAndComments(pCh);
		Assert(strnicmp(pCh,"<Compulsory>",strlen("<Compulsory>"))==0,"Expected <Compulsory>");
		pCh+=strlen("<Compulsory>");

		while (true)
		{
			pCh=SkipWhiteSpaceAndComments(pCh);
			if (*pCh=='{')
			{
				pCh=DoSingleTest(pCh,&Match,true,true);
			}
			else
			{
				break;
			}
		}

		Assert(strnicmp(pCh,"<Optional>",strlen("<Optional>"))==0,"Expected <Optional>");
		pCh+=strlen("<Optional>");

		while (true)
		{
			pCh=SkipWhiteSpaceAndComments(pCh);
			if (*pCh=='{')
			{
				pCh=DoSingleTest(pCh,&Match,false,true);
			}
			else
			{
				break;
			}
		}

		pCh=SkipWhiteSpaceAndComments(pCh);
		Assert(strnicmp(pCh,"<EndFormat>",strlen("<EndFormat>"))==0,"Expected <EndFormat>");
		pCh+=strlen("<EndFormat>");
		pCh=SkipWhiteSpaceAndComments(pCh);
	}
	else
	{
		pCh=SkipWhiteSpaceAndComments(pCh);
		Assert(strnicmp(pCh,"<Compulsory>",strlen("<Compulsory>"))==0,"Expected <Compulsory>");
		pCh+=strlen("<Compulsory>");

		while (true)
		{
			pCh=SkipWhiteSpaceAndComments(pCh);
			if (*pCh=='{')
			{
				pCh=SkipTest(pCh);
			}
			else
			{
				break;
			}
		}

		Assert(strnicmp(pCh,"<Optional>",strlen("<Optional>"))==0,"Expected <Optional>");
		pCh+=strlen("<Optional>");

		while (true)
		{
			pCh=SkipWhiteSpaceAndComments(pCh);
			if (*pCh=='{')
			{
				pCh=SkipTest(pCh);
			}
			else
			{
				break;
			}
		}

		Assert(strnicmp(pCh,"<EndFormat>",strlen("<EndFormat>"))==0,"Expected <EndFormat>");
		pCh+=strlen("<EndFormat>");

		pCh=SkipWhiteSpaceAndComments(pCh);
	}

	return pCh;
}

#define MAX_FORMAT_FILE_CHARS 100
#define FORMAT_FILE_SUB_PATH "data\\scripts\\formats.f"

int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
	char *pPath=getenv("SKATE4_PATH");
	Assert(pPath!=NULL,"Check.exe requires that the environment variable SKATE4_PATH is set.");

	char pFormatFilename[MAX_FORMAT_FILE_CHARS+1];
	Assert(strlen(pPath)+1+strlen(FORMAT_FILE_SUB_PATH)<=MAX_FORMAT_FILE_CHARS,"Filename too long.");
	strcpy(pFormatFilename,pPath);
	if (pFormatFilename[strlen(pFormatFilename)-1]!='\\') strcat(pFormatFilename,"\\");
	strcat(pFormatFilename,FORMAT_FILE_SUB_PATH);

	FILE *ifp=fopen(pFormatFilename,"rb");
	Assert(ifp!=NULL,"Could not open data\\scripts\\formats.f");
	fseek(ifp,0,SEEK_END);
	long FileSize=ftell(ifp);
	fseek(ifp,0,SEEK_SET);
	pFormats=(char*)malloc(FileSize+1); // +1 for a terminating 0
	Assert(pFormats!=NULL,"Could not allocate memory to hold data\\scripts\\formats.f");
	Assert(fread(pFormats,FileSize,1,ifp)==1,"fread failed");
	fclose(ifp);
	ifp=NULL;
	// Put a terminating 0 on the end so that the end of file can be detected easily
	// without having to compare some counter to FileSize.
	pFormats[FileSize]=0; 

	Assert(lpCmdLine!=NULL,"NULL lpCmdLine !!");
	Assert(lpCmdLine[0],"Empty string sent to check.exe");
	// Open the .qb file for checking.
	ifp=fopen(lpCmdLine,"rb");
	if (ifp==NULL)
	{
		MessageBox(NULL,"Check.exe could not open the above qb file.",lpCmdLine,MB_OK);
		exit(EXIT_FAILURE);
	}
	fseek(ifp,0,SEEK_END);
	FileSize=ftell(ifp);
	fseek(ifp,0,SEEK_SET);
	pQBFile=(uint8*)malloc(FileSize);
	if (pQBFile==NULL)
	{
		MessageBox(NULL,"Check.exe could not allocate memory to hold the qb file.",lpCmdLine,MB_OK);
		exit(EXIT_FAILURE);
	}
	Assert(fread(pQBFile,FileSize,1,ifp)==1,"fread failed");
	fclose(ifp);
	ifp=NULL;

	// Record the checksum names listed in the .qb for future reference if needed, eg
	// when displaying any unrequired component names.
	const uint8 *pToken=pQBFile;
	while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)
	{
		if (*pToken==ESCRIPTTOKEN_CHECKSUM_NAME)
		{
			AddChecksumName(*(uint32*)(pToken+1),(char*)(pToken+5));
		}
		pToken=SkipToken(pToken);
	}

	const char *pCh=pFormats;
	bool MatchFound=false;
	while (*pCh)
	{
		bool Match;
		pCh=CheckFormat(pCh,&Match);
		if (Match) MatchFound=true;
		// This is so that an error that may be common to a load of formats won't
		// mean that the designers will have to click through them all.
		if (ErrorFound) break;
	}

	if (!MatchFound)
	{
		Error("No match found with any current format.");
	}

	// Clean up
	if (pFormats)
	{
		free (pFormats);
		pFormats=NULL;
	}
	if (pQBFile)
	{
		free (pQBFile);
		pQBFile=NULL;
	}

	if (ErrorFound)
	{
		return EXIT_FAILURE;
	}
	else
	{
		return EXIT_SUCCESS;
	}
}



