#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include <core/defines.h>

#include <SceneConv.h>
#include <PS2/PS2Conv.h>
#include <GC/GameCubeConv.h>
#include <Xbox/XBoxConv.h>
#include <PCViewer/PCViewerConv.h>

#include "Utility.h"
#include "VirtualFile.h"
#include "ngcdll\ngcdll.h"
#include "ps2dll\ps2dll.h"
#include "xboxdll\xboxdll.h"
#include "pcdll\pcdll.h"

#define vMAJOR_REV		2
#define vMINOR_REV		0

bool	gTextureOverride = false;
int		gTextureOverrideMapping[Utils::vNUM_PLATFORMS] = { 0, 1, 2 };

static bool s_force_overwrite = true;
static bool s_batch_mode = false;
static bool s_optimized = false;
static bool s_generate_weight_map = false;

int		run_utility( char* tool_name, char* filename, int platform )
{
	char scene_drive[_MAX_DRIVE];
	char scene_path[_MAX_PATH];
	char orig_scene_name[_MAX_FNAME];
	char scene_name[_MAX_FNAME];
	char scene_ext[_MAX_EXT];
	int  err;

	_splitpath( filename, scene_drive, scene_path, scene_name, scene_ext );
		
	char *proj_root_path = getenv( "PROJ_ROOT" );
	if( proj_root_path == NULL )
	{		
		fprintf(stderr, "You must first define your PROJ_ROOT environment variable\n");
		exit(1);
	}

	char dst_scene[_MAX_PATH];

	strcpy( orig_scene_name, scene_name );
	if( s_optimized )
	{
		strcat( scene_name, "_net" );
		sprintf( dst_scene, "%s%s%s%s.%s", scene_drive, scene_path, scene_name, scene_ext, Utils::GetPlatformExt(platform) );		
	}
	else
	{
		sprintf( dst_scene, "%s.%s", filename, Utils::GetPlatformExt(platform) );
	}

	if( !s_force_overwrite )
	{
		char exe_drive[_MAX_PATH];
		char exe_path[_MAX_PATH];
		char exe_name[_MAX_PATH];
		char exe_ext[_MAX_PATH];
		_splitpath( tool_name, exe_drive, exe_path, exe_name, exe_ext );

		char exe_fullpath[_MAX_PATH];
		sprintf( exe_fullpath, "%s\\bin\\win32\\%s.exe", proj_root_path, exe_name );

		if ( Utils::UpToDate( filename, dst_scene, exe_fullpath ) )
		{
			// doesn't need updating based on the timestamps
			return 0;
		}
	}

	printf( "%s: ", filename );

	char col_name[_MAX_PATH];
	sprintf( col_name, "%s%s%s.col.%s", scene_drive, scene_path, scene_name, Utils::GetPlatformExt(platform) );

	char dict_name[_MAX_PATH];
	sprintf( dict_name, "%s%s%s.tex", scene_drive, scene_path, orig_scene_name );
	
	char usg_name[_MAX_PATH];
	sprintf( usg_name, "%s%s%s.usg.%s", scene_drive, scene_path, scene_name, Utils::GetPlatformExt(platform) );
	
	char geom_name[_MAX_PATH];
	sprintf( geom_name, "%s%s%s.geom.%s", scene_drive, scene_path, scene_name, Utils::GetPlatformExt(platform) );
	
	char cas_name[_MAX_PATH];
	sprintf( cas_name, "%s%s%s.cas.%s", scene_drive, scene_path, scene_name, Utils::GetPlatformExt(platform) );

	char weightmap_name[_MAX_PATH];
	sprintf( weightmap_name, "%s%s%s.wgt.%s", scene_drive, scene_path, scene_name, Utils::GetPlatformExt(platform) );

	char lmp_name[_MAX_PATH];
	sprintf( lmp_name,  "%s%s%s.lmp", scene_drive, scene_path, scene_name );

	switch( platform )
	{
		case Utils::vPLATFORM_PS2:
		{
			err = PS2DLL_export( filename, dict_name, col_name, dst_scene, platform, usg_name, cas_name, geom_name, weightmap_name, s_optimized, s_generate_weight_map );

//			IPS2Converter* ps2_conv;
//			ps2_conv = GetPS2Converter();
//			ps2_conv->LoadScene( filename );
//			ps2_conv->LoadTextureDictionary( dict_name );
//			strcat( dict_name, "." );
//			strcat( dict_name, Utils::GetPlatformExt(platform) );
//			ps2_conv->CollapseCollsionVertices();
//			ps2_conv->GenerateCollisionBSP();
//			ps2_conv->SaveCollisionData( col_name );
//			ps2_conv->ConvertData();
//			ps2_conv->SaveScene( dst_scene );
//			ps2_conv->SaveTextureDictionary( dict_name, usg_name );
//			ps2_conv->SaveGeom( dict_name, dst_scene, geom_name );
//			ps2_conv->SaveCASFlags( cas_name );
//			ResetPS2Converter();	// reset for next use
			printf( "Done.\n" );
			break;
		}
		case Utils::vPLATFORM_NGC:
		{
			err = NGCDLL_export( filename, dict_name, col_name, dst_scene, platform, usg_name, cas_name, weightmap_name, s_optimized );

//			IGameCubeConverter* gc_conv;
//			gc_conv = GetGameCubeConverter();
//			gc_conv->LoadScene( filename );
//			gc_conv->LoadTextureDictionary( dict_name );
//			gc_conv->CollapseCollsionVertices();
//			gc_conv->GenerateCollisionBSP();
//			gc_conv->SaveCollisionData( col_name );
//			gc_conv->ConvertData();
//			gc_conv->SaveScene( dst_scene );
//			strcat( dict_name, "." );
//			strcat( dict_name, Utils::GetPlatformExt(platform) );
//			gc_conv->SaveTextureDictionary( dict_name, usg_name );
//			gc_conv->SaveCASFlags( cas_name );
//			ResetGameCubeConverter();	// reset for next use
			printf( "Done.  \n" );
			break;
		}
		case Utils::vPLATFORM_XBOX:
		{
			err = XBOXDLL_export( filename, dict_name, col_name, dst_scene, platform, usg_name, cas_name, weightmap_name, s_optimized );

//			IXBoxConverter* xb_conv;
//			xb_conv = GetXBoxConverter();
//			xb_conv->LoadScene( filename );
//			xb_conv->LoadTextureDictionary( dict_name );
//			xb_conv->CollapseCollsionVertices();
//			xb_conv->GenerateCollisionBSP();
//			xb_conv->SaveCollisionData( col_name );
//			xb_conv->ConvertData();
//			xb_conv->SaveScene( dst_scene );
//			strcat( dict_name, "." );
//			strcat( dict_name, Utils::GetPlatformExt(platform) );
//			xb_conv->SaveTextureDictionary( dict_name, usg_name );
//			xb_conv->SaveCASFlags( cas_name );
//			ResetXBoxConverter();	// reset for next use
			break;
		}
		case Utils::vPLATFORM_PCVIEWER:
		{
			err = PCDLL_export( filename, dict_name, lmp_name, col_name, dst_scene, usg_name, cas_name, false );

//			IPCViewerConverter* pcview_conv;
//			pcview_conv = GetPCViewerConverter();
//			pcview_conv->LoadScene( filename );
//			pcview_conv->LoadTextureDictionary( dict_name );
//			pcview_conv->LoadLightmaps( lmp_name );
//			pcview_conv->SaveCollisionData( col_name );
//			pcview_conv->ConvertData();
//			pcview_conv->SaveScene( dst_scene );
//			strcat( dict_name, ".pcv" );
//			pcview_conv->SaveTextureDictionary( dict_name, usg_name );
//			pcview_conv->SaveCASFlags( cas_name );
			break;
		}
	}

	return err;
}

int		main( int argc, char* argv[] )
{
	char* filename = NULL;
	char* batchname = NULL;
	int platform = Utils::vPLATFORM_NONE;

	if( argc > 2 )
	{
		int i;

		for( i = 1; i < argc; i++ )
		{
			if( argv[i][0] == '-' )
			{
				switch( argv[i][1] )
				{
					case 'P':
					case 'p':
						switch( argv[i][2] )
						{
							case 'P':
							case 'p':
								platform = Utils::vPLATFORM_PS2;
								break;
							case 'G':
							case 'g':
								platform = Utils::vPLATFORM_NGC;
								break;
							case 'X':
							case 'x':
								platform = Utils::vPLATFORM_XBOX;
								break;
							case 'C':
							case 'c':
								platform = Utils::vPLATFORM_PCVIEWER;
								break;
						}
						break;
					case 'B':
					case 'b':
						s_batch_mode = true;
						batchname = &argv[i][2];
						break;
					/*case 'O':
					case 'o':
						s_optimized = true;
						break;*/

					case 'F':
					case 'f':
						filename = &argv[i][2];
						break;
					case 'C':	// check timestamp
					case 'c':
						// by default, we no longer do the
						// timestamp comparison.  the utilities
						// now assume that if it is called,
						// it is supposed to re-convert the files
						// unconditionally.  it's up to the
						// convertassets program to do the 
						// timestamp comparisons
						s_force_overwrite = false;
						break;
					case 'T':
					case 't':
					{
						int src, dst;

						gTextureOverride = true;
						src = 0;
						dst = 0;
						switch( argv[i][2] )
						{
							case 'P':
							case 'p':
								src = Utils::vPLATFORM_PS2;
								break;
							case 'G':
							case 'g':
								src = Utils::vPLATFORM_NGC;
								break;
							case 'X':
							case 'x':
								src = Utils::vPLATFORM_XBOX;
								break;
						}							
						switch( argv[i][3] )
						{
							case 'P':
							case 'p':
								dst = Utils::vPLATFORM_PS2;
								break;
							case 'G':
							case 'g':
								dst = Utils::vPLATFORM_NGC;
								break;
							case 'X':
							case 'x':
								dst = Utils::vPLATFORM_XBOX;
								break;
						}	
						gTextureOverrideMapping[src] = dst;
						break;
					}
				}
			}
		}	
	}

	if( ( platform == Utils::vPLATFORM_NONE ) ||
		( filename == NULL && !s_batch_mode ))
	{	
#ifdef _DEBUG
		printf( "\nSceneConv v%d.%d (Debug) Built %s %s\n", vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#else
		printf( "\nSceneConv v%d.%d (Release) Built %s %s\n", vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#endif
		printf( "Neversoft Entertainment\n" );
		printf( "\nUsage: SceneConv -f<filename> -p[p | g | x]\n" );
		printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
		printf( "Ex: SceneConv -f%PROJ_ROOT%/data/levels/test/test.scn -pp\n" );
		return 0;
	}

	if ( Utils::TestArg( argc, argv, 'w' ) )
	{
		// used for generating the weight map, for cutscene heads
		s_generate_weight_map = true;
	}
	else
	{
		s_generate_weight_map = false;
	}

	if ( s_batch_mode )
	{
		IoUtils::CVirtualInputFile theBatchFile;
		if ( !theBatchFile.Load( batchname ) )
		{
			printf( "Error!  Could not open %s\n", batchname );
			exit(1);
		}

		printf( "Running batch mode on %s\n", batchname );

		// If an error occurs during a batch process the final return
		// code will be 0 if no errors occurred or the error value of
		// the last error to occur
		int err, lastErr = 0;

		char lineBuf[_MAX_PATH];

		while ( theBatchFile.ReadString( lineBuf, _MAX_PATH ) )
		{
			err = run_utility( argv[0], lineBuf, platform );
			if (err < 0)
			{
				lastErr = err;
				break;		// we want to exit out as soon as there is an error
			}

			// If this is a scene file, run it through once more with net optimization enabled
			char ext[_MAX_PATH];
			_splitpath( lineBuf, NULL, NULL, NULL, ext );
			if( stricmp( ext, ".scn" ) == 0 )
			{
				s_optimized = true;
				err = run_utility( argv[0], lineBuf, platform );
				if (err < 0)
				{
					lastErr = err;
					break;		// we want to exit out as soon as there is an error
				}
			}

			s_optimized = false;
		}

		return lastErr;
	}
	else
	{
		int err;
		char ext[_MAX_EXT];

		err = run_utility( argv[0], filename, platform );
		if( err >= 0 )
		{		
			// If this is a scene file, run it through once more with net optimization enabled
			_splitpath( filename, NULL, NULL, NULL, ext );
			if( stricmp( ext, ".scn" ) == 0 )
			{
				s_optimized = true;
				return run_utility( argv[0], filename, platform );			
			}			
		}

		return err;		
	}
}
