#include <windows.h>
#include <assert.h>
#include <SceneConv.h>
#include "strip.h"
#include "striper.h"
#include "xbstrip.h"

class NxStripper: public INxStripper
{
public:	
	// From INxStripper
	void	Reset( void );
	void	AddFace( NxFace* face );
	bool	Stripify( bool use_adc = true );
	bool	StripifyForXbox( void );
	bool	StripifyForNgc( int flags );
	NxMeshList*	GetMeshList( void );
	void	SetObject( NxObject* object );	
	void	SetPlatform( int platform );

	NxMeshList		m_MeshList;
	NxObject*		m_Object;
	int				m_Platform;
};

static	NxStripper	s_stripper;


void AddLODFace( NxObject *p_object, NxMeshList *p_meshlist, int lod_level, int i0, int i1, int i2 )
{
	// We need to select a mesh to which the face given by i0, i1, i2 will be added.
	// The simplest case is when this face matches a face in the original mesh data (more likely for
	// the higher resolution LOD levels).
	bool	matched = false;
	int		mesh;
	int		old_indices[3];
	for( mesh = 0; mesh < p_meshlist->m_NumMeshes[0]; ++mesh )
	{
		NxMesh *p_mesh = p_meshlist->m_Meshes[0][mesh];
		for( int tri = 0; tri < p_mesh->m_NumFaces; ++tri )
		{
			old_indices[0]		= p_mesh->m_Topology[tri][0];
			old_indices[1]		= p_mesh->m_Topology[tri][1];
			old_indices[2]		= p_mesh->m_Topology[tri][2];

			if(( i0 != old_indices[0] ) && ( i0 != old_indices[1] ) && ( i0 != old_indices[2] ))
			{
				continue;
			}

			if(( i1 != old_indices[0] ) && ( i1 != old_indices[1] ) && ( i1 != old_indices[2] ))
			{
				continue;
			}
							
			if(( i2 != old_indices[0] ) && ( i2 != old_indices[1] ) && ( i2 != old_indices[2] ))
			{
				continue;
			}
							
			// This tri matches a tri in the original mesh list.
			// Just add it to the corresponding mesh list for this LOD level.
//			printf( "Tri: %d exactly matched original tri.\n", face );
			matched = true;
			break;
		}
		if( matched )
			break;
	}
					
	if( !matched )
	{
		// We need to figure which mesh this face belongs to. Find the first mesh which contains all three
		// of the face indices.
		for( mesh = 0; mesh < p_meshlist->m_NumMeshes[0]; ++mesh )
		{
			unsigned int	match_bitfield	= 0;
			NxMesh			*p_mesh			= p_meshlist->m_Meshes[0][mesh];

			for( int tri = 0; tri < p_mesh->m_NumFaces; ++tri )
			{
				if(( i0 == p_mesh->m_Topology[tri][0] ) ||
				   ( i0 == p_mesh->m_Topology[tri][1] ) ||
				   ( i0 == p_mesh->m_Topology[tri][2] ))
				{
					match_bitfield |= 1;
				}
				if(( i1 == p_mesh->m_Topology[tri][0] ) ||
				   ( i1 == p_mesh->m_Topology[tri][1] ) ||
				   ( i1 == p_mesh->m_Topology[tri][2] ))
				{
					match_bitfield |= 2;
				}
				if(( i2 == p_mesh->m_Topology[tri][0] ) ||
				   ( i2 == p_mesh->m_Topology[tri][1] ) ||
				   ( i2 == p_mesh->m_Topology[tri][2] ))
				{
					match_bitfield |= 4;
				}
				if( match_bitfield == 0x07 )
				{
					matched = true;
//					printf( "Tri: %d had indices that were all found in mesh: %d\n", face, mesh );
					break;
				}
			}
			if( matched )
				break;
		}
	}

	if( !matched )
	{
		// We failed to match this tri up at all...
//		printf( "Failed to match tri: %d\n", face );
		return;
	}

	// Add this matched tri to the mesh 'mesh' at LOD level 'lod'.
	NxMesh *p_mesh = p_meshlist->m_Meshes[0][mesh];

	// See if we have a mesh with a matching material checksum for the LOD level.
	bool found = false;
	for( int lod_mesh = 0; lod_mesh < p_meshlist->m_NumMeshes[lod_level]; ++lod_mesh )
	{
		if( p_meshlist->m_Meshes[lod_level][lod_mesh]->m_MatChecksum == p_mesh->m_MatChecksum )
		{
			// Just add this face to the matched mesh.
			NxMesh *p_existing_mesh						= p_meshlist->m_Meshes[lod_level][lod_mesh];
			int num_faces								= p_meshlist->m_Meshes[lod_level][lod_mesh]->m_NumFaces;
			p_existing_mesh->m_Topology[num_faces][0]	= i0;
			p_existing_mesh->m_Topology[num_faces][1]	= i1;
			p_existing_mesh->m_Topology[num_faces][2]	= i2;
			p_existing_mesh->m_NumFaces					= num_faces + 1;
			found										= true;
			break;
		}
	}
	if( !found )
	{
		// Need to create a new mesh, with the same material checksum as the matched mesh.
		NxMesh *p_new_mesh								= new NxMesh( p_meshlist->m_Meshes[0][mesh]->m_MatChecksum );
		p_new_mesh->m_Object							= p_object;
		p_new_mesh->m_Topology[0][0]					= i0;
		p_new_mesh->m_Topology[0][1]					= i1;
		p_new_mesh->m_Topology[0][2]					= i2;
		p_new_mesh->m_NumFaces							= 1;

		// Add this newly created mesh to the meshlist.
		p_meshlist->AppendMesh( p_new_mesh, lod_level );
	}
}



void SimplifyMultipassFace( NxObject *p_object, NxMeshList *p_meshlist, NxLODFace *p_face, int mesh_write_level )
{
	int			set;
	NxVertex	*p_verts[3];
	int			new_indices[4][3]	= {{ -1, -1, -1 }, { -1, -1, -1 }, { -1, -1, -1 }, { -1, -1, -1 }};

	p_verts[0]						= p_object->m_OriginalVerts + p_face->v[0];
	p_verts[1]						= p_object->m_OriginalVerts + p_face->v[1];
	p_verts[2]						= p_object->m_OriginalVerts + p_face->v[2];

	// We want to find up to four sets of three verts that match these original verts in position,
	// color and normal, and *one* set of uv's.
	for( set = 0; set < 4; ++set )
	{
		for( int c = 0; c < 3; ++c )
		{
			for( int v = 0; v < p_object->m_NumVerts; ++v )
			{
				// Check position.
				if(( p_object->m_Verts[v].m_Pos[0] != p_verts[c]->m_Pos[0] ) ||
				   ( p_object->m_Verts[v].m_Pos[1] != p_verts[c]->m_Pos[1] ) ||
				   ( p_object->m_Verts[v].m_Pos[2] != p_verts[c]->m_Pos[2] ))
				{
					continue;
				}

				// Check normal.
				if(( p_object->m_Verts[v].m_Normal[0] != p_verts[c]->m_Normal[0] ) ||
				   ( p_object->m_Verts[v].m_Normal[1] != p_verts[c]->m_Normal[1] ) ||
				   ( p_object->m_Verts[v].m_Normal[2] != p_verts[c]->m_Normal[2] ))
				{
					continue;
				}
				
				// Check color.
				if(( p_object->m_Verts[v].m_Color[0] != p_verts[c]->m_Color[0] ) ||
				   ( p_object->m_Verts[v].m_Color[1] != p_verts[c]->m_Color[1] ) ||
				   ( p_object->m_Verts[v].m_Color[2] != p_verts[c]->m_Color[2] ))
				{
					continue;
				}

				// Check to see if uv(set) in the current verts matches uv(0) in the original verts.
				if(( p_object->m_Verts[v].m_TexCoord[set][0] != p_verts[c]->m_TexCoord[0][0] ) ||
				   ( p_object->m_Verts[v].m_TexCoord[set][1] != p_verts[c]->m_TexCoord[0][1] ))
				{
					continue;
				}

				// We have a match. Check to see that we haven't already written to this vertex.
				if( new_indices[set][c] != -1 )
				{
					printf( "WARNING: Writing twice to set (%d %d)\n", set, c );
				}
				new_indices[set][c]	= v;
			}
		}
	}

	// Now go through and try to identify a mesh in which any valid set of three verts reside.
	for( set = 0; set < 4; ++set )
	{
		if(( new_indices[set][0] != -1 ) && ( new_indices[set][1] != -1 ) && ( new_indices[set][2] != -1 ))
		{
			AddLODFace( p_object, p_meshlist, mesh_write_level, new_indices[set][0], new_indices[set][1], new_indices[set][2] );
		}
	}
}



// This is the function that takes index-set based LOD data and build separate levels of meshes.
void Simplify( NxObject *p_object, NxMeshList *p_meshlist )
{
	// Check that this object has LOD information.
	if(( p_object->m_Flags & NxObject::mHASINTLODINFO ) == 0 )
	{
		return;
	}
	assert( p_object->m_LODinfo != NULL );
	
	// Check for duplicate verts (which at this stage shouldn't exist).
//	for( int ii = 0; ii < p_object->m_NumVerts; ++ii )
//	{
//		for( int jj = 0; jj < p_object->m_NumVerts; ++jj )
//		{
//			if( jj == ii )
//				continue;
//
//			if( p_object->m_Verts[ii] == p_object->m_Verts[jj] )
//			{
//				printf( "%d and %d match\n", ii, jj );
//				assert( 0 );
//			}
//		}
//	}
	
	// This is the mesh level at which we will write the current LOD level data. Mesh level 0 is the original data,
	// which is why this value starts at 1.
	int mesh_write_level = 1;
		
	// Work through each LOD mesh (the highest level LOD mesh will contain the same initial data as the regular mesh).
	for( int lod = 1; lod < p_object->m_LODLevels; ++lod, ++mesh_write_level )
	{
		printf( "Processing LOD level %d.\n", lod );

		NxLODInfo *p_info = p_object->m_LODinfo + lod;

		for( int face = 0; face < p_info->numFaces; ++face )
		{
			// Obtain pointers to the three (original) vertices forming this face.
			NxVertex	*p_verts[3];
			int			new_indices[3]	= { -1, -1, -1 };
			NxLODFace	*p_face			= p_info->faces + face;
			p_verts[0]					= p_object->m_OriginalVerts + p_face->v[0];
			p_verts[1]					= p_object->m_OriginalVerts + p_face->v[1];
			p_verts[2]					= p_object->m_OriginalVerts + p_face->v[2];

			// Now scan through (current) vertex list to see where these vertices appear.
			for( int c = 0; c < 3; ++c )
			{
				for( int v = 0; v < p_object->m_NumVerts; ++v )
				{
					if( p_object->m_Verts[v] == *p_verts[c] )
					{
						new_indices[c] = v;
						break;
					}
				}

				if( new_indices[c] == -1 )
				{
					// We failed to find this vertex in the list. Exit the loop.
					break;
				}
			}

			if(( new_indices[0] == -1 ) || ( new_indices[1] == -1 ) || ( new_indices[2] == -1 ))
			{
				// We failed to find this vertex in the list.
				printf( "Failed to match vertices for LOD face: %d for LOD level: %d.\n", face, lod );

				// It is possible that the geometry was split up to deal with multipass issues, most
				// likely for Ps2, but also for specifically flagged Xbox and Gamecube meshes. Try a different
				// approach here just in case...
				SimplifyMultipassFace( p_object, p_meshlist, p_face, mesh_write_level );

				// Skip to next face.				
				continue;
			}
				
			// We matched up the vertices in this face. Add it to the meshes for this LOD level, if possible.
			AddLODFace( p_object, p_meshlist, mesh_write_level, new_indices[0], new_indices[1], new_indices[2] );
		}
	}
}



INxStripper*	GetNxStripper( void )
{
	return &s_stripper;
}

void NxStripper::Reset( void )
{
	m_MeshList.Reset();
}


void NxStripper::SetPlatform( int platform )
{
	m_Platform = platform;
}


void NxStripper::SetObject( NxObject* object )
{
	m_Object = object;
}

void NxStripper::AddFace( NxFace* face )
{
	int i;
	bool existed;
	FlagType face_shadow_flags, mesh_shadow_flags;

	existed = false;	
	face_shadow_flags = face->m_FaceFlags & ( NxFace::mFD_NO_SKATER_SHADOW | NxFace::mFD_NO_SKATER_SHADOW_WALL );
	mesh_shadow_flags = 0;
	// On Xbox, no_shadow and no_shadow wall are distinct
	if( m_Platform == Utils::vPLATFORM_XBOX )
	{		
		if( face_shadow_flags & NxFace::mFD_NO_SKATER_SHADOW )
		{
			mesh_shadow_flags |= NxObject::mNO_SHADOW;
		}
		if( face_shadow_flags & NxFace::mFD_NO_SKATER_SHADOW_WALL )
		{
			mesh_shadow_flags |= NxObject::mNO_SHADOW_WALL;
		}
	}
	else
	{
		// On NGC and PS2, either flag indicates NO SHADOW for now
		if( face_shadow_flags )
		{
			mesh_shadow_flags = NxObject::mNO_SHADOW | NxObject::mNO_SHADOW_WALL;
		}
	}
	for( i = 0; i < m_MeshList.m_NumMeshes[0]; i++ )
	{
		if(	( m_MeshList.m_Meshes[0][i]->m_MatChecksum == face->m_MatChecksum ) &&
			( m_MeshList.m_Meshes[0][i]->m_ShadowFlags == mesh_shadow_flags ))
		{			
			NxMesh* mesh;

			mesh = m_MeshList.m_Meshes[0][i];
			
			assert( mesh->m_NumFaces < NxMesh::vMAX_FACES ); 

			mesh->m_Topology[mesh->m_NumFaces][0] = face->m_Vertex[0];
			mesh->m_Topology[mesh->m_NumFaces][1] = face->m_Vertex[1];
			mesh->m_Topology[mesh->m_NumFaces][2] = face->m_Vertex[2];

			// for cas flags (need to copy over the collision
			// tris' face flags into the renderable tris' face flags)
			mesh->m_CASFaceFlags[mesh->m_NumFaces] = (unsigned long)face->m_CASFaceFlags;
	
			mesh->m_NumFaces++;
			existed = true;
			break;
		}
	}

	// If the texture index is new, create a new mesh to represent these tri's
	if( !existed )
	{
		NxMesh* new_mesh;
		
		new_mesh = new NxMesh( face->m_MatChecksum );
		new_mesh->m_Object =  m_Object;
		new_mesh->m_ShadowFlags = mesh_shadow_flags;
		m_MeshList.AppendMesh( new_mesh );
		new_mesh->m_Topology[0][0] = face->m_Vertex[0];
		new_mesh->m_Topology[0][1] = face->m_Vertex[1];
		new_mesh->m_Topology[0][2] = face->m_Vertex[2];

		// for cas flags (need to copy over the collision
		// tris' face flags into the renderable tris' face flags)
		new_mesh->m_CASFaceFlags[0] = (unsigned long)face->m_CASFaceFlags;

		new_mesh->m_NumFaces++;	
	}
}



bool NxStripper::StripifyForNgc( int flags )
{
	// Create the topology for each mesh.
	int i, j;
	for( i = 0; i < m_Object->m_NumFaces; i++ )
	{
		AddFace( &m_Object->m_Faces[i] );
	}

	if( m_MeshList.m_NumMeshes[0] > 0 )
	{
		Simplify( m_Object, &m_MeshList );
	}
	
	// At this point we should have one mesh per material, so we can pass each mesh's topology to the stripper.
	for( i = 0; i < m_MeshList.m_NumMeshes[0]; i++ )
	{
		NxMesh *mesh = m_MeshList.m_Meshes[0][i];

		// Create a list of triangles usable by the Xbox stripper code.
		unsigned short *p_tri_indices_in = new unsigned short[mesh->m_NumFaces * 3];
		for( j = 0; j < mesh->m_NumFaces; ++j )
		{
			p_tri_indices_in[j * 3 + 0] = mesh->m_Topology[j][0];
			p_tri_indices_in[j * 3 + 1] = mesh->m_Topology[j][1];
			p_tri_indices_in[j * 3 + 2] = mesh->m_Topology[j][2];
		}

		unsigned long	num_output_indices;
		unsigned short	*p_output_indices;
		
//		if( flags & NxObject::mSKINNED )
//		{
//			NgcStripify(	mesh->m_NumFaces,		// Number of triangles.
//						p_tri_indices_in,		// Ptr to triangle indices.
//						&num_output_indices,	// Number of output indices.
//						&p_output_indices,		// Output indices.
//						OPTIMIZE_FOR_INDICES | OUTPUT_TRISTRIP | SINGLE_MESH );
//
//			// Copy the stripped verts back into the mesh.
//			mesh->m_NumStripVerts = num_output_indices + 2;
//			mesh->m_VertStrip		= new NxStripVert[num_output_indices+2];
//			mesh->m_VertStrip[0].m_Index		= num_output_indices;
//			mesh->m_VertStrip[0].m_StripVert	= false;
//			mesh->m_VertStrip[(num_output_indices+2)-1].m_Index		= 0;
//			mesh->m_VertStrip[(num_output_indices+2)-1].m_StripVert	= false;
//			for( j = 0; j < (int) num_output_indices; ++j )
//			{
//				mesh->m_VertStrip[j+1].m_Index		= p_output_indices[j];
//				mesh->m_VertStrip[j+1].m_StripVert	= false;
//			}
//		}
//		else
		{
			NgcStripify(	mesh->m_NumFaces,		// Number of triangles.
						p_tri_indices_in,		// Ptr to triangle indices.
						&num_output_indices,	// Number of output indices.
						&p_output_indices,		// Output indices.
						OPTIMIZE_FOR_INDICES | OUTPUT_TRISTRIP );

			// Copy the stripped verts back into the mesh.
			mesh->m_NumStripVerts = num_output_indices;
			mesh->m_VertStrip		= new NxStripVert[num_output_indices];
			for( j = 0; j < (int) num_output_indices; ++j )
			{
				mesh->m_VertStrip[j].m_Index		= p_output_indices[j];
				mesh->m_VertStrip[j].m_StripVert	= false;
			}
		}

		
		delete [] p_tri_indices_in;
		delete [] p_output_indices;
	}

	return true;
}

bool NxStripper::StripifyForXbox( void )
{
	// Create the topology for each mesh.
	int i, j;
	for( i = 0; i < m_Object->m_NumFaces; i++ )
	{
		AddFace( &m_Object->m_Faces[i] );
	}

	if( m_MeshList.m_NumMeshes[0] > 0 )
	{
		Simplify( m_Object, &m_MeshList );
	}
	
	// At this point we should have one mesh per material, so we can pass each mesh's topology to the stripper.
	for( int lod = 0; lod < 8; ++lod )
	{
		for( i = 0; i < m_MeshList.m_NumMeshes[lod]; i++ )
		{
			NxMesh *mesh = m_MeshList.m_Meshes[lod][i];

			// Create a list of triangles usable by the Xbox stripper code.
			unsigned short *p_tri_indices_in = new unsigned short[mesh->m_NumFaces * 3];
			for( j = 0; j < mesh->m_NumFaces; ++j )
			{
				p_tri_indices_in[j * 3 + 0] = mesh->m_Topology[j][0];
				p_tri_indices_in[j * 3 + 1] = mesh->m_Topology[j][1];
				p_tri_indices_in[j * 3 + 2] = mesh->m_Topology[j][2];
			}

			unsigned long	num_output_indices;
			unsigned short	*p_output_indices;
		
			XBStripify(	mesh->m_NumFaces,		// Number of triangles.
						p_tri_indices_in,		// Ptr to triangle indices.
						&num_output_indices,	// Number of output indices.
						&p_output_indices,		// Output indices.
						OPTIMIZE_FOR_CACHE | OUTPUT_TRISTRIP );


			// Copy the stripped verts back into the mesh.
			mesh->m_NumStripVerts = num_output_indices;
			mesh->m_VertStrip		= new NxStripVert[num_output_indices];
			for( j = 0; j < (int) num_output_indices; ++j )
			{
				mesh->m_VertStrip[j].m_Index		= p_output_indices[j];
				mesh->m_VertStrip[j].m_StripVert	= false;
			}
		
			delete [] p_tri_indices_in;
			delete [] p_output_indices;
		}
	}

	return true;
}



bool NxStripper::Stripify( bool use_adc )
{	
	int i;

	//m_Object->OptimizeVertList();

	// Create the topology for each mesh
	for( i = 0; i < m_Object->m_NumFaces; i++ )
	{
		AddFace( &m_Object->m_Faces[i] );
	}

	// At this point we should have one mesh per material (eventually for each texture)
	// so we can pass each mesh's topology to the stripper
	for( i = 0; i < m_MeshList.m_NumMeshes[0]; i++ )
	{
		udword j, k;
		int num_verts, count;
		uword* Refs;
		NxMesh* mesh;

		mesh = m_MeshList.m_Meshes[0][i];

		STRIPERCREATE sc;
		sc.DFaces			= (unsigned int *) mesh->m_Topology;//Topology;
		sc.NbFaces			= mesh->m_NumFaces;
		sc.AskForWords		= true;
		sc.ConnectAllStrips	= false;
		sc.OneSided			= false;
		sc.SGIAlgorithm		= true;

		Striper Strip;
		if( Strip.Init(sc) == false )
		{
			return false;
		}

		STRIPERRESULT sr;
		if( Strip.Compute(sr) == false )
		{
			return false;
		}

#ifdef DEBUG_STRIPS
		DebugPrint("Number of strips: %d\n", sr.NbStrips);
		Refs = (uword*)sr.StripRuns;
		for( j= 0 ; j < sr.NbStrips; j++ )
		{
			DebugPrint( "Strip %d:   ", j);
			udword NbRefs = sr.StripLengths[j];
			for( k = 0; k < NbRefs; k++ )
			{
				DebugPrint( "%d ", *Refs++);
			}
			DebugPrint( "\n");
		}
#endif
		num_verts = 0;
		// Count the number of verts needed to strip this mesh
		for( j = 0; j < sr.NbStrips; j++ )
		{
			num_verts += sr.StripLengths[j];
		}
		
		if( num_verts > 0 )
		{
			count	= 0;
			Refs	= (uword*) sr.StripRuns;

			if( use_adc )
			{
				mesh->m_NumStripVerts = num_verts;
				mesh->m_VertStrip = new NxStripVert[num_verts];

				// Now populate the strip vertex array with indices to those verts and their
				// ADC bits
				for( j = 0; j < sr.NbStrips; j++ )
				{
					udword NbRefs = sr.StripLengths[j];
					for( k = 0; k < NbRefs; k++ )
					{
						// The first two verts of a strip are the start of a new strip, rather
						// than a continuation of the last. So their ADC bits are off
						if( k < 2 )
						{
							mesh->m_VertStrip[count].m_StripVert = false;
						}
						else
						{
							mesh->m_VertStrip[count].m_StripVert = true;
						}
						mesh->m_VertStrip[count].m_Index = *Refs++;
						count++;
					}				
				}
			}
			else
			{
				// Due to the addition of degenerate verts, will possibly need a bigger array.
				mesh->m_VertStrip = new NxStripVert[num_verts * 4];

				// Now populate the strip vertex array with indices to those verts.
				// Use degenerate tris rather than adc bits (which are PS2 specific) to link strips.
				for( j = 0; j < sr.NbStrips; j++ )
				{
					udword NbRefs = sr.StripLengths[j];
					for( k = 0; k < NbRefs; k++ )
					{
						// The first two verts of a strip are the start of a new strip, rather
						// than a continuation of the last. So their ADC bits are off
						if( k < 2 )
						{
							mesh->m_VertStrip[count].m_StripVert = false;
						}
						else
						{
							mesh->m_VertStrip[count].m_StripVert = true;
						}

						if(( j > 0 ) && ( k == 0 ))
						{
							// Need to add verts here, since the ADC flags are ignored.
							// To join disparate tris, 012 and 345, need index array 0,1,2,[2,3],3,4,5
							mesh->m_VertStrip[count].m_Index = mesh->m_VertStrip[count - 1].m_Index;
							++count;
						}
						
						mesh->m_VertStrip[count].m_Index = *Refs++;
						count++;

						if(( j > 0 ) && ( k == 0 ))
						{
							// See comment above.
							mesh->m_VertStrip[count].m_Index = mesh->m_VertStrip[count - 1].m_Index;
							++count;
						}
					}				
				}

				// Now we know exactly how many verts were added.
				mesh->m_NumStripVerts = count;
			}
		}
	}

	return true;
}

NxMeshList*	NxStripper::GetMeshList( void )
{
	return &m_MeshList;
}
