//-------------------------------------------------------------------
//-------------------------------------------------------------------------------
//
//	File:
//		NxPNG.h
//
//	Description:
//		Neversoft PNG Exporter. Will export PNGs as 1, 2, 4 and 8-bit images and optionally 4 or 8-bit with alpha.
//
//	Use:
//		Format modules are called from the Save, Save as,
//		and Save a copy dialogs.
//
//-------------------------------------------------------------------------------

//-------------------------------------------------------------------------------
//	Includes
//-------------------------------------------------------------------------------

#include "NxPNG.h"
#include "png.h"
#include "ImageData.h"
#include "Quantizer.h"
#include "Quant.h"
#include <admdialog.h>
#include <admimage.h>
#include <admdrawer.h>
#include "NxPNG-sym.h"

//-------------------------------------------------------------------------------
//	Prototypes
//-------------------------------------------------------------------------------

static void DoReadPrepare (GPtr globals);
static void DoReadStart (GPtr globals);
static void DoReadContinue (GPtr globals);
static void DoReadFinish (GPtr globals);
static void DoOptionsPrepare (GPtr globals);
static void DoOptionsStart (GPtr globals);
static void DoOptionsContinue (GPtr globals);
static void DoOptionsFinish (GPtr globals);
static void DoEstimatePrepare (GPtr globals);
static void DoEstimateStart (GPtr globals);
static void DoEstimateContinue (GPtr globals);
static void DoEstimateFinish (GPtr globals);
static void DoWritePrepare (GPtr globals);
static void DoWriteStart (GPtr globals);
static void DoWriteContinue (GPtr globals);
static void DoWriteFinish (GPtr globals);
static void DoFilterFile (GPtr globals);

static void AddComment (GPtr globals);
static void GetHistory (GPtr globals, int16 index, Str255 s);

static Boolean CheckIdentifier (char identifier []);
static void SetIdentifier (char identifier []);
static Boolean CheckForServices (GPtr globals);
static int32 RowBytes (GPtr globals);
static Boolean AllocatePixelBuffer (GPtr globals);
static void DisposePixelBuffer (GPtr globals);

static void ReadSome (GPtr globals, int32 count, void *buffer);
static void WriteSome (GPtr globals, int32 count, void *buffer);
static void ReadRow (GPtr globals);
static void WriteRow (GPtr globals);
static void DisposeImageResources (GPtr globals);

static void DoReadICCProfile(GPtr globals);
static void DoWriteICCProfile(GPtr globals);

static void png_error_cb( png_structp png_struct, png_const_charp error );
static void png_warning_cb( png_structp png_struct, png_const_charp warning );
static void png_write( png_structp png_struct, png_bytep data, png_size_t size );
static void png_read( png_structp png_struct, png_bytep data, png_size_t size );

extern bool read_png( png_structp* in_png_ptr, png_infop* in_info_ptr );


//-------------------------------------------------------------------------------
//	Globals -- Define global variables for plug-in scope.
//-------------------------------------------------------------------------------

SPBasicSuite * sSPBasic = NULL;
ADMDialogSuite* sADMDialog = NULL;	
ADMImageSuite* sADMImage = NULL;	
ADMItemSuite* sADMItem = NULL;
ADMDrawerSuite* sADMDrawer = NULL;
ADMDialogRef sDialogRef = NULL;

int sNumColors = 0;
bool sIndexed = false;
bool sInterlaced = false;
png_infop sPngPtr = NULL;
GPtr sGlobals;

//-------------------------------------------------------------------------------
//
//	PluginMain / main
//
//	All calls to the plug-in module come through this routine.
//	It must be placed first in the resource.  To achieve this,
//	most development systems require this be the first routine
//	in the source.
//
//	The entrypoint will be "pascal void" for Macintosh,
//	"void" for Windows.
//
//	Inputs:
//		const short selector						Host provides selector indicating
//													what command to do.
//
//		FormatRecord *formatParamBlock				Host provides pointer to parameter
//													block containing pertinent data
//													and callbacks from the host.
//													See PIFormat.h.
//
//	Outputs:
//		FormatRecord *formatParamBlock				Host provides pointer to parameter
//													block containing pertinent data
//													and callbacks from the host.
//													See PIFormat.h.
//
//		long *data									Use this to store a handle to our
//													global parameters structure, which
//													is maintained by the host between
//													calls to the plug-in.
//
//		short *result								Return error result or noErr.  Some
//													errors are handled by the host, some
//													are silent, and some you must handle.
//													See PIGeneral.h.
//
//-------------------------------------------------------------------------------

DLLExport MACPASCAL void PluginMain (const short selector,
						             FormatRecord *formatParamBlock,
						             long *data,
						             short *result)
{
	//---------------------------------------------------------------------------
	//	(1) Check for about box request.
	//
	// 	The about box is a special request; the parameter block is not filled
	// 	out, none of the callbacks or standard data is available.  Instead,
	// 	the parameter block points to an AboutRecord, which is used
	// 	on Windows.
	//---------------------------------------------------------------------------

	if (selector == formatSelectorAbout)
	{
		sSPBasic = ((AboutRecordPtr)formatParamBlock)->sSPBasic;
		DoAbout((AboutRecordPtr)formatParamBlock);
	}
	else
	{ // do the rest of the process as normal:

		sSPBasic = formatParamBlock->sSPBasic;

		Ptr globalPtr = NULL;		// Pointer for global structure
		GPtr globals = NULL; 		// actual globals

		//-----------------------------------------------------------------------
		//	(2) Allocate and initalize globals.
		//
		// 	AllocateGlobals requires the pointer to result, the pointer to the
		// 	parameter block, a pointer to the handle procs, the size of our local
		// 	"Globals" structure, a pointer to the long *data, a Function
		// 	Proc (FProcP) to the InitGlobals routine.  It automatically sets-up,
		// 	initializes the globals (if necessary), results result to 0, and
		// 	returns with a valid pointer to the locked globals handle or NULL.
		//-----------------------------------------------------------------------
		
		globalPtr = AllocateGlobals ((uint32)result,
									 (uint32)formatParamBlock,
									 formatParamBlock->handleProcs,
									 sizeof(Globals),
						 			 data,
						 			 InitGlobals);
		
		if (globalPtr == NULL)
		{ // Something bad happened if we couldn't allocate our pointer.
		  // Fortunately, everything's already been cleaned up,
		  // so all we have to do is report an error.
		  
		  *result = memFullErr;
		  return;
		}
		
		// Get our "globals" variable assigned as a Global Pointer struct with the
		// data we've returned:
		globals = (GPtr)globalPtr;
		sGlobals = globals;

		//-----------------------------------------------------------------------
		//	(3) Dispatch selector.
		//-----------------------------------------------------------------------

		switch (selector)
		{
			case formatSelectorReadPrepare:
				DoReadPrepare(globals);
				break;
			case formatSelectorReadStart:
				DoReadStart(globals);
				break;
			case formatSelectorReadContinue:
				DoReadContinue(globals);
				break;
			case formatSelectorReadFinish:
				DoReadFinish(globals);
				break;

			case formatSelectorOptionsPrepare:
				DoOptionsPrepare(globals);
				break;
			case formatSelectorOptionsStart:
				DoOptionsStart(globals);
				break;
			case formatSelectorOptionsContinue:
				DoOptionsContinue(globals);
				break;
			case formatSelectorOptionsFinish:
				DoOptionsFinish(globals);
				break;

			case formatSelectorEstimatePrepare:
				DoEstimatePrepare(globals);
				break;
			case formatSelectorEstimateStart:
				DoEstimateStart(globals);
				break;
			case formatSelectorEstimateContinue:
				DoEstimateContinue(globals);
				break;
			case formatSelectorEstimateFinish:
				DoEstimateFinish(globals);
				break;

			case formatSelectorWritePrepare:
				DoWritePrepare(globals);
				break;
			case formatSelectorWriteStart:
				DoWriteStart(globals);
				break;
			case formatSelectorWriteContinue:
				DoWriteContinue(globals);
				break;
			case formatSelectorWriteFinish:
				DoWriteFinish(globals);
				break;

			case formatSelectorFilterFile:
				DoFilterFile(globals);
				break;
		}
			
		//-----------------------------------------------------------------------
		//	(4) Unlock data, and exit resource.
		//
		//	Result is automatically returned in *result, which is
		//	pointed to by gResult.
		//-----------------------------------------------------------------------	
		
		// unlock handle pointing to parameter block and data so it can move
		// if memory gets shuffled:
		if ((Handle)*data != NULL)
			PIUnlockHandle((Handle)*data);
	
	} // about selector special		
	
} // end PluginMain

//-------------------------------------------------------------------------------
//
//	InitGlobals
//	
//	Initalize any global values here.  Called only once when global
//	space is reserved for the first time.
//
//	Inputs:
//		Ptr globalPtr		Standard pointer to a global structure.
//
//	Outputs:
//		Initializes any global values with their defaults.
//
//-------------------------------------------------------------------------------

void InitGlobals (Ptr globalPtr)
{	
	// create "globals" as a our struct global pointer so that any
	// macros work:
	GPtr globals = (GPtr)globalPtr;
	
	// Initialize global variables:
	gQueryForParameters = true;
	ValidateParameters (globals);
	
} // end InitGlobals

//-------------------------------------------------------------------------------
//
//	ValidateParameters
//
//	Initialize parameters to default values.
//
//	Inputs:
//		GPtr globals		Pointer to global structure.
//
//	Outputs:
//		gWhatShape			Default: iShapeTriangle.
//
//		gCreate				Default: iCreateSelection.
//
//		gIdleAmount			Default: 2.0 seconds.
//
//-------------------------------------------------------------------------------

void ValidateParameters (GPtr globals)
{
	gPixelBuffer = NULL;
	gPixelData = NULL;
	
	gRowBytes = 0;
	
	gFooRead = false;
	gBarWrite = true;

	PIResetString(gHistory);

} // end ValidateParameters


/*****************************************************************************/

static Boolean CheckIdentifier (char identifier [])
	{
	
	return identifier[0] == 'o' &&
		   identifier[1] == 'n' &&
		   identifier[2] == 'e' &&
		   identifier[3] == 'b' &&
		   identifier[4] == 'r' &&
		   identifier[5] == 'a' &&
		   identifier[6] == 'i' &&
		   identifier[7] == 'n';
	
	}

/*****************************************************************************/

static void SetIdentifier (char identifier [])
	{
	
	identifier[0] = 'o';
    identifier[1] = 'n';
    identifier[2] = 'e';
    identifier[3] = 'b';
    identifier[4] = 'r';
    identifier[5] = 'a';
    identifier[6] = 'i';
    identifier[7] = 'n';
	
	}

/*****************************************************************************/

static Boolean CheckForServices (GPtr globals)
	{
	
	return WarnBufferProcsAvailable () &&
		   WarnAdvanceStateAvailable ();
	// handle procs have upgraded for 6.0 but we want to run in 5.5
	// WarnHandleProcsAvailable ();
	
	}

/*****************************************************************************/

static int32 RowBytes (GPtr globals)
	{
	
	return (gStuff->imageSize.h * (( gStuff->depth * gStuff->planes ) >> 3 ));	
	
	}

static int32 ImageBytes(GPtr globals)
	{

	return (gStuff->imageSize.h * gStuff->imageSize.v * (int32) gStuff->depth + 7) >> 3;

	}

/*****************************************************************************/

static Boolean AllocatePixelBuffer (GPtr globals)
	{
	
	BufferID buffer;
	
	if (gResult != noErr) return FALSE;

	/* We will want a buffer that is one line wide. */
	
	gPixelBuffer = 0;
	
	gRowBytes = RowBytes (globals);

	if (!TSR (AllocateBuffer (gRowBytes, &buffer))) return FALSE;
	
	gPixelBuffer = buffer;
	
	gPixelData = LockBuffer (gPixelBuffer, FALSE);
	
	return TRUE;
	
	}

/*****************************************************************************/

static void DisposePixelBuffer (GPtr globals)
	{
	
	if (gPixelBuffer != 0)
		{
		
		FreeBuffer (gPixelBuffer);
		
		gPixelBuffer = 0;
		gPixelData = 0;
		
		}
	
	}

/*****************************************************************************/

static void DoReadPrepare (GPtr globals)
	{
	
	gStuff->maxData = 0;
	
	}

/*****************************************************************************/

static void ReadSome (GPtr globals, int32 count, void *buffer)
	{
	
	int32 readCount = count;
	
	if (gResult != noErr)
		return;
	
	gResult = FSRead (gStuff->dataFork, &readCount, buffer);
	
	if (gResult == noErr && readCount != count)
		gResult = eofErr;
	
	}

/*****************************************************************************/

static void WriteSome (GPtr globals, int32 count, void *buffer)
	{
	
	int32 writeCount = count;
	
	if (gResult != noErr)
		return;
	
	gResult = FSWrite (gStuff->dataFork, &writeCount, buffer);
	
	if (gResult == noErr && writeCount != count)
		gResult = dskFulErr;
	
	}

/*****************************************************************************/

static void ReadRow (GPtr globals)
	{
	
	ReadSome (globals, gRowBytes, gPixelData);
	
	}

/*****************************************************************************/

static void WriteRow (GPtr globals)
	{
	
	WriteSome (globals, gRowBytes, gPixelData);
	
	}

/*****************************************************************************/

static void DisposeImageResources (GPtr globals)
	{
	
	if (gStuff->imageRsrcData)
		{
		
		PIDisposeHandle (gStuff->imageRsrcData);
		
		gStuff->imageRsrcData = NULL;
		
		gStuff->imageRsrcSize = 0;
		
		}
	
	}

/*****************************************************************************/

static void DoReadStart (GPtr globals)
{
	
	// If you add fmtCanCreateThumbnail to the FormatFlags PiPL property
	// you will get called to create a thumbnail. The only way to tell
	// that you are processing a thumbnail is to check openForPreview in the
	// FilterRecord. You do not need to parse the entire file. You need to
	// process enough for a thumbnail view and you need to do it quickly.

	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
		                                          NULL, png_error_cb, png_warning_cb );

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		return;
	}
	
	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return;
	}

	// Hook up our own read function. Since we did not open the file ourselves, we need Photoshop
	// to do the file reading itself.
	png_set_read_fn( png_ptr, globals, png_read );

	if( read_png( &png_ptr, &info_ptr ))
	{
		sPngPtr = info_ptr;

		// Fill in Photoshop's internal file format descriptor with the png info
		gStuff->imageSize.v = (short) info_ptr->height;
		gStuff->imageSize.h = (short) info_ptr->width;
		gStuff->depth = ( info_ptr->bit_depth < 8 ) ? 8 : info_ptr->bit_depth;
		gStuff->planes = info_ptr->channels;

		// If there is a transparency chunk, then the resultant indexed image will have 2 color planes : an index
		// plane and an alpha plane
		if( png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS ))
		{
			gStuff->planes++;
		}

		switch( info_ptr->color_type )
		{			
			case PNG_COLOR_TYPE_PALETTE:	
			{
				int i, num_pal_entries;

				gStuff->imageMode = plugInModeIndexedColor;
				if( info_ptr->bit_depth == 1 )
				{
					num_pal_entries = 2;
					gStuff->lutCount = 2;
				}
				else if( info_ptr->bit_depth == 2 )
				{
					num_pal_entries = 4;
					gStuff->lutCount = 4;
				}
				else if( info_ptr->bit_depth == 4 )
				{
					num_pal_entries = 16;
					gStuff->lutCount = 16;
				}
				else if( info_ptr->bit_depth == 8 )
				{
					num_pal_entries = 256;
					gStuff->lutCount = 256;
				}
				else
				{
					return;
				}

				// Read in the palette
				for( i = 0; i < num_pal_entries; i++ )
				{
					gStuff->redLUT[i] = info_ptr->palette[i].red;
					gStuff->greenLUT[i] = info_ptr->palette[i].green;
					gStuff->blueLUT[i] = info_ptr->palette[i].blue;
				}

				// Clear out the rest of the palette
				for( i = num_pal_entries; i < 256; i++ )
				{
					gStuff->redLUT[i] = 0;
					gStuff->greenLUT[i] = 0;
					gStuff->blueLUT[i] = 0;
				}

				break;
			}
			case PNG_COLOR_TYPE_RGB:
				gStuff->imageMode = plugInModeRGBColor;
				break;
			case PNG_COLOR_TYPE_RGBA:
				gStuff->imageMode = plugInModeRGBColor;
				break;
			case PNG_COLOR_TYPE_GRAY:				
			case PNG_COLOR_TYPE_GA:				
			default:
				// We don't support these (yet)
				return;
		}
	}
}

/*****************************************************************************/

static void DoReadContinue (GPtr globals)
{
	
	int32 done;
	int32 total;
	int16 row;
	int16 col;
	unsigned char* dst, *src;
	
	/* Dispose of the image resource data if it exists. */
	
	DisposeImageResources (globals);
	
	/* Set up the progress variables. */
	
	done = 0;
	total = gStuff->imageSize.v;
		
	/* Next, we will allocate the pixel buffer. */
	
	AllocatePixelBuffer (globals);
	
	/* Set up to start returning chunks of data. */
	
	gStuff->theRect.left = 0;
	gStuff->theRect.right = gStuff->imageSize.h;
	
	gStuff->colBytes = ((gStuff->depth * gStuff->planes ) + 7) >> 3;
	gStuff->planeBytes = 1;
	if (gStuff->imageMode == plugInModeIndexedColor)
	{
		if( gStuff->planes != 2 )
		{
			gStuff->planeBytes = 0;
		}
	}
	gStuff->rowBytes = gRowBytes;
	
	gStuff->data = gPixelData;
	if (gStuff->depth == 16)
		gStuff->maxValue = 0x8000;

	// Tell Photoshop that we intend to write all planes of data together rather than one-at-a-time
	gStuff->loPlane = 0;
	gStuff->hiPlane = gStuff->planes - 1;	
	for (row = 0; gResult == noErr && row < gStuff->imageSize.v; ++row)
	{			
		// Tell Photoshop that we intend to write one row at-a-time
		gStuff->theRect.top = row;
		gStuff->theRect.bottom = row + 1;
			
		dst = (unsigned char*) gPixelData;
		// If our source bit depth is 8, we can just do a byte-wise row copy
		if( sPngPtr->bit_depth == 8 )
		{
			// If there is a transparency chunk, we need to manually interleave indices and transparency data
			if( gStuff->planes == 2 )
			{				
				for( col = 0; col < gStuff->imageSize.h; col++ )
				{
					*dst++ = sPngPtr->row_pointers[row][col];
					if( sPngPtr->row_pointers[row][col] < sPngPtr->num_trans )
					{
						*dst++ = sPngPtr->trans[sPngPtr->row_pointers[row][col]];
					}
					else
					{
						// If we have no transparency data for this palette entry, just write an opaque value
						*dst++ = 255;
					}					
				}
			}
			else
			{
				memcpy( gPixelData, sPngPtr->row_pointers[row], sPngPtr->rowbytes );
			}
		}
		else
		{
			unsigned int i, col;
			unsigned char color_data[16384];
			unsigned char* dst;
			int pixel_count;
			
			// otherwise, we need to copy the values manually into the pixelData buffer
			// since it's expecting byte-length values
			memcpy( color_data, sPngPtr->row_pointers[row], sPngPtr->rowbytes );
			dst = (unsigned char*) gPixelData;
			src = color_data;
			pixel_count = 0;
			switch( sPngPtr->bit_depth )
			{
				case 1:
					for( i = 0; i < sPngPtr->rowbytes; i++ )
					{
						unsigned char byte, color;
						
						// Write the first 4-bit color as a 8-bit color value since Photoshop only works at 8-bit
						byte = color_data[i];
						color = byte & 0x80;
						color >>= 7;
						*dst++ = color;
						pixel_count++;
						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}
						
						color = byte & 0x40;
						color >>= 6;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x20;
						color >>= 5;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x10;
						color >>= 4;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x08;
						color >>= 3;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x04;
						color >>= 2;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x02;
						color >>= 1;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x01;
						*dst++ = color;
						pixel_count++;
					}
					break;
				case 2:
					for( i = 0; i < sPngPtr->rowbytes; i++ )
					{
						unsigned char byte, color;
						
						// Write the first 4-bit color as a 8-bit color value since Photoshop only works at 8-bit
						byte = color_data[i];
						color = byte & 0xC0;
						color >>= 6;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x30;
						color >>= 4;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}

						color = byte & 0x0C;
						color >>= 2;
						*dst++ = color;
						pixel_count++;

						if( pixel_count == gStuff->imageSize.h )
						{
							break;
						}
						
						color = byte & 0x03;
						*dst++ = color;
						pixel_count++;
					}
					break;
				case 4:
					if( gStuff->planes == 2 )
					{
						for( col = 0; col < sPngPtr->rowbytes; col++ )
						{
							unsigned char byte, color;
							
							byte = color_data[col];
							//color = byte & 0x0F;
							color = byte >> 4;//& 0x0F;
							
							// Write the first 4-bit color as a 8-bit color value since Photoshop only works at 8-bit
							*dst++ = color;
							if( color < sPngPtr->num_trans )
							{
								*dst++ = sPngPtr->trans[color];
							}
							else
							{
								*dst++ = 255;
							}							

							pixel_count++;
							if( pixel_count == gStuff->imageSize.h )
							{
								break;
							}

							color = (byte & 0x0F);//( byte >> 4 );
							*dst++ = color;
							if( color < sPngPtr->num_trans )
							{
								*dst++ = sPngPtr->trans[color];
							}
							else
							{
								*dst++ = 255;
							}							
							pixel_count++;
						}					
					}
					else
					{					
						for( col = 0; col < sPngPtr->rowbytes; col++ )
						{
							unsigned char byte, color;
							
							// Write the first 4-bit color as a 8-bit color value since Photoshop only works at 8-bit
							byte = color_data[col];
							//color = byte & 0x0F;
							color = ( byte >> 4 );
							*dst++ = color;
							pixel_count++;
							if( pixel_count == gStuff->imageSize.h )
							{
								break;
							}

							color = byte & 0x0F;
							//color = ( byte >> 4 );
							*dst++ = color;
							pixel_count++;
						}					
					}
					break;
				default:
					return;
			}
		}
			
		// AdvanceState() actually copies the row of data to the Photoshop target
		if (gResult == noErr)
			gResult = AdvanceState ();
				
		// Update the progress bar
		UpdateProgress (++done, total);	
	}

	gStuff->data = NULL;
	
	DisposePixelBuffer (globals);

	DoReadICCProfile (globals);
	
}

void	PreviewDrawProc( ADMItemRef preview, ADMDrawerRef drawer )
{
	ASRect rect;
	ASPoint point;

	ADMImageRef image = sADMImage->Create( 100, 100 );
	ASWindowRef window = sADMDialog->GetWindowRef( sDialogRef );	
	rect.left = 50;
	rect.top = 50;
	rect.right = 1000;
	rect.bottom = 500;

	point.h = 100;
	point.v = 100;

	//ADMDrawerRef drawer = sADMDrawer->Create( sADMDrawer->GetADMWindowPort( window ), &rect, kADMDefaultFont);
	sADMDrawer->DrawADMImage( drawer, image, &point );
}

/*****************************************************************************/

static void DoReadFinish (GPtr globals)
	{
	
	/* Dispose of the image resource data if it exists. */
	
	DisposeImageResources (globals);
	WriteScriptParamsOnRead (globals); // should be different for read/write
	AddComment (globals); // write a history comment

	}

ASErr ASAPI InitProc( ADMDialogRef reference )
{
	ADMItemRef dlgItem;
	
	GPtr globals = (GPtr) sADMDialog->GetUserData( reference );

	dlgItem = sADMDialog->GetItem( reference, IDC_8_BIT );
	sADMItem->SetBooleanValue( dlgItem, true );

	if (gStuff->imageMode == plugInModeIndexedColor)
	{
		dlgItem = sADMDialog->GetItem( reference, IDC_TRUECOLOR );
		sADMItem->Enable( dlgItem, false );
	}

	if((gStuff->planes == 2 ) || ( gStuff->planes == 4 ))
	{
		dlgItem = sADMDialog->GetItem( reference, IDC_2_BIT );
		sADMItem->Enable( dlgItem, false );

		dlgItem = sADMDialog->GetItem( reference, IDC_1_BIT );
		sADMItem->Enable( dlgItem, false );
	}

	return kNoErr;
}

/****************************************************************************/

static void DoOptionsPrepare (GPtr globals)
{
	SPErr error;

	gStuff->maxData = ( 10 * ( 1024 * 1000 ));
		
	error = sSPBasic->AcquireSuite( kADMDialogSuite, kADMDialogSuiteVersion,  (const void **) &sADMDialog );
	error = sSPBasic->AcquireSuite( kADMItemSuite, kADMItemSuiteVersion, (const void **) &sADMItem );
	error = sSPBasic->AcquireSuite( kADMDialogSuite, kADMDialogSuiteVersion,  (const void **) &sADMDialog );
	error = sSPBasic->AcquireSuite( kADMImageSuite, kADMImageSuiteVersion,  (const void **) &sADMImage );
	error = sSPBasic->AcquireSuite( kADMDrawerSuite, kADMDrawerSuiteVersion,  (const void **) &sADMDrawer );

	sInterlaced = false;

	if (gStuff->imageMode != plugInModeIndexedColor)
	{
		sDialogRef = sADMDialog->Create((struct SPPlugin *) gStuff->plugInRef, "NxPNG Parameters", IDD_PNG_PARAMS, kADMModalDialogStyle, InitProc, globals );	
		int32 dlg_result = sADMDialog->DisplayAsModal( sDialogRef );
		if( dlg_result == IDCANCEL )
		{
			*globals->result = userCanceledErr;
			sADMDialog->Destroy( sDialogRef );	

		}
	}
	else
	{
		sIndexed = true;

		if( gStuff->lutCount > 16 )
		{
			sNumColors = 256;
		}
		else if( gStuff->lutCount > 4 )
		{
			sNumColors = 16;
		}
		else if( gStuff->lutCount > 2 )
		{
			sNumColors = 4;
		}
		else
		{
			sNumColors = 2;
		}
	}
}

/*****************************************************************************/

static void DoOptionsStart (GPtr globals)
{
	/* Check for the needed services. */
	
	if (!TSC ((Boolean)(!CheckForServices (globals)))) return;
	if (gStuff->imageMode == plugInModeIndexedColor) return;	// we only need these options for truecolor images

	gStuff->data = NULL;	

	ADMItemRef item;

	item = sADMDialog->GetItem( sDialogRef, IDC_INTERLACED );
	if( sADMItem->GetBooleanValue( item ))
	{
		sInterlaced = true;
	}	
	else
	{
		sInterlaced = false;
	}

	item = sADMDialog->GetItem( sDialogRef, IDC_TRUECOLOR );
	if( sADMItem->GetBooleanValue( item ))
	{
		sIndexed = false;
	}
	else
	{
		sIndexed = true;
		item = sADMDialog->GetItem( sDialogRef, IDC_8_BIT );
		if( sADMItem->GetBooleanValue( item ))
		{
			sNumColors = 256;
		}
		else
		{
			item = sADMDialog->GetItem( sDialogRef, IDC_4_BIT );
			if( sADMItem->GetBooleanValue( item ))
			{
				sNumColors = 16;
			}
			else
			{
				item = sADMDialog->GetItem( sDialogRef, IDC_2_BIT );
				if( sADMItem->GetBooleanValue( item ))
				{
					sNumColors = 4;
				}
				else
				{
					sNumColors = 2;
				}
			}
		}	
	}
}

/*****************************************************************************/

static void DoOptionsContinue (GPtr globals)
{
	#ifdef __PIMWCW__
		#pragma unused (globals) // remove this when you write this routine
	#endif
}

/*****************************************************************************/

static void DoOptionsFinish (GPtr globals)
{
	#ifdef __PIMWCW__
		#pragma unused (globals) // remove this when you write this routine
	#endif
}

/*****************************************************************************/

static void DoEstimatePrepare (GPtr globals)
{
	gStuff->maxData = 0;

}

/*****************************************************************************/

static void DoEstimateStart (GPtr globals)
{
	
	int32 dataBytes;
	
	/* Check for the needed services. */
	
	if (!TSC ((Boolean)(!CheckForServices (globals)))) return;

	dataBytes = sizeof (FileHeader) +
				gStuff->imageRsrcSize +
				RowBytes (globals) * gStuff->planes * gStuff->imageSize.v;
					  
	if (gStuff->imageMode == plugInModeIndexedColor)
		dataBytes += 3 * sizeof (LookUpTable);
		
	gStuff->minDataBytes = dataBytes;
	gStuff->maxDataBytes = dataBytes;
	
	gStuff->data = NULL;

}

/*****************************************************************************/

static void DoEstimateContinue (GPtr globals)
{
	#ifdef __PIMWCW__
		#pragma unused (globals) // remove this when you write this routine
	#endif
}

/*****************************************************************************/

static void DoEstimateFinish (GPtr globals)
{
	#ifdef __PIMWCW__
		#pragma unused (globals) // remove this when you write this routine
	#endif
}

/*****************************************************************************/

static void DoWritePrepare (GPtr globals)
{
	gStuff->maxData = 0;	
}

void png_write( png_structp png_struct, png_bytep data, png_size_t size )
{
	GPtr globals = (GPtr) png_struct->io_ptr;
	WriteSome( globals, size, data );
}

void png_read( png_structp png_struct, png_bytep data, png_size_t size )
{
	GPtr globals = (GPtr) png_struct->io_ptr;
	ReadSome( globals, size, data );
}

void png_flush( png_structp png_struct )
{
}

void png_error_cb( png_structp png_struct, png_const_charp error )
{
	printf( error );
}

void png_warning_cb( png_structp png_struct, png_const_charp warning )
{
	printf( warning );
}

/*****************************************************************************/

void Dither24( ImageData* ditheredImage, ImageData* quantizedImage, ImageData* origImage )
{
	Quantizer quant;

	// Matrix for determining position of neighboring pixels (Reduced to affected pixels)
	const Point2D PosMatrix[4] = {	Point2D( 1, 0),
									Point2D(-1, 1),
									Point2D( 0, 1),
									Point2D( 1, 1) };

	// Floyd-Steinberg error diffusion matrix showing percent of color error
	// that should be propagated over the above positions
	const float ErrMatrix[4] = { 7.0f / 7.0f,
		                         3.0f / 7.0f,
								 5.0f / 7.0f,
								 1.0f / 7.0f };

	*ditheredImage = *quantizedImage;

	color24* sBuf = (color24*)origImage->buf;
	color24* dBuf = (color24*)ditheredImage->buf;

	TreeData* ctree = quant.BuildColorTree(quantizedImage);

	//return;

	int x,y;
	int width  = origImage->width;
	int height = origImage->height;

	int rDiff, gDiff, bDiff;

	int ydir,xdir;
	ydir = 1;
	xdir = 1;

	for(y = 0; y < height; y+=ydir)
	{
		if (xdir > 0)
			x = 0;
		else
			x = width;

		int oldy = y;

		for(; (xdir > 0) ? (x < width) : (x > -1); x+=xdir)
		{
			y++;
			color24* ptColor = &sBuf[y*width+x];
			color24*  newClr = &dBuf[y*width+x];

//			rDiff = newClr->r - ptColor->r;
//			gDiff = newClr->g - ptColor->g;
//			bDiff = newClr->b - ptColor->b;

			rDiff = ptColor->r - newClr->r;
			gDiff = ptColor->g - newClr->g;
			bDiff = ptColor->b - newClr->b;


			// Update neighboring pixels
			for(int i=0;i<4;i++)
			{
				Point2D ptSample = Point2D(x+PosMatrix[i].x,y+PosMatrix[i].y);

				if (ptSample.x > -1 && ptSample.x < width &&
					ptSample.y > -1 && ptSample.y < height)
				{
					color24* ptNP = &dBuf[(y+PosMatrix[i].y)*width+x+PosMatrix[i].x];
					
					float tval;
					
					tval = ptNP->r + rDiff * ErrMatrix[i];

					// Perform clamping on the computed value
					if (tval < 0)
						tval = 0;

					if (tval > 255)
						tval = 255;

					ptNP->r = (unsigned char)tval;
					
					tval = ptNP->g + gDiff * ErrMatrix[i];

					// Perform clamping on the computed value
					if (tval < 0)
						tval = 0;

					if (tval > 255)
						tval = 255;

					ptNP->g = (unsigned char)tval;

					tval = ptNP->b + bDiff * ErrMatrix[i];

					// Perform clamping on the computed value
					if (tval < 0)
						tval = 0;

					if (tval > 255)
						tval = 255;

					ptNP->b = (unsigned char)tval;

					//color32 color;
					__int32 dist = -1;
					//quant.FindLinearMatch(ptNP->r,ptNP->g,ptNP->b,255,ctree,&color,&dist);
					//color = quant.FindColorMatch(ctree,ptNP->r,ptNP->g,ptNP->b);

					color32 color = quant.GetClosestColor32(ctree->root,ptNP->r,ptNP->g,ptNP->b);
					ptNP->r = color.r;
					ptNP->g = color.g;
					ptNP->b = color.b;

				}
			}
			y = oldy;
		}
	

		if (xdir == 1)
			xdir = -1;
		else
			xdir = 1;

	}

	delete ctree;
}

void SetImageDataAndPalette( ImageData* dst_image, png_color* palette, unsigned char** pRows, int max_colors )
{
	int i;
	png_byte* color_dst;
	
	dst_image->buf = (unsigned char*) malloc(dst_image->width * dst_image->height );
	dst_image->imageSize = dst_image->width * dst_image->height;
	dst_image->color_type = PNG_COLOR_TYPE_PALETTE;
	dst_image->bit_depth = 8;
	dst_image->nChannels = 1;		
	dst_image->palette = (png_color*)malloc(sizeof(png_color) * max_colors );
	dst_image->nPalEntries = max_colors;
	memcpy( dst_image->palette, palette, sizeof( png_color ) * max_colors );
	color_dst = (png_byte*) dst_image->buf;
	for( i = 0; i < (int) dst_image->height; i++ )
	{
		memcpy( color_dst, pRows[i], dst_image->width );
		color_dst += dst_image->width;
	}
}

bool WritePNG( GPtr globals)
{
	int width, height, bit_depth, pixel_depth, color_type, interlace_type, compression_type, filter_type, i, num_pal_entries;
	int dst_bit_depth;
	png_color*      palette;			
	bool			has_transparency;
	unsigned char** pRows;				// Pointer to scanlines of data in image	
	int row, col;
	
	height = gStuff->imageSize.v;
	width = gStuff->imageSize.h;		
	bit_depth = gStuff->depth;		
	filter_type = PNG_FILTER_TYPE_DEFAULT;
	compression_type = PNG_COMPRESSION_TYPE_DEFAULT;
	if( sInterlaced )
	{
		interlace_type = PNG_INTERLACE_ADAM7;
	}
	else
	{
		interlace_type = PNG_INTERLACE_NONE;
	}

	has_transparency = false;
	num_pal_entries = 256;
	palette = NULL;

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
		                                          NULL, png_error_cb, png_warning_cb );

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	// Hook up our own write function. Since we didn't open the file ourselves, we need to let Photoshop
	// write data to the output stream itself
	png_set_write_fn( png_ptr, globals, png_write, png_flush );

	if( sIndexed )
	{
		color_type = PNG_COLOR_TYPE_PALETTE;		
		if( gStuff->imageMode == plugInModeIndexedColor)
		{
			// Two planes in an indexed image indicates that alpha values are interleaved with palette indices
			if( gStuff->planes == 2 )
			{
				pixel_depth = gStuff->depth;
				// PlaneBytes needs to be set if the image is not a simple indexed image without alpha
				gStuff->planeBytes = 1;
				has_transparency = true;
			}
			else
			{
				gStuff->planeBytes = 0;
				pixel_depth = gStuff->depth * gStuff->planes;	
			}		
		}
		else
		{			
			pixel_depth = gStuff->depth * gStuff->planes;
			if( pixel_depth == 32 )
			{
				has_transparency = true;
			}
			// PlaneBytes needs to be set if the image is not a simple indexed image without alpha			
			gStuff->planeBytes = 1;
		}
	}
	else
	{		
		pixel_depth = gStuff->depth * gStuff->planes;
		if( pixel_depth == 32 )
		{
			color_type = PNG_COLOR_TYPE_RGB_ALPHA;			
		}
		else
		{
			color_type = PNG_COLOR_TYPE_RGB;
		}
		// PlaneBytes needs to be set if the image is not a simple indexed image without alpha			
		gStuff->planeBytes = 1;
	}

	dst_bit_depth = 8;
	if( sIndexed )
	{
		switch( sNumColors )
		{
			case 16:
				dst_bit_depth = 4;
				break;
			case 4:
				dst_bit_depth = 2;
				break;
			case 2:
				dst_bit_depth = 1;
				break;
			default:				
				break;
		}		
	}
	
	// Now read in the source image from Photoshop's internal format
	
	// Tell Photoshop we want to consider the entire image
	gStuff->theRect.left = 0;
	gStuff->theRect.right = gStuff->imageSize.h;

	// Tell Photoshop how many bytes per pixel and per row
	gStuff->colBytes = ((gStuff->depth * gStuff->planes ) + 7) >> 3;
	gStuff->rowBytes = gRowBytes;

	// Read in the palette if one exists
	if (gStuff->imageMode == plugInModeIndexedColor)
	{		
		palette = (png_color*)malloc(sizeof(png_color)*num_pal_entries);			
		for( i = 0; i < num_pal_entries; i++ )
		{
			palette[i].red = gStuff->redLUT[i];
			palette[i].green = gStuff->greenLUT[i];
			palette[i].blue = gStuff->blueLUT[i];
		}		
	}
	
	// Now read in the pixel data
	gStuff->data = gPixelData;		

	// We want all of the color planes at once instead of one-at-a-time
	gStuff->loPlane = 0;
	gStuff->hiPlane = gStuff->planes - 1;
	
	// Store the image data in a series of rows
	pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);
	for (row = 0; gResult == noErr && row < height; ++row)
	{	
		gStuff->theRect.top = row;
		gStuff->theRect.bottom = row + 1;
		
		// AdvanceState() actually copies pixel data into the global row of data pointed to by gPixelData
		if (gResult == noErr)
			gResult = AdvanceState ();
			
		pRows[row] = (png_bytep) malloc( gRowBytes );
		memcpy( pRows[row], gPixelData, gRowBytes );
				
		// Update the progress bar
		UpdateProgress (row, height);		
	}		
	
	// If the destination image is an indexed format
	if( sIndexed )
	{
		Quantizer  quant;
		ImageData src_image, dst_image;
		unsigned char* data;
		int col_index, alpha_index;
		bool should_quantize;

		should_quantize = true;

		// If Photoshop is handing us a non-transparent image that is already color-reduced,
		// don't quantize it. Leave its palette unchanged.
		if(	( gStuff->lutCount <= sNumColors ) &&
			( has_transparency == false ) &&
			( gStuff->imageMode == plugInModeIndexedColor))
		{
			should_quantize = false;
		}

		if( should_quantize )
		{
			// If there's transparency info in this paletted image, we need to take all image data
			// as input and quantize the palette with all data					
			src_image.bit_depth = 8;
			src_image.nChannels = ( has_transparency == true ) ? 4 : 3;
			src_image.width = width;
			src_image.height = height;
			src_image.color_type = ( has_transparency == true) ? PNG_COLOR_TYPE_RGBA : PNG_COLOR_TYPE_RGB;
			src_image.compression_type = compression_type;
			src_image.color_type = color_type;
			src_image.filter_type = filter_type;
			src_image.buf = new unsigned char[ src_image.nChannels * width * height ];
			data = src_image.buf;
			
			// Add every pixel of the src image to a copy to be quantized
			for( row = 0; row < height; row++ )
			{
				if( gStuff->imageMode == plugInModeIndexedColor)
				{
					for( col = 0; col < width; col++ )
					{				
						if( has_transparency )
						{
							// If it's an indexed image with alpha, its pixels (palette offsets) are interveaved
							// with alpha values (0-255 range)
							alpha_index = ( col * 2 ) + 1;
							col_index = col * 2;
							*data++ = palette[pRows[row][col_index]].red;
							*data++ = palette[pRows[row][col_index]].green;
							*data++ = palette[pRows[row][col_index]].blue;				
							*data++ = pRows[row][alpha_index];
						}
						else
						{
							// Otherwise, it's all palette offsets
							*data++ = palette[pRows[row][col]].red;
							*data++ = palette[pRows[row][col]].green;
							*data++ = palette[pRows[row][col]].blue;									
						}								
					}
				}
				else
				{
					memcpy( data, pRows[row], src_image.nChannels * width );
					data += src_image.nChannels * width;
				}
			}

			NxQuantizeImage( &dst_image, &src_image, sNumColors );
		}
		else
		{
			dst_image.bit_depth = 8;
			dst_image.nChannels = 3;
			dst_image.width = width;
			dst_image.height = height;
			dst_image.color_type = ( has_transparency == true) ? PNG_COLOR_TYPE_RGBA : PNG_COLOR_TYPE_RGB;
			dst_image.compression_type = compression_type;
			dst_image.color_type = color_type;
			dst_image.filter_type = filter_type;
			dst_image.buf = new unsigned char[ dst_image.nChannels * width * height ];
			SetImageDataAndPalette( &dst_image, palette, pRows, sNumColors );
		}		
		
		// free the old row info, since it no longer applies to the quantized palette
		for (row = 0; gResult == noErr && row < height; ++row)
		{	
			free( pRows[ row ] );
		}
		// Now, reallocate rows according to destination pixel format
		for (row = 0; gResult == noErr && row < height; ++row)
		{	
			unsigned char byte;
			unsigned char* dst_byte;
			
			pRows[row] = (png_bytep) malloc((( dst_bit_depth * width ) + 7 ) >> 3 );
			dst_byte = pRows[row];
			for( col = 0; col < width; col++ )
			{
				switch( dst_bit_depth )
				{
					case 1:
					{
						switch( col & 0x07 )
						{
						case 0:
							byte = dst_image.buf[ ( width * row ) + col ];
							break;
						case 1:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 2:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 3:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 4:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 5:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 6:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 7:
							byte <<= 1;
							byte |= dst_image.buf[ ( width * row ) + col ];
							*dst_byte++ = byte;
							break;
						}						
						break;
					}
					case 2:
					{
						switch( col & 0x03 )
						{
						case 0:
							byte = dst_image.buf[ ( width * row ) + col ];
							break;
						case 1:
							byte <<= 2;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 2:
							byte <<= 2;
							byte |= dst_image.buf[ ( width * row ) + col ];
							break;
						case 3:
							byte <<= 2;
							byte |= dst_image.buf[ ( width * row ) + col ];
							*dst_byte++ = byte;
							break;
						}						
						break;
					}
					case 4:
					{
						if(( col & 1 ) == 0 )
						{
							byte = dst_image.buf[ ( width * row ) + col ];
						}
						else
						{
							byte <<= 4;
							byte |= dst_image.buf[ ( width * row ) + col ];
							*dst_byte++ = byte;							
						}						
						break;
					}
					case 8:
					{
						pRows[row][col] = dst_image.buf[ ( width * row ) + col ];
						break;
					}
				}
			}			
		}		

		// Set up the PNG header with the destination image format
		png_set_IHDR(png_ptr,info_ptr,
	             width,height,
				 dst_bit_depth,
				 color_type,
				 interlace_type,
				 compression_type,
				 filter_type);

		// Set the PNG palette
		if ( dst_image.nPalEntries > 0)
		{
			png_set_PLTE(png_ptr,info_ptr, dst_image.palette, sNumColors );
		}
		
		if ( dst_image.nPals > 0 )
		{
			png_set_sPLT(png_ptr,info_ptr, dst_image.pals, dst_image.nPals );
		}

		// Set tRNS data
		if( has_transparency )
		{
			png_set_tRNS(png_ptr,info_ptr, dst_image.tRNSpal, dst_image.tRNSentries, &dst_image.tRNStrans_values );
		}		
		
		// Write out the rest of the image info
		png_write_info(png_ptr,info_ptr);
		png_write_image(png_ptr,pRows);		

		if( palette != NULL )
		{
			free( palette );
		}		
	}
	else
	{
		// Set up the PNG header with the destination image format
		png_set_IHDR(png_ptr,info_ptr,
	             width,height,
				 dst_bit_depth,
				 color_type,
				 interlace_type,
				 compression_type,
				 filter_type);

		png_write_info(png_ptr,info_ptr);
		png_write_image(png_ptr,pRows);
		
	}	

	png_write_end(png_ptr,info_ptr);
	png_destroy_write_struct(&png_ptr,&info_ptr);
	for (row = 0; gResult == noErr && row < height; ++row)
	{	
		free( pRows[ row ] );
	}
	free(pRows);
	
	return true;
}

static void DoWriteStart (GPtr globals)
{	
	/* See if I have any of my comments in this file */
	/* I don't do anything with them but it was fun. */
	int16 resourceCount = CountPIResources(histResource);

	while (resourceCount)
		{
		Handle resourceHandle = GetPIResource(histResource,resourceCount--);
		Ptr resourcePtr = PILockHandle(resourceHandle, FALSE);
		PIUnlockHandle(resourceHandle);
		}

	ReadScriptParamsOnWrite (globals); // read script params here
	
    if (gResult != noErr) return;
		
	/* Check for the needed services. */
	
	if (!TSC ((Boolean)(!CheckForServices (globals)))) return;

	/* Write the header. */
	
	gResult = SetFPos (gStuff->dataFork, fsFromStart, 0);

	AllocatePixelBuffer (globals);

	WritePNG( globals );	
		
	gStuff->data = NULL;
	
	DisposePixelBuffer (globals);

	DoWriteICCProfile (globals);
	
}

/*****************************************************************************/

static void DoWriteContinue (GPtr globals)
{
	#ifdef __PIMWCW__
		#pragma unused (globals) // remove this when you write this routine
	#endif
}

/*****************************************************************************/

static void DoWriteFinish (GPtr globals)
{
	WriteScriptParamsOnWrite (globals); // should be different for read/write
	sADMDialog->Destroy( sDialogRef );	
}

/*****************************************************************************/

static void DoFilterFile (GPtr globals)
{
	
	FileHeader header;
	
	/* Exit if we have already encountered an error. */

    if (gResult != noErr) return;
		
	/* Read the file header. */
	   
	if (!TSR (SetFPos (gStuff->dataFork, fsFromStart, 0))) return;
	
	ReadSome (globals, sizeof (FileHeader), &header);
	
	if (gResult != noErr) return;
	
	/* Check the identifier. */
	
	if (!CheckIdentifier (header.identifier))
		{
		gResult = formatCannotRead;
		return;
		}
	
}

/*****************************************************************************/

/* This routine adds a history entry to the file when incoming. */

static void AddComment (GPtr globals)
{
	
	/* We will attempt to add a resource showing that we made this change. */
	
	Str255 s, s0;
	Handle h;
	
	if (!ResourceProcsAvailable (NULL))
		return;
		
	PIGetString(kHistoryEntry, s);
	// Pulls a pascal string from the resource tree.
	
	NumToString(gStuff->dataFork, s0);
	// Need some unique data for this entry
	
	PIParamText(s, s0, NULL, NULL);

	h = PIPString2Handle(s);
		
	if (h != NULL)
	{
		PILockHandle(h, FALSE);
		(void) AddPIResource (histResource, h);
		PIUnlockHandle(h);
		PIDisposeHandle(h);
		h = NULL;
	}
	
	if (gHistory[0] > 0)
	{ // we had an old history.  Write that back out, as well.
		h = PIPString2Handle(gHistory);
		
		if (h != NULL)
		{ // got our handle. Write it out.
			gHistory[ (gHistory[0]=0)+1 ] = 0; // clear string
			PILockHandle(h, FALSE);
			(void) AddPIResource (histResource, h);
			PIUnlockHandle(h);
			PIDisposeHandle(h);
			h = NULL;
		}
	} // no earlier history
} // end AddComment

/**************************************************************************/
// Returns a history entry in a string

static void GetHistory (GPtr globals, int16 index, Str255 s)
{	
	int16 currentResources = CountPIResources(histResource);
	
	PIResetString(s);
	
	if (currentResources >= index)
	{
		Handle h = GetPIResource (histResource, index);	
		if (h != NULL) PIHandle2PString(h, s);
		// just a read-only reference.  Do NOT dispose.
	}
}

static void DoReadICCProfile(GPtr globals)
{
	if (gStuff->canUseICCProfiles)
	{
		ReadSome(globals, sizeof(gStuff->iCCprofileSize), &gStuff->iCCprofileSize);
		if (gResult == noErr)
		{
			gStuff->iCCprofileData = PINewHandle(gStuff->iCCprofileSize);
			Ptr data = PILockHandle(gStuff->iCCprofileData, FALSE);
			if (gStuff->iCCprofileData != NULL && data != NULL)
			{
				ReadSome(globals, gStuff->iCCprofileSize, data);
				PIUnlockHandle(gStuff->iCCprofileData);
			}
		}
		else
		{
			gResult = noErr; // it's an old file or one without ICC info, no worries
		}
	}
}

static void DoWriteICCProfile(GPtr globals)
{
	if (gStuff->canUseICCProfiles)
	{
		if (gStuff->iCCprofileSize && gStuff->iCCprofileData)
		{
			WriteSome(globals, sizeof(gStuff->iCCprofileSize), &gStuff->iCCprofileSize);
			Ptr data = PILockHandle(gStuff->iCCprofileData, FALSE);
			if (data != NULL)
			{
				WriteSome(globals, gStuff->iCCprofileSize, data);
				PIUnlockHandle(gStuff->iCCprofileData);
			}
		}
	}
}
