#include "stdafx.h"
#include "engine.h"
#include "stdio.h"
#include "string.h"
#include "stdlib.h"
#include "resource.h"
#include "utils.h"
#include "mycode.h"
#include "checksum.h"
#include "core/crc.h"
#include "checksumdownloadprogress.h"
#include "monitor.h"
#include "mytreectrl.h"

#include "sk/gamenet/scriptdebugger.h"
#include "gel/scripting/symboltype.h"
#include "gel/scripting/checksum.h"

int WM_DEBUGGER_HELP = RegisterWindowMessage ("DEBUGGER_HELP");
int WM_DEBUGGER_REFRESH = RegisterWindowMessage ("DEBUGGER_REFRESH");
int WM_DEBUGGER_INSERT = RegisterWindowMessage ("DEBUGGER_INSERT");
int WM_DEBUGGER_DELETE = RegisterWindowMessage ("DEBUGGER_DELETE");


//#include "sk/modules/skate/skate.h"
// This enum is in skate.h, but I get a compile error if I try and include it.
// TODO: Either fix the compile error or factor this enum out into a seperate
// header file.
enum
{
	SKATE_TYPE_UNDEFINED 		= 0,
	SKATE_TYPE_SKATER 			= 1,
	SKATE_TYPE_PED				= 2,
	SKATE_TYPE_CAR				= 3,
	SKATE_TYPE_GAME_OBJ			= 4,
	SKATE_TYPE_BOUNCY_OBJ		= 5,
	SKATE_TYPE_CASSETTE			= 6,
	SKATE_TYPE_ANIMATING_OBJECT	= 7,
	SKATE_TYPE_CROWN			= 8,
	SKATE_TYPE_PARTICLE			= 9,
	SKATE_TYPE_REPLAY_DUMMY		= 10,
};


S4Bytes Read4Bytes(const uint8 *p_long)
{
    S4Bytes four_bytes;
	four_bytes.mUInt=*(uint32*)p_long;
    return four_bytes;
}

S2Bytes Read2Bytes(const uint8 *p_short)
{
    S2Bytes two_bytes;
    two_bytes.mUInt=*(uint16*)p_short;
    return two_bytes;
}

uint8 *InsertCArrayIntoTree(CMyTreeCtrl *p_tree, CMyTreeEntry *p_parent, uint8 *p_array_data)
{
	uint8 *p_data=p_array_data;

	char p_index_buffer[1024];
	char p_value_buffer[1024];

	// Get the type and size
	Script::ESymbolType type=(Script::ESymbolType)*p_data++;
	uint32 size=*p_data++;
	size+=*p_data++<<8; // 2 byte size.

	for (uint32 i=0; i<size; ++i)
	{
		sprintf(p_index_buffer,"[%d]=",i);
		p_value_buffer[0]=0;

		switch (type)
		{
			case ESYMBOLTYPE_INTEGER:
			{
				sprintf(p_value_buffer,"%d",Read4Bytes(p_data).mInt);
				p_data+=4;
				break;
			}		
			case ESYMBOLTYPE_FLOAT:
			{
				sprintf(p_value_buffer,"%f",Read4Bytes(p_data).mFloat);
				p_data+=4;
				break;
			}		
			case ESYMBOLTYPE_NAME:
			{
				sprintf(p_value_buffer,"%s",Script::FindChecksumName(Read4Bytes(p_data).mChecksum));
				p_data+=4;
				break;
			}		
			case ESYMBOLTYPE_STRING:
			{
				sprintf(p_value_buffer,"\"%s\"",(const char*)p_data);
				p_data+=strlen((char*)p_data)+1;
				break;
			}		
			case ESYMBOLTYPE_LOCALSTRING:
			{
				sprintf(p_value_buffer,"'%s'",(const char*)p_data);
				p_data+=strlen((char*)p_data)+1;
				break;
			}		
			case ESYMBOLTYPE_PAIR:
			{
				float x=Read4Bytes(p_data).mFloat;
				p_data+=4;
				float y=Read4Bytes(p_data).mFloat;
				p_data+=4;
				sprintf(p_value_buffer,"(%f,%f)",x,y);
				break;
			}	
			case ESYMBOLTYPE_VECTOR:
			{
				float x=Read4Bytes(p_data).mFloat;
				p_data+=4;
				float y=Read4Bytes(p_data).mFloat;
				p_data+=4;
				float z=Read4Bytes(p_data).mFloat;
				p_data+=4;
				sprintf(p_value_buffer,"(%f,%f,%f)",x,y,z);
				break;
			}	
			case ESYMBOLTYPE_STRUCTURE:
			{
				sprintf(p_value_buffer,"{}");
				break;
			}	
			case ESYMBOLTYPE_ARRAY:
			{
				sprintf(p_value_buffer,"[]");
				break;
			}	
			case ESYMBOLTYPE_NONE:
				// Empty array
				break;
			default:
				ASSERT(0);
				break;
		}

		strcat(p_index_buffer,p_value_buffer);
		CMyTreeEntry *p_new_tree_entry=p_tree->AddEntry(p_index_buffer,p_parent);
		if (type==ESYMBOLTYPE_STRUCTURE)
		{
			p_data=InsertCStructIntoTree(p_tree,p_new_tree_entry,p_data);
		}
		else if (type==ESYMBOLTYPE_ARRAY)
		{
			p_data=InsertCArrayIntoTree(p_tree,p_new_tree_entry,p_data);
		}
	}

	return p_data;
}

uint8 *InsertCStructIntoTree(CMyTreeCtrl *p_tree, CMyTreeEntry *p_parent, uint8 *p_struct_data)
{
	char p_name_buf[1024];
	char p_value_buf[1024];

	// Scan through the buffer adding the components until ESYMBOLTYPE_NONE is reached.
	while (true)
    {
		float x,y,z;
		
		// Get the type and the name checksum.
		Script::ESymbolType type=(Script::ESymbolType)*p_struct_data++;
		if (type==ESYMBOLTYPE_NONE)
		{
			// All done.
			break;
		}

		// Get the name checksum, which may be stored as an index into a table of checksums
		// to save space.
		uint32 name=0;
		if (type&MASK_8_BIT_NAME_LOOKUP)			
		{
			Dbg_MsgAssert(!(type&MASK_16_BIT_NAME_LOOKUP),("Eh? Both lookup-table flags set ?"));
			name=Script::Get8BitIndexedChecksum(*p_struct_data++);
		}
		else if (type&MASK_16_BIT_NAME_LOOKUP)			
		{
			//CArray *p_table=GetArray(0x25231f42/*WriteToBuffer_CompressionLookupTable_16*/);
			//name=p_table->GetChecksum(Read2Bytes(p_struct_data).mUInt);

			// TODO: Add code to do the above lookup.
			name=0x73559503; // Unresolved16BitCompressedName
			p_struct_data+=2;
		}
		else
		{
			// It's not stored as an index, but as the full 4 byte checksum.
			name=Read4Bytes(p_struct_data).mChecksum;
			p_struct_data+=4;
		}

		if (name)
		{
			sprintf(p_name_buf,"%s=",Script::FindChecksumName(name));
		}
		else
		{
			p_name_buf[0]=0;
		}
		p_value_buf[0]=0;

		type=(Script::ESymbolType)( ((uint32)type) & ~(MASK_8_BIT_NAME_LOOKUP | MASK_16_BIT_NAME_LOOKUP) );
			
        switch (type)
        {
        case ESYMBOLTYPE_INTEGER:
			sprintf(p_value_buf,"%d",Read4Bytes(p_struct_data).mInt);
			p_struct_data+=4;
            break;
        case ESYMBOLTYPE_FLOAT:
			sprintf(p_value_buf,"%f",Read4Bytes(p_struct_data).mFloat);
			p_struct_data+=4;
            break;
        case ESYMBOLTYPE_NAME:
			sprintf(p_value_buf,"%s",Script::FindChecksumName(Read4Bytes(p_struct_data).mChecksum));
			p_struct_data+=4;
            break;
		case ESYMBOLTYPE_ZERO_FLOAT:
			sprintf(p_value_buf,"%f",0.0f);
			break;	
		case ESYMBOLTYPE_ZERO_INTEGER:
			sprintf(p_value_buf,"0");
			break;	
		case ESYMBOLTYPE_INTEGER_ONE_BYTE:
			sprintf(p_value_buf,"%d",*(sint8*)p_struct_data++);
			break;
		case ESYMBOLTYPE_UNSIGNED_INTEGER_ONE_BYTE:
			sprintf(p_value_buf,"%d",*p_struct_data++);
			break;
		case ESYMBOLTYPE_INTEGER_TWO_BYTES:
			sprintf(p_value_buf,"%d",Read2Bytes(p_struct_data).mInt);
			p_struct_data+=2;
			break;	
		case ESYMBOLTYPE_UNSIGNED_INTEGER_TWO_BYTES:
			sprintf(p_value_buf,"%d",Read2Bytes(p_struct_data).mUInt);
			p_struct_data+=2;
			break;	
        case ESYMBOLTYPE_STRING:
			sprintf(p_value_buf,"\"%s\"",(char *)p_struct_data);
			p_struct_data+=strlen((char *)p_struct_data);
			++p_struct_data; // Skip over the terminator too.
			break;
        case ESYMBOLTYPE_LOCALSTRING:
			sprintf(p_value_buf,"'%s'",(char *)p_struct_data);
			p_struct_data+=strlen((char *)p_struct_data);
			++p_struct_data; // Skip over the terminator too.
            break;
        case ESYMBOLTYPE_PAIR:
			x=Read4Bytes(p_struct_data).mFloat;
			p_struct_data+=4;
			y=Read4Bytes(p_struct_data).mFloat;
			p_struct_data+=4;
			sprintf(p_value_buf,"(%f,%f)",x,y);
            break;
        case ESYMBOLTYPE_VECTOR:
			x=Read4Bytes(p_struct_data).mFloat;
			p_struct_data+=4;
			y=Read4Bytes(p_struct_data).mFloat;
			p_struct_data+=4;
			z=Read4Bytes(p_struct_data).mFloat;
			p_struct_data+=4;
			sprintf(p_value_buf,"(%f,%f,%f)",x,y,z);
            break;
		case ESYMBOLTYPE_STRUCTURE:
			sprintf(p_value_buf,"{}");
			break;	
		case ESYMBOLTYPE_ARRAY:
			sprintf(p_value_buf,"[]");
			break;	
        default:
			ASSERT(0);
            break;
        }

		strcat(p_name_buf,p_value_buf);

		// If it is a CPU time value then stick a microseconds units indication on the end.
		if (name==0xac0d53de/*CPUTime*/)
		{
			strcat(p_name_buf,"s");
		}

		// If it is a CPU time value and it is a member of a substructure, then
		// instead of adding it as a new entry put the time value after the parent
		// structure's name.
		// This is to make the time values visible at a glance rather than having
		// to open each substructure to see it.
		if (name==0xac0d53de/*CPUTime*/ && p_parent)
		{
			strcat(p_parent->mpText,"   ");
			strcat(p_parent->mpText,p_value_buf);
			strcat(p_parent->mpText,"s");
		}
		else
		{
			CMyTreeEntry *p_new_tree_entry=p_tree->AddEntry(p_name_buf,p_parent);
			if (type==ESYMBOLTYPE_STRUCTURE)
			{
				p_struct_data=InsertCStructIntoTree(p_tree,p_new_tree_entry,p_struct_data);
			}
			else if (type==ESYMBOLTYPE_ARRAY)
			{
				p_struct_data=InsertCArrayIntoTree(p_tree,p_new_tree_entry,p_struct_data);
			}
		}
    }

	return p_struct_data;
}

const char *GetObjectTypeName(int type)
{
	static char sp_foo[100];

	switch (type)
	{
		case SKATE_TYPE_UNDEFINED:
			return "Undefined (0)";
		case SKATE_TYPE_SKATER: 			
			return "Skater";
		case SKATE_TYPE_PED:
			return "Pedestrian";			
		case SKATE_TYPE_CAR:
			return "Car";			
		case SKATE_TYPE_GAME_OBJ:			
			return "Game Object";			
		case SKATE_TYPE_BOUNCY_OBJ:		
			return "Bouncy Object";			
		case SKATE_TYPE_CASSETTE:			
			return "Cassette";			
		case SKATE_TYPE_ANIMATING_OBJECT:	
			return "Animating Object";			
		case SKATE_TYPE_CROWN:	
			return "Crown";			
		case SKATE_TYPE_PARTICLE:			
			return "Particle";			
		case SKATE_TYPE_REPLAY_DUMMY:		
			return "Replay Dummy";			
		default:
			sprintf(sp_foo,"Unknown (0x%08x)",type);
			return sp_foo;
	}
}

const char *GetWaitTypeName(Script::EWaitType waitType)
{
	static char sp_foo[100];

	switch (waitType)
	{
		case Script::WAIT_TYPE_NONE:
			return "None";
		case Script::WAIT_TYPE_COUNTER:
			return "Counter";
		case Script::WAIT_TYPE_TIMER:
			return "Timer";
		case Script::WAIT_TYPE_BLOCKED:
			return "Blocked";
		case Script::WAIT_TYPE_OBJECT_MOVE:
			return "Object Move";
		case Script::WAIT_TYPE_OBJECT_ANIM_FINISHED:
			return "Object Anim Finished";
		case Script::WAIT_TYPE_OBJECT_JUMP_FINISHED:
			return "Object Jump Finished";
		case Script::WAIT_TYPE_OBJECT_ROTATE:
			return "Object Rotate";
		case Script::WAIT_TYPE_STREAM_FINISHED:
			return "Stream Finished";
		case Script::WAIT_TYPE_ONE_PER_FRAME:
			return "One per frame";
		default:
			sprintf(sp_foo,"Unknown (%d)",waitType);
			return sp_foo;
	}
}

const char *GetSingleStepModeName(Script::ESingleStepMode mode)
{
	static char sp_foo[100];

	switch (mode)
	{
		case Script::OFF:
			return "OFF";
		case Script::WAITING:
			return "WAITING";
		case Script::STEP_INTO:
			return "STEP_INTO";
		case Script::STEP_OVER:
			return "STEP_OVER";
		default:
			sprintf(sp_foo,"Unknown (%d)",mode);
			return sp_foo;
	}
}

const char *GenerateFullQFileName(const char *p_name)
{
	static char p_full_source_file_name[1024];

	char *p_project_root=getenv("PROJ_ROOT");
	if (p_project_root)
	{
		bool got_root_already=true;
		const char *p_a=p_project_root;
		const char *p_b=p_name;
		while (*p_a)
		{
			if (tolower(*p_a)!=tolower(*p_b) || *p_b==0)
			{
				got_root_already=false;
				break;
			}
			++p_a;
			++p_b;
		}
		
		if (got_root_already)
		{
			strcpy(p_full_source_file_name,p_name);
		}
		else
		{
			sprintf(p_full_source_file_name,"%s\\data\\%s",p_project_root,p_name);
		}

		// Remove the 'b' on the end so that the extension is just .q
		p_full_source_file_name[strlen(p_full_source_file_name)-1]=0;

		// TODO: Decide whether it should actually be a qn file ?

		return p_full_source_file_name;
	}
	else
	{
		return NULL;
	}
}

// Translates a keypress to a window message.
// This is used in the OnKeyDown handler for various things, eg the CWatchListTree class.
// It is needed because if an element within a document window is selected, then
// pressing F5 will call the OnKeyDown handler for that element, but not for the
// parent window.
// By using this function, the elements can have their OnKeyDown call this for their
// parent (using GetParent() for p_window), thus making F5 take effect for the parent
// even if a child is selected.
// The parent window then also uses this function to send itself a message in its OnKeyDown
// function, by passing 'this' for p_window.
void ConvertKeyPressToMessage(UINT nChar, CWnd *p_window)
{
	switch (nChar)
	{
		case 112: // F1
			p_window->SendMessage(WM_DEBUGGER_HELP);
			break;
		case 116: // F5
			p_window->SendMessage(WM_DEBUGGER_REFRESH);
			break;
		case 45: // Insert
			p_window->SendMessage(WM_DEBUGGER_INSERT);
			break;
		case 46: // Delete
			p_window->SendMessage(WM_DEBUGGER_DELETE);
			break;
		default:
			break;
	}
}

uint32 ConvertHexStringToUint32(const char *p_hex)
{
	if (p_hex[0]=='0' && p_hex[1]=='x')
	{
		p_hex+=2;
		uint32 val=0;
		int len=strlen(p_hex);
		for (int i=0; i<len; ++i)
		{
			val<<=4;
			if (*p_hex>='0' && *p_hex<='9')
			{
				val+=*p_hex-'0';
			}
			else if (*p_hex>='a' && *p_hex<='f')
			{
				val+=*p_hex-'a'+10;
			}
			else
			{
				val+=*p_hex-'A'+10;
			}
			++p_hex;
		}
		return val;
	}

	return 0;
}

// Basically the same as GenerateCRCFromString except it notices if the string
// is preceded by 0x and returns the hex value following the 0x if it is.
uint32 ConvertNameToChecksum(const char *p_name)
{
	if (p_name[0]=='0' && p_name[1]=='x')
	{
		// Sometimes a name might be a hex value, for instance when a checksum cannot
		// be resolved due to the latest checksum names not having been downloaded.
		// Also the skater's id is always 0x00000000

		// Convert the hex string to a uint32 value.
		return ConvertHexStringToUint32(p_name);
	}
	else
	{
		// Get the checksum from the name.
		return Crc::GenerateCRCFromString(p_name);
	}
}

void RequestObjectInfo(const char *p_object_name)
{
	uint32 object_id=ConvertNameToChecksum(p_object_name);

	// Send a request for the object's info to the game.
	// If the game does find the object, it will send a packet of object info,
	// which will then bring up an object window when received.
	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_SEND_COMPOSITE_OBJECT_INFO;
	msg.m_Length=4;
	msg.m_Data=&object_id;
	gClient->EnqueueMessageToServer(&msg);
}

void RequestScriptGlobalInfo(const char *p_name)
{
	uint32 id=ConvertNameToChecksum(p_name);

	// Send a request for the script global's info to the game.
	// If the game does find the global, it will send a packet of info,
	// which will then bring up a script-global window when received.
	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_SEND_SCRIPT_GLOBAL_INFO;
	msg.m_Length=4;
	msg.m_Data=&id;
	gClient->EnqueueMessageToServer(&msg);
}

void SetBreakpointOnScript(uint32 script_name)
{
	uint32 p_data[3];
	p_data[0]=true;
	p_data[1]=Dbg::CHECKSUM_IS_SCRIPT_NAME;
	p_data[2]=script_name;

	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_WATCH_SCRIPT;
	msg.m_Length=sizeof(p_data);
	msg.m_Data=p_data;
	gClient->EnqueueMessageToServer(&msg);
}

// This gets called when a tree entry is double-clicked.
// It parses the text to see whether it is referring to a script or an object, and
// brings up the appropriate window.
void ParseTreeEntryText(CString text)
{
	// Check whether the text is specifying a CScript instance.
	if (text.Left(14).CompareNoCase("m_unique_id=0x")==0)
	{
		// It is, so call WatchScript, which will request the script info
		// from the game code and hence bring up a script window when the
		// info is received.
		uint32 script_id=ConvertHexStringToUint32(text.GetString()+12);
		WatchScript(script_id);
		return;
	}

	// Now check whether the text is referring to an object.
	if (text.Left(7).CompareNoCase("object=")==0)
	{
		// If preceded by object= then the text on the rhs of the = is the object name.
		RequestObjectInfo(text.GetString()+7);
		return;
	}

	// Check if is the skater the way it is listed in the composite object list window
	if (text.Left(9).CompareNoCase("skater={}")==0)
	{
		RequestObjectInfo("0x00000000");
		return;
	}

	int equals_position=text.Find("=");
	// equals_position will be -1 if there is no '=' sign in the string, so after
	// incrementing it will be 0, hence pointing to the start of the string.
	// If there is an '=' sign, after incrementing it will be pointing to whatever
	// follows the =
	++equals_position;

	const char *p_name=text.GetString()+equals_position;

	// If the = is followed by {} then a cpu time value, which will end in s, then
	// it might be the the name preceding the = is a composite object name such as
	// in the list of composite objects.
	if (equals_position && text.Right(2).CompareNoCase("s")==0)
	{
		RequestObjectInfo(text.Left(equals_position-1));
		return;
	}

	// Try requesting object info for the name.
	RequestObjectInfo(p_name);

	// Maybe it is the name of some script global, so request global info for it too.
	RequestScriptGlobalInfo(p_name);
}

void BringUpHelpFile(const char *p_name)
{
	char p_foo[1024];
	sprintf(p_foo,"c:\\NxDocs\\Tools\\Script_Debugger\\%s.mht",p_name);

	ShellExecute(gHWnd,"open","explorer.exe",p_foo,"",SW_SHOW );
}

