#include "FuncEnter.h"

/*****************************************************************************
 *<
	FILE: pivot.cpp

	DESCRIPTION:  Snaps to pivots

	CREATED BY: John Hutchinson		

	HISTORY: created 12/12/96

 *>	Copyright (c) 1994, All Rights Reserved.
 *****************************************************************************/

#include "link/linkUI.h"

//master include file for osnaps
#include "osnapapi.h"

#include "pivotdata.h"
#include "resource.h"

#define SUBCOUNT 2
#define NOSUB -1
#define PIV_SUB 0
#define BBOX_SUB 1

typedef Tab<Point3> Point3Tab;

extern TCHAR *GetString(int id);

#define PIVOT_OSNAP_CLASS_ID Class_ID(0x24960fd5, 0x343a4913)

class PivotSnap : public Osnap {

private:
	OsnapMarker markerdata[SUBCOUNT];
	TSTR name[SUBCOUNT];
	HBITMAP tools;
	HBITMAP masks;

public:

	PivotSnap();//constructor
	virtual ~PivotSnap();

	virtual int numsubs(){ FUNC_ENTER("PivotSnap::numsubs"); return SUBCOUNT;} //the number of subsnaps this guy has
	virtual TSTR *snapname(int index); // the snaps name to be displayed in the UI
	virtual boolean ValidInput(SClass_ID scid, Class_ID cid);// supports anything 
	Class_ID ClassID() { FUNC_ENTER("PivotSnap::ClassID");  return PIVOT_OSNAP_CLASS_ID; }

	virtual OsnapMarker *GetMarker(int index){ FUNC_ENTER("PivotSnap::GetMarker"); return &(markerdata[index]);} 
	virtual HBITMAP getTools(){ FUNC_ENTER("PivotSnap::getTools"); return tools;} 
	virtual HBITMAP getMasks(){ FUNC_ENTER("PivotSnap::getMasks"); return masks;} 
	virtual WORD HiliteMode(){ FUNC_ENTER("PivotSnap::HiliteMode"); return HILITE_BOX;}
	virtual WORD AccelKey(int index); //virtual key codes
	virtual void Snap(Object* pobj, IPoint2 *p, TimeValue t);
};



TSTR *PivotSnap::snapname(int index){ FUNC_ENTER("PivotSnap::snapname"); 
	return &name[index];
}


WORD PivotSnap::AccelKey(int index){ FUNC_ENTER("PivotSnap::AccelKey"); 
	switch (index){
	case PIV_SUB:
		return 0x4F;
		break;
	case BBOX_SUB:
		return 0x42;
		break;
	default:
		return 0;
	}
}


PivotSnap::PivotSnap()
{ FUNC_ENTER("PivotSnap::PivotSnap"); 
	tools = LoadBitmap(hInstance, MAKEINTRESOURCE(IDB_ICONS));
	masks = LoadBitmap(hInstance, MAKEINTRESOURCE(IDB_MASK));
	name[PIV_SUB]= TSTR(GetString(IDS_PIVOT));
	name[BBOX_SUB]= TSTR(GetString(IDS_BBOX));
	markerdata[0]=OsnapMarker(5,mark0verts,mark0es);
	markerdata[1]=OsnapMarker(13,mark1verts,mark1es);
};

PivotSnap::~PivotSnap()
{ FUNC_ENTER("PivotSnap::~PivotSnap"); 
	DeleteObject(tools);
	DeleteObject(masks);
}

boolean PivotSnap::ValidInput(SClass_ID scid, Class_ID cid){ FUNC_ENTER("PivotSnap::ValidInput"); 

	//JH 6/17/99
	//The implementation of the snap manager assumes that any nodes which record hits
	//will still be around when the scenetraversal ends. With the introduction of autogrid
	//this assumption is clearly wrong since the temporary grid object comes and goes.
	//The short term fix is to avoid snapping to these grid objects.
	if(theman->GetNode() == GetCOREInterface()->GetActiveGrid())
		return FALSE;

	if (cid==vLINK_OBJ_CLASS_ID)	// Don't snap to the link objects
		return FALSE;

	boolean c_ok = FALSE, sc_ok = FALSE;
	sc_ok |= (scid == GEOMOBJECT_CLASS_ID)? TRUE : FALSE;
	sc_ok |= (scid == SHAPE_CLASS_ID)? TRUE : FALSE;
	sc_ok |= (scid == CAMERA_CLASS_ID)? TRUE : FALSE;
	sc_ok |= (scid == LIGHT_CLASS_ID)? TRUE : FALSE;
	sc_ok |= (scid == HELPER_CLASS_ID)? TRUE : FALSE;
//	c_ok |= (cid == Class_ID(0xe44f10b3,0))? TRUE : FALSE; 
	return sc_ok;
}

static BOOL EssentiallyEmpty(Box3 &box)
{ FUNC_ENTER("EssentiallyEmpty"); 
	Point3 p2(0.01f,0.01f,0.01f);
	Point3 p1(-0.01f,-0.01f,-0.01f);
	if(box.pmin == p1 && box.pmax == p2)
		return TRUE;
	return FALSE;
}

void PivotSnap::Snap(Object* pobj, IPoint2 *p, TimeValue t)
{ FUNC_ENTER("PivotSnap::Snap"); 	
	// This snap computes the bounding box points of a node as 
	// well as the pivot point

	//local copy of the cursor position
	Point2 fp = Point2((float)p->x, (float)p->y);

	//In this snap mode we actually need to get a pointer to the node so that we 
	// can check for WSM's and compute the pivot point
	INode *inode = theman->GetNode();
	Matrix3 atm; //This will hold the nodes tm before WSMs

	//See if this guys has any spacewarps applied
	BOOL wsm = (BOOL) inode->GetProperty(PROPID_HAS_WSM);

	//If it does then we'll need to get a meaningful tm as follows
	if(wsm)
		atm = inode->GetObjTMBeforeWSM(t);

	//get the node's bounding box
	Box3 box;
	box.Init();
	pobj->GetDeformBBox(t, box, NULL );

	if(EssentiallyEmpty(box))
		pobj->GetLocalBoundBox(t, inode, theman->GetVpt() , box);

		//We need a hitmesh which shows the bounding box of the node
	//This automatic variable gets passed to the hitmesh copy constructor
	// in every case
	HitMesh thehitmesh, *phitmesh;
	thehitmesh.setNumVerts(8);
	for(int jj = 0;jj<8;++jj)
		thehitmesh.setVert(jj,box[jj]);


	BOOL got_one= FALSE;

	//Compute all the hit point candidates
	if(	GetActive(PIV_SUB))
	{
		got_one = FALSE;
		Point3 *pivpt;

		if(wsm)
			pivpt = new Point3(atm.GetTrans() - inode->GetObjOffsetPos());
		else
			pivpt = new Point3(-inode->GetObjOffsetPos());

		//Make a hitmesh
		phitmesh = new HitMesh(thehitmesh);

		//now register a hit with the osnap manager
		theman->RecordHit(new OsnapHit(*pivpt, this, PIV_SUB, phitmesh));
	}

	if(	GetActive(BBOX_SUB))
	{

		//set up our highlight mesh
		for(int ii = 0;ii<8;++ii)
		{
			phitmesh = new HitMesh(thehitmesh);

			theman->RecordHit(new OsnapHit(box[ii], this, BBOX_SUB, phitmesh));
		}
	}


};


//static PivotSnap thePivotSnap;

//===========================================================================\
// | The Class Descriptor
//===========================================================================/

class OsnapClassDesc:public ClassDesc {
	public:
	// The IsPublic() method should return TRUE if the plug-in can be picked
	// and assigned by the user. Some plug-ins may be used privately by other
	// plug-ins implemented in the same DLL and should not appear in lists for
	// user to choose from, so these plug-ins would return FALSE.
	int 			IsPublic() { FUNC_ENTER("OsnapClassDesc::IsPublic");  return 0; }
	// This is the method that actually creates a new instance of
	// a plug-in class.  By having the system call this method,
	// the plug-in may use any memory manager it wishes to 
	// allocate its objects.  The system calls the correspoding 
	// DeleteThis() method of the plug-in to free the memory.  Our 
	// implementations use 'new' and 'delete'.
	void *			Create(BOOL loading = FALSE) { FUNC_ENTER("OsnapClassDesc::Create"); return new PivotSnap();}
//	void *			Create(OsnapManager *pman) { return new PivotSnap(pman); }
	// This is used for debugging purposes to give the class a 
	// displayable name.  It is also the name that appears on the button
	// in the MAX user interface.
	const TCHAR *	ClassName() { FUNC_ENTER("OsnapClassDesc::ClassName");  return _T("PivotSnap"); }
	// The system calls this method at startup to determine the type of object
	// this is.  In our case, we're a geometric object so we return 
	// GEOMOBJECT_CLASS_ID.  The possible options are defined in PLUGAPI.H
	SClass_ID		SuperClassID() { FUNC_ENTER("OsnapClassDesc::SuperClassID");  return OSNAP_CLASS_ID; }
	// The system calls this method to retrieve the unique
	// class id for this object.
	Class_ID		ClassID() { FUNC_ENTER("OsnapClassDesc::ClassID");  
		return PIVOT_OSNAP_CLASS_ID; }
	// The category is selected
	// in the bottom most drop down list in the create branch.
	// If this is set to be an exiting category (i.e. "Primatives", ...) then
	// the plug-in will appear in that category. If the category doesn't
	// yet exists then it is created.  We use the new How To category for
	// all the example plug-ins in the How To sections.
	const TCHAR* 	Category() { FUNC_ENTER("OsnapClassDesc::Category");  return _T(""); }
	};

// Declare a static instance of the class descriptor.
static OsnapClassDesc sampDesc;
// This function returns the address of the descriptor.  We call it from 
// the LibClassDesc() function, which is called by the system when loading
// the DLLs at startup.
ClassDesc* GetPivotDesc() { FUNC_ENTER("GetPivotDesc");  return &sampDesc; }
