/*
	SymbolDebug.cpp
	This file is responsible for providing debug facilities that make use of
	debug symbols in PDB files generated by the Visual C++ compiler.
	Uncertain if imagehlp maintains all the same compatibility for COFF info

	IMPORTANT:  You MUST have at least version 6.1 of DbgHelp.dll (tested on 6.1.17.1)

	8-13-03  aml
*/

#include <string.h>
#include <stdio.h>
#include <windows.h>
#include <imagehlp.h>
#include <conio.h>
#include "SymbolDebug.h"
#include "../../DebugLink/DebugLink.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

//#define DUMP_SYMBOL_DEBUG_OUTPUT		// If defined program flow is dumped to debug panel
//#define POST_DEBUG_STATE				// If defined program debug state is posted so other apps can view it realtime

HANDLE ghProcess = NULL;

#define MAX_DEPTH 512

void * Stack[MAX_DEPTH * 2];    // NOT multi-thread safe
int    SP          = 0;
int    depth       = 0;

DebugState gcurDebugState;

typedef struct
{
    const void  * function;     // function address
    const char * parameter;    // formatting string for parameter decoding, '?' for class method
    const char * returnval;    // formatting string for return value decoding,
}   Signature;

static IMAGEHLP_SYMBOL* gpSymbol = NULL;
static SYMBOL_INFO*     gpSymbolInfo = NULL;
static bool gbSymbolsInitialized = false;

bool SymbolsActive()
{ FUNC_ENTER("SymbolsActive"); 
	return gbSymbolsInitialized;
}

void Expect(Signature * pSig);  // expecting a call to record its address in its signature

void _stdcall ExitFunc(unsigned * pStack)
{ FUNC_ENTER("ExitFunc"); 

#ifdef SYMBOLS_ENABLED
	if (!gbSymbolsInitialized)
	{
		depth --;
		SP --;
		pStack[0] = (unsigned) Stack[SP]; // change return address to point to original caller
		return;
	}

	IMAGEHLP_LINE line;
	line.SizeOfStruct = sizeof(IMAGEHLP_LINE);
	line.Key          = NULL;
	line.LineNumber   = 0;
	line.FileName     = NULL;
	line.Address      = 0;

	DWORD displace = 0;

	// Determine the actual line and file in source code that this occurs in	
	if (!SymGetLineFromAddr(ghProcess, (DWORD)Stack[SP-1], &displace, &line))
	{
		line.FileName   = "<Unknown File>";
		line.LineNumber = 0;
	}

	DWORD64 displace64 = 0;

	if (!SymFromAddr(ghProcess, (DWORD)Stack[SP-1], &displace64, gpSymbolInfo))
	{
		strcpy(gpSymbolInfo->Name, "<Unknown>");
	}

	// Store this to the current debug state
	if (gpSymbolInfo)
	{
		strcpy(gcurDebugState.lastFunction, gpSymbolInfo->Name);
	}
	else
		strcpy(gcurDebugState.lastFunction, "<unresolved>");

	if (line.FileName)
		strcpy(gcurDebugState.lastFile, line.FileName);
	else
		strcpy(gcurDebugState.lastFile, "<Unknown File>");

	gcurDebugState.lastLine = line.LineNumber;
#ifdef POST_DEBUG_STATE
	PostDebugState(&gcurDebugState);
#endif

	depth --;

#ifdef DUMP_SYMBOL_DEBUG_OUTPUT
    char        temp[260];
	char        outstr[256];
	memset(temp, ' ', 260);

	sprintf(outstr, "%s(%i) : ", line.FileName, line.LineNumber);

	wsprintf(temp+depth*2, "returns 0x%x", pStack[0]);
	strcat(outstr, temp);

    OutputDebugString(outstr);
    OutputDebugString("\n");
#endif

    SP --;
    pStack[0] = (unsigned) Stack[SP]; // change return address to point to original caller
#endif

}

void _stdcall EnterFunc0(unsigned * pStack);
void _stdcall EnterFunc(unsigned * pStack);

#ifdef SYMBOLS_ENABLED
extern "C" __declspec(naked) void __cdecl _pexit(void)
{ FUNC_ENTER("_pexit"); 
    _asm 
    { 
        push   eax        // function return value and placehold for return to original caller
        pushad            // save all general purpose registers 
        mov    eax, esp   // current stack pointer  
        add    eax, 32    // stack pointer before pushad
        push   eax        // push pointer to where EAX is saved as parameter to ExitFunc

        call   ExitFunc + 5

        popad             // restore general registers
        ret               // return to original caller  
    }
}

extern "C" __declspec(naked) void __cdecl _penter(void)
{ 
    _asm 
    { 
        pushad              // save all general purpose registers
        mov    eax, esp     // current stack pointer
        add    eax, 32      // stack pointer before pushad
        push   eax          // push pointer to return address as parameter to EnterFunc0
        
        //call   EnterFunc0 + 5
		call   EnterFunc + 5

        popad               // restore general purpose registers
        ret                 // start executing original function
    }
} 
#endif

void _stdcall EnterFunc(unsigned * pStack)
{ FUNC_ENTER("EnterFunc"); 

#ifdef SYMBOLS_ENABLED
	if (!gbSymbolsInitialized)
		return;

    void      * pCaller;

    // pStack[-8] : EDI     general purpose registers saved on stack
    // pStack[-7] : ESI
    // pStack[-6] : EBP
    // pStack[-5] : ESP
    // pStack[-4] : EBX
    // pStack[-3] : EDX     usually 'this' pointer for an object
    // pStack[-2] : ECX
    // pStack[-1] : EAX

    // pStack[0]  : return address to the caller of _penter
    // pStack[0]-5: first byte of caller function, for example WinMain
    // pStack[1]  : return address to the caller function
    // pStack[2]  : parameter1
    // pStack[3]  : parameter2
    // pStack[4]  : parameter3
    // ...
    // pStack[N+1]: parameterN

    pCaller = (void *) (pStack[0] - 5); // the instruction for calling _penter is 5 bytes long

	// Look up the symbolic data for the function that we will return to so we can log it
	DWORD displace = 0;

	IMAGEHLP_LINE line;
	line.SizeOfStruct = sizeof(IMAGEHLP_LINE);
	line.Key          = NULL;
	line.LineNumber   = 0;
	line.FileName     = NULL;
	line.Address      = 0;

	if (!SymGetSymFromAddr(ghProcess, (DWORD)pCaller, &displace, gpSymbol))
	{
#ifdef DUMP_SYMBOL_DEBUG_OUTPUT		
		char         temp[260];
		char         strout[1024];
		memset(temp, ' ', 260);

		wsprintf(temp+depth*2, "Entered function but <symbol unknown>");
		sprintf(strout, "<Unknown File>(0) : %s\n", temp);	

		OutputDebugString(strout);
#endif
	}
	else
	{
		// Determine the actual line and file in source code that this occurs in	
		if (!SymGetLineFromAddr(ghProcess, (DWORD)gpSymbol->Address, &displace, &line))
		{
			line.FileName   = "<Unknown File>";
			line.LineNumber = 0;
		}

#ifdef DUMP_SYMBOL_DEBUG_OUTPUT
		char         temp[260];
		char         strout[1024];
		memset(temp, ' ', 260);

		
		wsprintf(temp+depth*2, "Entered function %s", gpSymbol->Name);
		sprintf(strout, "%s(%i) : %s\n", line.FileName, line.LineNumber, temp);

		OutputDebugString(strout);
#endif
	}

	// Store this to the current debug state
	if (gpSymbol)
	{
		strcpy(gcurDebugState.lastFunction, gpSymbol->Name);
	}
	else
		strcpy(gcurDebugState.lastFunction, "<not initailzed yet>");

	if (line.FileName)
		strcpy(gcurDebugState.lastFile, line.FileName);
	else
		strcpy(gcurDebugState.lastFile, "<Unknown File>");

	gcurDebugState.lastLine = line.LineNumber;
#ifdef POST_DEBUG_STATE
	PostDebugState(&gcurDebugState);
#endif

    Stack[SP++] = (void *) pStack[1]; // save return address to original caller   
    pStack[1]   = (unsigned) _pexit;  // HACK stack to link to _pexit  

    depth ++;
#endif
}

void _stdcall EnterFunc0(unsigned * pStack)
{ FUNC_ENTER("EnterFunc0"); 
	__asm
	{
		push pStack
		call EnterFunc + 5
	}
}

int main(void);

BOOL CALLBACK enumSymbols(LPSTR SymbolName, ULONG SymbolAddress, ULONG SymbolSize, void* pData)
{ FUNC_ENTER("enumSymbols"); 
	printf("\t%s\t%x\t%i\n", SymbolName, SymbolAddress, SymbolSize);
	return TRUE;
}

static DWORD gModuleBase = 0;

BOOL CALLBACK enumModules(LPSTR ModuleName, ULONG BaseOfDll, void* pData)
{ FUNC_ENTER("enumModules"); 
#ifdef SYMBOLS_ENABLED
	printf("%s\t%x\n", ModuleName, BaseOfDll);

	if (gModuleBase == 0)
		gModuleBase = BaseOfDll;

	if (!SymLoadModule(ghProcess, 
		          NULL, 
				  NULL,
				  NULL,
				  BaseOfDll,
				  0))
	{
		printf("Failed to load module\n");
	}

	SymEnumerateSymbols(ghProcess, BaseOfDll, enumSymbols, NULL);
#endif
	return TRUE;
}


BOOL CALLBACK lineCallback(PSRCCODEINFO lineInfo, void* pData)
{ FUNC_ENTER("lineCallback"); 
	printf("Obj: %s  File: %s  Line: %i\n", lineInfo->Obj, lineInfo->FileName, lineInfo->LineNumber);
	return TRUE;
}


void DumpSymbols()
{ FUNC_ENTER("DumpSymbols"); 

#ifdef SYMBOLS_ENABLED
	LPAPI_VERSION version = ImagehlpApiVersion();
	printf("ImageHelp Version: %i.%i (Rev %i)\n", version->MajorVersion, version->MinorVersion, version->Revision);
	while(getch() == 0){}

	SymEnumerateModules(ghProcess, enumModules, NULL);
	SymEnumLines(ghProcess, gModuleBase, NULL, NULL, lineCallback, NULL);
#endif
}

void InitSymbols()
{ FUNC_ENTER("InitSymbols"); 
#ifdef SYMBOLS_ENABLED
	ghProcess = GetCurrentProcess();

	OutputDebugString("Loading Debug Symbols...\n");
	if (!SymInitialize(ghProcess, NULL, TRUE))
	{
#ifdef DUMP_SYMBOL_DEBUG_OUTPUT
		OutputDebugString("Failed to locate symbolic debug info for application.\n");
#endif
	}
	else
	{
#ifdef ENUM_SYMBOLS
		DumpSymbols();	
#endif

		// Initialize the global symbol structure that will be used to lookup symbol data
		gpSymbol = (IMAGEHLP_SYMBOL*)malloc(sizeof(IMAGEHLP_SYMBOL) + 256);
		gpSymbol->SizeOfStruct = sizeof(IMAGEHLP_SYMBOL);
		gpSymbol->Address = 0;
		gpSymbol->Size = 0;
		gpSymbol->Flags = 0;
		gpSymbol->MaxNameLength = 256;

		gpSymbolInfo = (SYMBOL_INFO*)malloc(sizeof(SYMBOL_INFO) + 256);
		gpSymbolInfo->SizeOfStruct = sizeof(SYMBOL_INFO);
		gpSymbolInfo->TypeIndex    = 0;
		gpSymbolInfo->Reserved[0]  = 0;
		gpSymbolInfo->Reserved[1]  = 0;
		gpSymbolInfo->Size         = 0;
		gpSymbolInfo->Reserved[0]  = 0;
		gpSymbolInfo->ModBase      = 0;		// Filled in on ExitFunc
		gpSymbolInfo->Flags        = IMAGEHLP_SYMBOL_FUNCTION;
		gpSymbolInfo->Value        = 0;
		gpSymbolInfo->Address      = 0;		// Filled in on ExitFunc
		gpSymbolInfo->Register     = 0;		// ?
		gpSymbolInfo->Scope        = 0;		// For DIA info  (unused for our purposes)
		gpSymbolInfo->Tag          = 0;		// ? PDB Classification
		gpSymbolInfo->NameLen      = 0;
		gpSymbolInfo->MaxNameLen   = 256;
		gpSymbolInfo->Name[0]      = 0;

		gbSymbolsInitialized = true;
		//SetDebugInit(true);
		//DumpSymbols();
	}
#endif

}

int ShutdownSymbols()
{ FUNC_ENTER("ShutdownSymbols"); 
#ifdef SYMBOLS_ENABLED
	gbSymbolsInitialized = false;
	//SetDebugInit(false);

	SymCleanup(ghProcess);
	free(gpSymbol);
	free(gpSymbolInfo);
	OutputDebugString("Debug Symbols have been flushed.\n");
#endif

	return 0;
}
