#include "FuncEnter.h"

/*
	PropBufGen.cpp
	Property Buffer Generator
	2-13-01
*/

#include "../Trigger/Trigger.h"
#include "PropBufGen.h"
#include "PropFlags.h"
#include "ParseFuncs.h"

// Modes for template definition metacommands
#define TEMPLATEMODE_PRINT   "print"
#define TEMPLATEMODE_PRINTIF "printif"

#define DEBUGDUMP(filename,x) { FILE* fp = fopen(filename,"w"); fprintf(fp, "%s\n", (char*)x); fclose(fp); }

PropBufGen::PropBufGen()
{ FUNC_ENTER("PropBufGen::PropBufGen"); 
	
}

PropBufGen::~PropBufGen()
{ FUNC_ENTER("PropBufGen::~PropBufGen"); 

}

CStr PropBufGen::GetClassProp(INode* node)
{ FUNC_ENTER("PropBufGen::GetClassProp"); 
	CStr ClassProp;

	if (!node->GetUserPropString("Class",ClassProp))
		node->GetUserPropString("class",ClassProp);

	if (ClassProp==CStr(""))
	{
		Object* obj=node->EvalWorldState(0).obj;

		if (!obj)
			return CStr("");

		if(obj->ClassID()==vTRIGGER_CLASS_ID)
		{
			ClassProp=TRIGGER_CLASS;
		}
		else if (obj->ClassID()==Class_ID(SIMPLE_CAM_CLASS_ID,0) ||
			     obj->ClassID()==Class_ID(LOOKAT_CAM_CLASS_ID,0))
		{
			ClassProp=CAMERA_CLASS;
		}
		else
		{
			ClassProp=DEFAULT_CLASS;
		}
	}

	return ClassProp;
}

CStr PropBufGen::GetTypeProp(INode* node)
{ FUNC_ENTER("PropBufGen::GetTypeProp"); 
	CStr TypeProp;

	if (!node->GetUserPropString("Type",TypeProp))
	{
		if (!node->GetUserPropString("type",TypeProp))
		{
			CStr ClassProp = GetClassProp(node);
			
			// Find the class and use the first defined type
			ConfigClass cclass;
			cclass.name = ClassProp;
			Link<ConfigClass>* link = ConfigDBFind(&cclass);

			if (!link)
				return CStr("");

			if (link->data.types.GetHead() == NULL)
				return CStr("");

			return link->data.types.GetHead()->data.name;
		}
	}
	
	return TypeProp;
}

CStr PropBufGen::BuildPropBuffer(INode* node,
								 CStr strClass,
						         CStr strType,
						         DWORD flags,
						         LinkList<ConfigProp>* props,
						         CStr strCmds,
						         CStr strScript,
						         CStr strCluster,
								 LinkList<ConfigProgram>* programs,
								 LinkList<ConfigScript>* scripts,
								 bool bConvertToDefaultDelim)
{ FUNC_ENTER("PropBufGen::BuildPropBuffer"); 
	// Dump the node out to the property buffer
	CStr applyBuf;
	CStr eol="\r\n";

	// Dump class/type
	if (strClass.Length()==0 && node)
	{
		strClass=GetClassProp(node);
		strType=GetTypeProp(node);
	}

	applyBuf+=CStr("Class = ")+strClass+eol;
	
	if (strType!=CStr(""))
		applyBuf+=CStr("Type = ")+strType+eol;

	// Dump flags
//	if (flags & CCLASS_CREATEDATSTART)
//		applyBuf+=CStr("CreatedAtStart")+eol;
//	
//	if (flags & CCLASS_OCCLUDER)
//		applyBuf+=CStr("Occluder")+eol;
//
//	if (flags & CCLASS_ABSENTINNETGAMES)
//		applyBuf+=CStr("AbsentInNetGames")+eol;
//
//	if (flags & CCLASS_TRICKOBJECT)
//		if (strCluster.Length()>0)
//			applyBuf+=CStr("TrickObject Cluster = ")+strCluster+eol;
//		else
//			applyBuf+=CStr("TrickObject")+eol;

	// Convert properties to default delim where appropriate
	if (bConvertToDefaultDelim)
		ConvertToDefaultDelim(strClass,strType,props);

	// Dump props

	Link<ConfigProp>* lprop=props->GetHead();

	while(lprop)
	{
		applyBuf+=lprop->data.name+CStr(" = ")+lprop->data.value+eol;

		lprop=lprop->next;
	}

	// Dump property extended data
	lprop=props->GetHead();

	while(lprop)
	{
		// Don't add extended property data if it exists within script.ini
		if (GetDynUICmds(strClass,strType,lprop->data.name).Length()==0)
		{
			if (lprop->data.extdata.Length()>0)
			{
				// Don't add eol unless necessary
				if (lprop->data.extdata.Substr(lprop->data.extdata.Length()-eol.Length(),eol.Length())==eol)
					applyBuf+=lprop->data.extdata;
				else
					applyBuf+=lprop->data.extdata+eol;
			}
		}

		lprop=lprop->next;
	}

	// Dump commands
	applyBuf+=strCmds;

	// Dump program
	if (programs &&
		programs->GetSize() > 0)
	{
		Link<ConfigProgram>* curprog = programs->GetHead();

		while(curprog)
		{
			if (curprog->data.scriptContext.Length()>0)
			{
				if (curprog->data.type == ConfigProgram::CFGPROG_VOLATILE)
					applyBuf+=CStr("// @programvolatile | ")+curprog->data.filename+CStr(" | ")+curprog->data.scriptContext+eol;
				else
					applyBuf+=CStr("// @program | ")+curprog->data.filename+CStr(" | ")+curprog->data.scriptContext+eol;
			}
			else
			{
				if (curprog->data.type == ConfigProgram::CFGPROG_VOLATILE)
					applyBuf+=CStr("// @programvolatile | ")+curprog->data.filename+eol;
				else
					applyBuf+=CStr("// @program | ")+curprog->data.filename+eol;
			}

			curprog = curprog->next;
		}
	}

	// Dump out script
	applyBuf+=CStr("script")+eol;
	
	if (strScript==CStr(vUNKNOWN))
		strScript="";

	// Don't add eol unless necessary
	if (strScript.Length() > 0)
	{
		if (strScript.Substr(strScript.Length()-eol.Length(),eol.Length())==eol)
			applyBuf+=strScript;
		else
			applyBuf+=strScript+eol;
	}

	applyBuf+=CStr("endscript")+eol;

	// Build the information for the alternate scripts
	if (scripts)
	{	
		Link<ConfigScript>* curscript = scripts->GetHead();

		while(curscript)
		{
			//applyBuf += CStr("altscript ") + curscript->data.name + CStr(" ") + curscript->data.filename + eol;
			applyBuf += CStr("altscript ") + curscript->data.name + eol;
			applyBuf += curscript->data.buffer + eol;

			applyBuf+=CStr("endscript")+eol;
			curscript = curscript->next;
		}
	}

	// Apply the new user properties to the node
	return applyBuf;
}

// Returns true if the buffer already existed
bool PropBufGen::AssignAltBuf(INode* node,CStr bufName,CStr buf)
{ FUNC_ENTER("PropBufGen::AssignAltBuf"); 
	LinkList<ConfigProp> cprops;
	DWORD flags;
	CStr  unkBuf;
	CStr  clusterBuf;
	LinkList<ConfigProgram> programs;
	CStr  script;
	LinkList<ConfigScript> scripts;
	CStr  propBuffer;

	node->GetUserPropBuffer(propBuffer);

	CStr strClass = GetClassName(propBuffer);
	CStr strType  = GetTypeName(propBuffer);

	script = ParseConfigProps(&cprops,&flags,propBuffer,&unkBuf,&clusterBuf,&programs,&scripts);

	ConfigScript srch;
	srch.name = bufName;
	Link<ConfigScript>* link = scripts.Find(&srch);

	if (!link)
	{
		srch.buffer = buf;
		scripts.Add(&srch);
		PropBufGen::DumpNode(node,strClass,strType,flags,&cprops,unkBuf,script,clusterBuf,&programs,&scripts);
		return false;
	}

	link->data.buffer = buf;
	PropBufGen::DumpNode(node,strClass,strType,flags,&cprops,unkBuf,script,clusterBuf,&programs,&scripts);
	return true;
}

void PropBufGen::RemoveAltBuf(INode* node, CStr bufName)
{ FUNC_ENTER("PropBufGen::RemoveAltBuf"); 
	LinkList<ConfigProp> cprops;
	DWORD flags;
	CStr  unkBuf;
	CStr  clusterBuf;
	LinkList<ConfigProgram> programs;
	CStr  script;
	LinkList<ConfigScript> scripts;
	CStr  propBuffer;

	node->GetUserPropBuffer(propBuffer);

	CStr strClass = GetClassName(propBuffer);
	CStr strType  = GetTypeName(propBuffer);

	script = ParseConfigProps(&cprops,&flags,propBuffer,&unkBuf,&clusterBuf,&programs,&scripts);

	ConfigScript srch;
	srch.name = bufName;
	Link<ConfigScript>* link = scripts.Find(&srch);

	if (link)
		scripts.Remove(link);

	PropBufGen::DumpNode(node,strClass,strType,flags,&cprops,unkBuf,script,clusterBuf,&programs,&scripts);
}

CStr PropBufGen::GetAltBuf(INode* node,CStr bufName)
{ FUNC_ENTER("PropBufGen::GetAltBuf"); 
	CStr propBuffer;
	LinkList<ConfigScript> scripts;

	node->GetUserPropBuffer(propBuffer);

	ParseConfigProps(NULL,NULL,propBuffer,NULL,NULL,NULL,&scripts);

	ConfigScript srch;
	srch.name = bufName;

	Link<ConfigScript>* link = scripts.Find(&srch);

	if (!link)
		return CStr("");

	return link->data.buffer;
}

void PropBufGen::BuildList(LinkList<CStr>* list, CStr buf, bool bShaveEntries)
{ FUNC_ENTER("PropBufGen::BuildList"); 
	int pos = 0;
	int len = buf.Length();

	while (pos < len)
	{
		CStr line = GetRemainLineExact(buf,&pos);

		if (bShaveEntries)
			line = Shave(line);

		list->Add(&line);
	}
}

bool PropBufGen::BuildList(CStr Filename,LinkList<CStr>* list)
{ FUNC_ENTER("PropBufGen::BuildList"); 
	FILE* fp;
	char lineBuf[1024];

	list->Clear();

	fp = fopen(Filename,"r");

	if (!fp)
		return false;

	while(!feof(fp))
	{
		if (fgets(lineBuf,1023,fp))
			list->Add(&Shave(lineBuf));
	}

	fclose(fp);

	return true;
}

CStr PropBufGen::BuildBuf(LinkList<CStr>* list)
{ FUNC_ENTER("PropBufGen::BuildBuf"); 
	CStr buf;
	Link<CStr>* curline = list->GetHead();

	while(curline)
	{
		buf += curline->data + CStr("\n");	
		curline = curline->next;
	}

	return buf;
}

CStr PropBufGen::BuildBufCRLF(LinkList<CStr>* list)
{ FUNC_ENTER("PropBufGen::BuildBufCRLF"); 
	CStr buf;
	Link<CStr>* curline = list->GetHead();

	while(curline)
	{
		buf += curline->data + CStr("\r\n");	
		curline = curline->next;
	}

	return buf;
}

CStr PropBufGen::BuildPartialList(LinkList<CStr>* list,Link<CStr>* start,Link<CStr>* end)
{ FUNC_ENTER("PropBufGen::BuildPartialList"); 
	CStr buf;

	if (!start)
		start = list->GetHead();

	if (!end)
		end = list->GetTail();

	Link<CStr>* curline = start;

	if (!curline)
		return CStr("");

	do
	{
		buf += curline->data + CStr("\n");	
		curline = curline->next;

	} while(curline && curline != end);

	return buf;
}

CStr PropBufGen::GetMetaCmds(CStr buf)
{ FUNC_ENTER("PropBufGen::GetMetaCmds"); 
	int pos = 0;
	CStr strLine;
	CStr strFinalBuf;

	while(pos < buf.Length())
	{
		strLine = GetRemainLinePartial(buf, &pos);

		if (strstr(strLine, "// @"))
			strFinalBuf += strLine + CStr("\r\n");
	}

	return strFinalBuf;
}

CStr PropBufGen::GetUniqueClassTypeRecord(CStr className, CStr typeName)
{ FUNC_ENTER("PropBufGen::GetUniqueClassTypeRecord"); 
	// Find the class we're looking for
	Link<ConfigClass>* curNode = configDB.GetHead();

	while(curNode)
	{
		if (curNode->data.name == className)
		{
			CStr classBuf = curNode->data.strCmds;
			CStr typeBuf  = GetTypeRecord(&curNode->data, typeName, true);
			
			return CombinePropBuffers(classBuf, typeBuf, className, typeName, false);
		}

		curNode = curNode->next;
	}

	return CStr("");
}

CStr PropBufGen::CombinePropBuffers(CStr& buf1, CStr& buf2, CStr& strClass, CStr& strType, bool bConvertToDefaultDelim)
{ FUNC_ENTER("PropBufGen::CombinePropBuffers"); 
	LinkList<ConfigProp>   cprops1, cprops2, cprops3;
	LinkList<ConfigProgram> programs1, programs2, programs3;
	LinkList<ConfigScript>  scripts1, scripts2, scripts3;
	LinkList<CStr>          omitList;
	DWORD flags1, flags2, flags3;
	CStr  unkBuf1, unkBuf2, unkBuf3;
	CStr  cluster1, cluster2, cluster3;
	CStr  strScript1, strScript2, strScript3;
	CStr  tmpBuf1, tmpBuf2;

	tmpBuf1 = buf1;
	tmpBuf2 = buf2;

	// Copy the meta commands from each buffer so all the meta commands
	// are processed on each buffer
	tmpBuf1 += GetMetaCmds(buf2);
	tmpBuf2 += GetMetaCmds(buf1);

	//DEBUGDUMP("c:\\Combine1.txt", tmpBuf1);	
	//DEBUGDUMP("c:\\Combine2.txt", tmpBuf2);

	strScript1 = ParseConfigProps(&cprops1, &flags1, tmpBuf1, &unkBuf1, &cluster1, &programs1, &scripts1);
	strScript2 = ParseConfigProps(&cprops2, &flags2, tmpBuf2, &unkBuf2, &cluster2, &programs2, &scripts2);

	strScript3 = strScript1 + strScript2;

	GetOmissions(strClass, strType, &omitList);

	Link<ConfigProp> *curprop, *fndprop;
	ConfigProp cprop;

	// Build up the new properties list
	// Properties from buf2 take precedence over buf1
	
	// Add buf1 properties
	curprop = cprops1.GetHead();

	while(curprop)
	{
		cprop.name = curprop->data.name;

		// Don't add properties that should be omitted
		if (omitList.Find(&cprop.name))
		{
			curprop = curprop->next;
			continue;
		}

		// Omit class and type fields
		CStr lcName = cprop.name;
		lcName.toLower();

		if (lcName == CStr("class"))
		{
			curprop = curprop->next;
			continue;
		}

		if (lcName == CStr("type"))
		{
			curprop = curprop->next;
			continue;
		}

		fndprop = cprops2.Find(&cprop);

		if (fndprop)
		{
			cprop         = fndprop->data;
			
			if (curprop->data.extdata != fndprop->data.extdata)
				cprop.extdata = curprop->data.extdata + fndprop->data.extdata;
			
			cprops3.Add(&cprop);
		}
		else
		{
			cprops3.Add(&curprop->data);
		}

		curprop = curprop->next;
	}

	// Add buf2 properties
	curprop = cprops2.GetHead();

	while(curprop)
	{
		// Don't add properties that should be omitted
		if (omitList.Find(&cprop.name))
		{
			curprop = curprop->next;
			continue;
		}

		// Omit class and type fields
		cprop.name = curprop->data.name;
		cprop.name.toLower();
		if (cprop.name == CStr("class"))
		{
			curprop = curprop->next;
			continue;
		}

		if (cprop.name == CStr("type"))
		{
			curprop = curprop->next;
			continue;
		}

		cprops3.AddToTailUnique(&curprop->data);
		curprop = curprop->next;
	}

	// Add buf1 programs
	Link<ConfigProgram>* curprog = programs1.GetHead();

	while(curprog)
	{
		programs3.AddToTailUnique(&curprog->data);
		curprog = curprog->next;
	}

	// Add buf2 programs
	curprog = programs2.GetHead();

	while(curprog)
	{
		programs3.AddToTailUnique(&curprog->data);
		curprog = curprog->next;		
	}

	// Add buf1 config scripts
	Link<ConfigScript>* curscript = scripts1.GetHead();

	while(curscript)
	{
		scripts3.AddToTailUnique(&curscript->data);
		curscript = curscript->next;
	}

	// Add buf2 config scripts
	curscript = scripts2.GetHead();

	while(curscript)
	{
		scripts3.AddToTailUnique(&curscript->data);
		curscript = curscript->next;
	}

	flags3  = flags1 | flags2;
	unkBuf3 = unkBuf1 + unkBuf2;

	if (cluster2 == CStr(""))
		cluster3 = cluster1;
	else
		cluster3 = cluster2;


	CStr outbuf;
	outbuf = BuildPropBuffer(NULL,
		                   strClass,
						   strType,
						   flags3,
						   &cprops3,
						   unkBuf3,
						   strScript3,
						   cluster3,
						   &programs3,
						   &scripts3,
						   bConvertToDefaultDelim);

	// Add omissions
	Link<CStr>* curomit = omitList.GetHead();

	while(curomit)
	{
		outbuf += "Omit ";
		outbuf += curomit->data;
		outbuf += "\r\n";

		curomit = curomit->next;
	}

	// Dump property data
	curprop = cprops1.GetHead();

	while(curprop)
	{
		outbuf += curprop->data.extdata;
		outbuf += "\r\n";
		curprop = curprop->next;
	}

	curprop = cprops2.GetHead();

	while(curprop)
	{
		outbuf += curprop->data.extdata;
		outbuf += "\r\n";
		curprop = curprop->next;
	}

	//DEBUGDUMP("c:\\Combine3.txt", outbuf);

	return outbuf;
}

CStr PropBufGen::BuildTemplateBuffer(LinkList<ConfigProp>* cprops, CStr classTypeRec)
{ FUNC_ENTER("PropBufGen::BuildTemplateBuffer"); 
	// Scan through the buffer for @template and @templateopt lines and build a template buffer
	char* loc = strstr(classTypeRec, "// @template | ");
	CStr strTemplateMode;
	CStr strCheckProp;
	CStr strCheckPropValue;
	CStr strTemplateLine;
	CStr strTemplateBuffer;
	char token[256];
	char eol[3];
	eol[0] = 10;
	eol[1] = 13;
	eol[2] = 0;

	while(loc)
	{
		int pos = 0;
		loc+=12;
		strCheckProp = "";
		strCheckPropValue = "";

		GetToken(loc,&pos,token);		// This is undefined
		GetToken(loc,&pos,token);		// This is the template mode
		StripToken(token);
		strTemplateMode = token;

		if (strTemplateMode == CStr(TEMPLATEMODE_PRINTIF))
		{
			GetToken(loc,&pos,token);	// This is the property to check
			StripToken(token);
			strCheckProp = token;

			GetToken(loc,&pos,token);	// This is the value to check for
			StripToken(token);
			strCheckPropValue = token;
		}

		GetToken(loc,&pos,token);		// This is the template string
		StripToken(token);
		strTemplateLine = token;

		// Determine if the template string should be added to the template buffer
		if (strCheckProp == CStr(""))
		{
			// No restrictions template string should be added
			strTemplateBuffer += strTemplateLine + CStr(eol);
		}
		else
		{
			ConfigProp cpropCheck;
			cpropCheck.name = strCheckProp;
			
			Link<ConfigProp>* link = cprops->Find(&cpropCheck);

			if (link)
			{
				if (link->data.value == strCheckPropValue)
					strTemplateBuffer += strTemplateLine + CStr(eol);
			}
		}

		loc = strstr(loc, "// @template | ");
	}

	return strTemplateBuffer;
}

CStr PropBufGen::ProcessTemplateBuffer(LinkList<ConfigProp>* cprops, CStr tempBuf)
{ FUNC_ENTER("PropBufGen::ProcessTemplateBuffer"); 
	Link<ConfigProp>* link = cprops->GetHead();
	CStr buf = tempBuf;

	while(link)
	{
		buf = ReplaceStr(buf, CStr("<") + link->data.name + CStr(">"), link->data.value);
		link = link->next;
	}

	return buf;
}
