#include "FuncEnter.h"

/*
	GwRender.h
	Graphics Window Rendering Functions

	The standard MAX graphics window class doesn't draw much beyond polyline primitives
	this file contains drawing functions for other primitives

	aml - 4-16-03
*/

#include "GwRender.h"
#include <assert.h>

void GetCirclePts(Point3* pts, float radius, Plane2D plane, int nPts)
{ FUNC_ENTER("GetCirclePts"); 
	int    cnt = 0;
	float  incr = TWOPI / nPts;

	for(float i = 0; i < TWOPI; i += incr)
	{
		switch(plane)
		{
		case PLANE_XY:
			pts[cnt].x = cos(i) * radius;
			pts[cnt].y = sin(i) * radius;
			pts[cnt].z = 0.0f;
			break;

		case PLANE_YZ:
			pts[cnt].x = 0.0f;
			pts[cnt].y = cos(i) * radius;
			pts[cnt].z = sin(i) * radius;
			break;

		case PLANE_XZ:
			pts[cnt].x = cos(i) * radius;
			pts[cnt].y = 0.0f;
			pts[cnt].z = sin(i) * radius;
			break;

		default:
			OutputDebugString("GetCirclePts:  Unknown 2D Plane constant.\n");
			assert(0);
		}

		cnt++;
	}
}

void DrawCircle(GraphicsWindow* gw, float radius, Plane2D plane, int nPts)
{ FUNC_ENTER("DrawCircle"); 
	// The graphics window doesn't have a method for drawing circles, only poly lines
	// so we'll need to approximate a circle using polylines
	// Px = cos(p) * r
	// Py = sin(p) * r

	Point3* pts = new Point3[nPts];
	GetCirclePts( pts, radius, plane, nPts );
	gw->polyline( nPts, pts, NULL, NULL, FALSE, NULL );

	delete [] pts;
}

void ApplyPtTransform(Point3* pts, int nPts, Matrix3 mat)
{ FUNC_ENTER("ApplyPtTransform"); 
	for(int i = 0; i < nPts; i++)
		pts[i] = pts[i] * mat;
}

void DrawCircleSphere(GraphicsWindow* gw, float radius, int nPts)
{ FUNC_ENTER("DrawCircleSphere"); 
	// The graphics window doesn't have a method for drawing circles, only poly lines
	// so we'll need to approximate a circle using polylines
	// Px = cos(p) * r
	// Py = sin(p) * r

	Matrix3 mat;
	int     i;
	Point3* pts = new Point3[nPts];

	// Plane XY
	mat.IdentityMatrix();
	GetCirclePts( pts, radius, PLANE_XY, nPts );

	// 8 circles should be drawn each rotated 45 degrees more
	for (i = 0; i < 8; i++)
	{
		mat.RotateX( DegToRad(45.0f) );
		ApplyPtTransform( pts, nPts, mat );
		gw->polyline( nPts, pts, NULL, NULL, FALSE, NULL );
	}

	// Plane YZ
	mat.IdentityMatrix();
	GetCirclePts( pts, radius, PLANE_YZ, nPts );

	// 8 circles should be drawn each rotated 45 degrees more
	for (i = 0; i < 8; i++)
	{
		mat.RotateY( DegToRad(45.0f) );
		ApplyPtTransform( pts, nPts, mat );
		gw->polyline( nPts, pts, NULL, NULL, FALSE, NULL );
	}

	delete [] pts;
}
