#include "FuncEnter.h"

/*
	LightmapGen.cpp
	This class is responsible for generating lightmaps
	aml - 9-18-02
*/

#include "LightmapGen.h"
#include "path.h"
#include "../ImageData.h"
#include "../misc/maxutil.h"

#define MAP_YZ	 0	// X is largest
#define MAP_XZ   1  // Y is largest
#define MAP_XY   2  // Z is largest

extern Interface* gInterface;
extern DWORD WINAPI ProgressFunc(LPVOID arg);

#define LMAP_SIZE 16			// The size of generated lightmaps
								// (Larger is more accurate but requires more VRAM)

LightmapGen::LightmapGen()
{ FUNC_ENTER("LightmapGen::LightmapGen"); 
	curdir = 0;
	curmap = 0;
	sceneFaces = NULL;
	faceLights = NULL;
	maps = NULL;
}

LightmapGen::~LightmapGen()
{ FUNC_ENTER("LightmapGen::~LightmapGen"); 
	if (sceneFaces)
		delete [] sceneFaces;

	if (maps)
		delete [] maps;
}

float LightmapGen::GetAng(Point3 v1, Point3 v2)
{ FUNC_ENTER("LightmapGen::GetAng"); 
	v1 = v1.Normalize();
	v2 = v2.Normalize();

	float dot = DotProd(v1, v2);

	// Clamp extents
	if (dot > 1.0f)
		dot = 1.0f;

	if (dot < -1.0f)
		dot = -1.0f;

	return RadToDeg(acosf(dot));
}

// This function determines if the given point falls within the line segment defined by start and end
bool LightmapGen::PtInLine(Point3 start, Point3 end, Point3 pt)
{ FUNC_ENTER("LightmapGen::PtInLine"); 
	// Working Eqs.
	// P = Po + t * v  (parametric line eq)
	// X = x + t * v.x
	// Y = y + t * v.y
	// Z = z + t * v.z
	// P - Po = t * v
	// (P - Po) / v = t
	// t = (P - Po) / v

	// Start is always 0, end is always 1 by parametric def

	// Compute direction vector for line
	Point3 v = end - start;

	// We know the time of the initial point is 0
	// Compute time of the ending point
	// X - x = t * v.x
	// (X - x) / v = t
	// t = (X - x) / v
	// [x,y,z] == start
	// [X,Y,Z] == any

	// End times will always be 1 by definition

	// Compute time of unknown point
	float tPX, tPY, tPZ;
	bool  bInfX, bInfY, bInfZ;			// True if result is infinite
	
	bInfX = false;
	bInfY = false;
	bInfZ = false;

	// If v is 0 the value doesn't change on the line
	// which means if the pt value doesn't equal the start (or end) its
	// already failed to be on the line

	if (v.x == 0)
	{
		if (pt.x != start.x)
			return false;

		bInfX = true;
	}
	else
		tPX = (pt.x - start.x) / v.x;

	if (v.y == 0)
	{
		if (pt.y != start.y)
			return false;

		bInfY = true;
	}
	else
		tPY = (pt.y - start.y) / v.y;

	if (v.z == 0)
	{
		if (pt.z != start.z)
			return false;

		bInfZ = true;
	}
	else
		tPZ = (pt.z - start.z) / v.z;

	// If the point is on the line segment the component times should be equal (or infinite)
	// and it should fall within the start and end times 0 and 1
	
	// Check tPX
	if (!bInfX)
	{
		if (!bInfY)
			if (tPX != tPY)
				return false;

		if (!bInfZ)
			if (tPX != tPZ)
				return false;

		if (tPX >= 0.0f && tPX <= 1.0f)
			return true;

		return false;
	}

	// Check tPY
	if (!bInfY)
	{
		if (!bInfX)
			if (tPY != tPX)
				return false;

		if (!bInfZ)
			if (tPY != tPZ)
				return false;

		if (tPY >= 0.0f && tPY <= 1.0f)
			return true;

		return false;
	}

	// Check tPZ
	if (!bInfZ)
	{
		if (!bInfX)
			if (tPZ != tPX)
				return false;

		if (!bInfY)
			if (tPZ != tPY)
				return false;

		if (tPZ >= 0.0f && tPZ <= 1.0f)
			return true;

		return false;
	}

	// Every time is infinite (must be on line)
	return true;
}

bool LightmapGen::PtInTri(Point3 tri0, Point3 tri1, Point3 tri2, Point3 pt)
{ FUNC_ENTER("LightmapGen::PtInTri"); 
	// Sub-divide the triangle into 3 sub triangles inscribed within the original triangle
	// Then we can add the inner angles, if they add to 360 degrees we're inside the triangle
	// if not we're outside
	Point3 inedge0 = pt - tri0;
	Point3 inedge1 = pt - tri1;
	Point3 inedge2 = pt - tri2;

	float ang1 = GetAng(inedge0, inedge1);
	float ang2 = GetAng(inedge1, inedge2);
	float ang3 = GetAng(inedge2, inedge0);

	float sum = ang1 + ang2 + ang3;

	if ( sum > 359.0 && sum < 361.0 )
		return true;

	return false;
}

bool LightmapGen::RayTriIntersect(Point3  tri0, Point3 tri1, Point3 tri2,
	                              Ray     ray,
				                  Point3* intersect)
{ FUNC_ENTER("LightmapGen::RayTriIntersect"); 
	// Working Eqs.
	// Ax + By + Cz + D == 0  (plane)
	// t = -(Ax+By+Cz+D) / (Ai+Bj+Ck) (intersection time of ray and plane)

	// First we need to determine the equation for the plane that the triangle
	// exists in.  We'll start by finding the normal of the plane by taking
	// the cross product of two edges

	Point3 edge1, edge2, norm;
	edge1 = tri1 - tri0;
	edge2 = tri2 - tri1;
	norm  = edge1 ^ edge2;

	// Equivalent to: D = -(Ax+By+Cz)
	float D = -(DotProd(norm, tri0));

	// Now we have the eq of the plane, check to see if the ray intersects
	// the plane
	
	float dot = DotProd(norm, ray.dir);

	// If the dot product of the plane's normal and the direction of our ray is 0
	// then we know they are perpendicular.  If the direction of the ray is perpendicular
	// to the normal of the plane, then we know the ray is parallel to the plane.
	if (dot > -0.0001 && dot < 0.0001f)
		return false;

	// The ray must intersect the appropriate side of the triangle
	if (dot > 0.0f)
		return false;

	// We now know the ray intersects the triangle's infinite plane
	// Determine the point at which the intersection occurs
	// t = -(Ax+By+Cz+D) / (Ai+Bj+Ck) (intersection time of ray and plane)
	// [x,y,z] = pt on plane
	// [A,B,C] = plane normal
	float t = -(norm.x * tri0.x + norm.y * tri0.y + norm.z * tri0.z) / dot;

	// Now that we have the time of intersection we can use the parametric
	// form of a line to compute the actual intersection coordinates
	// P = Po + t * v  (parametric line eq)
	Point3 hitPt;
	hitPt = ray.p + t * ray.dir;

	// Ensure that the hit point lies within the triangle
	if (PtInTri(tri0, tri1, tri2, hitPt))
	{
		if (intersect)
			*intersect = hitPt;

		return true;
	}

	return false;
}

bool  LightmapGen::LineSegTriIntersect(Point3  tri0, Point3 tri1, Point3 tri2,
		                               Point3  segStart, Point3 segEnd,
						               Point3* intersect)
{ FUNC_ENTER("LightmapGen::LineSegTriIntersect"); 
	// Working Eqs.
	// Ax + By + Cz + D == 0  (plane)
	// t = -(Ax+By+Cz+D) / (Ai+Bj+Ck) (intersection time of ray and plane)

	// First we need to determine the equation for the plane that the triangle
	// exists in.  We'll start by finding the normal of the plane by taking
	// the cross product of two edges

	Point3 edge1, edge2, norm;
	edge1 = tri1 - tri0;
	edge2 = tri2 - tri1;
	norm  = edge1 ^ edge2;

	// Equivalent to: D = -(Ax+By+Cz)
	float D = -(DotProd(norm, tri0));

	// Now we have the eq of the plane, check to see if the ray intersects
	// the plane

	Point3 dir = segEnd - segStart;

	float dot = DotProd(norm, dir);

	// If the dot product of the plane's normal and the direction of our ray is 0
	// then we know they are perpendicular.  If the direction of the ray is perpendicular
	// to the normal of the plane, then we know the ray is parallel to the plane.
	if (dot == 0.0f)
		return false;

	// We now know the ray intersects the triangle's infinite plane
	// Determine the point at which the intersection occurs
	// t = -(Ax+By+Cz+D) / (Ai+Bj+Ck) (intersection time of ray and plane)
	// [x,y,z] = pt on plane
	// [A,B,C] = plane normal
	float t = -(norm.x * tri0.x + norm.y * tri0.y + norm.z * tri0.z) / dot;

	// Now that we have the time of intersection we can use the parametric
	// form of a line to compute the actual intersection coordinates
	// P = Po + t * v  (parametric line eq)
	Point3 hitPt;
	hitPt = segStart + t * dir;

	// Determine if the point that intersected is actually within the line
	if (!PtInLine(segStart, segEnd, hitPt))
		return false;

	// Ensure that the hit point lies within the triangle
	if (PtInTri(tri0, tri1, tri2, hitPt))
	{
		if (intersect)
			*intersect = hitPt;

		return true;
	}

	return false;
}

float LightmapGen::Dist(Point3 pt1, Point3 pt2)
{ FUNC_ENTER("LightmapGen::Dist"); 
	// Standard Distance equation
	return sqrt((pt2.x-pt1.x)*(pt2.x-pt1.x))+((pt2.y-pt1.y)*(pt2.y-pt1.y))+((pt2.z-pt1.z)*(pt2.z-pt1.z));
}

float LightmapGen::Dist2(Point3 pt1, Point3 pt2)
{ FUNC_ENTER("LightmapGen::Dist2"); 
	// Standard Distance equation (returns sqr of distance for speed)
	return ((pt2.x-pt1.x)*(pt2.x-pt1.x))+((pt2.y-pt1.y)*(pt2.y-pt1.y))+((pt2.z-pt1.z)*(pt2.z-pt1.z));
}

// This is the amount of space that a single triangles map
// should take up (this should typically be very small) and
// exist as
void LightmapGen::SetTriMapSize(int width, int height)
{ FUNC_ENTER("LightmapGen::SetTriMapSize"); 
	triMapWidth  = width;
	triMapHeight = height;

	//
}

Link<LightData>* LightmapGen::AddLight(Point3 pos, Point3 dir, Color color)
{ FUNC_ENTER("LightmapGen::AddLight"); 
	LightData ldata;

	ldata.ray.p   = pos;
	ldata.ray.dir = dir;
	ldata.color   = color;
	ldata.type    = LIGHTTYPE_DIRECTIONAL;

	return lightList.Add(&ldata);
}

Link<LightData>* LightmapGen::AddLight(Point3 pos, Color color)
{ FUNC_ENTER("LightmapGen::AddLight"); 
	LightData ldata;

	ldata.ray.p   = pos;
	ldata.color   = color;
	ldata.type    = LIGHTTYPE_OMNI;

	return lightList.Add(&ldata);
}

void LightmapGen::ClearLights()
{ FUNC_ENTER("LightmapGen::ClearLights"); 
	lightList.Clear();
}

// Triangle goes in UVs on the map for the object come out
void LightmapGen::LightTri(Point3     vert[3],
						   UVVert*    uv,
						   ImageData* img)
{ FUNC_ENTER("LightmapGen::LightTri"); 
	// For now, each triangle will have an individual file for test purposes
	
	// Fill out image data
	img->width = LMAP_SIZE;
	img->height = LMAP_SIZE;
	img->bit_depth = 8;
	img->nChannels = 1;
	img->imageSize = img->width * img->height;
	img->pixel_depth = 8;
	img->nPalEntries = 0;
	img->buf = (unsigned char*)malloc(sizeof(unsigned char) * img->width * img->height);

	// Find the extents of the triangle
	Point3  min, max;
	Point3 imin, imax;
	//UVVert uv[3];
	UVVert uvMin, uvMax;

	min    = vert[0];
	max    = vert[0];
	imin.x = imin.y = imin.z = 0;
	imax.x = imax.y = imax.z = 0;

	for(int i = 1; i < 3; i++)
	{
		if (vert[i].x < min.x)
		{
			imin.x = i;
			min.x = vert[i].x;
		}

		if (vert[i].y < min.y)
		{
			imin.y = i;
			min.y = vert[i].y;
		}

		if (vert[i].z < min.z)
		{
			imin.z = i;
			min.z = vert[i].z;
		}

		if (vert[i].x > max.x)
		{
			imax.x = i;
			max.x = vert[i].x;
		}

		if (vert[i].y > max.y)
		{
			imax.y = i;
			max.y = vert[i].y;
		}

		if (vert[i].z > max.z)
		{
			imax.z = i;
			max.z = vert[i].z;
		}
	}

	// Determine which axis to perform planar mapping on
	Point3 edge[3], norm, start, edgeH, edgeV;
	edge[0] = vert[1] - vert[0];
	edge[1] = vert[2] - vert[1];
	edge[2] = vert[0] - vert[2];

	norm = edge[0] ^ edge[1];

	float largeComponent = fabs(norm.x);
	int   mapMode        = MAP_YZ;

	if (fabs(norm.y) > largeComponent)
	{
		largeComponent = fabs(norm.y);
		mapMode = MAP_XZ;
	}

	if (fabs(norm.z) > largeComponent)
	{
		largeComponent = norm.z;
		mapMode = MAP_XY;
	}

	switch(mapMode)
	{
	case MAP_YZ:
		{
			for(int v = 0; v < 3; v++)
			{
				uv[v].x = vert[v].y;
				uv[v].y = vert[v].z;
			}
		}
		break;

	case MAP_XZ:
		{
			for(int v = 0; v < 3; v++)
			{
				uv[v].x = vert[v].x;
				uv[v].y = vert[v].z;
			}
		}
		break;

	case MAP_XY:
		{
			for(int v = 0; v < 3; v++)
			{
				uv[v].x = vert[v].x;
				uv[v].y = vert[v].y;
			}
		}
		break;
	}

	// Determine the min/max of the UVs
	uvMin = uv[0];
	uvMax = uv[0];

	int v;

	for(v = 0; v < 3; v++)
	{
		if (uv[v].x < uvMin.x)
			uvMin.x = uv[v].x;

		if (uv[v].y < uvMin.y)
			uvMin.y = uv[v].y;

		if (uv[v].x > uvMax.x)
			uvMax.x = uv[v].x;

		if (uv[v].y > uvMax.y)
			uvMax.y = uv[v].y;
	}

	// Map UVs from 0 to 1
	UVVert delta;

	delta.x = uvMax.x - uvMin.x;
	delta.y = uvMax.y - uvMin.y;

	for(v = 0; v < 3; v++)
	{
		// Map to start
		uv[v].x -= uvMin.x;
		uv[v].y -= uvMin.y;

		// Limit to range
		if (delta.x == 0.0f)
			delta.x = 0.0001f;

		if (delta.y == 0.0f)
			delta.y = 0.0001f;

		uv[v].x /= delta.x;
		uv[v].y /= delta.y;
	}

	// Compute plane distance
	// D = -(Ax+By+Cz)
	float D = -(norm.x * vert[0].x + norm.y * vert[0].y + norm.z * vert[0].z);

	// Compute edges to interpolate on
	switch(mapMode)
	{
	case MAP_YZ:
		{
			// Compute X
			// Ax+By+Cz+D = 0
			// Ax = -By-Cz-D
			// -Ax = By+Cz+D
			// x = -(By+Cz+D) / A

			// Compute start vector (min, min) 
			float x;
			
			if (norm.x == 0.0f)
				x = 0;
			else
				x = -(norm.y * uvMin.x + norm.z * uvMin.y + D) / norm.x;
			
			start = Point3(x, uvMin.x, uvMin.y);

			// Compute Horiz vector (max, min)	// ??? Verify (Flip)
			if (norm.x == 0.0f)
				x = 0;
			else
				x = -(norm.y * uvMax.x + norm.z * uvMin.y + D) / norm.x;

			edgeH = Point3(x, uvMin.y, uvMin.x);

			// Compute Vert vector (min, max) // ??? Verify (Flip)
			if (norm.x == 0.0f)
				x = 0;
			else
				x = -(norm.y * uvMin.x + norm.z * uvMax.y +  D) / norm.x;

			edgeV = Point3(x, uvMax.y, uvMin.x);
		}
		break;

	case MAP_XZ:
		{
			// Compute Y
			// Ax+By+Cz+D = 0
			// By = -Ax-Cz-D
			// -By = Ax+Cz+D
			// y = -(Ax+Cz+D) / B

			// Compute start vector (min, min)
			float y;

			if (norm.y == 0.0f)
				y = 0;
			else
				y = -(norm.x * uvMin.x + norm.z * uvMin.y + D) / norm.y;

			start = Point3(uvMin.x, y, uvMin.y);

			// Compute Horiz vector (max, min) /// ??? Verify
			if (norm.y == 0.0f)
				y = 0;
			else
				y = -(norm.x * uvMax.x + norm.z * uvMin.y + D) / norm.y;

			edgeH = Point3(uvMax.x, y, uvMin.y);

			// Compute Vert vector (min, max) /// ??? Verify
			if (norm.y == 0.0f)
				y = 0;
			else
				y = -(norm.x * uvMin.x + norm.z * uvMax.y + D) / norm.y;

			edgeV = Point3(uvMin.x, y, uvMax.y);
		}
		break; 

	case MAP_XY:
		{
			// Compute Z
			// Ax+By+Cz+D = 0
			// Cz = -Ax-By-D
			// -Cz = Ax+By+D
			// z = -(Ax+By+D) / C

			// Compute start vector (min, min)
			float z;

			if (norm.z == 0.0f)
				z = 0;
			else
				z = -(norm.x * uvMin.x + norm.y * uvMin.y + D) / norm.z;

			start = Point3(uvMin.x, uvMin.y, z);
			
			// Compute Horiz vector (max, min) /// ??? Verify
			if (norm.z == 0.0f)
				z = 0;
			else
				z = -(norm.x * uvMax.x + norm.y * uvMin.y + D) / norm.z;

			edgeH = Point3(uvMax.x, uvMin.y, z);

			// Compute Vert vector (min, max) /// ??? Verify
			if (norm.z == 0.0f)
				z = 0;
			else
				z = -(norm.x * uvMin.x + norm.y * uvMax.y + D) / norm.z;

			edgeV = Point3(uvMin.x, uvMax.y, z);
		}
		break;
	}

	// Make computed edges relative to start
	edgeH = edgeH - start;
	edgeV = edgeV - start;

	// We now know the extents of the triangle, and can interpolate accross it
	// Build Lightmap
	for(int posY = 0; posY < LMAP_SIZE; posY++)
	{
		for(int posX = 0; posX < LMAP_SIZE; posX++)
		{
			// Compute interp percent by current pos
			float UPer = (float)posX / (float)LMAP_SIZE;
			float VPer = (float)posY / (float)LMAP_SIZE;

			Point3 curEdgeH = edgeH * UPer;
			Point3 curEdgeV = edgeV * VPer;

			Point3 worldPos = start + curEdgeH + curEdgeV;

			unsigned char color = 255;
			float ambient = 0.0f;

			// For now we'll just choose an intensity based on the distance from
			// this point to the distance of our light source(s)
			Link<LightData>* curlink = lightList.GetHead();

			while(curlink)
			{
				// We'll use Lambert's formula to calculate the lighting
				Point3 energy  = curlink->data.ray.p - worldPos;
				//Point3 energy  = worldPos - curlink->data.ray.p;
				Point3 triNorm = norm.Normalize();
				energy = energy.Normalize();
				//energy.x = fabs(energy.x);
				//energy.y = fabs(energy.y);
				//energy.z = fabs(energy.z);

				//float const falloff = 10.0f;
				float dist = Dist(worldPos, curlink->data.ray.p);

				if (dist > 40.0f)
				{
					//dist = 1 / dist;
					dist = 40.0f;
				}

				dist = 40.0f / dist;

				float dot  = DotProd(energy, triNorm);
				float b = DotProd(energy, triNorm) * dist;
				//float b = DotProd(energy, triNorm) * 1000.0f / Dist(worldPos, curlink->data.ray.p);
				//float b = DotProd(energy, triNorm);// / Dist(worldPos, curlink->data.ray.p);

				// Compute ambient light into brightness
				// (Since we invert later ambient light is subtracted)
				//b *= ambient;

				// Clamp it
				if (b > 1.0f)
					b = 1.0f;

				if (b < 0.0f)
					b = 0.0f;

				// Scale the brightness to 1 byte of accuracy
				// (Doubling range and clamping for greater varience within the map)
				// Using the straight 0 - 255 scale generates very slight gradients
				//b *= 80.0f;
				b *= 255.0f;
				
				if (b > 255.0f)
					b = 255.0f;
				
				color = b;

				// Clamp it


				/*
				float d = Dist(worldPos, curlink->data.ray.p) / 8.0f;
			
				// Clamp color
				if ((float)color - d < 0.0f)
					color = 0;
				else
					color -= d;
				*/

				curlink = curlink->next;
			}

			// Write the pixel into the lightmap texture
			img->buf[posY * img->width + posX] = color;
		}
	}


	/*
	// Test write
	unsigned char color = 128;

	for(int y = 0; y < img.height; y++)
	{
		for(int x = 0; x < img.width; x++)
		{
			img.buf[y * img.width + x] = color;
		}

		color++;
	}
	*/

	img->WritePNG("c:\\lighttest.png");
}

static int allFaces = 0;

void LightmapGen::glLightUpdate(GLRenderer* gl, void* pData)
{ FUNC_ENTER("LightmapGen::glLightUpdate"); 
	LightmapGen* pthis = (LightmapGen*)pData;

	glEnable(GL_DEPTH_TEST);
	glDisable(GL_TEXTURE_2D);
	glDepthFunc(GL_LESS);

	Point3 dir[6] = { Point3(0.00001f,0.8999f,0.0f),	// Up		(Must be different from up vector for lookat matrix comp)
		              Point3(0.00001f,-1.0f,0.0f),	// Down		(Must be different from up vector for lookat matrix comp)
					  Point3(-1,0,0),				// Left
					  Point3(1,0,0),				// Right
					  Point3(0,0,-1),				// In
					  Point3(0,0,1),				// Out
					};

	//for(int cdir = 0; cdir < 6; cdir++)
	Link<LightData>* link = pthis->lightList.GetHead();

	Point3 trans = link->data.ray.p;

	// Convert trans to the game's coordinate system
	float temp = trans.y;
	trans.y = trans.z;
	trans.z = -temp;

	//int cdir = 5;
	{
		static float rot = 0.0f;

		Matrix3 rotmat;
		rotmat.IdentityMatrix();
		//rotmat.RotateX(rot+=0.01f);

		Point3 dir2 = dir[pthis->curdir] * rotmat;

		Point3 viewPos = trans + dir2;
		//viewPos = viewPos.Normalize();

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		
		gluPerspective(90.0, 1.0, 0.1, 10000.0);
		gluLookAt(trans.x, trans.y, trans.z,
				  viewPos.x, viewPos.y, viewPos.z,
				  0.0f,1.0f,0.0f);

		//glRotatef(rot+=0.1f,0.0f,1.0f,0.0f);

		glViewport(0, 0, 640, 480);

		int curFace = 1;
		allFaces = 1;

		for(int i = 0; i < pthis->scene->m_Objects.Count(); i++)
		{
			/* From NxObject
			int				m_NumFaces;	
			NxFace*			m_Faces;
			int				m_NumVerts;
			NxVertex*		m_Verts;
			*/

			NxObject* obj = pthis->scene->m_Objects[i];

			for(int faceIdx = 0; faceIdx < obj->m_NumFaces; faceIdx++)
			{
				// Determine color from first 24-bits of triangle index
				unsigned char r = curFace & 0xFF;
				unsigned char g = (curFace & 0xFF00) >> 8;
				unsigned char b = (curFace & 0xFF0000) >> 16;

//				char tbuf[256];

				//r=255;
				//g=255;
				b=255;

				glColor3ub(r,g,b);
				//glColor3f(1.0f,1.0f,1.0f);

				curFace++;
				allFaces++;

				glBegin(GL_TRIANGLES);

				glVertex3f(obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[0]].m_Pos.x,
						   obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[0]].m_Pos.y,
						   obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[0]].m_Pos.z);

//				sprintf(tbuf, "[%f,%f,%f], ", obj->m_Verts[obj->m_Faces[i].m_Vertex[0]].m_Pos.x,
//											  obj->m_Verts[obj->m_Faces[i].m_Vertex[0]].m_Pos.z,
//											  obj->m_Verts[obj->m_Faces[i].m_Vertex[0]].m_Pos.y);
//
//				OutputDebugString(tbuf);

				glVertex3f(obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[1]].m_Pos.x,
						   obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[1]].m_Pos.y,
						   obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[1]].m_Pos.z);

//				sprintf(tbuf, "[%f,%f,%f], ", obj->m_Verts[obj->m_Faces[i].m_Vertex[1]].m_Pos.x,
//											  obj->m_Verts[obj->m_Faces[i].m_Vertex[1]].m_Pos.z,
//											  obj->m_Verts[obj->m_Faces[i].m_Vertex[1]].m_Pos.y);
//
//				OutputDebugString(tbuf);

				glVertex3f(obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[2]].m_Pos.x,
						   obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[2]].m_Pos.y,
						   obj->m_Verts[obj->m_Faces[faceIdx].m_Vertex[2]].m_Pos.z);

//				sprintf(tbuf, "[%f,%f,%f]\n", obj->m_Verts[obj->m_Faces[i].m_Vertex[2]].m_Pos.x,
//											  obj->m_Verts[obj->m_Faces[i].m_Vertex[2]].m_Pos.z,
//											  obj->m_Verts[obj->m_Faces[i].m_Vertex[2]].m_Pos.y);
//
//				OutputDebugString(tbuf);

				glEnd();
			}
		}
	}
}

void LightmapGen::ProcLight(NxScene* scene, INode* node)
{ FUNC_ENTER("LightmapGen::ProcLight"); 
	Matrix3 TM    = node->GetNodeTM(0);
	Point3  trans = TM.GetTrans();
	Color   color(255,255,255);

	// Trans must be converted into game coord system since it is being computed
	// against scene data that has already been put in the game coord system
	float temp;
	temp    = trans.y;
	trans.y = trans.z;
	trans.z = -temp;

	Link<LightData>* link = AddLight(trans, color);

	// Do PVS preprocessing
	// The idea here is to render the entire scene with GL assigning a unique color
	// to each triangle.  This should be more than sufficient with 24-bit color
	// 255*255*255 gives us 16581375 color combinations.  We render the scene 6 times as we
	// would generate a cubic environment map.  Render Top (0,1,0), Bottom (0.-1,0), Left, Right, In, Out
	// And then add the faces to the PVS list based on their color, (we'll shift the component colors
	// to build a 24-bit face index value and vice-versa)
	// Additionally, we'll tag the face so we know what lights hit it from an NxFace
	
	//HDC hdc = InitContext();
	//MakeCurrent();

	InitWindow();
	//ShowWindow();

	// Need to render the scene from all 6 lighting directions
	for(curdir = 0; curdir < 6; curdir++)
	{
		SetCallback(glLightUpdate, this);
		allFaces = 0;
		Update();
		MakeCurrent();

		// Analyze the generated scene
		unsigned char* buf = (unsigned char*)malloc(sizeof(unsigned char) * 640 * 480 * 3);

		glPixelStorei(GL_UNPACK_ROW_LENGTH,0);
		glPixelStorei(GL_UNPACK_SKIP_ROWS,0);
		glPixelStorei(GL_UNPACK_SKIP_PIXELS,0);
		glPixelStorei(GL_UNPACK_ALIGNMENT,1);
																																																																								
		glReadPixels(0, 0, 640, 480, GL_RGB, GL_UNSIGNED_BYTE, buf);

		// Scan through the directional render from the lightsource and match up the
		// color indicies with the corresponding NxFaces so we can build up PVS info
		// for the light

		color24* imgcolor = (color24*)buf;

		// Force all
		for(int idx = 0; idx < allFaces - 1; idx++)
		{
			link->data.pvs.AddUnique(&(sceneFaces[idx]));

			// Tag the face with this light
			faceLights[idx].AddUnique(&link);
		}

		/*
		for(int idx = 0; idx < 640 * 480; idx++)
		{
			// Build Face index from color
			//int faceIdx = ((imgcolor[idx].r) | (imgcolor[idx].g << 8) | (imgcolor[idx].b << 16)) - 1;
			int faceIdx = ((imgcolor[idx].r) | (imgcolor[idx].g << 8)) - 1;

			if (faceIdx >= 0)
			{
				link->data.pvs.AddUnique(&(sceneFaces[faceIdx]));

				// Tag the face with this light
				faceLights[faceIdx].AddUnique(&link);
			}
		}
		*/

		ImageData img;

		img.width = 640;
		img.height = 480;
		img.bit_depth = 8;
		img.nChannels = 3;
		img.imageSize = img.width * img.height;
		img.pixel_depth = 8;
		img.nPalEntries = 0;
		img.color_type = PNG_COLOR_TYPE_RGB;
		img.buf = (unsigned char*)malloc(sizeof(unsigned char) * img.width * img.height * img.nChannels);
		memcpy(img.buf, buf, 640*480*3);

		char bufDir[256], bufFile[256];
		_itoa(curdir,bufDir,10);

		strcpy(bufFile,"c:\\fbuffer");
		strcat(bufFile, bufDir);
		strcat(bufFile,".png");

		img.WritePNG(bufFile);

		free(buf);

		EndCurrent();
	}
	//scene->m_Objects.Count();
}

void LightmapGen::GenerateLightmaps(NxScene* scene)
{ FUNC_ENTER("LightmapGen::GenerateLightmaps"); 
	INodeTab nodes;
	this->scene = scene;

	// Initialize scene face pointers
	int nSceneTris = NumSceneTris();
	int i, j;

	lightList.Clear();

	if (sceneFaces)
		delete [] sceneFaces;

	if (faceLights)
		delete [] faceLights;

	sceneFaces = new FaceObj[nSceneTris];
	faceLights = new LinkList< Link<LightData>* >[nSceneTris];

	int nObjects = scene->m_Objects.Count();
	int curface = 0;

	for(i = 0; i < nObjects; i++)
	{
		NxObject* obj = scene->m_Objects[i];
		int nFaces = obj->m_NumFaces;

		for(j = 0; j < nFaces; j++)
		{
			sceneFaces[curface].face = &(obj->m_Faces[j]);
			sceneFaces[curface].obj  = obj;
			curface++;
		}
	}

	GetAllNodes(nodes, gInterface->GetRootNode());

	// Enumerate all the lights in the scene
	gInterface->ProgressStart(_T("Building scene light PVS"), TRUE, ProgressFunc, NULL);

	for(i = 0; i < nodes.Count(); i++)
	{
		// Process this node if it happens to be a light
		// (Only processing omni lights for now
		Object* obj = nodes[i]->EvalWorldState(0).obj;

		if (obj->ClassID() == Class_ID(OMNI_LIGHT_CLASS_ID, 0))
		{
			ProcLight(scene, nodes[i]);
		}

		gInterface->ProgressUpdate( i * 100 / nodes.Count() );
	}

	gInterface->ProgressEnd();

	int nLightMaps = NumLightMaps();

	// Allocate lightmaps
	if (maps)
		delete [] maps;

	maps = new ImageData[nLightMaps];

	// Num we'll build up the actual light maps
	gInterface->ProgressStart(_T("Building lightmaps"), TRUE, ProgressFunc, NULL);

	// Scan through all the faces in the scene
	curface = 0;
	for(i = 0; i < nObjects; i++)
	{
		NxObject* obj = scene->m_Objects[i];

		int nFaces = obj->m_NumFaces;

		for(j = 0; j < nFaces; j++)
		{
			if (faceLights[curface].GetSize() > 0)
			{
				Point3 pt[3];
				pt[0] = obj->m_Verts[obj->m_Faces[j].m_Vertex[0]].m_Pos;
				pt[1] = obj->m_Verts[obj->m_Faces[j].m_Vertex[1]].m_Pos;
				pt[2] = obj->m_Verts[obj->m_Faces[j].m_Vertex[2]].m_Pos;

				sceneFaces[curface].face->m_LightmapChecksum = curmap + 1;

				// TODO: Add map to texture dictionary if possible
				LightTri(pt, sceneFaces[curface].face->m_LightmapUV, &(maps[curmap]), &(faceLights[curface]));
				curmap++;
			}

			gInterface->ProgressUpdate( curface * 100 / nSceneTris );
			curface++;
		}
	}
	
	gInterface->ProgressEnd();

	/*
	delete [] sceneFaces;
	delete faceLights;

	sceneFaces = NULL;
	faceLights = NULL;
	*/
}

int LightmapGen::NumLightMaps()
{ FUNC_ENTER("LightmapGen::NumLightMaps"); 
	// This function computes the number of lightmaps that will be generated after
	// PVS data has been generated
	int nLightMaps = 0;

	if (!sceneFaces ||
		!faceLights)
		return 0;

	int nFaces = NumSceneTris();

	for(int i = 0; i < nFaces; i++)
	{
		if (faceLights[i].GetSize() > 0)
			nLightMaps++;
	}

	return nLightMaps;
}

int LightmapGen::NumSceneTris()
{ FUNC_ENTER("LightmapGen::NumSceneTris"); 
	// This function will determine the total number of NxFaces used to construct the
	// entire scene
	int nObjects = scene->m_Objects.Count();
	int nFaces = 0;

	for(int i = 0; i < nObjects; i++)
	{
		nFaces += scene->m_Objects[i]->m_NumFaces;
	}

	return nFaces;
}

bool LightmapGen::IsLit(Point3     worldPos,
						LightData* ldata,
						Point3     excludeTri0,
						Point3     excludeTri1,
						Point3     excludeTri2)
{ FUNC_ENTER("LightmapGen::IsLit"); 
	// Check if the line segment from the worldPos to the lightsource
	// intersects any triangles within the light's PVS data set

	Link<FaceObj>* curface = ldata->pvs.GetHead();

	while(curface)
	{
		NxVertex* verts = curface->data.obj->m_Verts;

		if (verts[curface->data.face->m_Vertex[0]].m_Pos != excludeTri0 &&
			verts[curface->data.face->m_Vertex[1]].m_Pos != excludeTri1 &&
			verts[curface->data.face->m_Vertex[2]].m_Pos != excludeTri2)
		{
//			if (LineSegTriIntersect(verts[curface->data.face->m_Vertex[0]].m_Pos,
//									verts[curface->data.face->m_Vertex[1]].m_Pos,
//									verts[curface->data.face->m_Vertex[2]].m_Pos,
//									worldPos,
//									ldata->ray.p))

			ldata->ray.dir = worldPos - ldata->ray.p;
			//ldata->ray.dir = ldata->ray.p - worldPos;
			ldata->ray.dir.Normalize();

			if (RayTriIntersect(verts[curface->data.face->m_Vertex[0]].m_Pos,
				                verts[curface->data.face->m_Vertex[1]].m_Pos,
								verts[curface->data.face->m_Vertex[2]].m_Pos,
								ldata->ray,
								NULL))
			{
				return false;
			}			
		}

		curface = curface->next;
	}

	return true;
}

// Triangle goes in UVs on the map for the object come out
void LightmapGen::LightTri(Point3     vert[3],
						   UVCoord*   uv,
						   ImageData* img,
						   LinkList< Link<LightData>* >* lights )
{ FUNC_ENTER("LightmapGen::LightTri"); 
	// For now, each triangle will have an individual file for test purposes
	
	// Fill out image data
	img->width = LMAP_SIZE;
	img->height = LMAP_SIZE;
	img->bit_depth = 8;
	img->nChannels = 1;
	img->imageSize = img->width * img->height;
	img->pixel_depth = 8;
	img->nPalEntries = 0;
	img->buf = (unsigned char*)malloc(sizeof(unsigned char) * img->width * img->height);

	// Find the extents of the triangle
	Point3  min, max;
	Point3 imin, imax;
	//UVVert uv[3];
	UVCoord uvMin, uvMax;

	min    = vert[0];
	max    = vert[0];
	imin.x = imin.y = imin.z = 0;
	imax.x = imax.y = imax.z = 0;

	for(int i = 1; i < 3; i++)
	{
		if (vert[i].x < min.x)
		{
			imin.x = i;
			min.x = vert[i].x;
		}

		if (vert[i].y < min.y)
		{
			imin.y = i;
			min.y = vert[i].y;
		}

		if (vert[i].z < min.z)
		{
			imin.z = i;
			min.z = vert[i].z;
		}

		if (vert[i].x > max.x)
		{
			imax.x = i;
			max.x = vert[i].x;
		}

		if (vert[i].y > max.y)
		{
			imax.y = i;
			max.y = vert[i].y;
		}

		if (vert[i].z > max.z)
		{
			imax.z = i;
			max.z = vert[i].z;
		}
	}

	// Determine which axis to perform planar mapping on
	Point3 edge[3], norm, start, edgeH, edgeV;
	edge[0] = vert[1] - vert[0];
	edge[1] = vert[2] - vert[1];
	edge[2] = vert[0] - vert[2];

	norm = edge[0] ^ edge[1];

	float largeComponent = fabs(norm.x);
	int   mapMode        = MAP_YZ;

	if (fabs(norm.y) > largeComponent)
	{
		largeComponent = fabs(norm.y);
		mapMode = MAP_XZ;
	}

	if (fabs(norm.z) > largeComponent)
	{
		largeComponent = norm.z;
		mapMode = MAP_XY;
	}

	switch(mapMode)
	{
	case MAP_YZ:
		{
			for(int v = 0; v < 3; v++)
			{
				uv[v].u = vert[v].y;
				uv[v].v = vert[v].z;
			}
		}
		break;

	case MAP_XZ:
		{
			for(int v = 0; v < 3; v++)
			{
				uv[v].u = vert[v].x;
				uv[v].v = vert[v].z;
			}
		}
		break;

	case MAP_XY:
		{
			for(int v = 0; v < 3; v++)
			{
				uv[v].u = vert[v].x;
				uv[v].v = vert[v].y;
			}
		}
		break;
	}

	// Determine the min/max of the UVs
	uvMin = uv[0];
	uvMax = uv[0];

	int v;

	for(v = 0; v < 3; v++)
	{
		if (uv[v].u < uvMin.u)
			uvMin.u = uv[v].u;

		if (uv[v].v < uvMin.v)
			uvMin.v = uv[v].v;

		if (uv[v].u > uvMax.u)
			uvMax.u = uv[v].u;

		if (uv[v].v > uvMax.v)
			uvMax.v = uv[v].v;
	}

	// Map UVs from 0 to 1
	UVCoord delta;

	delta.u = uvMax.u - uvMin.u;
	delta.v = uvMax.v - uvMin.v;

	for(v = 0; v < 3; v++)
	{
		// Map to start
		uv[v].u -= uvMin.u;
		uv[v].v -= uvMin.v;

		// Limit to range
		if (delta.u == 0.0f)
			delta.u = 0.0001f;

		if (delta.v == 0.0f)
			delta.v = 0.0001f;

		uv[v].u /= delta.u;
		uv[v].v /= delta.v;
	}

	// Compute plane distance
	// D = -(Ax+By+Cz)
	float D = -(norm.x * vert[0].x + norm.y * vert[0].y + norm.z * vert[0].z);

	// Compute edges to interpolate on
	switch(mapMode)
	{
	case MAP_YZ:
		{
			// Compute X
			// Ax+By+Cz+D = 0
			// Ax = -By-Cz-D
			// -Ax = By+Cz+D
			// x = -(By+Cz+D) / A

			// Compute start vector (min, min) 
			float x;
			
			if (norm.x == 0.0f)
				x = 0;
			else
				x = -(norm.y * uvMin.u + norm.z * uvMin.v + D) / norm.x;
			
			start = Point3(x, uvMin.u, uvMin.v);

			// Compute Horiz vector (max, min)	// ??? Verify (Flip)
			if (norm.x == 0.0f)
				x = 0;
			else
				x = -(norm.y * uvMax.u + norm.z * uvMin.v + D) / norm.x;

			edgeH = Point3(x, uvMin.u, uvMin.v);

			// Compute Vert vector (min, max) // ??? Verify (Flip)
			if (norm.x == 0.0f)
				x = 0;
			else
				x = -(norm.y * uvMin.u + norm.z * uvMax.v +  D) / norm.x;

			edgeV = Point3(x, uvMax.v, uvMin.u);
		}
		break;

	case MAP_XZ:
		{
			// Compute Y
			// Ax+By+Cz+D = 0
			// By = -Ax-Cz-D
			// -By = Ax+Cz+D
			// y = -(Ax+Cz+D) / B

			// Compute start vector (min, min)
			float y;

			if (norm.y == 0.0f)
				y = 0;
			else
				y = -(norm.x * uvMin.u + norm.z * uvMin.v + D) / norm.y;

			start = Point3(uvMin.u, y, uvMin.v);

			// Compute Horiz vector (max, min) /// ??? Verify
			if (norm.y == 0.0f)
				y = 0;
			else
				y = -(norm.x * uvMax.u + norm.z * uvMin.v + D) / norm.y;

			edgeH = Point3(uvMax.u, y, uvMin.v);

			// Compute Vert vector (min, max) /// ??? Verify
			if (norm.y == 0.0f)
				y = 0;
			else
				y = -(norm.x * uvMin.u + norm.z * uvMax.v + D) / norm.y;

			edgeV = Point3(uvMin.u, y, uvMax.v);
		}
		break; 

	case MAP_XY:
		{
			// Compute Z
			// Ax+By+Cz+D = 0
			// Cz = -Ax-By-D
			// -Cz = Ax+By+D
			// z = -(Ax+By+D) / C

			// Compute start vector (min, min)
			float z;

			if (norm.z == 0.0f)
				z = 0;
			else
				z = -(norm.x * uvMin.u + norm.y * uvMin.v + D) / norm.z;

			start = Point3(uvMin.u, uvMin.v, z);
			
			// Compute Horiz vector (max, min) /// ??? Verify
			if (norm.z == 0.0f)
				z = 0;
			else
				z = -(norm.x * uvMax.u + norm.y * uvMin.v + D) / norm.z;

			edgeH = Point3(uvMax.u, uvMin.v, z);

			// Compute Vert vector (min, max) /// ??? Verify
			if (norm.z == 0.0f)
				z = 0;
			else
				z = -(norm.x * uvMin.u + norm.y * uvMax.v + D) / norm.z;

			edgeV = Point3(uvMin.u, uvMax.v, z);
		}
		break;
	}

	// Make computed edges relative to start
	edgeH = edgeH - start;
	edgeV = edgeV - start;

	// We now know the extents of the triangle, and can interpolate accross it
	// Build Lightmap
	for(int posY = 0; posY < LMAP_SIZE; posY++)
	{
		for(int posX = 0; posX < LMAP_SIZE; posX++)
		{
			// Compute interp percent by current pos
			float UPer = (float)posX / (float)LMAP_SIZE;
			float VPer = (float)posY / (float)LMAP_SIZE;

			Point3 curEdgeH = edgeH * UPer;
			Point3 curEdgeV = edgeV * VPer;

			Point3 worldPos = start + curEdgeH + curEdgeV;

			unsigned char color = 255;
			float ambient = 0.0f;

			// For now we'll just choose an intensity based on the distance from
			// this point to the distance of our light source(s) on this face
			Link< Link<LightData>* >* curlink = lights->GetHead();

			while(curlink)
			{
				// We'll use Lambert's formula to calculate the lighting
				Point3 energy  = curlink->data->data.ray.p - worldPos;
				//Point3 energy  = worldPos - curlink->data.ray.p;
				Point3 triNorm = norm.Normalize();
				energy = energy.Normalize();
				//energy.x = fabs(energy.x);
				//energy.y = fabs(energy.y);
				//energy.z = fabs(energy.z);

				//float const falloff = 10.0f;
				float dist = Dist(worldPos, curlink->data->data.ray.p);

				if (dist > 40.0f)
				{
					//dist = 1 / dist;
					dist = 40.0f;
				}

				dist = 40.0f / dist;

				float dot  = DotProd(energy, triNorm);
				float b = dot * dist;
				//float b = DotProd(energy, triNorm) * 1000.0f / Dist(worldPos, curlink->data.ray.p);
				//float b = DotProd(energy, triNorm);// / Dist(worldPos, curlink->data.ray.p);

				// Compute ambient light into brightness
				// (Since we invert later ambient light is subtracted)
				//b *= ambient;

				// Clamp it
				if (b > 1.0f)
					b = 1.0f;

				if (b < 0.0f)
					b = 0.0f;

				// Scale the brightness to 1 byte of accuracy
				// (Doubling range and clamping for greater varience within the map)
				// Using the straight 0 - 255 scale generates very slight gradients
				//b *= 80.0f;
				b *= 255.0f;
				
				if (b > 255.0f)
					b = 255.0f;

				// TODO: Raytracing, err casting for now, since we're not bouncing rays or absorbing energy
				//       Check ray-tri intersection with all other tris in the light's
				//       PVS list to determine if the light actually reaches this triangle or is obscured

				// Check for shadows
				// If we draw a line from this point (on the triangle we're lighting) to the light source
				// and it intersects a different triangle we know that the brightness of this point should
				// actually be 0 (unless we start computing transparency but, that's another issue)
				if (IsLit(worldPos,
					      &(curlink->data->data),
						  vert[0],
						  vert[1],
						  vert[2]))
					color = b;
				else
					color = 0;

				// Clamp it


				/*
				float d = Dist(worldPos, curlink->data.ray.p) / 8.0f;
			
				// Clamp color
				if ((float)color - d < 0.0f)
					color = 0;
				else
					color -= d;
				*/

				curlink = curlink->next;
			}

			// Write the pixel into the lightmap texture
			img->buf[posY * img->width + posX] = color;
		}
	}


	/*
	// Test write
	unsigned char color = 128;

	for(int y = 0; y < img.height; y++)
	{
		for(int x = 0; x < img.width; x++)
		{
			img.buf[y * img.width + x] = color;
		}

		color++;
	}
	*/

	//img->WritePNG("c:\\lighttest.png");
}

bool LightmapGen::SaveLightmaps()
{ FUNC_ENTER("LightmapGen::SaveLightmaps"); 
	// Save out a lightmap database file
	// I've decided to make this a seperate file as opposed to stuffing it into the
	// texture dictionary since:
	//
	// 1) All lightmaps use the same greyscale format
	// 2) We WILL later share maps on the same texture
	// 3) Additional tmap and material data is non-relevant for lightmaps
	// 4) Prevents lightmaps from needing to be reprocessed with texture changes
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr filename = seo.m_SceneName;
	filename += ".lmp";
	char* project_root = getenv( APP_ENV );

	filename = CStr(project_root) + CStr(LEVEL_PATH) + seo.m_SceneName + CStr("\\") + filename;

	gInterface->ProgressStart(_T("Saving Lightmap data"), TRUE, ProgressFunc, NULL);

	FILE* fp = fopen(filename,"wb");

	if (!fp)
		return false;

	int nMaps = curmap;
	
	const unsigned long version = 0x0001;
	fwrite(&version, sizeof(unsigned long), 1, fp);

	fwrite(&nMaps, sizeof(int), 1, fp);

	for(int i = 0; i < nMaps; i++)
	{
		fwrite(&maps[i].width, sizeof(unsigned long), 1, fp);
		fwrite(&maps[i].height, sizeof(unsigned long), 1, fp);

		// All maps are grey-scale 1 channel 1 byte per channel
		fwrite(maps[i].buf, maps[i].width * maps[i].height, 1, fp);

		gInterface->ProgressUpdate( i * 100 / nMaps );
	}

	fclose(fp);

	gInterface->ProgressEnd();

	return true;
}
