/**********************************************************************
 *<
	FILE: PolyEdUI.cpp

	DESCRIPTION: Editable Polygon Mesh Object / Edit Polygon Mesh Modifier UI code

	CREATED BY: Steve Anderson

	HISTORY: created Nov 9, 1999

 *>	Copyright (c) 1999, All Rights Reserved.
 **********************************************************************/

#include "EPoly.h"
#include "PolyEdit.h"
#include "MeshDLib.h"
#include "MaxIcon.h"

#include <FaceFlags/FaceFlags.h>

static int	checkbox_to_flag_table[][2] = {
			{ mFD_SKATABLE, IDC_SKATABLE },
			{ mFD_NOT_SKATABLE, IDC_NOT_SKATABLE },
			{ mFD_WALL_RIDABLE, IDC_WALL_RIDABLE },
			{ mFD_VERT, IDC_VERT_POLY },
			{ mFD_NON_COLLIDABLE, IDC_NON_COLLIDABLE },
			{ mFD_TRIGGER, IDC_TRIGGER },
			{ mFD_CAMERA_COLLIDABLE, IDC_CAM_COLLIDABLE },
			{ mFD_NO_SKATER_SHADOW, IDC_NO_SKATER_SHADOW },
			{ mFD_SKATER_SHADOW, IDC_SKATER_SHADOW },
			{ mFD_NO_SKATER_SHADOW_WALL, IDC_NO_SKATER_SHADOW_WALL },
			{ mFD_UNDER_OK, IDC_UNDER_OK },
			{ mFD_INVISIBLE, IDC_INVISIBLE },
			{ 0, 0 }
};

extern Interface* gInterface;

// Class Descriptor
// (NOTE: this must be in the same file as, and previous to, the ParamBlock2Desc declaration.)
class EditablePolyObjectClassDesc : public ClassDesc2 {
public:
	int 			IsPublic() { return 0; }
	void *			Create(BOOL loading = FALSE);
	const TCHAR *	ClassName() {return _T("Editable Poly"); }
	SClass_ID		SuperClassID() {return GEOMOBJECT_CLASS_ID;}
	Class_ID		ClassID() {return EPOLYOBJ_CLASS_ID;}
	const TCHAR* 	Category() {return _T("Editable Objects");}
	const TCHAR*	InternalName() { return _T("EditablePolyMesh"); }	// returns fixed parsable name (scripter-visible name)
	HINSTANCE		HInstance() { return hInstance; }			// returns owning module handle
};

static EditablePolyObjectClassDesc editPolyObjectDesc;
ClassDesc2* GetEditablePolyDesc() {return &editPolyObjectDesc;}

static FPInterfaceDesc epfi (
	EPOLY_INTERFACE, _T("EditablePoly"), IDS_SCA_E_POLY, &editPolyObjectDesc, FP_MIXIN,
	epfn_hide, _T("Hide"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_unhide_all, _T("unhideAll"), 0, TYPE_bool, 0, 1,
		_T("mnSelLevel"), 0, TYPE_INT,
	epfn_named_selection_copy, _T("namedSelCopy"), 0, TYPE_VOID, 0, 1,
		_T("Name"), 0, TYPE_STRING,
	epfn_named_selection_paste, _T("namedSelPaste"), 0, TYPE_VOID, 0, 1,
		_T("useRenameDialog"), 0, TYPE_bool,
	epfn_create_vertex, _T("createVertex"), 0, TYPE_INT, 0, 3,
		_T("point"), 0, TYPE_POINT3,
		_T("pointInLocalCoords"), 0, TYPE_bool, f_keyArgDefault, true,
		_T("select"), 0, TYPE_bool, f_keyArgDefault, false,
	epfn_create_edge, _T("createEdge"), 0, TYPE_INT, 0, 3,
		_T("vertex1"), 0, TYPE_INDEX,
		_T("vertex2"), 0, TYPE_INDEX,
		_T("select"), 0, TYPE_bool, f_keyArgDefault, false,
	epfn_create_face, _T("createFace"), 0, TYPE_INT, 0, 3,
		_T("vertexArray"), 0, TYPE_INDEX_BP,
		_T("faceDegree"), 0, TYPE_INT,
		_T("select"), 0, TYPE_bool, f_keyArgDefault, false,
	epfn_cap_holes, _T("capHoles"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_delete, _T("delete"), 0, TYPE_bool, 0, 3,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
		_T("deleteIsoVerts"), 0, TYPE_bool, f_keyArgDefault, true,
	//epfn_attach, _T("attach"), 0, TYPE_VOID, 0, 3,
		//_T("nodePointer"), 0, TYPE_INODE,
		//_T("canUndo"), 0, TYPE_BOOL_BR,
		//_T("myNode"), 0, TYPE_INODE,
	//epfn_multi_attach, _T("multiAttach"), 0, TYPE_VOID, 0, 2,
		//_T("nodeTable"), 0, TYPE_INODE_TAB_BR,
		//_T("myNode"), 0, TYPE_INODE,
	epfn_detach_to_element, _T("detachToElement"), 0, TYPE_bool, 0, 3,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
		_T("keepOriginal"), 0, TYPE_bool, f_keyArgDefault, false,
	//epfn_detach_to_object, _T("detachToObject"), 0, TYPE_bool, 0, 5,
		//_T("newNodeName"), 0, TYPE_STRING,
		//_T("mnSelLevel"), 0, TYPE_INT,
		//_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
		//_T("keepOriginal"), 0, TYPE_bool, f_keyArgDefault, false,
		//_T("myNode"), 0, TYPE_INODE,
	epfn_split_edges, _T("splitEdges"), 0, TYPE_bool, 0, 1,
		_T("edgeFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_break_verts, _T("breakVerts"), 0, TYPE_bool, 0, 1,
		_T("vertFlags"), 0, TYPE_DWORD,
	epfn_divide_face, _T("divideFace"), 0, TYPE_INDEX, 0, 3,
		_T("faceID"), 0, TYPE_INDEX,
		_T("vertexCoefficients"), 0, TYPE_FLOAT_TAB_BR,
		_T("select"), 0, TYPE_bool, f_keyArgDefault, false,
	epfn_divide_edge, _T("divideEdge"), 0, TYPE_INDEX, 0, 3,
		_T("edgeID"), 0, TYPE_INDEX,
		_T("proportion"), 0, TYPE_FLOAT,
		_T("select"), 0, TYPE_bool, f_keyArgDefault, false,
	epfn_collapse, _T("collapse"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_extrude_faces, _T("extrudeFaces"), 0, TYPE_VOID, 0, 2,
		_T("amount"), 0, TYPE_FLOAT,
		_T("faceFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_bevel_faces, _T("bevelFaces"), 0, TYPE_VOID, 0, 3,
		_T("height"), 0, TYPE_FLOAT,
		_T("outline"), 0, TYPE_FLOAT,
		_T("faceFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_chamfer_vertices, _T("chamferVertices"), 0, TYPE_VOID, 0, 1,
		_T("amount"), 0, TYPE_FLOAT,
	epfn_chamfer_edges, _T("chamferEdges"), 0, TYPE_VOID, 0, 1,
		_T("amount"), 0, TYPE_FLOAT,
	epfn_slice, _T("slice"), 0, TYPE_bool, 0, 4,
		_T("slicePlaneNormal"), 0, TYPE_POINT3,
		_T("slicePlaneCenter"), 0, TYPE_POINT3,
		_T("flaggedFacesOnly"), 0, TYPE_bool, f_keyArgDefault, false,
		_T("faceFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_in_slice_plane_mode, _T("inSlicePlaneMode"), 0, TYPE_bool, 0, 0,
	epfn_cut_vertex, _T("cutVertices"), 0, TYPE_INDEX, 0, 3,
		_T("startVertex"), 0, TYPE_INDEX,
		_T("endPosition"), 0, TYPE_POINT3,
		_T("viewDirection"), 0, TYPE_POINT3,
	epfn_cut_edge, _T("cutEdges"), 0, TYPE_INDEX, 0, 5,
		_T("startEdge"), 0, TYPE_INDEX,
		_T("startProportion"), 0, TYPE_FLOAT,
		_T("endEdge"), 0, TYPE_INDEX,
		_T("endProportion"), 0, TYPE_FLOAT,
		_T("viewDirection"), 0, TYPE_POINT3,
	epfn_cut_face, _T("cutFaces"), 0, TYPE_INDEX, 0, 4,
		_T("startFace"), 0, TYPE_INDEX,
		_T("startPosition"), 0, TYPE_POINT3,
		_T("endPosition"), 0, TYPE_POINT3,
		_T("viewDirection"), 0, TYPE_POINT3,
	epfn_weld_verts, _T("weldVerts"), 0, TYPE_bool, 0, 3,
		_T("vertex1"), 0, TYPE_INDEX,
		_T("vertex2"), 0, TYPE_INDEX,
		_T("destinationPoint"), 0, TYPE_POINT3,
	epfn_weld_edges, _T("weldEdges"), 0, TYPE_bool, 0, 2,
		_T("edge1"), 0, TYPE_INDEX,
		_T("edge2"), 0, TYPE_INDEX,
	epfn_weld_flagged_verts, _T("weldFlaggedVertices"), 0, TYPE_bool, 0, 1,
		_T("vertexFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_weld_flagged_edges, _T("weldFlaggedEdges"), 0, TYPE_bool, 0, 1,
		_T("edgeFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_create_shape, _T("createShape"), 0, TYPE_bool, 0, 4,
		_T("shapeName"), 0, TYPE_STRING,
		_T("curved"), 0, TYPE_bool,
		_T("myNode"), 0, TYPE_INODE,
		_T("edgeFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_make_planar, _T("makePlanar"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_move_to_plane, _T("moveToPlane"), 0, TYPE_bool, 0, 4,
		_T("planeNormal"), 0, TYPE_POINT3,
		_T("planeOffset"), 0, TYPE_FLOAT,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_align_to_grid, _T("alignToGrid"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_align_to_view, _T("alignToView"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_delete_iso_verts, _T("deleteIsoVerts"), 0, TYPE_bool, 0, 0,
	epfn_meshsmooth, _T("meshSmooth"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_tessellate, _T("tessellate"), 0, TYPE_bool, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("flags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_force_subdivision, _T("forceSubdivision"), 0, TYPE_VOID, 0, 0,
	epfn_set_diagonal, _T("setDiagonal"), 0, TYPE_VOID, 0, 3,
		_T("face"), 0, TYPE_INDEX,
		_T("corner1"), 0, TYPE_INDEX,
		_T("corner2"), 0, TYPE_INDEX,
	epfn_retriangulate, _T("retriangulate"), 0, TYPE_bool, 0, 1,
		_T("faceFlags"), 0, TYPE_DWORD,
	epfn_flip_normals, _T("flipNormals"), 0, TYPE_bool, 0, 1,
		_T("faceFlags"), 0, TYPE_DWORD,
	epfn_select_by_mat, _T("selectByMaterial"), 0, TYPE_VOID, 0, 2,
		_T("materialID"), 0, TYPE_INDEX,
		_T("clearCurrentSelection"), 0, TYPE_bool, f_keyArgDefault, true,
	epfn_select_by_smooth_group, _T("selectBySmoothGroup"), 0, TYPE_VOID, 0, 2,
		_T("smoothingGroups"), 0, TYPE_DWORD,
		_T("clearCurrentSelection"), 0, TYPE_bool, f_keyArgDefault, true,
	epfn_autosmooth, _T("autosmooth"), 0, TYPE_VOID, 0, 0,
	epfn_button_op, _T("buttonOp"), 0, TYPE_VOID, 0, 1,
		_T("buttonOpID"), 0, TYPE_ENUM, epolyEnumButtonOps,
	epfn_toggle_command_mode, _T("toggleCommandMode"), 0, TYPE_VOID, 0, 1,
		_T("commandModeID"), 0, TYPE_ENUM, epolyEnumCommandModes,
	epfn_enter_pick_mode, _T("enterPickMode"), 0, TYPE_VOID, 0, 1,
		_T("pickModeID"), 0, TYPE_ENUM, epolyEnumPickModes,
	epfn_exit_command_modes, _T("exitCommandModes"), 0, TYPE_VOID, 0, 0,

	// Flag Accessor methods:
	epfn_get_vertices_by_flag, _T("getVerticesByFlag"), 0, TYPE_bool, 0, 3,
		_T("vertexSet"), 0, TYPE_BITARRAY_BR,
		_T("flagsRequested"), 0, TYPE_DWORD,
		_T("flagMask"), 0, TYPE_DWORD, f_keyArgDefault, 0,
	epfn_get_edges_by_flag, _T("getEdgesByFlag"), 0, TYPE_bool, 0, 3,
		_T("edgeSet"), 0, TYPE_BITARRAY_BR,
		_T("flagsRequested"), 0, TYPE_DWORD,
		_T("flagMask"), 0, TYPE_DWORD, f_keyArgDefault, 0,
	epfn_get_faces_by_flag, _T("getFacesByFlag"), 0, TYPE_bool, 0, 3,
		_T("faceSet"), 0, TYPE_BITARRAY_BR,
		_T("flagsRequested"), 0, TYPE_DWORD,
		_T("flagMask"), 0, TYPE_DWORD, f_keyArgDefault, 0,

	epfn_set_vertex_flags, _T("setVertexFlags"), 0, TYPE_VOID, 0, 4,
		_T("vertexSet"), 0, TYPE_BITARRAY_BR,
		_T("flagsToSet"), 0, TYPE_DWORD,
		_T("flagMask"), 0, TYPE_DWORD, f_keyArgDefault, 0,
		_T("generateUndoRecord"), 0, TYPE_bool, f_keyArgDefault, true,
	epfn_set_edge_flags, _T("setEdgeFlags"), 0, TYPE_VOID, 0, 4,
		_T("edgeSet"), 0, TYPE_BITARRAY_BR,
		_T("flagsToSet"), 0, TYPE_DWORD,
		_T("flagMask"), 0, TYPE_DWORD, f_keyArgDefault, 0,
		_T("generateUndoRecord"), 0, TYPE_bool, f_keyArgDefault, true,
	epfn_set_face_flags, _T("setFaceFlags"), 0, TYPE_VOID, 0, 4,
		_T("faceSet"), 0, TYPE_BITARRAY_BR,
		_T("flagsToSet"), 0, TYPE_DWORD,
		_T("flagMask"), 0, TYPE_DWORD, f_keyArgDefault, 0,
		_T("generateUndoRecord"), 0, TYPE_bool, f_keyArgDefault, true,

	// Data accessor methods:
	epfn_reset_slice_plane, _T("resetSlicePlane"), 0, TYPE_VOID, 0, 0,
	epfn_get_slice_plane, _T("getSlicePlane"), 0, TYPE_VOID, 0, 3,
		_T("planeNormal"), 0, TYPE_POINT3_BR,
		_T("planeCenter"), 0, TYPE_POINT3_BR,
		_T("planeSize"), 0, TYPE_FLOAT_BP,
	epfn_set_slice_plane, _T("setSlicePlane"), 0, TYPE_VOID, 0, 3,
		_T("planeNormal"), 0, TYPE_POINT3_BR,
		_T("planeCenter"), 0, TYPE_POINT3_BR,
		_T("planeSize"), 0, TYPE_FLOAT,
	epfn_get_vertex_data, _T("getVertexData"), 0, TYPE_FLOAT, 0, 4,
		_T("vertexDataChannel"), 0, TYPE_INT,
		_T("numberSelected"), 0, TYPE_INT_BP,
		_T("uniformData"), 0, TYPE_bool_BP,
		_T("vertexFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_get_edge_data, _T("getEdgeData"), 0, TYPE_FLOAT, 0, 4,
		_T("edgeDataChannel"), 0, TYPE_INT,
		_T("numberSelected"), 0, TYPE_INT_BP,
		_T("uniformData"), 0, TYPE_bool_BP,
		_T("edgeFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_set_vertex_data, _T("setVertexData"), 0, TYPE_VOID, 0, 3,
		_T("vertexDataChannel"), 0, TYPE_INT,
		_T("value"), 0, TYPE_FLOAT,
		_T("vertexFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_set_edge_data, _T("setEdgeData"), 0, TYPE_VOID, 0, 3,
		_T("edgeDataChannel"), 0, TYPE_INT,
		_T("value"), 0, TYPE_FLOAT,
		_T("edgeFlags"), 0, TYPE_DWORD, f_keyArgDefault, MN_SEL,
	epfn_reset_vertex_data, _T("resetVertexData"), 0, TYPE_VOID, 0, 1,
		_T("vertexDataChannel"), 0, TYPE_INT,
	epfn_reset_edge_data, _T("resetEdgeData"), 0, TYPE_VOID, 0, 1,
		_T("edgeDataChannel"), 0, TYPE_INT,
	epfn_begin_modify_perdata, _T("beginModifyPerData"), 0, TYPE_VOID, 0, 2,
		_T("mnSelLevel"), 0, TYPE_INT,
		_T("dataChannel"), 0, TYPE_INT,
	epfn_in_modify_perdata, _T("inModifyPerData"), 0, TYPE_bool, 0, 0,
	epfn_end_modify_perdata, _T("endModifyPerData"), 0, TYPE_VOID, 0, 1,
		_T("success"), 0, TYPE_bool,
	epfn_get_mat_index, _T("getMaterialIndex"), 0, TYPE_INT, 0, 1,
		_T("determined"), 0, TYPE_bool_BP,
	epfn_set_mat_index, _T("setMaterialIndex"), 0, TYPE_VOID, 0, 2,
		_T("index"), 0, TYPE_INT,
		_T("faceFlags"), 0, TYPE_DWORD,
	epfn_get_smoothing_groups, _T("getSmoothingGroups"), 0, TYPE_VOID, 0, 3,
		_T("faceFlag"), 0, TYPE_DWORD,
		_T("anyFaces"), 0, TYPE_DWORD_BP,
		_T("allFaces"), 0, TYPE_DWORD_BP,
	epfn_set_smoothing_groups, _T("setSmoothingGroups"), 0, TYPE_VOID, 0, 3,
		_T("bitValues"), 0, TYPE_DWORD,
		_T("bitMask"), 0, TYPE_DWORD, 
		_T("faceFlags"), 0, TYPE_DWORD,

	epfn_collapse_dead_structs, _T("collapeDeadStrctures"), 0, TYPE_VOID, 0, 0,
	epfn_propagate_component_flags, _T("propogateComponentFlags"), 0, TYPE_INT, 0, 7,
		_T("mnSelLevelTo"), 0, TYPE_INT,
		_T("flagSetTo"), 0, TYPE_DWORD,
		_T("mnSelLevelFrom"), 0, TYPE_INT,
		_T("flagTestFrom"), 0, TYPE_DWORD,
		_T("allBitsMustMatch"), 0, TYPE_bool, f_keyArgDefault, false,
		_T("set"), 0, TYPE_bool, f_keyArgDefault, true,
		_T("undoable"), 0, TYPE_bool, f_keyArgDefault, false,

	enums,
		epolyEnumButtonOps, 29,
			_T("HideSelection"), epop_hide,
			_T("UnhideAll"), epop_unhide,
			_T("NamedSelectionCopy"), epop_ns_copy,
			_T("NamedSelectionPaste"), epop_ns_paste,
			_T("Cap"), epop_cap,
			_T("Delete"), epop_delete,
			_T("Detach"), epop_detach,
			_T("AttachList"), epop_attach_list,
			_T("SplitEdges"), epop_split,
			_T("BreakVertex"), epop_break,
			_T("Collapse"), epop_collapse,
			_T("ResetSlicePlane"), epop_reset_plane,
			_T("Slice"), epop_slice,
			_T("WeldSelected"), epop_weld_sel,
			_T("CreateShape"), epop_create_shape,
			_T("MakePlanar"), epop_make_planar,
			_T("AlignGrid"), epop_align_grid,
			_T("AlignView"), epop_align_view,
			_T("RemoveIsoVerts"), epop_remove_iso_verts,
			_T("MeshSmooth"), epop_meshsmooth,
			_T("Tessellate"), epop_tessellate,
			_T("Update"), epop_update,
			_T("SelectByVertexColor"), epop_selby_vc,
			_T("Retriangulate"), epop_retriangulate,
			_T("FlipNormals"), epop_flip_normals,
			_T("SelectByMatID"), epop_selby_matid,
			_T("SelectBySmoothingGroups"), epop_selby_smg,
			_T("Autosmooth"), epop_autosmooth,
			_T("ClearSmoothingGroups"), epop_clear_smg,

		epolyEnumCommandModes, 15,
			_T("CreateVertex"), epmode_create_vertex,
			_T("CreateEdge"), epmode_create_edge,
			_T("CreateFace"), epmode_create_face,
			_T("DivideEdge"), epmode_divide_edge,
			_T("DivideFace"), epmode_divide_face,
			//_T("ExtrudeVertex"), epmode_extrude_vertex,
			//_T("ExtrudeEdge"), epmode_extrude_edge,
			_T("ExtrudeFace"), epmode_extrude_face,
			_T("ChamferVertex"), epmode_chamfer_vertex,
			_T("ChamferEdge"), epmode_chamfer_edge,
			_T("Bevel"), epmode_bevel, 
			_T("SlicePlane"), epmode_sliceplane,
			_T("CutVertex"), epmode_cut_vertex,
			_T("CutEdge"), epmode_cut_edge,
			_T("CutFace"), epmode_cut_face,
			_T("Weld"), epmode_weld,
			_T("EditTriangulation"), epmode_edit_tri,

		epolyEnumPickModes, 1,
			_T("Attach"),	epmode_attach, 

	end
);

FPInterfaceDesc *::EPoly::GetDesc () {
	return &epfi;
}

void *EditablePolyObjectClassDesc::Create (BOOL loading) {
	AddInterface (&epfi);
	return new EditPoly::EditPolyObject;
}

namespace EditPoly
{

static INT_PTR CALLBACK FaceFlagsDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
static INT_PTR CALLBACK SelectByFlagsDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
static INT_PTR CALLBACK ExtFaceFlagsDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);
static INT_PTR CALLBACK FacePassDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

static BOOL bLockPassUpdate = FALSE;

// Subobject levels for new (4.0) interface:
static GenSubObjType SOT_Vertex(1);
static GenSubObjType SOT_Edge(2);
static GenSubObjType SOT_Border(9);
static GenSubObjType SOT_Poly(4);
static GenSubObjType SOT_Element(5);

// Parameter block 2 contains all Editable parameters:
 
static ParamBlockDesc2 ep_param_block ( ep_block, _T("ePolyParameters"),
									   0, GetEditablePolyDesc(),
									   P_AUTO_CONSTRUCT|P_AUTO_UI|P_MULTIMAP,
									   0,
    // map rollups
	9,	ep_select, IDD_EP_SELECT, IDS_SELECT, 0, 0, NULL,
		ep_softsel, IDD_EP_SOFTSEL, IDS_SOFTSEL, 0, APPENDROLL_CLOSED, NULL,
		ep_geom, IDD_EP_GEOM, IDS_EDIT_GEOM, 0, 0, NULL,
		ep_subdiv, IDD_EP_SUBDIV, IDS_SUBDIVISION, 0, APPENDROLL_CLOSED, NULL,
		ep_surface, IDD_EP_SURF_OBJECT, IDS_SURFACE_PROPERTIES, 0, 0, NULL,
		ep_vertex, IDD_EP_SURF_VERT, IDS_SURFACE_PROPERTIES, 0, 0, NULL,
		ep_edge, IDD_EP_SURF_EDGE, IDS_SURFACE_PROPERTIES, 0, 0, NULL,
		ep_face, IDD_EP_SURF_FACE, IDS_SURFACE_PROPERTIES, 0, 0, NULL,
		ep_displacement, IDD_EP_DISP_APPROX, IDS_DISPLACEMENT, 0, 0, NULL,

	ep_by_vertex, _T("selByVertex"), TYPE_BOOL, P_TRANSIENT, IDS_SEL_BY_VERTEX,
		p_default, FALSE,
		p_ui, ep_select, TYPE_SINGLECHEKBOX, IDC_SEL_BYVERT,
		end,

	ep_ignore_backfacing, _T("ignoreBackfacing"), TYPE_BOOL, P_TRANSIENT, IDS_IGNORE_BACKFACING,
		p_default, FALSE,
		p_ui, ep_select, TYPE_SINGLECHEKBOX, IDC_IGNORE_BACKFACES,
		end,

	ep_ss_use, _T("useSoftSel"), TYPE_BOOL, 0, IDS_USE_SOFTSEL,
		p_default, FALSE,
		p_enable_ctrls, 4, ep_ss_affect_back,
			ep_ss_falloff, ep_ss_pinch, ep_ss_bubble,
		p_ui, ep_softsel, TYPE_SINGLECHEKBOX, IDC_USE_SOFTSEL,
		end,

	ep_ss_edist_use, _T("ssUseEdgeDist"), TYPE_BOOL, 0, IDS_USE_EDIST,
		p_default, FALSE,
		p_enable_ctrls, 1, ep_ss_edist,
		p_ui, ep_softsel, TYPE_SINGLECHEKBOX, IDC_USE_EDIST,
		end,

	ep_ss_edist, _T("ssEdgeDist"), TYPE_INT, 0, IDS_EDGE_DIST,
		p_default, 1,
		p_range, 1, 9999999,
		p_ui, ep_softsel, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_EDIST, IDC_EDIST_SPIN, .2f,
		end,

	ep_ss_affect_back, _T("affectBackfacing"), TYPE_BOOL, 0, IDS_AFFECT_BACKFACING,
		p_default, TRUE,
		p_ui, ep_softsel, TYPE_SINGLECHEKBOX, IDC_AFFECT_BACKFACING,
		end,

	ep_ss_falloff, _T("falloff"), TYPE_WORLD, P_ANIMATABLE, IDS_FALLOFF,
		p_default, 20.0f,
		p_range, 0.0f, 999999.f,
		p_ui, ep_softsel, TYPE_SPINNER, EDITTYPE_POS_UNIVERSE,
			IDC_FALLOFF, IDC_FALLOFFSPIN, SPIN_AUTOSCALE,
		end,

	ep_ss_pinch, _T("pinch"), TYPE_WORLD, P_ANIMATABLE, IDS_PINCH,
		p_default, 0.0f,
		p_ui, ep_softsel, TYPE_SPINNER, EDITTYPE_UNIVERSE,
			IDC_PINCH, IDC_PINCHSPIN, SPIN_AUTOSCALE,
		end,

	ep_ss_bubble, _T("bubble"), TYPE_WORLD, P_ANIMATABLE, IDS_BUBBLE,
		p_default, 0.0f,
		p_ui, ep_softsel, TYPE_SPINNER, EDITTYPE_UNIVERSE,
			IDC_BUBBLE, IDC_BUBBLESPIN, SPIN_AUTOSCALE,
		end,

	ep_extrusion_type, _T("extrusionType"), TYPE_INT, P_TRANSIENT, IDS_EXTRUSION_TYPE,
		p_default, 0,	// Group normal
		p_ui, ep_geom, TYPE_RADIO, 3, IDC_EXTYPE_GROUP, IDC_EXTYPE_LOCAL, IDC_EXTYPE_BY_POLY,
		end,

	ep_split, _T("split"), TYPE_BOOL, P_TRANSIENT, IDS_SPLIT,
		p_default, FALSE,
		p_ui, ep_geom, TYPE_SINGLECHEKBOX, IDC_SPLIT,
		end,

	ep_weld_threshold, _T("weldThreshold"), TYPE_WORLD, P_TRANSIENT, IDS_WELD_THRESHOLD,
		p_default, .1f,
		p_range, 0.0f, 9999999.0f,
		p_ui, ep_geom, TYPE_SPINNER, EDITTYPE_POS_UNIVERSE,
			IDC_W_THR, IDC_W_THR_SPIN, SPIN_AUTOSCALE,
		end,

	ep_weld_pixels, _T("weldPixels"), TYPE_INT, P_TRANSIENT, IDS_WELD_PIXELS,
		p_default, 4,
		p_ui, ep_geom, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_T_THR, IDC_T_THR_SPIN, .5f,
		end,

	ep_ms_smoothness, _T("smoothness"), TYPE_FLOAT, P_TRANSIENT, IDS_MS_SMOOTHNESS,
		p_default, 1.0f,
		p_range, 0.0f, 1.0f,
		p_ui, ep_subdiv, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_MS_SMOOTH, IDC_MS_SMOOTHSPIN, .001f,
		end,

	ep_ms_sep_smooth, _T("separateBySmoothing"), TYPE_BOOL, P_TRANSIENT, IDS_MS_SEP_BY_SMOOTH,
		p_default, FALSE,
		p_ui, ep_subdiv, TYPE_SINGLECHEKBOX, IDC_MS_SEP_BY_SMOOTH,
		end,

	ep_ms_sep_mat, _T("separateByMaterial"), TYPE_BOOL, P_TRANSIENT, IDS_MS_SEP_BY_MATID,
		p_default, FALSE,
		p_ui, ep_subdiv, TYPE_SINGLECHEKBOX, IDC_MS_SEP_BY_MATID,
		end,

	ep_tess_type, _T("tesselateBy"), TYPE_INT, P_TRANSIENT, IDS_TESS_BY,
		p_default, 0,
		p_ui, ep_subdiv, TYPE_RADIO, 2, IDC_TESS_EDGE, IDC_TESS_FACE,
		end,

	ep_tess_tension, _T("tessTension"), TYPE_FLOAT, P_TRANSIENT, IDS_TESS_TENSION,
		p_default, 0.0f,
		p_ui, ep_subdiv, TYPE_SPINNER, EDITTYPE_FLOAT,
			IDC_TESS_TENSION, IDC_TESS_TENSIONSPIN, .01f,
		end,

	ep_surf_subdivide, _T("surfSubdivide"), TYPE_BOOL, 0, IDS_SUBDIVIDE,
		p_default, FALSE,
		p_ui, ep_surface, TYPE_SINGLECHEKBOX, IDC_USE_NURMS,
		end,

	ep_surf_subdiv_smooth, _T("subdivSmoothing"), TYPE_BOOL, 0, IDS_SMOOTH_SUBDIV,
		p_default, TRUE,
		p_ui, ep_surface, TYPE_SINGLECHEKBOX, IDC_SMOOTH_SUBDIV,
		end,

	ep_surf_iter, _T("iterations"), TYPE_INT, P_ANIMATABLE, IDS_ITER,
		p_default, 0,
		p_range, 0, 10,
		p_ui, ep_surface, TYPE_SPINNER, EDITTYPE_POS_INT, 
			IDC_ITER, IDC_ITERSPIN, .2f,
		end,

	ep_surf_thresh, _T("surfaceSmoothness"), TYPE_FLOAT, P_ANIMATABLE, IDS_SHARP,
		p_default, 1.0f,
		p_range, 0.0f, 1.0f,
		p_ui, ep_surface, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_SHARP, IDC_SHARPSPIN, .002f,
		end,

	ep_surf_use_riter, _T("useRenderIterations"), TYPE_BOOL, 0, IDS_USE_RITER,
		p_default, FALSE,
		p_ui, ep_surface, TYPE_SINGLECHEKBOX, IDC_USE_RITER,
		p_enable_ctrls, 1, ep_surf_riter,
		end,

	ep_surf_riter, _T("renderIterations"), TYPE_INT, P_ANIMATABLE, IDS_RITER,
		p_default, 0,
		p_range, 0, 10,
		p_ui, ep_surface, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_R_ITER, IDC_R_ITERSPIN, .2f,
		end,

	ep_surf_use_rthresh, _T("useRenderSmoothness"), TYPE_BOOL, 0, IDS_USE_RSHARP,
		p_default, FALSE,
		p_ui, ep_surface, TYPE_SINGLECHEKBOX, IDC_USE_RSHARP,
		p_enable_ctrls, 1, ep_surf_rthresh,
		end,

	ep_surf_rthresh, _T("renderSmoothness"), TYPE_FLOAT, P_ANIMATABLE, IDS_RSHARP,
		p_default, 1.0f,
		p_range, 0.0f, 1.0f,
		p_ui, ep_surface, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_R_SHARP, IDC_R_SHARPSPIN, .002f,
		end,

	ep_surf_sep_mat, _T("sepByMats"), TYPE_BOOL, 0, IDS_MS_SEP_BY_MATID,
		p_default, FALSE,
		p_ui, ep_surface, TYPE_SINGLECHEKBOX, IDC_MS_SEP_BY_MATID,
		end,

	ep_surf_sep_smooth, _T("sepBySmGroups"), TYPE_BOOL, 0, IDS_MS_SEP_BY_SMOOTH,
		p_default, FALSE,
		p_ui, ep_surface, TYPE_SINGLECHEKBOX, IDC_MS_SEP_BY_SMOOTH,
		end,

	ep_surf_update, _T("update"), TYPE_INT, 0, IDS_UPDATE_OPTIONS,
		p_default, 0, // Update Always.
		p_range, 0, 2,
		p_ui, ep_surface, TYPE_RADIO, 3,
			IDC_UPDATE_ALWAYS, IDC_UPDATE_RENDER, IDC_UPDATE_MANUAL,
		end,

	ep_vert_sel_color, _T("vertSelectionColor"), TYPE_RGBA, P_TRANSIENT, IDS_VERT_SELECTION_COLOR,
		p_default, Point3(1,1,1),
		p_ui, ep_vertex, TYPE_COLORSWATCH, IDC_VERT_SELCOLOR,
		end,

	ep_vert_color_selby, _T("vertSelectBy"), TYPE_INT, P_TRANSIENT, IDS_VERT_SELECT_BY,
		p_default, 0,
		p_ui, ep_vertex, TYPE_RADIO, 2, IDC_SEL_BY_COLOR, IDC_SEL_BY_ILLUM,
		end,

	ep_vert_selc_r, _T("vertSelectionRedRange"), TYPE_INT, P_TRANSIENT, IDS_VERT_SELECTION_RED_RANGE,
		p_default, 10,
		p_range, 0, 255,
		p_ui, ep_vertex, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_VERT_SELR, IDC_VERT_SELRSPIN, .5f,
		end,

	ep_vert_selc_g, _T("vertSelectionGreenRange"), TYPE_INT, P_TRANSIENT, IDS_VERT_SELECTION_GREEN_RANGE,
		p_default, 10,
		p_range, 0, 255,
		p_ui, ep_vertex, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_VERT_SELG, IDC_VERT_SELGSPIN, .5f,
		end,

	ep_vert_selc_b, _T("vertSelectionBlueRange"), TYPE_INT, P_TRANSIENT, IDS_VERT_SELECTION_BLUE_RANGE,
		p_default, 10,
		p_range, 0, 255,
		p_ui, ep_vertex, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_VERT_SELB, IDC_VERT_SELBSPIN, .5f,
		end,

	ep_face_smooth_thresh, _T("autoSmoothThreshold"), TYPE_ANGLE, P_TRANSIENT, IDS_FACE_AUTOSMOOTH_THRESH,
		p_default, PI/4.0f,	// 45 degrees.
		p_range, 0.0f, 180.0f,
		p_ui, ep_face, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_SMOOTH_THRESH, IDC_SMOOTH_THRESHSPIN, SPIN_AUTOSCALE,
		end,

	ep_sd_use, _T("useSubdivisionDisplacement"), TYPE_BOOL, 0, IDS_USE_DISPLACEMENT,
		p_default, false,
		p_ui, ep_displacement, TYPE_SINGLECHEKBOX, IDC_SD_ENGAGE,
		end,

	ep_sd_split_mesh, _T("displaceSplitMesh"), TYPE_BOOL, 0, IDS_SD_SPLITMESH,
		p_default, true,
		p_ui, ep_displacement, TYPE_SINGLECHEKBOX, IDC_SD_SPLITMESH,
		end,

	ep_sd_method, _T("displaceMethod"), TYPE_INT, 0, IDS_SD_METHOD,
		p_default, 3,
		p_ui, ep_displacement, TYPE_RADIO, 4, IDC_SD_TESS_REGULAR,
			IDC_SD_TESS_SPATIAL, IDC_SD_TESS_CURV, IDC_SD_TESS_LDA,
		end,

	ep_sd_tess_steps, _T("displaceSteps"), TYPE_INT, P_ANIMATABLE, IDS_SD_TESS_STEPS,
		p_default, 2,
		p_range, 0, 100,
		p_ui, ep_displacement, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_SD_TESS_U, IDC_SD_TESS_U_SPINNER, .5f,
		end,

	ep_sd_tess_edge, _T("displaceEdge"), TYPE_FLOAT, P_ANIMATABLE, IDS_SD_TESS_EDGE,
		p_default, 20.0f,
		p_range, 0.0f, 9999999.0f,
		p_ui, ep_displacement, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_SD_TESS_EDGE, IDC_SD_TESS_EDGE_SPINNER, .1f,
		end,

	ep_sd_tess_distance, _T("displaceDistance"), TYPE_FLOAT, P_ANIMATABLE, IDS_SD_TESS_DISTANCE,
		p_default, 20.0f,
		p_range, 0.0f, 9999999.0f,
		p_ui, ep_displacement, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_SD_TESS_DIST, IDC_SD_TESS_DIST_SPINNER, .1f,
		end,

	ep_sd_tess_angle, _T("displaceAngle"), TYPE_ANGLE, P_ANIMATABLE, IDS_SD_TESS_ANGLE,
		p_default, PI/18.0f,	// 10 degrees.
		p_range, 0.0f, 180.0f,
		p_ui, ep_displacement, TYPE_SPINNER, EDITTYPE_POS_FLOAT,
			IDC_SD_TESS_ANG, IDC_SD_TESS_ANG_SPINNER, .1f,
		end,

	ep_sd_view_dependent, _T("viewDependent"), TYPE_BOOL, 0, IDS_SD_VIEW_DEPENDENT,
		p_default, false,
		p_ui, ep_displacement, TYPE_SINGLECHEKBOX, IDC_SD_VIEW_DEP,
		end,

	ep_asd_style, _T("displaceStyle"), TYPE_INT, 0, IDS_SD_DISPLACE_STYLE,
		p_default, 0,
		p_ui, ep_advanced_displacement, TYPE_RADIO, 3, IDC_ASD_GRID,
			IDC_ASD_TREE, IDC_ASD_DELAUNAY,
		end,

	ep_asd_min_iters, _T("displaceMinLevels"), TYPE_INT, 0, IDS_SD_MIN_LEVELS,
		p_default, 0,
		p_range, 0, 2,
		p_ui, ep_advanced_displacement, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_ASD_MIN_ITERS, IDC_ASD_MIN_ITERS_SPIN, .5f,
		end,

	ep_asd_max_iters, _T("displaceMaxLevels"), TYPE_INT, 0, IDS_SD_MAX_LEVELS,
		p_default, 2,
		p_range, 0, 7,
		p_ui, ep_advanced_displacement, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_ASD_MAX_ITERS, IDC_ASD_MAX_ITERS_SPIN, .5f,
		end,

	ep_asd_max_tris, _T("displaceMaxTris"), TYPE_INT, 0, IDS_SD_MAX_TRIS,
		p_default, 20000,
		p_range, 0, 99999999,
		p_ui, ep_advanced_displacement, TYPE_SPINNER, EDITTYPE_POS_INT,
			IDC_ASD_MAX_TRIS, IDC_ASD_MAX_TRIS_SPIN, .5f,
		end,

	end
);

// -- Misc. Window procs ----------------------------------------

static int createCurveType   = IDC_CURVE_SMOOTH;

static INT_PTR CALLBACK CurveNameDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	static TSTR *name = NULL;

	switch (msg) {
	case WM_INITDIALOG:
		name = (TSTR*)lParam;
		SetWindowText (GetDlgItem(hWnd,IDC_CURVE_NAME), name->data());
		CenterWindow (hWnd,GetParent(hWnd));
		SendMessage(GetDlgItem(hWnd,IDC_CURVE_NAME), EM_SETSEL,0,-1);			
		CheckDlgButton(hWnd,createCurveType,TRUE);
		return FALSE;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
			name->Resize(GetWindowTextLength(GetDlgItem(hWnd,IDC_CURVE_NAME))+1);
			GetWindowText(GetDlgItem(hWnd,IDC_CURVE_NAME), name->data(), name->length()+1);
			if (IsDlgButtonChecked(hWnd,IDC_CURVE_SMOOTH)) createCurveType = IDC_CURVE_SMOOTH;
			else createCurveType = IDC_CURVE_LINEAR;
			EndDialog(hWnd,1);
			break;
		
		case IDCANCEL:
			EndDialog(hWnd,0);
			break;
		}
		break;

	default:
		return 0;
	}
	return 1;
}

bool GetCreateShapeName (Interface *ip, TSTR &name, bool &ccSmooth) {
	HWND hMax = ip->GetMAXHWnd();
	name = GetString(IDS_SHAPE);
	ip->MakeNameUnique (name);
	bool ret = DialogBoxParam (hInstance,
		MAKEINTRESOURCE(IDD_CREATECURVE),
		hMax, CurveNameDlgProc, (LPARAM)&name) ? true : false;
	ccSmooth = (createCurveType == IDC_CURVE_LINEAR) ? false : true;
	return ret;
}

static BOOL detachToElem = FALSE;
static BOOL detachAsClone = FALSE;

static void SetDetachNameState(HWND hWnd) {
	if (detachToElem) {
		EnableWindow(GetDlgItem(hWnd,IDC_DETACH_NAMELABEL),FALSE);
		EnableWindow(GetDlgItem(hWnd,IDC_DETACH_NAME),FALSE);
	} else {
		EnableWindow(GetDlgItem(hWnd,IDC_DETACH_NAMELABEL),TRUE);
		EnableWindow(GetDlgItem(hWnd,IDC_DETACH_NAME),TRUE);
	}
}

static INT_PTR CALLBACK DetachDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	static TSTR *name = NULL;

	switch (msg) {
	case WM_INITDIALOG:
		name = (TSTR*)lParam;
		SetWindowText(GetDlgItem(hWnd,IDC_DETACH_NAME), name->data());
		CenterWindow(hWnd,GetParent(hWnd));
		SendMessage(GetDlgItem(hWnd,IDC_DETACH_NAME), EM_SETSEL,0,-1);
		CheckDlgButton (hWnd, IDC_DETACH_ELEM, detachToElem);
		CheckDlgButton (hWnd, IDC_DETACH_CLONE, detachAsClone);
		if (detachToElem) SetFocus (GetDlgItem (hWnd, IDOK));
		else SetFocus (GetDlgItem (hWnd, IDC_DETACH_NAME));
		SetDetachNameState(hWnd);
		return FALSE;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
			name->Resize (GetWindowTextLength(GetDlgItem(hWnd,IDC_DETACH_NAME))+1);
			GetWindowText (GetDlgItem(hWnd,IDC_DETACH_NAME),
				name->data(), name->length()+1);
			EndDialog(hWnd,1);
			break;

		case IDC_DETACH_ELEM:
			detachToElem = IsDlgButtonChecked(hWnd,IDC_DETACH_ELEM);
			SetDetachNameState(hWnd);
			break;

		case IDC_DETACH_CLONE:
			detachAsClone = IsDlgButtonChecked (hWnd, IDC_DETACH_CLONE);
			break;

		case IDCANCEL:
			EndDialog(hWnd,0);
			break;
		}
		break;

	default:
		return 0;
	}
	return 1;
}

bool GetDetachObjectName (Interface *ip, TSTR &name, bool &elem, bool &clone) {
	HWND hMax = ip->GetMAXHWnd();
	name = GetString(IDS_OBJECT);
	ip->MakeNameUnique (name);
	if (DialogBoxParam(hInstance, MAKEINTRESOURCE(IDD_DETACH),
			hMax, DetachDlgProc, (LPARAM)&name)) {
		elem = detachToElem ? true : false;
		clone = detachAsClone ? true : false;
		return true;
	} else {
		return false;
	}
}

static int cloneTo = IDC_CLONE_ELEM;

static void SetCloneNameState(HWND hWnd) {
	switch (cloneTo) {
	case IDC_CLONE_ELEM:
		EnableWindow(GetDlgItem(hWnd,IDC_CLONE_NAME),FALSE);
		break;
	case IDC_CLONE_OBJ:
		EnableWindow(GetDlgItem(hWnd,IDC_CLONE_NAME),TRUE);
		break;
	}
}

static INT_PTR CALLBACK CloneDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	static TSTR *name = NULL;

	switch (msg) {
	case WM_INITDIALOG:
		name = (TSTR*)lParam;
		SetWindowText(GetDlgItem(hWnd,IDC_CLONE_NAME), name->data());
		CenterWindow(hWnd, GetParent(hWnd));

		CheckRadioButton (hWnd, IDC_CLONE_OBJ, IDC_CLONE_ELEM, cloneTo);
		if (cloneTo == IDC_CLONE_OBJ) {
			SetFocus(GetDlgItem(hWnd,IDC_CLONE_NAME));
			SendMessage(GetDlgItem(hWnd,IDC_CLONE_NAME), EM_SETSEL,0,-1);
		} else SetFocus (GetDlgItem (hWnd, IDOK));
		SetCloneNameState(hWnd);
		return FALSE;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
			name->Resize (GetWindowTextLength(GetDlgItem(hWnd,IDC_CLONE_NAME))+1);
			GetWindowText (GetDlgItem(hWnd,IDC_CLONE_NAME),
				name->data(), name->length()+1);
			EndDialog(hWnd,1);
			break;

		case IDCANCEL:
			EndDialog (hWnd, 0);
			break;

		case IDC_CLONE_ELEM:
		case IDC_CLONE_OBJ:
			cloneTo = LOWORD(wParam);
			SetCloneNameState(hWnd);
			break;
		}
		break;

	default:
		return 0;
	}
	return 1;
}

BOOL GetCloneObjectName (Interface *ip, TSTR &name) {
	HWND hMax = ip->GetMAXHWnd();
	name = GetString(IDS_OBJECT);
	if (ip) ip->MakeNameUnique (name);
	DialogBoxParam (hInstance, MAKEINTRESOURCE(IDD_CLONE), hMax, CloneDlgProc, (LPARAM)&name);
	return (cloneTo==IDC_CLONE_OBJ);
}

class SelectDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	bool updateNumSel;

	SelectDlgProc () { epol = NULL; updateNumSel = true; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void DeleteThis() { }
	void RefreshSelType (HWND hWnd);
	void SetEnables (HWND hWnd);
	void SetNumSelLabel (HWND hWnd);
};

static SelectDlgProc theSelectDlgProc;

static int butIDs[] = { 0, IDC_SELVERTEX, IDC_SELEDGE,
	IDC_SELBORDER, IDC_SELFACE, IDC_SELELEMENT };

void SelectDlgProc::RefreshSelType (HWND hWnd) {
	ICustToolbar *iToolbar = GetICustToolbar (GetDlgItem (hWnd, IDC_SELTYPE));
	if (iToolbar) {
		ICustButton  *but;
		for (int i=1; i<6; i++) {
			but = iToolbar->GetICustButton (butIDs[i]);
			if (!but) continue;
			but->SetCheck (epol->GetEPolySelLevel()==i);
			ReleaseICustButton (but);
		}
	}
	ReleaseICustToolbar(iToolbar);
}

void SelectDlgProc::SetEnables (HWND hSel) {
	bool fac = (meshSelLevel[epol->GetEPolySelLevel()] == MNM_SL_FACE);
	bool edg = (meshSelLevel[epol->GetEPolySelLevel()] == MNM_SL_EDGE);
	bool obj = (epol->GetEPolySelLevel() == EP_SL_OBJECT);
	bool vtx = (epol->GetEPolySelLevel() == EP_SL_VERTEX);

	EnableWindow (GetDlgItem (hSel, IDC_SEL_BYVERT), fac||edg);
	EnableWindow (GetDlgItem (hSel, IDC_IGNORE_BACKFACES), !obj);

	ICustButton *but = GetICustButton (GetDlgItem (hSel, IDC_HIDE));
	if (but) {
		but->Enable (vtx||fac);
		ReleaseICustButton (but);
	}

	but = GetICustButton (GetDlgItem (hSel, IDC_UNHIDEALL));
	if (but) {
		but->Enable (vtx||fac);
		ReleaseICustButton (but);
	}

	EnableWindow (GetDlgItem (hSel, IDC_NAMEDSEL_LABEL), !obj);
	if (but = GetICustButton (GetDlgItem (hSel, IDC_COPY_NS))) {
		but->Enable (!obj);
		ReleaseICustButton(but);
	}
	if (but = GetICustButton (GetDlgItem (hSel, IDC_PASTE_NS))) {
		but->Enable (!obj && (GetMeshNamedSelClip (namedClipLevel[epol->GetEPolySelLevel()])));
		ReleaseICustButton(but);
	}
}

void SelectDlgProc::SetNumSelLabel (HWND hWnd) {	
	static TSTR buf;
	if (!updateNumSel) {
		SetDlgItemText (hWnd, IDC_NUMSEL_LABEL, buf);
		return;
	}
	updateNumSel = false;

	int num, j, lastsel;
	switch (epol->GetEPolySelLevel()) {
	case EP_SL_OBJECT:
		buf.printf (GetString (IDS_OBJECT_SEL));
		break;

	case EP_SL_VERTEX:
		num=0;
		for (j=0; j<epol->GetMeshPtr()->numv; j++) {
			if (!epol->GetMeshPtr()->v[j].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
			num++;
			lastsel=j;
		}
		if (num==1) {
			buf.printf (GetString(IDS_WHICHVERTSEL), lastsel+1);
		} else buf.printf (GetString(IDS_NUMVERTSELP), num);
		break;

	case EP_SL_FACE:
	case EP_SL_ELEMENT:
		num=0;
		for (j=0; j<epol->GetMeshPtr()->numf; j++) {
			if (!epol->GetMeshPtr()->f[j].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
			num++;
			lastsel=j;
		}
		if (num==1) {
			buf.printf (GetString(IDS_WHICHFACESEL), lastsel+1);
		} else buf.printf(GetString(IDS_NUMFACESELP),num);
		break;

	case EP_SL_EDGE:
	case EP_SL_BORDER:
		num=0;
		for (j=0; j<epol->GetMeshPtr()->nume; j++) {
			if (!epol->GetMeshPtr()->e[j].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
			num++;
			lastsel=j;
		}
		if (num==1) {
			buf.printf (GetString(IDS_WHICHEDGESEL), lastsel+1);
		} else buf.printf(GetString(IDS_NUMEDGESELP),num);
		break;
	}

	SetDlgItemText (hWnd, IDC_NUMSEL_LABEL, buf);
}

BOOL SelectDlgProc::DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
							 UINT msg, WPARAM wParam, LPARAM lParam) {
	ICustToolbar *iToolbar;

	switch (msg) {
	case WM_INITDIALOG:
		iToolbar = GetICustToolbar(GetDlgItem(hWnd,IDC_SELTYPE));
		iToolbar->SetImage (GetPolySelImageHandler()->LoadImages());
		iToolbar->AddTool(ToolButtonItem(CTB_CHECKBUTTON,0,5,0,5,24,23,24,23,IDC_SELVERTEX));
		iToolbar->AddTool(ToolButtonItem(CTB_CHECKBUTTON,1,6,1,6,24,23,24,23,IDC_SELEDGE));
		iToolbar->AddTool(ToolButtonItem(CTB_CHECKBUTTON,2,7,2,7,24,23,24,23,IDC_SELBORDER));
		iToolbar->AddTool(ToolButtonItem(CTB_CHECKBUTTON,3,8,3,8,24,23,24,23,IDC_SELFACE));
		iToolbar->AddTool(ToolButtonItem(CTB_CHECKBUTTON,4,9,4,9,24,23,24,23,IDC_SELELEMENT));
		ReleaseICustToolbar(iToolbar);
		RefreshSelType (hWnd);
		SetEnables (hWnd);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_SELVERTEX:
			if (epol->GetEPolySelLevel() == EP_SL_VERTEX)
				epol->SetEPolySelLevel (EP_SL_OBJECT);
			else epol->SetEPolySelLevel (EP_SL_VERTEX);
			break;

		case IDC_SELEDGE:
			if (epol->GetEPolySelLevel() == EP_SL_EDGE)
				epol->SetEPolySelLevel (EP_SL_OBJECT);
			else epol->SetEPolySelLevel (EP_SL_EDGE);
			break;

		case IDC_SELBORDER:
			if (epol->GetEPolySelLevel() == EP_SL_BORDER)
				epol->SetEPolySelLevel (EP_SL_OBJECT);
			else epol->SetEPolySelLevel (EP_SL_BORDER);
			break;

		case IDC_SELFACE:
			if (epol->GetEPolySelLevel() == EP_SL_FACE)
				epol->SetEPolySelLevel (EP_SL_OBJECT);
			else epol->SetEPolySelLevel (EP_SL_FACE);
			break;

		case IDC_SELELEMENT:
			if (epol->GetEPolySelLevel() == EP_SL_ELEMENT)
				epol->SetEPolySelLevel (EP_SL_OBJECT);
			else epol->SetEPolySelLevel (EP_SL_ELEMENT);
			break;

		case IDC_HIDE: epol->EpActionButtonOp (epop_hide); break;
		case IDC_UNHIDEALL: epol->EpActionButtonOp (epop_unhide); break;
		case IDC_COPY_NS: epol->EpActionButtonOp (epop_ns_copy); break;
		case IDC_PASTE_NS: epol->EpActionButtonOp (epop_ns_paste); break;
		}
		break;

	case WM_PAINT:
		if (updateNumSel) SetNumSelLabel (hWnd);
		return FALSE;

	case WM_NOTIFY:
		if(((LPNMHDR)lParam)->code != TTN_NEEDTEXT) break;
		LPTOOLTIPTEXT lpttt;
		lpttt = (LPTOOLTIPTEXT)lParam;				
		switch (lpttt->hdr.idFrom) {
		case IDC_SELVERTEX:
			lpttt->lpszText = GetString (IDS_VERTEX);
			break;
		case IDC_SELEDGE:
			lpttt->lpszText = GetString (IDS_EDGE);
			break;
		case IDC_SELBORDER:
			lpttt->lpszText = GetString(IDS_BORDER);
			break;
		case IDC_SELFACE:
			lpttt->lpszText = GetString(IDS_FACE);
			break;
		case IDC_SELELEMENT:
			lpttt->lpszText = GetString(IDS_ELEMENT);
			break;
		}
		break;

	default:
		return FALSE;
	}
	return TRUE;
}

#define GRAPHSTEPS 20

class SoftselDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	SoftselDlgProc () { epol = NULL; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void DrawCurve (TimeValue t, HWND hWnd, HDC hdc);
	void DeleteThis() { }
	void SetEnables (HWND hSS);
};

static SoftselDlgProc theSoftselDlgProc;

void SoftselDlgProc::DrawCurve (TimeValue t, HWND hWnd,HDC hdc) {
	float pinch, falloff, bubble;
	IParamBlock2 *pblock = epol->getParamBlock ();
	if (!pblock) return;

	pblock->GetValue (ep_ss_falloff, t, falloff, FOREVER);
	pblock->GetValue (ep_ss_pinch, t, pinch, FOREVER);
	pblock->GetValue (ep_ss_bubble, t, bubble, FOREVER);

	TSTR label = FormatUniverseValue(falloff);
	SetWindowText(GetDlgItem(hWnd,IDC_FARLEFTLABEL), label);
	SetWindowText(GetDlgItem(hWnd,IDC_NEARLABEL), FormatUniverseValue (0.0f));
	SetWindowText(GetDlgItem(hWnd,IDC_FARRIGHTLABEL), label);

	Rect rect, orect;
	GetClientRectP(GetDlgItem(hWnd,IDC_SS_GRAPH),&rect);
	orect = rect;

	SelectObject(hdc,GetStockObject(NULL_PEN));
	SelectObject(hdc,GetStockObject(WHITE_BRUSH));
	Rectangle(hdc,rect.left,rect.top,rect.right,rect.bottom);	
	SelectObject(hdc,GetStockObject(NULL_BRUSH));
	
	rect.left   += 3;
	rect.right  -= 3;
	rect.top    += 20;
	rect.bottom -= 20;
	
	SelectObject(hdc,CreatePen(PS_DOT,0,GetSysColor(COLOR_BTNFACE)));
	MoveToEx(hdc,orect.left,rect.top,NULL);
	LineTo(hdc,orect.right,rect.top);
	MoveToEx(hdc,orect.left,rect.bottom,NULL);
	LineTo(hdc,orect.right,rect.bottom);
	MoveToEx(hdc,(rect.left+rect.right)/2,orect.top,NULL);
	LineTo(hdc,(rect.left+rect.right)/2,orect.bottom);
	DeleteObject(SelectObject(hdc,GetStockObject(BLACK_PEN)));
	
	MoveToEx(hdc,rect.left,rect.bottom,NULL);
	for (int i=0; i<=GRAPHSTEPS; i++) {
		float dist = falloff * float(abs(i-GRAPHSTEPS/2))/float(GRAPHSTEPS/2);		
		float y = AffectRegionFunction(dist,falloff,pinch,bubble);
		int ix = rect.left + int(float(rect.w()-1) * float(i)/float(GRAPHSTEPS));
		int	iy = rect.bottom - int(y*float(rect.h()-2)) - 1;
		if (iy<orect.top) iy = orect.top;
		if (iy>orect.bottom-1) iy = orect.bottom-1;
		LineTo(hdc, ix, iy);
	}
	
	WhiteRect3D(hdc,orect,TRUE);
}

void SoftselDlgProc::SetEnables (HWND hSS) {
	ISpinnerControl *spin;
	EnableWindow (GetDlgItem (hSS, IDC_USE_SOFTSEL), epol->GetEPolySelLevel());
	IParamBlock2 *pblock = epol->getParamBlock ();
	int softSel, useEdgeDist;
	pblock->GetValue (ep_ss_use, TimeValue(0), softSel, FOREVER);
	pblock->GetValue (ep_ss_edist_use, TimeValue(0), useEdgeDist, FOREVER);
	bool enable = (epol->GetEPolySelLevel() && softSel) ? TRUE : FALSE;
	EnableWindow (GetDlgItem (hSS, IDC_USE_EDIST), enable);
	EnableWindow (GetDlgItem (hSS, IDC_AFFECT_BACKFACING), enable);
	spin = GetISpinner (GetDlgItem (hSS, IDC_EDIST_SPIN));
	spin->Enable (enable && useEdgeDist);
	ReleaseISpinner (spin);
	spin = GetISpinner (GetDlgItem (hSS, IDC_FALLOFFSPIN));
	spin->Enable (enable);
	ReleaseISpinner (spin);
	spin = GetISpinner (GetDlgItem (hSS, IDC_PINCHSPIN));
	spin->Enable (enable);
	ReleaseISpinner (spin);
	spin = GetISpinner (GetDlgItem (hSS, IDC_BUBBLESPIN));
	spin->Enable (enable);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hSS, IDC_FALLOFF_LABEL), enable);
	EnableWindow (GetDlgItem (hSS, IDC_PINCH_LABEL), enable);
	EnableWindow (GetDlgItem (hSS, IDC_BUBBLE_LABEL), enable);
}

BOOL SoftselDlgProc::DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
							  UINT msg, WPARAM wParam, LPARAM lParam) {
	Rect rect;
	PAINTSTRUCT ps;
	HDC hdc;

	switch (msg) {
	case WM_INITDIALOG:
		SetEnables (hWnd);
		break;
		
	case WM_PAINT:
		hdc = BeginPaint(hWnd,&ps);
		DrawCurve (t, hWnd, hdc);
		EndPaint(hWnd,&ps);
		return FALSE;

	case CC_SPINNER_CHANGE:
		switch (LOWORD(wParam)) {
		case IDC_FALLOFFSPIN:
		case IDC_PINCHSPIN:
		case IDC_BUBBLESPIN:
			GetClientRectP(GetDlgItem(hWnd,IDC_SS_GRAPH),&rect);
			InvalidateRect(hWnd,&rect,FALSE);
			epol->InvalidateSoftSelectionCache();
			break;
		case IDC_EDIST_SPIN:
			epol->InvalidateDistanceCache ();
			break;
		}
		map->RedrawViews (t, REDRAW_INTERACTIVE);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_USE_SOFTSEL:
		case IDC_USE_EDIST:
			SetEnables (hWnd);
			epol->InvalidateDistanceCache ();
			map->RedrawViews (t);
			break;

		case IDC_EDIST:
			epol->InvalidateDistanceCache ();
			map->RedrawViews (t);
			break;

		case IDC_AFFECT_BACKFACING:
			epol->InvalidateSoftSelectionCache ();
			map->RedrawViews (t);
			break;
		}
		break;

	default:
		return FALSE;
	}
	return TRUE;
}

class GeomDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	GeomDlgProc () { epol = NULL; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void SetEnables (HWND hGeom);
	void DeleteThis() { }
};

static GeomDlgProc theGeomDlgProc;

void GeomDlgProc::SetEnables (HWND hGeom) {
	int sl = epol->GetEPolySelLevel ();
	int msl = meshSelLevel[sl];
	BOOL edg = (msl == MNM_SL_EDGE);
	BOOL vtx = (sl == EP_SL_VERTEX);
	BOOL fac = (msl == MNM_SL_FACE);
	BOOL brdr = (sl == EP_SL_BORDER);
	BOOL elem = (sl == EP_SL_ELEMENT);
	BOOL obj = (sl == EP_SL_OBJECT);

	ISpinnerControl *spin;
	ICustButton *but;

	but = GetICustButton (GetDlgItem (hGeom, IDC_CREATE));
	but->Enable (!obj);
	if (brdr) {
		but->SetText (GetString (IDS_CAP));
		but->SetType (CBT_PUSH);
	} else {
		but->SetText (GetString (IDS_CREATE));
		but->SetType (CBT_CHECK);
	}
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_DELETE));
	but->Enable (!obj && !brdr);
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_DETACH));
	but->Enable (!brdr);
	if (obj) but->SetText (GetString (IDS_ATTACH_LIST));
	else {
		if (edg) but->SetText (GetString (IDS_SPLIT));
		else but->SetText (GetString (IDS_DETACH));
	}
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_DIVIDE));
	but->Enable (!obj);
	if (edg||fac) {
		but->SetText (GetString (IDS_DIVIDE));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
	} else {
		but->SetText (GetString (IDS_BREAK));
		but->SetType (CBT_PUSH);
	}
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_COLLAPSE));
	but->Enable (!obj && !elem);
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_EXTRUDE));
	but->Enable (fac);
	ReleaseICustButton (but);

	spin = GetISpinner (GetDlgItem (hGeom, IDC_EXTRUDESPINNER));
	spin->Enable (fac);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hGeom, IDC_EXTRUSION_LABEL), fac);

	but = GetICustButton (GetDlgItem (hGeom, IDC_BEVEL));
	but->Enable (!obj);
	if (vtx||edg) but->SetText (GetString (IDS_CHAMFER));
	else but->SetText (GetString (IDS_BEVEL));
	ReleaseICustButton (but);

	if (vtx||edg) SetDlgItemText (hGeom, IDC_OUTLINE_LABEL, GetString (IDS_CHAMFER));
	else SetDlgItemText (hGeom, IDC_OUTLINE_LABEL, GetString (IDS_OUTLINE));
	EnableWindow (GetDlgItem (hGeom, IDC_OUTLINE_LABEL), !obj);

	spin = GetISpinner (GetDlgItem (hGeom, IDC_OUTLINESPINNER));
	spin->Enable (!obj);
	if (vtx||edg) spin->SetLimits(0.0f, 9999999.0f, FALSE);
	else spin->SetLimits(-9999999.0f, 9999999.0f, FALSE);
	ReleaseISpinner (spin);

	EnableWindow (GetDlgItem (hGeom, IDC_EXTRUSION_TYPE_LABEL), fac);
	EnableWindow (GetDlgItem (hGeom, IDC_EXTYPE_GROUP), fac);
	EnableWindow (GetDlgItem (hGeom, IDC_EXTYPE_LOCAL), fac);
	EnableWindow (GetDlgItem (hGeom, IDC_EXTYPE_BY_POLY), fac);

	// It would be nice if Slice Plane were always active, but we can't make it available
	// at the object level, since the transforms won't work.
	but = GetICustButton (GetDlgItem (hGeom, IDC_SLICEPLANE));
	but->Enable (!obj);
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_RESET_PLANE));
	but->Enable (!obj);
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_SLICE));
	but->Enable (epol->EpfnInSlicePlaneMode());
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_RESET_PLANE));
	but->Enable (epol->EpfnInSlicePlaneMode());
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_CUT));
	but->Enable (!obj);
	ReleaseICustButton (but);

	EnableWindow (GetDlgItem (hGeom, IDC_SPLIT), !obj);

	bool weld = (vtx||edg) && (!brdr);
	EnableWindow (GetDlgItem (hGeom, IDC_WELD_SECTION), weld);

	but = GetICustButton (GetDlgItem (hGeom, IDC_WELD));
	but->Enable (weld);
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_WELD_TO));
	but->Enable (weld);
	ReleaseICustButton (but);

	spin = GetISpinner (GetDlgItem (hGeom, IDC_W_THR_SPIN));
	spin->Enable (weld);
	ReleaseISpinner (spin);

	spin = GetISpinner (GetDlgItem (hGeom, IDC_T_THR_SPIN));
	spin->Enable (weld);
	ReleaseISpinner (spin);

	EnableWindow (GetDlgItem (hGeom, IDC_PIXELS_TEXT), weld);

	but = GetICustButton (GetDlgItem (hGeom, IDC_CREATE_CURVE));
	but->Enable (edg);
	ReleaseICustButton (but);

	but = GetICustButton (GetDlgItem (hGeom, IDC_MAKEPLANAR));
	but->Enable (!obj);
	ReleaseICustButton (but);

	// We decided these should be always-active, though they have limited utility at object level.
	//but = GetICustButton (GetDlgItem (hGeom, IDC_ALIGN_VIEW));
	//but->Enable (!obj);
	//ReleaseICustButton (but);

	//but = GetICustButton (GetDlgItem (hGeom, IDC_ALIGN_GRID));
	//but->Enable (!obj);
	//ReleaseICustButton (but);

	// Remove Iso Verts always active.
}

BOOL GeomDlgProc::DlgProc (TimeValue t, IParamMap2 *pmap, HWND hWnd,
						   UINT msg, WPARAM wParam, LPARAM lParam) {
	ISpinnerControl *spin;
	ICustButton *but;
	TSTR name;
	int msl;

	switch (msg) {
	case WM_INITDIALOG:
		// Set up the "depressed" color for the command-mode buttons
		but = GetICustButton(GetDlgItem(hWnd,IDC_CREATE));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		but = GetICustButton(GetDlgItem(hWnd,IDC_ATTACH2));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);					

		but = GetICustButton(GetDlgItem(hWnd,IDC_DIVIDE));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		but = GetICustButton(GetDlgItem(hWnd,IDC_EXTRUDE));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		but = GetICustButton(GetDlgItem(hWnd,IDC_BEVEL));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		but = GetICustButton(GetDlgItem(hWnd,IDC_SLICEPLANE));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		but = GetICustButton(GetDlgItem(hWnd,IDC_CUT));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		but = GetICustButton(GetDlgItem(hWnd,IDC_WELD_TO));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		// Set up spinners
		spin = GetISpinner(GetDlgItem(hWnd,IDC_EXTRUDESPINNER));
		spin->SetLimits(-9999999.0f, 9999999.0f, FALSE);
		spin->LinkToEdit (GetDlgItem (hWnd,IDC_EXTRUDEAMOUNT), EDITTYPE_UNIVERSE);
		spin->SetScale (.1f);
		ReleaseISpinner (spin);

		spin = GetISpinner(GetDlgItem(hWnd,IDC_OUTLINESPINNER));
		spin->SetLimits(-9999999.0f, 9999999.0f, FALSE);
		spin->LinkToEdit (GetDlgItem (hWnd,IDC_OUTLINEAMOUNT), EDITTYPE_UNIVERSE);
		spin->SetScale (.1f);
		ReleaseISpinner (spin);

		SetEnables(hWnd);
		break;

	case CC_SPINNER_BUTTONDOWN:
		msl = epol->GetMNSelLevel ();
		switch (LOWORD(wParam)) {
		case IDC_EXTRUDESPINNER:
			epol->EpfnBeginExtrude (msl, MN_SEL, t);
			break;

		case IDC_OUTLINESPINNER:
			switch (msl) {
			case MNM_SL_VERTEX:
			case MNM_SL_EDGE:
				epol->EpfnBeginChamfer (msl, t);
				break;
			case MNM_SL_FACE:
				epol->EpfnBeginBevel (msl, MN_SEL, false, t);
				break;
			}
			break;
		}
		break;

	case CC_SPINNER_BUTTONUP:
		switch (LOWORD(wParam)) {
		case IDC_EXTRUDESPINNER:
			epol->EpfnEndExtrude (HIWORD(wParam)?true:false, t);
			pmap->RedrawViews (t, REDRAW_END);
			break;

		case IDC_OUTLINESPINNER:
			switch (epol->GetEPolySelLevel()) {
			case EP_SL_VERTEX:
			case EP_SL_EDGE:
			case EP_SL_BORDER:
				epol->EpfnEndChamfer (HIWORD(wParam) ? true : false, t);
				break;
			case EP_SL_FACE:
			case EP_SL_ELEMENT:
				epol->EpfnEndBevel (HIWORD(wParam) ? true : false, t);
				break;
			}
			pmap->RedrawViews (t, REDRAW_END);
			break;
		}
		break;

	case CC_SPINNER_CHANGE:
		spin = (ISpinnerControl*)lParam;
		msl = epol->GetMNSelLevel ();

		switch (LOWORD(wParam)) {
		case IDC_EXTRUDESPINNER:
			if (!HIWORD(wParam)) {
				// We get this value for HIWORD(wParam) either when the user
				// enters data into the spinner manually, or once when right-click
				// cancelling a spin.  If the former, we need to start the extrude.
				// If the latter, we're already in the extrude and this line is harmless.
				epol->EpfnBeginExtrude(msl, MN_SEL, t);
			}
			epol->EpfnDragExtrude (spin->GetFVal(), t);
			break;

		case IDC_OUTLINESPINNER:
			switch (msl) {
			case EP_SL_VERTEX:
			case EP_SL_EDGE:
				if (!HIWORD(wParam)) {
					// We get this value for HIWORD(wParam) either when the user
					// enters data into the spinner manually, or once when right-click
					// cancelling a spin.  If the former, we need to start the chamfer.
					// If the latter, we're already in the chamfer and this line is harmless.
					epol->EpfnBeginChamfer (msl, t);
				}
				epol->EpfnDragChamfer (spin->GetFVal (), t);
				break;

			default:
				if (!HIWORD(wParam)) {
					// We get this value for HIWORD(wParam) either when the user
					// enters data into the spinner manually, or once when right-click
					// cancelling a spin.  If the former, we need to start the bevel.
					// If the latter, we're already in the bevel and this line is harmless.
					epol->EpfnBeginBevel (msl, MN_SEL, false, t);
				}
				epol->EpfnDragBevel (spin->GetFVal (), 0.0f, t);
				break;
			}
			pmap->RedrawViews (t,REDRAW_INTERACTIVE);
			break;
		}
		break;

	case WM_CUSTEDIT_ENTER:
		// Note: The Begin and Drag parts of this operation are done above in CC_SPINNER_CHANGE.
		// Here, we just successfully end.
		msl = epol->GetMNSelLevel ();
		switch (LOWORD(wParam)) {
		case IDC_EXTRUDEAMOUNT:
			epol->EpfnEndExtrude (true, t);
			break;
		case IDC_OUTLINEAMOUNT:
			switch (msl) {
			case EP_SL_VERTEX:
			case EP_SL_EDGE:
				epol->EpfnEndChamfer (true, t);
				break;
			default:
				epol->EpfnEndBevel (true, t);
				break;
			}
			break;
		}
		pmap->RedrawViews (t, REDRAW_END);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_CREATE:
			switch (epol->GetEPolySelLevel()) {
			case EP_SL_OBJECT:
				break;
			case EP_SL_VERTEX:
				epol->EpActionToggleCommandMode (epmode_create_vertex);
				break;
			case EP_SL_EDGE:
				epol->EpActionToggleCommandMode (epmode_create_edge);
				break;
			case EP_SL_BORDER:
				epol->EpActionButtonOp (epop_cap);
				break;
			case EP_SL_FACE:
			case EP_SL_ELEMENT:
				epol->EpActionToggleCommandMode (epmode_create_face);
				break;
			}
			break;

		case IDC_DELETE:
			epol->EpActionButtonOp (epop_delete);
			break;

		case IDC_ATTACH2:
			epol->EpActionEnterPickMode (epmode_attach);
			break;

		case IDC_DETACH:
			switch (epol->GetMNSelLevel()) {
			case MNM_SL_OBJECT:
				// Really an attach multiple button.
				epol->EpActionButtonOp (epop_attach_list);
				break;
			case MNM_SL_VERTEX:
			case MNM_SL_FACE:
				epol->EpActionButtonOp (epop_detach);
				break;
			case MNM_SL_EDGE:
				// Really a "split" button.
				epol->EpActionButtonOp (epop_split);
				break;
			}
			break;

		case IDC_DIVIDE:
			switch (epol->GetMNSelLevel()) {
			case MNM_SL_OBJECT: break;
			case MNM_SL_VERTEX:
				epol->EpActionButtonOp (epop_break);
				break;
			case MNM_SL_EDGE:
				epol->EpActionToggleCommandMode (epmode_divide_edge);
				break;
			case MNM_SL_FACE:
				epol->EpActionToggleCommandMode (epmode_divide_face);
				break;
			}
			break;

		case IDC_COLLAPSE:
			epol->EpActionButtonOp (epop_collapse);
			break;

		case IDC_EXTRUDE:
			switch (epol->GetMNSelLevel()) {
			case MNM_SL_OBJECT: break;
			case MNM_SL_VERTEX:
				epol->EpActionToggleCommandMode (epmode_extrude_vertex);
				break;
			case MNM_SL_EDGE:
				epol->EpActionToggleCommandMode (epmode_extrude_edge);
				break;
			case MNM_SL_FACE:
				epol->EpActionToggleCommandMode (epmode_extrude_face);
				break;
			}
			break;

		case IDC_BEVEL:
			switch (epol->GetMNSelLevel()) {
			case MNM_SL_VERTEX:
				epol->EpActionToggleCommandMode (epmode_chamfer_vertex);
				break;
			case MNM_SL_EDGE:
				epol->EpActionToggleCommandMode (epmode_chamfer_edge);
				break;
			case MNM_SL_FACE:
				epol->EpActionToggleCommandMode (epmode_bevel);
				break;
			}
			break;

		case IDC_SLICEPLANE:
			epol->EpActionToggleCommandMode (epmode_sliceplane);
			break;

		case IDC_CUT:
			switch (epol->GetMNSelLevel()) {
			case MNM_SL_VERTEX:
				epol->EpActionToggleCommandMode (epmode_cut_vertex);
				break;
			case MNM_SL_EDGE:
				epol->EpActionToggleCommandMode (epmode_cut_edge);
				break;
			case MNM_SL_FACE:
				epol->EpActionToggleCommandMode (epmode_cut_face);
				break;
			}
			break;

		case IDC_SLICE:
			epol->EpActionButtonOp (epop_slice);
			break;

		case IDC_RESET_PLANE:
			epol->EpActionButtonOp (epop_reset_plane);
			break;

		case IDC_WELD:
			epol->EpActionButtonOp (epop_weld_sel);
			break;

		case IDC_WELD_TO:
			epol->EpActionToggleCommandMode (epmode_weld);
			break;

		case IDC_MAKEPLANAR:
			epol->EpActionButtonOp (epop_make_planar);
			break;

		case IDC_ALIGN_VIEW:
			epol->EpActionButtonOp (epop_align_view);
			break;

		case IDC_ALIGN_GRID:
			epol->EpActionButtonOp (epop_align_grid);
			break;

		case IDC_REMOVE_ISO_VERTS:
			epol->EpActionButtonOp (epop_remove_iso_verts);
			break;

		case IDC_CREATE_CURVE:
			epol->EpActionButtonOp (epop_create_shape);
			break;
		}
		break;

	default:
		return FALSE;
	}

	return TRUE;
}

class SubdivDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly* epol;
	SubdivDlgProc () { epol = NULL; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void SetEnables (HWND hSubdiv);
	void DeleteThis() { }
};

static SubdivDlgProc theSubdivDlgProc;

void SubdivDlgProc::SetEnables (HWND hSubdiv) {
	int tessByFaces;
	epol->getParamBlock()->GetValue (ep_tess_type, TimeValue(0), tessByFaces, FOREVER);
	ISpinnerControl *spin;
	spin = GetISpinner (GetDlgItem (hSubdiv, IDC_TESS_TENSIONSPIN));
	spin->Enable (!tessByFaces);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hSubdiv, IDC_TESS_TENSION_LABEL), !tessByFaces);
}

BOOL SubdivDlgProc::DlgProc (TimeValue t, IParamMap2 *pmap, HWND hWnd,
						   UINT msg, WPARAM wParam, LPARAM lParam) {
	switch (msg) {
	case WM_INITDIALOG:
		SetEnables (hWnd);
		break;
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_MESHSMOOTH:
			epol->EpActionButtonOp (epop_meshsmooth);
			break;
		case IDC_TESSELLATE:
			epol->EpActionButtonOp (epop_tessellate);
			break;
		case IDC_TESS_EDGE:
		case IDC_TESS_FACE:
			SetEnables (hWnd);
			break;
		}
	}
	return FALSE;
}

class SurfaceDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	SurfaceDlgProc () { epol = NULL; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void UpdateEnables (HWND hWnd, TimeValue t);
	void DeleteThis() { }
};

static SurfaceDlgProc theSurfaceDlgProc;

void SurfaceDlgProc::UpdateEnables (HWND hWnd, TimeValue t) {
	IParamBlock2 *pblock = epol->getParamBlock ();
	int useRIter, useRSmooth;
	pblock->GetValue (ep_surf_use_riter, t, useRIter, FOREVER);
	pblock->GetValue (ep_surf_use_rthresh, t, useRSmooth, FOREVER);
	EnableWindow (GetDlgItem (hWnd, IDC_RENDER_ITER_LABEL), useRIter);
	EnableWindow (GetDlgItem (hWnd, IDC_RENDER_SMOOTHNESS_LABEL), useRSmooth);
	int updateType;
	pblock->GetValue (ep_surf_update, t, updateType, FOREVER);
	ICustButton *but = GetICustButton (GetDlgItem (hWnd, IDC_RECALC));
	but->Enable (updateType);
	ReleaseICustButton (but);
}

BOOL SurfaceDlgProc::DlgProc (TimeValue t, IParamMap2 *pmap, HWND hWnd,
						   UINT msg, WPARAM wParam, LPARAM lParam) {
	switch (msg) {
	case WM_INITDIALOG:
		UpdateEnables (hWnd, t);
		break;
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_RECALC:
			epol->EpfnForceSubdivision ();
			break;
		case IDC_USE_RITER:
		case IDC_USE_RSHARP:
		case IDC_UPDATE_ALWAYS:
		case IDC_UPDATE_RENDER:
		case IDC_UPDATE_MANUAL:
			UpdateEnables (hWnd, t);
			break;
		}
	}
	return FALSE;
}

static void SetSmoothButtonState (HWND hWnd,DWORD bits,DWORD invalid,DWORD unused=0) {
	for (int i=IDC_SMOOTH_GRP1; i<IDC_SMOOTH_GRP1+32; i++) {
		if ( (unused&(1<<(i-IDC_SMOOTH_GRP1))) ) {
			ShowWindow(GetDlgItem(hWnd,i),SW_HIDE);
			continue;
		}

		if ( (invalid&(1<<(i-IDC_SMOOTH_GRP1))) ) {
			SetWindowText(GetDlgItem(hWnd,i),NULL);
			SendMessage(GetDlgItem(hWnd,i),CC_COMMAND,CC_CMD_SET_STATE,FALSE);
		} else {
			TSTR buf;
			buf.printf(_T("%d"),i-IDC_SMOOTH_GRP1+1);
			SetWindowText(GetDlgItem(hWnd,i),buf);
			SendMessage(GetDlgItem(hWnd,i),CC_COMMAND,CC_CMD_SET_STATE,
				(bits&(1<<(i-IDC_SMOOTH_GRP1)))?TRUE:FALSE);
		}
		InvalidateRect(GetDlgItem(hWnd,i),NULL,TRUE);
	}
}

static DWORD selectBySmoothGroups = 0;
static bool selectBySmoothClear = true;

static INT_PTR CALLBACK SelectBySmoothDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	static DWORD *param;
	int i;
	ICustButton *iBut;

	switch (msg) {
	case WM_INITDIALOG:
		param = (DWORD*)lParam;
		for (i=IDC_SMOOTH_GRP1; i<IDC_SMOOTH_GRP1+32; i++)
			SendMessage(GetDlgItem(hWnd,i),CC_COMMAND,CC_CMD_SET_TYPE,CBT_CHECK);
		SetSmoothButtonState(hWnd,selectBySmoothGroups,0,param[0]);
		CheckDlgButton(hWnd,IDC_CLEARSELECTION, selectBySmoothClear);
		CenterWindow(hWnd,GetParent(hWnd));
		break;

	case WM_COMMAND: 
		if (LOWORD(wParam)>=IDC_SMOOTH_GRP1 &&
			LOWORD(wParam)<=IDC_SMOOTH_GRP32) {
			iBut = GetICustButton(GetDlgItem(hWnd,LOWORD(wParam)));
			int shift = LOWORD(wParam) - IDC_SMOOTH_GRP1;
			if (iBut->IsChecked()) selectBySmoothGroups |= 1<<shift;
			else selectBySmoothGroups &= ~(1<<shift);
			ReleaseICustButton(iBut);
			break;
		}

		switch (LOWORD(wParam)) {
		case IDOK:
			selectBySmoothClear = IsDlgButtonChecked(hWnd,IDC_CLEARSELECTION) ? true : false;
			EndDialog(hWnd,1);					
			break;

		case IDCANCEL:
			EndDialog(hWnd,0);
			break;
		}
		break;			

	default:
		return FALSE;
	}
	return TRUE;
}

bool GetSelectBySmoothParams (Interface *ip, DWORD usedBits, DWORD & smG, bool & clear) {
	DWORD buttonsToUse = ~usedBits;
	if (DialogBoxParam(hInstance, MAKEINTRESOURCE(IDD_EP_SELECTBYSMOOTH),
		ip->GetMAXHWnd(), SelectBySmoothDlgProc, (LPARAM)&buttonsToUse)) {
		smG = selectBySmoothGroups;
		clear = selectBySmoothClear;
		return true;
	}
	return false;
}

static MtlID selectByMatID = 1;
static bool selectByMatClear = true;
static INT_PTR CALLBACK SelectByMatDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	ISpinnerControl *spin;

	switch (msg) {
	case WM_INITDIALOG:
		SetupIntSpinner (hWnd,IDC_MAT_IDSPIN,IDC_MAT_ID,1,MAX_MATID, selectByMatID);
		CheckDlgButton (hWnd,IDC_CLEARSELECTION, selectByMatClear);
		CenterWindow (hWnd,GetParent(hWnd));
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
			spin = GetISpinner(GetDlgItem(hWnd,IDC_MAT_IDSPIN));
			selectByMatID = (MtlID) spin->GetIVal();
			selectByMatClear = IsDlgButtonChecked(hWnd,IDC_CLEARSELECTION) ? true : false;
			ReleaseISpinner(spin);
			EndDialog(hWnd,1);					
			break;

		case IDCANCEL:
			EndDialog(hWnd,0);
			break;
		}
		break;

	default:
		return FALSE;
	}
	return TRUE;
}

bool GetSelectByMaterialParams (Interface *ip, MtlID & matId, bool & clear) {
	if (DialogBoxParam(hInstance, MAKEINTRESOURCE(IDD_EP_SELECTBYMAT),
		ip->GetMAXHWnd(), SelectByMatDlgProc, LPARAM(0))) {
		matId = selectByMatID;
		clear = selectByMatClear;
		return true;
	}
	return false;
}

class VertexDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	bool uiValid;
	VertexDlgProc () { epol = NULL; uiValid = FALSE; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void UpdatePerDataDisplay (TimeValue t, int channel, HWND hWnd);
	void DeleteThis() { }
	void UpdateAlphaDisplay (HWND hWnd, TimeValue t);
};

static VertexDlgProc theVertexDlgProc;

void VertexDlgProc::UpdatePerDataDisplay (TimeValue t, int channel, HWND hWnd) {
	if (!epol) return;
	ISpinnerControl *spin=NULL;
	switch (channel) {
	case VDATA_WEIGHT:
		spin = GetISpinner (GetDlgItem (hWnd, IDC_VS_WEIGHTSPIN));
		break;
	}
	if (!spin) return;

	int num;
	bool uniform;
	float value = epol->GetVertexDataValue (channel, &num, &uniform, MN_SEL, t);

	if (num == 0) {	// Nothing selected - use default weight value
		spin->SetValue (1.0f, false);
		spin->SetIndeterminate (true);
	} else {
		// Set the actual readout:
		if (!uniform) {	// Data not uniform
			spin->SetIndeterminate (TRUE);
		} else {
			spin->SetIndeterminate(FALSE);
			spin->SetValue (value, FALSE);
		}
	}
	ReleaseISpinner(spin);
}

void VertexDlgProc::UpdateAlphaDisplay (HWND hWnd, TimeValue t) {
	HWND hSpin = GetDlgItem (hWnd, IDC_VERT_ALPHASPIN);
	if (!hSpin) return;

	int num;
	bool uniform;
	float alpha;
	alpha = epol->GetVertexColor (&uniform, &num, MAP_ALPHA, MN_SEL, t).r;

	ISpinnerControl *spin = GetISpinner (hSpin);
	spin->SetIndeterminate (!uniform);
	spin->SetValue (alpha*100.0f, FALSE);
	ReleaseISpinner (spin);
}

BOOL VertexDlgProc::DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
							 UINT msg, WPARAM wParam, LPARAM lParam) {
	ISpinnerControl *spin;
	IColorSwatch *iCol;
	Color selByColor;
	int mp=0;

	switch (msg) {
	case WM_INITDIALOG:
		spin = SetupFloatSpinner (hWnd, IDC_VS_WEIGHTSPIN, IDC_VS_WEIGHT, 0.0f, 9999999.0f, 1.0f, .1f);
		spin->SetAutoScale(TRUE);
		ReleaseISpinner (spin);

		spin = SetupFloatSpinner (hWnd, IDC_VERT_ALPHASPIN,
			IDC_VERT_ALPHA, 0.0f, 100.0f, 100.0f, .1f);
		ReleaseISpinner (spin);

		// Cue an update based on the current selection
		uiValid = FALSE;
		break;

	case WM_PAINT:
		if (uiValid) return FALSE;
		iCol = GetIColorSwatch (GetDlgItem (hWnd,IDC_VERT_COLOR),
			epol->GetVertexColor (NULL, NULL, 0, MN_SEL, t),
			GetString (IDS_VERTEX_COLOR));
		ReleaseIColorSwatch(iCol);

		iCol = GetIColorSwatch (GetDlgItem (hWnd,IDC_VERT_ILLUM),
			epol->GetVertexColor (NULL, NULL, MAP_SHADING, MN_SEL, t),
			GetString (IDS_VERTEX_ILLUM));
		ReleaseIColorSwatch(iCol);

		UpdatePerDataDisplay (t, VDATA_WEIGHT, hWnd);
		UpdateAlphaDisplay (hWnd, t);

		int byIllum;
		epol->getParamBlock()->GetValue (ep_vert_sel_color, t, selByColor, FOREVER);
		epol->getParamBlock()->GetValue (ep_vert_color_selby, t, byIllum, FOREVER);
		iCol = GetIColorSwatch (GetDlgItem (hWnd, IDC_VERT_SELCOLOR), selByColor,
			GetString (byIllum ? IDS_SEL_BY_ILLUM : IDS_SEL_BY_COLOR));
		ReleaseIColorSwatch (iCol);
		uiValid = TRUE;
		return FALSE;

	case CC_COLOR_BUTTONDOWN:
		switch (LOWORD(wParam)) {
		case IDC_VERT_COLOR:
			theHold.Begin ();
			epol->BeginVertexColorModify (0);
			break;
		case IDC_VERT_ILLUM:
			theHold.Begin ();
			epol->BeginVertexColorModify (MAP_SHADING);
			break;
		}
		break;

	case CC_COLOR_BUTTONUP:
		switch (LOWORD(wParam)) {
		case IDC_VERT_COLOR:
			epol->EndVertexColorModify (HIWORD(wParam) ? true : false);
			if (HIWORD(wParam)) theHold.Accept (GetString(IDS_SET_VERT_COLOR));
			else theHold.Cancel ();
			break;
		case IDC_VERT_ILLUM:
			epol->EndVertexColorModify (HIWORD(wParam) ? true : false);
			if (HIWORD(wParam)) theHold.Accept (GetString(IDS_SET_VERT_ILLUM));
			else theHold.Cancel ();
			break;
		}
		break;

	case CC_COLOR_CHANGE:
		iCol = (IColorSwatch*)lParam;
		switch (LOWORD(wParam)) {
		case IDC_VERT_COLOR:
			epol->SetVertexColor (iCol->GetColor(), 0, MN_SEL, t);
			break;
		case IDC_VERT_ILLUM:
			epol->SetVertexColor (iCol->GetColor(), MAP_SHADING, MN_SEL, t);
			break;
		}
		break;

	case CC_SPINNER_BUTTONDOWN:
		switch (LOWORD(wParam)) {
		case IDC_VS_WEIGHTSPIN:
			epol->BeginPerDataModify (MNM_SL_VERTEX, VDATA_WEIGHT);
			break;
		case IDC_VERT_ALPHASPIN:
			theHold.Begin ();
			epol->BeginVertexColorModify (MAP_ALPHA);
		}
		break;

	case WM_CUSTEDIT_ENTER:
		switch (LOWORD(wParam)) {
		case IDC_VS_WEIGHT:
			map->RedrawViews (t, REDRAW_END);
			theHold.Begin ();
			epol->EndPerDataModify (true);
			theHold.Accept (GetString(IDS_CHANGEWEIGHT));
			break;

		case IDC_VERT_ALPHA:
			map->RedrawViews (t, REDRAW_END);
			epol->EndVertexColorModify (true);
			theHold.Accept (GetString (IDS_SET_VERT_ALPHA));
			break;
		}
		break;

	case CC_SPINNER_BUTTONUP:
		switch (LOWORD(wParam)) {
		case IDC_VS_WEIGHTSPIN:
			map->RedrawViews (t, REDRAW_END);
			if (HIWORD(wParam)) {
				theHold.Begin ();
				epol->EndPerDataModify (true);
				theHold.Accept (GetString(IDS_CHANGEWEIGHT));
			} else {
				epol->EndPerDataModify (false);
			}
			break;

		case IDC_VERT_ALPHASPIN:
			if (HIWORD(wParam)) {
				epol->EndVertexColorModify (true);
				theHold.Accept (GetString (IDS_SET_VERT_ALPHA));
			} else {
				epol->EndVertexColorModify (false);
				theHold.Cancel();
			}
			break;
		}
		break;

	case CC_SPINNER_CHANGE:
		spin = (ISpinnerControl*)lParam;
		switch (LOWORD(wParam)) {
		case IDC_VS_WEIGHTSPIN:
			epol->SetVertexDataValue (VDATA_WEIGHT, spin->GetFVal(), MN_SEL, t);
			break;
		case IDC_VERT_ALPHASPIN:
			if (!epol->InVertexColorModify ()) {
				theHold.Begin ();
				epol->BeginVertexColorModify (MAP_ALPHA);
			}
			float alpha;
			alpha = spin->GetFVal()/100.0f;
			Color clr;
			clr = Color(alpha,alpha,alpha);
			epol->SetVertexColor (clr, MAP_ALPHA, MN_SEL, t);
			break;
		}
		break;

	case WM_COMMAND:
		if (HIWORD(wParam) == 1) return FALSE;	// not handling keyboard shortcuts here.

		switch (LOWORD(wParam)) {
		case IDC_VERT_SELBYCOLOR:
			epol->EpActionButtonOp (epop_selby_vc);
			break;
		}
		break;

	default:
		return FALSE;
	}

	return TRUE;
}

class EdgeDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	bool uiValid;
	EdgeDlgProc () { epol = NULL; uiValid = FALSE; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void UpdatePerDataDisplay (TimeValue t, int channel, HWND hWnd);
	void DeleteThis() { }
};

static EdgeDlgProc theEdgeDlgProc;

void EdgeDlgProc::UpdatePerDataDisplay (TimeValue t, int channel, HWND hWnd) {
	if (!epol) return;
	ISpinnerControl *spin=NULL;
	float defaultValue=1.0f;
	switch (channel) {
	case EDATA_KNOT:
		spin = GetISpinner (GetDlgItem (hWnd, IDC_ES_WEIGHTSPIN));
		break;
	case EDATA_CREASE:
		spin = GetISpinner (GetDlgItem (hWnd, IDC_ES_CREASESPIN));
		defaultValue = 0.0f;
		break;
	}
	if (!spin) return;

	int num;
	bool uniform;
	float value = epol->GetEdgeDataValue (channel, &num, &uniform, MN_SEL, t);

	if (num == 0) {	// Nothing selected
		spin->SetIndeterminate (true);
		spin->SetValue (defaultValue, false);
	} else {
		if (!uniform) {	// Data not uniform
			spin->SetIndeterminate (TRUE);
		} else {
			spin->SetIndeterminate(FALSE);
			spin->SetValue (value, FALSE);
		}
	}
	ReleaseISpinner(spin);
}

BOOL EdgeDlgProc::DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
							 UINT msg, WPARAM wParam, LPARAM lParam) {
	ISpinnerControl *spin;

	switch (msg) {
	case WM_INITDIALOG:
		spin = SetupFloatSpinner (hWnd, IDC_ES_WEIGHTSPIN, IDC_ES_WEIGHT, 0.0f, 9999999.0f, 1.0f, .1f);
		spin->SetAutoScale(TRUE);
		ReleaseISpinner (spin);

		spin = SetupFloatSpinner (hWnd, IDC_ES_CREASESPIN, IDC_ES_CREASE, 0.0f, 1.0f, 1.0f, .1f);
		spin->SetAutoScale(TRUE);
		ReleaseISpinner (spin);

		UpdatePerDataDisplay (t, EDATA_KNOT, hWnd);
		UpdatePerDataDisplay (t, EDATA_CREASE, hWnd);

		break;

	case WM_PAINT:
		if (uiValid) return FALSE;
		UpdatePerDataDisplay (t, EDATA_KNOT, hWnd);
		UpdatePerDataDisplay (t, EDATA_CREASE, hWnd);
		uiValid = TRUE;
		return FALSE;

	case CC_SPINNER_BUTTONDOWN:
		switch (LOWORD(wParam)) {
		case IDC_ES_WEIGHTSPIN:
			epol->BeginPerDataModify (MNM_SL_EDGE, EDATA_KNOT);
			break;
		case IDC_ES_CREASESPIN:
			epol->BeginPerDataModify (MNM_SL_EDGE, EDATA_CREASE);
			break;
		}
		break;

	case WM_CUSTEDIT_ENTER:
	case CC_SPINNER_BUTTONUP:
		switch (LOWORD(wParam)) {
		case IDC_ES_WEIGHT:
		case IDC_ES_WEIGHTSPIN:
			map->RedrawViews (t,REDRAW_END);
			if (HIWORD(wParam) || msg==WM_CUSTEDIT_ENTER) {
				theHold.Begin ();
				epol->EndPerDataModify (true);
				theHold.Accept (GetString(IDS_CHANGEWEIGHT));
			} else {
				epol->EndPerDataModify (false);
			}
			break;

		case IDC_ES_CREASE:
		case IDC_ES_CREASESPIN:
			map->RedrawViews (t,REDRAW_END);
			if (HIWORD(wParam) || msg==WM_CUSTEDIT_ENTER) {
				theHold.Begin ();
				epol->EndPerDataModify (true);
				theHold.Accept (GetString(IDS_CHANGE_CREASE_VALS));
			} else {
				epol->EndPerDataModify (false);
			}
			break;
		}
		break;

	case CC_SPINNER_CHANGE:
		spin = (ISpinnerControl*)lParam;
		switch (LOWORD(wParam)) {
		case IDC_ES_WEIGHTSPIN:
			epol->SetEdgeDataValue (EDATA_KNOT, spin->GetFVal(), MN_SEL, t);
			break;
		case IDC_ES_CREASESPIN:
			epol->SetEdgeDataValue (EDATA_CREASE, spin->GetFVal(), MN_SEL, t);
			break;
		}
		break;

	default:
		return FALSE;
	}

	return TRUE;
}

class FaceDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	bool uiValid;
	FaceDlgProc () { epol = NULL; uiValid = FALSE; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void SetEnables (HWND hSurface);
	void DeleteThis() { }
	void UpdateAlphaDisplay (HWND hWnd, TimeValue t);
};

static FaceDlgProc theFaceDlgProc;

void FaceDlgProc::SetEnables (HWND hSurface) {
	if (!epol) return;
	bool element = (epol->GetEPolySelLevel()==EP_SL_ELEMENT) ? true : false;
	ICustButton *but;
	but = GetICustButton (GetDlgItem (hSurface, IDC_FS_FLIP_NORMALS));
	but->Enable (element);
	ReleaseICustButton (but);
}

void FaceDlgProc::UpdateAlphaDisplay (HWND hWnd, TimeValue t) {
	HWND hSpin = GetDlgItem (hWnd, IDC_VERT_ALPHASPIN);
	if (!hSpin) return;

	int num;
	bool uniform;
	float alpha;
	alpha = epol->GetFaceColor (&uniform, &num, MAP_ALPHA, MN_SEL, t).r;

	ISpinnerControl *spin = GetISpinner (hSpin);
	spin->SetIndeterminate (!uniform);
	spin->SetValue (alpha*100.0f, FALSE);
	ReleaseISpinner (spin);
}

BOOL FaceDlgProc::DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
							  UINT msg, WPARAM wParam, LPARAM lParam) {
	ISpinnerControl *spin;
	ICustButton *but;
	IColorSwatch *iCol;
	int i;

	switch (msg) {
	case WM_INITDIALOG:
		but = GetICustButton(GetDlgItem(hWnd,IDC_FS_EDIT_TRI));
		but->SetType(CBT_CHECK);
		but->SetHighlightColor(GREEN_WASH);
		ReleaseICustButton(but);

		SetupIntSpinner (hWnd, IDC_MAT_IDSPIN, IDC_MAT_ID, 1, MAX_MATID, 0);

		// NOTE: the following requires that the smoothing group ID's be sequential!
		for (i=IDC_SMOOTH_GRP1; i<IDC_SMOOTH_GRP1+32; i++) {
			SendMessage(GetDlgItem(hWnd,i),CC_COMMAND,
				CC_CMD_SET_TYPE,CBT_CHECK);
		}

		spin = SetupFloatSpinner (hWnd, IDC_VERT_ALPHASPIN,
			IDC_VERT_ALPHA, 0.0f, 100.0f, 100.0f, .1f);
		ReleaseISpinner (spin);

		// Cue an update based on the current face selection.
		uiValid = FALSE;
		SetEnables (hWnd);
		break;

	case WM_PAINT:
		if (uiValid) return FALSE;
		// Display the correct smoothing groups for the current selection:
		DWORD invalid, bits;
		epol->GetSmoothingGroups (MN_SEL, &invalid, &bits);
		invalid -= bits;
		SetSmoothButtonState(hWnd,bits,invalid);
		// Display the correct material index:
		MtlID mat;
		bool determined;
		mat = MtlID(epol->GetMatIndex (&determined));
		spin = GetISpinner(GetDlgItem(hWnd,IDC_MAT_IDSPIN));
		if (!determined) {
			spin->SetIndeterminate(TRUE);
		} else {
			spin->SetIndeterminate(FALSE);
			spin->SetValue (int(mat+1), FALSE);
		}
		ReleaseISpinner(spin);

		// Set the correct vertex color...
		iCol = GetIColorSwatch (GetDlgItem(hWnd,IDC_VERT_COLOR),
			epol->GetFaceColor (NULL, NULL, 0, MN_SEL, t),
			GetString(IDS_VERTEX_COLOR));
		ReleaseIColorSwatch(iCol);

		// ...and vertex illumination...
		iCol = GetIColorSwatch (GetDlgItem (hWnd,IDC_VERT_ILLUM),
			epol->GetFaceColor (NULL, NULL, MAP_SHADING, MN_SEL, t),
			GetString (IDS_VERTEX_ILLUM));
		ReleaseIColorSwatch(iCol);

		// ...and vertex alpha for the selected faces:
		UpdateAlphaDisplay (hWnd, t);

		// Make the Flip Normals button active only in element level:
		but = GetICustButton (GetDlgItem (hWnd, IDC_FS_FLIP_NORMALS));
		but->Enable (epol->GetEPolySelLevel() == EP_SL_ELEMENT);
		ReleaseICustButton(but);

		uiValid = TRUE;
		return FALSE;

	case CC_COLOR_BUTTONDOWN:
		theHold.Begin ();
		switch (LOWORD(wParam)) {
		case IDC_VERT_COLOR: epol->BeginVertexColorModify (0); break;
		case IDC_VERT_ILLUM: epol->BeginVertexColorModify (MAP_SHADING); break;\
		}
		break;

	case CC_COLOR_BUTTONUP:
		int stringID;
		switch (LOWORD(wParam)) {
		case IDC_VERT_COLOR:
			stringID = IDS_SET_VERT_COLOR;
			break;
		case IDC_VERT_ILLUM:
			stringID = IDS_SET_VERT_ILLUM;
			break;
		}
		epol->EndVertexColorModify (HIWORD(wParam) ? true : false);
		if (HIWORD(wParam)) theHold.Accept (GetString(stringID));
		else theHold.Cancel ();
		break;

	case CC_COLOR_CHANGE:
		iCol = (IColorSwatch*)lParam;
		switch (LOWORD(wParam)) {
		case IDC_VERT_COLOR:
			epol->SetFaceColor (iCol->GetColor(), 0, MN_SEL, t);
			break;
		case IDC_VERT_ILLUM:
			epol->SetFaceColor (iCol->GetColor(), MAP_SHADING, MN_SEL, t);
			break;
		}
		break;

	case CC_SPINNER_BUTTONDOWN:
		switch (LOWORD(wParam)) {
		case IDC_MAT_IDSPIN:
			theHold.Begin ();
			break;
		case IDC_VERT_ALPHASPIN:
			theHold.Begin ();
			epol->BeginVertexColorModify (MAP_ALPHA);
		}
		break;

	case WM_CUSTEDIT_ENTER:
	case CC_SPINNER_BUTTONUP:
		switch (LOWORD(wParam)) {
		case IDC_MAT_ID:
		case IDC_MAT_IDSPIN:
			epol->LocalDataChanged (PART_TOPO);
			epol->RefreshScreen ();
			if (HIWORD(wParam) || msg==WM_CUSTEDIT_ENTER) 
				 theHold.Accept(GetString(IDS_ASSIGN_MATID));
			else theHold.Cancel();
			break;

		case IDC_VERT_ALPHA:
		case IDC_VERT_ALPHASPIN:
			if (HIWORD(wParam) || (LOWORD(wParam)==IDC_VERT_ALPHA)) {
				epol->EndVertexColorModify (true);
				theHold.Accept (GetString (IDS_SET_VERT_ALPHA));
			} else {
				epol->EndVertexColorModify (false);
				theHold.Cancel();
			}
			break;
		}
		break;

	case CC_SPINNER_CHANGE:
		spin = (ISpinnerControl*)lParam;
		switch (LOWORD(wParam)) {
		case IDC_MAT_IDSPIN:
			if (!theHold.Holding()) theHold.Begin();
			epol->SetMatIndex(spin->GetIVal()-1, MN_SEL);
			break;
		case IDC_VERT_ALPHASPIN:
			if (!epol->InVertexColorModify ()) {
				theHold.Begin ();
				epol->BeginVertexColorModify (MAP_ALPHA);
			}
			float alpha;
			alpha = spin->GetFVal()/100.0f;
			Color clr;
			clr = Color(alpha,alpha,alpha);
			epol->SetFaceColor (clr, MAP_ALPHA, MN_SEL, t);
			break;
		}
		break;

	case WM_COMMAND:
		if (HIWORD(wParam) == 1) return FALSE;	// not handling keyboard shortcuts here.
		if (LOWORD(wParam)>=IDC_SMOOTH_GRP1 &&
			LOWORD(wParam)<=IDC_SMOOTH_GRP32) {
			ICustButton *iBut = GetICustButton(GetDlgItem(hWnd,LOWORD(wParam)));
			int bit = iBut->IsChecked() ? 1 : 0;
			int shift = LOWORD(wParam) - IDC_SMOOTH_GRP1;
			// NOTE: Clumsy - what should really happen, is EPoly::SetSmoothBits should be type bool.
			EditPolyObject *epo = (EditPolyObject *) epol;
			if (!epo->LocalSetSmoothBits(bit<<shift, 1<<shift, MN_SEL))
				iBut->SetCheck (false);
			ReleaseICustButton(iBut);
			break;
		}
		switch (LOWORD(wParam)) {
		case IDC_FS_EDIT_TRI:
			epol->EpActionToggleCommandMode (epmode_edit_tri);
			break;

		case IDC_FS_RETRIANGULATE:
			epol->EpActionButtonOp (epop_retriangulate);
			break;

		case IDC_FS_FLIP_NORMALS:
			epol->EpActionButtonOp (epop_flip_normals);
			break;

		case IDC_SELECT_BYID:
			epol->EpActionButtonOp (epop_selby_matid);
			break;

		case IDC_SELECTBYSMOOTH:
			epol->EpActionButtonOp (epop_selby_smg);
			break;

		case IDC_SMOOTH_CLEAR:
			epol->EpActionButtonOp (epop_clear_smg);
			break;

		case IDC_SMOOTH_AUTO:
			epol->EpActionButtonOp (epop_autosmooth); break;
		}
		break;

	default:
		return FALSE;
	}

	return TRUE;
}

class AdvancedDisplacementDlgProc : public ParamMap2UserDlgProc {
public:
	EPoly *epol;
	AdvancedDisplacementDlgProc () { epol = NULL; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void SetEnables (HWND hDisp);
	void DeleteThis() { }
};

static AdvancedDisplacementDlgProc theAdvancedDisplacementProc;

void AdvancedDisplacementDlgProc::SetEnables (HWND hDisp) {
	ISpinnerControl *spin;
	int style=0;
	if (epol && epol->getParamBlock())
		epol->getParamBlock()->GetValue (ep_asd_style, TimeValue(0), style, FOREVER);
	spin = GetISpinner (GetDlgItem (hDisp, IDC_ASD_MIN_ITERS_SPIN));
	spin->Enable (style<2);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_ASD_MIN_ITERS_LABEL), style<2);
	spin = GetISpinner (GetDlgItem (hDisp, IDC_ASD_MAX_ITERS_SPIN));
	spin->Enable (style<2);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_ASD_MAX_ITERS_LABEL), style<2);
	spin = GetISpinner (GetDlgItem (hDisp, IDC_ASD_MAX_TRIS_SPIN));
	spin->Enable (style==2);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_ASD_MAX_TRIS_LABEL), style==2);
}

BOOL AdvancedDisplacementDlgProc::DlgProc (TimeValue t, IParamMap2 *map,
						HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	switch (msg) {
	case WM_INITDIALOG:
		SetEnables (hWnd);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_ASD_GRID:
		case IDC_ASD_TREE:
		case IDC_ASD_DELAUNAY:
			SetEnables (hWnd);
			break;
		}
		break;
	}
	return false;
}

class DisplacementDlgProc : public ParamMap2UserDlgProc {
public:
	bool uiValid;
	EPoly *epol;
	DisplacementDlgProc () { epol = NULL; uiValid = false; }
	BOOL DlgProc (TimeValue t, IParamMap2 *map, HWND hWnd,
		UINT msg, WPARAM wParam, LPARAM lParam);
	void SetEnables (HWND hDisp);
	void DeleteThis() { }
	void InvalidateUI (HWND hWnd) { uiValid = false; if (hWnd) InvalidateRect (hWnd, NULL, false); }
};

static DisplacementDlgProc theDisplacementProc;

void DisplacementDlgProc::SetEnables (HWND hDisp) {
	BOOL useSD=false;
	int method=0;
	if (epol && epol->getParamBlock ()) {
		epol->getParamBlock()->GetValue (ep_sd_use, TimeValue(0), useSD, FOREVER);
		epol->getParamBlock()->GetValue (ep_sd_method, TimeValue(0), method, FOREVER);
	}

	// Enable buttons...
	EnableWindow (GetDlgItem (hDisp, IDC_SD_PRESET1), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_PRESET2), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_PRESET3), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_ADVANCED), useSD);

	// Enable grouping boxes:
	EnableWindow (GetDlgItem (hDisp, IDC_SD_PRESET_BOX), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_METHOD_BOX), useSD);

	// Enable radios, checkboxes:
	EnableWindow (GetDlgItem (hDisp, IDC_SD_SPLITMESH), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_TESS_REGULAR), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_TESS_SPATIAL), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_TESS_CURV), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_TESS_LDA), useSD);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_VIEW_DEP), useSD);

	// Enable spinners and their labels based both on useSD and method.
	bool useSteps = useSD && (method==0);
	ISpinnerControl *spin;
	spin = GetISpinner (GetDlgItem (hDisp, IDC_SD_TESS_U_SPINNER));
	spin->Enable (useSteps);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_STEPS_LABEL), useSteps);

	bool useEdge = useSD && ((method==1) || (method==3));
	spin = GetISpinner (GetDlgItem (hDisp, IDC_SD_TESS_EDGE_SPINNER));
	spin->Enable (useEdge);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_EDGE_LABEL), useEdge);

	bool useDist = useSD && (method>1);
	spin = GetISpinner (GetDlgItem (hDisp, IDC_SD_TESS_DIST_SPINNER));
	spin->Enable (useDist);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_DISTANCE_LABEL), useDist);

	spin = GetISpinner (GetDlgItem (hDisp, IDC_SD_TESS_ANG_SPINNER));
	spin->Enable (useDist);
	ReleaseISpinner (spin);
	EnableWindow (GetDlgItem (hDisp, IDC_SD_ANGLE_LABEL), useDist);

	uiValid = true;
}

class DispParamChangeRestore : public RestoreObj {
public:
	IParamMap2 *map;
	DispParamChangeRestore (IParamMap2 *m) : map(m) { }
	void Restore (int isUndo) { theDisplacementProc.InvalidateUI (map->GetHWnd()); }
	void Redo () { theDisplacementProc.InvalidateUI (map->GetHWnd()); }
};

BOOL DisplacementDlgProc::DlgProc (TimeValue t, IParamMap2 *pmap, HWND hWnd,
						   UINT msg, WPARAM wParam, LPARAM lParam) {
	BOOL ret;
	switch (msg) {
	case WM_INITDIALOG:
		uiValid = false;
		break;

	case WM_PAINT:
		if (uiValid) break;
		SetEnables (hWnd);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_SD_ENGAGE:
			uiValid = false;
			break;
		case IDC_SD_PRESET1:
			theHold.Begin();  // (uses restore object in paramblock.)
			epol->UseDisplacementPreset (0);
			theHold.Put (new DispParamChangeRestore (pmap));
			theHold.Accept(GetString(IDS_DISP_APPROX_CHANGE));
			uiValid = false;
			break;
		case IDC_SD_PRESET2:
			theHold.Begin();  // (uses restore object in paramblock.)
			epol->UseDisplacementPreset (1);
			theHold.Put (new DispParamChangeRestore (pmap));
			theHold.Accept(GetString(IDS_DISP_APPROX_CHANGE));
			uiValid = false;
			break;
		case IDC_SD_PRESET3:
			theHold.Begin();  // (uses restore object in paramblock.)
			epol->UseDisplacementPreset (2);
			theHold.Put (new DispParamChangeRestore (pmap));
			theHold.Accept(GetString(IDS_DISP_APPROX_CHANGE));
			uiValid = false;
			break;
		case IDC_SD_ADVANCED:
			// Make sure we're up to date...
			epol->SetDisplacementParams ();
			theAdvancedDisplacementProc.epol = epol;
			ret = CreateModalParamMap2 (ep_advanced_displacement, epol->getParamBlock(),
				t, hInstance, MAKEINTRESOURCE (IDD_EP_DISP_APPROX_ADV),
				hWnd, &theAdvancedDisplacementProc);
			if (ret) {
				epol->SetDisplacementParams ();
			} else {
				// Otherwise, revert back to values in polyobject.
				epol->UpdateDisplacementParams ();
			}
			break;

		default:
			InvalidateUI (hWnd);
			break;
		}
	}
	return FALSE;
}

//-----------------------------------------------------------------------
// EditPolyObject UI-related methods:

void EditPolyObject::RefreshSelType () {
	HWND hWnd = GetDlgHandle (ep_select);
	if (hWnd) {
		theSelectDlgProc.RefreshSelType (hWnd);
		theSelectDlgProc.SetEnables (hWnd);
	}
	SetSoftSelDlgEnables();
	SetGeomDlgEnables();
	SetSubdivDlgEnables ();
	UpdateSurfType ();
}

void EditPolyObject::InvalidateNumberSelected () {
	HWND hWnd = GetDlgHandle (ep_select);
	if (!hWnd) return;
	InvalidateRect (hWnd, NULL, FALSE);
	theSelectDlgProc.updateNumSel = true;
}

void EditPolyObject::SetSoftSelDlgEnables () {
	HWND hSS = GetDlgHandle (ep_softsel);
	if (!hSS) return;
	theSoftselDlgProc.epol = this;
	theSoftselDlgProc.SetEnables (hSS);
}

void EditPolyObject::SetGeomDlgEnables () {
	HWND hGeom = GetDlgHandle (ep_geom);
	if (!hGeom) return;
	theGeomDlgProc.epol = this;
	theGeomDlgProc.SetEnables (hGeom);
}

void EditPolyObject::SetSubdivDlgEnables () {
	HWND hSubdiv = GetDlgHandle (ep_subdiv);
	if (!hSubdiv) return;
	theSubdivDlgProc.epol = this;
	theSubdivDlgProc.SetEnables (hSubdiv);
}

void EditPolyObject::UpdateDisplacementParams () {
	// The first call we make to pblock->SetValue will overwrite all the PolyObject-level params,
	// so we need to cache copies.
	bool useDisp = GetDisplacement ();
	bool dispSplit = GetDisplacementSplit ();
	TessApprox dparams = DispParams ();

	pblock->SetValue (ep_sd_use, ip->GetTime(), useDisp);
	pblock->SetValue (ep_sd_split_mesh, ip->GetTime(), dispSplit);
	int method;
	switch (dparams.type) {
	case TESS_REGULAR: method=0; break;
	case TESS_SPATIAL: method=1; break;
	case TESS_CURVE: method=2; break;
	case TESS_LDA: method=3; break;
	}
	pblock->SetValue (ep_sd_method, ip->GetTime(), method);
	pblock->SetValue (ep_sd_tess_steps, ip->GetTime(), dparams.u);
	pblock->SetValue (ep_sd_tess_edge, ip->GetTime(), dparams.edge);
	pblock->SetValue (ep_sd_tess_distance, ip->GetTime(), dparams.dist);
	pblock->SetValue (ep_sd_tess_angle, ip->GetTime(), dparams.ang*PI/180.0f);
	pblock->SetValue (ep_sd_view_dependent, ip->GetTime(), dparams.view);
}

void EditPolyObject::SetDisplacementParams () {
	int val;
	TimeValue t = ip ? ip->GetTime() : GetCOREInterface()->GetTime();
	pblock->GetValue (ep_sd_use, t, val, FOREVER);
	SetDisplacement (val?true:false);
	pblock->GetValue (ep_sd_split_mesh, t, val, FOREVER);
	SetDisplacementSplit (val?true:false);
	pblock->GetValue (ep_sd_method, t, val, FOREVER);
	switch (val) {
	case 0: DispParams().type=TESS_REGULAR; break;
	case 1: DispParams().type=TESS_SPATIAL; break;
	case 2: DispParams().type=TESS_CURVE; break;
	case 3: DispParams().type=TESS_LDA; break;
	}
	pblock->GetValue (ep_sd_tess_steps, t, DispParams().u, FOREVER);
	pblock->GetValue (ep_sd_tess_edge, t, DispParams().edge, FOREVER);
	pblock->GetValue (ep_sd_tess_distance, t, DispParams().dist, FOREVER);
	pblock->GetValue (ep_sd_tess_angle, t, DispParams().ang, FOREVER);
	DispParams().ang *= 180.0f/PI;
	pblock->GetValue (ep_sd_view_dependent, t, DispParams().view, FOREVER);
}

void EditPolyObject::UseDisplacementPreset (int presetNumber) {
	// Make sure PolyObject settings are up to date:
	SetDisplacementParams ();
	// Use PolyObject method to change PolyObject settings:
	SetDisplacementApproxToPreset (presetNumber);
	// Copy new PolyObject settings to parameter block:
	UpdateDisplacementParams ();
}

void EditPolyObject::InvalidateSurfaceUI() {
	if (!pSurface) return;
	switch (selLevel) {
	case EP_SL_OBJECT:
		return;	// Nothing to invalidate.
	case EP_SL_VERTEX:
		theVertexDlgProc.uiValid = FALSE;
		break;
	case EP_SL_EDGE:
	case EP_SL_BORDER:
		theEdgeDlgProc.uiValid = FALSE;
		break;
	case EP_SL_FACE:
	case EP_SL_ELEMENT:
		theFaceDlgProc.uiValid = FALSE;
		break;
	}
	HWND hWnd = pSurface->GetHWnd();
	
	if (hWnd)
		InvalidateRect (hWnd, NULL, FALSE);
	
	if( hFaceFlags )
	{
		InvalidateRect (hFaceFlags, NULL, FALSE);
	}
	if( hPassFlags )
	{
		InvalidateRect (hPassFlags, NULL, FALSE);
	}
	if( hExtFaceFlags )
	{
		InvalidateRect (hExtFaceFlags, NULL, FALSE);
	}
}

int surfIDs[] = {ep_surface, ep_vertex, ep_edge, ep_face};
int surfDlgs[] = { IDD_EP_SURF_OBJECT, IDD_EP_SURF_VERT,
		IDD_EP_SURF_EDGE, IDD_EP_SURF_FACE };
ParamMap2UserDlgProc *surfProcs[] = { &theSurfaceDlgProc,
	 &theVertexDlgProc, &theEdgeDlgProc, &theFaceDlgProc };

void EditPolyObject::UpdateSurfType () {
	if (editObj != this) return;
	if (!ip) return;
	if (!pSurface) return;
	int msl = meshSelLevel[selLevel];
	HWND hSurf = pSurface->GetHWnd ();
	bool wasObj = GetDlgItem (hSurf, IDC_USE_NURMS) ? TRUE : FALSE;
	bool wasFace = GetDlgItem (hSurf, IDC_FS_EDIT_TRI) ? TRUE : FALSE;
	bool wasEdge = GetDlgItem (hSurf, IDC_ES_WEIGHT_LABEL) ? TRUE : FALSE;
	bool wasVert = GetDlgItem (hSurf, IDC_VS_WEIGHT_LABEL) ? TRUE : FALSE;
	if (wasObj && (msl == MNM_SL_OBJECT)) return;
	if (wasVert && (msl == MNM_SL_VERTEX)) return;
	if (wasEdge && (msl == MNM_SL_EDGE)) return;
	if (wasFace && (msl == MNM_SL_FACE)) {
		InvalidateSurfaceUI ();
		return;
	}

	if( hFaceFlags )
	{
		rsFaceFlags = IsRollupPanelOpen( hFaceFlags );
		ip->DeleteRollupPage( hFaceFlags );
		hFaceFlags = NULL;
	}
	if( hExtFaceFlags )
	{
		rsExtFaceFlags = IsRollupPanelOpen( hExtFaceFlags );
		ip->DeleteRollupPage( hExtFaceFlags );
		hExtFaceFlags = NULL;
	}
	if ( hPassFlags )
	{
		rsPassFlags = IsRollupPanelOpen( hPassFlags );
		ip->DeleteRollupPage( hPassFlags );
		hPassFlags = NULL;
	}

	if (pDisp && (msl != MNM_SL_OBJECT)) {
		rsDisp = IsRollupPanelOpen (pDisp->GetHWnd()) ? false : true;
		DestroyCPParamMap2 (pDisp);
		pDisp = NULL;
	}

	rsSurface = IsRollupPanelOpen (hSurf) ? FALSE : TRUE;
	HWND hFocus = GetFocus ();
	pSurface = CreateCPParamMap2 (surfIDs[msl], pblock, ip, hInstance,
		MAKEINTRESOURCE (surfDlgs[msl]), GetString (IDS_SURFACE_PROPERTIES),
		rsSurface ? APPENDROLL_CLOSED : 0, surfProcs[msl], hSurf);

	// Handle displacement mapping dialog:
	if (msl == MNM_SL_OBJECT) {
		if (!pDisp) pDisp = CreateCPParamMap2 (ep_displacement, pblock, ip, hInstance,
			MAKEINTRESOURCE (IDD_EP_DISP_APPROX),
			GetString (IDS_DISPLACEMENT),
			rsDisp ? APPENDROLL_CLOSED : 0, &theDisplacementProc);
	}

	// Verify: EP_SL_OBJECT
	if(( selLevel == EP_SL_FACE ) || ( selLevel == EP_SL_OBJECT ) || ( selLevel == EP_SL_ELEMENT ))
	{
		hFaceFlags = ip->AddRollupPage(hInstance, MAKEINTRESOURCE( IDD_FACE_FLAGS ),
			FaceFlagsDlgProc, GetString(IDS_FACE_FLAGS), (LPARAM)this, rsFaceFlags ? 0 : APPENDROLL_CLOSED);

		hPassFlags = ip->AddRollupPage(hInstance, MAKEINTRESOURCE( IDD_FACE_PASS ),
			FacePassDlgProc, "Face Multitexture Passes", (LPARAM)this, rsPassFlags ? 0 : APPENDROLL_CLOSED);

		hExtFaceFlags = ip->AddRollupPage(hInstance, MAKEINTRESOURCE( IDD_CASFLAGS ),
			ExtFaceFlagsDlgProc, "Extended Face Flags", (LPARAM)this, rsExtFaceFlags ? 0 : APPENDROLL_CLOSED);
	}

	SetFocus (hFocus);
}

// --- Begin/End Edit Params ---------------------------------

static bool oldShowEnd;

void EditPolyObject::BeginEditParams (IObjParam  *ip, ULONG flags,Animatable *prev) {
	this->ip = ip;
	editObj = this;

	theSelectDlgProc.epol = this;
	theSoftselDlgProc.epol = this;
	theGeomDlgProc.epol = this;
	theSubdivDlgProc.epol = this;
	theSurfaceDlgProc.epol = this;
	theVertexDlgProc.epol = this;
	theEdgeDlgProc.epol = this;
	theFaceDlgProc.epol = this;
	theDisplacementProc.epol = this;

	pSelect = CreateCPParamMap2 (ep_select, pblock, ip, hInstance,
		MAKEINTRESOURCE (IDD_EP_SELECT), GetString (IDS_SELECT),
		rsSelect ? APPENDROLL_CLOSED : 0, &theSelectDlgProc);
	pSoftsel = CreateCPParamMap2 (ep_softsel, pblock, ip, hInstance,
		MAKEINTRESOURCE (IDD_EP_SOFTSEL), GetString (IDS_SOFTSEL),
		rsSoftsel ? APPENDROLL_CLOSED : 0, &theSoftselDlgProc);
	pGeom = CreateCPParamMap2 (ep_geom, pblock, ip, hInstance,
		MAKEINTRESOURCE (IDD_EP_GEOM), GetString (IDS_EDIT_GEOM),
		rsGeom ? APPENDROLL_CLOSED : 0, &theGeomDlgProc);
	pSubdiv = CreateCPParamMap2 (ep_subdiv, pblock, ip, hInstance,
		MAKEINTRESOURCE (IDD_EP_SUBDIV), GetString (IDS_SUBDIVISION),
		rsSubdiv ? APPENDROLL_CLOSED : 0, &theSubdivDlgProc);
	int msl = meshSelLevel[selLevel];
	pSurface = CreateCPParamMap2 (surfIDs[msl], pblock, ip, hInstance,
		MAKEINTRESOURCE (surfDlgs[msl]), GetString (IDS_SURFACE_PROPERTIES),
		rsSurface ? APPENDROLL_CLOSED : 0, surfProcs[msl]);
	if (selLevel == EP_SL_OBJECT) {
		pDisp = CreateCPParamMap2 (ep_displacement, pblock, ip, hInstance,
			MAKEINTRESOURCE (IDD_EP_DISP_APPROX),
			GetString (IDS_DISPLACEMENT),
			rsDisp ? APPENDROLL_CLOSED : 0, &theDisplacementProc);
	}

	if(( selLevel == EP_SL_FACE ) || ( selLevel == EP_SL_OBJECT ) || ( selLevel == EP_SL_ELEMENT ))
	{
		hFaceFlags = ip->AddRollupPage(hInstance, MAKEINTRESOURCE( IDD_FACE_FLAGS ),
			FaceFlagsDlgProc, GetString(IDS_FACE_FLAGS), (LPARAM)this, rsFaceFlags ? 0 : APPENDROLL_CLOSED);

		hPassFlags = ip->AddRollupPage(hInstance, MAKEINTRESOURCE( IDD_FACE_PASS ),
			FacePassDlgProc, "Face Multitexture Passes", (LPARAM)this, rsPassFlags ? 0 : APPENDROLL_CLOSED);

		hExtFaceFlags = ip->AddRollupPage(hInstance, MAKEINTRESOURCE( IDD_CASFLAGS ),
			ExtFaceFlagsDlgProc, "Extended Face Flags", (LPARAM)this, rsExtFaceFlags ? 0 : APPENDROLL_CLOSED);
	}


	InvalidateNumberSelected ();
	UpdateDisplacementParams ();

	// Create sub object editing modes.
	moveMode       = new MoveModBoxCMode(this,ip);
	rotMode        = new RotateModBoxCMode(this,ip);
	uscaleMode     = new UScaleModBoxCMode(this,ip);
	nuscaleMode    = new NUScaleModBoxCMode(this,ip);
	squashMode     = new SquashModBoxCMode(this,ip);
	selectMode     = new SelectModBoxCMode(this,ip);

	// Add our command modes:
	createVertMode = new CreateVertCMode (this, ip);
	createEdgeMode = new CreateEdgeCMode (this, ip);
	createFaceMode = new CreateFaceCMode(this, ip);
	divideEdgeMode = new DivideEdgeCMode(this, ip);
	divideFaceMode = new DivideFaceCMode (this, ip);
	extrudeMode = new ExtrudeCMode (this, ip);
	chamferMode = new ChamferCMode (this, ip);
	bevelMode = new BevelCMode(this, ip);
	cutVertMode = new CutVertCMode (this, ip);
	cutEdgeMode = new CutEdgeCMode (this, ip);
	cutFaceMode = new CutFaceCMode (this, ip);
	weldMode = new WeldCMode (this, ip);
	editTriMode = new EditTriCMode (this, ip);

	// And our pick modes:
	attachPickMode = new AttachPickMode(this, ip);

	// Add our sub object type
	// TSTR type1(GetString(IDS_VERTEX));
	// TSTR type2(GetString (IDS_EDGE));
	// TSTR type3(GetString (IDS_BORDER));
	// TSTR type4(GetString (IDS_FACE));
	// TSTR type5(GetString (IDS_ELEMENT));
	// const TCHAR *ptype[] = {type1,type2,type3,type4,type5};
	// This call is obsolete. Please see BaseObject::NumSubObjTypes() and BaseObject::GetSubObjType()
	// ip->RegisterSubObjectTypes (ptype, 5);

	// Set show end result.
	oldShowEnd = ip->GetShowEndResult() ? TRUE : FALSE;
	ip->SetShowEndResult (GetFlag (EPOLY_DISP_RESULT));

	// Restore the selection level.
	ip->SetSubObjectLevel(selLevel);

	// We want del key, backspace input if in geometry level
	if (selLevel != EP_SL_OBJECT) {
		ip->RegisterDeleteUser(this);
		backspacer.SetEPoly(this);
		backspaceRouter.Register(&backspacer);
	}

	NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);		
	SetAFlag(A_OBJ_BEING_EDITED);	
}

void EditPolyObject::EndEditParams (IObjParam *ip, ULONG flags,Animatable *next)
{
	//if (!enabled) return;

	// Unregister del key, backspace notification
	if (selLevel != EP_SL_OBJECT) {
		ip->UnRegisterDeleteUser(this);
		backspacer.SetEPoly (NULL);
		backspaceRouter.UnRegister (&backspacer);
	}

	ExitAllCommandModes ();
	if (moveMode) delete moveMode;
	moveMode = NULL;
	if (rotMode) delete rotMode;
	rotMode = NULL;
	if (uscaleMode) delete uscaleMode;
	uscaleMode = NULL;
	if (nuscaleMode) delete nuscaleMode;
	nuscaleMode = NULL;
	if (squashMode) delete squashMode;
	squashMode = NULL;
	if (selectMode) delete selectMode;
	selectMode = NULL;

	if (createVertMode) delete createVertMode;
	createVertMode = NULL;
	if (createEdgeMode) delete createEdgeMode;
	createEdgeMode = NULL;
	if (createFaceMode) delete createFaceMode;
	createFaceMode = NULL;
	if (divideEdgeMode) delete divideEdgeMode;
	divideEdgeMode = NULL;
	if (divideFaceMode) delete divideFaceMode;
	divideFaceMode = NULL;
	if (extrudeMode) delete extrudeMode;
	extrudeMode = NULL;
	if (chamferMode) delete chamferMode;
	chamferMode = NULL;
	if (bevelMode) delete bevelMode;
	bevelMode = NULL;
	if (cutVertMode) delete cutVertMode;
	cutVertMode = NULL;
	if (cutEdgeMode) delete cutEdgeMode;
	cutEdgeMode = NULL;
	if (cutFaceMode) delete cutFaceMode;
	cutFaceMode = NULL;
	if (weldMode) delete weldMode;
	weldMode = NULL;
	if (editTriMode) delete editTriMode;
	editTriMode = NULL;

	if (tempData) {
		delete tempData;
		tempData = NULL;
	}
	if (tempMove) {
		delete tempMove;
		tempMove = NULL;
	}

	if( hFaceFlags ) 
	{
		rsFaceFlags = IsRollupPanelOpen( hFaceFlags );
		ip->DeleteRollupPage( hFaceFlags );
		hFaceFlags = NULL;
	}
	if( hExtFaceFlags )
	{
		rsExtFaceFlags = IsRollupPanelOpen( hExtFaceFlags );
		ip->DeleteRollupPage( hExtFaceFlags );
		hExtFaceFlags = NULL;
	}
	if ( hPassFlags )
	{
		rsPassFlags = IsRollupPanelOpen( hPassFlags );
		ip->DeleteRollupPage( hPassFlags );
		hPassFlags = NULL;
	}

	if (pSelect) {
		rsSelect = IsRollupPanelOpen (pSelect->GetHWnd())?FALSE:TRUE;
		DestroyCPParamMap2 (pSelect);
		pSelect = NULL;
	}
	if (pSoftsel) {
		rsSoftsel = IsRollupPanelOpen (pSoftsel->GetHWnd())?FALSE:TRUE;
		DestroyCPParamMap2 (pSoftsel);
		pSoftsel = NULL;
	}
	if (pGeom) {
		rsGeom = IsRollupPanelOpen (pGeom->GetHWnd())?FALSE:TRUE;
		DestroyCPParamMap2 (pGeom);
		pGeom = NULL;
	}
	if (pSubdiv) {
		rsSubdiv = IsRollupPanelOpen (pSubdiv->GetHWnd())?FALSE:TRUE;
		DestroyCPParamMap2 (pSubdiv);
		pSubdiv = NULL;
	}
	if (pSurface) {
		rsSurface = IsRollupPanelOpen (pSurface->GetHWnd())?FALSE:TRUE;
		DestroyCPParamMap2 (pSurface);
		pSurface = NULL;
	}
	if (pDisp) {
		rsDisp = IsRollupPanelOpen (pDisp->GetHWnd())?FALSE:TRUE;
		DestroyCPParamMap2 (pDisp);
		pDisp = NULL;
	}

	this->ip = NULL;
	editObj = NULL;

	// Reset show end result
	// NOTE: since this causes a pipeline reevaluation, it needs to come last.
	SetFlag (EPOLY_DISP_RESULT, ip->GetShowEndResult());
	ip->SetShowEndResult(oldShowEnd);

	ClearAFlag(A_OBJ_BEING_EDITED);
}

void EditPolyObject::ExitAllCommandModes (bool exSlice) {
	if (moveMode) ip->DeleteMode(moveMode);
	if (rotMode) ip->DeleteMode(rotMode);
	if (uscaleMode) ip->DeleteMode(uscaleMode);
	if (nuscaleMode) ip->DeleteMode(nuscaleMode);
	if (squashMode) ip->DeleteMode(squashMode);
	if (selectMode) ip->DeleteMode(selectMode);

	if (createVertMode) ip->DeleteMode (createVertMode);
	if (createEdgeMode) ip->DeleteMode (createEdgeMode);
	if (createFaceMode) ip->DeleteMode (createFaceMode);
	if (divideEdgeMode) ip->DeleteMode (divideEdgeMode);
	if (divideFaceMode) ip->DeleteMode (divideFaceMode);
	if (extrudeMode) ip->DeleteMode (extrudeMode);
	if (chamferMode) ip->DeleteMode (chamferMode);
	if (bevelMode) ip->DeleteMode (bevelMode);
	if (cutVertMode) ip->DeleteMode (cutVertMode);
	if (cutEdgeMode) ip->DeleteMode (cutEdgeMode);
	if (cutFaceMode) ip->DeleteMode (cutFaceMode);
	if (weldMode) ip->DeleteMode (weldMode);
	if (editTriMode) ip->DeleteMode (editTriMode);

	if (sliceMode && exSlice) ExitSliceMode ();
	ip->ClearPickMode();
}

void EditPolyObject::InvalidateDialogElement (int elem) {
	// No convenient way to get parammap pointer from element id,
	// so we invalidate this element in all parammaps.  (Harmless - does
	// nothing if elem is not present in a given parammap.)
	if (pGeom) pGeom->Invalidate (elem);
	if (pSelect) pSelect->Invalidate (elem);
	if (pSoftsel) pSoftsel->Invalidate (elem);
	if (pSubdiv) pSubdiv->Invalidate (elem);
	if (pSurface) pSurface->Invalidate (elem);
	if (pDisp) {
		pDisp->Invalidate (elem);
		if ((elem == ep_sd_use) || (elem == ep_sd_method)) {
			theDisplacementProc.InvalidateUI (pDisp->GetHWnd());
		}
	}
}

void EditPolyObject::UpdatePerDataDisplay (TimeValue t, int msl, int channel) {
	if (msl != meshSelLevel[selLevel]) return;
	if (!pSurface) return;
	HWND hWnd = pSurface->GetHWnd();
	switch (msl) {
	case MNM_SL_VERTEX:
		theVertexDlgProc.UpdatePerDataDisplay (t, channel, hWnd);
		break;
	case MNM_SL_EDGE:
		theEdgeDlgProc.UpdatePerDataDisplay (t, channel, hWnd);
		break;
	}
}

HWND EditPolyObject::GetDlgHandle (int dlgID) {
	if (!ip) return NULL;
	if (editObj != this) return NULL;
	if (dlgID<0) return NULL;
	IParamMap2 *pmap=NULL;
	switch (dlgID) {
	case ep_select: pmap = pSelect; break;
	case ep_softsel: pmap = pSoftsel; break;
	case ep_geom: pmap = pGeom; break;
	case ep_subdiv: pmap = pSubdiv; break;
	case ep_displacement: pmap = pDisp; break;
	case ep_surface:
	case ep_vertex:
	case ep_edge:
	case ep_face:
		pmap = pSurface; break;
	}
	if (!pmap) return NULL;
	return pmap->GetHWnd();
}

int EditPolyObject::NumSubObjTypes() { return 5; }

ISubObjType *EditPolyObject::GetSubObjType(int i) {
	static bool initialized = false;
	if (!initialized) {
		initialized = true;
		SOT_Vertex.SetName(GetString(IDS_VERTEX));
		SOT_Edge.SetName(GetString(IDS_EDGE));
		SOT_Border.SetName(GetString(IDS_BORDER));
		SOT_Poly.SetName(GetString(IDS_FACE));
		SOT_Element.SetName(GetString(IDS_ELEMENT));
	}

	switch(i) {
	case -1:
		if (selLevel > 0) return GetSubObjType (selLevel-1);
		break;
	case 0: return &SOT_Vertex;
	case 1: return &SOT_Edge;
	case 2: return &SOT_Border;
	case 3: return &SOT_Poly;
	case 4: return &SOT_Element;
	}
	return NULL;
}

static INT_PTR CALLBACK SelectByFlagsDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	static EditPolyObject* mod;
	static BOOL exact_match;
	static FlagType set_flags;

	switch (msg) {
	case WM_INITDIALOG:
		mod = (EditPolyObject*) lParam;
		exact_match = FALSE;
		set_flags = 0;
		//param = (DWORD*)lParam;
		/*for (i=IDC_SMOOTH_GRP1; i<IDC_SMOOTH_GRP1+32; i++)
			SendMessage(GetDlgItem(hWnd,i),CC_COMMAND,CC_CMD_SET_TYPE,CBT_CHECK);
		SetSmoothButtonState(hWnd,param[0],0,param[2]);
		CheckDlgButton(hWnd,IDC_CLEARSELECTION,param[1]);*/
		CenterWindow(hWnd,GetParent(hWnd));
		break;

	case WM_COMMAND: 
		if (LOWORD(wParam)>=IDC_SEL_SKATABLE &&
			LOWORD(wParam)<=IDC_SEL_RENDER_SEPARATE) 
		{
			int is_checked;
			is_checked = SendMessage( GetDlgItem(hWnd,LOWORD(wParam)), BM_GETCHECK, 0, 0 );
			
			int shift = LOWORD(wParam) - IDC_SEL_SKATABLE;				
			if( is_checked == BST_CHECKED ) set_flags |= 1<<shift;
			else set_flags &= ~(1<<shift);					
			break;
		}

		switch (LOWORD(wParam)) {
		/*case IDOK:
			param[1] = IsDlgButtonChecked(hWnd,IDC_CLEARSELECTION);					
			EndDialog(hWnd,1);					
			break;					*/
		case IDC_EXACT_MATCH:
			exact_match ^= 1;
			break;
		case IDC_SELECT_MATCHING:
			mod->SelectByFaceFlags( set_flags, exact_match == TRUE );
			break;
		case IDCANCEL:
			EndDialog(hWnd,0);
			break;
		}
		break;			

	default: return FALSE;
	}
	return TRUE;
}

void EditPolyObject::SetFaceFlags( FlagType flags )
{
	int i;
	if( !ip ) 
	{
		return;
	}

	BitArray faceSel;

	MNMesh &mesh = GetMesh();
	mesh.getFaceSel(faceSel);

	if( faceSel.NumberSet() == 0 ) 
	{
		return;
	}

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		return;
	}

	FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
	if( fdc == NULL ) 
	{		
		// The mesh does not have our face-data channel so we will add it here
		fdc = new FaceFlagsData();
		fdc->FacesCreated( 0, mesh.FNum());

		pFDMgr->AddFaceDataChan( fdc );
	}

	for( i = 0; i < faceSel.GetSize(); i++ )
	{
		if( faceSel[i] )
		{
			FlagType new_value;

			fdc->GetValue( i, new_value );
			new_value |= flags;
			fdc->SetValue( i, new_value );	
		}		
	}
}

void EditPolyObject::ClearFaceFlags( FlagType flags )
{
	int i;
	if( !ip ) 
	{
		return;
	}

	MNMesh &mesh = GetMesh();
	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	if( faceSel.NumberSet() == 0 ) 
	{
		return;
	}

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		return;
	}

	FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
	if( fdc == NULL ) 
	{		
		// The mesh does not have our face-data channel so we will add it here
		fdc = new FaceFlagsData();
		fdc->FacesCreated( 0, mesh.FNum());

		pFDMgr->AddFaceDataChan( fdc );
	}

	for( i = 0; i < faceSel.GetSize(); i++ )
	{
		if( faceSel[i] )
		{
			FlagType new_value;

			fdc->GetValue( i, new_value );
			new_value &= ~flags;
			fdc->SetValue( i, new_value );	
		}		
	}
}

int	EditPolyObject::GetFaceFlagsSet( HWND hwnd, FlagType &diff_flags )
{
	FlagType flags = 0;
	int i;
	
	diff_flags = 0;
	for( i = 0; checkbox_to_flag_table[i][0]; i++ )
	{
		HWND button;
		button = GetDlgItem( hwnd, checkbox_to_flag_table[i][1] );
		if( SendMessage( button, BM_GETCHECK, 0, 0 ) == BST_CHECKED )
		{	
			flags |= checkbox_to_flag_table[i][0];
		}
		else if( SendMessage( button, BM_GETCHECK, 0, 0 ) == BST_INDETERMINATE )
		{
			diff_flags |= checkbox_to_flag_table[i][0];
		}
	}
	
	return flags;
}

void EditPolyObject::SetFaceFlagMask( HWND parent, FlagType mask, FlagType diff_flags )
{
	int i;
	HWND hwnd;
	
	for( i = 0; checkbox_to_flag_table[i][0]; i++ )
	{
		hwnd = GetDlgItem( parent, checkbox_to_flag_table[i][1] );
		if( mask & checkbox_to_flag_table[i][0] )
		{			
			SendMessage( hwnd, BM_SETCHECK, BST_CHECKED, 0 );;		
		}
		else if( diff_flags & checkbox_to_flag_table[i][0] )
		{			
			SendMessage( hwnd, BM_SETCHECK, BST_INDETERMINATE, 0 );;		
		}
		else
		{
			SendMessage( hwnd, BM_SETCHECK, BST_UNCHECKED, 0 );;		
		}
	}	
}

void EditPolyObject::DisplayFaceFlags( void )
{
	// Neversoft addition for face flags
	if(	/*( selLevel == SL_POLY ) ||*/
		( selLevel == EP_SL_OBJECT ) ||
		( selLevel == EP_SL_ELEMENT ) ||
		( selLevel == EP_SL_FACE ))
	{
		FlagType mask, diff;
		
		mask = 0;
		diff = 0;
		
		BitArray faceSel;
		MNMesh &mesh = GetMesh();
		mesh.getFaceSel(faceSel);

		if( faceSel.NumberSet() == 0 )
		{
			SetFaceFlagMask( hFaceFlags, 0, 0 );
			UpdatePassFlags();
		}
		else
		{			
			// Get the face-data channel from the incoming mesh
			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
			if ( pFDMgr == NULL )
			{
				SetFaceFlagMask( hFaceFlags, 0, 0 );
				UpdatePassFlags();
				return;
			}

			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			if( fdc == NULL ) 
			{		
				SetFaceFlagMask( hFaceFlags, 0, 0 );
				UpdatePassFlags();
				return;
			}

			fdc->GetValue( faceSel, mask, diff );	
			SetFaceFlagMask( hFaceFlags, mask, diff );
			UpdatePassFlags();
		}
	}
}

void UpdateCASFaceFlagsSel(HWND hWnd, EditPolyObject* eo)
{
	bool buttonStates[32] = {false};		// The last state of the button
	bool buttonInconsistent[32] = {false};	// True if the button isn't consistent accross all faces
	bool buttonUpdated[32] = {false};		// Used for determining first pass

	MNMesh& mesh=eo->GetMesh();

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));
	if (fdc == NULL)
		return;

	// Uncheck all buttons to start
	int id;
	
	for(id=0;id<32;id++)
	{
		// Turn the button on
		HWND hwndButton = GetDlgItem(hWnd, IDC_CMATCHF1+id);
		ICustButton* button = GetICustButton(hwndButton);
		
		button->SetCheck(false);
		ReleaseICustButton(button);
	}

	int numFaces=mesh.FNum();
	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	for(int i=0;i<numFaces;i++)
	{
		if (faceSel[i])
		{
			CASFlagType flag;
			// Turn on buttons based on data within the channel
			fdc->GetValue(i,flag);

			for(id=0;id<32;id++)
			{
				if (flag & (1 << id))
				{
					// Turn the button on
					HWND hwndButton = GetDlgItem(hWnd, IDC_CMATCHF1+id);
					ICustButton* button = GetICustButton(hwndButton);
					
					if (buttonUpdated[id]==true && buttonStates[id]==false)
						buttonInconsistent[id]=true;

					buttonStates[id]=true;
					button->SetCheck(true);

					if (buttonInconsistent[id])
					{
						button->SetHighlightColor(RGB(0,255,0));
						button->SetCheckHighlight(true);
					}
					else
					{
						button->SetHighlightColor(RGB(255,255,255));
						button->SetCheckHighlight(true);
					}

					ReleaseICustButton(button);
				}
				else
				{
					// Turn the button off
					HWND hwndButton = GetDlgItem(hWnd, IDC_CMATCHF1+id);
					ICustButton* button = GetICustButton(hwndButton);

					if (buttonStates[id] || buttonInconsistent[id])
					{
						// This button was turned on before
						buttonInconsistent[id]=true;
						buttonStates[id]=true;
						button->SetCheck(true);
						button->SetHighlightColor(RGB(0,255,0));
						button->SetCheckHighlight(true);
					}
					else
					{
						buttonStates[id]=false;
						button->SetCheck(false);
						button->SetHighlightColor(RGB(255,255,255));
						button->SetCheckHighlight(true);
					}

					ReleaseICustButton(button);
				}

				buttonUpdated[id]=true;
			}
		}
	}
}

void CASFlagSet(HWND hWnd, EditPolyObject* eo)
{
	bool buttonStates[32] = {false};	// Fills array to 0's

	MNMesh& mesh=eo->GetMesh();

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		//SetFaceFlagMask( hFaceFlags, 0, 0 );
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

	if (fdc == NULL)
	{
		// The CAS Face data channel doesn't exist, add it
		fdc = new CASFaceFlagsData();
		pFDMgr->AddFaceDataChan( fdc );
		fdc->FacesCreated( 0, mesh.FNum() );
	}

	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	for(int i=0;i<mesh.FNum();i++)
	{
		if (faceSel[i])
		{
			CASFlagType flag=0;

			// Build flag values
			for(int id=0;id<32;id++)
			{
				HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
				ICustButton* button = GetICustButton(hwndButton);

				if (button->IsChecked())
					flag |= (1 << id);

				ReleaseICustButton(button);
			}

			fdc->SetValue(i,flag);
		}
	}
}

void CASFlagSetMask(HWND hWnd, EditPolyObject* eo)
{
	bool buttonStates[32] = {false};	// Fills array to 0's

	MNMesh& mesh=eo->GetMesh();

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		//SetFaceFlagMask( hFaceFlags, 0, 0 );
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

	if (fdc == NULL)
	{
		// The CAS Face data channel doesn't exist, add it
		fdc = new CASFaceFlagsData();
		pFDMgr->AddFaceDataChan( fdc );
		fdc->FacesCreated( 0, mesh.FNum() );
	}

	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	for(int i=0;i<mesh.FNum();i++)
	{
		if (faceSel[i])
		{
			CASFlagType flag;
			fdc->GetValue(i,flag);

			// Build flag values
			for(int id=0;id<32;id++)
			{
				HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
				ICustButton* button = GetICustButton(hwndButton);

				if (button->IsChecked())
					flag |= (1 << id);

				ReleaseICustButton(button);
			}

			fdc->SetValue(i,flag);
		}
	}
}

void CASFlagClearMask(HWND hWnd, EditPolyObject* eo)
{
	bool buttonStates[32] = {false};	// Fills array to 0's

	MNMesh& mesh=eo->GetMesh();

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		//SetFaceFlagMask( hFaceFlags, 0, 0 );
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

	if (fdc == NULL)
	{
		// The CAS Face data channel doesn't exist, add it
		fdc = new CASFaceFlagsData();
		pFDMgr->AddFaceDataChan( fdc );
		fdc->FacesCreated( 0, mesh.FNum() );
	}

	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	for(int i=0;i<mesh.FNum();i++)
	{
		if (faceSel[i])
		{
			CASFlagType flag;
			fdc->GetValue(i,flag);

			// Build flag values
			for(int id=0;id<32;id++)
			{
				HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
				ICustButton* button = GetICustButton(hwndButton);

				if (button->IsChecked())
					flag &= ~(1 << id);

				ReleaseICustButton(button);
			}

			fdc->SetValue(i,flag);
		}
	}
}

void CASFlagClear(HWND hWnd)
{
	for(int id=0;id<32;id++)
	{
		HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
		ICustButton* button = GetICustButton(hwndButton);

		button->SetCheck(false);

		ReleaseICustButton(button);
	}
}

void CASInitButtons(HWND hWnd)
{
	for(int id=0;id<32;id++)
	{
		HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
		ICustButton* button = GetICustButton(hwndButton);

		if (button)
		{
			button->SetType(CBT_CHECK);
			ReleaseICustButton(button);
		}

		hwndButton = GetDlgItem(hWnd, IDC_CMATCHF1+id);
		button = GetICustButton(hwndButton);

		if (button)
		{
			button->SetType(CBT_CHECK);
			button->SetCheckHighlight(true);
			button->SetHighlightColor(RGB(255,255,255));
			ReleaseICustButton(button);
		}
	}
}

void CASSelMatch(HWND hWnd, EditPolyObject* eo)
{
	MNMesh& mesh=eo->GetMesh();

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		//SetFaceFlagMask( hFaceFlags, 0, 0 );
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

	if (fdc == NULL)
	{
		// The CAS Face data channel doesn't exist, add it
		fdc = new CASFaceFlagsData();
		pFDMgr->AddFaceDataChan( fdc );
		fdc->FacesCreated( 0, mesh.FNum() );
	}

	BitArray faceSel;
	mesh.getFaceSel(faceSel);
	faceSel.ClearAll();
	mesh.FaceSelect(faceSel);

	for(int i=0;i<mesh.FNum();i++)
	{
		// Go through the mesh and find faces with the appropriate flags set
		CASFlagType flag;
		fdc->GetValue(i,flag);

		if (IsDlgButtonChecked(hWnd,IDC_EXACTMATCH)==BST_UNCHECKED)
		{
			// 1 || 2

			// Find the faces with the specified CAS flags
			for(int id=0;id<32;id++)
			{
				HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
				ICustButton* button = GetICustButton(hwndButton);

				if (button->IsChecked())
				{
					if (flag & (1 << id))
					{
						faceSel.Set(i);
						ReleaseICustButton(button);
						break;
					}
				}

				ReleaseICustButton(button);
			}
		}
		else
		{
			// 1 && 2

			// Find the faces with the specified CAS flags
			bool bAllFlags=true;
			for(int id=0;id<32;id++)
			{
				HWND hwndButton = GetDlgItem(hWnd, IDC_LOCKF1+id);
				ICustButton* button = GetICustButton(hwndButton);

				if (button->IsChecked())
				{
					if (!(flag & (1 << id)))
					{
						bAllFlags=false;
						ReleaseICustButton(button);
						break;
					}
				}

				ReleaseICustButton(button);
			}

			// Check if all the flags are set (and we should thus select the face
			if (bAllFlags)
				faceSel.Set(i);
		}
	}

	mesh.FaceSelect(faceSel);
	gInterface->ForceCompleteRedraw();
}

static INT_PTR CALLBACK ExtFaceFlagsDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	//EditTriObject *eo = (EditTriObject*)GetWindowLongPtr(hWnd,GWLP_USERDATA);
	static EditPolyObject* eo;

	switch(msg)
	{
	case WM_INITDIALOG:
		eo = (EditPolyObject*)lParam;

		CASInitButtons(hWnd);
		UpdateCASFaceFlagsSel(hWnd, eo);
		
		//SetWindowLongPtr(hWnd,GWLP_USERDATA,lParam);
		//eo->hFaceFlags = hWnd;
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_SELMATCHING:
			// Select matching faces
			CASSelMatch(hWnd, eo);
			UpdateCASFaceFlagsSel(hWnd, eo);
			return TRUE;
		
		case IDC_MASKSET:
			CASFlagSetMask(hWnd, eo);
			UpdateCASFaceFlagsSel(hWnd, eo);
			return TRUE;

		case IDC_MASKCLEAR:
			CASFlagClearMask(hWnd, eo);
			UpdateCASFaceFlagsSel(hWnd, eo);
			return TRUE;

		case IDC_EXTFLAGSET:
			CASFlagSet(hWnd, eo);
			UpdateCASFaceFlagsSel(hWnd, eo);
			return TRUE;
		
		case IDC_EXTFLAGCLEAR:
			CASFlagClear(hWnd);
			return TRUE;

		default:
			if (LOWORD(wParam)>=IDC_LOCKF1 && LOWORD(wParam)<=IDC_LOCKF32)
			{
				HWND hwndButton = GetDlgItem(hWnd, LOWORD(wParam));
				ICustButton* button = GetICustButton(hwndButton);

				if (button)
				{
					if (button->IsChecked())
						button->SetCheck(true);
					else
						button->SetCheck(false);

					ReleaseICustButton(button);
				}
			}

			if (LOWORD(wParam)>=IDC_CMATCHF1 && LOWORD(wParam)<=IDC_CMATCHF32)
			{
				HWND hwndButton = GetDlgItem(hWnd, LOWORD(wParam));
				ICustButton* button = GetICustButton(hwndButton);

				if (button)
				{
					// Don't allow setting change
					if (button->IsChecked())
						button->SetCheck(false);
					else
						button->SetCheck(true);

					ReleaseICustButton(button);
				}				
			}
		}
	}

	return FALSE;
}

static ProcCheckbox(HWND hWnd, int ID)
{
	switch(IsDlgButtonChecked(hWnd,ID))
	{
	case BST_INDETERMINATE:
		CheckDlgButton(hWnd,ID,BST_UNCHECKED);
		break;
	}
}

static INT_PTR CALLBACK FacePassDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	if (bLockPassUpdate)
		return FALSE;

	EditPolyObject *eo = (EditPolyObject*)GetWindowLongPtr(hWnd,GWLP_USERDATA);

	switch(msg)
	{
		case WM_INITDIALOG:
			eo = (EditPolyObject*)lParam;		
			SetWindowLongPtr(hWnd,GWLP_USERDATA,lParam);
			eo->hPassFlags = hWnd;
			eo->UpdatePassFlags();
			break;

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
			case IDC_PASS1:
				ProcCheckbox(hWnd, IDC_PASS1);
				eo->SetPassFlags();
				break;

			case IDC_PASS2:
				ProcCheckbox(hWnd, IDC_PASS2);
				eo->SetPassFlags();
				break;

			case IDC_PASS3:
				ProcCheckbox(hWnd, IDC_PASS3);
				eo->SetPassFlags();
				break;

			case IDC_PASS4:
				ProcCheckbox(hWnd, IDC_PASS4);
				eo->SetPassFlags();
				break;

			case IDC_FINDFACES:
				eo->FindPassFaces();
				break;
			}
			break;

		default:
			return FALSE;
	}

	return TRUE;
}

static INT_PTR CALLBACK FaceFlagsDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) 
{
	EditPolyObject *eo = (EditPolyObject*)GetWindowLongPtr(hWnd,GWLP_USERDATA);
	
	switch (msg) 
	{
		case WM_INITDIALOG:
			eo = (EditPolyObject*)lParam;		
			SetWindowLongPtr(hWnd,GWLP_USERDATA,lParam);
			eo->hFaceFlags = hWnd;
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) 
			{
				case IDC_SELECT_FACES:
				{
					if( DialogBoxParam( hInstance, MAKEINTRESOURCE(IDD_NS_SEL_FACE_FACES),
						gInterface->GetMAXHWnd(), SelectByFlagsDlgProc, (LPARAM) eo ))
					{
					}
					break;
				}
				// The order here matters. The flags are inferred from this order so it must match the actual flags				
				case IDC_SKATABLE:
				case IDC_NOT_SKATABLE:
				case IDC_WALL_RIDABLE:
				case IDC_VERT_POLY:
				case IDC_NON_COLLIDABLE:
				case IDC_TRIGGER:
				case IDC_CAM_COLLIDABLE:
				case IDC_NO_SKATER_SHADOW:
				case IDC_SKATER_SHADOW:
				case IDC_NO_SKATER_SHADOW_WALL:
				case IDC_UNDER_OK:
				case IDC_INVISIBLE:
				case IDC_RENDER_SEPARATE:
				{
					FlagType face_flags, flag_set, diff_flags;
					bool currently_checked;
					HWND current_checkbox;

					current_checkbox = GetDlgItem( hWnd, LOWORD(wParam));
					currently_checked = ( SendMessage( current_checkbox, BM_GETCHECK, 0, 0 ) == BST_CHECKED );
					if( !currently_checked )
					{
						SendMessage( current_checkbox, BM_SETCHECK, 0, 0 );
					}

					face_flags = ( 1 << ( LOWORD(wParam) - IDC_SKATABLE ));
					flag_set = eo->GetFaceFlagsSet( hWnd, diff_flags );
					// If they're setting a flag, 
					if( currently_checked )
					{
						if( ValidFlagConfiguration( flag_set ) == false )
						{
							MessageBox( hWnd, "Invalid Flag Configuration", "Invalid!", MB_OK );
							SendMessage( current_checkbox, BM_SETCHECK, BST_UNCHECKED, 0 );
							break;
						}
					}
					
					if( !currently_checked )
					{
						eo->ClearFaceFlags( face_flags );
					}
					else
					{
						eo->SetFaceFlags( face_flags );
					}
					break;
				}										

				default:
					break;
			}

		default:
			return FALSE;
	}

	return TRUE;
}

void EditPolyObject::SelectByCASFaceFlags( CASFlagType &flags, bool exact_match )
{
	int i;
	theHold.Begin();
	BitArray nfs;
	
	nfs.SetSize (GetMesh().FNum());
	nfs.ClearAll ();
	
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>( GetMesh().GetInterface( FACEDATAMGR_INTERFACE ));
	
	if( pFDMgr )
	{	
		CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

		if (!fdc)
			return;

		for( i = 0; i < GetMesh().FNum(); i++ ) 
		{			
			CASFlagType value;

			fdc->GetValue( i, value );
			if( exact_match )
			{
				if( value == flags )
				{
					nfs.Set(i);
				}
			}
			else
			{
				if( value & flags )
				{
					nfs.Set(i);
				}
			}
			
		}
	}
	
	SetFaceSel (nfs, this, ip->GetTime());
	theHold.Accept( "Select by Face Flags" );
	LocalDataChanged ();
	ip->RedrawViews (ip->GetTime());
}

void EditPolyObject::SetCASFaceFlags( CASFlagType flags)
{
	int i;
	if( !ip ) 
	{
		return;
	}

	MNMesh &mesh = GetMesh();

	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	if( faceSel.NumberSet() == 0 ) 
	{
		return;
	}

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));
	if( fdc == NULL ) 
	{		
		// The mesh does not have our face-data channel so we will add it here
		fdc = new CASFaceFlagsData();
		fdc->FacesCreated( 0, mesh.FNum());

		pFDMgr->AddFaceDataChan( fdc );
	}

	for( i = 0; i < faceSel.GetSize(); i++ )
	{
		if( faceSel[i] )
		{
			CASFlagType new_value;

			fdc->GetValue( i, new_value );
			new_value |= flags;
			fdc->SetValue( i, new_value );	
		}		
	}

	collapseCASFlagData = *fdc;
}

void EditPolyObject::ClearCASFaceFlags( CASFlagType flags )
{
	int i;
	if( !ip ) 
	{
		return;
	}

	MNMesh &mesh = GetMesh();

	BitArray faceSel;
	mesh.getFaceSel(faceSel);

	if( faceSel.NumberSet() == 0 ) 
	{
		return;
	}

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if ( pFDMgr == NULL )
	{
		return;
	}

	CASFaceFlagsData* fdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));
	if( fdc == NULL ) 
	{		
		// The mesh does not have our face-data channel so we will add it here
		fdc = new CASFaceFlagsData();
		fdc->FacesCreated( 0, mesh.FNum());

		pFDMgr->AddFaceDataChan( fdc );
	}

	for( i = 0; i < faceSel.GetSize(); i++ )
	{
		if( faceSel[i] )
		{
			FlagType new_value;

			fdc->GetValue( i, new_value );
			new_value &= ~flags;
			fdc->SetValue( i, new_value );	
		}		
	}

	collapseCASFlagData = *fdc;
}

void EditPolyObject::DisplayCASFaceFlags( void )
{
	// Neversoft addition for face flags
	if(	( selLevel == EP_SL_OBJECT ) ||
		( selLevel == EP_SL_ELEMENT ) ||
		( selLevel == EP_SL_FACE ))
	{
		UpdateCASFaceFlagsSel(hExtFaceFlags, this);
	}
}

void EditPolyObject::UpdatePassFlags()
{
	if(	( selLevel == EP_SL_OBJECT ) ||
		( selLevel == EP_SL_ELEMENT ) ||
		( selLevel == EP_SL_FACE ))
	{
		bLockPassUpdate = TRUE;

		MNMesh& mesh   = GetMesh();
		int numFaces = mesh.FNum();

		// Clear all the currently set flags (if any) from the UI
		CheckDlgButton(hPassFlags,IDC_PASS1,BST_CHECKED);
		CheckDlgButton(hPassFlags,IDC_PASS2,BST_UNCHECKED);
		CheckDlgButton(hPassFlags,IDC_PASS3,BST_UNCHECKED);
		CheckDlgButton(hPassFlags,IDC_PASS4,BST_UNCHECKED);
		
		// Get the face-data channel from the incoming mesh
		IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
		if ( pFDMgr == NULL )
		{
			bLockPassUpdate = FALSE;
			return;
		}

		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
		if( fdc == NULL ) 
		{
			bLockPassUpdate = FALSE;
			return;
		}

		FlagType flag;
		UINT     state;

		// Find the first selected face
		int firstSelFace = -1;
		int i;

		BitArray faceSel;
		mesh.getFaceSel(faceSel);

		for(i=0;i<numFaces;i++)
		{
			if (faceSel[i])
			{
				fdc->GetValue(i,flag);

				if (flag & mFD_PASS_1_DISABLED)
					CheckDlgButton(hPassFlags,IDC_PASS1,BST_UNCHECKED);

				if (flag & mFD_PASS_2_ENABLED)
					CheckDlgButton(hPassFlags,IDC_PASS2,BST_CHECKED);

				if (flag & mFD_PASS_3_ENABLED)
					CheckDlgButton(hPassFlags,IDC_PASS3,BST_CHECKED);

				if (flag & mFD_PASS_4_ENABLED)
					CheckDlgButton(hPassFlags,IDC_PASS4,BST_CHECKED);

				firstSelFace = i;
				break;
			}
		}

		// Check for any faces that create inconsistencies with the first selected face
		for(i=firstSelFace + 1;i<numFaces;i++)
		{
			if (faceSel[i])
			{
				fdc->GetValue(i,flag);

				state = IsDlgButtonChecked(hPassFlags,IDC_PASS1);

				if ((  (flag & mFD_PASS_1_DISABLED) && state==BST_CHECKED) ||
					( !(flag & mFD_PASS_1_DISABLED) && state==BST_UNCHECKED))
				{
					CheckDlgButton(hPassFlags,IDC_PASS1,BST_INDETERMINATE);
				}

				state = IsDlgButtonChecked(hPassFlags,IDC_PASS2);

				if ((  (flag & mFD_PASS_2_ENABLED) && state==BST_UNCHECKED) ||
					( !(flag & mFD_PASS_2_ENABLED) && state==BST_CHECKED))
				{
					CheckDlgButton(hPassFlags,IDC_PASS2,BST_INDETERMINATE);
				}

				state = IsDlgButtonChecked(hPassFlags,IDC_PASS3);

				if ((  (flag & mFD_PASS_3_ENABLED) && state==BST_UNCHECKED) ||
					( !(flag & mFD_PASS_3_ENABLED) && state==BST_CHECKED))
				{
					CheckDlgButton(hPassFlags,IDC_PASS3,BST_INDETERMINATE);
				}

				state = IsDlgButtonChecked(hPassFlags,IDC_PASS4);

				if ((  (flag & mFD_PASS_4_ENABLED) && state==BST_UNCHECKED) ||
					( !(flag & mFD_PASS_4_ENABLED) && state==BST_CHECKED))
				{
					CheckDlgButton(hPassFlags,IDC_PASS4,BST_INDETERMINATE);
				}
			}
		}

	}

	bLockPassUpdate = FALSE;
}

void EditPolyObject::SetPassFlags()
{
	if(	( selLevel == EP_SL_OBJECT ) ||
		( selLevel == EP_SL_ELEMENT ) ||
		( selLevel == EP_SL_FACE ))
	{
		MNMesh& mesh   = GetMesh();
		int numFaces = mesh.FNum();
		
		BitArray faceSel;
		mesh.getFaceSel(faceSel);

		if( faceSel.NumberSet() == 0 )
			return;

		// Get the face-data channel from the incoming mesh
		IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
		if ( pFDMgr == NULL )
			return;

		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
		if( fdc == NULL ) 
		{
			fdc = new FaceFlagsData();
			pFDMgr->AddFaceDataChan( fdc );
			fdc->FacesCreated( 0, mesh.FNum() );
		}

		FlagType flag;

		for(int i=0;i<numFaces;i++)
		{
			if (faceSel[i])
			{
				fdc->GetValue(i,flag);

				switch(IsDlgButtonChecked(hPassFlags,IDC_PASS1))
				{
				case BST_UNCHECKED:
					flag |= mFD_PASS_1_DISABLED;
					break;

				case BST_CHECKED:
					flag &= ~mFD_PASS_1_DISABLED;
					break;
					// No action for INDETERMINATE
				}
				
				switch(IsDlgButtonChecked(hPassFlags,IDC_PASS2))
				{
				case BST_CHECKED:
					flag |= mFD_PASS_2_ENABLED;
					break;

				case BST_UNCHECKED:
					flag &= ~mFD_PASS_2_ENABLED;
					break;
				}

				switch(IsDlgButtonChecked(hPassFlags,IDC_PASS3))
				{
				case BST_CHECKED:
					flag |= mFD_PASS_3_ENABLED;
					break;

				case BST_UNCHECKED:
					flag &= ~mFD_PASS_3_ENABLED;
					break;
				}

				switch(IsDlgButtonChecked(hPassFlags,IDC_PASS4))
				{
				case BST_CHECKED:
					flag |= mFD_PASS_4_ENABLED;
					break;

				case BST_UNCHECKED:
					flag &= ~mFD_PASS_4_ENABLED;
					break;
				}

				fdc->SetValue(i,flag);
			}
		}

		collapseFlagData = *fdc;
	}
}

static INT_PTR CALLBACK FindPassDlgProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static EditPolyObject* epo;

	switch(msg)
	{
	case WM_INITDIALOG:
		epo = (EditPolyObject*)lParam;
		CheckRadioButton(hWnd,IDC_SEARCHALL,IDC_SEARCHANY,IDC_SEARCHALL);
		break;

	case WM_CLOSE:
		EndDialog(hWnd,0);
		break;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_DONE:
			EndDialog(hWnd,0);
			break;

		case IDC_FINDFACES:
			{
				__int64 i64SearchMask = 0;

				// Build search mask
				if (IsDlgButtonChecked(hWnd,IDC_PASS1)==BST_UNCHECKED)
					i64SearchMask |= mFD_PASS_1_DISABLED;

				if (IsDlgButtonChecked(hWnd,IDC_PASS2)==BST_CHECKED)
					i64SearchMask |= mFD_PASS_2_ENABLED;

				if (IsDlgButtonChecked(hWnd,IDC_PASS3)==BST_CHECKED)
					i64SearchMask |= mFD_PASS_3_ENABLED;

				if (IsDlgButtonChecked(hWnd,IDC_PASS4)==BST_CHECKED)
					i64SearchMask |= mFD_PASS_4_ENABLED;
			
				// Select the pass faces
				MNMesh& mesh = epo->GetMesh();

				// Get the face-data channel from the incoming mesh
				IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
				if ( pFDMgr == NULL )
				{
					//SetFaceFlagMask( hFaceFlags, 0, 0 );
					break;
				}

				FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

				BitArray faceSel;
				mesh.getFaceSel(faceSel);
				faceSel.ClearAll();

				FlagType flag;

				for(int i=0;i<mesh.FNum();i++)
				{
					if (fdc)
						fdc->GetValue(i,flag);
					else
						flag = 0;

					if (IsDlgButtonChecked(hWnd,IDC_SEARCHALL))
					{
						if ((flag & i64SearchMask) == i64SearchMask)
							faceSel.Set(i);
					}

					if (IsDlgButtonChecked(hWnd,IDC_SEARCHANY))
					{
						if ((flag & i64SearchMask) || !(flag & mFD_PASS_1_DISABLED) == !(i64SearchMask & mFD_PASS_1_DISABLED))
							faceSel.Set(i);
					}
				}

				mesh.FaceSelect(faceSel);
				gInterface->ForceCompleteRedraw();
			}
			break;
		}
		break;

	default:
		return FALSE;
	}

	return TRUE;
}

void EditPolyObject::FindPassFaces()
{
	DialogBoxParam(hInstance,
	               MAKEINTRESOURCE(IDD_PASS_SELECT),
	               hPassFlags,
			       FindPassDlgProc,
				   (LPARAM)this);

	UpdatePassFlags();
}


	// CAS Face Flags
	//void SelectByCASFaceFlags( CASFlagType &flags, bool exact_match );
	//void SetCASFaceFlags( CASFlagType flags );
	//void ClearCASFaceFlags( CASFlagType flags );
	//void DisplayCASFaceFlags( void );

	// Pass Flags
	//void UpdatePassFlags();
	//void SetPassFlags();
	//void FindPassFaces();



}	// end namespace EditPoly
