#include "FuncEnter.h"

/*
	FindMap.cpp
	Debug plugin for locating specific maps within the scene
*/

#include "FindMap.h"
#include "Resource.h"
#include "../Material/NExtMat.h"
#include "../Material/NExtMultiMat.h"
#include "../Texture/NExtTexture.h"
#include "../misc/maxutil.h"
#include "../PropEdit/ParseFuncs.h"
#include "../UI/PathSelect.h"
#include "../UI/OkToAll.h"
#include "../misc/find.h"
#include <io.h>
#include <sys/stat.h>
#include <direct.h>

#define ERR_MAP_NOT_FOUND "Map Not Found!"

static FindMapToolClassDesc theFindMapToolDesc;
ClassDesc2* GetFindMapToolDesc() { FUNC_ENTER("GetFindMapToolDesc");  return &theFindMapToolDesc; }
/*
bool FindMapTool::FindFile(char* path, char* file, char* location)
{
	// Calling findfirst/findnext is too slow as there appears to
	// be no way of not running through all files no luck with FindFirstEx
	// despite flags either (also seemingly incorrect ver???)
	
	// Resorting to the rather less elegant, but effective way of getting
	// the data from a fake list box with LB_DIR <sigh>

	_chdir(path);

	char WorkingDir[256];
	_getcwd(WorkingDir,255);

	pbar->SetCaption(WorkingDir);

	// Check if the file exists in the current directory
	FILE* fp = fopen(file, "r");

	if (fp)
	{
		fclose(fp);

		if (location)
		{
			strcpy(location, WorkingDir);
			strcat(location, "\\");
			strcat(location, file);
		}

		return true;
	}

	char wildcard[2048];
	strcpy(wildcard, WorkingDir);
	strcat(wildcard, "\\*.*");

	SendDlgItemMessage(hwnd, IDC_FAKEDIRLIST, LB_RESETCONTENT, 0, 0);
	SendDlgItemMessage(hwnd, IDC_FAKEDIRLIST, LB_DIR, (WPARAM)DDL_DIRECTORY|DDL_EXCLUSIVE, (LPARAM)wildcard);
	int count = SendDlgItemMessage(hwnd, IDC_FAKEDIRLIST, LB_GETCOUNT, 0, 0);

	char buf[1024];

	for(int i = 0; i < count; i++)
	{
		SendDlgItemMessage(hwnd, IDC_FAKEDIRLIST, LB_RESETCONTENT, 0, 0);
		SendDlgItemMessage(hwnd, IDC_FAKEDIRLIST, LB_DIR, (WPARAM)DDL_DIRECTORY|DDL_EXCLUSIVE, (LPARAM)wildcard);
		SendDlgItemMessage(hwnd, IDC_FAKEDIRLIST, LB_GETTEXT, (WPARAM)i, (LPARAM)buf);
		char* name = buf + 1;
		name[strlen(name)-1] = 0;

		if (strcmp(name, ".") == 0 ||
			strcmp(name, "..") == 0)
			continue;

		if (FindFile(name,file,location))
			return true;

		_chdir(WorkingDir);
	}

	return false;
}
*/

bool FindMapTool::FindMapFile(char* path, char* file, char* location)
{ FUNC_ENTER("FindMapTool::FindMapFile"); 
	MapLocator mapLocator;
	mapLocator.mapName = file;

	Link<MapLocator>* link = mapList.Find(&mapLocator);

	if (link)
	{
		if (link->data.bWasLocated)
		{
			strcpy(location, link->data.mapPath);
			return true;
		}
		else
			return false;
	}

	bool bFound = FindFile(path, file, location);

	if (bFound)
	{
		mapLocator.mapPath = location;
		mapLocator.bWasLocated = true;
		mapList.Add(&mapLocator);
		return true;
	}

	mapLocator.bWasLocated = false;
	mapList.Add(&mapLocator);
	return false;
}

bool FindMapTool::FindFile(char* path, char* file, char* location)
{ FUNC_ENTER("FindMapTool::FindFile"); 
	WIN32_FIND_DATA fdata;
	_chdir(path);

	char WorkingDir[256];
	_getcwd(WorkingDir,255);

	pbar->SetCaption(WorkingDir);

	// Check if the file exists in the current directory
	FILE* fp = fopen(file, "r");

	if (fp)
	{
		fclose(fp);

		if (location)
		{
			strcpy(location, WorkingDir);
			strcat(location, "\\");
			strcat(location, file);
		}

		return true;
	}

	HANDLE hFindFirst = FindFirstFile("*.", &fdata);

	if (!hFindFirst)
		return false;

	BOOL hFile = TRUE;

	while(hFile)
	{
		if (fdata.dwFileAttributes & _A_SUBDIR)
		{
			if (strcmp(fdata.cFileName,".")==0 ||
				strcmp(fdata.cFileName,"..")==0)
			{
				hFile = FindNextFile(hFindFirst, &fdata);
				continue;
			}

			if (FindFile(fdata.cFileName,file,location))
			{
				FindClose(hFindFirst);
				return true;
			}

			_chdir(WorkingDir);
		}

		hFile = FindNextFile(hFindFirst, &fdata);
	}

	FindClose(hFindFirst);
	return false;
}


FindMapTool::FindMapTool()
{ FUNC_ENTER("FindMapTool::FindMapTool"); 

}

FindMapTool::~FindMapTool()
{ FUNC_ENTER("FindMapTool::~FindMapTool"); 

}

void FindMapTool::BeginEditParams(Interface *ip,IUtil *iu)
{ FUNC_ENTER("FindMapTool::BeginEditParams"); 
	this->ip = ip;
	this->iu = iu;

	hwnd = ip->AddRollupPage(hInstance,MAKEINTRESOURCE(IDD_FINDMAP),DlgProc,"FindMap Tool");
	SetWindowLong(hwnd, GWL_USERDATA, (LONG)this);

	IEditMapName        = GetICustEdit(GetDlgItem(hwnd, IDC_MAPEDIT));
	IEditMapReplaceName = GetICustEdit(GetDlgItem(hwnd, IDC_MAPREPLACEEDIT));

}

void FindMapTool::EndEditParams(Interface *ip,IUtil *iu)
{ FUNC_ENTER("FindMapTool::EndEditParams"); 
	ReleaseICustEdit(IEditMapName);
	ReleaseICustEdit(IEditMapReplaceName);

	ip->DeleteRollupPage(hwnd);
}

void FindMapTool::SelectionSetChanged(Interface *ip,IUtil *iu)
{ FUNC_ENTER("FindMapTool::SelectionSetChanged"); 

}

BOOL FindMapTool::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("FindMapTool::DlgProc"); 
	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_MAPFIND:
			{
				FindMapTool* pthis = (FindMapTool*)GetWindowLong(hwnd, GWL_USERDATA);
				if (pthis)
					pthis->MapFind();
			}
			return TRUE;

		case IDC_MAPREPLACE:
			{
				FindMapTool* pthis = (FindMapTool*)GetWindowLong(hwnd, GWL_USERDATA);
				if (pthis)
					pthis->MapReplace();
			}
			return TRUE;

		case IDC_ADDPATH:
			{
				PathSelectDlg* pselect = new PathSelectDlg(hInstance, hwnd);

				if (!pselect->WasCancelled())
					SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_ADDSTRING, 0, (LPARAM)(char*)pselect->GetPath());

				delete pselect;
			}
			return TRUE;

		case IDC_REMOVEPATH:
			{
				int idx = SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_GETCURSEL, 0, 0);

				if (idx != LB_ERR)
					SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_DELETESTRING, (WPARAM)idx, 0);
			}
			return TRUE;

		case IDC_LOADPATHS:
			LoadPaths(hwnd);
			return TRUE;

		case IDC_SAVEPATHS:
			SavePaths(hwnd);
			return TRUE;

		case IDC_FIXPATHS:
			{
				FindMapTool* pthis = (FindMapTool*)GetWindowLong(hwnd, GWL_USERDATA);
				pthis->FixPaths();
			}
			return TRUE;
		}
	}

	return FALSE;
}

void FindMapTool::SelectNodes(Mtl* mtl)
{ FUNC_ENTER("FindMapTool::SelectNodes"); 
	INodeTab nodes;
	GetAllNodes(nodes);

	int nNodes = nodes.Count();

	for(int i = 0; i < nNodes; i++)
	{
		Mtl* nodemtl = nodes[i]->GetMtl();

		if (nodemtl)
		{
			if (nodemtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID)
			{
				int nMtls = nodemtl->NumSubMtls();

				for(int curmtl = 0; curmtl < nMtls; curmtl++)
				{
					Mtl* submtl = nodemtl->GetSubMtl(curmtl);

					if (submtl == mtl)
					{
						// Verify that this mtlID exists within the mesh
						Object* obj = nodes[i]->EvalWorldState(0).obj;
						
						if (obj->CanConvertToType(triObjectClassID))
						{
							TriObject* triobj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
							Mesh& mesh = triobj->GetMesh();

							for(int face = 0; face < mesh.numFaces; face++)
							{
								if (mesh.getFaceMtlIndex(face) % nMtls == curmtl)	
								{
									ip->SelectNode(nodes[i], FALSE);
									goto cont;
								}
							}
						}

						break;
					}
				}
			}
			else
			{
				if (nodemtl == mtl)
					ip->SelectNode(nodes[i], FALSE);
			}
cont:;
		}
	}
}

// Processes sele
void FindMapTool::ProcMtl(Mtl* mtl, char* wildcard)
{ FUNC_ENTER("FindMapTool::ProcMtl"); 
	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*>(mtl);

		int nPasses = next_mat->GetNumPasses();

		for(int pass = 0; pass < nPasses; pass++)
		{
			Texmap* tmap = next_mat->GetTexmap(pass);

			if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
			{
				INExtTexture* next_tex = dynamic_cast<INExtTexture*>(tmap);

				for(int plat = 0; plat < vNUM_PLATFORMS; plat++)
				{
					BitmapInfo* bmp = next_tex->GetBaseMap( plat );
					if (MatchPattern(CStr(bmp->Name()), CStr(wildcard)))
					{
						if (IsDlgButtonChecked(hwnd, IDC_FORCEMTLEDITOR) == BST_CHECKED)
						{
							ip->PutMtlToMtlEditor(mtl);
						}
						else
						{
							SelectNodes(mtl);
							if (ip->GetSelNodeCount() == 0)
							{
								ip->PutMtlToMtlEditor(mtl);
							}
						}
					}
				}
			}
		}
	}
	else if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID)
	{
		int nSubs = mtl->NumSubMtls();

		for(int cursub = 0; cursub < nSubs; cursub++)
		{
			Mtl* submtl = mtl->GetSubMtl(cursub);

			if (submtl)
				ProcMtl(submtl, wildcard);
		}
	}
}

// Processes sele
void FindMapTool::ProcMtlReplace(Mtl* mtl, char* wildcard, char* replace)
{ FUNC_ENTER("FindMapTool::ProcMtlReplace"); 
	CStr name = mtl->GetName();

	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*>(mtl);

		//int nPasses = next_mat->GetNumPasses();
		int nPasses = 4;

		for(int pass = 0; pass < nPasses; pass++)
		{
			Texmap* tmap = next_mat->GetTexmap(pass);

			if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
			{
				INExtTexture* next_tex = dynamic_cast<INExtTexture*>(tmap);

				for(int plat = 0; plat < vNUM_PLATFORMS; plat++)
				{
					// Replace manual mipmaps
					for(int i = 0; i < 5; i++)
					{
						BitmapInfo* bmp = next_tex->GetManualMap(plat, i);

						if (bmp)
						{
							CStr name = bmp->Name();

							if (name != CStr("None"))
							{
								if (MatchPattern(CStr(bmp->Name()), CStr(wildcard)))
								{
									CStr strWildcard;
									CStr strName = bmp->Name();

									strWildcard = wildcard;
									strWildcard = ReplaceStr(strWildcard, "*", "");
									strWildcard = ReplaceStr(strWildcard, "?", "");

									strName     = ReplaceStr(strName, strWildcard, replace, false);
									
									bmp->SetName(strName);
								}
							}
						}
					}

					// Replace basemaps
					if (plat == 1 &&
						next_tex->GetBaseMap( plat ) == next_tex->GetBaseMap(0))
						continue;

					if (plat == 2 &&
						(next_tex->GetBaseMap( plat ) == next_tex->GetBaseMap(0) ||
						 next_tex->GetBaseMap( plat ) == next_tex->GetBaseMap(1)))
						 continue;

					BitmapInfo* bmp = next_tex->GetBaseMap( plat );
					CStr name = bmp->Name();
					if (MatchPattern(CStr(bmp->Name()), CStr(wildcard)))
					{
						CStr strWildcard;
						CStr strName = bmp->Name();

						strWildcard = wildcard;
						strWildcard = ReplaceStr(strWildcard, "*", "");
						strWildcard = ReplaceStr(strWildcard, "?", "");

						strName     = ReplaceStr(strName, strWildcard, replace, false);
						
						if (strName == CStr(""))
							strName = "None";

						if (bmp)
							bmp->SetName(strName);
					}
				}
			}
		}
	}
	else if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID ||
			 mtl->ClassID() == Class_ID(MULTI_CLASS_ID,0))
	{
		int nSubs = mtl->NumSubMtls();

		for(int cursub = 0; cursub < nSubs; cursub++)
		{
			Mtl* submtl = mtl->GetSubMtl(cursub);

			if (submtl)
				ProcMtlReplace(submtl, wildcard, replace);
		}
	}
}

void FindMapTool::MapFind()
{ FUNC_ENTER("FindMapTool::MapFind"); 
	// Scan through all the materials in the scene and find any that match the wildcard
	// specified in the map name edit box
	char mapWildcard[256];
	IEditMapName->GetText(mapWildcard, 255);

	ip->ClearNodeSelection();

	MtlBaseLib* mtlLib = ip->GetSceneMtls();

	int nMtls = mtlLib->Count();

	pbar = new ProgressBar(hInstance,hwnd,"Searching for map...",0,nMtls);
	pbar->AllowCancel(true);

	for(int i = 0; i < nMtls; i++)
	{
		Mtl* mtl = (Mtl*)(*mtlLib)[i];
		ProcMtl(mtl, mapWildcard);

		pbar->SetVal(i);

		if (pbar->WasCanceled())
		{
			delete pbar;
			pbar = NULL;
			return;
		}
	}

	delete pbar;
	pbar = NULL;

	ip->ForceCompleteRedraw();
}

void FindMapTool::MapReplace()
{ FUNC_ENTER("FindMapTool::MapReplace"); 
	// Scan through all the materials in the scene and find any that match the wildcard
	// specified in the map name edit box
	char mapWildcard[256];
	char mapReplace[256];
	IEditMapName->GetText(mapWildcard, 255);
	IEditMapReplaceName->GetText(mapReplace, 255);

	ip->ClearNodeSelection();

	MtlBaseLib* mtlLib = ip->GetSceneMtls();

	int nMtls = mtlLib->Count();

	pbar = new ProgressBar(hInstance,hwnd,"Replacing map...",0,nMtls);
	pbar->AllowCancel(true);

	for(int i = 0; i < nMtls; i++)
	{
		Mtl* mtl = (Mtl*)(*mtlLib)[i];
		CStr name = mtl->GetName();
		ProcMtlReplace(mtl, mapWildcard, mapReplace);

		pbar->SetVal(i);

		if (pbar->WasCanceled())
		{
			delete pbar;
			pbar = NULL;
			return;
		}
	}

	delete pbar;
	pbar = NULL;

	ip->ForceCompleteRedraw();
}

void FindMapTool::LoadPaths(HWND hwnd)
{ FUNC_ENTER("FindMapTool::LoadPaths"); 
	OPENFILENAME ofn;
	char filename[256]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=gInterface->GetMAXHWnd();
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Path Files (*.pth)\0*.pth\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Load Paths...";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".pth");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	if (filename && strlen(filename) > 0)
	{
		FILE* fp = fopen(filename, "r");

		if (!fp)
		{
			char strErr[256];
			sprintf(strErr, "Failed to open file '%s' for reading", filename);
			MessageBox(hwnd, strErr, "Error reading in paths", MB_ICONSTOP|MB_OK);
			return;
		}
		
		SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_RESETCONTENT, 0, 0);

		char line[2048];
		
		while(!feof(fp))
		{
			if (!fgets(line, 2047, fp))
				break;

			int len = strlen(line);
			if (len > 0)
				line[len-1] = 0;

			SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_ADDSTRING, (WPARAM)0, (LPARAM)line);
		}
	}
}

void FindMapTool::SavePaths(HWND hwnd)
{ FUNC_ENTER("FindMapTool::SavePaths"); 
	OPENFILENAME ofn;
	char filename[256]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=gInterface->GetMAXHWnd();
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Path Files (*.pth)\0*.pth\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Save Paths...";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".pth");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);

	if (filename && strlen(filename) > 0)
	{
		char bufItem[2048];
		int count = SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_GETCOUNT, 0, 0);

		FILE* fp = fopen(filename, "w");

		if (!fp)
		{
			char strErr[256];
			sprintf(strErr, "Failed to open file '%s' for writting", filename);
			MessageBox(hwnd, strErr, "Error writting paths", MB_ICONSTOP|MB_OK);
			return;
		}

		for(int i = 0; i < count; i++)
		{
			SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_GETTEXT, (WPARAM)i, (LPARAM)bufItem);
			fprintf(fp, "%s\n", bufItem);
		}

		fclose(fp);
	}
}

void FindMapTool::FixPaths()
{ FUNC_ENTER("FindMapTool::FixPaths"); 
	// Scan through all the materials in the scene and find any that match the wildcard
	// specified in the map name edit box
	MessageBoxReset(ERR_MAP_NOT_FOUND);
	mapList.Clear();
	
	ip->ClearNodeSelection();

	MtlBaseLib* mtlLib;
	int nMtls;

	int i;
	int nSlotReplace, nMatLibReplace;

	HWND hwndMtlEditor = GetMtlEditor();

	if (!hwndMtlEditor)
	{
		MessageBox(ip->GetMAXHWnd(), "ABORTED! You MUST manually open the material editor before performing the Fix Maps operation.", "Material Editor not open", MB_ICONSTOP|MB_OK);
		return;
	}

	nChanges = 0;

	// Fix material editor slots , there are 24 material slots
	pbar = new ProgressBar(hInstance,hwnd,"Fixing material slot names...",0,24);

	for(i = 0; i < 24; i++)
	{
		Mtl* mtl = (Mtl*)ip->GetMtlSlot(i);
		ProcMtlReplaceFixPaths(mtl);

		pbar->SetVal(i);

		if (pbar->WasCanceled())
		{
			delete pbar;
			pbar = NULL;
			return;
		}
	}

	delete pbar;

	nSlotReplace = nChanges;
	////////////////////////////////// End slot fix

	nChanges = 0;

	mtlLib = &ip->GetMaterialLibrary();
	nMtls  = mtlLib->Count();

	// Fix material library contents
	pbar = new ProgressBar(hInstance,hwnd,"Fixing Material Library map paths...",0,nMtls);
	pbar->AllowCancel(true);

	for(i = 0; i < nMtls; i++)
	{
		Mtl* mtl = (Mtl*)(*mtlLib)[i];
		CStr name = mtl->GetName();
		ProcMtlReplaceFixPaths(mtl);

		pbar->SetVal(i);

		if (pbar->WasCanceled())
		{
			delete pbar;
			pbar = NULL;
			return;
		}
	}

	delete pbar;
	pbar = NULL;

	nMatLibReplace = nChanges;
	//////////////////////////////// End Fix Mtl Lib

	nChanges = 0;

	mtlLib = ip->GetSceneMtls();
	nMtls = mtlLib->Count();

	pbar = new ProgressBar(hInstance,hwnd,"Fixing map paths...",0,nMtls);
	pbar->AllowCancel(true);

	for(i = 0; i < nMtls; i++)
	{
		Mtl* mtl = (Mtl*)(*mtlLib)[i];
		CStr name = mtl->GetName();
		ProcMtlReplaceFixPaths(mtl);

		pbar->SetVal(i);

		if (pbar->WasCanceled())
		{
			delete pbar;
			pbar = NULL;
			return;
		}
	}

	delete pbar;
	pbar = NULL;

	if (IsDlgButtonChecked(hwnd, IDC_RECURSE) == BST_CHECKED)
	{
		char buf[20000];
		sprintf(buf,"FixPaths replaced %i scene maps.\n%i material lib maps\n %i material editor maps\n", nChanges, nMatLibReplace, nSlotReplace);
		
		Link<MapLocator>* curlink = mapList.GetHead();

		if (curlink)
			strcat(buf, "  The following had problems:\n\n");

		while(curlink)
		{
			if (!curlink->data.bWasLocated)
			{
				strcat(buf, curlink->data.mapName);
				strcat(buf, "\n");
			}

			curlink = curlink->next;
		}

		MessageBox(hwnd, buf, "FixPaths Completed", MB_ICONINFORMATION|MB_OK);
	}

	SendMessage(hwndMtlEditor, WM_CLOSE, 0, 0);
	ip->ForceCompleteRedraw();
}

CStr FindMapTool::ReplaceMap(CStr file)
{ FUNC_ENTER("FindMapTool::ReplaceMap"); 
	CStr newname;

	// Strip out the filename
	char* name = strrchr(file, '\\');
	name++;

	// Try to find this file in any of the scan directories
	int count = SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_GETCOUNT, 0, 0);
	char itemPath[2048];
	FILE* fp;

	for(int i = 0; i < count; i++)
	{
		SendDlgItemMessage(hwnd, IDC_PATHLIST, LB_GETTEXT, (WPARAM)i, (LPARAM)itemPath);

		// Construct the path we want to test
		newname = itemPath;
		newname += "\\";
		newname += name;

		fp = fopen(newname, "r");

		if (fp)
		{
			fclose(fp);

			CStr name1 = file;
			CStr name2 = newname;

			name1.toLower();
			name2.toLower();

			if (name1 != name2)
				nChanges++;

			return newname;
		}

		// If we're set to recurse paths recurse all directories below the given
		if (IsDlgButtonChecked(hwnd, IDC_RECURSE) == BST_CHECKED)
		{
			char location[2048];

			if (FindMapFile(itemPath, name, location))
			{
				CStr name1 = file;
				CStr name2 = newname;

				name1.toLower();
				name2.toLower();

				if (name1 != name2)
					nChanges++;

				return CStr(location);
			}
		}
	}

	// Couldn't find a match (warn the user)
	if (IsDlgButtonChecked(hwnd, IDC_CLEARMISSINGMAPS) == BST_CHECKED)
	{
		char strErr[2048];
		sprintf(strErr, "The map file '%s' could not be located in any of the scan directories.\nDo you want this map to be cleared?", (char*)file);
		int bResult = MessageBoxAll(hwnd, strErr, ERR_MAP_NOT_FOUND, MB_YESNO);

		if (bResult == IDYES)
		{
			if (file != CStr("None"))
				nChanges++;

			return CStr("None");
		}
		else
			return file;
	}

	char strErr[2048];
	sprintf(strErr, "The map file '%s' could not be located in any of the scan directories.", (char*)file);
	MessageBoxAll(hwnd, strErr, ERR_MAP_NOT_FOUND, MB_OK);

	return file;
}

BOOL FindMapTool::EnumChildProc(HWND hwnd, LPARAM lParam)
{ FUNC_ENTER("FindMapTool::EnumChildProc"); 
	HWND* pval = (HWND*)lParam;

	if (IsWindowVisible(hwnd))
	{
		char wndText[256];
		GetWindowText(hwnd, wndText, 255);

		// For some reason the material editor isn't considered a child
		// of the main MAX window, but the material editor's parent is the
		// main MAX window (Spy++)
		if (strstr(wndText, "Material Editor"))
		{
			if (GetParent(hwnd) == gInterface->GetMAXHWnd())
			{
				*pval = hwnd;
				return FALSE;
			}
		}
	}

	return TRUE;
}

HWND FindMapTool::GetMtlEditor()
{ FUNC_ENTER("FindMapTool::GetMtlEditor"); 
	// Ugly, yes, but no SDK method seems to exist for this
	HWND  starthwnd = GetDesktopWindow();		// Isn't child from main MAX window oddly enough
	HWND  hwndMtlEditor = NULL;
	//bool  bOpen = false;

	EnumChildWindows(starthwnd, EnumChildProc, (LPARAM)&hwndMtlEditor);
	return hwndMtlEditor;
}

void FindMapTool::ProcMtlReplaceFixPaths(Mtl* mtl)
{ FUNC_ENTER("FindMapTool::ProcMtlReplaceFixPaths"); 
	CStr name = mtl->GetName();

	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		INExtMaterial* next_mat = dynamic_cast<INExtMaterial*>(mtl);

		//int nPasses = next_mat->GetNumPasses();
		int nPasses = 4;

		for(int pass = 0; pass < nPasses; pass++)
		{
			Texmap* tmap = next_mat->GetTexmap(pass);

			if (tmap && tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
			{
				INExtTexture* next_tex = dynamic_cast<INExtTexture*>(tmap);

				for(int plat = 0; plat < vNUM_PLATFORMS; plat++)
				{
					// Replace manual mipmaps
					for(int i = 0; i < 5; i++)
					{
						BitmapInfo* bmp = next_tex->GetManualMap(plat, i);

						if (bmp)
						{
							CStr name = bmp->Name();

							if (name != CStr("None"))
							{
								if (name.Length() > 0)
									bmp->SetName(ReplaceMap(name));
							}
						}
					}

					// Replace basemaps
					if (plat == 1 &&
						next_tex->GetBaseMap( plat ) == next_tex->GetBaseMap(0))
						continue;

					if (plat == 2 &&
						(next_tex->GetBaseMap( plat ) == next_tex->GetBaseMap(0) ||
						 next_tex->GetBaseMap( plat ) == next_tex->GetBaseMap(1)))
						 continue;

					BitmapInfo* bmp = next_tex->GetBaseMap( plat );
					CStr name = bmp->Name();

					if (name.Length() > 0)
						bmp->SetName(ReplaceMap(name));
				}
			}
		}
	}
	else if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID ||
			 mtl->ClassID() == Class_ID(MULTI_CLASS_ID,0))
	{
		int nSubs = mtl->NumSubMtls();

		for(int cursub = 0; cursub < nSubs; cursub++)
		{
			Mtl* submtl = mtl->GetSubMtl(cursub);

			if (submtl)
				ProcMtlReplaceFixPaths(submtl);
		}
	}
}
