/*****************************************************************************
**																			**
**			              Neversoft Entertainment							**
**																		   	**
**				   Copyright (C) 1999 - All Rights Reserved				   	**
**																			**
******************************************************************************
**																			**
**	Project:		Core Library											**
**																			**
**	Module:			Support (SPT)	 										**
**																			**
**	File name:		classnode.cpp											**
**																			**
**	Created by:		06/14/99	-	mjb										**
**																			**
**	Description:	Class tracker ClassNode.								**
**																			**
*****************************************************************************/


/*****************************************************************************
**							  	  Includes									**
*****************************************************************************/
 
#ifdef __NOPT_FULL_DEBUG__

#include <core/defines.h>
#include <core/support.h>

/*****************************************************************************
**							  DBG Information								**
*****************************************************************************/


namespace Spt
{



/*****************************************************************************
**								  Externals									**
*****************************************************************************/


/*****************************************************************************
**								   Defines									**
*****************************************************************************/


/*****************************************************************************
**								Private Types								**
*****************************************************************************/


/*****************************************************************************
**								 Private Data								**
*****************************************************************************/


/*****************************************************************************
**								 Public Data								**
*****************************************************************************/

bool	ClassNode::placement = false;
  
/*****************************************************************************
**							  Private Prototypes							**
*****************************************************************************/


/*****************************************************************************
**							   Private Functions							**
*****************************************************************************/

/******************************************************************************
 *
 * Function:	ClassNode::InstanceNode::InstanceNode ( InstanceNode* p, const Class* inst )
 *
 * Description: ClassNode::InstanceNode constructor
 *				appends new ClassNode to start of instance list
 *
 * Parameters:	head	= pointer to head of instance list
 *				inst	= pointer to instance
 * 
 *****************************************************************************/

ClassNode::InstanceNode::InstanceNode( InstanceNode* head, const Class& inst )
: instance( inst ), next( head )
{
	

	if ( next )
	{ 
		next->prev = this;
	}

	prev = NULL;
}

/******************************************************************************
 *
 * Function:	ClassNode::InstanceNode::~InstanceNode( void )
 *
 * Description: ClassNode::InstanceNode destructor
 *				removes ClassNode from instance list
 *
 *****************************************************************************/

ClassNode::InstanceNode::~InstanceNode( void )
{
	
	
	if ( prev )
	{
		prev->next = next;
	}

	if ( next )
	{
		next->prev = prev;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void* 	ClassNode::InstanceNode::operator new( size_t size )
{
	return Dbg::NewInstanceNode( size );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void 	ClassNode::InstanceNode::operator delete( void* mem )
{
	Dbg::DeleteInstanceNode( mem );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void* 	ClassNode::operator new( size_t size )
{
	return Dbg::NewClassNode( size );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void 	ClassNode::operator delete( void* mem )
{
	Dbg::DeleteClassNode( mem );
}

/*****************************************************************************
**							   Public Functions								**
*****************************************************************************/

/******************************************************************************
 *
 * Function:	ClassNode::ClassNode ( ClassNode* p, char* name )
 *
 * Description: ClassNode constructor			
 *
 * Parameters:	p		= pointer to parent ClassNode
 *				name	= name of the class this ClassNode represents
 * 
 *****************************************************************************/

ClassNode::ClassNode( ClassNode* p, const char* name )
: class_name( name ), parent( p )
{
	 
	
	child_count		= 0;
	child			= NULL;
	next			= NULL;
	prev			= NULL;

	instance_count	= 0;
	placement_count	= 0;
	instance_list	= NULL;
	placement_list	= NULL;

	if ( parent )
	{
		if (( next = parent->child ))
		{
			next->prev = this;
		}
		parent->child = this;
		parent->child_count++;
	}
}

/******************************************************************************
 *
 * Function:	ClassNode::~ClassNode( void )
 *
 * Description: destroys a classnode and all it's children
 *
 *****************************************************************************/

ClassNode::~ClassNode( void )
{
	 
	
	Dbg_MsgAssert(( instance_count == 0 ), "%d instance(s) of %s still exist", 
									GetNumberOfInstances ( true ), GetName() );
	Dbg_MsgAssert(( placement_count == 0 ), "%d placement instance(s) of %s still exist", 
									GetNumberOfPlacementInstances ( true ), GetName() );
//	Unfortunately we can't guarantee this because of the order that destructors
//	are called with multiple inheritence.
///	Dbg_MsgAssert (( child == NULL ), ( "Child still connected" ));
///	Dbg_MsgAssert (( child_count == 0 ), ( "child count error" ));

	
	if ( child )
	{
		child->parent = NULL;
	}

	if ( prev )
	{
		prev->next = next;
	}
	else if ( parent )
	{
		parent->child = next;
	}

	if ( next )
	{
		next->prev = prev;
	}

	if ( parent )
	{
		parent->child_count--;
	}
}

/******************************************************************************
 *
 * Function:	ClassNode::RegisterInstance ( const Class& inst )
 *
 * Description: registers an instance of class to ClassNode
 *
 * Parameters:	ptr			= pointer to instance
 *
 *****************************************************************************/

void ClassNode::RegisterInstance( const Class& inst )
{
	 

	if( placement )
	{
		placement_count++;
		inst.m_is_placement = true;
		placement_list = new InstanceNode( placement_list, inst );
	}
	else
	{
		instance_count++;
		inst.m_is_placement = false;
		instance_list = new InstanceNode( instance_list, inst );
	}

}


/******************************************************************************
 *
 * Function:	ClassNode::UnregisterInstance ( const Class& inst ) 
 *
 * Description: unregisters an instance from ClassNode
 *
 * Parameters:	ptr = pointer to instance
 *
 *****************************************************************************/

void ClassNode::UnregisterInstance( const Class& inst )
{
	 
	

	InstanceNode**	p_it = ( inst.m_is_placement ) ? &placement_list : &instance_list;
	sint32*			p_count = ( inst.m_is_placement ) ? &placement_count : &instance_count;
	
	InstanceNode* iterator = *p_it;

	while ( iterator && ( &iterator->instance != &inst ))
	{
		iterator = iterator->next;
	}

	Dbg_MsgAssert( iterator, "Instance not found %x", &inst );

	if ( iterator == *p_it )
	{
		*p_it = iterator->next;
	}

	delete iterator;
	
	(*p_count)--;
	Dbg_Assert( (*p_count) >= 0 );

}


/******************************************************************************
 *
 * Function:	ClassNode::GetName()
 *
 * Description: returns the name of the class associated with the ClassNode
 *
 * Returns:		name of the class
 *
 ******************************************************************************/

const char*		ClassNode::GetName( void ) const
{
	 

	return class_name;
}


/******************************************************************************
 *
 * Function:	ClassNode::GetNumberOfInstances ( bool exact )
 *
 * Description:	retrieves number of instances of given class
 *
 * Parameters:	exact = true if derived classes should not be taken
 *				into account
 * 
 * Returns:		number of instances of type sint32
 *
 ******************************************************************************/

sint32 ClassNode::GetNumberOfInstances( bool exact ) const
{
	 

	if ( !exact ) 
	{  
		return instance_count;
	}
	else
	{
		sint sum = 0;

		for ( ClassNode* ClassNode = child; ClassNode; ClassNode = ClassNode->next )
		{
			sum += ClassNode->instance_count;
		}

		return instance_count - sum;
	}
}

/******************************************************************************
 *
 * Function:	ClassNode::GetNumberOfPlacementInstances ( bool exact )
 *
 * Description:	retrieves number of placement instances of given class
 *
 * Parameters:	exact = true if derived classes should not be taken
 *				into account
 * 
 * Returns:		number of instances of type sint32
 *
 ******************************************************************************/

sint32 ClassNode::GetNumberOfPlacementInstances( bool exact ) const
{
	 

	if ( !exact ) 
	{  
		return placement_count;
	}
	else
	{
		sint sum = 0;

		for ( ClassNode* ClassNode = child; ClassNode; ClassNode = ClassNode->next )
		{
			sum += ClassNode->placement_count;
		}

		return placement_count - sum;
	}
}


/******************************************************************************
 *
 * Function:	ClassNode::IsSame()
 *
 * Description: check's whether the class represented by 'this' is 
 *				the same as the given ClassNode ClassNode
 *
 * Parameters:	t = reference to const ClassNode
 *
 * Returns:		true if the class is the same
 *
 *****************************************************************************/

bool ClassNode::IsSame( const ClassNode& t ) const
{
	 

	return ( this == &t );
}


/******************************************************************************
 *
 * Function:	ClassNode::IsDerivedOrSame()
 *
 * Description: check's whether the class represented by 'this' is 
 *				derived from or the same as the given ClassNode ClassNode
 *
 * Parameters:	t = reference to const ClassNode
 *
 * Returns:		true if the class is derived or same
 *
 *****************************************************************************/

bool ClassNode::IsDerivedOrSame( const ClassNode& t ) const
{
	 

	if ( IsSame( t )) 
	{
		return true;
	}
	else if ( parent )
	{
		return parent->IsDerivedOrSame( t );
	}
	
	return false;
}


/******************************************************************************
 *
 * Function:	ClassNode::DumpHierarchy ( ostream &os, flags< Class::filter > flgs, sint tab )
 *
 * Description: dumps information about ClassNode to given output stream
 *
 * Parameters:	os			= reference to nOStream
 *				flgs		= information filter
 *									eINSTANCE_NAMES		displays instance names
 *									eANON_INSTANCES		displays anonymous instances
 *									eDERIVED_INSTANCES	displays derived instances
 *				tab			= number of spaces to insert before each line
 *
 *****************************************************************************/

void ClassNode::DumpHierarchy( ostream& os, flags< Class::filter > flgs, sint tab ) const
{
	 

	for ( sint i = 0; i < tab; i++ )
	{
		os << " ";
	}
	
	os << class_name << " Inst " << GetNumberOfInstances ( true ) << " (total " << GetNumberOfInstances ( false ) << ")";
	
	if ( flgs.Test ( Class::vPLACEMENT_INSTANCES ) && GetNumberOfPlacementInstances ( false ))
	{
		os << " Placement " << GetNumberOfPlacementInstances ( true ) << " (total " << GetNumberOfPlacementInstances ( false ) << ")";
	}
	
	os << "\n";

	if ( flgs.Test ( Class::vINSTANCE_NAMES ))
	{
		DumpInstanceNames( os, flgs, tab );
		
		if ( placement_count && flgs.Test ( Class::vPLACEMENT_INSTANCES ))
		{
			DumpPlacementInstanceNames( os, flgs, tab );
		}
	}

	if( child )
	{
		child->DumpHierarchy( os, flgs, tab+3 );
	}

	if ( next && tab )
	{
		next->DumpHierarchy( os, flgs, tab );
	}

}

/******************************************************************************
 *
 * Function:		ClassNode::DumpInstanceNames ( ostream& os, flags< Class::filter > flgs, sint tab )  
 *
 * Description:		dumps instances of class
 *
 * Parameters:		os			= reference nOStream
 *					flgs		= information filter
 *									vANON_INSTANCES		displays anonymous instances
 *									vDERIVED_INSTANCES	displays derived instances
 *					tab			= number of spaces to insert before each line
 *
 ******************************************************************************/

void ClassNode::DumpInstanceNames( ostream& os, flags< Class::filter > flgs, sint tab ) const
{
	

	for ( InstanceNode* iterator = instance_list; iterator; iterator = iterator->next )
	{
		const Class& instance = iterator->instance;

		if (( flgs.Test( Class::vANON_INSTANCES )	 || instance.is_named() ) && 
			( flgs.Test( Class::vDERIVED_INSTANCES ) || IsSame ( *instance.vClassNode())))

		{
			Dbg_MsgAssert( instance.vClassNode()->IsDerivedOrSame ( *this ), "Type mismatch" );

			for ( sint i = 0; i < tab; i++ )
			{
				os << " ";
			}

			os << "(" << (void*)&instance << ") ";
			os << instance.GetName() << "\n";
		}
	}
}

/******************************************************************************
 *
 * Function:		ClassNode::DumpInstanceNames ( ostream& os, flags< Class::filter > flgs, sint tab )  
 *
 * Description:		dumps instances of class
 *
 * Parameters:		os			= reference nOStream
 *					flgs		= information filter
 *									vANON_INSTANCES		displays anonymous instances
 *									vDERIVED_INSTANCES	displays derived instances
 *					tab			= number of spaces to insert before each line
 *
 ******************************************************************************/

void ClassNode::DumpPlacementInstanceNames( ostream& os, flags< Class::filter > flgs, sint tab ) const
{
	

	for ( InstanceNode* iterator = placement_list; iterator; iterator = iterator->next )
	{
		const Class& instance = iterator->instance;

		if (( flgs.Test( Class::vANON_INSTANCES )	 || instance.is_named() ) && 
			( flgs.Test( Class::vDERIVED_INSTANCES ) || IsSame ( *instance.vClassNode())))

		{
			Dbg_MsgAssert( instance.vClassNode()->IsDerivedOrSame ( *this ), "Type mismatch" );

			for ( sint i = 0; i < tab; i++ )
			{
				os << " ";
			}

			os << "(" << (void*)&instance << ") ";
			os << instance.GetName() << "\n";
		}
	}
}

/******************************************************************************/

} // namespace Spt

#endif // __NOPT_FULL_DEBUG__
