/*
	MultiList.cpp
	This is a list control that can contain lists within lists for
	selecting things from a hierarchy
*/

#include "MultiList.h"
#include "../PropEdit/ParseFuncs.h"

#define CUSTIDC_BUTTON 1
#define MENU_DELIM     "/"
#define MENUEVAL_DELIM "__"

MultiList::MultiList(HINSTANCE hInstance) :
	UIControl(hInstance, CLASS)
{
	hMenu = CreateMenu();
	hPopupMenu = CreateMenu();
	lastMenuID = 1;
	lastSelID = -1;
	oldSelID  = -1;

	bDispFullPath = false;
	bEvalFullPath = false;
	bGroupText    = false;
	bAck = true;

	AppendMenu(hMenu, MF_POPUP, (UINT)hPopupMenu, "Popup");
	AppendMenu(hPopupMenu,MF_STRING,0,"[No Change]");
	SetMenuDefaultItem(hPopupMenu,0,TRUE);
}

MultiList::MultiList(HINSTANCE hInstance,HWND hwnd) :
	UIControl(hInstance, CLASS)
{
	hMenu = CreateMenu();
	hPopupMenu = CreateMenu();
	lastMenuID = 1;
	lastSelID = -1;
	oldSelID = -1;

	bDispFullPath = false;
	bEvalFullPath = false;
	bGroupText    = false;
	bAck = true;

	AppendMenu(hMenu, MF_POPUP, (UINT)hPopupMenu, "Popup");
	AppendMenu(hPopupMenu,MF_STRING,0,"[No Change]");
	SetMenuDefaultItem(hPopupMenu,0,TRUE);

	Attach(hwnd);
}

MultiList::~MultiList()
{
	Destroy();
}

MultiList::MultiList(MultiList& right) :
	UIControl(right.hInstance, CLASS)
{
	hMenu = CreateMenu();
	hPopupMenu = CreateMenu();
	lastMenuID = 1;
	lastSelID = -1;
	oldSelID  = -1;

	bDispFullPath = false;
	bEvalFullPath = false;
	bAck = true;

	AppendMenu(hMenu, MF_POPUP, (UINT)hPopupMenu, "Popup");
	AppendMenu(hPopupMenu,MF_STRING,0,"[No Change]");
	SetMenuDefaultItem(hPopupMenu,0,TRUE);

	hwndButton = NULL;

	strValue    = right.strValue;
	strDelimOut = right.strDelimOut;
	curValue    = right.curValue;
	lastValue   = right.lastValue;

	Link<MenuDesc>* link = right.menus.GetHead();

	while(link)
	{
		AddItem(link->data.menuName);
		link = link->next;
	}
}

MultiList& MultiList::operator = (MultiList& right)
{
	Reset();

	hwndButton = NULL;

	strValue    = right.strValue;
	strDelimOut = right.strDelimOut;
	curValue    = right.curValue;
	lastValue   = right.lastValue;

	Link<MenuDesc>* link = right.menus.GetHead();

	while(link)
	{
		AddItem(link->data.menuName);
		link = link->next;
	}

	return *this;
}

void MultiList::Destroy()
{
	// Destroy all the submens
	Link<MenuDesc>* curmenu = menus.GetHead();

	while(curmenu)
	{
		Link<MenuDesc>* nextmenu = curmenu->next;
		
		DestroyMenu(curmenu->data.hMenu);
		menus.Remove(curmenu);

		curmenu = nextmenu;
	}

	DestroyMenu(hMenu);
	DestroyMenu(hPopupMenu);
}

void MultiList::ResetKeepText()
{
	Destroy();

	menus.Clear();
	names.Clear();

	hMenu = CreateMenu();
	hPopupMenu = CreateMenu();
	lastMenuID = 1;
	lastSelID = -1;
	oldSelID = -1;

	//SetWindowText(hwndButton,"");

	AppendMenu(hMenu, MF_POPUP, (UINT)hPopupMenu, "Popup");
	AppendMenu(hPopupMenu,MF_STRING,0,"[No Change]");
	SetMenuDefaultItem(hPopupMenu,0,TRUE);
}

void MultiList::Reset()
{
	ResetKeepText();
	SetWindowText(hwndButton,"");
}

void MultiList::BuildControlUI()
{
	RECT wndRect;
	GetClientRect(hwnd,&wndRect);

	hwndButton = CreateWindow("button",
		                      "",
							  WS_CHILD|WS_VISIBLE,
							  0,
							  0,
							  wndRect.right - wndRect.left,
							  wndRect.bottom - wndRect.top,
							  hwnd,
							  (HMENU)CUSTIDC_BUTTON,
							  hInstance,
							  NULL);
}

LRESULT MultiList::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	// Maintain compatibility with list boxes and combo boxes
	case WM_SETTEXT:
		{
			/*
			char* txt = (char*)lParam;
			
			if (strlen(txt)==0)
				SetWindowText(hwndButton,"");
			else
				SetWindowText(hwndButton,(char*)lParam);
			*/

			SetValue((char*)lParam);

			if (bAck)
			{
				HWND hwndParent = GetParent(hwnd);
				int  ID         = GetWindowLong(hwnd,GWL_ID);

				SendMessage(hwndParent,WM_COMMAND,MAKEWPARAM(ID,CBN_SELCHANGE),(LPARAM)hwnd);
			}
		}
		return TRUE;

	case WM_GETTEXT:
		{
			CStr buf = GetValue();

			char* str    = (char*)lParam;
			int   maxlen = (int)wParam;

			if (CStr(buf)==CStr(""))
			{
				str[0] = '\0';
				return 1;
			}

			for(int i=0;i<buf.Length() && i<maxlen-1;i++)
				str[i] = buf[i];

			str[i] = '\0';

			return strlen(str);
		}
		return TRUE;

	case WM_GETTEXTLAST:
		{
			/*
			char buf[256];
			//GetWindowText(hwndButton,buf,255);

			if (oldSelID <= 0)
			{
				*((char*)lParam) = 0;
				return 0;
			}

			if (bEvalFullPath)
			{
				if (strDelimOut.Length()>0)
				{
					CStr buf2 = ReplaceStr(names[oldSelID-1].path,"/",strDelimOut);

					if (buf2.Length()>0)
						strcpy(buf,buf2 + strDelimOut + names[oldSelID-1].name);
					else
						strcpy(buf, names[oldSelID-1].name);
				}
				else
					if (names[oldSelID-1].path.Length()>0)
						strcpy(buf,names[oldSelID-1].path + CStr("/") + names[oldSelID-1].name);
					else
						strcpy(buf,names[oldSelID-1].name);
			}
			else
				strcpy(buf,names[oldSelID-1].name);

			char* str    = (char*)lParam;
			int   maxlen = (int)wParam;

			if (CStr(buf)==CStr(""))
			{
				str[0] = '\0';
				return 1;
			}

			for(int i=0;i<strlen(buf) && i<maxlen-1;i++)
				str[i] = buf[i];

			str[i] = '\0';

			return strlen(str);
			*/

			char* str = (char*)lParam;
			int   maxlen = (int)wParam;

			for(int i=0;i<lastValue.Length() && i<maxlen-1;i++)
				str[i] = lastValue[i];
			
			str[i] = 0;

			return i-1;
		}

	case WM_RESETLAST:
		{
			ResetLast();
			return TRUE;
		}

	case LB_RESETCONTENT:
	case CB_RESETCONTENT:
		Reset();
		return TRUE;

	case LB_ADDSTRING:
	case CB_ADDSTRING:
			return AddItem((char*)lParam);
		
	case LB_SETITEMDATA:
	case CB_SETITEMDATA:
		{
			SetUserData(wParam,lParam);
			return TRUE;
		}

	case LB_GETITEMDATA:
	case CB_GETITEMDATA:
		{
			return GetUserData(wParam);
		}

	case LB_GETCURSEL:
	case CB_GETCURSEL:
		{
			if (lastSelID != -1)
				return lastSelID-1;
			else
				return -1;
		}

	case LB_SETCURSEL:
	case CB_SETCURSEL:
		{
			if (wParam < names.GetSize())
			{
				if (lastSelID == -1)
					oldSelID = wParam+1;
				else
					oldSelID = lastSelID;

				SetValue(names[wParam].path + CStr("/") + names[wParam].name,false);

				if (bAck)
				{
					HWND hwndParent = GetParent(hwnd);
					int  ID         = GetWindowLong(hwnd,GWL_ID);

					SendMessage(hwndParent,WM_COMMAND,MAKEWPARAM(ID,CBN_SELCHANGE),(LPARAM)hwnd);
				}

				lastSelID = wParam+1;
				return TRUE;
			}
			
			return FALSE;
		}

	case CB_GETLBTEXTLEN:
		{
			if ((int)wParam < names.GetSize() && (int)wParam > -1)
			{
				CStr name;

				if (bEvalFullPath)
				{
					if (names[wParam].path.Length()>0)
						name = names[wParam].path + CStr("/") + names[wParam].name;
					else
						name = names[wParam].name;

					if (strDelimOut.Length()>0)
						name = ReplaceStr(name,"/",strDelimOut);
				}
				else
					name = names[wParam].name;

				return name.Length();
			}
			else
				return CB_ERR;
		}

	case CB_GETLBTEXT:
		{
			CStr name;

			if (bEvalFullPath)
			{
				if (names[wParam].path.Length()>0)
					name = names[wParam].path + CStr("/") + names[wParam].name;
				else
					name = names[wParam].name;

				if (strDelimOut.Length()>0)
					name = ReplaceStr(name,"/",strDelimOut);
			}
			else
				name = names[wParam].name;
			
			strcpy((char*)lParam,(char*)name);

			return name.Length();
		}

	case CB_FINDSTRING:
		{
			NameDesc nd;
			Link<NameDesc>* link;
			SplitNamePath(nd, (char*)lParam);

			if (bEvalFullPath)
				link = names.Find(&nd);
			else
				link = names.Find(NameDesc::CompareName,&nd);

			return names.GetIndex(link);
		}

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case CUSTIDC_BUTTON:
			InvalidateRect(hwndButton,NULL,FALSE);
			OpenMenu();
			return TRUE;
		}
		break;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

int MultiList::AddItem(char* str)
{
	CStr  newStr = ReplaceStr(str,MENUEVAL_DELIM,MENU_DELIM);
	char* pos;
	char* dpos = newStr;
	CStr  strPath;
	HMENU hcurMenu = hPopupMenu;
	LinkList<MenuDesc>* curMenus = &menus;

	while((pos=strstr(dpos,"/"))!=0)
	{
		char* cpos = dpos;
		CStr curentry;

		while(cpos!=pos)
		{
			char chr[2];
			chr[0] = *cpos;
			chr[1] = 0;
		
			curentry += chr;
			cpos++;
		}

		// Process the current entry
		MenuDesc mdesc;
		mdesc.menuName = curentry;

		Link<MenuDesc>* refmenu = curMenus->Find(&mdesc);

		if (refmenu)
		{
			hcurMenu = refmenu->data.hMenu;
			curMenus = &refmenu->data.submenus;

			if (strPath.Length()>0)
				strPath += CStr("/") + refmenu->data.menuName;
			else
				strPath += refmenu->data.menuName;
		}
		else
		{
			// If more /'s exist in the string it must be a new menu
			Link<MenuDesc>* cm;

			mdesc.hMenu = CreateMenu();
			cm = curMenus->Add(&mdesc);
			AppendMenu(hcurMenu,MF_POPUP,(UINT)mdesc.hMenu,curentry);

			hcurMenu = mdesc.hMenu;
			curMenus = &cm->data.submenus;

			if (strPath.Length()>0)
				strPath += CStr("/") + mdesc.menuName;
			else
				strPath += mdesc.menuName;
		}

		dpos = pos+1;
	}

	// There are no more /'s in the string this is a menu item
	AppendMenu(hcurMenu,MF_STRING,lastMenuID++,dpos);

	NameDesc nd;
	nd.name = dpos;
	nd.path = strPath;
	names.Add(&nd);

	oldSelID = -1;

	//return lastMenuID-1;
	return lastMenuID-2;
}

int MultiList::AddItemUnique(char* str)
{
	NameDesc nd;
	SplitNamePath(nd,str);	

	Link<NameDesc>* link = names.Find(&nd);

	if (!link)
		return AddItem(str);

	return -1;
}

void MultiList::OpenMenu()
{
	POINT pt;
	int   ID;

	GetCursorPos(&pt);

	oldSelID = lastSelID;

	ID = (INT)TrackPopupMenu(hPopupMenu,TPM_LEFTBUTTON|TPM_RETURNCMD,
						     pt.x,pt.y,
							 0,hwnd,NULL);

	InvalidateRect(hwndButton,NULL,FALSE);

	if (ID < 1)
		return;

	lastSelID = ID;

	if (ID-1 < names.GetSize())
		if (names[ID-1].path.Length()>0)
			SetValue(names[ID-1].path + CStr("/") + names[ID-1].name);
		else
			SetValue(names[ID-1].name);

	if (bAck)
	{
		HWND hwndParent = GetParent(hwnd);
		int  ID         = GetWindowLong(hwnd,GWL_ID);

		SendMessage(hwndParent,WM_COMMAND,MAKEWPARAM(ID,CBN_SELCHANGE),(LPARAM)hwnd);
	}

	SetFocus(GetParent(hwnd));
	SetFocus(GetParent(hwnd));
}

CStr MultiList::GetValue()
{	
	if (lastSelID <= 0)
		return CStr("");

	CStr strOut;

	if (bEvalFullPath)
	{
		if (names[lastSelID-1].path.Length()>0)
			strOut = names[lastSelID-1].path + CStr("/") + names[lastSelID-1].name;
		else
			strOut = names[lastSelID-1].name;

		if (strDelimOut.Length()>0)
			strOut = ReplaceStr(strOut,"/",strDelimOut);
	}
	else
		strOut = names[lastSelID-1].name;

	return strOut;
}

CStr MultiList::GetAbsValue()
{
	if (lastSelID < 1)
		return CStr("");

	if (names[lastSelID-1].path.Length()>0)
		return names[lastSelID-1].path + CStr("/") + names[lastSelID-1].name;
	else
		return names[lastSelID-1].name;
}

void MultiList::SetValue(CStr val,bool bNotify)
{
	lastValue = curValue;

	//SetWindowText(hwndButton,val);
	if (strDelimOut.Length()>0)
		val = ReplaceStr(val,strDelimOut,"/");

	// Find the value in the list so we can match it to a selection ID
	NameDesc nd;
	SplitNamePath(nd, val);
	
	Link<NameDesc>* ndesc;

	ndesc = names.Find(&nd);

	if (!ndesc)
		ndesc = names.Find(NameDesc::CompareName,&nd);

	if (ndesc)
	{
		CStr outVal;

		//oldSelID = lastSelID;
		lastSelID = names.GetIndex(ndesc)+1;

		if (bDispFullPath)
		{
			if (ndesc->data.path.Length()>0)
				outVal = ndesc->data.path + CStr("/") + ndesc->data.name;
			else
				outVal = ndesc->data.name;
		}
		else
			outVal = ndesc->data.name;

		if (bGroupText)
			outVal = CStr("[") + outVal + "]";

		SetWindowText(hwndButton, (char*)outVal);
	}
	else
	{
		lastSelID = -1;
		SetWindowText(hwndButton,"");
	}

	//if (oldSelID != lastSelID)
	{
		HWND hwndParent = GetParent(hwnd);
		int  ID         = GetWindowLong(hwnd,GWL_ID);

		if (bAck)
		{
			SendMessage(hwndParent,WM_CHILDUPDATED,(WPARAM)ID,0);
		}
	}

	curValue = val;
	
	if (curValue.Length()>0 && curValue[0] == '/')
		curValue = curValue.Substr(1,curValue.Length());

	InvalidateRect(hwndButton,NULL,FALSE);
}

DWORD MultiList::GetUserData(int id)
{
	return names[id].data;
}

void MultiList::SetUserData(int id,DWORD val)
{
	names[id].data = val;
}

void MultiList::SortMenus()
{
	// TODO:
}

bool MultiList::SplitNamePath(NameDesc& nd, char* str)
{
	int len = strlen(str);
	char* buf = new char[len+1];
	strcpy(buf,str);

	char* pos = strrchr(buf,'/');

	if (!pos)
	{
		nd.name = str;
		nd.path = "";

		delete [] buf;
		return false;
	}

	// Terminate string at last backslash
	*pos = 0;

	nd.path = buf;
	nd.name = pos + 1;

	delete [] buf;

	return true;
}

void MultiList::ResetLast()
{
	oldSelID = lastSelID;
	lastValue = curValue;
}

void MultiList::SetFont(HFONT hFont)
{
	SendMessage(hwndButton,WM_SETFONT,(WPARAM)hFont,MAKELPARAM(TRUE,0));
	InvalidateRect(hwndButton,NULL,TRUE);
}

void MultiList::SetGroupText(bool bVal)
{
	bGroupText = bVal;
	CStr value = GetValue();
	
	if (bGroupText)
		value = CStr("[") + value + "]";

	SetWindowText(hwndButton,(char*)value);
}
