/*
	TriggerUI.cpp
	This file contains the UI and assignment system
	for Trigger Objects
	2-13-01
*/

#include "Trigger.h"
#include "Next.h"
#include "TriggerUI.h"
#include "../AppData.h"
#include "../PropEdit/PropEdit.h"
#include "../PropEdit/ConfigData.h"
#include "Resource.h"

#define DEFAULT_TRIGGER_CLASS "RailNode"
extern  PropEditor* pPropEdit;

HWND gLastUIhwnd=NULL;

TriggerUI::TriggerUI(Interface* ip)
{
	this->ip=ip;
	propList=NULL;
	contNode=NULL;
	hwndUI=NULL;
	IColor=NULL;
	IEdit =NULL;
	
	if (ip->GetSelNodeCount()>0)
		lastNode = ip->GetSelNode(0);
	else
		lastNode = NULL;

	// If a new trigger object is created it should contain
	// the defaults of the last created trigger automatically
	AppDataChunk* appdata;
	ReferenceTarget* scene=ip->GetScenePointer();
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERSETTINGS_ID);

	if (appdata)
	{
		TriggerUIData* pUIData=(TriggerUIData*)appdata->data;

		bLinkLast    = pUIData->bLinkLast;
		bAssignProps = pUIData->bAssignProps;
		bShowNames   = pUIData->bShowNames;
	}
	else
	{
		bLinkLast    = false;
		bAssignProps = false;
		bShowNames   = false;
	}

	// Acquire the last property buffer
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERPROP_ID);
	
	if (appdata)
		propBuf=(char*)appdata->data;
}

TriggerUI::~TriggerUI()
{
	if (IColor)
		ReleaseIColorSwatch(IColor);

	if (IEdit)
		ReleaseICustEdit(IEdit);
}

BOOL TriggerUI::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	TriggerUI* pthis=(TriggerUI*)GetWindowLong(hwnd,GWL_USERDATA);

	if (!pthis)
		return TRUE;

	switch(msg)
	{
	case CC_COLOR_CHANGE:
		pthis->UpdateExtData();
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_EDITNAME:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				pthis->UpdateExtData();
				return TRUE;
			}

			return FALSE;

		case IDC_CLASSLIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				pthis->UpdateClassSel();
				return TRUE;
			}

			return FALSE;

		case IDC_TYPELIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				pthis->UpdateTypeSel();
				return TRUE;
			}

		case IDC_CREATEDATSTART:
		case IDC_ABSENTINNETGAMES:
			pthis->UpdateChunk();
			return TRUE;

		case IDC_CLUSTER:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				pthis->UpdateChunk();
				return TRUE;
			}

			return FALSE;

		case IDC_LINKLAST:
			if (IsDlgButtonChecked(hwnd,IDC_LINKLAST)==BST_CHECKED)
				pthis->bLinkLast=true;
			else
				pthis->bLinkLast=false;

			pthis->UpdateChunk();
			return TRUE;

		case IDC_SHOWNAMES:
			if (IsDlgButtonChecked(hwnd,IDC_SHOWNAMES)==BST_CHECKED)
				pthis->bShowNames=true;
			else
				pthis->bShowNames=false;

			pthis->UpdateChunk();
			pthis->ip->ForceCompleteRedraw();
			return TRUE;

		case IDC_OUTPUTPROPS:
			if (IsDlgButtonChecked(hwnd,IDC_OUTPUTPROPS)==BST_CHECKED)
				pthis->bAssignProps=true;
			else
				pthis->bAssignProps=false;

			pthis->UpdateChunk();
			return TRUE;
		}
	};

	return FALSE;
}

void TriggerUI::UpdateClassSel(bool bUpdateChunk)
{
	char buf[256];

	int index=SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_GETCURSEL,0,0);
	SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);

	ConfigClass* cclass=pPropEdit->GetConfigClass(buf);

	if (cclass)
	{
		Link<ConfigType>* curNode=cclass->types.GetHead();

		SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_RESETCONTENT,0,0);

		while(curNode)
		{
			SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_ADDSTRING,(WPARAM)0,(LPARAM)(char*)curNode->data.name);
			curNode=curNode->next;
		}
	}

	// Select the first item in the type list
	SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_SETCURSEL,(WPARAM)0,0);
	UpdateTypeSel(bUpdateChunk);
}

void TriggerUI::UpdateTypeSel(bool bUpdateChunk)
{
	if (propList)
	{
		char bufClass[256], bufType[256];
		int  index;

		index=SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_GETCURSEL,0,0);
		SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_GETLBTEXT,(WPARAM)index,(LPARAM)bufClass);

		index=SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_GETCURSEL,0,0);
		SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_GETLBTEXT,(WPARAM)index,(LPARAM)bufType);

		if (bUpdateChunk)
		{
			propList->Clear();

			CStr strType;
			CStr strClassRec;

			strClassRec = pPropEdit->GetClassTypeRecord(bufClass,bufType);
			
			// Acquire the properties and add them to the property list
			ParseConfigProps(&props,&flags,strClassRec,&strCmds,&strCluster,&programs);

			if (flags & CCLASS_CREATEDATSTART)
				CheckDlgButton(hwndUI,IDC_CREATEDATSTART,BST_CHECKED);
			else
				CheckDlgButton(hwndUI,IDC_CREATEDATSTART,BST_UNCHECKED);

			if (flags & CCLASS_ABSENTINNETGAMES)
				CheckDlgButton(hwndUI,IDC_ABSENTINNETGAMES,BST_CHECKED);
			else
				CheckDlgButton(hwndUI,IDC_ABSENTINNETGAMES,BST_UNCHECKED);

			if (flags & CCLASS_TRICKOBJECT)
				SetDlgItemText(hwndUI,IDC_CLUSTER,(char*)strCluster);

			BuildPropList(propList,&props);
			propList->BuildUI();

			UpdateChunk();
		}
	}
}

void TriggerUI::ChangeCB(PropList* plist,void* pData)
{
	TriggerUI* pthis=(TriggerUI*)pData;
	pthis->UpdateChunk();
}

void TriggerUI::BeginEditParams(IObjParam *ip, ULONG flags, Animatable *prev)
{
	if (ip->GetCommandPanelTaskMode()==TASK_MODE_CREATE)
	{
		TriggerLayout::BeginEditParams(ip,flags,prev);

		propList=new PropList(hInstance,1);
		propList->HasApply(false);

		hwndUI=ip->AddRollupPage(hInstance,MAKEINTRESOURCE(IDD_TRIGGERUI),DlgProc,"Trigger Create Parameters");
		gLastUIhwnd=hwndUI;
		SetWindowLong(hwndUI,GWL_USERDATA,(LONG)this);

		IColor=GetIColorSwatch(GetDlgItem(hwndUI,IDC_COLOR),RGB(0,255,0),_T("Trigger Creation color"));
		IEdit=GetICustEdit(GetDlgItem(hwndUI,IDC_EDITNAME));

		propList->Attach(GetDlgItem(hwndUI,IDC_PROPS));
		propList->SetChangeCB(ChangeCB,this);

		// Set settings
		if (bLinkLast)
			CheckDlgButton(hwndUI,IDC_LINKLAST,BST_CHECKED);
		else
			CheckDlgButton(hwndUI,IDC_LINKLAST,BST_UNCHECKED);
		
		if (bShowNames)
			CheckDlgButton(hwndUI,IDC_SHOWNAMES,BST_CHECKED);
		else
			CheckDlgButton(hwndUI,IDC_SHOWNAMES,BST_UNCHECKED);

		if (bAssignProps)
			CheckDlgButton(hwndUI,IDC_OUTPUTPROPS,BST_CHECKED);
		else
			CheckDlgButton(hwndUI,IDC_OUTPUTPROPS,BST_UNCHECKED);

		// Read the TriggerObject types out of scripts.ini (already loaded from PE)
		//if (ParseScriptIni())
		{
			SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_RESETCONTENT,0,0);

			Link<ConfigClass>* curNode = pPropEdit->GetConfigDBHead();

			while(curNode)
			{
				SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_ADDSTRING,0,(LPARAM)(char*)curNode->data.name);
				curNode=curNode->next;
			}
		}

		// Set the class selection default
		if (propBuf.Length()>0)
		{
			// Set defaults
			//AddGlobalDefaults(propBuf);
			ParseConfigProps(&props,&flags,propBuf,&strCmds,&strCluster);
			pPropEdit->AddGlobalDefaults(propBuf,&props,&flags,&strCmds);

			if (flags & CCLASS_CREATEDATSTART)
				CheckDlgButton(hwndUI,IDC_CREATEDATSTART,BST_CHECKED);
			else
				CheckDlgButton(hwndUI,IDC_CREATEDATSTART,BST_UNCHECKED);

			if (flags & CCLASS_ABSENTINNETGAMES)
				CheckDlgButton(hwndUI,IDC_ABSENTINNETGAMES,BST_CHECKED);
			else
				CheckDlgButton(hwndUI,IDC_ABSENTINNETGAMES,BST_UNCHECKED);

			if (flags & CCLASS_TRICKOBJECT)
				SetDlgItemText(hwndUI,IDC_CLUSTER,(char*)strCluster);

			CStr strClass,strType;
			strClass = GetClassName(propBuf);
			strType  = GetTypeName(propBuf);

			propList->Clear();
		
			// Convert defaults
			pPropEdit->ConvertToDefaults(strClass,strType,&props);

			// Select the appropriate class
			int index=SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)strClass);
			SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_SETCURSEL,(WPARAM)index,0);

			UpdateClassSel(false);

			// Select the appropriate type
			index=SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)strType);
			SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_SETCURSEL,(WPARAM)index,0);

			propList->Clear();
			BuildPropList(propList,&props);
			propList->BuildUI();
		}
		else
		{
			int index=SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_FINDSTRING,(WPARAM)-1,(LPARAM)DEFAULT_TRIGGER_CLASS);
		
			if (index!=CB_ERR)
				SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_SETCURSEL,(WPARAM)index,0);

			UpdateClassSel();
		}

		GetExtData();
	}
}

void TriggerUI::EndEditParams(IObjParam *ip, ULONG flags, Animatable *next)
{
	TriggerLayout::EndEditParams(ip,flags,next);

	if (propList)
	{
		delete propList;
		propList=NULL;

		// Using config data from the PE now (can't clear)
		//configDB.Clear();

		if (IColor)
		{
			ReleaseIColorSwatch(IColor);
			IColor=NULL;
		}

		if (IEdit)
		{
			ReleaseICustEdit(IEdit);
			IEdit=NULL;
		}

		ip->DeleteRollupPage(hwndUI);
		hwndUI=NULL;
	}
}

void TriggerUI::TriggerCreated(RefMakerHandle rm)
{
	if (ip->GetSelNodeCount()>0)
		lastNode = ip->GetSelNode(0);
	else
		lastNode = NULL;
	
	if (rm->ClassID()==Class_ID(BASENODE_CLASS_ID,0))
	{
		INode* node=(INode*)rm;

		Object* obj=node->EvalWorldState(0).obj;
		
		if (obj && obj->ClassID()==vTRIGGER_CLASS_ID)
			contNode=node;
	}
}

void CenterPivot(INode* node)
{
	// Pivots should no longer be centered  4-2-01  aml
	/*
	TimeValue t = gInterface->GetTime();
	Point3 amt;
	Point3 ot = node->GetObjectTM(t).GetTrans();
	Point3 nt = node->GetNodeTM(t).GetTrans();

	amt = ot-nt;
	//node->Move( t, node->GetNodeTM(t), -19.0f/2.0f * Point3(0.0f, 0.5f, -0.5f), TRUE, FALSE, PIV_PIVOT_ONLY);
	node->Move( t, node->GetNodeTM(t), -19.0f/2.0f * Point3(0.0f, 0.75f, -0.5f), TRUE, FALSE, PIV_PIVOT_ONLY);
	*/
}

void TriggerUI::TriggerLink(BOOL endOfChain)
{
	// Retrieve chunk from the scene indicating the last trigger created
	// and then update it to indicate this newly created trigger
	if (!contNode)
		return;
	
	//INode* lastNode=NULL;

	CenterPivot(contNode);

	AppDataChunk* appdata;

	ReferenceTarget* scene=ip->GetScenePointer();
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LASTTRIGGER_ID);

	if (hwndUI)
	{
		// Name the node
		char buf[256];
		CStr strName;
		
		if (IEdit)
			IEdit->GetText(buf,255);
		
		strName=buf;
		ip->MakeNameUnique(strName);
		contNode->SetName(strName);

		// Set the node color
		if (IColor)
			contNode->SetWireColor(IColor->GetColor());
	}

	if (appdata)
	{
		ULONG nodeID=*((ULONG*)appdata->data);

		if (bLinkLast)
		{
			// Do the link
			ILinkMan* linkman=GetLinkMan();

			/*
			lastNode=ip->GetINodeByHandle(nodeID);

			ip->SelectNode(lastNode);
			*/

			ip->SelectNode(lastNode);
			ip->SelectNode(contNode,0);
			linkman->ChainLink(false);
			ip->SelectNode(contNode);
		}

		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LASTTRIGGER_ID);
	}

	// Apply the stored property buffer
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERPROP_ID);

	if (appdata && bAssignProps)
		contNode->SetUserPropBuffer(CStr((char*)appdata->data));

	// Add the newly created node as the last trigger now
	ULONG* pNodeID=(ULONG*)malloc(sizeof(ULONG));
	*pNodeID=contNode->GetHandle();

	if (endOfChain)
	{
		// Uncheck the link last trigger option
		if (gLastUIhwnd)
			CheckDlgButton(gLastUIhwnd,IDC_LINKLAST,BST_UNCHECKED);

		bLinkLast=false;
		UpdateTriggerSettings();
	}

	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LASTTRIGGER_ID,sizeof(ULONG),pNodeID);
}

void TriggerUI::UpdateTriggerSettings()
{
	ReferenceTarget* scene=ip->GetScenePointer();

	TriggerUIData* pUIData=(TriggerUIData*)malloc(sizeof(TriggerUIData));
		
	pUIData->bLinkLast    = bLinkLast;
	pUIData->bShowNames   = bShowNames;
	pUIData->bAssignProps = bAssignProps;
		
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERSETTINGS_ID);
	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERSETTINGS_ID,sizeof(TriggerUIData),pUIData);
}

void TriggerUI::UpdateChunk()
{
	ReferenceTarget* scene=ip->GetScenePointer();

	UpdateTriggerSettings();

	strClass=GetClass();
	strType=GetType();

	// Acquire flags
	flags=0;

	if (IsDlgButtonChecked(hwndUI,IDC_CREATEDATSTART)==BST_CHECKED)
		flags|=CCLASS_CREATEDATSTART;

	if (IsDlgButtonChecked(hwndUI,IDC_ABSENTINNETGAMES)==BST_CHECKED)
		flags|=CCLASS_ABSENTINNETGAMES;

	char strCluster[256];
	GetDlgItemText(hwndUI,IDC_CLUSTER,strCluster,255);

	if (strlen(strCluster)>0)
		flags|=CCLASS_TRICKOBJECT;
	
	// Acquire settings from property list
	int count=propList->NumProps();

	for(int i=0;i<count;i++)
	{
		CStr strValue;
		propList->GetValue(i,strValue);
		props[i].value=strValue;
	}

	propBuf=pPropEdit->BuildPropBuffer(NULL,
								    strClass,
									strType,
									flags,
									&props,
									strCmds,
									"",				// No scripts
									strCluster,
									&programs);

	char* PropBuf=(char*)malloc(propBuf.Length()+1);
	strcpy(PropBuf,(char*)propBuf);

	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERPROP_ID);
	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERPROP_ID,propBuf.Length()+1,PropBuf);
}

CStr TriggerUI::GetClass()
{
	char buf[256];

	int index=SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_GETCURSEL,0,0);
	SendDlgItemMessage(hwndUI,IDC_CLASSLIST,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);

	return CStr(buf);
}

CStr TriggerUI::GetType()
{
	char buf[256];

	int index=SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_GETCURSEL,0,0);
	SendDlgItemMessage(hwndUI,IDC_TYPELIST,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);

	return CStr(buf);
}

void TriggerUI::UpdateExtData()
{
	if (hwndUI)
	{
		ExtTriggerData* pextdata=(ExtTriggerData*)malloc(sizeof(ExtTriggerData));
		
		if (IEdit)
			IEdit->GetText(pextdata->name,255);
		else
			pextdata->name[0]='\0';
		
		if (IColor)
			pextdata->color=IColor->GetColor();
		else
			pextdata->color=RGB(0,255,0);

		ReferenceTarget* scene=ip->GetScenePointer();

		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_EXTTRIGGERSETTINGS_ID);
		scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_EXTTRIGGERSETTINGS_ID,sizeof(ExtTriggerData),pextdata);
	}
}

void TriggerUI::GetExtData()
{
	ReferenceTarget* scene=ip->GetScenePointer();

	AppDataChunk* appdata;
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_EXTTRIGGERSETTINGS_ID);

	if (appdata)
	{
		ExtTriggerData* pExtData=(ExtTriggerData*)appdata->data;
		
		// Set name
		if (strlen(pExtData->name)==0)
			strcpy(pExtData->name,"TRG_");

		if (IEdit)
			IEdit->SetText(pExtData->name);

		// Set color
		if (IColor)
			IColor->SetColor(pExtData->color);
	}
	else
	{
		IEdit->SetText("TRG_");
	}
}

void TriggerUI::DestroyRetainedData()
{
	ReferenceTarget* scene = gInterface->GetScenePointer();
	
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERPROP_ID);
	//scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LASTTRIGGER_ID);
	//scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERSETTINGS_ID);
}
