#include <resource.h>
#include <max.h>
#include "LinkMan.h"
#include "LinkUI.h"
#include "LinkOptions.h"
#include <hold.h>

class LinkMan: public ILinkMan
{	
	bool    m_bLocked;

public:
	LinkMan( void );

	// From ILinkMan
	bool    IsLocked( void );		// Returns true if selection related ops should be denied
	void    SetLock( bool bVal );
	void	ManyToOneLink( void );
	void	OneToManyLink( void );
	void	ChainLink( bool closed, bool enumerate = true );
	void	ReverseChainLink( bool closed );
	void	Unlink( void );
	void	SetLinkColor( Color color );
	Color	GetLinkColor( void );
	void	SetDisplayMode( int mode );
	int		GetDisplayMode( void );
	void	SetSelectChildren( bool select );
	bool	GetSelectChildren( void );
	void	SetSelectParents( bool select );
	bool	GetSelectParents( void );
	void	RecalculateReferences( void );

	void	IncrementSystemSelectOpID( void );
	int		GetSystemSelectID( void );

	bool	IsMerging( void );
	void	OnPreMerge( void );
	void	OnPostMerge( void );
	void	OnPostOpen( void );
	void	OnPreSave( void );
	
	void	Link( INode *from, INode *to, class LinkRestoreObj *restore_obj = NULL );	
	bool	AlreadyLinked( INode *from, INode *to, bool &circular_ref );
		
	void	EnumerateSelectedNodes( bool reverse = false );
	bool	CancelledLinkOp( void );
	
	Tab		<INode *> m_SelectedNodes;
	Color	m_Color;
	bool	m_Merging;
	int		m_DisplayMode;
	bool	m_SelectChildren;
	bool	m_SelectParents;
	int		m_SystemSelectOpID;
};

class LinkRestoreObj : public RestoreObj
{
public:
	LinkRestoreObj( LinkMan *link_man ) : m_LinkMan( link_man ) {}
	void Redo();
	void Restore( int isUndo );
	TSTR Description() { return TSTR(_T("NExtLinkRestore")); }

	LinkMan *m_LinkMan;
};

void LinkRestoreObj::Redo( void )
{	
	m_LinkMan->RecalculateReferences();
}

void LinkRestoreObj::Restore( int isUndo )
{
	m_LinkMan->RecalculateReferences();
}

LinkMan::LinkMan( void )
{
	m_Color = Color( 0.5f, 0.5f, 0.5f );
	m_Merging = false;
	m_DisplayMode = DISPLAY_MODE_ALL;
	m_SelectChildren = false;
	m_SelectParents = false;
	m_SystemSelectOpID = 0;
	m_bLocked = false;
}

void LinkMan::SetLock(bool bVal)
{
	m_bLocked = bVal;
}

Color	LinkMan::GetLinkColor( void )
{
	return m_Color;
}

void	LinkMan::SetLinkColor( Color color )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_selected;
	int num_children = root->NumberOfChildren();
	int i;

	num_selected = gInterface->GetSelNodeCount();

	m_Color = m_Color;

	if( num_selected == 1 )
	{
		for( i = 0; i < num_children; i++ )
		{
			node = root->GetChildNode(i);
			obj = node->EvalWorldState(0).obj;

			if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
			{
				LinkObject* link = (LinkObject*) obj;
				if( link->from_node->Selected())
				{
					link->m_Color = color;
				}
			}
		}	
	}
	else if( num_selected > 1 )
	{
		for( i = 0; i < num_children; i++ )
		{
			node = root->GetChildNode(i);
			obj = node->EvalWorldState(0).obj;

			if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
			{
				LinkObject* link = (LinkObject*) obj;
				if( link->from_node->Selected() && link->to_node->Selected())
				{
					link->m_Color = color;
				}
			}
		}
	}

	gInterface->ForceCompleteRedraw();
}

void	LinkMan::EnumerateSelectedNodes( bool reverse )
{
	int i, num_selected;
	INode *sel_node;
	Object *obj;

	m_SelectedNodes.Resize( 0 );
	num_selected = gInterface->GetSelNodeCount();
	for( i = 0; i < num_selected; i++ )
	{
		sel_node = gInterface->GetSelNode( i );
		obj = sel_node->EvalWorldState(0).obj;
		if( obj && obj->ClassID() != vLINK_OBJ_CLASS_ID )
		{
			if( reverse )
			{
				m_SelectedNodes.Insert( 0, 1, &sel_node );
			}
			else
			{
				m_SelectedNodes.Append( 1, &sel_node );
			}
		}
	}	
}

bool	LinkMan::IsLocked( void )
{
	return m_bLocked;
}

void	LinkMan::ManyToOneLink( void )
{	
	int num_selected;
	INode *to, *from;
	int i;
	bool circular_ref, cancelled;
	LinkRestoreObj *restore_obj;

	EnumerateSelectedNodes();

	cancelled = false;
	num_selected = m_SelectedNodes.Count();
	if( num_selected == 1 )
	{
#ifdef LINK_TO_SELF_ENABLED
		// The last one will be considered the target node for the link
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );;
		theHold.Put( restore_obj );
		
		to = m_SelectedNodes[0];
		from = to;
		if( AlreadyLinked( from, to, circular_ref ) == false )
		{
			if( circular_ref )
			{
				if( CancelledLinkOp())
				{
					cancelled = true;		
					break;
				}
			}
			Link( from, to, restore_obj );
		} 		
		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "Many To One Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());
#else
		return;
#endif
	}
	else
	{
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );

		to = m_SelectedNodes[num_selected - 1];
		for( i = 0; i < (num_selected - 1); i++ )
		{
			from = m_SelectedNodes[i];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;
						break;						
					}
				}
				Link( from, to, restore_obj );
			} 
		}

		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "Many To One Link" ));		
		}
		gInterface->RedrawViews( gInterface->GetTime());
	}

	m_SelectedNodes.Resize( 0 );
}

void	LinkMan::OneToManyLink( void )
{
	int num_selected;
	bool circular_ref, cancelled;
	INode *to, *from;
	int i;
	LinkRestoreObj *restore_obj;

	EnumerateSelectedNodes();

	cancelled = false;
	num_selected = m_SelectedNodes.Count();
	if( num_selected == 1 )
	{
#ifdef LINK_TO_SELF_ENABLED
		// The last one will be considered the target node for the link
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );
		from = m_SelectedNodes[0];
		to = from;
		if( AlreadyLinked( from, to, circular_ref ) == false )
		{
			if( circular_ref )
			{
				if( CancelledLinkOp())
				{
					cancelled = true;		
					break;
				}
			}
			Link( from, to, restore_obj );
		}
		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{		
			theHold.Accept(_T( "One To Many Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());	
#else
		return;
#endif
	}
	else
	{
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );

		from = m_SelectedNodes[0];
		for( i = 1; i < num_selected; i++ )
		{
			to = m_SelectedNodes[i];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;		
						break;
					}
				}
				Link( from, to, restore_obj );
			} 
		}

		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "One To Many Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());	
		
	}	

	m_SelectedNodes.Resize( 0 );
}

void	LinkMan::ReverseChainLink( bool closed )
{
	EnumerateSelectedNodes( true );
	ChainLink( closed, false );	
}

void	LinkMan::ChainLink( bool closed, bool enumerate )
{
	int num_selected;
	bool circular_ref, cancelled;
	INode *to, *from;
	int i;
	LinkRestoreObj *restore_obj;

	if( enumerate )
	{
		EnumerateSelectedNodes();
	}

	cancelled = false;
	num_selected = m_SelectedNodes.Count();
	if( num_selected == 1 )
	{
#ifdef LINK_TO_SELF_ENABLED
		if( closed )		
		{
			if( theHold.Holding() == false ) 
			{
				theHold.Begin();
			}

			restore_obj = new LinkRestoreObj( this );
			theHold.Put( restore_obj );

			// The last one will be considered the target node for the link
			from = m_SelectedNodes[0];

			to = from;
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;		
						break;
					}
				}
				Link( from, to, restore_obj );
			} 

			if( cancelled )
			{
				theHold.Cancel();
			}
			else
			{
				theHold.Accept(_T( "Chain Link" ));
			}
			gInterface->RedrawViews( gInterface->GetTime());
			
		}
#else		
		return;		
#endif
	}
	else
	{
		if( theHold.Holding() == false ) 
		{
			theHold.Begin();
		}

		restore_obj = new LinkRestoreObj( this );
		theHold.Put( restore_obj );

		for( i = 0; i < ( num_selected - 1 ); i++ )
		{
			from = m_SelectedNodes[i];
			to = m_SelectedNodes[i + 1];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;		
						break;
					}
				}
				Link( from, to, restore_obj );
			} 
		}
		if( !cancelled && closed )
		{
			from = m_SelectedNodes[num_selected - 1];
			to = m_SelectedNodes[0];
			if( AlreadyLinked( from, to, circular_ref ) == false )
			{
				if( circular_ref )
				{
					if( CancelledLinkOp())
					{
						cancelled = true;								
					}
				}
				if( !cancelled )
				{
					Link( from, to, restore_obj );
				}
			} 
		}

		if( cancelled )
		{
			theHold.Cancel();
		}
		else
		{
			theHold.Accept(_T( "Chain Link" ));
		}
		gInterface->RedrawViews( gInterface->GetTime());
		
	}

	m_SelectedNodes.Resize( 0 );
}

void	LinkMan::RecalculateReferences( void )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	int num_selected = gInterface->GetSelNodeCount();
	int i;
	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link_obj = (LinkObject*) obj;
			link_obj->ref_count = 0;			
		}
	}

	for( i = 0; i < num_selected; i++ )
	{
		IncrementSystemSelectOpID();
		node = gInterface->GetSelNode( i );		
		node->NotifyDependents( FOREVER, PART_ALL, REFMSG_SELECT_SURROUNDING_NODES );		
	}

}

void	LinkMan::Unlink( void )
{
	m_bLocked=true;

	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	LinkRestoreObj *restore_obj;
	int i;
	
	Tab<INode *> del_list;

	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;
			if( link->from_node && link->to_node )
			{
				if( link->from_node->Selected() && link->to_node->Selected())
				{
					del_list.Append( 1, &node );					
				}
			}
		}
	}	

	theHold.Begin();
	restore_obj = new LinkRestoreObj( this );;
	theHold.Put( restore_obj );
	for( i = 0; i < del_list.Count(); i++ )
	{
		gInterface->DeleteNode( del_list[i], FALSE );
		
	}
	theHold.Accept( _T("Unlink"));
	
	RecalculateReferences();

	gInterface->RedrawViews( gInterface->GetTime());

	m_bLocked=false;
}

void	LinkMan::OnPreSave( void )
{
	int i, num_children;
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;	
	Tab<INode *> del_list;
	LinkOptions options;
	
	num_children = root->NumberOfChildren();	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;			
			if(	( link->to_node == NULL ) ||
				( link->from_node == NULL ))
			{
				del_list.Append( 1, &node );
			}
		}		
	}

	for( i = 0; i < del_list.Count(); i++ )
	{
		gInterface->DeleteNode( del_list[i], FALSE );			
	}

	options.m_DisplayMode = m_DisplayMode;
	options.m_SelectChildren = m_SelectChildren;
	options.m_SelectParents = m_SelectParents;
	SetLinkOptions( &options );
}

void	LinkMan::OnPostOpen( void )
{
	int i, num_children;
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;	
	Tab<INode *> del_list;
	LinkOptions *options;
	
	num_children = root->NumberOfChildren();	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;			
			link->link_node = node;
			
			if(	( link->to_node == NULL ) ||
				( link->from_node == NULL ))
			{
				del_list.Append( 1, &node );
			}
		}		
	}

	for( i = 0; i < del_list.Count(); i++ )
	{
		gInterface->DeleteNode( del_list[i], FALSE );			
	}

	if( options = GetLinkOptions())
	{
		m_SelectChildren = options->m_SelectChildren;
		m_SelectParents = options->m_SelectParents;
		m_DisplayMode = options->m_DisplayMode;

		if( m_SelectChildren || m_SelectParents )
		{
			if( m_DisplayMode == ILinkMan::DISPLAY_MODE_SYSTEM )
			{
				SetDisplayMode( ILinkMan::DISPLAY_MODE_ALL );
			}
		}
		else
		{
			SetDisplayMode( m_DisplayMode );
		}
		
		LinkUI::SetSelectChildren( m_SelectChildren );
		LinkUI::SetSelectParents( m_SelectParents );
	}
}

bool	LinkMan::IsMerging( void )
{
	return m_Merging;
}

void	LinkMan::OnPreMerge( void )
{
	m_Merging = true;
}

void	LinkMan::OnPostMerge( void )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	int i;
	
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;
			if( link->to_node == NULL )
			{
				char *node_name;
				
				node_name = GetLinkToObjectName( link->link_node );				
				if( node_name )
				{
					link->to_node = gInterface->GetINodeByName( node_name );
				}
			}

			if( link->from_node == NULL )
			{
				char *node_name;

				node_name = GetLinkFromObjectName( link->link_node );
				if( node_name )
				{
					link->from_node = gInterface->GetINodeByName( node_name );
				}
			}
		}
	}
	
	m_Merging = false;
}

void	LinkMan::IncrementSystemSelectOpID( void )
{
	m_SystemSelectOpID++;
}

int		LinkMan::GetSystemSelectID( void )
{
	return m_SystemSelectOpID;
}

void	LinkMan::SetDisplayMode( int mode )
{
	m_DisplayMode = mode;	
	if( m_DisplayMode == ILinkMan::DISPLAY_MODE_SYSTEM )
	{
		RecalculateReferences();
	}
	gInterface->ForceCompleteRedraw();
}

int		LinkMan::GetDisplayMode( void )
{
	return m_DisplayMode;
}

void	LinkMan::SetSelectChildren( bool select )
{
	m_SelectChildren = select;
	if( m_SelectChildren == false )
	{
		RecalculateReferences();
	}
}

bool	LinkMan::GetSelectChildren( void )
{
	return m_SelectChildren;
}

void	LinkMan::SetSelectParents( bool select )
{
	m_SelectParents = select;
	if( m_SelectParents == false )
	{
		RecalculateReferences();
	}
}

bool	LinkMan::GetSelectParents( void )
{
	return m_SelectParents;
}

static LinkMan link_man;
ILinkMan* GetLinkMan( void )
{
	return &link_man;
}

bool	LinkMan::AlreadyLinked( INode *from, INode *to, bool &circular_ref )
{
	INode* root = gInterface->GetRootNode();
	INode* node;
	Object* obj;
	int num_children = root->NumberOfChildren();
	int i;

	circular_ref = false;
	for( i = 0; i < num_children; i++ )
	{
		node = root->GetChildNode(i);
		obj = node->EvalWorldState(0).obj;

		if( obj->ClassID() == vLINK_OBJ_CLASS_ID )
		{
			LinkObject* link = (LinkObject*) obj;
			if(( link->to_node == from ) && ( link->from_node == to ))
			{
				circular_ref = true;
			}
			if(( link->from_node == from ) && ( link->to_node == to ))
			{
				return true;
			}
		}
	}

	return false;
}

void	LinkMan::Link( INode *from, INode *to, LinkRestoreObj *restore_obj )
{
	Matrix3 tm;
	Matrix3 link_tm( 1 );
	Point3 from_pos, to_pos, link_pos;
	LinkObject* link;
	Object *to_obj, *from_obj;
	
	to_obj = to->EvalWorldState(0).obj;
	from_obj = from->EvalWorldState(0).obj;

	if( ( to_obj->ClassID() == vLINK_OBJ_CLASS_ID ) ||
		( from_obj->ClassID() == vLINK_OBJ_CLASS_ID ))
	{
		return;
	}

	link = new LinkObject;
	
	link->link_node = gInterface->CreateObjectNode( link );

	// center the new link node between its target references
	tm = from->GetNodeTM( 0 );
	from_pos = tm.GetTrans();

	tm = to->GetNodeTM( 0 );
	to_pos = tm.GetTrans();

	link_pos = ( from_pos + to_pos ) / 2;
	link_tm.SetTrans( link_pos );
	link->link_node->SetNodeTM( 0, link_tm );

	link->from_node = from;
	link->to_node = to;

	gInterface->SelectNode( link->link_node, 0 );

	link->MakeRefByID( FOREVER, LinkObject::REF_ID_FROM, from );
	link->MakeRefByID( FOREVER, LinkObject::REF_ID_TO, to );
	
	RecalculateReferences();
	
	SetLinkFromObjectName( link->link_node, from->GetName());
	SetLinkToObjectName( link->link_node, to->GetName());
}

bool	LinkMan::CancelledLinkOp( void )
{
	return ( MessageBox( NULL, "This operation will create a cyclical reference. Proceed?", "Warning", MB_OKCANCEL ) == IDCANCEL );
}