/*
	GenExport.cpp
	Generalized Animatable Exporter
	1-28-01
*/

#include "GenExport.h"
#include "..\PropEdit\ParseFuncs.h"
#include "iparamb2.h"

int KeySize[] =
{
	sizeof(ITCBKey),
	sizeof(ITCBFloatKey),
	sizeof(ITCBPoint3Key),
	sizeof(ITCBRotKey),
	sizeof(ITCBScaleKey),
	sizeof(IBezFloatKey),
	sizeof(IBezPoint3Key),
	sizeof(IBezQuatKey),
	sizeof(IBezScaleKey),
	sizeof(ILinFloatKey),
	sizeof(ILinPoint3Key),
	sizeof(ILinRotKey),
	sizeof(ILinScaleKey),
};

#define PROP_TOKEN_C    '|'
#define PROP_TOKEN_S    "|"
#define SUBPROP_TOKEN_C '/'
#define SUBPROP_TOKEN_S "/"

ExportData::~ExportData()
{
	int size=keys.Count();

	for(int i=0;i<size;i++)
	{
		if (keys[i].key)
			delete keys[i].key;
	}
}

void SearchKey::AddExportAnim(CStr str)
{
	animStr+=CStr(PROP_TOKEN_S)+str;
}

int SearchKey::NumProps()
{
	// Returns the number of properties that are to be exported
	return CountChar(animStr,PROP_TOKEN_C)+1;
}

int SearchKey::NumSubProps(int Index)
{
	// Returns the number of sub properties in the given property
	CStr strProp=GetProp(Index);

	return CountChar(strProp,SUBPROP_TOKEN_C)+1;
}

CStr SearchKey::GetProp(int Index)
{
	int index=0;
	int animLen=animStr.Length();

	for(int i=0;i<animLen;i++)
	{
		if (Index==index)
		{
			CStr strProp=animStr.Substr(i,animLen);
			int pos=Instr(strProp,PROP_TOKEN_S);

			if (pos==-1)
				pos=animLen;

			strProp=strProp.Substr(0,pos);
			return strProp;
		}

		index++;
		i=Instr(animStr.Substr(i,animLen),PROP_TOKEN_S);
	}

	// index out of range or invalid animStr
	return CStr("");
}

CStr SearchKey::GetSubProp(int Index,int SubIndex)
{
	CStr strProp=GetProp(Index);
	
	int subindex=0;
	int animLen=strProp.Length();

	for(int i=0;i<animLen;i++)
	{
		if (SubIndex==subindex)
		{
			CStr strSubProp=strProp.Substr(i,animLen);
			int pos=Instr(strSubProp,SUBPROP_TOKEN_S);
			
			if (pos==-1)
				pos=animLen;
			
			strSubProp=strSubProp.Substr(0,pos);
			return strSubProp;
		}

		subindex++;
		i=Instr(strProp.Substr(i,animLen),SUBPROP_TOKEN_S);
	}

	// index out of range or invalid animStr
	return CStr("");
}


GenExporter::GenExporter(Interface* ip)
{
	this->ip=ip;
}

GenExporter::~GenExporter()
{

}

void GenExporter::ClearSearchKeys()
{
	searchKeys.ZeroCount();
}

void GenExporter::ClearExportData()
{
	exportDB.ZeroCount();
}

void GenExporter::AddSearchKey(SearchKey& skey)
{
	searchKeys.Append(1,&skey);
}

// Traverses TrackView animatable hierarchy to find the animatable we want to export
Animatable* GenExporter::FindAnimatable(Animatable* anim,CStr name)
{
	if (anim==NULL)
		return NULL;

	int numKeys=anim->NumKeys();

	char strMsg[256];
	sprintf(strMsg,"NumKeys: %i",numKeys);

	MessageBox(NULL,strMsg,"Number of Keys",MB_OK);

	GetPBlockVal(anim,name);

	if (GetControlInterface(anim))
		MessageBox(NULL,"Got a controller","Blah",MB_OK);

	int numSubs=anim->NumSubs();

	for(int i=0;i<numSubs;i++)
	{
		Animatable* subanim;

		MessageBox(NULL,anim->SubAnimName(i),"Subanimatable enum debug",MB_OK);

		if (anim->SubAnimName(i)==name)
			return anim->SubAnim(i);

		subanim=FindAnimatable(anim->SubAnim(i),name);

		if (subanim)
			return subanim;
	}

	return NULL;
}

Control* GenExporter::FindAnimCont(Animatable* anim,CStr name)
{
	Control* control;
	CStr     strName;

	if (anim==NULL)
		return NULL;

	anim->GetClassName(strName);

	if (strName==name)
	{
		// This is the correct animatable try to get it's controller
		control=GetControlInterface(anim);
		return control;
	}

	// Scan any parameter blocks for our controller
	control=GetPBlockCont(anim,name);

	if (control)
		return control;

	int numSubs=anim->NumSubs();

	for(int i=0;i<numSubs;i++)
	{
		if (anim->SubAnimName(i)==name)
		{
			// Get the Sub anim's controller
			control=GetControlInterface(anim->SubAnim(i));
			return control;
		}

		control=FindAnimCont(anim->SubAnim(i),name);

		if (control)
			return control;
	}

	return NULL;
}

void* GenExporter::FindAnimProp(Animatable* anim,CStr name)
{
	Control* control;
	CStr     strName;

	if (anim==NULL)
		return NULL;

	anim->GetClassName(strName);

	if (strName==name)
	{
		// Get the animatable's property
		return anim->GetProperty(0);
	}

	// Scan any parameter blocks for our controller
	control=GetPBlockCont(anim,name);

	if (control)
		return control;

	int numSubs=anim->NumSubs();

	for(int i=0;i<numSubs;i++)
	{
		if (anim->SubAnimName(i)==name)
		{
			// Get the Sub anim's property value
			return anim->GetProperty(i);
		}

		control=FindAnimCont(anim->SubAnim(i),name);

		if (control)
			return control;
	}

	return NULL;
}

bool GenExporter::GetPBlockVal(Animatable* anim,CStr name)
{
	// Search the param blocks of this animatable
	int NumBlocks=anim->NumParamBlocks();

	for(int k=0;k<NumBlocks;k++)
	{
		IParamBlock2* pblock=anim->GetParamBlock(k);

		MessageBox(NULL,(char*)pblock->GetLocalName(),"Parameter Block",MB_OK);

		int NumParams=pblock->NumParams();
		
		for(int m=0;m<NumParams;m++)
		{
			CStr valName=pblock->GetLocalName(m);
			MessageBox(NULL,(char*)valName,"Parameter Block Value",MB_OK);

			if (valName==name)
			{
				MessageBox(NULL,"Search Found!","Found param block value through anim tree",MB_OK);
				return true;
			}
		}
	}

	return false;
}

Control* GenExporter::GetPBlockCont(Animatable* anim,CStr name)
{
	// Search the param blocks of this animatable
	int NumBlocks=anim->NumParamBlocks();

	for(int k=0;k<NumBlocks;k++)
	{
		IParamBlock2* pblock=anim->GetParamBlock(k);

		int NumParams=pblock->NumParams();
		
		for(int m=0;m<NumParams;m++)
		{
			CStr valName=pblock->GetLocalName(m);

			if (valName==name)
				return pblock->GetController(m);
		}
	}

	return NULL;
}

void* GenExporter::MakeKey(ContType type)
{
	// Allocate new key based on controller type
	// These keys will be freed when the ExportData structure is destroyed
	switch(type)
	{
	case CONTTYPE_FLOAT:
		return new float;
		break;

	case CONTTYPE_POSITION:
	case CONTTYPE_POINT3:
		return new Point3;

	case CONTTYPE_ROTATION:
		return new Quat;

	case CONTTYPE_SCALE:
		return new ScaleValue;

	case CONTTYPE_TRANSFORM:
		return new Matrix3;
	}

	return NULL;
}

bool GenExporter::ExportKeys(ExportInfo& expInfo,Object* obj,SearchKey& searchKey)
{
	// TODO: Complete

	int numProps=searchKey.NumProps();

	for(int i=0;i<numProps;i++)
	{
		int numSubProps=searchKey.NumSubProps(i);

		for(int j=0;j<numSubProps;j++)
		{
			Control* control = FindAnimCont(obj,searchKey.GetSubProp(i,j));

			ExportData  newExpData;
			expInfo.exportList.Append(1,&newExpData);
			
			ExportData& expData=expInfo.exportList[expInfo.exportList.Count()-1];

			if (searchKey.fps>0)
			{
				int      fpsMAX       = GetFrameRate();
				Interval animRange    = ip->GetAnimRange();
				int      numFrames    = animRange.End()/GetTicksPerFrame()-animRange.Start()/GetTicksPerFrame();
				int      numReqFrames = numFrames/fpsMAX*searchKey.fps;
				void*    KeyVal;

				// sample the output
				if (control)
				{
					expData.numKeys=numFrames;
					expData.type=searchKey.conttype;
					expData.keys.Resize(numFrames);
					expData.keys.SetCount(numFrames);

					for(int m=0;m<numFrames;m++)
					{
						KeyVal=MakeKey(searchKey.conttype);
						control->GetValue(GetTicksPerFrame()*m,KeyVal,FOREVER,CTRL_ABSOLUTE);

						expData.keys[m].key   = KeyVal;
						expData.keys[m].time  = (float)m/(float)searchKey.fps;

						float flt;
						flt=*((float*)KeyVal);
					} 
				}
			}
		}
	}

	return true;
}

void GenExporter::ExportNode(INode* node)
{
	int i;
	Object* obj=node->EvalWorldState(0).obj;

	// Check if this object is one of the nodes we're supposed to export data for
	int SearchKeyCount=searchKeys.Count();

	for(i=0;i<SearchKeyCount;i++)
	{
		if (obj->ClassID()==searchKeys[i].cid)
		{
			ExportInfo expInfo;
			expInfo.node=node;
			exportDB.Append(1,&expInfo);
			ExportKeys(exportDB[exportDB.Count()-1],obj,searchKeys[i]);
		}
	}

	int numChildren=node->NumberOfChildren();

	for(i=0;i<numChildren;i++)
	{
		INode* child=node->GetChildNode(i);
		ExportNode(child);
	}
}

Tab<ExportInfo>* GenExporter::Export()
{
	INode* root=ip->GetRootNode();

	int numChildren=root->NumberOfChildren();

	for(int i=0;i<numChildren;i++)
	{
		INode* child=root->GetChildNode(i);
		ExportNode(child);
	}

	return &exportDB;
}

Object* GetObj(SearchKey& skey,INode* node)
{
	Object* obj=node->EvalWorldState(0).obj;

	if (obj->ClassID()==skey.cid)
	{

	}

	return NULL;
}

template<class T>
bool GetKeyFrames(Interface* ip,GenKey<T>* keys,SearchKey& skey)
{
	INode* root=ip->GetRootNode();

	int numChildren=root->NumberOfChildren();

	for(int i=0;i<numChildren;i++)
	{
		INode* child=root->GetChild(i);
		Object* obj=child->EvalWorldState(0).obj;

		if (obj)
		{
			// Check if the search key matches 

		}
	}
}
