/*
	AnimExporter.cpp
	Animation Exporter
	1-11-01
*/

#include <direct.h>			// for _mkdir
#include <io.h>				// for _access, _chmod
#include <process.h>
#include <sys/stat.h>		// for access flags
#include "AnimExporter.h"
#include "../../../../include/AnimExportFmt.h"
#include "../Misc/GenCrc.h"
#include "../Misc/mth.h"
#include "AnimTol.h"

#define  WRITE_ACCESS     0x02
#define  REMOVE_DBLKEYS
//#define  USE_EULERFIX

static double RtQuatDiff( Mth::Quat& p_in1, Mth::Quat& p_in2 )
{
	double f_cos_theta =	( p_in1[X] * p_in2[X] ) + ( p_in1[Y] * p_in2[Y] ) +
							( p_in1[Z] * p_in2[Z] ) + ( p_in1[W] * p_in2[W] );

	bool   b_obtuse_theta = ( f_cos_theta < 0.0 );

	if( b_obtuse_theta )
	{
		// Flip one of the quaternions.
		f_cos_theta = ( f_cos_theta < -1.0 ) ? 1.0 : -f_cos_theta;
		p_in2[X] =-p_in2[X];
		p_in2[Y] =-p_in2[Y];
		p_in2[Z] =-p_in2[Z];
		p_in2[W] =-p_in2[W];
    }
    else
    {
		f_cos_theta = ( f_cos_theta > 1.0 ) ? 1.0 : f_cos_theta;
    }

	return f_cos_theta;
}

static bool IsQLinear( const Gfx::SAnimQFrame* p_data, unsigned int start, unsigned int end, double error )
{
	Mth::Quat at_start_frame;
	Mth::Quat at_end_frame;
	unsigned int i;

	// If just a two key run, always linear.
	if(( end - start ) <= 1 )
		return TRUE;

	at_start_frame	= p_data[start].q;
	at_end_frame	= p_data[end].q;
	for( i = start + 1; i < end; ++i )
	{
		Mth::Quat at_this_frame				= p_data[i].q;
		Mth::Quat interpolated_at_this_frame;
		double diff_at_this_frame;
		
		interpolated_at_this_frame = Mth::Slerp(at_start_frame,
			                                    at_end_frame,
										        (float)(i - start) / (float)(end - start));

		//RpSkinAnimQuatSlerp( &at_start_frame, &at_end_frame, (float)( i - start ) / (float)( end - start ), &interpolated_at_this_frame );
		
		diff_at_this_frame			= RtQuatDiff( at_this_frame, interpolated_at_this_frame );

		// The diff is measured in the scale [0.0,1.0], with 1.0 meaning an exact match.
		if( diff_at_this_frame < error )
			return FALSE;
	}
	return TRUE;
}

static bool IsFloatLinear( const GenKey<float>* vals, unsigned int start, unsigned int end, double error )
{
	double at_start_frame;
	double at_end_frame;
	double delta;
	unsigned int i;

	// If just a two key run, always linear.
	if(( end - start ) <= 1 )
		return TRUE;

	// Scan through and calculate the difference between each inbetween point and the line
	// joining start and end points. Too large a difference results in this set of points no longer being
	// considered linear.
	at_start_frame	= vals[start].key;
	at_end_frame	= vals[end].key;
	delta			= at_end_frame - at_start_frame;

	for( i = start + 1; i < end; ++i )
	{
		double at_this_frame			= vals[i].key;
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );

		if( diff_at_this_frame > error )
			return FALSE;
	}

	return TRUE;
}

static bool IsTLinear( const Gfx::SAnimTFrame* p_data, unsigned int start, unsigned int end, double error_x, double error_y, double error_z )
{
	double at_start_frame;
	double at_end_frame;
	double delta;
	unsigned int i;

	// If just a two key run, always linear.
	if(( end - start ) <= 1 )
		return TRUE;

	// Scan through and calculate the 'x' axis difference between each inbetween point and the line
	// joining start and end points. Too large a difference results in this set of points no longer being
	// considered linear.
	at_start_frame	= p_data[start].t[X];
	at_end_frame		= p_data[end].t[X];
	delta			= at_end_frame - at_start_frame;

	for( i = start + 1; i < end; ++i )
	{
		double at_this_frame			= p_data[i].t[X];
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );

		if( diff_at_this_frame > error_x )
			return FALSE;
	}

	at_start_frame	= p_data[start].t[Y];
	at_end_frame	= p_data[end].t[Y];
	delta			= at_end_frame - at_start_frame;
	for( i = start + 1; i < end; ++i )
	{
		double at_this_frame			= p_data[i].t[Y];
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );

		if( diff_at_this_frame > error_y )
			return FALSE;
	}

	at_start_frame	= p_data[start].t[Z];
	at_end_frame	= p_data[end].t[Z];
	delta			= at_end_frame - at_start_frame;
	for( i = start + 1; i < end; ++i )
	{
		double at_this_frame			= p_data[i].t[Z];
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );

		if( diff_at_this_frame > error_z )
			return FALSE;
	}
	return TRUE;
}

int AnimExporter::CompressRawQFrameData( Gfx::SAnimQFrame* p_newdata, Gfx::SAnimQFrame* p_data, int num_keys, double error )
{
//	Dbg_MemberFunction;

	int num_linear_keys;
	int current_index;
	int next_index;
	int next_write_index;
	int newdata_write_index=0;	// aml

	// Need at least 3 points to be able to compress data.
	if(( num_keys ) < 3 )
	{
		for(int i=0;i<num_keys;i++)
			p_newdata[i]=p_data[i];

		return num_keys;
	}

	num_linear_keys = 0;
	current_index = 0;
	next_index = 1;

	// The compressed frames are just written over the existing frames (since the data will never
	// be larger). This index keeps track of the next 'write' index.
	next_write_index = 0;
	
	while( current_index < ( num_keys - 1 ))
	{
		unsigned int run_length;

		// Continue stepping along a straight line...
		if( IsQLinear( p_data, current_index, next_index, error ) )
		{
			++next_index;
			if( next_index >= num_keys )
			{
				// Need to write out a key where the current index is.
				++num_linear_keys;
				//p_data[next_write_index++] = p_data[current_index];
				p_newdata[newdata_write_index++]=p_data[current_index];	// aml
				break;
			}
			else
				continue;
		}

		// ...until a curve is found.
		--next_index;
		run_length = next_index - current_index;		

		if( run_length > 1 )
		{
			// A linear section of at least 3 keys was found.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++]=p_data[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
		else
		{
			// Just write out the key.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++]=p_data[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
	}

	// Now just need to write out the end key.
	++num_linear_keys;
	p_newdata[newdata_write_index++]=p_data[num_keys - 1];	// aml

	//return num_linear_keys;
	return newdata_write_index;
}

int AnimExporter::CompressRawTFrameData( Gfx::SAnimTFrame* p_newdata, Gfx::SAnimTFrame* p_data, int num_keys, double error )
{
//	Dbg_MemberFunction;

	double min_x =  10000.0;
	double max_x = -10000.0;
	double min_y =  10000.0;
	double max_y = -10000.0;
	double min_z =  10000.0;
	double max_z = -10000.0;

	double error_x;
	double error_y;
	double error_z;

	int num_linear_keys;
	int current_index;
	int next_index;
	int next_write_index;
	int newdata_write_index=0;	// aml
	unsigned int i;

	// Need at least 3 points to be able to compress data.
	if(( num_keys ) < 3 )
	{
		for(int i=0;i<num_keys;i++)
			p_newdata[i]=p_data[i];

		return num_keys;
	}

	// Scan through the data, calculating min and max values...
	for( i = 0; i < num_keys; ++i )
	{
		if( p_data[i].t[X] > max_x )
			max_x = p_data[i].t[X];

		if( p_data[i].t[X] < min_x )
			min_x = p_data[i].t[X];

		if( p_data[i].t[Y] > max_y )
			max_y = p_data[i].t[Y];

		if( p_data[i].t[Y] < min_y )
			min_y = p_data[i].t[Y];

		if( p_data[i].t[Z] > max_z )
			max_z = p_data[i].t[Z];

		if( p_data[i].t[Z] < min_z )
			min_z = p_data[i].t[Z];
	}

	// ...and the error as a percentage of the range.
	error_x = ( fabs( max_x - min_x ) * error ) / 100.0;
	error_y = ( fabs( max_y - min_y ) * error ) / 100.0;
	error_z = ( fabs( max_z - min_z ) * error ) / 100.0;

	// Limit error from getting too small (in runs with very little or no variation).
	if( error_x < 0.005 )
		error_x = 0.005f;
	if( error_y < 0.005 )
		error_y = 0.005f;
	if( error_z < 0.005 )
		error_z = 0.005f;

	num_linear_keys = 0;
	current_index = 0;
	next_index = 1;

	// The compressed frames are just written over the existing frames (since the data will never
	// be larger). This index keeps track of the next 'write' index.
	next_write_index = 0;
	
	while( current_index < ( num_keys - 1 ))
	{
		unsigned int run_length;

		// Continue stepping along a straight line...
		if( IsTLinear( p_data, current_index, next_index, error_x, error_y, error_z ) )
		{
			++next_index;
			if( next_index >= num_keys )
			{
				// Need to write out a key where the current index is.
				++num_linear_keys;
				//p_data[next_write_index++] = p_data[current_index];
				p_newdata[newdata_write_index++] = p_data[current_index];	// aml
				break;
			}
			else
				continue;
		}

		// ...until a curve is found.
		--next_index;
		run_length = next_index - current_index;		

		if( run_length > 1 )
		{
			// A linear section of at least 3 keys was found.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++] = p_data[current_index];
			current_index = next_index;
			++next_index;
			continue;
		}
		else
		{
			// Just write out the key.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++] = p_data[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
	}

	// Now just need to write out the end key.
	++num_linear_keys;
	//p_data[next_write_index++] = p_data[num_keys - 1];
	p_newdata[newdata_write_index++] = p_data[num_keys - 1];	// aml

	//return num_linear_keys;
	return newdata_write_index;	// aml
}

int AnimExporter::CompressFloatKeys( GenKey<float>* newvals, GenKey<float>* vals, int num_keys, double Serror )
{
//	Dbg_MemberFunction;

	double min =  10000.0;
	double max = -10000.0;

	double error;

	int num_linear_keys;
	int current_index;
	int next_index;
	int next_write_index;
	int newdata_write_index=0;	// aml
	unsigned int i;

	// Need at least 3 points to be able to compress data.
	if(( num_keys ) < 3 )
	{
		for(int i=0;i<num_keys;i++)
			newvals[i]=vals[i];

		return num_keys;
	}

	// Scan through the data, calculating min and max values...
	for( i = 0; i < num_keys; ++i )
	{
		if( vals[i].key > max )
			max = vals[i].key;

		if( vals[i].key < min )
			min = vals[i].key;
	}

	// ...and the error as a percentage of the range.
	error = ( fabs( max - min ) * Serror ) / 100.0;

	// Limit error from getting too small (in runs with very little or no variation).
	if( error < 0.005 )
		error = 0.005f;

	num_linear_keys = 0;
	current_index = 0;
	next_index = 1;

	// The compressed frames are just written over the existing frames (since the data will never
	// be larger). This index keeps track of the next 'write' index.
	next_write_index = 0;
	
	while( current_index < ( num_keys - 1 ))
	{
		unsigned int run_length;

		// Continue stepping along a straight line...
		if ( IsFloatLinear( vals, current_index, next_index, error ) )
		{
			++next_index;
			if( next_index >= num_keys )
			{
				// Need to write out a key where the current index is.
				++num_linear_keys;
				//p_data[next_write_index++] = p_data[current_index];
				newvals[newdata_write_index++] = vals[current_index];	// aml
				break;
			}
			else
				continue;
		}

		// ...until a curve is found.
		--next_index;
		run_length = next_index - current_index;		

		if( run_length > 1 )
		{
			// A linear section of at least 3 keys was found.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			newvals[newdata_write_index++] = vals[current_index];
			current_index = next_index;
			++next_index;
			continue;
		}
		else
		{
			// Just write out the key.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			newvals[newdata_write_index++] = vals[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
	}

	// Now just need to write out the end key.
	++num_linear_keys;
	//p_data[next_write_index++] = p_data[num_keys - 1];
	newvals[newdata_write_index++] = vals[num_keys - 1];	// aml

	//return num_linear_keys;
	return newdata_write_index;	// aml
}

AnimExporter::AnimExporter()
{
	Qframes=NULL;
	Tframes=NULL;
	compQframes=NULL;
	compTframes=NULL;
	skeleton=NULL;
	bRemoveReadOnly=false;
	bSwapCoordSystem=false;

	numCompressedQFrames=NULL;
	numCompressedTFrames=NULL;
}

AnimExporter::~AnimExporter()
{
	if (Qframes)
		FreeFrames(&Qframes);

	if (Tframes)
		FreeFrames(&Tframes);

	if (compQframes)
		FreeFrames(&compQframes);

	if (compTframes)
		FreeFrames(&compTframes);

	if (skeleton)
		delete skeleton;
}

INode* AnimExporter::GetRoot(INode* node)
{
	assert(skeleton);
	return skeleton->GetRootBone(node);

	/*
	assert(node!=NULL);
	CStr name=node->GetName();


	INode* lastNode=node;

	while(node && !node->IsRootNode())
	{
		lastNode=node;
		node=node->GetParentNode();

		CStr name2=node->GetName();
	}
	

	// node should be the scene root
	return lastNode;
	*/
}

int AnimExporter::CountNodes(INode* node)
{
	assert(skeleton);
	return skeleton->GetCount();

	/*	
	CSkeletonData* skel=new CSkeletonData(node);
	int count=skel->GetCount();
	delete skel;
	*/
	
	/*
	int numChildren=node->NumberOfChildren();
	int count;

	if (!node->IsHidden())
		count=1;
	else
		count=0;

	for(int i=0;i<numChildren;i++)
	{
		INode* child=node->GetChildNode(i);

		if (!child->IsHidden())
			count+=CountNodes(node->GetChildNode(i));
	}

	return count;
	*/
}

bool AnimExporter::ExportNodes(INode* node)
{
	int fps3DS;								// Frames per second MAX is set to
	int num3DSFrames;						// Number of frames @ the selected 3DS FPS
	int reqFrames;							// Number of frames required for game's FPS rate

	// Don't export the scene root, but export it's children
	if (node->IsRootNode())
	{
		int numChildren=node->NumberOfChildren();
		
		for(int i=0;i<numChildren;i++)
			if (!ExportNodes(node->GetChildNode(i)))
				return false;

		return true;
	}

	fps3DS=GetFrameRate();
	num3DSFrames=end-start+1;
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;

	Gfx::SAnimQFrame Qframe;
	Gfx::SAnimTFrame Tframe;

	// Dump out animation frames at game FPS rate
	//for(int i=0;i<reqFrames;i++)
	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3 tm;
		Matrix3 parentTM;
		INode*  parentNode;

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// Build the game's structures for output
		/*
		if (node==rootbone)
		{
			tm.IdentityMatrix();
			parentTM.IdentityMatrix();
		}
		else
		*/

		// Get TM for Quats
		{
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bRotateRoot)
			{
				if (node==rootbone)
				{
					tm.RotateX(DegToRad(-90));
				}
			}

			if (bSwapCoordSystem)
			{	
				tm.RotateX(DegToRad(-90));
				//tm.RotateY(DegToRad(-90));
			}

			tm.NoScale();
			parentNode = node->GetParentNode();

			if (bSwapCoordSystem)
			{
				//parentTM = node->GetObjectTM(curFrame*GetTicksPerFrame());
				parentTM = node->GetNodeTM(curFrame*GetTicksPerFrame());
			}
			else
				parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			parentTM.NoScale();
			tm = tm * Inverse(parentTM);
		}

		Quat    quat=Quat(tm);
		Qframe.q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		//Qframe.time=(float)i/(float)OUTPUT_FPS;
		Qframe.time=(float)i/(float)fps3DS;

		// Get TM for Translations
		{
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bSwapCoordSystem)
			{
				//tm.RotateX(DegToRad(-90));
			}

			tm.NoScale();
			parentNode = node->GetParentNode();
			parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			parentTM.NoScale();
			tm = tm * Inverse(parentTM);
		}

		Point3 trans=tm.GetTrans();

		if (bSwapCoordSystem)
		{
			Tframe.t[X]=trans[X];
			Tframe.t[Y]=trans[Z];
			Tframe.t[Z]=-trans[Y];
		}
		else
		{
			Tframe.t[0]=trans[0];
			Tframe.t[1]=trans[1];
			Tframe.t[2]=trans[2];
		}

		//Tframe.time=(float)i/(float)OUTPUT_FPS;
		Tframe.time=(float)i/(float)fps3DS;
			
		// Dump transform to file
		fwrite(&Qframe,sizeof(Gfx::SAnimQFrame),1,fpExport);
		fwrite(&Tframe,sizeof(Gfx::SAnimTFrame),1,fpExport);
	}

	// Dump children
	for(i=0;i<node->NumberOfChildren();i++)
		if (!ExportNodes(node->GetChildNode(i)))
			return false;

	return true;
}

bool AnimExporter::Export(INode* root,char* Filename,char* ModelName,int start,int end)
{
	this->start=start;
	this->end=end;

	fpExport=fopen(Filename,"wb");

	if (!fpExport)
		return false;

	int numTransforms=CountNodes(root);
	fwrite(&numTransforms,sizeof(int),1,fpExport);  // Number of Transforms
	ExportNodes(root);

	fclose(fpExport);
	return true;
}

bool AnimExporter::GetAnim(INode* root,int start,int end,float errorQ,float errorT,TolData* tdata,bool bCompress,
						   int** numCompressedQFrames, Gfx::SAnimQFrame** compQFrames,
						   int** numCompressedTFrames, Gfx::SAnimTFrame** compTFrames,
						   int*  numNodes)
{
	if (Qframes)
		FreeFrames(&Qframes);

	if (Tframes)
		FreeFrames(&Tframes);

	if (numCompressedQFrames)
		delete [] numCompressedQFrames;

	if (numCompressedTFrames)
		delete [] numCompressedTFrames;

	if (*numNodes != 1)
		*numNodes = CountNodes(root);

	int fps3DS=GetFrameRate();						// 3DS MAX Frame rate
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	// Allocate space for the uncompressed animation data
	Qframes=new Gfx::SAnimQFrame[*numNodes*num3DSFrames];
	Tframes=new Gfx::SAnimTFrame[*numNodes*num3DSFrames];

	// Allocate space for the compressed animation data

	// Compiler not liking *compQFrames = new Gfx::SAnimQFrame[*numNodes*num3DSFrames]
	// (thinks it's incorrect type) reassigning to a local var works ????
	Gfx::SAnimQFrame** pcompQframes = compQFrames;
	Gfx::SAnimTFrame** pcompTframes = compTFrames;

	*pcompQframes = new Gfx::SAnimQFrame[*numNodes*num3DSFrames];
	*pcompTframes = new Gfx::SAnimTFrame[*numNodes*num3DSFrames];

	*numCompressedQFrames=new int[*numNodes];
	*numCompressedTFrames=new int[*numNodes];

	this->start=start;
	this->end=end;

	if (*numNodes == 1)
		GetQFrames(Qframes,root,true);
	else
		GetQFrames(Qframes,root,false);

	// Fix Flipped Euler rotations
	for(int i=0;i<*numNodes;i++)
	{
		INode* node;

		if (*numNodes != 1)
			node=skeleton->GetBone(i);
		else
			node=root;

		CStr name=node->GetName();

		FixEulerFlips(Qframes+i*num3DSFrames,num3DSFrames);
	}

	if (*numNodes == 1)
		GetTFrames(Tframes,root,true);
	else
		GetTFrames(Tframes,root,false);

	for(int curNode=0;curNode<*numNodes;curNode++)
	{
		if (!bCompress)
		{
			(*numCompressedQFrames)[curNode]=num3DSFrames;
			(*numCompressedTFrames)[curNode]=num3DSFrames;
		}
		else
		{
			// Handle variable compression entries
			INode* node;
			
			if (*numNodes != 1)
				node = skeleton->GetBone(curNode);
			else
				node = root;

			CStr   strClass;
			float  varErrorQ, varErrorT;

			if (!node->GetUserPropString("Class",strClass))
				node->GetUserPropString("class",strClass);

			TolRecord tdataFind;
			tdataFind.name = node->GetName();
			Link<TolRecord>* trLink;

			if (tdata)
			{
				trLink = tdata->DB.Find(&tdataFind);
			}
			else
				trLink=NULL;

			if (trLink)
			{
				if (trLink->data.bHasRot)
				{
					varErrorQ = trLink->data.valRot;
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
				}

				if (trLink->data.bHasTran)
				{
					varErrorT = trLink->data.valTran;
				}
				else
				{
					varErrorT = errorT;
				}
			}
			else
			{
				/* Leaving this in for now, just in case */
				if (strClass==CStr("VarAnimTolerance"))
				{
					node->GetUserPropFloat("RotTolerance",varErrorQ);
					node->GetUserPropFloat("TransTolerance",varErrorT);

					// Scale the rotation compression value appropriately
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
					varErrorT = errorT;
				}
			}

			(*numCompressedQFrames)[curNode]=CompressRawQFrameData( (*pcompQframes)+(curNode*num3DSFrames), Qframes+(curNode*num3DSFrames), num3DSFrames, varErrorQ );
			(*numCompressedTFrames)[curNode]=CompressRawTFrameData( (*pcompTframes)+(curNode*num3DSFrames), Tframes+(curNode*num3DSFrames), num3DSFrames, varErrorT );
		
			// Ensure that the last frame of animation is ALWAYS exported
			/*	
			Gfx::SAnimQFrame* qframes = (*pcompQframes)+(curNode*num3DSFrames);
			Gfx::SAnimTFrame* tframes = (*pcompTframes)+(curNode*num3DSFrames);

			int nQFrames = (*numCompressedQFrames)[curNode];
			int nTFrames = (*numCompressedTFrames)[curNode];

			float finalTime = ((float)end-(float)start+1.0f) / GetFrameRate();

			// If the last keyframe isn't at the final time, add the uncompressed key
			if (qframes[nQFrames-1].time < finalTime)
			{
				qframes[nQFrames] = Qframes[curNode*num3DSFrames-1];
				(*numCompressedQFrames)[curNode]++;
			}

			if (tframes[nTFrames-1].time < finalTime)
			{
				tframes[nTFrames] = Tframes[curNode*num3DSFrames-1];
				(*numCompressedTFrames)[curNode]++;
			}
			*/
		}
	}

	if (!bCompress)
	{
		for(int x=0;x<(*numNodes)*num3DSFrames;x++)
		{
			(*pcompQframes)[x]=Qframes[x];
			(*pcompTframes)[x]=Tframes[x];
		}
	}

	if (*numCompressedQFrames>0 ||
		*numCompressedTFrames>0)
		return true;

	return false;
}

bool AnimExporter::GetAnim(INode* root,int start,int end,float errorQ,float errorT,TolData* tdata,bool bCompress)
{
	if (Qframes)
		FreeFrames(&Qframes);

	if (Tframes)
		FreeFrames(&Tframes);

	if (compQframes)
		FreeFrames(&compQframes);

	if (compTframes)
		FreeFrames(&compTframes);

	if (numCompressedQFrames)
		delete [] numCompressedQFrames;

	if (numCompressedTFrames)
		delete [] numCompressedTFrames;

	int numNodes=CountNodes(root);					// Number of nodes (bones)	
	int fps3DS=GetFrameRate();						// 3DS MAX Frame rate
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	// Allocate space for the uncompressed animation data
	Qframes=new Gfx::SAnimQFrame[numNodes*num3DSFrames];
	Tframes=new Gfx::SAnimTFrame[numNodes*num3DSFrames];

	// Allocate space for the compressed animation data
	compQframes=new Gfx::SAnimQFrame[numNodes*num3DSFrames];
	compTframes=new Gfx::SAnimTFrame[numNodes*num3DSFrames];

	numCompressedQFrames=new int[numNodes];
	numCompressedTFrames=new int[numNodes];

	this->start=start;
	this->end=end;

	GetQFrames(Qframes,root);

	// Fix Flipped Euler rotations
	for(int i=0;i<numNodes;i++)
	{
		INode* node=skeleton->GetBone(i);
		CStr name=node->GetName();

//		if (name==CStr("burnq_board"))
//			__asm int 3;

		FixEulerFlips(Qframes+i*num3DSFrames,num3DSFrames);
	}

	GetTFrames(Tframes,root);

	for(int curNode=0;curNode<numNodes;curNode++)
	{
		if (!bCompress)
		{
			numCompressedQFrames[curNode]=num3DSFrames;
			numCompressedTFrames[curNode]=num3DSFrames;
		}
		else
		{
			// Handle variable compression entries
			INode* node = skeleton->GetBone(curNode);
			CStr   strClass;
			float  varErrorQ, varErrorT;

			if (!node->GetUserPropString("Class",strClass))
				node->GetUserPropString("class",strClass);

			TolRecord tdataFind;
			tdataFind.name = node->GetName();
			Link<TolRecord>* trLink;

			if (tdata)
			{
				trLink = tdata->DB.Find(&tdataFind);
			}
			else
				trLink=NULL;

			if (trLink)
			{
				if (trLink->data.bHasRot)
				{
					varErrorQ = trLink->data.valRot;
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
				}

				if (trLink->data.bHasTran)
				{
					varErrorT = trLink->data.valTran;
				}
				else
				{
					varErrorT = errorT;
				}
			}
			else
			{
				/* Leaving this in for now, just in case */
				if (strClass==CStr("VarAnimTolerance"))
				{
					node->GetUserPropFloat("RotTolerance",varErrorQ);
					node->GetUserPropFloat("TransTolerance",varErrorT);

					// Scale the rotation compression value appropriately
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
					varErrorT = errorT;
				}
			}

			numCompressedQFrames[curNode]=CompressRawQFrameData( compQframes+(curNode*num3DSFrames), Qframes+(curNode*num3DSFrames), num3DSFrames, varErrorQ );
			numCompressedTFrames[curNode]=CompressRawTFrameData( compTframes+(curNode*num3DSFrames), Tframes+(curNode*num3DSFrames), num3DSFrames, varErrorT );

			// RETURN HERE!
			// Ensure that the last frame of animation is ALWAYS exported		
			/*
			Gfx::SAnimQFrame* qframes = compQframes+(curNode*num3DSFrames);
			Gfx::SAnimTFrame* tframes = compTframes+(curNode*num3DSFrames);

			int nQFrames = numCompressedQFrames[curNode];
			int nTFrames = numCompressedTFrames[curNode];

			float finalTime = ((float)end-(float)start+1.0f) / GetFrameRate();

			// If the last keyframe isn't at the final time, add the uncompressed key
			if (qframes[nQFrames-1].time < finalTime)
			{
				qframes[nQFrames] = Qframes[curNode*num3DSFrames-1];
				qframes[nQFrames].time = finalTime;
				numCompressedQFrames[curNode]++;
			}

			if (tframes[nTFrames-1].time < finalTime)
			{
				tframes[nTFrames] = Tframes[curNode*num3DSFrames-1];
				tframes[nTFrames].time = finalTime;
				numCompressedTFrames[curNode]++;
			}
			*/
		}
	}

	if (!bCompress)
	{
		for(int x=0;x<numNodes*num3DSFrames;x++)
		{
			compQframes[x]=Qframes[x];
			compTframes[x]=Tframes[x];
		}
	}

	if (numCompressedQFrames>0 ||
		numCompressedTFrames>0)
		return true;




	return false;
}

///// temp
void AnimExporter::BuildNodeQFrames(Gfx::SAnimQFrame* qframes,INode* node)
{
	int num3DSFrames;
	int reqFrames;

	int fps3DS=GetFrameRate();
	num3DSFrames=end-start+1;
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;

	Quat prevQuat;

	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3  tm,parentTM;
		Quat     quat,quatParent;
		Interval interval=FOREVER;
		
		Control* TMController = node->GetTMController();
		Control* rot          = TMController->GetRotationController();
		Control* parentRot    = NULL;

		INode*   parentNode   = node->GetParentNode();

		TMController = parentNode->GetTMController();

		if (TMController)
			parentRot = TMController->GetRotationController();

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// If there's no rotation controller, grab the rotation from the node TM
		if (!rot)
		{
			// Build the game's structures for output
			if (bSwapCoordSystem)
			{
				//tm=node->GetObjectTM(curFrame*GetTicksPerFrame());
				tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
				tm.RotateX(DegToRad(-90));
				//tm.RotateY(DegToRad(-90));
			}
			else
				tm=node->GetNodeTM(curFrame*GetTicksPerFrame());	
			
			tm.NoScale();
			parentNode = node->GetParentNode();

			if (bSwapCoordSystem)
			{
				//parentTM=parentNode->GetObjectTM(curFrame*GetTicksPerFrame());
				parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
			}
			else
				parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			parentTM.NoScale();
			tm = tm * Inverse(parentTM);
		}
		else
		{
			// A rotation controller exists, grab the rotation
			rot->GetValue(curFrame*GetTicksPerFrame(),&quat,interval,CTRL_ABSOLUTE);
			interval=FOREVER;

			quat.MakeMatrix(tm);

			if (parentRot)
			{
				parentRot->GetValue(curFrame*GetTicksPerFrame(),&quatParent,interval,CTRL_ABSOLUTE);
				quatParent.MakeMatrix(parentTM);
			}
			else
			{
				// No parent rotation controller (use identity)
				//parentTM=Matrix3(1);

				parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
			}

			tm=tm*node->GetObjectTM(curFrame*GetTicksPerFrame());
			parentTM=parentTM*parentNode->GetObjectTM(curFrame*GetTicksPerFrame());

			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
			parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bSwapCoordSystem)
				tm.RotateX(DegToRad(-90));
			
			tm.NoScale();
			parentTM.NoScale();
			tm = tm * Inverse(parentTM);
		}

		quat=Quat(tm);

		if ( i != 0 )
		{
			quat.MakeClosest(prevQuat);
		}
		else
		{
			// make sure they're all anims start off with a positive w.
			// (only do this on frame 0!)
			if (quat.w<0.0f)
			{
				quat.x=-quat.x;
				quat.y=-quat.y;
				quat.z=-quat.z;
				quat.w=-quat.w;
			}
		}

		prevQuat = quat;

		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		qframes[i].time=(float)i/(float)fps3DS;
	}
}

/*
// Fill out an array of QFrames for a single node for all frames at the game's FPS rate
void AnimExporter::BuildNodeQFrames(Gfx::SAnimQFrame* qframes,INode* node)
{
	int num3DSFrames;							// Number of frames @ the selected 3DS FPS
	int reqFrames;								// Number of frames required for game's FPS rate

	int fps3DS=GetFrameRate();					// 3DS MAX Frame rate
	num3DSFrames=end-start;						// Number of frames in animation at 3DS MAX's frame rate
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	// Dump out animation frames at game's FPS rate
	//for(int i=0;i<reqFrames;i++)
	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3 tm;
		Matrix3 parentTM;
		INode*  parentNode;

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// Build the game's structures for output
		if (bSwapCoordSystem)
		{
			//tm=node->GetObjectTM(curFrame*GetTicksPerFrame());
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
			tm.RotateX(DegToRad(-90));
			//tm.RotateY(DegToRad(-90));
		}
		else
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());	
		
		tm.NoScale();
		parentNode = node->GetParentNode();

		if (bSwapCoordSystem)
		{
			//parentTM=parentNode->GetObjectTM(curFrame*GetTicksPerFrame());
			parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
		}
		else
			parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

		parentTM.NoScale();
		tm = tm * Inverse(parentTM);

		Quat    quat=Quat(tm);

		if (quat.w<0.0f)
		{
			quat.x=-quat.x;
			quat.y=-quat.y;
			quat.z=-quat.z;
			quat.w=-quat.w;
		}

		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		//qframes[i].time=(float)i/(float)OUTPUT_FPS;
		qframes[i].time=(float)i/(float)fps3DS;
	}
}
*/

// Creates a single TFrame for a single node for the given frame at the game's FPS rate
void AnimExporter::BuildNodeTFrames(Gfx::SAnimTFrame* tframes,INode* node)
{
	int num3DSFrames;							// Number of frames @ the selected 3DS FPS
	int reqFrames;								// Number of frames required for game's FPS rate

	int fps3DS=GetFrameRate();					// 3DS MAX Frame rate
	num3DSFrames=end-start+1;						// Number of frames in animation at 3DS MAX's frame rate
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	// Dump out animation frames at game's FPS rate
	//for(int i=0;i<reqFrames;i++)
	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3 tm;
		Matrix3 parentTM;
		INode*  parentNode;
		
		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// Build the game's structures for output
		/*
		if (node==rootbone)
		{
			tm.IdentityMatrix();
			parentTM.IdentityMatrix();
		}
		else
		*/
		{
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bRotateRoot)
			{
				if (node==rootbone)
				{
					tm.RotateX(DegToRad(-90));
				}
			}


			if (bSwapCoordSystem)
			{
				//tm.RotateX(DegToRad(-90));
			}

			tm.NoScale();
			parentNode = node->GetParentNode();
			parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			parentTM.NoScale();
			tm = tm * Inverse(parentTM);
		}

		Point3 trans=tm.GetTrans();

		if (bSwapCoordSystem)
		{
			tframes[i].t[X]=trans[X];
			tframes[i].t[Y]=trans[Z];
			tframes[i].t[Z]=-trans[Y];
		}
		else
		{
			tframes[i].t[X]=trans[X];
			tframes[i].t[Y]=trans[Y];
			tframes[i].t[Z]=trans[Z];
		}

		//tframes[i].time=(float)i/(float)OUTPUT_FPS;
		tframes[i].time=(float)i/(float)fps3DS;
	}
}

// Builds a list of SAnimQFrames for each rotation in the model
// at the game's FPS rate
void AnimExporter::GetQFrames(Gfx::SAnimQFrame* qframes,INode* root,bool bOneNode)
{
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate


	if (!bOneNode)
	{
		int numBones=skeleton->GetCount();
		for(int i=0;i<numBones;i++)
		{
			INode* node=skeleton->GetBone(i);
			BuildNodeQFrames(qframes+num3DSFrames*i,node);
		}
	}
	else
		BuildNodeQFrames(qframes,root);
}

// Builds a list of SAnimTFrames for each translation in the model
// at the game's FPS rate
void AnimExporter::GetTFrames(Gfx::SAnimTFrame* tframes,INode* root,bool bOneNode)
{
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate


	if (!bOneNode)
	{
		int numBones=skeleton->GetCount();
		for(int i=0;i<numBones;i++)
		{
			INode* node=skeleton->GetBone(i);
			BuildNodeTFrames(tframes+num3DSFrames*i,node);
		}
	}
	else
		BuildNodeTFrames(tframes,root);
}

void AnimExporter::FreeFrames(Gfx::SAnimQFrame** frame)
{
	if (frame)
		delete [] *frame;

	*frame=NULL;
}

void AnimExporter::FreeFrames(Gfx::SAnimTFrame** frame)
{
	if (frame)
		delete [] *frame;

	*frame=NULL;
}

int AnimExporter::GetNodeName(CStr* strNames,INode* node,int ID)
{
	if (node->IsHidden())
		ID--;
	else
		strNames[ID]=node->GetName();

	int numChildren=node->NumberOfChildren();

	INode* newNode;

	for(int i=0;i<numChildren;i++)
	{
		newNode=node->GetChildNode(i);
		ID=GetNodeName(strNames,newNode,++ID);
	}

	return ID;
}


int AnimExporter::GetParentName(CStr* strNames,INode* node,int ID)
{
	INode* parent=node->GetParentNode();

	if (node->IsHidden())
		ID--;
	else
		if (parent && !parent->IsHidden())
			strNames[ID]=parent->GetName();
		else
			strNames[ID]="";

	int numChildren=node->NumberOfChildren();

	INode* newNode;

	for(int i=0;i<numChildren;i++)
	{
		newNode=node->GetChildNode(i);
		ID=GetParentName(strNames,newNode,++ID);
	}

	return ID;
}

void AnimExporter::GetNodeNames(CStr* strNames,INode* root)
{
	int numChildren=root->NumberOfChildren();
	int ID=0;

	INode* node;

	if (!root->IsHidden())
		strNames[ID]=root->GetName();
	else
		ID=-1;

	for(int i=0;i<numChildren;i++)
	{
		node=root->GetChildNode(i);
		ID=GetNodeName(strNames,node,++ID);
	}
}

void AnimExporter::GetParentNames(CStr* strNames,INode* root)
{
	int numChildren=root->NumberOfChildren();
	int ID=0;

	INode* node;
	INode* parent=root->GetParentNode();

	if (!root->IsHidden())
	{
		if (parent && !parent->IsHidden())
			strNames[ID]=root->GetName();
		else
			strNames[ID]="";
	}
	else
		ID=-1;


	for(int i=0;i<numChildren;i++)
	{
		node=root->GetChildNode(i);
		ID=GetParentName(strNames,node,++ID);
	}
}

void AnimExporter::HideDummy(INode* node)
{
	CStr name=node->GetName();

	if (name==CStr("Dummy_scale"))
		node->Hide(TRUE);

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		HideDummy(newnode);
	}
}

void AnimExporter::UnhideDummy(INode* node)
{
	CStr name=node->GetName();

	if (name==CStr("Dummy_scale"))
		node->Hide(FALSE);

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		UnhideDummy(newnode);
	}
}

void AnimExporter::UnhideDummies()
{
	INode* node=gInterface->GetRootNode();

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		UnhideDummy(newnode);
	}
}

void AnimExporter::HideDummies()
{
	INode* node=gInterface->GetRootNode();

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		HideDummy(newnode);
	}
}

bool AnimExporter::ApproxEqual(float v1,float v2)
{
	if (v1 > v2-0.00005f &&
		v1 < v2+0.00005f)
		return true;

	return false;
}

void AnimExporter::RemoveDblKeys(int numNodes,int numFrames)
{
	for(int i=0;i<numNodes;i++)
	{
		int curFrame=i*numFrames;

		if (numCompressedQFrames[i]==2)
		{
			if (ApproxEqual(compQframes[curFrame].q[X],compQframes[curFrame+1].q[X]) &&
				ApproxEqual(compQframes[curFrame].q[Y],compQframes[curFrame+1].q[Y]) &&
				ApproxEqual(compQframes[curFrame].q[Z],compQframes[curFrame+1].q[Z]) &&
				ApproxEqual(compQframes[curFrame].q[W],compQframes[curFrame+1].q[W]))
			{
				//compQframes[curFrame].time=compQframes[curFrame+1].time;
				numCompressedQFrames[i]=1;
			}
		}

		if (numCompressedTFrames[i]==2)
		{
			if (ApproxEqual(compTframes[curFrame].t[0],compTframes[curFrame+1].t[0]) &&
			    ApproxEqual(compTframes[curFrame].t[1],compTframes[curFrame+1].t[1]) &&
				ApproxEqual(compTframes[curFrame].t[2],compTframes[curFrame+1].t[2]))
			{
				//compTframes[curFrame].time=compTframes[curFrame+1].time;
				numCompressedTFrames[i]=1;
			}
		}
	}
}

void AnimExporter::RemoveDblKeys(int numNodes, int numFrames,
								 Gfx::SAnimQFrame* compQframes,
								 Gfx::SAnimTFrame* compTframes,
								 int* numCompressedQFrames,
								 int* numCompressedTFrames)
{
	for(int i=0;i<numNodes;i++)
	{
		int curFrame=i*numFrames;

		if (numCompressedQFrames[i]==2)
		{
			if (compQframes[curFrame].q[X]==compQframes[curFrame+1].q[X] &&
				compQframes[curFrame].q[Y]==compQframes[curFrame+1].q[Y] &&
				compQframes[curFrame].q[Z]==compQframes[curFrame+1].q[Z] &&
				compQframes[curFrame].q[W]==compQframes[curFrame+1].q[W])
			{
				//compQframes[curFrame].time=compQframes[curFrame+1].time;
				numCompressedQFrames[i]=1;
			}
		}

		if (numCompressedTFrames[i]==2)
		{
			if (compTframes[curFrame].t[0]==compTframes[curFrame+1].t[0] &&
			    compTframes[curFrame].t[1]==compTframes[curFrame+1].t[1] &&
				compTframes[curFrame].t[2]==compTframes[curFrame+1].t[2])
			{
				//compTframes[curFrame].time=compTframes[curFrame+1].time;
				numCompressedTFrames[i]=1;
			}
		}
	}
}

bool AnimExporter::ExportAnim(HWND hwnd,INode* node,
							  char* Filename,
							  char* ModelName,
							  int start,
							  int end,
							  float errorQ,
							  float errorT,
							  TolData* tdata,
							  bool bOneFrame,
							  bool bDebug,
							  bool bUseCompression,
							  bool bCompressTime,
							  bool bRotateRoot)
{
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	this->bCompressTime = bCompressTime;
	this->bRotateRoot   = bRotateRoot;

	if (skeleton)
		delete skeleton;
	skeleton=new CSkeletonData(node);
	rootbone=skeleton->GetRootBone(node);

	//UnhideDummies();

	INode* root=GetRoot(node);
	int numNodes=CountNodes(root);

	if (!root)
		return false;

	CStr name2=node->GetName();
	CStr name=root->GetName();

	// Write out animation file
	fpExport=fopen(Filename,"wb");

	if (!fpExport)
	{
		if (bRemoveReadOnly)
		{
			if (_access(Filename,WRITE_ACCESS)!=0)
			{
				if (_chmod(Filename, _S_IREAD | _S_IWRITE)==-1)
				{
					MessageBox(NULL,"Couldn't change file attribute setting from read only.","Export Animations",MB_ICONSTOP|MB_OK);
					return false;
				}

				fpExport=fopen(Filename,"wb");
			}

			if (!fpExport)
				return false;
		}
		else
		{
			// Check the file attributes to see if the file is read-only
			if (_access(Filename,WRITE_ACCESS)!=0)
			{
				char ErrMsg[256];
				int  rVal;
				sprintf(ErrMsg,"The file '%s' does not have write access enabled.\nWould you like to manually remove the Read-Only flag on all items?",(char*)Filename);
				rVal=MessageBox(hwnd,ErrMsg,"Export Animations",MB_YESNOCANCEL|MB_ICONWARNING);

				if (rVal==IDYES)
				{
					bRemoveReadOnly=true;
					rVal=IDNO;
				}

				if (rVal==IDNO)
				{
					if (_chmod(Filename, _S_IREAD | _S_IWRITE)==-1)
					{
						MessageBox(NULL,"An error occurred while changing the file's permission settings!","Export Animations",MB_ICONSTOP|MB_OK);
						return false;
					}

					return ExportAnim(hwnd,node,Filename,ModelName,start,end,errorQ,errorT,tdata,bOneFrame,bDebug,bUseCompression);
				}
				
				return false;
			}

			return false;
		}
	}

	GetAnim(root,start,end,errorQ,errorT,tdata,bUseCompression);
	
	if (bOneFrame)
		RemoveDblKeys(numNodes,num3DSFrames);

	// Compute total QFrames and TFrames
	int QKeys=0;
	int TKeys=0;
	int i;

	for(i=0;i<numNodes;i++)
	{
		QKeys+=numCompressedQFrames[i];
		TKeys+=numCompressedTFrames[i];		
	}
	
	// Output the stored data in THPS4 format
	//CSkeletonData* skeleton=new CSkeletonData(node);

	// Output file header
	SHeader header;
	header.version=ANIMFILE_VERSION;
	header.flags=INTERMEDIATE_FORMAT;
	
	if (!bUseCompression)
		header.flags|=UNCOMPRESSED_FORMAT;

	if (bRotateRoot)
		header.flags|=PREROTATEDROOT;

	if (bCompressTime)
		header.flags|=COMPRESSEDTIME;

	header.skeletonName=skeleton->GetChecksum();			// hard-coded default for now
	header.duration=(float)(num3DSFrames-1)/(float)fps3DS;
	header.numBones=numNodes;
	header.numQKeys=QKeys;
	header.numTKeys=TKeys;
	header.numUserDefinedKeys=0;



	if (!header.Write(fpExport))
	{
		delete skeleton;
		skeleton=NULL;
		return false;
	}

	// Output Skeleton data

	// Output using CSkeleton instead
	if (!skeleton->Write(fpExport))
	{
		delete skeleton;
		skeleton=NULL;
		return false;
	}

	///////////////////////////////////////////////////

	// Output compressed rotation data
	SQData sqdata(numNodes,QKeys);
	int QKeyPos=0;

	FILE* fpDebug=NULL;

	if (bDebug)
	{
		fpDebug=fopen("c:\\debug.txt","w");

		fprintf(fpDebug,"Duration: %f\n",header.duration);
		fprintf(fpDebug,"NumTKeys: %i\n",header.numTKeys);
		fprintf(fpDebug,"NumQKeys: %i\n",header.numQKeys);

		if (!fpDebug)
			MessageBox(NULL,"Failed to open c:\\debug.txt","Debug output",MB_OK|MB_ICONWARNING);
	}

	for(i=0;i<numNodes;i++)
	{
		sqdata.numKeysPerBone[i]=numCompressedQFrames[i];

		if (bDebug)
		{
			CStr name=skeleton->GetBone(i)->GetName();
			fprintf(fpDebug,"Node: %s\n",(char*)name);
		}

		for(int j=0;j<numCompressedQFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
				float ftmp = compQframes[curFrame].time * (float)OUTPUT_FPS;
				*val = Round(ftmp);
			}
			else
				sqdata.theQKeys[QKeyPos].time = compQframes[curFrame].time;

			sqdata.theQKeys[QKeyPos].qx   = compQframes[curFrame].q[X];
			sqdata.theQKeys[QKeyPos].qy   = compQframes[curFrame].q[Y];
			sqdata.theQKeys[QKeyPos].qz   = compQframes[curFrame].q[Z];
			sqdata.theQKeys[QKeyPos].real = compQframes[curFrame].q[W];

			if (bDebug)
			{
				float EulerAng[3],Deg1[3];
				Quat quat = Quat(sqdata.theQKeys[QKeyPos].qx,
					             sqdata.theQKeys[QKeyPos].qy,
					             sqdata.theQKeys[QKeyPos].qz,
					             sqdata.theQKeys[QKeyPos].real);
				
				QuatToEuler(quat,EulerAng);
				
				Deg1[X]=RadToDeg(EulerAng[X]);
				Deg1[Y]=RadToDeg(EulerAng[Y]);
				Deg1[Z]=RadToDeg(EulerAng[Z]);

				if (bCompressTime)
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
				else
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%f\n",i,j,compQframes[curFrame].time);

				fprintf(fpDebug,"Q[node=%i frame=%i].qx=%f\tEulerX: %f\tDeg: %f\tFlip: %i\n",i,j,compQframes[curFrame].q[X],EulerAng[X],Deg1[X],compQframes[curFrame].bFlip[X]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qy=%f\tEulerY: %f\tDeg: %f\tFlip: %i\n",i,j,compQframes[curFrame].q[Y],EulerAng[Y],Deg1[Y],compQframes[curFrame].bFlip[Y]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qz=%f\tEulerZ: %f\tDeg: %f\tFlip: %i\n",i,j,compQframes[curFrame].q[Z],EulerAng[Z],Deg1[Z],compQframes[curFrame].bFlip[Z]);
				fprintf(fpDebug,"Q[node=%i frame=%i].real=%f\n\n",i,j,compQframes[curFrame].q[W]);
			}

			QKeyPos++;
		}
	}

	if (!sqdata.Write(fpExport))
		return false;

	// Output compressed translation data
	STData stdata(numNodes,TKeys);
	int TKeyPos=0;

	for(i=0;i<numNodes;i++)
	{
		stdata.numKeysPerBone[i]=numCompressedTFrames[i];
		
		if (bDebug)
		{
			CStr name=skeleton->GetBone(i)->GetName();
			fprintf(fpDebug,"Node: %s\n",(char*)name);
		}

		for(int j=0;j<numCompressedTFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&stdata.theTKeys[TKeyPos].time;
				float ftmp = compTframes[curFrame].time * (float)OUTPUT_FPS;
				*val = Round(ftmp);
			}
			else
				stdata.theTKeys[TKeyPos].time = compTframes[curFrame].time;
	
			stdata.theTKeys[TKeyPos].tx   = compTframes[curFrame].t[0];
			stdata.theTKeys[TKeyPos].ty   = compTframes[curFrame].t[1];
			stdata.theTKeys[TKeyPos].tz   = compTframes[curFrame].t[2];

			if (bDebug)
			{
				if (bCompressTime)
					fprintf(fpDebug,"T[node=%i frame=%i].time=%i\n",i,j,*((int*)&stdata.theTKeys[TKeyPos].time));
				else
					fprintf(fpDebug,"T[node=%i frame=%i].time=%f\n",i,j,compTframes[curFrame].time);

				fprintf(fpDebug,"T[node=%i frame=%i].tx=%f\n",i,j,compTframes[curFrame].t[0]);
				fprintf(fpDebug,"T[node=%i frame=%i].ty=%f\n",i,j,compTframes[curFrame].t[1]);
				fprintf(fpDebug,"T[node=%i frame=%i].tz=%f\n\n",i,j,compTframes[curFrame].t[2]);
			}

			TKeyPos++;
		}
	}

	if (bDebug)
		fclose(fpDebug);

	if (!stdata.Write(fpExport))
		return false;

	fclose(fpExport);

	//HideDummies();

	delete skeleton;
	skeleton=NULL;

	return true;
}

unsigned int AnimExporter::GetChecksum(char* Filename)
{
	FILE* fp=fopen(Filename,"rb");

	if (!fp)
		return 0;
	
	unsigned int checksum;

	SHeader header;
	if (!header.Read(fp))
	{
		fclose(fp);
		return 0;
	}

	// The next 4 bytes in the file should be the skeleton checksum
	if (!fread(&checksum,sizeof(unsigned int),1,fp))
	{
		fclose(fp);
		return 0;
	}

	fclose(fp);
	return checksum;
}

// This copy method will sample the from controller and smooth out all flips
// Nikolai 1-15-99

// This is based off the code in \MAXSDK\SAMPLES\CONTROLLERS\EULRCTRL.CPP
// in EulerRotation::Copy(Control *from)
// This should be done BEFORE compression
//#define THRESHHOLD 3.140350877   // Allow rotations of up to 179 degrees per frame (not quite PI)
#define THRESHHOLD 1.0

void AnimExporter::FixEulerFlips(Gfx::SAnimQFrame* pdata,int numKeys)
{
	// Don't fix Euler flips, so we can check this in
#ifndef USE_EULERFIX
	return;
#endif

	if (numKeys<2)
		return;

	/*
	for(int curKey=0;curKey<numKeys;curKey++)
	{
		float EulerAng[3];

		Quat quat = Quat(pdata[curKey].q[X],
						 pdata[curKey].q[Y],
						 pdata[curKey].q[Z],
						 pdata[curKey].q[W]);

		QuatToEuler(quat,EulerAng);
		
		EulerAng[X]=RadToDeg(EulerAng[X]);
		EulerAng[Y]=RadToDeg(EulerAng[Y]);
		EulerAng[Z]=RadToDeg(EulerAng[Z]);

		int i;
		i=0;
	}
	*/

	Quat qPrev;
	Quat qCurr;

	float   eaCurr[3];
	float   eaPrev[3];
	float   EulerAng[3] = {0,0,0};
	Matrix3 tmPrev,tmCurr;

	// Fill out a MAX Quat with our quat
	qPrev.x = pdata[0].q[X];
	qPrev.y = pdata[0].q[Y];
	qPrev.z = pdata[0].q[Z];
	qPrev.w = pdata[0].q[W];
	
	float dEuler[3],f;	

	// Here we sample over the time range, to detect flips
	for(int curKey=1;curKey<numKeys;curKey++)
	{
		//from->GetValue(time,&qCurr,iv);
		qCurr.x = pdata[curKey].q[X];
		qCurr.y = pdata[curKey].q[Y];
		qCurr.z = pdata[curKey].q[Z];
		qCurr.w = pdata[curKey].q[W];
		
		QuatToEuler(qPrev,eaPrev);
		QuatToEuler(qCurr,eaCurr);

		float prevDeg[3],currDeg[3];		// Switch to degrees for DEBUG
		prevDeg[0]=RadToDeg(eaPrev[0]);
		prevDeg[1]=RadToDeg(eaPrev[1]);
		prevDeg[2]=RadToDeg(eaPrev[2]);

		currDeg[0]=RadToDeg(eaCurr[0]);
		currDeg[1]=RadToDeg(eaCurr[1]);
		currDeg[2]=RadToDeg(eaCurr[2]);

		// The Euler/Quat ratio is the relation of the angle difference in Euler space to 
		// the angle difference in Quat space. If this ration is bigger than PI the rotation 
		// between the two time steps contains a flip

		qPrev.MakeMatrix(tmPrev);
		qCurr.MakeMatrix(tmCurr);

		f = GetEulerMatAngleRatio(tmPrev,tmCurr,eaPrev,eaCurr,EULERTYPE_XYZ);	

		bool bFlip[3];
		bFlip[X]=false;
		bFlip[Y]=false;
		bFlip[Z]=false;

		QuatToEuler(qPrev,eaPrev);
		QuatToEuler(qCurr,eaCurr);

		if ( f > PI )
		//if (1)
		{
			for(int j=0;j<3;j++)
			{
				// find the sign flip (Over 110 degrees for a frame, we flip)
				if ((eaPrev[j]<0 && eaCurr[j]>0) ||
					(eaPrev[j]>0 && eaCurr[j]<0))
				{
					// The values should be close together omiting the sign
					float fval=fabs(RadToDeg(eaCurr[j])+RadToDeg(eaPrev[j]));
					
					// But the distance should be far away from the other if it was flipped
					float fdist=fabs(RadToDeg(eaPrev[j])-RadToDeg(eaCurr[j]));

					GetEulerMatAngleRatio(tmPrev,tmCurr,eaPrev,eaCurr,EULERTYPE_XYZ);

					if (fval<35 && fdist>100)
					{
						char strErr[256];
						sprintf(strErr,"Frame: %i  Flip Axis: %i\nPrevDeg: %f  CurrDeg: %f\nDiff: %f",curKey,j,RadToDeg(eaPrev[j]),RadToDeg(eaCurr[j]),fval);

						MessageBox(NULL,strErr,"Debug",MB_OK);

						//EulerAng[j] = eaCurr[j];// * -1.0f;	// Flip it back
						EulerAng[j] = (2*PI - (float) (fabs(eaCurr[j]) + fabs(eaPrev[j]))) * (eaPrev[j] > 0 ? 1 : -1);
						bFlip[j]=true;
					}
					else
						EulerAng[j] = eaCurr[j];
				}
			}
		}
		else
		{
			for(int j=0 ; j < 3 ; j++)
			{
				dEuler[j] = eaCurr[j]-eaPrev[j];
				EulerAng[j] += dEuler[j];
			}
		}
		
		/*
		if(  f > PI)
		{
			// We found a flip here
			for(int j=0 ; j < 3 ; j++)
			{				
				// find the sign flip :
				if(fabs((eaCurr[j]-eaPrev[j])) < 2*PI-THRESHHOLD )
					dEuler[j] = eaCurr[j]-eaPrev[j];
				else
				{
					char strErr[256];
					sprintf(strErr,"Frame: %i  Flip Axis: %i\nPrevDeg: %f  CurrDeg: %f\nDiff: %f",curKey,j,RadToDeg(eaPrev[j]),RadToDeg(eaCurr[j]),RadToDeg(f));

					MessageBox(NULL,strErr,"Debug",MB_OK);

					// unflip the flip
					dEuler[j] = (2*PI - (float) (fabs(eaCurr[j]) + fabs(eaPrev[j]))) * (eaPrev[j] > 0 ? 1 : -1);
				}
				
				EulerAng[j] += dEuler[j];
			}
		}
		else
		{
			// Add up the angle difference
			for(int j=0 ; j < 3 ; j++)
			{
				dEuler[j] = eaCurr[j]-eaPrev[j];
				EulerAng[j] += dEuler[j];
			}
		}
		*/

		// Write out our new key
		Quat qTemp;
		EulerToQuat(EulerAng,qTemp,EULERTYPE_XYZ);
		pdata[curKey].q[X] = -qTemp.x;
		pdata[curKey].q[Y] = -qTemp.y;
		pdata[curKey].q[Z] = -qTemp.z;
		pdata[curKey].q[W] = -qTemp.w;
		pdata[curKey].bFlip[X]=bFlip[X];
		pdata[curKey].bFlip[Y]=bFlip[Y];
		pdata[curKey].bFlip[Z]=bFlip[Z];
		qPrev = qCurr;
	}
}
