/*
	ObjectIO.cpp
	Functions responsible for Reading and Posting NxObjects to/from memory
*/

#include "stdafx.h"
#include "ObjectIO.h"
#include "MemIO.h"

NxObject* ReadObject(void *pMem)
{
	NxObject* pObj = new NxObject;

	unsigned char* pPos = (unsigned char*)pMem;

	Read(&pPos, sizeof(int), &pObj->m_NumFaces);
	pObj->m_Faces = new NxFace[pObj->m_NumFaces];

	Read(&pPos, sizeof(NxFace) * pObj->m_NumFaces, pObj->m_Faces);
	Read(&pPos, sizeof(int), &pObj->m_NumVerts);
	pObj->m_Verts = new NxVertex[pObj->m_NumVerts];

	Read(&pPos, sizeof(NxVertex) * pObj->m_NumVerts, pObj->m_Verts);

	GetString(&pPos, pObj->m_Name);

	Read(&pPos, sizeof(float), &pObj->m_BoundingBox.m_Min[0]);
	Read(&pPos, sizeof(float), &pObj->m_BoundingBox.m_Min[1]);
	Read(&pPos, sizeof(float), &pObj->m_BoundingBox.m_Min[2]);
	Read(&pPos, sizeof(float), &pObj->m_BoundingBox.m_Max[0]);
	Read(&pPos, sizeof(float), &pObj->m_BoundingBox.m_Max[1]);
	Read(&pPos, sizeof(float), &pObj->m_BoundingBox.m_Max[2]);

	Read(&pPos, sizeof(int), &pObj->m_Flags);
	Read(&pPos, sizeof(int), &pObj->m_NumUVSets);
	Read(&pPos, sizeof(unsigned long), &pObj->m_SkeletonChecksum);
	Read(&pPos, sizeof(int), &pObj->m_CASRemoveFlags);
	Read(&pPos, sizeof(float), &pObj->m_KBias);
	Read(&pPos, sizeof(int), &pObj->m_ParentMatrixIndex);

	Read(&pPos, sizeof(unsigned int), &pObj->m_LODVersion);
	Read(&pPos, sizeof(int), &pObj->m_LODFlags);
	
	if (pObj->m_LODFlags & NxObject::mMASTER)
	{
		Read(&pPos, sizeof(int), &pObj->m_LODMaster.m_NumLODLevels);
		pObj->m_LODMaster.m_LODLevels = new NxLODLevel[pObj->m_LODMaster.m_NumLODLevels];

		Read(&pPos, sizeof(NxLODLevel) * pObj->m_LODMaster.m_NumLODLevels, pObj->m_LODMaster.m_LODLevels);
	}

	if (pObj->m_LODFlags & NxObject::mSLAVE)
	{
		Read(&pPos, sizeof(unsigned long), &pObj->m_LODSlave.m_masterCRC);
	}

	Read(&pPos, sizeof(unsigned int), &pObj->m_Version);
	Read(&pPos, sizeof(int), &pObj->m_ParentCRC);
	
	Read(&pPos, sizeof(int), &pObj->m_NumChildren);
	pObj->m_ChildCRCs = new unsigned long[pObj->m_NumChildren];

	Read(&pPos, sizeof(unsigned long) * pObj->m_NumChildren, pObj->m_ChildCRCs);

	Read(&pPos, sizeof(int), &pObj->m_LODLevels);
	pObj->m_LODinfo = new NxLODInfo[pObj->m_LODLevels];
	Read(&pPos, sizeof(NxLODInfo) * pObj->m_LODLevels, pObj->m_LODinfo);

	Read(&pPos, sizeof(NxObject::BillboardType), &pObj->m_BillboardType);
	Read(&pPos, sizeof(Point3), &pObj->m_BillboardOrigin);
	Read(&pPos, sizeof(Point3), &pObj->m_PivotPos);
	Read(&pPos, sizeof(Point3), &pObj->m_PivotAxis);

	return pObj;
}

int StoreObject(NxObject* obj, void *pMem)
{
	unsigned char* pPos = (unsigned char*)pMem;
	int size = 0;

	size += Write(&pPos, sizeof(int), &obj->m_NumFaces);
	size += Write(&pPos, sizeof(NxFace) * obj->m_NumFaces, obj->m_Faces);
	size += Write(&pPos, sizeof(int), &obj->m_NumVerts);
	size += Write(&pPos, sizeof(NxVertex) * obj->m_NumVerts, obj->m_Verts);

	size += WriteString(&pPos, obj->m_Name);

	size += Write(&pPos, sizeof(float), &obj->m_BoundingBox.m_Min[0]);
	size += Write(&pPos, sizeof(float), &obj->m_BoundingBox.m_Min[1]);
	size += Write(&pPos, sizeof(float), &obj->m_BoundingBox.m_Min[2]);
	size += Write(&pPos, sizeof(float), &obj->m_BoundingBox.m_Max[0]);
	size += Write(&pPos, sizeof(float), &obj->m_BoundingBox.m_Max[1]);
	size += Write(&pPos, sizeof(float), &obj->m_BoundingBox.m_Max[2]);

	size += Write(&pPos, sizeof(int), &obj->m_Flags);
	size += Write(&pPos, sizeof(int), &obj->m_NumUVSets);
	size += Write(&pPos, sizeof(unsigned long), &obj->m_SkeletonChecksum);
	size += Write(&pPos, sizeof(int), &obj->m_CASRemoveFlags);
	size += Write(&pPos, sizeof(float), &obj->m_KBias);
	size += Write(&pPos, sizeof(int), &obj->m_ParentMatrixIndex);

	size += Write(&pPos, sizeof(unsigned int), &obj->m_LODVersion);
	size += Write(&pPos, sizeof(int), &obj->m_LODFlags);
	
	if (obj->m_LODFlags & NxObject::mMASTER)
	{
		size += Write(&pPos, sizeof(int), &obj->m_LODMaster.m_NumLODLevels);
		size += Write(&pPos, sizeof(NxLODLevel) * obj->m_LODMaster.m_NumLODLevels, obj->m_LODMaster.m_LODLevels);
	}

	if (obj->m_LODFlags & NxObject::mSLAVE)
	{
		size += Write(&pPos, sizeof(unsigned long), &obj->m_LODSlave.m_masterCRC);
	}

	size += Write(&pPos, sizeof(unsigned int), &obj->m_Version);
	size += Write(&pPos, sizeof(int), &obj->m_ParentCRC);
	
	size += Write(&pPos, sizeof(int), &obj->m_NumChildren);
	size += Write(&pPos, sizeof(unsigned long) * obj->m_NumChildren, obj->m_ChildCRCs);

	size += Write(&pPos, sizeof(int), &obj->m_LODLevels);
	size += Write(&pPos, sizeof(NxLODInfo) * obj->m_LODLevels, obj->m_LODinfo);

	size += Write(&pPos, sizeof(NxObject::BillboardType), &obj->m_BillboardType);
	size += Write(&pPos, sizeof(Point3), &obj->m_BillboardOrigin);
	size += Write(&pPos, sizeof(Point3), &obj->m_PivotPos);
	size += Write(&pPos, sizeof(Point3), &obj->m_PivotAxis);

	return size;
}

///////////////////////////////////////////////////////////////////////////////////////

void ReadTexture(NxTexture* pTexture, void *pMem)
{
	unsigned char* pPos = (unsigned char*)pMem;
	
	Read(&pPos, sizeof(int), &pTexture->m_PixelFormat);
	Read(&pPos, sizeof(int), &pTexture->m_PaletteFormat);

	Read(&pPos, sizeof(int) * NxTexture::vMAX_MIP_LEVELS, pTexture->m_Width);
	Read(&pPos, sizeof(int) * NxTexture::vMAX_MIP_LEVELS, pTexture->m_Height);

	Read(&pPos, sizeof(int), &pTexture->m_Bpp);
	Read(&pPos, sizeof(int), &pTexture->m_PaletteBpp);
	Read(&pPos, sizeof(int), &pTexture->m_TotalPaletteDataSize);
	Read(&pPos, sizeof(int), &pTexture->m_TotalTexelDataSize);

	Read(&pPos, sizeof(int) * NxTexture::vMAX_MIP_LEVELS, pTexture->m_TexelDataSize);
	
	for(int i = 0; i < NxTexture::vMAX_MIP_LEVELS; i++)
	{
		Read(&pPos, sizeof(int), &pTexture->m_TexelDataSize[i]);
		Read(&pPos, pTexture->m_TexelDataSize[i], &pTexture->m_TexelData[i]);
	}

	Read(&pPos, pTexture->m_TotalPaletteDataSize, pTexture->m_PaletteData);
	Read(&pPos, sizeof(int), &pTexture->m_NumPaletteEntries);
	Read(&pPos, sizeof(int), &pTexture->m_MipLevels);
	Read(&pPos, sizeof(int), &pTexture->m_Flags);
	Read(&pPos, sizeof(int), &pTexture->m_PlatFlags);
	
	GetString(&pPos, pTexture->m_Name);

	Read(&pPos, sizeof(unsigned long), &pTexture->m_Checksum);
	Read(&pPos, sizeof(float), &pTexture->m_LastDrawOrder);	
	Read(&pPos, sizeof(int), &pTexture->m_LastGroupIndex);
	Read(&pPos, sizeof(int), &pTexture->m_LastGroupId);
	Read(&pPos, sizeof(int), &pTexture->m_GroupFlags);
	Read(&pPos, sizeof(bool), &pTexture->m_AlreadyExported);
}

int  StoreTexture(NxTexture* pTexture, void *pMem)
{
	unsigned char* pPos = (unsigned char*)pMem;
	int size = 0;

	size += Write(&pPos, sizeof(int), &pTexture->m_PixelFormat);
	size += Write(&pPos, sizeof(int), &pTexture->m_PaletteFormat);

	size += Write(&pPos, sizeof(int) * NxTexture::vMAX_MIP_LEVELS, pTexture->m_Width);
	size += Write(&pPos, sizeof(int) * NxTexture::vMAX_MIP_LEVELS, pTexture->m_Height);

	size += Write(&pPos, sizeof(int), &pTexture->m_Bpp);
	size += Write(&pPos, sizeof(int), &pTexture->m_PaletteBpp);
	size += Write(&pPos, sizeof(int), &pTexture->m_TotalPaletteDataSize);
	size += Write(&pPos, sizeof(int), &pTexture->m_TotalTexelDataSize);

	size += Write(&pPos, sizeof(int) * NxTexture::vMAX_MIP_LEVELS, pTexture->m_TexelDataSize);
	
	for(int i = 0; i < NxTexture::vMAX_MIP_LEVELS; i++)
	{
		size += Write(&pPos, sizeof(int), &pTexture->m_TexelDataSize[i]);
		size += Write(&pPos, pTexture->m_TexelDataSize[i], &pTexture->m_TexelData[i]);
	}

	size += Write(&pPos, pTexture->m_TotalPaletteDataSize, pTexture->m_PaletteData);
	size += Write(&pPos, sizeof(int), &pTexture->m_NumPaletteEntries);
	size += Write(&pPos, sizeof(int), &pTexture->m_MipLevels);
	size += Write(&pPos, sizeof(int), &pTexture->m_Flags);
	size += Write(&pPos, sizeof(int), &pTexture->m_PlatFlags);
	
	size += WriteString(&pPos, pTexture->m_Name);

	size += Write(&pPos, sizeof(unsigned long), &pTexture->m_Checksum);
	size += Write(&pPos, sizeof(float), &pTexture->m_LastDrawOrder);	
	size += Write(&pPos, sizeof(int), &pTexture->m_LastGroupIndex);
	size += Write(&pPos, sizeof(int), &pTexture->m_LastGroupId);
	size += Write(&pPos, sizeof(int), &pTexture->m_GroupFlags);
	size += Write(&pPos, sizeof(bool), &pTexture->m_AlreadyExported);

	return size;
}

NxMaterial* ReadMaterial(void *pMem)
{
	unsigned char* pPos = (unsigned char*)pMem;

	NxMaterial* pMtl = new NxMaterial;

	GetString(&pPos, pMtl->m_Name);
	// m_MaxMtl must be relinked by m_Name on completion

	Read(&pPos, sizeof(unsigned long), &pMtl->m_Checksum);
	Read(&pPos, sizeof(unsigned long), &pMtl->m_NewChecksum);	
	Read(&pPos, sizeof(bool), &pMtl->m_Transparent);
	Read(&pPos, sizeof(int), &pMtl->m_NumPasses);
	Read(&pPos, sizeof(int), &pMtl->m_AlphaCutoff);
	Read(&pPos, sizeof(int), &pMtl->m_Terrain);
	Read(&pPos, sizeof(bool), &pMtl->m_WarnedAlready);
	Read(&pPos, sizeof(bool), &pMtl->m_OneSided);
	Read(&pPos, sizeof(bool), &pMtl->m_TwoSided);
	Read(&pPos, sizeof(bool), &pMtl->m_Invisible);
	Read(&pPos, sizeof(float), &pMtl->m_DrawOrder);
	Read(&pPos, sizeof(int), &pMtl->m_BasePass);
	Read(&pPos, sizeof(bool), &pMtl->m_Sorted);
	Read(&pPos, sizeof(int), &pMtl->m_CutoffFunc);
	Read(&pPos, sizeof(bool), &pMtl->m_water);
	Read(&pPos, sizeof(bool), &pMtl->m_grassify);
	Read(&pPos, sizeof(float), &pMtl->m_grassHeight);
	Read(&pPos, sizeof(int), &pMtl->m_grassLayers);
	Read(&pPos, sizeof(int), &pMtl->m_NumWibbleSequences);
	
	for(int i = 0; i < pMtl->m_NumWibbleSequences; i++)
	{
		Read(&pPos, sizeof(int), &pMtl->m_WibbleSequences[i].m_NumFrames);
		Read(&pPos, sizeof(WibbleKeyframe) * pMtl->m_WibbleSequences[i].m_NumFrames, pMtl->m_WibbleSequences[i].m_WibbleFrames);
	}

	// Fortunately NxMaterial pass contains no variably sized fields so we can do one big memcpy
	Read(&pPos, sizeof(NxMaterialPass) * vMAX_MATERIAL_PASSES, pMtl->m_Passes);

	Read(&pPos, sizeof(bool), &pMtl->m_UseSpecular);
	Read(&pPos, sizeof(float), &pMtl->m_SpecularPower);
	Read(&pPos, sizeof(float) * 3, pMtl->m_SpecularColor);

	return pMtl;
}

int  StoreMaterial(NxMaterial* pMtl, void *pMem)
{
	unsigned char* pPos = (unsigned char*)pMem;
	int size = 0;

	size += WriteString(&pPos, pMtl->m_Name);
	// m_MaxMtl must be relinked by m_Name on completion

	size += Write(&pPos, sizeof(unsigned long), &pMtl->m_Checksum);
	size += Write(&pPos, sizeof(unsigned long), &pMtl->m_NewChecksum);	
	size += Write(&pPos, sizeof(bool), &pMtl->m_Transparent);
	size += Write(&pPos, sizeof(int), &pMtl->m_NumPasses);
	size += Write(&pPos, sizeof(int), &pMtl->m_AlphaCutoff);
	size += Write(&pPos, sizeof(int), &pMtl->m_Terrain);
	size += Write(&pPos, sizeof(bool), &pMtl->m_WarnedAlready);
	size += Write(&pPos, sizeof(bool), &pMtl->m_OneSided);
	size += Write(&pPos, sizeof(bool), &pMtl->m_TwoSided);
	size += Write(&pPos, sizeof(bool), &pMtl->m_Invisible);
	size += Write(&pPos, sizeof(float), &pMtl->m_DrawOrder);
	size += Write(&pPos, sizeof(int), &pMtl->m_BasePass);
	size += Write(&pPos, sizeof(bool), &pMtl->m_Sorted);
	size += Write(&pPos, sizeof(int), &pMtl->m_CutoffFunc);
	size += Write(&pPos, sizeof(bool), &pMtl->m_water);
	size += Write(&pPos, sizeof(bool), &pMtl->m_grassify);
	size += Write(&pPos, sizeof(float), &pMtl->m_grassHeight);
	size += Write(&pPos, sizeof(int), &pMtl->m_grassLayers);
	size += Write(&pPos, sizeof(int), &pMtl->m_NumWibbleSequences);
	
	for(int i = 0; i < pMtl->m_NumWibbleSequences; i++)
	{
		size += Write(&pPos, sizeof(int), &pMtl->m_WibbleSequences[i].m_NumFrames);
		size += Write(&pPos, sizeof(WibbleKeyframe) * pMtl->m_WibbleSequences[i].m_NumFrames, pMtl->m_WibbleSequences[i].m_WibbleFrames);
	}

	// Fortunately NxMaterial pass contains no variably sized fields so we can do one big memcpy
	size += Write(&pPos, sizeof(NxMaterialPass) * vMAX_MATERIAL_PASSES, pMtl->m_Passes);

	size += Write(&pPos, sizeof(bool), &pMtl->m_UseSpecular);
	size += Write(&pPos, sizeof(float), &pMtl->m_SpecularPower);
	size += Write(&pPos, sizeof(float) * 3, pMtl->m_SpecularColor);

	return size;
}

