/*
	LevelEditorDlg.cpp
	Main Level Editor Dialog Window
	aml - 9-26-03
*/

#include "stdafx.h"
#include "LevelEditorDlg.h"
#include "resource.h"
#include "RenderWindow.h"
#include "../InterMaxLink/InterMaxLink.h"	// Exports for InterMaxLink.dll for InterCommunication with 3DSMAX
#include "../InterMax/DataNotifyCodes.h"	// InterMax.gup message notification codes for IPC with 3DSMAX
#include "../Max5/Next/InterLink/InterLinkCodes.h"
#include <stdio.h>
#include <commdlg.h>

extern int nVerts, nFaces;
extern imlVert* verts;
extern imlFace* faces;

/// LevelEditorDlg

LevelEditorDlg::LevelEditorDlg(HINSTANCE hInstance, HWND hwndInterLink) :
	MSDlgWindow(hInstance, MAKEINTRESOURCE(IDD_MAIN), NULL),
	MSWindow(hInstance)
{
	this->hInstance     = hInstance;
	this->hwndInterLink = hwndInterLink;

	bInterLinkActive = false;
	bCloseOnDetach   = false;

	pRenderWindowTL = new RenderWindow(hInstance);
	pRenderWindowTR = new RenderWindow(hInstance);
	pRenderWindowBL = new RenderWindow(hInstance);
	pRenderWindowBR = new RenderWindow(hInstance);

	pRenderWindowTL->Attach(GetDlgItem(MSDlgWindow::hwnd, IDC_VIEWERTL));
	pRenderWindowTR->Attach(GetDlgItem(MSDlgWindow::hwnd, IDC_VIEWERTR));
	pRenderWindowBL->Attach(GetDlgItem(MSDlgWindow::hwnd, IDC_VIEWERBL));
	pRenderWindowBR->Attach(GetDlgItem(MSDlgWindow::hwnd, IDC_VIEWERBR));

	SendMessage(hwndInterLink, WM_IML_ATTACH, 0, (LPARAM)MSWindow::hwnd);
}

LevelEditorDlg::~LevelEditorDlg()
{
	delete pRenderWindowTL;
	delete pRenderWindowTR;
	delete pRenderWindowBL;
	delete pRenderWindowBR;
}

LRESULT LevelEditorDlg::WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_IML_ATTACH_ACK:
		{
			bInterLinkActive = true;
		}
		return TRUE;

	case WM_IML_DETACH_ACK:
		{
			bInterLinkActive = false;

			if (bCloseOnDetach)
			{
				EndDialog(MSDlgWindow::hwnd, 0);
				PostQuitMessage(0);
			}
		}
		return TRUE;

	case WM_IML_MTLEXISTS:
		{
			if (worldDB.GetMaterial((unsigned long)wParam))
				return TRUE;

			return FALSE;
		}
	}

	return DefWindowProc(hwnd, msg, wParam, lParam);
}

// Loads a file piece and should initiate construction of that piece in MAX
bool LevelEditorDlg::LoadTestPiece(HWND hwnd, char* filename)
{
	FILE* fp = fopen(filename, "rb");

	if (!fp)
	{
		char buf[256];
		sprintf(buf, "Couldn't open file '%s'", filename);
		MessageBox(hwnd, buf, "LoadTestPiece", MB_ICONWARNING|MB_OK);
		return false;
	}

	fread(&nVerts, sizeof(int), 1, fp);

	if (verts)
		free(verts);

	verts = (imlVert*)malloc(sizeof(imlVert) * nVerts);
	fread(verts, sizeof(imlVert), nVerts, fp);

	fread(&nFaces, sizeof(int), 1, fp);
	
	if (faces)
		free(faces);

	faces = (imlFace*)malloc(sizeof(imlFace) * nFaces);
	fread(faces, sizeof(imlFace), nFaces, fp);

	fclose(fp);

	PostIML();
	return true;
}

bool LevelEditorDlg::LoadTestPiece(HWND hwnd)
{
	OPENFILENAME ofn;
	char filename[256]="";
	char filetitle[128]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Test Piece Files (*.tpc)\0*.tpc\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=filetitle;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir="";
	ofn.lpstrTitle="Load Scene";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".tpc");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	// Check if aborted
	if (ofn.lpstrFile[0] == 0)
		return false;

	return LoadTestPiece(hwnd, filename);
}

// Saves the currently loaded MAX piece to a file
bool LevelEditorDlg::SaveTestPiece(HWND hwnd, char* filename)
{
	FILE* fp = fopen(filename, "wb");

	if (!fp)
	{
		char buf[256];
		sprintf(buf, "Failed to save '%s'", filename);
		MessageBox(hwnd, buf, "SaveTestPiece", MB_ICONWARNING|MB_OK);
		return false;
	}

	fwrite(&nVerts, sizeof(int), 1, fp);
	fwrite(verts, sizeof(imlVert), nVerts, fp);
	fwrite(&nFaces, sizeof(int), 1, fp);
	fwrite(faces, sizeof(imlFace), nFaces, fp);

	fclose(fp);
	return true;
}

bool LevelEditorDlg::SaveTestPiece(HWND hwnd)
{
	OPENFILENAME ofn;
	char filename[2048]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Test Pieces (*.tpc)\0*.tpc\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=0;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Save Test Piece";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".cdb");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);

	// Exit if cancelled
	if (strlen(filename)==0)
		return false;

	return SaveTestPiece(hwnd, filename);
}

bool LevelEditorDlg::OpenLevel(HWND hwnd, char* filename)
{
	if (!worldDB.LoadScene(filename))
	{
		char buf[256];
		sprintf(buf, "Failed to open '%s'.", filename);
		MessageBox(hwnd, buf, "Open Level", MB_ICONWARNING|MB_OK);
		return false;
	}

	return true;
}

bool LevelEditorDlg::SaveLevel(HWND hwnd, char* filename)
{
	if (!worldDB.SaveScene(filename))
	{
		char buf[256];
		sprintf(buf, "Failed to save '%s'.", filename);
		MessageBox(hwnd, buf, "Save Level", MB_ICONWARNING|MB_OK);
		return false;
	}

	return true;
}

bool LevelEditorDlg::OpenLevel(HWND hwnd)
{
	OPENFILENAME ofn;
	char filename[256]="";
	char filetitle[128]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Scene Files (*.scn)\0*.scn\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=filetitle;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir="";
	ofn.lpstrTitle="Load Scene";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".q");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	// Check if aborted
	if (ofn.lpstrFile[0] == 0)
		return false;

	return OpenLevel(hwnd, ofn.lpstrFile);
}

bool LevelEditorDlg::SaveLevel(HWND hwnd)
{
	OPENFILENAME ofn;
	char filename[256]="";
	char filetitle[128]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Scene Files (*.scn)\0*.scn\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=filetitle;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir="";
	ofn.lpstrTitle="Save Scene";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".q");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);

	// Check if aborted
	if (ofn.lpstrFile[0] == 0)
		return false;

	return SaveLevel(hwnd, ofn.lpstrFile);
}

BOOL LevelEditorDlg::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_CLOSE:
		if (bInterLinkActive)
		{
			bCloseOnDetach = true;
			SendMessage(hwndInterLink, WM_IML_DETACH, 0, 0);
			return TRUE;
		}
		else
		{
			EndDialog(hwnd, 0);
			PostQuitMessage(0);
		}

		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case ID_FILE_LOAD_SCN:
			OpenLevel(hwnd);
			return TRUE;

		case ID_FILE_SAVE_SCN:
			SaveLevel(hwnd);
			return TRUE;

		case ID_FILE_SAVETESTPIECE:
			SaveTestPiece(hwnd);
			return TRUE;

		case ID_FILE_LOADTESTPIECE:
			LoadTestPiece(hwnd);
			return TRUE;

		case ID_FILE_EXIT:
			PostQuitMessage(0);
			return TRUE;
		}
	}

	return FALSE;
}

void LevelEditorDlg::Update()
{
	pRenderWindowTL->Update();
	pRenderWindowTR->Update();
	pRenderWindowBL->Update();
	pRenderWindowBR->Update();
}

void LevelEditorDlg::PostIML()
{
	unsigned char* pMem = NULL;
	while(!pMem){ pMem = (unsigned char*)imlLockSection(); if (!pMem) imlUpdate(); }		// Wait for lock
	unsigned char* pPos = pMem;
	int size = 0;

	*((int*)pPos) = nVerts;
	pPos += sizeof(int);
	size += sizeof(int);

	memcpy(pPos, verts, sizeof(imlVert) * nVerts);
	pPos += sizeof(imlVert) * nVerts;
	size += sizeof(imlVert) * nVerts;

	*((int*)pPos) = nFaces;
	pPos += sizeof(int);
	size += sizeof(int);

	memcpy(pPos, faces, sizeof(imlFace) * nFaces);
	pPos += sizeof(imlFace) * nFaces;
	size += sizeof(imlFace) * nFaces;

	imlUnlockSection(NOTIFYDATA_NEWGEOMBUILD, size);
}
