/*
	ImageViewer.cpp
	This app will display data in the .img format
	Currently just supports img.ps2
	aml - 7-30-03
*/

#include <windows.h>
#include <assert.h>
#include <stdio.h>
#include "Resource.h"

HINSTANCE ghInstance;

HDC hdcBmp = NULL;
HBITMAP   hBitmap = NULL;
unsigned char* pPalette = NULL;
unsigned char* image = NULL;
int pixelDepth;
int imgWidth, imgHeight;
int nPaletteEntries;
int paletteDepth;

enum TextureFormat
{
	v32_BIT,
	v24_BIT,
	v16_BIT,
	v8_BIT,
	v4_BIT,		
	v8_BIT_GRAY,
	v4_BIT_GRAY,
	v2_BIT,
	v1_BIT,
	v2_BIT_GRAY,
	v1_BIT_GRAY,
	vUNKNOWN,
};

enum PSPixelMode
{
	vPSMCT32 = 0,
	vPSMCT24 = 1,
	vPSMCT16 = 2,
	vPSMT8   = 19,
	vPSMT4   = 20,
};

#pragma pack(push,1)		// Force single-byte alignment
struct TGAHeader
{
	unsigned char  IDLength;
	unsigned char  ColorMapType;
	unsigned char  ImageType;
	unsigned short CMapStart;
	unsigned short CMapLength;
	unsigned char  CMapDepth;
	unsigned short XOffset;
	unsigned short YOffset;
	unsigned short Width;
	unsigned short Height;
	unsigned char  PixelDepth;
	unsigned char  ImageDescriptor;
};

struct TGAFooter
{
	unsigned long ExtensionOffset;
	unsigned long DeveloperOffset;
	char          Signature[18];
};

struct color24
{
	unsigned char b, g, r;
};

#pragma pack(pop)

// Assumming 0rrrrrgggggbbbbb
#define RED_MASK   0x7C00
#define GREEN_MASK 0x3E0
#define BLUE_MASK  0x1F

TextureFormat GetTextureFormat(int depth)
{
	switch(depth)
	{
	case 1:
		return v1_BIT;

	case 2:
		return v2_BIT;

	case 4:
		return v4_BIT;

	case 8:
		return v8_BIT;

	case 16:
		return v16_BIT;

	case 24:
		return v24_BIT;

	case 32:
		return v32_BIT;

	default:
		return vUNKNOWN;
	}
}

// Returns the number of bytes per entry for a given bit depth
float GetBytesPerDepth(TextureFormat depth)
{
	switch(depth)
	{
	case v2_BIT:
		return 0.25f;

	case v4_BIT:
		return 0.5f;

	case v8_BIT:
		return 1;

	case v16_BIT:
		return 2;

	case v24_BIT:
		return 3;

	case v32_BIT:
		return 4;

	default:
		return 0;		// Unsupported depth
	}
}

void DrawImage(HDC hdc)
{
	if (!pPalette || !image)
		return;

	if (hdcBmp)
		DeleteDC(hdcBmp);

	//hdcBmp = CreateDC("DISPLAY", NULL, NULL, NULL);
	hdcBmp = CreateCompatibleDC(hdc);
	assert(hdcBmp);

	BITMAPINFO* bi = (BITMAPINFO*)malloc(sizeof(BITMAPINFOHEADER) + sizeof(RGBQUAD) * nPaletteEntries);

	bi->bmiHeader.biSize          = sizeof(BITMAPINFOHEADER);
	bi->bmiHeader.biWidth         = imgWidth;
	bi->bmiHeader.biHeight        = imgHeight;
	bi->bmiHeader.biPlanes        = 1;
	bi->bmiHeader.biBitCount      = pixelDepth;
	bi->bmiHeader.biCompression   = BI_RGB;
	bi->bmiHeader.biSizeImage     = (int)(GetBytesPerDepth(GetTextureFormat(pixelDepth)) * (float)imgWidth * (float)imgHeight);
	bi->bmiHeader.biXPelsPerMeter = 0;
	bi->bmiHeader.biYPelsPerMeter = 0;
	bi->bmiHeader.biClrUsed       = 0;
	bi->bmiHeader.biClrImportant  = 0;

	switch(paletteDepth)
	{
	case 16:
		{
			short* color = (short*)pPalette;

			for(int i = 0; i < nPaletteEntries; i++)
			{
				// Assuming 555
				bi->bmiColors[i].rgbRed      = ((*color) & RED_MASK)   >> 10;
				bi->bmiColors[i].rgbGreen    = ((*color) & GREEN_MASK) >> 5;
				bi->bmiColors[i].rgbBlue     = ((*color) & BLUE_MASK);
				bi->bmiColors[i].rgbReserved = 0;
				color++;
			}
		}
		break;

	case 24:
		{
			unsigned char* color = pPalette;

			for(int i = 0; i < nPaletteEntries; i++)
			{
				bi->bmiColors[i].rgbRed      = *color;
				bi->bmiColors[i].rgbGreen    = *(color + 1);
				bi->bmiColors[i].rgbBlue     = *(color + 2);
				bi->bmiColors[i].rgbReserved = 0;

				color += 3;
			}
		}
		break;

	case 32:
		{
			unsigned char* color = pPalette;

			for(int i = 0; i < nPaletteEntries; i++)
			{
				bi->bmiColors[i].rgbRed      = *color;
				bi->bmiColors[i].rgbGreen    = *(color + 1);
				bi->bmiColors[i].rgbBlue     = *(color + 2);
				bi->bmiColors[i].rgbReserved = 0;
				
				// Alpha unused
				color += 4;
			}
		}
		break;
	}

	if (hBitmap)
		DeleteObject(hBitmap);
	
	//hBitmap = CreateDIBitmap(hdc, (BITMAPINFOHEADER*)bi, CBM_INIT, image, bi,  DIB_RGB_COLORS);
	void* pImageData;
	hBitmap = CreateDIBSection(hdcBmp, bi, DIB_RGB_COLORS, (void**)&pImageData, NULL, 0);
	SelectObject(hdcBmp, hBitmap);
	memcpy(pImageData, image, bi->bmiHeader.biSizeImage);
}

TextureFormat ConvertPSDepth(PSPixelMode mode)
{
	switch(mode)
	{
	case vPSMCT32:
		return v32_BIT;

	case vPSMCT24:
		return v24_BIT;

	case vPSMCT16:
		return v16_BIT;

	case vPSMT8:
		return v8_BIT;

	case vPSMT4:
		return v4_BIT;

	default:
		return (TextureFormat)0;
	}
}

int GetDecimalDepth(TextureFormat format)
{
	switch(format)
	{
	case v32_BIT:
		return 32;

	case v24_BIT:
		return 24;
		
	case v16_BIT:
		return 16;

	case v8_BIT:
		return 8;

	case v4_BIT:
		return 4;

	case v8_BIT_GRAY:
		return 8;

	case v4_BIT_GRAY:
		return 4;

	case v2_BIT:
		return 2;

	case v1_BIT:
		return 1;

	case v2_BIT_GRAY:
		return 2;

	case v1_BIT_GRAY:
		return 1;

	default:
		return 0;
	}
}

int GetNumPaletteEntries(TextureFormat bppMode)
{
	// Only 8 bit and 4 bit modes have palettes
	switch(bppMode)
	{
	case v8_BIT:
		return 1 << 8;

	case v4_BIT:
		return 1 << 4;

	default:
		return 0;
	}
}

void LoadPS2File(HWND hwndOwner)
{
	OPENFILENAME ofn;

	char filename[1024]  = "";

	ofn.lStructSize       = sizeof(OPENFILENAME);
	ofn.hwndOwner         = hwndOwner;
	ofn.hInstance         = ghInstance;
	ofn.lpstrFilter       = "PS2 Image Files (*.img.ps2)\0*.img.ps2\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter    = 0;
	ofn.nFilterIndex      = 0;
	ofn.lpstrFile         = filename;
	ofn.nMaxFile          = 1024;
	ofn.lpstrFileTitle    = NULL;
	ofn.nMaxFileTitle     = 0;
	ofn.lpstrInitialDir   = NULL;
	ofn.lpstrTitle        = "Select .img.ps2 file to load";
	ofn.Flags             = 0;
	ofn.nFileOffset       = 0;
	ofn.nFileExtension    = 0;
	ofn.lpstrDefExt       = ".img.ps2";
	ofn.lCustData         = 0;
	ofn.lpfnHook          = NULL;
	ofn.lpTemplateName    = NULL;

	GetOpenFileName(&ofn);

	if (*filename == 0)
		return;

	// Load in the file
	FILE* fp = fopen(filename, "rb");

	if (!fp)
	{
		char buf[256];
		sprintf(buf, "Failed to open '%s'.", filename);
		MessageBox(hwndOwner, buf, "Failed to open image", MB_ICONSTOP|MB_OK);
		return;
	}

/*
Offset		Size (Bytes)	Field
---------------------------------------------------------------
0x00			4			Version (currently 1)
0x04			4			Basename of file in CRC
format
0x08			4			PS2 TW (log2 Width)
0x0c			4			PS2 TH (log2 Height)
0x10			4			PS2 PSM (Bit depth)
0x14			4			PS2 CPSM (CLUT bit
depth)
0x18			4			PS2 MXL (Number of
mipmaps)
0x1c			2			Original image width
0x1e			2			Original image height
0x20			Upto 1K		CLUT data
0x20+CLUT_size	W*H*depth		Texture mipmaps
*/
	int version;
	unsigned long crc;
	int log2Width, log2Height;
	TextureFormat bitDepth;
	TextureFormat clutDepth;
	int nMips;
	short width, height;

	fread(&version, sizeof(int), 1, fp);
	fread(&crc, sizeof(unsigned long), 1, fp);
	fread(&log2Width, sizeof(int), 1, fp);
	fread(&log2Height, sizeof(int), 1, fp);
	fread(&bitDepth, sizeof(TextureFormat), 1, fp);
	fread(&clutDepth, sizeof(TextureFormat), 1, fp);
	fread(&nMips, sizeof(int), 1, fp);
	fread(&width, sizeof(short), 1, fp);
	fread(&height, sizeof(short), 1, fp);

	bitDepth = ConvertPSDepth((PSPixelMode)bitDepth);

	pixelDepth   = GetDecimalDepth(bitDepth);
	paletteDepth = GetDecimalDepth(clutDepth);
	imgWidth     = width;
	imgHeight    = height;

	if (pixelDepth == 2)
	{
		MessageBox(hwndOwner, "2-bit images cannot currently be displayed in this viewer.", "Image can't be displayed", MB_ICONWARNING|MB_OK);
		fclose(fp);
		return;
	}

	// Check for gray scale images
	if (bitDepth  == v8_BIT_GRAY ||
		bitDepth  == v4_BIT_GRAY ||
		bitDepth  == v2_BIT_GRAY ||
		bitDepth  == v1_BIT_GRAY ||
		clutDepth == v8_BIT_GRAY ||
		clutDepth == v4_BIT_GRAY ||
		clutDepth == v2_BIT_GRAY ||
		clutDepth == v1_BIT_GRAY)
	{
		MessageBox(hwndOwner, "Gray scale images are not currently supported by this viewer.", "Grayscale not supported", MB_ICONWARNING|MB_OK);
		fclose(fp);
		return;
	}

	// Load in the CLUT
	int nCLUTEntries = GetNumPaletteEntries(bitDepth);
	nPaletteEntries = nCLUTEntries;

	if (pPalette)
	{
		free(pPalette);
		pPalette = NULL;
	}

	if (nCLUTEntries > 0)
	{
		// Ensure that the palette is in a valid format
		if (clutDepth != v16_BIT &&
			clutDepth != v24_BIT &&
			clutDepth != v32_BIT)
		{
			MessageBox(hwndOwner, "Palettized image uses a non-supported palette depth.  Only 16/24/32 bitdepths are supported.", "Invalid Palette Depth", MB_ICONSTOP|MB_OK);
			fclose(fp);
			return;
		}

		int bytesPerEntry = (int)GetBytesPerDepth(clutDepth);
		pPalette = (unsigned char*)malloc(bytesPerEntry * nCLUTEntries);
		
		fread(pPalette, bytesPerEntry, nCLUTEntries, fp);		
	}

	// Read in the image (only the first mip gets read)
	if (image)
	{
		free(image);
		image = NULL;
	}

	image = (unsigned char*)malloc((int)(GetBytesPerDepth(bitDepth) * (float)width * (float)height));
	fread(image, (int)(GetBytesPerDepth(bitDepth) * (float)width * (float)height), 1, fp);

	fclose(fp);

	HDC hdcMain = GetDC(hwndOwner);
	DrawImage(hdcMain);
	ReleaseDC(hwndOwner, hdcMain);

	InvalidateRect(hwndOwner, NULL, TRUE);
}

void SaveTGA(HWND hwndOwner)
{
	OPENFILENAME ofn;

	char filename[1024]  = "";

	ofn.lStructSize       = sizeof(OPENFILENAME);
	ofn.hwndOwner         = hwndOwner;
	ofn.hInstance         = ghInstance;
	ofn.lpstrFilter       = "Targa Files (*.tga)\0*.tga\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter    = 0;
	ofn.nFilterIndex      = 0;
	ofn.lpstrFile         = filename;
	ofn.nMaxFile          = 1024;
	ofn.lpstrFileTitle    = NULL;
	ofn.nMaxFileTitle     = 0;
	ofn.lpstrInitialDir   = NULL;
	ofn.lpstrTitle        = "Enter name of Targa image to save as";
	ofn.Flags             = 0;
	ofn.nFileOffset       = 0;
	ofn.nFileExtension    = 0;
	ofn.lpstrDefExt       = ".tga";
	ofn.lCustData         = 0;
	ofn.lpfnHook          = NULL;
	ofn.lpTemplateName    = NULL;

	GetSaveFileName(&ofn);

	if (*filename == 0)
		return;

	FILE* fp = fopen(filename, "wb");

	TGAHeader header;

	header.IDLength        = 0;
	header.ColorMapType    = 0;	// No palette
	header.ImageType       = 2;	// True color no colormap no RLE
	header.CMapStart       = 0;
	header.CMapLength      = 0;
	header.CMapDepth       = 0;
	header.XOffset         = 0;
	header.YOffset         = 0;
	header.Width           = imgWidth;
	header.Height          = imgHeight;
	header.PixelDepth      = 24;
	header.ImageDescriptor = 0;

	fwrite(&header, sizeof(TGAHeader), 1, fp);
	
	color24* imageOut = (color24*)malloc(sizeof(color24) * imgHeight * imgWidth);
	color24* imageOutPos = imageOut;

	// Scan through the image converting to 24-bit truecolor
	HDC hdcMain = GetDC(hwndOwner);

	for(int y = imgHeight - 1; y >= 0; y--)
	{
		for(int x = 0; x < imgWidth; x++)
		{
			COLORREF color = GetPixel(hdcMain,x,y);

			imageOutPos->r = GetRValue(color);
			imageOutPos->g = GetGValue(color);
			imageOutPos->b = GetBValue(color);
			imageOutPos++;
		}
	}

	ReleaseDC(hwndOwner, hdcMain);

	fwrite(imageOut, sizeof(color24), imgHeight * imgWidth, fp);
	fclose(fp);
}

LRESULT CALLBACK WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_CREATE:

		return TRUE;

	case WM_DESTROY:
		PostQuitMessage(0);
		return TRUE;

	case WM_PAINT:
		{
			PAINTSTRUCT ps;
			HDC         hdc;
			hdc = BeginPaint(hwnd, &ps);
			if (hBitmap)
			{
				BitBlt(hdc,
					   0,
					   0,
					   imgWidth,
					   imgHeight,
					   hdcBmp,
					   0,
					   0,
					   SRCCOPY);
			}

			//DrawImage(hdc);
			EndPaint(hwnd, &ps);

			return 0;
		}

	case WM_COMMAND:
		{
			switch(LOWORD(wParam))
			{
			case IDM_LOADPS2:
				LoadPS2File(hwnd);
				break;

			case IDM_SAVETGA:
				SaveTGA(hwnd);
				break;

			case IDM_EXIT:
				PostQuitMessage(0);
				return TRUE;
			}

		break;
		}
	}
	
	return DefWindowProc(hwnd, msg, wParam, lParam);
}

int WINAPI WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, char* cmdline, int nShow)
{
	MSG      msg;
	WNDCLASS wndclass;
	HWND     hwnd;

	ghInstance = hInstance;

	wndclass.style         = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc   = WndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = (HCURSOR)LoadCursor(hInstance, IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	wndclass.lpszMenuName  = MAKEINTRESOURCE(IDR_MAINMENU);
	wndclass.lpszClassName = "ImageViewerApp";

	if (!RegisterClass(&wndclass))
		return 0;

	hwnd = CreateWindow("ImageViewerApp",
		                "NeverSoft ImageViewer",
						WS_OVERLAPPEDWINDOW,
						0,
						0,
						640,
						480,
						NULL,
						NULL,
						hInstance,
						NULL);

	if (!hwnd)
		return 0;
	
	ShowWindow(hwnd, SW_SHOW);

	while(GetMessage(&msg, NULL, 0, 0))
	{
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	return 0;
}
