// KSH 16 Jan 2002

// CheckChanges will determine whether any of the files in the search paths
// listed in the specified qb file have changed.
// A change means that either one of the files' dates has changed,
// or some files have been added or removed.
// If changes are detected, CheckChanges will return an error code.
// Hence a batch file can call CheckChanges, then test whether ERRORCODE is zero.
// If ERRORCODE is zero, no changes have occurred.

// Example: CheckChanges levelfiles.qb
// (The qb file is assumed to be in the %%SKATE4_PATH%%\\bin\\win32\\ directory)

#include "stdafx.h"
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <io.h>
#include <conio.h>
#include <process.h>
#include <time.h>
#include <fstream.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "c:\skate4\code\gel\scripting\tokens.h"

using namespace Script;

// This is so that skiptoken.cpp can be #included.
#define Dbg_MsgAssert( _c, _params )	Assert( _c, _params )

void cleanup_log_file_buffers();

void Assert(int condition, char *p_message, ...)
{
	if (condition) return;

	char p_buf[1000];
	va_list args;
	
	va_start(args, p_message);
	vsprintf( p_buf, p_message, args);
	va_end( args );

	printf("CheckChanges assertion failed: %s",p_buf);
	printf("\n");

	cleanup_log_file_buffers();
	exit(1);
}


typedef unsigned char uint8;
typedef unsigned int uint32;

#include <c:/skate4/code/gel/scripting/skiptoken.cpp>

// This is so that when changes are detected, it will only print a message for the
// first one, otherwise time could be wasted with printf'ing tons of messages.
// (CheckChanges needs to be quick)
bool gPrintedMessageAlready=false;

unsigned long CRC_TABLE[256] = // CRC polynomial 0xedb88320
{
      0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
      0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
      0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
      0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
      0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
      0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
      0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
      0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
      0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
      0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
      0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
      0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
      0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
      0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
      0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
      0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
      0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
      0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
      0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
      0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
      0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
      0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
      0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
      0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
      0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
      0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
      0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
      0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
      0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
      0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
      0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
      0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
      0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
      0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
      0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
      0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
      0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
      0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
      0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
      0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
      0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
      0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
      0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
      0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
      0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
      0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
      0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
      0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
      0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
      0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
      0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
      0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
      0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
      0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
      0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
      0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
      0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
      0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
      0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
      0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
      0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
      0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
      0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
      0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

uint32 GenerateChecksum(const char *pString)
{
	// Initializing the CRC to all one bits avoids failure of detection
	// should entire data stream get cyclically bit-shifted by one position.
	// The calculation of the probability of this happening is left as
	// an exercise for the reader.
	unsigned long rc = 0xffffffff;
	int Len=strlen(pString);
	for (int i=0; i<Len; i++)
	{
		char ch=pString[i];
        // Convert to lower case.
        if (ch>='A' && ch<='Z') ch='a'+ch-'A';
		rc = CRC_TABLE[(rc^ch) & 0xff] ^ ((rc>>8) & 0x00ffffff);
	}

	return rc;
}

uint32 *gpOldLogFile=NULL;
uint32 OldLogFileNumEntries=0;
#define LOG_FILE_MAX_ENTRIES 10000
uint32 *gpNewLogFile=NULL;
uint32 NewLogFileNumEntries=0;

void import_log_file(const char *p_name)
{
	if (gpOldLogFile)
	{
		free(gpOldLogFile);
		gpOldLogFile=NULL;
	}
	OldLogFileNumEntries=0;

	// Open it and load it into memory.
	ifstream log_file(p_name, ios::in | ios::binary);
	log_file.seekg(0, ios::end);
	int log_file_size = log_file.tellg();
	if (!log_file_size )
	{
		log_file.close();
		return;
	}
	log_file.seekg(0, ios::beg);
	gpOldLogFile = (uint32*)malloc(log_file_size);
	Assert(gpOldLogFile!=NULL,"Could not allocate memory for log file %s",p_name);
	log_file.read((char*)gpOldLogFile, log_file_size);
	log_file.close();


	Assert((log_file_size&7)==0,"Log file %s has size %d, not a multiple of 8 ?",p_name,log_file_size);
	OldLogFileNumEntries=log_file_size/8;
}

void create_new_log_file_buffer()
{
	if (gpNewLogFile)
	{
		free(gpNewLogFile);
		gpNewLogFile=NULL;
	}
	gpNewLogFile=(uint32*)malloc(LOG_FILE_MAX_ENTRIES*8);
	NewLogFileNumEntries=0;
}

void cleanup_log_file_buffers()
{
	if (gpNewLogFile)
	{
		free(gpNewLogFile);
		gpNewLogFile=NULL;
	}
	NewLogFileNumEntries=0;
	if (gpOldLogFile)
	{
		free(gpOldLogFile);
		gpOldLogFile=NULL;
	}
	OldLogFileNumEntries=0;
}

void export_log_file(const char *p_name)
{
	fstream log_file(p_name, ios::out | ios::binary);
	if (gpNewLogFile)
	{
		log_file.write((char*)gpNewLogFile,NewLogFileNumEntries*8);
	}
	log_file.flush();
	log_file.close();
}

void add_file_to_new_log_file(const char *p_name)
{
	Assert(p_name!=NULL,"NULL p_name");

	// Get the date info for the file.
	struct stat stat_buf;
	int return_code=stat(p_name,&stat_buf);
	Assert(return_code!=-1,"File %s does not exist ??",p_name);
	Assert(return_code==0,"Unexpected return_code when running stat on %s",p_name);

	Assert(gpNewLogFile!=NULL,"NULL gpNewLogFile");
	Assert(NewLogFileNumEntries<LOG_FILE_MAX_ENTRIES,"Need to increase LOG_FILE_MAX_ENTRIES in CheckChanges");
	
	uint32 *p_new=gpNewLogFile+2*NewLogFileNumEntries;
	*p_new++=GenerateChecksum(p_name);
	*p_new=stat_buf.st_mtime;
	++NewLogFileNumEntries;
}

// p_path is the path, not ending in a '\'
// p_wildcard is for example "*.qb"
void add_files_to_new_log_file( char *p_path, char *p_wildcard )
{
	struct _finddata_t file_info;
	long handle;
	char p_search_string[_MAX_PATH];

	Assert(p_path!=NULL,"NULL p_path ");
	Assert(p_wildcard!=NULL,"NULL p_wildcard ");
	Assert(*p_wildcard=='*',"Expected *p_wildcard to be '*'");

	sprintf( p_search_string, "%s\\*.*", p_path );
	handle = _findfirst( p_search_string, &file_info );
	if( handle >= 0 )
	{
		char p_ext[_MAX_EXT];

		_splitpath( file_info.name, NULL, NULL, NULL, p_ext );
		if( file_info.attrib & _A_SUBDIR )
		{
			if( ( stricmp( file_info.name, "." ) != 0 ) &&
				( stricmp( file_info.name, ".." ) != 0 ))
			{
				char p_new_path[_MAX_PATH];

				sprintf( p_new_path, "%s\\%s", p_path, file_info.name );
				add_files_to_new_log_file( p_new_path, p_wildcard );
			} 
		}
		else if( stricmp( p_ext, p_wildcard+1 ) == 0 )
		{
			char p_full_name[1000];
			sprintf(p_full_name,"%s\\%s",p_path,file_info.name);
			add_file_to_new_log_file(p_full_name);
		}		

		while( _findnext( handle, &file_info ) == 0 )
		{
			_splitpath( file_info.name, NULL, NULL, NULL, p_ext );
			if( file_info.attrib & _A_SUBDIR )
			{
				if( ( stricmp( file_info.name, "." ) != 0 ) &&
					( stricmp( file_info.name, ".." ) != 0 ))
				{
					char p_new_path[_MAX_PATH];

					sprintf( p_new_path, "%s\\%s", p_path, file_info.name );
					add_files_to_new_log_file( p_new_path, p_wildcard );
				}
			}
			else if( stricmp( p_ext, p_wildcard+1 ) == 0 )
			{
				char p_full_name[1000];
				sprintf(p_full_name,"%s\\%s",p_path,file_info.name);
				add_file_to_new_log_file(p_full_name);
			}		
		}
	}
}

int main(int argc, char* argv[])
{
	if (argc==1)
	{
		printf("Compiled on: %s %s\n\n",__DATE__,__TIME__);
		printf("Example: CheckChanges levelfiles.qb\n");
		printf("(The qb file is assumed to be in the %%SKATE4_PATH%%\\bin\\win32\\ directory)\n\n");
		printf("CheckChanges will determine whether any of the files in the search paths\n");
		printf("listed in the specified qb file have changed.\n");
		printf("A change means that either one of the files' dates has changed,\n");
		printf("or some files have been added or removed.\n");
		printf("If changes are detected, CheckChanges will return an error code.\n");
		printf("Hence a batch file can call CheckChanges, then test whether ERRORCODE is zero.\n");
		printf("If ERRORCODE is zero, no changes have occurred.\n\n");
		exit(1);
	}

	// Open the file specified on the command line, assuming that it is in skate4\bin\win32
	char p_datafiles_qb_name[1000];

	char *p_path=getenv("SKATE4_PATH");
	Assert(p_path!=NULL,"CheckChanges requires that the environment variable SKATE4_PATH is set.");

	strcpy(p_datafiles_qb_name,p_path);
	if (p_datafiles_qb_name[strlen(p_datafiles_qb_name)-1]!='\\')
	{
		strcat(p_datafiles_qb_name,"\\");
	}
	strcat(p_datafiles_qb_name,"bin\\win32\\");
	strcat(p_datafiles_qb_name,argv[1]);

	// Open it and load it into memory.
	ifstream datafiles_qb(p_datafiles_qb_name, ios::in | ios::binary);
	datafiles_qb.seekg(0, ios::end);
	int datafiles_qb_size = datafiles_qb.tellg();
	if (!datafiles_qb_size )
	{
		datafiles_qb.close();
		Assert(0,"Could not open %s",p_datafiles_qb_name);
		return 0;
	}
	datafiles_qb.seekg(0, ios::beg);
	uint8 *p_datafiles_qb = (uint8*)malloc(datafiles_qb_size);
	Assert(p_datafiles_qb!=NULL,"Could not allocate memory for qb file %s",p_datafiles_qb_name);
	datafiles_qb.read(p_datafiles_qb, datafiles_qb_size);
	datafiles_qb.close();

	// Import the log file, which has the same name as the qb just with .log on the end.
	char p_log_file_name[1000];
	strcpy(p_log_file_name,p_datafiles_qb_name);
	strcat(p_log_file_name,".log");
	import_log_file(p_log_file_name);

	create_new_log_file_buffer();

	// Parse the qb, looking for strings.
	const uint8 *p_token=p_datafiles_qb;
	while (*p_token!=ESCRIPTTOKEN_ENDOFFILE)
	{
		if (*p_token==ESCRIPTTOKEN_STRING)
		{
			char *p_name=(char *)(p_token+5);
			
			char p_path[1000];
			strcpy(p_path,p_name);
			char *p_end=p_path+strlen(p_path)-1;
			while (p_end>p_path && *p_end!='\\')
			{
				--p_end;
			}
			if (p_end>p_path)
			{
				*p_end=0;
				add_files_to_new_log_file(p_path,p_end+1);
			}
		}                                                       
                            
		p_token=SkipToken(p_token);
	}

	free(p_datafiles_qb);
	p_datafiles_qb=NULL;

	export_log_file(p_log_file_name);

	bool match=true;
	if (!gpOldLogFile)
	{
		match=false;
	}
	if (!gpNewLogFile)
	{
		match=false;
	}
	if (gpOldLogFile && gpNewLogFile)
	{
		if (NewLogFileNumEntries > OldLogFileNumEntries)
		{
			match=false;
			if (NewLogFileNumEntries==OldLogFileNumEntries+1)
			{
				printf("CheckChanges: New file added ...\n");
			}
			else
			{
				printf("CheckChanges: New files added ...\n");
			}
		}
		else if (NewLogFileNumEntries < OldLogFileNumEntries)
		{
			match=false;
			if (NewLogFileNumEntries==OldLogFileNumEntries-1)
			{
				printf("CheckChanges: File removed ...\n");
			}
			else
			{
				printf("CheckChanges: Files removed ...\n");
			}
		}
		else
		{
			// Matching number of files.
			uint32 *p_old=gpOldLogFile;
			uint32 *p_new=gpNewLogFile;
			for (uint32 i=0; i<NewLogFileNumEntries; ++i)
			{
				if (*p_old++!=*p_new++)
				{
					match=false;
					printf("CheckChanges: File(s) deleted/added/renamed ...\n");
					break;
				}
				if (*p_old++!=*p_new++)
				{
					match=false;
					printf("CheckChanges: File date changed ...\n");
					break;
				}
			}
		}
	}

	cleanup_log_file_buffers();

	if (match)
	{
		return 0;
	}
	else
	{
		return -1;
	}
}



