//****************************************************************************
//* MODULE:         Tools/AnimConv
//* FILENAME:       main.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  1/16/2002
//****************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#define __USE_OLD_STREAMS__
#include <fstream.h>

#include "AnimConv.h"
#include "Utility.h"
#include "VirtualFile.h"

#define	vMAJOR_REV		1
#define vMINOR_REV	 	0
#define vEXE_NAME		"AnimConv"

static bool s_force_overwrite = false;
static bool s_generate_uncompressed = false;

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_usage()
{
#ifdef _DEBUG
	printf( "\n%s v%d.%d (Debug) Built %s %s\n", vEXE_NAME, vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#else
	printf( "\n%s v%d.%d (Release) Built %s %s\n", vEXE_NAME, vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#endif
	printf( "Neversoft Entertainment, 2002\n" );
	printf( "\nUsage: AnimConv -b<batchfile> -f<filename> -c<compressname> -p[p | g | x] -o\n" );
	printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
	printf( "Ex: AnimConv -f%%PROJ_ROOT%%/data/anims/test/test.ska -pp\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void run_utility( AnimConverter* pAnimConverter, const char* pSrcName, CompressTable* pQCompressTable, CompressTable* pTCompressTable, int platform )
{
	pAnimConverter->Reset();

	if ( !pAnimConverter->Load( pSrcName ) )
	{
		printf( "Error!  %s is in bad format\n", pSrcName );
		exit(1);
	}

	char pDstName[_MAX_PATH];
	sprintf( pDstName, "%s.%s", pSrcName, Utils::GetPlatformExt(platform) );

	bool reverseByteOrder = false;
	switch( platform )
	{
		case Utils::vPLATFORM_PS2:
		case Utils::vPLATFORM_XBOX:
			reverseByteOrder = false;
			break;
		case Utils::vPLATFORM_NGC:
			reverseByteOrder = true;
			break;
	}

	// from now on, all of the platforms generate the compressed version
	pAnimConverter->SaveCompressed(pDstName, pQCompressTable, pTCompressTable, reverseByteOrder);

	if ( s_generate_uncompressed )
	{
		// the uncompressedStream gets exported so that
		// we can run the animation compressor on it
		strcpy( pDstName, pSrcName );
		pDstName[strlen(pDstName)-3] = 's';
		pDstName[strlen(pDstName)-2] = 'k';
		pDstName[strlen(pDstName)-1] = 'u';
		strcat( pDstName, "." );
		strcat( pDstName, Utils::GetPlatformExt(platform) );
		pAnimConverter->Save(pDstName, reverseByteOrder);
	}

	// success
	printf( "." );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void main( int argc, char* argv[] )
{
	char* pSrcName = NULL;
	char* pBatchFileName = NULL;
	char* pCompressTableName = NULL;
	int platform = Utils::vPLATFORM_NONE;

	CompressTable theQCompressTable;
	CompressTable theTCompressTable;

	char* pArg = NULL;

	if ( pArg = Utils::TestArg( argc, argv, '?' ) )
	{
		print_usage();
		return;
	}

	if ( pArg = Utils::TestArg( argc, argv, 'P' ) )
	{
		switch ( *pArg )
		{
			case 'P':
			case 'p':
				platform = Utils::vPLATFORM_PS2;
				break;
			case 'G':
			case 'g':
				platform = Utils::vPLATFORM_NGC;
				break;
			case 'X':
			case 'x':
				platform = Utils::vPLATFORM_XBOX;
				break;
		}
	}
	 
	// check for invalid platform
	if ( platform == Utils::vPLATFORM_NONE )
	{
		print_usage();
		return;
	}
	
	char compressTableRoot[_MAX_PATH];
	char* proj_path = getenv( "PROJ_ROOT" );
	if( proj_path == NULL )
	{		
		printf( "You must first define your PROJ_ROOT environment variable\n" );
	}

	// force it to generate uncompressed files
	if ( pArg = Utils::TestArg( argc, argv, 'U' ) )
	{
		printf( "Generating uncompressed files\n" );
		s_generate_uncompressed = true;
	}

	if ( pArg = Utils::TestArg( argc, argv, 'C' ) )
	{
		pCompressTableName = pArg;
	}
	else
	{
		sprintf( compressTableRoot, "%s\\data\\anims\\standardkey", proj_path );
		pCompressTableName = compressTableRoot;
	}

	// test whether the compress table exists
	if ( pCompressTableName )
	{
		char tableName[_MAX_PATH];

		sprintf( tableName, "%sQ.bin", pCompressTableName );
		if ( !IoUtils::file_exists( tableName ) )
		{
			printf( "Couldn't find compress table %s.  Generating uncompressed files.\n", tableName );
			s_generate_uncompressed = true;
		}
	
		sprintf( tableName, "%sT.bin", pCompressTableName );
		if ( !IoUtils::file_exists( tableName ) )
		{
			printf( "Couldn't find compress table %s.  Generating uncompressed files.\n", tableName );
			s_generate_uncompressed = true;
		}

		if ( s_generate_uncompressed )
		{
			pCompressTableName = NULL;
		}
	}

	// now initialize the compress tables
	if ( pCompressTableName )
	{
		char qTableName[_MAX_PATH];
		sprintf( qTableName, "%sQ.bin", pCompressTableName );
		if ( !theQCompressTable.Init( qTableName ) )
		{
			exit(1);
		}

		char tTableName[_MAX_PATH];
		sprintf( tTableName, "%sT.bin", pCompressTableName );
		if ( !theTCompressTable.Init( tTableName ) )
		{
			exit(1);
		}
	}

	if ( pArg = Utils::TestArg( argc, argv, 'O' ) )
	{
		s_force_overwrite = true;
	}

	// create the converter
	AnimConverter theAnimConverter;
	
	// check for debug mode
	bool debugOutput = false;
	if ( pArg = Utils::TestArg( argc, argv, 'D' ) )
	{
		debugOutput = true;
	}
	theAnimConverter.SetDebugMode( debugOutput );

	// check for batch mode
	if ( pArg = Utils::TestArg( argc, argv, 'B' ) )
	{
		pBatchFileName = pArg;

		fstream batchStream( pBatchFileName, ios::in | ios::nocreate );
		if ( !batchStream.is_open() )
		{
			printf( "Error!  Could not open %s\n", pBatchFileName );
			batchStream.close();
			exit(1);
		}

		while ( !batchStream.eof() )
		{
			char lineBuf[256];
			batchStream.getline( &lineBuf[0], 256 );
			if ( batchStream.eof() )
			{
				break;
			}

			run_utility( &theAnimConverter, lineBuf, &theQCompressTable, &theTCompressTable, platform );
		}

		batchStream.close();
	}
	else if ( pArg = Utils::TestArg( argc, argv, 'F' ) )
	{
		pSrcName = pArg;
		run_utility( &theAnimConverter, pSrcName, &theQCompressTable, &theTCompressTable, platform );
	}
	else
	{
		print_usage();
		return;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/