/*
	AnimComp.cpp
	1-16-01
*/

#include <math.h>
#include <stdio.h>
#include <search.h>
#include <string.h>
#include "AnimComp.h"
#include "..\animconv\ps2AnimConv.h"
#include <core/defines.h>
#include <core/math.h>

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

struct SUniqueItem
{
	short x, y, z, count;
};
const int MAX_UNIQUE_Q_TABLE_SIZE = 400000;
static SUniqueItem uniqueQTable[MAX_UNIQUE_Q_TABLE_SIZE];
	
const int MAX_UNIQUE_T_TABLE_SIZE = 24000;	
static SUniqueItem uniqueTTable[MAX_UNIQUE_T_TABLE_SIZE];
int uniqueQCount = 0;
int uniqueTCount = 0;

const int NUM_BUCKETS = 256;

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int compareUniqueItem( const void* s1, const void* s2)
{
	if ( ((SUniqueItem*)s1)->count > ((SUniqueItem*)s2)->count )
	{
		return -1;
	}
	else if ( ((SUniqueItem*)s1)->count < ((SUniqueItem*)s2)->count )
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_unique_table( SUniqueItem* pTable, int& tableCount )
{
	// first sort it
	qsort( pTable, tableCount, sizeof(SUniqueItem), compareUniqueItem);

	int heavyCount = 0;
	int totalTableCount = 0;
	const int HEAVY_VALUE = 512;

	for ( int i = 0; i < tableCount; i++ )
	{
		if ( i < HEAVY_VALUE )
		{
			printf( "count[%d] = %d\n", i, pTable[i].count );
			heavyCount += pTable[i].count;
		}
		totalTableCount += pTable[i].count;
	}

	printf( "%d items of %d fit into the first %d 48-bit buckets\n", heavyCount, totalTableCount, HEAVY_VALUE );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool in_unique_table( SUniqueItem* pTable,
						 int& tableCount,
						 short qx,
				   		 short qy, 
						 short qz )
{
	for ( int i = 0; i < NUM_BUCKETS; i++ )
	{
		if ( pTable[i].x == qx && pTable[i].y == qy && pTable[i].z == qz )
		{
			return true;
		}
	}

	return false;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void check_unique_table( SUniqueItem* pTable,
						 int& tableCount,
						 int maxSize,
						 short qx,
						 short qy, 
						 short qz )
{
	for ( int i = 0; i < tableCount; i++ )
	{
		if ( pTable[i].x == qx && pTable[i].y == qy && pTable[i].z == qz )
		{
			pTable[i].count++;
			return;
		}
	}

	if ( tableCount < maxSize )
	{
		pTable[tableCount].x = qx;
		pTable[tableCount].y = qy;
		pTable[tableCount].z = qz;
		pTable[tableCount].count = 1;
		tableCount++;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int intcmp( const void* s1, const void* s2 )
{
	if ( ((SRepeatData*)s1)->count > ((SRepeatData*)s2)->count )
	{
		return -1;
	}
	else if ( ((SRepeatData*)s1)->count < ((SRepeatData*)s2)->count )
	{
		return 1;
	}
	else
	{
		return 0;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void AnimCompressor::StartSecondPass()
{
	// first sort it
	qsort( uniqueQTable, uniqueQCount, sizeof(SUniqueItem), compareUniqueItem);
	qsort( uniqueTTable, uniqueTCount, sizeof(SUniqueItem), compareUniqueItem);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int is_hires( int flags )
{
	return ( flags & nxBONEDANIMFLAGS_CAMERADATA ) || ( flags & nxBONEDANIMFLAGS_OBJECTANIMDATA );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

AnimCompressor::AnimCompressor()
{
	Reset();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

AnimCompressor::~AnimCompressor()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void AnimCompressor::Reset()
{
	for ( int i = 0; i < 65536; i++ )
	{
		m_repeatQ[i].count = 0;
		m_repeatT[i].count = 0;
		m_repeatQ[i].data = i;
		m_repeatT[i].data = i;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimCompressor::SaveQ48Table( iostream& outStream, int platform )
{
	qsort( uniqueQTable, uniqueQCount, sizeof(SUniqueItem), compareUniqueItem);

	qsort( m_repeatQ, 65536, sizeof(int), intcmp );

	for ( int i = 0; i < NUM_BUCKETS; i++  )
	{
		outStream.write( (const char*)&uniqueQTable[i].x, sizeof(short) );
		outStream.write( (const char*)&uniqueQTable[i].y, sizeof(short) );
		outStream.write( (const char*)&uniqueQTable[i].z, sizeof(short) );
		outStream.write( (const char*)&m_repeatQ[i].data, sizeof(short) );
	}

	return 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimCompressor::SaveT48Table( iostream& outStream, int platform )
{
	qsort( uniqueTTable, uniqueTCount, sizeof(SUniqueItem), compareUniqueItem);

	qsort( m_repeatT, 65536, sizeof(int), intcmp );

	for ( int i = 0; i < NUM_BUCKETS; i++  )
	{
		outStream.write( (const char*)&uniqueTTable[i].x, sizeof(short) );
		outStream.write( (const char*)&uniqueTTable[i].y, sizeof(short) );
		outStream.write( (const char*)&uniqueTTable[i].z, sizeof(short) );
		outStream.write( (const char*)&m_repeatT[i].data, sizeof(short) );
	}

	return 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void AnimCompressor::PrintContents()
{
	qsort( m_repeatQ, 65536, sizeof(int), intcmp );
	qsort( m_repeatT, 65536, sizeof(int), intcmp );

	int uniqueQ = 0;
	int uniqueT = 0;
		
	int totalQ = 0;
	int totalT = 0;

	int i = 0;

	const int PRINT_BUCKETS = 512;

	for ( i = 0; i < 65536; i++ )
	{
		if ( i < PRINT_BUCKETS )
		{
			uniqueQ += m_repeatQ[i].count;
			uniqueT += m_repeatT[i].count;
		}
					
		totalQ += m_repeatQ[i].count;
		totalT += m_repeatT[i].count;
	}	

	for ( i = 0; i < PRINT_BUCKETS; i++ )
	{
		printf( "q-bucket[%d]=%d\n", i, m_repeatQ[i].count );
	}
		
	for ( i = 0; i < PRINT_BUCKETS; i++ )
	{
		printf( "t-bucket[%d]=%d\n", i, m_repeatT[i].count );
	}

	printf( "%d items of %d fit into the first %d short-buckets\n", uniqueQ, totalQ, PRINT_BUCKETS );
	printf( "%d items of %d fit into the first %d short-buckets\n", uniqueT, totalT, PRINT_BUCKETS );

	print_unique_table( uniqueQTable, uniqueQCount );
	print_unique_table( uniqueTTable, uniqueTCount );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimCompressor::LoadAnimFirstPass( iostream& animStream, int platform )
{
	// read in header information
	SBonedAnimFileHeader theBonedAnimHeader;
	if ( !animStream.read( (char*)&theBonedAnimHeader, sizeof(SBonedAnimFileHeader) ) )
	{
		printf( "No file header found" );
		return 1;
	}

	// read in platform-specific header information
	SPlatformFileHeader thePlatformHeader;
	if ( !animStream.read( (char*)&thePlatformHeader, sizeof(SPlatformFileHeader) ) )
	{
		printf( "No plat file header found" );
		return 1;
	}

	if ( is_hires( theBonedAnimHeader.flags ) )
	{
		// if it's a hi-res packet, 
		// then we don't try to compress it (yet)
		return 0;
	}

	if ( theBonedAnimHeader.flags & nxBONEDANIMFLAGS_USECOMPRESSTABLE )
	{
		// already compressed,
		// so don't try to do it again
		return 1;
	}

	int i = 0;

	CStandardAnimFramePointers thePointers;
	for ( i = 0; i < thePlatformHeader.numBones; i++ )
	{
		if ( !animStream.read( (char*)&thePointers, sizeof(CStandardAnimFramePointers) ) )
		{
			printf( "Failed reading bone pointers" );
			return 1;
		}
	}

	CStandardAnimQKey* pQKey = new CStandardAnimQKey[thePlatformHeader.numQKeys];

	if ( !animStream.read( (char*)pQKey, thePlatformHeader.numQKeys*sizeof(CStandardAnimQKey) ) )
	{
		Dbg_Message( "Failed while reading Q data" );
		return 1;
	}

	for ( i = 0; i < thePlatformHeader.numQKeys; i++ )
	{
		check_unique_table( uniqueQTable, uniqueQCount, MAX_UNIQUE_Q_TABLE_SIZE,
			pQKey[i].qx, pQKey[i].qy, pQKey[i].qz );
	}

	CStandardAnimTKey* pTKey = new CStandardAnimTKey[thePlatformHeader.numTKeys];

	if ( !animStream.read( (char*)pTKey, thePlatformHeader.numTKeys*sizeof(CStandardAnimTKey) ) )
	{
		Dbg_Message( "Failed while reading T data" );
		return 1;
	}

	for ( i = 0; i < thePlatformHeader.numTKeys; i++ )
	{
		check_unique_table( uniqueTTable, uniqueTCount, MAX_UNIQUE_T_TABLE_SIZE,
			pTKey[i].tx, pTKey[i].ty, pTKey[i].tz );
	}

	return 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimCompressor::LoadAnimSecondPass( iostream& animStream, int platform )
{
	// read in header information
	SBonedAnimFileHeader theBonedAnimHeader;
	if ( !animStream.read( (char*)&theBonedAnimHeader, sizeof(SBonedAnimFileHeader) ) )
	{
		printf( "No file header found" );
		return 1;
	}

	// read in platform-specific header information
	SPlatformFileHeader thePlatformHeader;
	if ( !animStream.read( (char*)&thePlatformHeader, sizeof(SPlatformFileHeader) ) )
	{
		printf( "No plat file header found" );
		return 1;
	}

	if ( is_hires( theBonedAnimHeader.flags ) )
	{
		// if it's a hi-res packet, 
		// then we don't try to compress it (yet)
		return 0;
	}

	if ( theBonedAnimHeader.flags & nxBONEDANIMFLAGS_USECOMPRESSTABLE )
	{
		// already compressed,
		// so don't try to do it again
		return 1;
	}

	int i = 0;

	CStandardAnimFramePointers thePointers;
	for ( i = 0; i < thePlatformHeader.numBones; i++ )
	{
		if ( !animStream.read( (char*)&thePointers, sizeof(CStandardAnimFramePointers) ) )
		{
			printf( "Failed reading bone pointers" );
			return 1;
		}
	}

	// read Q data
	CStandardAnimQKey theQKey;
	for ( i = 0; i < thePlatformHeader.numQKeys; i++ )
	{
		if ( !animStream.read( (char*)&theQKey, sizeof(CStandardAnimQKey) ) )
		{
			Dbg_Message( "Failed while reading Q data" );
			return 1;
		}

		if ( !in_unique_table( uniqueQTable, uniqueQCount, theQKey.qx, theQKey.qy, theQKey.qz ) )
		{
			m_repeatQ[(uint16)theQKey.qx].count++;
			m_repeatQ[(uint16)theQKey.qy].count++;
			m_repeatQ[(uint16)theQKey.qz].count++;
		}
	}

	// read T data
	CStandardAnimTKey theTKey;
	for ( i = 0; i < thePlatformHeader.numTKeys; i++ )
	{
		if ( !animStream.read( (char*)&theTKey, sizeof(CStandardAnimTKey) ) )
		{
			Dbg_Message( "Failed while reading T data" );
			return 1;
		}

		if ( !in_unique_table( uniqueTTable, uniqueTCount, theTKey.tx, theTKey.ty, theTKey.tz ) )
		{
			m_repeatT[(uint16)theTKey.tx].count++;
			m_repeatT[(uint16)theTKey.ty].count++;
			m_repeatT[(uint16)theTKey.tz].count++;
		}
	}

	return 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

