/*
	AnimExportFmt.h
	Animation Export Format for THPS4/5
*/

#ifndef __ANIMEXPORTFMT__
#define __ANIMEXPORTFMT__

#define ANIMVER3SUPPORT

#ifdef ANIMVER3SUPPORT
#define ANIMFILE_VERSION      3				// Partial anim bones are to no longer be stripped out
											// bone mask section gets included after file table names
#else
#define ANIMFILE_VERSION      2
#endif

// Key types
#define KEYTYPE_CHANGEPARENTKEY			0
#define KEYTYPE_CHANGEFOV				1
#define KEYTYPE_CHANGECAM               2	// Defines switch to another camera (Camera switches are
											// being defined as NoteTrack in trackview for the camera)

#define KEYTYPE_RUNSCRIPT				4
#define KEYTYPE_ENABLEOBJ				5	// Defines the time at which a specific object should be activated
#define KEYTYPE_DISABLEOBJ				6	// Defines the time at which a specific object should be deactivated

// Size of each individual key in bytes for each key type
#define KEYSIZE_CHANGEPARENTKEY        16
#define KEYSIZE_CHANGEFOV              16
#define KEYSIZE_RUNSCRIPT              16
#define KEYSIZE_CHANGECAM              44
#define KEYSIZE_ENABLEOBJ              16
#define KEYSIZE_DISABLEOBJ             16

// Flags
#define THPS3_FORMAT		(1<<31)		// Specifies an old animation from THPS3
#define INTERMEDIATE_FORMAT	(1<<30)		// Should always be set when exporting from plugin
#define UNCOMPRESSED_FORMAT (1<<29)		// The animation is strictly sampled (no compression applied)
#define COMPRESSEDTIME      (1<<26)		// Time values are given in integers (frames) as opposed to floats (time)
#define CONTAINS_CAMDATA    (1<<27)		// The exported animation is for camera data
#define PREROTATEDROOT      (1<<25)		// Root node is rotated -90 on X axis
#define OBJECTANIM			(1<<24)		// Animated Object
#define CUSTOMKEYSAT60FPS	(1<<21)		// Custom keys (FOV) are exported at 60 fps  (always set)
#define CUTSCENE            (1<<20)		// This animation is part of a cutscene
#define PARTIALANIM         (1<<19)		// This animation contains only a subset of the bones within the full model

struct SHeader
{
    uint32      version;
        // 1 == Starting version for THPS4

    uint32      flags;          
        // (1 << 31) = COMPRESSED  
        // (1 << 30) = THPS4 FORMAT

    float       duration;
        // In seconds    

    uint32      skeletonName;
        // user name for this skeleton
        // (i.e. human, bird, car) - for now, just hardcode to "human"

    uint32      numBones;
        // Number of bones in this animation

    uint32      numQKeys;
        // Number of quat keys

    uint32      numTKeys;
        // NUmber of translation keys

    uint32      numUserDefinedKeys;
        // Number of parent keys - for now, just hardcode to 0

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

class SSkeletonData
{
	bool bSuccessfulLoad;

public:
	uint32      checksum;
	uint32      numBones;

    uint32*     boneNames/*[ numBones ]*/;

    uint32*     parentData/*[ numBones ]*/;

    uint32*     flipTable/*[ numBones ]*/;

	uint32*     mask/*[ numBones ]*/;

	uint32      version;		// Not saved, stored from header

	SSkeletonData(SHeader* header);
	SSkeletonData(SHeader* header, int numBones);
	~SSkeletonData();

	bool Read(FILE* fp,int numBones);
	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
};

struct SQKey
{
    float       time;   
    float       qx, qy, qz;
    float       real;

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

class SQData
{
	int  numBones;
	int  totalNumKeys;
	bool bSuccessfulLoad;

	void Init(int numBones,int totalNumKeys);

public:
    uint32*     numKeysPerBone/*[ numBones ]*/;
    
    SQKey*      theQKeys/*[ totalNumKeys ]*/;

	SQData();
	SQData(int numBones,int totalNumKeys);

	~SQData();

	bool Read(FILE* fp,int numBones,int totalNumKeys);
	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
};

struct STKey
{
    float       time;
    float       tx, ty, tz;

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

class STData
{
	int  numBones;
	int  totalNumKeys;
	bool bSuccessfulLoad;

	void Init(int numBones,int totalNumKeys);

public:
    uint32*     numKeysPerBone/*[ numBones ]*/;
    
    STKey*      theTKeys/*[ totalNumKeys ]*/;

	STData();
	STData(int numBones,int totalNumKeys);

	~STData();

	bool Read(FILE* fp,int numBones,int totalNumKeys);
	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
};

struct SChangeParentKey
{
    float       time;
    short       parent;

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

struct SUserDefinedKey
{
	float		timeStamp;	// 4  bytes
	uint32      keyType;	// 4  bytes
	uint32      size;		// 4  bytes
							// 12 bytes total

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);

	// Overloads for sorting purposes
	inline int  operator==(SUserDefinedKey& right) {  return (timeStamp == right.timeStamp);  }
	inline int  operator< (SUserDefinedKey& right) {  return (timeStamp <  right.timeStamp);  }
	inline int  operator<=(SUserDefinedKey& right) {  return (timeStamp <= right.timeStamp);  }
	inline int  operator> (SUserDefinedKey& right) {  return (timeStamp >  right.timeStamp);  }
	inline int  operator>=(SUserDefinedKey& right) {  return (timeStamp >= right.timeStamp);  }
};

struct ChangeParentKey: public SUserDefinedKey
{
	uint32		newParentBoneId;

	ChangeParentKey();
	ChangeParentKey(float time,uint32 id);

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

struct ChangeFovKey: public SUserDefinedKey
{
	float		newFov;

	ChangeFovKey();
	ChangeFovKey(float time,float fov);

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

struct RunScriptKey: public SUserDefinedKey
{
	uint32		scriptName;

	RunScriptKey();
	RunScriptKey(float time,uint32 name);

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

struct EnableObjKey: public SUserDefinedKey
{
	uint32		arrayName;

	EnableObjKey();
	EnableObjKey(float time,uint32 name);

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

struct DisableObjKey: public SUserDefinedKey
{
	uint32		arrayName;

	DisableObjKey();
	DisableObjKey(float time,uint32 name);

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

struct ChangeCamKey: public SUserDefinedKey
{
	float x,y,z,w;		// Cam position
	float rx,ry,rz,rw;	// Quat CamRotation

	ChangeCamKey();
	ChangeCamKey(float time,
		         float posX, float posY, float posZ,
		         float rotX, float rotY, float rotZ, float rotW);

	bool Write(FILE* fp);
	bool Write(unsigned char* pData, unsigned int* pos);
	bool Read(FILE* fp);
};

unsigned long GetUserKeyMemoryUsage(SUserDefinedKey* keys, int nKeys);
void          SortUserKeys(SUserDefinedKey* keys, int nKeys);

#endif
