// []-----------------------------------------------------------------------------------------------------------[]
//	|	File				:	utils.c																																											|
//	|																																																						|
//	|								Copyright(c) 2000 Sega Europe.																															|
//	|																																																						|
//	|	Author			:	Elliott Martin, Sega Europe (martine@soe.sega.co.uk).																				|
//	|																																																						|
//	|	Description	:	Functions to access the transport devices, set-up and control them and configure the stack	|
//	|								to dial an ISP.																																							|
// []-----------------------------------------------------------------------------------------------------------[]

// TAB WIDTH == 2

// Header files:
// -------------
#include <shinobi.h>
#include <transport.h>
#include <nt_utl.h>
#include <ngdc.h>
#include <ngppp.h>
#include <ngeth.h>
#include <ngdhcpc.h>
#include <ngdns.h>

#include "utils.h"
#include "debug.h"


// Private function prototypes:
// ----------------------------
static void	ChangeDeviceParams(int, int);


// Private enumerated types:
// -------------------------
enum	// States used when running external modem reset script or hardware resetting internal modem...
{
// External modem reset states:
	EXTMDM_RESET_INIT = 1,
	EXTMDM_RESET_SCR_PRE_ESC,
	EXTMDM_RESET_SCR_POST_ESC,
	EXTMDM_RESET_ESC_CODE,

// Internal modem reset states:
	INTMDM_RESET_INIT,
	INTMDM_RESET_DROP_LINES,
	INTMDM_RESET_WAIT_FOR_DSR
};


// External global variables:
// --------------------------
#ifdef TR_USE_DHCP
extern NGdhcpoption	g_trDhcpOpts[];
extern NGuint				g_trDhcpOptsMask;
extern NGuint				g_trDhcpUp;
#endif
extern NGdev				g_trDev;
extern NGdevcb			*g_trDevcb;
extern NGmdmstate		g_trMdmstate;
extern NGifnet			g_trDhcpIfnet;


// Public global variables:
// ------------------------
char	g_trCountryInit[MAX_COUNTRY_INIT_STR];	// AT command used to initialise the modem for a specific country.
char	g_trModemFlags[MAX_MODEM_FLAGS_STR];		// Internal AT command flags to initialise the modem.
char	g_trUserInitStr[MAX_USER_INIT_STR];			// User-defined AT command string.
char	g_trDialStr[MAX_DIAL_STR];							// AT command string to dial the modem.


// Private global variables:
// -------------------------
static NGmdmscript g_trMdmResetScript[] = { // Modem reset script (for external modem).
// Action				Answer			Output		Goto		Retval	Timeout(secs)
	{"ATE0",			"OK",				NULL,			 1,			NG_EOK,			1},	// Disable command echo.
	{"ATH",				"OK",				NULL,			 2,			NG_EOK,			1},	// Force hang-up.
	{"ATZ",				"OK",				NULL,			-1,			NG_EOK,			2},	// Soft-reset.
	{NULL,				"ERROR",		NULL,			-1,			NG_EINVAL,	1},
	{NULL,				NULL,				NULL,			-1,			NG_EOK,			0}
};

char g_trConnectSpeedSerial[64] = "";	// Modem/Serial Connection speed strings (should be no longer than 64 bytes!).


char *g_trConnectSpeedDCLan[] = {	// DC LAN Connection speed strings
	"DC LAN not connected",   // DC LAN card not connected.
	"10 Mbps (half-duplex)",	// 10 Mbps connection speed at half-duplex.
	"10 Mbps (full-duplex)",	// 10 Mbps connection speed at full-duplex.
	"100 Mbps (half-duplex)",	// 100 Mbps connection speed at half-duplex.
	"100 Mbps (full-duplex)",	// 100 Mbps connection speed at full-duplex.
	"Unknown DC LAN speed",		// Unknown DC LAN connection speed.
};




// ============================================================================================================ //




// ------------------- //
//  InitModemScript()  //
// ------------------- //
int InitModemScript(NGmdmscript *script)
{
	int	dev_flags, err;

// Open the device...
	err = ngDevioOpen(&g_trDev, 0, &g_trDevcb);

	if(!err) {
	// Set-up the device parameters...
		dev_flags = (NG_DEVCF_SER_DTR | NG_DEVCF_SER_RTS | NG_DEVCF_SER_NOHUPCL);
		ChangeDeviceParams(dev_flags, 0);
		
	// Initialise the modem script...
		err = ngModemInit(&g_trMdmstate, g_trDevcb, script);
	}

	return err;
}




// ---------------------- //
//  ChangeDeviceParams()  //
// ---------------------- //
static void ChangeDeviceParams(int set_flags, int clear_flags)
{
	int	size, dev_flags;

// Set-up the serial device parameters...
	size = sizeof(dev_flags);

	ngDevioIoctl(g_trDevcb, NG_IOCTL_DEVCTL, NG_DEVCTL_GCFLAGS, &dev_flags, &size);	// Get the device params.

	dev_flags &= ~clear_flags;																											// Clear any flags as needed.
	dev_flags |= set_flags;																													// Set any new flags as needed.

	ngDevioIoctl(g_trDevcb, NG_IOCTL_DEVCTL, NG_DEVCTL_SCFLAGS, &dev_flags, &size);	// Set the new device params.
}




// ------------------- //
//  SetupDialParams()  //
// ------------------- //
int SetupDialParams(int transport, char *dial, int size)
{
// Copy the country init string to the connection script (not supported on external modems)...
	if(transport == TR_TRANSPORT_EXTMODEM)	strncpy(g_trCountryInit, "AT", sizeof(g_trCountryInit));
	else																		ntInfGetIniMdm(g_trCountryInit, sizeof(g_trCountryInit));
	
// Copy the modem init flags from flash RAM as an AT command...
	ntInfBuildFlagString(g_trModemFlags, sizeof(g_trModemFlags));

// Append the user-defined AT string...
	ntInfGetModemInit(0, g_trUserInitStr, sizeof(g_trUserInitStr));
//crt -- 
strcpy(g_trUserInitStr,"ATM1L3");

// If the dial string is valid then copy it to the connection script...
	if(!dial || !dial[0]) {
		DEBUG_PRINT(("The specified dial string was not valid\n"));
		return NG_EINVAL;	// There is no dial string so we can't dial anything!
	}
	
	else strncpy(g_trDialStr, dial, size);

	return NG_EOK;
}




// -------------------- //
//  SetupLoginParams()  //
// -------------------- //
int SetupLoginParams(NGifnet *ifnet, int isp_area)
{
	static	char	user[MAX_USERNAME_STR], pass[MAX_PASSWORD_STR];	// Store flash RAM login username and password.
					char	*u, *p;
					int		err;

// Get the login username from flash RAM...
	err = ntInfGetLoginId(isp_area, user, sizeof(user));

	if(err) {
	  DEBUG_PRINT(("Could not retrieve the login username from flash RAM (%d)\n", err));
		return NG_EINVAL;
  }

	else u = user;

// Get the login password from flash RAM...
	err = ntInfGetLoginPasswd(isp_area, pass, sizeof(pass));

	if(err) {
	  DEBUG_PRINT(("Could not retrieve the login password from flash RAM (%d)\n", err));
		return NG_EINVAL;
  }

	else p = pass;

// Copy the login options into the stack interface...
	if(user[0]) {
							err = ngIfSetOption(ifnet, NG_PPPIFO_AUTH_USER,		(void *)&u);
		if(!err)	err = ngIfSetOption(ifnet, NG_PPPIFO_AUTH_SECRET, (void *)&p);
	}
	
	if(err)	return err;
	else		return NG_EOK;
}




// --------------------- //
//  ConfigStaticDCLan()  //
// --------------------- //
int ConfigStaticDCLan(NGifnet *ifnet, Uint32 flag)
{
	struct in_addr	ip, subnet, gateway, dns1, dns2;	// Used to set-up the static configuration.
	int							err;

// Check that a static configuration has been fully enabled in flash RAM...
	if(!(flag & TCP_FIXED_ADDRESS) || !(flag & TCP_USE_GATEWAY)) return NG_EACCES;
 
// Set-up the static IP address...
	err = ntInfGetIPAddr(3, &ip);

	if(err || !ip.s_addr) {
		DEBUG_PRINT(("Could not retrieve the DC LAN static IP address from flash RAM (%d)\n", err));
		return NG_EFAULT;
	}

	err = ngIfSetOption(ifnet, NG_IFO_ADDR, (void *)&ip.s_addr);
	
	if(err) {
		DEBUG_PRINT(("Could not store the DC LAN static IP address in the stack (%d)\n", err));
		return err;
	}

// Set-up the static subnet mask...
	err = ntInfGetSubnetMask(&subnet);

	if(err || !subnet.s_addr) {
		DEBUG_PRINT(("Could not retrieve the DC LAN static subnet mask from flash RAM (%d)\n", err));
		return NG_EFAULT;
	}

	err = ngIfSetOption(ifnet, NG_IFO_NETMASK, (void *)&subnet.s_addr);
	
	if(err) {
		DEBUG_PRINT(("Could not store the DC LAN static subnet mask in the stack (%d)\n", err));
		return err;
	}
	
// Set-up the default gateway...
	err = ntInfGetPrimaryGateway(&gateway);

	if(err || !gateway.s_addr) {
		DEBUG_PRINT(("Could not retrieve the DC LAN static gateway from flash RAM (%d)\n", err));
		return NG_EFAULT;
	}

	err = ngRouteDefault(gateway.s_addr);

	if(err) {
		DEBUG_PRINT(("Could not add a default LAN gateway to the routing table (%d)\n", err));
		return err;
	}

// Set-up the static DNS servers...
	if(flag & TCP_USE_DNS) {
		err = ntInfGetPrimaryDnsAddress(3, &dns1);

    if(err || !dns1.s_addr) { DEBUG_PRINT(("Could not retrieve the Primary DNS address from flash RAM (%d)\n", err)); }
		
		ntInfGetSecondaryDnsAddress(3, &dns2);
		ngDnsInit(NULL, dns1.s_addr, dns2.s_addr);
	}

	return NG_EOK;
}




#ifdef TR_USE_DHCP
// -------------------- //
//  ResetDhcpOptions()  //
// -------------------- //
void ResetDhcpOptions(void)
{
// Reset the variables used to process the DHCP options...
	g_trDhcpOptsMask				= 0;
	g_trDhcpOpts[0].do_tag	= TAG_END;
	g_trDhcpUp							= FALSE;
}




// ----------------- //
//  AddDhcpOption()  //
// ----------------- //
void AddDhcpOption(int tag, int size, void *data)
{
	NGdhcpoption *p = g_trDhcpOpts; // Pointer to the global DHCP options structure.

// Find the end of the options structure...
	while(p->do_tag != TAG_END) p++;

// Add the new DHCP options...
	p->do_tag		= tag;
	p->do_size	= size;
	p->do_data	= data;

	p++;

	p->do_tag = TAG_END;
}




// ----------------- //
//  GetDhcpOption()  //
// ----------------- //
int GetDhcpOption(int tag, void **data)
{
	NGdhcpoption *p = g_trDhcpOpts; // Pointer to the global DHCP options structure.
	
// Make sure that the data pointer is valid...
	if(!data)	return NG_EINVAL;

// Find the matching tag in the options structure...
	while(p->do_tag != TAG_END) {
		if(p->do_tag == tag) {	// If the specified tag has been found...
			*data = p->do_data;
			return NG_EOK;
		}

		else p++;	// Skip to the next option.
	}

// No data was found...
	return NG_EINVAL;
}




// ---------------- //
//  ShutdownDhcp()  //
// ---------------- //
int ShutdownDhcp(NGifnet *ifnet, NGuint *is_dhcp_up)
{
	int err;

// Release the IP address back to the DHCP server...
	err = ngDhcpcRelease(ifnet);

	if((err == NG_EINVAL) || !err)	// Either DHCP is already shutdown or no error occured...
		*is_dhcp_up = FALSE;

	return err;
}
#endif	// TR_USE_DHCP




// ----------- //
//  InitDns()  //
// ----------- //
void InitDns(char *domain, NGuint flash_dns1, NGuint flash_dns2, NGuint server_dns1, NGuint server_dns2)
{
	NGuint	dns1, dns2;		// DNS server addresses to use (either from flash or access server).
	/*Sint32	country_code;	// Country code this Dreamcast is set to.

// Choose which DNS params to use depending on the country code...
	syCfgGetCountryCode(&country_code);

	switch(country_code) {
	case SYD_CFG_AMERICA:	// Flash settings override server provided DNS:
		dns1 = flash_dns1 ? flash_dns1 : server_dns1;
		dns2 = flash_dns2 ? flash_dns2 : server_dns2;
		break;

	case SYD_CFG_EUROPE:	// Server provided DNS settings override flash:
	default:
		dns1 = server_dns1 ? server_dns1 : flash_dns1;
		dns2 = server_dns2 ? server_dns2 : flash_dns2;
		break;
	}*/

// Work out which DNS address to use to initialise the stack...
	dns1 = server_dns1 ? server_dns1 : flash_dns1;
	dns2 = server_dns2 ? server_dns2 : flash_dns2;

// Initialise the DNS system (NOTE: A zero primary DNS value disables DNS)...
	ngDnsInit(domain, dns1, dns2);
}




// -------------------------- //
//  StoreModemConnectSpeed()  //
// -------------------------- //
int StoreModemConnectSpeed(void)
{
	int		ch;			// Character read from the modem state string.
	int		i = 1;	// Prevents overwriting of the array boundaries.
	char	*ptr;		// Pointer to the formatted connection string.

// Copy the start of modem connection line...
	strcpy(g_trConnectSpeedSerial, g_trMdmstate.mdst_buf);
  i		= strlen(g_trConnectSpeedSerial);
	ptr	= g_trConnectSpeedSerial + i;

// Read the connection string while we still have room in the buffer...
	while(++i < sizeof(g_trConnectSpeedSerial)) {
	// Read the next character in the connection string...
		ch = ngDevioReadByte(g_trDevcb, 0);

	// If a character was read...
		if(ch >= 0) {
			if(ch < ' ')	break;				// We have reached the end of the connection string.
			else					*ptr++ = ch;	// Store the character into the string.
		}

		else if(ch != NG_EWOULDBLOCK) {	// We received an error (not blocking warning)...
  		*ptr = 0;
			return ch;
		}
	}

	*ptr = '\0';		// Terminate the string.
	return NG_EOK;
}




// ----------------------- //
//  StoreSerialBaudRate()  //
// ----------------------- //
int StoreSerialBaudRate(void)
{
	unsigned int	baud_rate;		// Baud rate of the serial port.
	int						size, err;

// Get the speed from the serial device...
	size = sizeof(baud_rate);

	err = ngDevioIoctl(g_trDevcb, NG_IOCTL_DEVCTL, NG_DEVCTL_GIOSPEED, (void *)&baud_rate, &size);

	if(err < 0) return err;	// A negative error code...

	sprintf(g_trConnectSpeedSerial, "%u bps", baud_rate);
	return NG_EOK;
}



// ------------------------ //
//  GetDCLanConnectSpeed()  //
// ------------------------ //
int GetDCLanConnectSpeed(void)
{
	int		dclan_speed;	// Stores the internal representation of the DC LAN card speed.
	int		err;

// Get the DC LAN connection speed from the stack...
// (The reason 'g_trDhcpIfnet' is used here for both DHCP and PPPoE is because
//	it interfaces directly to the ethernet layer, whereas 'g_trPppoeIfnet' only
//	interfaces to the PPP layer above ethernet.  As the LAN speed option doesn't
//	exist in the PPP layer, calling the function below with the 'g_trPppoeIfnet'
//	parameter would be meaningless and return an invalid result.)
	err = ngIfGetOption(&g_trDhcpIfnet, NG_ETHIFO_DEV2, (void *)&dclan_speed);

	if(err) dclan_speed = -1;	// Force an error (unknown speed).

	switch(dclan_speed) {
	case DCLAN_AUTO:			return DCLAN_NOT_CONNECTED;	// No connection speed - is card disconnected?
	case DCLAN_10BaseT:		return DCLAN_10MBPS_HALF;		// 10Mbps, half-duplex:
	case DCLAN_10BaseTX:	return DCLAN_10MBPS_FULL;		// 10Mbps, full-duplex:
	case DCLAN_100BaseT:	return DCLAN_100MBPS_HALF;	// 100Mbps, half-duplex:
	case DCLAN_100BaseTX:	return DCLAN_100MBPS_FULL;	// 100Mbps, full-duplex:
	default:							return DCLAN_UNKNOWN;				// Unknown:
	}

	DEBUG_PRINT(("DC LAN card returned an unknown connection speed\n"));
}



// ----------------- //
//  ResetExtModem()  //
// ----------------- //
int ResetExtModem(void)
{
	static	bool	is_command_mode = false;					// Used to flag when in command mode.
	static	int		reset_state = EXTMDM_RESET_INIT;	// Tracks the current reset state of the modem.
	static	int		t_start = 0;											// Used to wait for 2 secs when sending the escape code.
					int		err;
					static bool once = false;

// Reset the external modem using a sequence of Hayes AT commands...
// (This is because a hardware reset does not work for the external modem due to
//	the serial interface on the SH4 not correctly supporting all hardware lines.)
	switch(reset_state) {
	case EXTMDM_RESET_INIT:	// Initialise the modem reset sequence:
		err = ngDevioOpen(&g_trDev, 0, &g_trDevcb);

		if(!err) {
			if(is_command_mode) {	// If we are now in command mode...
				err					= ngModemInit(&g_trMdmstate, g_trDevcb, g_trMdmResetScript);
				reset_state = EXTMDM_RESET_SCR_POST_ESC;
			}

			else reset_state = EXTMDM_RESET_ESC_CODE;//EXTMDM_RESET_SCR_PRE_ESC;
		}

		if(err)	goto reset_state;
		else		return NG_EWOULDBLOCK;

	case EXTMDM_RESET_SCR_PRE_ESC:	// Execute the modem script (pre escape code	: '+++'):
	case EXTMDM_RESET_SCR_POST_ESC:	// Execute the modem script (post escape code	: '+++'):
		err = ngModemPoll(&g_trMdmstate);

		switch(err) {
		case NG_EWOULDBLOCK:	return err;				// Blocking warning - continue polling the script.
		case NG_EOK:					goto reset_state;	// Successfully completed the modem reset script.
		
		case NG_ETIMEDOUT:	// The current script action timed out (error):
		// If this is before we have sent the escape code (timeout on first AT command)...
			if(reset_state == EXTMDM_RESET_SCR_PRE_ESC) {
				reset_state = EXTMDM_RESET_ESC_CODE;
				return NG_EWOULDBLOCK;
			}

			else {	// The second attempt at the script has timed out...
				DEBUG_PRINT(("External modem reset script timed out on command '%s', after sending the escape code\n", g_trMdmstate.mdst_currentp->mdm_action));
				goto reset_state;
			}
			
		default:	// Undefined error (not blocking warning?):
			DEBUG_PRINT(("An undefined error occured during the external modem reset ('%s' : %d)\n", g_trMdmstate.mdst_currentp->mdm_action, err));
			goto reset_state;
		}

		break;

	case EXTMDM_RESET_ESC_CODE:
	// If this is the first time we have reached here then send the escape code...
		if(!t_start) {
			err = ngDevioWrite(g_trDevcb, "+++", 3, 0);

			if(err == 3) t_start = ngOSClockGetTime();	// Start the reset timer.
		}

	// Pause for 2 seconds to give the modem time to switch to command mode.
		else if((ngOSClockGetTime() - t_start) >= (2 * NG_CLOCK_FREQ)) {
			ngDevioClose(g_trDevcb);
			is_command_mode = true;								// Flag that we are now in command mode.
			t_start					= 0;									// Reset the timer.
			reset_state			= EXTMDM_RESET_INIT;	// The escape code has been sent.  Poll the script again.
		}

		return NG_EWOULDBLOCK;
	}

// Error handling...
reset_state:
	ngDevioClose(g_trDevcb);
	is_command_mode = false;
	reset_state			= EXTMDM_RESET_INIT;
	if(!once) { once = true; return -6; } else once = false;
	return err;
}




// ----------------- //
//  ResetIntModem()  //
// ----------------- //
int ResetIntModem(int droptime, int timeout)
{
	static	unsigned int	t_start;													// Tracks the droptime and timeout periods.
	static	int						reset_state = INTMDM_RESET_INIT;	// Tracks the current reset state of the modem.
					int						dev_flags, size;									// Used to get the device parameters.
					int						err;

// Process the current reset state...
	switch(reset_state) {
	case INTMDM_RESET_INIT:
	// Open the serial device...
		err = ngDevioOpen(&g_trDev, 0, &g_trDevcb);

		if(err) return err;

	// Start the internal modem reset procedure...
		ChangeDeviceParams(0, NG_DEVCF_SER_DTR);	// Clear DTR (hardware line reset).
		t_start			= ngOSClockGetTime();					// Store the start value of the drop time period.
		reset_state = INTMDM_RESET_DROP_LINES;		// Set the next state.
		return NG_EWOULDBLOCK;

	case INTMDM_RESET_DROP_LINES:
	// Check to see if the drop time period has expired...
		if((ngOSClockGetTime() - t_start) >= (droptime * NG_CLOCK_FREQ)) {
			ChangeDeviceParams(NG_DEVCF_SER_DTR, 0);	// Set DTR.
			t_start			= ngOSClockGetTime();					// Store the start value of the timeout period.
			reset_state = INTMDM_RESET_WAIT_FOR_DSR;	// Set the next state.
		}

		return NG_EWOULDBLOCK;

	case INTMDM_RESET_WAIT_FOR_DSR:
	// Get the current device params...
		size = sizeof(dev_flags);
		ngDevioIoctl(g_trDevcb, NG_IOCTL_DEVCTL, NG_DEVCTL_GCFLAGS, &dev_flags, &size);

	// Check if the modem has been reset...
		if(dev_flags & NG_DEVCF_SER_DSR) err = NG_EOK;	// The modem has been reset!

	// Check to see if we have timed out waiting for DSR to come back up...
		else if((ngOSClockGetTime() - t_start) >= (timeout * NG_CLOCK_FREQ))	err = NG_ETIMEDOUT;
		else																																	return NG_EWOULDBLOCK;

	// If we reach here, either the modem has been reset or it timed out...
		ngDevioClose(g_trDevcb);
		reset_state = INTMDM_RESET_INIT;
		return err;
	}
}




// ------------------- //
//  SetupSerialPort()  //
// ------------------- //
void SetupSerialPort(void)
{
	Uint32				flag;				// Flag in flash RAM that stores the serial port settings.
	unsigned long	baud_rate;	// Stores the baud rate as configured in flash RAM.
	unsigned long	char_bits;	// Stores the number of bits per character as configured in flash RAM.
	unsigned long	parity;			// Stores the parity mode as configured in flash RAM.
	unsigned long	stop_bits;	// Stores the number of stop bits as configured in flash RAM.
	unsigned long	flow_ctrl;	// Stores the flow control mode as configured in flash RAM.
	int						size, err;

//  Get the flag from flash RAM...
	err = ntInfGetCommonFlag(&flag);

	if(err) { DEBUG_PRINT(("Could not read the communications flag from flash RAM (%d)\n", err)); return; }

// Store the baud rate...
	switch(flag & NTINF_COMM_SERIAL_BAUDMASK) {
	case NTINF_COMM_SERIAL_B300:		baud_rate = 300;		break;
	case NTINF_COMM_SERIAL_B1200:		baud_rate = 1200;		break;
	case NTINF_COMM_SERIAL_B2400:		baud_rate = 2400;		break;
	case NTINF_COMM_SERIAL_B4800:		baud_rate = 4800;		break;
	case NTINF_COMM_SERIAL_B9600:		baud_rate = 9600;		break;
	case NTINF_COMM_SERIAL_B14400:	baud_rate = 14400;	break;
	case NTINF_COMM_SERIAL_B19200:	baud_rate = 19200;	break;
	case NTINF_COMM_SERIAL_B38400:	baud_rate = 38400;	break;
	case NTINF_COMM_SERIAL_B57600:	baud_rate = 57600;	break;
	case NTINF_COMM_SERIAL_B230400:	baud_rate = 230400;	break;
	case NTINF_COMM_SERIAL_B460800:	baud_rate = 460800;	break;
	default:												baud_rate = 115200;	break;
	}

// Store the number of character bits...
	if(flag & NTINF_COMM_SERIAL_7BIT) char_bits = NG_DEVCF_SER_CS7;
	else															char_bits = NG_DEVCF_SER_CS8;

// Store the parity mode...
	if(flag & NTINF_COMM_SERIAL_PARITY) {
		if(flag & NTINF_COMM_SERIAL_PAR_EVEN)	parity = NG_DEVCF_SER_PAREVN;
		else																	parity = NG_DEVCF_SER_PARODD;
	}
	else parity = NG_DEVCF_SER_PARDIS;	// Disable parity.

// Store the number of stop bits...
	if((flag & NTINF_COMM_SERIAL_STOP2) == NTINF_COMM_SERIAL_STOP2)	stop_bits = NG_DEVCF_SER_CSTOPB2;
	else																														stop_bits = NG_DEVCF_SER_CSTOPB1;

// Store the flow control mode...
	if(!(flag & NTINF_COMM_SERIAL_NFLOW)) {
		if(flag & NTINF_COMM_SERIAL_SOFTFLOW)	flow_ctrl = NG_DEVIF_SFLOW | NG_DEVOF_SFLOW;
		else																	flow_ctrl = NG_DEVIF_HFLOW | NG_DEVOF_HFLOW;
	}
	else flow_ctrl = 0;

// Change the device settings based on the information gathered above...
	size	= sizeof(baud_rate);
	err		= ngDevioIoctl(g_trDevcb, NG_IOCTL_DEVCTL, NG_DEVCTL_SIOSPEED, (void *)&baud_rate, &size);

	if(err) { DEBUG_PRINT(("Could not set the serial port baud rate (%d)\n", err)); }

	ChangeDeviceParams((char_bits | parity | stop_bits | flow_ctrl), 0);
}
