// []-----------------------------------------------------------------------------------------------------------[]
//	|	File				:	transport.c																																									|
//	|																																																						|
//	|								Copyright(c) 2000 Sega Europe.																															|
//	|																																																						|
//	|	Author			:	Elliott Martin, Sega Europe (martine@soe.sega.co.uk).																				|
//	|																																																						|
//	|	Description	:	Functions to probe for the transport devices, set-up and control them and configure the			|
//	|								stack to dial an ISP.																																				|
//	|																																																						|
//	|	History			:	dd/mm/yyyy																																									|
//	|								----------																																									|
//	|								22/08/2000 - Elliott Martin - Version 1.00																									|
//	|										First working release of the device-probing PPP/socket demo.														|
//	|																																																						|
//	|								31/08/2000 - Elliott Martin - Version 1.01																									|
//	|										Seperated out the transport specific functions so as to hide the implementation from		|
//	|										the user.  They now exist more as an API lying behind the stack state machine (see			|
//	|										'states.c').																																						|
//	|																																																						|
//	|								07/09/2000 - Elliott Martin - Version 1.02																									|
//	|										Added a function to retrieve ISP info from flash RAM. Unfortunately, this has to be			|
//	|										exposed to the main app so it is no longer completely hidden behind the state machine.	|
//	|																																																						|
//	|								14/09/2000 - Elliott Martin - Version 1.03																									|
//	|										Modified the direct serial probing function to return a value that either indicates			|
//	|										what type of direct serial connection was found (RAS/non-RAS) or an error.							|
//	|																																																						|
//	|								19/09/2000 - Elliott Martin - Version 1.04																									|
//	|										Re-wrote the internal and external modem reset functions to make them totally						|
//	|										asynchronous.  Replaced ProbeForExternalModem() and ProbeForInternalModem() with				|
//	|										ProbeForModem() (both functions were using EXACTLY the same code).  Renamed							|
//	|										SetupDeviceParams() to ChangeDeviceParams() and changed the code so that device flags		|
//	|										could both be set and cleared.  Removed the function GetFlashInfo() because I decided		|
//	|										that this should remain application specific, so 'transport.c' is once again completely	|
//	|										hidden behind the state machine.  Changed the ProbeForModem() function to make it				|
//	|										totally asynchronous (reflects changes made to device probing state).  Improved the			|
//	|										handling of 'devcb' pointers so that they always feedback to the application's main			|
//	|										'state_data->devcb' pointer, providing dynamic updates of the device I/O state.  Added	|
//	|										some code to check that we don't overflow the array boundary in GetConnectSpeed().			|
//	|										Updated header filenames to ensure compliancy with NexGenIP version 1.2.								|
//	|																																																						|
//	|								27/09/2000 - Elliott Martin - Version 1.05																									|
//	|										Combined all old modem script functions into one function (InitModemScript()) to				|
//	|										reflect the asynchronous nature of the state machine.  Moved some modem scripts into		|
//	|										their own header file to make them accessible to the higher level functions (now allows	|
//	|										them to	be passed into InitModemScript() as a parameter).  Added new functions to copy	|
//	|										the ISP info supplied by the state machine (either from flash RAM or user-defined) into |
//	|										the stack and modem scripts.  Added new functions (GetModemConnectSpeed(), GetDCLAN...	|
//	|										...ConnectSpeed() & GetSerialBaudRate()) to get the connection speeds of all types of		|
//	|										transport.  These functions check the sizes of the arrays passed to them and,	if they		|
//	|										are too small, they alter the 'max_buf_len' param to inform the calling function of the |
//	|										correct buffer size required.																														|
// []-----------------------------------------------------------------------------------------------------------[]

// TAB WIDTH == 2

// Header files:
// -------------
#include <shinobi.h>
#include <ngdc.h>
#include <ngnet.h>
#include <ngappp.h>
#include <ngeth.h>
#include <ngip/icmp.h>
#include <ngsocket.h>
#include "transport.h"


// Private function prototypes:
// ----------------------------
static void	ChangeDeviceParams(NGdevcb **, int, int);


// Private enumerated types:
// -------------------------
static enum	// States used when running external modem reset script or hardware resetting internal modem.
{
// External modem reset states:
	EXTMDM_RESET_INIT = 1,
	EXTMDM_RESET_SCR_PRE_ESC,
	EXTMDM_RESET_SCR_POST_ESC,
	EXTMDM_RESET_ESC_CODE,

// Internal modem reset states:
	INTMDM_RESET_INIT,
	INTMDM_RESET_DROP_LINES,
	INTMDM_RESET_WAIT_FOR_DSR
};


// Global variables:
// -----------------
char gDialStr[105];			// AT command string to dial the modem (used in modem dial script).
char gModemInitStr[32];	// AT command string to initialise modem (used in modem dial script).

static NGmdmscript gMdmResetScript[] = // Modem reset script (for external modem).
{
// Action				Answer			Output		Goto		Retval	Timeout(secs)
	{"AT",				"OK",				NULL,			 1,			0,			20},	// Enough time to wait before sending escape code.
	{"ATH0",			"OK",				NULL,			 2,			0,			10},
	{"ATZ",				"OK",				NULL,			-1,			0,			10},
	{NULL,				NULL,				NULL,			-1,			0,			20}
};




// ============================================================================================================ //




// ------------------- //
//  InitModemScript()  //
// ------------------- //
int InitModemScript(NGdev *dev, NGdevcb **devcb, NGmdmstate *mdmstate, NGmdmscript *script)
{
	int	dev_flags, error;

// Open the device...
	error = ngDevioOpen(dev, 0, devcb);

	if(!error)
	{
	// Set-up the device parameters...
		dev_flags = (NG_DEVCF_SER_DTR | NG_DEVCF_SER_RTS | NG_DEVCF_SER_NOHUPCL);
		ChangeDeviceParams(devcb, dev_flags, 0);
		
	// Initialise the modem script...
		error = ngModemInit(mdmstate, *devcb, script);

		if(error)
			ngDevioClose(*devcb);
	}

	return error;
}




// ---------------------- //
//  ChangeDeviceParams()  //
// ---------------------- //
static void ChangeDeviceParams(NGdevcb **devcb, int set_flags, int clear_flags)
{
	int	size, dev_flags;

// Set-up the serial device parameters...
	size = sizeof(dev_flags);

	ngDevioIoctl(*devcb, NG_IOCTL_DEVCTL, NG_DEVCTL_GCFLAGS, &dev_flags, &size);	// Get the device params.

	dev_flags &= ~clear_flags;																										// Clear any flags as needed.
	dev_flags |= set_flags;																												// Set any new flags as needed.

	ngDevioIoctl(*devcb, NG_IOCTL_DEVCTL, NG_DEVCTL_SCFLAGS, &dev_flags, &size);	// Set the new device params.
}




// ------------------- //
//  SetupDialParams()  //
// ------------------- //
int SetupDialParams(char *init, char *dial1)
{
// Check the parameters...
	if(!init || !dial1)
		return NG_EINVAL;

// Copy the modem init string to the modem dial script...
	strcpy(gModemInitStr, init);

// Copy the main dial string to the modem dial script...
	strcpy(gDialStr, dial1);

	return NG_EOK;
}




// -------------------- //
//  SetupLoginParams()  //
// -------------------- //
int SetupLoginParams(NGifnet *ifnet, char *user, char *pass)
{
// Check the parameters...
	if(!user || !pass)
		return NG_EINVAL;

// Copy the login options into the stack interface...
	if(ngIfSetOption(ifnet, NG_PPPIFO_AUTH_USER,		(void *)&user) != 0)
		return ngOSGetErrno();

	if(ngIfSetOption(ifnet, NG_PPPIFO_AUTH_SECRET,	(void *)&pass) != 0)
		return ngOSGetErrno();

	return NG_EOK;
}




// ------------------ //
//  SetupDnsParams()  //
// ------------------ //
int SetupDnsParams(NGifnet *ifnet, unsigned int *dns1, unsigned int *dns2)
{
// TODO : BUG HERE - FIX!

// Copy the login options into the stack interface...
	if(ngIfSetOption(ifnet, NG_PPPIFO_IPCP_DNS1_ADDR,	(void *)&dns1) != 0)
		return ngOSGetErrno();

	if(ngIfSetOption(ifnet, NG_PPPIFO_IPCP_DNS2_ADDR,	(void *)&dns2) != 0)
		return ngOSGetErrno();

	return NG_EOK;
}




// ------------------------ //
//  GetModemConnectSpeed()  //
// ------------------------ //
int GetModemConnectSpeed(NGmdmstate *mdmstate, NGdevcb *devcb, char *buf, int *max_buf_len)
{
	int		ch;					// Character read from the modem state string.
	int		i = 1;			// Loop counter.
	char	*ptr;				// Pointer to the formatted connection string.
	char	temp[128];	// Used to store string before 'buf' size is checked (SHOULDN'T be bigger than 128 bytes).

// Copy the start of modem connection line...
	strcpy(temp, mdmstate->mdst_buf);
	ptr = temp + strlen(temp);

// Read the connection string while we still have room in the buffer...
	while(i++ < sizeof(temp))	// Only loop until array index 127 because we need 1 byte for '\0' at end of 'temp[]'.
	{
	// Read the next character in the connection string...
		ch = ngDevioReadByte(devcb, 0);

	// If a character was read...
		if(ch >= 0)
		{
			if(ch < ' ')	// We have reached the end of the connection string.
				break;
		
			else *ptr++ = ch;	// Store the character into the string.
		}

		else if(ch != NG_EWOULDBLOCK)	// We received an error (not blocking warning)...
		{
  		*ptr = 0;
			return ch;
		}
	}

	*ptr = '\0';	// Terminate the string.

// Check that the buffer provided is big enough to hold the string...
	if(strlen(temp) >= *max_buf_len)
	{
		*max_buf_len = strlen(temp) + 1;	// Change the value to show the correct size required (inc. '\0').
		return NG_ERANGE;
	}

	else	// The buffer is big enough.  Copy the string...
	{
		strcpy(buf, temp);
		return NG_EOK;
	}
}




// --------------------- //
//  GetSerialBaudRate()  //
// --------------------- //
int GetSerialBaudRate(NGdevcb *devcb, char *buf, int *max_buf_len)
{
	unsigned int	baud_rate;	// Baud rate of the serial port.
	int						size;
	char					temp[32];		// Used to store string before 'buf' size is checked (SHOULDN'T be bigger than 32 bytes).

// Get the speed from the serial device...
	size = sizeof(baud_rate);

	ngDevioIoctl(devcb, NG_IOCTL_DEVCTL, NG_DEVCTL_GIOSPEED, (void *)&baud_rate, &size);
	sprintf(temp, "%u bps", baud_rate);

// Check that the buffer provided is big enough to hold the string...
	if(strlen(temp) >= *max_buf_len)
	{
		*max_buf_len = strlen(temp) + 1;	// Change the value to show the correct size required (inc. '\0').
		return NG_ERANGE;
	}

	else	// The buffer is big enough.  Copy the string...
	{
		strcpy(buf, temp);
		return NG_EOK;
	}
}



// ------------------------ //
//  GetDCLANConnectSpeed()  //
// ------------------------ //
int GetDCLANConnectSpeed(NGifnet *ifnet, char *buf, int *max_buf_len)
{
	int		dclan_speed;	// Stores the internal representation of the DC LAN card speed.
	int		error;
	char	temp[32];			// Used to store string before 'buf' size is checked (SHOULDN'T be bigger than 32 bytes).

// Get the DC LAN connection speed from the stack...
	error = ngIfGetOption(ifnet, NG_ETHIFO_DEV2, (void *)&dclan_speed);

	if(error)
		dclan_speed = -1;	// Force an error (unknown speed).

	switch(dclan_speed)
	{
		case DCLAN_AUTO:			// No connection speed - this is an error!
			strcpy(temp, "not connected");
			break;

		case DCLAN_10BaseT:		// 10Mbps, half duplex:
			strcpy(temp, "10Mbps (half duplex)");
			break;

		case DCLAN_10BaseTX:	// 10Mbps, full duplex:
			strcpy(temp, "10Mbps (full duplex)");
			break;

		case DCLAN_100BaseT:	// 100Mbps, half duplex:
			strcpy(temp, "100Mbps (half duplex)");
			break;

		case DCLAN_100BaseTX:	// 100Mbps, full duplex:
			strcpy(temp, "100Mbps (full duplex)");
			break;

		default:							// Unknown:
			strcpy(temp, "unknown speed");
			break;
	}

// Check that the buffer provided is big enough to hold the string...
	if(strlen(temp) >= *max_buf_len)
	{
		*max_buf_len = strlen(temp) + 1;	// Change the value to show the correct size required (inc. '\0').
		return NG_ERANGE;
	}

	else	// The buffer is big enough.  Copy the string...
	{
		strcpy(buf, temp);
		return NG_EOK;
	}
}



// ----------------- //
//  ResetExtModem()  //
// ----------------- //
int ResetExtModem(int reset_state, NGdev *dev, NGdevcb **devcb, NGmdmstate *mdmstate)
{
	static	bool	is_command_mode = false;	// Used to flag when in command mode.
	static	int		t_start = 0;							// Used to wait for 2 secs when sending the escape code.
					int		error;

// Reset the external modem using a sequence of Hayes commands...
// (This is because a hardware reset does not work for the external modem due to
//	the serial interface on the SH4 not correctly supporting all hardware lines.)
	switch(reset_state)
	{
		case EXTMDM_RESET_INIT:	// Initialise the modem reset sequence:
			ngDevioClose(*devcb);	// Close the device before trying to re-open it (saves creating new instances of 
														//																															pointer in memory).
			error = ngDevioOpen(dev, 0, devcb);

			if(!error)
				error = ngModemInit(mdmstate, *devcb, gMdmResetScript);
			
			if(!error)
			{
				if(is_command_mode)	// If we are now in command mode...
					return EXTMDM_RESET_SCR_POST_ESC;

				else return EXTMDM_RESET_SCR_PRE_ESC;
			}

			else return error;

		case EXTMDM_RESET_SCR_PRE_ESC:		// Execute the modem script (pre escape code '+++'):
		case EXTMDM_RESET_SCR_POST_ESC:	// Execute the modem script (post escape code '+++'):
			error = ngModemPoll(mdmstate);

			switch(error)
			{
				case NG_EOK:		// Successfully completed the modem reset script:
					ngDevioClose(*devcb);
					is_command_mode = false;
					return NG_EOK;
				
				case NG_ETIMEDOUT:	// The current script action timed out (error):
					if(reset_state == EXTMDM_RESET_SCR_PRE_ESC)	// If this is before we have sent the escape code...
						return EXTMDM_RESET_ESC_CODE;

					else	// The second attempt at the script has timed out...
					{
						ngDevioClose(*devcb);
						return error;
					}

				case NG_EWOULDBLOCK:	// Blocking warning.  Continue polling the script:
					return reset_state;
					
				default:	// Undefined error (not blocking warning?):
					ngDevioClose(*devcb);
					return error;
			}

			break;

		case EXTMDM_RESET_ESC_CODE:
		// If this is the first time we have reached here then send the escape code...
			if(!t_start)
			{
				error = ngDevioWrite(*devcb, "+++", 3, 0);

				if(error == 3)
					t_start = syTmrGetCount();	// Start the reset timer.
			}

		// Pause for 2 seconds to give the modem time to switch to command mode.
			else if(syTmrCountToMicro(syTmrDiffCount(t_start, syTmrGetCount())) >= 2000000)
			{
				is_command_mode = true;		// Flag that we are now in command mode.
				t_start					= 0;			// Reset the timer.
				return EXTMDM_RESET_INIT;	// The escape code has been sent.  Poll the script again.
			}

			return reset_state;
	}
}




// ----------------- //
//  ResetIntModem()  //
// ----------------- //
int ResetIntModem(NGdev *dev, NGdevcb **devcb, int droptime, int timeout)
{
	static	int	reset_state = INTMDM_RESET_INIT;	// Tracks the current reset state of the modem.
	static	int	t_start;													// Tracks the droptime and timeout periods.
					int dev_flags, size;									// Used to get the device parameters.
					int	error;

// Process the current reset state...
	switch(reset_state)
	{
		case INTMDM_RESET_INIT:
		// Close the device before trying to re-open it (saves creating new instances of pointer in memory).
			ngDevioClose(*devcb);

		// Open the serial device...
			error = ngDevioOpen(dev, 0, devcb);

			if(error)
				return error;

		// Start the internal modem reset procedure...
			ChangeDeviceParams(devcb, 0, NG_DEVCF_SER_DTR);	// Clear DTR (hardware line reset).
			t_start			= syTmrGetCount();									// Store the start value of the drop time period.
			reset_state = INTMDM_RESET_DROP_LINES;					// Set the next state.
			return NG_EWOULDBLOCK;

		case INTMDM_RESET_DROP_LINES:
		// Check to see if the drop time period has expired...
			if(syTmrCountToMicro(syTmrDiffCount(t_start, syTmrGetCount())) >= (droptime * 1000000))
			{
				ChangeDeviceParams(devcb, NG_DEVCF_SER_DTR, 0);	// Set DTR.
				t_start			= syTmrGetCount();									// Store the start value of the timeout period.
				reset_state = INTMDM_RESET_WAIT_FOR_DSR;				// Set the next state.
			}

			return NG_EWOULDBLOCK;

		case INTMDM_RESET_WAIT_FOR_DSR:
		// Get the current device params...
			size = sizeof(dev_flags);
			ngDevioIoctl(*devcb, NG_IOCTL_DEVCTL, NG_DEVCTL_GCFLAGS, &dev_flags, &size);

		// Check if the modem has been reset...
			if(dev_flags & NG_DEVCF_SER_DSR)
				error = NG_EOK;	// The modem has been reset!

		// Check to see if we have timed out waiting for DSR to come back up...
			else if(syTmrCountToMicro(syTmrDiffCount(t_start, syTmrGetCount())) >= (timeout * 1000000))
				error = NG_ETIMEDOUT;

			else return NG_EWOULDBLOCK;

		// If we reach here, either the modem has been reset or it timed out...
			ngDevioClose(*devcb);
			reset_state = INTMDM_RESET_INIT;
			return error;
	}
}