// []-----------------------------------------------------------------------------------------------------------[]
//	|	File				:	states.c																																										|
//	|																																																						|
//	|								Copyright(c) 2000 Sega Europe.																															|
//	|																																																						|
//	|	Author			:	Elliott Martin, Sega Europe (martine@soe.sega.co.uk).																				|
//	|																																																						|
//	|	Description	:	State machine handling functions used to process stack states.															|
// []-----------------------------------------------------------------------------------------------------------[]

// TAB WIDTH == 2

// Header files:
// -------------
#include <shinobi.h>

#include <transport.h>

#include <ngppp.h>
#include <ngdc.h>
#include <ngeth.h>

#ifdef TR_USE_DHCP
#include <ngdhcpc.h>
#endif

#ifdef TR_USE_PPPOE
#include <ngpppoe.h>
#endif

#include "states.h"
#include "utils.h"
#include "debug.h"


// Macro definitions:
// ------------------
#define DC_LAN_TIMEOUT	5 * NG_CLOCK_FREQ	// 5 second timeout when looking for a DHCP/PPPoE server.


// Private function prototypes:
// ----------------------------
static int	PollEthernetDev(void);
static int	PollModemScript(void);
static void	EnableEthernetProtocol(int);
static void	DisableEthernetProtocol(int);
static char *GetScriptErrStr(int);


// External variables:
// -------------------
extern NGdevcb	*g_trDevcb;

#ifdef TR_USE_DHCP
extern NGifnet				g_trDhcpIfnet;
extern NGuint					g_trDhcpUp;
extern struct in_addr	g_trFlashGateway; // Used to store the Gateway settings retrieved from flash RAM.
extern char						g_trFlashHost[];  // Used to store the machine's host name retrieved from flash RAM.
#endif

#ifdef TR_USE_PPPOE
extern NGifnet	g_trPppoeIfnet;
#endif

extern NGifnet				*g_trIfnet;
extern NGmdmstate			g_trMdmstate;
extern struct in_addr	g_trFlashDns[];
extern bool						g_trStackInit;
extern int						g_trTransport;
extern int						g_trDisconnectStatus;
extern TR_DIAL_INFO		g_trDialInfo;
extern char						g_trCountryInit[];
extern char						g_trModemFlags[];
extern char						g_trUserInitStr[];
extern char						g_trDialStr[];


// Public global variables:
// ------------------------
NGmdmscript g_trSerRasScript[] = {	// RAS specific null-modem script (direct serial connection).
// Action							Answer						Output		Goto		Retval								Timeout(secs)
	{"CLIENTCLIENT",		"CLIENTSERVER",		NULL,			-1,			TR_RAS_SCR_CONNECTED,	1},	// Serial RAS script.
	{NULL,							"ERROR",					NULL,			-1,			NG_EINVAL,						1},
	{NULL,							NULL,							NULL,			-1,			0,										0}
};

TR_FUNC_PTR	g_trfpStateHandlingFunc;			// Next state handling function to call.
TR_FUNC_PTR	g_trfpProbeFinishedCallback;	// Called when all device probing has finished,
TR_FUNC_PTR	g_trfpDialFailureCallback;		// Called if the current dial attempt fails.
TR_FUNC_PTR	g_trfpConnectionCallback;			// Called while the stack is connected to an ISP.
TR_FUNC_PTR	g_trfpDisconnectCallback;			// Called just before disconnecting.
TR_FUNC_PTR	g_trfpDeviceResetCallback;		// Called after the device has been reset.


// Private global variables:
// -------------------------

static NGmdmscript g_trMdmDialScript[] = {	// Process all the stages involved in dialing and connecting to an ISP.
// Action							Answer						Output		Goto	Retval											Timeout(secs)
	{"ATE0",						"OK",							NULL,			 1,		NG_EOK,												5},	// Disable command echo.
	{g_trCountryInit,		"OK",							NULL,			 2,		NG_EOK,												5},	// Set-up country code.
	{g_trModemFlags,		"OK",							NULL,			 3,		NG_EOK,												5},	// Internal modem flags.
	{g_trUserInitStr,		"OK",							NULL,			 4,		NG_EOK,												5},	// User-defined AT string.
	{g_trDialStr,				"CONNECT",				NULL,			-1,		TR_DIAL_SCR_CONNECTED,				90},
	{NULL,							"NO CARRIER",			NULL,			-1,		TR_DIAL_SCR_ERR_NO_CARRIER,		0},
	{NULL,							"NO ANSWER",			NULL,			-1,		TR_DIAL_SCR_ERR_NO_ANSWER,		0},
	{NULL,							"NO DIALTONE",		NULL,			-1,		TR_DIAL_SCR_ERR_NO_DIALTONE,	0},
	{NULL,							"NO DIAL TONE",		NULL,			-1,		TR_DIAL_SCR_ERR_NO_DIALTONE,	0},
	{NULL,							"BUSY",						NULL,			-1,		TR_DIAL_SCR_ERR_LINE_BUSY,		0},
	{NULL,							"ERROR",					NULL,			-1,		NG_EINVAL,										0},
	{NULL,							NULL,							NULL,			-1,		NG_EOK,												0}
};

static int		g_trDCLanTimeoutStart;	// Stores the time when we started looking for a DHCP/PPPoE server.
static bool		g_trPppNegotiation;			// Flags when PPP negotiation is taking place (used in login failure check).




// ============================================================================================================ //




// ------------ //
//  StartDev()  //
// ------------ //
void StartDev(void)
{
#ifdef TR_USE_DHCP
  static NGuint					DhcpSubnet;
  static NGuint					DhcpGateway;
  static struct in_addr	DhcpDns[2];
#endif
	int err = NG_EOK;

// If we are using DC LAN...
	if(g_trTransport & TR_COMPONENT_ETHERNET) {
		Uint32	flag;		// Stores the TCP/IP info area flags.
		int			speed;	// Stores the DC LAN link connection speed as set in flash RAM.

		ntInfGetTCPIPInfoFlag(&flag);	// Get the DC LAN flag.

	// Find out which link connection speed is configured in flash RAM...
		speed = flag & TCP_LINK_NEGO_MASK;

		switch(speed) {
		default:										speed = DCLAN_AUTO;				break;
		case TCP_LINK_NEGO_10BT:		speed = DCLAN_10BaseT;		break;
		case TCP_LINK_NEGO_10BTX:		speed = DCLAN_10BaseTX;		break;
		case TCP_LINK_NEGO_100BT:		speed = DCLAN_100BaseT;		break;
		case TCP_LINK_NEGO_100BTX:	speed = DCLAN_100BaseTX;	break;
		}
		
		err = ngIfSetOption(g_trIfnet, NG_ETHIFO_DEV1, (void *)&speed);
		if(err)	{ DEBUG_PRINT(("Could not set the link connection speed of the DC LAN card (%d)\n", err)); }

	// Record the time we started searching for a DHCP or PPPoE server (used for timeout)...
		g_trDCLanTimeoutStart = ngOSClockGetTime();

	// If DHCP is enabled in flash RAM then start looking for a DHCP server...
	#ifdef TR_USE_DHCP
		if(!(flag & TCP_DYNAMIC_NO_DHCP)) {
    // Reset the DHCP options structure.
      ResetDhcpOptions();

    // Pass the hostname to the DHCP server in case it is needed for authentication...
			if(*g_trFlashHost)
    		AddDhcpOption(TAG_HOST_NAME,	strlen(g_trFlashHost),	(void *)g_trFlashHost);

			AddDhcpOption(TAG_SUBNET_MASK,	sizeof(DhcpSubnet),			(void *)&DhcpSubnet);

			if(!(flag & TCP_USE_GATEWAY))
				AddDhcpOption(TAG_GATEWAY,		sizeof(DhcpGateway),		(void *)&DhcpGateway);

			if(!(flag & TCP_USE_DNS))
				AddDhcpOption(TAG_DOMAIN_SERVER,	sizeof(DhcpDns),		(void *)DhcpDns);

		// Start looking for a DHCP server...
			err = ngDhcpcStart(&g_trDhcpIfnet);

			if(err) {
				DEBUG_PRINT(("Could not start the DHCP service (%d)\n", err));
				err = TR_STATUS_ERR_DHCP_INIT;
			}
		}
	#endif	// TR_USE_DHCP

	// If PPPoE is enabled in flash RAM then start looking for a PPPoE server...
	#ifdef TR_USE_PPPOE
		if(flag & TCP_DYNAMIC_PPPoE) {
			err = ngIfOpen((NGifnet *)&g_trPppoeIfnet);

			if(!err) err = ngPppoeStart(&g_trPppoeIfnet);

			if(err) {
				DEBUG_PRINT(("Could not start PPPoE service (%d)\n", err));
				err = TR_STATUS_ERR_PPPOE_INIT;
				
			#ifdef TR_USE_DHCP
				ShutdownDhcp(&g_trDhcpIfnet, &g_trDhcpUp);	// Stop DHCP.
			#endif
			}
		}
	#endif	// TR_USE_PPPOE
	}

// If we are using a standard modem device then start the dial-up procedure...
	if(g_trTransport & TR_COMPONENT_SCRIPT) {
		err = InitModemScript((g_trTransport & TR_COMPONENT_MODEM) ? g_trMdmDialScript : g_trSerRasScript);

		if(err) {
			DEBUG_PRINT(("Could not initialise the connection script (%d)\n", err));
			ngDevioClose(g_trDevcb);
			err = TR_STATUS_ERR_SCR_INIT;
		}
	}

// If this a serial based transport device...
	if(g_trTransport & TR_COMPONENT_SERIAL) {
		SetupSerialPort();	// Configure the serial port based on flash RAM settings.

	// If we are using a non-RAS serial connection then start PPP straight away...
		if(!(g_trTransport & TR_COMPONENT_SCRIPT)) {
			err = StartPpp();
			
			if(!err) {
				SetStackState(TR_STATE_POLL_PPP);
				return;
			}
		}
	}

// If there was an error, reset the device.  Otherwise we can start running the connection process...
	if(err) {
		SetDisconnectStatus(err);
		SetStackState(TR_STATE_RESET_DEV);
	}
		
	else SetStackState(TR_STATE_POLL_DEV);	// Continue with the connection procedure.
}




// ---------------- //
//  StatePollDev()  //
// ---------------- //
void StatePollDev(void)
{
	int	t_timeout; // Used to test if the search for a DHCP/PPPoE server has timed out.
	int err;

// If we are using the DC LAN card then poll the DHCP and PPPoE interfaces...
	if(g_trTransport & TR_COMPONENT_ETHERNET) {
		t_timeout = ngOSClockGetTime() - g_trDCLanTimeoutStart;	// Check server probe timeout.

		if(t_timeout > DC_LAN_TIMEOUT) {
			DEBUG_PRINT(("Timeout occured while looking for a DHCP/PPPoE server\n", __FILE__, __LINE__));
		
		#ifdef TR_USE_DHCP
			ShutdownDhcp(&g_trDhcpIfnet, &g_trDhcpUp);
		#endif
		#ifdef TR_USE_PPPOE
    	ngPppoeStop(&g_trPppoeIfnet);
		#endif

			SetDisconnectStatus(TR_STATUS_ERR_DCLAN_TIMEOUT);
			SetStackState(TR_STATE_RESET_DEV);
			return;
		}

		else {
		// Poll the ethernet device to see if we have found a DHCP or PPPoE server...
			err = PollEthernetDev();

			switch(err) {
			case NG_EOK:	// A DHCP or PPPoE server was found!
				if(g_trTransport & TR_COMPONENT_DHCP)
					SetStackState(TR_STATE_CONNECTED);	// A DHCP server was found and DHCP has been enabled.

				else if(g_trTransport & TR_COMPONENT_PPP)
					SetStackState(TR_STATE_POLL_PPP);		// A PPPoE server was found and PPPoE has been enabled.
				
			// Drop through to return below...

			case NG_EWOULDBLOCK:
			default:
				return;	// Error.
			}
		}
	}

// Else we are using a modem or RAS serial connection so poll the modem script...
	else if(g_trTransport & TR_COMPONENT_SCRIPT) {
		err = PollModemScript();

		switch(err) {
		case NG_EWOULDBLOCK: return;	// Blocking warning.  Continue polling.

		case NG_EOK:	// Successfully found a modem or RAS serial connection!
			g_trDialInfo.attempt = TR_DIAL_NONE;	// Prevent further dial attempts (modem only).
			break;
			
		default:	// An error occured:
			DEBUG_PRINT(("Modem script error (%d) on command '%s'\n", err, g_trMdmstate.mdst_currentp->mdm_action));

			if(g_trTransport & TR_COMPONENT_MODEM) {	// Dial a backup number?
				Uint32 flag;

				err = ntInfGetFlag(&flag);

				if(err) { DEBUG_PRINT(("Could not retrieve the main flash RAM info flags (%d)\n", err)); }

				else if(!(flag & FIXED_NUMBER))
					g_trfpDialFailureCallback();	// Inform the application that we are trying a backup number next.

				else g_trDialInfo.attempt = TR_DIAL_NONE;	// Prevent further dial attempts.
			}

			SetStackState(TR_STATE_RESET_DEV);
			break;
		}
	}

	ngDevioClose(g_trDevcb);
	
// If the script was successful then start the PPP session...
	if(!err) {
		if(StartPpp() == NG_EOK) SetStackState(TR_STATE_POLL_PPP);
		else										 SetStackState(TR_STATE_RESET_DEV);
	}
}




// ------------------- //
//  PollEthernetDev()  //
// ------------------- //
static int PollEthernetDev(void)
{
	Uint32	flag;					// Stores the TCP/IP info area flags.
	int  	  *pAddr;       // Retrieve different address returned by DHCP
	int			err;

#ifdef TR_USE_PPPOE
// Check to see if a PPPoE server has been found...
	if(ngPppoeGetState(&g_trPppoeIfnet) == NG_POEIFS_SESSION) {
	#ifdef TR_USE_DHCP
	// Stop looking for a DHCP server and enable PPPoE...
		err = ShutdownDhcp(&g_trDhcpIfnet, &g_trDhcpUp);

		if(err)	{ DEBUG_PRINT(("Could not shutdown the DHCP service (%d)\n", err)); }
	#endif	// TR_USE_DHCP

		EnableEthernetProtocol(TR_COMPONENT_PPP);
		g_trIfnet = (NGifnet *)&g_trPppoeIfnet;	// Dereference the interface for use in the state machine.

	// Start the normal PPP session...
		err = StartPpp();

		if(err) {
			SetDisconnectStatus(TR_STATUS_ERR_PPPOE_FAILED);
			return err;
		}

		else return NG_EOK;
	}
#endif	// TR_USE_PPPOE
    
// If a DHCP server has been found then set-up the DC LAN connection...
#ifdef TR_USE_DHCP
  if(g_trDhcpUp) {
	// Stop the stack looking for a PPPoE server...
	#ifdef TR_USE_PPPOE
		ngPppoeStop(&g_trPppoeIfnet);
	#endif

	// Enable DHCP in the transport field and dereference the interface...
		EnableEthernetProtocol(TR_COMPONENT_DHCP);
		g_trIfnet = (NGifnet *)&g_trDhcpIfnet;

	// Set-up the network subnet mask...
		err = GetDhcpOption(TAG_SUBNET_MASK, (void **)&pAddr);

		if(err) { 
			DEBUG_PRINT(("Could not retrieve the subnet mask returned by the DHCP server\n"));
			SetDisconnectStatus(TR_STATUS_ERR_DHCP_INIT);
			return err;
		}

		else ngIfSetOption(g_trIfnet, NG_IFO_NETMASK, (void *)pAddr);

		err = GetDhcpOption(TAG_GATEWAY, (void **)&pAddr);

		if(err || !*pAddr) { 
      DEBUG_PRINT(("Could not retrieve the default gateway returned by the DHCP server\n"));

			pAddr = (int*)&g_trFlashGateway.s_addr;

  		if(!*pAddr) {
				DEBUG_PRINT(("Could not find any gateway stored in flash RAM\n"));
				err = NG_EFAULT;
			}

      else err = 0;
    }

		if(err) {
			SetDisconnectStatus(TR_STATUS_ERR_DHCP_INIT);
			return err;
		}

		ngRouteDefault(*pAddr);	// Add the gateway to the routing table (default route).

	// Initialise the DNS servers (pass in both flash and DHCP values)...
		err = GetDhcpOption(TAG_DOMAIN_SERVER, (void **)&pAddr);
    
		if(err || !pAddr[0]) {
			DEBUG_PRINT(("Could not retrieve the DNS servers returned by the DHCP server\n"));
  		pAddr[0] = pAddr[1] = 0;
    }

		InitDns(NULL, g_trFlashDns[0].s_addr, g_trFlashDns[1].s_addr, pAddr[0], pAddr[1]);
		
		return NG_EOK;	// The DC LAN card should now be connected to the LAN.
	}
#endif	// TR_USE_DHCP

// No servers have been found yet.  Continue polling...
	return NG_EWOULDBLOCK;
}




// ------------------- //
//  PollModemScript()  //
// ------------------- //
static int PollModemScript(void)
{
	int err, state;

// Get and process the current modem script state...
	state = ngModemPoll(&g_trMdmstate);

	switch(state) {
	case NG_EWOULDBLOCK: return state;

	case TR_DIAL_SCR_CONNECTED:	// Modem successfully connected!
		err = StoreModemConnectSpeed();

		if(err)	{ DEBUG_PRINT(("Could not get the down/up link speed of the modem connection (%d)\n", err)); }

	// Drop through to the next case in order to return NG_EOK...

	case TR_RAS_SCR_CONNECTED: return NG_EOK;	// RAS serial connection successful!

	default:	// An error occured:
		if(state > NG_EOK) {
			DEBUG_PRINT(("Could not establish a modem or RAS serial connection ('%s')\n", GetScriptErrStr(state)));
			SetDisconnectStatus(state);	// Set the disconnection status as defined in the modem script.
		}

		else if(state == NG_ETIMEDOUT) SetDisconnectStatus(TR_STATUS_ERR_SCR_TIMEOUT);	// Script timeout.

		else { DEBUG_PRINT(("Could not establish a modem or RAS serial connection (%d)\n", state)); }

		return state;
	}
}




// ------------ //
//  StartPpp()  //
// ------------ //
int StartPpp(void)
{
	int err = NG_EOK;

// Flag that we are now in PPP negotiation...
	g_trPppNegotiation = true;

// Open the interface (don't open it if we are using PPPoE because it should already be open)...
	if(!(g_trTransport & TR_COMPONENT_ETHERNET)) err = ngIfOpen(g_trIfnet);

// Start the PPP negotiation process...
	if(!err) {
		err = ngPppStart(g_trIfnet);

		if(err) {
			DEBUG_PRINT(("Could not start PPP negotiation (%d)\n", err));
			ngIfClose(g_trIfnet);
		}
	}

	else { DEBUG_PRINT(("Could not open the PPP interface (%d)\n", err)); }

	return (err ? TR_STATUS_ERR_PPP_INIT : NG_EOK);
}




// ---------------- //
//  StatePollPpp()  //
// ---------------- //
void StatePollPpp(void)
{
	NGuint	ppp_dns1, ppp_dns2;	// DNS servers returned by the PPP server.
	int			ppp_state;					// Current PPP negotiation state of the stack.
	int			err;

// Process the current PPP state...
	ppp_state = ngPppGetState(g_trIfnet);

	switch(ppp_state) {
	case NG_PIFS_DEAD:	// Disconnected:
		if(!(g_trTransport & TR_COMPONENT_ETHERNET)) ngIfClose(g_trIfnet);	// Close the normal PPP interface...
	
	#ifdef TR_USE_PPPOE
		else ngPppoeStop(g_trIfnet);	// Shutdown the PPPoE session (do not close the interface yet).
	#endif

	// Check if we were in the process of PPP negotiation (PPP login failure)...
		if(g_trPppNegotiation) {
			SetDisconnectStatus(TR_STATUS_ERR_PPP_NEG);
			g_trPppNegotiation = false;
		}

	// Reset the device...
		SetStackState(TR_STATE_RESET_DEV);
		return;

	case NG_PIFS_IFUP:	// A connection has been established:
	// We are no longer authenticating PPP...
		g_trPppNegotiation = false;

	// Get the DNS server parameters returned by the PPP server...
    ppp_dns1 = ppp_dns2 = 0;
		
		err	= ngIfGetOption(g_trIfnet, NG_PPPIFO_IPCP_DNS1_ADDR, (void *)&ppp_dns1);
					ngIfGetOption(g_trIfnet, NG_PPPIFO_IPCP_DNS2_ADDR, (void *)&ppp_dns2);

	// Check that we have valid values...
		if(err || !ppp_dns1) { DEBUG_PRINT(("PPP Server did not return a primary DNS address\n")); }

	// Initialise the NexGenIP DNS system (pass in both flash and PPP values)...
		InitDns(NULL, g_trFlashDns[0].s_addr, g_trFlashDns[1].s_addr, ppp_dns1, ppp_dns2);
		SetStackState(TR_STATE_CONNECTED);
    return;
	}
}




// -------------------- //
//  StateResetDevice()  //
// -------------------- //
void StateResetDevice(void)
{
	static	bool	callback_called = false;	// Flags when the disconnection callback has been called.
					int		err;

// If no further connection attempts should be made then call the disconnection callback...
	if(g_trDialInfo.attempt == TR_DIAL_BACKUP2) g_trDialInfo.attempt = TR_DIAL_NONE;

	if((g_trDialInfo.attempt == TR_DIAL_NONE) && !callback_called) {
		g_trfpDisconnectCallback();
		callback_called = true;
	}

// Perform a reset specific to the type of device in use...
	switch(g_trTransport) {
	case TR_TRANSPORT_EXTMODEM:	// Reset the external modem:
		err = ResetExtModem();

		switch(err) {
		case NG_EWOULDBLOCK: return;
			
		case NG_EOK:	// Check if we should now dial a backup number:
			if(g_trDialInfo.attempt >= TR_DIAL_PRIMARY) g_trDialInfo.attempt++;
			break;
	
		default:
			g_trDialInfo.attempt = TR_DIAL_NONE;	// An error occured so prevent further dial attempts.
			break;
	}

	break;

	case TR_TRANSPORT_INTMODEM:	// Reset the internal modem (drop hardware lines for 2 secs, timeout after 5 secs):
		err = ResetIntModem(2, 5);
		
		switch(err) {
		case NG_EWOULDBLOCK: return;

		case NG_EOK:	// Check if we should now dial a backup number:
			if(g_trDialInfo.attempt >= TR_DIAL_PRIMARY) g_trDialInfo.attempt++;
			break;

		default:
			DEBUG_PRINT(("Could not reset the internal modem hardware lines (%d)\n", err));
			g_trDialInfo.attempt = TR_DIAL_NONE;	// An error occured so prevent further dial attempts.
			break;
		}

		break;

	case TR_TRANSPORT_DCLAN_STATIC:

#ifdef TR_USE_DHCP
	case TR_TRANSPORT_DCLAN_DHCP:
		DisableEthernetProtocol(TR_COMPONENT_DHCP);	// Disable DHCP.
#endif	// TR_USE_DHCP

#ifdef TR_USE_PPPOE
	case TR_TRANSPORT_DCLAN_PPPOE:	// Check if the PPP session has been shutdown yet:
		if(ngPppoeGetState(&g_trPppoeIfnet) != NG_POEIFS_DOWN) return;
		ngIfClose((NGifnet *)&g_trPppoeIfnet);			// PPPoE has now been shutdown.
		DisableEthernetProtocol(TR_COMPONENT_PPP);	// Disable PPPoE.
#endif	// TR_USE_PPPOE
	  
		break;
	}

// The device has been reset.  If we are using a modem then redial, otherwise return to an idle state...
	if((g_trTransport & TR_COMPONENT_MODEM) && (g_trDialInfo.attempt > TR_DIAL_PRIMARY))
		trConnect(g_trTransport, g_trDialInfo.net_info);

	else {
		g_trDialInfo.attempt	= TR_DIAL_PRIMARY;	// Reset the dial attempt counter.
		callback_called				= false;

		SetStackState(TR_STATE_IDLE);
		g_trfpDeviceResetCallback();

		ngExit(0);													// Shutdown the NexGenIP library and free its resources.
		g_trStackInit = false;							// Flag that the stack has been shutdown.
		g_trTransport = TR_TRANSPORT_NONE;	// Clear the chosen transport device.
	}
}




// ----------------- //
//  SetStackState()  //
// ----------------- //
void SetStackState(int state)
{
// Set-up the state handling function pointer to call the required state function...
	switch(state) {
	case TR_STATE_IDLE:
		g_trfpStateHandlingFunc = NULL;
		break;

	case TR_STATE_POLL_DEV:
		g_trfpStateHandlingFunc = StatePollDev;
		break;

	case TR_STATE_RESET_DEV:
		g_trfpStateHandlingFunc = StateResetDevice;
		break;

	case TR_STATE_POLL_PPP:
		g_trfpStateHandlingFunc = StatePollPpp;
		break;

	case TR_STATE_CONNECTED:
		g_trfpStateHandlingFunc = g_trfpConnectionCallback;
		break;
	}
}




// ----------------------- //
//  SetDisconnectStatus()  //
// ----------------------- //
void SetDisconnectStatus(int status)
{
// Store the last disconnection status (as long as one hasn't been stored already)...
	if(!status || !g_trDisconnectStatus) g_trDisconnectStatus = status;
}




// -------------------------- //
//  EnableEthernetProtocol()  //
// -------------------------- //
static void EnableEthernetProtocol(int protocol)
{
// Check that this function is being called for the Ethernet device...
	if(!(g_trTransport & TR_COMPONENT_ETHERNET)) return;

// Enable the Ethernet protocol specified...
	g_trTransport |= protocol;
}




// --------------------------- //
//  DisableEthernetProtocol()  //
// --------------------------- //
static void DisableEthernetProtocol(int protocol)
{
// Check that this function is being called for the Ethernet device...
	if(!(g_trTransport & TR_COMPONENT_ETHERNET)) return;

// Disable the Ethernet protocol specified...
	g_trTransport &= ~protocol;
}




// ------------------- //
//  GetScriptErrStr()  //
// ------------------- //
static char *GetScriptErrStr(int err)
{
	switch(err) {
	case TR_STATUS_ERR_SCR_NO_CARRIER:	return "TR_STATUS_ERR_SCR_NO_CARRIER";
	case TR_STATUS_ERR_SCR_NO_ANSWER:		return "TR_STATUS_ERR_SCR_NO_ANSWER";
	case TR_STATUS_ERR_SCR_NO_DIALTONE:	return "TR_STATUS_ERR_SCR_NO_DIALTONE";
	case TR_STATUS_ERR_SCR_LINE_BUSY:		return "TR_STATUS_ERR_SCR_LINE_BUSY";
	}
}