// []-----------------------------------------------------------------------------------------------------------[]
//	|	File				:	stack_cfg.c																																									|
//	|																																																						|
//	|								Copyright(c) 2000 Sega Europe.																															|
//	|																																																						|
//	|	Author			:	Elliott Martin, Sega Europe (martine@soe.sega.co.uk).																				|
//	|																																																						|
//	|	Description	:	Global data structures for NexGenIP stack configuration.																		|
// []-----------------------------------------------------------------------------------------------------------[]

// TAB WIDTH == 2

// Header files:
// -------------
#include <shinobi.h>
#include <transport.h>
#include <ngdc.h>
#include <ngip\ethernet.h>
#include <ngeth.h>
#include <ngtcp.h>
#include <ngudp.h>
#include <ngappp.h>

#ifdef TR_USE_DHCP
#include <ngbootpc.h>
#include <ngdhcpc.h>
#endif

#ifdef TR_USE_PPPOE
#include <ngpppoe.h>
#endif

#include <ngsocket.h>
#include <ngmdm.h>
#include "utils.h"
#include "debug.h"


// Function prototypes:
// --------------------
static void *StaticBufAlloc(Uint32, NGuint *);

#ifdef TR_USE_DHCP
static void DhcpCallbackProc(int, NGuint, NGdhcpclient *);
#endif


// Macro definitions:
// ------------------
#define	NG_MAX_STACK_MEM		400000	// 400KB static memory buffer.
#define NG_MAX_BUF					100			// Maximum number of message buffers.
#define NG_MAX_SOCK					8				// Maximum number of sockets.
#define	NG_MAX_DEVCBS				2				// Maximum number of simultaneous device control blocks.
#define NG_MAX_DEVIO_SIZE		2048		// Maximum size of the device I/O buffers.
#define NG_MAX_DHCP_OPTS		10			// Maximum number of DHCP options that can be stored.

#define CFG_GENERAL_OPTIONS																																									\
	NG_BUFO_MAX,								NG_CFG_INT(NG_MAX_BUF),																												\
	NG_BUFO_ALLOC_F,						NG_CFG_FNC(StaticBufAlloc),	/* Used to manually allocate stack memory.		*/	\
	/*NG_BUFO_FREE_F,						NG_CFG_FNC(YourFreeFunc),		// Used to free any manually allocated memory.*/	\
	NG_BUFO_HEADER_SIZE,				NG_CFG_INT(sizeof(NGetherhdr)),																								\
	NG_BUFO_DATA_SIZE,					NG_CFG_INT(ETHERMTU),																													\
	NG_SOCKO_MAX,								NG_CFG_INT(NG_MAX_SOCK),																											\
	NG_SOCKO_TABLE,							NG_CFG_PTR(&g_trSocks),																												\
	NG_RTO_CLOCK_FREQ,					NG_CFG_INT(NG_CLOCK_FREQ),																										\
	NG_DEVCBO_TABLE,						NG_CFG_PTR(g_trDeviocb),																											\
	NG_DEVCBO_MAX,							NG_CFG_INT(NG_MAX_DEVCBS)

#define CFG_PROTOCOLS_OPTIONS																																								\
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_TCP),																											\
	NG_TCPO_TCB_MAX,						NG_CFG_INT(NG_MAX_SOCK),																											\
	NG_TCPO_TCB_TABLE,					NG_CFG_PTR(&g_trTcpcb),																												\
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_UDP),																											\
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_RAWIP),																										\
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_IP),																											\
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_PPP)

#define CFG_DEVICE_OPTIONS(device, driver)																																	\
	NG_CFG_DEVADD,							NG_CFG_PTR(&device),						/* Add a device object to the stack.			*/	\
	NG_CFG_DRIVER,							NG_CFG_PTR(&driver),						/* Add a device driver to the stack.			*/	\
	NG_DEVO_BAUDS,							NG_CFG_LNG(115200UL),						/* 115,200 bps baud rate (recommended).		*/	\
  NG_DEVO_CLKBASE,						NG_CFG_LNG(SH4_PPHI_FREQ),			/* Peripheral clock frequency (50Hz).			*/	\
	NG_DEVO_IBUF_PTR,						NG_CFG_PTR(&g_trDevIbuf),				/* Device input buffer.										*/	\
	NG_DEVO_OBUF_PTR,						NG_CFG_PTR(&g_trDevObuf),				/* Device output buffer.									*/	\
	NG_DEVO_IBUF_LEN,						NG_CFG_INT(sizeof(g_trDevIbuf)),																							\
	NG_DEVO_OBUF_LEN,						NG_CFG_INT(sizeof(g_trDevObuf))

#define CFG_INTERFACE_OPTIONS(ifnet, name)																																	\
	NG_CFG_IFADDWAIT,						NG_CFG_PTR(&ifnet),							/* Add a network interface object.				*/	\
	NG_CFG_DRIVER,							NG_CFG_PTR(&ngNetDrv_AHDLC),		/* Add the network interface driver.			*/	\
  NG_IFO_NAME,                NG_CFG_PTR(name)            		/* Add a name for the interface.     			*/

#define CFG_PPP_OPTIONS(device)																																							\
	NG_APPPIFO_DEVICE,					NG_CFG_PTR(&device),						/* Add the device to the PPP interface.		*/	\
	NG_PPPIFO_DEFAULT_ROUTE,		NG_CFG_TRUE,																																	\
  NG_PPPIFO_LCP_ASYNC,        NG_CFG_FALSE,										/* Disable ACCM table (esc. char. codes).	*/	\
	NG_PPPIFO_LCP_PAP,					NG_CFG_TRUE,										/* Enable simple PAP authentication.			*/	\
	NG_PPPIFO_LCP_CHAP,					NG_CFG_TRUE,										/* Enable secure CHAP authentication.			*/	\
	NG_PPPIFO_IPCP_VJCOMP,			NG_CFG_TRUE,										/* Enable Van Jacobson  compression.			*/	\
	NG_PPPIFO_IPCP_VJTABLE,			NG_CFG_PTR(&g_trVjTable),																											\
	NG_PPPIFO_IPCP_VJMAX,				NG_CFG_INT(sizeof(g_trVjTable) / sizeof(g_trVjTable[0])),											\
  NG_PPPIFO_IPCP_DNS1,        NG_CFG_TRUE,										/* Enable DNS resolvers (2 DNS servers).	*/	\
  NG_PPPIFO_IPCP_DNS2,        NG_CFG_TRUE


// Private global variables:
// -------------------------
static NGubyte			g_trDevIbuf[NG_MAX_DEVIO_SIZE];	// Device input buffer.
static NGubyte			g_trDevObuf[NG_MAX_DEVIO_SIZE];	// Device output buffer.

#ifdef DEBUG
static NGubyte			g_trSerDbgIbuf[NG_MAX_DEVIO_SIZE];	// Input buffer for serial debug device.
static NGubyte			g_trSerDbgObuf[NG_MAX_DEVIO_SIZE];	// Output buffer for serial debug device.
#endif

static NGdevcb			g_trDeviocb[NG_MAX_DEVCBS];			// Device control block structure (for one device only).
static NGsock				g_trSocks[NG_MAX_SOCK];					// Sockets.
static NGtcpcb			g_trTcpcb[NG_MAX_SOCK];					// TCP protocol control blocks.
static NGpppvjent		g_trVjTable[32];								// Van Jacobson compression table (MUST be 32 bytes).
static NGarpent			g_trArpTable[10];								// ARP protocol table.

#ifdef TR_USE_DHCP
static NGdhcpclient	g_trDhcpc;											// DHCP client protocol.
#endif


// Public global variables:
// ------------------------
#ifdef TR_USE_DHCP
NGdhcpoption		g_trDhcpOpts[NG_MAX_DHCP_OPTS];	// Required DHCP options (will be set at run-time).
NGuint					g_trDhcpOptsMask;								// Used to check if all DHCP options have been received.
NGuint					g_trDhcpUp;											// 'TRUE' when the DHCP interface is active.
#endif

#ifdef TR_USE_PPPOE
NGpppoeifnet	g_trPppoeIfnet;				// PPPoE interface on DC LAN.
#endif

#ifdef DEBUG
NGdev					g_trSerDbgDev;				// Serial device used for debug output.
#endif

NGdev					g_trDev;				      // Loaded device.
NGdevcb				*g_trDevcb;			      // Pointer to the device control block structure.
NGethifnet		g_trDhcpIfnet;				// DHCP interface on DC LAN (needed for default ethernet set-up).
NGapppifnet		g_trPppIfnet;					// PPP network interface configuration.
NGifnet				*g_trIfnet;			      // Pointer to the loaded interface.
NGmdmstate		g_trMdmstate;		      // Modem script state.
int						g_trTransport;	      // Transport method currently in use.
int						g_trDisconnectStatus;	// Last disconnection status.




// *********************************************************************************************** //
// **** NOTE : The groups of options in the arrays below MUST be declared in the order shown! **** //
// *********************************************************************************************** //
// ----------------------------------- //
//  Serial Device Stack Configuration  //
// ----------------------------------- //
NGcfgent g_trSerDevStackCfg[] = {	// This includes both external modems and direct serial connections.
	CFG_GENERAL_OPTIONS,
	CFG_PROTOCOLS_OPTIONS,
	CFG_DEVICE_OPTIONS(g_trDev, ngSerDrv_SH4SCIF),
	CFG_INTERFACE_OPTIONS(g_trPppIfnet, "Serial"),
	CFG_PPP_OPTIONS(g_trDev),
	NG_CFG_END
};


// ------------------------------------ //
//  Internal Modem Stack Configuration  //
// ------------------------------------ //
NGcfgent g_trIntMdmStackCfg[] = {
  CFG_GENERAL_OPTIONS,
  CFG_PROTOCOLS_OPTIONS,
  CFG_DEVICE_OPTIONS(g_trDev, ngSerDrv_Trisignal),

#ifdef DEBUG	// Used to set-up the serial port for debug output...
  NG_CFG_DEVADD,			NG_CFG_PTR(&g_trSerDbgDev),
  NG_CFG_DRIVER,			NG_CFG_PTR(&ngSerDrv_SH4SCIF),
  NG_DEVO_BAUDS,			NG_CFG_LNG(115200UL),
  NG_DEVO_CLKBASE,		NG_CFG_LNG(SH4_PPHI_FREQ),
  NG_DEVO_IBUF_PTR,		NG_CFG_PTR(&g_trSerDbgIbuf),
  NG_DEVO_IBUF_LEN,		NG_CFG_INT(sizeof(g_trSerDbgIbuf)),
  NG_DEVO_OBUF_PTR,		NG_CFG_PTR(&g_trSerDbgObuf),
  NG_DEVO_OBUF_LEN,		NG_CFG_INT(sizeof(g_trSerDbgObuf)),
#endif

  CFG_INTERFACE_OPTIONS(g_trPppIfnet, "Internal Modem"),
  CFG_PPP_OPTIONS(g_trDev),
	NG_PPPIFO_MODEM,	NG_CFG_TRUE,	// This is only needed by the internal modem to detect hardware disconnection.
	NG_CFG_END
};




// ********************************************************** //
// **** NOTE : This is a DC LAN card specific structure. **** //
// ********************************************************** //
NGcfgent g_trDCLanStackCfg[] = {
// Configure the general options (must come first!)...
	NG_BUFO_MAX,								NG_CFG_INT(NG_MAX_BUF),
	NG_BUFO_ALLOC_F,						NG_CFG_FNC(StaticBufAlloc),
	NG_BUFO_HEADER_SIZE,				NG_CFG_INT(sizeof(NGetherhdr) + 30),	// This is very important for the Dreamcast DC LAN Driver (DMA padding)!
	NG_BUFO_DATA_SIZE,					NG_CFG_INT(ETHERMTU + 31),						// Same here as the DMA controller might write before and after packet for performance reasons ;)
	NG_BUFO_INPQ_MAX,						NG_CFG_INT(8),
	NG_SOCKO_MAX,								NG_CFG_INT(NG_MAX_SOCK),
	NG_SOCKO_TABLE,							NG_CFG_PTR(&g_trSocks),
	NG_RTO_CLOCK_FREQ,					NG_CFG_INT(NG_CLOCK_FREQ),

// Configure protocol options...
  CFG_PROTOCOLS_OPTIONS,			// Macro from above.
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_ARP),
	NG_ARPO_MAX,								NG_CFG_INT(sizeof(g_trArpTable) / sizeof(g_trArpTable[0])),
	NG_ARPO_TABLE,							NG_CFG_PTR(&g_trArpTable),
	NG_ARPO_EXPIRE,							NG_CFG_INT(600),	// Time of expiration for a resolved ARP enquiry (ms?).

// Configure the interface...
	NG_CFG_IFADD,								NG_CFG_PTR(&g_trDhcpIfnet),
	NG_CFG_DRIVER,							NG_CFG_PTR(&ngNetDrv_DCLAN),
	NG_IFO_NAME,							  NG_CFG_PTR("DC LAN DHCP"),
	NG_ETHIFO_DEV1,							NG_CFG_INT(DCLAN_AUTO),	// Sets the speed of the DC LAN card (automatic).
															// (Other values : DCLAN_10BaseT, DCLAN_10BaseTX, DCLAN_100BaseT, DCLAN_100BaseTX)
															// In the current version of the driver, only 10BaseT and 100BaseTX are supported.
#ifdef TR_USE_DHCP
// Configure DHCP options...
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_DHCPC),
	NG_DHCPCO_CLIENTADD,				NG_CFG_PTR(&g_trDhcpc),
	NG_DHCPCO_IF,								NG_CFG_PTR(&g_trDhcpIfnet),
	NG_DHCPCO_MODE,							NG_CFG_INT(DHCPC_MATCHFIRST),
	NG_DHCPCO_CALLBACK,					NG_CFG_PTR(DhcpCallbackProc),
	NG_DHCPCO_OPTLIST,					NG_CFG_PTR(g_trDhcpOpts),
#endif	// TR_USE_DHCP

#ifdef TR_USE_PPPOE
// Add the PPPoE protocol...
	NG_CFG_PROTOADD,						NG_CFG_PTR(&ngProto_PPPOE),

// Configure the PPPoE interface...
	NG_CFG_IFADDWAIT,						NG_CFG_PTR(&g_trPppoeIfnet),
	NG_CFG_DRIVER,							NG_CFG_PTR(&ngNetDrv_PPPOE),
	NG_IFO_NAME,								NG_CFG_PTR("DC LAN PPPoE"),
	NG_IFO_OUTQ_MAX,						NG_CFG_INT(6),
	NG_PPPOEIFO_ETHIF,					NG_CFG_PTR(&g_trDhcpIfnet),
	NG_PPPOEIFO_MAXERR,					NG_CFG_INT(5),
	NG_PPPOEIFO_TIMEOUT,				NG_CFG_INT(25),

	NG_PPPIFO_DEFAULT_ROUTE,		NG_CFG_TRUE,
	NG_PPPIFO_PASSIVE,					NG_CFG_FALSE,
	NG_PPPIFO_SILENT,						NG_CFG_FALSE,
	NG_PPPIFO_RESTART,					NG_CFG_FALSE,
	NG_PPPIFO_LCP_ASYNC,				NG_CFG_FALSE,
	NG_PPPIFO_LCP_ACOMP,				NG_CFG_FALSE,	// Required by PPPoE RFC.
	NG_PPPIFO_LCP_PCOMP,				NG_CFG_FALSE,	// Recommended by PPPoE RFC.
	NG_PPPIFO_LCP_PAP,					NG_CFG_TRUE,
	NG_PPPIFO_LCP_CHAP,					NG_CFG_TRUE,
	NG_PPPIFO_IPCP_VJCOMP,			NG_CFG_TRUE,
	NG_PPPIFO_IPCP_VJTABLE,			NG_CFG_PTR(&g_trVjTable),																												
	NG_PPPIFO_IPCP_VJMAX,				NG_CFG_INT(sizeof(g_trVjTable) / sizeof(g_trVjTable[0])),
	NG_PPPIFO_IPCP_DNS1,				NG_CFG_TRUE,
	NG_PPPIFO_IPCP_DNS2,				NG_CFG_TRUE,
#endif	// TR_USE_PPPOE

	NG_CFG_END
};




// ------------------ //
//  StaticBufAlloc()  //
// ------------------ //
static void *StaticBufAlloc(Uint32 size, NGuint *phaddr)
{
	static char buf[NG_MAX_STACK_MEM];	// Static memory buffer for the stack.

// Check that there isn't more memory being requested than is available...
	if(size > NG_MAX_STACK_MEM) {
		DEBUG_PRINT(("Buffer allocation requested (%d bytes) exceeds capacity (%d bytes).\n", size, NG_MAX_STACK_MEM));
		return NULL;
	}

// Success!
	return buf;
}




#ifdef TR_USE_DHCP
// -------------------- //
//  DhcpCallbackProc()  //
// -------------------- //
static void DhcpCallbackProc(int event, NGuint incoming_mask, NGdhcpclient *dcp)
{
	NGdhcpoption	*p;
	NGuint				bit, mask;

// Process the DHCP event...
	switch(event) {
	case UP:
		g_trDhcpOptsMask |= (1 << 31);
		break;

	case DOWN:
		g_trDhcpOptsMask &= ~(1 << 31);
		break;
	}

	g_trDhcpOptsMask |= incoming_mask;

// Scan all options that request an answer...
	for(bit = 1, mask = (1 << 31), p = g_trDhcpOpts; bit; bit <<= 1, p++) {
		switch(p->do_tag) {
		case TAG_END:
			bit = 0;
			break;
	
		case TAG_HOST_NAME:
		case TAG_GATEWAY:
		case TAG_DOMAIN_SERVER:
			break;
	
		default:
			mask |= bit;
		}
	}

	g_trDhcpUp = ((g_trDhcpOptsMask & mask) == mask);
}
#endif	// TR_USE_DHCP
