// GameSpy Peer SDK C Test App
// Dan "Mr. Pants" Schoenblum
// dan@gamespy.com

/*************
** INCLUDES **
*************/
#include <stdlib.h>
#include <stdio.h>
#if defined(applec) || defined(THINK_C) || defined(__MWERKS__) && !defined(__KATANA__) && !defined(__mips64)
	#include "::peer.h"
#else
	#include "../peer.h"
#endif

#ifdef __KATANA__
void NUMsg(char *in_format, ...);
#define printf NUMsg
#endif


#ifdef UNDER_CE
void RetailOutputA(CHAR *tszErr, ...);
#define printf RetailOutputA
#endif

char RtoS[][16] = 
{
	"TitleRoom",
	"GroupRoom",
	"StagingRoom"
};

/*************
** SETTINGS **
*************/
char nick[256];
int profileID;
char title[256];
char qrSecretKey[256];
char engineName[256];
char engineSecretKey[256];
int engineMaxUpdates;

PEERBool stop;

/**************
** FUNCTIONS **
**************/
void DisconnectedCallback
(
	PEER peer,
	const char * reason,
	void * param
)
{
	printf("Disconnected: %s\n", reason);
}

void RoomMessageCallback
(
	PEER peer,
	RoomType roomType,
	const char * nick,
	const char * message,
	MessageType messageType,
	void * param
)
{
	printf("(%s) %s: %s\n", RtoS[roomType], nick, message);
	if(strcasecmp(message, "quit") == 0)
		stop = PEERTrue;
	else if((strlen(message) > 5) && (strncasecmp(message, "echo", 4) == 0))
		peerMessageRoom(peer, roomType, message + 5, messageType);
}

void PlayerMessageCallback
(
	PEER peer,
	const char * nick,
	const char * message,
	MessageType messageType,
	void * param
)
{
	printf("(PRIVATE) %s: %s\n", nick, message);
}

void ReadyChangedCallback
(
	PEER peer,
	const char * nick,
	PEERBool ready,
	void * param
)
{
	if(ready)
		printf("%s is ready\n", nick);
	else
		printf("%s is not ready\n", nick);
}

void GameStartedCallback
(
	PEER peer,
	unsigned int IP,
	const char * message,
	void * param
)
{
	IN_ADDR addr;
	addr.s_addr = IP;
	printf("The game is starting at %s\n", inet_ntoa(addr));
	if(message && message[0])
		printf(": %s\n", message);
	else
		printf("\n");
}

void PlayerJoinedCallback
(
	PEER peer,
	RoomType roomType,
	const char * nick,
	void * param
)
{
	printf("%s joined the %s\n", nick, RtoS[roomType]);
}

void PlayerLeftCallback
(
	PEER peer,
	RoomType roomType,
	const char * nick,
	const char * reason,
	void * param
)
{
	printf("%s left the %s\n", nick, RtoS[roomType]);
}

void PlayerChangedNickCallback
(
	PEER peer,
	RoomType roomType,
	const char * oldNick,
	const char * newNick,
	void * param
)
{
	printf("%s in %s changed nicks to %s\n", oldNick, RtoS[roomType], newNick);
}

PEERBool connectSuccess;
void ConnectCallback
(
	PEER peer,
	PEERBool success,
	void * param
)
{
	connectSuccess = success;
}

void EnumPlayersCallback
(
	PEER peer,
	PEERBool success,
	RoomType roomType,
	int index,
	const char * nick,
	int flags,
	void * param
)
{
	if(!success)
	{
		printf("Enum %s players failed\n", RtoS[roomType]);
		return;
	}

	if(index == -1)
	{
		printf("--------------------\n");
		return;
	}

	printf("%d: %s", index, nick);
	if(flags & PEER_FLAG_OP)
		printf(" (host)\n");
	else
		printf("\n");
}

static void ListingGamesCallback
(
	PEER peer,
	PEERBool success,
	const char * name,
	GServer server,
	PEERBool staging,
	int msg,
	int progress,
	void * param
)
{
	if(success)
	{
		char *msgname;
		switch (msg)
		{
		case PEER_CLEAR:
			msgname = "PEER_CLEAR";
			break;
		case PEER_ADD:
			msgname = "PEER_ADD";
			break;
		case PEER_UPDATE:
			msgname = "PEER_UPDATE";
			break;
		case PEER_REMOVE:
			msgname = "PEER_REMOVE";
			break;
		default:
			msgname = "ERROR!";
		}
		printf("game: %s\n", msgname);
		if(server)
			printf("  server name: %s numplayers: %d ping: %d\n", ServerGetStringValue(server,"hostname","UNKNOWN"), ServerGetIntValue(server,"numplayers",0), ServerGetPing(server));
	}

}

void ListGroupRoomsCallback
(
	PEER peer,
	PEERBool success,
	int groupID,
	GServer server,
	const char * name,
	int numWaiting,
	int maxWaiting,
	int numGames,
	int numPlaying,
	void * param
)
{
	if(success && (groupID > 0))
	{
		printf("group: %s\n", name);
	}
}

PEERBool joinSuccess;
void JoinCallback
(
	PEER peer,
	PEERBool success,
	PEERJoinResult result,
	RoomType roomType,
	void * param
)
{
	joinSuccess = success;

	if(success)
	{
		printf("Listing %s players:\n", RtoS[roomType]);
		peerEnumPlayers(peer, roomType, EnumPlayersCallback, NULL);
	}
}

void NickErrorCallback
(
	PEER peer,
	int type,
	const char * badNick,
	void * param
)
{
	static int errCount;

	if(errCount < 20)
		sprintf(nick,"peerc%u",(unsigned int)current_time());
	else
		nick[0] = '\0';
	errCount++;

	peerRetryWithNick(peer, nick);
}

#if defined(__KATANA__) || defined(__mips64)
int test_main(int argc, char **argv)
#else
int main(int argc, char **argv)
#endif
{
	PEER peer;
	PEERCallbacks callbacks;
	unsigned long stopTime;
	PEERBool pingRooms[NumRooms];
	PEERBool crossPingRooms[NumRooms];

	// Defaults.
	////////////
	sprintf(nick,"peerc%u",(unsigned int)current_time());
	profileID = 0;
	strcpy(title, "gmtest");
	strcpy(qrSecretKey, "HA6zkS");
	strcpy(engineName, "gmtest");
	strcpy(engineSecretKey, "HA6zkS");
	engineMaxUpdates = 10;

	// Setup the callbacks.
	///////////////////////
	memset(&callbacks, 0, sizeof(PEERCallbacks));
	callbacks.disconnected = DisconnectedCallback;
	callbacks.gameStarted = GameStartedCallback;
	callbacks.playerChangedNick = PlayerChangedNickCallback;
	callbacks.playerJoined = PlayerJoinedCallback;
	callbacks.playerLeft = PlayerLeftCallback;
	callbacks.readyChanged = ReadyChangedCallback;
	callbacks.roomMessage = RoomMessageCallback;
	callbacks.playerMessage = PlayerMessageCallback;
	callbacks.param = NULL;

	// Init.
	////////
	peer = peerInitialize(&callbacks);
	if(!peer)
	{
		printf("Failed to init\n");
		return 1;
	}

	// Ping/cross-ping in every room.
	/////////////////////////////////
	pingRooms[TitleRoom] = PEERTrue;
	pingRooms[GroupRoom] = PEERTrue;
	pingRooms[StagingRoom] = PEERTrue;
	crossPingRooms[TitleRoom] = PEERTrue;
	crossPingRooms[GroupRoom] = PEERTrue;
	crossPingRooms[StagingRoom] = PEERTrue;

	// Set the title.
	/////////////////
	if(!peerSetTitle(peer, title, qrSecretKey, engineName, engineSecretKey, engineMaxUpdates, pingRooms, crossPingRooms))
	{
		peerShutdown(peer);
		printf("Failed to set the title\n");
		return 1;
	}

	// Connect.
	///////////
	peerConnect(peer, nick, profileID, NickErrorCallback, ConnectCallback, NULL, PEERTrue);
	if(!connectSuccess)
	{
		peerShutdown(peer);
		printf("Failed to connect\n");
		return 1;
	}

	// Join the title room.
	///////////////////////
	peerJoinTitleRoom(peer, JoinCallback, NULL, PEERTrue);
	if(!joinSuccess)
	{
		peerDisconnect(peer);
		peerShutdown(peer);
		printf("Failed to join the title room\n");
		return 1;
	}

	peerListGroupRooms(peer, ListGroupRoomsCallback, NULL, PEERFalse);

	peerStartListingGames(peer, NULL, ListingGamesCallback, NULL);

	// Loop for a while.
	////////////////////
	stopTime = (current_time() + 120000);
	do
	{
		peerThink(peer);

		msleep(1);
	}
	while((current_time() < stopTime) && !stop);

	peerStopListingGames(peer);

	// Leave the title room.
	////////////////////////
	peerLeaveRoom(peer, TitleRoom, NULL);

	// Disconnect.
	//////////////
	peerDisconnect(peer);

	// Shutdown.
	////////////
	peerShutdown(peer);
	printf("All done!\n");
	return 0;
}