/******
mwinsock.c
GameSpy Mac Sockets Library
  
Copyright 1999 GameSpy Industries, Inc

Suite E-204
2900 Bristol Street
Costa Mesa, CA 92626
(714)549-7689
Fax(714)549-0757

******/

/*
If you get a negative return value, it means the error wasn't translated
to a winsock error correctly. Please report the error # and call so that
it can be added! */
#include "mwinsock.h"
#include <OTDebug.h>
#include <Gestalt.h>
#include <time.h>

#define DOYIELD
#ifdef DOYIELD
	#include <Threads.h>
#endif

typedef struct socket_s
{
	EndpointRef ep;
	InetAddress *pbindaddress;
	TCall *pcall;
	int protocol;


} socket_t;


#define MAX_SOCKETS 1024
static InetSvcRef g_inet_services = nil;
static int g_wsarefct = 0;
static int g_lasterror = 0;
static socket_t *g_sockets[MAX_SOCKETS];
static Boolean gHaveIPSingleLinkMultihoming;
enum
{
	kOTIPSingleLinkMultihomingVersion = 0x01300000
};

//check for common problems and clear the error flag
#define DEFAULTCHECKS 	if (g_wsarefct == 0) return SetError(WSANOTINITIALISED); if (s >= MAX_SOCKETS || g_sockets[s] == NULL) return SetError(WSAENOTSOCK); g_lasterror = 0;

#ifdef DOYIELD

static pascal void YieldingNotifier(void *fs, OTEventCode code, OTResult result, void* cookie)
	// This simple notifier checks for kOTSyncIdleEvent and
	// when it gets one calls the Thread Manager routine
	// YieldToAnyThread.  Open Transport sends kOTSyncIdleEvent
	// whenever it's waiting for something, eg data to arrive
	// inside a sync/blocking OTRcv call.  In such cases, we
	// yield the processor to some other thread that might
	// be doing useful work.
	//
{
	#pragma unused(result)
	#pragma unused(cookie)
	#pragma unused(fs)

	
	switch (code) {
		case kOTSyncIdleEvent:
			YieldToAnyThread();
			break;
		default:
			break;
	}
}
#endif

int PASCAL FAR WSAStartup(WORD wVersionRequired, LPWSADATA lpWSAData)
{	
	OSStatus err;
	NumVersionVariant otVersion;
	
	if (wVersionRequired > 0x101)
		return WSAVERNOTSUPPORTED;
	lpWSAData->wVersion = 0x101;
	lpWSAData->wHighVersion = 0x101;
	OTStrCopy(lpWSAData->szDescription,"crt mwinsock 0.1");
	OTStrCopy(lpWSAData->szSystemStatus,"rocking");
	lpWSAData->iMaxSockets = MAX_SOCKETS;
	lpWSAData->iMaxUdpDg = 65536;
	lpWSAData->lpVendorInfo = "Copyright 1999, GameSpy Industries";
	
	if (g_wsarefct++ != 0)
		return 0;
	err = InitOpenTransport();
	
	if (err == noErr) 
	{
		g_inet_services = OTOpenInternetServices(kDefaultInternetServicesPath, 0, &err);
		if (err == noErr)
		{
			gHaveIPSingleLinkMultihoming = ( Gestalt(gestaltOpenTptVersions, (long *) &otVersion) == noErr
											&& (otVersion.whole >= kOTIPSingleLinkMultihomingVersion ) );
			return 0; //init success
		}
		
	}
	OTMemzero(g_sockets, sizeof(g_sockets[0]) * MAX_SOCKETS);
	return WSASYSNOTREADY;
}


int PASCAL FAR WSACleanup(void)
{
	int i;
	
	if (--g_wsarefct != 0)
		return 0;
	//close all remaining sockets
	
	for (i = 0 ; i < MAX_SOCKETS ; i++)
		if (g_sockets[i] != NULL)
			closesocket(i);

	if (g_inet_services != nil) 
	{
		OTCloseProvider(g_inet_services);
	}

	CloseOpenTransport();
	return 0;
}

int PASCAL FAR WSAGetLastError(void)
{
	return g_lasterror;
}

static void freesocket(SOCKET socket)
{
	if (g_sockets[socket] != NULL)
		OTFreeMem(g_sockets[socket]);
	g_sockets[socket] = NULL;				
}

static SOCKET SetError(int error)
{
	g_lasterror = error;
	return INVALID_SOCKET;
}

SOCKET PASCAL FAR socket (int af, int type, int protocol)
{
	int sock;
	OSStatus err;
	
	if (g_wsarefct == 0)
		return SetError(WSANOTINITIALISED);
		
	if (af != AF_INET) //only supported
		return SetError(WSAEAFNOSUPPORT);
	if (type != SOCK_STREAM && type != SOCK_DGRAM)
		return SetError(WSAESOCKTNOSUPPORT);
	
	sock = 0;
	while (g_sockets[sock] != NULL && sock < MAX_SOCKETS)
		sock++;
	if (sock == MAX_SOCKETS)
		return SetError(WSAEMFILE);
		
	//init the structure	
	g_sockets[sock] = OTAllocMem(sizeof(socket_t));
	if (!g_sockets[sock])
		return SetError(WSAENOBUFS);

	g_sockets[sock]->pbindaddress = NULL;
	g_sockets[sock]->pcall = NULL;
	
	if (protocol == IPPROTO_IP) // try to pick the right one!!
		protocol = (type == SOCK_STREAM) ? IPPROTO_TCP : IPPROTO_UDP;
		
	switch (protocol)
	{
		case IPPROTO_TCP:
					g_sockets[sock]->ep = OTOpenEndpoint(OTCreateConfiguration(kTCPName), 0, nil, &err);
				break;
		case IPPROTO_UDP:
					g_sockets[sock]->ep = OTOpenEndpoint(OTCreateConfiguration(kUDPName), 0, nil, &err);
				break;
		default:
				freesocket(sock);
				return SetError(WSAEPROTONOSUPPORT);
	}
	g_sockets[sock]->protocol = protocol;
	if (err != noErr)
	{
		freesocket(sock);
		return SetError(WSAENETDOWN);
	}
	
	OTSetBlocking(g_sockets[sock]->ep);
	OTSetSynchronous(g_sockets[sock]->ep);
	
//	DoNegotiateIPReuseAddrOption(g_sockets[sock]->ep, true);
#ifdef DOYIELD
	OTInstallNotifier(g_sockets[sock]->ep, &YieldingNotifier,nil);
	OTUseSyncIdleEvents(g_sockets[sock]->ep, true);
#endif
	return sock;

}


int PASCAL FAR closesocket (SOCKET s)
{
	DEFAULTCHECKS;

	if (g_sockets[s]->pcall != NULL)
	{
		if (g_sockets[s]->protocol == IPPROTO_TCP)
			OTSndDisconnect(g_sockets[s]->ep,g_sockets[s]->pcall);
		if (g_sockets[s]->pcall->addr.buf != NULL)
			OTFreeMem(g_sockets[s]->pcall->addr.buf);
		OTFreeMem(g_sockets[s]->pcall);
	}
	if (g_sockets[s]->pbindaddress != NULL)
	{
		OTUnbind(g_sockets[s]->ep);
		OTFreeMem(g_sockets[s]->pbindaddress);
	}
	OTCloseProvider(g_sockets[s]->ep);
	OTFreeMem(g_sockets[s]);
	g_sockets[s] = NULL;
	return 0;
}

int PASCAL FAR shutdown (SOCKET s, int how)
{
	#pragma unused(how)

	DEFAULTCHECKS;

	if (g_sockets[s]->pcall != NULL)
	{
		if (g_sockets[s]->protocol == IPPROTO_TCP)
		{
			OTResult err = OTSndOrderlyDisconnect(g_sockets[s]->ep);
			if (err == noErr)
				err = OTRcvOrderlyDisconnect(g_sockets[s]->ep);
		}
		if (g_sockets[s]->pcall->addr.buf != NULL)
			OTFreeMem(g_sockets[s]->pcall->addr.buf);
		OTFreeMem(g_sockets[s]->pcall);
		g_sockets[s]->pcall = NULL;

	} else
		if (g_sockets[s]->protocol == IPPROTO_TCP)
			return SetError(WSAENOTCONN);
	return 0;
}

static OSStatus BindAddress(EndpointRef ep, InetHost ipAddr, InetPort port,InetAddress *retaddr)
{
	TBind bindReq, bretaddr;
	InetAddress ipAddress;
	
	OTInitInetAddress(&ipAddress, port, ipAddr);
	bindReq.addr.buf = (UInt8 *) &ipAddress;
	bindReq.addr.len = sizeof(ipAddress);
	bindReq.qlen = 5; 
//note: we have to set the backlog queue here, instead of in listen!!
	if (retaddr != NULL)
	{
		bretaddr.addr.buf = (UInt8 *)retaddr;
		bretaddr.addr.maxlen = sizeof(InetAddress);
		return OTBind(ep, &bindReq, &bretaddr);
	}
	else
	 	return OTBind(ep, &bindReq, nil);

} 
static OSStatus BindInterface(EndpointRef ep, InetInterfaceInfo* interfaceInfo, SInt32 interfaceIndex,InetPort port,InetAddress *retaddr)
{
	OSStatus err;
	InetHost *secondaryAddressBuffer;
	UInt32   numberOfSecondaryAddresses;
	UInt32   addressIndex;

	secondaryAddressBuffer = nil;
	
	// First run the server for the interfaces primary address.
	
	err = BindAddress(ep,interfaceInfo->fAddress, port, retaddr);
	
	numberOfSecondaryAddresses = interfaceInfo->fIPSecondaryCount;
	
	if ( err == noErr && gHaveIPSingleLinkMultihoming && numberOfSecondaryAddresses > 0 ) 
	{

		// Allocate a buffer for the secondary address info.
		
		secondaryAddressBuffer = (InetHost *) OTAllocMem( numberOfSecondaryAddresses * sizeof(InetHost) );
		if (secondaryAddressBuffer == nil)
			err = kENOMEMErr;
		
		// Ask OT for the list of secondary addresses on this interface.
		
		if (err == noErr)
			err = OTInetGetSecondaryAddresses(secondaryAddressBuffer, &numberOfSecondaryAddresses, interfaceIndex);
		
		// Start a server for each secondary address.
		
		addressIndex = 0;
		while (err == noErr && addressIndex < numberOfSecondaryAddresses) 
		{
			err = BindAddress(ep, secondaryAddressBuffer[addressIndex], port, NULL);
			if (err == noErr)
				addressIndex++;
		}
	}

	// Clean up.
	
	if (secondaryAddressBuffer != nil)
		OTFreeMem(secondaryAddressBuffer);

	return (err);
}

static OSStatus BindAllInterfaces(EndpointRef ep, InetPort port,InetAddress *retaddr)
{
	OSStatus err;
	InetInterfaceInfo info;
	SInt32 interfaceIndex;
	Boolean done;
	
	done = false;
	interfaceIndex = 0; 
	do {
		done = ( OTInetGetInterfaceInfo(&info, interfaceIndex) != noErr );
		if ( ! done ) {
			err = BindInterface(ep, &info, interfaceIndex, port, retaddr);
			interfaceIndex += 1;
		}
	} while (err == noErr && !done);
	return (err);
}

int PASCAL FAR bind (SOCKET s, const struct sockaddr FAR *addr, int namelen)
{
	#pragma unused(namelen)
	OSStatus err;
	struct sockaddr_in *saddr = (struct sockaddr_in *)addr;
	
	DEFAULTCHECKS;
	if (g_sockets[s]->pbindaddress != NULL)
		return SetError(WSAEINVAL);
	g_sockets[s]->pbindaddress = OTAllocMem(sizeof(InetAddress));
	if (!g_sockets[s]->pbindaddress)
		return SetError(WSAENOBUFS);
	if (saddr->sin_addr.S_un.S_addr == INADDR_ANY)
		err = BindAllInterfaces(g_sockets[s]->ep, saddr->sin_port, g_sockets[s]->pbindaddress);
	else
		err = BindAddress(g_sockets[s]->ep, saddr->sin_addr.S_un.S_addr, saddr->sin_port, g_sockets[s]->pbindaddress);
	
	if (err == kOTAddressBusyErr || err == kOTNoAddressErr)
		return SetError(WSAEADDRINUSE);
	if (err != noErr)
		return SetError(err);
	return 0;	
}

int PASCAL FAR listen (SOCKET s, int backlog)
{
	#pragma unused(backlog)
	//we don't need to do anything here -- bound sockets automatically listen
	DEFAULTCHECKS;
	//should we check for a UDP socket here?
	return 0;
}

SOCKET PASCAL FAR accept (SOCKET s, struct sockaddr FAR *addr, int FAR *addrlen)
{
	TCall call;
	InetAddress inaddr;
	OSStatus err;
	SOCKET snew;
	struct sockaddr_in *saddr = (struct sockaddr_in *)addr;
	
	DEFAULTCHECKS;
	//should we check for a UDP socket here?
	if (addrlen != NULL)
	{
		if (*addrlen < sizeof(struct sockaddr_in))
			return SetError(WSAEFAULT);
		*addrlen = sizeof(struct sockaddr_in);
	}
	OTMemzero(&call, sizeof(TCall));
	call.addr.buf = (UInt8 *)&inaddr;
	call.addr.maxlen = sizeof(InetAddress);
	
	err = OTListen(g_sockets[s]->ep,&call);
	if (err != noErr)
	{
		if (err == kOTNoDataErr || err == kEAGAINErr || err == kEWOULDBLOCKErr)
			return SetError(WSAEWOULDBLOCK);
		else
			return SetError(err);
	}	
	
	snew = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP); //this only works with this type of socket
	if (snew == INVALID_SOCKET)
		return SetError(WSAENOBUFS);
		
	g_sockets[snew]->pcall = OTAllocMem(sizeof(TCall));
	if (!g_sockets[snew]->pcall)
		return SetError(WSAENOBUFS);
	OTMemzero(g_sockets[snew]->pcall, sizeof(TCall));
	g_sockets[snew]->pcall->addr.buf = OTAllocMem(sizeof(InetAddress));
	if (!g_sockets[snew]->pcall->addr.buf)
		return SetError(WSAENOBUFS);
	g_sockets[snew]->pcall->addr.maxlen = sizeof(InetAddress);
	g_sockets[snew]->pcall->sequence = call.sequence;
	err = OTAccept(g_sockets[s]->ep, g_sockets[snew]->ep, g_sockets[snew]->pcall);
	if (err == kOTLookErr)
	{
		err = OTLook(g_sockets[s]->ep);
		if (err == T_LISTEN)
		{
			err = OTListen(g_sockets[s]->ep,g_sockets[snew]->pcall);
			err = OTAccept(g_sockets[s]->ep, g_sockets[snew]->ep, g_sockets[snew]->pcall);
		}
	
	}
	if (err != noErr)
		return SetError(err);
		
	if (!OTIsBlocking(g_sockets[s]->ep))
		OTSetNonBlocking(g_sockets[snew]->ep);

	if (addr != NULL)
	{
		saddr->sin_family = inaddr.fAddressType;
		saddr->sin_port = inaddr.fPort;
		saddr->sin_addr.S_un.S_addr = inaddr.fHost;
	}
	return snew;
}

int PASCAL FAR recv (SOCKET s, char FAR * buf, int len, int flags)
{
	OTResult err;
	OTFlags oflags = flags;
	int ret;
	
	DEFAULTCHECKS;
	
	if (g_sockets[s]->pcall == NULL) //they haven't called connect!
		return SetError(WSAENOTCONN);
	if (len < 1)
		return SetError(WSAEMSGSIZE);
		
	if (g_sockets[s]->protocol == IPPROTO_UDP)
	{
		struct sockaddr_in saddr;
		int saddrlen = sizeof(saddr);
		InetAddress *paddr = (InetAddress *)g_sockets[s]->pcall->addr.buf; 
		saddr.sin_family = AF_INET;
		saddr.sin_port = paddr->fPort;
		saddr.sin_addr.S_un.S_addr = paddr->fHost;
		return recvfrom(s,buf,len,flags,(struct sockaddr *)&saddr, &saddrlen);
	} 

	/* Stupid open transport tries to wait for LEN characters of data during the rcv
		we trick it by just waiting for the first char, then getting the length of the
		rest and getting just that ammount
		was: err = OTRcv(g_sockets[s]->ep, buf, len, &oflags);*/
	err = OTRcv(g_sockets[s]->ep, buf, 1, &oflags);	
	if (err > 0 && len > 1)
	{
		size_t newlen;
		err = OTCountDataBytes(g_sockets[s]->ep,&newlen);
		if (err == noErr)
		{
			if (newlen > len - 1)
				newlen = len - 1;
			err = OTRcv(g_sockets[s]->ep,buf+1,newlen,&oflags);
			if (err >= 0)
				err++; //for the first char!!
		}
	}
	if (err >= 0)
		return err;	
	if (err == kOTNoDataErr || err == kEAGAINErr || err == kEWOULDBLOCKErr)
		return SetError(WSAEWOULDBLOCK);
	if (err == kOTBufferOverflowErr)
		return SetError(WSAEMSGSIZE);
	if (err == kENETRESETErr)
		return SetError(WSAENETRESET);
	if (err == kECONNABORTEDErr)
		return SetError(WSAECONNABORTED);
	if (err == kECONNRESETErr)
		return SetError(WSAECONNRESET);
	if (err == kESHUTDOWNErr)
		return SetError(WSAESHUTDOWN);	
	if (err == kOTLookErr)
	{
		err = OTLook(g_sockets[s]->ep);
		if (err == T_DISCONNECT)
		{
			OTRcvDisconnect(g_sockets[s]->ep,NULL);
			return SetError(WSAECONNRESET);
		}
		if (err == T_ORDREL)
		{
			OTRcvOrderlyDisconnect(g_sockets[s]->ep);
			OTSndOrderlyDisconnect(g_sockets[s]->ep);
			return 0;
		}
	}
	return SetError(err);	
}

int PASCAL FAR recvfrom (SOCKET s, char FAR * buf, int len, int flags,
                         struct sockaddr FAR *from, int FAR * fromlen)
{
	TUnitData udata;
	InetAddress inaddr;
	OTFlags oflags = flags;
	OTResult err;
	struct sockaddr_in *saddr = (struct sockaddr_in *)from;

	DEFAULTCHECKS;
	//should we check for a UDP socket here?
	if (fromlen != NULL)
	{
		if (*fromlen < sizeof(struct sockaddr_in))
			return SetError(WSAEFAULT);
		*fromlen = sizeof(struct sockaddr_in);
	}
	
	OTMemzero(&udata, sizeof(TUnitData));
	udata.addr.buf = (UInt8 *)&inaddr;
	udata.addr.maxlen = sizeof(InetAddress);
	udata.udata.buf = (unsigned char *)buf;
	udata.udata.maxlen = len;
	
	err = OTRcvUData(g_sockets[s]->ep,&udata, &oflags);
	if (err == noErr) //got data!
	{
		if (saddr != NULL)
		{
			saddr->sin_family = inaddr.fAddressType;
			saddr->sin_port = inaddr.fPort;
			saddr->sin_addr.S_un.S_addr = inaddr.fHost;
		}
		if (oflags == T_MORE) 
		{ //all full, need to dump remainder of datagram to do what winsock would
			unsigned char *tempbuff;
			size_t newlen;
			err = OTCountDataBytes(g_sockets[s]->ep, &newlen);
			if (err != noErr) //just bail
				return SetError(WSAEMSGSIZE);
			tempbuff = OTAllocMem(newlen);
			if (!tempbuff)
				return SetError(WSAENOBUFS);
			oflags = 0;
			udata.udata.buf = tempbuff;
			udata.udata.maxlen = newlen;
			udata.udata.len = 0;
			err = OTRcvUData(g_sockets[s]->ep,&udata, &oflags);
			//we just discard the data
			OTFreeMem(tempbuff);
			return SetError(WSAEMSGSIZE);
		}
		else
			return udata.udata.len;
	}
	if (err == kOTNoDataErr || err == kEAGAINErr || err == kEWOULDBLOCKErr)
		return SetError(WSAEWOULDBLOCK);
	if (err == kOTBufferOverflowErr)
		return SetError(WSAEMSGSIZE);
	return SetError(err);	
}

static int bindlocalsocket(SOCKET s)
{
	int err;
	g_sockets[s]->pbindaddress = OTAllocMem(sizeof(InetAddress));
	if (!g_sockets[s]->pbindaddress)
		return SetError(WSAENOBUFS);
	err = BindAddress(g_sockets[s]->ep, 0, 0, g_sockets[s]->pbindaddress);
	return err;
}

int PASCAL FAR connect (SOCKET s, const struct sockaddr FAR *name, int namelen)
{
	#pragma unused(namelen)
	OTResult err;
	struct sockaddr_in *saddr = (struct sockaddr_in *)name;

	DEFAULTCHECKS;
	if (g_sockets[s]->pbindaddress == NULL) //we need to bind it!!
	{
		bindlocalsocket(s);
	}
	
	if (g_sockets[s]->pcall != NULL)
	{
		if (g_sockets[s]->protocol == IPPROTO_TCP)
			return SetError(WSAEISCONN);
		else //udp sock, free the old connect info
		{
			if (g_sockets[s]->pcall->addr.buf != NULL)
				OTFreeMem(g_sockets[s]->pcall->addr.buf);
			OTFreeMem(g_sockets[s]->pcall);
		}
	}
	g_sockets[s]->pcall = OTAllocMem(sizeof(TCall));
	if (!g_sockets[s]->pcall)
		return SetError(WSAENOBUFS);
	OTMemzero(g_sockets[s]->pcall, sizeof(TCall));
	g_sockets[s]->pcall->addr.buf = OTAllocMem(sizeof(InetAddress));
	if (!g_sockets[s]->pcall->addr.buf)
		return SetError(WSAENOBUFS);
	g_sockets[s]->pcall->addr.len = sizeof(InetAddress);
	OTInitInetAddress((InetAddress *)(g_sockets[s]->pcall->addr.buf), saddr->sin_port, saddr->sin_addr.S_un.S_addr);
	if (g_sockets[s]->protocol == IPPROTO_UDP)
		return 0; //no need to connect udp!
	err = OTConnect(g_sockets[s]->ep, g_sockets[s]->pcall, NULL);
	if (err == noErr)
		return 0;
	OTFreeMem(g_sockets[s]->pcall->addr.buf);
	OTFreeMem(g_sockets[s]->pcall);
	g_sockets[s]->pcall = NULL;
	if (err == kOTNoDataErr || err == kEAGAINErr || err == kEWOULDBLOCKErr)
		return SetError(WSAEWOULDBLOCK);
	if (err == kOTBadAddressErr)
		return SetError(WSAEADDRNOTAVAIL);
	if (err == kENETDOWNErr)
		return SetError(WSAENETDOWN);
	if (err == kECONNREFUSEDErr)
		return SetError(WSAECONNREFUSED);
	if (err == kENETUNREACHErr)
		return SetError(WSAENETUNREACH);
	if (err == kETIMEDOUTErr)
		return SetError(WSAETIMEDOUT);
	if (err == kOTLookErr)
	{
		err = OTLook(g_sockets[s]->ep);
		if (err == T_DISCONNECT)
		{
			OTRcvDisconnect(g_sockets[s]->ep,NULL);
			return SetError(WSAECONNREFUSED);
		}
	}
	return SetError(err);		
}

int PASCAL FAR send (SOCKET s, const char FAR * buf, int len, int flags)
{
	OTResult err;
	OTFlags oflags = flags;
	DEFAULTCHECKS;
	
	if (g_sockets[s]->pcall == NULL) //they haven't called connect!
		return SetError(WSAENOTCONN);
		
	if (g_sockets[s]->protocol == IPPROTO_UDP)
	{
		struct sockaddr_in saddr;
		int saddrlen = sizeof(saddr);
		InetAddress *paddr = (InetAddress *)g_sockets[s]->pcall->addr.buf; 
		saddr.sin_family = AF_INET;
		saddr.sin_port = paddr->fPort;
		saddr.sin_addr.S_un.S_addr = paddr->fHost;
		return sendto(s,buf,len,flags,(struct sockaddr *)&saddr, saddrlen);
	} 
	
	err = OTSnd(g_sockets[s]->ep, (void *)buf, len, 0); //we don't pass flags cause the value doesnt match to OT
	if (err >= 0)
		return err;
	if (err == kOTNoDataErr || err == kEAGAINErr || err == kEWOULDBLOCKErr || err == kOTFlowErr)
		return SetError(WSAEWOULDBLOCK);
	if (err == kENETRESETErr)
		return SetError(WSAENETRESET);
	if (err == kECONNABORTEDErr)
		return SetError(WSAECONNABORTED);
	if (err == kECONNRESETErr)
		return SetError(WSAECONNRESET);
	if (err == kESHUTDOWNErr)
		return SetError(WSAESHUTDOWN);	
	return SetError(err);	
}

int PASCAL FAR sendto (SOCKET s, const char FAR * buf, int len, int flags,
                       const struct sockaddr FAR *to, int tolen)
{
	#pragma unused(flags)
	TUnitData udata;
	InetAddress inaddr;
	OTResult err;
	struct sockaddr_in *saddr = (struct sockaddr_in *)to;

	DEFAULTCHECKS;
	//should we check for a UDP socket here?
	if (tolen < sizeof(struct sockaddr_in))
		return SetError(WSAEFAULT);
	
	if (g_sockets[s]->pbindaddress == NULL) //we need to bind it!!
	{
		bindlocalsocket(s);
	}
	
	OTInitInetAddress(&inaddr, saddr->sin_port, saddr->sin_addr.S_un.S_addr);
	OTMemzero(&udata, sizeof(TUnitData));
	udata.addr.buf = (UInt8 *)&inaddr;
	udata.addr.len = sizeof(InetAddress);
	udata.udata.buf = (unsigned char *)buf;
	udata.udata.len = len;
	
	err = OTSndUData(g_sockets[s]->ep,&udata);
	if (err >= 0)
		return err;
	if (err == kOTBadDataErr)
		return WSAEMSGSIZE;
	if (err == kOTNoDataErr || err == kEAGAINErr || err == kEWOULDBLOCKErr || err == kOTFlowErr)
		return SetError(WSAEWOULDBLOCK);
	if (err == kENETRESETErr)
		return SetError(WSAENETRESET);
	if (err == kECONNABORTEDErr)
		return SetError(WSAECONNABORTED);
	if (err == kECONNRESETErr)
		return SetError(WSAECONNRESET);
	if (err == kESHUTDOWNErr)
		return SetError(WSAESHUTDOWN);	
	return SetError(err);	
}

static int CheckFDSet(fd_set *set)
{
	int i;
	SOCKET s;
	if (set == NULL)
		return 0;
	for (i = 0 ; i < set->fd_count ; i++)
	{
		s = set->fd_array[i];
		DEFAULTCHECKS;
		if (g_sockets[s]->protocol == IPPROTO_TCP && g_sockets[s]->pcall == NULL && g_sockets[s]->pbindaddress == NULL)
			return SetError(WSAENOTCONN); //idiot check (not part of the spec, but helpful)
	}
	return 0;
}


int PASCAL FAR select (int nfds, fd_set FAR *readfds, fd_set FAR *writefds,
                       fd_set FAR *exceptfds, const struct timeval FAR *timeout)
{
	#pragma unused(nfds)
	time_t endtime;
	fd_set  wset, rset ,eset;
	int ntotal = 0;
	OTResult err;
	int i;

		//first, check that all the sockets are valid
	if (CheckFDSet(readfds) == SOCKET_ERROR || CheckFDSet(writefds) == SOCKET_ERROR || CheckFDSet(exceptfds) == SOCKET_ERROR)
		return SOCKET_ERROR;
	//copy all and except into a batch
	if (readfds != NULL)
	{
		OTMemcpy(&rset, readfds, sizeof(fd_set));
		readfds->fd_count = 0;
	} else
		rset.fd_count = 0;
	if (exceptfds != NULL)
	{
		OTMemcpy(&eset, exceptfds, sizeof(fd_set));
		exceptfds->fd_count = 0;
	} else
		eset.fd_count = 0;
	//copy write into another batch
	if (writefds != NULL)
	{
		OTMemcpy(&wset, writefds, sizeof(fd_set));
		writefds->fd_count = 0;
	}
	else
		wset.fd_count = 0;
		
	if (timeout == NULL)
		endtime = 0xFFFFFFFF;
	else
		endtime = time(NULL) + timeout->tv_sec; //we only use sec accuracy since we don't have a more accurate timer
	do
	{ //check for connects

		for (i = 0 ; i < wset.fd_count ; i++)
		{
			err = OTGetEndpointState(g_sockets[wset.fd_array[i]]->ep);
			if (err < 0)
				return SetError(err);
			if (err == T_DATAXFER)
			{
				FD_SET(wset.fd_array[i], writefds);
				ntotal++;
			}
		}
		for (i = 0 ; i < rset.fd_count ; i++)
		{
			err = OTLook(g_sockets[rset.fd_array[i]]->ep);
			if (err < 0)
				return SetError(err);
			if (err == 0)
				continue;
			if (err == T_CONNECT)
				OTRcvConnect(g_sockets[rset.fd_array[i]]->ep, NULL);
			if (err == T_DISCONNECT)
				OTRcvDisconnect(g_sockets[rset.fd_array[i]]->ep,NULL);
			ntotal++;
			FD_SET(rset.fd_array[i], readfds);				
		}
		for (i = 0 ; i < eset.fd_count ; i++)
		{
			err = OTLook(g_sockets[eset.fd_array[i]]->ep);
			if (err < 0)
				return SetError(err);
			if (err == T_DISCONNECT)
				OTRcvDisconnect(g_sockets[eset.fd_array[i]]->ep,NULL);
			if (err == T_EXDATA || err == T_DISCONNECT)
			{
				ntotal++;
				FD_SET(eset.fd_array[i], exceptfds);
			}				
		}
		YieldToAnyThread();
	} while (time(NULL) < endtime && ntotal == 0);
	return ntotal;
}

struct hostent FAR * PASCAL FAR gethostbyaddr(const char FAR * addr,
                                              int len, int type)
 {  
 	#pragma unused(len)
 	#pragma unused(type)
 	struct in_addr iaddr;
 	struct hostent *hent;
 	OTResult err;
 	int i;
 	
 	iaddr.S_un.S_addr = *(InetHost *)addr;
 	hent = gethostbyname(inet_ntoa(iaddr)); //fill in the rest of the struct
 	//now set the name if it did ok
 	if (hent == NULL)
 		return hent;
 	err = OTInetAddressToName(g_inet_services, *(InetHost *)addr, hent->h_name);
 	if (err != noErr)
	{
		g_lasterror = WSAHOST_NOT_FOUND;
		return NULL;
	}
	i = OTStrLength(hent->h_name);
	if (i > 0 && hent->h_name[i-1] == '.') //get rid of the OT trailing period
		hent->h_name[i-1] = '\0';
	return hent;
 }

struct hostent FAR * PASCAL FAR gethostbyname(const char FAR * name)
{
	static char *aliases = NULL;
	static InetHost *paddrs[kMaxHostAddrs];
	static InetHostInfo hinfo;
	static struct hostent hent;
	int i;
	OTResult err;

	err = OTInetStringToAddress(g_inet_services, (char *)name, &hinfo);
	if (err != noErr)
	{
		g_lasterror = WSAHOST_NOT_FOUND;
		return NULL;
	}
	hent.h_name = hinfo.name;
	i = OTStrLength(hent.h_name);
	if (i > 0 && hent.h_name[i-1] == '.') //get rid of the OT trailing period
		hent.h_name[i-1] = '\0';
	hent.h_aliases = &aliases;
	hent.h_addrtype = 2;
	hent.h_length = 4;
	
	for (i = 0 ; i < kMaxHostAddrs ; i++)
		if (hinfo.addrs[i] != 0)
			paddrs[i] = &hinfo.addrs[i];
		else
			paddrs[i] = NULL;
	hent.h_addr_list = (char **)paddrs;
	 
	return &hent;
}

//can't figure out how to get this on the mac??!
int PASCAL FAR gethostname (char FAR * name, int namelen)
{
	if (namelen < 10)
		return SetError(WSAEFAULT);
	OTStrCopy(name,"localhost");
	return 0;
}


int PASCAL FAR ioctlsocket (SOCKET s, long cmd, u_long FAR *argp)
{
	OTResult err;
	DEFAULTCHECKS;
	
	switch (cmd)
	{
		case FIONBIO:
			if (*argp == 0)
				err = OTSetBlocking(g_sockets[s]->ep);
			else
				err = OTSetNonBlocking(g_sockets[s]->ep);
		break;
		case FIONREAD:
				err = OTCountDataBytes(g_sockets[s]->ep, argp);
				if (err == kOTNoDataErr)
				{
					err = 0;
					*argp = 0;
				}
		break;
		case SIOCATMARK:
			*argp = 0; //we don't support OOB data (URGENT in OT terms)
			err = 0;
		break;
	
	}
	return (err == 0) ? 0 : SetError(err);
}



static int GetNameHelper (SOCKET s, struct sockaddr FAR *name,
                            int FAR * namelen, int forpeer)
{
	TBind bretaddr;
	InetAddress ipAddress;
	OTResult err;
	struct sockaddr_in *saddr = (struct sockaddr_in *)name;
	
	DEFAULTCHECKS;
	
	if (*namelen < sizeof(struct sockaddr_in))
		return SetError(WSAEFAULT);
	*namelen = sizeof(struct sockaddr_in);
	
	bretaddr.addr.buf = (UInt8 *)&ipAddress;
	bretaddr.addr.maxlen = sizeof(InetAddress);
	if (forpeer)
		err = OTGetProtAddress(g_sockets[s]->ep, NULL, &bretaddr);
	else
		err = OTGetProtAddress(g_sockets[s]->ep, &bretaddr, NULL);
	
	saddr->sin_family = AF_INET;
	saddr->sin_port = ipAddress.fPort;
	saddr->sin_addr.S_un.S_addr = ipAddress.fHost;
	if (err == 0)
	{
		if (bretaddr.addr.len == 0)
			return (forpeer) ? SetError(WSAENOTCONN) : SetError(WSAEINVAL);
		return 0;
	}	
	return SetError(err);
}

int PASCAL FAR getpeername (SOCKET s, struct sockaddr FAR *name,
                            int FAR * namelen)
{
	return GetNameHelper(s, name, namelen, 1);
}                   

int PASCAL FAR getsockname (SOCKET s, struct sockaddr FAR *name,
                            int FAR * namelen)
{
	return GetNameHelper(s, name, namelen, 0);
}

static int OptionHelper(SOCKET s, int level, int optname, char FAR * optval, int FAR *optlen, int set)
{
	TOption optionin, optionout;
	TOptMgmt req, ret;
	OTResult err;
	
	DEFAULTCHECKS;

	if (*optlen != sizeof(optionin.value[0])) //we don't support anything but ints for now
		return SetError(WSAENOPROTOOPT);
	switch (optname)
	{
		case SO_REUSEADDR:
		case SO_BROADCAST:
		case SO_DONTROUTE:
			level = INET_IP;
			break;
		case SO_KEEPALIVE:
		case SO_OOBINLINE:
			level = INET_TCP;
			break;
	}
	
	
	req.opt.buf = (unsigned char *)&optionin;
	req.opt.maxlen = sizeof(TOption);
	req.opt.len = sizeof(TOption);
	req.flags = (set) ? T_NEGOTIATE : T_CURRENT;
	ret.opt.buf = (unsigned char *)&optionout;
	ret.opt.maxlen = sizeof(TOption);
	ret.opt.len = sizeof(TOption);

	optionin.len = sizeof(TOption);
	optionin.level = level;
	optionin.name = optname;
	optionin.value[0] = *(UInt32 *)optval;
	
	err = OTOptionManagement(g_sockets[s]->ep, &req, &ret);
	if (err != 0)
		return SetError(WSAENOPROTOOPT);
	if (optionout.status == T_SUCCESS)
	{
		if (!set)
			 *(UInt32 *)optval = optionout.value[0]; 
		return 0;
	} else
		return SetError(WSAENOPROTOOPT);
	
}


int PASCAL FAR getsockopt (SOCKET s, int level, int optname,
                           char FAR * optval, int FAR *optlen)
{
	return OptionHelper(s,level,optname, optval, optlen,0);
}

int PASCAL FAR setsockopt (SOCKET s, int level, int optname,
                           const char FAR * optval, int optlen)
{
	return OptionHelper(s,level,optname, (char *)optval, &optlen,1);
}                   

unsigned long PASCAL FAR inet_addr (const char FAR * cp)
{
	InetHost res;
	if (OTInetStringToHost((char *)cp,&res) == noErr)
		return res;
	else
		return INADDR_NONE;
}

char FAR * PASCAL FAR inet_ntoa (struct in_addr in)
{
	static char addrbuf[16];
	OTInetHostToString(in.S_un.S_addr,addrbuf);
	return addrbuf;
}

u_long PASCAL FAR ntohl (u_long netlong)
{
	return netlong; //yeah! on mac it's the same
}

u_short PASCAL FAR ntohs (u_short netshort)
{
	return netshort;
}

u_long PASCAL FAR htonl (u_long hostlong)
{
	return hostlong;
}

u_short PASCAL FAR htons (u_short hostshort)
{
	return hostshort;
}

int __WSAFDIsSet(SOCKET fd, fd_set *set)
{
	int i;
	for (i = 0 ; i < set->fd_count ; i++)
		if (set->fd_array[i] == fd)
			return 1;
	return 0;
}

