/*
	SymbolTrace.cpp

	This app adds a debug macro to the beginning of each function recursively
	for all .cpp files in a project.

	aml - 8-18-03
*/

#include <windows.h>
#include <stdio.h>
#include <imagehlp.h>
#include <psapi.h>
#include "llist.h"

//
// DIA enums (These correspond to symbol tag data)
//

enum SymTagEnum
{
    SymTagNull,
    SymTagExe,
    SymTagCompiland,
    SymTagCompilandDetails,
    SymTagCompilandEnv,
    SymTagFunction,
    SymTagBlock,
    SymTagData,
    SymTagAnnotation,
    SymTagLabel,
    SymTagPublicSymbol,
    SymTagUDT,
    SymTagEnum,
    SymTagFunctionType,
    SymTagPointerType,
    SymTagArrayType,
    SymTagBaseType,
    SymTagTypedef,
    SymTagBaseClass,
    SymTagFriend,
    SymTagFunctionArgType,
    SymTagFuncDebugStart,
    SymTagFuncDebugEnd,
    SymTagUsingNamespace,
    SymTagVTableShape,
    SymTagVTable,
    SymTagCustom,
    SymTagThunk,
    SymTagCustomType,
    SymTagManagedType,
    SymTagDimension,
    SymTagMax
};

bool bDumpModules    = false;
bool bDumpSymbols    = false;
bool bIsDLL          = false;
bool bIncludeHeaders = false;
bool bOmitInline     = false;

void* pModuleBase;

int  gnSymbols               = 0;
int  gnFunctions             = 0;
int  gnFunctionsInstrumented = 0;

SYMBOL_INFO* gpSymbolInfo = NULL;

void tolower(char* str)
{
	int len = strlen(str);

	for(int i = 0; i < len; i++)
		str[i] = _tolower(str[i]);
}

struct FileUpdateRec
{
	char Filename[1024];

	FileUpdateRec() {}
	FileUpdateRec(char* buf) { strcpy(Filename, buf); }

	bool operator ==(FileUpdateRec& rec)
	{
		if (strcmp(Filename, rec.Filename) == 0)
			return true;

		return false;
	}
};

LinkList<FileUpdateRec> fileList;

void AdvanceLines(FILE* fpSrc, FILE* fpDest, int nLines)
{
	char buf[1024];

	while(nLines > 0)
	{
		if (!fgets(buf, 1024, fpSrc))
			return;

		fputs(buf, fpDest);
		nLines--;
	}
}

BOOL CALLBACK ProcSymbolCB(LPSTR SymbolName, ULONG SymbolAddress, ULONG SymbolSize, void* pData)
{
	DWORD64 displace64 = 0;

	// Lookup symbol
	gpSymbolInfo->SizeOfStruct = sizeof(SYMBOL_INFO);

	if (SymFromAddr(GetCurrentProcess(), SymbolAddress, &displace64, gpSymbolInfo))
	{
		gpSymbolInfo->SizeOfStruct = sizeof(SYMBOL_INFO);

		/*
		if (strcmp(gpSymbolInfo->Name, "ShutdownSymbols") == 0)
		{
			__asm int 3;
		}
		*/

		// If this symbol is a function we need to process it
		//if (gpSymbolInfo->Flags & SYMFLAG_FUNCTION)	// Flags aren't valid :(
		
		// Grr, flags don't ever seem to be valid for some reason in relation to symbol data
		// eventually tracked down the DIA enums for codeview tags (which seems correct)
		if (gpSymbolInfo->Tag == SymTagFunction &&
			strstr(gpSymbolInfo->Name, "$") == NULL)
		{
			// Determine where the function starts in respect to the source code that generated it
			IMAGEHLP_LINE line;
			line.SizeOfStruct = sizeof(IMAGEHLP_LINE);
			line.Key          = NULL;
			line.LineNumber   = 0;
			line.FileName     = NULL;
			line.Address      = 0;

			DWORD displace = 0;

			if (SymGetLineFromAddr(GetCurrentProcess(), (DWORD)SymbolAddress, &displace, &line))
			{
				// If this exists within a header file skip it
				if (strstr(line.FileName, ".h") ||
					strstr(line.FileName, ".inl"))
					return TRUE;

				printf("Func: %s\nin %s at line %i.\n\n", gpSymbolInfo->Name, line.FileName, line.LineNumber);
				gnFunctions++;

				// OK, now we have all our info, just open up the file and insert our hook into the source code
				// We want to start by making a backup
				FileUpdateRec newRec(line.FileName);

				if (!fileList.Find(&newRec))
				{
					char newFile[1024];
					strcpy(newFile, line.FileName);
					strcat(newFile, ".bak");

					CopyFile(line.FileName, newFile, TRUE);
					fileList.AddToTail(&newRec);
				}

				FILE* fp = fopen(line.FileName, "r");
				FILE* fpOut = fopen("temp.tmp", "w");

				if (fp && fpOut)
				{
					AdvanceLines(fp, fpOut, line.LineNumber - 1);

					char sline[1024];
					fgets(sline, 1024, fp);

					// Don't process inline if told not to
					if (bOmitInline && strstr(sline, "inline"))
					{
						fclose(fp);
						fclose(fpOut);
						DeleteFile("temp.tmp");
						return TRUE;
					}

					// Don't process line if already instrumented
					if (strstr(sline, "FUNC_ENTER"))
					{
						fclose(fp);
						fclose(fpOut);
						DeleteFile("temp.tmp");
						return TRUE;			
					}

					char* fstart = strstr(sline, "{");

					// Abort if not at the start of a definition
					if (!fstart)
					{
						fclose(fp);
						fclose(fpOut);
						DeleteFile("temp.tmp");
						return TRUE;
					}

					*fstart = 0;
					fstart++;

					printf("%s{ FUNC_ENTER(\"%s\"); %s", sline, gpSymbolInfo->Name, fstart);
					fprintf(fpOut, "%s{ FUNC_ENTER(\"%s\"); %s", sline, gpSymbolInfo->Name, fstart);
					
					AdvanceLines(fp, fpOut, 99999999);
					fclose(fp);
					fclose(fpOut);

					BOOL bVal = CopyFile("temp.tmp", line.FileName, FALSE);
					DWORD err = GetLastError();
					gnFunctionsInstrumented++;
				}
			}
		}
	}

	return TRUE;
}

BOOL CALLBACK EnumModulesCB(LPSTR ModuleName, ULONG BaseOfDll, void* pData)
{
	printf("%s\t\t%x\n", ModuleName, BaseOfDll);
	return TRUE;
}

BOOL CALLBACK EnumSymbolCB(LPSTR SymbolName, ULONG SymbolAddress, ULONG SymbolSize, void* pData)
{
	if (strcmp(SymbolName, "main")==0)
		__asm int 3;

	printf("%s\t%x\t%i\n", SymbolName, SymbolAddress, SymbolSize);
	gnSymbols++;

	return TRUE;
}

void UpdateIncludes()
{
	Link<FileUpdateRec>* link = fileList.GetHead();
	char line[1024];
	int  nUpdates = 0;

	while(link)
	{
		FILE* fpSrc = fopen(link->data.Filename, "r");

		if (!fpSrc)
		{
			link = link->next;
			continue;
		}
		
		fgets(line, 1024, fpSrc);

		if (strstr(line, "#include \"FuncEnter.h\""))
		{
			fclose(fpSrc);
			link = link->next;
			continue;
		}

		FILE* fpOut = fopen("temp.tmp", "w");

		if (!fpOut)
		{
			fclose(fpSrc);
			link = link->next;
			continue;
		}

		fprintf(fpOut, "#include \"FuncEnter.h\"\n\n");
		fputs(line, fpOut);
		AdvanceLines(fpSrc, fpOut, 99999999);

		fclose(fpSrc);
		fclose(fpOut);
		CopyFile("temp.tmp", link->data.Filename, FALSE);

		printf("Updated includes for %s.\n", link->data.Filename);
		nUpdates++;
		link = link->next;
	}

	printf("Updated includes on %i of %i modified files.\n", nUpdates, fileList.GetSize());
}

int main(int argc, char* argv[])
{
	int   iModuleArg = 0;

	if (argc < 2)
	{
		printf("SymbolTrace [filename] [-d] [-dll] [-s]\n\n");
		
		printf("SymbolTrace inserts function start macros into source code\n");
		printf("based on debug symbol info in the compiled file.\n\n");

		printf("-d\tDump module names/base addresses\n");
		printf("-dll\tFile is a DLL\n");
		printf("-s\tDump symbols\n");
		printf("-h\tInclude header files in instrumentation\n");
		printf("-i\tOmit inline functions (must start on same line as inline keyword)\n");
		return 0;
	}

	for(int i = 2; i < argc; i++)
	{
		if (strcmp(argv[i], "-d") == 0)
			bDumpModules = true;

		if (strcmp(argv[i], "-dll") == 0)
			bIsDLL = true;

		if (strcmp(argv[i], "-s") == 0)
			bDumpSymbols = true;

		if (strcmp(argv[i], "-h") == 0)
			bIncludeHeaders = true;

		if (strcmp(argv[i], "-i") == 0)
			bOmitInline = true;
	}

	// We'll create the process in a suspended state so that we can gain access
	// to a process handle for it that we can use for SymInitialize to gain access
	// to the application's debug symbol table
	STARTUPINFO startInfo;
	startInfo.cb = sizeof(STARTUPINFO);
	startInfo.lpReserved      = NULL;
	startInfo.lpDesktop       = NULL;
	startInfo.lpTitle         = NULL;
	startInfo.dwX             = 0;
	startInfo.dwY             = 0;
	startInfo.dwXSize         = 640;
	startInfo.dwYSize         = 480;
	startInfo.dwXCountChars   = 0;
	startInfo.dwYCountChars   = 0;
	startInfo.dwFillAttribute = 0;

	PROCESS_INFORMATION procInfo;
	ZeroMemory(&procInfo, sizeof(PROCESS_INFORMATION));

	if (!bIsDLL)
	{
		if (!CreateProcess(argv[1],
						   "",
						   NULL,
						   NULL,
						   FALSE,
						   CREATE_SUSPENDED,
						   NULL,
						   NULL,
						   &startInfo,
						   &procInfo))
		{
			printf("Failed to initiate application in suspended state.\n");
			return -1;
		}

		MODULEINFO moduleInfo;
		HMODULE    hModule = GetModuleHandle(NULL);
		
		GetModuleInformation(GetCurrentProcess(), hModule, &moduleInfo, sizeof(MODULEINFO));
		pModuleBase = moduleInfo.lpBaseOfDll;	
	}
	else
	{
		MODULEINFO moduleInfo;
		HMODULE    hModule = LoadLibraryEx(argv[1], NULL, DONT_RESOLVE_DLL_REFERENCES);

		//SymLoadModule

		if (!hModule)
		{
			printf("Failed to load library '%s'.\n", argv[1]);
			return 0;
		}

		GetModuleInformation(GetCurrentProcess(), hModule, &moduleInfo, sizeof(MODULEINFO));
		pModuleBase = moduleInfo.lpBaseOfDll;
	}

	if (!SymInitialize(GetCurrentProcess(), "C:\\SKATE5\\Tools\\src\\Max5\\Next\\NExt___Win32_Hybrid", TRUE))
	{
		printf("Failed to initalize symbol information\n");
		DWORD err = GetLastError();
		return -1;
	}

	SymSetOptions(SYMOPT_LOAD_ANYTHING|SYMOPT_LOAD_LINES|SYMOPT_DEBUG);

	if (bDumpModules)
	{
		SymEnumerateModules(GetCurrentProcess(), EnumModulesCB, NULL);
		SymCleanup(GetCurrentProcess());
		return 0;
	}

	if (bDumpSymbols)
	{
		// Now scan through all the function symbols in the process
		gnSymbols = 0;
		SymEnumerateSymbols(GetCurrentProcess(), (DWORD)pModuleBase, EnumSymbolCB, NULL);
		SymCleanup(GetCurrentProcess());
		printf("%i Symbols.\n", gnSymbols);
		return 0;
	}
	
	// Process symbols
	gpSymbolInfo = (SYMBOL_INFO*)malloc(sizeof(SYMBOL_INFO) + 256);
	gpSymbolInfo->SizeOfStruct = sizeof(SYMBOL_INFO);
	gpSymbolInfo->TypeIndex    = 0;
	gpSymbolInfo->Reserved[0]  = 0;
	gpSymbolInfo->Reserved[1]  = 0;
	gpSymbolInfo->Size         = 0;
	gpSymbolInfo->Reserved[0]  = 0;
	gpSymbolInfo->ModBase      = 0;		// Filled in on ExitFunc
	gpSymbolInfo->Flags        = IMAGEHLP_SYMBOL_FUNCTION;
	gpSymbolInfo->Value        = 0;
	gpSymbolInfo->Address      = 0;		// Filled in on ExitFunc
	gpSymbolInfo->Register     = 0;		// ?
	gpSymbolInfo->Scope        = 0;		// For DIA info  (unused for our purposes)
	gpSymbolInfo->Tag          = 0;		// DIA info but, publicly available, identifies symbol type 
	gpSymbolInfo->NameLen      = 0;
	gpSymbolInfo->MaxNameLen   = 256;
	gpSymbolInfo->Name[0]      = 0;

	/*
	char buf[1024];
	if (!FindDebugInfoFile("C:\\SKATE5\\Tools\\src\\Max5\\Next\\NExt___Win32_Hybrid\\Next.pdb", "C:\\SKATE5\\Tools\\src\\Max5\\Next\\NExt___Win32_Hybrid", buf))
	{
		__asm int 3;
	}
	*/

	SymEnumerateSymbols(GetCurrentProcess(), (DWORD)pModuleBase, ProcSymbolCB, NULL);
	printf("Instrumented %i of %i functions.\n", gnFunctionsInstrumented, gnFunctions);

	// Run through all the files we processed and add an include line for the macro
	UpdateIncludes();

	SymCleanup(GetCurrentProcess());

	if (!bIsDLL)
		TerminateProcess(procInfo.hProcess, 0);

	//free(gpSymbol);
	free(gpSymbolInfo);
	return 0;
}
