/*
	FaceFlagger.cpp
	A utility plugin for automatically applying face flags
	aml
*/

#include "FaceFlagger.h"
#include "Resource.h"
#include "../misc/maxutil.h"

// Conversion factor for Max units to feet
//#define  MINPREVENT_LENGTH  0.666667f  // (8 in)
#define  MINPREVENT_LENGTH  8.0f  // (8 in)
#define  SIDE_A   0
#define  SIDE_B   1
#define  SIDE_C   2

#define DEG90  1.57080		// 90 Degrees in radians PI/2

static FaceFlaggerClassDesc theFaceFlaggerDesc;
ClassDesc2* GetFaceFlaggerDesc() { return &theFaceFlaggerDesc; }

static FaceFlagger* theFaceFlagger;

FaceFlagger::FaceFlagger()
{

}

FaceFlagger::~FaceFlagger()
{

}

void FaceFlagger::GetEdgeSizes(Mesh& mesh, Face face, 
							   float& sAB, float& sBC, float& sAC)
{
	Point3 A, B, C;

	A = mesh.verts[face.v[0]];
	B = mesh.verts[face.v[1]];
	C = mesh.verts[face.v[2]];
	
	Point3 T = B - A;

	/*
	sAB = Length(B - A) * UNITS_TO_FEET;
	sBC = Length(B - C) * UNITS_TO_FEET;
	sAC = Length(C - A) * UNITS_TO_FEET;
	*/

	sAB = Length(B - A);
	sBC = Length(B - C);
	sAC = Length(C - A);
}

void FaceFlagger::GetEdgeMinMax(float s1, float s2, float s3,
								float& min, float& max)
{
	min = max = s1;

	if (s2 < min)
		min = s2;

	if (s2 > max)
		max = s2;

	if (s3 < min)
		min = s3;

	if (s3 > max)
		max = s3;
}

void FaceFlagger::AutoAssignFaceFlags()
{
	INodeTab nodelist;
	GetAllNodes(nodelist);

	int nNodes = nodelist.Count();

	for(int i = 0; i < nNodes; i++)
		AutoAssignFaceFlags(nodelist[i]);
}

void FaceFlagger::AutoAssignFaceFlagsSel()
{
	int nNodes = ip->GetSelNodeCount();

	if (nNodes > 0)
	{
		INode* node = ip->GetSelNode(0);
		AutoAssignFaceFlags(node);
	}
}

float FaceFlagger::GetMaxAltEdgeLength(Mesh& mesh, FaceFlagsData* fdc, AdjEdgeList* elist, int face, int edge)
{
	float sAB, sBC, sAC;			// Sizes of each edge of the face
	float smallest, largest;

	int idx = elist->edges[edge].OtherFace(face);	

	if (idx == -1)
		return 0.0f;

	GetEdgeSizes(mesh,
		         mesh.faces[idx],
				 sAB, sBC, sAC);
		         
	GetEdgeMinMax(sAB, sBC, sAC,
		          smallest, largest);

	return largest;
}

// Note: This check is the same for Camera Collideable
bool FaceFlagger::CheckFaceWallRideable(Matrix3 TM, Mesh& mesh, FaceFlagsData* fdc, AdjEdgeList* elist, int face)
{
	// (Not set if a face is manually flagged as non-collidable)
 	if (fdc->m_data[face] & mFD_NON_COLLIDABLE)
		return false;

	float fminEdge        = GetMinEdgeWR();			// If the edges are greater than this then the face should be flagged
	float fminPreventEdge = GetMinPreventEdgeWR();	// If the edges an edge is less than this then the face shouldn't be flagged
	float fvertTol        = DegToRad(GetVertTolWR());

	// Check if this face should be flagged
	float sAB, sBC, sAC;			// Sizes of each edge of the face
	float smallest, largest;

	GetEdgeSizes(mesh, 
				 mesh.faces[face],
				 sAB, sBC, sAC);

	GetEdgeMinMax(sAB, sBC, sAC,
				  smallest, largest);

	// If the smallest edge of the face is less than the min prevent edge value
	// then this face shouldn't be flagged
	if (smallest <= fminPreventEdge)
		return false;

	// If any edge is greater than fminEdge then the face should be flagged
	if (largest >= fminEdge)
	{
		// The flag should only be set to true if it's normal lies at a near vertical
		// (within the vertical tolerance) amount to the up vector

		Point3 vUp(0.0f, 0.0f, 1.0f);

		Point3 pt0 = mesh.verts[mesh.faces[face].v[0]] * TM;
		Point3 pt1 = mesh.verts[mesh.faces[face].v[1]] * TM;
		Point3 pt2 = mesh.verts[mesh.faces[face].v[2]] * TM;

		Point3 vNormal = (pt1 - pt0) ^ (pt2 - pt1);

		//vNormal.x = fabs(vNormal.x);
		//vNormal.y = fabs(vNormal.y);
		//vNormal.z = fabs(vNormal.z);

		float    ang = acos(DotProd(vUp, vNormal) / (vUp.Length() * vNormal.Length()));
		float angdeg = RadToDeg(ang);

		// Must fall within 90-tol degrees from Up Vector and 90+tol degrees from Up Vector
		if (ang > DEG90-fabs(fvertTol) &&
			ang < DEG90+fabs(fvertTol))
			return true;
		
		return false;
	}
/*ll
	// If any shared edge is greater than fminEdge then the face should be flagged
	int idxEdgeAB = elist->FindEdge(mesh.faces[face].v[SIDE_A], mesh.faces[face].v[SIDE_B]);
	int idxEdgeBC = elist->FindEdge(mesh.faces[face].v[SIDE_B], mesh.faces[face].v[SIDE_C]);
	int idxEdgeAC = elist->FindEdge(mesh.faces[face].v[SIDE_A], mesh.faces[face].v[SIDE_C]);

	if (GetMaxAltEdgeLength(mesh, fdc, elist, face, idxEdgeAB) >= fminEdge)
		return true;

	if (GetMaxAltEdgeLength(mesh, fdc, elist, face, idxEdgeBC) >= fminEdge)
		return true;

	if (GetMaxAltEdgeLength(mesh, fdc, elist, face, idxEdgeAC) >= fminEdge)
		return true;
*/
	return false;
}

// Note: This check is the same for Camera Collideable
bool FaceFlagger::CheckFaceCameraCollidable(Matrix3 TM, Mesh& mesh, FaceFlagsData* fdc, AdjEdgeList* elist, int face)
{
	// (Not set if a face is manually flagged as non-collidable)
 	if (fdc->m_data[face] & mFD_NON_COLLIDABLE)
		return false;

	float fminEdge        = GetMinEdgeCC();			// If the edges are greater than this then the face should be flagged
	float fminPreventEdge = GetMinPreventEdgeCC();	// If the edges an edge is less than this then the face shouldn't be flagged
	
	// Check if this face should be flagged
	float sAB, sBC, sAC;			// Sizes of each edge of the face
	float smallest, largest;

	GetEdgeSizes(mesh, 
				 mesh.faces[face],
				 sAB, sBC, sAC);

	GetEdgeMinMax(sAB, sBC, sAC,
				  smallest, largest);

	// If the smallest edge of the face is less than the min prevent edge value
	// then this face shouldn't be flagged
	if (smallest <= fminPreventEdge)
		return false;

	// If any edge is greater than fminEdge then the face should be flagged
	if (largest >= fminEdge)
		return true;
/*
	// If any shared edge is greater than fminEdge then the face should be flagged
	int idxEdgeAB = elist->FindEdge(mesh.faces[face].v[SIDE_A], mesh.faces[face].v[SIDE_B]);
	int idxEdgeBC = elist->FindEdge(mesh.faces[face].v[SIDE_B], mesh.faces[face].v[SIDE_C]);
	int idxEdgeAC = elist->FindEdge(mesh.faces[face].v[SIDE_A], mesh.faces[face].v[SIDE_C]);

	if (GetMaxAltEdgeLength(mesh, fdc, elist, face, idxEdgeAB) >= fminEdge)
		return true;

	if (GetMaxAltEdgeLength(mesh, fdc, elist, face, idxEdgeBC) >= fminEdge)
		return true;

	if (GetMaxAltEdgeLength(mesh, fdc, elist, face, idxEdgeAC) >= fminEdge)
		return true;
*/
	return false;
}

void FaceFlagger::ScanFaces(Matrix3 TM, Mesh& mesh, FaceFlagsData* fdc, AdjEdgeList* elist)
{
	for(int i = 0; i < mesh.numFaces; i++)
	{
		if (CheckFaceWallRideable(TM, mesh,fdc,elist,i))
			fdc->m_data[i] |= mFD_WALL_RIDABLE;
		else
			if (IsDlgButtonChecked(hwndRollup, IDC_CLEARWR) == BST_CHECKED)
				fdc->m_data[i] &= ~mFD_WALL_RIDABLE;

		if (CheckFaceCameraCollidable(TM, mesh,fdc,elist,i))
			fdc->m_data[i] |= mFD_NON_CAMERA_COLLIDABLE;
		else
			if (IsDlgButtonChecked(hwndRollup, IDC_CLEARCC) == BST_CHECKED)
				fdc->m_data[i] &= ~mFD_NON_CAMERA_COLLIDABLE;
	}
}

void FaceFlagger::AutoAssignFaceFlags(INode* node)
{
	if (!node)
		return;

	Object* obj = node->EvalWorldState(0).obj;
	
	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

		Mesh&        mesh = triObj->GetMesh();
		AdjEdgeList elist(mesh);

		IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
		if (pFDMgr)
		{
			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

			if (!fdc)
			{
				fdc = new FaceFlagsData();
				pFDMgr->AddFaceDataChan( fdc );
				fdc->FacesCreated( 0, mesh.getNumFaces() );
			}

			ScanFaces(node->GetNodeTM(0), mesh, fdc, &elist);
		}

		if (triObj != obj)
			triObj->DeleteThis();
	}
}

void FaceFlagger::AutoAssignFaceFlagsOld()
{
	INodeTab nodelist;
	//GetAllNodes(nodelist);
	GetSelSetNodes(nodelist);

	int nNodes = nodelist.Count();

	// Min edge was 4.0 ft
	float fminEdge = minEdgeSpinWR->GetFVal() * INCHES_TO_FEET * FEET_TO_UNITS;

	for(int i = 0; i < nNodes; i++)
	{
		Object* obj = nodelist[i]->EvalWorldState(0).obj;

		if (obj && obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
			Mesh&        mesh = triObj->GetMesh();
			AdjEdgeList elist(mesh);

			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
			if (pFDMgr)
			{
				FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

				if (!fdc)
				{
					fdc = new FaceFlagsData();
					pFDMgr->AddFaceDataChan( fdc );
					fdc->FacesCreated( 0, mesh.getNumFaces() );
				}

//------>		// Scan through all the triangles of the mesh for this node, and flag them appropriately
				for(int i = 0; i < mesh.numFaces; i++)
				{
					float sAB, sBC, sAC;			// Sizes of each edge of the face
					float smallest, largest;

					GetEdgeSizes(mesh, 
								 mesh.faces[i],
								 sAB, sBC, sAC);

					GetEdgeMinMax(sAB, sBC, sAC,
						          smallest, largest);


					// Auto assign Camera Collidable =============================================
					
					// (Not set if a face is manually flagged as non-collidable)
 					if (!(fdc->m_data[i] & mFD_NON_COLLIDABLE))
					{
						// Should only be set if an edge is 4 ft or greater
						if ((sAB > fminEdge ||
							 sBC > fminEdge ||
							 sAC > fminEdge))
						{
							// Not set if an edge is shared with a poly that is 4 ft or greater
							int idxEdgeAB = elist.FindEdge(mesh.faces[i].v[SIDE_A], mesh.faces[i].v[SIDE_B]);
							int idxEdgeBC = elist.FindEdge(mesh.faces[i].v[SIDE_B], mesh.faces[i].v[SIDE_C]);
							int idxEdgeAC = elist.FindEdge(mesh.faces[i].v[SIDE_A], mesh.faces[i].v[SIDE_C]);

							float esAB, esBC, esAC;

							// Check if the other face sharing the edges of the current face has a size > 4 ft
							idxEdgeAB = elist.edges[idxEdgeAB].OtherFace(i);
							
							if (idxEdgeAB > -1)
							{
								GetEdgeSizes(mesh, mesh.faces[idxEdgeAB], esAB, esBC, esAC);						
							}
							else
							{
								esAB = 0.0f;
								esBC = 0.0f;
								esAC = 0.0f;
							}

							if ((esAB > fminEdge ||
								 esBC > fminEdge ||
								 esAC > fminEdge))
							{
								idxEdgeBC = elist.edges[idxEdgeBC].OtherFace(i);

								if (idxEdgeBC > -1)
								{
									GetEdgeSizes(mesh, mesh.faces[idxEdgeBC], esAB, esBC, esAC);
								}
								else
								{
									esAB = 0.0f;
									esBC = 0.0f;
									esAC = 0.0f;
								}

								if ((esAB > fminEdge ||
									 esBC > fminEdge ||
									 esAC > fminEdge))
								{
									idxEdgeAC = elist.edges[idxEdgeAC].OtherFace(i);

									if (idxEdgeAC > -1)
									{
										GetEdgeSizes(mesh, mesh.faces[idxEdgeAC], esAB, esBC, esAC);
									}
									else
									{
										esAB = 0.0f;
										esBC = 0.0f;
										esAC = 0.0f;
									}

									if ((esAB > fminEdge ||
										 esBC > fminEdge ||
										 esAC > fminEdge))
									{
										// Not flagged if an original edge is less than 8 inches (0.666667f ft)
										if (!(smallest < 0.666667f))
										{
											// All conditions met, flag the face
											if (!(fdc->m_data[i] & mFD_NON_CAMERA_COLLIDABLE))
											{
												fdc->m_data[i] |= mFD_NON_CAMERA_COLLIDABLE;
												char str[256];
												sprintf(str, "Face %i Flagged CAMERA_COLLIDABLE\n", i);
												OutputDebugString(str);
											}
											else
											{
												char str[256];
												sprintf(str, "Face %i CAMERA_COLLIDABLE Rejected due to NON flag\n", i);
												OutputDebugString(str);
											}

											// Wallrideable has the same conditions
											if (!(fdc->m_data[i] & mFD_NON_WALL_RIDABLE))
											{
												fdc->m_data[i] |= mFD_WALL_RIDABLE;
												char str[256];
												sprintf(str, "Face %i Flagged WALL_RIDEABLE\n", i);
												OutputDebugString(str);
											}
											else
											{
												char str[256];
												sprintf(str, "Face %i WALL_RIDEABLE Rejected due to NON flag\n", i);
												OutputDebugString(str);
											}
										}
										else
										{
											OutputDebugString("Rejected due to smallest edge < 8 in.\n");
										}
									}
								}
							}
						}
						else
						{
							char str[256];
							sprintf(str, "Face %i rejected orig not greater than 4 ft\n", i);
							OutputDebugString(str);

						}

					}
					// ===========================================================================
				}

				

//------>		///////////
			}


			if (triObj != obj)
				triObj->DeleteThis();
		}
	}
}

BOOL CALLBACK FaceFlagger::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	// This can be static because there can only ever be one instance of a particular
	// utility plugin at a time
	
	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_AUTOASSIGN:
			theFaceFlagger->AutoAssignFaceFlags();
			MessageBox(theFaceFlagger->ip->GetMAXHWnd(), "Auto face flag assignment (all) completed.", "Auto Face Flagger", MB_ICONINFORMATION|MB_OK);
			return TRUE;

		case IDC_AUTOASSIGNSEL:
			theFaceFlagger->AutoAssignFaceFlagsSel();
			MessageBox(theFaceFlagger->ip->GetMAXHWnd(), "Auto face flag assignment (sel) completed.", "Auto Face Flagger", MB_ICONINFORMATION|MB_OK);
			return TRUE;
		}
	}

	return FALSE;
}

void FaceFlagger::BeginEditParams(Interface* ip, IUtil* iu)
{
	theFaceFlagger = this;
	this->ip = ip;
	hwndRollup = ip->AddRollupPage(hInstance, MAKEINTRESOURCE(IDD_FACEFLAGGER), DlgProc, "Face Flagger Util",(LPARAM)this);

	// Setup wall-rideable interface
	minEdgeWR     = GetICustEdit(GetDlgItem(hwndRollup, IDC_WR_MINEDGE));
	minEdgeSpinWR = GetISpinner(GetDlgItem(hwndRollup, IDC_WR_MINEDGESPIN));
	minEdgeSpinWR->LinkToEdit(GetDlgItem(hwndRollup, IDC_WR_MINEDGE), EDITTYPE_POS_UNIVERSE);

	minEdgeSpinWR->SetLimits(0.0f, 1000.0f);
	minEdgeSpinWR->SetValue(4.0f * 12.0f, TRUE);
	minEdgeSpinWR->SetAutoScale(TRUE);

	minPreventEdgeWR     = GetICustEdit(GetDlgItem(hwndRollup, IDC_WR_PREVENTEDGE));
	minPreventEdgeSpinWR = GetISpinner(GetDlgItem(hwndRollup, IDC_WR_PREVENTEDGESPIN));
	minPreventEdgeSpinWR->LinkToEdit(GetDlgItem(hwndRollup, IDC_WR_PREVENTEDGE), EDITTYPE_POS_UNIVERSE);

	minPreventEdgeSpinWR->SetLimits(0.0f, 1000.0f);
	minPreventEdgeSpinWR->SetValue(MINPREVENT_LENGTH, TRUE);
	minPreventEdgeSpinWR->SetAutoScale(TRUE);

	vertTolWR     = GetICustEdit(GetDlgItem(hwndRollup, IDC_WR_VERTTOL));
	vertTolSpinWR = GetISpinner(GetDlgItem(hwndRollup, IDC_WR_VERTTOLSPIN));
	vertTolSpinWR->LinkToEdit(GetDlgItem(hwndRollup, IDC_WR_VERTTOL), EDITTYPE_FLOAT);

	vertTolSpinWR->SetLimits(0.0f, 1000.0f);
	vertTolSpinWR->SetValue(0.0f, TRUE);
	vertTolSpinWR->SetAutoScale(TRUE);

	// Setup Camera-Collideable interface
	minEdgeCC     = GetICustEdit(GetDlgItem(hwndRollup, IDC_CC_MINEDGE));
	minEdgeSpinCC = GetISpinner(GetDlgItem(hwndRollup, IDC_CC_MINEDGESPIN));
	minEdgeSpinCC->LinkToEdit(GetDlgItem(hwndRollup, IDC_CC_MINEDGE), EDITTYPE_POS_UNIVERSE);

	minEdgeSpinCC->SetLimits(0.0f, 1000.0f);
	minEdgeSpinCC->SetValue(4.0f * 12.0f, TRUE);
	minEdgeSpinCC->SetAutoScale(TRUE);

	minPreventEdgeCC     = GetICustEdit(GetDlgItem(hwndRollup, IDC_CC_PREVENTEDGE));
	minPreventEdgeSpinCC = GetISpinner(GetDlgItem(hwndRollup, IDC_CC_PREVENTEDGESPIN));
	minPreventEdgeSpinCC->LinkToEdit(GetDlgItem(hwndRollup, IDC_CC_PREVENTEDGE), EDITTYPE_POS_UNIVERSE);

	minPreventEdgeSpinCC->SetLimits(0.0f, 1000.0f);
	minPreventEdgeSpinCC->SetValue(MINPREVENT_LENGTH, TRUE);
	minPreventEdgeSpinCC->SetAutoScale(TRUE);

	CheckDlgButton(hwndRollup, IDC_CLEARWR, BST_CHECKED);
	CheckDlgButton(hwndRollup, IDC_CLEARCC, BST_CHECKED);

	// Attempt to retrieve settings from FaceFlagger.ini
	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\FaceFlagger.ini";

	char buf[256];
	GetPrivateProfileString("WallRideable","minEdge","48.0",buf,255,(char*)appDir);
	minEdgeSpinWR->SetValue((float)atof(buf), TRUE);

	GetPrivateProfileString("WallRideable","minPreventEdge","8.0",buf,255,(char*)appDir);
	minPreventEdgeSpinWR->SetValue((float)atof(buf), TRUE);

	GetPrivateProfileString("WallRideable","vertTol","0.0",buf,255,(char*)appDir);
	vertTolSpinWR->SetValue((float)atof(buf), TRUE);

	GetPrivateProfileString("CameraCollidable","minEdge","48.0",buf,255,(char*)appDir);
	minEdgeSpinCC->SetValue((float)atof(buf), TRUE);

	GetPrivateProfileString("CameraCollidable","minPreventEdge","8.0",buf,255,(char*)appDir);
	minPreventEdgeSpinCC->SetValue((float)atof(buf), TRUE);
}

void FaceFlagger::EndEditParams(Interface* ip, IUtil* iu)
{
	// Dump the last settings out to a FaceFlagger.ini
	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\FaceFlagger.ini";

	char buf[256];
	sprintf(buf, "%f", minEdgeSpinWR->GetFVal());
	WritePrivateProfileString("WallRideable","minEdge",buf,appDir);

	sprintf(buf, "%f", minPreventEdgeSpinWR->GetFVal());
	WritePrivateProfileString("WallRideable","minPreventEdge",buf,appDir);

	sprintf(buf, "%f", vertTolSpinWR->GetFVal());
	WritePrivateProfileString("WallRideable","vertTol",buf,appDir);

	sprintf(buf, "%f", minEdgeSpinCC->GetFVal());
	WritePrivateProfileString("CameraCollidable","minEdge",buf,appDir);

	sprintf(buf, "%f", minPreventEdgeSpinCC->GetFVal());
	WritePrivateProfileString("CameraCollidable","minPreventEdge",buf,appDir);

	// Release controls
	ReleaseICustEdit(minEdgeWR);
	ReleaseICustEdit(minEdgeCC);
	ReleaseISpinner(minEdgeSpinWR);
	ReleaseISpinner(minEdgeSpinCC);
	ReleaseICustEdit(minPreventEdgeWR);
	ReleaseICustEdit(minPreventEdgeCC);
	ReleaseISpinner(minPreventEdgeSpinWR);
	ReleaseISpinner(minPreventEdgeSpinCC);
	ReleaseICustEdit(vertTolWR);
	ReleaseISpinner(vertTolSpinWR);	

	ip->DeleteRollupPage(hwndRollup);
}

void FaceFlagger::SelectionSetChanged(Interface* ip, IUtil* iu)
{
	
}
