/*
	ViewportUI.h
	This class contains functionality for overriding MAX's
	viewport windows and hooking into their message loops
	so we can intercept mouse and other actions normally
	not possible through SDK calls alone.
	aml - 4/1/03
*/

#include "ViewportUI.h"

#define MAX_VIEWPORT_CLASS  "OGLWindow"		// This is the class name of the MAX viewport windows
#define MAX_VIEWPANEL_CLASS "ViewPanel"		// This is the class name of the MAX viewport panel which contains
											// all the MAX viewports as children

static ViewportUI* gpViewportUIman;

ViewportUI::ViewportUI(Interface* ip)
{
	hwndViewports = NULL;
	wndProcOrig   = NULL;

	this->ip = ip;
	GetViewports();
	gpViewportUIman = this;

#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(viewport_change_handler, this, NOTIFY_VIEWPORT_CHANGE);
#endif
}

ViewportUI::~ViewportUI()
{
	UnRegisterNotification(viewport_change_handler, this, NOTIFY_VIEWPORT_CHANGE);

	if (hwndViewports)
		delete [] hwndViewports;

	if (wndProcOrig)
		delete [] wndProcOrig;
}

void ViewportUI::GetViewports()
{
	// Determine the number of MAX viewport windows
	hwndViewpanel = NULL;
	EnumChildWindows(ip->GetMAXHWnd(), findViewpanel, (LPARAM)&hwndViewpanel);
	assert(hwndViewpanel);
	
	numViewports  = 0;
	EnumChildWindows(hwndViewpanel, countViewports, (LPARAM)&numViewports);
	
	if (hwndViewports)
		delete [] hwndViewports;

	if (wndProcOrig)
		delete [] wndProcOrig;

	hwndViewports = new HWND[numViewports];
	wndProcOrig   = new WNDPROC[numViewports];
	curStoreIdx   = 0;
	EnumChildWindows(hwndViewpanel, storeViewports, (LPARAM)this);
}

LRESULT ViewportUI::HookWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	WNDPROC origProc = gpViewportUIman->GetOrigWndProc(hwnd);
	assert(origProc);

	switch(msg)
	{
	case WM_LBUTTONDOWN:
		MessageBox(hwnd, "WM_LBUTTONDOWN", "Message Recieved", MB_OK);
		break;
	}

	return CallWindowProc(origProc, hwnd, msg, wParam, lParam);
}

// Install our wndproc hook into all the viewport windows
void ViewportUI::HookViewports()
{
	for(int i = 0; i < numViewports; i++)
	{
		wndProcOrig[i] = (WNDPROC)GetWindowLong(hwndViewports[i], GWL_WNDPROC);
		SetWindowLong(hwndViewports[i], GWL_WNDPROC, (LONG)HookWndProc);
	}
}

void ViewportUI::UnHookViewports()
{
	for(int i = 0; i < numViewports; i++)
	{
		SetWindowLong(hwndViewports[i], GWL_WNDPROC, (LONG)wndProcOrig[i]);
	}
}

BOOL CALLBACK ViewportUI::findViewpanel(HWND hwnd, LPARAM lParam)
{
	char bufClassName[256];
	GetClassName(hwnd, bufClassName, 255);

	if (strstr(bufClassName, MAX_VIEWPANEL_CLASS) == 0)
	{
		*((HWND*)lParam) = hwnd;
		return FALSE;
	}

	return TRUE;
}

BOOL CALLBACK ViewportUI::countViewports(HWND hwnd, LPARAM lParam)
{
	char bufClassName[256];
	GetClassName(hwnd, bufClassName, 255);

	if (strstr(bufClassName, MAX_VIEWPORT_CLASS) == 0)
		(*((int*)lParam))++;

	return TRUE;
}

BOOL CALLBACK ViewportUI::storeViewports(HWND hwnd, LPARAM lParam)
{
	char bufClassName[256];
	GetClassName(hwnd, bufClassName, 255);

	if (strstr(bufClassName, MAX_VIEWPORT_CLASS) == 0)
	{
		ViewportUI* pVptUI = (ViewportUI*)lParam;
		pVptUI->hwndViewports[pVptUI->curStoreIdx++] = hwnd;
	}

	return TRUE;
}

void ViewportUI::viewport_change_handler(void *param, NotifyInfo *info)
{
	OutputDebugString("HANDLER: viewport_change_handler\n");
	ViewportUI* pVptUI = (ViewportUI*)param;
	pVptUI->UnHookViewports();
	pVptUI->GetViewports();
	pVptUI->HookViewports();
}

WNDPROC ViewportUI::GetOrigWndProc(int viewportIdx)
{
	if (!wndProcOrig)
		return NULL;

	if (viewportIdx < 0 ||
		viewportIdx > numViewports)
		return NULL;

	return wndProcOrig[viewportIdx];
}

WNDPROC ViewportUI::GetOrigWndProc(HWND hwndViewport)
{
	if (!wndProcOrig)
		return NULL;

	for(int i = 0; i < numViewports; i++)
	{
		if (hwndViewports[i] == hwndViewport)
			return wndProcOrig[i];
	}

	return NULL;
}
