/*
	QueryBuilder.cpp
	The query builder generates a list of Find-A-Node queries from level property data
*/

#include "QueryBuilder.h"
#include "../UI/PropList.h"
#include "ConfigData.h"
#include "exprlib.h"			// For matching by evaluating expressions
#include "ParseFuncs.h"
#include "PropEdit.h"
#include "Resource.h"

extern PropEditor* pPropEdit;	// So we can get access to the scripts.ini database

void QueryBuilderDlg::FocusCB(PropList* plist, int idx, void* pData)
{
	QueryBuilderDlg* pthis = (QueryBuilderDlg*)pData;

	if (pthis->lastFocusIdx != -1)
		plist->SetColor(pthis->lastFocusIdx, PROPCOLOR_GRAY);

	plist->SetColor(idx, PROPCOLOR_WHITE);
	pthis->lastFocusIdx = idx;
}

QueryBuilderDlg::QueryBuilderDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_QUERYBUILDER),hwndParent)
{
	this->ip = ip;
	lastFocusIdx = -1;

	CheckDlgButton(hwnd,IDC_MATCHANY,BST_CHECKED);
	CheckDlgButton(hwnd,IDC_CONVERTDEF,BST_CHECKED);

	// Build and attach the property list to Visual UI
	plist = new PropList(hInstance,1);
	plist->Attach(GetDlgItem(hwnd,IDC_QBPROPLIST));
	plist->HasApply(false);
	plist->SetFocusCB(FocusCB, this);

	BuildSNList();
	BuildAPList();
}

QueryBuilderDlg::~QueryBuilderDlg()
{

}

void QueryBuilderDlg::RegOKCB(void(*Func)(QueryBuilderDlg*,void*),void* pData)
{
	fpOkCB    = Func;
	pOkCBData = pData;
}

BOOL QueryBuilderDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_APALL:
			APAll();			
			return TRUE;

		case IDC_APNONE:
			APNone();
			return TRUE;

		case IDC_APINVERT:
			APInvert();
			return TRUE;

		case IDC_SNALL:
			SNAll();
			return TRUE;

		case IDC_SNNONE:
			SNNone();
			return TRUE;

		case IDC_SNINVERT:
			SNInvert();
			return TRUE;

		case IDC_ADDTOPL:
			AddToPropList();
			return TRUE;

		case IDC_REMOVEFROMPL:
			RemoveFromPropList();
			return TRUE;

		case IDC_RESET:
			ResetPropList();
			return TRUE;

		case IDC_RESCAN:
			Rescan();
			return TRUE;

		case IDC_NODESELECT:
			NodeSelect();
			return TRUE;

		case IDC_SNEDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				SNEditUpdate();
				return TRUE;
			}
			break;

		case IDC_APEDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				APEditUpdate();
				return TRUE;
			}
			break;

		case IDCANCEL:
			Hide();
			return TRUE;

		case IDOK:
			Process();
			return TRUE;
		}
	}

	return FALSE;
}

void QueryBuilderDlg::BuildAPList(INode* root)
{
	// We'll scan through all nodes in the scene and
	if (!root)
	{
		root = ip->GetRootNode();
		SendDlgItemMessage(hwnd,IDC_APLIST,LB_RESETCONTENT,0,0);
		propNames.Clear();
	}

	int nKids = root->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = root->GetChildNode(i);
		BuildAPList(child);
	}

	if (root != ip->GetRootNode())
	{
		// Scan the properties of this node
		CStr propBuffer;
		root->GetUserPropBuffer(propBuffer);

		LinkList<ConfigProp> propList;
		ParseConfigProps(&propList,NULL,propBuffer,NULL,NULL,NULL,NULL);

		// Append unique property names to the list
		Link<ConfigProp>* curprop = propList.GetHead();

		while(curprop)
		{
			propNames.AddUnique(&curprop->data.name);
			curprop = curprop->next;
		}
	}
	else
	{
		// Add prop names to the list
		Link<CStr>* curname = propNames.GetHead();

		while(curname)
		{
			SendDlgItemMessage(hwnd, IDC_APLIST, LB_ADDSTRING, 0, (LPARAM)(char*)curname->data);
			curname = curname->next;
		}
	}
}

void QueryBuilderDlg::BuildSNList(INode* root)
{
	// Add all the nodes in the scene to the list
	if (!root)
	{
		root = ip->GetRootNode();
		SendDlgItemMessage(hwnd,IDC_SNLIST,LB_RESETCONTENT,0,0);
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		BuildSNList(child);
	}

	if (root != ip->GetRootNode())
	{
		int index = SendDlgItemMessage(hwnd,IDC_SNLIST,LB_ADDSTRING,0,(LPARAM)(char*)root->GetName());
		SendDlgItemMessage(hwnd,IDC_SNLIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)root);
	}
}

void QueryBuilderDlg::APAll()
{
	int count = SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETCOUNT,0,0);
	SendDlgItemMessage(hwnd,IDC_APLIST,LB_SELITEMRANGE,(WPARAM)TRUE,MAKELPARAM(0,count-1));
}

void QueryBuilderDlg::APNone()
{
	int count = SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETCOUNT,0,0);
	SendDlgItemMessage(hwnd,IDC_APLIST,LB_SELITEMRANGE,(WPARAM)FALSE,MAKELPARAM(0,count-1));
}

void QueryBuilderDlg::APInvert()
{
	int count = SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETSEL,(WPARAM)i,0))
			SendDlgItemMessage(hwnd,IDC_APLIST,LB_SETSEL,(WPARAM)FALSE,(LPARAM)i);
		else
			SendDlgItemMessage(hwnd,IDC_APLIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)i);
	}
}

void QueryBuilderDlg::SNAll()
{
	int count = SendDlgItemMessage(hwnd,IDC_SNLIST,LB_GETCOUNT,0,0);
	SendDlgItemMessage(hwnd,IDC_SNLIST,LB_SELITEMRANGE,(WPARAM)TRUE,MAKELPARAM(0,count-1));	
}

void QueryBuilderDlg::SNNone()
{
	int count = SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETCOUNT,0,0);
	SendDlgItemMessage(hwnd,IDC_SNLIST,LB_SELITEMRANGE,(WPARAM)FALSE,MAKELPARAM(0,count-1));
}

void QueryBuilderDlg::SNInvert()
{
	int count = SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_SNLIST,LB_GETSEL,(WPARAM)i,0))
			SendDlgItemMessage(hwnd,IDC_SNLIST,LB_SETSEL,(WPARAM)FALSE,(LPARAM)i);
		else
			SendDlgItemMessage(hwnd,IDC_SNLIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)i);
	}
}

void QueryBuilderDlg::AddToPropList()
{
	plist->SaveValues();
	plist->DestroyUI();	
	
	// Add any selected properties from the available properties list
	int nSel = SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETSELCOUNT, 0, 0);
	int* selItems = new int[nSel];

	SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETSELITEMS, (WPARAM)nSel,(LPARAM)selItems);

	int i;

	for(i = 0; i < nSel; i++)
	{
		int idx = selItems[i];
		char name[1024];

		SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETTEXT, (WPARAM)idx, (LPARAM)name);
		plist->AddEdit(name,"Enter any valid wildcard (*,?)");
	}

	delete [] selItems;

	int nItems = SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETCOUNT, 0, 0);

	for(i = 0; i < nItems; i++)
	{
		if (SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETSEL, (WPARAM)i, 0))
		{
			SendDlgItemMessage(hwnd, IDC_APLIST, LB_DELETESTRING, (WPARAM)i, 0);
			i = -1;
		}

		nItems = SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETCOUNT, 0, 0);
	}

	plist->BuildUI();
	plist->RestoreValues();
}

void QueryBuilderDlg::RemoveFromPropList()
{
	if (lastFocusIdx == -1)
		return;

	CStr name = plist->GetName(lastFocusIdx);

	// Clear the property out of the proplist
	plist->SaveValues();
	plist->DestroyUI();

	if (lastFocusIdx != -1)
	{
		plist->DeleteProp(lastFocusIdx);
	}

	plist->BuildUI();
	plist->RestoreValues();

	// Add the property back to the listbox property list
	SendDlgItemMessage(hwnd, IDC_APLIST, LB_ADDSTRING, (WPARAM)0, (LPARAM)(char*)name);
}

void QueryBuilderDlg::Process()
{
	// Scan through the nodes selected for processing
	int nFoundNodes = 0;
	int nSel = SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETSELCOUNT, 0, 0);
	int* selItems = new int[nSel];

	ip->ClearNodeSelection();

	SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETSELITEMS, (WPARAM)nSel,(LPARAM)selItems);

	for(int nodeIdx = 0; nodeIdx < nSel; nodeIdx++)
	{
		int idx = selItems[nodeIdx];
		bool bAbort;

		INode* node = (INode*)SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETITEMDATA, (WPARAM)idx, 0);

		// Now that we have the node, see if it matches our search criteria
		CStr propBuffer;
		node->GetUserPropBuffer(propBuffer);

		int nProps = plist->NumProps();

		LinkList<ConfigProp> propList;
		ParseConfigProps(&propList,NULL,propBuffer,NULL,NULL,NULL,NULL);
		
		if (IsDlgButtonChecked(hwnd,IDC_CONVERTDEF) == BST_CHECKED)
		{
			CStr className = GetClassName(propBuffer);
			CStr typeName  = GetTypeName(propBuffer);
			pPropEdit->ConvertToDefaults(className, typeName, &propList);
		}

		if (IsDlgButtonChecked(hwnd,IDC_MATCHANY) == BST_CHECKED)
		{
			Link<ConfigProp>* curprop = propList.GetHead();

			bAbort = false;

			while(curprop)
			{
				// Check if this is one of the properties we're matching
				for(int i = 0; i < nProps; i++)
				{
					CStr name = plist->GetName(i);
					CStr value;
					plist->GetValue(i,value);

					if (curprop->data.name == name &&
						IsMatch(curprop->data.value, value))
					{
						ip->SelectNode(node, 0);
						nFoundNodes++;
						bAbort = true;
						break;
					}
				}

				if (bAbort)
					break;

				curprop = curprop->next;
			}
		}
		else if (IsDlgButtonChecked(hwnd,IDC_MATCHALL) == BST_CHECKED)
		{
			// To select the node, all the properties must match all the given wildcards
			Link<ConfigProp>* curprop = propList.GetHead();

			bool bSelectNode = true;

			while(curprop)
			{
				for(int i = 0; i < nProps; i++)
				{
					CStr name = plist->GetName(i);
					CStr value;
					plist->GetValue(i,value);

					if (curprop->data.name == name &&
						!IsMatch(curprop->data.value, value))
					{
						bSelectNode = false;
						bAbort      = true;
						break;
					}
				}

				if (bAbort)
					break;

				curprop = curprop->next;
			}

			if (bSelectNode)
			{
				ip->SelectNode(node, 0);
				nFoundNodes++;
			}
		}
	}

	delete [] selItems;

	char strMsg[256];
	sprintf(strMsg,"Found %i nodes that have matching search criteria.", nFoundNodes);

	MessageBox(hwnd, strMsg, "Search Completed", MB_ICONINFORMATION | MB_OK);
	ip->ForceCompleteRedraw();
}

void QueryBuilderDlg::ResetPropList()
{
	plist->Clear();
	BuildSNList();
	BuildAPList();
}

void QueryBuilderDlg::Rescan()
{
	ResetPropList();
	BuildSNList();
	BuildAPList();
}

void QueryBuilderDlg::NodeSelect()
{
	// This will select all the nodes in the SN list that are currently selected
	int nSel = SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETSELCOUNT, 0, 0);
	int* selItems = new int[nSel];

	ip->ClearNodeSelection();

	SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETSELITEMS, (WPARAM)nSel, (LPARAM)selItems);

	for(int i = 0; i < nSel; i++)
	{
		int idx = selItems[i];

		INode* node = (INode*)SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETITEMDATA, (WPARAM)idx, 0);
		ip->SelectNode(node, 0);
	}

	delete [] selItems;
}

void QueryBuilderDlg::SNEditUpdate()
{
	char buf[256];
	char bufItem[256];
	GetWindowText(GetDlgItem(hwnd,IDC_SNEDIT), buf, 254);
	strcat(buf,"*");
	
	// Clear out current selection
	SendDlgItemMessage(hwnd, IDC_SNLIST, LB_SETSEL, (WPARAM)FALSE, (LPARAM)-1);

	// Select all the items within the list that match our wildcard
	int nItems = SendDlgItemMessage(hwnd,IDC_SNLIST,LB_GETCOUNT,0,0);

	for(int i = 0; i < nItems; i++)
	{
		SendDlgItemMessage(hwnd, IDC_SNLIST, LB_GETTEXT, (WPARAM)i, (LPARAM)bufItem);
		
		if (MatchPattern(CStr(bufItem), CStr(buf)))
			SendDlgItemMessage(hwnd, IDC_SNLIST, LB_SETSEL, (WPARAM)TRUE, (LPARAM)i);
	}
}

void QueryBuilderDlg::APEditUpdate()
{
	char buf[256];
	char bufItem[256];
	GetWindowText(GetDlgItem(hwnd,IDC_APEDIT), buf, 254);
	strcat(buf,"*");
	
	// Clear out current selection
	SendDlgItemMessage(hwnd, IDC_APLIST, LB_SETSEL, (WPARAM)FALSE, (LPARAM)-1);

	// Select all the items within the list that match our wildcard
	int nItems = SendDlgItemMessage(hwnd,IDC_APLIST,LB_GETCOUNT,0,0);

	for(int i = 0; i < nItems; i++)
	{
		SendDlgItemMessage(hwnd, IDC_APLIST, LB_GETTEXT, (WPARAM)i, (LPARAM)bufItem);
		
		if (MatchPattern(CStr(bufItem), CStr(buf)))
			SendDlgItemMessage(hwnd, IDC_APLIST, LB_SETSEL, (WPARAM)TRUE, (LPARAM)i);
	}
}

bool QueryBuilderDlg::IsMatch(CStr orig, CStr pattern)
{
	if (strstr(pattern,"#"))
	{
		// Parse as mathematical expression
		pattern = ReplaceStr(pattern, "#", orig);

		Expr expr;
		int  err;

		err = expr.load(pattern);
		
		if (err != EXPR_NORMAL)
			return false;
		else
		{
			float   ans[3];
			float*  sRegs;
			Point3* vRegs;
			int     sCount = expr.getVarCount(SCALAR_VAR);
			int     vCount = expr.getVarCount(VECTOR_VAR);

			sRegs = new float[sCount];
			vRegs = new Point3[vCount];

			err = expr.eval(ans, sCount, sRegs, vCount, vRegs);

			delete [] sRegs;
			delete [] vRegs;

			if (expr.getExprType() != SCALAR_EXPR)
				return false;

			// Return match if the evaluated expression is true, if not return false
			if (ans[0])
				return true;
			else
				return false;
		}
	}
	else
	{
		// Parse as string
		if (MatchPattern(orig, pattern))
			return true;

		return false;
	}
}
