/*
	FindInFiles.cpp
	Find in Files
*/

#include "PropEdit.h"
#include <io.h>
#include <direct.h>
#include "max.h"
#include "FindInFiles.h"
#include "ParseFuncs.h"
#include "Resource.h"

FindInFilesDlg::FindInFilesDlg(HINSTANCE hInstance,HWND hwndParent,PropEditor* pPropEdit,RichText* pRichText) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_FINDINFILES),hwndParent)
{
	this->hInstance=hInstance;
	this->pPropEdit=pPropEdit;
	this->pRichText=pRichText;

	bUserWarned=false;
	bAbort=false;
}

FindInFilesDlg::~FindInFilesDlg()
{
	
}

BOOL FindInFilesDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_CLOSE:
		Hide();
		return TRUE;
	
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_FINDLIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				ListSel();
				break;
			}
			return TRUE;

		case IDOK:
			Find();
			return TRUE;

		case IDCANCEL:
			SendMessage(hwnd,WM_CLOSE,0,0);
			return TRUE;

		case IDC_SEARCHDIR:
			GetSearchDir();
			return TRUE;
		}

		return FALSE;
	}

	return FALSE;
}

void FindInFilesDlg::ListSel()
{
	if (!bUserWarned)
	{
		int rVal=MessageBox(hwnd,"This operation will update the contents of the script window.\nAre you sure?","Script Window Update",MB_OKCANCEL|MB_ICONWARNING);

		if (rVal==IDCANCEL)
			return;
	}

	bUserWarned=true;

	// Parse filename
	char buf[2048];
	char dummy[1024];
	char filename[512];
	int  line;

	int index=SendDlgItemMessage(hwnd,IDC_FINDLIST,LB_GETCURSEL,0,0);
	SendDlgItemMessage(hwnd,IDC_FINDLIST,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);
	
	sscanf(buf,"%s (%i) %s",filename,&line,dummy);

	pPropEdit->LoadText(filename);
	pRichText->HighlightLineNum(line-1);

}

void FindInFilesDlg::GetSearchDir()
{
	OPENFILENAME ofn;
	char filename[256]="*.q";

	// Fill out the arduously long openfilename struct
	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Script Files (*.q; *.qn)\0*.q;*.qn\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=0;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Select the directory path to search for files";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt="*.q";
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	char* pos=strrchr(filename,'\\');
	
	if (!pos)
		return;

	*pos='\0';

	strcat(filename,"\\*.q");

	// Copy directory to edit box
	SetDlgItemText(hwnd,IDC_SRCHDIR,filename);
}

void FindInFilesDlg::Find()
{
	char strDir[256];
	char strSearch[256];
	char strSrch[256];

	bAbort=false;

	GetDlgItemText(hwnd,IDC_SRCHDIR,strDir,256);
	GetDlgItemText(hwnd,IDC_SRCHSTR,strSearch,256);

	// If not set to use wildcards, convert the string into a wildcard
	// so our operation is the same
	if (!IsDlgButtonChecked(hwnd,IDC_USEWILDCARDS))
	{
		strcpy(strSrch,"*");
		strcat(strSrch,strSearch);
		strcat(strSrch,"*");
	}

	// Seperate prefix wildcard
	char* pos=strrchr(strDir,'\\');
	
	if (!pos)
		return;

	*pos='\0';

	SendDlgItemMessage(hwnd,IDC_FINDLIST,LB_RESETCONTENT,0,0);
	
	int count=CountFiles(strDir,pos+1);
	SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETRANGE,0,MAKELPARAM(0,count));
	SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETPOS,0,0);
	
	Find(strDir,pos+1,strSrch);
}

int FindInFilesDlg::CountFiles(char* sdir,char* prefix)
{
	_finddata_t fdata;
	int         count=0;

	_chdir(sdir);
	long handle=_findfirst("*.*",&fdata);
	int  bContinue;

	if (handle)
		bContinue=0;

	while(bContinue==0)
	{
		// Recurse subdirectories
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")!=0 &&
				strcmp(fdata.name,"..")!=0)
			{
				char strPath[1024];
				strcpy(strPath,sdir);
				strcat(strPath,"\\");
				strcat(strPath,fdata.name);

				count+=CountFiles(strPath,prefix);
				_chdir(sdir);
			}
		}
		else
		{
			char strPath[1024];
			strcpy(strPath,sdir);
			strcat(strPath,"\\");
			strcat(strPath,fdata.name);

			if (MatchPattern(CStr(fdata.name),CStr(prefix)))
				count++;
		}

		bContinue=_findnext(handle,&fdata);
	}

	return count;
}

bool FindInFilesDlg::MatchContent(FILE* fp,char* wildcard,char* buf,int* line)
{
	char  tempBuf[512];

	if (!fp)
		return false;

	while(!feof(fp))
	{
		fgets(tempBuf,510,fp);
		(*line)++;

		if (MatchPattern(CStr(tempBuf),CStr(wildcard)))
		{
			tempBuf[511]='\0';
			strcpy(buf,tempBuf);
			return true;
		}

		// Check for cancel
		MSG msg;

		if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
		{
			if (msg.message==WM_LBUTTONDOWN)
			{
				bAbort=true;
				return false;
			}
		}
	}
	
	return false;
}

void FindInFilesDlg::Find(char* sdir,char* prefix,char* search)
{
	_finddata_t fdata;

	_chdir(sdir);
	long handle=_findfirst("*.*",&fdata);
	int  bContinue=-1;

	if (handle)
		bContinue=0;

	while(bContinue==0)
	{
		// Recurse subdirectories
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")!=0 &&
				strcmp(fdata.name,"..")!=0)
			{
				char strPath[1024];
				strcpy(strPath,sdir);
				strcat(strPath,"\\");
				strcat(strPath,fdata.name);

				Find(strPath,prefix,search);
				_chdir(sdir);
			}
		}
		else
		{
			char strPath[2048];
			strcpy(strPath,sdir);
			strcat(strPath,"\\");
			strcat(strPath,fdata.name);

			if (MatchPattern(CStr(fdata.name),CStr(prefix)))
			{
				FILE* fp;
				char buf[1024]="";
				int  line=0;
				fp=fopen(fdata.name,"r");

				while(MatchContent(fp,search,buf,&line))
				{
					char strNew[1024];
					char strNum[256];
					_itoa(line,strNum,10);

					strcpy(strNew,strPath);
					strcat(strNew," (");
					strcat(strNew,strNum);
					strcat(strNew,")        ");
					strcat(strNew,buf);

					SendDlgItemMessage(hwnd,IDC_FINDLIST,LB_ADDSTRING,0,(LPARAM)strNew);

					// Check for cancel
					MSG msg;

					if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
					{
						if (msg.message==WM_LBUTTONDOWN)
							return;
					}
				}

				fclose(fp);

				if (bAbort)
					return;

				// Check for cancel
				MSG msg;

				if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
				{
					if (msg.message==WM_LBUTTONDOWN)
						return;
				}

				// Update progress bar
				int val=SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_GETPOS,0,0);
				SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETPOS,(WPARAM)val+1,0);
			}
		}

		bContinue=_findnext(handle,&fdata);
	}
}
