#include <Engine/Engine.h>
#include <Sk/Gamenet/ExportMsg.h>
#include <Export/Export.h>
#include <Export/ExportOptions.h>
#include <Export/ModelExport.h>
#include <Export/SkinExport.h>
#include <Export/TextureExporter.h>
#include <Export/Strip/Strip.h>
#include <Misc/GenCrc.h>
#include <Misc/HelperFuncs.h>
#include <Misc/Util.h>
#include <Export/Skeleton.h>
#include "phyexp.h"
#include <process.h>
#include "path.h"
#include "../UI/OKtoAll.h"
#include "../../genlib/virtualfile.h"
#include "../misc/mdir.h"

#define MULTIRES_CLASS_ID  Class_ID(0x6a9e4c6b, 0x494744dd)
#define PB_VERTEXCOUNT     1
#define PB_VERTEXPERCENT   2
#define PB_GENERATE        10

extern bool gbAbort;	// Flag to force entire export to abort
extern CStr GetModelName();

class SkinExporter : public ISkinExporter
{
public:
	bool	SpawnPostExportProcessor( void );
	void	EnumerateExportableNodes( void );
	bool	SaveTextures( void );
	bool	SaveSkin( char* skinName = NULL );
	bool	SaveGeometry( IoUtils::CVirtualOutputFile &file );
	bool	GetSkinData( NxObject* pObject, INode* pNode, unsigned long& checksum, INode* pNodeRoot = NULL);

	// From Exporter
	bool	DoExport( ExportType type );
	bool	Save( void );

	void	SetExportModel(NxModel* model) { m_model = model; }

	void	enumerate_selected_nodes( void );
	void	enumerate_visible_nodes( INode* root );
	void	enumerate_selection_set_nodes( void );

private:
	void	EvalLODData( INode* node, NxObject* nxobj );
	void	EvalLODData( Modifier* mod, INode* node, NxObject* nxobj );
	int		DisableMultiRes(INode* node);
	int		EnableMultiRes(INode* node);

	bool	DoIndividualSkinExport( char* skinName = NULL );

	Tab <INode *>		m_exportable_nodes;	
	SkinExportOptions	m_export_options;
	NxModel*			m_model;
	bool                m_LitVertWarned;		// Set true if user has been warned of lit verticies
	CStr                strSkinName;
};

static	SkinExporter	s_skin_exporter;

// Utility functions for skin export...
static void ShowSkinExportErrorMessage(char* pErrorMessage)
{
    MessageBoxAll( gInterface->GetMAXHWnd(),
                   pErrorMessage,
                   "Skin Export",
                   MB_OK|MB_ICONERROR);
}

void	SkinExporter::enumerate_selected_nodes( void )
{
	int i;
	INode* sel_node;
	
	for( i = 0; i < gInterface->GetSelNodeCount(); i++ )
	{
		sel_node = gInterface->GetSelNode( i );
		if( IsExportable( sel_node ))
		{
			m_exportable_nodes.Append( 1, &sel_node );
		}
	}
}

void	SkinExporter::enumerate_visible_nodes( INode* root )
{
	INode* child;
	int i;
	
	for( i = 0; i < root->NumberOfChildren(); i++ )
	{
		child = root->GetChildNode( i );
		// Add only visible, non-frozen nodes
		if(	( child->IsHidden() == false ) &&
			( child->IsFrozen() == false ))
		{
			if( IsExportable( child ))
			{
				m_exportable_nodes.Append( 1, &child );			
			}
		}

		// Recursively add all visible children
		if( child->NumberOfChildren() > 0 )
		{
			enumerate_visible_nodes( child );
		}
	}
}

void	SkinExporter::enumerate_selection_set_nodes( void )
{
	int i, num_sel_sets;

	// If they haven't entered a set, just return
	if( m_export_options.m_ExportSet == TSTR( "" ))
	{
		return;
	}

	// Find the set that matches the name of the proposed export set
	num_sel_sets = gInterface->GetNumNamedSelSets();
	for( i = 0; i < num_sel_sets; i++ )
	{
		if( m_export_options.m_ExportSet == TSTR( gInterface->GetNamedSelSetName( i )))
		{
			int j, num_nodes;

			num_nodes = gInterface->GetNamedSelSetItemCount( i );
			for( j = 0; j < num_nodes; j++ )
			{
				INode* sel_node;

				sel_node = gInterface->GetNamedSelSetItem( i, j );

				m_exportable_nodes.Append( 1, &sel_node );
			}

			break;
		}
	}
}

void	SkinExporter::EnumerateExportableNodes( void )
{
	m_exportable_nodes.ZeroCount();
	
	if( m_export_options.m_ExportSelected )
	{
		enumerate_selected_nodes();
	}
	else if( m_export_options.m_ExportVisibleOnly )
	{
		enumerate_visible_nodes( gInterface->GetRootNode());
	}
	else if( m_export_options.m_ExportSelectionSet )
	{
		enumerate_selection_set_nodes();
	}	
}

ISkinExporter*	GetSkinExporter( void )
{
	return &s_skin_exporter;
}

DWORD WINAPI SkinExportProgressFunc(LPVOID arg) 
{
    return(0);
}

bool	SkinExporter::SaveGeometry( IoUtils::CVirtualOutputFile &file )
{
	int i, j, k, num_obj, num_faces, num_verts;

	FILE* fpWeights = fopen("c:\\vweights.txt","w");

	num_obj = m_model->m_Objects.Count();
	file.Write((const char*) &num_obj, sizeof( int ));
	gInterface->ProgressStart(_T("Exporting Geometry"), TRUE, 
							SkinExportProgressFunc, NULL );
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxFace* face_list;
		NxObject* object;
		Point3 center;
		float radius;
		unsigned long checksum;

		object = m_model->m_Objects[i];
		vert_list = object->m_Verts;
		face_list = object->m_Faces;
		num_faces = object->m_NumFaces;
		num_verts = object->m_NumVerts;

		
		// Write out the object's checksum		
		checksum = GenerateCRC( object->m_Name );
		file.Write((const char*) &checksum, sizeof( unsigned long ));		

		// Write out the flags which describe the object's properties
		object->m_Flags |= NxObject::mHASVERSIONINFO;
		file.Write((const char*) &object->m_Flags, sizeof( int ));

		// Write out the object's skeleton checksum
		file.Write((const char*) &object->m_SkeletonChecksum, sizeof( unsigned long ));		

		// Write out the number of sets of UVs
		file.Write((const char*) &object->m_NumUVSets, sizeof( int ));

		// Write out bounding box/sphere data
		file.Write((const char*) &object->m_BoundingBox.pmin, sizeof( Point3 ));
		file.Write((const char*) &object->m_BoundingBox.pmax, sizeof( Point3 ));

		center = object->m_BoundingBox.Center();
		radius = (( object->m_BoundingBox.pmax - object->m_BoundingBox.pmin ).Length()) / 2.0f;
		file.Write((const char*) &center, sizeof( Point3 ));
		file.Write((const char*) &radius, sizeof( float ));

		file.Write((const char*) &num_verts, sizeof( int ));
		for( j = 0; j < num_verts; j++ )
		{
			NxVertex* vert;			
			
			vert = &vert_list[j];

			// Write out first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					file.Write((const char*) &vert->m_TexCoord[k].x, sizeof( float ));
					file.Write((const char*) &vert->m_TexCoord[k].y, sizeof( float ));
				}
			}

			// Write out colors
			if( object->m_Flags & NxObject::mCOLORED )
			{
				if ( m_export_options.m_WarnOnLitVerts &&
					 !(vert->m_Color.r == 1.0f &&
					   vert->m_Color.g == 1.0f &&
					   vert->m_Color.b == 1.0f))
				{
					if( !m_LitVertWarned )
					{
						MessageBoxAll(gInterface->GetMAXHWnd(),"WARNING!  You are exporting a skin with lit verticies.","Skin exported with lit verticies",MB_ICONWARNING|MB_OK);
						m_LitVertWarned = true;
					}

					ReportWarning( "WARNING!  You are exporting a skin with lit verticies." );
				}

				file.Write((const char*) &vert->m_Color.r, sizeof( float ));
				file.Write((const char*) &vert->m_Color.g, sizeof( float ));
				file.Write((const char*) &vert->m_Color.b, sizeof( float ));
				file.Write((const char*) &vert->m_Color.a, sizeof( float ));
			}				

			// Write out normals
			if( object->m_Flags & ( NxObject::mNORMALS | NxObject::mSS_NORMALS ))
			{					
				file.Write((const char*) &vert->m_Normal.x, sizeof( float ));
				file.Write((const char*) &vert->m_Normal.y, sizeof( float ));
				file.Write((const char*) &vert->m_Normal.z, sizeof( float ));
			}

			// Write out weighting info
			if( object->m_Flags & NxObject::mSKINNED )
			{
				if( vert->m_NumWeights == 0 )
				{
					int l;

					l = 1;
				}
				file.Write((const char*) &vert->m_Weight[0], sizeof( float ));
				file.Write((const char*) &vert->m_Weight[1], sizeof( float ));
				file.Write((const char*) &vert->m_Weight[2], sizeof( float ));

				file.Write((const char*) &vert->m_WeightedIndex[0], sizeof( int ));
				file.Write((const char*) &vert->m_WeightedIndex[1], sizeof( int ));
				file.Write((const char*) &vert->m_WeightedIndex[2], sizeof( int ));

				if ( object->m_Flags & NxObject::mHAS4WEIGHTS)
				{
					file.Write((const char*) &vert->m_Weight[3], sizeof( float ));
					file.Write((const char*) &vert->m_WeightedIndex[3], sizeof( int ));
				}

				fprintf(fpWeights, "%i: Weights: %f, %f, %f, %f  Index: %i, %i, %i, %i\n", j,
					                                                   vert->m_Weight[0],
																	   vert->m_Weight[1],
																	   vert->m_Weight[2],
																	   vert->m_Weight[3],
																	   vert->m_WeightedIndex[0],
																	   vert->m_WeightedIndex[1],
																	   vert->m_WeightedIndex[2],
																	   vert->m_WeightedIndex[3]);
			}

			// Write out vc wibble data
			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				file.Write((const char*) &vert->m_WibbleIndex, sizeof( char ));
				file.Write((const char*) &vert->m_WibbleOffset, sizeof( char ));	
			}

			// Write out positions				
			file.Write((const char*) &vert->m_Pos.x, sizeof( float ));
			file.Write((const char*) &vert->m_Pos.y, sizeof( float ));
			file.Write((const char*) &vert->m_Pos.z, sizeof( float ));		

			fprintf(fpWeights, "%i: Pos: (%f,%f,%f)\n", j,
				                                        vert->m_Pos.x,
														vert->m_Pos.y,
														vert->m_Pos.z);
		}

		fclose(fpWeights);

		file.Write((const char*) &num_faces, sizeof( int ));
		for( j = 0; j < num_faces; j++ )
		{
			// Write out the material checksum for this face
			file.Write((const char*) &face_list[j].m_MatChecksum, sizeof( unsigned long ));

			// Write out the face normal
			file.Write((const char*) &face_list[j].m_Normal, sizeof( Point3 ));

			// Versioning data isn't saved with NxObject.  So we are using this flag
			// to indicate the addition of CAS Face data
			face_list[j].m_FaceFlags |= mFD_CASFACEFLAGSEXIST;
			face_list[j].m_MinFaceFlags |= mFD_CASFACEFLAGSEXIST;

			// Write out the face flags
			file.Write((const char*) &face_list[j].m_FaceFlags, sizeof( FlagType ));

			// Write out the minimalist face flags
			file.Write((const char*) &face_list[j].m_MinFaceFlags, sizeof( FlagType ));

			// Write out the CAS face flags
			file.Write((const char*) &face_list[j].m_CASFaceFlags, sizeof( CASFlagType ));

			// Write out the vertex indices
			file.Write((const char*) face_list[j].m_Vertex, sizeof( int ) * 3 );
		}

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Write out CAS face removal flags
			file.Write((const char*)&object->m_CASRemoveFlags, sizeof( int ));
		}

		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Write out the KBias
			file.Write((const char*)&object->m_KBias,sizeof( float ));
		}

		// Write out LOD data if appropriate
		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Write out the LOD info
			file.Write((const char*)&object->m_LODVersion,sizeof(unsigned int));
			file.Write((const char*)&object->m_LODFlags,sizeof(int));

			if (object->m_LODFlags & NxObject::mMASTER)
			{
				file.Write((const char*)&object->m_LODMaster.m_NumLODLevels,sizeof(int));
				file.Write((const char*)object->m_LODMaster.m_LODLevels,sizeof(NxLODLevel) * object->m_LODMaster.m_NumLODLevels);
			}

			if (object->m_LODFlags & NxObject::mSLAVE)
			{
				file.Write((const char*)&object->m_LODSlave.m_masterCRC,sizeof(int));
			}
		}

		if (object->m_Flags & NxObject::mHASINTLODINFO)
		{
			file.Write((const char*)&object->m_LODLevels,sizeof(int));
			
			for(int lodlev = 0; lodlev < object->m_LODLevels; lodlev++)
			{
				file.Write((const char*)&object->m_LODinfo[lodlev].numFaces, sizeof(int));

				file.Write((const char*)object->m_LODinfo[lodlev].faces,
					       sizeof(NxLODFace) * object->m_LODinfo[lodlev].numFaces);
			}

			/*
			// Dump LOD data to debug file
			FILE* fp = fopen("c:\\blah.txt", "w");

			fprintf(fp, "Number of LOD levels: %i\n", object->m_LODLevels);
			
			for(lodlev = 0; lodlev < object->m_LODLevels;lodlev++)
			{
				fprintf(fp, "NumFaces: %i\n", object->m_LODinfo[lodlev].numFaces);

				for(int curface = 0; curface < object->m_LODinfo[lodlev].numFaces; curface++)
				{
					fprintf(fp, "\tFace %i: A: %i B: %i C: %i\n", 
						    curface,
							object->m_LODinfo[lodlev].faces[curface].v[0],
							object->m_LODinfo[lodlev].faces[curface].v[1],
							object->m_LODinfo[lodlev].faces[curface].v[2]);

					fprintf(fp, "\t(A: %i B: %i C: %i)\n",
							object->m_Verts[object->m_LODinfo[lodlev].faces[curface].v[0]].m_MaxVert,
							object->m_Verts[object->m_LODinfo[lodlev].faces[curface].v[1]].m_MaxVert,
							object->m_Verts[object->m_LODinfo[lodlev].faces[curface].v[2]].m_MaxVert);
				}
			}

			fclose(fp);
			*/
		}

		if (object->m_Flags & NxObject::mBILLBOARD)
		{
			file.Write((const char*)&object->m_BillboardType, sizeof(NxObject::BillboardType));
			file.Write((const char*)&object->m_BillboardOrigin, sizeof(Point3));
			file.Write((const char*)&object->m_PivotPos, sizeof(Point3));
			file.Write((const char*)&object->m_PivotAxis, sizeof(Point3));
		}

		// Write out version info
		file.Write((const char*)&object->m_Version,sizeof(unsigned int));

		// Write out relative info v1
		file.Write((const char*)&object->m_ParentCRC,sizeof(unsigned long));
		file.Write((const char*)&object->m_NumChildren,sizeof(int));
		file.Write((const char*)object->m_ChildCRCs,sizeof(unsigned long) * object->m_NumChildren);

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			//file.close();
			return false;
		}
		gInterface->ProgressUpdate( i * 100 / num_obj );
	}

	gInterface->ProgressEnd();
	
	return true;
}

bool	SkinExporter::SpawnPostExportProcessor( void )
{
	int result;
	char *args[5];
	char path[_MAX_PATH], conv_path[_MAX_PATH];	
	TSTR scene_name, scene_dir, base_name,tex_name;
	char* project_root;
	
	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		if (strSkinName.Length() > 0)
			base_name = strSkinName;
		else
			base_name = m_export_options.m_SkinName;
	}	

	/*
	scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;	
	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	*/

	if (m_export_options.m_DirPath.Length()==0)
		scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
	else
		scene_dir = m_export_options.m_DirPath;

	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	tex_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".tex" );

	sprintf( path, "-f%s", scene_name );
	sprintf( conv_path, "%s\\bin\\win32\\sceneconv.exe", project_root );

	args[0] = conv_path;
	if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		args[1] = "-px";
	}
	else 
	{
		args[1] = "-pp";
	}
	args[2] = path;	

	if (UseSceneConvOptimize())
	{
		args[3] = "-o";
		args[4] = NULL;
	}
	else
		args[3] = NULL; 

	//result = spawnv( _P_WAIT, args[0], args );
	result = ExecuteCommand( args );

	if (result < 0)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Sceneconv has failed.  If you are exporting without textures you should re-export with textures.", "SceneConv Failed", MB_ICONSTOP|MB_OK);
		return false;
	}

	if( m_export_options.m_ViewUsgFile )
	{
		char usg_file[_MAX_PATH];
		char* args[3];

		sprintf( usg_file, "%s/%s.usg.ps2", scene_dir, base_name );
		args[0] = "notepad.exe";
		args[1] = usg_file;
		args[2] = NULL;

		ExecuteCommand( args, false );
	}

	if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];

		sprintf( path, "%s.xbx", scene_name );
		sprintf( copy_path, "%s\\bin\\win32\\copyx.bat", project_root );
		args[0] = copy_path;
		args[1] = path;
		args[2] = NULL;

		//result = spawnv( _P_WAIT, args[0], args );
		result = ExecuteCommand( args );

		sprintf( path, "%s.xbx", tex_name );
		args[1] = path;
		
		//result = spawnv( _P_WAIT, args[0], args );
		result = ExecuteCommand( args );
	}
	else
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];
		char* proj_root, *path_ptr;

		proj_root = getenv( APP_ENV );

		// We need to copy the .scn.ngc file to the ndata directory
		sprintf( path, "%s.ngc", scene_name );
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );

		// We need to copy the .scn.ngc.gd file to the ndata directory
		sprintf( path, "%s.ngc.gd", scene_name);
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );

		// We need to copy the .tex.ngc file to the ndata directory
		sprintf( path, "%s.ngc", tex_name );
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );		
	}
	return true;
}

bool	SkinExporter::SaveTextures( void )
{
	TSTR texfile_name, usgfile_name, base_name;
	char* project_root;	
	ITextureExporter* tex_exp;
	
	tex_exp = GetTextureExporter();
	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_SkinName;
	}

	if (strlen(m_export_options.m_DirPath)>0)
	{
		texfile_name = TSTR(m_export_options.m_DirPath) + TSTR( "/" ) + base_name + TSTR( ".tex" );
		usgfile_name = TSTR(m_export_options.m_DirPath) + TSTR( "/" )+ base_name + TSTR( ".usg" );

		if (m_export_options.m_DirWarn)
			if (!tex_exp->VerifyTexturePaths( base_name ))
				return false;
	}
	else
	{
		texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name + TSTR( "/" ) + base_name + TSTR( ".tex" );
		usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name + TSTR( "/" ) + base_name + TSTR( ".usg" );

		if (m_export_options.m_DirWarn)
			if (!tex_exp->VerifyTexturePaths( TSTR( "models/" ) + base_name ))
				return false;
	}
	
	return tex_exp->SaveTextureDictionary( texfile_name, usgfile_name );
}

bool	SkinExporter::SaveSkin( char* skinName )
{
	//fstream file;
	IoUtils::CVirtualOutputFile file;
	TSTR skn_name, skin_dir, base_name;
	char* project_root;
	int version;
	DWORD attribs;
	bool sky_export;
	
	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";

		skin_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
		//CreateDirectory( skin_dir, NULL );	
		MDir(skin_dir);
		CreateDirectory( "c:/temp", NULL );	// for image processing

		skn_name = skin_dir + TSTR( "/" ) + TSTR( base_name ) + TSTR( ".skin" );
	}
	else
	{
		if (skinName)
			base_name = skinName;
		else
			base_name = m_export_options.m_SkinName;
	}

	if (m_export_options.m_DirPath.Length()==0)
	{
		skin_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
		//CreateDirectory( skin_dir, NULL );
		MDir(skin_dir);
		CreateDirectory( "c:/temp", NULL );	// for image processing

		skn_name = skin_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	}
	else
	{
		skin_dir = m_export_options.m_DirPath;
		//CreateDirectory( skin_dir, NULL );	
		MDir(skin_dir);
		CreateDirectory( "c:/temp", NULL );	// for image processing
		
		skn_name = skin_dir + TSTR( "/" ) + base_name + TSTR( ".skin" );
	}

	attribs = GetFileAttributes( skn_name );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			int result;
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", skn_name );
			result = MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_CHECKOUT_UNLOCK );
			if( result == IDWRITABLE )
			{
				// break the lock, if necessary
				SetFileAttributes( skn_name, FILE_ATTRIBUTE_NORMAL );
			}
			else if( result == IDCHECKOUT )
			{
				char *args[4];
				int process_result;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = skn_name;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );
				process_result = ExecuteCommand( args );
				attribs = GetFileAttributes( skn_name );
				if(( attribs & FILE_ATTRIBUTE_READONLY ) || ( attribs == -1 ))
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Skin",MB_ICONSTOP|MB_OK);
					return false;
				}
			}
			else
			{
				return false;
			}
		}
	}

	
	//file.open( skn_name, ios::out | ios::binary );
	file.Init( vMAX_SKIN_FILE_SIZE );

	version = NxMaterial::vVERSION_NUMBER;
	file.Write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	file.Write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	file.Write((const char*) &version, sizeof( int ));
	sky_export = false;
	file.Write((const char*) &sky_export, sizeof( bool ));
	if( SaveMaterials( file, m_model ) == false )
	{
		//file.close();
		file.Uninit();
		return false;
	}

	if( SaveGeometry( file ) == false )
	{
		//file.close();
		file.Uninit();
		return false;	
	}	

	//file.close();
	file.Save( skn_name );
	return true;	
}

bool	SkinExporter::Save( void )
{
	if (strSkinName.Length() > 0)
		return( SaveSkin(strSkinName) && SaveTextures());
	else
		return( SaveSkin() && SaveTextures());		
}

bool	SkinExporter::DoIndividualSkinExport( char* skinName )
{
	CStr export_name;
	IModelExporter* model_exp;
	ITextureExporter* tex_exp;

	model_exp = GetModelExporter();
	tex_exp = GetTextureExporter();
	tex_exp->Reset();	
	
	if( m_exportable_nodes.Count() == 0 )
	{
		// Nothing to export
		return false;
	}

	if( !m_exportable_nodes[0]->GetUserPropString( "ExportName", export_name ) &&
		!m_export_options.m_ExportBySkinName ||
		m_export_options.m_ExportBySkinName && !m_exportable_nodes[0]->GetUserPropString( "ExportName", export_name ))
		//m_export_options.m_ExportBySkinName && !m_exportable_nodes[0]->GetUserPropString( "SkinExportName", export_name ))
	{
		// If the object has no name, don't export it
		ShowSkinExportErrorMessage( "You must assign the ExportName property to this object" );
		return false;
	}

	if (skinName)
		strcpy( m_export_options.m_SkinName, skinName );
	else
		strcpy( m_export_options.m_SkinName, export_name );

	// At this point, we have gathered a list of nodes which represent the exportable geometry.
	// Now parse these nodes and store relavent data in our own generic database
	ModelExportOptions model_exp_options;
	GetModelExportOptions(&model_exp_options);

	model_exp_options.m_AlwaysLOD      = m_export_options.m_AlwaysLOD;
	model_exp_options.m_LODLowerLimit  = m_export_options.m_LODLowerLimit;
	model_exp_options.m_LODNumLevels   = m_export_options.m_LODNumLevels;
	model_exp_options.m_LODProgressive = m_export_options.m_LODProgressive;

	model_exp->SetExportOptions(&model_exp_options);

	m_model = model_exp->ExtractModelData( m_exportable_nodes, true, m_export_options.m_Scale );
	// If something went wrong or it was cancelled, don't continue
	if( m_model == NULL )
	{
		return false;
	}

	// Load the textures needed for the current export set
	tex_exp->LoadTextureData();

	if (skinName)
		strSkinName = skinName;
	else
		strSkinName = "";

	// Here output the model and its materials, either via the network or to disk
	if( Save())
	{		
		if( SpawnPostExportProcessor())
		{
			// Here do quickview stuff
			if( m_export_type == vRESET_AND_RELOAD ||
				m_export_type == vRESET_AND_RELOAD_NOTEX)
			{
				char exec_str[128];
				char *viewer_root;

				if( viewer_root = getenv( APP_ENV ))
				{			
					char viewer_app[_MAX_DIR + _MAX_FNAME];

					sprintf( viewer_app, "%s\\build\\NGPSgnu\\%s", viewer_root, vVIEWER_APP_NAME );
					sprintf( exec_str, "ps2run -r %s", viewer_app );
					WinExec( exec_str, SW_HIDE );
				}
			}
			else if( m_export_type == vQUICK_VIEW )
			{
				Net::MsgQuickview msg;

				// The scn, tex and usg files have been exported to temp "quick" files.
				// Tell the engine, via the network, to load these files into the current scene
				sprintf( msg.m_Filename, m_export_options.m_SkinName );

				if( GetQuickviewPlatform() == vQUICKVIEW_PS2 )
				{
					Net::MsgDesc msgdesc;
					msgdesc.m_Id     = Net::vMSG_ID_QUICKVIEW;
					msgdesc.m_Length = sizeof( Net::MsgQuickview );
					msgdesc.m_Data   = &msg;

					gClient->EnqueueMessageToServer( &msgdesc );
				}
				else if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
				{
					char* server_ip;

					server_ip = getenv( vXBOX_SERVER_IP_VARIABLE );
					if( server_ip == NULL )
					{
						MessageBox( gInterface->GetMAXHWnd(), "You must set up your XBOX_VIEWER_IP environment variable if you want real-time updates",
										"Warning", MB_OK|MB_TASKMODAL);		
					}
					else
					{
						gClient->SendMessageTo( Net::vMSG_ID_QUICKVIEW, sizeof( Net::MsgQuickview ), &msg,
												inet_addr( server_ip ), Net::vEXPORT_COMM_PORT, 0 );						
					}
				}				
				else	// NGC
				{
					char* server_ip;

					server_ip = getenv( vNGC_SERVER_IP_VARIABLE );
					if( server_ip == NULL )
					{
						MessageBox( gInterface->GetMAXHWnd(), "You must set up your NGC_VIEWER_IP environment variable if you want real-time updates",
										"Warning", MB_OK|MB_TASKMODAL);		
					}
					else
					{
						char msg_data[Net::Manager::vMAX_PACKET_SIZE];
						bool result;
						char* byte_size;
						unsigned short msg_len;

						msg_len = sizeof( Net::MsgQuickview );
						byte_size = (char*) &msg_len;
						msg_data[0] = Net::vMSG_ID_QUICKVIEW;
						msg_data[2] = *byte_size;
						byte_size++;
						msg_data[1] = *byte_size;
						memcpy( &msg_data[3], &msg, msg_len );					
						result = gClient->SendTo( inet_addr( server_ip ), Net::vEXPORT_COMM_PORT, 
								msg_data, msg_len + Net::Manager::vMSG_HEADER_LENGTH_WITH_SIZE, 0 );
					}
				}				
			}
		}
	}

	if (m_model)
	{
		delete m_model;
		m_model = NULL;
	}
	
	return true;
}

bool	SkinExporter::DoExport( ExportType type )
{
	// Reset all warning mesages
	MessageBoxResetAll();
	m_export_type = type;

	m_LitVertWarned = false;
	GetSkinExportOptions( &m_export_options );
	
	if (m_export_options.m_ExportBySkinName)
	{
		//INodeTab nodeList;
		//GetAllNodes(nodeList);
		EnumerateExportableNodes();
		//nodeList = m_exportable_nodes;

		for(int i = 0; i < m_exportable_nodes.Count(); i++)
		{
			CStr strExportName;
			//if (!nodeList[i]->GetUserPropString("SkinExportName", strExportName))
			if (!m_exportable_nodes[i]->GetUserPropString("ExportName", strExportName))
				continue;

			// Ensure all child nodes exist within the export system
			//GetAllNodes(m_exportable_nodes, nodeList[i]);
			//EnumerateExportableNodes();
			
			if (!DoIndividualSkinExport(strExportName))
				return false;
		}
	}
	else
	{
		EnumerateExportableNodes();
		
		if (!DoIndividualSkinExport())
			return false;
	}

	return true;
}

int	get_remapped_vertex_index( INode* pNode, int nxVertexIndex )
{
	Object* pObject = pNode->EvalWorldState(0).obj;
	MaxAssert( pObject->CanConvertToType( triObjectClassID ));
	TriObject* pTriObject = (TriObject*) pObject->ConvertToType( 0, triObjectClassID );
	Mesh* pMesh = &pTriObject->mesh;

	// given the nxvertex index, we can get the face number
	int faceNum = nxVertexIndex / 3;
	int vertNum = nxVertexIndex % 3;

	// given the face number, we can get the MAX vertex index
	MaxAssert( faceNum >= 0 && faceNum < pMesh->getNumFaces() );
	Face* pFace = &pMesh->faces[faceNum];

	int iResult = pFace->v[vertNum];

	if (pTriObject != pObject)
		pTriObject->DeleteThis();

	return iResult;
}

int get_remapped_vertex_index( Mesh* pMesh, int nxVertexIndex )
{
	// given the nxvertex index, we can get the face number
	int faceNum = nxVertexIndex / 3;
	int vertNum = nxVertexIndex % 3;

	// given the face number, we can get the MAX vertex index
	MaxAssert( faceNum >= 0 && faceNum < pMesh->getNumFaces() );
	Face* pFace = &pMesh->faces[faceNum];

	return pFace->v[vertNum];
}

bool	SkinExporter::GetSkinData( NxObject* pObject, INode* pNode, unsigned long& checksum, INode* pNodeRoot )
{
    // Validate pointers
    MaxAssert( pObject );
    MaxAssert( pNode );

	int multiResMode = DisableMultiRes(pNode);

	Modifier* pPhysiqueModifier = FindPhysiqueModifier( pNode );
    
	checksum = 0;
    if ( !pPhysiqueModifier )
    {
        ShowSkinExportErrorMessage( "Selected object does not contain physique modifier" );

		if (multiResMode)
			EnableMultiRes(pNode);

        return false;
    }
    
    // should be able to grab the interface to the exporters safely...
    IPhysiqueExport* pPhysiqueExport = (IPhysiqueExport*)pPhysiqueModifier->GetInterface(I_PHYINTERFACE);    
    MaxAssert( pPhysiqueExport );

    // create a ModContext Export Interface for the specific node of the Physique Modifier
    IPhyContextExport* pModContextExport = (IPhyContextExport*)pPhysiqueExport->GetContextInterface(pNode);
    MaxAssert( pModContextExport );

    // as of CS2, only rigid vertices were supported by Character Studio
    // the new version of CS may have better support, but that's a feature
    // that can be implemented at a later date...
    pModContextExport->ConvertToRigid( true );
    pModContextExport->AllowBlending( true );

    int numVerts = pObject->m_NumVerts;//pModContextExport->GetNumberVertices();    

    // the physique modifier should have reported the
    // same number of verts as the NxObject
//    MaxAssert( pObject->m_NumVerts == numVerts );

    bool badVertexTypeFound = false;
	bool tooManyWeightsFound = false;
	bool exceededMaxUsedWeights = false;
	int  errWeights;
	int  i;

	CSkeletonData* pSkeletonData = NULL;

	Object*       obj = pNode->EvalWorldState(0).obj;
	TriObject* triobj = NULL;
	Mesh*       pMesh = NULL;

	if (obj->CanConvertToType(triObjectClassID))
	{
		triobj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
		pMesh  = &triobj->mesh;
	}

    for ( i = 0; i < numVerts; i++ )
    {
		char sMsg[256];
		sprintf(sMsg, "Export PHY: Vert %i\n", i);
		OutputDebugString(sMsg);

		// AML: This version was reconverting the mesh everytime it grabbed a vert and
		//      would eventually crash running out of memory as the triobject was never deleted
		//int remapped_index = get_remapped_vertex_index( pNode, i );
		int remapped_index = get_remapped_vertex_index( pMesh, i );

        // GJ:  Theoretically, we should be getting the
        // verts in the same order as grabbing them from the mesh.
        // If not, we'll have to re-map them somehow (perhaps
        // by comparing the actual coordinates).

        NxVertex* pNxVertex = &(pObject->m_Verts[i]);

        IPhyVertexExport* pVertexExport;
        pVertexExport = pModContextExport->GetVertexInterface( remapped_index );

		if (!pVertexExport)
		{
			OutputDebugString("\\Export\\SkinExport.cpp: Physique pVertexExport interface not found!");

			if (multiResMode)
				EnableMultiRes(pNode);
			
			return false;
		}

        MaxAssert( pVertexExport );

        // In THPS3, the exporter had a max limit of 4 weights per vertex.
        // Now, the exporter will not care about the limit;
        // it's up to the platform-specific processor to compress
        // them down to N verts.

        switch ( pVertexExport->GetVertexType() )
        {
            case RIGID_BLENDED_TYPE:
            {
                pNxVertex->m_Weighted = true;
                IPhyBlendedRigidVertex* pBlendedVertex = (IPhyBlendedRigidVertex*)pVertexExport;
                int numWeights = pBlendedVertex->GetNumberNodes();

				char sbuf[256];
				sprintf(sbuf, "Weights: %i\n", numWeights);
				OutputDebugString(sbuf);

				if ( numWeights > NxVertex::vMAX_WEIGHTS_PER_VERTEX )
				{
					tooManyWeightsFound = true;
					errWeights=numWeights;
					break;
				}
				else if( numWeights > NxVertex::vMAX_USED_WEIGHTS_PER_VERTEX )				
				{
					exceededMaxUsedWeights = true;
				}

				pNxVertex->m_NumWeights = numWeights;
				float runningWeightTotal = 0.0f;
				for ( int j = 0; j < numWeights; j++ )
                {
                    INode* pBone = pBlendedVertex->GetNode(j);
					if ( !pSkeletonData )
					{
						if (pNodeRoot)
							pSkeletonData = new CSkeletonData( pNodeRoot, true, false, true);
						else
							pSkeletonData = new CSkeletonData( pBone );
					}
                    pNxVertex->m_WeightedIndex[j] = pSkeletonData->GetBoneIndex( pBone );
                    pNxVertex->m_Weight[j] = pBlendedVertex->GetWeight( j );
					runningWeightTotal += pBlendedVertex->GetWeight( j );
                }
				// the total weight for all the verts should equal 1.0
				if ((short)( runningWeightTotal * 4096.0f + 0.5f ) != 4096)
				{
					MessageBox(gInterface->GetMAXHWnd(), "Skin doesn't have proper weighting on verticies.  Verify vertex weighting and reexport.", "Skin Exporter", MB_ICONWARNING|MB_OK);
					return false;
				}

				MaxAssert( (short)( runningWeightTotal * 4096.0f + 0.5f ) == 4096 );
            }
            break;
            case RIGID_TYPE:
            {
                pNxVertex->m_Weighted = true;
                IPhyRigidVertex* pRigidVertex = (IPhyRigidVertex*)pVertexExport;
                INode* pBone = pRigidVertex->GetNode();
				if ( !pSkeletonData )
				{
					if (pNodeRoot)
						pSkeletonData = new CSkeletonData( pNodeRoot, true, false, true );
					else
						pSkeletonData = new CSkeletonData( pBone );
				}
				pNxVertex->m_NumWeights = 1;

				// DEBUG:
				CStr boneName = pBone->GetName();
				char strMsg[256];
				sprintf(strMsg, "Assigning weighting for node '%s'.\n", (char*)pBone->GetName());
				OutputDebugString(strMsg);

                pNxVertex->m_WeightedIndex[0] = pSkeletonData->GetBoneIndex( pBone );
                pNxVertex->m_Weight[0] = 1.0f;
            }
            break;
            default:  
                // unrecognized type
                badVertexTypeFound = true;
                break;
        }

        pModContextExport->ReleaseVertexInterface( pVertexExport );
        pVertexExport = NULL;

        // NOT SURE IF THE FOLLOWING LINE IS NECESSARY:
        pModContextExport->ReleaseVertexInterface( pVertexExport );
        pVertexExport = NULL;
    }

	// Clean up the allocated TriObject if necessary
	if (triobj != obj)
		triobj->DeleteThis();

	checksum = pSkeletonData->GetChecksum();
	delete pSkeletonData;
	
    pPhysiqueExport->ReleaseContextInterface( pModContextExport );            

    pPhysiqueModifier->ReleaseInterface( I_PHYINTERFACE, pPhysiqueExport );                    

    // TODO:
    // NORMALIZE THE WEIGHTS?
    // CHECK FOR > MAX WEIGHTS
    
    if ( badVertexTypeFound )
    {
        ShowSkinExportErrorMessage( "Unrecognized vertex type found (must be rigid or rigid-blended." );

		if (multiResMode)
			EnableMultiRes(pNode);
        
		return false;
    }

	if ( tooManyWeightsFound )
    {
		char strErr[256];
		sprintf(strErr,"Too many weights found on vertex.\nNode Name: %s\nVert Index: %i\nNum Weights: %i (Limit: %i)",
			    (char*)pNode->GetName(),i,errWeights,NxVertex::vMAX_WEIGHTS_PER_VERTEX);

        ShowSkinExportErrorMessage( strErr );

		// Force the entire export to abort
		gbAbort=true;

		if (multiResMode)
			EnableMultiRes(pNode);

        return false;
    }
	if( exceededMaxUsedWeights )
	{
		char strErr[256];
		sprintf(strErr,"Some vertices are weighted to more than %d bones. These vertices will be normalized to use %d bones. Node Name: %s",
			NxVertex::vMAX_USED_WEIGHTS_PER_VERTEX, NxVertex::vMAX_USED_WEIGHTS_PER_VERTEX, (char*)pNode->GetName());

        ShowSkinExportErrorMessage( strErr );

	}

	if (multiResMode)
		EnableMultiRes(pNode);

//	EvalLODData( pNode, pObject );

	return true;
}

void	SkinExporter::EvalLODData(Modifier* mod, INode* node, NxObject* nxobj)
{
	IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);

	// Go through and evaluate the object at different LOD levels
	
	float start = m_export_options.m_LODLowerLimit;
	float step  = (100.0f - start) / (float)m_export_options.m_LODNumLevels;

	int idx = 0;
	Interval interval = FOREVER;

	float origper, origcount;
	pblk->GetValue(PB_VERTEXPERCENT, 0, origper, interval);
	interval = FOREVER;
	pblk->GetValue(PB_VERTEXCOUNT, 0, origcount, interval);

	if (m_export_options.m_LODProgressive)
	{
		Object* obj = node->EvalWorldState(0).obj;
		if (obj->CanConvertToType(triObjectClassID))
		{
			// Determine the number of verticies within the mesh
			TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			int nverts = mesh.numVerts;

			if (triObj != obj)
				triObj->DeleteThis();

			nxobj->m_LODLevels = nverts;
			nxobj->m_LODinfo   = new NxLODInfo[nverts];

			for(int cvert = nverts; cvert > 0; cvert--)
			{
				pblk->SetValue(PB_VERTEXCOUNT, 0, cvert);

				Object* obj = node->EvalWorldState(0).obj;
				if (obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

					Mesh& mesh = triObj->GetMesh();

					nxobj->m_LODinfo[idx].numFaces = mesh.numFaces;
					nxobj->m_LODinfo[idx].faces = new NxLODFace[mesh.numFaces];

					for(int cface = 0; cface < mesh.numFaces; cface++)
					{
						// Converts from MAX Vertex indicies to the indicies of the expanded verts
						nxobj->m_LODinfo[idx].faces[cface].v[0] = nxobj->m_MaxVerts[mesh.faces[cface].v[0]];
						nxobj->m_LODinfo[idx].faces[cface].v[1] = nxobj->m_MaxVerts[mesh.faces[cface].v[1]];
						nxobj->m_LODinfo[idx].faces[cface].v[2] = nxobj->m_MaxVerts[mesh.faces[cface].v[2]];

						Point3 vert;
						
						vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_MaxVert];
						
						if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_Pos)
							MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

						vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_MaxVert];
						
						if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_Pos)
							MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

						vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_MaxVert];
						
						if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_Pos)
							MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);
					}

					if (triObj != obj)
						triObj->DeleteThis();

					idx++;
				}
			}
		}
	}
	else
	{
		nxobj->m_LODLevels = m_export_options.m_LODNumLevels;
		nxobj->m_LODinfo   = new NxLODInfo[m_export_options.m_LODNumLevels];

		for(float per = 100.0f; per >= start; per -= step)
		{
			pblk->SetValue(PB_VERTEXPERCENT, 0, per);

			Object* obj = node->EvalWorldState(0).obj;
			if (obj->CanConvertToType(triObjectClassID))
			{
				TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

				Mesh& mesh = triObj->GetMesh();

				nxobj->m_LODinfo[idx].numFaces = mesh.numFaces;
				nxobj->m_LODinfo[idx].faces = new NxLODFace[mesh.numFaces];

				for(int cface = 0; cface < mesh.numFaces; cface++)
				{
					// Converts from MAX Vertex indicies to the indicies of the expanded verts
					nxobj->m_LODinfo[idx].faces[cface].v[0] = nxobj->m_MaxVerts[mesh.faces[cface].v[0]];
					nxobj->m_LODinfo[idx].faces[cface].v[1] = nxobj->m_MaxVerts[mesh.faces[cface].v[1]];
					nxobj->m_LODinfo[idx].faces[cface].v[2] = nxobj->m_MaxVerts[mesh.faces[cface].v[2]];

					Point3 vert;
					/*
					vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_MaxVert];
					
					if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_Pos)
						MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

					vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_MaxVert];
					
					if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_Pos)
						MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

					vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_MaxVert];
					
					if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_Pos)
						MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);
					*/
				}

				if (triObj != obj)
					triObj->DeleteThis();

				idx++;
			}
		}
	}

	nxobj->m_Flags |= NxObject::mHASINTLODINFO;

	pblk->SetValue(PB_VERTEXPERCENT, 0, origper);
	pblk->SetValue(PB_VERTEXCOUNT, 0, origcount);
}

int		SkinExporter::DisableMultiRes(INode* node)
{
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				int state = mod->IsEnabled();
				mod->DisableMod();
				return state;
			}
		}
	}	

	return FALSE;
}

int		SkinExporter::EnableMultiRes(INode* node)
{
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				int state = mod->IsEnabled();
				mod->EnableMod();
				return state;
			}
		}
	}
	
	return FALSE;
}

void	SkinExporter::EvalLODData(INode* node, NxObject* nxobj)
{
	// Check if the node has a MultiRes modifier applied to it
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				EvalLODData(mod, node, nxobj);
				return;
			}
		}
	}

	if (m_export_options.m_AlwaysLOD)
	{
		Modifier* mod = (Modifier*)CreateInstance(OSM_CLASS_ID, MULTIRES_CLASS_ID);
		ModContext* mc = new ModContext(new Matrix3(1), NULL, NULL);

		IDerivedObject* dobj;

		if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
		{
			dobj = (IDerivedObject*)obj;
			//dobj->AddModifier(mod, mc, dobj->NumModifiers());
			dobj->AddModifier(mod, mc);
		}
		else
		{
			dobj = CreateDerivedObject(obj);

			// Place at the bottom of the stack so it doesn't interfere with the physique modifier
			dobj->AddModifier(mod, mc, dobj->NumModifiers());
			node->SetObjectRef(dobj);
		}

		Object* cobj = node->EvalWorldState(0).obj;

		IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);
		pblk->SetValue(PB_GENERATE, 0, 1);	// Force MultiRes Generate Function
		EvalLODData(mod, node, nxobj);
		//dobj->DeleteModifier(dobj->NumModifiers()-1);
		//dobj->DeleteModifier(0);

		pblk->SetValue(PB_VERTEXPERCENT, 0, 100.0f);
	}
}
