 /**********************************************************************  
 *<
	FILE: PolyEdOps.cpp

	DESCRIPTION: Editable Polygon Mesh Object

	CREATED BY: Steve Anderson

	HISTORY: created Nov. 9, 1999

 *>	Copyright (c) 1999, All Rights Reserved.
 **********************************************************************/

#include "EPoly.h"
#include "PolyEdit.h"
#include "MeshDLib.h"
#include "macrorec.h"
#include "decomp.h"
#include "spline3d.h"
#include "splshape.h"
#include "shape.h"

namespace EditPoly
{

//---------------------------------------------------------
// EditPolyObject implementations of EPoly methods:

void EditPolyObject::LocalDataChanged (DWORD parts) {
	bool sel = (parts & PART_SELECT) ? TRUE : FALSE;
	bool topo = (parts & PART_TOPO) ? TRUE : FALSE;
	bool geom = (parts & PART_GEOM) ? TRUE : FALSE;
	bool vertCol = (parts & PART_VERTCOLOR) ? true : false;
	InvalidateTempData (parts);
	if (sel) {
		InvalidateNumberSelected ();
		InvalidateNamedSelDropDown ();
	}
	if (topo||sel||vertCol) InvalidateSurfaceUI ();
	if (topo) {
		SynchContArray(mm.numv);
	}
	if (geom||topo) {
		mm.InvalidateGeomCache ();
		mm.freeRVerts ();
	}
	subdivValid.SetEmpty ();
	NotifyDependents(FOREVER, parts, REFMSG_CHANGE);
}

void EditPolyObject::EpfnForceSubdivision () {
	SetFlag (EPOLY_FORCE);
	subdivValid.SetEmpty ();
	NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
}

void EditPolyObject::RefreshScreen () {
#ifdef _DEBUG
	if (!mm.CheckAllData ()) {
		mm.MNDebugPrint ();
		DbgAssert (0);
	}
	DbgAssert (mm.GetFlag (MN_MESH_FILLED_IN));
#endif
	if (ip) {
		ip->RedrawViews (ip->GetTime ());
		UpdateNamedSelDropDown ();
	}
}

int EditPolyObject::EpfnPropagateComponentFlags (int slTo, DWORD flTo, int slFrom, DWORD flFrom, bool ampersand, bool set, bool undoable) {
	if (undoable && theHold.Holding())
		theHold.Put (new ComponentFlagRestore(this, slTo));
	int res = mm.PropegateComponentFlags (slTo, flTo, slFrom, flFrom, ampersand, set);
	DWORD flag =0;
	if (flTo & MN_SEL) flag |= PART_SELECT;
	if (flTo & (MN_HIDDEN | MN_EDGE_INVIS)) flag |= PART_DISPLAY;
	if (flTo & MN_HIDDEN) flag |= PART_GEOM;	// Bounding box change -sca
	if (res && flag) LocalDataChanged (flag);
	return res;
}


bool EditPolyObject::EpfnHide (int msl, DWORD flag) {
	int i;
	switch (msl) {
	case MNM_SL_VERTEX:
		if (theHold.Holding())
			theHold.Put (new ComponentFlagRestore(this, MNM_SL_VERTEX));
		mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_HIDDEN, MNM_SL_VERTEX, flag);
		mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_SEL, MNM_SL_VERTEX, MN_HIDDEN, false, false);
		break;

	case MNM_SL_FACE:
		if (theHold.Holding()) {
			theHold.Put (new ComponentFlagRestore(this, MNM_SL_VERTEX));
			theHold.Put (new ComponentFlagRestore(this, MNM_SL_FACE));
		}
		// Hide the verts first:
		mm.PropegateComponentFlags (MNM_SL_VERTEX,
			MN_HIDDEN, MNM_SL_FACE, flag, TRUE);
		for (i=0; i<mm.numf; i++) {
			if (mm.f[i].GetFlag (MN_DEAD)) continue;
			if (mm.f[i].GetFlag (flag)) {
				mm.f[i].SetFlag (MN_HIDDEN);
				mm.f[i].ClearFlag (MN_SEL);
			}
		}
		break;

	default:
		return false;
	}
	LocalDataChanged (PART_DISPLAY|PART_SELECT|PART_GEOM);	// sca GEOM added because of bounding box change.
	return true;
}

bool EditPolyObject::EpfnUnhideAll (int msl) {
	int i;
	switch (msl) {
	case MNM_SL_VERTEX:
		if (theHold.Holding())
			theHold.Put (new ComponentFlagRestore (this, MNM_SL_VERTEX));
		mm.ClearVFlags (MN_HIDDEN);
		break;

	case MNM_SL_FACE:
		if (theHold.Holding())  {
			theHold.Put (new ComponentFlagRestore (this, MNM_SL_FACE));
			theHold.Put (new ComponentFlagRestore (this, MNM_SL_VERTEX));
		}
		// We don't clear vertex hide flags on most vertices here, only those on previously hidden faces.
		// (to be consistent with EMesh behavior).
		for (i=0; i<mm.numf; i++) {
			if (mm.f[i].GetFlag (MN_DEAD)) continue;
			if (mm.f[i].GetFlag (MN_HIDDEN)) {
				for (int j=0; j<mm.f[i].deg; j++) mm.v[mm.f[i].vtx[j]].ClearFlag (MN_HIDDEN);
			}
			mm.f[i].ClearFlag (MN_HIDDEN);
		}
		break;

	default:
		return false;
	}
	LocalDataChanged (PART_DISPLAY|PART_GEOM);	// sca GEOM added because of bbox change.
	return true;
}

bool EditPolyObject::EpfnDelete (int msl, DWORD flag, bool delIsoVerts) {
	bool ret=false;
	switch (msl) {
	case MNM_SL_OBJECT:
		return false;
	case MNM_SL_VERTEX:
		ret = DeleteVerts (flag);
		if (ret) CollapseDeadStructs();
		break;
	case MNM_SL_EDGE:
		// Actually joins polygons together.
		ret = DeleteEdges (flag);
		if (ret) CollapseDeadStructs ();
		break;
	case MNM_SL_FACE:
		ret = DeleteFaces (flag, delIsoVerts);
		if (ret) CollapseDeadStructs ();
		break;
	}
	return ret;
}

void EditPolyObject::EpActionButtonOp (int opcode) {
	TSTR name;
	int i, index;
	bool delIsoVerts, ret;
	bool createCurveSmooth;
	int msl = meshSelLevel[selLevel];
	TimeValue t = ip ? ip->GetTime() : TimeValue (0);

	switch (opcode) {
	case epop_hide:
		theHold.Begin ();
		if (EpfnHide (msl, MN_SEL)) {
			theHold.Accept (GetString (IDS_HIDE_SELECTED));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_unhide:
		theHold.Begin ();
		if (EpfnUnhideAll (msl)) {
			theHold.Accept (GetString (IDS_UNHIDE_ALL));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_ns_copy:
		index = SelectNamedSet();
		NamedSelectionCopy (index);
		break;

	case epop_ns_paste:
		EpfnNamedSelectionPaste (true);
		break;

	case epop_cap:
		theHold.Begin ();
		if (!EpfnCapHoles ()) {
			theHold.Cancel ();
		} else {
			theHold.Accept (GetString (IDS_CAP));
			RefreshScreen ();
		}
		break;

	case epop_delete:
		if (msl == MNM_SL_FACE) {
			delIsoVerts = mm.PropegateComponentFlags (MNM_SL_VERTEX, 0, MNM_SL_FACE, MN_SEL, TRUE) ? true : false;
			if (ip && delIsoVerts) {
				TSTR str1 = GetString(IDS_DELETE_ISOLATED);
				TSTR str2 = GetString(IDS_DELETE_FACES);
				if (IDYES!=MessageBox (ip->GetMAXHWnd(), str1,str2, MB_ICONQUESTION|MB_YESNO)) {
					delIsoVerts = FALSE;
				}
			}
		}
		theHold.Begin ();
		if (!EpfnDelete (msl, MN_SEL, delIsoVerts)) {
			theHold.Cancel ();
		} else {
			switch (msl) {
			case MNM_SL_VERTEX:
				theHold.Accept (GetString (IDS_DELETE_VERTICES));
				break;
			case MNM_SL_EDGE:
				theHold.Accept (GetString (IDS_DELETE_EDGES));
				break;
			case MNM_SL_FACE:
				theHold.Accept (GetString (IDS_DELETE_FACES));
				break;
			}
			RefreshScreen ();
		}
		break;

	case epop_attach_list:
		if (ip) {
			AttachHitByName proc(this);
			ip->DoHitByNameDialog(&proc);
		}
		break;

	case epop_detach:
		if (!editObj || !ip) break;
		if (mm.VertexTempSel().NumberSet() == 0) break;	// defect 261157 (4)
		bool elem, asClone;
		if (GetDetachObjectName (ip, name, elem, asClone)) {
			if (elem) EpfnDetachToElement (msl, MN_SEL, asClone);
			else {
				ModContextList mcList;
				INodeTab nodes;
				ip->GetModContexts(mcList,nodes);
				EpfnDetachToObject (name, msl, MN_SEL, asClone, nodes[0], ip->GetTime());
				nodes.DisposeTemporary ();
			}
			RefreshScreen ();
		}
		break;

	case epop_split:
		if (selLevel != EP_SL_EDGE) break;
		theHold.Begin ();
		if (EpfnSplitEdges (MN_SEL)) {
			theHold.Accept (GetString (IDS_SPLIT_EDGES));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_break:
		if (selLevel != EP_SL_VERTEX) break;
		theHold.Begin ();
		if (EpfnBreakVerts ()) {
			theHold.Accept (GetString (IDS_BREAK_VERTS));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_collapse:
		if ((selLevel > EP_SL_OBJECT) && (selLevel<EP_SL_ELEMENT)) {
			theHold.Begin();
			if (EpfnCollapse (msl, MN_SEL)) {
				theHold.Accept(GetString(IDS_COLLAPSE));
				RefreshScreen ();
			} else theHold.Cancel ();
		}
		break;

	case epop_reset_plane:
		if (sliceMode && selLevel) {
			theHold.Begin ();
			EpResetSlicePlane ();
			theHold.Accept (GetString (IDS_RESET_SLICE_PLANE));
		}
		break;

	case epop_slice:
		if (sliceMode && selLevel) {
			theHold.Begin ();
			Point3 planeNormal, planeCenter;
			EpGetSlicePlane (planeNormal, planeCenter);
			if (EpfnSlice (planeNormal, planeCenter, msl==MNM_SL_FACE, MN_SEL)) {
				theHold.Accept (GetString (IDS_SCA_SLICE));
				RefreshScreen ();
			} else {
				theHold.Cancel ();
			}
		}
		break;

	case epop_weld_sel:
		int actionID, failureID;
		ret = false;
		theHold.Begin ();
		switch (meshSelLevel[selLevel]) {
		case MNM_SL_VERTEX:
			ret = EpfnWeldFlaggedVerts (MN_SEL)?true:false;
			actionID = IDS_WELD_VERTS;
			failureID = IDS_NO_VERTS_TO_WELD;
			break;
		case MNM_SL_EDGE:
			ret = EpfnWeldFlaggedEdges (MN_SEL)?true:false;
			actionID = IDS_WELD_EDGES;
			failureID = IDS_NO_EDGES_TO_WELD;
			break;
		}
		if (ret) {
			theHold.Accept (GetString (actionID));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
			if (ip) {
				TSTR buf1 = GetString (failureID);
				TSTR buf2 = GetString (actionID);
				MessageBox (ip->GetMAXHWnd(), buf1, buf2, MB_OK|MB_TASKMODAL);
			}
		}
		break;

	case epop_create_shape:
		if (meshSelLevel[selLevel] != MNM_SL_EDGE) break;
		for (i=0; i<mm.nume; i++) if (mm.e[i].GetFlag (MN_SEL)) break;
		if (i>= mm.nume) {
			TSTR buf1 = GetString(IDS_CREATECURVE);
			TSTR buf2 = GetString(IDS_NOEDGESSELECTED);
			MessageBox (ip->GetMAXHWnd(),buf2,buf1,MB_ICONEXCLAMATION|MB_OK);
			break;
		}
		if (GetCreateShapeName (ip, name, createCurveSmooth)) {
			theHold.Begin ();
			ModContextList mcList;
			INodeTab nodes;
			ip->GetModContexts(mcList,nodes);
			if (EpfnCreateShape (name, createCurveSmooth, nodes[0], MN_SEL)) {
				theHold.Accept (GetString(IDS_CREATECURVE));
				RefreshScreen ();
			} else {
				theHold.Cancel ();
			}
			nodes.DisposeTemporary ();
		}
		break;

	case epop_make_planar:
		theHold.Begin ();
		if (EpfnMakePlanar (msl, MN_SEL, t)) {
			theHold.Accept(GetString(IDS_MAKE_PLANAR));
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_align_view:
		theHold.Begin ();
		if (EpfnAlignToView (msl, MN_SEL)) {
			theHold.Accept(GetString(IDS_ALIGN_TO_VIEW));
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_align_grid:
		theHold.Begin ();
		if (EpfnAlignToGrid (msl, MN_SEL)) {
			theHold.Accept(GetString(IDS_ALIGN_TO_GRID));
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_remove_iso_verts:
		theHold.Begin ();
		if (!EpfnDeleteIsoVerts ()) {
			theHold.Cancel ();
		} else {
			theHold.Accept (GetString (IDS_DELETE_ISO_VERTS));
			RefreshScreen ();
		}
		break;

	case epop_meshsmooth:
		theHold.Begin();
		if (EpfnMeshSmooth (msl, MN_SEL)) {
			theHold.Accept(GetString(IDS_MESHSMOOTH));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_tessellate:
		theHold.Begin ();
		if (EpfnTessellate (msl, MN_SEL)) {
			theHold.Accept(GetString(IDS_TESSELLATE));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_update:
		EpfnForceSubdivision ();
		break;

	case epop_selby_vc:
		BOOL add, sub;
		add = GetKeyState(VK_CONTROL)<0;
		sub = GetKeyState(VK_MENU)<0;
		int selByIllum;
		pblock->GetValue (ep_vert_color_selby, t, selByIllum, FOREVER);
		EpfnSelectVertByColor (add, sub, selByIllum ? -1 : 0, t);
		break;

	case epop_retriangulate:
		if (meshSelLevel[selLevel] != MNM_SL_FACE) break;
		theHold.Begin ();
		if (EpfnRetriangulate (MN_SEL)) {
			theHold.Accept (GetString (IDS_RETRIANGULATE));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_flip_normals:
		if (selLevel != EP_SL_ELEMENT) break;
		theHold.Begin ();
		if (EpfnFlipNormals (MN_SEL)) {
			theHold.Accept (GetString (IDS_FLIP_NORMALS));
			RefreshScreen ();
		} else {
			theHold.Cancel ();
		}
		break;

	case epop_selby_matid:
		MtlID material;
		bool clear;
		if (GetSelectByMaterialParams (ip, material, clear)) {
			theHold.Begin();
			EpfnSelectByMat (material-1, clear, t);
			theHold.Accept(GetString(IDS_SELECT_BY_MATID));
			RefreshScreen ();
		}
		break;

	case epop_selby_smg:
		DWORD smBits, usedBits;
		GetSmoothingGroups (0, &usedBits);
		if (GetSelectBySmoothParams (ip, usedBits, smBits, clear)) {
			theHold.Begin ();
			EpfnSelectBySmoothGroup (smBits, clear, t);
			theHold.Accept(GetString(IDS_SEL_BY_SMGROUP));
			RefreshScreen ();
		}
		break;

	case epop_autosmooth:
		theHold.Begin ();
		EpfnAutoSmooth (t);
		theHold.Accept (GetString (IDS_AUTOSMOOTH));
		RefreshScreen ();
		break;

	case epop_clear_smg:
		DWORD bitmask=0;
		bitmask = ~bitmask;
		SetSmoothBits (0, bitmask, MN_SEL);
		break;
	}
}

void EditPolyObject::MoveSelection(int level, TimeValue t, Matrix3& partm,
								   Matrix3& tmAxis, Point3& val, BOOL localOrigin) {
	Transform (level, t, partm, tmAxis, localOrigin, TransMatrix(val), 0);
}

void EditPolyObject::Move (TimeValue t, Matrix3& partm, Matrix3& tmAxis, 
		Point3& val, BOOL localOrigin) {
	Transform(meshSelLevel[selLevel], t, partm, tmAxis, localOrigin, TransMatrix(val), 0);
	macroRecorder->FunctionCall(_T("move"), 2, 0, mr_meshsel, meshSelLevel[selLevel], this, mr_point3, &val);  // JBW : macrorecorder
}

void EditPolyObject::RotateSelection(int level, TimeValue t, Matrix3& partm,
									 Matrix3& tmAxis, Quat& val, BOOL localOrigin) {
	Matrix3 mat;
	val.MakeMatrix(mat);
	Transform(level, t, partm, tmAxis, localOrigin, mat, 1);
}

void EditPolyObject::Rotate (TimeValue t, Matrix3& partm, Matrix3& tmAxis, 
		Quat& val, BOOL localOrigin) {
	Matrix3 mat;
	val.MakeMatrix(mat);
	Transform(meshSelLevel[selLevel], t, partm, tmAxis, localOrigin, mat, 1);
	macroRecorder->FunctionCall(_T("rotate"), 2, 0, mr_meshsel, meshSelLevel[selLevel], this, mr_quat, &val);  // JBW : macrorecorder
}

void EditPolyObject::ScaleSelection(int level, TimeValue t, Matrix3& partm, 
									Matrix3& tmAxis, Point3& val, BOOL localOrigin) {
	Transform (level, t, partm, tmAxis, localOrigin, ScaleMatrix(val), 2);	
}

void EditPolyObject::Scale (TimeValue t, Matrix3& partm, Matrix3& tmAxis, 
		Point3& val, BOOL localOrigin) {
	Transform (meshSelLevel[selLevel], t, partm, tmAxis, localOrigin, ScaleMatrix(val), 2);	
	macroRecorder->FunctionCall(_T("scale"), 2, 0, mr_meshsel, meshSelLevel[selLevel], this, mr_point3, &val);  // JBW : macrorecorder
}

void EditPolyObject::Transform (int sl, TimeValue t, Matrix3& partm, Matrix3 tmAxis, 
		BOOL localOrigin, Matrix3 xfrm, int type) {
	if (!ip) return;
	if (sl == MNM_SL_OBJECT) return;

	if (sliceMode) {
		// Special case -- just transform slicing plane.
		theHold.Put (new TransformPlaneRestore (this));
		Matrix3 tm  = partm * Inverse(tmAxis);
		Matrix3 itm = Inverse(tm);
		Matrix3 myxfm = tm * xfrm * itm;
		Point3 myTrans, myScale;
		Quat myRot;
		DecomposeMatrix (myxfm, myTrans, myRot, myScale);
		float factor;
		switch (type) {
		case 0: sliceCenter += myTrans; break;
		case 1: sliceRot *= myRot; break;
		case 2:
			factor = (float) exp(log(myScale[0]*myScale[1]*myScale[2])/3.0);
			sliceSize *= factor;
			break;
		}
		NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
		ip->RedrawViews(ip->GetTime());
		return;
	}

	// Get our node transform
	ModContextList mcList;
	INodeTab nodes;
	ip->GetModContexts(mcList,nodes);
	// nodeTm is no longer needed due to changes below (-sca, 1/29/01)
	//Matrix3 nodeTm = nodes[0]->GetObjectTM(t);

	// Get axis type:
	int numAxis = ip->GetNumAxis();

	// Get soft selection parameters:
	int softSel, edgeIts=1, useEdgeDist=FALSE, affectBack=FALSE;
	float falloff, pinch, bubble;
	Interval frvr = FOREVER;
	pblock->GetValue (ep_ss_use, t, softSel, frvr);
	if (softSel) {
		pblock->GetValue (ep_ss_edist_use, t, useEdgeDist, frvr);
		if (useEdgeDist) pblock->GetValue (ep_ss_edist, t, edgeIts, frvr);
		pblock->GetValue (ep_ss_affect_back, t, affectBack, frvr);
		pblock->GetValue (ep_ss_falloff, t, falloff, frvr);
		pblock->GetValue (ep_ss_pinch, t, pinch, frvr);
		pblock->GetValue (ep_ss_bubble, t, bubble, frvr);
	}

	// Special case for vertices: Only individual axis when moving in local space
	if ((sl==MNM_SL_VERTEX) && (numAxis==NUMAXIS_INDIVIDUAL)) {
		if (ip->GetRefCoordSys()!=COORDS_LOCAL || 
			ip->GetCommandMode()->ID()!=CID_SUBOBJMOVE) {
			numAxis = NUMAXIS_ALL;
		}
	}

	// Selected vertices - either directly or indirectly through selected faces or edges.
	BitArray sel = GetMesh().VertexTempSel ();
	if (!sel.NumberSet()) {
		nodes.DisposeTemporary ();
		return;
	}

	int i, nv = GetMesh().numv;
	if (!nv) return;
	Tab<Point3> delta;
	delta.SetCount (nv);
	for (i=0; i<nv; i++) delta[i] = Point3(0,0,0);

	// Compute the transforms
	if (numAxis==NUMAXIS_INDIVIDUAL && sl != MNM_SL_VERTEX) {
		// Do each cluster one at a time
		int count;
		Tab<int> *vclust = NULL;
		if (sl == MNM_SL_EDGE) count = TempData()->EdgeClusters()->count;
		else count = TempData()->FaceClusters()->count;
		vclust = TempData()->VertexClusters(sl);
		float *clustDist=NULL, *ws=NULL;
		Tab<float> weightSum;
		Matrix3 tm, itm;
		if (softSel) {
			weightSum.SetCount(nv);
			ws = weightSum.Addr(0);
			for (i=0; i<nv; i++) ws[i] = 0.0f;
		}
		for (int j=0; j<count; j++) {
			tmAxis = ip->GetTransformAxis (nodes[0], j);
			tm  = partm * Inverse(tmAxis);
			itm = Inverse(tm);
			tm *= xfrm;
			if (softSel) clustDist = TempData()->ClusterDist(sl, MN_SEL, j, useEdgeDist, edgeIts)->Addr(0);
			for (i=0; i<nv; i++) {
				if (sel[i]) {
					if ((*vclust)[i]!=j) continue;
					Point3 & old = GetMesh().v[i].p;
					delta[i] = (tm*old)*itm - old;
				} else {
					if (!softSel) continue;
					if (clustDist[i] < 0) continue;
					if (clustDist[i] > falloff) continue;
					float af = AffectRegionFunction (clustDist[i], falloff, pinch, bubble);
					ws[i] += af;
					Point3 & old = GetMesh().v[i].p;
					delta[i] = ((tm*old)*itm - old) * af;
				}
			}
		}
		if (softSel) {
			for (i=0; i<nv; i++) {
				if (sel[i]) continue;
				if (ws[i] <= 1) continue;
				delta[i] /= ws[i];
			}
		}
	} else {
		Matrix3 tm  = partm * Inverse(tmAxis);
		Matrix3 itm = Inverse(tm);
		tm *= xfrm;
		// Normals are no longer needed; we use the MNMesh::GetVertexSpace method now.  -sca, 1/29/01
		//Point3 *vn = (numAxis == NUMAXIS_INDIVIDUAL) ? TempData()->VertexNormals()->Addr(0) : 0;
		float *vsw = softSel ? TempData()->VSWeight(useEdgeDist, edgeIts, !affectBack,
				falloff, pinch, bubble)->Addr(0) : NULL;
		for (i=0; i<nv; i++) {
			if (!sel[i]) {
				if (!vsw || !vsw[i]) continue;
			}
			Point3 & old = mm.v[i].p;
			if (numAxis == NUMAXIS_INDIVIDUAL) {
				// Changed for 4.patch by Steve A, 1/25/00,
				// Previous code was obselete, dates back to before proper
				// SubObjectTransforms were established.
				//MatrixFromNormal (vn[i], tm);
				//tm  = partm * Inverse(tm*nodeTm);
				Matrix3 axis2obj;
				mm.GetVertexSpace (i, axis2obj);
				axis2obj.SetTrans(old);
				Matrix3 obj2axis = Inverse (axis2obj);
				delta[i] = ((old*obj2axis)*xfrm)*axis2obj - old;
			} else {
				delta[i] = itm*(tm*old)-old;
			}
			if (!sel[i]) delta[i] *= vsw[i];
		}
	}

	nodes.DisposeTemporary ();
	DragMove (delta, this, t);
}

void EditPolyObject::TransformStart(TimeValue t) {
	if (!ip) return;
	ip->LockAxisTripods(TRUE);
	DragMoveInit();
}

void EditPolyObject::TransformHoldingFinish (TimeValue t) {
	if (!ip) return;
	DragMoveAccept (t);
}

void EditPolyObject::TransformFinish(TimeValue t) {
	if (!ip) return;
	ip->LockAxisTripods(FALSE);
}

void EditPolyObject::TransformCancel (TimeValue t) {
	DragMoveClear ();
	if (!ip) return;
	ip->LockAxisTripods(FALSE);
}

void EditPolyObject::CloneSelSubComponents(TimeValue t) {
	if (selLevel == EP_SL_OBJECT) return;
	if (selLevel == EP_SL_EDGE) return;
	if (selLevel == EP_SL_BORDER) return;
	if (!ip) return;

	theHold.Begin();
	theHold.Put (new ComponentFlagRestore (this, meshSelLevel[selLevel]));
	theHold.Put (new CreateOnlyRestore (this));

	switch (selLevel) {
	case EP_SL_VERTEX:
		mm.CloneVerts ();
		break;
	case EP_SL_FACE:
	case EP_SL_ELEMENT:
		mm.CloneFaces ();
		break;
	}
	theHold.Accept (IDS_CLONE);

	LocalDataChanged (PART_TOPO|PART_GEOM|PART_SELECT);
	RefreshScreen ();
}

void EditPolyObject::AcceptCloneSelSubComponents(TimeValue t) {
	switch (selLevel) {
	case EP_SL_OBJECT:
	case EP_SL_EDGE:
	case EP_SL_BORDER:
		return;
	}
	TSTR name;
	if (!GetCloneObjectName(ip, name)) return;
	if (!ip) return;

	ModContextList mcList;
	INodeTab nodes;
	ip->GetModContexts(mcList,nodes);
	EpfnDetachToObject (name, meshSelLevel[selLevel], MN_SEL, false, nodes[0], t);
	nodes.DisposeTemporary ();
	RefreshScreen ();
}

// Edit Geometry ops

int EditPolyObject::EpfnCreateVertex(Point3 pt, bool pt_local, bool select) {
	if (!pt_local) {
		if (!ip) return -1;
		// Put the point in object space:
		ModContextList mcList;
		INodeTab nodes;
		ip->GetModContexts(mcList,nodes);
		pt = pt * Inverse(nodes[0]->GetObjectTM(ip->GetTime()));
		nodes.DisposeTemporary();
	}

	if (theHold.Holding())
		theHold.Put (new CreateOnlyRestore (this));
	int ret = mm.NewVert (pt);
	if (select) {
		mm.v[ret].SetFlag (MN_SEL);
		LocalDataChanged (PART_GEOM|PART_SELECT);
	}
	else 
		LocalDataChanged (PART_GEOM);
	return ret;
}

int EditPolyObject::EpfnCreateEdge (int v1, int v2, bool select) {
	DbgAssert (v1>=0);
	DbgAssert (v2>=0);
	Tab<int> v1fac = mm.vfac[v1];
	int i, j, ff, v1pos, v2pos=-1;
	for (i=0; i<v1fac.Count(); i++) {
		MNFace & mf = mm.f[v1fac[i]];
		for (j=0; j<mf.deg; j++) {
			if (mf.vtx[j] == v2) v2pos = j;
			if (mf.vtx[j] == v1) v1pos = j;
		}
		if (v2pos<0) continue;
		ff = v1fac[i];
		break;
	}

	if (v2pos<0) return -1;

	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	int nf, ne;
	mm.SeparateFace (ff, v1pos, v2pos, nf, ne, TRUE, select);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	if (select) 
		LocalDataChanged (PART_TOPO|PART_SELECT);
	else
		LocalDataChanged (PART_TOPO);
	return ne;
}

int EditPolyObject::EpfnCreateFace(int *v, int deg, bool select) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}

	int ret = mm.CreateFace (deg, v);
	if (ret<0) {
		if (tchange) {
			theHold.Cancel ();
			delete tchange;
		}
		return -1;
	}
	mm.RetriangulateFace (ret);
	if (select) mm.f[ret].SetFlag (MN_SEL);
	mm.ClearFlag (MN_MESH_NO_BAD_VERTS);

	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	if (select) 
		LocalDataChanged (PART_TOPO|PART_SELECT);
	else
		LocalDataChanged (PART_TOPO);
	return ret;
}

bool EditPolyObject::EpfnCapHoles (int msl, DWORD targetFlags) {
	MNMeshBorder brdr;
	mm.GetBorder (brdr, msl, targetFlags);
	for (int i=0; i<brdr.Num(); i++) if (brdr.LoopTarg (i)) break;
	if (i>= brdr.Num()) return false;	// Nothing to do.

	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	mm.FillInBorders (&brdr);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_TOPO|PART_SELECT);
	return true;
}

void EditPolyObject::CollapseDeadVerts () {
	int i, max;

	// Check that we have at least one dead vertex.
	max = mm.numv;
	for (i=0; i<max; i++) if (mm.v[i].GetFlag (MN_DEAD)) break;
	if (i>=max) return;

	if (theHold.Holding ()) theHold.Put (new CollapseDeadVertsRestore (this));

	max = cont.Count ();
	if (mm.numv < max) max = mm.numv;
	for (i=max-1; i>=0; i--) {
		if (!mm.v[i].GetFlag (MN_DEAD)) continue;
		if (cont[i]) cont[i]->DeleteThis ();
		cont.Delete (i, 1);
	}
	mm.CollapseDeadVerts ();
	if (cont.Count() != mm.numv) SynchContArray (mm.numv);
	else masterCont->SetNumSubControllers(mm.numv, TRUE);
	LocalDataChanged (PART_TOPO|PART_GEOM);
}

void EditPolyObject::CollapseDeadEdges () {
	if (theHold.Holding ()) theHold.Put (new CollapseDeadEdgesRestore (this));
	mm.CollapseDeadEdges ();
	LocalDataChanged (PART_TOPO);
}

void EditPolyObject::CollapseDeadFaces () {
	if (theHold.Holding ()) theHold.Put (new CollapseDeadFacesRestore (this));
	mm.CollapseDeadFaces ();
	LocalDataChanged (PART_TOPO);
}

void EditPolyObject::CollapseDeadStructs () {
	CollapseDeadFaces ();
	CollapseDeadEdges ();
	CollapseDeadVerts ();
	RefreshScreen ();
}

bool EditPolyObject::DeleteVerts (DWORD flag) {
	if (mm.numv<1) return false;
	DbgAssert (mm.GetFlag (MN_MESH_FILLED_IN));
	DbgAssert (mm.vedg);
	DbgAssert (mm.vfac);

	// First, find out if any of these vertices are present and/or used.
	int i,j;
	bool present=FALSE, used=FALSE;
	for (i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (!mm.v[i].GetFlag (flag)) continue;
		present = TRUE;
		if (mm.vfac[i].Count()) {
			used = TRUE;
			break;
		}
	}
	if (!present) return false;	// Nothing to do.

	if (used) {
		// Delete the relevant faces and edges first.
		mm.ClearFFlags (MN_WHATEVER);
		for (; i<mm.numv; i++) {
			if (mm.v[i].GetFlag (MN_DEAD)) continue;
			if (!mm.v[i].GetFlag (flag)) continue;
			int ct;
			if ((ct=mm.vfac[i].Count()) == 0) continue;
			for (j=0; j<ct; j++) mm.f[mm.vfac[i][j]].SetFlag (MN_WHATEVER);
		}
		DeleteFaces (MN_WHATEVER, FALSE);
	}
	if (theHold.Holding ()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_VERTEX));
	for (i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (!mm.v[i].GetFlag (flag)) continue;
		mm.v[i].SetFlag (MN_DEAD);
	}

	LocalDataChanged (PART_GEOM|PART_TOPO|PART_SELECT);
	return true;
}

bool EditPolyObject::EpfnDeleteIsoVerts () {
	if (mm.numv<1) return false;
	DbgAssert (mm.GetFlag (MN_MESH_FILLED_IN));
	DbgAssert (mm.vfac);

	bool ret = false;
	if (theHold.Holding ()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_VERTEX));
	for (int i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (mm.vfac[i].Count()) continue;
		mm.v[i].SetFlag (MN_DEAD);
		ret = true;
	}
	LocalDataChanged (PART_GEOM|PART_SELECT);
	if (ret) CollapseDeadVerts ();
	return ret;
}

bool EditPolyObject::DeleteEdges (DWORD flag) {
	if (mm.numv<1) return false;
	DbgAssert (mm.GetFlag (MN_MESH_FILLED_IN));
	DbgAssert (mm.vedg);
	DbgAssert (mm.vfac);
	for (int i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (mm.e[i].f2<0) continue;
		if (mm.e[i].f1 == mm.e[i].f2) continue;
		if (mm.e[i].GetFlag (flag)) break;
	}
	if (i>= mm.nume) return false;
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}

	for (; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (mm.e[i].f2<0) continue;
		if (mm.e[i].f1 == mm.e[i].f2) continue;
		if (!mm.e[i].GetFlag (flag)) continue;

		mm.RemoveEdge (i);
	}

	if (theHold.Holding()) {
		tchange->After ();
		theHold.Put (tchange);
	}
	return true;
}

bool EditPolyObject::DeleteFaces (DWORD flag, bool delIsoVerts) {
	if (mm.numv<1) return false;
	DbgAssert (mm.GetFlag (MN_MESH_FILLED_IN));
	DbgAssert (mm.vedg);
	DbgAssert (mm.vfac);
	for (int i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (MN_DEAD)) continue;
		if (mm.f[i].GetFlag (flag)) break;
	}
	if (i>= mm.numf) return false;
	TopoChangeRestore *tchange=NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before();
	}
	DWORD invalid = PART_TOPO|PART_SELECT;
	flag |= MN_DEAD;	// so we also remove connections to dead faces, if any creep in...
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if ((mm.e[i].f2>-1) && mm.f[mm.e[i].f2].GetFlag(flag))
			mm.e[i].f2 = -1;
		if (!mm.f[mm.e[i].f1].GetFlag (flag)) continue;
		mm.e[i].f1 = -1;
		if (mm.e[i].f2 > -1) mm.e[i].Invert ();
		else mm.e[i].SetFlag (MN_DEAD);
	}
	for (i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		Tab<int> & vf = mm.vfac[i];
		Tab<int> & ve = mm.vedg[i];
		int j;
		if (j=vf.Count ()) {
			for (j--; j>=0; j--) {
				if (mm.f[vf[j]].GetFlag (flag)) vf.Delete (j,1);
			}
			if (!vf.Count() && delIsoVerts) {
				mm.v[i].SetFlag (MN_DEAD);
				invalid |= PART_GEOM;
			}
		}
		for (j=ve.Count()-1; j>=0; j--) {
			if (mm.e[ve[j]].GetFlag (MN_DEAD)) ve.Delete (j, 1);
		}
	}
	for (i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (flag)) mm.f[i].SetFlag (MN_DEAD);
	}
	// This operation can create 2-boundary vertices:
	mm.ClearFlag (MN_MESH_NO_BAD_VERTS);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}

	LocalDataChanged (invalid);
	return true;
}

void EditPolyObject::EpfnAttach (INode *node, bool & canUndo, INode *myNode, TimeValue t) {
	// First get the object
	BOOL del = FALSE;
	PolyObject *obj = NULL;
	ObjectState os = node->GetObjectRef()->Eval(t);
	if (os.obj->IsSubClassOf(polyObjectClassID)) obj = (PolyObject *) os.obj;
	else {
		if (!os.obj->CanConvertToType(polyObjectClassID)) return;
		obj = (PolyObject*)os.obj->ConvertToType (t, polyObjectClassID);
		if (obj!=os.obj) del = TRUE;
	}

	theHold.Begin();

	// Combine the materials of the two nodes.
	int mat2Offset=0;
	Mtl *m1 = myNode->GetMtl();
	Mtl *m2 = node->GetMtl();
	bool condenseMe = FALSE;
	if (m1 && m2 && (m1 != m2)) {
		if (attachMat==ATTACHMAT_IDTOMAT) {
			FitPolyMeshIDsToMaterial (mm, m1);
			FitPolyMeshIDsToMaterial (obj->mm, m2);
			if (condenseMat) condenseMe = TRUE;
		}

		// the theHold calls here were a vain attempt to make this all undoable.
		// This should be revisited in the future so we don't have to use the SYSSET_CLEAR_UNDO.
		theHold.Suspend ();
		if (attachMat==ATTACHMAT_MATTOID) {
			m1 = FitMaterialToPolyMeshIDs (GetMesh(), m1);
			m2 = FitMaterialToPolyMeshIDs (obj->GetMesh(), m2);
		}
		Mtl *multi = CombineMaterials(m1, m2, mat2Offset);
		if (attachMat == ATTACHMAT_NEITHER) mat2Offset = 0;
		theHold.Resume ();
		// We can't be in face subobject mode, else we screw up the materials:
		DWORD oldSL = selLevel;
		selLevel = EP_SL_OBJECT;
		myNode->SetMtl(multi);
		selLevel = oldSL;
		m1 = multi;
		canUndo = TRUE;	//DS 10/16/00: Undo should work now.
	}
	if (!m1 && m2) {
		// This material operation seems safe.
		// We can't be in face subobject mode, else we screw up the materials:
		DWORD oldSL = selLevel;
		if (oldSL>EP_SL_EDGE) selLevel = EP_SL_OBJECT;
		myNode->SetMtl (m2);
		if (oldSL>EP_SL_EDGE) selLevel = oldSL;
		m1 = m2;
	}

	// Construct a transformation that takes a vertex out of the space of
	// the source node and puts it into the space of the destination node.
	Matrix3 relativeTransform = node->GetObjectTM(t) *
		Inverse(myNode->GetObjectTM(t));

	theHold.Put (new CreateOnlyRestore (this));
	int ovnum = mm.numv;
	int ofnum = mm.numf;
	mm += obj->mm;
	// Note that following code only modifies created elements,
	// so it's still ok to use the CreateOnlyRestore.
	for (int i=ovnum; i<mm.numv; i++) mm.v[i].p = relativeTransform * mm.v[i].p;
	if (relativeTransform.Parity()) {
		mm.ClearFFlags (MN_WHATEVER);
		for (i=ofnum; i<mm.numf; i++) {
			if (!mm.f[i].GetFlag (MN_DEAD)) mm.f[i].SetFlag (MN_WHATEVER);
		}
		mm.FlipElementNormals (MN_WHATEVER);
	}
	if (mat2Offset) {
		for (i=ofnum; i<mm.numf; i++) mm.f[i].material += mat2Offset;
	}
	GetCOREInterface()->DeleteNode (node);
	theHold.Accept (GetString (IDS_ATTACH_OBJECT));

	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);

	if (m1 && condenseMe) {
		// Following clears undo stack.
		m1 = CondenseMatAssignments (GetMesh(), m1);
	}
	if (del) delete obj;
}

void EditPolyObject::EpfnMultiAttach (INodeTab &nodeTab, INode *myNode, TimeValue t) {
	bool canUndo = TRUE;
	if (nodeTab.Count() > 1) theHold.SuperBegin ();
	for (int i=0; i<nodeTab.Count(); i++) EpfnAttach (nodeTab[i], canUndo, myNode, t);
	if (nodeTab.Count() > 1) theHold.SuperAccept (GetString (IDS_ATTACH_LIST));
	if (!canUndo) GetSystemSetting (SYSSET_CLEAR_UNDO);
	RefreshScreen ();
}

bool EditPolyObject::EpfnDetachToElement (int msl, DWORD flags, bool keepOriginal) {
	// Verify that we have some components that may be detached.
	int i;
	if (msl == MNM_SL_FACE) {
		for (i=0; i<mm.numf; i++) {
			if (mm.f[i].GetFlag (MN_DEAD)) continue;
			if (mm.f[i].GetFlag (flags)) break;
		}
		if (i==mm.numf) return false;
	} else {
		// NOTE: won't work for MNM_SL_OBJECT!
		mm.ClearFFlags (MN_WHATEVER);
		if (mm.PropegateComponentFlags (MNM_SL_FACE, MN_WHATEVER, msl, flags) == 0)
			return false;
		flags = MN_WHATEVER;
	}

	bool localHold = !theHold.Holding();
	if (localHold) theHold.Begin ();
	TopoChangeRestore *tchange = new TopoChangeRestore (this);
	tchange->Before ();
	if (keepOriginal) mm.CloneFaces (flags, true);
	else mm.DetachFaces (flags);
	tchange->After ();
	theHold.Put (tchange);
	if (localHold) theHold.Accept (GetString (IDS_DETACH));
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return true;
}

// NOTE: Still depends on ip for node creation...
// now getting via GetCOREInterface()
bool EditPolyObject::EpfnDetachToObject (TSTR name, int msl, DWORD flags, bool keepOriginal, INode *myNode, TimeValue t) {
	theHold.Begin ();
	// First, make the detachables into their own element:
	if (!EpfnDetachToElement (msl, flags, keepOriginal)) {
		theHold.Cancel ();
		return FALSE;
	}

	// we want to deal with the faces detached by DetachToElement.
	// If we're not in that SO level, such faces are indicated by MN_WHATEVER.
	if (msl != MNM_SL_FACE) flags = MN_WHATEVER;

	// Animation confuses things.
	SuspendAnimate();
	AnimateOff();

	PolyObject *newObj = CreateEditablePolyObject();
	// Copy our parameters (such as soft selection, subdivision) to new object.
	newObj->ReplaceReference(EPOLY_PBLOCK,pblock->Clone());

	// Put detached portion into newObj
	TopoChangeRestore *tchange = new TopoChangeRestore (this);
	tchange->Before ();
	mm.DetachElementToObject (newObj->mm, flags, true);
	// SCA Fix 1/29/01: the newObj mesh was always being created with the NONTRI flag empty,
	// leading to problems in rendering and a couple other areas.  Instead we copy the
	// NONTRI flag from our current mesh.  Note that this may result in the NONTRI flag
	// being set on a fully triangulated detached region, but that's acceptable; leaving out
	// the NONTRI flag when it is non-triangulated is much more serious.
	newObj->mm.CopyFlags (mm, MN_MESH_NONTRI);
	tchange->After ();
	theHold.Put (tchange);

	// Add the object to the scene. Give it the given name
	// and set its transform to ours.
	// Also, give it our material.
	INode *node = GetCOREInterface()->CreateObjectNode(newObj);
	Matrix3 ntm = myNode->GetNodeTM(t);
	if (GetCOREInterface()->GetINodeByName(name) != node) {	// Just in case name = "Object01" or some such default.
		TSTR uname = name;
		if (GetCOREInterface()->GetINodeByName (uname)) GetCOREInterface()->MakeNameUnique(uname);
		node->SetName(uname);
	}
	node->SetNodeTM(t,ntm);
	node->CopyProperties (myNode);
	node->FlagForeground(t,FALSE);
	node->SetMtl(myNode->GetMtl());
	node->SetObjOffsetPos (myNode->GetObjOffsetPos());
	node->SetObjOffsetRot (myNode->GetObjOffsetRot());
	node->SetObjOffsetScale (myNode->GetObjOffsetScale());

	theHold.Accept (GetString (IDS_DETACH));
	ResumeAnimate();
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return true;
}

bool EditPolyObject::EpfnSplitEdges (DWORD flag) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	if (!mm.SplitFlaggedEdges (flag)) {
		if (tchange) delete tchange;
		return false;
	}
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_TOPO|PART_GEOM|PART_SELECT);
	return true;
}

bool EditPolyObject::EpfnBreakVerts (DWORD flag) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	if (!mm.SplitFlaggedVertices (flag)) {
		if (tchange) {
			delete tchange;
		}
		return false;
	}
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_TOPO|PART_GEOM|PART_SELECT);
	return true;
}

int EditPolyObject::EpfnDivideFace (int face, Tab<float> &bary, bool select) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	int nv = mm.DivideFace (face, bary);
	if (select) mm.v[nv].SetFlag (MN_SEL);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	if (select)
		LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
	else
		LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE-PART_SELECT);
	return nv;
}

int EditPolyObject::EpfnDivideEdge (int edge, float prop, bool select) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	int nv = mm.SplitEdge (edge, prop);
	if (select) mm.v[nv].SetFlag (MN_SEL);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	if (select)
		LocalDataChanged (PART_TOPO|PART_GEOM|PART_SELECT);
	else
		LocalDataChanged (PART_TOPO|PART_GEOM);
	return nv;
}

void EditPolyObject::EpfnExtrudeFaces (float amount, DWORD flag, TimeValue t) {
	EpfnBeginExtrude (MNM_SL_FACE, flag, t);
	EpfnDragExtrude (amount, t);
	EpfnEndExtrude (true, t);
}

void EditPolyObject::EpfnBevelFaces (float height, float outline, DWORD flag, TimeValue t) {
	EpfnBeginBevel (MNM_SL_FACE, flag, true, t);
	EpfnDragBevel (outline, height, t);
	EpfnEndBevel (true, t);
}

void EditPolyObject::EpfnChamferVertices (float amount, TimeValue t) {
	EpfnBeginChamfer (MNM_SL_VERTEX, t);
	EpfnDragChamfer (amount, t);
	EpfnEndChamfer (true, t);
}

void EditPolyObject::EpfnChamferEdges (float amount, TimeValue t) {
	EpfnBeginChamfer (MNM_SL_EDGE, t);
	EpfnDragChamfer (amount, t);
	EpfnEndChamfer (true, t);
}

bool EditPolyObject::DoExtrusion (int msl, DWORD flag) {
	theHold.Begin();
	TopoChangeRestore *tchange = new TopoChangeRestore (this);
	tchange->Before ();
	int extType;
	pblock->GetValue (ep_extrusion_type, TimeValue(0), extType, FOREVER);
	bool abort=TRUE;

	if (flag != MN_SEL) {
		// will need to rebuild face clusters.
		TempData()->Invalidate (PART_TOPO);
	}
	switch (msl) {
	case MNM_SL_VERTEX:
		abort = TRUE;	// Some day, we'll have a vertex extrude operation.
		break;
	case MNM_SL_EDGE:
		abort = TRUE;	// Some day, we'll have an edge extrude operation.
		break;
	case MNM_SL_FACE:
		switch (extType) {
		case 0:	// Group normals
			abort = !mm.ExtrudeFaceClusters (*(TempData()->FaceClusters(flag)));
			if (!abort) {
				TempData()->Invalidate (PART_TOPO);	// Forces reevaluation of face clusters.
				mm.GetExtrudeDirection (TempData()->ChamferData(),
					TempData()->FaceClusters(flag),
					TempData()->ClusterNormals (MNM_SL_FACE, flag)->Addr(0));
			}
			break;

		case 1:	// Local normals
			abort = !mm.ExtrudeFaceClusters (*(TempData()->FaceClusters(flag)));
			if (!abort) {
				TempData()->Invalidate (PART_TOPO);	// Forces reevaluation of face clusters.
				// NOTE: This method should accept a face flag, but doesn't, so it finds
				// the extrusion direction for faces with the MN_SEL flag!
				mm.GetExtrudeDirection (TempData()->ChamferData());
			}
			break;

		case 2:	// poly-by-poly extrusion.
			abort = !mm.ExtrudeFaces (flag);
			if (!abort) {
				TempData()->Invalidate (PART_TOPO);
				// NOTE: This method should accept a face flag, but doesn't, so it finds
				// the extrusion direction for faces with the MN_SEL flag!
				mm.GetExtrudeDirection (TempData()->ChamferData());
			}
			break;
		}
		break;
	}
	if (abort) {
		delete tchange;
        theHold.Cancel();
		return false;
	}
	tchange->After ();
	theHold.Put (tchange);
	theHold.Accept (GetString (IDS_EXTRUDE));
	LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
	RefreshScreen ();
	return true;
}

void EditPolyObject::EpfnBeginExtrude (int msl, DWORD flag, TimeValue t) {	
	if (inExtrude) return;
	inExtrude = TRUE;
	theHold.SuperBegin();
	if (!DoExtrusion(msl, flag)) {
		theHold.SuperCancel ();
		inExtrude = FALSE;
		return;
	}
	BitArray vset;
	if ((msl != MNM_SL_VERTEX) || !(flag & MN_WHATEVER)) {
		mm.ClearVFlags (MN_WHATEVER);
		// NOTE: following won't work for msl==MNM_SL_OBJECT
		mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_WHATEVER, msl, flag);
	}
	mm.getVerticesByFlag (vset, (msl == MNM_SL_VERTEX) ? flag : MN_WHATEVER);
	PlugControllersSel (t, vset);
	DragMoveInit();
}

void EditPolyObject::EpfnEndExtrude (bool accept, TimeValue t) {
	if (inExtrude) {
		inExtrude = FALSE;
		TempData()->freeChamferData ();

		theHold.Begin ();
		DragMoveAccept (t);
		theHold.Accept(GetString(IDS_MOVE));
		if (accept) theHold.SuperAccept(GetString(IDS_EXTRUDE));
		else {
			theHold.SuperCancel();
			InvalidateTempData ();
		}
	}

	HWND hGeom = GetDlgHandle (ep_geom);
	if (hGeom) {
		ISpinnerControl *spin = GetISpinner(GetDlgItem(hGeom,IDC_EXTRUDESPINNER));
		if (spin) {
			spin->SetValue(0,FALSE);
			ReleaseISpinner(spin);
		}
	}
}

void EditPolyObject::EpfnDragExtrude (float amount, TimeValue t) {
	if (!inExtrude) return;
	DragMoveRestore ();

	MNChamferData *chamData = TempData()->ChamferData();
	if (chamData == NULL) return;
	Tab<Point3> delta;
	delta.SetCount (mm.numv);
	chamData->GetDelta (amount, delta);
	DragMove (delta, this, t);
}

static bool ExtDone=FALSE;

void EditPolyObject::EpfnBeginBevel (int msl, DWORD flag, bool doExtrude, TimeValue t) {
	if (inBevel) return;
	inBevel = TRUE;
	theHold.SuperBegin();
	if (doExtrude && !DoExtrusion (MNM_SL_FACE, flag)) {
		theHold.SuperCancel ();
		inBevel = FALSE;
		return;
	}
	ExtDone = doExtrude ? true : false;
	BitArray vset;
	if ((msl != MNM_SL_VERTEX) || !(flag & MN_WHATEVER)) {
		mm.ClearVFlags (MN_WHATEVER);
		// NOTE: following won't work for msl==MNM_SL_OBJECT
		mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_WHATEVER, msl, flag);
	}
	mm.getVerticesByFlag (vset, (msl == MNM_SL_VERTEX) ? flag : MN_WHATEVER);
	PlugControllersSel(t,vset);
	DragMoveInit ();
}

void EditPolyObject::EpfnEndBevel (bool accept, TimeValue t) {
	if (!inBevel) return;
	inBevel = FALSE;
	TempData()->freeBevelInfo();
	TempData()->freeChamferData ();

	theHold.Begin ();
	DragMoveAccept (t);
	theHold.Accept(GetString(IDS_BEVEL));
	if (accept) theHold.SuperAccept(GetString(ExtDone ? IDS_BEVEL : IDS_OUTLINE));
	else {
		theHold.SuperCancel();
		InvalidateTempData ();
	}
	ExtDone = false;

	HWND hGeom = GetDlgHandle (ep_geom);
	if (hGeom) {
		ISpinnerControl *spin = GetISpinner(GetDlgItem(hGeom,IDC_OUTLINESPINNER));
		if (spin) {
			spin->SetValue(0,FALSE);
			ReleaseISpinner(spin);
		}
		spin = GetISpinner(GetDlgItem(hGeom,IDC_EXTRUDESPINNER));
		if (spin) {
			spin->SetValue(0,FALSE);
			ReleaseISpinner(spin);
		}
	}
}

void EditPolyObject::EpfnDragBevel (float amount, float height, TimeValue t) {
	if (!inBevel) return;
	DragMoveRestore ();

	MNChamferData *chamData = TempData()->ChamferData();
	if (chamData == NULL) return;

	int i;
	Tab<Point3> delta;
	delta.SetCount (mm.numv);
	if (height) chamData->GetDelta (height, delta);
	else {
		for (i=0; i<mm.numv; i++) delta[i] = Point3(0,0,0);
	}

	if (amount) {
		int extType;
		pblock->GetValue (ep_extrusion_type, TimeValue(0), extType, FOREVER);
		Tab<Point3> *outDir;
		if (extType == 0) outDir = TempData()->OutlineDir (MESH_EXTRUDE_CLUSTER);
		else outDir = TempData()->OutlineDir (MESH_EXTRUDE_LOCAL);
		Point3 *od = outDir->Addr(0);
		for (i=0; i<mm.numv; i++) delta[i] += od[i]*amount;
	}

	DragMove (delta, this, t);

	HWND hGeom = GetDlgHandle (ep_geom);
	if (hGeom) {
		ISpinnerControl *spin = GetISpinner(GetDlgItem(hGeom,IDC_OUTLINESPINNER));
		if (spin) {
			spin->SetValue(amount,FALSE);
			ReleaseISpinner(spin);
		}
		spin = GetISpinner(GetDlgItem(hGeom,IDC_EXTRUDESPINNER));
		if (spin) {
			spin->SetValue(height,FALSE);
			ReleaseISpinner(spin);
		}
	}
}

void EditPolyObject::DoChamfer (int msl) { // This assumes that we want flag MN_SEL - should be generalized.
	theHold.Begin();
	TopoChangeRestore *tchange = new TopoChangeRestore (this);
	tchange->Before ();
	MNChamferData *mcd = TempData()->ChamferData();
	bool ret = FALSE;
	switch (msl) {
	case MNM_SL_VERTEX:
		ret = mm.ChamferVertices (MN_SEL, mcd);
		break;
	case MNM_SL_EDGE:
		ret = mm.ChamferEdges (MN_SEL, mcd);
		break;
	}
	if (!ret) { 
		delete tchange;
        theHold.Cancel();
		return;
	}
	tchange->After ();
	theHold.Put (tchange);
	theHold.Accept (GetString (IDS_CHAMFER));
	LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
	RefreshScreen ();
}

void EditPolyObject::EpfnBeginChamfer (int msl, TimeValue t) {
	if (inChamfer) return;
	inChamfer = TRUE;
	theHold.SuperBegin();
	DoChamfer (msl);
	if (!TempData()->ChamferData()) return;
	if (!TempData()->ChamferData()->vmax.Count()) return;
	BitArray sel;
	sel.SetSize (mm.numv);
	sel.ClearAll ();
	float *vmp = TempData()->ChamferData()->vmax.Addr(0);
	for (int i=0; i<mm.numv; i++) if (vmp[i]) sel.Set(i);
	PlugControllersSel (t,sel);
	DragMoveInit();
}

void EditPolyObject::EpfnEndChamfer (bool accept, TimeValue t) {
	if (!inChamfer) return;
	inChamfer = FALSE;
	TempData()->freeChamferData();

	theHold.Begin ();
	DragMoveAccept (t);
	theHold.Accept(GetString(IDS_MOVE));
	if (accept) theHold.SuperAccept(GetString(IDS_CHAMFER));
	else {
		theHold.SuperCancel();
		InvalidateTempData ();
	}

	HWND hGeom = GetDlgHandle (ep_geom);
	if (hGeom) {
		ISpinnerControl *spin = GetISpinner(GetDlgItem(hGeom,IDC_OUTLINESPINNER));
		if (spin) {
			spin->SetValue(0,FALSE);
			ReleaseISpinner(spin);
		}
	}
}

void EditPolyObject::EpfnDragChamfer (float amount, TimeValue t) {
	if (!inChamfer) return;
	DragMoveRestore ();
	if (amount<=0) return;
	if (!TempData()->ChamferData()) return;

	// Do DragMap's first, because they don't call RefreshScreen.
	Tab<UVVert> mapDelta;
	for (int mp=-NUM_HIDDENMAPS; mp<mm.numm; mp++) {
		if (mm.M(mp)->GetFlag (MN_DEAD)) continue;
		TempData()->ChamferData()->GetMapDelta (mm, mp, amount, mapDelta);
		DragMap (mp, mapDelta, this, t);
	}

	Tab<Point3> delta;
	TempData()->ChamferData()->GetDelta (amount, delta);
	DragMove (delta, this, t);
}

bool EditPolyObject::EpfnCreateShape (TSTR name, bool createCurveSmooth, INode *myNode, DWORD edgeFlag) {
	bool ret = CreateCurveFromMeshEdges (GetMesh(), myNode, GetCOREInterface(), name, createCurveSmooth, edgeFlag);
	return ret;
}

bool EditPolyObject::EpfnMakePlanar (int msl, DWORD flag, TimeValue t) {
	//if (msl == MNM_SL_OBJECT) return false;
	Tab<Point3> delta;
	delta.SetCount (mm.numv);

	if (!mm.MakeFlaggedPlanar (msl, flag, delta.Addr(0))) {
		// Nothing was moved.
		return false;
	}

	ApplyDelta (delta, this, t);
	return true;
}

bool EditPolyObject::EpfnAlignToGrid (int msl, DWORD flag) {
	// We'll need the viewport or construction plane transform:
	if (!ip) return false;
	Matrix3 atm, otm, res;
	ViewExp *vpt = ip->GetActiveViewport();
	float zoff;
	vpt->GetConstructionTM(atm);
	ip->ReleaseViewport (vpt);

	// We'll also need our own transform:
	ModContextList mcList;
	INodeTab nodes;
	ip->GetModContexts(mcList,nodes);
	otm = nodes[0]->GetObjectTM(ip->GetTime());
	nodes.DisposeTemporary();
	res = atm*Inverse (otm);	// screenspace-to-objectspace.

	// For ZNorm, we want the object-space unit vector pointing into the screen:
	Point3 ZNorm (0,0,-1);
	ZNorm = Normalize (VectorTransform (res, ZNorm));

	// Find the z-depth of the construction plane, in object space:
	zoff = DotProd (ZNorm, res.GetTrans());

	EpfnMoveToPlane (ZNorm, zoff, msl, flag);
	return true;
}

bool EditPolyObject::EpfnMoveToPlane (Point3 ZNorm, float zoff, int msl, DWORD flag, TimeValue t) {
	// Target appropriate vertices with flags:
	if ((msl != MNM_SL_VERTEX) || (flag&MN_WHATEVER == 0))
		mm.ClearVFlags (MN_WHATEVER);
	if (msl == MNM_SL_OBJECT) {
		for (int i=0; i<mm.numv; i++) mm.v[i].SetFlag (MN_WHATEVER, !mm.v[i].GetFlag (MN_DEAD));
	} else mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_WHATEVER, msl, flag);

	Tab<Point3> delta;
	delta.SetCount (mm.numv);
	if (!mm.MoveVertsToPlane (ZNorm, zoff, MN_WHATEVER, delta.Addr(0))) return false;	// nothing happened.
	ApplyDelta (delta, this, t);
	return true;
}

bool EditPolyObject::EpfnAlignToView (int msl, DWORD flag) {
	// We'll need the viewport or construction plane transform:
	if (!ip) return false;
	Matrix3 atm, otm, res;
	ViewExp *vpt = ip->GetActiveViewport();
	float zoff;
	vpt->GetAffineTM(atm);
	atm = Inverse(atm);
	ip->ReleaseViewport (vpt);

	// We'll also need our own transform:
	ModContextList mcList;
	INodeTab nodes;
	ip->GetModContexts(mcList,nodes);
	otm = nodes[0]->GetObjectTM(ip->GetTime());
	nodes.DisposeTemporary();
	res = atm*Inverse (otm);	// screenspace-to-objectspace.

	// Target appropriate vertices with flags:
	if ((msl != MNM_SL_VERTEX) || (flag&MN_WHATEVER == 0))
		mm.ClearVFlags (MN_WHATEVER);
	if (msl == MNM_SL_OBJECT) {
		for (int i=0; i<mm.numv; i++) mm.v[i].SetFlag (MN_WHATEVER, !mm.v[i].GetFlag (MN_DEAD));
	} else mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_WHATEVER, msl, flag);

	BitArray sel = GetMesh().VertexTempSel();
	// For ZNorm, we want the object-space unit vector pointing into the screen:
	Point3 ZNorm (0,0,-1);
	ZNorm = Normalize (VectorTransform (res, ZNorm));

	// Find the average z-depth for the current selection.
	zoff = 0.0f;
	int ct=0;
	for (int i=0; i<mm.numv; i++) {
		if (!mm.v[i].GetFlag (MN_WHATEVER)) continue;
		zoff += DotProd (ZNorm, mm.v[i].p);
		ct++;
	}
	zoff /= float(ct);

	Tab<Point3> delta;
	delta.SetCount (mm.numv);
	if (!mm.MoveVertsToPlane (ZNorm, zoff, MN_WHATEVER, delta.Addr(0))) return false;	// nothing happened.
	ApplyDelta (delta, this, ip->GetTime());
	return true;
}

bool EditPolyObject::EpfnCollapse (int msl, DWORD flag) {
	if (msl == MNM_SL_OBJECT) return false;

	TopoChangeRestore *tchange=NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	Tab<bool> flagBackup;
	int i;
// LAM - defect 292184 - was always operating on MN_SEL rather than flag
// this is messy now, check with Steve to see why setting edge MN_SEL rather than MN_TARG
//	if (msl != MNM_SL_EDGE) {	
		// In order to collapse other SO levels, we turn them into edge selections.
		// In order to make use of edge clusters (to make this easy), we need to backup the current edge
		// selection, replace it with the new selection, and replace the original edge selection later.
		flagBackup.SetCount (mm.nume);
		for (i=0; i<mm.nume; i++) flagBackup[i] = mm.e[i].GetFlag (MN_SEL) ? true : false;

		if (msl != MNM_SL_EDGE || flag != MN_SEL)
			mm.ClearEFlags (MN_SEL);
		// at vertex level, we require that edges have both ends selected to be targeted.
		// at face level, any edge with at least one selected face is targeted.
		if (msl == MNM_SL_VERTEX) 
			mm.PropegateComponentFlags (MNM_SL_EDGE, MN_SEL, msl, flag, true);
		else if (msl != MNM_SL_EDGE || flag != MN_SEL)
			mm.PropegateComponentFlags (MNM_SL_EDGE, MN_SEL, msl, flag, false);

		TempData()->Invalidate (PART_SELECT);
//	}

	// Collect average locations ourselves, since WeldEdge averages in piecemeal fashion.
	Tab<Point3> pointSum;
	Tab<int> pointNum;
	pointSum.SetCount (mm.numv);
	pointNum.SetCount (mm.numv);
	for (i=0; i<mm.numv; i++) {
		pointSum[i] = mm.v[i].p;
		pointNum[i] = 1;
	}

	// Perform topological welding.
	bool ret=false;
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!mm.e[i].GetFlag (MN_SEL)) continue;
		int v1 = mm.e[i].v1;
		int v2 = mm.e[i].v2;
		if (mm.WeldEdge (i)) {
			pointSum[v1] += pointSum[v2];
			pointNum[v1] += pointNum[v2];
			ret = true;
		}
	}

	// Then set all the welded vertices to the correct locations - edge by edge above produces wrong geometric result.
	if (ret) {
		for (i=0; i<mm.numv; i++) {
			if (mm.v[i].GetFlag (MN_DEAD)) continue;
			if (pointNum[i] < 2) continue;
			mm.v[i].p = pointSum[i] / float(pointNum[i]);
		}
	}

// LAM - defect 292184 - was always operating on MN_SEL rather than flag
	// Cleanup from non-edge SO level
//	if (msl != MNM_SL_EDGE) {
		mm.ClearEFlags (MN_SEL);
		for (i=0; i<mm.nume; i++) mm.e[i].SetFlag (MN_SEL, flagBackup[i]);
		TempData()->Invalidate (PART_SELECT);
//	}

	// If no collapsing occurred, but we're in vertex level, try a weld.
	if (!ret && (msl == MNM_SL_VERTEX)) {
		ret = mm.WeldBorderVerts (999999.0f, flag);
	}

	if (ret) {
		if (tchange) {
			tchange->After ();
			theHold.Put (tchange);
		}
		CollapseDeadStructs ();
		LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
		return true;
	} else {
		if (tchange) delete tchange;
		return false;
	}
}

bool EditPolyObject::EpfnMeshSmooth (int msl, DWORD flag) {
	float smoothness;
	int sepSmooth, sepMat;
	pblock->GetValue (ep_ms_smoothness, TimeValue(0), smoothness, FOREVER);
	pblock->GetValue (ep_ms_sep_smooth, TimeValue(0), sepSmooth, FOREVER);
	pblock->GetValue (ep_ms_sep_mat, TimeValue(0), sepMat, FOREVER);
	switch (msl) {
		int i;
	case MNM_SL_OBJECT:
		mm.ClearVFlags (MN_TARG);
		for (i=0; i<mm.numv; i++) if (!mm.v[i].GetFlag (MN_DEAD)) mm.v[i].SetFlag (MN_TARG);
		break;
	case MNM_SL_VERTEX:
		if ((flag & MN_TARG) == 0) mm.ClearVFlags (MN_TARG);
		for (i=0; i<mm.numv; i++) if (mm.v[i].GetFlag (flag)) break;
		if (i==mm.numv) return false;
		if (flag != MN_TARG) {
			for (; i<mm.numv; i++) if (mm.v[i].GetFlag (flag)) mm.v[i].SetFlag (MN_TARG);
		}
		break;
	default:
		mm.ClearVFlags (MN_TARG);
		int num = mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_TARG, msl, flag);
		if (!num) return false;
	}
	if (smoothness < 1.0f) {
		mm.DetargetVertsBySharpness (1.0f - smoothness);
	}
	mm.ClearEFlags (MN_EDGE_NOCROSS|MN_EDGE_MAP_SEAM);
	mm.SetMapSeamFlags ();
	mm.FenceOneSidedEdges ();
	if (sepMat) mm.FenceMaterials ();
	if (sepSmooth) mm.FenceSmGroups ();
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore(this);
		tchange->Before ();
	}
	mm.CubicNURMS (NULL, NULL, MN_SUBDIV_NEWMAP);
	mm.ClearEFlags (MN_EDGE_NOCROSS|MN_EDGE_MAP_SEAM);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	CollapseDeadFaces ();
	LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
	return true;
}

bool EditPolyObject::EpfnTessellate (int msl, DWORD flag) {
	float tens;
	int type;
	pblock->GetValue (ep_tess_tension, TimeValue(0), tens, FOREVER);
	pblock->GetValue (ep_tess_type, TimeValue(0), type, FOREVER);
	switch (msl) {
		int i, num;
	case MNM_SL_OBJECT:
		mm.ClearFFlags (MN_TARG);
		for (i=0; i<mm.numf; i++) if (!mm.f[i].GetFlag (MN_DEAD)) mm.f[i].SetFlag (MN_TARG);
		break;
	case MNM_SL_FACE:
		if ((flag & MN_TARG) == 0) mm.ClearFFlags (MN_TARG);
		for (i=0; i<mm.numf; i++) if (mm.f[i].GetFlag (flag)) break;
		if (i==mm.numf) return false;
		if (flag != MN_TARG) {
			for (; i<mm.numf; i++) if (mm.f[i].GetFlag (flag)) mm.f[i].SetFlag (MN_TARG);
		}
		break;
	default:
		mm.ClearFFlags (MN_TARG);
		num = mm.PropegateComponentFlags (MNM_SL_FACE, MN_TARG, msl, flag);
		if (!num) return false;
	}
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore(this);
		tchange->Before ();
	}
	if (type) mm.TessellateByCenters ();
	else mm.TessellateByEdges (tens/400.0f);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	CollapseDeadFaces ();
	LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
	return true;
}

void EditPolyObject::EpResetSlicePlane () {
	if (theHold.Holding ()) theHold.Put (new TransformPlaneRestore (this));
	Box3 box = mm.getBoundingBox ();
	sliceCenter = (box.pmax + box.pmin)*.5f;
	sliceRot.Identity();
	box.pmax -= box.pmin;
	sliceSize = (box.pmax.x > box.pmax.y) ? box.pmax.x : box.pmax.y;
	if (box.pmax.z > sliceSize) sliceSize = box.pmax.z;
	sliceSize *= .52f;
	if (sliceSize < 1) sliceSize = 1.0f;
	NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	RefreshScreen ();
}

void EditPolyObject::EpGetSlicePlane (Point3 & planeNormal, Point3 & planeCenter, float *planeSize) {
	if (theHold.Holding ()) theHold.Put (new TransformPlaneRestore (this));
	Matrix3 rotMatrix;
	sliceRot.MakeMatrix (rotMatrix);
	planeNormal = Point3(0.0f,0.0f,1.0f) * rotMatrix;
	planeCenter = sliceCenter;
	if (planeSize) *planeSize = sliceSize;
}

void EditPolyObject::EpSetSlicePlane (Point3 & planeNormal, Point3 & planeCenter, float planeSize) {
	sliceCenter = planeCenter;
	sliceSize = planeSize;
	float len = Length (planeNormal);
	Point3 pnorm = planeNormal/len;
	Point3 axis = Point3(0,0,1)^pnorm;
	float angle = acosf (pnorm.z);	// pnorm.z is the dot product with (0,0,1).
	sliceRot.Set (AngAxis (axis, angle));
}

bool EditPolyObject::EpfnSlice (Point3 planeNormal, Point3 planeCenter,
								bool flaggedFacesOnly, DWORD faceFlags) {
	if (flaggedFacesOnly) {
		for (int i=0; i<mm.numf; i++) {
			if (mm.f[i].GetFlag (MN_DEAD)) continue;
			if (mm.f[i].GetFlag (faceFlags)) break;
		}
		if (i==mm.numf) return false;
	}
	float offset = DotProd (planeNormal, planeCenter);
	BOOL sliceSplit;
	pblock->GetValue (ep_split, TimeValue(0), sliceSplit, FOREVER);
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	if (!mm.Slice (planeNormal, offset, MNEPS,
		sliceSplit?true:false, FALSE, flaggedFacesOnly, faceFlags)) {
		if (tchange) delete tchange;
		return false;
	}
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return true;
}

int EditPolyObject::EpfnCutVertex (int startv, Point3 destination, Point3 projDir) {
	int splitme;
	pblock->GetValue (ep_split, TimeValue(0), splitme, FOREVER);
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	int ret = mm.Cut (startv, destination, projDir, splitme?true:false);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return ret;
}

int EditPolyObject::EpfnCutEdge (int e1, float prop1, int e2, float prop2, Point3 projDir) {
	int splitme;
	pblock->GetValue (ep_split, TimeValue(0), splitme, FOREVER);
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	int vret;
	vret = mm.CutEdge (e1, prop1, e2, prop2, projDir, splitme?true:false);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return vret;
}

int EditPolyObject::EpfnCutFace (int f1, Point3 start, Point3 dest, Point3 projDir) {
	int splitme;
	pblock->GetValue (ep_split, TimeValue(0), splitme, FOREVER);
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	f1 = mm.CutFace (f1, start, dest, projDir, splitme?true:false);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return f1;
}

bool EditPolyObject::EpfnWeldVerts (int v1, int v2, Point3 destination) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore(this);
		tchange->Before();
	}
	// If vertices v1 and v2 share an edge, then take a collapse type approach;
	// Otherwise, weld them if they're suitable (open verts, etc.)
	int i;
	bool ret=false;
	for (i=0; i<mm.vedg[v1].Count(); i++) {
		if (mm.e[mm.vedg[v1][i]].OtherVert(v1) == v2) break;
	}
	if (i<mm.vedg[v1].Count()) {
		ret = mm.WeldEdge (mm.vedg[v1][i]);
		if (mm.v[v1].GetFlag (MN_DEAD)) mm.v[v2].p = destination;
		else mm.v[v1].p = destination;
	} else {
		ret = mm.WeldBorderVerts (v1, v2, &destination);
	}
	if (!ret) {
		if (tchange) delete tchange;
		return false;
	}
	if (tchange) {
		tchange->After();
		theHold.Put (tchange);
	}
	CollapseDeadStructs ();
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return true;
}

bool EditPolyObject::EpfnWeldEdges (int e1, int e2) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore(this);
		tchange->Before();
	}
	bool ret = mm.WeldBorderEdges (e1, e2);
	if (!ret) {
		if (tchange) delete tchange;
		return false;
	}
	if (tchange) {
		tchange->After();
		theHold.Put (tchange);
	}
	CollapseDeadStructs ();
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return true;
}

bool EditPolyObject::EpfnWeldFlaggedVerts (DWORD flag) {
	float thresh;
	pblock->GetValue (ep_weld_threshold, TimeValue(0), thresh, FOREVER);
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	if (mm.WeldBorderVerts (thresh, flag)) {
		if (tchange) {
			tchange->After ();
			theHold.Put (tchange);
		}
		CollapseDeadStructs ();
		LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
		return true;
	}

	// Otherwise, no weldable vertices.
	// Check for collapsable vertices:
	// (This code was copied from EpfnCollapse.)
	// In order to collapse vertices, we turn them into edge selections, where the edges are shorter than the
	// weld threshold.
	// In order to make use of edge clusters (to make this easy), we need to backup the current edge
	// selection, replace it with the new selection, and replace the original edge selection later.
	Tab<bool> flagBackup;
	int i;
	flagBackup.SetCount (mm.nume);
	for (i=0; i<mm.nume; i++) flagBackup[i] = mm.e[i].GetFlag (MN_SEL) ? true : false;

	mm.ClearEFlags (MN_SEL);

	float threshSq = thresh*thresh;
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!mm.v[mm.e[i].v1].GetFlag (flag)) continue;
		if (!mm.v[mm.e[i].v2].GetFlag (flag)) continue;
		if (LengthSquared (mm.P(mm.e[i].v1) - mm.P(mm.e[i].v2)) > threshSq) continue;
		mm.e[i].SetFlag (MN_SEL);
	}
	TempData()->Invalidate (PART_SELECT);

	// Collect average locations ourselves, since WeldEdge averages in piecemeal fashion.
	Tab<Point3> pointSum;
	Tab<int> pointNum;
	pointSum.SetCount (mm.numv);
	pointNum.SetCount (mm.numv);
	for (i=0; i<mm.numv; i++) {
		pointSum[i] = mm.v[i].p;
		pointNum[i] = 1;
	}

	// Perform topological welding.
	bool ret=false;
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!mm.e[i].GetFlag (MN_SEL)) continue;
		int v1 = mm.e[i].v1;
		int v2 = mm.e[i].v2;
		if (mm.WeldEdge (i)) {
			pointSum[v1] += pointSum[v2];
			pointNum[v1] += pointNum[v2];
			ret = true;
		}
	}

	// Then set all the welded vertices to the correct locations - edge by edge above produces wrong geometric result.
	if (ret) {
		for (i=0; i<mm.numv; i++) {
			if (mm.v[i].GetFlag (MN_DEAD)) continue;
			if (pointNum[i] < 2) continue;
			mm.v[i].p = pointSum[i] / float(pointNum[i]);
		}
	}

	// Restore previous edge selections:
	mm.ClearEFlags (MN_SEL);
	for (i=0; i<mm.nume; i++) mm.e[i].SetFlag (MN_SEL, flagBackup[i]);
	TempData()->Invalidate (PART_SELECT);

	if (ret) {
		if (tchange) {
			tchange->After ();
			theHold.Put (tchange);
		}
		CollapseDeadStructs ();
		LocalDataChanged (PART_ALL-PART_SUBSEL_TYPE);
		return true;
	} else {
		if (tchange) delete tchange;
		return false;
	}
}

bool EditPolyObject::EpfnWeldFlaggedEdges (DWORD flag) {
	float thresh;
	pblock->GetValue (ep_weld_threshold, TimeValue(0), thresh, FOREVER);
	mm.ClearVFlags (MN_WHATEVER);
	mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_WHATEVER, MNM_SL_EDGE, flag);
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	if (!mm.WeldBorderVerts (thresh, MN_WHATEVER)) {
		if (tchange) delete tchange;
		return false;
	}
	// Check for edges for which one end has been welded but not the other.
	// This is made possible by special code in MNMesh::WeldBorderVerts
	// that clears the weld flags from the welded vertices - if the weld flags are not MN_SEL.
	for (int i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!mm.e[i].GetFlag (flag)) continue;
		int v1 = mm.e[i].v1;
		int v2 = mm.e[i].v2;
		// If both were welded, or both were not welded, continue.
		if (mm.v[v1].GetFlag (MN_WHATEVER) == mm.v[v2].GetFlag (MN_WHATEVER)) continue;

		// Ok, one's been welded but the other hasn't.  See if there's another weld we can do to seal it up.
		int weldEnd = mm.v[v1].GetFlag (MN_WHATEVER) ? 1 : 0;
		int unweldEnd = 1-weldEnd;
		int va = mm.e[i][weldEnd];
		int vb = mm.e[i][unweldEnd];

		// First of all, if the welded vert only has 2 flagged, open edges, it's clear they should be welded together.
		Tab<int> elist;
		for (int j=0; j<mm.vedg[va].Count(); j++) {
			int eid = mm.vedg[va][j];
			if (eid == i) continue;
			if (mm.e[eid].f2 > -1) continue;
			if (!mm.e[eid].GetFlag (flag)) continue;
			if (mm.e[eid][unweldEnd] != va) continue;	// should have va at the opposite end from edge i.
			elist.Append (1, &eid);
		}
		if (elist.Count() != 1) {
			// Give up for now.  (Perhaps make better solution later.)
			continue;
		}
		int vc = mm.e[elist[0]].OtherVert (va);
		if (!mm.v[vc].GetFlag (MN_WHATEVER)) continue;
		// Ok, now we know which vertices to weld.
		Point3 dest = (mm.v[vb].p + mm.v[vc].p)*.5f;
		if (mm.WeldBorderVerts (vb, vc, &dest)) {
			mm.v[vb].ClearFlag (MN_WHATEVER);
			mm.v[vc].ClearFlag (MN_WHATEVER);
		}
	}
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	CollapseDeadStructs ();
	LocalDataChanged (PART_ALL - PART_SUBSEL_TYPE);
	return true;
}

void EditPolyObject::EpfnAutoSmooth (TimeValue t) {
	for (int i=0; i<mm.numf; i++) if (mm.f[i].FlagMatch (MN_DEAD|MN_SEL, MN_SEL)) break;
	if (i==mm.numf) return;
	float angle;
	pblock->GetValue (ep_face_smooth_thresh, t, angle, FOREVER);
	SmGroupRestore *smr = NULL;
	if (theHold.Holding ()) smr = new SmGroupRestore (this);
	mm.AutoSmooth (angle, true, false);
	if (smr) {
		smr->After ();
		theHold.Put (smr);
	}
	LocalDataChanged (PART_TOPO);
}

void EditPolyObject::EpfnSetDiagonal (int face, int corner1, int corner2) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	mm.SetDiagonal (face, corner1, corner2);
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_TOPO|PART_SELECT);
}

bool EditPolyObject::EpfnRetriangulate (DWORD flag) {
//	if (meshSelLevel[selLevel] != MNM_SL_FACE) return false;
	bool ret = false;
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	int i;
	for (i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (MN_DEAD)) continue;
		if (!mm.f[i].GetFlag (flag)) continue;
		ret = true;
		mm.RetriangulateFace (i);
	}
	if (!ret) {
		delete tchange;
		return false;
	}
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_TOPO);
	return true;
}

bool EditPolyObject::EpfnFlipNormals (DWORD flag) {
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	if (!mm.FlipElementNormals (flag)) {
		if (tchange) delete tchange;
		return false;
	}
	if (tchange) {
		tchange->After ();
		theHold.Put (tchange);
	}
	LocalDataChanged (PART_TOPO);
	return true;
}

// Vertex / Edge data operations:

float EditPolyObject::GetVertexDataValue (int channel, int *numSel, bool *uniform, DWORD flags, TimeValue t) {
	if (numSel) *numSel = 0;
	if (uniform) *uniform = TRUE;
	float ret=1.0f;
	float *vd=NULL;
	int i, found=0;
	memcpy (&ret, VertexDataDefault(channel), sizeof(float));
	vd = mm.vertexFloat (channel);
	for (i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (!mm.v[i].GetFlag (flags)) continue;
		if (!found && vd) ret = vd[i];
		found++;
		if (vd && (ret != vd[i])) break;
	}
	if (uniform && (i<mm.numv)) *uniform = FALSE;
	if (numSel) *numSel = found;
	return ret;
}

float EditPolyObject::GetEdgeDataValue (int channel, int *numSel, bool *uniform, DWORD flags, TimeValue t) {
	if (numSel) *numSel = 0;
	if (uniform) *uniform = TRUE;
	float ret=1.0f;
	float *vd=NULL;
	int i, found=0;
	memcpy (&ret, EdgeDataDefault(channel), sizeof(float));
	vd = mm.edgeFloat (channel);
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!mm.e[i].GetFlag (flags)) continue;
		if (!found && vd) ret = vd[i];
		found++;
		if (vd && (ret != vd[i])) break;
	}
	if (uniform && (i<mm.nume)) *uniform = FALSE;
	if (numSel) *numSel = found;
	return ret;
}

bool HasMinimum (int msl, int channel, float *minim) {
	switch (msl) {
	case MNM_SL_VERTEX:
		switch (channel) {
		case VDATA_WEIGHT:
			*minim = MIN_WEIGHT;
			return TRUE;
		}
		break;
	case MNM_SL_EDGE:
		switch (channel) {
		case EDATA_KNOT:
			*minim = MIN_WEIGHT;
			return TRUE;
		case EDATA_CREASE:
			*minim = 0.0f;
			return TRUE;
		}
		break;
	}
	return FALSE;
}

bool HasMaximum (int msl, int channel, float *maxim) {
	switch (msl) {
	case MNM_SL_VERTEX:
		switch (channel) {
		case VDATA_WEIGHT:
			*maxim = MAX_WEIGHT;
			return TRUE;
		}
		break;
	case MNM_SL_EDGE:
		switch (channel) {
		case EDATA_KNOT:
			*maxim = MAX_WEIGHT;
			return TRUE;
		case EDATA_CREASE:
			*maxim = 1.0f;
			return TRUE;
		}
		break;
	}
	return FALSE;
}

static PerDataRestore *perDataRestore = NULL;
void EditPolyObject::BeginPerDataModify (int mnSelLevel, int channel) {
	DebugPrint ("Beginning PerData Modification\n");
	if (perDataRestore) delete perDataRestore;
	perDataRestore = new PerDataRestore (this, mnSelLevel, channel);
}

bool EditPolyObject::InPerDataModify () {
	return (perDataRestore) ? true : false;
}

void EditPolyObject::EndPerDataModify (bool success) {
	DebugPrint ("Ending PerData Modification\n");
	if (!perDataRestore) return;
	if (success) {
		if (theHold.Holding()) theHold.Put (perDataRestore);
		else delete perDataRestore;
	} else {
		perDataRestore->Restore (FALSE);
		delete perDataRestore;
	}
	perDataRestore = NULL;
}

static MapChangeRestore *mchange = NULL;
void EditPolyObject::BeginVertexColorModify (int mp) {
	if ((mm.numm <= mp) || mm.M(mp)->GetFlag (MN_DEAD)) InitVertColors (mp);
	if (mchange) delete mchange;
	mchange = new MapChangeRestore (this, mp);
}

bool EditPolyObject::InVertexColorModify () {
	return mchange ? true : false;
}

void EditPolyObject::EndVertexColorModify (bool success) {
	if (!mchange) return;
	if (success) {
		if (theHold.Holding()) {
			if (mchange->ReduceMem ()) theHold.Put (mchange);
			else delete mchange;
		}
		else delete mchange;
	} else {
		mchange->Restore (FALSE);
		delete mchange;
	}
	mchange = NULL;
}

void EditPolyObject::SetVertexDataValue (int channel, float value, DWORD flags, TimeValue t) {
	if (!perDataRestore) BeginPerDataModify (MNM_SL_VERTEX, channel);
	float minim=0.0f, maxim=1.0f;
	if (HasMinimum (MNM_SL_VERTEX, channel, &minim)) {
		if (value < minim) value = minim;
	}
	if (HasMaximum (MNM_SL_VERTEX, channel, &maxim)) {
		if (value > maxim) value = maxim;
	}

	int i;
	bool change=false;
	float *defaultValue = (float *) VertexDataDefault (channel);
	bool setToDefault = (value == *defaultValue);
	float *vd = mm.vertexFloat (channel);
	if (!vd) {
		if (setToDefault) return;
		mm.setVDataSupport (channel, TRUE);
		vd = mm.vertexFloat (channel);
	}
	bool allFlagged = true;
	for (i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (!mm.v[i].GetFlag (flags)) {
			allFlagged = false;
			continue;
		}
		vd[i] = value;
		change=true;
	}
	if (setToDefault && allFlagged) mm.setVDataSupport (channel, false);
	if (change) LocalDataChanged (PART_GEOM);
	if (change) RefreshScreen ();
}

void EditPolyObject::SetEdgeDataValue (int channel, float value, DWORD flags, TimeValue t) {
	if (!perDataRestore) BeginPerDataModify (MNM_SL_EDGE, channel);
	float minim=0.0f, maxim=1.0f;
	if (HasMinimum (MNM_SL_EDGE, channel, &minim)) {
		if (value < minim) value = minim;
	}
	if (HasMaximum (MNM_SL_EDGE, channel, &maxim)) {
		if (value > maxim) value = maxim;
	}

	int i;
	bool change=false;
	float *defaultValue = (float *) EdgeDataDefault (channel);
	bool setToDefault = (value == *defaultValue);
	float *vd = mm.edgeFloat (channel);
	if (!vd) {
		if (setToDefault) return;
		mm.setEDataSupport (channel, TRUE);
		vd = mm.edgeFloat (channel);
	}
	bool allFlagged = true;
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!mm.e[i].GetFlag (flags)) {
			allFlagged = false;
			continue;
		}
		vd[i] = value;
		change=true;
	}
	if (setToDefault && allFlagged) mm.setEDataSupport (channel, false);
	if (change) LocalDataChanged (PART_TOPO);
	if (change) RefreshScreen ();
}

void EditPolyObject::ResetVertexData (int channel) {
	mm.freeVData (channel);
	LocalDataChanged (PART_GEOM);
}

void EditPolyObject::ResetEdgeData (int channel) {
	mm.freeEData (channel);
	LocalDataChanged (PART_TOPO);
}

void EditPolyObject::InitVertColors (int mp) {
	if (theHold.Holding ()) theHold.Put (new InitVertColorRestore(this, mp));
	if (mp>=mm.numm) mm.SetMapNum (mp+1);
	mm.M(mp)->ClearFlag (MN_DEAD);
	mm.M(mp)->setNumFaces (mm.numf);
	mm.M(mp)->setNumVerts (mm.numv);
	for (int i=0; i<mm.numv; i++) mm.M(mp)->v[i] = UVVert(1,1,1);
	for (i=0; i<mm.numf; i++) mm.M(mp)->f[i] = mm.f[i];
	LocalDataChanged (PART_VERTCOLOR);
}

Color EditPolyObject::GetVertexColor (bool *uniform, int *num, int mp, DWORD flags, TimeValue t) {
	static Color white(1,1,1), black(0,0,0);
	int i;
	Color col=white;
	bool init=false;
	if (uniform) *uniform = true;
	if (num) *num = 0;

	if ((mm.numm <= mp) || (mm.M(mp)->GetFlag (MN_DEAD))) {
		if (num) {
			for (i=0; i<mm.numv; i++) {
				if (mm.v[i].GetFlag (MN_DEAD)) continue;
				if (mm.v[i].GetFlag (flags)) (*num)++;
			}
		}
		return white;
	}
	MNMapFace *cf = mm.M(mp)->f;
	UVVert *cv = mm.M(mp)->v;
	if (!cf || !cv) {
		if (num) {
			for (i=0; i<mm.numv; i++) {
				if (mm.v[i].GetFlag (MN_DEAD)) continue;
				if (mm.v[i].GetFlag (flags)) (*num)++;
			}
		}
		return white;
	}

	for (i=0; i<mm.numf; i++) {
		int *tt = cf[i].tv;
		int *vv = mm.f[i].vtx;
		for (int j=0; j<mm.f[i].deg; j++) {
			if (!mm.v[vv[j]].GetFlag (flags)) continue;
			if (num) (*num)++;
			if (!init) {
				col = cv[tt[j]];
				init = TRUE;
			} else {
				Color ac = cv[tt[j]];
				if (ac!=col) {
					if (uniform) *uniform = false;
					return black;
				}
			}
		}
	}
	return col;
}

void EditPolyObject::SetVertexColor (Color clr, int mp, DWORD flags, TimeValue t) {
	if (!mchange) BeginVertexColorModify (mp);
	for (int i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (mm.v[i].GetFlag (flags)) break;
	}
	if (i>=mm.numv) return;
	if ((mm.numm <= mp) || mm.M(mp)->GetFlag (MN_DEAD)) InitVertColors (mp);
	UVVert uvColor(clr);
	mm.SetVertColor (uvColor, mp, flags);
	LocalDataChanged ((mp<1) ? PART_VERTCOLOR : PART_TEXMAP);
	RefreshScreen ();
}

void EditPolyObject::EpfnSelectVertByColor (BOOL add, BOOL sub, int mp, TimeValue t) {
	MNMapFace *cf;
	UVVert *cv;
	if ((mp>=mm.numm) || (mp<-NUM_HIDDENMAPS) || mm.M(mp)->GetFlag (MN_DEAD)) {
		cf = NULL;
		cv = NULL;
	} else {
		cf = mm.M(mp)->f;
		cv = mm.M(mp)->v;
	}

	int deltaR, deltaG, deltaB;
	Color selColor;
	pblock->GetValue (ep_vert_sel_color, t, selColor, FOREVER);
	pblock->GetValue (ep_vert_selc_r, t, deltaR, FOREVER);
	pblock->GetValue (ep_vert_selc_g, t, deltaG, FOREVER);
	pblock->GetValue (ep_vert_selc_b, t, deltaB, FOREVER);

	UVVert clr = selColor;
	float dr = float(deltaR)/255.0f;
	float dg = float(deltaG)/255.0f;
	float db = float(deltaB)/255.0f;

	theHold.Begin();

	BitArray nvs;
	if (add || sub) {
		nvs = GetVertSel ();
		nvs.SetSize (mm.numv, TRUE);
	} else {
		nvs.SetSize (mm.numv);
		nvs.ClearAll();
	}

	Point3 col(1,1,1);
	for (int i=0; i<mm.numf; i++) {
		for (int j=0; j<mm.f[i].deg; j++) {
			if (cv && cf) col = cv[cf[i].tv[j]];
			if ((float)fabs(col.x-clr.x) > dr) continue;
			if ((float)fabs(col.y-clr.y) > dg) continue;
			if ((float)fabs(col.z-clr.z) > db) continue;
			if (sub) nvs.Clear(mm.f[i].vtx[j]);
			else nvs.Set(mm.f[i].vtx[j]);
		}
	}
	SetVertSel (nvs, this, t);
	theHold.Accept (GetString (IDS_SEL_BY_COLOR));
	LocalDataChanged ();
	RefreshScreen ();
}

Color EditPolyObject::GetFaceColor (bool *uniform, int *num, int mp, DWORD flags, TimeValue t) {
	static Color white(1,1,1), black(0,0,0);
	if (uniform) *uniform = true;
	if (num) *num = 0;
	int i;
	if ((mp>=mm.numm) || (mp<-NUM_HIDDENMAPS) || (mm.M(mp)->GetFlag (MN_DEAD))) {
		if (num) {
			for (i=0; i<mm.numf; i++) {
				if (!mm.f[i].GetFlag (MN_DEAD) && mm.f[i].GetFlag (flags)) (*num)++;
			}
		}
		return white;
	}
	BOOL init=FALSE;
	Color col=white;

	MNMapFace *cf = mm.M(mp)->f;
	UVVert *cv = mm.M(mp)->v;

	for (i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (MN_DEAD)) continue;
		if (!mm.f[i].GetFlag (flags)) continue;
		if (num) (*num)++;
		int *tt = cf[i].tv;
		for (int j=0; j<cf[i].deg; j++) {
			if (!init) {
				col = cv[tt[j]];
				init = TRUE;
			} else {
				Color ac = cv[tt[j]];
				if (ac!=col) {
					if (uniform) *uniform = false;
					return black;
				}
			}
		}
	}
	return col;
}

void EditPolyObject::SetFaceColor (Color clr, int mp, DWORD flags, TimeValue t) {
	if (!mchange) BeginVertexColorModify (mp);
	for (int i=0; i<mm.numf; i++) {
		if (!mm.f[i].GetFlag (MN_DEAD) && mm.f[i].GetFlag (flags)) break;
	}
	if (i>=mm.numf) return;
	if ((mm.numm <= mp) || mm.M(mp)->GetFlag (MN_DEAD)) InitVertColors (mp);
	UVVert uvColor(clr);
	mm.SetFaceColor (uvColor, mp, flags);
	LocalDataChanged ((mp<1) ? PART_VERTCOLOR : PART_TEXMAP);
	RefreshScreen ();
}

// Face surface operations:
int EditPolyObject::GetMatIndex(bool * determined) {
	MtlID mat = 0;
	*determined = false;
	for (int j=0; j<mm.numf; j++) {
		if (!mm.f[j].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
		if (!(*determined)) {
			mat = mm.f[j].material;
			*determined = true;
		} else if (mm.f[j].material != mat) {
			*determined = false;
			return mat;
		}
	}
	return mat;
}

// Note: good reasons for handling theHold.Begin/Accept at higher level.
void EditPolyObject::SetMatIndex (int index, DWORD flags) {
	if (theHold.Holding ()) {
		theHold.Put (new MtlIDRestore (this));
	}
	for (int j=0; j<mm.numf; j++) {
		if (!mm.f[j].FlagMatch (flags|MN_DEAD, flags)) continue;
		mm.f[j].material = MtlID(index);
	}
	LocalDataChanged (PART_TOPO);
	RefreshScreen ();
}

void EditPolyObject::EpfnSelectByMat (int index, bool clear, TimeValue t) {
	MtlID matIndex = MtlID(index);
	BitArray ns;
	if (clear) {
		ns.SetSize (mm.numf);
		ns.ClearAll ();
	} else {
		ns = GetFaceSel ();
	}
	for (int j=0; j<mm.numf; j++) {
		if (mm.f[j].GetFlag (MN_DEAD)) continue;
		if (mm.f[j].material == matIndex) ns.Set(j);
	}
	SetFaceSel (ns, this, t);
	LocalDataChanged ();
}

void EditPolyObject::GetSmoothingGroups (DWORD flag, DWORD *anyFaces, DWORD *allFaces) {
	DWORD localAny;
	if (allFaces) {
		*allFaces = ~0;
		if (!anyFaces) anyFaces = &localAny;
	}
	if (!anyFaces) return;
	*anyFaces = 0;
	for (int j=0; j<mm.numf; j++) {
		if (mm.f[j].GetFlag (MN_DEAD)) continue;
		if (flag && !mm.f[j].GetFlag (flag)) continue;
		*anyFaces |= mm.f[j].smGroup;
		if (allFaces) *allFaces &= mm.f[j].smGroup;
	}
	if (allFaces) *allFaces &= *anyFaces;
}

bool EditPolyObject::LocalSetSmoothBits (DWORD bits, DWORD bitmask, DWORD flags) {
	for (int i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (MN_DEAD)) continue;
		if (mm.f[i].GetFlag (flags)) break;
	}
	if (i==mm.numf) return false;

	theHold.Begin();
	SmGroupRestore *smr = new SmGroupRestore (this);
	bits &= bitmask;
	for (int j=0; j<mm.numf; j++) {
		if (mm.f[j].GetFlag (MN_DEAD)) continue;
		if (!mm.f[j].GetFlag (flags)) continue;
		mm.f[j].smGroup &= ~bitmask;
		mm.f[j].smGroup |= bits;
	}
	smr->After ();
	theHold.Put (smr);
	theHold.Accept(GetString(IDS_ASSIGN_SMGROUP));
	LocalDataChanged (PART_TOPO);
	RefreshScreen ();
	return true;
}

void EditPolyObject::EpfnSelectBySmoothGroup(DWORD bits,BOOL clear, TimeValue t) {
	BitArray nfs;
	if (clear) {
		nfs.SetSize (mm.numf);
		nfs.ClearAll ();
	} else {
		nfs = GetFaceSel ();
	}
	for (int j=0; j<mm.numf; j++) {
		if (mm.f[j].smGroup & bits) nfs.Set(j);
	}
	SetFaceSel (nfs, this, t);
	LocalDataChanged ();
}

//----Globals----------------------------------------------
// Move to class Interface or someplace someday?
bool CreateCurveFromMeshEdges (MNMesh & mesh, INode *onode, Interface *ip, TSTR & name, bool curved, DWORD flag) {
	SuspendAnimate();
	AnimateOff();

	SplineShape *shape = (SplineShape*)GetSplineShapeDescriptor()->Create(0);	
	BitArray done;
	done.SetSize (mesh.nume);

	for (int i=0; i<mesh.nume; i++) {
		if (done[i]) continue;
		if (mesh.e[i].GetFlag (MN_DEAD)) continue;
		if (!mesh.e[i].GetFlag (flag)) continue;

		// The array of points for the spline
		Tab<Point3> pts;

		// Add the first two points.
		pts.Append(1,&mesh.v[mesh.e[i].v1].p,10);
		pts.Append(1,&mesh.v[mesh.e[i].v2].p,10);
		int nextv = mesh.e[i].v2, start = mesh.e[i].v1;

		// Mark this edge as done
		done.Set(i);

		// Trace along selected edges
		// Use loopcount/maxLoop just to avoid a while(1) loop.
		int loopCount, maxLoop=mesh.nume;
		for (loopCount=0; loopCount<maxLoop; loopCount++) {
			Tab<int> & ve = mesh.vedg[nextv];
			for (int j=0; j<ve.Count(); j++) {
				if (done[ve[j]]) continue;
				if (mesh.e[ve[j]].GetFlag (flag)) break;
			}
			if (j==ve.Count()) break;
			if (mesh.e[ve[j]].v1 == nextv) nextv = mesh.e[ve[j]].v2;
			else nextv = mesh.e[ve[j]].v1;

			// Mark this edge as done
			done.Set(ve[j]);

			// Add this vertex to the list
			pts.Append(1,&mesh.v[nextv].p,10);
		}
		int lastV = nextv;

		// Now trace backwards
		nextv = start;
		for (loopCount=0; loopCount<maxLoop; loopCount++) {
			Tab<int> & ve = mesh.vedg[nextv];
			for (int j=0; j<ve.Count(); j++) {
				if (done[ve[j]]) continue;
				if (mesh.e[ve[j]].GetFlag (flag)) break;
			}
			if (j==ve.Count()) break;
			if (mesh.e[ve[j]].v1 == nextv) nextv = mesh.e[ve[j]].v2;
			else nextv = mesh.e[ve[j]].v1;

			// Mark this edge as done
			done.Set(ve[j]);

			// Add this vertex to the list
			pts.Insert(0,1,&mesh.v[nextv].p);
		}
		int firstV = nextv;

		// Now weve got all th points. Create the spline and add points
		Spline3D *spline = new Spline3D(KTYPE_AUTO,KTYPE_BEZIER);					
		int max = pts.Count();
		if (firstV == lastV) {
			max--;
			spline->SetClosed ();
		}
		if (curved) {
			for (int j=0; j<max; j++) {
				int prvv = j ? j-1 : ((firstV==lastV) ? max-1 : 0);
				int nxtv = (max-1-j) ? j+1 : ((firstV==lastV) ? 0 : max-1);
				float prev_length = Length(pts[j] - pts[prvv])/3.0f;
				float next_length = Length(pts[j] - pts[nxtv])/3.0f;
				Point3 tangent = Normalize (pts[nxtv] - pts[prvv]);
				SplineKnot sn (KTYPE_BEZIER, LTYPE_CURVE, pts[j],
						pts[j] - prev_length*tangent, pts[j] + next_length*tangent);
				spline->AddKnot(sn);
			}
		} else {
			for (int j=0; j<max; j++) {
				SplineKnot sn(KTYPE_CORNER, LTYPE_LINE, pts[j],pts[j],pts[j]);
				spline->AddKnot(sn);
			}
			spline->ComputeBezPoints();
		}
		shape->shape.AddSpline(spline);
	}

	shape->shape.InvalidateGeomCache();
	shape->shape.UpdateSels();

	INode *node = ip->CreateObjectNode (shape);
	INode *nodeByName = ip->GetINodeByName (name);
	if (nodeByName != node) {
		if (nodeByName) ip->MakeNameUnique(name);
		node->SetName (name);
	}
	Matrix3 ntm = onode->GetNodeTM(ip->GetTime());
	node->SetNodeTM (ip->GetTime(),ntm);
	node->FlagForeground (ip->GetTime(),FALSE);
	node->SetMtl (onode->GetMtl());
	node->SetObjOffsetPos (onode->GetObjOffsetPos());
	node->SetObjOffsetRot (onode->GetObjOffsetRot());
	node->SetObjOffsetScale (onode->GetObjOffsetScale());	
	ResumeAnimate();
	return true;
}

}	// end namespace EditPoly
