#include <Stdlib.h>
#include <String>
#include <stdio.h>
#include <assert.h>
#include <SceneConv.h>
#include <List/Search.h>

#include "Utility.h"
#include "VirtualFile.h"

using namespace std;

SceneConverter::SceneConverter( void )
{
	m_scene = NULL;
	m_textures = NULL;
	m_lightmaps = NULL;
	m_generate_optimized_level = false;

	// special debug mode that generates a WGT file for doing cutscene head scaling
	// (ideally, this data will come from the exporter eventually)
	m_generate_weight_map = false;

	mp_weight_map_vertices = NULL;
	m_num_weight_map_vertices = 0;
}

SceneConverter::~SceneConverter( void )
{
	if ( m_scene )
	{
		delete m_scene;
	}

	if ( m_textures ) 
	{
		delete[] m_textures;
	}

	if ( m_lightmaps )
	{
		delete [] m_lightmaps;
	}

	if ( mp_weight_map_vertices )
	{
		delete [] mp_weight_map_vertices;
	}
}

bool SceneConverter::should_generate_optimized_level( void )
{
	return m_generate_optimized_level;
}

// Version History:
// 1:	Created
// 2:	Removed Mipmap L
//		Added int for address U
//		Added int for address V
//		Added char for alpha cutoff

bool	SceneConverter::load_materials( IoUtils::CVirtualInputFile* pInputFile, int material_version )
{
	int i, num_materials;

	pInputFile->Read((char *) &num_materials, sizeof( int ));
	
	m_scene->m_NumMaterials = num_materials;	
	for( i = 0; i < num_materials; i++ )
	{
		NxMaterial* cur_mat;
		int j, flags;

		cur_mat = new NxMaterial;		
		pInputFile->Read((char *) &cur_mat->m_Checksum, sizeof( unsigned long ));
		
		pInputFile->Read((char *) &flags, sizeof( int ));		
		if ( material_version >= 0x000F )
		{
			pInputFile->Read((char *) &cur_mat->m_materialName, sizeof( unsigned long ));
			//cur_mat->m_Checksum = cur_mat->m_materialName;
		}
		else
		{
			cur_mat->m_materialName = cur_mat->m_Checksum;
		}

		/*
		if( flags & NxMaterial::mNEWFORMATCRC )
		{
			// The old checksum will be overwritten with the new version if the flag exists
			pInputFile->Read((char *) &cur_mat->m_Checksum, sizeof( unsigned long ));
		}
		*/
		if( flags & NxMaterial::mTRANSPARENT )
		{
			cur_mat->m_GroupFlags |= NxMaterial::mGROUP_TRANSPARENT;			
		}
		if( flags & NxMaterial::mINVISIBLE )
		{
			cur_mat->m_Invisible = true;
		}
		if( flags & NxMaterial::mTWO_SIDED )
		{
			cur_mat->m_TwoSided = true;
		}
		if( flags & NxMaterial::mONE_SIDED )
		{
			cur_mat->m_OneSided = true;
		}

		pInputFile->Read((char *) &cur_mat->m_Terrain, sizeof( int ));
		if( material_version >= 0x0002 )
		{
			pInputFile->Read((char *) &cur_mat->m_AlphaCutoff, sizeof( int ));
		}		

		if( material_version >= 0x0005 )
		{
			pInputFile->Read((char *) &cur_mat->m_CutoffFunc, sizeof( int ));
		}

		if( material_version >= 0x0004 )
		{
			pInputFile->Read((char *) &cur_mat->m_DrawOrder, sizeof( float ));			
			pInputFile->Read((char *) &cur_mat->m_Sorted, sizeof( bool ));			
		}

		if( material_version >= 0x0008 )
		{
			pInputFile->Read((char *) &cur_mat->m_grassify, sizeof( bool ));
		}

		if( material_version >= 0x0009 )
		{
			pInputFile->Read((char *) &cur_mat->m_grassHeight, sizeof( float ));
			pInputFile->Read((char *) &cur_mat->m_grassLayers, sizeof( int ));
		}

		if( material_version >= 0x000E )
		{
			pInputFile->Read((char *) &cur_mat->m_water, sizeof( bool ));
		}
		
		if( material_version >= 0x000D )
		{
			pInputFile->Read((char *) &cur_mat->m_BasePass, sizeof( int ));
		}

		// Specular data
		if( material_version >= 0x000A )
		{
			if (flags & NxMaterial::mSPECULAR)
			{
				pInputFile->Read((char *) &cur_mat->m_SpecularPower, sizeof( float ));
				pInputFile->Read((char *) &cur_mat->m_SpecularColor, sizeof(float)*3);
			}
		}

		if( flags & NxMaterial::mVC_WIBBLE )
		{
			int k, num_sequences;

			pInputFile->Read((char *) &num_sequences, sizeof( int ));
			for( j = 0; j < num_sequences; j++ )
			{
				VCWibbleSequence* new_seq;

				new_seq = new VCWibbleSequence;
				new_seq->m_Index = j + 1;	// sequences are one-based
				pInputFile->Read((char *) &new_seq->m_NumFrames, sizeof( int ));
				new_seq->m_WibbleFrames = new WibbleKeyframe[new_seq->m_NumFrames];
				for( k = 0; k < new_seq->m_NumFrames; k++ )
				{
					pInputFile->Read((char *) &new_seq->m_WibbleFrames[k].m_Time, sizeof( int ));
					pInputFile->Read((char *) &new_seq->m_WibbleFrames[k].m_Color, 4 * sizeof( float ));
				}
				cur_mat->m_WibbleSequences.AddToTail( new_seq );
			}
		}

		pInputFile->Read((char *) &cur_mat->m_NumPasses, sizeof( int ));
		for( j = 0; j < cur_mat->m_NumPasses; j++ )
		{
			NxMaterialPass* pass = &cur_mat->m_Passes[j];
			int k, flags, mmag, mmin;
			unsigned long tex_checksums[ Utils::vNUM_PLATFORMS ];

			// In version 0x000C we moved the flags to the top of the loop
			if( material_version >= 0x000C )
			{
				pInputFile->Read((char*) &pass->m_Flags, sizeof( unsigned long ));
				if( pass->m_Flags & NxMaterialPass::ANIMATED_TEXTURE )
				{
					NxAnimatedTexture* anim_tex;
					int frame;

					anim_tex = &pass->m_AnimatedTexture;
					pInputFile->Read((char *) &anim_tex->m_NumKeyframes, sizeof( int ));
					anim_tex->m_Keyframes = new NxAnimatedTextureKeyframe[anim_tex->m_NumKeyframes];
					pInputFile->Read((char *) &anim_tex->m_Period, sizeof( int ));
					pInputFile->Read((char *) &anim_tex->m_Iterations, sizeof( int ));
					pInputFile->Read((char *) &anim_tex->m_Phase, sizeof( int ));
					for( frame = 0; frame < anim_tex->m_NumKeyframes; frame++ )
					{
						NxAnimatedTextureKeyframe* key_frame;

						key_frame = &anim_tex->m_Keyframes[frame];
						pInputFile->Read((char *) &key_frame->m_Time, sizeof( unsigned int ));
						pInputFile->Read((char *) key_frame->m_TexChecksum, Utils::vNUM_PLATFORMS * sizeof( unsigned long ));
					}
				}
				else
				{
					pInputFile->Read((char *) tex_checksums, Utils::vNUM_PLATFORMS * sizeof( unsigned long ));
					for( k = 0; k < Utils::vNUM_PLATFORMS; k++ )
					{
						pass->SetTextureChecksum( k, tex_checksums[k] );
					}
				}
			}
			else
			{
				pInputFile->Read((char *) tex_checksums, Utils::vNUM_PLATFORMS * sizeof( unsigned long ));
				for( k = 0; k < Utils::vNUM_PLATFORMS; k++ )
				{
					pass->SetTextureChecksum( k, tex_checksums[k] );
				}
			}
			
			pInputFile->Read((char *) &flags, sizeof( int ));

			pInputFile->Read((char*) &pass->m_BlendMode, sizeof( int ));

			if( material_version >= 0x0002 )
			{
				pInputFile->Read((char*) &pass->m_AddressModeU, sizeof( int ));
				pInputFile->Read((char*) &pass->m_AddressModeV, sizeof( int ));
			}
			else
			{
				// Default to something sensible.
				pass->m_AddressModeU = 0;
				pass->m_AddressModeV = 0;

			}
			pInputFile->Read((char*) &pass->m_FixedAlpha, sizeof( int ));
			// Read in surface properties of the material
			
			if ( material_version >= 0x0003 )
			{
				pInputFile->Read((char*) &pass->m_HasColor, sizeof( bool ));		// aml
			}
			else
			{
				pass->m_HasColor=true;
			}

			pInputFile->Read((char*) &pass->m_Color[0], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Color[1], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Color[2], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Color[3], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Diffuse[0], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Diffuse[1], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Diffuse[2], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Specular[0], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Specular[1], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Specular[2], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Ambient[0], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Ambient[1], sizeof( float ));
			pInputFile->Read((char*) &pass->m_Ambient[2], sizeof( float ));

			// Read in UV wibble params
			if( flags & NxMaterial::mUV_WIBBLE )
			{
				// Flag that this pass has UV wibbling enabled.
				pass->m_UVWibbleEnabled = true;

				pInputFile->Read((char*) &pass->m_UVel, sizeof( float ));
				pInputFile->Read((char*) &pass->m_VVel, sizeof( float ));
				pInputFile->Read((char*) &pass->m_UFrequency, sizeof( float ));
				pInputFile->Read((char*) &pass->m_VFrequency, sizeof( float ));
				pInputFile->Read((char*) &pass->m_UAmplitude, sizeof( float ));
				pInputFile->Read((char*) &pass->m_VAmplitude, sizeof( float ));
				pInputFile->Read((char*) &pass->m_UPhase, sizeof( float ));
				pInputFile->Read((char*) &pass->m_VPhase, sizeof( float ));
			}

			if( material_version >= 0x000B )
			{
				if( flags & NxMaterial::mENVIRONMENT )
				{
					pInputFile->Read((char*) &pass->m_EnvTileU, sizeof( float ));
					pInputFile->Read((char*) &pass->m_EnvTileV, sizeof( float ));
				}
			}

			// Read in VC wibble sequences
			if( flags & NxMaterial::mVC_WIBBLE )
			{
				int num_sequences;

				pInputFile->Read((char*) &num_sequences, sizeof( int ));
			}

			if( flags & NxMaterial::mENVIRONMENT )
			{
				pass->m_MappingMode = vMAPPING_ENVIRONMENT;
			}

			// Read in MMAG and MMIN		
			pInputFile->Read((char*) &mmag, sizeof( int ));
			pInputFile->Read((char*) &mmin, sizeof( int ));
			pass->m_MagFilteringMode = (FilteringMode) mmag;
			pass->m_MinFilteringMode = (FilteringMode) mmin;

			// Read in K
			pInputFile->Read((char*) &pass->m_MipMapK, sizeof( float ));
			if( material_version < 0x0002 )
			{
				float mip_map_l;
				pInputFile->Read((char*) &mip_map_l, sizeof( float ));		
			}

			// In version 0x000C we moved the flags to the top of the loop
			if (( material_version >= 0x000A ) && ( material_version < 0x000C ))
			{
				pInputFile->Read((char*) &pass->m_Flags, sizeof( unsigned long ));
			}
		}

		// If the user has created a material that blends with the frame-buffer but has not flagged it as
		// transparent, we will nudge it just a bit so that it renders right after opaque materials
		if(	( cur_mat->m_Passes[0].m_BlendMode != vBLEND_MODE_DIFFUSE ) &&
			( !( cur_mat->m_GroupFlags & NxMaterial::mGROUP_TRANSPARENT )))
		{
			cur_mat->m_GroupFlags |= NxMaterial::mGROUP_TRANSPARENT;
			cur_mat->m_DrawOrder = 1.0f;
		}

		// Keep the materials in sorted order based on draw order id
		if( cur_mat->m_Sorted )
		{
			// Sort "sorted" materials to the end of the list
			// I add one to make sure that we won't get a value of 0, which would merge sorted materials
			// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
			// sort to the end of the list
			cur_mat->SetPri(( cur_mat->m_DrawOrder + 1.0f ) * 1000.0f );
		}
		else
		{
			cur_mat->SetPri( cur_mat->m_DrawOrder );
		}
		
		m_scene->m_Materials.AddNode( cur_mat );
		m_scene->AddMaterialToLookup( cur_mat );
	}

	return true;
}

bool	SceneConverter::temp_write_weight_map_file( char* path )
{
	printf( "Writing weight map file %s...\n", path );

	NxObject* object = get_cas_object();

	if ( !object )
	{
		return false;
	}

	if ( !(object->m_Flags & NxObject::mSKINNED) )
	{
		// should assert?
		return false;
	}

	IoUtils::CVirtualOutputFile theWeightMapFile;

	// 10-meg buffer
	theWeightMapFile.Init(10*1024*1024);

	theWeightMapFile.Write( (const char*)&object->m_NumVerts, sizeof(int), false );		

	NxVertex* vert_list = object->m_Verts;
	for ( int j = 0; j < object->m_NumVerts; j++ )
	{
		NxVertex* vert;
		vert = &vert_list[j];

		theWeightMapFile.Write( (const char*)&vert->m_Weight[0], sizeof(float), false );
		theWeightMapFile.Write( (const char*)&vert->m_Weight[1], sizeof(float), false );
		theWeightMapFile.Write( (const char*)&vert->m_Weight[2], sizeof(float), false );
	}

	for ( int j = 0; j < object->m_NumVerts; j++ )
	{
		NxVertex* vert;
		vert = &vert_list[j];

		theWeightMapFile.Write( (const char*)&vert->m_WeightedIndex[0], sizeof(int), false );
		theWeightMapFile.Write( (const char*)&vert->m_WeightedIndex[1], sizeof(int), false );
		theWeightMapFile.Write( (const char*)&vert->m_WeightedIndex[2], sizeof(int), false );
	}

	// don't do anything special about breaking the lock
	// it's only a temporary function anyway...
	if ( !theWeightMapFile.Save( path ) )
	{
		Utils::Assert( 0, "Couldn't write map file %s?", path );
		return false;
	}

	return true;
}

bool	SceneConverter::load_weight_map_file( char* path )
{
	IoUtils::CVirtualInputFile theWeightMapFile;
	if ( !theWeightMapFile.Load( path ) )
	{
		Utils::Assert( 0, "Couldn't open weight map file %s", theWeightMapFile );
		return false;
	}
				
	printf( "Reading weight map file %s...\n", path );

	NxObject* object = get_cas_object();

	if ( !object )
	{
		return false;
	}

	if ( !(object->m_Flags & NxObject::mSKINNED) )
	{
		// should assert?
		return false;
	}

	int numVerts;
	theWeightMapFile.Read( (char*)&numVerts, sizeof(int) );

	if ( numVerts != object->m_NumVerts )
	{
		Utils::Assert( 0, "Expected same number of verts in weight map file (SKIN=%d WGT=%d)", object->m_NumVerts, numVerts );
		return false;
	}
	
	NxVertex* vert_list = object->m_Verts;
	for ( int j = 0; j < object->m_NumVerts; j++ )
	{
		NxVertex* vert;
		vert = &vert_list[j];

		if ( !theWeightMapFile.Read( (char*)&vert->m_MeshScalingWeight[0], sizeof(float) ) )
		{
			return false;
		}

		if ( !theWeightMapFile.Read( (char*)&vert->m_MeshScalingWeight[1], sizeof(float) ) )
		{
			return false;
		}

		if ( !theWeightMapFile.Read( (char*)&vert->m_MeshScalingWeight[2], sizeof(float) ) )
		{
			return false;
		}
	}

	for ( int j = 0; j < object->m_NumVerts; j++ )
	{
		NxVertex* vert;
		vert = &vert_list[j];

		if ( !theWeightMapFile.Read( (char*)&vert->m_MeshScalingWeightedIndex[0], sizeof(int) ) )
		{
			return false;
		}

		if ( !theWeightMapFile.Read( (char*)&vert->m_MeshScalingWeightedIndex[1], sizeof(int) ) )
		{
			return false;
		}

		if ( !theWeightMapFile.Read( (char*)&vert->m_MeshScalingWeightedIndex[2], sizeof(int) ) )
		{
			return false;
		}
	}

	return true;
}

bool	SceneConverter::load_objects( IoUtils::CVirtualInputFile* pInputFile, int mesh_version )
{
	int i, j, k;
	bool bReadSkeletalInfo = false;
	int  numRemovedObjects = 0;				// Number of removed objects for net games

	pInputFile->Read((char*) &m_scene->m_NumObjects, sizeof( int ));	 
	//m_scene->m_Objects = new NxObject[ m_scene->m_NumObjects ];	
	NxObject* tmpObjects = new NxObject[ m_scene->m_NumObjects ];
	//m_scene->m_Objects = new NxObject[ m_scene->m_NumObjects ];

	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxVertex* vert_list;		
		NxFace* face_list;
		NxObject* object;

		//object = &m_scene->m_Objects[i];
		object = &tmpObjects[i];
			
		// Read in the object's checksum
		pInputFile->Read((char*) &object->m_Checksum, sizeof( int ));		

		// Read in the flags which describe the object's properties
		pInputFile->Read((char*) &object->m_Flags, sizeof( int ));

		// If -o is set, we'll need to remove any objects that are flagged as mABSENTINNETGAMES later
		if (m_generate_optimized_level)
		{
			if (object->m_Flags & NxObject::mABSENTINNETGAMES)
				numRemovedObjects++;
		}

		if( object->m_Flags & NxObject::mSKELETALMODEL )
		{
			pInputFile->Read((char*) &object->m_ParentMatrixIndex, sizeof( int ));
			bReadSkeletalInfo = true;
		}

		if( object->m_Flags & NxObject::mSKINNED )
		{
			pInputFile->Read((char*) &object->m_SkeletonChecksum, sizeof( int ));
		
			// also allocate the buffer for create-a-skater flags here
			object->mp_CASData = new NxCASData[NxObject::vMAXCASDATA];
		}
		
		// Read in the number of sets of UVs
		pInputFile->Read((char*) &object->m_NumUVSets, sizeof( int ));

		// Read in bounding box/sphere data
		pInputFile->Read((char*) &object->m_BoundingBox.m_Min, 3 * sizeof( float ));
		pInputFile->Read((char*) &object->m_BoundingBox.m_Max, 3 * sizeof( float ));

		// Copy into collision bbox in case we don't make a new one
		object->m_CollisionBoundingBox = object->m_BoundingBox;

		pInputFile->Read((char*) &object->m_BoundingSphere.m_Center, 3 * sizeof( float ));
		pInputFile->Read((char*) &object->m_BoundingSphere.m_Radius, sizeof( float ));

		pInputFile->Read((char*) &object->m_NumVerts, sizeof( int ));
		object->m_Verts = new NxVertex[ object->m_NumVerts ];
		vert_list = object->m_Verts;
		for( j = 0; j < object->m_NumVerts; j++ )
		{
			NxVertex* vert;
			
			vert = &vert_list[j];

			// Read in first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					pInputFile->Read((char*) &vert->m_TexCoord[k][0], sizeof( float ));
					pInputFile->Read((char*) &vert->m_TexCoord[k][1], sizeof( float ));
				}
			}

			// Read in colors
			if( object->m_Flags & NxObject::mCOLORED )
			{
				pInputFile->Read((char*) &vert->m_Color[0], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Color[1], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Color[2], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Color[3], sizeof( float ));
			}				

			// Read in normals
			if( object->m_Flags & ( NxObject::mNORMALS | NxObject::mSS_NORMALS ))
			{					
				pInputFile->Read((char*) &vert->m_Normal[0], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Normal[1], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Normal[2], sizeof( float ));
			}

			if(  object->m_Flags & NxObject::mSKINNED)
			{
				pInputFile->Read((char*) &vert->m_Weight[0], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Weight[1], sizeof( float ));
				pInputFile->Read((char*) &vert->m_Weight[2], sizeof( float ));

				pInputFile->Read((char*) &vert->m_WeightedIndex[0], sizeof( int ));
				pInputFile->Read((char*) &vert->m_WeightedIndex[1], sizeof( int ));
				pInputFile->Read((char*) &vert->m_WeightedIndex[2], sizeof( int ));

				if ( object->m_Flags & NxObject::mHAS4WEIGHTS )
				{
					pInputFile->Read((char*) &vert->m_Weight[3], sizeof( float ));
					pInputFile->Read((char*) &vert->m_WeightedIndex[3], sizeof( int ));
				}
				else
				{
					vert->m_Weight[3]        = 0.0f;
					vert->m_WeightedIndex[3] = 0;
				}
			}

			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				pInputFile->Read((char*) &vert->m_WibbleIndex, sizeof( char ));
				pInputFile->Read((char*) &vert->m_WibbleOffset, sizeof( char ));
			}

			// Read in positions				
			pInputFile->Read((char*) &vert->m_Pos[0], sizeof( float ));
			pInputFile->Read((char*) &vert->m_Pos[1], sizeof( float ));
			pInputFile->Read((char*) &vert->m_Pos[2], sizeof( float ));		

			// Round vert positions to nearest tenth of an inch
			vert->m_Pos[0] = ((int) (( vert->m_Pos[0] * 16.0f ) + ((vert->m_Pos[0] >0.0f) ? 0.5f : -0.5f) )) / 16.0f;
			vert->m_Pos[1] = ((int) (( vert->m_Pos[1] * 16.0f ) + ((vert->m_Pos[1] >0.0f) ? 0.5f : -0.5f) )) / 16.0f;
			vert->m_Pos[2] = ((int) (( vert->m_Pos[2] * 16.0f ) + ((vert->m_Pos[2] >0.0f) ? 0.5f : -0.5f) )) / 16.0f;
		}
		
		// Copy the verts over into the original verts field.
		object->m_OriginalVerts = new NxVertex[object->m_NumVerts];
		memcpy( object->m_OriginalVerts, object->m_Verts, sizeof( NxVertex ) * object->m_NumVerts );
		
		pInputFile->Read((char*) &object->m_NumFaces, sizeof( int ));
		object->m_Faces = new NxFace[ object->m_NumFaces ];
		face_list = object->m_Faces;		
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			FlagType min_flags;

			// Read in the material checksum for this face
			pInputFile->Read((char*) &face_list[j].m_MatChecksum, sizeof( unsigned long ));

			// Read in the face normal
			pInputFile->Read((char*) &face_list[j].m_Normal, 3 * sizeof( float ));

			// Read in the face flags
			pInputFile->Read((char*) &face_list[j].m_FaceFlags, sizeof( FlagType ));

			// Read in the minimalist face flags
			if( mesh_version >= 0x0002 )
			{
				pInputFile->Read((char*) &min_flags, sizeof( FlagType ));
				if( should_generate_optimized_level())
				{
					face_list[j].m_FaceFlags = min_flags;
				}
			}		

			if (face_list[j].m_FaceFlags & NxFace::mFD_CASFACEFLAGSEXIST)	// (versioning data isn't exported for NxObject)
			{
				// Read in CAS face flags
				pInputFile->Read((char*) &face_list[j].m_CASFaceFlags, sizeof( CASFlagType ));
			}

			face_list[j].m_PassFlags = 0;
			if( !( face_list[j].m_FaceFlags & NxFace::mFD_PASS_1_DISABLED ))
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_1;
			}
			if( face_list[j].m_FaceFlags & NxFace::mFD_PASS_2_ENABLED )
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_2;
			}
			if( face_list[j].m_FaceFlags & NxFace::mFD_PASS_3_ENABLED )
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_3;
			}
			if( face_list[j].m_FaceFlags & NxFace::mFD_PASS_4_ENABLED )
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_4;
			}

			// Read in the vertex indices
			pInputFile->Read((char*) face_list[j].m_Vertex, sizeof( int ) * 3 );

			// Read in the lightmap data for this face
			if (face_list[j].m_FaceFlags & NxFace::mFD_LIGHTMAPPED)
			{
				pInputFile->Read((char*)&face_list[j].m_LightmapChecksum, sizeof(unsigned long) );
				pInputFile->Read((char*) face_list[j].m_LightmapUV, sizeof(UVCoord) * 3 );
			}
			else
			{
				face_list[j].m_LightmapChecksum = 0;
				face_list[j].m_LightmapUV[0].u  = 0;
				face_list[j].m_LightmapUV[0].v  = 0;
				face_list[j].m_LightmapUV[1]    = face_list[j].m_LightmapUV[0];
				face_list[j].m_LightmapUV[2]    = face_list[j].m_LightmapUV[0];
			}
		}		

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Read in CAS face removal flags
			pInputFile->Read((char*)&object->m_CASRemoveFlags, sizeof( int ));
		}

		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Read in KBias
			pInputFile->Read((char*)&object->m_KBias,sizeof( float ));
		}

		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Read in the LOD info
			unsigned int version;
			pInputFile->Read((char*)&version,sizeof(unsigned int));

			if (version >= 0x0001)
			{
				pInputFile->Read((char*)&object->m_LODFlags,sizeof(int));

				if (object->m_LODFlags & NxObject::mMASTER)
				{
					pInputFile->Read((char*)&object->m_LODMaster.m_NumLODLevels,sizeof(int));

					if (object->m_LODMaster.m_LODLevels)
						delete [] object->m_LODMaster.m_LODLevels;

					object->m_LODMaster.m_LODLevels = new NxLODLevel[object->m_LODMaster.m_NumLODLevels];

					pInputFile->Read((char*)object->m_LODMaster.m_LODLevels,sizeof(NxLODLevel) * object->m_LODMaster.m_NumLODLevels);
					for (int lidx = 0; lidx < object->m_LODMaster.m_NumLODLevels; lidx++)
					{
						//printf("Reading LOD %x with distance of %f\n", object->m_LODMaster.m_LODLevels[lidx].m_ObjectCRC, object->m_LODMaster.m_LODLevels[lidx].m_Distance);
						Utils::Assert(object->m_Checksum != object->m_LODMaster.m_LODLevels[lidx].m_ObjectCRC,"Something LOD related!");
					}
				}
				
				if (object->m_LODFlags & NxObject::mSLAVE)
				{
					pInputFile->Read((char*)&object->m_LODSlave.m_masterCRC,sizeof(unsigned long));
					Utils::Assert(object->m_Checksum != object->m_LODSlave.m_masterCRC,"Something LOD related again!");
				}
			}
		}

		if (object->m_Flags & NxObject::mHASINTLODINFO)
		{
			pInputFile->Read((char*)&object->m_LODLevels,sizeof(int));
			
			object->m_LODinfo = new NxLODInfo[object->m_LODLevels];

			for(int lodlev = 0; lodlev < object->m_LODLevels; lodlev++)
			{
				pInputFile->Read((char*)&object->m_LODinfo[lodlev].numFaces, sizeof(int));

				object->m_LODinfo[lodlev].faces = new NxLODFace[object->m_LODinfo[lodlev].numFaces];

				pInputFile->Read((char*)object->m_LODinfo[lodlev].faces,
					              sizeof(NxLODFace) * object->m_LODinfo[lodlev].numFaces);

				// If userdefined LOD distances are included, load them in
				if (object->m_Flags & NxObject::mHASLODDISTS)
					pInputFile->Read((char*)&object->m_LODinfo[lodlev].distance, sizeof(float));
			}

			/////////////// RETURN HERE!
		}

		if (object->m_Flags & NxObject::mBILLBOARD)
		{
			pInputFile->Read((char*)&object->m_BillboardType, sizeof(NxObject::BillboardType));
			pInputFile->Read((char*)&object->m_BillboardOrigin, sizeof(float) * 3);
			pInputFile->Read((char*)&object->m_PivotPos, sizeof(float) * 3);
			pInputFile->Read((char*)&object->m_PivotAxis, sizeof(float) * 3);
		}

		if (object->m_Flags & NxObject::mHASVERSIONINFO)
		{
			unsigned int version;
			pInputFile->Read((char*)&version,sizeof(unsigned int));

			if (version >= 0x0001)
			{
				// Read in relative info
				pInputFile->Read((char*)&object->m_ParentCRC,sizeof(unsigned long));
				pInputFile->Read((char*)&object->m_NumChildren,sizeof(int));
				
				object->m_ChildCRCs = new unsigned long[object->m_NumChildren];
				pInputFile->Read((char*)object->m_ChildCRCs,sizeof(unsigned long) * object->m_NumChildren);
			}
			// The mSS_NORMALS flag is not valid before object version 4
			if( version < 0x0004)
			{
				object->m_Flags &= ~NxObject::mSS_NORMALS;
			}
		}
	}

	// Objects loaded in, reorder into the actual scene object array
	// skipping anything flagged as absent in net games if appropriate
	int curObj = 0;
	m_scene->m_Objects = new NxObject[m_scene->m_NumObjects - numRemovedObjects];

	for(i = 0; i < m_scene->m_NumObjects; i++)
	{
		if (m_generate_optimized_level)
		{
			if (!(tmpObjects[i].m_Flags & NxObject::mABSENTINNETGAMES))
			{
				m_scene->m_Objects[curObj] = tmpObjects[i];
				curObj++;
			}
		}
		else
		{
			m_scene->m_Objects[curObj] = tmpObjects[i];
			curObj++;
		}
	}

	delete [] tmpObjects;
	m_scene->m_NumObjects -= numRemovedObjects;

	// Init master LOD pointers
	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;
		
		object = &m_scene->m_Objects[i];

		if (object->m_Flags & NxObject::mHASLODINFO && object->m_LODFlags & NxObject::mSLAVE)
		{
			object->m_LODSlave.mp_master_object = m_scene->FindObject(object->m_LODSlave.m_masterCRC);
			assert(object != object->m_LODSlave.mp_master_object);
		}
	}

	if (bReadSkeletalInfo)
	{
		int version;
		pInputFile->Read((char*)&version,sizeof(int));

		pInputFile->Read((char*)&m_scene->m_Model.nBones,sizeof(int));
		m_scene->m_Model.bones = new NxModel::BoneDesc[m_scene->m_Model.nBones];

		pInputFile->Read((char*)m_scene->m_Model.bones,sizeof(NxModel::BoneDesc) * m_scene->m_Model.nBones);
	}

	return true;
}

// Create a unique animation sequence for every pair of wibble index/offset that we find
bool	SceneConverter::create_new_vc_wibble_sequences( void )
{
	NxMaterial* material;
	unsigned long last_mat_checksum;
	int i, j;

	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;
		NxFace* face;
		NxVertex* vert;
		bool has_valid_wibble_data;
		
		last_mat_checksum = 0xFFFFFFFF;
		material = NULL;
		object = &m_scene->m_Objects[i];		
		if( object->m_Flags & NxObject::mVCWIBBLE )
		{
			has_valid_wibble_data = false;
			for( j = 0; j < object->m_NumVerts; j++ )
			{
				vert = &object->m_Verts[j];
				face = &object->m_Faces[j/3];
				if( face->m_MatChecksum != last_mat_checksum )
				{
					material = m_scene->GetMaterial( face->m_MatChecksum );
					last_mat_checksum = face->m_MatChecksum;
				}

				vert->m_WibbleIndex = material->GetWibbleSequence( vert->m_WibbleIndex, vert->m_WibbleOffset );				
				if( vert->m_WibbleIndex == 0 )
				{
					vert->m_WibbleOffset = 0;
				}
				else
				{
					has_valid_wibble_data = true;
				}
			}		
			if( has_valid_wibble_data == false )
			{
				object->m_Flags &= ~NxObject::mVCWIBBLE;
			}
		}
	}

	return true;
}

// Some faces only use certain passes of a multi-pass material.  Create a new material
// that represents just the passes that are used
bool	SceneConverter::create_new_multipass_materials( void )
{
	int i, j, k;	

	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;
		NxMaterial* material, *new_mat;
		unsigned long last_mat_checksum;

		last_mat_checksum = 0xFFFFFFFF;
		material = NULL;
		object = &m_scene->m_Objects[i];
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxFace* face;			
			unsigned long new_mat_checksum;
			int pass_flags, diff_flags;

			face = &object->m_Faces[j];			
			if( face->m_PassFlags == 0 )
			{
				material = m_scene->GetMaterial( face->m_MatChecksum );
				material->m_Used = true;
				last_mat_checksum = face->m_MatChecksum;
				continue;
			}

			pass_flags = face->m_PassFlags;
			
			if( face->m_MatChecksum != last_mat_checksum )
			{
				material = m_scene->GetMaterial( face->m_MatChecksum );
				last_mat_checksum = face->m_MatChecksum;
			}		

			diff_flags = 0;
			for( k = 0; k < material->m_NumPasses; k++ )
			{
				if(( pass_flags & ( 1 << k )) == 0 )
				{
					diff_flags |= ( 1 << k );
				}
			}

			new_mat_checksum = face->m_MatChecksum + ( diff_flags * ( vMAX_MATERIAL_PASSES + 1 ));
			new_mat = m_scene->GetMaterial( new_mat_checksum );
			if( new_mat == NULL )				
			{
				m_scene->CopyMaterial( face->m_MatChecksum, new_mat_checksum, pass_flags );

				// Now adjust the name checksum of the new material to indicate that it contains only a subset of the passes
				// that the original material contained.
				if( ( GetPlatformFlags() == mPLAT_XBOX ) || ( GetPlatformFlags() == mPLAT_NGC ) )
				{
					new_mat = m_scene->GetMaterial( new_mat_checksum );
					Utils::Assert( new_mat != NULL, "Failed to get material" );

					// We shift right here because at a later stage there is another separation process for geometry that is flagged
					// 'render as separate', which will also modify the material name. That modification uses the low 3 bits of the
					// name, so this one uses the next 4 bits. Very messy, needs reworking at some point.
					new_mat->m_materialName	+= ( pass_flags << 3 );
				}
			}
			else
			{
				new_mat->m_Used = true;
			}

			face->m_MatChecksum = new_mat_checksum;
		}
	}

	return true;
}


NxObject*	SceneConverter::get_cas_object()
{
	// it is considered a create-a-skater object
	// if there is (only) 1 skinned object in the
	// scene

	int num_obj = m_scene->m_NumObjects;

	NxObject* object;
		
	object = &m_scene->m_Objects[0];

	if ( num_obj != 1 )
	{
		if ( object->m_Flags & NxObject::mSKINNED )
		{
			char msg[512];
			sprintf( msg, "Was expecting only 1 object in skinned model (found %d)", num_obj );
			Utils::Assert( 0, msg );
		}
		return NULL;
	}

	if ( object->m_Flags & NxObject::mSKINNED )
	{
		return object;
	}
	else
	{
		return NULL;
	}
}

bool	SceneConverter::LoadScene( char* path )
{
	int material_version, mesh_version, version;
	
	_splitpath( path, NULL, NULL, m_scene_name, NULL );

	m_scene = new NxScene;

		IoUtils::CVirtualInputFile theInputFile;
		if ( !theInputFile.Load( path ) )
		{
			return false;
		}

		// material version
		theInputFile.Read((char*) &material_version, sizeof( int ));
		// mesh version
		theInputFile.Read((char*) &mesh_version, sizeof( int ));
		// vertex version
		theInputFile.Read((char*) &version, sizeof( int ));	
		theInputFile.Read((char*) &m_sky, sizeof( bool ));	

		if( load_materials( &theInputFile, material_version ) == false )
		{
			return false;
		}

		if( load_objects( &theInputFile, mesh_version ) == false )
		{
			return false;
		}

		// temporary for generating weight maps
		// (eventually will be exported directly into the appropriate SKIN files)
		if ( m_generate_weight_map )
		{
			char lowerCasePath[_MAX_PATH];
			for ( int i = 0; i < strlen(path); i++ )
			{
				lowerCasePath[i] = path[i];
				if ( lowerCasePath[i] >= 'A' && lowerCasePath[i] <= 'Z' )
				{
					lowerCasePath[i] = lowerCasePath[i] - 'A' + 'a';
				}
			}
			lowerCasePath[strlen(path)] = 0;

			char* inputWeightMapNames[] = {
				"head_cas_male01b.skin",
				"head_cas_male02b.skin",
				"head_cas_female01b.skin",
				"head_cas_female02b.skin",
				NULL
			};

			char* outputWeightMapNames[] = {
				"head_cas_male01.wgt",
				"head_cas_male02.wgt",
				"head_cas_female01.wgt",
				"head_cas_female02.wgt",
				NULL
			};

			char** ppInputWeightMapName = &inputWeightMapNames[0];
			char** ppOutputWeightMapName = &outputWeightMapNames[0];

			while ( *ppInputWeightMapName && *ppOutputWeightMapName )
			{
				char* pActualName = strstr( lowerCasePath, *ppInputWeightMapName );

				if ( pActualName  )
				{
					// change extension
					strcpy( pActualName, *ppOutputWeightMapName );
					temp_write_weight_map_file( lowerCasePath );
					break;
				}

				ppInputWeightMapName++;
				ppOutputWeightMapName++;
			}
		}

		m_weight_map_loaded = false;

		// load weight map here, if the WGT file exists
		// (which should only exist for cutscene heads...)  
		// eventually, the exporter should export this data 
		// directly into the SKIN file
		{
			char lowerCasePath[_MAX_PATH];
			for ( int i = 0; i < strlen(path); i++ )
			{
				lowerCasePath[i] = path[i];
				if ( lowerCasePath[i] >= 'A' && lowerCasePath[i] <= 'Z' )
				{
					lowerCasePath[i] = lowerCasePath[i] - 'A' + 'a';
				}
			}
			lowerCasePath[strlen(path)] = 0;

			// change extension
			char weightFileName[_MAX_PATH];
			strcpy( weightFileName, lowerCasePath );
			char* pExt = strstr( weightFileName, ".skin" );
			if ( pExt )	// it's a skinned file
			{
				Utils::Assert( pExt != NULL, "No extension in %s?", weightFileName );
				strcpy( pExt, ".wgt" );

				if ( Utils::GetTimeStamp( weightFileName ) != -1 )
				{
					printf( "Loading weight map file %s\n", weightFileName );

					// weight file exists
					if ( !load_weight_map_file( weightFileName ) )
					{
						Utils::Assert( 0, "Couldn't read weight map file %s", lowerCasePath );

						return false;
					}
					else
					{
						m_weight_map_loaded = true;
						Utils::Assert( mp_weight_map_vertices == NULL, "weight map array already exists" );
						mp_weight_map_vertices = new NxVertex[vMAX_WEIGHT_MAP_VERTICES];
						m_num_weight_map_vertices = 0;
					}
				}
			}
		}

	if( create_new_vc_wibble_sequences() == false )
	{
		return false;
	}

	if( create_new_multipass_materials() == false )
	{
		return false;
	}

	m_scene->RemoveUnusedMaterials();

	return true;
}

bool	SceneConverter::LoadLightmaps( char* path )
{
	// Load in the lightmap .lmp file
	IoUtils::CVirtualInputFile file;
	if ( !file.Load( path ) )
	{
		return false;
	}

	unsigned long version;

	file.Read((char*) &version, sizeof( unsigned long ));
	file.Read((char*) &m_NumLightmaps, sizeof( int ));

	// Allocate memory for each lightmap that will be used
	if (m_lightmaps)
		delete [] m_lightmaps;
	
	m_lightmaps = new NxTexture[m_NumLightmaps];

	for(int i = 0; i < m_NumLightmaps; i++)
	{
		file.Read((char*) &m_lightmaps[i].m_Width[0], sizeof( int ));
		file.Read((char*) &m_lightmaps[i].m_Height[0], sizeof( int ));

		// Lightmaps are not mipped (at least not currently)
		m_lightmaps[i].m_MipLevels = 0;

		// Allocate memory for the actual texel data
		m_lightmaps[i].m_TexelData[0] = new char[m_lightmaps[i].m_Width[0] * m_lightmaps[i].m_Height[0]];

		file.Read(m_lightmaps[i].m_TexelData[0], m_lightmaps[i].m_Width[0] * m_lightmaps[i].m_Height[0]);

		m_lightmaps[i].m_Bpp = 8;
		m_lightmaps[i].m_PaletteBpp = 0;
		m_lightmaps[i].m_PaletteData = NULL;
		m_lightmaps[i].m_NumPaletteEntries = 0;
		m_lightmaps[i].m_Checksum = i;
		m_lightmaps[i].m_TexelDataSize[0] = m_lightmaps[i].m_Width[0] * m_lightmaps[i].m_Height[0];
		m_lightmaps[i].m_TotalTexelDataSize = m_lightmaps[i].m_TexelDataSize[0];
		m_lightmaps[i].m_TotalPaletteDataSize = 0;
		m_lightmaps[i].m_PixelFormat = NxTexture::v8_BIT;
		m_lightmaps[i].m_PaletteFormat = 0;
		m_lightmaps[i].m_Name[0] = 0;
		m_lightmaps[i].m_Flags = 0;
		m_lightmaps[i].m_PlatFlags = Utils::vPLATFORM_PS2 | Utils::vPLATFORM_NGC | Utils::vPLATFORM_XBOX;
		m_lightmaps[i].m_LastDrawOrder = 0;
		m_lightmaps[i].m_LastGroupIndex = 0;
		m_lightmaps[i].m_LastGroupId = 0;
		m_lightmaps[i].m_GroupFlags = 0;	
	}


	return true;
}

// Version History:
// 1:	Created
// 2:	Added an int name length indicator
//		Added the texture's path

bool	SceneConverter::LoadTextureDictionary( char* path )
{
	int i, version;
	bool all_valid;
	
	IoUtils::CVirtualInputFile theInputFile;
	if ( !theInputFile.Load( path ) )
	{
		m_num_textures = 0;
		return false;
	}

	// dictionary version
	theInputFile.Read((char*) &version, sizeof( int ));

	// Whether or not all textures withing are valid
	if( version >= 7 )
	{
		theInputFile.Read((char*) &all_valid, sizeof( bool ));
	}

	theInputFile.Read((char*) &m_num_textures, sizeof( int ));
	m_textures = new NxTexture[m_num_textures];
	for( i = 0; i < m_num_textures; i++ )
	{		
		NxTexture* texture;
		int j;
		int width, height;
		
		texture = &m_textures[i];
			
		if( version >= 2 )
		{
			int length;

			theInputFile.Read((char *) &length, sizeof( int ));
			if( length > 0 )
			{
				theInputFile.Read((char *) texture->m_Name, length );
				texture->m_Name[length] = '\0';
			}
		}

		theInputFile.Read((char *) &texture->m_PlatFlags, sizeof( int ));		
		theInputFile.Read((char*) &texture->m_Checksum, sizeof( unsigned long ));
				
		theInputFile.Read((char *) &texture->m_Width[0], sizeof( int ));
		theInputFile.Read((char *) &texture->m_Height[0], sizeof( int ));
		
		// Read in pixel/clut storage modes
		theInputFile.Read((char *) &texture->m_Flags, sizeof( int ));		
		theInputFile.Read((char *) &texture->m_PixelFormat, sizeof( int ));
		theInputFile.Read((char *) &texture->m_PaletteFormat, sizeof( int ));			
		theInputFile.Read((char *) &texture->m_MipLevels, sizeof( int ));

		// Check for a checksum that indicates an invalid texture
		if( texture->m_Checksum == vINVALID_CHECKSUM )
		{
			continue;
		}

		if( texture->IsPaletted())
		{				
			int bytes_per_entry;

			switch( texture->m_PixelFormat )
			{
				case NxTexture::v8_BIT:
					texture->m_NumPaletteEntries = 256;					
					break;
				case NxTexture::v4_BIT:
					texture->m_NumPaletteEntries = 16;
					break;
				default:
					return false;
			}

			switch( texture->m_PaletteFormat )
			{
				case NxTexture::v32_BIT:
					texture->m_PaletteBpp = 32;
					bytes_per_entry = 4;
					break;
				case NxTexture::v24_BIT:
					texture->m_PaletteBpp = 24;
					bytes_per_entry = 3;
					break;
				case NxTexture::v16_BIT:
					texture->m_PaletteBpp = 16;
					bytes_per_entry = 2;
					break;
				default:
					return false;
			}
			
			texture->m_TotalPaletteDataSize = texture->m_NumPaletteEntries * bytes_per_entry;
			texture->m_PaletteData = new char[ texture->m_NumPaletteEntries * bytes_per_entry ];
			theInputFile.Read((char *) texture->m_PaletteData, texture->m_NumPaletteEntries * bytes_per_entry );
		}

		switch( texture->m_PixelFormat )
		{
			case NxTexture::v32_BIT:
				texture->m_Bpp = 32;
				break;
			case NxTexture::v24_BIT:
				texture->m_Bpp = 24;
				break;
			case NxTexture::v16_BIT:
				texture->m_Bpp = 16;
				break;
			case NxTexture::v8_BIT:
				texture->m_Bpp = 8;
				break;
			case NxTexture::v4_BIT:
				texture->m_Bpp = 4;
				break;
			case NxTexture::v8_BIT_GRAY:
				texture->m_Bpp = 8;
				break;
			case NxTexture::v4_BIT_GRAY:
				texture->m_Bpp = 4;
				break;
			default:
				return false;
		}

		width = texture->m_Width[0];
		height = texture->m_Height[0];
		texture->m_TotalTexelDataSize = 0;
		for( j = 0; j <= texture->m_MipLevels; j++ )
		{
			if ( version >= 0x0006 )
			{
				int length;
				unsigned int time[2];

				theInputFile.Read((char *) &length, sizeof( int ));
				if( length > 0 )
				{
					char name_buff[256];	// we don't really use these right now, so we'll just skip them
					theInputFile.Read((char *) name_buff, length );
					name_buff[length] = '\0';
				}

				theInputFile.Read((char *) time, 2 * sizeof( unsigned int ));
			}

			if ( version >= 0x0005 )
			{
				theInputFile.Read((char*) &texture->m_Width[j], sizeof(int));
				theInputFile.Read((char*) &texture->m_Height[j], sizeof(int));

				width = texture->m_Width[j];
				height = texture->m_Height[j];
			}

			int texel_data_size, bytes_per_row;

			bytes_per_row = (( width * texture->m_Bpp ) + 7 ) >> 3;
			texel_data_size = height * bytes_per_row;
			texture->m_TexelDataSize[j] = texel_data_size;
			texture->m_TotalTexelDataSize += texel_data_size;
			texture->m_TexelData[j] = new char[ texel_data_size ];
			// Read in the texel data
			theInputFile.Read((char *) texture->m_TexelData[j], texel_data_size );
			
			if ( version < 0x0005 )
			{
				// < v5 computes width/height (not in file)
				texture->m_Width[j] = width;
				texture->m_Height[j] = height;
			}

			// In versions later than 5, we write the width and height of the
			// mip directly into the file  (aml)
			if ( version < 0x0005 )
			{
				width >>= 1;
				height >>= 1;			

				if( version <= 0x0003 )
				{
					if( width < 1 )
					{
						width = 1;
					}
					if( height < 1 )
					{
						height = 1;
					}
				}
				else
				{
					if( width < NxTexture::vMIN_MIP_WIDTH )
					{
						width = NxTexture::vMIN_MIP_WIDTH;
					}
					if( height < NxTexture::vMIN_MIP_HEIGHT )
					{
						height = NxTexture::vMIN_MIP_HEIGHT;
					}
				}
			}	// end < v5
		}		
	}	

	LimitMipMaps();
	return true;
}

NxTexture*	SceneConverter::GetTextureByChecksum( unsigned long checksum )
{
	int i;

	for( i = 0; i < m_num_textures; i++ )
	{
		if( m_textures[i].m_Checksum == checksum )
		{
			return &m_textures[i];
		}
	}

	return NULL;
}

// This function is meant to limit a texture's mipmaps to (at the smallest) 8x8. At the present
// this is sufficient for NGC and is the minimum for PS2, while the Xbox can and will use
// all the way down to 1x1
void	SceneConverter::LimitMipMaps( void )
{
	int i, j;

	for( i = 0; i < m_num_textures; i++ )
	{
		for( j = 0; j <= m_textures[i].m_MipLevels; j++ )
		{
			if( m_textures[i].m_Width[j] <= 8 )
			{
				m_textures[i].m_MipLevels = j;
				break;
			}
			else if( m_textures[i].m_Height[j] <= 8 )
			{
				m_textures[i].m_MipLevels = j;
				break;
			}
		}		
	}
}

#define MAX_FACE_INDICIES 10000

//static unsigned short	s_face_index_buffer[MAX_FACE_INDICIES];
static unsigned short	s_seq_face_index_buffer[MAX_FACE_INDICIES] = { 0xFFFF }; // Set to uninitialized
//static int				s_num_face_indicies;

static const int		s_max_face_per_leaf = 20;		// maximum number faces per leaf
static const int		s_max_tree_levels = 8;			// maximum number of levels in a tree

// This function creates the collision BSP trees.
bool	SceneConverter::GenerateCollisionBSP( void )
{
	if (!m_scene)
	{
		return false;
	}

	// reset the face index buffer so 
	// that it gets rebuilt in batch mode...
	s_seq_face_index_buffer[0] = 0xffff;

	m_scene->m_NumCollBSPTreeNodes = 0;
	for( int i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;
		
		object = &m_scene->m_Objects[i];
		object->InitBSPTree();
		m_scene->m_NumCollBSPTreeNodes += NxCollBSPNode::CountBSPNodes(object->mp_BSPRoot);
	}

	return true;
}

// This function removes duplicate verts for the collision data.
bool	SceneConverter::CollapseCollsionVertices( bool round_verts )
{
	if (!m_scene)
	{
		return false;
	}

	// Make temp palette out for now
	NxFaceInfoPalette face_info_palette;

	for( int i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;
		
		object = &m_scene->m_Objects[i];

		// First find Non-Collidable faces
		for( int j = 0; j < object->m_NumFaces; j++ )
		{
			if ((object->m_Faces[j].m_FaceFlags & NxFace::mFD_NON_COLLIDABLE) && 
				!(object->m_Faces[j].m_FaceFlags & (NxFace::mFD_TRIGGER | NxFace::mFD_VERT | NxFace::mFD_EXPORT_COLLISION)))
			{
				object->m_Faces[j].m_NonCollidable = true;
				object->m_Verts[ j * 3 ].m_NonCollidable = true;
				object->m_Verts[ (j * 3) + 1 ].m_NonCollidable = true;
				object->m_Verts[ (j * 3) + 2 ].m_NonCollidable = true;
			} else {
				NxMaterial *material = m_scene->GetMaterial( object->m_Faces[j].m_MatChecksum );
				unsigned short terrain = 0;
				if( material )
				{
					terrain = (short) material->m_Terrain;
				}

				face_info_palette.AddFaceInfo((unsigned short) object->m_Faces[j].m_FaceFlags, terrain);
			}
		}

		assert(object->mp_CollVerts == NULL);
		assert(object->mp_ToCollVertIdxs == NULL);

		object->mp_CollVerts = new NxVertex[object->m_NumVerts];		// Reserve maximum number of verts
		object->mp_ToCollVertIdxs = new int[object->m_NumVerts];

		bool init_bounding_box = round_verts;
		int new_count = 0;
		for (int idx = 0; idx < object->m_NumVerts; idx++)
		{
			const NxVertex & orig_vertex = object->m_Verts[idx];

			// First check if we even need the vert
			if (orig_vertex.m_NonCollidable)
			{
				object->mp_ToCollVertIdxs[idx] = -1;
				continue;
			}

			int nidx;
			bool found = false;
			for (nidx = 0; nidx < new_count; nidx++)
			{
				if (orig_vertex.PosAndColorSame(object->mp_CollVerts[nidx]))
				{
					found = true;
					break;
				}
			}

			if (found)
			{
				object->mp_ToCollVertIdxs[idx] = nidx;
			} else {
				object->mp_CollVerts[new_count] = orig_vertex;

				// Do rounding, if requested
				if (round_verts)
				{
					for (int axis = 0; axis < 3; axis++)
					{
						int fixed_value = (int) floor((object->mp_CollVerts[new_count].m_Pos[axis] * COLLISION_SUB_INCH_PRECISION) + 0.5f);
						object->mp_CollVerts[new_count].m_Pos[axis] = ((float) fixed_value) * COLLISION_RECIPROCAL_SUB_INCH_PRECISION;
					}

					// Also update collision bounding box
					if (init_bounding_box)
					{
						object->m_CollisionBoundingBox.Set(object->mp_CollVerts[new_count].m_Pos,
														   object->mp_CollVerts[new_count].m_Pos);
						init_bounding_box = false;
					}
					else
					{
						object->m_CollisionBoundingBox.AddPoint(object->mp_CollVerts[new_count].m_Pos);
					}
				}
				object->mp_ToCollVertIdxs[idx] = new_count++;
			}
		}

		object->m_NumCollVerts = new_count;

		// Now check the faces
		object->mp_CollFaces = new NxFace[object->m_NumFaces];		// Reserve maximum number of faces

		object->m_NumCollFaces = 0;
		for (int fidx = 0; fidx < object->m_NumFaces; fidx++)
		{
			if (!object->m_Faces[fidx].m_NonCollidable)
			{
				object->mp_CollFaces[object->m_NumCollFaces++] = object->m_Faces[fidx];
			}
		}

		//assert(object->m_NumCollFaces);
	}

	//printf("Face Info palette size of %d.\n", face_info_palette.m_num_entries);

	return true;
}

NxScene::NxScene( void ) : m_MaterialLookup(12)
{
	m_NumMaterials = 0;	
	m_NumCollBSPTreeNodes = 0;
	m_Objects = NULL;
}

NxScene::~NxScene( void )
{
	Lst::Search< NxMaterial > sh;
	NxMaterial* material, *next;

	for( material = sh.FirstItem( m_Materials ); material;
			material = next )
	{
		next = sh.NextItem();
		delete material;
	}

	if (m_Objects)
		delete [] m_Objects;
}

NxModel::NxModel( void )
{
	nBones = 0;
	bones = NULL;
}

NxModel::~NxModel()
{
	if (bones)
		delete [] bones;
}

VCWibbleSequence::VCWibbleSequence( void )
: Lst::Node< VCWibbleSequence > ( this )
{
	m_NumFrames = 0;
	m_Index = 0;
	m_Offset = 0;
	m_WibbleFrames = NULL;
	m_Phase = 0;
}

void			NxScene::RemoveUnusedMaterials( void )
{
	Lst::Search< NxMaterial > sh;
	NxMaterial* material, *next;

	for( material = sh.FirstItem( m_Materials ); material;
			material = next )
	{
		next = sh.NextItem();
		if( !material->m_Used )
		{
			delete material;
		}
	}
}

NxMaterial*		NxScene::GetMaterial( unsigned long checksum )
{
#if 1
	NxMaterial* pMaterial = m_MaterialLookup.GetItem( checksum );
	return pMaterial;
#else
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;

	for( material = sh.FirstItem( m_Materials ); material;
			material = sh.NextItem())
	{
		if( material->m_Checksum == checksum )
		{
			return material;
		}
	}
#endif

	return NULL;
}

NxMaterial*		NxScene::CopyMaterial( unsigned long old_checksum, unsigned long new_checksum, int pass_flags )
{
	NxMaterial* src_mat, *new_mat;
	int i, cur_pass, num_passes;
	VCWibbleSequence* seq, *new_seq;
	Lst::Search< VCWibbleSequence > sh;

	src_mat = GetMaterial( old_checksum );
	assert( src_mat );

	num_passes = 0;
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		if( pass_flags & ( NxMaterial::mGROUP_PASS_1 << i ))
		{
			num_passes++;
		}
	}

	assert( num_passes > 0 );
	assert( num_passes < src_mat->m_NumPasses );

	new_mat = new NxMaterial;
	new_mat->m_NumPasses = num_passes;
	new_mat->m_Checksum = new_checksum;
	new_mat->m_materialName = src_mat->m_materialName;
	new_mat->m_AlphaCutoff = src_mat->m_AlphaCutoff;
	new_mat->m_CutoffFunc = src_mat->m_CutoffFunc;
	new_mat->m_DrawOrder = src_mat->m_DrawOrder;
	new_mat->m_BasePass = src_mat->m_BasePass;
	new_mat->m_Invisible = src_mat->m_Invisible;
	new_mat->m_Sorted = src_mat->m_Sorted;
	new_mat->m_Terrain = src_mat->m_Terrain;

	// new aml
	new_mat->m_GroupFlags = src_mat->m_GroupFlags;
	new_mat->m_ShouldExpand = src_mat->m_ShouldExpand;
	new_mat->m_GroupId = src_mat->m_GroupId;
	new_mat->m_water = src_mat->m_water;
	new_mat->m_grassify = src_mat->m_grassify;
	new_mat->m_grassHeight = src_mat->m_grassHeight;
	new_mat->m_grassLayers = src_mat->m_grassLayers;
	new_mat->m_Pass = src_mat->m_Pass;
	
	new_mat->m_TwoSided = src_mat->m_TwoSided;
	new_mat->m_OneSided = src_mat->m_OneSided;
	new_mat->m_SpecularPower = src_mat->m_SpecularPower;
	new_mat->m_SpecularColor[0] = src_mat->m_SpecularColor[0];
	new_mat->m_SpecularColor[1] = src_mat->m_SpecularColor[1];
	new_mat->m_SpecularColor[2] = src_mat->m_SpecularColor[2];

	///////////
	new_mat->SetPri( src_mat->GetPri());
	new_mat->m_Used = true;

	for( seq = sh.FirstItem( src_mat->m_WibbleSequences ); seq; seq = sh.NextItem())
	{
		new_seq = new VCWibbleSequence;
		new_seq->m_Index = seq->m_Index;
		new_seq->m_Offset = seq->m_Offset;
		new_seq->m_NumFrames = seq->m_NumFrames;
		new_seq->m_Phase = seq->m_Phase;
		new_seq->m_WibbleFrames = new WibbleKeyframe[new_seq->m_NumFrames];
		for( i = 0; i < new_seq->m_NumFrames; i++ )
		{
			new_seq->m_WibbleFrames[i] = seq->m_WibbleFrames[i];
		}

		new_mat->m_WibbleSequences.AddToTail( new_seq );
	}

	cur_pass = 0;
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		if( pass_flags & ( NxMaterial::mGROUP_PASS_1 << i ))
		{
			new_mat->m_Passes[cur_pass++] = src_mat->m_Passes[i];			
		}
	}
	
	m_Materials.AddNode( new_mat );
	AddMaterialToLookup( new_mat );

	m_NumMaterials++;

	return new_mat;
}

int				NxScene::GetNumVisibleMaterials( void )
{
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;
	int count;

	count = 0;
	for( material = sh.FirstItem( m_Materials ); material;
			material = sh.NextItem())
	{
		if( material->m_Invisible )
		{
			continue;
		}

		count++;
	}

	return count;
}

bool			NxScene::ContainsSortedUntexturedMaterials( int platform )
{
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;
	
	for( material = sh.FirstItem( m_Materials ); material;
			material = sh.NextItem())
	{
		if( material->m_Invisible )
		{
			continue;
		}

		// Ignore animated bitmaps in this check
		if( ( material->m_Sorted ) &&
			(( material->m_Passes[0].m_Flags & NxMaterialPass::ANIMATED_TEXTURE ) == 0 ) &&
			( material->m_Passes[0].GetTextureChecksum( platform ) == 0 ))
		{
			return true;
		}
	}

	return false;
}

NxObject*		NxScene::FindObject( unsigned long checksum )
{
	NxObject* object;

	for(int i = 0; i < m_NumObjects; i++)
	{
		object = &(m_Objects[i]);
		if( object->m_Checksum == checksum )
		{
			return object;
		}
	}

	return NULL;
}

NxAnimatedTextureKeyframe::NxAnimatedTextureKeyframe( void )
{
	int i;

	m_Time = 0;
	for( i = 0; i < Utils::vNUM_PLATFORMS; i++ )
	{
		m_TexChecksum[i] = 0;
	}
}

NxAnimatedTexture::NxAnimatedTexture( void )
{
	m_NumKeyframes = 0;
	m_Period = 0;
	m_Iterations = 0;
	m_Keyframes = NULL;
	m_Phase = 0;
}

NxAnimatedTexture::~NxAnimatedTexture( void )
{
	if( m_Keyframes )
	{
		delete [] m_Keyframes;
	}
}

NxMaterialPass::NxMaterialPass( void )
{
	int i;

	for( i = 0; i < Utils::vNUM_PLATFORMS; i++ )
	{
		m_tex_checksum[i] = 0;
	}

	m_BlendMode = vBLEND_MODE_DIFFUSE;
	m_FixedAlpha = 0;
	m_MappingMode = vMAPPING_EXPLICIT;		// Explicit or procedural (eg. environment-mapping)	
	m_MinFilteringMode = vFILTERING_LINEAR;	// Point/Bi-linear
	m_MagFilteringMode = vFILTERING_LINEAR;	// Point/Bi-linear/Tri-linear
	m_UVWibbleEnabled = false;		
	m_UVel = 0;
	m_VVel = 0;	
	m_UAmplitude = 0;
	m_VAmplitude = 0;
	m_UPhase = 0;
	m_VPhase = 0;
	m_UFrequency = 0;
	m_VFrequency = 0;
	m_EnvTileU = 3.0f;
	m_EnvTileV = 3.0f;
	m_Color[0] = 0.5f;
	m_Color[1] = 0.5f;
	m_Color[2] = 0.5f;
	m_Ambient[0] = 0.5f;
	m_Ambient[1] = 0.5f;
	m_Ambient[2] = 0.5f;
	m_Diffuse[0] = 0.5f;
	m_Diffuse[1] = 0.5f;
	m_Diffuse[2] = 0.5f;
	m_Specular[0] = 0.5f;
	m_Specular[1] = 0.5f;
	m_Specular[2] = 0.5f;
	m_MipMapK = -8.0f;		
	m_Flags = 0;
}

void			NxMaterialPass::SetTextureChecksum( int platform, unsigned long tex_checksum )
{
	m_tex_checksum[platform] = tex_checksum;
}

unsigned long	NxMaterialPass::GetTextureChecksum( int platform )
{
	if( gTextureOverride )
	{
		platform = gTextureOverrideMapping[platform];
	}
	return m_tex_checksum[platform];
}

unsigned long	NxMaterialPass::GetTextureChecksum( int index, int platform )
{
	if( m_Flags & ANIMATED_TEXTURE )
	{
		Dbg_Assert( index < m_AnimatedTexture.m_NumKeyframes );
		if( gTextureOverride )
		{
			platform = gTextureOverrideMapping[platform];
		}
		
		return m_AnimatedTexture.m_Keyframes[index].m_TexChecksum[platform];		
	}
	else
	{
		return GetTextureChecksum( platform );
	}
}

int		NxMaterialPass::GetNumTextures( int platform )
{
	if( m_Flags & ANIMATED_TEXTURE )
	{
		return m_AnimatedTexture.m_NumKeyframes;
	}
	else
	{
		if( GetTextureChecksum( platform ) != 0 )
		{
			return 1;
		}
	}

	return 0;
}

void	NxMaterialPass::operator = ( NxMaterialPass& pass )
{
	int i;

	// Most of the pass can be copied using memcpy. But the animated texture frames
	// need to be dynamically-allocated
	memcpy( this, &pass, sizeof( NxMaterialPass ));
	if( pass.m_Flags & ANIMATED_TEXTURE )
	{
		m_AnimatedTexture.m_Keyframes = new NxAnimatedTextureKeyframe[pass.m_AnimatedTexture.m_NumKeyframes];
		for( i = 0; i < pass.m_AnimatedTexture.m_NumKeyframes; i++ )
		{
			NxAnimatedTextureKeyframe* dst_key, *src_key;

			src_key = &pass.m_AnimatedTexture.m_Keyframes[i];
			dst_key = &m_AnimatedTexture.m_Keyframes[i];

			*dst_key = *src_key;
		}
	}
}

NxMaterial::NxMaterial( void )
: Lst::Node< NxMaterial > ( this )
{
	int i;

	m_Checksum = 0;
	m_GroupFlags = 0;
	m_Invisible = false;
	m_ShouldExpand = false;
	m_NumPasses = 1;
	m_Terrain = 0;
	m_AlphaCutoff = 1;
	m_DrawOrder = 0;
	m_BasePass = 0;
	m_Sorted = false;
	m_GroupId = -1;
	m_CutoffFunc = 0;
	m_Pass = 0;
	m_TwoSided = false;	
	m_OneSided = false;
	m_Used = false;
	m_water = false;
	m_grassify = false;
	m_grassHeight = 0;
	m_grassLayers = 0;
	m_SpecularPower    = 0.0f;
	m_SpecularColor[0] = 0.0f;
	m_SpecularColor[1] = 0.0f;
	m_SpecularColor[2] = 0.0f;

	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		m_Passes[i].m_OrigPass = i;
	}
}

int		NxMaterial::GetWibbleSequence( int base_seq, int offset )
{
	Lst::Search< VCWibbleSequence > sh;
	VCWibbleSequence* seq, *new_seq;
	int i, new_index;
	bool found_sequence;

	new_index = 1;	// indices are 1-based
	found_sequence = false;
	for( seq = sh.FirstItem( m_WibbleSequences ); seq; seq = sh.NextItem())
	{
		if( seq->m_Index == base_seq )
		{
			found_sequence = true;
			if( seq->m_Offset == offset )
			{
				return new_index;
			}
		}
		new_index++;
	}

	// If there isn't even a matching sequence that means that this vertex should not wibble.
	// Return a value of zero indicating that it shouldn't wibble.  Perhaps the index was accidentally 
	// set in Max. Or perhaps a new material was applied to the object which did not have wibble 
	// sequences defined. Or perhaps the Max wibble data channels are corrupt

	if( found_sequence == false )
	{
		return 0;
	}

	// If we've gotten here, that means a suitable sequence does not yet exist. Create it now
	new_seq = new VCWibbleSequence;
	new_seq->m_Index = base_seq;
	new_seq->m_Offset = offset;
	// Now copy the animation frames from the original, but compute the phase based on the keyframe offset
	for( seq = sh.FirstItem( m_WibbleSequences ); seq; seq = sh.NextItem())
	{
		if( seq->m_Index == base_seq )
		{
			new_seq->m_NumFrames = seq->m_NumFrames;
			new_seq->m_WibbleFrames = new WibbleKeyframe[new_seq->m_NumFrames];
			for( i = 0; i < new_seq->m_NumFrames; i++ )
			{
				new_seq->m_WibbleFrames[i] = seq->m_WibbleFrames[i];				
			}

			// GJ:  For some reason, it's possible to get an offset
			// this is outside the valid range of keyframes, perhaps
			// due to MAX corruption?  If this happens, make sure
			// it's some predictable number so that my diff won't fail.
			if ( offset >= seq->m_NumFrames )
			{
				offset %= seq->m_NumFrames;
			}

			new_seq->m_Phase = seq->m_WibbleFrames[offset].m_Time;

			break;
		}
	}

	m_WibbleSequences.AddToTail( new_seq );

	return new_index;
}

NxFace::NxFace( void )
{
	m_Invisible = false;
	m_NonCollidable = false;
	m_CASFaceFlags = 0;
	m_PassFlags = 0;	
}
					
NxFaceInfoPalette::NxFaceInfoPalette()
{
	ClearPalette();
}

void NxFaceInfoPalette::ClearPalette()
{
	m_num_entries = 0;
}

int NxFaceInfoPalette::AddFaceInfo(unsigned short flags, unsigned short terrain_type)
{
	for (int i = 0; i < m_num_entries; i++)
	{
		if ((flags == m_face_info[i].m_flags) && (terrain_type == m_face_info[i].m_terrain_type))
		{
			return i;
		}
	}

	// Put this assert back when we use the palette
	//assert(m_num_entries < MAX_PALETTE_ENTRIES);

	if (m_num_entries == MAX_PALETTE_ENTRIES)
	{
		return -1;
	}

	m_face_info[m_num_entries].m_flags = flags;
	m_face_info[m_num_entries].m_terrain_type = terrain_type;

	return m_num_entries++;
}

NxVertex::NxVertex( void )
{
	int i;

	m_Pos[0] = 0;
	m_Pos[1] = 0;
	m_Pos[2] = 0;
	m_Normal[0] = 0;
	m_Normal[1] = 0;
	m_Normal[2] = 0;
	
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		m_TexCoord[i][0] = 0;
		m_TexCoord[i][1] = 0;		
	}
	m_Color[0] = 0.5f;
	m_Color[1] = 0.5f;
	m_Color[2] = 0.5f;
	m_Color[3] = 0.5f;
	m_WibbleIndex = 0;
	m_WibbleOffset = 0;
	m_Pass = 0;	
	m_Invisible = false;
	m_NonCollidable = false;
	m_NumWeights = 0;

	for( i = 0; i < vMAX_WEIGHTS_PER_VERTEX; i++ )
	{
		m_Weight[i] = 0;
		m_WeightedIndex[i] = 0;

		m_MeshScalingWeight[i] = 0;
		m_MeshScalingWeightedIndex[i] = -1;
	}
}

bool	NxVertex::operator==( NxVertex& vertex )
{
	int j,k,m;

	// Check position
	if (m_Pos[0] != vertex.m_Pos[0] ||
		m_Pos[1] != vertex.m_Pos[1] ||
		m_Pos[2] != vertex.m_Pos[2])
		return false;

	// Check UVs
	for(j=0;j<vMAX_MATERIAL_PASSES;j++)
	{
		if (m_TexCoord[j][0] != vertex.m_TexCoord[j][0] ||
			m_TexCoord[j][1] != vertex.m_TexCoord[j][1])
			return false;
	}
				
	// Check Color
	if (m_Color[0] != vertex.m_Color[0] ||
		m_Color[1] != vertex.m_Color[1] ||
		m_Color[2] != vertex.m_Color[2] ||
		m_Color[3] != vertex.m_Color[3])
		return false;

	if (m_WibbleIndex  != vertex.m_WibbleIndex  ||
		m_WibbleOffset != vertex.m_WibbleOffset ||
		m_Pass         != vertex.m_Pass         ||
		m_NumWeights   != vertex.m_NumWeights)
		return false;

	// Check WeightedIndicies
	for(k=0;k<m_NumWeights;k++)
	{
		if (m_WeightedIndex[k] != vertex.m_WeightedIndex[k])
			return false;
	}

	// Check Weights
	for(m=0;m<m_NumWeights;m++)
	{
		if (m_Weight[m]!=vertex.m_Weight[m])
			return false;
	}

	return true;

	
/*  aml:  only checking the first value of most arrays

	int i;

	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		if( m_TexCoord[i] != vertex.m_TexCoord[i] )
		{
			return false;
		}
	}

	if( ( m_Pos == vertex.m_Pos ) &&		
		( m_Normal == vertex.m_Normal ) &&		
		( m_Color == vertex.m_Color ) &&
		( m_WibbleIndex == vertex.m_WibbleIndex ) &&
		( m_WibbleOffset == vertex.m_WibbleOffset ) &&
		( m_Pass == vertex.m_Pass ))
	{
		return true;
	}
	return false;
*/
}





bool NxVertex::PosAndColorSame( const NxVertex& vertex ) const
{
	// Check position
	if (m_Pos[0] != vertex.m_Pos[0] ||
		m_Pos[1] != vertex.m_Pos[1] ||
		m_Pos[2] != vertex.m_Pos[2])
		return false;
				
	// Check Color
	if (m_Color[0] != vertex.m_Color[0] ||
		m_Color[1] != vertex.m_Color[1] ||
		m_Color[2] != vertex.m_Color[2] ||
		m_Color[3] != vertex.m_Color[3])
		return false;

	return true;
}

void NxVertex::SortVertexWeights( void )
{
	// Will sort weights (and corresponding bone indices too) into order.
	for( int i = 0; i < vMAX_WEIGHTS_PER_VERTEX - 1; ++i )
	{
		for( int w = i + 1; w < vMAX_WEIGHTS_PER_VERTEX; ++w )
		{
			// Sort high values up the list.
			if( m_Weight[i] < m_Weight[w] )
			{
				float t				= m_Weight[i];
				m_Weight[i]			= m_Weight[w];
				m_Weight[w]			= t;

				unsigned short st	= m_WeightedIndex[i];
				m_WeightedIndex[i]	= m_WeightedIndex[w];
				m_WeightedIndex[w]	= st;
			}
		}
	}
}

void NxVertex::NormalizeVertexWeights( int numWeights )
{	
	int i;

	float runningTotal = 0.0f;
	for ( i = 0; i < numWeights; i++ )
	{
		runningTotal += m_Weight[i];
	}

	for ( i = 0; i < numWeights; i++ )
	{
		m_Weight[i] /= runningTotal;
	}
}

void NxBoundingBox::Set(const float min[3], const float max[3])
{
	m_Min[0] = min[0];
	m_Min[1] = min[1];
	m_Min[2] = min[2];

	m_Max[0] = max[0];
	m_Max[1] = max[1];
	m_Max[2] = max[2];
}

void NxBoundingBox::AddPoint(const float point[3])
{
	// Adjust min/max points
	if (point[0] < m_Min[0])
		m_Min[0] = point[0];
	if (point[1] < m_Min[1])
		m_Min[1] = point[1];
	if (point[2] < m_Min[2])
		m_Min[2] = point[2];

	if (point[0] > m_Max[0])
		m_Max[0] = point[0];
	if (point[1] > m_Max[1])
		m_Max[1] = point[1];
	if (point[2] > m_Max[2])
		m_Max[2] = point[2];
}

NxLODLevel::NxLODLevel( void )
{
	m_Distance = 0.0f;
	m_ObjectCRC = 0;
}

const int	NxObject::sFaceLargeSize = 10;
const int	NxObject::sFaceSmallSize = 8;
const int	NxObject::sVertexLargeSize = 12;
const int	NxObject::sVertexSmallSize = 6;

NxObject::NxObject( void )
{
	m_Checksum = 0;
	m_SkeletonChecksum = 0;
	m_NumVerts = 0;
	m_Verts = NULL;
	m_OriginalVerts = NULL;
	m_NumFaces = 0;
	m_Faces = NULL;
	m_BoundingBox.m_Max[0] = 0;
	m_BoundingBox.m_Max[1] = 0;
	m_BoundingBox.m_Max[2] = 0;
	m_BoundingBox.m_Min[0] = 0;
	m_BoundingBox.m_Min[1] = 0;
	m_BoundingBox.m_Min[2] = 0;
	m_CollisionBoundingBox = m_BoundingBox;
	m_BoundingSphere.m_Center[0] = 0;
	m_BoundingSphere.m_Center[1] = 0;
	m_BoundingSphere.m_Center[2] = 0;
	m_BoundingSphere.m_Radius = 0;
	m_Flags = 0;
	m_NumUVSets = 0;
	m_NumCASData = 0;
	mp_BSPRoot = NULL;
	mp_CollVerts = NULL;
	mp_ToCollVertIdxs = NULL;
	mp_CollFaces = NULL;
	m_NumCollVerts = 0;
	m_NumCollFaces = 0;
	m_CollUseFixedVerts = 0;
	m_NumCASData = 0;
	mp_CASData = NULL;
	m_LODLevels = 0;
	m_LODFlags = 0;
	m_LODMaster.m_NumLODLevels = 0;
	m_LODMaster.m_LODLevels = NULL;
	m_LODinfo = NULL;
	m_ParentCRC = 0;
	m_NumChildren = 0;
	m_ChildCRCs = NULL;
}

NxObject::~NxObject( void )
{
	int i;

	if( m_Verts )
	{
		delete [] m_Verts;
	}

	if( m_OriginalVerts )
	{
		delete [] m_OriginalVerts;
	}

	if( m_Faces )
	{
		delete [] m_Faces;
	}

	if ( mp_BSPRoot )
	{
		delete mp_BSPRoot;
	}

	if (mp_CollVerts)
	{
		delete [] mp_CollVerts;
	}

	if (mp_ToCollVertIdxs)
	{
		delete [] mp_ToCollVertIdxs;
	}

	if (mp_CollFaces)
	{
		delete [] mp_CollFaces;
	}

	for( i = 0; i < m_MeshList.m_NumMeshes[0]; i++ )
	{
		delete m_MeshList.m_Meshes[0][i];
	}

	if ( mp_CASData )
	{
		delete[] mp_CASData;
		mp_CASData = NULL;
	}

	if ( (m_LODFlags & mMASTER) && m_LODMaster.m_LODLevels )
	{
		delete [] m_LODMaster.m_LODLevels;
		m_LODMaster.m_LODLevels = NULL;
		m_LODMaster.m_NumLODLevels = 0;
		m_LODFlags = 0;
	}

	if ( m_ChildCRCs )
	{
		delete [] m_ChildCRCs;
		m_ChildCRCs = NULL;
		m_ParentCRC = 0;
		m_NumChildren = 0;
	}

	if ( m_LODinfo )
	{
		delete [] m_LODinfo;
		m_LODinfo = NULL;
	}

	m_MeshList.Reset();
}

// Assignment operator  (REMEMBER TO UPDATE ME ON ADDITION OF NEW FIELDS !)
NxObject&	NxObject::operator =(NxObject& obj)
{
	m_Checksum         = obj.m_Checksum;
	m_SkeletonChecksum = obj.m_SkeletonChecksum;
	m_NumFaces         = obj.m_NumFaces;

	if (obj.m_Faces)
	{
		m_Faces = new NxFace[m_NumFaces];
		memcpy(m_Faces, obj.m_Faces, m_NumFaces * sizeof(NxFace));
	}

	m_NumVerts         = obj.m_NumVerts;

	if (obj.m_Verts)
	{
		m_Verts = new NxVertex[m_NumVerts];
		memcpy(m_Verts, obj.m_Verts, m_NumVerts * sizeof(NxVertex));
	}

	if (obj.m_OriginalVerts)
	{
		m_OriginalVerts = new NxVertex[m_NumVerts];
		memcpy(m_OriginalVerts, obj.m_OriginalVerts, m_NumVerts * sizeof(NxVertex));
	}

	m_BoundingBox          = obj.m_BoundingBox;
	m_CollisionBoundingBox = obj.m_CollisionBoundingBox;
	m_BoundingSphere       = obj.m_BoundingSphere;
	m_Flags                = obj.m_Flags;
	m_MeshList             = obj.m_MeshList;
	m_NumUVSets            = obj.m_NumUVSets;
    m_CASRemoveFlags       = obj.m_CASRemoveFlags;

	// Collision data must manually be copied for now  (assignment intented pre collision processing)
	mp_BSPRoot             = NULL;
	mp_CollVerts           = NULL;
	mp_ToCollVertIdxs      = NULL;
	mp_CollFaces           = NULL;
	m_NumCollVerts         = 0;
	m_NumCollFaces         = 0;

	m_CollUseFixedVerts    = obj.m_CollUseFixedVerts;
	m_KBias                = obj.m_KBias;
	m_ParentMatrixIndex    = obj.m_ParentMatrixIndex;
	m_LODFlags             = obj.m_LODFlags;

	if (m_LODFlags & mMASTER)
	{
		m_LODMaster.m_NumLODLevels = obj.m_LODMaster.m_NumLODLevels;
		m_LODMaster.m_LODLevels    = new NxLODLevel[m_LODMaster.m_NumLODLevels];

		memcpy(m_LODMaster.m_LODLevels, obj.m_LODMaster.m_LODLevels, sizeof(NxLODLevel) * m_LODMaster.m_NumLODLevels);
	}
	else if (m_LODFlags & mSLAVE)
	{
		m_LODSlave.m_masterCRC      = obj.m_LODSlave.m_masterCRC;
		m_LODSlave.mp_master_object = obj.m_LODSlave.mp_master_object;
	}

	m_ParentCRC   = obj.m_ParentCRC;
	m_NumChildren = obj.m_NumChildren;

	//if (m_NumChildren)
	if (obj.m_ChildCRCs)
	{
		m_ChildCRCs   = new unsigned long[m_NumChildren];
		memcpy(m_ChildCRCs, obj.m_ChildCRCs, sizeof(unsigned long) * m_NumChildren);
	}
	else
		m_ChildCRCs   = NULL;

	m_LODLevels = obj.m_LODLevels;

	//if (m_LODLevels)
	if (obj.m_LODinfo)
	{
		int i;
		m_LODinfo = new NxLODInfo[m_LODLevels];
		
		for(i = 0; i < m_LODLevels; i++)
			m_LODinfo[i] = obj.m_LODinfo[i];
	}
	else
		m_LODinfo = NULL;

	m_BillboardType = obj.m_BillboardType;
	
	m_BillboardOrigin[0] = obj.m_BillboardOrigin[0];
	m_BillboardOrigin[1] = obj.m_BillboardOrigin[1];
	m_BillboardOrigin[2] = obj.m_BillboardOrigin[2];

	m_PivotPos[0] = obj.m_PivotPos[0];
	m_PivotPos[1] = obj.m_PivotPos[1];
	m_PivotPos[2] = obj.m_PivotPos[2];

	m_PivotAxis[0] = obj.m_PivotAxis[0];
	m_PivotAxis[1] = obj.m_PivotAxis[1];
	m_PivotAxis[2] = obj.m_PivotAxis[2];

	mp_CASData = new NxCASData[vMAXCASDATA];

	if (obj.mp_CASData)
		memcpy(mp_CASData, obj.mp_CASData, sizeof(NxCASData) * vMAXCASDATA);

	m_NumCASData = obj.m_NumCASData;

	return *this;
}

bool	NxObject::flag_invisible_vertices_and_faces( NxScene* scene )
{
	int i;
	bool invisible;

	invisible = true;
	for( i = 0; i < m_NumFaces; i++ )
	{
		NxMaterial* material;

		material = scene->GetMaterial( m_Faces[i].m_MatChecksum );
		if( ( material && material->m_Invisible ) ||
			( m_Faces[i].m_FaceFlags & NxFace::mFD_INVISIBLE ))
		{
			m_Faces[i].m_Invisible = true;
			m_Verts[ i * 3 ].m_Invisible = true;
			m_Verts[ (i * 3) + 1 ].m_Invisible = true;
			m_Verts[ (i * 3) + 2 ].m_Invisible = true;
		}
		else
		{
			invisible = false;
		}
	}

	return invisible;
}



void NxObject::OptimizeVertList( NxScene* scene )
{	
	NxVertex* unique_verts;
	int i, j, num_unique_verts;
	int* remapped_verts;
	int num_visible_faces;
	bool invisible;

	if( m_NumVerts == 0 )
	{
		// Nothing to optimize
		return;
	}
	
	invisible = flag_invisible_vertices_and_faces( scene );

	// If the entire object is invisible, just return.  Nothing to do....
	if( invisible )
	{
		m_Flags |= mINVISIBLE;
	}
	
	num_unique_verts = 0;

	unique_verts = new NxVertex[ m_NumVerts ];
	remapped_verts = new int[ m_NumVerts ];

	// Initialize the "instructions" for remapping verts. By default, just remap to same
	// array position
	for( i = 0; i < m_NumVerts; i++ )
	{
		remapped_verts[i] = i;
	}

	for( i = 0; i < m_NumVerts; i++ )
	{
		bool unique;

		unique = true;
		int nUnique = 0;

		// Check if this vertex is already represented in our list of unique verts
		for( j = 0; j < num_unique_verts; j++ )
		{			
			// Skip invisible verts. They won't be included in the renderable mesh
			if( m_Verts[i].m_Invisible )
			{
				continue;
			}
			// If so, just point it to the already-existing vertex
			if( unique_verts[j] == m_Verts[i] )
			{
				remapped_verts[i] = j;			
				unique = false;
				break;
			}			
		}

		if( unique )
		{
			// This is a new vertex. Add it to the list and remap the original to this spot
			unique_verts[num_unique_verts] = m_Verts[i];
			remapped_verts[i] = num_unique_verts;


			num_unique_verts++;
		}
	}

	// Eliminate invisible faces
	num_visible_faces = 0;


	for( i = 0; i < m_NumFaces; i++ )
	{
		if( m_Faces[i].m_Invisible == false )
		{
			num_visible_faces++;
		}
	}
	
	if( num_visible_faces != m_NumFaces )
	{
		NxFace* new_faces;

		new_faces = new NxFace[ num_visible_faces ];

		j = 0;
		for( i = 0; i < m_NumFaces; i++ )
		{			
			if( m_Faces[i].m_Invisible == false )
			{
				memcpy( &new_faces[j], &m_Faces[i], sizeof( NxFace ));

				new_faces[j].m_Index = j++;
			}
		}

		delete [] m_Faces;
		m_NumFaces = num_visible_faces;
		m_Faces = new_faces;


	}

	// Now go through each face and change vertex indices to match remapped vert indices
	for( i = 0; i < m_NumFaces; i++ )
	{
		for( j = 0; j < 3; j++ )
		{
			m_Faces[i].m_Vertex[j] = remapped_verts[m_Faces[i].m_Vertex[j]];
		}
	}
	
	delete [] m_Verts;
	m_NumVerts = num_unique_verts;
	m_Verts = new NxVertex[ m_NumVerts ];

	for( i = 0; i < m_NumVerts; i++ )
	{
		m_Verts[i] = unique_verts[i];
	}

	delete [] unique_verts;
	delete [] remapped_verts;
}

const int				NxCollBSPNode::s_bsp_node_size = 8;	// size of instance on game side
const int				NxCollBSPLeaf::s_bsp_leaf_size = 8;

NxCollBSPNode::NxCollBSPNode()
{
	mp_less_branch = NULL;
	mp_greater_branch = NULL;
	m_less_branch_idx = -1;
	m_greater_branch_idx = -1;
	m_array_offset = 0;
	m_split_axis = 0;
	m_split_point = 0.0f;
}

NxCollBSPNode::~NxCollBSPNode()
{
	if (mp_less_branch)
	{
		delete mp_less_branch;
	}

	if (mp_greater_branch)
	{
		delete mp_greater_branch;
	}
}

NxCollBSPLeaf::NxCollBSPLeaf()
{
	mp_face_idx_array = NULL;
	m_num_faces = 0;
	m_split_axis = 3;
}

NxCollBSPLeaf::~NxCollBSPLeaf()
{
	if (mp_face_idx_array)
	{
		delete[] mp_face_idx_array;
	}
}

bool	NxObject::InitBSPTree()
{
	// Initialize sequence face index list, if not done yet
	if (s_seq_face_index_buffer[0] != 0)
	{
		for (int i = 0; i < MAX_FACE_INDICIES; i++)
		{
			s_seq_face_index_buffer[i] = i;
		}
	}

	assert(m_NumCollFaces < MAX_FACE_INDICIES);

	// Create the tree
	mp_BSPRoot = create_bsp_tree(m_BoundingBox, s_seq_face_index_buffer, m_NumCollFaces);

	return mp_BSPRoot != NULL;
}
bool	NxObject::calc_split_faces(int axis, float axis_distance, unsigned short *p_face_indexes,
										  int num_faces, int & less_faces, int & greater_faces,
										  unsigned short *p_less_face_indexes, unsigned short *p_greater_face_indexes)
{
	less_faces = greater_faces = 0;

	for (int i = 0; i < num_faces; i++)
	{
		bool less = false, greater = false;
		NxFace *p_face = &(mp_CollFaces[p_face_indexes[i]]);

		// Check the face
		for (int j = 0; j < 3; j++)
		{
			int vidx = p_face->m_Vertex[j];
			if (m_Verts[vidx].m_Pos[axis] < axis_distance)
			{
				less = true;
			} else if (m_Verts[vidx].m_Pos[axis] >= axis_distance)
			{
				greater = true;
			}
		}

		assert(less || greater);

		// Increment counts and possibly put in new array
		if (less)
		{
			if (p_less_face_indexes)
			{
				p_less_face_indexes[less_faces] = p_face_indexes[i];
			}
			less_faces++;
		}
		if (greater)
		{
			if (p_greater_face_indexes)
			{
				p_greater_face_indexes[greater_faces] = p_face_indexes[i];
			}
			greater_faces++;
		}
	}

	return true;
}

NxCollBSPLeaf * NxObject::create_bsp_leaf(unsigned short *p_face_indexes, int num_faces)
{
	NxCollBSPLeaf *p_bsp_leaf = new NxCollBSPLeaf;

	p_bsp_leaf->m_split_axis = 3;
	p_bsp_leaf->mp_less_branch = NULL;
	p_bsp_leaf->mp_greater_branch = NULL;

	// Make new array in BottomUp memory
	p_bsp_leaf->m_num_faces = num_faces;
	p_bsp_leaf->mp_face_idx_array = new unsigned short[num_faces];
	for (int i = 0; i < num_faces; i++)
	{
		p_bsp_leaf->mp_face_idx_array[i] = p_face_indexes[i];
	}

	return p_bsp_leaf;
}

NxCollBSPNode * NxObject::create_bsp_tree(const NxBoundingBox & bbox, unsigned short *p_face_indexes, int num_faces, int level)
{
	if ((num_faces <= s_max_face_per_leaf) || (level == s_max_tree_levels)) // Check if this should be a leaf
	{
		return create_bsp_leaf(p_face_indexes, num_faces);
	} else {																// Create Node

		// Find initial splits on the three axis
		int i_mid_split[3];
		float mid_width[3], mid_split[3];
		mid_width[0] = ((bbox.m_Max[0] - bbox.m_Min[0]) * 0.5f);
		mid_width[1] = ((bbox.m_Max[1] - bbox.m_Min[1]) * 0.5f);
		mid_width[2] = ((bbox.m_Max[2] - bbox.m_Min[2]) * 0.5f);
		i_mid_split[0] = (int) (((mid_width[0] + bbox.m_Min[0]) * COLLISION_SUB_INCH_PRECISION) + 0.5f);	// Round to nearest 1/16th
		i_mid_split[1] = (int) (((mid_width[1] + bbox.m_Min[1]) * COLLISION_SUB_INCH_PRECISION) + 0.5f);	// Round to nearest 1/16th
		i_mid_split[2] = (int) (((mid_width[2] + bbox.m_Min[2]) * COLLISION_SUB_INCH_PRECISION) + 0.5f);	// Round to nearest 1/16th
		mid_split[0] = i_mid_split[0] * COLLISION_RECIPROCAL_SUB_INCH_PRECISION;
		mid_split[1] = i_mid_split[1] * COLLISION_RECIPROCAL_SUB_INCH_PRECISION;
		mid_split[2] = i_mid_split[2] * COLLISION_RECIPROCAL_SUB_INCH_PRECISION;

		// Find the weighting of the three potential splits
		int less_faces[3], greater_faces[3];
		calc_split_faces(0, mid_split[0], p_face_indexes, num_faces, less_faces[0], greater_faces[0]);
		calc_split_faces(1, mid_split[1], p_face_indexes, num_faces, less_faces[1], greater_faces[1]);
		calc_split_faces(2, mid_split[2], p_face_indexes, num_faces, less_faces[2], greater_faces[2]);

		// Figure out best split
		int best_axis = -1;
		float best_diff = -1;
		const int duplicate_threshold = (num_faces * 7) / 10;	// tunable
		for (int axis = 0; axis <= 2; axis++)
		{
			float new_diff = (float) abs(less_faces[axis] - greater_faces[axis]);
			int duplicates = less_faces[axis] + greater_faces[axis] - num_faces;

			if (duplicates >= duplicate_threshold)
				continue;

			new_diff += duplicates;				// tunable
			new_diff /= mid_width[axis];		// tunable

			if ((best_axis < 0) || (new_diff < best_diff))
			{
				best_axis = axis;
				best_diff = new_diff;
			}
		}

		if (best_axis < 0)			// Couldn't make a good split, give up
		{
			return create_bsp_leaf(p_face_indexes, num_faces);
		}

		// Allocate new temp arrays for the face indexes
		unsigned short *p_less_face_indexes = new unsigned short[less_faces[best_axis]];
		unsigned short *p_greater_face_indexes = new unsigned short[greater_faces[best_axis]];

		// Now fill in the array
		calc_split_faces(best_axis, mid_split[best_axis], p_face_indexes, num_faces, 
						 less_faces[best_axis], greater_faces[best_axis],
						 p_less_face_indexes, p_greater_face_indexes);

		// And the new bboxes
		NxBoundingBox less_bbox(bbox), greater_bbox(bbox);
		less_bbox.m_Max[best_axis] = mid_split[best_axis];
		greater_bbox.m_Min[best_axis] = mid_split[best_axis];
	
		// And now calculate the branches
		NxCollBSPNode *p_bsp_tree = new NxCollBSPNode;
		p_bsp_tree->m_split_axis = best_axis;
		p_bsp_tree->m_split_point = mid_split[best_axis];
		p_bsp_tree->mp_less_branch = create_bsp_tree(less_bbox, p_less_face_indexes, less_faces[best_axis], level + 1);
		p_bsp_tree->mp_greater_branch = create_bsp_tree(greater_bbox, p_greater_face_indexes, greater_faces[best_axis], level + 1);

		// Free temp arrays
		delete p_less_face_indexes;
		delete p_greater_face_indexes;

		return p_bsp_tree;
	}
}
	
int NxCollBSPNode::CountBSPNodes( NxCollBSPNode *p_bsp_node )
{
	if (p_bsp_node)
	{
		return 1 + CountBSPNodes(p_bsp_node->mp_less_branch) + CountBSPNodes(p_bsp_node->mp_greater_branch);
	} else {
		return 0;
	}
}

int	NxCollBSPNode::AssignNodesToArray( NxCollBSPNode *p_bsp_node, NxCollBSPNode **p_bsp_array,
									   int & cur_array_idx, int & cur_array_offset, bool root_node )
{
	if (p_bsp_node)
	{
		//int assigned_index = cur_array_idx;

		// If root of tree, assign node to array
		if (root_node)
		{
			p_bsp_array[cur_array_idx++] = p_bsp_node;
			p_bsp_node->m_array_offset = cur_array_offset;
			cur_array_offset += p_bsp_node->InstanceSize();
		}

		// By assigning the branches together, we can guarantee that the nodes are contiguous (and need only one pointer) 
		if (p_bsp_node->mp_less_branch)
		{
			Utils::Assert(p_bsp_node->mp_greater_branch != NULL, "Only less BSP branch is not NULL");

			p_bsp_array[cur_array_idx++] = p_bsp_node->mp_less_branch;
			p_bsp_node->mp_less_branch->m_array_offset = cur_array_offset;
			p_bsp_node->m_less_branch_idx = cur_array_offset;
			cur_array_offset += p_bsp_node->mp_less_branch->InstanceSize();
		}
		if (p_bsp_node->mp_greater_branch)
		{
			Utils::Assert(p_bsp_node->mp_less_branch != NULL, "Only greater BSP branch is not NULL");

			p_bsp_array[cur_array_idx++] = p_bsp_node->mp_greater_branch;
			p_bsp_node->mp_greater_branch->m_array_offset = cur_array_offset;
			p_bsp_node->m_greater_branch_idx = cur_array_offset;
			cur_array_offset += p_bsp_node->mp_greater_branch->InstanceSize();
		}

		// Do recursion
		AssignNodesToArray(p_bsp_node->mp_less_branch, p_bsp_array, cur_array_idx, cur_array_offset, false);
		AssignNodesToArray(p_bsp_node->mp_greater_branch, p_bsp_array, cur_array_idx, cur_array_offset, false);

		//return assigned_index;
		return p_bsp_node->m_array_offset;
	} else {
		return -1;
	}
}

bool NxObject::AddCASData( unsigned long mask, unsigned long data0, unsigned long data1 )
{
	if ( mp_CASData && m_NumCASData < vMAXCASDATA )
	{
		mp_CASData[m_NumCASData].m_Mask		= mask;
		mp_CASData[m_NumCASData].m_Data		= data0;
		mp_CASData[m_NumCASData].m_Data1	= data1;
		m_NumCASData++;
		return true;
	}
	else
	{
		return false;
	}
}
	
NxMesh::NxMesh( unsigned long mat_checksum )
: Lst::Node< NxMesh > ( this ), m_MatChecksum( mat_checksum )
{
	m_NumStripVerts = 0;
	m_VertStrip = NULL;
	m_NumFaces = 0;	
	m_Object;
	m_Flags = 0;
	m_ShadowFlags = 0;
	for ( int i = 0; i < vMAX_FACES; i++ )
	{
		m_CASFaceFlags[i] = 0;
	}
}

NxMesh::~NxMesh( void )
{
	if( m_VertStrip )
	{
		delete [] m_VertStrip;
	}
}

unsigned long NxMesh::FindCASFaceFlags( int vertIndex0, int vertIndex1, int vertIndex2 )
{
	int vertIndex[3];
	vertIndex[0] = vertIndex0;
	vertIndex[1] = vertIndex1;
	vertIndex[2] = vertIndex2;

	int i, j;

	// lame sort 3 verts
	for ( i = 0; i < 2; i++ )
	{
		if ( vertIndex[i] > vertIndex[i+1] )
		{
			int swap = vertIndex[i];
			vertIndex[i] = vertIndex[i+1];
			vertIndex[i+1] = swap;
			i = -1;
		}
	}

	for ( j = 0; j < m_NumFaces; j++ )
	{
		int topologyIndex[3];
		topologyIndex[0] = m_Topology[j][0];
		topologyIndex[1] = m_Topology[j][1];
		topologyIndex[2] = m_Topology[j][2];

		for ( i = 0; i < 2; i++ )
		{
			if ( topologyIndex[i] > topologyIndex[i+1] )
			{
				int swap = topologyIndex[i];
				topologyIndex[i] = topologyIndex[i+1];
				topologyIndex[i+1] = swap;
				i = -1;
			}
		}

		if ( topologyIndex[0] == vertIndex[0] 
			&& topologyIndex[1] == vertIndex[1]
			&& topologyIndex[2] == vertIndex[2] )
		{
			return m_CASFaceFlags[j];
		}
	}

	printf( "err: couldn't find topography %d %d %d!\n", vertIndex[0],
		vertIndex[1],
		vertIndex[2] );
	return 0;
}


void NxMesh::SetWibbleFlags( void )
{
	int i;
	NxVertex* vert_list;
	bool has_wibble_data;

	m_Flags &= ~NxObject::mVCWIBBLE;

	// If the object didn't have any wibble data, this mesh won't have any
	if(( m_Object->m_Flags & NxObject::mVCWIBBLE ) == 0 )
	{
		return;
	}

	has_wibble_data = false;
	vert_list = m_Object->m_Verts;
	for( i = 0; i < m_NumStripVerts; i++ )
	{
		NxVertex* vert;

		vert = &vert_list[m_VertStrip[i].m_Index];
		if( vert->m_WibbleIndex > 0 )
		{
			m_Flags |= NxObject::mVCWIBBLE;
			return;
		}
	}

}

NxMeshGroup::NxMeshGroup( void )
: Lst::Node< NxMeshGroup > ( this )
{
}

NxMeshList::NxMeshList( void )
{
	for( int l = 0; l < 8; ++l )
	{
		m_NumMeshes[l] = 0;
		m_Meshes[l] = new NxMesh*[MESH_BLOCK_SIZE];
		m_MaxMeshes[l] = MESH_BLOCK_SIZE;
	}
}

NxMeshList::~NxMeshList( void )
{
	for( int l = 0; l < 8; ++l )
	{
		delete m_Meshes[l];
	}
}

void	NxMeshList::Reset( void )
{
	for( int l = 0; l < 8; ++l )
	{
		for( int i = 0; i < m_NumMeshes[l]; i++ )
		{
			m_Meshes[l][i] = NULL;
		}
		m_NumMeshes[l] = 0;
	}
}

void	NxMeshList::AppendMesh( NxMesh* mesh, int lod )
{
//	Utils::Assert( m_NumMeshes[lod] < MAX_NUM_MESHES, "NxMeshList array overrun" );

	if ( m_NumMeshes[lod] >= m_MaxMeshes[lod] )
	{
		NxMesh** pp_old_meshes = m_Meshes[lod];
		m_MaxMeshes[lod] += MESH_BLOCK_SIZE;
		m_Meshes[lod] = new NxMesh*[m_MaxMeshes[lod]];
		memcpy( m_Meshes[lod], pp_old_meshes, sizeof( NxMesh* ) * m_NumMeshes[lod] );
	}

	m_Meshes[lod][ m_NumMeshes[lod]++ ] = mesh;
}

void	NxMeshList::operator = ( NxMeshList& mesh_list )
{
	for( int l = 0; l < 8; ++l )
	{
		m_NumMeshes[l] = mesh_list.m_NumMeshes[l];
		for( int i = 0; i < m_NumMeshes[l]; i++ )
		{
			m_Meshes[l][i] = mesh_list.m_Meshes[l][i];
		}
	}
}

NxTexture::NxTexture( void )
{
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		m_TexelData[i] = NULL;		
		m_MipLevels = 0;		
	}

	m_AlreadyExported = false;
	m_GroupFlags = 0;
	m_Flags = 0;
	m_PlatFlags = 0;
	m_TotalPaletteDataSize = 0;
	m_TotalTexelDataSize = 0;
	m_PaletteData = NULL;
	m_Bpp = 0;
	m_PaletteBpp = 0;
	m_LastGroupIndex = vNO_GROUP;
	m_LastGroupId = vNO_GROUP;
	m_LastDrawOrder = 0;
	strcpy( m_Name, "None" );
}

NxTexture::~NxTexture( void )
{
	int i;

	for( i = 0; i <= m_MipLevels; i++ )
	{
		if ( m_TexelData[i] )
		{
			delete [] m_TexelData[i];		
		}
	}
	
	if ( m_PaletteData )
	{
		delete [] m_PaletteData;
	}
}

bool NxTexture::IsPaletted( void )
{
	return (( m_PixelFormat == v8_BIT ) || ( m_PixelFormat == v4_BIT ));
}



int	NxTexture::GetTotalDataSize( void )
{
	return( m_TotalPaletteDataSize + m_TotalTexelDataSize );
}



bool NxTexture::Convert4BitPixelFormatTo8BitPixelFormat( void )
{
	if( !IsPaletted() || ( m_PixelFormat != v4_BIT ))
	{
		return false;
	}

	// Reset this so we count up from scratch.
	m_TotalTexelDataSize = 0;

	for( int i = 0; i <= m_MipLevels; ++i )
	{
		int	src, dst;
		unsigned char*	p_8bit_texel_data;
		int	num_texels, bytes_per_row, row_count;
		
		num_texels = m_Width[i] * m_Height[i];
		p_8bit_texel_data = new unsigned char[num_texels];
		
		/*int texel_index = 0;
		for( int p = 0; p < m_TexelDataSize[i]; ++p )
		{
			// Read indices.
			unsigned int index0 = ((unsigned char)m_TexelData[i][p] >> 0 ) & 0xF;
			unsigned int index1 = ((unsigned char)m_TexelData[i][p] >> 4 ) & 0xF;

			p_8bit_texel_data[texel_index++] = index0;
			if( texel_index < num_texels )
			{
				p_8bit_texel_data[texel_index++] = index1;
			}
		}*/

		// I'm sure this loop could be more straight-forward, but
		// for now, I'm just concerned with correctness. -SG
		src = 0;
		dst = 0;
		bytes_per_row = (( m_Width[i] * 4 ) + 7 ) >> 3;
		for( int j = 0; j < m_Height[i]; ++j )
		{
			row_count = 0;
			for( int k = 0; k < bytes_per_row; ++k )
			{
				// Read indices.
				unsigned int index0 = ((unsigned char)m_TexelData[i][src] >> 0 ) & 0xF;
				unsigned int index1 = ((unsigned char)m_TexelData[i][src] >> 4 ) & 0xF;

				p_8bit_texel_data[dst++] = index0;
				row_count++;
				if( row_count < m_Width[i] )
				{
					p_8bit_texel_data[dst++] = index1;
					row_count++;
				}				
				src++;
			}
		}

		delete [] m_TexelData[i];

		m_TexelData[i]			= (char*)p_8bit_texel_data;
		m_TexelDataSize[i]		= num_texels;
		m_TotalTexelDataSize	+= m_TexelDataSize[i];
	}
			
	// Change the format.
	m_PixelFormat = v8_BIT;

	return true;
}



bool NxTexture::Convert16BitPaletteFormatTo32BitPaletteFormat( void )
{
	if( !IsPaletted() || ( m_PaletteFormat != v16_BIT ))
	{
		return false;
	}

	unsigned int*	p_32bit_palette_data	= new unsigned int[m_NumPaletteEntries];

	for( int p = 0; p < m_NumPaletteEntries; ++p )
	{
		unsigned int entry	= ((unsigned short*)m_PaletteData )[p];
		entry				= (( entry & 0x1F ) << 3 ) | ((( entry >> 5 ) & 0x1F ) << 11 ) | ((( entry >> 10 ) & 0x1F ) << 19 ) | (( entry & 0x8000 ) ? 0xFF000000 : 0x00 );

		p_32bit_palette_data[p] = entry;
	}

	delete [] m_PaletteData;

	m_PaletteData			= (char*)p_32bit_palette_data;
	m_TotalPaletteDataSize	*= 2;

	m_PaletteFormat = v32_BIT;

	return true;
}



bool NxTexture::Convert24BitPaletteFormatTo32BitPaletteFormat( void )
{
	if( !IsPaletted() || ( m_PaletteFormat != v24_BIT ))
	{
		return false;
	}

	unsigned int	*p_32bit_palette_data	= new unsigned int[m_NumPaletteEntries];
	unsigned char	*p_24bit_palette_data	= (unsigned char*)m_PaletteData;
	for( int p = 0; p < m_NumPaletteEntries; ++p )
	{
		unsigned int red	= *p_24bit_palette_data++;
		unsigned int grn	= *p_24bit_palette_data++;
		unsigned int blu	= *p_24bit_palette_data++;
		unsigned int entry	= red | ( grn << 8 ) | ( blu << 16 ) | 0xFF000000;

		p_32bit_palette_data[p] = entry;
	}

	delete [] m_PaletteData;

	m_PaletteData			= (char*)p_32bit_palette_data;
	m_TotalPaletteDataSize	= ( m_TotalPaletteDataSize * 4 ) / 3;

	m_PaletteFormat = v32_BIT;

	return true;
}



bool NxTexture::Convert32BitRGBAPaletteFormatTo32BitBGRAPaletteFormat( void )
{
	if( !IsPaletted() || ( m_PaletteFormat != v32_BIT ))
	{
		return false;
	}

	unsigned int *p_32bit_palette_data	= (unsigned int*)m_PaletteData;
	for( int p = 0; p < m_NumPaletteEntries; ++p )
	{
		unsigned int old		= *p_32bit_palette_data;
		unsigned int red		= old & 0xFF;
		unsigned int grn		= ( old >> 8 ) & 0xFF;
		unsigned int blu		= ( old >> 16 ) & 0xFF;
		unsigned int alp		= ( old >> 24 ) & 0xFF;
		*p_32bit_palette_data++	= blu | ( grn << 8 ) | ( red << 16 ) | ( alp << 24 );
	}

	return true;
}



__forceinline int Ftoi_ASM( const float f )
{
    __asm cvttss2si eax, f        // return int(f)
}

__forceinline void AdjustChrominance( unsigned int &red, unsigned int &grn, unsigned int &blu, float s )
{
	float r		= ((float)red ) / 255.0f;
	float g		= ((float)grn ) / 255.0f;
	float b		= ((float)blu ) / 255.0f;
	
	// Approximate values for each component's contribution to luminance.
    // Based upon the NTSC standard described in ITU-R Recommendation BT.709.
	float grey	= ( r * 0.2125f ) + ( g * 0.7154f ) + ( b * 0.0721f );
    
	float new_r	= grey + s * ( r - grey );
	float new_g	= grey + s * ( g - grey );
	float new_b	= grey + s * ( b - grey );

	new_r		= ( new_r > 1.0f ) ? 1.0f : ( new_r < 0.0f ) ? 0.0f : new_r;
	new_g		= ( new_g > 1.0f ) ? 1.0f : ( new_g < 0.0f ) ? 0.0f : new_g;
	new_b		= ( new_b > 1.0f ) ? 1.0f : ( new_b < 0.0f ) ? 0.0f : new_b;

	red			= Ftoi_ASM( new_r * 255.0f );
	grn			= Ftoi_ASM( new_g * 255.0f );
	blu			= Ftoi_ASM( new_b * 255.0f );
}


bool NxTexture::Convert32BitRGBAPixelFormatTo32BitBGRAPixelFormat( void )
{
	if( m_PixelFormat != v32_BIT )
	{
		return false;
	}

	for( int i = 0; i <= m_MipLevels; ++i )
	{
		unsigned int *p_32bit_texel_data	= (unsigned int*)m_TexelData[i];
		for( int p = 0; p < ( m_Width[i] * m_Height[i] ); ++p )
		{
			unsigned int old		= *p_32bit_texel_data;
			unsigned int red		= old & 0xFF;
			unsigned int grn		= ( old >> 8 ) & 0xFF;
			unsigned int blu		= ( old >> 16 ) & 0xFF;
			unsigned int alp		= ( old >> 24 ) & 0xFF;

			AdjustChrominance( red, grn, blu, 1.115f );
			
			*p_32bit_texel_data++	= blu | ( grn << 8 ) | ( red << 16 ) | ( alp << 24 );
		}
	}

	return true;
}



bool NxTexture::ConvertTo32BitPixelFormat( void )
{
	// Currently only supports 16 and 32 bit palettes (if a palette is present).
	if( IsPaletted() && ( m_PaletteFormat != v32_BIT ) && ( m_PaletteFormat != v24_BIT ) && ( m_PaletteFormat != v16_BIT ))
	{
		return false;
	}

	switch( m_PixelFormat )
	{
		case v4_BIT:
		{
			// Just convert to 8 bit pixel format, and fall through to next case.
			if( !Convert4BitPixelFormatTo8BitPixelFormat())
			{
				return false;
			}
		}
	
		case v8_BIT:
		{
			// Reset this so we count up from scratch.
			m_TotalTexelDataSize = 0;

			// Convert to 32 bit palette if not already.
			if( m_PaletteFormat == v16_BIT )
			{
				if( !Convert16BitPaletteFormatTo32BitPaletteFormat())
				{
					return false;
				}
			}
			if( m_PaletteFormat == v24_BIT )
			{
				if( !Convert24BitPaletteFormatTo32BitPaletteFormat())
				{
					return false;
				}
			}

			for( int i = 0; i <= m_MipLevels; ++i )
			{
				unsigned int	texel_index			= 0;
				unsigned int*	p_32bit_texel_data	= new unsigned int[m_Width[i] * m_Height[i]];

				unsigned char*	p_texel_data	= (unsigned char*)m_TexelData[i];
				for( int p = 0; p < m_TexelDataSize[i]; ++p )
				{
					// Read indices.
					unsigned int entry;
					unsigned int index0					= *p_texel_data++;

					// Must be in 32 bit palette format at this stage.
					entry								= ((unsigned int*)m_PaletteData )[index0];
					p_32bit_texel_data[texel_index++]	= entry;
				}

				delete [] m_TexelData[i];

				m_TexelData[i]			= (char*)p_32bit_texel_data;
				m_TexelDataSize[i]		*= 4;
				m_TotalTexelDataSize	+= m_TexelDataSize[i];
			}
			
			// Now we can free up the palette data...
			delete [] m_PaletteData;
			m_PaletteData = NULL;

			m_TotalPaletteDataSize		= 0;

			// ...and change format..
			m_PixelFormat = v32_BIT;

			return true;
		}

		case v24_BIT:
		{
			// Reset this so we count up from scratch.
			m_TotalTexelDataSize = 0;

			for( int i = 0; i <= m_MipLevels; ++i )
			{
				unsigned int*	p_32bit_texel_data	= new unsigned int[m_Width[i] * m_Height[i]];
				unsigned char*	p_24bit_texel_data	= (unsigned char*)m_TexelData[i];

				for( int p = 0; p < ( m_Width[i] * m_Height[i] ); ++p )
				{
					// Read indices.
					unsigned int red		= *p_24bit_texel_data++;
					unsigned int grn		= *p_24bit_texel_data++;
					unsigned int blu		= *p_24bit_texel_data++;
					p_32bit_texel_data[p]	= red | ( grn << 8 ) | ( blu << 16 ) | 0xFF000000UL;
				}

				delete [] m_TexelData[i];

				m_TexelData[i]			= (char*)p_32bit_texel_data;
				m_TexelDataSize[i]		= ( m_TexelDataSize[i] * 4 ) / 3;
				m_TotalTexelDataSize	+= m_TexelDataSize[i];
			}

			// Change the format.
			m_PixelFormat = v32_BIT;

			return true;
		}

		default:
		{
			break;
		}
	}

	// Cannot convert.
	return false;
}

void	SceneConverter::Sort(float* val0,float* val1,float* val2)
{
	float v0=*val0, v1=*val1, v2=*val2;
	float min=v0,max=v0;

	if (v0<v1 && v0<v2)
		min=v0;

	if (v1<v0 && v1<v2)
		min=v1;

	if (v2<v0 && v2<v1)
		min=v2;

	if (v0>v1 && v0>v2)
		max=v0;

	if (v1>v0 && v1>v2)
		max=v1;

	if (v2>v0 && v2>v1)
		max=v2;

	*val0=min;
	*val2=max;

	if (v0!=min && v0!=max)
	{
		*val1=v0;
		return;
	}

	if (v1!=min && v1!=max)
	{
		*val1=v1;
		return;
	}

	*val1=v2;
}

bool	SceneConverter::HasValidUVs(NxMaterial* material, NxObject* obj, NxFace* face, int pass )
{
	int channel;

	iTotal++;

	// Get clamping parameters
	bool UClamp = material->m_Passes[pass].m_AddressModeU==NxTexture::vADDRESS_MODE_CLAMP;
	bool VClamp = material->m_Passes[pass].m_AddressModeV==NxTexture::vADDRESS_MODE_CLAMP;

	int vert0 = face->m_Vertex[0];
	int vert1 = face->m_Vertex[1];
	int vert2 = face->m_Vertex[2];
	float u0,v0;
	float u1,v1;
	float u2,v2;

	channel = get_source_uv_channel( face->m_PassFlags, pass );

	u0 = obj->m_Verts[vert0].m_TexCoord[channel][0];
	v0 = obj->m_Verts[vert0].m_TexCoord[channel][1];
	u1 = obj->m_Verts[vert1].m_TexCoord[channel][0];
	v1 = obj->m_Verts[vert1].m_TexCoord[channel][1];
	u2 = obj->m_Verts[vert2].m_TexCoord[channel][0];
	v2 = obj->m_Verts[vert2].m_TexCoord[channel][1];
	
	// UVs are always valid if not clamped
	if ((!UClamp || !VClamp))
	{
		return true;
	}

	// Sort the uv's left to right and top to bottom
	Sort(&u0,&u1,&u2);
	Sort(&v0,&v1,&v2);

	//printf("obj: %x u0: %f  u1: %f  u2: %f\n",obj,u0,u1,u2);
	//printf("obj: %x v0: %f  v1: %f  v2: %f\n",obj,v0,v1,v2);

	// Check uv's
	if ( (u2<0.0f || u0>1.0f) &&
		 (v2<0.0f || v0>1.0f))
	{
		iReject++;
		return false;
	}

	return true;
}

int	SceneConverter::get_source_uv_channel( int pass_flags, int src_pass )
{
	int i, pass_count;

	pass_count = 0;
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		if( pass_flags & ( 1 << i ))
		{
			if( src_pass == pass_count )
			{
				return i;
			}
			pass_count++;
		}
	}

	// GJ:  This was happening on THPS4 as well, but
	// assert messages were not enabled in the release
	// builds, so no one noticed it.  It's probably
	// worth tracking down why it fires off in the first
	// place, but I'll leave that for someone else to do.
//	Utils::Assert( 0, "get_source_uv_channel returned bad value!" );

	return 0;
}



void SceneConverter::SeparateMultiPassMeshes( void )
{
	for( int i = 0; i < m_scene->m_NumObjects; i++ )
	{
		int j, k, l, total_verts, total_faces;
		NxObject* object;
		NxVertex* new_vert_list;
		NxFace* new_face_list;		
		int num_new_faces, num_new_verts, dst_vert_idx, dst_face_idx;

		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		
		// First, count up the number of extra faces we'll have to create as a result of multi-pass
		// rendering, which creates a new mesh per pass		
		num_new_faces = 0;
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxMaterial* material;

			NxFace* face=&object->m_Faces[j];

			// On xbox, we'll only separate passes for faces that are flagged for that behavior
			if(( face->m_FaceFlags & NxFace::mFD_RENDER_SEPARATE ) == 0 )
			{
				continue;
			}

			material = m_scene->GetMaterial( face->m_MatChecksum );
			if(( material == NULL ) || ( material->m_NumPasses == 1 ))
			{
				continue;
			}

			if( face->m_PassFlags == 0 )
			{
				continue;
			}

			material->m_ShouldExpand = true;
			
			// We'll  have to create a new face for every extra pass
			for( int pass=1; pass < material->m_NumPasses; pass++ )
			{
				if( HasValidUVs(material, object, face, pass ))
				{
					num_new_faces++;
				}
			}
				
		}

		num_new_verts = num_new_faces * 3;

		total_faces = object->m_NumFaces + num_new_faces;
		total_verts = object->m_NumVerts + num_new_verts;

		new_vert_list = new NxVertex[ total_verts ];
		new_face_list = new NxFace[ total_faces ];

		// First, copy all of the existing faces and verts into the new array
		memcpy( new_vert_list, object->m_Verts, sizeof( NxVertex ) * object->m_NumVerts );
		memcpy( new_face_list, object->m_Faces, sizeof( NxFace ) * object->m_NumFaces );

		// Now fill the rest of the array with the new verts and faces from the extra passes
		dst_vert_idx = object->m_NumVerts;
		dst_face_idx = object->m_NumFaces;

		int eCount=0;

		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxFace* src_face, *dst_face;
			NxMaterial* material;
			int src_uv_channel;
			NxVertex src_verts[3];

			src_face = &new_face_list[j];
			material = m_scene->GetMaterial( src_face->m_MatChecksum );
			if( material == NULL )
			{
				continue;
			}

			if( src_face->m_PassFlags == 0 )
			{
				continue;
			}

			src_verts[0] = new_vert_list[src_face->m_Vertex[0]];
			src_verts[1] = new_vert_list[src_face->m_Vertex[1]];
			src_verts[2] = new_vert_list[src_face->m_Vertex[2]];
			if( src_face->m_FaceFlags & NxFace::mFD_RENDER_SEPARATE )
			{				
				// Even for 1-pass materials, we need to make sure that the face is using the appropriate
				// set of UVs. A 1-pass face may now use UV channel 3 if it wishes.
				src_uv_channel = get_source_uv_channel( src_face->m_PassFlags, 0 );
				if( src_uv_channel != 0 )
				{						
					for( k = 0; k < 3; k++ )
					{
						// Now copy over pass-specific info into the new vertex (which represents a subsequent pass)
						new_vert_list[src_face->m_Vertex[k]].m_TexCoord[0][0] = src_verts[k].m_TexCoord[src_uv_channel][0];
						new_vert_list[src_face->m_Vertex[k]].m_TexCoord[0][1] = src_verts[k].m_TexCoord[src_uv_channel][1];					
					}
				}
				// Add each extra face to the face list
				// Add each faces' verts to the vert list
				for( k = 1; k < material->m_NumPasses; k++ )
				{
					src_uv_channel = get_source_uv_channel( src_face->m_PassFlags, k );
					if (HasValidUVs(material, object, src_face, k ))
					//if (HasValidUVs(material, object, src_face, src_uv_channel))
					{
						eCount++;

						dst_face = &new_face_list[ dst_face_idx ];

						*dst_face = *src_face;
						dst_face->m_Pass = ( k );
						dst_face->m_MatChecksum = src_face->m_MatChecksum + k + 1;
						dst_face->m_Vertex[0] = dst_vert_idx;
						dst_face->m_Vertex[1] = dst_vert_idx + 1;
						dst_face->m_Vertex[2] = dst_vert_idx + 2;
					
						for( l = 0; l < 3; l++ )
						{
							new_vert_list[dst_vert_idx] = src_verts[l];
							// Now copy over pass-specific info into the new vertex (which represents a subsequent pass)
							//new_vert_list[dst_vert_idx].m_TexCoord[0][0] = new_vert_list[dst_vert_idx].m_TexCoord[k][0];
							//new_vert_list[dst_vert_idx].m_TexCoord[0][1] = new_vert_list[dst_vert_idx].m_TexCoord[k][1];
							new_vert_list[dst_vert_idx].m_TexCoord[0][0] = src_verts[l].m_TexCoord[src_uv_channel][0];
							new_vert_list[dst_vert_idx].m_TexCoord[0][1] = src_verts[l].m_TexCoord[src_uv_channel][1];

							dst_vert_idx++;
						}
						dst_face_idx++;
					}
				}			
			}
			else
			{
				for( k = 0; k < material->m_NumPasses; k++ )
				{
					src_uv_channel = get_source_uv_channel( src_face->m_PassFlags, k );
					if (HasValidUVs(material, object, src_face, src_uv_channel ))
					{						
						for( l = 0; l < 3; l++ )
						{
							// Now copy over pass-specific info into the new vertex (which represents a subsequent pass)
							new_vert_list[src_face->m_Vertex[l]].m_TexCoord[k][0] = src_verts[l].m_TexCoord[src_uv_channel][0];
							new_vert_list[src_face->m_Vertex[l]].m_TexCoord[k][1] = src_verts[l].m_TexCoord[src_uv_channel][1];
						}
					}
				}
			}
		}
		
		// Now remap the first-pass faces to the new 1-pass material that represents
		// pass 1 of the original multi-pass material
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxFace* src_face;
			NxMaterial* material;

			src_face = &new_face_list[j];
			material = m_scene->GetMaterial( src_face->m_MatChecksum );
			if( material == NULL )
			{
				continue;
			}

			// Only materials with more than one pass will be split into separate layers...
			if( material->m_NumPasses <= 1 )
			{
				continue;
			}

			// If this face doesn't use Pass1, then don't need to remap it.
			if(( src_face->m_PassFlags & NxMaterial::mGROUP_PASS_1 ) == 0 )
			{
				continue;
			}

			if( src_face->m_FaceFlags & NxFace::mFD_RENDER_SEPARATE )
			{				
				// A new 1-pass material that represents the first pass of this multi-pass material
				// will be created whose checksum will be one greater than the original.
				src_face->m_MatChecksum++;
			}
		}

		// We don't need the originals anymore
		delete [] object->m_Verts;
		delete [] object->m_Faces;

		// Point the object to its new topology
		object->m_NumVerts = total_verts;
		object->m_NumFaces = total_faces;		
		
		object->m_Verts = new_vert_list;
		object->m_Faces = new_face_list;
	}	
}

void	SceneConverter::ExpandMultiPassMaterials( void )
{
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;//, *next;
	VCWibbleSequence* seq, *new_seq;
	Lst::Search< VCWibbleSequence > vc_sh;
	
	for( material = sh.FirstItem( m_scene->m_Materials ); material;
			material = sh.NextItem())
	{
		if( material->m_ShouldExpand )
		{			
			int i, j;

			for( i = 0; i < material->m_NumPasses; i++ )
			{
				NxMaterial* new_mat;

				new_mat = new NxMaterial;
				new_mat->m_Pass = i;								
				new_mat->m_NumPasses = 1;
				new_mat->m_Checksum = material->m_Checksum + i + 1;
				new_mat->m_materialName = material->m_materialName + i + 1;
				new_mat->m_GroupFlags = material->m_GroupFlags & NxMaterial::mGROUP_TRANSPARENT;				
				new_mat->m_GroupFlags |= ( NxMaterial::mGROUP_PASS_1 << i );
				new_mat->m_AlphaCutoff = material->m_AlphaCutoff;
				new_mat->m_DrawOrder = material->m_DrawOrder + ( i * 0.01f );	// Increase the draw order every pass
				new_mat->m_Sorted = material->m_Sorted;
				new_mat->m_Passes[0] = material->m_Passes[i];				

				if( new_mat->m_Sorted )
				{
					// Sort "sorted" materials to the end of the list
					// I add one to make sure that we won't get a value of 0, which would merge sorted materials
					// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
					// sort to the end of the list
					new_mat->SetPri(( new_mat->m_DrawOrder + 1.0f ) * 1000.0f );
				}
				else
				{
					new_mat->SetPri( new_mat->m_DrawOrder );
				}

				for( seq = vc_sh.FirstItem( material->m_WibbleSequences ); seq; seq = vc_sh.NextItem())
				{
					new_seq = new VCWibbleSequence;
					new_seq->m_Index = seq->m_Index;
					new_seq->m_Offset = seq->m_Offset;
					new_seq->m_NumFrames = seq->m_NumFrames;
					new_seq->m_Phase = seq->m_Phase;
					new_seq->m_WibbleFrames = new WibbleKeyframe[new_seq->m_NumFrames];
					for( j = 0; j < new_seq->m_NumFrames; j++ )
					{
						new_seq->m_WibbleFrames[j] = seq->m_WibbleFrames[j];
					}

					new_mat->m_WibbleSequences.AddToTail( new_seq );
				}

				m_scene->m_Materials.AddNodeBackwards( new_mat );

				// Continually adjust the checksum until we know it does not clash with an existing material checksum.
				while( m_scene->GetMaterial( new_mat->m_Checksum ))
				{
					++new_mat->m_Checksum;
				}
				m_scene->AddMaterialToLookup( new_mat );
				m_scene->m_NumMaterials++;
			}
		}
		else
		{
			material->m_GroupFlags |= NxMaterial::mGROUP_PASS_1;
		}
	}	

	// Now remove the original, multi-pass materials
	/*for( material = sh.FirstItem( m_scene->m_Materials ); material;	material = next )
	{		
		next = sh.NextItem();
		if( material->m_NumPasses > 1 )
		{
			material->Remove();
			m_scene->m_NumMaterials--;
		}
	}*/
}

void SceneConverter::WriteUSGDuplicates(IoUtils::CVirtualOutputFile* pUsageFile)
{
	// Write out any duplicate texture entries
	Link<CRCEntry>* curlink = usedCRCs.GetHead();
	char buf[256];

	while(curlink)
	{
		if (curlink->data.CRCs.GetSize()>1)
		{
			sprintf(buf,"Duplicate Entry! Name: %s\n",curlink->data.Name);
			pUsageFile->Write((const char*)buf,strlen(buf));

			Link<unsigned long>* curCRC = curlink->data.CRCs.GetHead();

			while(curCRC)
			{
				sprintf(buf,"\t--> CRC: (%x)\n",curCRC->data);
				pUsageFile->Write((const char*)buf,strlen(buf));
				curCRC = curCRC->next;
			}
		}

		curlink = curlink->next;
	}
}

void SceneConverter::AddUSGTexture(NxTexture* texture)
{
	CRCEntry crcEntry;
	char* name = strrchr(texture->m_Name,'\\');

	if (name)
		name++;
	else
		name = "";

	strcpy(crcEntry.Name,name);
	
	Link<CRCEntry>* link = usedCRCs.AddToTailFindUnique(&crcEntry);
	link->data.CRCs.AddToTailUnique(&texture->m_Checksum);
}

void SceneConverter::EnableOptimization( bool enable )
{
	m_generate_optimized_level = enable;
}

void SceneConverter::EnableWeightMapGeneration( bool generate_weight_map )
{
	m_generate_weight_map = generate_weight_map;
}