#include <memory.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <Strip/Strip.h>
#include <PS2/PS2Conv.h>
//#include <List/Search.h>
//#include <Util.h>
#include <Misc/GenCRC.h>
#include <windows.h>
#include <assert.h>
#include <ngcdll/progmesh.h>

#include "VirtualFile.h"

#include "core/defines.h"
#include "core/math.h"
#include "gfx/ngps/nx/dma.h"
#include "gfx/ngps/nx/vif.h"
#include "gfx/ngps/nx/vu1.h"
#include "gfx/ngps/nx/vu1code.h"
#include "gfx/ngps/nx/gif.h"
#include "gfx/ngps/nx/gs.h"
#include "gfx/ngps/nx/material.h"
#include "gfx/ngps/nx/texture.h"
#include "gfx/ngps/nx/mesh.h"
#include "gfx/ngps/nx/group.h"
#include "gfx/ngps/nx/mikemath.h"
#include "gfx/ngps/nx/geomnode.h"
#include "gfx/ngps/nx/render.h"

#include "gfx/NxHierarchy.h"

// From util.h
int	GetLog2( int value );
bool IsPowerOfTwo( int value );
bool FileExists( char* path );
bool FileIsNewer( char* file1, char* file2 );

bool	gTextureOverride = false;
int		gTextureOverrideMapping[Utils::vNUM_PLATFORMS] = { 0, 1, 2 };

// Mick's solution to some linking problems...
namespace Dbg
{
	void Assert(char *p_i, unsigned int x, char *y) {}
	void pad_printf(char const *p_z, ...) {}
	char mybuffer[256];
	char *sprintf_pad = mybuffer;
}

class PS2Converter : public IPS2Converter
{
public:
	PS2Converter() { iReject=0; iTotal=0; }

	bool	ConvertData( void );
	bool	SaveCollisionData( char* path );
	bool	SaveScene( char* path );
	bool	SaveGeom( char* tex_path, char* scn_path, char* geom_path );
	bool	SaveTextureDictionary( char* path, char* usg_path );
	bool	SaveCASFlags( char* path );
	bool	SaveWeightMap( char* path );
	int		GetPlatformFlags( void );

	bool	SaveShadowVolumes( IoUtils::CVirtualOutputFile * p_file );

	void	InitializeTempTextureData( void );
	float	GetAverageGroupPriority( int group );
	int		NumTexturesInGroup( int group );
	int		SaveTextureGroup( IoUtils::CVirtualOutputFile* pOutFile, IoUtils::CVirtualOutputFile* pUsgFile, int group, bool sorted, int index );
	bool	SaveMaterials( IoUtils::CVirtualOutputFile* pOutFile );
	bool	SaveGeometry( IoUtils::CVirtualOutputFile* pOutFile );
	__int64	GetBlendParameters( int blend_mode, int fixed );
	int		GetTargetTextureDataSize( NxTexture* texture );
	int		GetTotalVRAMUsed( NxTexture* texture );
	int		GetTotalVRAMUsed( NxMaterial* material );

	void	ExpandGrassGeometry( void );
	void	ExpandMultiPassMaterials( void );
	void	SeparateMultiPassMeshes( void );
	void	OptimizeVertexLists( void );
	void	ApplyMaterialColors( void );
	void	CreateTriStrips( void );
	void	CreateTextureGroups( void );
	void	CreateMeshGroups( void );	

	bool	LoadTextures( IoUtils::CVirtualInputFile* pInputFile, int version );
	bool	LoadMaterials( IoUtils::CVirtualInputFile* pInputFile, int version );
	bool	LoadMeshes( IoUtils::CVirtualInputFile* pInputFile );
	void	LoadVertices( IoUtils::CVirtualInputFile* pInputFile, NxPs2::sMesh *pMesh, NxPs2::sMaterial *pMat, NxPs2::sGroup *pGroup);
	void	BeginModelPrim( uint32 Regs, uint NReg, uint Prim, uint Pre, uint Addr, uint Step );
	void	EndModelPrim( uint Eop );
	void	BeginModelMultiPrim( uint32 Regs, uint NReg, uint Prim, uint NLoop, uint Addr, NxPs2::sMesh *pMesh, uint Step);
	void	EndModelMultiPrim( NxPs2::sMesh *pMesh );
	void	BeginVertex( NxPs2::sMesh *pMesh, NxPs2::sMaterial *pMat );
	void	EndVertex( void );
	void	VertexST16( uint8 *pData );
	void	VertexST32( uint8 *pData );
	void	VertexSTFloat( uint8 *pData );
	void	VertexNormal( uint8 *pData );
	void	VertexWeights( uint8 *pData );
	void	VertexSkinNormal( uint8 *pNormData, uint8 *pTOData );
	void	VertexRGBA( uint8 *pData, bool wibbleVC, uint32 seq );
	void	VertexRGB( uint8 *pData);
	void	VertexXYZ16( uint8 *pData, bool IsSkin, Mth::Vector &sphere );
	void	VertexXYZ32( uint8 *pData, bool IsSkin, Mth::Vector &sphere );
	void	VertexXYZFloat( uint8 *pData );
	void	CreateNodes( void );
	bool	SaveGeom( IoUtils::CVirtualOutputFile* pOutputFile );

	sint8	FindHierarchyBoneIndex(uint32 checksum);

private:
	bool	already_in_group( NxMaterial* material, int group_index );
	int		get_last_group_index( NxMaterial* material );
	void	set_last_group_index( NxMaterial* material, int group_index );
	void	set_last_draw_order( NxMaterial* material, int draw_order );

	Lst::Head< NxMeshGroup >	m_MeshGroups;
	int		m_NumSortedTextureGroups;
	int		m_NumUnsortedTextureGroups;
	char	m_tex_buff[1024 * 1024 * 4];
};
	
static PS2Converter*	sp_ps2_conv = NULL;
static int	s_first_sorted_texture_group = 0;
static int	s_first_unsorted_texture_group = 0;;
static int	s_num_first_pass_faces = 0;
static int  s_num_multi_pass_faces = 0;

extern NxPs2::sMesh GMeshes[6000];


IPS2Converter* GetPS2Converter( void )
{
	if ( sp_ps2_conv == NULL )
	{
		sp_ps2_conv = new PS2Converter;
		s_first_sorted_texture_group = 0;
		s_first_unsorted_texture_group = 0;
		s_num_first_pass_faces = 0;
		s_num_multi_pass_faces = 0;
	}

	return sp_ps2_conv;
}

void ResetPS2Converter()
{
	if ( sp_ps2_conv )
	{
		delete sp_ps2_conv;
		sp_ps2_conv = NULL;
	}
}

void	PS2Converter::ExpandGrassGeometry( void )
{
	int i;

	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		int j, k, l, total_verts, total_faces;
		NxObject* object;
		NxVertex* new_vert_list;
		NxFace* new_face_list;		
		int num_new_faces, num_new_verts, dst_vert_idx, dst_face_idx;

		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}

		// First, count up the number of extra faces we'll have to create as a result of grass
		// rendering, which creates a new mesh per grass level		
		num_new_faces = 0;
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxMaterial* material;
			NxFace* face=&object->m_Faces[j];

			material = m_scene->GetMaterial( face->m_MatChecksum );
			if(( material == NULL ) || !( material->m_grassify ))
			{
				continue;
			}

			if ( HasValidUVs(material, object, face, 0 ) )
			{
				num_new_faces += material->m_grassLayers;
			}
		}

		num_new_verts = num_new_faces * 3;

		total_faces = object->m_NumFaces + num_new_faces;
		total_verts = object->m_NumVerts + num_new_verts;

		new_vert_list = new NxVertex[ total_verts ];
		new_face_list = new NxFace[ total_faces ];

		// First, copy all of the existing faces and verts into the new array
		memcpy( new_vert_list, object->m_Verts, sizeof( NxVertex ) * object->m_NumVerts );
		memcpy( new_face_list, object->m_Faces, sizeof( NxFace ) * object->m_NumFaces );

		// Now fill the rest of the array with the new verts and faces from the extra passes
		dst_vert_idx = object->m_NumVerts;
		dst_face_idx = object->m_NumFaces;

		int eCount=0;

		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxFace* src_face, *dst_face;
			NxMaterial* material;
			int src_uv_channel;
			NxVertex src_verts[3];

			src_face = &new_face_list[j];
			material = m_scene->GetMaterial( src_face->m_MatChecksum );
			if( ( material == NULL )  || !( material->m_grassify ))
			{
				continue;
			}
			
			if (!HasValidUVs(material, object, src_face, 0 ))
			{
				continue;
			}

			src_verts[0] = new_vert_list[src_face->m_Vertex[0]];
			src_verts[1] = new_vert_list[src_face->m_Vertex[1]];
			src_verts[2] = new_vert_list[src_face->m_Vertex[2]];

			int num_grass_layers = material->m_grassLayers;
			float layer_height = material->m_grassHeight / num_grass_layers;

			// Add each extra face to the face list
			// Add each faces' verts to the vert list
			for( k = 0; k < num_grass_layers; k++ )
			{
				src_uv_channel = get_source_uv_channel( src_face->m_PassFlags, 0 );

				eCount++;

				dst_face = &new_face_list[ dst_face_idx ];

				// Note that the material checksum is case-sensitve (so we use GenerateCRC(string, length))
				char material_name[64];
				sprintf( material_name, "Grass-Grass_Layer%d", k );
				uint32 layer_mat_checksum = GenerateCRC(material_name, strlen( material_name ));

				*dst_face = *src_face;
				dst_face->m_Pass = ( 0 );
				dst_face->m_FaceFlags |= NxFace::mFD_NO_SKATER_SHADOW;
				dst_face->m_MatChecksum = layer_mat_checksum;
				dst_face->m_Vertex[0] = dst_vert_idx;
				dst_face->m_Vertex[1] = dst_vert_idx + 1;
				dst_face->m_Vertex[2] = dst_vert_idx + 2;

				for( l = 0; l < 3; l++ )
				{
					new_vert_list[dst_vert_idx] = src_verts[l];
					new_vert_list[dst_vert_idx].m_Pos[1] += (layer_height * (k + 1));
			
					// Calculate uv's based on xz's
					float u	= src_verts[l].m_Pos[0] / 48.0f;
					float v	= src_verts[l].m_Pos[2] / 48.0f;

					// Now copy over pass-specific info into the new vertex (which represents a subsequent pass)
					new_vert_list[dst_vert_idx].m_TexCoord[src_uv_channel][0] = u;
					new_vert_list[dst_vert_idx].m_TexCoord[src_uv_channel][1] = v;

					dst_vert_idx++;
				}
				dst_face_idx++;

				// Adjust layer material
				NxMaterial* layer_material = m_scene->GetMaterial(layer_mat_checksum);
				if (!layer_material)
				{
					printf("Error: Can't find grass material %s\n", material_name);
					continue;
				}

				// DrawOrder must be above 1500 so shadow is drawn first
				float base_draw_order = Mth::Max(1501.0f, material->m_DrawOrder);
				if (layer_material->m_DrawOrder != (base_draw_order + k))
				{
					layer_material->m_GroupFlags |= NxMaterial::mGROUP_TRANSPARENT;
					layer_material->m_DrawOrder = base_draw_order + k;

					if( layer_material->m_Sorted )
					{
						// Sort "sorted" materials to the end of the list
						// I add one to make sure that we won't get a value of 0, which would merge sorted materials
						// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
						// sort to the end of the list
						layer_material->SetPri(( layer_material->m_DrawOrder + 1 ) * 1000.0f );
					}
					else
					{
						layer_material->SetPri( layer_material->m_DrawOrder );
					}

					// Re-add to list
					layer_material->Remove();
					m_scene->RemoveMaterialFromLookup(layer_material);
					m_scene->m_Materials.AddNodeBackwards(layer_material);
					m_scene->AddMaterialToLookup(layer_material);
				}
			}			
		}
		
		// We don't need the originals anymore
		delete [] object->m_Verts;
		delete [] object->m_Faces;

		// Point the object to its new topology
		assert(dst_vert_idx == total_verts);
		assert(dst_face_idx == total_faces);
		object->m_NumVerts = total_verts;
		object->m_NumFaces = total_faces;		
		
		object->m_Verts = new_vert_list;
		object->m_Faces = new_face_list;
	}

	// Clear these stat variables
	iReject=0; iTotal=0;
}

NxMaterial* s_tempMaterial[5000];
int s_tempCount = 0;

void	PS2Converter::ExpandMultiPassMaterials( void )
{
	int i, j;
	Lst::Head< NxMaterial > pass_list[vMAX_MATERIAL_PASSES];
	Lst::Search< NxMaterial > sh;
	NxMaterial* material, *next;
	VCWibbleSequence* seq, *new_seq;
	Lst::Search< VCWibbleSequence > vc_sh;
	
	// first look for all the multi materials
	s_tempCount = 0;
	for( material = sh.FirstItem( m_scene->m_Materials ); material; )
	{
		next = sh.NextItem();

		if( material->m_NumPasses > 1 )
		{
			// remove it from the list and lookup
			material->Remove();
			m_scene->m_NumMaterials--;
			m_scene->RemoveMaterialFromLookup( material );
			s_tempMaterial[s_tempCount++] = material;
		}
		else
		{
			material->m_GroupFlags |= NxMaterial::mGROUP_PASS_1;
		}

		material = next;
	}

	for( int z = 0; z < s_tempCount; z++ )
	{
		material = s_tempMaterial[z];	

		for( i = 0; i < material->m_NumPasses; i++ )
		{
			NxMaterial* new_mat;

			new_mat = new NxMaterial;
			new_mat->m_materialName = material->m_materialName;
			new_mat->m_Pass = i;				
			new_mat->m_NumPasses = 1;
			new_mat->m_Checksum = material->m_Checksum + i;
			new_mat->m_GroupFlags = material->m_GroupFlags & NxMaterial::mGROUP_TRANSPARENT;				
			new_mat->m_GroupFlags |= ( NxMaterial::mGROUP_PASS_1 << i );
			new_mat->m_AlphaCutoff = material->m_AlphaCutoff;
			new_mat->m_DrawOrder = material->m_DrawOrder;
			new_mat->m_Sorted = material->m_Sorted;
			new_mat->m_BasePass = material->m_BasePass;
			new_mat->m_OneSided = material->m_OneSided;
			new_mat->m_Passes[0] = material->m_Passes[i];

			if( new_mat->m_Sorted )
			{
				// Sort "sorted" materials to the end of the list
				// I add one to make sure that we won't get a value of 0, which would merge sorted materials
				// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
				// sort to the end of the list
				new_mat->SetPri(( new_mat->m_DrawOrder + 1 ) * 1000.0f );
			}
			else
			{
				new_mat->SetPri( new_mat->m_DrawOrder );
			}

			if ( (i+1 < material->m_NumPasses) &&
				 ((material->m_Passes[i+1].m_BlendMode == vBLEND_MODE_BLEND_PREVIOUS_MASK) ||
				  (material->m_Passes[i+1].m_BlendMode == vBLEND_MODE_BLEND_INVERSE_PREVIOUS_MASK)))
			{
				new_mat->m_Passes[0].m_Flags |= NxMaterialPass::FORCE_ALPHA;
			}

			for( seq = vc_sh.FirstItem( material->m_WibbleSequences ); seq; seq = vc_sh.NextItem())
			{
				new_seq = new VCWibbleSequence;
				new_seq->m_Index = seq->m_Index;
				new_seq->m_Offset = seq->m_Offset;
				new_seq->m_Phase = seq->m_Phase;
				new_seq->m_NumFrames = seq->m_NumFrames;
				new_seq->m_WibbleFrames = new WibbleKeyframe[new_seq->m_NumFrames];
				for( j = 0; j < new_seq->m_NumFrames; j++ )
				{
					new_seq->m_WibbleFrames[j] = seq->m_WibbleFrames[j];
				}
				new_mat->m_WibbleSequences.AddToTail( new_seq );
			}

			pass_list[i].AddNodeBackwards( new_mat );
			//m_scene->m_Materials.AddNodeBackwards( new_mat );
			//m_scene->AddMaterialToLookup( new_mat );
			//m_scene->m_NumMaterials++;
		}
	}
	
	for( i = 0; i < vMAX_MATERIAL_PASSES; i++ )
	{
		for( material = sh.FirstItem( pass_list[i] ); material; 
				material = next )
		{
			next = sh.NextItem();
			material->Remove();
			m_scene->m_Materials.AddNodeBackwards( material );
			m_scene->AddMaterialToLookup( material );
			m_scene->m_NumMaterials++;
		}
	}

	for( z=0; z<s_tempCount; z++ )
	{
		delete s_tempMaterial[z];
		s_tempCount = 0;
	}
}

void	PS2Converter::OptimizeVertexLists( void )
{
	int i;

	NxObject* pObject = &m_scene->m_Objects[0];		
	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		pObject->OptimizeVertList( m_scene );
		pObject++;
	}
}

void	PS2Converter::ApplyMaterialColors( void )
{
	Lst::Search< VCWibbleSequence > sh;
	Lst::Search< NxMaterial > mat_sh;
	VCWibbleSequence* seq;
	NxMaterial* material;
	int i, j;

	// First, apply the material colors to verts that use the material
	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;		

		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			int k;			

			material = m_scene->GetMaterial( object->m_Faces[j].m_MatChecksum );
			if( material )
			{
				for( k = 0; k < 3; k++ )
				{
					object->m_Verts[ object->m_Faces[j].m_Vertex[k]].m_Color[0] *= material->m_Passes[0].m_Color[0];
					object->m_Verts[ object->m_Faces[j].m_Vertex[k]].m_Color[1] *= material->m_Passes[0].m_Color[1];
					object->m_Verts[ object->m_Faces[j].m_Vertex[k]].m_Color[2] *= material->m_Passes[0].m_Color[2];
					if( material->m_Passes[0].m_Flags & NxMaterialPass::IGNORE_VERTEX_ALPHA )
					{
						object->m_Verts[ object->m_Faces[j].m_Vertex[k]].m_Color[3] = 1.0f;
					}					
				}
			}
		}
	}

	// Next, apply the material colors to wibble sequences for this material		
	for( material = mat_sh.FirstItem( m_scene->m_Materials ); material; material = mat_sh.NextItem())
	{
		for( seq = sh.FirstItem( material->m_WibbleSequences ); seq; seq = sh.NextItem())
		{
			for( i = 0; i < seq->m_NumFrames; i++ )
			{
				seq->m_WibbleFrames[i].m_Color[0] *= material->m_Passes[0].m_Color[0];
				seq->m_WibbleFrames[i].m_Color[1] *= material->m_Passes[0].m_Color[1];
				seq->m_WibbleFrames[i].m_Color[2] *= material->m_Passes[0].m_Color[2];
				seq->m_WibbleFrames[i].m_Color[3] = ( seq->m_WibbleFrames[i].m_Color[3] / 2 ) + 0.5f;	// alpha on the PS2 is opaque at 0.5
			}
		}
	}
}

void	PS2Converter::SeparateMultiPassMeshes( void )
{
	int i;

	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		int j, k, l, total_verts, total_faces;
		NxObject* object;
		NxVertex* new_vert_list;
		NxFace* new_face_list;		
		int num_new_faces, num_new_verts, dst_vert_idx, dst_face_idx;

		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}

		s_num_first_pass_faces += object->m_NumFaces;

		// First, count up the number of extra faces we'll have to create as a result of multi-pass
		// rendering, which creates a new mesh per pass		
		num_new_faces = 0;
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxMaterial* material;

			material = m_scene->GetMaterial( object->m_Faces[j].m_MatChecksum );
			if(( material == NULL ) || ( material->m_NumPasses == 1 ))
			{
				continue;
			}			

			NxFace* face=&object->m_Faces[j];

			if( face->m_PassFlags == 0 )
			{
				continue;
			}

			// We'll  have to create a new face for every extra pass
			for( int pass=1; pass < material->m_NumPasses; pass++ )
				if ( HasValidUVs(material, object, face, pass ) )
					num_new_faces++;

				//num_new_faces += ( material->m_NumPasses - 1 );
		}

		num_new_verts = num_new_faces * 3;

		total_faces = object->m_NumFaces + num_new_faces;
		total_verts = object->m_NumVerts + num_new_verts;

		s_num_multi_pass_faces += num_new_faces;

		new_vert_list = new NxVertex[ total_verts ];
		new_face_list = new NxFace[ total_faces ];

		// First, copy all of the existing faces and verts into the new array
		memcpy( new_vert_list, object->m_Verts, sizeof( NxVertex ) * object->m_NumVerts );
		memcpy( new_face_list, object->m_Faces, sizeof( NxFace ) * object->m_NumFaces );

		// Now fill the rest of the array with the new verts and faces from the extra passes
		dst_vert_idx = object->m_NumVerts;
		dst_face_idx = object->m_NumFaces;

		int eCount=0;

		for( j = 0; j < object->m_NumFaces; j++ )
		{
			NxFace* src_face, *dst_face;
			NxMaterial* material;
			int src_uv_channel;
			NxVertex src_verts[3];

			src_face = &new_face_list[j];
			material = m_scene->GetMaterial( src_face->m_MatChecksum );
			if( material == NULL )
			{
				continue;
			}

			if( src_face->m_PassFlags == 0 )
			{
				continue;
			}
			
			src_verts[0] = new_vert_list[src_face->m_Vertex[0]];
			src_verts[1] = new_vert_list[src_face->m_Vertex[1]];
			src_verts[2] = new_vert_list[src_face->m_Vertex[2]];

			// Even for 1-pass materials, we need to make sure that the face is using the appropriate
			// set of UVs. A 1-pass face may now use UV channel 3 if it wishes.
			src_uv_channel = get_source_uv_channel( src_face->m_PassFlags, 0 );
			if( src_uv_channel != 0 )
			{						
				for( k = 0; k < 3; k++ )
				{
					// Now copy over pass-specific info into the new vertex (which represents a subsequent pass)
					new_vert_list[src_face->m_Vertex[k]].m_TexCoord[0][0] = src_verts[k].m_TexCoord[src_uv_channel][0];
					new_vert_list[src_face->m_Vertex[k]].m_TexCoord[0][1] = src_verts[k].m_TexCoord[src_uv_channel][1];					
				}
			}
			// Add each extra face to the face list
			// Add each faces' verts to the vert list
			for( k = 1; k < material->m_NumPasses; k++ )
			{
				src_uv_channel = get_source_uv_channel( src_face->m_PassFlags, k );
				if (HasValidUVs(material, object, src_face, k ))
				//if (HasValidUVs(material, object, src_face, src_uv_channel))
				{
					eCount++;

					dst_face = &new_face_list[ dst_face_idx ];

					*dst_face = *src_face;
					dst_face->m_Pass = ( k );
					dst_face->m_MatChecksum = src_face->m_MatChecksum + k;
					dst_face->m_Vertex[0] = dst_vert_idx;
					dst_face->m_Vertex[1] = dst_vert_idx + 1;
					dst_face->m_Vertex[2] = dst_vert_idx + 2;
				
					for( l = 0; l < 3; l++ )
					{
						new_vert_list[dst_vert_idx] = src_verts[l];
						// Now copy over pass-specific info into the new vertex (which represents a subsequent pass)
						//new_vert_list[dst_vert_idx].m_TexCoord[0][0] = new_vert_list[dst_vert_idx].m_TexCoord[k][0];
						//new_vert_list[dst_vert_idx].m_TexCoord[0][1] = new_vert_list[dst_vert_idx].m_TexCoord[k][1];
						new_vert_list[dst_vert_idx].m_TexCoord[0][0] = src_verts[l].m_TexCoord[src_uv_channel][0];
						new_vert_list[dst_vert_idx].m_TexCoord[0][1] = src_verts[l].m_TexCoord[src_uv_channel][1];

						dst_vert_idx++;
					}
					dst_face_idx++;
				}
			}			
		}
		
		// We don't need the originals anymore
		delete [] object->m_Verts;
		delete [] object->m_Faces;

		// Point the object to its new topology
		object->m_NumVerts = total_verts;
		object->m_NumFaces = total_faces;		
		
		object->m_Verts = new_vert_list;
		object->m_Faces = new_face_list;
	}

	printf("\nRejected %i unnecessary faces of %i due to clamping.\n",iReject, iTotal);
	printf("First pass faces: %d	Multi-pass faces: %d\n", s_num_first_pass_faces, s_num_multi_pass_faces );
}

void	PS2Converter::CreateMeshGroups( void )
{
	int i, j;
	bool unique;
	
	for( i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;

		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		for( j = 0; j < object->m_MeshList.m_NumMeshes[0]; j++ )
		{
			NxMesh* mesh;
			NxMaterial* material;

			mesh = object->m_MeshList.m_Meshes[0][j];
			material = m_scene->GetMaterial( mesh->m_MatChecksum );
			if( material )
			{
				int group_num;
				Lst::Search< NxMeshGroup > group_sh;
				NxMeshGroup* mesh_group;

				if( material->m_Invisible )
				{
					continue;
				}

				group_num = material->m_GroupId;

				unique = true;
				for( mesh_group = group_sh.FirstItem( m_MeshGroups ); mesh_group;
						mesh_group = group_sh.NextItem())
				{
					if( mesh_group->m_GroupNumber == group_num )
					{
						if( material->m_Sorted )
						{
							// Sort "sorted" materials to the end of the list
							// I add one to make sure that we won't get a value of 0, which would merge sorted materials
							// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
							// sort to the end of the list
							mesh->SetPri(( material->m_DrawOrder + 1 ) * 1000.0f );
						}
						else
						{
							mesh->SetPri( material->m_DrawOrder );
						}

						//mesh_group->m_Meshes.AddToTail( mesh );
						mesh_group->m_Meshes.AddNodeBackwards( mesh );
						unique = false;
						break;
					}
				}

				if( unique )
				{
					NxMeshGroup* new_group;
					
					new_group = new NxMeshGroup;
					new_group->m_GroupNumber = group_num;						
					
					if( material->m_Sorted )
					{
						// Sort "sorted" materials to the end of the list
						// I add one to make sure that we won't get a value of 0, which would merge sorted materials
						// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
						// sort to the end of the list
						mesh->SetPri(( material->m_DrawOrder + 1 ) * 1000.0f );
					}
					else
					{
						mesh->SetPri( material->m_DrawOrder );
					}

					m_MeshGroups.AddToTail( new_group );

					new_group->m_Meshes.AddNodeBackwards( mesh );
				}				
			}
		}		
	}
}

void	PS2Converter::set_last_group_index( NxMaterial* material, int group_index )
{
	int i;
	NxTexture* tex;

	// By this time, all we have are one-pass materials
	for( i = 0; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
	{			
		tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));
		if( tex )
		{
			tex->m_LastGroupIndex = group_index;
		}
	}
}

void	PS2Converter::set_last_draw_order( NxMaterial* material, int draw_order )
{
	int i;
	NxTexture* tex;

	// By this time, all we have are one-pass materials
	for( i = 0; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
	{			
		tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));
		if( tex )
		{
			tex->m_LastDrawOrder = draw_order;
		}
	}
}

int		PS2Converter::get_last_group_index( NxMaterial* material )
{
	int i, group_index;
	NxTexture* tex;

	// By this time, all we have are one-pass materials
	tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( 0, Utils::vPLATFORM_PS2 ));
	if( tex == NULL )
	{
		return vNO_GROUP;
	}
	group_index = tex->m_LastGroupIndex;
	for( i = 1; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
	{			
		tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));
		if( tex )
		{
			if( tex->m_LastGroupIndex != group_index )
			{
				return vNO_GROUP;
			}
		}
	}

	return group_index;	
}

bool	PS2Converter::already_in_group( NxMaterial* material, int group_index )
{
	int i;
	NxTexture* tex;

	// By this time, all we have are one-pass materials
	for( i = 0; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
	{			
		tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));
		if( tex )
		{
			// Check if this texture is already in the current group. If not, the entire material is
			// not in that group
			if( tex->m_LastGroupIndex != group_index )
			{
				return false;
			}
		}
	}

	return true;
}

//#define DEBUG_GROUP_CREATION

void	PS2Converter::CreateTextureGroups( void )
{
	int group_size, group_id, group_index;
	char group_name[128];
	float cur_draw_order;
	int	cur_tex_size;
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;
	bool reached_sorted_mat, reached_trans_mat, first_trans_mat;
	NxTexture* tex;

	int tex_num = 0;

	group_index = 0;
	group_size = 0;
	m_NumUnsortedTextureGroups = 0;	
	s_first_unsorted_texture_group = 0;
	s_first_sorted_texture_group = 0;
	reached_sorted_mat = false;
	reached_trans_mat = false;
	
	int group_pass = 0;
	
	bool	filled_a_group = false;

	int group_passes[10000];  // for remembering which pass went with which group
	
	#ifdef	DEBUG_GROUP_CREATION	
	printf ("\n\nGroup 0\n");
	#endif
	
	// Create the texture groups by looping through all materials and
	// adding their textures to groups (space-permitting)
	for( material = sh.FirstItem( m_scene->m_Materials ); material;
			material = sh.NextItem())
	{		
		// We're only concerned with renderable materials
		if( material->m_Invisible )
		{
			continue;
		}	

		// If we've hit a sorted material, end this loop. We now move on to
		// sorted texture group creations
		if( material->m_Sorted )
		{
			reached_sorted_mat = true;
			cur_draw_order = material->m_DrawOrder;
			break;
		}

		// If we've hit a transparent material, end this loop. We now move on to
		// transparent texture group creations
		if( material->m_DrawOrder > 0 )
		{
			reached_trans_mat = true;
			cur_draw_order = material->m_DrawOrder;
			break;
		}
		
		// if we've changed which pass we are using, then start a new group
		// but only if we've filled up a group (with the first pass)
		// otherwise peds and small model would generate too many groups
		
		if (material->m_Pass != group_pass && filled_a_group)
		{
				
				group_pass = material->m_Pass;
				group_size = 0;
				group_index++;
				group_passes[group_index] = material->m_Pass;
				m_NumUnsortedTextureGroups++;
				#ifdef	DEBUG_GROUP_CREATION	
				printf ("Pass change caused group change\n");
				printf ("\n\nGroup %d Pass %d\n",group_index,group_pass);
				#endif
		}
		

		// By this time, all we have are one-pass materials
		tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( 0, Utils::vPLATFORM_PS2 ));
		if( tex )
		{
			// Check if this texture is already in the current group. If so, skip it
			if( already_in_group( material, group_index ))//tex->m_LastGroupIndex == group_index )
			{
				sprintf( group_name, "%s_%d", m_scene_name, group_index );
				group_id = GenerateCRC( group_name );
				material->m_GroupId = group_id;
				continue;
			}
			else if( get_last_group_index( material ) != vNO_GROUP )//tex->m_LastGroupIndex != vNO_GROUP )	// If the texture has been in a previous group
			{
				// If the draw order has not changed, it is still safe to render meshes that use this
				// material with a previous texture group -- ONLY IF WE DON'T COMPROMISE MULTI-PASS ORDERING.
				// So, for now, only perform this optimization on first passes
			    if(	( tex->m_LastDrawOrder == material->m_DrawOrder ) 
					//&& ( material->m_Pass == 0 )			  
					&& ( material->m_Pass == group_passes[get_last_group_index( material )] )
					)
				{
					sprintf( group_name, "%s_%d", m_scene_name, tex->m_LastGroupIndex );
					group_id = GenerateCRC( group_name );
					material->m_GroupId = group_id;
					continue;
				}
				else
				{
					#ifdef	DEBUG_GROUP_CREATION	
					printf ("Dupe with group %d (material = %d, group_pass = %d\n",get_last_group_index( material ), material->m_Pass, group_passes[get_last_group_index( material )]);
					#endif
				}
			}
			
			cur_tex_size = GetTotalVRAMUsed( material );
			// tex size of a given material must not exceed the group ceiling
			Dbg_Assert( cur_tex_size <= vTEX_GROUP_SIZE_CEILING );	
			group_size += cur_tex_size;
			if( group_size > vTEX_GROUP_SIZE_CEILING )
			{
				#ifdef	DEBUG_GROUP_CREATION	
				printf ("Total %d (last %d)\n",group_size,cur_tex_size);
				#endif
				group_size = cur_tex_size;
				group_index++;
				m_NumUnsortedTextureGroups++;
				#ifdef	DEBUG_GROUP_CREATION	
				printf ("\n\nGroup %d\n",group_index);
				#endif
				filled_a_group = true;
			}
			#ifdef	DEBUG_GROUP_CREATION	
			printf ("%3d Material order %f, pass %d OrigPass %d uses %d, total %d\n",tex_num++,material->m_DrawOrder,material->m_Pass, material->m_Passes[0].m_OrigPass,cur_tex_size,group_size);
			#endif
			set_last_group_index( material, group_index );
			set_last_draw_order( material, material->m_DrawOrder );
		}		

		// remember what pass this group contains, so we can effectivly find dupes of multi-passes
		group_passes[group_index] = material->m_Pass;
		
		sprintf( group_name, "%s_%d", m_scene_name, group_index );
		group_id = GenerateCRC( group_name );
		material->m_GroupId = group_id;
		
		if( m_NumUnsortedTextureGroups == 0 )
		{
			s_first_unsorted_texture_group = group_id;
			m_NumUnsortedTextureGroups++;
		}
	}

	if( reached_trans_mat )
	{
		// We only want to increment to a new group number if there were actually any opaque textures.
		// Otherwise, start at zero
		if( m_NumUnsortedTextureGroups > 0 )
		{
			group_index++;
		}
		group_size = 0;
		first_trans_mat = true;
		
		#ifdef	DEBUG_GROUP_CREATION	
		printf ("\n\nFirst TRANSPARENT Group %d\n",group_index);
		#endif
		// Create the texture groups by looping through all materials and
		// adding their textures to groups (space-permitting)
		for( material = material; material; material = sh.NextItem())
		{
			// We're only concerned with renderable materials
			if( material->m_Invisible )
			{
				continue;
			}	

			Dbg_Assert( material->m_DrawOrder > 0);				

			// If we've hit a sorted material, end this loop. We now move on to
			// sorted texture group creations
			if( material->m_Sorted )
			{
				reached_sorted_mat = true;
				cur_draw_order = material->m_DrawOrder;
				break;
			}

			// By this time, all we have are one-pass materials
			tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( 0, Utils::vPLATFORM_PS2 ));
			if( tex )
			{
				// Check if this texture is already in the current group. If so, skip it
				if( already_in_group( material, group_index ))//tex->m_LastGroupIndex == group_index )
				{
					sprintf( group_name, "%s_%d", m_scene_name, group_index );
					group_id = GenerateCRC( group_name );
					material->m_GroupId = group_id;
					continue;
				}
				else if( get_last_group_index( material ) != vNO_GROUP )//tex->m_LastGroupIndex != vNO_GROUP )	// If the texture has been in a previous group
				{
					// If the draw order has not changed, it is still safe to render meshes that use this
					// material with a previous texture group -- ONLY IF WE DON'T COMPROMISE MULTI-PASS ORDERING.
					// So, for now, only perform this optimization on first passes
					if(	( tex->m_LastDrawOrder == material->m_DrawOrder ) &&
						( material->m_Pass == 0 ))
					{
						sprintf( group_name, "%s_%d", m_scene_name, tex->m_LastGroupIndex );
						group_id = GenerateCRC( group_name );
						material->m_GroupId = group_id;
						continue;
					}
					else
					{
						#ifdef	DEBUG_GROUP_CREATION	
						printf ("Dupe with group %d (material = %d, group_pass = %d\n",get_last_group_index( material ), material->m_Pass, group_passes[get_last_group_index( material )]);
						#endif
					}
				}
				
				cur_tex_size = GetTotalVRAMUsed( material );
				Dbg_Assert( cur_tex_size <= vTEX_GROUP_SIZE_CEILING );	
				group_size += cur_tex_size;
				if( group_size > vTEX_GROUP_SIZE_CEILING )
				{
					#ifdef	DEBUG_GROUP_CREATION	
					printf ("Total %d (last %d)\n",group_size,cur_tex_size);
					#endif
					group_size = cur_tex_size;
					group_index++;
					#ifdef	DEBUG_GROUP_CREATION	
					printf ("\n\nGroup %d\n",group_index);
					#endif
					m_NumUnsortedTextureGroups++;
				}
				#ifdef	DEBUG_GROUP_CREATION	
				printf ("%3d Material order %f, pass %d  uses %d, total %d\n",tex_num++,material->m_DrawOrder,material->m_Pass,cur_tex_size,group_size);
				#endif
				set_last_group_index( material, group_index );//tex->m_LastGroupIndex = group_index;
				set_last_draw_order( material, material->m_DrawOrder );//tex->m_LastDrawOrder = material->m_DrawOrder;
			}			

			sprintf( group_name, "%s_%d", m_scene_name, group_index );
			group_id = GenerateCRC( group_name );
			material->m_GroupId = group_id;
			
			if( first_trans_mat )
			{
				s_first_unsorted_texture_group = group_id;
				m_NumUnsortedTextureGroups++;
				first_trans_mat = false;
			}
		}
	}
	
	// We only want to increment to a new group number if there were actually any unsorted textures.
	// Otherwise, start at zero
	if( m_NumUnsortedTextureGroups > 0 )
	{
		group_index++;
	}
	#ifdef	DEBUG_GROUP_CREATION	
	printf ("\n\nFirst SORTED Group %d\n",group_index);
	#endif
	m_NumSortedTextureGroups = 0;
	group_size = 0;
	if( reached_sorted_mat )
	{
		while( material )
		{
			for( material = material; material; material = sh.NextItem())
			{
				// Materials should be sorted such that "sorted" materials are at the end
				Dbg_Assert( material->m_Sorted );				
				if( material->m_DrawOrder != cur_draw_order )
				{
					printf( "Error: Scene has more than one priority for sorted groups\n" );
					Dbg_Assert( 0 );	// Should not have more than one sorted group
					cur_draw_order = material->m_DrawOrder;
					group_index++;	// the next set of sorted materials should start a new group
					break;
				}

				// By this time, all we have are one-pass materials
				tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( 0, Utils::vPLATFORM_PS2 ));
				if( tex )
				{
					// Check if this texture is already in the current group. If so, skip it
					if( already_in_group( material, group_index ))//tex->m_LastGroupIndex == group_index )
					{
						sprintf( group_name, "%s_%d", m_scene_name, group_index );
						group_id = GenerateCRC( group_name );
						material->m_GroupId = group_id;
						continue;
					}

					cur_tex_size = GetTotalVRAMUsed( material );
					Dbg_Assert( cur_tex_size <= vSORTED_TEX_GROUP_SIZE_CIELING );	
					group_size += cur_tex_size;
					if( group_size > vSORTED_TEX_GROUP_SIZE_CIELING )
					{
						#ifdef	DEBUG_GROUP_CREATION	
						printf ("Total %d (last %d)\n",group_size,cur_tex_size);
						#endif
						group_size = 0;
						#ifdef	DEBUG_GROUP_CREATION	
						printf ("\n\nGroup %d\n",group_index);
						#endif
						group_index++;
						m_NumSortedTextureGroups++;
						printf( "Sorted Group Size Limit Exceeded!  Too many sorted materials!\n" );
						Dbg_Assert( 0 );
					}
					#ifdef	DEBUG_GROUP_CREATION	
					printf ("%3d Material order %f, pass %d  uses %d, total %d\n",tex_num++,material->m_DrawOrder,material->m_Pass,cur_tex_size,group_size);
					#endif
				
					set_last_group_index( material, group_index );//tex->m_LastGroupIndex = group_index;
				}				
				
				sprintf( group_name, "%s_%d", m_scene_name, group_index );
				group_id = GenerateCRC( group_name );
				material->m_GroupId = group_id;				
				
				if( m_NumSortedTextureGroups == 0 )
				{
					s_first_sorted_texture_group = group_id;
					m_NumSortedTextureGroups++;
				}				
			}			
		}
	}
}

void	PS2Converter::CreateTriStrips( void )
{
	INxStripper* stripper;
	NxObject* object;
	int i, j, num_objects;
	
	stripper = GetNxStripper();
	num_objects = m_scene->m_NumObjects;
	
	for( i = 0; i < num_objects; i++ )
	{
		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		stripper->Reset();
		stripper->SetObject( object );
		stripper->SetPlatform( Utils::vPLATFORM_PS2 );
		if( stripper->Stripify())
		{
			object->m_MeshList = *stripper->GetMeshList();
			for( j = 0; j < object->m_MeshList.m_NumMeshes[0]; j++ )
			{
				NxMesh* mesh = object->m_MeshList.m_Meshes[0][j];
				NxMaterial* material;

				// By default, a mesh inherits the flags of the object.  However,
				// afterwards, we strip the mesh of some object flags that did not
				// apply to this particular mesh
				mesh->m_Flags = object->m_Flags;

				// Now check to see if this mesh is actually textured.  It is possible
				// that some of the object was textured, but that does not mean that all of it is.				
				material = m_scene->GetMaterial( mesh->m_MatChecksum );
				if( material )
				{
					mesh->m_Flags |= PassToFlags( material->m_Pass + material->m_BasePass );
					if( material->m_Passes[0].GetTextureChecksum( 0, Utils::vPLATFORM_PS2 ) == 0 )
					{
						mesh->m_Flags &= ~NxObject::mTEXTURED;
					}
					if (material->m_Passes[0].m_Flags & NxMaterialPass::COLOR_LOCKED)
					{
						mesh->m_Flags |= NxObject::mCOLOR_LOCKED;
					}
					if (material->m_Passes[0].m_Flags & NxMaterialPass::UNLIT)
					{
						mesh->m_Flags |= NxObject::mUNLIT;
					}
					if( material->m_OneSided == false )
					{
						mesh->m_Flags &= ~ NxObject::mSS_NORMALS;
					}
				}
				else
				{
					mesh->m_Flags &= ~NxObject::mTEXTURED;
				}
			}
		}
	}
}

bool	PS2Converter::ConvertData( void )
{
	//ExpandGrassGeometry();
	SeparateMultiPassMeshes();
	ExpandMultiPassMaterials();
	ApplyMaterialColors();
	OptimizeVertexLists();
	CreateTriStrips();
	CreateTextureGroups();
	CreateMeshGroups();	
	
	return true;
}

bool	PS2Converter::SaveMaterials( IoUtils::CVirtualOutputFile* pOutFile )
{
	int i, j, num_materials;
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;

	num_materials = m_scene->GetNumVisibleMaterials();
	pOutFile->Write((const char*) &num_materials, sizeof( int ),false);
	for( material = sh.FirstItem( m_scene->m_Materials ); material;
			material = sh.NextItem())
	{
		int flags, num_sequences, mmag, mmin;
		__int64 equation;
		unsigned long tex_checksum;

		if( material->m_Invisible )
		{
			continue;
		}

		pOutFile->Write((const char*) &material->m_Checksum, sizeof( unsigned long ), false);

		// *** Flag output was moved here, after the checksum in version 5 since the loader
		// will need the flag data to parse the subsequent data
		flags = 0;
		if( material->m_Passes[0].GetTextureChecksum( Utils::vPLATFORM_PS2 ) != 0 )
		{
			flags |= NxMaterial::mTEXTURED;
		}
		if( material->m_Passes[0].m_Flags & NxMaterialPass::ANIMATED_TEXTURE )
		{
			flags |= NxMaterial::mANIMATED_TEX;
		}
		if( material->m_Passes[0].m_MappingMode == vMAPPING_ENVIRONMENT )
		{
			flags |= NxMaterial::mENVIRONMENT;
		}
		if( material->m_Passes[0].m_UVWibbleEnabled )
		{
			flags |= NxMaterial::mUV_WIBBLE;
		}
		if( material->m_WibbleSequences.CountItems() > 0 )
		{
			flags |= NxMaterial::mVC_WIBBLE;
		}
		if( material->m_GroupFlags & NxMaterial::mGROUP_TRANSPARENT )
		{
			flags |= NxMaterial::mTRANSPARENT;
		}
		if (material->m_Passes[0].m_Flags & NxMaterialPass::FORCE_ALPHA)
		{
			flags |= NxMaterial::mFORCE_ALPHA;
		}
		// For now, everything is smooth
		flags |= NxMaterial::mSMOOTH;
		pOutFile->Write((const char*) &flags, sizeof( int ));		
		
		if( vPS2_MATERIAL_VERSION_NUMBER == 0x0002 )
		{
			unsigned char cut_off;
			
			cut_off = (unsigned char) material->m_AlphaCutoff;
			pOutFile->Write((const char*) &cut_off, sizeof( unsigned char ), false);
		}
		else if( vPS2_MATERIAL_VERSION_NUMBER >= 0x0003 )
		{
			pOutFile->Write((const char*) &material->m_AlphaCutoff, sizeof( int ), false);
		}

		if( flags & NxMaterial::mANIMATED_TEX )
		{
			NxAnimatedTexture* anim_tex;

			anim_tex = &material->m_Passes[0].m_AnimatedTexture;
			pOutFile->Write((const char*) &anim_tex->m_NumKeyframes, sizeof( int ), false);
			pOutFile->Write((const char*) &anim_tex->m_Period, sizeof( int ), false);
			pOutFile->Write((const char*) &anim_tex->m_Iterations, sizeof( int ), false);
			pOutFile->Write((const char*) &anim_tex->m_Phase, sizeof( int ), false);
			for( j = 0; j < anim_tex->m_NumKeyframes; j++ )
			{
				NxAnimatedTextureKeyframe* frame;

				frame = &anim_tex->m_Keyframes[j];
				pOutFile->Write((const char*) &frame->m_Time, sizeof( unsigned int ), false);

				tex_checksum = material->m_Passes[0].GetTextureChecksum( j, Utils::vPLATFORM_PS2 );
				pOutFile->Write((const char*) &tex_checksum, sizeof( unsigned long ), false);
			}
		}
		else
		{
			tex_checksum = material->m_Passes[0].GetTextureChecksum( Utils::vPLATFORM_PS2 );
			pOutFile->Write((const char*) &tex_checksum, sizeof( unsigned long ), false);
		}
		
		if( vPS2_MATERIAL_VERSION_NUMBER >= 0x0003 )
		{
			pOutFile->Write((const char*) &material->m_GroupId, sizeof( int ), false);
		}
		
		// Write out PS2 alpha-blending params
		equation = GetBlendParameters( material->m_Passes[0].m_BlendMode, 
											material->m_Passes[0].m_FixedAlpha );
		pOutFile->Write((const char*) &equation, sizeof( __int64 ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_AddressModeU, sizeof( int ), false);
		pOutFile->Write((const char*) &material->m_Passes[0].m_AddressModeV, sizeof( int ));
		// Write out surface properties of the material
		pOutFile->Write((const char*) &material->m_Passes[0].m_Diffuse[0], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Diffuse[1], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Diffuse[2], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Specular[0], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Specular[1], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Specular[2], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Ambient[0], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Ambient[1], sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_Ambient[2], sizeof( float ));

		// Write out UV wibble params
		if( flags & NxMaterial::mUV_WIBBLE )
		{
			pOutFile->Write((const char*) &material->m_Passes[0].m_UVel, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_VVel, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_UFrequency, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_VFrequency, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_UAmplitude, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_VAmplitude, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_UPhase, sizeof( float ));
			pOutFile->Write((const char*) &material->m_Passes[0].m_VPhase, sizeof( float ));

			// write two empty words
			float temp;
			temp=0.0f;
			pOutFile->Write((const char*) &temp, sizeof( float ));
			pOutFile->Write((const char*) &temp, sizeof( float ));
		}

		// Write out VC wibble sequences
		if( flags & NxMaterial::mVC_WIBBLE )
		{
			VCWibbleSequence* seq;
			Lst::Search< VCWibbleSequence > vc_sh;

			num_sequences = material->m_WibbleSequences.CountItems();
			pOutFile->Write((const char*) &num_sequences, sizeof( int ));
			for( seq = vc_sh.FirstItem( material->m_WibbleSequences ); seq; seq = vc_sh.NextItem())
			{
				pOutFile->Write((const char*) &seq->m_NumFrames, sizeof( int ));
				pOutFile->Write((const char*) &seq->m_Phase, sizeof( int ));
				for( i = 0; i < seq->m_NumFrames; i++ )
				{
					pOutFile->Write((const char*) &seq->m_WibbleFrames[i].m_Time, sizeof( int ));
					pOutFile->Write((const char*) seq->m_WibbleFrames[i].m_Color, 4 * sizeof( float ));
				}
			}		
		}

		// Write out MMAG and MMIN		
		mmag = (int) material->m_Passes[0].m_MagFilteringMode;
		mmin = (int) material->m_Passes[0].m_MinFilteringMode;
		pOutFile->Write((const char*) &mmag, sizeof( int ));
		pOutFile->Write((const char*) &mmin, sizeof( int ));

		// Write out K
		pOutFile->Write((const char*) &material->m_Passes[0].m_MipMapK, sizeof( float ));

		// Write out u and v reflection map scale values
		pOutFile->Write((const char*) &material->m_Passes[0].m_EnvTileU, sizeof( float ));
		pOutFile->Write((const char*) &material->m_Passes[0].m_EnvTileV, sizeof( float ));
	}

	return true;
}


bool	PS2Converter::SaveGeometry( IoUtils::CVirtualOutputFile* pOutFile )
{
	int i, num_verts, num_mesh_groups, num_faces;
	Lst::Search< NxMeshGroup > group_sh;
	NxMeshGroup* group;
	Mth::Vector billboard_verts[4];
	Mth::Vector billboard_texcoords[4];
	
	num_mesh_groups = m_MeshGroups.CountItems();
	pOutFile->Write((const char*) &num_mesh_groups, sizeof( int ));

	// count total number of meshes across all groups
	int total_num_meshes = 0;
	for( group = group_sh.FirstItem( m_MeshGroups ); group;
			group = group_sh.NextItem())
	{
		total_num_meshes += group->m_Meshes.CountItems();
	}
	if (vPS2_MESH_VERSION_NUMBER >= 4)
		pOutFile->Write((const char *) &total_num_meshes, sizeof( int ));


	// this counter is used to match up the face flags
	// to the appropriate tri-stripped vert in the mesh
	unsigned long skinned_vertex_index = 0;
	int total_num_faces = 0;
	NxPs2::sMesh *p_mesh = GMeshes;
	for( group = group_sh.FirstItem( m_MeshGroups ); group;
			group = group_sh.NextItem())
	{
		int num_meshes, mesh_id;	
		Lst::Search< NxMesh > mesh_sh;
		NxMesh* mesh;
		
		mesh_id = group->m_GroupNumber;
		pOutFile->Write((const char*) &mesh_id, sizeof( int ));
		num_meshes = group->m_Meshes.CountItems();
		pOutFile->Write((const char*) &num_meshes, sizeof( int ));

		for(mesh=mesh_sh.FirstItem( group->m_Meshes ); mesh; mesh=mesh_sh.NextItem(), p_mesh++)
		{
			NxVertex* vert_list;
			NxFace* face_list;
			NxObject* object;
			unsigned long mat_checksum;
			Mth::Vector center, vertex, min, max, half_vec;
			float radius, len;
			NxMaterial* material;
			
			material = m_scene->GetMaterial( mesh->m_MatChecksum );
			object = mesh->m_Object;
			vert_list = object->m_Verts;
			face_list = object->m_Faces;
			num_faces = object->m_NumFaces;
			total_num_faces += mesh->m_NumFaces;

#if 0
			// calculate object's bounding sphere
			diag_length = 0;
			for( i = 0; i < 3; i++ )
			{
				center[i] = ( object->m_BoundingBox.m_Max[i] + object->m_BoundingBox.m_Min[i] ) / 2;
				diag[i] = ( object->m_BoundingBox.m_Max[i] - object->m_BoundingBox.m_Min[i] );
				diag_length += ( diag[i] * diag[i] );
			}
			diag_length = sqrtf( diag_length );
			radius = diag_length / 2.0f;
#else
			// calculate a tighter bounding sphere for the object...

			// use centre of bounding box as centre of sphere (not optimal, but usually good)
			for( i = 0; i < 3; i++ )
			{
				center[i] = ( object->m_BoundingBox.m_Max[i] + object->m_BoundingBox.m_Min[i] ) / 2;
			}

			// start with radius 0, and grow the sphere each time a point is outside
			radius = 0.0f;
			for (i=0; i<object->m_NumVerts; i++)
			{
				vertex[0] = object->m_Verts[i].m_Pos[0];
				vertex[1] = object->m_Verts[i].m_Pos[1];
				vertex[2] = object->m_Verts[i].m_Pos[2];
				len = (vertex - center).Length();
				if (len > radius)
					radius = len;
			}
#endif

			// Write out the object's checksum
			pOutFile->Write((const char*) &object->m_Checksum, sizeof( unsigned long ));

			// Write out the object's bounding sphere for mesh version 2 onwards
			if (vPS2_MESH_VERSION_NUMBER >= 2)
			{
				unsigned int lod_master_checksum = 0;
				float lod_far_dist = MAX_LOD_DIST;

				if (object->m_Flags & NxObject::mHASLODINFO) {
					//printf("Object %x has LOD info\n", object->m_Checksum);
					// Master LOD
					if (object->m_LODFlags & NxObject::mMASTER /*&& object->m_LODMaster.m_NumLODLevels > 1*/) {
						lod_far_dist = object->m_LODMaster.m_LODLevels[0].m_Distance;

						//printf("Writing master %x with distance of %f\n", object->m_Checksum, lod_far_dist);
					}

					// Slave LOD
					if (object->m_LODFlags & NxObject::mSLAVE) {
						NxObject *p_master = object->m_LODSlave.mp_master_object;

						lod_master_checksum = object->m_LODSlave.m_masterCRC;
						for (int idx = 0; idx < p_master->m_LODMaster.m_NumLODLevels - 1; idx++)
						{
							if (p_master->m_LODMaster.m_LODLevels[idx].m_ObjectCRC == object->m_Checksum)
							{
								// Use next distance
								lod_far_dist = p_master->m_LODMaster.m_LODLevels[idx + 1].m_Distance;
								//printf("Found slave distance %f from master %x\n", lod_far_dist, lod_master_checksum);
							}

							assert(idx != (p_master->m_LODMaster.m_NumLODLevels - 1));
						}

						//printf("Writing slave %x with distance of %f\n", object->m_Checksum, lod_far_dist);
					}
				}


				pOutFile->Write((const char *) &lod_master_checksum, sizeof (unsigned int));
				pOutFile->Write((const char *) &lod_far_dist, sizeof (float));

				// Parent/Child info
				pOutFile->Write((const char *) &object->m_ParentCRC, sizeof( unsigned long ));
				pOutFile->Write((const char *) &object->m_NumChildren, sizeof( int ));
				if (object->m_NumChildren > 0)
				{
					pOutFile->Write((const char *) object->m_ChildCRCs, sizeof(unsigned long) * object->m_NumChildren);
				}

				pOutFile->Write((const char*) &center, 3 * sizeof( float ));
				pOutFile->Write((const char*) &radius, sizeof( float ));
			}

			// Write out the material's checksum
			mat_checksum = mesh->m_MatChecksum;
			pOutFile->Write((const char*) &mat_checksum, sizeof( unsigned long ));

			mesh->m_Flags |= mesh->m_ShadowFlags;			
			if( material->m_Pass > 0 )
			{	
				mesh->m_Flags |= NxObject::mNO_SHADOW;
			}			
			mesh->SetWibbleFlags();

			if (object->m_Flags & NxObject::mBILLBOARD)
			{
				mesh->m_Flags |= MESHFLAG_BILLBOARD;
				if (object->m_BillboardType == NxObject::BBT_AXIAL_ALIGNED)
				{
					mesh->m_Flags |= MESHFLAG_AXIAL;
				}
				else
				{
					mesh->m_Flags &= ~MESHFLAG_AXIAL;
				}
			}

			// set the short-axis billboard flag
			if (mesh->m_Flags & MESHFLAG_AXIAL)
			{
				Mth::Vector vert0,vert1,vert2, edgevec0, edgevec1, axis;
				axis.Set(object->m_PivotAxis[0], object->m_PivotAxis[1], object->m_PivotAxis[2], 0.0f);
				vert0[0] = vert_list[mesh->m_VertStrip[0].m_Index].m_Pos[0];
				vert0[1] = vert_list[mesh->m_VertStrip[0].m_Index].m_Pos[1];
				vert0[2] = vert_list[mesh->m_VertStrip[0].m_Index].m_Pos[2];
				vert1[0] = vert_list[mesh->m_VertStrip[1].m_Index].m_Pos[0];
				vert1[1] = vert_list[mesh->m_VertStrip[1].m_Index].m_Pos[1];
				vert1[2] = vert_list[mesh->m_VertStrip[1].m_Index].m_Pos[2];
				vert2[0] = vert_list[mesh->m_VertStrip[2].m_Index].m_Pos[0];
				vert2[1] = vert_list[mesh->m_VertStrip[2].m_Index].m_Pos[1];
				vert2[2] = vert_list[mesh->m_VertStrip[2].m_Index].m_Pos[2];
				edgevec0 = (vert1 - vert0);
				edgevec1 = (vert2 - vert0);
				if (fabs(Mth::DotProduct(edgevec0, axis)) >
					fabs(Mth::DotProduct(edgevec1, axis)))
				{
					mesh->m_Flags |= MESHFLAG_SHORTAXIS;
				}
			}

			// Write out the flags which describe the object's properties
			pOutFile->Write((const char*) &mesh->m_Flags, sizeof( int ));

			// for mesh version 2 onwards, we calculate bounding volumes for individual meshes
			// (otherwise the object info gets used)
			if (vPS2_MESH_VERSION_NUMBER >= 2)
			{

				// calculate bounding box for the mesh
				min[0] = min[1] = min[2] = 1e30f;
				max[0] = max[1] = max[2] = -1e30f;
				num_verts = mesh->m_NumStripVerts;
				for (i=0; i<num_verts; i++)
				{
					vertex[0] = vert_list[mesh->m_VertStrip[i].m_Index].m_Pos[0];
					vertex[1] = vert_list[mesh->m_VertStrip[i].m_Index].m_Pos[1];
					vertex[2] = vert_list[mesh->m_VertStrip[i].m_Index].m_Pos[2];

					if (vertex[0] < min[0])
						min[0] = vertex[0];
					if (vertex[1] < min[1])
						min[1] = vertex[1];
					if (vertex[2] < min[2])
						min[2] = vertex[2];

					if (vertex[0] > max[0])
						max[0] = vertex[0];
					if (vertex[1] > max[1])
						max[1] = vertex[1];
					if (vertex[2] > max[2])
						max[2] = vertex[2];
				}

				// calculate bounding sphere for the mesh
				center = (max+min)*0.5f;
				radius = 0.0f;
				for (i=0; i<num_verts; i++)
				{
					vertex[0] = vert_list[mesh->m_VertStrip[i].m_Index].m_Pos[0];
					vertex[1] = vert_list[mesh->m_VertStrip[i].m_Index].m_Pos[1];
					vertex[2] = vert_list[mesh->m_VertStrip[i].m_Index].m_Pos[2];
					len = (vertex - center).Length();
					if (len > radius)
						radius = len;
				}
			}

			// Write out bounding box data
			if (vPS2_MESH_VERSION_NUMBER >= 3)
			{
				// object box details
				for (i=0; i<3; i++)
				{
					half_vec[i] = 0.5f * (object->m_BoundingBox.m_Max[i] - object->m_BoundingBox.m_Min[i]);
				}
				pOutFile->Write((const char*) &half_vec, 3 * sizeof(float));

				// mesh box details
				half_vec = 0.5f * (max - min);
				pOutFile->Write((const char*) &half_vec, 3 * sizeof(float));
			}
			else
			{
				pOutFile->Write((const char*) &min, 3 * sizeof( float ));
				pOutFile->Write((const char*) &max, 3 * sizeof( float ));
			}

			// Write out bounding sphere data
			pOutFile->Write((const char*) &center, 3 * sizeof( float ));
			pOutFile->Write((const char*) &radius, sizeof( float ));

			// pass number
			if (vPS2_MESH_VERSION_NUMBER >= 5)
			{
				// GJ:  we want to use the original pass number, 
				// not the actual pass number...  this is because
				// unused passes get stripped out of the list
				NxMaterialPass* pPass = &material->m_Passes[0];
				pOutFile->Write((const char*) &pPass->m_OrigPass, sizeof( int ));
			}

			// material name
			if (vPS2_MESH_VERSION_NUMBER >= 6)
			{
				pOutFile->Write((const char*) &material->m_materialName, sizeof( int ));
			}

			num_verts = mesh->m_NumStripVerts;
			pOutFile->Write((const char*) &num_verts, sizeof( int ));

			// loop over verts, writing out data for each			
			NxVertex *vert, *prev_vert, *prev_prev_vert;				
			unsigned char red, green, blue, alpha;
			int adc, j;
			Mth::Vector p0,p1,p2,n0,n1,n2,n,cp;
			float min_u, max_u, mid_u, min_v, max_v, mid_v, tcu, tcv;
			float mesh_min_u, mesh_max_u, mesh_min_v, mesh_max_v;
			sint16 tcu16,tcv16;
			mesh_min_u = mesh_min_v = 1000000.0f;
			mesh_max_u = mesh_max_v = -1000000.0f;
			prev_vert = NULL;
			vert = NULL;
			prev_prev_vert = NULL;
			for( i = 0; i < num_verts; i++ )
			{
				// if it's a new substrip, calculate texture reduction values
				if ((mesh->m_Flags & NxObject::mTEXTURED) &&
					((i==0) ||
					 (i<=num_verts-3 && !mesh->m_VertStrip[i].m_StripVert && !mesh->m_VertStrip[i+1].m_StripVert)))
				{
					j=i;
					min_u = min_v = 1000000.0f;
					max_u = max_v = -1000000.0f;

					do
					{
						vert = &vert_list[mesh->m_VertStrip[j].m_Index];
						tcu = vert->m_TexCoord[0][0];
						tcv = vert->m_TexCoord[0][1];
						if (tcu < min_u)
							min_u = tcu;
						if (tcu > max_u)
							max_u = tcu;
						if (tcv < min_v)
							min_v = tcv;
						if (tcv > max_v)
							max_v = tcv;
						j++;
					} while (!((j==num_verts) ||
							   (j==num_verts-1 && !mesh->m_VertStrip[j].m_StripVert) ||
							   (j<=num_verts-2 && !mesh->m_VertStrip[j].m_StripVert && !mesh->m_VertStrip[j+1].m_StripVert)));

					mid_u = (min_u + max_u) * 0.5f;
					mid_u = (float)(int)(mid_u + ((mid_u > 0.0f) ? 0.5 : -0.5));
					mid_v = (min_v + max_v) * 0.5f;
					mid_v = (float)(int)(mid_v + ((mid_v > 0.0f) ? 0.5 : -0.5));
				}

				// special treatment for billboards
				if (object->m_Flags & NxObject::mBILLBOARD)
				{
					if ((i&3)==0)
					{
						billboard_verts[0][0] = vert_list[mesh->m_VertStrip[i+0].m_Index].m_Pos[0];
						billboard_verts[0][1] = vert_list[mesh->m_VertStrip[i+0].m_Index].m_Pos[1];
						billboard_verts[0][2] = vert_list[mesh->m_VertStrip[i+0].m_Index].m_Pos[2];
						billboard_verts[1][0] = vert_list[mesh->m_VertStrip[i+1].m_Index].m_Pos[0];
						billboard_verts[1][1] = vert_list[mesh->m_VertStrip[i+1].m_Index].m_Pos[1];
						billboard_verts[1][2] = vert_list[mesh->m_VertStrip[i+1].m_Index].m_Pos[2];
						billboard_verts[2][0] = vert_list[mesh->m_VertStrip[i+2].m_Index].m_Pos[0];
						billboard_verts[2][1] = vert_list[mesh->m_VertStrip[i+2].m_Index].m_Pos[1];
						billboard_verts[2][2] = vert_list[mesh->m_VertStrip[i+2].m_Index].m_Pos[2];
						billboard_verts[3][0] = vert_list[mesh->m_VertStrip[i+3].m_Index].m_Pos[0];
						billboard_verts[3][1] = vert_list[mesh->m_VertStrip[i+3].m_Index].m_Pos[1];
						billboard_verts[3][2] = vert_list[mesh->m_VertStrip[i+3].m_Index].m_Pos[2];

						billboard_texcoords[0][0] = vert_list[mesh->m_VertStrip[i+0].m_Index].m_TexCoord[0][0];
						billboard_texcoords[0][1] = vert_list[mesh->m_VertStrip[i+0].m_Index].m_TexCoord[0][1];
						billboard_texcoords[1][0] = vert_list[mesh->m_VertStrip[i+1].m_Index].m_TexCoord[0][0];
						billboard_texcoords[1][1] = vert_list[mesh->m_VertStrip[i+1].m_Index].m_TexCoord[0][1];
						billboard_texcoords[2][0] = vert_list[mesh->m_VertStrip[i+2].m_Index].m_TexCoord[0][0];
						billboard_texcoords[2][1] = vert_list[mesh->m_VertStrip[i+2].m_Index].m_TexCoord[0][1];
						billboard_texcoords[3][0] = vert_list[mesh->m_VertStrip[i+3].m_Index].m_TexCoord[0][0];
						billboard_texcoords[3][1] = vert_list[mesh->m_VertStrip[i+3].m_Index].m_TexCoord[0][1];
					}
				}

				// advance vertex queue and get new vertex pointer
				prev_prev_vert = prev_vert;
				prev_vert = vert;
				vert = &vert_list[mesh->m_VertStrip[i].m_Index];

				// Write out first texture coordinate set
				// and maintain min & max u & v over whole mesh
				if( mesh->m_Flags & NxObject::mTEXTURED )
				{
					if (!(object->m_Flags & NxObject::mBILLBOARD))
					{
						tcu = vert->m_TexCoord[0][0];
						tcv = vert->m_TexCoord[0][1];
					}
					else if (!(object->m_BillboardType == NxObject::BBT_AXIAL_ALIGNED))
					{
						switch (i&3)
						{
						case 0:
							tcu = billboard_texcoords[1][0];
							tcv = billboard_texcoords[1][1];
							break;
						case 1:
							tcu = billboard_texcoords[0][0];
							tcv = billboard_texcoords[0][1];
							break;
						case 2:
							tcu = billboard_texcoords[3][0];
							tcv = billboard_texcoords[3][1];
							break;
						case 3:
							tcu = billboard_texcoords[2][0];
							tcv = billboard_texcoords[2][1];
							break;
						}
					}
					else
					{
						Mth::Vector edgevec0, edgevec1, axis;
						axis.Set(object->m_PivotAxis[0], object->m_PivotAxis[1], object->m_PivotAxis[2], 0.0f);
						edgevec0 = (billboard_verts[1] - billboard_verts[0]);
						edgevec1 = (billboard_verts[2] - billboard_verts[0]);
						if (fabs(Mth::DotProduct(edgevec0, axis)) <
							fabs(Mth::DotProduct(edgevec1, axis)))
						{
							switch (i&3)
							{
							case 0:
								tcu = billboard_texcoords[1][0];
								tcv = billboard_texcoords[1][1];
								break;
							case 1:
								tcu = billboard_texcoords[0][0];
								tcv = billboard_texcoords[0][1];
								break;
							case 2:
								tcu = billboard_texcoords[3][0];
								tcv = billboard_texcoords[3][1];
								break;
							case 3:
								tcu = billboard_texcoords[2][0];
								tcv = billboard_texcoords[2][1];
								break;
							}
						}
						else
						{
							switch (i&3)
							{
							case 0:
								tcu = billboard_texcoords[1][0];
								tcv = billboard_texcoords[1][1];
								break;
							case 1:
								tcu = billboard_texcoords[3][0];
								tcv = billboard_texcoords[3][1];
								break;
							case 2:
								tcu = billboard_texcoords[0][0];
								tcv = billboard_texcoords[0][1];
								break;
							case 3:
								tcu = billboard_texcoords[2][0];
								tcv = billboard_texcoords[2][1];
								break;
							}
						}
					}

					if ((material->m_Passes[0].m_AddressModeU == 0)	// if not clamped in u
						&& !( mesh->m_Flags & NxObject::mSKINNED ))	// and not skinned
					{
						tcu -= mid_u;
					}
					if( mesh->m_Flags & NxObject::mSKINNED )
					{
						tcu16 = (sint16)(tcu*4096.0f);
						pOutFile->Write((const char*) &tcu16, sizeof(sint16));
					}
					else
					{
						pOutFile->Write((const char*) &tcu, sizeof( float ));
					}
					if (tcu < mesh_min_u)
					{
						mesh_min_u = tcu;
					}
					if (tcu > mesh_max_u)
					{
						mesh_max_u = tcu;
					}

					if ((material->m_Passes[0].m_AddressModeV == 0)	// if not clamped in v
						&& !( mesh->m_Flags & NxObject::mSKINNED ))	// and not skinned
					{
						tcv -= mid_v;
					}
					if( mesh->m_Flags & NxObject::mSKINNED )
					{
						tcv16 = (sint16)(tcv*4096.0f);
						pOutFile->Write((const char*) &tcv16, sizeof(sint16));
					}
					else
					{
						pOutFile->Write((const char*) &tcv, sizeof( float ));
					}
					if (tcv < mesh_min_v)
					{
						mesh_min_v = tcv;
					}
					if (tcv > mesh_max_v)
					{
						mesh_max_v = tcv;
					}
				}

				// Write out color
				if( mesh->m_Flags & NxObject::mCOLORED )
				{
					red = (unsigned char) (( vert->m_Color[0] * 255.0f ) + 0.5f );
					green = (unsigned char) (( vert->m_Color[1] * 255.0f ) + 0.5f );
					blue = (unsigned char) (( vert->m_Color[2] * 255.0f ) + 0.5f );
					// Scale alpha down for PS2 to be between 0 and 128
					alpha = (unsigned char) (( vert->m_Color[3] * 128.0f ) + 0.5f );
					pOutFile->Write((const char*) &red, sizeof( char ));
					pOutFile->Write((const char*) &green, sizeof( char ));
					pOutFile->Write((const char*) &blue, sizeof( char ));
					pOutFile->Write((const char*) &alpha, sizeof( char ));
				}				

				// Write out normal
				if( mesh->m_Flags & NxObject::mNORMALS )
				{					
					sint16 coord;
					coord = (sint16)(vert->m_Normal[0] * 32767.0f);
					if (vert->m_Normal[2] < 0.0f)
					{
						coord |= 0x0001;
					}
					else
					{
						coord &= 0xFFFE;
					}
					pOutFile->Write((const char*) &coord, sizeof(sint16));
					coord = (sint16)(vert->m_Normal[1] * 32767.0f);
					pOutFile->Write((const char*) &coord, sizeof(sint16));

					//coord = (sint16)(vert->m_Normal[2] * 32767.0f);
					//pOutFile->Write((const char*) &coord, sizeof(sint16));
					//coord = 0;
					//pOutFile->Write((const char*) &coord, sizeof(sint16));	// padding
				}

				if( mesh->m_Flags & NxObject::mSKINNED )
				{
					// sort the weights, then use only the first 3.
					vert->SortVertexWeights();
					vert->NormalizeVertexWeights(3);

					sint16 weight;
					weight = (sint16)(vert->m_Weight[0] * 32767.0f);
					pOutFile->Write((const char*) &weight, sizeof(sint16));
					weight = (sint16)(vert->m_Weight[1] * 32767.0f);
					pOutFile->Write((const char*) &weight, sizeof(sint16));
					//weight = (sint16)(vert->m_Weight[2] * 32767.0f);
					//pOutFile->Write((const char*) &weight, sizeof(sint16));

					uint8 pad=0;
					pOutFile->Write((const char*) &vert->m_WeightedIndex[0], sizeof(uint8));
					pOutFile->Write((const char*) &vert->m_WeightedIndex[1], sizeof(uint8));
					pOutFile->Write((const char*) &vert->m_WeightedIndex[2], sizeof(uint8));
					pOutFile->Write((const char*) &pad,						 sizeof(uint8));


					// XXXXXXXXXXXXXXX
					if ( vert->m_MeshScalingWeight[0] != 0.0f )
					{
						Utils::Assert( mp_weight_map_vertices != NULL, "weight map verts were uninitialized" );
						Utils::Assert( m_num_weight_map_vertices == skinned_vertex_index, "weight map index mismatch" );

 						NxVertex* pVertex = &mp_weight_map_vertices[m_num_weight_map_vertices];
						*pVertex = *vert;
						m_num_weight_map_vertices++;

//						int weight0 = vert->m_MeshScalingWeightedIndex[0];
//						int weight1 = vert->m_MeshScalingWeightedIndex[1];
//						int weight2 = vert->m_MeshScalingWeightedIndex[2];
//						printf( "\t%d, %d, %d, // vertex %d\n", weight0, weight1, weight2, skinned_vertex_index );
					}

#if 0
					if ( vert->m_MeshScalingWeight[0] != 0.0f )
					{
						printf( "\t%d, %d, %d, // vertex %d\n", weight0, weight1, weight2, skinned_vertex_index );
						printf( "\t%ff, %ff, %ff, // vertex %d\n", weight0, weight1, weight2, skinned_vertex_index );
					}
#endif
					// XXXXXXXXXXXXXXX

				}

				// Write out position
				if (!(object->m_Flags & NxObject::mBILLBOARD))
				{
					if (mesh->m_Flags & NxObject::mSKINNED)
					{
						sint16 coord;
						coord = (sint16)(vert->m_Pos[0] * SUB_INCH_PRECISION);
						pOutFile->Write((const char*) &coord, sizeof(sint16));
						coord = (sint16)(vert->m_Pos[1] * SUB_INCH_PRECISION);
						pOutFile->Write((const char*) &coord, sizeof(sint16));
						coord = (sint16)(vert->m_Pos[2] * SUB_INCH_PRECISION);
						pOutFile->Write((const char*) &coord, sizeof(sint16));
					}
					else
					{
						pOutFile->Write((const char*) &vert->m_Pos[0], sizeof( float ));
						pOutFile->Write((const char*) &vert->m_Pos[1], sizeof( float ));
						pOutFile->Write((const char*) &vert->m_Pos[2], sizeof( float ));
					}
				}
				else	// special treatment for billboards
				{
					Mth::Vector pvw, pvl, edgevec0, edgevec1, axis, centre;
					centre = 0.25f * (billboard_verts[0] + billboard_verts[1] + billboard_verts[2] + billboard_verts[3]);
					pvw[0] = object->m_PivotPos[0];
					pvw[1] = object->m_PivotPos[1];
					pvw[2] = -object->m_PivotPos[2];
					pvl = pvw - centre;
					switch (i&3)
					{
					case 0:	// world pivot
						pOutFile->Write((const char*) &pvw[0], sizeof( float ));
						pOutFile->Write((const char*) &pvw[1], sizeof( float ));
						pOutFile->Write((const char*) &pvw[2], sizeof( float ));
						break;
					case 1: // dimensions
						axis.Set(object->m_PivotAxis[0], object->m_PivotAxis[1], object->m_PivotAxis[2], 0.0f);
						float half_edge0, half_edge1;
						edgevec0 = (billboard_verts[1] - billboard_verts[0]);
						edgevec1 = (billboard_verts[2] - billboard_verts[0]);
						//if (fabs(Mth::DotProduct(edgevec0, axis)) <
						//	fabs(Mth::DotProduct(edgevec1, axis)))
						//{
						//	half_edge0 = 0.5f * edgevec0.Length();
						//	half_edge1 = 0.5f * edgevec1.Length();
						//}
						//else
						//{
						//	half_edge0 = 0.5f * edgevec0.Length();
						//	half_edge1 = 0.5f * edgevec1.Length();
						//}
						half_edge0 = 0.5f * edgevec0.Length();
						half_edge1 = 0.5f * edgevec1.Length();
						pOutFile->Write((const char*) &half_edge0, sizeof( float ));
						pOutFile->Write((const char*) &half_edge1, sizeof( float ));
						float zero;
						zero = 0.0f;
						pOutFile->Write((const char*) &zero, sizeof( float ));
						break;
					case 2:	// local pivot
						pOutFile->Write((const char*) &pvl[0], sizeof( float ));
						pOutFile->Write((const char*) &pvl[1], sizeof( float ));
						pOutFile->Write((const char*) &pvl[2], sizeof( float ));
						break;
					case 3:	// axis
						pOutFile->Write((const char*) &object->m_PivotAxis[0], sizeof( float ));
						pOutFile->Write((const char*) &object->m_PivotAxis[1], sizeof( float ));
						pOutFile->Write((const char*) &object->m_PivotAxis[2], sizeof( float ));
						break;
					}
				}

				if( mesh->m_VertStrip[i].m_StripVert )
				{
					adc = vADC_ON;
					
					// TODO:  need to handle the geom version, as well
					if( object->m_Flags & NxObject::mSKINNED )
					{
						Dbg_Assert( i != 0 && i != 1 );

						unsigned long cas_face_flags = mesh->FindCASFaceFlags( 
							mesh->m_VertStrip[i-2].m_Index,
							mesh->m_VertStrip[i-1].m_Index,
							mesh->m_VertStrip[i].m_Index );
						
						if ( cas_face_flags != 0 )
						{
							// GENERATES SOME TEST DATA
					//		static int xxx = 0;
					//		xxx++;
					//		cas_face_flags = ( xxx & 0x1 );
							
							if ( !object->AddCASData( cas_face_flags, skinned_vertex_index ) )
							{
								printf( "ERROR:  Increase number of cas data from %d!\n", NxObject::vMAXCASDATA );
								return false;
							}
						}
					}
				}
				else
				{
					adc = vADC_OFF;
				}

				// put wibble index in upper halfword of ADC word
				if( mesh->m_Flags & NxObject::mVCWIBBLE )
				{
					adc |= vert->m_WibbleIndex << 16;
				}

				// get verts and normals
				if (i>=2)
				{
					p0.Set(prev_prev_vert->m_Pos[0], prev_prev_vert->m_Pos[1], prev_prev_vert->m_Pos[2], 1.0f);
					n0.Set(prev_prev_vert->m_Normal[0], prev_prev_vert->m_Normal[1], prev_prev_vert->m_Normal[2], 0.0f);
				}
				else
				{
					p0.Set(0,0,0,1);
					n0.Set(0,0,0,0);
				}
				if (i>=1)
				{
					p1.Set(prev_vert->m_Pos[0], prev_vert->m_Pos[1], prev_vert->m_Pos[2], 1.0f);
					n1.Set(prev_vert->m_Normal[0], prev_vert->m_Normal[1], prev_vert->m_Normal[2], 0.0f);
				}
				else
				{
					p1.Set(0,0,0,1);
					n1.Set(0,0,0,0);
				}
				p2.Set(vert->m_Pos[0], vert->m_Pos[1], vert->m_Pos[2], 1.0f);
				n2.Set(vert->m_Normal[0], vert->m_Normal[1], vert->m_Normal[2], 0.0f);
				//n = n0+n1+n2;
				//cp = Mth::CrossProduct(p1-p0, p2-p0);
				//printf("p0: (%g,%g,%g)\n", p0[0], p0[1], p0[2]);
				//printf("p1: (%g,%g,%g)\n", p1[0], p1[1], p1[2]);
				//printf("p2: (%g,%g,%g)\n", p2[0], p2[1], p2[2]);
				//printf("n0: (%g,%g,%g)\n", n0[0], n0[1], n0[2]);
				//printf("n1: (%g,%g,%g)\n", n1[0], n1[1], n1[2]);
				//printf("n2: (%g,%g,%g)\n", n2[0], n2[1], n2[2]);
				//printf("n:  (%g,%g,%g)\n", n[0], n[1], n[2]);
				//printf("cp: (%g,%g,%g)\n", cp[0], cp[1], cp[2]);
				//printf("adc is %s\n", adc&0x8000 ? "set" : "clear");
				//printf("normals are%s included\n", (mesh->m_Flags & NxObject::mNORMALS) ? "" : "n't");

				// set 'sense' bit for backface culling
				if ((mesh->m_Flags & NxObject::mSS_NORMALS) &&
					(Mth::DotProduct(n0+n1+n2, Mth::CrossProduct(p1-p0, p2-p0)) > 0.0f))
				{
					adc |= 0x00000080;
				}

				skinned_vertex_index++;
				if (mesh->m_Flags & NxObject::mSKINNED)
				{
					pOutFile->Write((const char*) &adc, sizeof(uint16));			
				}
				else
				{
					pOutFile->Write((const char*) &adc, sizeof(uint32));			
				}
			}

			// set texture compression flag
			if ((mesh_min_u >= -8.0f          ) &&
				(mesh_max_u <=  7.99975585938f) &&
				(mesh_min_v >= -8.0f          ) &&
				(mesh_max_v <=  7.99975585938f))
			{
				p_mesh->Flags = MESHFLAG_ST16;
			}
			else
			{
				p_mesh->Flags = 0;
			}
		}	
	}

	// Write out hierarchy data, if any
	int num_hierarchy_objects = m_scene->m_Model.nBones;
	assert(num_hierarchy_objects < 256);
	pOutFile->Write((const char*) &num_hierarchy_objects, sizeof( int ));

	//printf("\nNum Bones %d\n", num_hierarchy_objects);

	for (uint8 bone_idx = 0; bone_idx < num_hierarchy_objects; bone_idx++) {
		uint32 cur_checksum = m_scene->m_Model.bones[bone_idx].crc;
		uint32 parent_checksum;
		NxObject *p_cur_object = m_scene->FindObject(cur_checksum);
		int16 parent_index = -1;
		//NxObject *p_parent_object = NULL;

		assert(p_cur_object);
		parent_checksum = p_cur_object->m_ParentCRC;
		if (parent_checksum)
		{
			//p_parent_object = m_scene->FindObject(parent_checksum);
			//assert(p_parent_object);
		
			// Find parent index
			for (int i = 0; i < num_hierarchy_objects; i++)
			{
				if (parent_checksum == m_scene->m_Model.bones[i].crc)
				{
					parent_index = i;
					break;
				}
			}

			assert(parent_index >= 0);
		}

		// Write the data
		pOutFile->Write((const char*) &cur_checksum, sizeof( int ));
		pOutFile->Write((const char*) &parent_checksum, sizeof( int ));
		pOutFile->Write((const char*) &parent_index, sizeof( short ));
		pOutFile->Write((const char*) &bone_idx, sizeof( char ));

		// Padding
		uint32 pad = 0;
		pOutFile->Write((const char*) &pad, sizeof( char ));
		pOutFile->Write((const char*) &pad, sizeof( int ));

		// Write the matrix
		pOutFile->Write((const char*) &m_scene->m_Model.bones[bone_idx].mat, 16 * sizeof(float));

#if 0
		printf("Bone %d\n", bone_idx);
		printf("Checksum %x\n", cur_checksum);
		printf("Parent Checksum %x\n", parent_checksum);
		printf("Parent Index %x\n", parent_index);
		Mth::Matrix mat = m_scene->m_Model.bones[bone_idx].mat;
		printf("[ %f %f %f %f]\n", mat[0][0], mat[0][1], mat[0][2], mat[0][3]);
		printf("[ %f %f %f %f]\n", mat[1][0], mat[1][1], mat[1][2], mat[1][3]);
		printf("[ %f %f %f %f]\n", mat[2][0], mat[2][1], mat[2][2], mat[2][3]);
		printf("[ %f %f %f %f]\n", mat[3][0], mat[3][1], mat[3][2], mat[3][3]);
#endif
	}

	return true;
}

#define OUTPUT_COLL_BSP

bool	PS2Converter::SaveCollisionData( char* path )
{
	bool success = false;

	IoUtils::CVirtualOutputFile theOutputFile;

	// 10-meg buffer
	theOutputFile.Init(10*1024*1024);

	int i, j, num_obj, version;

	const float one_f = 1.0f;
	const int zero = 0;
	const int one = 1;

	printf( "Collision Data...." );

#ifdef OUTPUT_COLL_BSP
	// Reserve space for BSP nodes
	NxCollBSPNode **p_node_array = new NxCollBSPNode *[m_scene->m_NumCollBSPTreeNodes];
	int node_array_index = 0;
	int node_array_offset = 0;
#endif // OUTPUT_COLL_BSP

	version = vPS2_COLLISION_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	// Number of collision objects
	num_obj = m_scene->m_NumObjects;
	int non_collision_objects = 0;
//	for( i = 0; i < num_obj; i++ )
//	{
//		if (m_scene->m_Objects[i].m_NumCollVerts == 0)
//		{
//			non_collision_objects++;
//		}
//	}
	int num_collision_objects = num_obj - non_collision_objects;
	theOutputFile.Write((const char*) &num_collision_objects, sizeof( int ));

	// Calculate the size of the vert and face arrays
	int total_num_verts_large = 0, total_num_verts_small = 0;
	int total_num_faces_large = 0, total_num_faces_small = 0;
	float box_diff = 0.0f;
	for ( i = 0; i < num_obj; i++ )
	{
		NxObject* object;
		object = &m_scene->m_Objects[i];

		// Skip empty objects
//		if (object->m_NumCollVerts == 0)
//			continue;

		// Figure out if we can use 16-bit verts and write result out
		object->m_CollUseFixedVerts = GENERATE_FIXED_POINT_COLLISION;
		int fixed_diff;
		for (int axis = 0; axis < 3; axis++)
		{
			fixed_diff = (int) (((object->m_CollisionBoundingBox.m_Max[axis] - object->m_CollisionBoundingBox.m_Min[axis]) * COLLISION_SUB_INCH_PRECISION)); //+ 0.5f); Lets not round it here
			if (fixed_diff > 0xFFFF)
			{
				object->m_CollUseFixedVerts = 0;
				break;
			}

			for( j = 0; j < object->m_NumCollVerts; j++ )
			{
				// We don't know why this condition exists, but we need to make sure it doesn't happen here.
				if (object->mp_CollVerts[j].m_Pos[axis] < object->m_CollisionBoundingBox.m_Min[axis])
				{
					object->m_CollUseFixedVerts = 0;
					if (box_diff < (object->m_CollisionBoundingBox.m_Min[axis] - object->mp_CollVerts[j].m_Pos[axis]))
					{
						box_diff = (object->m_CollisionBoundingBox.m_Min[axis] - object->mp_CollVerts[j].m_Pos[axis]);
					}
					//printf("Collision object %x has verts outside of bounding box (diff %f)...\n", object->m_Checksum, object->mp_CollVerts[j].m_Pos[axis] - object->m_CollisionBoundingBox.m_Min[axis]);
					//break;
				}
			}
		}

		if (object->m_CollUseFixedVerts)
		{
			total_num_verts_small += m_scene->m_Objects[i].m_NumCollVerts;
#if !GENERATE_FIXED_POINT_COLLISION
			total_num_verts_small += m_scene->m_Objects[i].m_NumCollVerts % 2;		// Make sure we are 128-bit aligned
#endif
		}
		else
		{
			total_num_verts_large += m_scene->m_Objects[i].m_NumCollVerts;
		}

		if (m_scene->m_Objects[i].UseFaceSmall())
		{
			total_num_faces_small += m_scene->m_Objects[i].m_NumCollFaces;
		} else {
			total_num_faces_large += m_scene->m_Objects[i].m_NumCollFaces;
		}
	}
	if (box_diff > 0.0f)
	{
		//printf("Found bounding box error %f\n", box_diff);
	}

	// Output the size of the vert and face arrays
	int sum_verts = total_num_verts_large + total_num_verts_small;
	theOutputFile.Write((const char*) &sum_verts, sizeof( int ));
	theOutputFile.Write((const char*) &total_num_faces_large, sizeof( int ));
	theOutputFile.Write((const char*) &total_num_faces_small, sizeof( int ));

	// Padding
	theOutputFile.Write((const char*) &total_num_verts_large, sizeof( int ));
	theOutputFile.Write((const char*) &total_num_verts_small, sizeof( int ));
	theOutputFile.Write((const char*) &zero, sizeof( int ));

	int master_vert_idx = 0, master_face_idx = 0, master_face_offset = 0, master_vert_offset = 0;
	int master_large_vert_offset = 0, master_small_vert_offset = total_num_verts_large * NxObject::sVertexLargeSize;
	for( i = 0; i < num_obj; i++ )
	{
		NxObject* object;
		unsigned short num_verts, num_faces;

		object = &m_scene->m_Objects[i];
			
		// Skip empty objects
//		if (object->m_NumCollVerts == 0)
//			continue;

		// Write out the flags which describe the object's properties
		theOutputFile.Write((const char*) &object->m_Checksum, sizeof( unsigned long ));			
		
		// Padding for m_Flags
		theOutputFile.Write((const char*) &zero, sizeof( short ));
		
		num_verts = object->m_NumCollVerts;
		theOutputFile.Write((const char*) &num_verts, sizeof( unsigned short ));

		num_faces = object->m_NumCollFaces;
		theOutputFile.Write((const char*) &num_faces, sizeof( unsigned short ));
		
		// m_use_face_small
		if (object->UseFaceSmall())
		{
			theOutputFile.Write((const char*) &one, sizeof( char ));
		} else {
			theOutputFile.Write((const char*) &zero, sizeof( char ));
		}

		// m_use_fixed_verts
		theOutputFile.Write((const char*) &object->m_CollUseFixedVerts, sizeof( char ));

		// Using offset for pointer mp_faces
		//theOutputFile.Write((const char*) &master_face_idx, sizeof( int ));
		theOutputFile.Write((const char*) &master_face_offset, sizeof( int ));

		// Write out bounding box/sphere data
		theOutputFile.Write((const char*) &object->m_CollisionBoundingBox.m_Min, 3 * sizeof( float ));
		theOutputFile.Write((const char*) &one_f, sizeof( float ));		// Vector pad
		theOutputFile.Write((const char*) &object->m_CollisionBoundingBox.m_Max, 3 * sizeof( float ));
		theOutputFile.Write((const char*) &one_f, sizeof( float ));		// Vector pad

#if 0	// Not exporting bounding sphere now
		float center[3], diag[3];
		float radius, diag_length;

		diag_length = 0;
		for( j = 0; j < 3; j++ )
		{
			center[j] = ( object->m_BoundingBox.m_Max[j] + object->m_BoundingBox.m_Min[j] ) / 2;
			diag[j] = ( object->m_BoundingBox.m_Max[j] - object->m_BoundingBox.m_Min[j] );
			diag_length += ( diag[j] * diag[j] );
		}
		diag_length = sqrtf( diag_length );
		radius = diag_length / 2.0f;
		theOutputFile.Write((const char*) &center, 3 * sizeof( float ));
		theOutputFile.Write((const char*) &radius, sizeof( float ));
#endif

		// Using index for pointers mp_verts //_pos and mp_vert_rgba
#if GENERATE_FIXED_POINT_COLLISION
		if (object->m_CollUseFixedVerts)
		{
			theOutputFile.Write((const char*) &master_small_vert_offset, sizeof( int ));
		}
		else
		{
			theOutputFile.Write((const char*) &master_large_vert_offset, sizeof( int ));
		}
#else
		theOutputFile.Write((const char*) &master_vert_offset, sizeof( int ));
		//theOutputFile.Write((const char*) &master_vert_idx, sizeof( int ));
#endif

#ifdef OUTPUT_COLL_BSP
		// Put BSP tree into array and write out first index
		int node_offset = NxCollBSPNode::AssignNodesToArray(object->mp_BSPRoot, p_node_array,
														   node_array_index, node_array_offset);
		Dbg_Assert(node_array_index <= m_scene->m_NumCollBSPTreeNodes);
		theOutputFile.Write((const char*) &node_offset, sizeof( int ));
#else
		// Padding for rest of class
		theOutputFile.Write((const char*) &zero, sizeof( int ));
#endif // OUTPUT_COLL_BSP
		// Intensity index (same as vert index)
		theOutputFile.Write((const char*) &master_vert_idx, sizeof( int ));
		theOutputFile.Write((const char*) &zero, sizeof( int ));

		master_vert_idx += num_verts;
		master_face_idx += num_faces;
		if (object->UseFaceSmall())
		{
			master_face_offset += (num_faces * NxObject::sFaceSmallSize);
		} else {
			master_face_offset += (num_faces * NxObject::sFaceLargeSize);
		}
		if (object->m_CollUseFixedVerts)
		{
			master_vert_offset += (num_verts * NxObject::sVertexSmallSize);
			master_small_vert_offset += (num_verts * NxObject::sVertexSmallSize);

#if !GENERATE_FIXED_POINT_COLLISION
			int align = master_vert_offset % 16;
			if (align)	// Align to 128 bits for now
			{
				master_vert_offset += (16 - align);
			}
#endif
		}
		else
		{
			master_vert_offset += (num_verts * NxObject::sVertexLargeSize);
			master_large_vert_offset += (num_verts * NxObject::sVertexLargeSize);
		}
	}

	// Export array of vert positions and colors
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		vert_list = object->mp_CollVerts;

		for( j = 0; j < object->m_NumCollVerts; j++ )
		{
			NxVertex* vert;	
			unsigned char red, green, blue, alpha, intensity;
			
			vert = &vert_list[j];

			// Calc color values
			red = (unsigned char) (( vert->m_Color[0] * 128.0f ) + 0.5f );
			green = (unsigned char) (( vert->m_Color[1] * 128.0f ) + 0.5f );
			blue = (unsigned char) (( vert->m_Color[2] * 128.0f ) + 0.5f );
			alpha = (unsigned char) (( vert->m_Color[3] * 128.0f ) + 0.5f );
			intensity = (unsigned char)( ( (int)red + (int)green + (int)blue ) / 3 );

			if (object->m_CollUseFixedVerts)
			{
#if !GENERATE_FIXED_POINT_COLLISION		// Output the small ones last for the new format
				for (int axis = 0; axis < 3; axis++)
				{
					float delta = ((vert->m_Pos[axis] - object->m_CollisionBoundingBox.m_Min[axis]) * COLLISION_SUB_INCH_PRECISION); // + 0.5f; Let's not round here
					unsigned short fixed_pos = (unsigned short) delta;	// Convert to integer
					theOutputFile.Write((const char*) &fixed_pos, sizeof( short ));
				}

				// Write out the colors of the verts (potentially used to tint the skater)
				theOutputFile.Write((const char*) &zero, sizeof( char ));
				theOutputFile.Write((const char*) &intensity, sizeof( char ));
#endif
			}
			else
			{
				// Write out positions				
				theOutputFile.Write((const char*) &vert->m_Pos[0], sizeof( float ));
				theOutputFile.Write((const char*) &vert->m_Pos[1], sizeof( float ));
				theOutputFile.Write((const char*) &vert->m_Pos[2], sizeof( float ));	

#if !GENERATE_FIXED_POINT_COLLISION
				// Write out the colors of the verts (potentially used to tint the skater)
				//theOutputFile.Write((const char*) &red, sizeof( char ));
				//theOutputFile.Write((const char*) &green, sizeof( char ));
				theOutputFile.Write((const char*) &zero, sizeof( char ));
				theOutputFile.Write((const char*) &intensity, sizeof( char ));
				theOutputFile.Write((const char*) &blue, sizeof( char ));
				theOutputFile.Write((const char*) &alpha, sizeof( char ));
#endif
			}
		}

#if !GENERATE_FIXED_POINT_COLLISION
		// Pad fixed verts if necessary.  Assumes fixed vert takes up 8 bytes.
		if (object->m_CollUseFixedVerts && (object->m_NumCollVerts % 2))
		{
			theOutputFile.Write((const char*) &zero, sizeof( int ));	// 4 bytes
			theOutputFile.Write((const char*) &zero, sizeof( int ));	// 4 bytes
		}
#endif
	}

#if GENERATE_FIXED_POINT_COLLISION
	// Export array of small vert positions
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		vert_list = object->mp_CollVerts;

		for( j = 0; j < object->m_NumCollVerts; j++ )
		{
			NxVertex* vert;	
			
			vert = &vert_list[j];
			if (object->m_CollUseFixedVerts)
			{
				for (int axis = 0; axis < 3; axis++)
				{
					float delta = ((vert->m_Pos[axis] - object->m_CollisionBoundingBox.m_Min[axis]) * COLLISION_SUB_INCH_PRECISION); // + 0.5f; Let's not round here
					unsigned short fixed_pos = (unsigned short) delta;	// Convert to integer
					theOutputFile.Write((const char*) &fixed_pos, sizeof( short ));
				}
			}
		}
	}

	// Export array of vert colors
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		vert_list = object->mp_CollVerts;

		for( j = 0; j < object->m_NumCollVerts; j++ )
		{
			NxVertex* vert;				
			unsigned char red, green, blue, alpha, intensity;
			
			vert = &vert_list[j];

			// Write out the colors of the verts (potentially used to tint the skater)
			red = (unsigned char) (( vert->m_Color[0] * 128.0f ) + 0.5f );
			green = (unsigned char) (( vert->m_Color[1] * 128.0f ) + 0.5f );
			blue = (unsigned char) (( vert->m_Color[2] * 128.0f ) + 0.5f );
			alpha = (unsigned char) (( vert->m_Color[3] * 128.0f ) + 0.5f );
			intensity = (unsigned char)( ( (int)red + (int)green + (int)blue ) / 3 );
			//theOutputFile.Write((const char*) &red, sizeof( char ));
			//theOutputFile.Write((const char*) &green, sizeof( char ));
			//theOutputFile.Write((const char*) &blue, sizeof( char ));
			//theOutputFile.Write((const char*) &alpha, sizeof( char ));
			theOutputFile.Write((const char*) &intensity, sizeof( char ));
		}	
	}

	// Pad to 32-bit boundary
	int remainder = (master_vert_offset + total_num_verts_large + total_num_verts_small) % 4;
	if (remainder) {
		for (i = 0; i < (4 - remainder); i++)
		{
			theOutputFile.Write((const char*) &zero, sizeof( char ));
		}
	}
#endif

	// Export array of faces
	int total_small_faces = 0;
	int total_large_faces = 0;
	for( i = 0; i < num_obj; i++ )
	{
		NxFace* face_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		face_list = object->mp_CollFaces;

		// Skip empty objects
//		if (object->m_NumCollVerts == 0)
//			continue;

		for( j = 0; j < object->m_NumCollFaces; j++ )
		{
			NxFace* face;
			NxMaterial* material;

			face = &face_list[j];

			// Write out the face's flags
			//theOutputFile.Write((const char*) &face->m_FaceFlags, sizeof( FlagType ));
			unsigned short flags = (unsigned short) face->m_FaceFlags;
			theOutputFile.Write((const char*) &flags, sizeof( unsigned short ));

			// Write out the face's terrain type. We used to write out the material's checksum and we let
			// the game find the terrain type via indirection, but it seems wasteful when we can get it here.
			material = m_scene->GetMaterial( face->m_MatChecksum );
			short terrain = 0;
			if( material )
			{
				terrain = (short) material->m_Terrain;
			}
			theOutputFile.Write((const char *) &terrain, sizeof( short ));

			// Write out the face's vertex indices
			unsigned short vidx0, vidx1, vidx2;
			vidx0 = (unsigned short) object->mp_ToCollVertIdxs[face->m_Vertex[0]];
			vidx1 = (unsigned short) object->mp_ToCollVertIdxs[face->m_Vertex[1]];
			vidx2 = (unsigned short) object->mp_ToCollVertIdxs[face->m_Vertex[2]];

			if (object->UseFaceSmall())
			{
				theOutputFile.Write((const char*) &vidx0, sizeof( unsigned char ));
				theOutputFile.Write((const char*) &vidx1, sizeof( unsigned char ));
				theOutputFile.Write((const char*) &vidx2, sizeof( unsigned char ));
				theOutputFile.Write((const char*) &zero, sizeof( unsigned char ));

				total_small_faces++;
			} else {
				theOutputFile.Write((const char*) &vidx0, sizeof( unsigned short ));
				theOutputFile.Write((const char*) &vidx1, sizeof( unsigned short ));
				theOutputFile.Write((const char*) &vidx2, sizeof( unsigned short ));
				//theOutputFile.Write((const char*) &zero, sizeof( unsigned short ));

				total_large_faces++;
			}

			// Write out the face's material checksum -- used to get at the material's properties
			//theOutputFile.Write((const char*) &face->m_MatChecksum, sizeof( unsigned long ));
		}
	}

	// Pad to 32-bit boundary
	if (total_large_faces % 2) {
		theOutputFile.Write((const char*) &zero, sizeof( short ));
	}

#ifdef OUTPUT_COLL_BSP
	// Export BSP node arrray size
	theOutputFile.Write((const char*) &node_array_offset, sizeof( int ));

	// Export array of BSP nodes
	//printf("\n*** We got %d BSP nodes and leaves\n", node_array_index);
	Dbg_Assert(node_array_index == m_scene->m_NumCollBSPTreeNodes);
	int master_face_index = 0;
	int node_offset_check = 0;
	for( i = 0; i < node_array_index; i++ )
	{
		NxCollBSPNode *p_bsp_node = p_node_array[i];
		NxCollBSPLeaf *p_bsp_leaf = NULL;
		node_offset_check += 8;
		if (p_bsp_node->m_split_axis == 3)
		{
			p_bsp_leaf = static_cast<NxCollBSPLeaf *>(p_bsp_node);
		}

		if (p_bsp_leaf)
		{
			// m_split axis must always be in line with the low byte of m_split_point
			theOutputFile.Write((const char*) &(p_bsp_node->m_split_axis), sizeof( char ));
			theOutputFile.Write((const char*) &zero, sizeof( char ));
			theOutputFile.Write((const char*) &(p_bsp_leaf->m_num_faces), sizeof( unsigned short ));
		} else {
			//theOutputFile.Write((const char*) &zero, sizeof( unsigned short ));
			int iSplitPoint = (int) (p_bsp_node->m_split_point * COLLISION_SUB_INCH_PRECISION);
			iSplitPoint = (iSplitPoint << NxCollBSPNode::NUM_AXIS_BITS) | p_bsp_node->m_split_axis;
			theOutputFile.Write((const char*) &iSplitPoint, sizeof( int ));
			//theOutputFile.Write((const char*) &(p_bsp_node->m_split_point), sizeof( float ));
		}

		if (p_bsp_leaf)
		{
			// Write out index to master face index array
			theOutputFile.Write((const char*) &master_face_index, sizeof( int ));
			master_face_index += p_bsp_leaf->m_num_faces;
		} else {
			// Write out indexes to master node array
			theOutputFile.Write((const char*) &(p_bsp_node->mp_less_branch->m_array_offset), sizeof( int ));
//			theOutputFile.Write((const char*) &(p_bsp_node->mp_greater_branch->m_array_offset), sizeof( int ));
		}
	}

	Dbg_Assert(node_array_offset == node_offset_check);

	// Export array of BSP face indexes
	for( i = 0; i < node_array_index; i++ )
	{
		// Check for leaf
		if (p_node_array[i]->m_split_axis == 3)
		{
			NxCollBSPLeaf *p_bsp_leaf = static_cast<NxCollBSPLeaf *>(p_node_array[i]);
			for (int j = 0; j < p_bsp_leaf->m_num_faces; j++)
			{
				theOutputFile.Write((const char*) &(p_bsp_leaf->mp_face_idx_array[j]), sizeof( unsigned short ));
			}
		}
	}

	if (p_node_array)
	{
		delete p_node_array;
	}
#endif // OUTPUT_COLL_BSP

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}

bool	PS2Converter::SaveScene( char* path )
{
	bool success = false;
	
	IoUtils::CVirtualOutputFile theOutputFile;

	// 10-meg buffer
	theOutputFile.Init(10*1024*1024);

	int version;

	printf( "Scene...." );

	version = vPS2_MATERIAL_VERSION_NUMBER;
	theOutputFile.Write( (const char*) &version, sizeof(int), false );
	version = vPS2_MESH_VERSION_NUMBER;
	theOutputFile.Write( (const char*) &version, sizeof(int), false );
	version = vPS2_VERTEX_VERSION_NUMBER;
	theOutputFile.Write( (const char*) &version, sizeof(int), false );

	if( SaveMaterials( &theOutputFile ) == false )
	{
		goto save_error;
	}

	if( SaveGeometry( &theOutputFile ) == false )
	{
		goto save_error;
	}

	SaveShadowVolumes( &theOutputFile );

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}

float	PS2Converter::GetAverageGroupPriority( int group )
{
	NxMaterial* material;
	Lst::Search< NxMaterial > sh;
	float least, greatest;
	bool first;

	least = 0;
	greatest = 0;
	first = true;
	for( material = sh.FirstItem( m_scene->m_Materials ); material;
			material = sh.NextItem())
	{
		if( material->m_GroupId == group )
		{			
			if( first )
			{
				least = material->m_DrawOrder;
				greatest = material->m_DrawOrder;
				first = false;
			}			
			else
			{
				if( material->m_DrawOrder < least )
				{
					least = material->m_DrawOrder;
				}
				if( material->m_DrawOrder > greatest )
				{
					greatest = material->m_DrawOrder;
				}
			}		
		}
	}

	return (( least + greatest ) / 2 );
}

void	PS2Converter::InitializeTempTextureData( void )
{
	int i;

	for( i = 0; i < m_num_textures; i++ )
	{
		NxTexture* texture;

		texture = &m_textures[i];
		texture->m_LastGroupIndex = vNO_GROUP;
		texture->m_LastGroupId = vNO_GROUP;
	}
}

int		PS2Converter::NumTexturesInGroup( int group )
{
	NxMaterial* material;
	Lst::Search< NxMaterial > sh;
	int i, count;

	// First, initialize the m_LastGroupId variable that we are about to use
	InitializeTempTextureData();

	count = 0;
	for( material = sh.FirstItem( m_scene->m_Materials ); material;
			material = sh.NextItem())
	{
		if( material->m_Invisible )
		{
			continue;
		}
		if( material->m_GroupId == group )
		{
			NxTexture* texture;
			
			// By this time, all we have are one-pass materials
			for( i = 0; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
			{
				texture = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));		

				if(	( texture == NULL ) ||
					( texture->m_Flags & NxTexture::mINVISIBLE ) ||
					( texture->m_LastGroupId == group ))// ensure that a texture is only written to a particular group once
				{
					continue;
				}				
				
				count++;
				texture->m_LastGroupId = group;
			}			
		}
	}

	return count;
}

__int64	PS2Converter::GetBlendParameters( int blend_mode, int fixed )
{
	BlendModes mode;
	__int64 equation, fixed_val;
	int a, b, c, d;	
	
	mode = (BlendModes) blend_mode;
	equation = 0;	
	switch( mode )
	{
		case vBLEND_MODE_GLOSS_MAP:				// Gloss map unsupported (Treat as diffuse)  aml
		case vBLEND_MODE_DIFFUSE:				// ( 0 - 0 ) * 0 + Src
			a = vPARAM_ZERO;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_SRC;
			break;
		case vBLEND_MODE_ADD:					// ( Src - 0 ) * Src + Dst
			a = vPARAM_SRC;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_ADD_FIXED:				// ( Src - 0 ) * Fixed + Dst
			a = vPARAM_SRC;
			b = vPARAM_ZERO;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_SUBTRACT:				// ( 0 - Src ) * Src + Dst
			a = vPARAM_ZERO;
			b = vPARAM_SRC;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_SUB_FIXED:				// ( 0 - Src ) * Fixed + Dst
			a = vPARAM_ZERO;
			b = vPARAM_SRC;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BLEND:					// ( Src - Dst ) * Src + Dst	
			a = vPARAM_SRC;
			b = vPARAM_DST;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BLEND_FIXED:			// ( Src - Dst ) * Fixed + Dst	
			a = vPARAM_SRC;
			b = vPARAM_DST;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_MODULATE:				// ( Dst - 0 ) * Src + 0
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_ZERO;
			break;
		case vBLEND_MODE_MODULATE_FIXED:			// ( Dst - 0 ) * Fixed + 0	
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_FIXED;
			d = vPARAM_ZERO;
			break;
		case vBLEND_MODE_BRIGHTEN:				// ( Dst - 0 ) * Src + Dst
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BRIGHTEN_FIXED:			// ( Dst - 0 ) * Fixed + Dst	
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BLEND_PREVIOUS_MASK:		// ( Src - Dst ) * Dst + Dst
			a = vPARAM_SRC;
			b = vPARAM_DST;
			c = vPARAM_DST;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BLEND_INVERSE_PREVIOUS_MASK:	// ( Dst - Src ) * Dst + Src
			a = vPARAM_DST;
			b = vPARAM_SRC;
			c = vPARAM_DST;
			d = vPARAM_SRC;
			break;
		case vBLEND_MODE_ADD_PREVIOUS_ALPHA:		// ( Src - 0 ) * Dst + Src
			a = vPARAM_SRC;
			b = vPARAM_ZERO;
			c = vPARAM_DST;
			d = vPARAM_SRC;
			break;
	}

	fixed_val = fixed;
	fixed_val <<= 32;

	equation = a | ( b << 2 ) | ( c << 4 ) | ( d << 6 ) | fixed_val;
		
	return equation;
}

int		PS2Converter::GetTargetTextureDataSize( NxTexture* texture )
{
	int i, size, palette_size, texel_data_size;

	size = 0;
	palette_size = 0;
	texel_data_size = 0;
	if( texture->IsPaletted())
	{
		// For PS2 exports, a texture will either have a 32 or 16-bit palette
		if( ( texture->m_PaletteFormat == NxTexture::v32_BIT ) ||
			( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT ))
		{
			palette_size = 4 * texture->m_NumPaletteEntries;
		}
		else
		{
			palette_size = 2 * texture->m_NumPaletteEntries;
		}
	}

	for( i = 0; i <= texture->m_MipLevels; i++ )
	{
		texel_data_size += texture->m_TexelDataSize[i];
	}

	return ( texel_data_size + palette_size );
}

int		PS2Converter::GetTotalVRAMUsed( NxTexture* texture )
{
	int i, BitsPerTexel, PageWidth, PageHeight, TBW, AdjustedHeight, AdjustedWidth, AdjustedBitsPerTexel, NumTexBytes, NumClutBytes, Width, Height;

	switch( texture->m_PixelFormat )
	{
		case NxTexture::v32_BIT:
			BitsPerTexel = 32;
			PageWidth  = 64;
			PageHeight = 32;
			break;
		case NxTexture::v24_BIT:
			if( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT )
			{
				BitsPerTexel = 24;
				PageWidth  = 64;
				PageHeight = 32;
			}
			else
			{
				BitsPerTexel = 16;
				PageWidth  = 64;
				PageHeight = 64;
			}
			break;
		case NxTexture::v16_BIT:
			BitsPerTexel = 16;
			PageWidth  = 64;
			PageHeight = 64;
			break;
		case NxTexture::v8_BIT:
			BitsPerTexel = 8;
			PageWidth  = 128;
			PageHeight = 64;
			break;
		case NxTexture::v4_BIT:
			BitsPerTexel = 4;
			PageWidth  = 128;
			PageHeight = 128;
			break;
		default:
			return 0;
	}

	AdjustedBitsPerTexel = BitsPerTexel;
	if (BitsPerTexel== 24)
	{
		AdjustedBitsPerTexel = 32;
	}

	NumTexBytes = 0;
	NumClutBytes = 0;

	for( i = 0; i <= texture->m_MipLevels; i++ )
	{
		Width = texture->m_Width[i];
		Height = texture->m_Height[i];
		

		// texture buffer width
		TBW = (Width+63) >> 6;
		if (BitsPerTexel<16 && TBW<2)
			TBW = 2;

		// adjusted dimensions based on vram page dimensions
		AdjustedWidth  = Width;
		AdjustedHeight = Height;

		if (AdjustedWidth < PageWidth && AdjustedHeight > PageHeight)
			AdjustedWidth = PageWidth;

		if (AdjustedWidth > PageWidth && AdjustedHeight < PageHeight)
			AdjustedHeight = PageHeight;

		if (TBW<<6 > AdjustedWidth)
			AdjustedWidth = TBW<<6;

		// space needed by texture (repeat for each mip)
		NumTexBytes += ((AdjustedWidth * AdjustedHeight * AdjustedBitsPerTexel >> 3)
					+ 0x1FFF) & 0xFFFFE000;		// round up to 8K boundary
	}

	bool pal_32;

	pal_32 = false;
	if(	( texture->m_PaletteBpp == 32 ) ||
		(	( texture->m_PaletteFormat == NxTexture::v24_BIT ) &&
			( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT )
		))
	{
		pal_32 = true;
	}

	// space needed by clut
	if (BitsPerTexel == 4)
		NumClutBytes = 256;
	else if (BitsPerTexel == 8)
		NumClutBytes = ( pal_32 ? 1024 : 512);

	return (NumClutBytes + NumTexBytes);
}

int		PS2Converter::GetTotalVRAMUsed( NxMaterial* material )
{
	int i, total;
	NxTexture* tex;

	total = 0;
	// By this time, all we have are one-pass materials
	for( i = 0; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
	{			
		tex = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));
		if( tex )
		{
			total += GetTotalVRAMUsed( tex );			
		}
	}

	return total;
}

int	PS2Converter::SaveTextureGroup( IoUtils::CVirtualOutputFile* pOutFile, IoUtils::CVirtualOutputFile* pUsgFile, int group, bool sorted, int index )
{
	int j, k, num_group_textures, group_flags, total, texture_data_size;
	float group_priority;
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;
	char usg_string[256]; 

	num_group_textures = NumTexturesInGroup( group );
	
	// Initialize the m_LastGroupId variable that we are about to use
	InitializeTempTextureData();

	pOutFile->Write((const char*) &group, sizeof( int ));
	group_flags = 0;
	if( sorted )
	{
		group_flags |= mTEX_GROUP_SORTED;
	}
	if( m_sky )
	{
		group_flags |= mTEX_GROUP_SKY;
	}
	
	group_priority = GetAverageGroupPriority( group );
	pOutFile->Write((const char*) &group_flags, sizeof( int ));
	pOutFile->Write((const char*) &group_priority, sizeof( float ));
	pOutFile->Write((const char*) &num_group_textures, sizeof( int ));

	// Clear out texture CRC tracking
	usedCRCs.Clear();

	sprintf( usg_string, "----------- START GROUP %d : %d Textures ------------\n", index, num_group_textures );
	pUsgFile->Write((const char*) usg_string, strlen( usg_string ));
	total = 0;
	for( material = sh.FirstItem( m_scene->m_Materials ); material; material = sh.NextItem())
	{		
		NxTexture* texture;
		int i, max_level;
		unsigned long checksum;		
				
		if( material->m_Invisible )
		{
			continue;
		}
		if( material->m_GroupId != group )
		{
			continue;
		}

		// By this time, all we have are one-pass materials
		for( i = 0; i < material->m_Passes[0].GetNumTextures( Utils::vPLATFORM_PS2 ); i++ )
		{			
			texture = GetTextureByChecksum( material->m_Passes[0].GetTextureChecksum( i, Utils::vPLATFORM_PS2 ));

			if(	( texture == NULL ) ||
				( texture->m_Flags & NxTexture::mINVISIBLE ))
			{
				continue;				
			}

			// ensure that a texture is only written to a particular group once
			if ( texture->m_LastGroupId == group )
			{
				continue;				
			}
			
			// Track this filename and associated CRCs so we can determine duplicate entries
			AddUSGTexture(texture);

			texture->m_LastGroupId = group;

			// save texture flags
			if (vPS2_TEXTURE_VERSION_NUMBER >= 5)
			{
				uint32 flags=0;
				if (material->m_Passes[0].m_Flags & NxMaterialPass::ANIMATED_TEXTURE)
					flags |= TEXFLAG_ANIMATED;
				pOutFile->Write((const char*) &flags, sizeof( unsigned long ));		
			}

			checksum = texture->m_Checksum;
			pOutFile->Write((const char*) &checksum, sizeof( unsigned long ));		
			if( texture->m_Checksum != vINVALID_CHECKSUM )
			{
				int log_width, log_height;
				int pixel_mode, clut_mode, out_bpp, out_pal_bpp;
				
				log_width = GetLog2( texture->m_Width[0] );
				log_height = GetLog2( texture->m_Height[0] );
				pOutFile->Write((const char*) &log_width, sizeof( int ));
				pOutFile->Write((const char*) &log_height, sizeof( int ));

				out_bpp = texture->m_Bpp;
				out_pal_bpp = texture->m_PaletteBpp;
				switch( texture->m_PixelFormat )
				{
					case NxTexture::v32_BIT:
						pixel_mode = vPSMCT32;
						break;
					case NxTexture::v24_BIT:
						if( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT )
						{
							pixel_mode = vPSMCT24;
							out_bpp = 24;						
						}
						else
						{
							pixel_mode = vPSMCT16;
							out_bpp = 16;						
						}
						break;
					case NxTexture::v16_BIT:
						pixel_mode = vPSMCT16;					
						break;
					case NxTexture::v8_BIT:
						pixel_mode = vPSMT8;					
						break;
					case NxTexture::v4_BIT:
						pixel_mode = vPSMT4;					
						break;
					default:
						return total;
				}

				switch( texture->m_PaletteFormat )
				{
					case NxTexture::v32_BIT:
						clut_mode = vPSMCT32; 					
						break;
					case NxTexture::v16_BIT:
						clut_mode = vPSMCT16;					
						break;
					case NxTexture::v24_BIT:
						if( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT )
						{
							clut_mode = vPSMCT32; 
							out_pal_bpp = 32;
						}
						else
						{
							clut_mode = vPSMCT16;
							out_pal_bpp = 16;						
						}
						break;
					default:
						return total;
				}			

				// Get the max mip level
				max_level = texture->m_MipLevels;
				
				// Write out pixel/clut storage modes
				pOutFile->Write((const char*) &pixel_mode, sizeof( int ));
				pOutFile->Write((const char*) &clut_mode, sizeof( int ));			

				if( texture->m_AlreadyExported )
				{
					// A max_level with the highest bit set signifies that no texture
					// data follows
					max_level |= 0x80000000;
					pOutFile->Write((const char*) &max_level, sizeof( int ));
					continue;
				}
				else
				{
					texture->m_AlreadyExported = true;
					pOutFile->Write((const char*) &max_level, sizeof( int ));
				}

				if( texture->IsPaletted())
				{
					// 16-byte align palette data
					pOutFile->Align(16);
	//				cur_pos = pOutFile->TellPos();
	//				if( cur_pos % 16 )
	//				{
	//					pOutFile->SeekPos( 16 - ( cur_pos % 16 ) + pOutFile->TellPos() );
	//				}				

					if( texture->m_PaletteFormat == NxTexture::v32_BIT )
					{
						int m;
						
						
						memcpy( m_tex_buff, texture->m_PaletteData, texture->m_TotalPaletteDataSize );
						
						// PS2's alpha range goes from 0 (fully-transparent) to 128 (fully-opaque)
						// so scale down the standard 0-255 range here
						for( m = 0; m < texture->m_TotalPaletteDataSize; m += 4 )
						{
							unsigned char cur_alpha;

							cur_alpha = m_tex_buff[m + 3];
							m_tex_buff[m + 3] = ( cur_alpha * 128 ) / 255;
						}
						pOutFile->Write((const char*) m_tex_buff, texture->m_TotalPaletteDataSize );
					}
					else
					{
						int size;

						if( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT )
						{
							
							int num_entries;
							unsigned char* src;
							unsigned char* dst;

							// Right now these are the only two we expect
							Dbg_Assert( texture->m_PaletteFormat == NxTexture::v24_BIT );

							// Convert 24-bit palette to 32-bit palette						
							src = (unsigned char*) texture->m_PaletteData;
							dst = (unsigned char*) m_tex_buff;
							num_entries = texture->m_NumPaletteEntries;
							for( j = 0; j < num_entries; j++ )
							{
								*dst++ = *src++;
								*dst++ = *src++;
								*dst++ = *src++;
								*dst++ = 128;	// fully opaque
							}
													
							size = (int) dst - (int) m_tex_buff;
						}
						else
						{
							int num_entries;
							unsigned char* src;
							unsigned short* dst;

							// Right now these are the only two we expect
							Dbg_Assert( texture->m_PaletteFormat == NxTexture::v24_BIT );

							// Convert 24-bit palette to 16-bit palette						
							src = (unsigned char*) texture->m_PaletteData;
							dst = (unsigned short*) m_tex_buff;
							num_entries = texture->m_NumPaletteEntries;
							for( j = 0; j < num_entries; j++ )
							{
								unsigned char red, blue, green;
								unsigned short color;

								red = ( *src++ ) >> 3;
								green = ( *src++ ) >> 3;
								blue = ( *src++ ) >> 3;
								
								color = 0x8000 | red | ( green << 5 ) | ( blue << 10 );
								*dst++ = color;						
							}

							size = (int) dst - (int) m_tex_buff;						
						}

						// Write out the texel data
						pOutFile->Write((const char*) m_tex_buff, size );
					}
				}

				for( j = 0; j <= texture->m_MipLevels; j++ )
				{
					// 16-byte align texture data
					pOutFile->Align(16);
	//				cur_pos = pOutFile->TellPos();
	//				if( cur_pos % 16 )
	//				{
	//					pOutFile->SeekPos( 16 - ( cur_pos % 16 ) + pOutFile->TellPos() );
	//				}				
					
					// 24-bits will be converted to 16bit for PS2 unless overridden by user
					if(	( texture->m_PixelFormat == NxTexture::v24_BIT ) &&
						( !( texture->m_Flags & NxTexture::mFORCE_BYTE_PER_COMPONENT )))
					{
						unsigned char* src;
						unsigned short* dst;
						int num_pixels;

						src = (unsigned char*) texture->m_TexelData[j];
						dst = (unsigned short*) m_tex_buff;
						num_pixels = texture->m_Height[j] * texture->m_Width[j];
						for( k = 0; k < num_pixels; k++ )
						{
							unsigned char red, blue, green;
							unsigned short color;

							red = ( *src++ ) >> 3;
							green = ( *src++ ) >> 3;
							blue = ( *src++ ) >> 3;
							
							color = red | ( green << 5 ) | ( blue << 10 );
							*dst++ = color;						
						}				

						// Write out the texel data
						pOutFile->Write((const char*) m_tex_buff, (int) dst - (int) m_tex_buff );
					}
					else if( texture->m_PixelFormat == NxTexture::v32_BIT )
					{
						int m;
						
						// PS2's alpha range goes from 0 (fully-transparent) to 128 (fully-opaque)
						// so scale down the standard 0-255 range here
						for( m = 0; m < texture->m_TexelDataSize[j]; m += 4 )
						{
							unsigned char cur_alpha;
							cur_alpha = texture->m_TexelData[j][m + 3];
							texture->m_TexelData[j][m + 3] = ( cur_alpha * 128 ) / 255;
						}

						// Write out the texel data
						pOutFile->Write((const char*) texture->m_TexelData[j], texture->m_TexelDataSize[j] );
					}
					else
					{
						// Write out the texel data
						pOutFile->Write((const char*) texture->m_TexelData[j], texture->m_TexelDataSize[j] );
					}					
				}

				texture_data_size = GetTargetTextureDataSize( texture );
				total += texture_data_size;				
				sprintf( usg_string, "%6d bytes %3dx%3d %2d bpp %2d pbpp %d levels (0x%08X) %s pass %d draw_order %f\n",
							GetTargetTextureDataSize( texture ),
							texture->m_Width[0], texture->m_Height[0], out_bpp,
							out_pal_bpp, texture->m_MipLevels + 1,
							texture->m_Checksum, texture->m_Name,
							material->m_Pass, material->m_DrawOrder );

				pUsgFile->Write((const char*) usg_string, strlen( usg_string ));
			}
			else
			{
				int pad;

				pad = -1;
				
				// Write out a header that signifies an empty texture
				pOutFile->Write((const char*) &pad, sizeof( int ));
				pOutFile->Write((const char*) &pad, sizeof( int ));
				pOutFile->Write((const char*) &pad, sizeof( int ));
				pOutFile->Write((const char*) &pad, sizeof( int ));
				pOutFile->Write((const char*) &pad, sizeof( int ));
			}
		}
	}

	return total;
}

bool	PS2Converter::SaveTextureDictionary( char* path, char* usg_path )
{
	bool success = false;
	char usg_string[512];

	int i, num_groups, version;	
	int total_num_textures, total_size;
	
	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init(10*1024*1024);

	IoUtils::CVirtualOutputFile theUsageFile;
	theUsageFile.Init(1*1024*1024);

	printf( "Texture Dictionary...." );
	
	version = vPS2_TEXTURE_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	total_num_textures = 0;
	num_groups = m_NumUnsortedTextureGroups + m_NumSortedTextureGroups;
	
	// If there are no texture groups, export a dummy group with one dummy texture
	if( num_groups == 0 )
	{
		int group, group_flags, num_tex, log_width, log_height, pixel_mode, clut_mode, max_level, color;
		unsigned long checksum;

		num_groups = 1;
		
		// Write out the number of texture groups
		theOutputFile.Write((const char*) &num_groups, sizeof( int ));

		total_num_textures = 1;
		theOutputFile.Write((const char*) &total_num_textures, sizeof( int ));

		group = s_first_unsorted_texture_group;
		theOutputFile.Write((const char*) &group, sizeof( int ));
		
		group_flags = 0;
		if( m_sky )
		{
			group_flags |= mTEX_GROUP_SKY;
		}

		theOutputFile.Write((const char*) &group_flags, sizeof( int ));
		num_tex = 1;
		theOutputFile.Write((const char*) &num_tex, sizeof( int ));
		checksum = 0;
		theOutputFile.Write((const char*) &checksum, sizeof( unsigned long ));
		
		log_width = 0;
		log_height = 0;
		theOutputFile.Write((const char*) &log_width, sizeof( int ));
		theOutputFile.Write((const char*) &log_height, sizeof( int ));

		pixel_mode = vPSMCT32;
		clut_mode = vPSMCT32;
		max_level = 0;
		// Write out pixel/clut storage modes
		theOutputFile.Write((const char*) &pixel_mode, sizeof( int ));
		theOutputFile.Write((const char*) &clut_mode, sizeof( int ));			
		theOutputFile.Write((const char*) &max_level, sizeof( int ));

		// 16-byte align texture data
		theOutputFile.Align(16);
//		cur_pos = theOutputFile.TellPos();
//		if( cur_pos % 16 )
//		{
//			theOutputFile.SeekPos( 16 - ( cur_pos % 16 ) + cur_pos );
//		}				
		
		color = 0x80808080;
		theOutputFile.Write((const char*) &color, sizeof( int ));
	}
	else
	{
		// Write out the number of texture groups
		theOutputFile.Write((const char*) &num_groups, sizeof( int ));

		//  count up and write out total number of textures
		for( i = 0; i < num_groups; i++ )
		{
			int group_id;
			char group_name[128];
			
			sprintf( group_name, "%s_%d", m_scene_name, i );
			group_id = GenerateCRC( group_name );						

			total_num_textures += NumTexturesInGroup( group_id );
		}
		
		theOutputFile.Write((const char*) &total_num_textures, sizeof( int ));

		total_size = 0;
		for( i = 0; i < num_groups; i++ )
		{
			char group_name[128];
			int group_id;
			bool sorted_group;

			sorted_group = false;
			if( i >= m_NumUnsortedTextureGroups )
			{
				sorted_group = true;
			}		

			sprintf( group_name, "%s_%d", m_scene_name, i );
			group_id = GenerateCRC( group_name );			
			
			total_size += SaveTextureGroup( &theOutputFile, &theUsageFile, group_id, sorted_group, i );
		}

		sprintf( usg_string, "-------------------------------------------\n" );
		theUsageFile.Write((const char*) usg_string, strlen( usg_string ));
		sprintf( usg_string, "Total size : %d Bytes\n", total_size );
		theUsageFile.Write((const char*) usg_string, strlen( usg_string ));
	}
	
	WriteUSGDuplicates(&theUsageFile);
	
	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	// break the lock, if necessary
	SetFileAttributes( usg_path, FILE_ATTRIBUTE_NORMAL );
	if ( !theUsageFile.Save( usg_path, IoUtils::vTEXT_MODE ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}

bool	PS2Converter::SaveCASFlags( char* path )
{
	NxObject* object;

	object = get_cas_object();

	// not a create-a-skater object
	if ( !object )
	{
		return false;
	}

	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init(1*1024*1024);

	int version;

	printf( "CAS flags...." );

	version = vPS2_CASFLAGS_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	unsigned int removalMask = 0;
	if ( object->m_Flags & NxObject::mHASCASREMOVEFLAGS )
	{
		removalMask = object->m_CASRemoveFlags;
	}
	theOutputFile.Write((const char*) &removalMask, sizeof( int ));

	theOutputFile.Write((const char*) &object->m_NumCASData, sizeof( int ));

	for ( int i = 0; i < object->m_NumCASData; i++ )
	{
		theOutputFile.Write((const char*) &object->mp_CASData[i].m_Mask, sizeof( unsigned long ));
		theOutputFile.Write((const char*) &object->mp_CASData[i].m_Data, sizeof( unsigned long ));
	}

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		return false;
	}

	return true;
}

bool	PS2Converter::SaveWeightMap( char* path )
{
	NxObject* object;

	object = get_cas_object();

	// not a create-a-skater object
	if ( !object )
	{
		return false;
	}

	if ( !m_weight_map_loaded )
	{
		return false;
	}

	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init(1*1024*1024);

	int version;

	printf( "Mesh scaling weight map...." );

	version = vPS2_WEIGHTMAP_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	Utils::Assert( mp_weight_map_vertices != NULL, "weight map vertices uninitialized?" );

	NxVertex* pVertices = mp_weight_map_vertices;
	int numVertices = m_num_weight_map_vertices;

	theOutputFile.Write((const char*) &numVertices, sizeof( int ));

	for ( int i = 0; i < numVertices; i++ )
	{
		NxVertex* pVertex = &pVertices[i];
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeight[0], sizeof( float ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeight[1], sizeof( float ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeight[2], sizeof( float ));
	}

	for ( int i = 0; i < numVertices; i++ )
	{
		NxVertex* pVertex = &pVertices[i];
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeightedIndex[0], sizeof( unsigned char ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeightedIndex[1], sizeof( unsigned char ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeightedIndex[2], sizeof( unsigned char ));
	}

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		return false;
	}

	return true;
}

int		PS2Converter::GetPlatformFlags( void )
{
	if( gTextureOverride )
	{
		return( 1 << gTextureOverrideMapping[Utils::vPLATFORM_PS2] );
	}
	return mPLAT_PS2;
}





//----------------------------------------------------------------------------------------
// some rather temporary code for producing the geom files...
const int vNUM_GTEXTURES = 6000;
const int vNUM_GMESHES = 6000;
const int vNUM_GGROUPS = 2000;
const int vNUM_GOBJECTCHECKSUMS = 2000;
const int vNUM_GCHILDOBJECTCHECKSUMS = 2000;
const int vNUM_GMASTERLODCHECKSUMS = 2000;
const int vNUM_GSLAVELODCHECKSUMS = 2000;
const int vNUM_GOBJECTLODFARDIST = 2000;
const int vNUM_GCHILDOBJECTLODFARDIST = 2000;
const int vNUM_GSLAVELODFARDIST = 2000;
const int vNUM_GCHILDOBJECTPARENTCHECKSUM = 2000;
const int vNUM_GOBJECTNUMCHILDREN = 2000;
const int vNUM_GOBJECTCHILDRENCRCS = 2000;
const int vNUM_GOBJECTSPHERES = 2000;
const int vNUM_GOBJECTBOXES = 2000;
const int vNUM_GCHILDOBJECTSPHERES = 2000;
const int vNUM_GCHILDOBJECTBOXES = 2000;
const int vNUM_GVCWIBBLEARRAY = 10000;

int GNumGroups,GNumTextures,GNumMaterials,GNumMeshes,GUnpackOffset,GNumObjects,GNumChildObjects,GNumSlaveLODs;
int GNumVerticesThisBuffer,GNumOutputVertices,GRestartTristrip;
int GMeshRegs,GMeshNReg,GMeshPrim,GMeshNLoop,GMeshAddr,GMeshStep;
uint32 *GpColour;
uint8  *GpColour24;
sint32 *GpVertex32, *GpTexCoords32, *GpSkinData, *GpWeights;
sint16 *GpVertex16, *GpTexCoords16, *GpNormal;
uint8 *GBuffer;
NxPs2::sTexture GTextures[vNUM_GTEXTURES];
NxPs2::sMaterial *GMaterials;
NxPs2::sMesh GMeshes[vNUM_GMESHES];
NxPs2::sGroup GGroups[vNUM_GGROUPS];
NxPs2::CGeomNode *GGeomNodes;
uint32 GObjectChecksums[vNUM_GOBJECTCHECKSUMS];
uint32 GChildObjectChecksums[vNUM_GCHILDOBJECTCHECKSUMS];
uint32 GMasterLODChecksums[vNUM_GMASTERLODCHECKSUMS];
uint32 GSlaveLODChecksums[vNUM_GSLAVELODCHECKSUMS];
float GObjectLODFarDist[vNUM_GOBJECTLODFARDIST];
float GChildObjectLODFarDist[vNUM_GCHILDOBJECTLODFARDIST];
float GSlaveLODFarDist[vNUM_GSLAVELODFARDIST];
uint32 GChildObjectParentChecksum[vNUM_GCHILDOBJECTPARENTCHECKSUM];
int GObjectNumChildren[vNUM_GOBJECTNUMCHILDREN];
unsigned long *GObjectChildrenCRCs[vNUM_GOBJECTCHILDRENCRCS];
Mth::Vector GObjectSpheres[vNUM_GOBJECTSPHERES];
Mth::Vector GObjectBoxes[vNUM_GOBJECTBOXES];
Mth::Vector GChildObjectSpheres[vNUM_GCHILDOBJECTSPHERES];
Mth::Vector GChildObjectBoxes[vNUM_GCHILDOBJECTBOXES];
uint32 GVCWibbleArray[vNUM_GVCWIBBLEARRAY];
sint32 GVCWibbleIndex;
Nx::CHierarchyObject *pGHierarchyArray = NULL;
int GNumHierarchyObjects = 0;

void ResetTempGlobals()
{
	// GJ:  Need to reset temp globals
	// in case we're running in batch mode
	// (otherwise the globals have
	// uninitialized data in them, which
	// sometimes messes things up)

	GNumGroups = 0;
	GNumTextures = 0;
	GNumMaterials = 0;
	GNumMeshes = 0;
	GUnpackOffset = 0;
	GNumObjects = 0;
	GNumChildObjects = 0;
	GNumSlaveLODs = 0;
	GNumVerticesThisBuffer = 0;
	GNumOutputVertices = 0;
	GRestartTristrip = 0;
	GMeshRegs = 0;
	GMeshNReg = 0;
	GMeshPrim = 0;
	GMeshNLoop = 0;
	GMeshAddr = 0;
	GMeshStep = 0;
	GpColour = NULL;
	GpColour24 = NULL;
	GpVertex32 = NULL;
	GpTexCoords32 = NULL;
	GpSkinData = NULL;
	GpWeights = NULL;
	GpVertex16 = NULL;
	GpTexCoords16 = NULL;
	GpNormal = NULL;
	GBuffer = NULL;
	memset( GTextures, 0, vNUM_GTEXTURES * sizeof(NxPs2::sTexture) );
	GMaterials = NULL;
	memset( GMeshes, 0, vNUM_GMESHES * sizeof(NxPs2::sMesh) );
	memset( GGroups, 0, vNUM_GGROUPS * sizeof(NxPs2::sGroup) );
	GGeomNodes = 0;
	memset( GObjectChecksums, 0, vNUM_GOBJECTCHECKSUMS * sizeof(uint32) );
	memset( GChildObjectChecksums, 0, vNUM_GCHILDOBJECTCHECKSUMS * sizeof(uint32) );
	memset( GMasterLODChecksums, 0, vNUM_GMASTERLODCHECKSUMS * sizeof(uint32) );
	memset( GSlaveLODChecksums, 0, vNUM_GSLAVELODCHECKSUMS * sizeof(uint32) );
	memset( GObjectLODFarDist, 0, vNUM_GOBJECTLODFARDIST * sizeof(float) );
	memset( GChildObjectLODFarDist, 0, vNUM_GCHILDOBJECTLODFARDIST * sizeof(float) );
	memset( GSlaveLODFarDist, 0, vNUM_GSLAVELODFARDIST * sizeof(float) );
	memset( GChildObjectParentChecksum, 0, vNUM_GCHILDOBJECTPARENTCHECKSUM * sizeof(uint32) );
	memset( GObjectNumChildren, 0, vNUM_GOBJECTNUMCHILDREN * sizeof(int) );
	memset( GObjectChildrenCRCs, 0, vNUM_GOBJECTCHILDRENCRCS * sizeof(unsigned long*) );
	memset( GObjectSpheres, 0, vNUM_GOBJECTSPHERES * sizeof(Mth::Vector) );
	memset( GObjectBoxes, 0, vNUM_GOBJECTBOXES * sizeof(Mth::Vector) );
	memset( GChildObjectSpheres, 0, vNUM_GCHILDOBJECTSPHERES * sizeof(Mth::Vector) );
	memset( GChildObjectBoxes, 0, vNUM_GCHILDOBJECTBOXES * sizeof(Mth::Vector) );
	memset( GVCWibbleArray, 0, vNUM_GVCWIBBLEARRAY * sizeof(uint32) );
	GVCWibbleIndex = 0;
	pGHierarchyArray = NULL;
	GNumHierarchyObjects = 0;
}

#define POS_EPSILON ( 1.0f / 16.0f )				// 1/16 pixel best accuracy.
#define HASH_SIZE (65536*4)
#define HASH_MASK (HASH_SIZE-1)

static int g_h_pos[HASH_SIZE];
static float g_u_pos[65536][3];
static int g_u_num_pos = 0;
static int g_num_pos = 0;

void init_hash_tables( void )
{
	for ( int lp = 0; lp < HASH_SIZE; lp++ )
	{
		g_h_pos[lp] = -1;
	}

	// Reset global pools.
	g_u_num_pos = 0;
	g_num_pos = 0;
}

int get_pos( float x, float y, float z )
{
	int key = ( (*(unsigned int*)&x) ^ ((*(unsigned int*)&y)<<4) ^ ((*(unsigned int*)&z)<<8) ) & HASH_MASK;

	while ( g_h_pos[key] != -1 )
	{
		if ( fabsf( x - g_u_pos[g_h_pos[key]][0] ) < POS_EPSILON )
		{
			if ( fabsf( y - g_u_pos[g_h_pos[key]][1] ) < POS_EPSILON )
			{
				if ( fabsf( z - g_u_pos[g_h_pos[key]][2] ) < POS_EPSILON )
				{
					return g_h_pos[key];
				}
			}
		}

		key++;
		key &= HASH_MASK;
	}

	return -1;
}

void add_pos( float x, float y, float z )
{
	int key = ( (*(unsigned int*)&x) ^ ((*(unsigned int*)&y)<<4) ^ ((*(unsigned int*)&z)<<8) ) & HASH_MASK;

	while ( g_h_pos[key] != -1 )
	{
		key++;
		key &= HASH_MASK;
	}

	g_u_pos[g_u_num_pos][0] = x;
	g_u_pos[g_u_num_pos][1] = y;
	g_u_pos[g_u_num_pos][2] = z;
	g_h_pos[key] = g_u_num_pos;
	g_u_num_pos++;
}

bool	PS2Converter::SaveShadowVolumes( IoUtils::CVirtualOutputFile * p_file )
{
//	// Pad file.
//	int offset = p_file->TellPos();
//	int aligned_offset = ( offset + 31 ) & ~31;
//	for ( int lp = 0; lp < ( aligned_offset - offset ); lp++ )
//	{
//		char zero = 0;
//		p_file->Write( (const char *)&zero, sizeof( char ), false );
//	}

	int i;
	int		num_objects	= m_scene->m_NumObjects;
	int ** mesh_matrix = NULL;
	int * num_mesh = NULL;
	Mesh ** pp_mesh = NULL;
	int total_mesh_groups = 0;
	NxMesh * p_mesh;
	NxMaterial  *p_material;
	int num_meshes;
	int num_verts;

//	printf( "\n%d objects", num_objects );

	init_hash_tables();

	// Stage 1: Create unique vertex lists.
	for( i = 0; i < num_objects; i++ )
	{
//		printf( "\nunique %d of %d (%d,%d,%d,%d)", i, num_objects, g_u_num_pos, g_u_num_nrm, g_u_num_col, g_u_num_uv );
		NxObject*	object		= m_scene->m_Objects + i;
		NxVertex*	vert_list	= object->m_Verts;
		NxFace*		face_list	= object->m_Faces;
		
		// Skip invisible objects.
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		
		// Check this checksum doesn't already clash.
		for( int check = 0; check < i; ++check )
		{
			NxObject* test_object = m_scene->m_Objects + check;
			if( test_object->m_Checksum == object->m_Checksum )
			{
				// Just set the object checksum to be something random for now...
				object->m_Checksum = rand();
			}
		}
		
		num_meshes	= object->m_MeshList.m_NumMeshes[0];
		num_verts	= object->m_NumVerts;

		// Add only referenced vertex data.
//		printf( "\nAdd object %d (%d/%d, worst %d) ", i, g_u_num_nrm, g_num_nrm, g_worst_nrm );
		for( int m = 0; m < num_meshes; ++m )
		{
//			printf( "\nAdd %d", m );
			p_mesh		= object->m_MeshList.m_Meshes[0][m];
			p_material	= m_scene->GetMaterial( p_mesh->m_MatChecksum );

			// Go through each mesh adding new vertices.
			for ( int v = 0; v < p_mesh->m_NumFaces; v++ )
			{
				// Add Position.
				int p[3];
				p[0] = p_mesh->m_Topology[v][0];
				p[1] = p_mesh->m_Topology[v][1];
				p[2] = p_mesh->m_Topology[v][2];

				for ( int pp = 0; pp < 3; pp++ )
				{
					if ( get_pos( object->m_Verts[p[pp]].m_Pos[0], object->m_Verts[p[pp]].m_Pos[1], object->m_Verts[p[pp]].m_Pos[2] ) == -1 )
					{
						add_pos( object->m_Verts[p[pp]].m_Pos[0], object->m_Verts[p[pp]].m_Pos[1], object->m_Verts[p[pp]].m_Pos[2] );
					}
					g_num_pos++;
				}
			}
		}
	}

//	printf( "\nPos: %d %d", g_num_pos, g_u_num_pos );

	for( i = 0; i < num_objects; i++ )
	{
//		printf( "\nunique %d of %d (%d,%d,%d,%d)", i, num_objects, g_u_num_pos, g_u_num_nrm, g_u_num_col, g_u_num_uv );
		NxObject*	object		= m_scene->m_Objects + i;
		NxVertex*	vert_list	= object->m_Verts;
		NxFace*		face_list	= object->m_Faces;
			
		// Skip invisible objects.
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		
		// Check this checksum doesn't already clash.
		for( int check = 0; check < i; ++check )
		{
			NxObject* test_object = m_scene->m_Objects + check;
			if( test_object->m_Checksum == object->m_Checksum )
			{
				// Just set the object checksum to be something random for now...
				object->m_Checksum = rand();
			}
		}

		if( object->m_Flags & NxObject::mSKINNED )
		{

			num_meshes	= object->m_MeshList.m_NumMeshes[0];
			num_verts	= object->m_NumVerts;

//			printf( "\n%d meshes", num_meshes );

			mesh_matrix = new int *[num_meshes];
			num_mesh = new int[num_meshes];
			int * mesh_assigned = new int[num_meshes];
			int num_mesh_groups = 0;

			for( int mm = 0; mm < num_meshes; ++mm ) mesh_matrix[mm] = new int[num_meshes];
			for( int ma = 0; ma < num_meshes; ++ma ) mesh_assigned[ma] = 0;

			// Build face arrays.
			struct sFace
			{
				int f[3];
			};
			sFace **p_face;
			p_face = new sFace *[num_meshes];

			for( int fmesh = 0; fmesh < num_meshes; ++fmesh )
			{
				NxMesh *p_mesh = object->m_MeshList.m_Meshes[0][fmesh];

				p_face[fmesh] = new sFace[p_mesh->m_NumFaces];

				for ( int v0 = 0; v0 < p_mesh->m_NumFaces; v0++ )
				{
					int p0 = p_mesh->m_Topology[v0][0];
					int p1 = p_mesh->m_Topology[v0][1];
					int p2 = p_mesh->m_Topology[v0][2];

					int pp0 = get_pos( object->m_Verts[p0].m_Pos[0], object->m_Verts[p0].m_Pos[1], object->m_Verts[p0].m_Pos[2] );
					int pp1 = get_pos( object->m_Verts[p1].m_Pos[0], object->m_Verts[p1].m_Pos[1], object->m_Verts[p1].m_Pos[2] );
					int pp2 = get_pos( object->m_Verts[p2].m_Pos[0], object->m_Verts[p2].m_Pos[1], object->m_Verts[p2].m_Pos[2] );

					p_face[fmesh][v0].f[0] = pp0;
					p_face[fmesh][v0].f[1] = pp1;
					p_face[fmesh][v0].f[2] = pp2;
				}
			}

			// Check connections..
			for( int mesh0 = 0; mesh0 < num_meshes; ++mesh0 )
			{
				NxMesh *p_mesh0		= object->m_MeshList.m_Meshes[0][mesh0];
//				printf( "\nMesh %d: %d", mesh0, p_mesh0->m_NumFaces );
				//printf( "\n%d meshes", num_meshes );

				num_mesh[mesh0] = 0;

				if ( mesh_assigned[mesh0] ) continue;

				mesh_matrix[num_mesh_groups][num_mesh[num_mesh_groups]] = mesh0;
				num_mesh[num_mesh_groups]++;
				mesh_assigned[mesh0] = 1;

				for( int mesh1 = 0; mesh1 < num_meshes; ++mesh1 )
				{
					if ( mesh0 == mesh1 ) continue;
					if ( mesh_assigned[mesh1] ) continue;

					NxMesh *p_mesh1		= object->m_MeshList.m_Meshes[0][mesh1];
				
					int connected = 0;
					for ( int v0 = 0; v0 < p_mesh0->m_NumFaces; v0++ )
					{
//						int p0 = p_mesh0->m_Topology[v0][0];
//						int p1 = p_mesh0->m_Topology[v0][1];
//						int p2 = p_mesh0->m_Topology[v0][2];
//
//						int pp0 = get_pos( object->m_Verts[p0].m_Pos[0], object->m_Verts[p0].m_Pos[1], object->m_Verts[p0].m_Pos[2] ); 
//						int pp1 = get_pos( object->m_Verts[p1].m_Pos[0], object->m_Verts[p1].m_Pos[1], object->m_Verts[p1].m_Pos[2] ); 
//						int pp2 = get_pos( object->m_Verts[p2].m_Pos[0], object->m_Verts[p2].m_Pos[1], object->m_Verts[p2].m_Pos[2] ); 

						int pp0 = p_face[mesh0][v0].f[0];
						int pp1 = p_face[mesh0][v0].f[1];
						int pp2 = p_face[mesh0][v0].f[2];

						for ( int v1 = 0; v1 < p_mesh1->m_NumFaces; v1++ )
						{
//							int q0 = p_mesh1->m_Topology[v1][0];
//							int q1 = p_mesh1->m_Topology[v1][1];
//							int q2 = p_mesh1->m_Topology[v1][2];
//
//							int qq0 = get_pos( object->m_Verts[q0].m_Pos[0], object->m_Verts[q0].m_Pos[1], object->m_Verts[q0].m_Pos[2] ); 
//							int qq1 = get_pos( object->m_Verts[q1].m_Pos[0], object->m_Verts[q1].m_Pos[1], object->m_Verts[q1].m_Pos[2] ); 
//							int qq2 = get_pos( object->m_Verts[q2].m_Pos[0], object->m_Verts[q2].m_Pos[1], object->m_Verts[q2].m_Pos[2] ); 

							int qq0 = p_face[mesh1][v1].f[0];
							int qq1 = p_face[mesh1][v1].f[1];
							int qq2 = p_face[mesh1][v1].f[2];

							if ( ( pp0 == qq0 ) || ( pp0 == qq1 ) || ( pp0 == qq2 ) ) connected++;
							if ( ( pp1 == qq0 ) || ( pp1 == qq1 ) || ( pp1 == qq2 ) ) connected++;
							if ( ( pp2 == qq0 ) || ( pp2 == qq1 ) || ( pp2 == qq2 ) ) connected++;
						}
					}
					if ( connected )
					{
						mesh_matrix[num_mesh_groups][num_mesh[num_mesh_groups]] = mesh1;
						num_mesh[num_mesh_groups]++;
						mesh_assigned[mesh1] = 1;
					}

//					printf( "\nMesh %2d: Connected %4d times with mesh %d", mesh0, connected, mesh1 );
				}
				num_mesh_groups++;
			}

			delete mesh_assigned;

			pp_mesh = new Mesh*[num_mesh_groups];
			total_mesh_groups = num_mesh_groups;



			// Delete temp face lists.
			for( int dmesh = 0; dmesh < num_meshes; ++dmesh )
			{
				delete p_face[dmesh];
			}
			delete p_face;


			for ( int gr = 0; gr < num_mesh_groups; gr++ )
			{
//				printf( "\nMesh Group %2d:", gr );
//				for ( int it = 0; it < num_mesh[gr]; it++ )
//				{
//					printf( " %2d", mesh_matrix[gr][it] );
//				}

				pp_mesh[gr] = new Mesh; 
				Mesh * mesh = pp_mesh[gr];
				mesh->numVerts = 0;
				mesh->verts = NULL;
				mesh->numFaces = 0;
				mesh->faces = new MeshFace[65536];

				int * vert_used= new int[g_u_num_pos];

				for ( int vv = 0; vv < g_u_num_pos; vv++ ) vert_used[vv] = 0;

				for ( int it = 0; it < num_mesh[gr]; it++ )
				{
					p_mesh		= object->m_MeshList.m_Meshes[0][mesh_matrix[gr][it]];
					p_material	= m_scene->GetMaterial( p_mesh->m_MatChecksum );

					// An array of mesh pointers to meshes of the same material, but different LOD levels.
					NxMesh *mesh_array[8] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }; 
					mesh_array[0] = p_mesh;

					// We need to figure out if there are lower LOD level meshes that correspond to this mesh (i.e. have the same material).
					unsigned int lod_level;
					for( lod_level = 1; lod_level < 8; ++lod_level )
					{
						if( object->m_MeshList.m_NumMeshes[lod_level] == 0 )
							break;

						for( int mitll = 0; mitll < object->m_MeshList.m_NumMeshes[lod_level]; ++mitll )
						{
							if( object->m_MeshList.m_Meshes[lod_level][mitll]->m_MatChecksum == p_mesh->m_MatChecksum )
							{
								// This level contains a corresponding mesh.
								mesh_array[lod_level] = object->m_MeshList.m_Meshes[lod_level][mitll];
								break;
							}
						}
						if( mesh_array[lod_level] == NULL )
						{
							break;
						}
					}

	//				printf( "\nMesh %d: %d faces", m, p_mesh->m_NumFaces );
					int fc = mesh->numFaces;
					for ( int v = 0; v < p_mesh->m_NumFaces; v++ )
					{
						int p0 = p_mesh->m_Topology[v][0]; 
						int p1 = p_mesh->m_Topology[v][1]; 
						int p2 = p_mesh->m_Topology[v][2]; 

						int r0 = get_pos( object->m_Verts[p0].m_Pos[0], object->m_Verts[p0].m_Pos[1], object->m_Verts[p0].m_Pos[2] ); 
						int r1 = get_pos( object->m_Verts[p1].m_Pos[0], object->m_Verts[p1].m_Pos[1], object->m_Verts[p1].m_Pos[2] ); 
						int r2 = get_pos( object->m_Verts[p2].m_Pos[0], object->m_Verts[p2].m_Pos[1], object->m_Verts[p2].m_Pos[2] ); 

	//					printf( "\nPoly %4d %4d %4d   %4d %4d %4d", p0, p1, p2, r0, r1, r2 );

						if ( r0!=r1 && r1!=r2 && r2!=r0 )
						{
	//						mesh->faces[mesh->numFaces].v[0] = p0;
	//						mesh->faces[mesh->numFaces].v[1] = p1;
	//						mesh->faces[mesh->numFaces].v[2] = p2;

							mesh->faces[mesh->numFaces].v[0] = r0;
							mesh->faces[mesh->numFaces].v[1] = r1;
							mesh->faces[mesh->numFaces].v[2] = r2;
							mesh->faces[mesh->numFaces].mesh = mesh_matrix[gr][it];

							vert_used[r0] = 1;
							vert_used[r1] = 1;
							vert_used[r2] = 1;

	//						printf( "\nMade Face: %4d %4d %4d", mesh->faces[mesh->numFaces].v[0], mesh->faces[mesh->numFaces].v[1], mesh->faces[mesh->numFaces].v[2] );
							mesh->numFaces++;
						}
					}
				}

				mesh->numVerts = 0;
				mesh->verts = new MeshVertex[g_u_num_pos];
				int * mesh_remap = new int[g_u_num_pos];
				for ( int vi = 0; vi < g_u_num_pos; vi++ )
				{
					if ( !vert_used[vi] ) continue;

					mesh->verts[mesh->numVerts].x = g_u_pos[vi][0];
					mesh->verts[mesh->numVerts].y = g_u_pos[vi][1];
					mesh->verts[mesh->numVerts].z = g_u_pos[vi][2];
	
					int index = -1;
					for ( int ovi = 0; ovi < object->m_NumVerts; ovi++ )
					{
						if ( get_pos( object->m_Verts[ovi].m_Pos[0], object->m_Verts[ovi].m_Pos[1], object->m_Verts[ovi].m_Pos[2] ) == vi )
						{
							index = object->m_Verts[ovi].m_WeightedIndex[0];
							break;
						}
					}
					if ( index == -1 ) printf( "\nVertex %4d, index %4d", vi, index );
	
					mesh->verts[mesh->numVerts].idx = index;
					mesh_remap[vi] = mesh->numVerts;
					mesh->numVerts++;
				}

				// Now, remap the faces we already added.
				for ( int f = 0; f < mesh->numFaces; f++ )
				{
					mesh->faces[f].v[0] = mesh_remap[mesh->faces[f].v[0]];
					mesh->faces[f].v[1] = mesh_remap[mesh->faces[f].v[1]];
					mesh->faces[f].v[2] = mesh_remap[mesh->faces[f].v[2]];
				}
			}
		}
	}
	
	Mesh * mesh = new Mesh;
	mesh->numFaces = 0;
	mesh->numVerts = 0;
	mesh->faces = NULL;
	mesh->verts = NULL;

//	if( mesh->numFaces )
	if( total_mesh_groups )
	{
//		for ( int vi = 0; vi < mesh->numVerts; vi++ )
//		{
//			bool found = false;
//			for ( int fi = 0; fi < mesh->numFaces; fi++ )
//			{
//				if ( ( mesh->faces[fi].v[0] == vi ) || ( mesh->faces[fi].v[1] == vi ) || ( mesh->faces[fi].v[2] == vi ) )
//				{
//					found = true;
//					break;
//				}
//			}
//			if ( !found ) printf( "\nExtra vert: %d", vi );
//		}

		printf( "Shadow Volumes...." );
		for ( int mm = 0; mm < total_mesh_groups; mm++ )
		{
			printf( "\nMesh Group %2d:", mm );

			Mesh * p_mesh = pp_mesh[mm];
		
			List<int> order;
			List<int> map;
			PMarg arg;

			arg.useedgelength	= true;
			arg.usecurvature	= true;
			arg.protecttexture	= false;
			arg.protectsmooth	= false;
			arg.protectvc		= false;
			arg.lockborder		= false;
			arg.lockselected	= false;

#define MIN_POLY 50
#define MAX_POLY 3000

#define MIN_REDUX 0.4f
#define MAX_REDUX 0.1f

//			printf( "\nsinf 0 %8.3f", sinf( 0.0f ) );
//			printf( "\nsinf 1 %8.3f", sinf( ( 3.141592f / 2.0f ) ) );
//			printf( "\nsinf 0 %8.3f", sinf( ( 3.141592f / 2.0f ) * 0.1f ) );
//			printf( "\nsinf 0 %8.3f", sinf( ( 3.141592f / 2.0f ) * 0.9f ) );

			// Calculate reduction factor.
			float factor;
			if ( p_mesh->numFaces > MIN_POLY )
			{
				if ( p_mesh->numFaces < MAX_POLY )
				{
					factor = MIN_REDUX - ( ( MIN_REDUX - MAX_REDUX ) * sinf( ( ( p_mesh->numFaces - (float)MIN_POLY ) / ( (float)MAX_POLY - (float)MIN_POLY ) ) * ( 3.141592f / 2.0f ) ) );
				}
				else
				{
					float factor = MAX_REDUX;
				}
			}
			else
			{
				factor = MIN_REDUX;
			}

			printf( "%8.3f (before: %4d,%4d) ", factor, p_mesh->numFaces, p_mesh->numVerts );
			ComputeProgressiveMesh( p_mesh, order, map, arg );
			DoProgressiveMesh( p_mesh, order, map, factor, 0, 0 );
			printf( "(after: %4d,%4d) ", p_mesh->numFaces, p_mesh->numVerts );

			for ( int it = 0; it < num_mesh[mm]; it++ )
			{
				printf( " %2d", mesh_matrix[mm][it] );
			}
		}

	
		// Combine back to 1 mesh.
		mesh->numFaces = 0;
		mesh->numVerts = 0;
		for ( int mm = 0; mm < total_mesh_groups; mm++ )
		{
			Mesh * p_mesh = pp_mesh[mm];

			mesh->numFaces += p_mesh->numFaces;
			mesh->numVerts += p_mesh->numVerts;
		}
		mesh->faces = new MeshFace[mesh->numFaces];
		mesh->verts = new MeshVertex[mesh->numVerts];
		int fo = 0;
		int vo = 0;
		for ( int mm = 0; mm < total_mesh_groups; mm++ )
		{
			Mesh * p_mesh = pp_mesh[mm];

			for ( int face = 0; face < p_mesh->numFaces; face++ )
			{
				mesh->faces[fo].v[0] = p_mesh->faces[face].v[0] + vo;
				mesh->faces[fo].v[1] = p_mesh->faces[face].v[1] + vo;
				mesh->faces[fo].v[2] = p_mesh->faces[face].v[2] + vo;
				mesh->faces[fo].mesh = p_mesh->faces[face].mesh;
				fo++;
			}

			for ( int vert = 0; vert < p_mesh->numVerts; vert++ )
			{
				mesh->verts[vo].x = p_mesh->verts[vert].x;
				mesh->verts[vo].y = p_mesh->verts[vert].y;
				mesh->verts[vo].z = p_mesh->verts[vert].z;
				mesh->verts[vo].idx = p_mesh->verts[vert].idx;
				vo++;
			}
		}
		printf( "\nFinal Faces after: %d, %d", mesh->numFaces, mesh->numVerts );

		for ( int fr = 0; fr < mesh->numFaces; fr++ )
		{
			bool remove = false;
			for ( int fd = 0; fd < mesh->numFaces; fd++ )
			{
				if ( fr == fd ) continue;
				if ( ( mesh->faces[fr].v[0] == mesh->faces[fd].v[0] ) &&
					 ( mesh->faces[fr].v[1] == mesh->faces[fd].v[1] ) &&
					 ( mesh->faces[fr].v[2] == mesh->faces[fd].v[2] ) ) remove = true;

				if ( ( mesh->faces[fr].v[0] == mesh->faces[fd].v[1] ) &&
					 ( mesh->faces[fr].v[1] == mesh->faces[fd].v[2] ) &&
					 ( mesh->faces[fr].v[2] == mesh->faces[fd].v[0] ) ) remove = true;
			
				if ( ( mesh->faces[fr].v[0] == mesh->faces[fd].v[2] ) &&
					 ( mesh->faces[fr].v[1] == mesh->faces[fd].v[0] ) &&
					 ( mesh->faces[fr].v[2] == mesh->faces[fd].v[1] ) ) remove = true;
			
				if ( ( mesh->faces[fr].v[0] == mesh->faces[fd].v[2] ) &&
					 ( mesh->faces[fr].v[1] == mesh->faces[fd].v[1] ) &&
					 ( mesh->faces[fr].v[2] == mesh->faces[fd].v[0] ) ) remove = true;
			
				if ( ( mesh->faces[fr].v[0] == mesh->faces[fd].v[1] ) &&
					 ( mesh->faces[fr].v[1] == mesh->faces[fd].v[0] ) &&
					 ( mesh->faces[fr].v[2] == mesh->faces[fd].v[2] ) ) remove = true;
			
				if ( ( mesh->faces[fr].v[0] == mesh->faces[fd].v[0] ) &&
					 ( mesh->faces[fr].v[1] == mesh->faces[fd].v[2] ) &&
					 ( mesh->faces[fr].v[2] == mesh->faces[fd].v[1] ) ) remove = true;
			}
			if ( remove )
			{
				mesh->faces[fr].mesh = mesh->faces[mesh->numFaces-1].mesh;
				mesh->faces[fr].v[0] = mesh->faces[mesh->numFaces-1].v[0];
				mesh->faces[fr].v[1] = mesh->faces[mesh->numFaces-1].v[1];
				mesh->faces[fr].v[2] = mesh->faces[mesh->numFaces-1].v[2];
				mesh->numFaces--;
				fr--;		// Force this entry to be looked at again.
			}
		}
		printf( "\nFaces after duplicate removal: %d, %d", mesh->numFaces, mesh->numVerts );
	}


	///////////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////////

	uint32 version = 0xDEAD1234;
	uint32 bytes = ( mesh->numVerts * 16 ) + ( mesh->numFaces * 12 );
	uint32 rounded_bytes = ( bytes + 31 ) & ~31;
	uint32 zero = 0;

	p_file->Write( (const char *)&version, sizeof( uint32 ), false );
	p_file->Write( (const char *)&mesh->numFaces, sizeof( uint32 ), false );
	p_file->Write( (const char *)&mesh->numVerts, sizeof( uint32 ), false );
	p_file->Write( (const char *)&rounded_bytes, sizeof( uint32 ), false );
	p_file->Write( (const char *)&zero, sizeof( uint32 ), false );
	p_file->Write( (const char *)&zero, sizeof( uint32 ), false );
	p_file->Write( (const char *)&zero, sizeof( uint32 ), false );
	p_file->Write( (const char *)&zero, sizeof( uint32 ), false );

	// Write out shadow volume data.
	// Needs to be here as it needs the skin data remap table.
	if ( mesh->numFaces )
	{
		// Shadow verts.
		for ( int v = 0; v < mesh->numVerts; v++ )
		{
			float x = mesh->verts[v].x;
			float y = mesh->verts[v].y;
			float z = mesh->verts[v].z;
			int	  i = mesh->verts[v].idx << 2;

			// Write out.
			p_file->Write( (const char *)&x, sizeof( float ), false );
			p_file->Write( (const char *)&y, sizeof( float ), false );
			p_file->Write( (const char *)&z, sizeof( float ), false );
			p_file->Write( (const char *)&i, sizeof( int ), false );
//			printf( "\nVert %4d: %8.3f %8.3f %8.3f", v, x, y, z );
		}

		// Shadow faces.
		for ( int f = 0; f < mesh->numFaces; f++ )
		{
			uint16 idx0 = mesh->faces[f].v[0];
			uint16 idx1 = mesh->faces[f].v[1];
			uint16 idx2 = mesh->faces[f].v[2];

			// Write out.
//			printf( "\nFace %4d: %4d %4d %4d", f, idx0, idx1, idx2 );
			p_file->Write( (const char *)&idx0, sizeof( uint16 ), false );
			p_file->Write( (const char *)&idx1, sizeof( uint16 ), false );
			p_file->Write( (const char *)&idx2, sizeof( uint16 ), false );
		}

		// Create neighbor list.
		struct sShadowEdge
		{
			short neighbor[3];
		};

		int num_shadow_faces = mesh->numFaces;
		sShadowEdge *p_edge = new sShadowEdge[num_shadow_faces];
		int s, t;

		// Find neighbor edges.
		for ( s = 0; s < num_shadow_faces; s++ )
		{
			int found01 = 0;
			int found12 = 0;
			int found20 = 0;

			p_edge[s].neighbor[0] = -1;
			p_edge[s].neighbor[1] = -1;
			p_edge[s].neighbor[2] = -1;

			int s0 = mesh->faces[s].v[0];
			int s1 = mesh->faces[s].v[1];
			int s2 = mesh->faces[s].v[2];

			for ( t = 0; t < num_shadow_faces; t++ )
			{
				if ( t == s ) continue;
				//if ( found01 && found12 && found20 ) break;

				int t0 = mesh->faces[t].v[0];
				int t1 = mesh->faces[t].v[1];
				int t2 = mesh->faces[t].v[2];

				if (
					( ( s0 == t0 ) && ( s1 == t1 ) ) ||
					( ( s0 == t1 ) && ( s1 == t2 ) ) ||
					( ( s0 == t2 ) && ( s1 == t0 ) ) ||
					( ( s0 == t0 ) && ( s1 == t2 ) ) ||
					( ( s0 == t2 ) && ( s1 == t1 ) ) ||
					( ( s0 == t1 ) && ( s1 == t0 ) )
				   )
				{
					p_edge[s].neighbor[0] = t;
					found01++;
//					continue;
				}

				if (
					( ( s1 == t0 ) && ( s2 == t1 ) ) ||
					( ( s1 == t1 ) && ( s2 == t2 ) ) ||
					( ( s1 == t2 ) && ( s2 == t0 ) ) ||
					( ( s1 == t0 ) && ( s2 == t2 ) ) ||
					( ( s1 == t2 ) && ( s2 == t1 ) ) ||
					( ( s1 == t1 ) && ( s2 == t0 ) )
				   )
				{
					p_edge[s].neighbor[1] = t;
					found12++;
//					continue;
				}

				if (
					( ( s2 == t0 ) && ( s0 == t1 ) ) ||
					( ( s2 == t1 ) && ( s0 == t2 ) ) ||
					( ( s2 == t2 ) && ( s0 == t0 ) ) ||
					( ( s2 == t0 ) && ( s0 == t2 ) ) ||
					( ( s2 == t2 ) && ( s0 == t1 ) ) ||
					( ( s2 == t1 ) && ( s0 == t0 ) )
				   )
				{
					p_edge[s].neighbor[2] = t;
					found20++;
//					continue;
				}
			}
			// If multiple connecting polys, treat as a neighborless edge.
			if ( found01 > 1 )
			{
				p_edge[s].neighbor[0] = -1;
			}
			if ( found12 > 1 )
			{
				p_edge[s].neighbor[1] = -1;
			}
			if ( found20 > 1 )
			{
				p_edge[s].neighbor[2] = -1;
			}

			// Turn neighborless edges into opposite tris.
			if ( p_edge[s].neighbor[0] == -1 ) p_edge[s].neighbor[0] = s;
			if ( p_edge[s].neighbor[1] == -1 ) p_edge[s].neighbor[1] = s;
			if ( p_edge[s].neighbor[2] == -1 ) p_edge[s].neighbor[2] = s;

//				printf( "\nFace %3d: %3d %3d %3d ( %3d %3d %3d ) %2d", s, mesh->faces[s].v[0], mesh->faces[s].v[1], mesh->faces[s].v[2], p_edge[s].neighbor[0], p_edge[s].neighbor[1], p_edge[s].neighbor[2], mesh->faces[s].mesh );
//				printf( " ( %8.3f %8.3f %8.3f )", ( mesh->verts[mesh->faces[s].v[0]].x + 20.0f ) * 20.0f, ( mesh->verts[mesh->faces[s].v[0]].y + 20.0f ) * 20.0f, ( mesh->verts[mesh->faces[s].v[0]].z + 20.0f ) * 20.0f );
//				printf( " ( %8.3f %8.3f %8.3f )", ( mesh->verts[mesh->faces[s].v[1]].x + 20.0f ) * 20.0f, ( mesh->verts[mesh->faces[s].v[1]].y + 20.0f ) * 20.0f, ( mesh->verts[mesh->faces[s].v[1]].z + 20.0f ) * 20.0f );
//				printf( " ( %8.3f %8.3f %8.3f )", ( mesh->verts[mesh->faces[s].v[2]].x + 20.0f ) * 20.0f, ( mesh->verts[mesh->faces[s].v[2]].y + 20.0f ) * 20.0f, ( mesh->verts[mesh->faces[s].v[2]].z + 20.0f ) * 20.0f );
		}


		for ( s = 0; s < num_shadow_faces; s++ )
		{
//			uint16 n0 = p_edge[s].neighbor[0];
//			uint16 n1 = p_edge[s].neighbor[1];
//			uint16 n2 = p_edge[s].neighbor[2];

			uint16 e0;
			uint16 e1;
			uint16 e2;

			uint16 n0 = mesh->faces[p_edge[s].neighbor[0]].v[0];
			uint16 n1 = mesh->faces[p_edge[s].neighbor[0]].v[1];
			uint16 n2 = mesh->faces[p_edge[s].neighbor[0]].v[2];

			uint16 s0 = mesh->faces[s].v[0];
			uint16 s1 = mesh->faces[s].v[1];
			uint16 s2 = mesh->faces[s].v[2];

			// 0-1
			e0 = mesh->faces[p_edge[s].neighbor[0]].v[0];
			e1 = mesh->faces[p_edge[s].neighbor[0]].v[1];
			e2 = mesh->faces[p_edge[s].neighbor[0]].v[2];

			if ( ( ( s0 == e0 ) && ( s1 == e1 ) ) || ( ( s1 == e0 ) && ( s0 == e1 ) ) )
			{
				n0 = e2;
			} else
			if ( ( ( s0 == e0 ) && ( s1 == e2 ) ) || ( ( s1 == e0 ) && ( s0 == e2 ) ) )
			{
				n0 = e1;
			} else
			if ( ( ( s0 == e1 ) && ( s1 == e2 ) ) || ( ( s1 == e1 ) && ( s0 == e2 ) ) )
			{
				n0 = e0;
			}

			// 1-2
			e0 = mesh->faces[p_edge[s].neighbor[1]].v[0];
			e1 = mesh->faces[p_edge[s].neighbor[1]].v[1];
			e2 = mesh->faces[p_edge[s].neighbor[1]].v[2];

			if ( ( ( s1 == e0 ) && ( s2 == e1 ) ) || ( ( s2 == e0 ) && ( s1 == e1 ) ) )
			{
				n1 = e2;
			} else
			if ( ( ( s1 == e0 ) && ( s2 == e2 ) ) || ( ( s2 == e0 ) && ( s1 == e2 ) ) )
			{
				n1 = e1;
			} else
			if ( ( ( s1 == e1 ) && ( s2 == e2 ) ) || ( ( s2 == e1 ) && ( s1 == e2 ) ) )
			{
				n1 = e0;
			}

			// 2-0
			e0 = mesh->faces[p_edge[s].neighbor[2]].v[0];
			e1 = mesh->faces[p_edge[s].neighbor[2]].v[1];
			e2 = mesh->faces[p_edge[s].neighbor[2]].v[2];

			if ( ( ( s2 == e0 ) && ( s0 == e1 ) ) || ( ( s0 == e0 ) && ( s2 == e1 ) ) )
			{
				n2 = e2;
			} else
			if ( ( ( s2 == e0 ) && ( s0 == e2 ) ) || ( ( s0 == e0 ) && ( s2 == e2 ) ) )
			{
				n2 = e1;
			} else
			if ( ( ( s2 == e1 ) && ( s0 == e2 ) ) || ( ( s0 == e1 ) && ( s2 == e2 ) ) )
			{
				n2 = e0;
			}


//			printf( "\nFace %3d: %3d %3d %3d (%3d %3d %3d) (%3d %3d %3d) (%3d %3d %3d) *** (%3d %3d %3d)", s, mesh->faces[s].v[0], mesh->faces[s].v[1], mesh->faces[s].v[2],
//					 mesh->faces[p_edge[s].neighbor[0]].v[0], mesh->faces[p_edge[s].neighbor[0]].v[1], mesh->faces[p_edge[s].neighbor[0]].v[2],
//					 mesh->faces[p_edge[s].neighbor[1]].v[0], mesh->faces[p_edge[s].neighbor[1]].v[1], mesh->faces[p_edge[s].neighbor[1]].v[2],
//					 mesh->faces[p_edge[s].neighbor[2]].v[0], mesh->faces[p_edge[s].neighbor[2]].v[1], mesh->faces[p_edge[s].neighbor[2]].v[2],
//					n0, n1, n2
//					);

//			pGDFile->Write( (const char *)&p_edge[s].neighbor[0], 2, true );
//			pGDFile->Write( (const char *)&p_edge[s].neighbor[1], 2, true );
//			pGDFile->Write( (const char *)&p_edge[s].neighbor[2], 2, true );
			p_file->Write( (const char *)&n0, sizeof( uint16 ), false );
			p_file->Write( (const char *)&n1, sizeof( uint16 ), false );
			p_file->Write( (const char *)&n2, sizeof( uint16 ), false );
//			printf( "\nEdge %4d: %4d %4d %4d", s, n0, n1, n2 );
		}

		if ( p_edge ) delete p_edge;
	}

	int pad = rounded_bytes - bytes;
	for ( int lp = 0; lp < pad; lp++ )
	{
		p_file->Write( (const char *)&zero, sizeof( char ), false );
	}

	return true;
}

bool	PS2Converter::SaveGeom( char* tex_path, char* scn_path, char* geom_path )
{
	// start all global vars in a known state
	ResetTempGlobals();

	bool success = false;

	// initialise buffer
	GBuffer = (uint8 *)malloc(16*1048576);
	NxPs2::dma::pLoc = GBuffer+4;		// allowing 4 bytes for header

	IoUtils::CVirtualInputFile theTexFile;
	IoUtils::CVirtualInputFile theScnFile;
	IoUtils::CVirtualOutputFile theGeomFile;
	
	if ( !theTexFile.Load( tex_path ) )
	{
		printf("couldn't open tex file for reading\n");
		goto save_error;
	}

	// open scene file
	if ( !theScnFile.Load( scn_path ) )
	{
		printf("couldn't open scene file for reading\n");
		goto save_error;
	}	

	printf( "Geom...." );

	// read version number
	int version, mat_version;
	theTexFile.Read((char *)&version, sizeof( int ));	// texture version
	
	if( LoadTextures( &theTexFile, version ) == false )
	{
		printf("couldn't load textures\n");
		goto save_error;
	}

//	// Skip shadow data.
//	uint32 data;
//	uint32 bytes;
//	theScnFile.Read((char *)&data, sizeof( int ));
//	theScnFile.Read((char *)&data, sizeof( int ));
//	theScnFile.Read((char *)&data, sizeof( int ));
//	theScnFile.Read((char *)&bytes, sizeof( int ));
//	theScnFile.Read((char *)&data, sizeof( int ));
//	theScnFile.Read((char *)&data, sizeof( int ));
//	theScnFile.Read((char *)&data, sizeof( int ));
//	theScnFile.Read((char *)&data, sizeof( int ));
//	for ( int lp = 0; lp < bytes; lp++ )
//	{
//		printf( "Read byte %d of %d\n", lp, bytes );
//		char d8;
//		theScnFile.Read((char *)&d8, sizeof( char ));
//	}

	// read version numbers
	theScnFile.Read((char *)&mat_version, sizeof( int ));	// material version
	theScnFile.Read((char *)&version, sizeof( int ));	// mesh version
	theScnFile.Read((char *)&version, sizeof( int ));	// vertex version

	if( LoadMaterials( &theScnFile, mat_version ) == false )
	{
		printf("couldn't load materials\n");
		goto save_error;
	}

	if( LoadMeshes( &theScnFile ) == false )
	{
		printf("couldn't load meshes\n");
		goto save_error;
	}

	// build the nodes from the data
	CreateNodes();

	// convert all pointers to offsets
	GGeomNodes[0].Preprocess(GBuffer);

	// write header
	((uint32 *)GBuffer)[0] = (uint8 *)GGeomNodes - GBuffer;

	// save all we need
	theGeomFile.Init( 10 * 1024 * 1024 );

	SaveGeom( &theGeomFile );

	// break the lock, if necessary
	SetFileAttributes( geom_path, FILE_ATTRIBUTE_NORMAL );
	if ( !theGeomFile.Save( geom_path ) )
	{
		printf("Couldn't open geom file %s for writing\n", geom_path);
		goto save_error;
	}

	success = true;

save_error:
	// free mem
	free(GBuffer);
	delete [] GMaterials;

	return success;
}



bool	PS2Converter::LoadTextures( IoUtils::CVirtualInputFile* pInputFile, int version )
{
	int i,j,bits_per_texel,palette_size,bits_per_clut_entry,num_textures_this_group;
	int width[7],height[7],TBW[7],num_clut_bytes,num_tex_bytes[7],num_vram_bytes[7];
	int adjusted_width[7], adjusted_height[7], page_width, page_height, adjusted_bits_per_texel;
	uint32 TW,TH,PSM,CPSM,k,TBP[7],next_TBP,CBP,last_CBP, vram_buffer_base, TexCount;
	int MXL;
	NxPs2::sTexture *p_tex;
	NxPs2::sGroup *p_group;
	int total_num_textures;

	pInputFile->Read( (char *)&GNumGroups, sizeof(int) );
	if ( version >= 3 )
		pInputFile->Read( (char *)&total_num_textures, sizeof(int) );

	// start with no textures
	GNumTextures = 0;
	p_tex = GTextures;

	// set buffer base (all pip models will clash, at this stage!)
	vram_buffer_base = 0x2BC0;

	for (i=0,p_group=GGroups; i<GNumGroups; i++,p_group++)
	{
		TexCount = 0;

		// get group checksum
		pInputFile->Read( (char *)&p_group->Checksum, sizeof(int) );

		// get group flags
		if( version >= 0x0002 )
		{
			pInputFile->Read( (char *)&p_group->flags, sizeof(int) );
		}
		else if (p_group->Checksum>=1000 && p_group->Checksum<2000)
		{
			p_group->flags = GROUPFLAG_SKY;
		}
		else
			p_group->flags = 0;

		// get group priority
		if (version >= 4)
		{
			pInputFile->Read((char *)&p_group->Priority, sizeof(float));
		}

		// get num textures in group
		pInputFile->Read( (char *)&num_textures_this_group, sizeof(int) );

		// set vram usage for this group
		p_group->VramStart = vram_buffer_base;
		p_group->VramEnd   = vram_buffer_base + 0x0A20;

		// advance buffer for next group
		vram_buffer_base ^= 0x1E20;

		// initialise base pointers
		next_TBP = p_group->VramStart;
		last_CBP = p_group->VramEnd;

		for (j=0; j<num_textures_this_group; j++,p_tex++)
		{
			p_tex->Flags = 0;

			if (vPS2_TEXTURE_VERSION_NUMBER >= 5)
			{
				pInputFile->Read( (char *)&p_tex->Flags, sizeof(int) );
			}
			pInputFile->Read( (char *)&p_tex->Checksum, sizeof(int) );
			p_tex->GroupChecksum = p_group->Checksum;
			pInputFile->Read( (char *)&TW, sizeof(int) );
			pInputFile->Read( (char *)&TH, sizeof(int) );
			pInputFile->Read( (char *)&PSM, sizeof(int) );
			pInputFile->Read( (char *)&CPSM, sizeof(int) );
			pInputFile->Read( (char *)&MXL, sizeof(int) );

			// bail if it's a non-texture
			if (TW == 0xFFFFFFFF)
				continue;

			// detect duplicated texture, signalled by a negative MXL
			if (MXL<0)
			{
				//printf("duplicate texture %08X\n", p_tex->Checksum);
				MXL &= 0x7FFFFFFF;						// adjust local MXL variable
				p_tex->Flags &= ~TEXFLAG_ORIGINAL;		// signal duplicate
			}
			else
			{
				// not a duplicate, must be the original
				p_tex->Flags |= TEXFLAG_ORIGINAL;
			}

			// quadword-align (original only)
			if (p_tex->Flags & TEXFLAG_ORIGINAL)
			{
				pInputFile->Align(16);
			}

			// bits per texel and palette size
			switch (PSM)
			{
			case PSMCT32:
				bits_per_texel = 32;
				palette_size  = 0;
				break;
			case PSMCT24:
				bits_per_texel = 24;
				palette_size  = 0;
				break;
			case PSMCT16:
				bits_per_texel = 16;
				palette_size  = 0;
				break;
			case PSMT8:
				bits_per_texel = 8;
				palette_size  = 256;
				break;
			case PSMT4:
				bits_per_texel = 4;
				palette_size  = 16;
				break;
			default:
				printf("Unknown PSM %d at index %d in texture file\n", PSM, i);
				return false;
			}

			// adjust bits per texel for 24-bit
			adjusted_bits_per_texel = bits_per_texel;
			if (bits_per_texel == 24)
			{
				adjusted_bits_per_texel = 32;
			}

			// bits per clut entry
			if (bits_per_texel < 16)
				switch (CPSM)
				{
				case PSMCT32:
					bits_per_clut_entry = 32;
					break;
				case PSMCT16:
					bits_per_clut_entry = 16;
					break;
				default:
					printf("Unknown CPSM %d in texture file\n", PSM);
					return false;
				}
			else
				bits_per_clut_entry = 0;

			// calculate texture dimensions
			for (k=0; k<=MXL; k++)
			{
				width[k]  = (TW>=0 ? 1<<TW : 0) >> k;
				height[k] = (TH>=0 ? 1<<TH : 0) >> k;
				TBW[k] = (width[k]+63) >> 6;
				if (bits_per_texel<16 && TBW[k]<2)
					TBW[k] = 2;
			}



			// get page size
			switch (PSM)
			{
			case PSMCT32:
			case PSMCT24:
				page_width  = 64;
				page_height = 32;
				break;
			case PSMCT16:
				page_width  = 64;
				page_height = 64;
				break;
			case PSMT8:
				page_width  = 128;
				page_height = 64;
				break;
			case PSMT4:
				page_width  = 128;
				page_height = 128;
				break;
			default:
				printf("Unknown PSM %d at index %d in texture file\n", PSM, i);
				exit(1);
			}


			// calculate adjusted dimensions based on vram page size
			for (k=0; k<=MXL; k++)
			{
				adjusted_width[k]  = width[k];
				adjusted_height[k] = height[k];

				if (adjusted_width[k] < page_width && adjusted_height[k] > page_height)
				{
					adjusted_width[k] = page_width;
				}

				if (adjusted_width[k] > page_width && adjusted_height[k] < page_height)
				{
					adjusted_height[k] = page_height;
				}	

				if (TBW[j]<<6 > adjusted_width[k])
				{
					adjusted_width[k] = TBW[k]<<6;
				}
			}



			// calculate sizes within file
			num_clut_bytes  = palette_size * bits_per_clut_entry >> 3;
			for (k=0; k<=MXL; k++)
			{
				num_tex_bytes[k]  = width[k] * height[k] * bits_per_texel >> 3;
				num_vram_bytes[k] = adjusted_width[k] * adjusted_height[k] * adjusted_bits_per_texel >> 3;
			}

			// calculate TBP
			TBP[0] = next_TBP;
			for (k=1; k<=MXL; k++)
				TBP[k] = (((TBP[k-1]<<8)+num_vram_bytes[k-1]+0x1FFF) & 0xFFFFE000) >> 8;

			// calculate CBP
			if (bits_per_texel >= 16)
				CBP = last_CBP;
			else if (bits_per_texel == 4)
				CBP = last_CBP - 1;
			else
				CBP = last_CBP - (CPSM==PSMCT32 ? 4 : 2);

			// calculate next TBP
			next_TBP = (((TBP[MXL]<<8)+num_vram_bytes[MXL]+0x1FFF) & 0xFFFFE000) >> 8;

			// report if VRAM would become over-packed
			if (next_TBP > CBP)
			{
				printf("no room for texture %d, group %d, dimensions %dx%dx%d\n",
							j,i,width[0],height[0],bits_per_texel);
			}

			// step past clut data and qword-align (original only)
			if (p_tex->Flags & TEXFLAG_ORIGINAL)
			{
				pInputFile->Skip(num_clut_bytes);
				pInputFile->Align(16);
			}
			
			// loop over mipmaps
			for (k=0; k<=MXL; k++)
			{
				// step past tex data for this level
				if (p_tex->Flags & TEXFLAG_ORIGINAL)
				{
					pInputFile->Skip(num_tex_bytes[k]);
				}
			}

			// offset to improve caching
			for (k=0; k<=MXL; k++)
			{
				if (TexCount & 1)
				{
					if ((bits_per_texel==32 || bits_per_texel==8)
							&& (width[k]  <= (page_width>>1))
							&& (height[k] <= page_height)
						||
						(bits_per_texel==16 || bits_per_texel==4)
							&& (width[k]  <=  page_width)
							&& (height[k] <= (page_height>>1)))
					{
						TBP[k] += 16;
					}
				}
				TexCount++;
			}

			// make entry in texture table
			p_tex->MXL        = MXL;
			p_tex->RegTEX0    = PackTEX0(TBP[0],TBW[0],PSM,TW,TH,(PSM!=PSMCT24),MODULATE,CBP,CPSM,0,0,1);
			p_tex->RegMIPTBP1 = PackMIPTBP1(TBP[1],TBW[1],TBP[2],TBW[2],TBP[3],TBW[3]);
			p_tex->RegMIPTBP2 = PackMIPTBP2(TBP[4],TBW[4],TBP[5],TBW[5],TBP[6],TBW[6]);

			// advance texture base pointer to 1st page after texture
			// and reduce clut base pointer to 1st page before clut (if there was one)
			TBP[0] = next_TBP;
			last_CBP = CBP;
		}

		GNumTextures += num_textures_this_group;
	}

	return true;
}

bool	PS2Converter::LoadMaterials( IoUtils::CVirtualInputFile* pInputFile, int mat_version )
{
	NxPs2::sMaterial *p_mat = NULL;
	NxPs2::sTexture *p_tex = NULL;
	uint32 texture_checksum = 0;
	uint32 group_checksum = 0;
	uint32 MXL = 0;
	uint32 MMAG = 0;
	uint32 MMIN = 0;
	uint32 K = 0;
	uint32 L = 0;
	uint32 num_seqs = 0;
	uint32 seq = 0;
	uint32 num_keys = 0;
	int i = 0;
	int j = 0;

	// get number of materials
	pInputFile->Read( (char *)&GNumMaterials, sizeof(int) );

	// allocate storage for materials
	GMaterials = new NxPs2::sMaterial[GNumMaterials];

	// loop over materials
	for (i=0,p_mat=GMaterials; i<GNumMaterials; i++,p_mat++)
	{
		// get material checksum
		pInputFile->Read( (char *)&p_mat->Checksum, sizeof(int) );

		// The flags were moved here, after the checksum, in version 5 since the parser
		// needs the flag info to correctly parse the material
		if (mat_version >= 5)
		{
			pInputFile->Read( (char *)&p_mat->Flags, sizeof(int) );
		}

		// get Aref
		if (vPS2_MATERIAL_VERSION_NUMBER >= 2)
		{
			if (vPS2_MATERIAL_VERSION_NUMBER >= 3)
			{
				pInputFile->Read((char *)&p_mat->Aref, 4);
			}
			else
			{
				pInputFile->Read((char *)&p_mat->Aref, 1);
			}

		}
		else
		{
			p_mat->Aref = 1;
		}

		if (mat_version >= 5)
		{
			texture_checksum = 0;
			if( p_mat->Flags & NxMaterial::mANIMATED_TEX )
			{
				int frame, num_keyframes;

				pInputFile->Read( (char *)&num_keyframes, sizeof(int) );
				pInputFile->Skip(12);
				for( frame = 0; frame < num_keyframes; frame++ )
				{
					pInputFile->Skip(4);
					if( frame == 0 )
					{
						// get texture checksum
						pInputFile->Read( (char *)&texture_checksum, sizeof(int) );
					}
					else
					{
						pInputFile->Skip(4);
					}
				}

				/*anim_tex = &material->m_Passes[0].m_AnimatedTexture;
				pOutFile->Write((const char*) &anim_tex->m_NumKeyframes, sizeof( int ), false);
				pOutFile->Write((const char*) &anim_tex->m_Period, sizeof( int ), false);
				pOutFile->Write((const char*) &anim_tex->m_Iterations, sizeof( int ), false);
				pOutFile->Write((const char*) &anim_tex->m_Phase, sizeof( int ), false);
				for( j = 0; j < anim_tex->m_NumKeyframes; j++ )
				{
					NxAnimatedTextureKeyframe* frame;

					frame = &anim_tex->m_Keyframes[j];
					pOutFile->Write((const char*) &frame->m_Time, sizeof( unsigned int ), false);

					tex_checksum = material->m_Passes[0].GetTextureChecksum( j, Utils::vPLATFORM_PS2 );
					pOutFile->Write((const char*) &tex_checksum, sizeof( unsigned long ), false);
				}*/				
			}
			else
			{
				// get texture checksum
				pInputFile->Read( (char *)&texture_checksum, sizeof(int) );
			}
		}
		else
		{
			// get texture checksum
			pInputFile->Read( (char *)&texture_checksum, sizeof(int) );
		}

		if (vPS2_MATERIAL_VERSION_NUMBER >= 3)
		{
			// get group checksum
			pInputFile->Read( (char *)&group_checksum, sizeof(int) );
		}

		// get material flags (moved to an earlier spot in version 5)
		if (mat_version < 5)
		{
			pInputFile->Read( (char *)&p_mat->Flags, sizeof(int) );
		}

		// get ALPHA register value
		pInputFile->Read( (char *)&p_mat->RegALPHA, sizeof(uint64) );

		// get CLAMP register value
		if (vPS2_MATERIAL_VERSION_NUMBER > 1)
		{
			uint32 ClampU,ClampV;
			pInputFile->Read( (char *)&ClampU, sizeof(int) );
			pInputFile->Read( (char *)&ClampV, sizeof(int) );
			p_mat->RegCLAMP = PackCLAMP(ClampU,ClampV,0,0,0,0);
		}
		else
			p_mat->RegCLAMP = PackCLAMP(REPEAT,REPEAT,0,0,0,0);

		// step past material colours, currently unsupported
		pInputFile->Skip(36);
//		pInputFile->SeekPos( 36 + pInputFile->TellPos() );

		// copy any uv wibble data
		if (p_mat->Flags & MATFLAG_UV_WIBBLE)
		{
			p_mat->pUVWibbleInfo = (float *)NxPs2::dma::pLoc;
			pInputFile->Read((char *)NxPs2::dma::pLoc, 40);
			NxPs2::dma::pLoc += 40;
		}

		// copy vc wibble mat data
		if (p_mat->Flags & MATFLAG_VC_WIBBLE)
		{
			p_mat->pVCWibbleInfo = (uint32 *)NxPs2::dma::pLoc;

			pInputFile->Read( (char *)&num_seqs, sizeof(uint32) );
			*(uint32 *)NxPs2::dma::pLoc = num_seqs;
			NxPs2::dma::pLoc += 4;

			for (seq=0; seq<num_seqs; seq++)
			{ 
				pInputFile->Read( (char *)&num_keys, sizeof(uint32) );
				*(uint32 *)NxPs2::dma::pLoc = num_keys;
				NxPs2::dma::pLoc += 4;

				pInputFile->Read((char *)NxPs2::dma::pLoc, 4+20*num_keys );	// phase-shift plus keyframes
				NxPs2::dma::pLoc += 4+20*num_keys;
			}
		}

		// if textured...
		if (texture_checksum)
		{
			// resolve texture checksum
			if (vPS2_MATERIAL_VERSION_NUMBER >= 3)
			{
				for (j=0,p_tex=GTextures; j<GNumTextures; j++,p_tex++)
				{
					if (p_tex->Checksum == texture_checksum && p_tex->GroupChecksum == group_checksum)
						break;
				}
			}
			else
			{
				for (j=0,p_tex=GTextures; j<GNumTextures; j++,p_tex++)
				{
					if (p_tex->Checksum == texture_checksum)
						break;
				}
			}
			if (j==GNumTextures)
			{
				printf("couldn't find texture checksum %08X\n", texture_checksum);
				return false;
			}

			// set texture pointer
			p_mat->pTex = p_tex;

			// get mipmap info
			MXL  = p_tex->MXL;
			pInputFile->Read( (char *)&MMAG, sizeof(uint32) );
			pInputFile->Read( (char *)&MMIN, sizeof(uint32) );
			pInputFile->Read( (char *)&K, sizeof(uint32) );
			K = (int) ((*(float *)&K) * 16.0f) & 0xFFF;
			
//			MXL = 0;     // No mipmappping
//			MMIN &= 6;	// Clear bottom bit of MMIN to remove trilinear filtering
//			MMIN |= 1;	// Set bottom bit of MMIN to SET trilinear 
//			MXL = 6;     // Full mipmappping

/*
sTexture members
	uint32 Checksum, GroupChecksum, MXL;
	uint64 RegTEX0, RegMIPTBP1, RegMIPTBP2;

	uint8  *mp_texture_buffer;
	uint32 m_texture_buffer_size;
	uint8  *mp_clut_buffer;
	uint32 m_clut_buffer_size;
*/


			int w = 1 << ((p_tex->RegTEX0 >> 26) & 0xf);
			int h = 1 << ((p_tex->RegTEX0 >> 30) & 0xf);
			int	PSM = ((p_tex->RegTEX0 >> 20) & 0x1f);
			int bpp = 8;
			if (PSM == 0x14)
			{
				bpp = 4;
			}
			int mem = w*h*bpp/8;
//			printf ("%4d x %4d %d bits, %6d\n",w,h,bpp, mem);
			

/*			
			if ( mem <= 1024) // if it will fit in the cache 
			{
				MXL = 2;	// then turn on full mip maps
				//MMIN = 5;   // with trilinear filtering  
				MMIN = 4;   // with bilinear filtering  
			}
*/

/*			
			else
			{
				MXL = 0;	// otherwise no mip mapping
			}
*/						
			if (vPS2_MATERIAL_VERSION_NUMBER >= 2)
				p_mat->RegTEX1 = PackTEX1(0,MXL,MMAG,MMIN,0,0,K);
			else
			{
				pInputFile->Read( (char *)&L, sizeof(uint32) );
				p_mat->RegTEX1 = PackTEX1(0,MXL,MMAG,MMIN,0,L,K);
			}
		}
		// otherwise just step past mipmap info
		else
		{
			p_mat->pTex = 0;
			pInputFile->Skip(12+4*(vPS2_MATERIAL_VERSION_NUMBER==1));
//			pInputFile->SeekPos( 12+4*(vPS2_MATERIAL_VERSION_NUMBER==1) + pInputFile->TellPos() );
		}

		// reflection-map scaling
		if (vPS2_MATERIAL_VERSION_NUMBER >= 4)
		{
			float temp;
			pInputFile->Read( (char *)&temp, sizeof(float) );
			p_mat->RefMapScaleU = (sint16)(temp * 256.0f);
			pInputFile->Read( (char *)&temp, sizeof(float) );
			p_mat->RefMapScaleV = (sint16)(temp * 256.0f);
		}
		else
		{
			p_mat->RefMapScaleU = 3*256;		// 1.7.8 fixed point
			p_mat->RefMapScaleV = 3*256;
		}

	}

	return true;
}

bool	PS2Converter::LoadMeshes( IoUtils::CVirtualInputFile* pInputFile )
{
	int i,j,k,num_mesh_groups,num_meshes_this_group;
	NxPs2::sMesh *p_mesh;
	NxPs2::sMaterial *p_mat;
	NxPs2::sGroup *p_group;
	uint32 material_checksum, group_checksum;
	Mth::Vector sphere;

	GNumMeshes = GNumObjects = GNumChildObjects = GNumSlaveLODs = 0;
	p_mesh = GMeshes;

	// get number of mesh groups; currently should match num texture groups
	pInputFile->Read( (char *)&num_mesh_groups, sizeof(int) );
	Dbg_MsgAssert( num_mesh_groups==GNumGroups, ("mismatch in number of groups\n") );

	// get total number of meshes
	int total_num_meshes;
	if (vPS2_MESH_VERSION_NUMBER >= 4)
		pInputFile->Read( (char *)&total_num_meshes, sizeof(int) );

	// iterate over mesh groups
	// GNumGroups is read as the number of textures loaded (not actual groups)
	// may not match num_mesh_groups if an object was
	// excluded from the list because it was flagged as invisible
	//for (i=0; i<GNumGroups; i++)	// aml
	for(i=0; i<num_mesh_groups; i++)
	{
		// get group checksum and number of meshes
		pInputFile->Read( (char *)&group_checksum, sizeof(uint32) );
		pInputFile->Read( (char *)&num_meshes_this_group, sizeof(uint32) );

		// resolve checksum
		for (j=0,p_group=GGroups; j<GNumGroups; j++,p_group++)
			if (p_group->Checksum == group_checksum)
				break;
		if (j==GNumGroups)
		{
			printf("Couldn't find group checksum #%d\n", group_checksum);
			return false;
		}

		// copy group info
		p_group->Checksum = group_checksum;
		p_group->pMeshes = p_mesh;
		p_group->NumMeshes = num_meshes_this_group;

		// loop over meshes
		for (j=0; j<num_meshes_this_group; j++,p_mesh++)
		{
			unsigned int lod_master_checksum = 0;
			float lod_far_dist = MAX_LOD_DIST;

			unsigned long parent_CRC, *child_CRCs = NULL;
			int num_children;

			// start a subroutine for this mesh
			NxPs2::dma::BeginSub3D();

			// start a ret tag
			NxPs2::dma::BeginTag(NxPs2::dma::ret, 0);

			// avoid having to set VIF1_OFFSET at the top level
			NxPs2::vif::OFFSET(0);

			// get object checksum
			pInputFile->Read( (char *)&p_mesh->Checksum, sizeof(uint32) );

			// get object bounding sphere, for mesh version 2
			if (vPS2_MESH_VERSION_NUMBER >= 2) {
				pInputFile->Read( (char *)&lod_master_checksum, sizeof(unsigned int) );
				pInputFile->Read( (char *)&lod_far_dist, sizeof(float) );

				// Parent/Child info
				pInputFile->Read((char *) &parent_CRC, sizeof( unsigned long ));
				pInputFile->Read((char *) &num_children, sizeof( int ));
				if (num_children > 0)
				{
					child_CRCs = new unsigned long[num_children];
					pInputFile->Read((char *) child_CRCs, sizeof(unsigned long) * num_children);
				}

				pInputFile->Read( (char *)&sphere, sizeof(Mth::Vector) );
			}

			bool master_LOD = (lod_master_checksum == 0);

			// get material checksum and look it up
			pInputFile->Read( (char *)&material_checksum, sizeof(uint32) );
			for (k=0,p_mat=GMaterials; k<GNumMaterials; k++,p_mat++)
				if (p_mat->Checksum == material_checksum)
					break;
			if (k==GNumMaterials)
			{
				printf("couldn't find material with checksum %08X\n", material_checksum);
				return false;
			}

			// set mesh's material
			p_mesh->pMaterial = p_mat;

			// get mesh flags
			uint32 flags;
			pInputFile->Read( (char *)&flags, sizeof(uint32) );
			p_mesh->Flags |= flags;

			// make it active
			p_mesh->Flags |= MESHFLAG_ACTIVE;

			// get bounding volume data
			pInputFile->Read( (char *)&p_mesh->ObjBox, 3*sizeof(float) );
			pInputFile->Read( (char *)&p_mesh->Box,	3*sizeof(float) );
			pInputFile->Read( (char *)&p_mesh->Sphere, 4*sizeof(float) );

			// pass number
			if (vPS2_MESH_VERSION_NUMBER >= 5)
			{
				int pass;
				pInputFile->Read( (char *)&pass, sizeof(int) );
			}

			// material name
			if (vPS2_MESH_VERSION_NUMBER >= 6)
			{
				int materialName;
				pInputFile->Read( (char *)&materialName, sizeof(int) );
			}

			// look up object checksum; add it if not already there
			if (master_LOD)
			{
				if (1 || parent_CRC == 0) {	// Garrett: For now, we aren't going to have children
					for (k=0; k<GNumObjects; k++)
					{
						if (GObjectChecksums[k] == p_mesh->Checksum)
							break;
					}
					if (k==GNumObjects)
					{
						GObjectChecksums[GNumObjects] = p_mesh->Checksum;
						GObjectSpheres[GNumObjects] = sphere;
						GObjectBoxes[GNumObjects] = *(Mth::Vector *)&p_mesh->ObjBox;
						GObjectLODFarDist[GNumObjects] = lod_far_dist;

						GObjectNumChildren[GNumObjects] = num_children;
						GObjectChildrenCRCs[GNumObjects] = child_CRCs;

						GNumObjects++;
					}
				} else {
					for (k=0; k<GNumChildObjects; k++)
					{
						if (GChildObjectChecksums[k] == p_mesh->Checksum)
							break;
					}
					if (k==GNumChildObjects)
					{
						GChildObjectChecksums[GNumChildObjects] = p_mesh->Checksum;
						GChildObjectSpheres[GNumChildObjects] = sphere;
						GChildObjectBoxes[GNumChildObjects] = *(Mth::Vector *)&p_mesh->ObjBox;
						GChildObjectLODFarDist[GNumChildObjects] = lod_far_dist;
						GChildObjectParentChecksum[GNumChildObjects] = parent_CRC;

						GNumChildObjects++;
					}
				}
			} else {
				for (k=0; k<GNumSlaveLODs; k++)
				{
					if (GSlaveLODChecksums[k] == p_mesh->Checksum)
						break;
				}
				if (k==GNumSlaveLODs)
				{
					GSlaveLODChecksums[GNumSlaveLODs] = p_mesh->Checksum;
					GSlaveLODFarDist[GNumSlaveLODs] = lod_far_dist;
					GMasterLODChecksums[GNumSlaveLODs] = lod_master_checksum;
					//printf("Found new slave %x with distance of %f\n", GSlaveLODChecksums[k], GSlaveLODFarDist[k]);
					GNumSlaveLODs++;
				}
			}

			// import vertices of this mesh
			LoadVertices(pInputFile, p_mesh, p_mat, p_group);

			// end the dma tag
			NxPs2::dma::EndTag();


			// store mesh data for wibbled vc's
			if (p_mat->Flags & MATFLAG_VC_WIBBLE)
			{
				uint32 *p_matdata = p_mat->pVCWibbleInfo;

				p_mesh->pVCWibbleInfo = (uint32 *)NxPs2::dma::pLoc;

				// copy num seqs
				int num_seqs = *p_matdata++;
				*(uint32 *)NxPs2::dma::pLoc = num_seqs;
				NxPs2::dma::pLoc += 4;

				// loop over seqs
				for (int seq=0; seq<num_seqs; seq++)
				{
					// set seq pointer
					*(uint32 **)NxPs2::dma::pLoc = p_matdata;
					NxPs2::dma::pLoc += 4;

					// leave space for num verts with this seq
					int *p_num_verts = (int *)NxPs2::dma::pLoc;
					NxPs2::dma::pLoc += 4;

					// loop over all wibbled verts for this mesh, copying verts with this seq
					int num_verts = 0;
					for (int i=0; i<GVCWibbleIndex; i+=2)
					{
						if (GVCWibbleArray[i] == seq)
						{
							*(uint32 **)NxPs2::dma::pLoc = (uint32 *)GVCWibbleArray[i+1];
							NxPs2::dma::pLoc += 4;
							num_verts++;
						}
					}

					// fill in num verts for this seq
					*p_num_verts = num_verts;

					// step material data pointer to next seq
					p_matdata += 5 * (*p_matdata) + 2;
				}
			}

			// align the dma pointer
			NxPs2::dma::Align(0,16);

			// end the model subroutine
			p_mesh->pSubroutine = NxPs2::dma::EndSub3D();
		}

		// add to total number for scene
		GNumMeshes += num_meshes_this_group;
	}

	// Read in hierarchy data, if any
	GNumHierarchyObjects = -1;
	pInputFile->Read((char*) &GNumHierarchyObjects, sizeof( int ));
	if (GNumHierarchyObjects) {
		pGHierarchyArray = new Nx::CHierarchyObject[GNumHierarchyObjects];
		pInputFile->Read((char*) pGHierarchyArray, GNumHierarchyObjects * sizeof( Nx::CHierarchyObject ));
	}

	return true;
}



void PS2Converter::LoadVertices( IoUtils::CVirtualInputFile* pInputFile, NxPs2::sMesh *pMesh, NxPs2::sMaterial *pMat, NxPs2::sGroup *pGroup)
{
	uint REGS, NREG, PRIM, ADDR, STEP;
	NxPs2::sTexture *p_tex;
	int i,num_vertices,ST_offset,colour_offset,normal_offset,skin_offset,XYZ_offset,vertex_size;
	uint8  vertex_data[64];

	// get number of vertices for this mesh
	pInputFile->Read( (char *)&num_vertices, sizeof(int) );

	// skip mesh if it has no vertices
	if (num_vertices==0)
		return;

	// work out giftag fields
	REGS = NxPs2::gs::XYZF2;
	//REGS = NxPs2::gs::XYZ2;
	NREG = 1;
	PRIM = TRISTRIP|ABE|FGE;
	//PRIM = TRISTRIP|ABE;
	ADDR = VU1_ADDR(Proj);

	if (pMesh->Flags & MESHFLAG_COLOURS)
	{
		REGS = REGS<<4 | NxPs2::gs::RGBAQ;
		NREG++;
	}

	if ((pMesh->Flags & MESHFLAG_NORMALS) && (pMat->Flags & MATFLAG_ENVIRONMENT))
	{
		ADDR = VU1_ADDR(Refl);
		REGS = REGS<<4 | NxPs2::gs::ST;
		NREG++;
		PRIM |= TME;
	}
	else if ((pMesh->Flags & MESHFLAG_NORMALS) && !(pMesh->Flags & MESHFLAG_UNLIT))
	{
		REGS = REGS<<4 | NxPs2::gs::NOP;
		NREG ++;
		if (pMesh->Flags & MESHFLAG_TEXTURE)
		{
			REGS = REGS<<4 | NxPs2::gs::ST;
			NREG++;
			PRIM |= TME;
			if (pMesh->pMaterial->Flags & MATFLAG_UV_WIBBLE)
			{
				ADDR = VU1_ADDR(LWibT);
			}
			else
			{
				ADDR = VU1_ADDR(LightT);
			}
		}
		else
		{
			ADDR = VU1_ADDR(Light);
		}
	}
	else if (pMesh->Flags & MESHFLAG_TEXTURE)
	{
		REGS = REGS<<4 | NxPs2::gs::ST;
		NREG++;
		PRIM |= TME;
		if (pMesh->pMaterial->Flags & MATFLAG_UV_WIBBLE)
		{
			ADDR = VU1_ADDR(WibbleT);
		}
		else
		{
			ADDR = VU1_ADDR(PTex);
		}
	}

	if (pMat->Flags & MATFLAG_SMOOTH)
		PRIM |= IIP;

	// override everything for skinned models!
	if (pMesh->Flags & MESHFLAG_SKINNED)
	{
		REGS = NxPs2::gs::XYZ2<<16 | NxPs2::gs::RGBAQ<<12 | NxPs2::gs::NOP<<8 | NxPs2::gs::NOP<<4 | NxPs2::gs::ST;
		NREG = 5;
		PRIM = TRISTRIP|ABE|IIP|TME;
		ADDR = VU1_ADDR(Skin);
	}

	// and override everything for billboards
	if (pMesh->Flags & MESHFLAG_BILLBOARD)
	{
		REGS = NxPs2::gs::ST | NxPs2::gs::RGBAQ<<4 | NxPs2::gs::XYZ2<<8;
		NREG = 3;
		PRIM = TRISTRIP|ABE|IIP|TME|FGE;
		STEP = 12;
		if (pMesh->Flags & MESHFLAG_AXIAL)
		{
			if (pMesh->Flags & MESHFLAG_SHORTAXIS)
			{
				ADDR = VU1_ADDR(SHAB);
			}
			else
			{
				ADDR = VU1_ADDR(LAB);
			}
		}
		else
		{
			ADDR = VU1_ADDR(SCAB);
		}
	}
	else
	{
		STEP = NREG;
	}

	// set giftag clamp flags, used for suppression of texcoord reduction
	if (pMat->RegCLAMP & 0x3)
		ADDR |= 0x1000;
	if (pMat->RegCLAMP & 0xC)
		ADDR |= 0x2000;

	// set single-sided flag
	if (pMesh->Flags & MESHFLAG_SINGLESIDED)
	{
		ADDR |= 0x0800;
	}

	// output GS context for material
	NxPs2::gs::BeginPrim(REL,0,0);
	NxPs2::gs::Reg1(NxPs2::gs::ALPHA_1,	pMat->RegALPHA);
	if ((pMat->RegALPHA & PackALPHA(3,3,0,3,0)) == PackALPHA(2,2,0,0,0)			// if diffuse blend mode
		&& (pMat->Aref <= 1))													// and not using alpha cutoff (except for A=0)
	{
		NxPs2::gs::Reg1(NxPs2::gs::TEST_1, PackTEST(0,0,0,0,0,0,1,ZGEQUAL));	// switch off alpha test
	}
	else
	{
		NxPs2::gs::Reg1(NxPs2::gs::TEST_1, PackTEST(1,AGEQUAL,pMat->Aref,KEEP,0,0,1,ZGEQUAL));
	}

	if (((pMat->RegALPHA & PackALPHA(3,3,0,3,0)) == PackALPHA(2,0,0,1,0)) || 
		((pMat->RegALPHA & PackALPHA(3,3,0,3,0)) == PackALPHA(0,2,0,1,0)))
	{
		NxPs2::gs::Reg1(NxPs2::gs::FOGCOL, PackFOGCOL(0x00,0x00,0x00));
	}

	// texture registers if necessary
	if (pMat->Flags & MATFLAG_TEXTURED)
	{
		p_tex = pMat->pTex;
		NxPs2::gs::Reg1(NxPs2::gs::TEX0_1,		p_tex->RegTEX0);
		NxPs2::gs::Reg1(NxPs2::gs::TEX1_1,		pMat->RegTEX1);
		NxPs2::gs::Reg1(NxPs2::gs::CLAMP_1,		pMat->RegCLAMP);
		if (p_tex->MXL > 0)
		{
			NxPs2::gs::Reg1(NxPs2::gs::MIPTBP1_1,	p_tex->RegMIPTBP1);
			if (p_tex->MXL > 3)
			{
				NxPs2::gs::Reg1(NxPs2::gs::MIPTBP2_1,	p_tex->RegMIPTBP2);
			}
		}
	}
	NxPs2::gs::EndPrim(0);

	// set maximum vu buffer size
	NxPs2::vu1::MaxBuffer = pMesh->Flags&MESHFLAG_SKINNED ? 200 : 300;

	// begin a batch of vertices
	BeginModelMultiPrim(REGS, NREG, PRIM, num_vertices, ADDR, pMesh, STEP);

	// work out vertex size and data offsets
	vertex_size = (pMesh->Flags & MESHFLAG_SKINNED) ? 8 : 16;
	XYZ_offset = skin_offset = normal_offset = colour_offset = ST_offset = 0;
	if (pMesh->Flags & MESHFLAG_SKINNED)
	{
		vertex_size   += 8;
		XYZ_offset    += 8;
	}
	if (pMesh->Flags & MESHFLAG_NORMALS)
	{
		vertex_size   += 4;
		XYZ_offset    += 4;
		skin_offset   += 4;
	}
	if (pMesh->Flags & MESHFLAG_COLOURS)
	{
		vertex_size   += 4;
		XYZ_offset    += 4;
		skin_offset   += 4;
		normal_offset += 4;
	}
	if (pMesh->Flags & MESHFLAG_TEXTURE)
	{
		int size = (pMesh->Flags & MESHFLAG_SKINNED) ? 4 : 8;
		vertex_size   += size;
		XYZ_offset    += size;
		skin_offset   += size;
		normal_offset += size;
		colour_offset += size;
	}

	// vc wibble init
	GVCWibbleIndex=0;

	// loop over vertices
	for (i=0; i<num_vertices; i++)
	{
		GUnpackOffset=1;

		pInputFile->Read( (char *)&vertex_data, vertex_size );

		BeginVertex(pMesh, pMat);

		// special case for skinned models
		if (pMesh->Flags & MESHFLAG_SKINNED)
		{
			// texture coords
			GUnpackOffset = 1;
			VertexSTFloat(vertex_data+ST_offset);

			// weights
			GUnpackOffset = 2;
			VertexWeights(vertex_data+skin_offset);

			// normal and transform offsets
			GUnpackOffset = 3;
			VertexSkinNormal(vertex_data+normal_offset, vertex_data+skin_offset+4);

			// colour
			GUnpackOffset = 4;
			VertexRGBA(vertex_data+colour_offset, ((uint16 *)(vertex_data+XYZ_offset))[7]!=0,
						(uint32)((uint16 *)(vertex_data+XYZ_offset))[7] - 1);

			// position
			GUnpackOffset = 5;
			if (pMesh->Box[0]<=2047.9 && pMesh->Box[1]<2047.9 && pMesh->Box[2]<2047.9)
				VertexXYZ16(vertex_data+XYZ_offset, true, pMesh->Sphere);
			else
				VertexXYZ32(vertex_data+XYZ_offset, true, pMesh->Sphere);
		}
		// special case for billboards

		else if (pMesh->Flags & MESHFLAG_BILLBOARD)
		{
			// texture coords
			GUnpackOffset = 1;
			VertexSTFloat(vertex_data+ST_offset);

			// colour
			GUnpackOffset = 2;
			VertexRGBA(vertex_data+colour_offset, ((uint16 *)(vertex_data+XYZ_offset))[7]!=0,
						(uint32)((uint16 *)(vertex_data+XYZ_offset))[7] - 1);

			// geometric data
			GUnpackOffset = 3;
			VertexXYZFloat(vertex_data+XYZ_offset);
		}

		else
		{
			// texture coords
			if ((pMesh->Flags & MESHFLAG_TEXTURE) && !(pMat->Flags & MATFLAG_ENVIRONMENT))
			{
				if (pMesh->Flags & MESHFLAG_ST16)
					VertexST16(vertex_data+ST_offset);
				else
					VertexST32(vertex_data+ST_offset);
			}

			// normal normals
			if ((pMesh->Flags & MESHFLAG_NORMALS) && ((pMat->Flags & MATFLAG_ENVIRONMENT) || !(pMesh->Flags & MESHFLAG_UNLIT)))
			{
				VertexNormal(vertex_data+normal_offset);
			}

			// colour
			if (pMesh->Flags & MESHFLAG_COLOURS)
			{	
				if (((pMat->RegALPHA & PackALPHA(3,3,0,3,0)) == PackALPHA(2,2,0,0,0))		// if diffuse blend mode
					&& (pMat->Aref <= 1)													// and not really using alpha cutoff
					&& (!(pMat->Flags & (MATFLAG_VC_WIBBLE|MATFLAG_FORCE_ALPHA))))			// and not vc-wibbled or force-alpha
				{
					VertexRGB(vertex_data+colour_offset);									// just rgb
				}
				else
				{
					VertexRGBA(vertex_data+colour_offset,									// else rgba
							   ((uint16 *)(vertex_data+XYZ_offset))[7]!=0,
								(uint32)((uint16 *)(vertex_data+XYZ_offset))[7] - 1);
				}
			}

			// position
			if (pMesh->Box[0]<=2047.9 && pMesh->Box[1]<2047.9 && pMesh->Box[2]<2047.9)
				VertexXYZ16(vertex_data+XYZ_offset, false, pMesh->Sphere);
			else
				VertexXYZ32(vertex_data+XYZ_offset, false, pMesh->Sphere);
		}

		EndVertex();
	}

	// finish batch of vertices
	EndModelMultiPrim(pMesh);
	
}

void	PS2Converter::BeginModelPrim(uint32 Regs, uint NReg, uint Prim, uint Pre, uint Addr, uint Step)
{
	NxPs2::vif::STCYCL(1,NReg);
	NxPs2::vif::UNPACK(0,V4_32,1,REL,UNSIGNED,0);
	NxPs2::gif::BeginTag1_extended(Regs, NReg, PACKED, Prim, Pre, Addr, Step);
}

void	PS2Converter::EndModelPrim(uint Eop)
{
	NxPs2::gif::EndTag1(Eop);
}

void	PS2Converter::BeginModelMultiPrim(uint32 Regs, uint NReg, uint Prim, uint NLoop, uint Addr, NxPs2::sMesh *pMesh, uint Step)
{
	// record tag info
	GMeshRegs  = Regs;
	GMeshNReg  = NReg;
	GMeshPrim  = Prim;
	GMeshNLoop = NLoop;
	GMeshAddr  = Addr;
	GMeshStep  = Step;

	// work out number of vertices to unpack first
	// (and go ahead with it for now, even if there's only room for 1 vertex)
	GNumVerticesThisBuffer = (NxPs2::vu1::MaxBuffer - ((NxPs2::vu1::Loc-NxPs2::vu1::Buffer)&0x3FF) - 1) / NReg;
	if (GNumVerticesThisBuffer > GMeshNLoop)
		GNumVerticesThisBuffer = GMeshNLoop;

	// zpush
	if (pMesh->Flags & MESHFLAG_PASS_BITS)
	{
		float z_push = 0.0f;

		if (pMesh->Flags & MESHFLAG_PASS_BIT_0)
		{
			z_push += 1.0f;
		}

		if (pMesh->Flags & MESHFLAG_PASS_BIT_1)
		{
			z_push += 2.0f;
		}

		if (pMesh->Flags & MESHFLAG_PASS_BIT_2)
		{
			z_push += 4.0f;
		}

		if (pMesh->Flags & MESHFLAG_PASS_BIT_3)
		{
			z_push += 8.0f;
		}

		NxPs2::vif::UNPACK(0,V4_32,1,REL,UNSIGNED,0);
		NxPs2::gif::Tag1(0, 0, 0, 0, 0, 0, 0, VU1_ADDR(AddZPush));
		NxPs2::vu1::Loc++;
		((uint32 *)NxPs2::dma::pLoc)[-1] |= (*(uint32 *)&z_push & 0xFFFF0000);
	}

	BeginModelPrim(Regs, NReg, Prim, 1, Addr, Step);
	GNumOutputVertices = 0;
}

void	PS2Converter::EndModelMultiPrim(NxPs2::sMesh *pMesh)
{
	// zpush
	if (pMesh->Flags & MESHFLAG_PASS_BITS)
	{
		float z_push = 0.0f;

		EndModelPrim(0);

		if (pMesh->Flags & MESHFLAG_PASS_BIT_0)
		{
			z_push += 1.0f;
		}

		if (pMesh->Flags & MESHFLAG_PASS_BIT_1)
		{
			z_push += 2.0f;
		}

		if (pMesh->Flags & MESHFLAG_PASS_BIT_2)
		{
			z_push += 4.0f;
		}

		if (pMesh->Flags & MESHFLAG_PASS_BIT_3)
		{
			z_push += 8.0f;
		}

		NxPs2::vif::UNPACK(0,V4_32,1,REL,UNSIGNED,0);
		NxPs2::gif::Tag1(0, 0, 0, 0, 0, 1, 0, VU1_ADDR(SubZPush));
		NxPs2::vu1::Loc++;
		((uint32 *)NxPs2::dma::pLoc)[-1] |= (*(uint32 *)&z_push & 0xFFFF0000);
	}
	else
	{
		EndModelPrim(1);
	}

	NxPs2::vif::MSCAL(VU1_ADDR(Parser));
}

void	PS2Converter::BeginVertex(NxPs2::sMesh *pMesh, NxPs2::sMaterial *pMat)
{
	if (GNumOutputVertices == GNumVerticesThisBuffer)
	{
		// end the model prim
		if ((pMesh->Flags & MESHFLAG_SKINNED) && (768-((NxPs2::vu1::Buffer+NxPs2::vif::UnpackSize*NxPs2::vif::CycleLength+1)&1023) < NxPs2::vu1::MaxBuffer))		{
			EndModelPrim(0);
			NxPs2::vif::UNPACK(0, V4_32, 1, REL, UNSIGNED, 0);
			NxPs2::gif::Tag1(0, 0, PACKED, 0, 0, 1, 0, VU1_ADDR(Jump));		// reset address to 0
			NxPs2::vu1::Loc = 0;

			// start VU1 execution
			NxPs2::vif::MSCAL(VU1_ADDR(Parser));
			NxPs2::vif::FLUSH();
		}
		else
		{
			EndModelPrim(1);

			// start VU1 execution
			NxPs2::vif::MSCAL(VU1_ADDR(Parser));
		}

		// reduce number still to go
		GMeshNLoop -= GNumOutputVertices;

		// work out number of vertices to unpack next
		GNumVerticesThisBuffer = NxPs2::vu1::MaxBuffer / GMeshNReg;
		if (GNumVerticesThisBuffer > GMeshNLoop)
			GNumVerticesThisBuffer = GMeshNLoop;

		// start a new buffalo'd

		// add a dummy gs context
		NxPs2::gs::BeginPrim(REL,0,0);
		NxPs2::gs::Reg1(NxPs2::gs::A_D_NOP,0);
		NxPs2::gs::EndPrim(0);

		BeginModelPrim(GMeshRegs, GMeshNReg, GMeshPrim, 1, GMeshAddr, GMeshStep);
		GNumOutputVertices = 0;

		// signal tristrip restart
		GRestartTristrip=1;
	}
}

void	PS2Converter::EndVertex(void)
{
	if (GNumOutputVertices==0 && GRestartTristrip)
	{
		GRestartTristrip = 0;
	}

	GNumOutputVertices++;
}

void	PS2Converter::VertexST16(uint8 *pData)
{
	static sint16 s0,t0,s1,t1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V2_16, REL, SIGNED, GUnpackOffset);
		GpTexCoords16 = (sint16 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 4;
		if (GRestartTristrip)
		{
			*GpTexCoords16++ = s0;
			*GpTexCoords16++ = t0;
			*GpTexCoords16++ = s1;
			*GpTexCoords16++ = t1;
			NxPs2::dma::pLoc += 2*4;
		}
		NxPs2::vif::EndUNPACK();
	}
	*GpTexCoords16++ = (sint16) (int) (((float *)pData)[0] * 4096.0f);
	*GpTexCoords16++ = (sint16) (int) (((float *)pData)[1] * 4096.0f);
	s0=s1, s1=GpTexCoords16[-2];
	t0=t1, t1=GpTexCoords16[-1];

	GUnpackOffset++;
}


void	PS2Converter::VertexST32(uint8 *pData)
{
	static sint32 s0,t0,s1,t1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V2_32, REL, SIGNED, GUnpackOffset);
		GpTexCoords32 = (sint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 8;
		if (GRestartTristrip)
		{
			*GpTexCoords32++ = s0;
			*GpTexCoords32++ = t0;
			*GpTexCoords32++ = s1;
			*GpTexCoords32++ = t1;
			NxPs2::dma::pLoc += 2*8;
		}
		NxPs2::vif::EndUNPACK();
	}
	*GpTexCoords32++ = (sint32) (((float *)pData)[0] * 4096.0f);
	*GpTexCoords32++ = (sint32) (((float *)pData)[1] * 4096.0f);
	s0=s1, s1=GpTexCoords32[-2];
	t0=t1, t1=GpTexCoords32[-1];

	GUnpackOffset++;
}


void	PS2Converter::VertexSTFloat(uint8 *pData)
{
	static sint32 s0,t0,s1,t1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::STMASK(0xE0);
		NxPs2::vif::BeginUNPACK(1, V2_32, REL, SIGNED, GUnpackOffset);
		GpTexCoords32 = (sint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 8;
		if (GRestartTristrip)
		{
			*GpTexCoords32++ = s0;
			*GpTexCoords32++ = t0;
			*GpTexCoords32++ = s1;
			*GpTexCoords32++ = t1;
			NxPs2::dma::pLoc += 2*8;
		}
		NxPs2::vif::EndUNPACK();
		NxPs2::vif::STMASK(0);
	}
	*GpTexCoords32++ = ((sint32 *)pData)[0];
	*GpTexCoords32++ = ((sint32 *)pData)[1];
	s0=s1, s1=GpTexCoords32[-2];
	t0=t1, t1=GpTexCoords32[-1];

	GUnpackOffset++;
}


void	PS2Converter::VertexNormal(uint8 *pData)
{
	static sint16 nx0,ny0,nz0,nx1,ny1,nz1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V3_16, REL, SIGNED, GUnpackOffset);
		GpNormal = (sint16 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 6;
		if (GRestartTristrip)
		{
			*GpNormal++ = nx0;
			*GpNormal++ = ny0;
			*GpNormal++ = nz0;
			*GpNormal++ = nx1;
			*GpNormal++ = ny1;
			*GpNormal++ = nz1;
			NxPs2::dma::pLoc += 2*6;
		}
		NxPs2::vif::EndUNPACK();
	}
	*GpNormal++ = ((sint16 *)pData)[0];
	*GpNormal++ = ((sint16 *)pData)[1];
	//*GpNormal++ = ((sint16 *)pData)[2];
	sint16 coord = (sint16)sqrtf(32767.0f*32767.0f - (float)((sint16 *)pData)[0] * (float)((sint16 *)pData)[0]
												   - (float)((sint16 *)pData)[1] * (float)((sint16 *)pData)[1]);
	if (((sint16 *)pData)[0] & 0x0001)
	{
		coord = -coord;
	}
	*GpNormal++ = coord;
	nx0=nx1, nx1=GpNormal[-3];
	ny0=ny1, ny1=GpNormal[-2];
	nz0=nz1, nz1=GpNormal[-1];

	GUnpackOffset++;
}

void	PS2Converter::VertexWeights(uint8 *pData)
{
	static sint32 wa0,wb0,wc0,wa1,wb1,wc1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V3_32, REL, SIGNED, GUnpackOffset);
		GpWeights = (sint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 12;
		if (GRestartTristrip)
		{
			*GpWeights++ = wa0;
			*GpWeights++ = wb0;
			*GpWeights++ = wc0;
			*GpWeights++ = wa1;
			*GpWeights++ = wb1;
			*GpWeights++ = wc1;
			NxPs2::dma::pLoc += 2*12;
		}
		NxPs2::vif::EndUNPACK();
	}
	*GpWeights++ = ((sint16 *)pData)[0];
	*GpWeights++ = ((sint16 *)pData)[1];
	*GpWeights++ = 0x7FFF - GpWeights[-1] - GpWeights[-2];
	wa0=wa1, wa1=GpWeights[-3];
	wb0=wb1, wb1=GpWeights[-2];
	wc0=wc1, wc1=GpWeights[-1];

	GUnpackOffset++;
}

void	PS2Converter::VertexSkinNormal(uint8 *pNormData, uint8 *pTOData)
{
	static sint32 nx0,ny0,nz0,nx1,ny1,nz1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V3_32, REL, SIGNED, GUnpackOffset);
		GpSkinData = (sint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 12;
		if (GRestartTristrip)
		{
			*GpSkinData++ = nx0;
			*GpSkinData++ = ny0;
			*GpSkinData++ = nz0;
			*GpSkinData++ = nx1;
			*GpSkinData++ = ny1;
			*GpSkinData++ = nz1;
			NxPs2::dma::pLoc += 2*12;
		}
		NxPs2::vif::EndUNPACK();
	}

#if 0
	// account for stupid axis convention of skinned model data
	float y = ((float *)pNormData)[1];
	float z = ((float *)pNormData)[2];
	((float *)pNormData)[1] = -z;
	((float *)pNormData)[2] = y;
#endif

	*GpSkinData++ = ((sint32) ((sint16 *)pNormData)[0] & 0xFFFFFC00) | (((uint8 *)pTOData)[0] << 2);
	*GpSkinData++ = ((sint32) ((sint16 *)pNormData)[1] & 0xFFFFFC00) | (((uint8 *)pTOData)[1] << 2);
	//*GpSkinData++ = ((sint32) ((sint16 *)pNormData)[2] & 0xFFFFFC00) | (((uint8 *)pTOData)[2] << 2);
	sint16 coord = (sint16)sqrtf(32767.0f*32767.0f - (float)((sint16 *)pNormData)[0] * (float)((sint16 *)pNormData)[0]
												   - (float)((sint16 *)pNormData)[1] * (float)((sint16 *)pNormData)[1]);
	if (((sint16 *)pNormData)[0] & 0x0001)
	{
		coord = -coord;
	}
	*GpSkinData++ = ((sint32)coord & 0xFFFFFC00) | (((uint8 *)pTOData)[2] << 2);

	nx0=nx1, nx1=GpSkinData[-3];
	ny0=ny1, ny1=GpSkinData[-2];
	nz0=nz1, nz1=GpSkinData[-1];

	GUnpackOffset++;
}



void	PS2Converter::VertexRGBA(uint8 *pData, bool wibbleVC, uint32 seq)
{
	static uint32 rgba0,rgba1;
	static bool wibbleVC0,wibbleVC1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V4_8, REL, UNSIGNED, GUnpackOffset);
		GpColour = (uint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 4;
		if (GRestartTristrip)
		{
			if (wibbleVC0)
			{
				GVCWibbleArray[GVCWibbleIndex++] = seq;
				GVCWibbleArray[GVCWibbleIndex++] = (uint32)GpColour;
			}
			*GpColour++ = rgba0;
			if (wibbleVC1)
			{
				GVCWibbleArray[GVCWibbleIndex++] = seq;
				GVCWibbleArray[GVCWibbleIndex++] = (uint32)GpColour;
			}
			*GpColour++ = rgba1;
			NxPs2::dma::pLoc += 2*4;
		}
		NxPs2::vif::EndUNPACK();
	}
	if (wibbleVC)
	{
		GVCWibbleArray[GVCWibbleIndex++] = seq;
		GVCWibbleArray[GVCWibbleIndex++] = (uint32)GpColour;
	}
	*GpColour++ = ((uint32 *)pData)[0];
	rgba0=rgba1, rgba1=GpColour[-1];
	wibbleVC0=wibbleVC1, wibbleVC1=wibbleVC;

	GUnpackOffset++;
}



void	PS2Converter::VertexRGB(uint8 *pData)
{
	static uint32 rgba0,rgba1;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V3_8, REL, UNSIGNED, GUnpackOffset);
		GpColour24 = NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 3;
		if (GRestartTristrip)
		{
			*GpColour24++ = rgba0;
			*GpColour24++ = rgba0 >> 8;
			*GpColour24++ = rgba0 >> 16;
			//*GpColour24++ = 0;
			*GpColour24++ = rgba1;
			*GpColour24++ = rgba1 >> 8;
			*GpColour24++ = rgba1 >> 16;
			//*GpColour24++ = 0;
			NxPs2::dma::pLoc += 2*3;
		}
		NxPs2::vif::EndUNPACK();
	}
	rgba0 = rgba1;
	rgba1 = ((uint32 *)pData)[0];
	*GpColour24++ = rgba1;
	*GpColour24++ = rgba1 >> 8;
	*GpColour24++ = rgba1 >> 16;
	//*GpColour24++ = 0;

	GUnpackOffset++;
}



void	PS2Converter::VertexXYZ16(uint8 *pData, bool IsSkin, Mth::Vector &sphere)
{
	static sint64 x0,y0,z0,x1,y1,z1,x2,y2,z2;
	sint64 det0,det1,det2;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::STMOD(1);
		NxPs2::vif::BeginUNPACK(0, V4_16, REL, SIGNED, GUnpackOffset);
		GpVertex16 = (sint16 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 8;
		if (GRestartTristrip)
		{
			*GpVertex16++ = x1;
			*GpVertex16++ = y1;
			*GpVertex16++ = z1;
			*GpVertex16++ = (sint16)0x8000;
			*GpVertex16++ = x2;
			*GpVertex16++ = y2;
			*GpVertex16++ = z2;
			*GpVertex16++ = (sint16)0x8000;
			NxPs2::dma::pLoc += 2*8;
		}
		NxPs2::vif::EndUNPACK();
		NxPs2::vif::STMOD(0);
	}
	
	if (IsSkin)
	{
		*GpVertex16++ = ((sint16 *)pData)[0] - (sint16)(sphere[0] * SUB_INCH_PRECISION);
		*GpVertex16++ = ((sint16 *)pData)[1] - (sint16)(sphere[1] * SUB_INCH_PRECISION);
		*GpVertex16++ = ((sint16 *)pData)[2] - (sint16)(sphere[2] * SUB_INCH_PRECISION);
		*GpVertex16++ = ((uint16 *)pData)[3];
	}
	else
	{
		*GpVertex16++ = (sint32) (((float *)pData)[0] * SUB_INCH_PRECISION) - (sint32)(sphere[0] * SUB_INCH_PRECISION);
		*GpVertex16++ = (sint32) (((float *)pData)[1] * SUB_INCH_PRECISION) - (sint32)(sphere[1] * SUB_INCH_PRECISION);
		*GpVertex16++ = (sint32) (((float *)pData)[2] * SUB_INCH_PRECISION) - (sint32)(sphere[2] * SUB_INCH_PRECISION);
		*GpVertex16++ = (sint32) ((uint32 *)pData)[3];
	}

	// advance vertex queue and cull triangle if zero area
	x0=x1, y0=y1, z0=z1;
	x1=x2, y1=y2, z1=z2;
	x2=GpVertex16[-4], y2=GpVertex16[-3], z2=GpVertex16[-2];
	det0 = y1*z2+y2*z0+y0*z1-y2*z1-y0*z2-y1*z0;
	det1 = z1*x2+z2*x0+z0*x1-z2*x1-z0*x2-z1*x0;
	det2 = x1*y2+x2*y0+x0*y1-x2*y1-x0*y2-x1*y0;
	if (det0*det0 + det1*det1 + det2*det2 == 0)
		GpVertex16[-1] = (sint16)0x8000;

	GUnpackOffset++;
}

void	PS2Converter::VertexXYZ32(uint8 *pData, bool IsSkin, Mth::Vector &sphere)
{
	static sint64 x0,y0,z0,x1,y1,z1,x2,y2,z2;
	sint64 det0,det1,det2;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::STMOD(1);
		NxPs2::vif::BeginUNPACK(0, V4_32, REL, SIGNED, GUnpackOffset);
		GpVertex32 = (sint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 16;
		if (GRestartTristrip)
		{
			*GpVertex32++ = x1;
			*GpVertex32++ = y1;
			*GpVertex32++ = z1;
			*GpVertex32++ = 0x8000;
			*GpVertex32++ = x2;
			*GpVertex32++ = y2;
			*GpVertex32++ = z2;
			*GpVertex32++ = 0x8000;
			NxPs2::dma::pLoc += 2*16;
		}
		NxPs2::vif::EndUNPACK();
		NxPs2::vif::STMOD(0);
	}
	
	if (IsSkin)
	{
		*GpVertex32++ = (sint32) ((sint16 *)pData)[0] - (sint16)(sphere[0] * SUB_INCH_PRECISION);
		*GpVertex32++ = (sint32) ((sint16 *)pData)[1] - (sint16)(sphere[1] * SUB_INCH_PRECISION);
		*GpVertex32++ = (sint32) ((sint16 *)pData)[2] - (sint16)(sphere[2] * SUB_INCH_PRECISION);
		*GpVertex32++ = (sint32) ((uint16 *)pData)[3];
	}
	else
	{
		*GpVertex32++ = (sint32) (((float *)pData)[0] * SUB_INCH_PRECISION) - (sint32)(sphere[0] * SUB_INCH_PRECISION);
		*GpVertex32++ = (sint32) (((float *)pData)[1] * SUB_INCH_PRECISION) - (sint32)(sphere[1] * SUB_INCH_PRECISION);
		*GpVertex32++ = (sint32) (((float *)pData)[2] * SUB_INCH_PRECISION) - (sint32)(sphere[2] * SUB_INCH_PRECISION);
		*GpVertex32++ = (sint32) ((uint32 *)pData)[3];
	}

	// advance vertex queue and cull triangle if zero area
	x0=x1, y0=y1, z0=z1;
	x1=x2, y1=y2, z1=z2;
	x2=GpVertex32[-4], y2=GpVertex32[-3], z2=GpVertex32[-2];
	det0 = y1*z2+y2*z0+y0*z1-y2*z1-y0*z2-y1*z0;
	det1 = z1*x2+z2*x0+z0*x1-z2*x1-z0*x2-z1*x0;
	det2 = x1*y2+x2*y0+x0*y1-x2*y1-x0*y2-x1*y0;
	if (det0*det0 + det1*det1 + det2*det2 == 0)
		GpVertex32[-1] = (sint16)0x8000;

	GUnpackOffset++;
}

void	PS2Converter::VertexXYZFloat(uint8 *pData)
{
	static sint32 x0,y0,z0,x1,y1,z1,x2,y2,z2;

	if (GNumOutputVertices==0)
	{
		NxPs2::vif::BeginUNPACK(0, V4_32, REL, SIGNED, GUnpackOffset);
		GpVertex32 = (sint32 *)NxPs2::dma::pLoc;
		NxPs2::dma::pLoc += GNumVerticesThisBuffer * 16;
		if (GRestartTristrip)
		{
			*GpVertex32++ = x1;
			*GpVertex32++ = y1;
			*GpVertex32++ = z1;
			*GpVertex32++ = 0;
			*GpVertex32++ = x2;
			*GpVertex32++ = y2;
			*GpVertex32++ = z2;
			*GpVertex32++ = 0;
			NxPs2::dma::pLoc += 2*16;
		}
		NxPs2::vif::EndUNPACK();
	}
	
	*GpVertex32++ = ((sint32 *)pData)[0];
	*GpVertex32++ = ((sint32 *)pData)[1];
	*GpVertex32++ = ((sint32 *)pData)[2];
	*GpVertex32++ = 0;

	// advance vertex queue
	x0=x1, y0=y1, z0=z1;
	x1=x2, y1=y2, z1=z2;
	x2=GpVertex32[-4], y2=GpVertex32[-3], z2=GpVertex32[-2];

	GUnpackOffset++;
}


// Finds the hierarchy bone index for a given checksum, if any
sint8	PS2Converter::FindHierarchyBoneIndex(uint32 checksum)
{
	for (int bidx = 0; bidx < GNumHierarchyObjects; bidx++)
	{
		if (checksum == pGHierarchyArray[bidx].GetChecksum())
		{
			return pGHierarchyArray[bidx].GetBoneIndex();
		}
	}

	return -1;
}


typedef struct
{
	NxPs2::CGeomNode *pNode;
	NxPs2::CGeomNode *pLastLeaf;
	float GroupPriority;
}
NodeSortStruct;


void	PS2Converter::CreateNodes( void )
{
	int i,j, num_objects;
	NxPs2::sMesh *p_mesh;
	NxPs2::CGeomNode *p_geomnode, *p_rootnode, *p_firstobjectnode, *p_firstchildnode, *p_firstslavenode, *p_firstmeshnode, *p_parent;
	NxPs2::sGroup *p_group;

	// set node array base inside dma buffer
	GGeomNodes = (NxPs2::CGeomNode *)NxPs2::dma::pLoc;

	p_rootnode = GGeomNodes;
	p_firstobjectnode = p_rootnode+1;
	num_objects = 0;

	// create object nodes
	if (vPS2_MESH_VERSION_NUMBER >= 2)
	{
		//printf("Num objects %d; Num children %d\n", GNumObjects, GNumChildObjects);
		for (i=0,p_geomnode=p_firstobjectnode; i<GNumObjects; i++,p_geomnode++)
		{
			// create a new object node
			p_geomnode->Init();
			p_geomnode->SetActive(true);
			p_geomnode->SetObject(true);
			p_geomnode->SetChecksum(GObjectChecksums[i]);
			p_geomnode->SetBoundingSphere( GObjectSpheres[i][0], GObjectSpheres[i][1], GObjectSpheres[i][2], GObjectSpheres[i][3] );
			p_geomnode->SetBoundingBox(GObjectBoxes[i][0], GObjectBoxes[i][1], GObjectBoxes[i][2]);
			p_geomnode->SetLODFarDist(GObjectLODFarDist[i]);
			p_geomnode->SetBoneIndex(FindHierarchyBoneIndex(GObjectChecksums[i]));
			p_geomnode->SetSibling(p_geomnode+1);
		}

		num_objects = GNumObjects;

		// Process children
		p_firstchildnode = p_geomnode;

		for (i=0,p_geomnode=p_firstchildnode; i<GNumChildObjects; i++,p_geomnode++)
		{
		// create a new object node
			p_geomnode->Init();
			p_geomnode->SetActive(true);
			p_geomnode->SetObject(true);
			p_geomnode->SetChecksum(GChildObjectChecksums[i]);
			p_geomnode->SetBoundingSphere( GChildObjectSpheres[i][0], GChildObjectSpheres[i][1], GChildObjectSpheres[i][2],
										   GChildObjectSpheres[i][3] );
			p_geomnode->SetBoundingBox(GChildObjectBoxes[i][0], GChildObjectBoxes[i][1], GChildObjectBoxes[i][2]);
			p_geomnode->SetLODFarDist(GChildObjectLODFarDist[i]);
			p_geomnode->SetBoneIndex(FindHierarchyBoneIndex(GChildObjectChecksums[i]));
			p_geomnode->SetSibling(NULL);

			// And add it to the parent
			for (j=0,p_parent=p_firstobjectnode; j<GNumObjects; j++,p_parent++)
			{
				if (p_parent->GetChecksum() == GChildObjectParentChecksum[i])
					break;
			}

			if (j == GNumObjects)
			{
				// Hack that takes care of missing empty objects
				p_parent--;
				assert(0);
			}

			p_parent->AddChild(p_geomnode);
			//printf("Should add child #%d of checksum %x to parent %x\n", i, GChildObjectChecksums[i], GChildObjectParentChecksum[i]);
			//printf("Adding child #%d of checksum %x to parent %x\n", i, GChildObjectChecksums[i], p_parent->GetChecksum());
		}

		num_objects += GNumChildObjects;

		p_firstslavenode = p_geomnode;

		for (i=0,p_geomnode=p_firstslavenode; i<GNumSlaveLODs; i++,p_geomnode++)
		{
			// create a new object node
			p_geomnode->Init();
			p_geomnode->SetActive(true);
			p_geomnode->SetObject(true);
			p_geomnode->SetChecksum(GSlaveLODChecksums[i]);
			p_geomnode->SetLODFarDist(GSlaveLODFarDist[i]);
			p_geomnode->SetSibling(NULL);

			//printf("Slave %x has distance of %f\n", GSlaveLODChecksums[i], GSlaveLODFarDist[i]);

			// And add it to the master
			for (j=0,p_parent=p_firstobjectnode; j<GNumObjects; j++,p_parent++)
			{
				if (p_parent->GetChecksum() == GMasterLODChecksums[i])
					break;
			}

			if (j == GNumObjects)				// Check children
			{
				for (j=0,p_parent=p_firstchildnode; j<GNumChildObjects; j++,p_parent++)
				{
					if (p_parent->GetChecksum() == GMasterLODChecksums[i])
						break;
				}
			}
			p_parent->SetSlaveLOD(p_geomnode);
		}

		num_objects += GNumSlaveLODs;
	}
	else
	{
		for (i=0,p_mesh=GMeshes; i<GNumMeshes; i++,p_mesh++)
		{
			for (j=0,p_geomnode=p_firstobjectnode; j<num_objects; j++,p_geomnode++)
			{
				if (p_geomnode->GetChecksum() == p_mesh->Checksum)
					break;
			}

			// do nothing if node already exists
			if (j < num_objects)
			{
				continue;
			}

			// create a new object node if necessary
			p_geomnode->Init();
			p_geomnode->SetActive(true);
			p_geomnode->SetObject(true);
			p_geomnode->SetChecksum(p_mesh->Checksum);
			p_geomnode->SetBoundingSphere( p_mesh->Sphere[0], p_mesh->Sphere[1], p_mesh->Sphere[2], p_mesh->Sphere[3] );
			p_geomnode->SetBoundingBox( 1.0e30f, 1.0e30f, 1.0e30f );
			p_geomnode->SetSibling(p_geomnode+1);

			num_objects++;
		}
	}

	// patch up sibling pointer of last normal object
	(p_firstobjectnode+GNumObjects-1)->SetSibling(NULL);

	p_firstmeshnode = p_geomnode = p_firstobjectnode+num_objects;

	// create leaf nodes for meshes
	for (i=0,p_mesh=GMeshes; i<GNumMeshes; i++,p_mesh++,p_geomnode++)
	{
		// find which group the mesh belongs to
		for (p_group=GGroups; p_group<GGroups+GNumGroups; p_group++)
			if ((p_mesh >= p_group->pMeshes) && (p_mesh < p_group->pMeshes+p_group->NumMeshes))
				break;

		// setup node
		p_geomnode->Init();
		p_geomnode->SetActive(true);
		p_geomnode->SetLeaf(true);
		p_geomnode->SetSkinned((p_mesh->Flags&MESHFLAG_SKINNED) ? true : false);
		p_geomnode->SetBillboard((p_mesh->Flags&MESHFLAG_BILLBOARD) ? true : false);
		p_geomnode->SetEnvMapped((p_mesh->pMaterial->Flags & MATFLAG_ENVIRONMENT) ? true : false);
		p_geomnode->SetChecksum(p_mesh->Checksum);
		p_geomnode->SetBoundingSphere(p_mesh->Sphere[0], p_mesh->Sphere[1], p_mesh->Sphere[2], p_mesh->Sphere[3]);
		p_geomnode->SetBoundingBox(p_mesh->Box[0], p_mesh->Box[1], p_mesh->Box[2]);
		p_geomnode->SetDmaTag(p_mesh->pSubroutine);
		p_geomnode->SetGroup((NxPs2::sGroup *)p_group->Checksum);
		p_geomnode->SetBlackFog(false);
		
		if (p_mesh->pMaterial->pTex && p_mesh->pMaterial->Flags & MATFLAG_TEXTURED)
		{
			p_geomnode->SetTextureChecksum(p_mesh->pMaterial->pTex->Checksum);
		}
		else
		{
			// untextured
			p_geomnode->SetTextureChecksum(0);
		}


		// find its parent object
		for (j=0,p_parent=p_firstobjectnode; j<num_objects; j++,p_parent++)
		{
			if (p_parent->GetChecksum() == p_mesh->Checksum)
				break;
		}

		p_parent->AddChild(p_geomnode);

		// z-push flags
		if (p_mesh->Flags & MESHFLAG_PASS_BIT_0)
			p_geomnode->SetZPush0(true);
		if (p_mesh->Flags & MESHFLAG_PASS_BIT_1)
			p_geomnode->SetZPush1(true);
		if (p_mesh->Flags & MESHFLAG_PASS_BIT_2)
			p_geomnode->SetZPush2(true);
		if (p_mesh->Flags & MESHFLAG_PASS_BIT_3)
			p_geomnode->SetZPush3(true);

		// 'no-shadow' flag
		if (p_mesh->Flags & MESHFLAG_NO_SHADOW)
			p_geomnode->SetNoShadow(true);

		// uv-wibble
		if (p_mesh->pMaterial->Flags & MATFLAG_UV_WIBBLE)
		{
			p_geomnode->SetUVWibblePointer(p_mesh->pMaterial->pUVWibbleInfo);
			p_geomnode->SetUVWibbled(true);
		}
		else if (p_mesh->pMaterial->Flags & MATFLAG_ENVIRONMENT)
		{
			uint16 uScale = (uint16)p_mesh->pMaterial->RefMapScaleU;
			uint16 vScale = (uint16)p_mesh->pMaterial->RefMapScaleV;
			uint32 value  = (uint32)uScale | (uint32)vScale << 16;
			p_geomnode->SetUVWibblePointer((float *)(void *)value);
		}
		else
		{
			p_geomnode->SetUVWibblePointer(NULL);
		}

		// vc-wibble
		if (p_mesh->pMaterial->Flags & MATFLAG_VC_WIBBLE)
		{
			p_geomnode->SetVCWibblePointer(p_mesh->pVCWibbleInfo);
			p_geomnode->SetVCWibbled(true);
		}
		else
		{
			p_geomnode->SetVCWibblePointer(NULL);
		}

		// black fog for additive and subtractive meshes
		if (((p_mesh->pMaterial->RegALPHA & PackALPHA(3,3,0,3,0)) == PackALPHA(2,0,0,1,0)) ||
			((p_mesh->pMaterial->RegALPHA & PackALPHA(3,3,0,3,0)) == PackALPHA(0,2,0,1,0)))
		{
			p_geomnode->SetBlackFog(true);
		}
	}

	// create root node
	p_rootnode->Init();
	if (num_objects > 0)
	{
		p_rootnode->SetActive(true);
		p_rootnode->SetChild(p_firstobjectnode);
	}

	// advance dma pointer beyond nodes
	NxPs2::dma::pLoc = (uint8 *)(p_firstmeshnode + GNumMeshes);

	for (int idx = 0; idx < GNumChildObjects; idx++)
	{
		delete [] GObjectChildrenCRCs[idx];
		GObjectChildrenCRCs[idx] = NULL;
	}

	// sorting

	int NodeCmp(const void *p0, const void *p1);
	NodeSortStruct *NodeSortArray = new NodeSortStruct[GNumObjects];
	NxPs2::CGeomNode *p_node, *p_leaf;


	// sort objects
	printf("Sorting objects....");
	for (i=0,p_node=p_firstobjectnode; i<GNumObjects; i++,p_node++)
	{
		for (p_leaf=p_node->GetChild(); p_leaf->GetSibling(); p_leaf=p_leaf->GetSibling())
			;

		for (j=0,p_group=GGroups; j<GNumGroups; j++,p_group++)
			if (p_group->Checksum == (uint)p_leaf->GetGroup())
				break;

		NodeSortArray[i].pNode = p_node;
		NodeSortArray[i].pLastLeaf = p_leaf;
		NodeSortArray[i].GroupPriority = p_group->Priority;
	}
	printf("ready...");

	qsort((void *)NodeSortArray, GNumObjects, sizeof(NodeSortStruct), NodeCmp);

	// relink objects
	if (GNumObjects > 0)	// crashed on scenes with no objects in them (or all objects removed) aml
	{
		for (i=0; i<GNumObjects-1; i++)
		{
			NodeSortArray[i].pNode->SetSibling(NodeSortArray[i+1].pNode);
		}
		NodeSortArray[i].pNode->SetSibling(NULL);
		p_rootnode->SetChild(NodeSortArray[0].pNode);
	}

	delete [] NodeSortArray;
}



bool	PS2Converter::SaveGeom( IoUtils::CVirtualOutputFile* pOutFile )
{
	int hierarchy_offset = 16;			// Where CHierarchyObject data actually starts (must be a multiple of 16)
	int offset = hierarchy_offset +		// Where CGeomNode data actually starts (must be a multiple of 16)
				 GNumHierarchyObjects * sizeof( Nx::CHierarchyObject );

	// Write CGeomNode data offset and hierarchy size
	pOutFile->Write( (const char *) &offset, sizeof(int) );
	pOutFile->Write( (const char *) &hierarchy_offset, sizeof(int) );
	pOutFile->Write( (const char *) &GNumHierarchyObjects, sizeof(int) );
	pOutFile->Write( (const char *) &GNumHierarchyObjects, sizeof(int) );   // used for padding now

	// Write hierarchy data here
	if (GNumHierarchyObjects)
	{
		pOutFile->Write((const char*) pGHierarchyArray, GNumHierarchyObjects * sizeof( Nx::CHierarchyObject ));

		delete [] pGHierarchyArray;
		pGHierarchyArray = NULL;
		GNumHierarchyObjects = 0;
	}

	// Write CGeomNode data
	pOutFile->Write( (char *)GBuffer, NxPs2::dma::pLoc - GBuffer );
	return true;
}


int NodeCmp(const void *p0, const void *p1)
{
	float gp0, gp1;
	gp0 = ((NodeSortStruct *)p0)->GroupPriority;
	gp1 = ((NodeSortStruct *)p1)->GroupPriority;
	NxPs2::CGeomNode *pL0,*pL1;
	pL0 = ((NodeSortStruct *)p0)->pLastLeaf;
	pL1 = ((NodeSortStruct *)p1)->pLastLeaf;
	return gp0 < gp1 ? -1 :
		   gp0 > gp1 ? +1 :
		   pL0 < pL1 ? -1 :
		   pL0 > pL1 ? +1 : 0;
}
