/*
	Quantizer.h
	Image quantization class (uses typical octtree method but, since we need alpha as well, this
	uses more memory in favor of speed with a hexideca tree (4-bit index keys) :)
*/

#ifndef __QUANTIZER__
#define __QUANTIZER__

#include "ImageData.h"
#include "llist.h"
#include "colors.h"
#include <math.h>

struct HDNode
{
	HDNode*        pChild[16];
	HDNode*        pParent;
	bool           bLeaf;		// As the tree is pruned not every level will have a depth of 7
	int            sumRed;		// Sum of all red component values for this color
	int            sumGreen;	// Sum of all green component values for this color
	int            sumBlue;		// Sum of all blue component values for this color
	int            sumAlpha;	// Sum of all alpha values for this color
	int            nPixels;		// Number of pixels using this color

	int            index;		// The index of the average color for this node in the tree 
								// (filled in after final construction)  Speeds up conversion to nearest colors

	Link<HDNode*>* link;		// link into the leaflist for this node

	HDNode()
	{
		for(int i=0;i<16;i++)
			pChild[i] = 0;

		pParent  = NULL;
		bLeaf    = false;
		sumRed   = 0;
		sumGreen = 0;
		sumBlue  = 0;
		sumAlpha = 0;
		nPixels  = 0;
		index    = -1;
	}

	void DeleteChild(int n)
	{
		for(int i=0;i<16;i++)
			if (n != i && pChild[i] == pChild[n])
				pChild[i] = NULL;

		delete pChild[n];
		pChild[n] = NULL;
	}

	~HDNode()
	{
		// The standard closest color tree method can leave some of the palette unused
		// To better optimize the palette we'll build the full tree and only reduce two of the children 
		// at a time, which can result in some child pointers pointing to the same node
		// This will prevent from freeing multiple times.
		for(int i=0;i<16;i++)
		{
			if (pChild[i])
			{
				for(int j=i+1;j<16;j++)
					if (pChild[j] == pChild[i])
						pChild[j] = NULL;

				delete pChild[i];
				pChild[i] = NULL;
			}
		}
	}
};

struct TreeData
{
	HDNode*           root;
	LinkList<HDNode*> leafList;
	int               nPalEntries;
	color32*          pal;

	TreeData()
	{
		root = NULL;
		nPalEntries = 0;
		pal = NULL;
	}

	~TreeData()
	{
		if (root)
			delete root;

		if (pal)
			delete [] pal;
	}
};

// Color component distribution
// This struct will be built per pixel, allowing us to disperse 
struct ColorDist
{
	unsigned int rSum, gSum, bSum, aSum;
};

class Quantizer
{
	ImageData*         srcImage;	// The source image that we're quantizing
	ImageData*         destImage;	// The image that we're generating
	HDNode*            root;		// Pointer to root of the color tree
	int                maxColors;	// Maximum number of colors allowed in palette
	LinkList<HDNode*>  leafList;	// Link list of leaf nodes in the tree for faster searches
	color32*           pal;			// The quantized palette
	int                palColors;	// Number of colors in final palette
	unsigned long      flags;

	void AddColor(HDNode** node,
		          HDNode*  parent,
				  int      level,
		          unsigned char r,unsigned char g,unsigned char b,unsigned char a=255,
				  LinkList<HDNode*>* list=NULL);

	void PruneTree(int nColors);
	void BuildPal();
	void CopyPal(ImageData* img);

	// Finds closest color match within the palette
	int GetClosestColor(unsigned char r,
		                unsigned char g,
						unsigned char b,
						unsigned char a=255,
						int           level=0,
						HDNode*       node=NULL);

	void Quant8Bit();
	void Quant16Bit();
	void Quant24Bit();
	void Quant32Bit();

	void QuantAdd();

	void FreeTree();

	// Finds squared distance between 2 colors in 4 dimensions
	inline __int32 Quantizer::Dist2_4D(unsigned char r1,
									   unsigned char g1,
									   unsigned char b1,
									   unsigned char a1,
									   unsigned char r2,
									   unsigned char g2,
									   unsigned char b2,
									   unsigned char a2)
{
	return (abs((__int32)r2-(__int32)r1)+
		    abs((__int32)g2-(__int32)g1)+
			abs((__int32)b2-(__int32)b1)+
			abs((__int32)a2-(__int32)a1));
}

public:
	enum
	{
		QUANT_PALETTE = 0x0001,		// Flag to convert the image to a palettized one (8-bit)
		QUANT_ALPHA   = 0x0002,		// Flag to store indexed alpha information in the image's tRNS chunk
	};
	
	Quantizer();
	~Quantizer();

	// Finds the closest linear match to a particular color from
	// a given point in the color tree
	void FindLinearMatch(unsigned char r,
						 unsigned char g,
						 unsigned char b,
						 unsigned char a,
		                 HDNode*  node, 
						 color32* bestMatch,
						 __int32* dist);

	inline color32 GetClosestColor32(unsigned char r,
		                             unsigned char g,
									 unsigned char b,
									 unsigned char a=255)
	{
		return pal[GetClosestColor(r,g,b,a)];
	}

	color32 FindColorMatch(TreeData* tdata,
		                   unsigned char r,
					       unsigned char g,
					       unsigned char b,
					       unsigned char a=255);

	color32 GetClosestColor32(HDNode* tree,
		                      unsigned char r,
							  unsigned char g,
							  unsigned char b,
							  unsigned char a=255,
							  int           level=0);
		                      
	TreeData* BuildColorTree(ImageData* image);

	void  AddImage(ImageData* image, int weight);
	void  ProcImage(ImageData* destImage, ImageData* srcImage, int nColors, unsigned long flags);
	int   NumColors();					// Returns number of unique colors in tree w/ leaf indexing
};

#endif
