#include "stdafx.h"
#include "engine.h"
#include "stdio.h"
#include "string.h"
#include "stdlib.h"
#include "resource.h"
#include "mycode.h"
#include "checksum.h"
#include "core/crc.h"
#include "checksumdownloadprogress.h"
#include "monitor.h"
#include "scriptlistview.h"
#include "watchlistview.h"
#include "printfview.h"
#include "ignore.h"
#include "objectlistview.h"
#include "utils.h"
#include "ps2tm.h"

#include "sk/gamenet/scriptdebugger.h"
#include "../monitormouse/monitormouse.h"
#include "gel/scripting/symboltype.h"
#include "gel/scripting/debugger.h"
#include "gel/scripting/checksum.h"
#include "gel/scripting/init.h"
#include <sys/config/config.h>

#include <gel\scripting\scriptcache.h>

// !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
// Warning: Comment this out before attempting to debug, otherwise it will crash your PC!!!
#define DO_READ_MOUSE

Engine* gpEngine = NULL;
int HandleChecksumNamesBufferSize(Net::MsgHandlerContext* context);
int HandleChecksumNames(Net::MsgHandlerContext* context);
int HandleCScriptList(Net::MsgHandlerContext* context);
int HandleCScriptInfo(Net::MsgHandlerContext* context);
int HandleScriptDied(Net::MsgHandlerContext* context);
int HandleCompressionLookupTable(Net::MsgHandlerContext* context);
int HandleBasicCScriptInfo(Net::MsgHandlerContext* context);
int HandleWatchList(Net::MsgHandlerContext* context);
int HandleCompositeObjectInfo(Net::MsgHandlerContext* context);
int HandleCompositeObjectList(Net::MsgHandlerContext* context);
int HandleScriptGlobalInfo(Net::MsgHandlerContext* context);
int HandleExceptionInfo(Net::MsgHandlerContext* context);
int HandleSplitPacket(Net::MsgHandlerContext* context);
int HandleRunningScript(Net::MsgHandlerContext* context);

extern CMonitorApp theApp;

bool gMouseEdgeIsOnLeft=true;
EPlatform gPlatform=PLATFORM_PS2;

CNewWatchWindowParams gNewWindowParams;

#define CHECKSUMS_TXT_FILE_NAME "\\bin\\win32\\checksums.txt"

Lst::HashTable<CWindowInfo> *gpWindowHashTable=NULL;
Lst::HashTable<CWindowInfo> *gpObjectWindowHashTable=NULL;
Lst::HashTable<CWindowInfo> *gpScriptGlobalWindowHashTable=NULL;

void CleanUp()
{
	#ifdef DO_READ_MOUSE
	ReleaseMouseHook();
	#endif

	if (gPlatform==PLATFORM_PS2)
	{
		PS2TM::UnInit();
	}

	Net::MsgDesc msg;
	msg.m_Id = Net::MSG_ID_DISCONN_REQ;
	gClient->EnqueueMessageToServer(&msg);
	gClient->SendData();
	Sleep(500);

	ExportChecksumNames();

	if (gpWindowHashTable)
	{
		delete gpWindowHashTable;
		gpWindowHashTable=NULL;
	}
	if (gpObjectWindowHashTable)
	{
		delete gpObjectWindowHashTable;
		gpObjectWindowHashTable=NULL;
	}
	if (gpScriptGlobalWindowHashTable)
	{
		delete gpScriptGlobalWindowHashTable;
		gpScriptGlobalWindowHashTable=NULL;
	}

	if( gpEngine )
	{
		delete gpEngine;
		gpEngine = NULL;
	}
}

void Assert(int Condition, char *pMessage)
{
	if (!Condition)
	{
		MessageBox(NULL,pMessage,"Monitor assertion failed",MB_OK);
		CleanUp();
		exit(EXIT_FAILURE);
	}
}

void Printf(const char *p_message)
{
	HWND h_window=CPrintfView::sGetWindow();
	if (h_window)
	{
		int WM_PRINTF = RegisterWindowMessage ("PRINTF");
		SendMessage(h_window,
		WM_PRINTF,
		(WPARAM) 0,
		(LPARAM) p_message);
	}
}

char *SkipWhiteSpace(char *p_ch)
{
	while (*p_ch==' ' || *p_ch=='\t' || *p_ch==0x0d || *p_ch==0x0a)
	{
		++p_ch;
	}
	return p_ch;
}

void LoadChecksumNames()
{
	// Open and read in the big database of checksum names.
	CFile checksums_file;
	char p_filename[1000];
	const char *p_path=getenv("PROJ_ROOT");
	if (!p_path)
	{
		Printf("Error! PROJ_ROOT needs to be set");
		return;
	}
	sprintf(p_filename,"%s%s",p_path,CHECKSUMS_TXT_FILE_NAME);

	if (!checksums_file.Open(TEXT(p_filename), CFile::modeRead))
	{
		return;
	}

	int size=checksums_file.GetLength();
	if (!size)
	{
		return;
	}

	char *p_file=new char[size+1]; // +1 for a terminating zero.

	int bytes_read=checksums_file.Read(p_file,size);
	ASSERT(bytes_read==size);
	checksums_file.Close();

	// Put a terminating zero on the end so that the end can be detected easily.
	p_file[size]=0;

	char *p_entry=p_file;
	bool done=false;
	while (*p_entry)
	{
		p_entry=SkipWhiteSpace(p_entry);
		if (*p_entry==0)
		{
			break;
		}

		ASSERT(p_entry[0]=='0');
		ASSERT(p_entry[1]=='x');
		ASSERT(p_entry[2] && p_entry[3] && p_entry[4] && p_entry[5] && p_entry[6] && p_entry[7] && p_entry[8] && p_entry[9] && p_entry[10]);
		p_entry+=10;
		
		p_entry=SkipWhiteSpace(p_entry);
		char p_name[200];
		int c=0;
		while (!(*p_entry==0x0d || *p_entry==0x0a || *p_entry==0))
		{
			p_name[c++]=*p_entry++;
		}
		p_name[c]=0;
		Script::AddChecksumName(Crc::GenerateCRCFromString(p_name),p_name);		

		p_entry=SkipWhiteSpace(p_entry);
	}

	delete [] p_file;
}

void ExportChecksumNames()
{
	CStdioFile checksums_file;
	char p_filename[1000];
	const char *p_path=getenv("PROJ_ROOT");
	if (!p_path)
	{
		Printf("Error! PROJ_ROOT needs to be set");
		return;
	}
	sprintf(p_filename,"%s%s",p_path,CHECKSUMS_TXT_FILE_NAME);

	if (!checksums_file.Open(TEXT(p_filename), CFile::modeWrite|CFile::modeCreate))
	{
		return;
	}

	char *p_start=NULL;
	char *p_end=NULL;
	Script::GetChecksumNamesBuffer(&p_start,&p_end);

	char p_checksum[100];

	while (p_start < p_end)
	{
		sprintf(p_checksum,"0x%08x ",Crc::GenerateCRCFromString(p_start));
		checksums_file.WriteString(p_checksum);
		checksums_file.WriteString(p_start);
		checksums_file.WriteString("\n");
		p_start+=strlen(p_start);
		++p_start;
	}

	checksums_file.Close();
}

void RemoveWindowFromHashTable(Lst::HashTable<CWindowInfo> *p_table, uint32 id)
{
	CWindowInfo *p_window_info=p_table->GetItem(id);
	if (p_window_info)
	{
		p_table->FlushItem(id);
		delete p_window_info;
	}
}

void RemoveAllWindowsFromHashTable(Lst::HashTable<CWindowInfo> *p_table)
{
	// Delete all the entries. I believe this is safe, it just means the
	// table will contain invalid pointers afterwards ...
	p_table->IterateStart();
	CWindowInfo *p_window_info=p_table->IterateNext();
	while (p_window_info)
	{
		delete p_window_info;
		p_window_info=p_table->IterateNext();
	}
	// Now remove all those invalid pointers.
	p_table->FlushAllItems();
}

void MyCodeInit()
{
	Config::Init(0,NULL);

	Spt::SingletonPtr< Script::CScriptCache> script_cache( true );

	CString edge=theApp.GetProfileString("Misc","MouseEdge","Left");
	if (edge=="Left")
	{
		gMouseEdgeIsOnLeft=true;
	}
	else
	{
		gMouseEdgeIsOnLeft=false;
	}

	CString platform=theApp.GetProfileString("Misc","Platform","PS2");
	if (platform=="PS2")
	{
		gPlatform=PLATFORM_PS2;
	}
	else if (platform=="XBox")
	{
		gPlatform=PLATFORM_XBOX;
	}

	gpEngine = new Engine;

	Script::AllocatePools();

	gpWindowHashTable = new Lst::HashTable<CWindowInfo>(8);
	gpObjectWindowHashTable = new Lst::HashTable<CWindowInfo>(8);
	gpScriptGlobalWindowHashTable = new Lst::HashTable<CWindowInfo>(8);

	if (gPlatform==PLATFORM_PS2)
	{
		PS2TM::Init();
	}

	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_CHECKSUM_NAMES_BUFFER_SIZE,
									 HandleChecksumNamesBufferSize);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_CHECKSUM_NAMES,
									 HandleChecksumNames);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_CSCRIPT_LIST,
									 HandleCScriptList);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_CSCRIPT_INFO,
									 HandleCScriptInfo);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_SCRIPT_DIED,
									 HandleScriptDied);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_COMPRESSION_LOOKUP_TABLE,
									 HandleCompressionLookupTable);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_BASIC_CSCRIPT_INFO,
									 HandleBasicCScriptInfo);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_WATCH_LIST,
									 HandleWatchList);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_COMPOSITE_OBJECT_INFO,
									 HandleCompositeObjectInfo);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_COMPOSITE_OBJECT_LIST,
									 HandleCompositeObjectList);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_SCRIPT_GLOBAL_INFO,
									 HandleScriptGlobalInfo);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_EXCEPTION_INFO,
									 HandleExceptionInfo);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_PACKET,
									 HandleSplitPacket);
	gClient->m_Dispatcher.AddHandler(GameNet::vMSG_ID_DBG_RUNNING_SCRIPT,
									 HandleRunningScript);

	// Load the checksum names from the database contained in bin\win32\checksums.txt
	LoadChecksumNames();

	// This will establish a connection with the game code, and then send a message
	// requesting the contents of the script array WriteToBuffer_CompressionLookupTable_8,
	// which is required in order to print the contents of script structures correctly.
	ConnectToServer();

	#ifdef DO_READ_MOUSE
	SetMouseHook(gHWnd);
	#endif
}

void SetMouseEdgeLeft()
{
	theApp.WriteProfileString("Misc","MouseEdge","Left");
	gMouseEdgeIsOnLeft=true;
}

void SetMouseEdgeRight()
{
	theApp.WriteProfileString("Misc","MouseEdge","Right");
	gMouseEdgeIsOnLeft=false;
}

void SetPlatformPS2()
{
	theApp.WriteProfileString("Misc","Platform","PS2");
	gPlatform=PLATFORM_PS2;
	MessageBox(NULL,"You will need to close and restart the debugger for this change to take effect.","Warning",MB_OK);
}

void SetPlatformXBox()
{
	theApp.WriteProfileString("Misc","Platform","XBox");
	gPlatform=PLATFORM_XBOX;
	MessageBox(NULL,"You will need to close and restart the debugger for this change to take effect.","Warning",MB_OK);	
}

void ConnectToServer()
{
	Printf("ConnectToServer");

	// Clear the list of ids of scripts to be ignored, because if the
	// user is reconnecting it is probably because they rebooted their PS2, in
	// which case the ids will start getting used again by new scripts.
	gDeadScriptsIdList.Clear();
	gIgnoredScriptsIdList.Clear();


	Net::MsgDesc msg;
	msg.m_Id = Net::MSG_ID_DISCONN_REQ;
	gClient->EnqueueMessageToServer(&msg);
	gClient->SendData();

	// Need to wait a bit for the last message to take effect.
	Sleep(500);

	msg.m_Id = Net::MSG_ID_CONNECTION_REQ;
	gClient->EnqueueMessageToServer(&msg);
}

void WatchScript(uint32 script_id)
{
	// Make sure that script-info packets received for this script
	// will not be ignored.
	gIgnoredScriptsIdList.RemoveId(script_id);

	// Tell the PS2 to start sending info about this script.
	uint32 p_data[3];

	p_data[0]=false; // Don't stop the script
	p_data[1]=Dbg::CHECKSUM_IS_SCRIPT_ID;
	p_data[2]=script_id;

	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_WATCH_SCRIPT;
	msg.m_Length=sizeof(p_data);
	msg.m_Data=p_data;
	gClient->EnqueueMessageToServer(&msg);
}

void ChangeObjectUpdateMode(uint32 objectId, Dbg::EObjectUpdateMode mode, bool on)
{
	/*
	char *p_mode="???";
	switch (mode)
	{
	case Dbg::EVERY_FRAME:
		p_mode="every-frame";
		break;
	case Dbg::SINGLE_STEP:
		p_mode="single-step";
		break;
	default:
		break;
	}*/

	Dbg::SObjectUpdateMode object_update_mode;
	object_update_mode.mObjectId=objectId;
	object_update_mode.mMode=mode;
	// Note: Need to cast the lhs to a uint32* cos MSVC appears to only copy 1 byte
	// over when copying the bool, leaving the other 3 bytes containing randomness.
	// The PS2 however will read all 4 bytes, leading to it sometimes think that 
	// the bool is true when it isn't.
	*(uint32*)&object_update_mode.mOn=on;

	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_SET_OBJECT_UPDATE_MODE;
	msg.m_Length=sizeof(Dbg::SObjectUpdateMode);
	msg.m_Data=&object_update_mode;
	gClient->EnqueueMessageToServer(&msg);
}

void SendViewObjectMessage(uint32 objectId, bool view_on)
{
	Dbg::SViewObject view_object_message;
	view_object_message.mObjectId=objectId;
	// Note: Need to cast the lhs to a uint32* cos MSVC appears to only copy 1 byte
	// over when copying the bool, leaving the other 3 bytes containing randomness.
	// The PS2 however will read all 4 bytes, leading to it sometimes think that 
	// the bool is true when it isn't.
	*(uint32*)&view_object_message.mDoViewObject=view_on;

	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_VIEW_OBJECT;
	msg.m_Length=sizeof(Dbg::SViewObject);
	msg.m_Data=&view_object_message;
	gClient->EnqueueMessageToServer(&msg);
}

static int sTotalBytesExpected=0;
static int sBytesLoaded=0;
static CChecksumDownloadProgress *spChecksumDownloadProgress=NULL;

int HandleChecksumNamesBufferSize(Net::MsgHandlerContext* context)
{
	//Assert(sBytesLeftToLoad==0,"Started a new checksum-names download before the last had finished ...");
	Assert(context->m_MsgLength==4,"Unexpected length for vMSG_ID_DBG_CHECKSUM_NAMES_BUFFER_SIZE message");
	Assert(sizeof(int)==4,"Eh? sizeof(int) not 4 ?");

	sTotalBytesExpected=*(int*)context->m_Msg;
	sBytesLoaded=0;

	if (spChecksumDownloadProgress==NULL)
	{
		spChecksumDownloadProgress=new CChecksumDownloadProgress;
		spChecksumDownloadProgress->Create(IDD_CHECKSUM_DOWNLOAD_PROGRESS,NULL);
		spChecksumDownloadProgress->ShowWindow(true);

		CProgressCtrl *p_progress=(CProgressCtrl*)(spChecksumDownloadProgress->GetDlgItem(IDC_CHECKSUM_DOWNLOAD_PROGRESS));

		p_progress->SetRange(0,100); //Set the range to between 0 and 100
		p_progress->SetStep(1); // Set the step amount
	}

	return Net::HANDLER_CONTINUE;
}

int HandleChecksumNames(Net::MsgHandlerContext* context)
{
	char *p_name=context->m_Msg;
	char *p_end=p_name+context->m_MsgLength;
	while (p_name < p_end)
	{
		Script::AddChecksumName(Crc::GenerateCRCFromString(p_name),p_name);
		p_name+=strlen(p_name);
		++p_name;
	}

	sBytesLoaded+=context->m_MsgLength;
	if (spChecksumDownloadProgress)
	{
		CProgressCtrl *p_progress=(CProgressCtrl*)(spChecksumDownloadProgress->GetDlgItem(IDC_CHECKSUM_DOWNLOAD_PROGRESS));
		p_progress->SetPos(sBytesLoaded*100/sTotalBytesExpected);

		if (sBytesLoaded==sTotalBytesExpected)
		{
			delete spChecksumDownloadProgress;
			spChecksumDownloadProgress=NULL;
		}
	}

	return Net::HANDLER_CONTINUE;
}

int HandleWatchList(Net::MsgHandlerContext* context)
{
	HWND h_window=CWatchListView::sGetWindow();
	if (!h_window)
	{
		theApp.mpWatchListDocTemplate->OpenDocumentFile(NULL);
	}

	h_window=CWatchListView::sGetWindow();
	if (h_window)
	{
		int WM_WATCH_LIST = RegisterWindowMessage ("WATCH_LIST");
		SendMessage(h_window,
		WM_WATCH_LIST,
		(WPARAM) context->m_MsgLength,
		(LPARAM) context->m_Msg);
	}

	return Net::HANDLER_CONTINUE;
}

int HandleCScriptList(Net::MsgHandlerContext* context)
{
	Printf("HandleCScriptList");

	HWND h_window=CScriptListView::sGetWindow();
	if (!h_window)
	{
		theApp.mpScriptListDocTemplate->OpenDocumentFile(NULL);
	}

	h_window=CScriptListView::sGetWindow();
	if (h_window)
	{
		int WM_SCRIPT_LIST = RegisterWindowMessage ("SCRIPT_LIST");
		SendMessage(h_window,
		WM_SCRIPT_LIST,
		(WPARAM) context->m_MsgLength,
		(LPARAM) context->m_Msg);
	}

	return Net::HANDLER_CONTINUE;
}

int HandleCScriptInfo(Net::MsgHandlerContext* context)
{
	uint32 id=((uint32*)context->m_Msg)[1];

	// Ignore late packets that may arrive after the script has died,
	// or after the user has chosen not to watch that script any more.
	if (gDeadScriptsIdList.IsInList(id) || gIgnoredScriptsIdList.IsInList(id))
	{
		return Net::HANDLER_CONTINUE;
	}

	// Find the window for this script id, or create it if it does not exist already.
	HWND window=theApp.GetOrCreateWatchWindow(id);
	
	// Then send it the script info so that it can display it.
	int WM_WATCH_INFO = RegisterWindowMessage ("WATCH_INFO");
	SendMessage(window,
				WM_WATCH_INFO,
				(WPARAM) context->m_MsgLength,
				(LPARAM) context->m_Msg);
	
	return Net::HANDLER_CONTINUE;
}

int HandleBasicCScriptInfo(Net::MsgHandlerContext* context)
{
	uint32 id=((uint32*)context->m_Msg)[0];

	// Ignore late packets that may arrive after the script has died,
	// or after the user has chosen not to watch that script any more.
	if (gDeadScriptsIdList.IsInList(id) || gIgnoredScriptsIdList.IsInList(id))
	{
		return Net::HANDLER_CONTINUE;
	}

	// Find the window for this script id, or create it if it does not exist already.
	HWND window=theApp.GetOrCreateWatchWindow(id);
	
	// Then send it the script info so that it can display it.
	int WM_BASIC_SCRIPT_INFO = RegisterWindowMessage ("BASIC_SCRIPT_INFO");
	SendMessage(window,
				WM_BASIC_SCRIPT_INFO,
				(WPARAM) context->m_MsgLength,
				(LPARAM) context->m_Msg);
	
	return Net::HANDLER_CONTINUE;
}

// This gets called when a script dies.
int HandleScriptDied(Net::MsgHandlerContext* context)
{
	Dbg_MsgAssert(context->m_MsgLength==4,("Expected m_MsgLength to be 4 for vMSG_ID_DBG_SCRIPT_DIED"));
	uint32 id=*(uint32*)context->m_Msg;

	// Add to the list of scripts to be ignored, so that any script-info packets 
	// for that script that arrive late get ignored.
	gDeadScriptsIdList.AddId(id);
	

	CWindowInfo *p_window_info=gpWindowHashTable->GetItem(id);
	if (p_window_info)
	{
		int WM_SCRIPT_DIED = RegisterWindowMessage ("SCRIPT_DIED");
		SendMessage(p_window_info->mViewHWND,WM_SCRIPT_DIED,(WPARAM)0,(LPARAM)0);
	}

	return Net::HANDLER_CONTINUE;
}

int HandleCompositeObjectInfo(Net::MsgHandlerContext* context)
{
	char p_foo[1024];
	sprintf(p_foo,"HandleCompositeObjectInfo, %d bytes",context->m_MsgLength);
    Printf(p_foo);

	uint32 id=((uint32*)context->m_Msg)[0];

	// Find the window for this script id, or create it if it does not exist already.
	HWND window=theApp.GetOrCreateObjectWindow(id);
	
	// Then send it the script info so that it can display it.
	int WM_OBJECT_INFO = RegisterWindowMessage ("OBJECT_INFO");
	SendMessage(window,
				WM_OBJECT_INFO,
				(WPARAM) context->m_MsgLength,
				(LPARAM) context->m_Msg);

	return Net::HANDLER_CONTINUE;
}

int HandleCompositeObjectList(Net::MsgHandlerContext* context)
{
	Printf("HandleCompositeObjectList");

	HWND h_window=CObjectListView::sGetWindow();
	if (!h_window)
	{
		theApp.mpObjectListDocTemplate->OpenDocumentFile(NULL);
	}

	h_window=CObjectListView::sGetWindow();
	if (h_window)
	{
		int WM_OBJECT_LIST = RegisterWindowMessage ("OBJECT_LIST");
		SendMessage(h_window,
		WM_OBJECT_LIST,
		(WPARAM) context->m_MsgLength,
		(LPARAM) context->m_Msg);
	}

	return Net::HANDLER_CONTINUE;
}

int HandleScriptGlobalInfo(Net::MsgHandlerContext* context)
{
	char p_foo[1024];
	sprintf(p_foo,"HandleScriptGlobalInfo, %d bytes",context->m_MsgLength);
    Printf(p_foo);

	uint32 *p_info=(uint32*)context->m_Msg;
	uint32 global_name=p_info[0];
	uint32 type=p_info[1];
	const char *p_source_file=(const char*)&p_info[2];
	
	switch (type)
	{
		case ESYMBOLTYPE_INTEGER:
		case ESYMBOLTYPE_FLOAT:
		case ESYMBOLTYPE_STRING:
		case ESYMBOLTYPE_LOCALSTRING:
		case ESYMBOLTYPE_PAIR:
		case ESYMBOLTYPE_VECTOR:
		case ESYMBOLTYPE_STRUCTURE:
		case ESYMBOLTYPE_ARRAY:
		case ESYMBOLTYPE_NAME:
		{
			// Find the window for this script id, or create it if it does not exist already.
			HWND window=theApp.GetOrCreateScriptGlobalWindow(global_name);
			
			// Then send it the info so that it can display it.
			int WM_SCRIPT_GLOBAL_INFO = RegisterWindowMessage ("SCRIPT_GLOBAL_INFO");
			SendMessage(window,
						WM_SCRIPT_GLOBAL_INFO,
						(WPARAM) context->m_MsgLength,
						(LPARAM) context->m_Msg);
		}
		break;

		case ESYMBOLTYPE_QSCRIPT:
		{
			// With scripts, give the user a choice of setting a breakpoint on the script.
			// Handy for things like exceptions.
			sprintf(p_foo,"%s is a script defined in '%s'\n\nWould you like to put a breakpoint on the script ?",Script::FindChecksumName(global_name),p_source_file);
			int result=MessageBox(NULL,p_foo,"Script",
								  // MB_DEFBUTTON2 means make the 2nd button, 
								  // ie "No" the default.
								  MB_YESNO | MB_ICONQUESTION | MB_DEFBUTTON2);
			if (result==IDYES)
			{
				SetBreakpointOnScript(global_name);
			}
			break;
		}

		case ESYMBOLTYPE_CFUNCTION:
			sprintf(p_foo,"%s is a C-function",Script::FindChecksumName(global_name));
			MessageBox(NULL,p_foo,"C-function",MB_OK);
			break;
		case ESYMBOLTYPE_MEMBERFUNCTION:
			sprintf(p_foo,"%s is a member function",Script::FindChecksumName(global_name));
			MessageBox(NULL,p_foo,"Member function",MB_OK);
			break;
		default:
			break;
	}

	return Net::HANDLER_CONTINUE;
}

int HandleExceptionInfo(Net::MsgHandlerContext* context)
{
	uint32 *p_info=(uint32*)context->m_Msg;

	char p_foo[1024];
	sprintf(p_foo,"Exception fired: Object '%s', Exception '%s' Script '%s'",
			Script::FindChecksumName(p_info[0]),
			Script::FindChecksumName(p_info[1]),
			Script::FindChecksumName(p_info[2]));
    Printf(p_foo);

	return Net::HANDLER_CONTINUE;
}

int HandleCompressionLookupTable(Net::MsgHandlerContext* context)
{
	uint32 *p_8bit_indexed_checksums=(uint32*)context->m_Msg;
	int num_checksums=context->m_MsgLength/4;

	Script::Set8BitIndexedChecksums(num_checksums,p_8bit_indexed_checksums);
	return Net::HANDLER_CONTINUE;
}



enum
{
	MAX_MESSAGES=3,
	MAX_PACKETS_PER_MESSAGE=4, // Max of 32.
};

class CReconstructedMessage
{
	void dispatch_message();

	static CReconstructedMessage sp_reconstructed_messages[MAX_MESSAGES];

	uint8 mp_message_buffer[Dbg::SPLIT_MESSAGE_MAX_SIZE * MAX_PACKETS_PER_MESSAGE];

	uint32 m_id;
	uint32 m_time; // The time the first packet came in, so that the message
				  // can be discarded if it gets too old before being completed.
	uint32 m_packets_received; // A bitfield, one bit for each packet received.

	int m_num_packets_received;
	int m_num_packets_expected;
	int m_packet_size[MAX_PACKETS_PER_MESSAGE];

	char m_original_id;

	uint32 m_bytes_received;

public:
	CReconstructedMessage();

	void AddPacket(Dbg::SSplitMessageHeader *p_header);
	static CReconstructedMessage *FindMessageWithId(uint32 id);
	static CReconstructedMessage *FindEmptyMessage();
};

CReconstructedMessage CReconstructedMessage::sp_reconstructed_messages[MAX_MESSAGES];

CReconstructedMessage::CReconstructedMessage()
{
	m_id=0;
	m_time=0;
	m_packets_received=0;
	m_num_packets_received=0;
	m_num_packets_expected=0;
	m_original_id=0;
	m_bytes_received=0;
}

void CReconstructedMessage::dispatch_message()
{
	Printf("Dispatching message");
	
	static Net::MsgHandlerContext context;
	context.m_MsgLength=m_bytes_received;
	context.m_Data=mp_message_buffer;

	switch (m_original_id)
	{
		case GameNet::vMSG_ID_DBG_COMPOSITE_OBJECT_INFO:
			HandleCompositeObjectInfo(&context);
			break;
		default:
			break;
	}
	
	m_id=0;
	m_original_id=0;
	m_bytes_received=0;
}

void CReconstructedMessage::AddPacket(Dbg::SSplitMessageHeader *p_header)
{
	char p_foo[1024];
	sprintf(p_foo,"AddPacket: Id=%x, i=%d, n=%d, size=%d",p_header->mThisMessageId,
		p_header->mThisMessageIndex,
		p_header->mTotalNumMessages,
		p_header->mThisMessageSize);
	Printf(p_foo);


	m_id=p_header->mThisMessageId;
	m_original_id=p_header->mOriginalMessageId;

	memcpy(mp_message_buffer+Dbg::SPLIT_MESSAGE_MAX_SIZE*p_header->mThisMessageIndex,
		   p_header+1,
		   p_header->mThisMessageSize);

	m_bytes_received+=p_header->mThisMessageSize;

	if (m_bytes_received==p_header->mOriginalMessageSize)
	{
		dispatch_message();
	}
}

CReconstructedMessage *CReconstructedMessage::FindMessageWithId(uint32 id)
{
	for (int i=0; i<MAX_MESSAGES; ++i)
	{
		if (sp_reconstructed_messages[i].m_id==id)
		{
			return &sp_reconstructed_messages[i];
		}
	}
	return NULL;
}

CReconstructedMessage *CReconstructedMessage::FindEmptyMessage()
{
	for (int i=0; i<MAX_MESSAGES; ++i)
	{
		if (sp_reconstructed_messages[i].m_id==0)
		{
			return &sp_reconstructed_messages[i];
		}
	}
	return NULL;
}

int HandleSplitPacket(Net::MsgHandlerContext* context)
{
	Dbg::SSplitMessageHeader *p_header=(Dbg::SSplitMessageHeader*)context->m_Msg;

	// First, see if a partially constructed message with this id exists, and if
	// so pass it the packet.
	CReconstructedMessage *p_mess=CReconstructedMessage::FindMessageWithId(p_header->mThisMessageId);
	if (p_mess)
	{
		p_mess->AddPacket(p_header);
		return Net::HANDLER_CONTINUE;
	}

	// Otherwise start constructing a new packet.
	p_mess=CReconstructedMessage::FindEmptyMessage();
	if (p_mess)
	{
		p_mess->AddPacket(p_header);
		return Net::HANDLER_CONTINUE;
	}

	Printf("Dropped a packet, buffer full ...");	
	return Net::HANDLER_CONTINUE;
}

int HandleRunningScript(Net::MsgHandlerContext* context)
{
	uint32 script_name=((uint32*)context->m_Msg)[0];
	uint32 num_return_addresses=((uint32*)context->m_Msg)[1];

	char p_foo[1000];
	for (int i=0; i<num_return_addresses; ++i)
	{
		p_foo[i]='.';
	}
	
	sprintf(p_foo+num_return_addresses,"Running script '%s'",Script::FindChecksumName(script_name));
	Printf(p_foo);

	return Net::HANDLER_CONTINUE;
}
