// MonitorView.cpp : implementation of the CMonitorView class
//

#include "stdafx.h"
#include "Monitor.h"

#include "MonitorDoc.h"
#include "MonitorView.h"
#include "mycode.h"
#include "checksum.h"
#include "utils.h"
#include "debuggerbutton.h"
#include "engine.h"
#include "sk/gamenet/scriptdebugger.h"
#include "mytreectrl.h"
#include "ignore.h"
#include "core/crc.h"
#include <gel/scripting/checksum.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

int WM_WATCH_INFO = RegisterWindowMessage ("WATCH_INFO");
int WM_BASIC_SCRIPT_INFO = RegisterWindowMessage ("BASIC_SCRIPT_INFO");

int WM_STOP_SCRIPT = RegisterWindowMessage ("STOP_SCRIPT");
int WM_STEP_INTO = RegisterWindowMessage ("STEP_INTO");
int WM_STEP_OVER = RegisterWindowMessage ("STEP_OVER");
int WM_START_SCRIPT = RegisterWindowMessage ("START_SCRIPT");

int WM_CHANGE_FILE = RegisterWindowMessage ("CHANGE_FILE");
int WM_SCRIPT_DIED = RegisterWindowMessage ("SCRIPT_DIED");

#define SLIDER_RANGE 10

/////////////////////////////////////////////////////////////////////////////
// CMonitorView

IMPLEMENT_DYNCREATE(CMonitorView, CView)

BEGIN_MESSAGE_MAP(CMonitorView, CView)
	//{{AFX_MSG_MAP(CMonitorView)
	ON_REGISTERED_MESSAGE(WM_WATCH_INFO, OnWatchInfo)
	ON_REGISTERED_MESSAGE(WM_BASIC_SCRIPT_INFO, OnBasicScriptInfo)
	ON_REGISTERED_MESSAGE(WM_STOP_SCRIPT, OnStopScript)
	ON_REGISTERED_MESSAGE(WM_STEP_INTO, OnStepInto)
	ON_REGISTERED_MESSAGE(WM_STEP_OVER, OnStepOver)
	ON_REGISTERED_MESSAGE(WM_START_SCRIPT, OnGo)
	ON_REGISTERED_MESSAGE(WM_CHANGE_FILE, OnChangeFile)
	ON_REGISTERED_MESSAGE(WM_SCRIPT_DIED, OnScriptDied)
	ON_WM_CREATE()
	ON_WM_HSCROLL()
	ON_WM_TIMER()
	ON_WM_MOUSEMOVE()
	ON_WM_PAINT()
	ON_WM_DESTROY()
	//}}AFX_MSG_MAP
	ON_WM_ERASEBKGND()
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CMonitorView construction/destruction

CMonitorView::CMonitorView()
{
	mp_edit=NULL;
	mp_info=NULL;

	mp_button_stop=NULL;
	mp_button_step_into=NULL;
	mp_button_step_over=NULL;
	mp_button_go=NULL;
	mp_auto_step_speed=NULL;

	mp_tree_function_params=NULL;
	mp_tree_callstack=NULL;

	m_id=0;
	m_num_callstack_entries=0;
	mp_edit_window_current_file_name[0]=0;
	m_current_displayed_callstack_entry=0;
	m_spinner=0;
	mp_info_bar_text[0]=0;
	m_slider_pos=SLIDER_RANGE;
}

CMonitorView::~CMonitorView()
{
	RemoveWindowFromHashTable(gpWindowHashTable,m_id);

	if (mp_tree_callstack)
	{
		delete mp_tree_callstack;
	}
	if (mp_tree_function_params)
	{
		delete mp_tree_function_params;
	}
	if (mp_edit)
	{
		delete mp_edit;
	}
	if (mp_info)
	{
		delete mp_info;
	}
	if (mp_button_stop)
	{
		delete mp_button_stop;
	}
	if (mp_button_step_into)
	{
		delete mp_button_step_into;
	}
	if (mp_button_step_over)
	{
		delete mp_button_step_over;
	}
	if (mp_button_go)
	{
		delete mp_button_go;
	}
	if (mp_auto_step_speed)
	{
		delete mp_auto_step_speed;
	}
}

BOOL CMonitorView::PreCreateWindow(CREATESTRUCT& cs)
{
	// TODO: Modify the Window class or styles here by modifying
	//  the CREATESTRUCT cs

	return CView::PreCreateWindow(cs);
}

/////////////////////////////////////////////////////////////////////////////
// CMonitorView drawing

void CMonitorView::OnDraw(CDC* pDC)
{
	CMonitorDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	// TODO: add draw code for native data here
}

/////////////////////////////////////////////////////////////////////////////
// CMonitorView diagnostics

#ifdef _DEBUG
void CMonitorView::AssertValid() const
{
	CView::AssertValid();
}

void CMonitorView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CMonitorDoc* CMonitorView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CMonitorDoc)));
	return (CMonitorDoc*)m_pDocument;
}
#endif //_DEBUG

static DWORD CALLBACK 
MyStreamInCallback(DWORD dwCookie, LPBYTE pbBuff, LONG cb, LONG *pcb)
{
   CFile* pFile = (CFile*) dwCookie;

   *pcb = pFile->Read(pbBuff, cb);

   return 0;
}

enum
{
	SCRIPT_WINDOW_WIDTH=975,
	SCRIPT_WINDOW_HEIGHT=470,
};

void CMonitorView::OnInitialUpdate() 
{
	CView::OnInitialUpdate();

	// Change the width and height so that all the controls I'm adding are visible.
	// The SWP_NOMOVE means keep the x,y coord of the top left corner the same as it
	// is currently, so that the passed x,y of 0,0 is ignored.
	GetParent()->SetWindowPos(NULL, 0,0, SCRIPT_WINDOW_WIDTH, SCRIPT_WINDOW_HEIGHT, SWP_NOMOVE);

	CFont info_bar_font;
	info_bar_font.CreateFont(
	   14,                        // nHeight
	   0,                         // nWidth
	   0,                         // nEscapement
	   0,                         // nOrientation
	   FW_NORMAL,                 // nWeight
	   FALSE,                     // bItalic
	   FALSE,                     // bUnderline
	   0,                         // cStrikeOut
	   ANSI_CHARSET,              // nCharSet
	   OUT_DEFAULT_PRECIS,        // nOutPrecision
	   CLIP_DEFAULT_PRECIS,       // nClipPrecision
	   DEFAULT_QUALITY,           // nQuality
	   DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
	   _T("Arial"));                 // lpszFacename

	mp_info = new CRichEditCtrl;
	mp_info->Create(ES_MULTILINE |
					ES_AUTOHSCROLL |
					ES_READONLY | 
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER | WS_VSCROLL | WS_HSCROLL | WS_THICKFRAME,
					CRect(10, 10, 950, 80), this, 111);
					
	mp_info->SetBackgroundColor(false,(COLORREF)0xc0c0c0);
	mp_info->SetFont(&info_bar_font);

	CFont font;
	font.CreateFont(
	   12,                        // nHeight
	   0,                         // nWidth
	   0,                         // nEscapement
	   0,                         // nOrientation
	   FW_NORMAL,                 // nWeight
	   FALSE,                     // bItalic
	   FALSE,                     // bUnderline
	   0,                         // cStrikeOut
	   ANSI_CHARSET,              // nCharSet
	   OUT_DEFAULT_PRECIS,        // nOutPrecision
	   CLIP_DEFAULT_PRECIS,       // nClipPrecision
	   DEFAULT_QUALITY,           // nQuality
	   DEFAULT_PITCH | FF_SWISS,  // nPitchAndFamily
	   _T("Courier"));            // lpszFacename

	mp_edit = new CRichEditCtrl;
	mp_edit->Create(ES_MULTILINE |
					ES_AUTOHSCROLL |
					ES_READONLY | 
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER | WS_VSCROLL | WS_HSCROLL | WS_THICKFRAME,
					CRect(10, 85, 600, 380), this, 112);
					
	mp_edit->SetBackgroundColor(false,(COLORREF)0xc0c0c0);
	mp_edit->SetFont(&font);

	// Set the tabs to be every 4 characters, the way everyone has slickedit set up.
	PARAFORMAT pf;
	pf.cbSize = sizeof(PARAFORMAT);
	pf.dwMask = PFM_TABSTOPS;
	pf.cTabCount=MAX_TAB_STOPS;
	for (int i=0; i<MAX_TAB_STOPS; ++i)
	{
		// The 4 means every 4 characters.
		// The 120 is to convert to twips
		pf.rgxTabs[i]=(i+1)*120*4;
	}
	mp_edit->SetParaFormat(pf);

	m_edit_old_width=590;
	m_edit_old_height=295;

	mp_button_step_into = new CDebuggerButton(WM_STEP_INTO);
	mp_button_step_into->Create("Step",
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER,
					CRect(10+0*90, 390, 90+0*90, 430), this, 114);

	mp_button_step_over = new CDebuggerButton(WM_STEP_OVER);
	mp_button_step_over->Create("Step Over",
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER,
					CRect(10+1*90, 390, 90+1*90, 430), this, 115);

	mp_button_stop = new CDebuggerButton(WM_STOP_SCRIPT);
	mp_button_stop->Create("Stop",
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER,
					CRect(10+2*90, 390, 90+2*90, 430), this, 113);
	// Disable for the moment cos it doesn't do anything yet.
	mp_button_stop->EnableWindow(FALSE);

	mp_auto_step_speed=new CSliderCtrl;
	mp_auto_step_speed->Create(TBS_AUTOTICKS |
							   TBS_BOTTOM |
							   WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER,
							   CRect(280, 390, 510, 430), this, 118);
	mp_auto_step_speed->SetRange(0,SLIDER_RANGE);
	mp_auto_step_speed->SetPos(SLIDER_RANGE);
	m_slider_pos=SLIDER_RANGE;

	

	mp_button_go = new CDebuggerButton(WM_START_SCRIPT);
	mp_button_go->Create("Go",
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER,
					CRect(520, 390, 600, 430), this, 116);

	// Create the tree for displaying the function params.
	mp_tree_function_params=new CMyTreeCtrl;
	mp_tree_function_params->Create(TVS_HASLINES | 
									TVS_LINESATROOT |
									TVS_HASBUTTONS |
									TVS_NOTOOLTIPS |
									WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER | WS_VSCROLL | WS_HSCROLL | WS_THICKFRAME,
									CRect(610, 85, 950, 170), this, 117);
	mp_tree_function_params->SetBkColor((COLORREF)0xc0c0c0);
	m_params_old_width=340;
	m_params_old_height=85;

	// Create the tree for displaying the callstack.
	mp_tree_callstack=new CMyTreeCtrl;
	mp_tree_callstack->Create(	TVS_HASLINES | 
					TVS_LINESATROOT |
					TVS_HASBUTTONS |
					TVS_NOTOOLTIPS |
					WS_CHILD | WS_VISIBLE | WS_TABSTOP | WS_BORDER | WS_VSCROLL | WS_HSCROLL | WS_THICKFRAME,
					CRect(610, 175, 950, 380), this, 118);
	mp_tree_callstack->SetBkColor((COLORREF)0xc0c0c0);
	m_callstack_old_width=340;
	m_callstack_old_height=205;
} 

void CMonitorView::update_spinner()
{
	++m_spinner;
	if (m_spinner>=8)
	{
		m_spinner=0;
	}
}

const char *CMonitorView::get_spinner()
{
	switch (m_spinner)
	{
	case 0:return "    ";
	case 1:return ".   ";
	case 2:return "..  ";
	case 3:return "... ";
	case 4:return "....";
	case 5:return " ...";
	case 6:return "  ..";
	case 7:return "   .";
	default: return "";
	}
}

void CMonitorView::handle_new_slider_pos(int new_pos)
{
	if (new_pos != m_slider_pos)
	{
		m_slider_pos=new_pos;

		KillTimer(1);

		// Translate the slider pos into a frame delay value.
		// The far right position means run the script at full speed, ie no
		// delay. The far left means totally stop the script so that it can
		// be single stepped.

		uint32 delay;
		if (m_slider_pos==0)
		{
			// For the far left position, send a special value indicating to
			// switch to single step mode.
			delay=0xffffffff;
		}
		else
		{
			// Otherwise convert so that the far right translates to a delay 
			// of 0 frames, the next mark left translates to a delay of 1, etc.
			delay=SLIDER_RANGE-m_slider_pos;
			
			if (delay)
			{
				SetTimer ( 1, 500/m_slider_pos, NULL );
			}

		}
	}
}

uint32 *CMonitorView::add_to_callstack_tree(int entryIndex, uint32 *p_data)
{
	uint8 *p_start=(uint8*)p_data;

	char p_buf[1024];
	uint32 script_name_checksum=*p_data++;
	uint32 line_number=*p_data++;
	char *p_source_file_name=(char*)p_data;
	int len=strlen(p_source_file_name)+1;
	len=(len+3)&~3;
	p_data+=len/4;

	Dbg_MsgAssert(entryIndex < MAX_CALLSTACK_ENTRIES,("Bad entryIndex of %d",entryIndex));
	mp_callstack_entries[entryIndex].mLineNumber=line_number;

	const char *p_full_file_name=GenerateFullQFileName(p_source_file_name);
	if (p_full_file_name)
	{
		Dbg_MsgAssert(strlen(p_full_file_name)<=MAX_FILE_NAME_CHARS,("File name '%s' too long",p_full_file_name));
		strcpy(mp_callstack_entries[entryIndex].mpFullFilename,p_full_file_name);
	}
	else
	{
		// This may happen if the environment variable PROJ_ROOT is not set, because
		// that will cause GenerateFullQFileName to return NULL.
		mp_callstack_entries[entryIndex].mpFullFilename[0]=0;
	}

	if (script_name_checksum)
	{
		sprintf(p_buf,"Script %s (%s, line %d)",Script::FindChecksumName(script_name_checksum),p_source_file_name,line_number);
	}
	else
	{
		sprintf(p_buf,"No script");
	}
	CMyTreeEntry *p_parent=mp_tree_callstack->AddEntry(p_buf);

	uint32 object_pointer=*p_data++;
	if (object_pointer)
	{
		uint32 object_type=*p_data++;
		uint32 object_id=*p_data++;
		bool objects_main_script_is_this=*p_data++;

		sprintf(p_buf,"Object of type %s",GetObjectTypeName(object_type));
		CMyTreeEntry *p_new_parent=mp_tree_callstack->AddEntry(p_buf,p_parent);
		
		sprintf(p_buf,"object=%s",Script::FindChecksumName(object_id));
		mp_tree_callstack->AddEntry(p_buf,p_new_parent);

		sprintf(p_buf,"CObject pointer=0x%08x",object_pointer);
		mp_tree_callstack->AddEntry(p_buf,p_new_parent);

		if (objects_main_script_is_this)
		{
			mp_tree_callstack->AddEntry("The object's main script is this script",p_new_parent);
		}
		else
		{
			mp_tree_callstack->AddEntry("The object's main script is not this script",p_new_parent);
		}


		CMyTreeEntry *p_tags_parent=mp_tree_callstack->AddEntry("Tags",p_new_parent);
		uint8 *p_old=(uint8*)p_data;
		p_data=(uint32*)InsertCStructIntoTree(mp_tree_callstack,p_tags_parent,(uint8*)p_data);

		uint32 bytes_read=(uint8*)p_data-p_old;
		int inc=( (bytes_read+3)&~3 )-bytes_read;
		p_data=(uint32*)(((uint8*)p_data)+inc);
	}
	else
	{
		mp_tree_callstack->AddEntry("No object",p_parent);
	}


	CMyTreeEntry *p_params_parent=mp_tree_callstack->AddEntry("Parameters",p_parent);
	p_data=(uint32*)InsertCStructIntoTree(mp_tree_callstack,p_params_parent,(uint8*)p_data);

	uint32 bytes_read=(uint8*)p_data-p_start;
	int inc=( (bytes_read+3)&~3 )-bytes_read;
	return (uint32*)(((uint8*)p_data)+inc);
}

void CMonitorView::update_file_window(bool update_highlight)
{
	if (m_current_displayed_callstack_entry >= m_num_callstack_entries)
	{
		if (m_num_callstack_entries==0)
		{
			//Printf("File window debug: This shouldn't happen ...");
			// Shouldn't happen, but just in case ...
			mp_edit->Clear();
			GetDocument()->SetTitle("");
			m_current_displayed_callstack_entry=0;
			mp_edit_window_current_file_name[0]=0;
			return;
		}
		else
		{
			m_current_displayed_callstack_entry=m_num_callstack_entries-1;
		}
	}

	char *p_filename=mp_callstack_entries[m_current_displayed_callstack_entry].mpFullFilename;
	int line_number=mp_callstack_entries[m_current_displayed_callstack_entry].mLineNumber;
	if (line_number==0)
	{
		//Printf("File window debug: line_number==0");
		mp_edit_window_current_file_name[0]=0;
		mp_edit->Clear();
		return;
	}

	if (p_filename[0]==0)
	{
		mp_edit->Clear();
		mp_edit->SetWindowText("Invalid source filename!\n\nCheck that your environment variable PROJ_ROOT\nis set to the correct path, eg \"c:\\skate5\"");
		mp_edit->UpdateWindow();
		return;
	}

	// TODO: Also compare the timestamp of the file, cos it might have changed
	// requiring a reload.
	if (stricmp(p_filename,mp_edit_window_current_file_name))
	{
		// The source file differs from what is currently loaded, so load in the new one.
		//mp_edit->Clear();
		mp_edit->SetSel(mp_edit->LineIndex(0),mp_edit->LineIndex(1));

		bool opened=false;
		CFile cFile;
		if (cFile.Open(TEXT(p_filename), CFile::modeRead))
		{
			//char p_blaa[1000];
			//sprintf(p_blaa,"Opened '%s' OK",p_filename);
			//Printf("File window debug: Q file opened OK");
			//Printf(p_blaa);
			opened=true;
		}
		else
		{
			char p_qn_name[MAX_FILE_NAME_CHARS+1];
			sprintf(p_qn_name,"%sn",p_filename);
			if (cFile.Open(TEXT(p_qn_name), CFile::modeRead))
			{
				//Printf("File window debug: QN file opened OK");
				opened=true;
			}
			else
			{
				char p_blaa[1000];
				sprintf(p_blaa,"File window debug: '%s' failed to open",p_qn_name);
				Printf(p_blaa);
			}
		}

		if (opened)
		{
			EDITSTREAM es;

			es.dwCookie = (DWORD) &cFile;
			es.pfnCallback = MyStreamInCallback; 
			mp_edit->StreamIn(SF_TEXT, es);	

			Dbg_MsgAssert(strlen(p_filename)<=MAX_FILE_NAME_CHARS,("Source file name %s too long",p_filename));
			strcpy(mp_edit_window_current_file_name,p_filename);
		}
	}

	long start,end;
	mp_edit->GetSel(start,end);
	int highlighted_line=mp_edit->LineFromChar(start)+1;
	// Only change the selected text if it no longer matches the required line_number.
	// This is so that when the script is stopped on a line, such as a wait command, the
	// text in the box can still be manually scrolled.
	if (highlighted_line != line_number || update_highlight==DO_UPDATE_HIGHLIGHT)
	{
		mp_edit->SetSel(mp_edit->LineIndex(line_number-1),mp_edit->LineIndex(line_number));
		mp_edit->HideSelection(FALSE, TRUE);

		// Selecting the line does not always bring it into view, so do a bit of logic
		// to ensure that it is always in view.
		int first_visible_line=mp_edit->GetFirstVisibleLine()+1;
		if (line_number-first_visible_line>=20)
		{
			// If the selected line is off the bottom of the edit box, scroll the contents to
			// bring it into view. The 20 is how many lines high the box is, and the +5
			// is just to bring it up a bit more.
			mp_edit->LineScroll(line_number-first_visible_line-20 +5);
		}
		else if (line_number<first_visible_line)
		{
			// If the selected line is off the top, scroll the contents down by passing
			// LineScroll a negative number. The -5 is just an extra bit to bring it
			// definitely into view.
			mp_edit->LineScroll(line_number-first_visible_line -5);
		}
	}

	mp_edit->UpdateWindow();
}

/////////////////////////////////////////////////////////////////////////////
// CMonitorView message handlers
afx_msg LRESULT CMonitorView::OnWatchInfo(WPARAM wParam, LPARAM lParam)
{
	update_spinner();

	uint32 *p_data=(uint32*)lParam;

	// Read out the script info from p_data.
	// See CScript::TransmitInfoToDebugger() in gel\scripting\script.cpp
	// to see the game code which writes out this info.
	uint32 cscript_pointer=*p_data++;
	uint32 script_id=*p_data++;
	float last_instruction_time=*((float*)p_data);
	++p_data;
	float total_instruction_time=*((float*)p_data);
	++p_data;
	sint32 originating_script_line_number=(sint32)*p_data++;
	uint32 originating_script_name_checksum=*p_data++;
	char *p_comment=(char*)p_data;
	int comment_length=strlen(p_comment)+1;
	comment_length=(comment_length+3)&~3;
	p_data+=comment_length/4;


	// Update the title
	char p_title[100];
	sprintf(p_title,"&CScript=0x%08x,   m_unique_id=0x%08x   %s",cscript_pointer,script_id,get_spinner());
	CDocument *p_doc=GetDocument();
	p_doc->SetTitle(p_title);

	// Update the last function params tree.
	mp_tree_function_params->RemoveAllEntries();
	CMyTreeEntry *p_function_params_parent=mp_tree_function_params->AddEntry("Last params sent");
	uint8 *p_old=(uint8*)p_data;
	p_data=(uint32*)InsertCStructIntoTree(mp_tree_function_params,p_function_params_parent,(uint8*)p_data);

	uint32 bytes_read=(uint8*)p_data-p_old;
	int inc=( (bytes_read+3)&~3 )-bytes_read;
	p_data=(uint32*)(((uint8*)p_data)+inc);
	mp_tree_function_params->CopyToDisplayedTree();

	// Update the callstack tree
	m_num_callstack_entries=*p_data++;

	mp_tree_callstack->RemoveAllEntries();

	for (uint32 i=0; i<m_num_callstack_entries; ++i)
	{
		p_data=add_to_callstack_tree(i,p_data);
	}

	mp_tree_callstack->CopyToDisplayedTree();


	// Update the file window, ie make sure it is displaying the correct file
	// and highlight position.
	update_file_window();

	// Display the various flags.
	uint32 spawned=*p_data++;
	uint32 wait_type=*p_data++;
	uint32 paused=*p_data++;
	uint32 not_session_specific=*p_data++;
	Script::ESingleStepMode single_step_mode=(Script::ESingleStepMode)*p_data++;

	char p_temp[MAX_INFO_BAR_CHARS+1];
	if (p_comment[0])
	{
		sprintf(p_temp,"Originating C-code info:\t%s\n",p_comment);
	}
	else
	{
		sprintf(p_temp,"Originating C-code info:\tN/A\n");
	}

	if (originating_script_name_checksum)
	{
		sprintf(p_temp+strlen(p_temp),"Originating script info:\tScript %s, Line %d\n",
				Script::FindChecksumName(originating_script_name_checksum),
				originating_script_line_number);
	}
	else
	{
		strcat(p_temp,"Originating script info:\tN/A\n");
	}

	if (spawned)
	{
		strcat(p_temp,"Spawned, ");
	}
	else
	{
		strcat(p_temp,"Not spawned, ");
	}

	if (paused)
	{
		strcat(p_temp,"Paused, ");
	}
	else
	{
		strcat(p_temp,"Not paused, ");
	}

	if (not_session_specific)
	{
		strcat(p_temp,"Not session specific, ");
	}
	else
	{
		strcat(p_temp,"Session specific, ");
	}

	sprintf(p_temp+strlen(p_temp),"mWaitType=%s, ",GetWaitTypeName((Script::EWaitType)wait_type));

	if (single_step_mode != Script::OFF)
	{
		strcat(p_temp,"Stopped in debugger");
	}
	else
	{
		strcat(p_temp,"Running");
	}

	// last_instruction_time will be <0 to indicate not to display it when not in 
	// single step mode.
	if (last_instruction_time>=0.0f)
	{
		char p_foo[100];
		sprintf(p_foo,"\nTime taken by last command = %f s \tTotal for this frame = %f s",last_instruction_time,total_instruction_time);
		strcat(p_temp,p_foo);

	}

	if (strcmp(p_temp,mp_info_bar_text))
	{
		strcpy(mp_info_bar_text,p_temp);
		mp_info->SetWindowText(mp_info_bar_text);
	}

	// If in single step mode, automatically collapse all the callstack entries up to
	// the last one, and make sure that one is expanded.
	// Then within that entry collapse all the entries apart from the last (the parameter)
	// and expand the last.
	if (single_step_mode!=Script::OFF)
	{
		// Expand the last entry in the callstack tree, collapsing those before.
		HTREEITEM h_last_item=mp_tree_callstack->CollapseAllSiblingsButExpandLast(mp_tree_callstack->GetRootItem());

		// Then expand the last sibling of the last entry, which will be the parameters
		// for the most recent script call.
		mp_tree_callstack->CollapseAllSiblingsButExpandLast(mp_tree_callstack->GetChildItem(h_last_item));

		// Make sure the file window is displaying the current PC
		m_current_displayed_callstack_entry = m_num_callstack_entries-1;
		update_file_window(DO_UPDATE_HIGHLIGHT);

		// Make sure that the function params are expanded to be visible.
		mp_tree_function_params->CollapseAllSiblingsButExpandLast(mp_tree_function_params->GetRootItem());
	}

	// These seem to be necessary to make sure the above changes take effect.
	mp_tree_callstack->UpdateWindow();
	mp_tree_function_params->UpdateWindow();

	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnBasicScriptInfo(WPARAM wParam, LPARAM lParam)
{
	uint32 *p_data=(uint32*)lParam;

	uint32 script_id=*p_data++;
	uint32 script_name_checksum=*p_data++;
	uint32 line_number=*p_data++;
	char *p_source_file_name=(char*)p_data;
	int len=strlen(p_source_file_name)+1;
	len=(len+3)&~3;
	p_data+=len/4;

	// TODO Fix this, load the filename and line number into callstack entry 0 ...
	//update_file_window(p_source_file_name,line_number);
	
	mp_tree_callstack->DeleteAllItems();
	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnStopScript(WPARAM wParam, LPARAM lParam)
{
	mp_auto_step_speed->SetPos(0);
	handle_new_slider_pos(0);

	Net::MsgDesc msg;
	msg.m_Data=&m_id;
	msg.m_Length=4;
	msg.m_Id = GameNet::vMSG_ID_DBG_STOP;
	gClient->EnqueueMessageToServer(&msg);

	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnStepInto(WPARAM wParam, LPARAM lParam)
{
	Net::MsgDesc msg;
	msg.m_Data=&m_id;
	msg.m_Length=4;
	msg.m_Id = GameNet::vMSG_ID_DBG_STEP_INTO;
	gClient->EnqueueMessageToServer(&msg);

	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnStepOver(WPARAM wParam, LPARAM lParam)
{
	Net::MsgDesc msg;
	msg.m_Data=&m_id;
	msg.m_Length=4;
	msg.m_Id = GameNet::vMSG_ID_DBG_STEP_OVER;
	gClient->EnqueueMessageToServer(&msg);

	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnGo(WPARAM wParam, LPARAM lParam)
{
	mp_auto_step_speed->SetPos(SLIDER_RANGE);
	handle_new_slider_pos(SLIDER_RANGE);

	Net::MsgDesc msg;
	msg.m_Data=&m_id;
	msg.m_Length=4;
	msg.m_Id = GameNet::vMSG_ID_DBG_GO;
	gClient->EnqueueMessageToServer(&msg);

	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnChangeFile(WPARAM wParam, LPARAM lParam)
{
	m_current_displayed_callstack_entry=wParam;
	update_file_window(DO_UPDATE_HIGHLIGHT);

	return (LRESULT)0;
}

afx_msg LRESULT CMonitorView::OnScriptDied(WPARAM wParam, LPARAM lParam)
{
	// Close this window.
	GetDocument()->OnCmdMsg(ID_FILE_CLOSE, 0, 0, 0);
	return (LRESULT)0;
}

int CMonitorView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;


	m_id=gNewWindowParams.mId;

	gNewWindowParams.mInfo.mViewHWND=m_hWnd;
	gNewWindowParams.mInfo.mpDocument=GetDocument();

	CWindowInfo *p_new=new CWindowInfo;
	*p_new=gNewWindowParams.mInfo;
	gpWindowHashTable->PutItem(m_id,p_new);
	
	return 0;
}

void CMonitorView::OnHScroll(UINT nSBCode, UINT nPos, CScrollBar* pScrollBar) 
{
	// TODO: Add your message handler code here and/or call default

	handle_new_slider_pos(((CSliderCtrl*)pScrollBar)->GetPos());

	CView::OnHScroll(nSBCode, nPos, pScrollBar);
}

void CMonitorView::OnTimer(UINT nIDEvent) 
{
	// TODO: Add your message handler code here and/or call default

	Net::MsgDesc msg;
	msg.m_Data=&m_id;
	msg.m_Length=4;
	msg.m_Id = GameNet::vMSG_ID_DBG_STEP_INTO;
	gClient->EnqueueMessageToServer(&msg);
	
	CView::OnTimer(nIDEvent);
}

void CMonitorView::OnMouseMove(UINT nFlags, CPoint point) 
{
	// TODO: Add your message handler code here and/or call default
	
	CView::OnMouseMove(nFlags, point);
}

void CMonitorView::get_window_position(CWnd *p_window, RECT *p_rect)
{
	RECT main_rect;
	GetWindowRect(&main_rect);

	p_window->GetWindowRect(p_rect);
	p_rect->left-=main_rect.left;
	p_rect->top-=main_rect.top;
	p_rect->right-=main_rect.left;
	p_rect->bottom-=main_rect.top;

	p_rect->left-=2;
	p_rect->top-=2;
	p_rect->right-=2;
	p_rect->bottom-=2;
}

bool CMonitorView::adjust_positions()
{
	// Get the current coords of the info window.
	RECT info_coords;
	get_window_position(mp_info,&info_coords);

	// Get the current coords of the file display window.
	RECT edit_coords;
	get_window_position(mp_edit,&edit_coords);


	// Lock the top of the file window to be 5 pixels below the info window.
	if (edit_coords.top != info_coords.bottom+5)
	{
		mp_edit->SetWindowPos(NULL, edit_coords.left, 
									info_coords.bottom+5, 
									m_edit_old_width,
									m_edit_old_height,
									0);
		return true;
	}

	// Get the current coords of the last-params tree.
	RECT params_coords;
	get_window_position(mp_tree_function_params,&params_coords);

	// Lock the top of the last-params tree to be 5 pixels below the info window.
	if (params_coords.top != info_coords.bottom+5)
	{
		mp_tree_function_params->SetWindowPos(NULL, params_coords.left, 
													info_coords.bottom+5, 
													m_params_old_width,
													m_params_old_height,
													0);
		return true;
	}


	// Get the current coords of the callstack tree.
	RECT callstack_coords;
	get_window_position(mp_tree_callstack,&callstack_coords);

	// Lock the top of the callstack tree to be 5 pixels below the last-params tree.
	if (callstack_coords.top != params_coords.bottom+5)
	{
		mp_tree_callstack->SetWindowPos(NULL, callstack_coords.left, 
											  params_coords.bottom+5, 
											  m_callstack_old_width,
											  m_callstack_old_height,
											  0);
		return true;
	}


	// Lock the tree windows to be next to the right edge of the file window.
	int required_x=edit_coords.right+10;
	if (required_x < 610)
	{
		// Except don't follow the file window too far to the left otherwise
		// the tree windows migh impinge on the buttons.
		required_x=610;
	}
	if (params_coords.left != required_x ||
		callstack_coords.left != required_x)
	{
		mp_tree_function_params->SetWindowPos(NULL, required_x, 
													params_coords.top, 
													m_params_old_width,
													m_params_old_height,
													0);
		mp_tree_callstack->SetWindowPos(NULL, required_x, 
											  callstack_coords.top, 
											  m_callstack_old_width,
											  m_callstack_old_height,
											  0);

		return true;
	}

	// Get the current coords of the step-into button.
	RECT button_coords;
	get_window_position(mp_button_step_into,&button_coords);

	// Lock the buttons & slider beneath the file window.
	if (button_coords.top != edit_coords.bottom+10)
	{
		int new_y=edit_coords.bottom+10;

		mp_button_step_into->SetWindowPos(NULL, 10+90*0,new_y,	80,40, 0);
		mp_button_step_over->SetWindowPos(NULL, 10+90*1,new_y,	80,40, 0);
		mp_button_stop->SetWindowPos	 (NULL, 10+90*2,new_y,	80,40, 0);
		mp_auto_step_speed->SetWindowPos (NULL, 280,new_y,		230,40, 0);
		mp_button_go->SetWindowPos		 (NULL, 520,new_y,		80,40, 0);
	}

	return false;
}

void CMonitorView::OnPaint() 
{
	CPaintDC dc(this); // device context for painting
	
	if (mp_edit==NULL ||
		mp_info==NULL ||
		mp_button_stop==NULL ||
		mp_button_step_into==NULL ||
		mp_button_step_over==NULL ||
		mp_button_go==NULL ||
		mp_auto_step_speed==NULL ||
		mp_tree_function_params==NULL ||
		mp_tree_callstack==NULL)
	{
		return;
	}

	// Fiddle with the positions until everything is OK.
	while (adjust_positions());

	// Remember the new widths & heights.
	RECT coords;
	get_window_position(mp_tree_function_params,&coords);
	m_params_old_height=coords.bottom-coords.top;
	m_params_old_width=coords.right-coords.left;

	get_window_position(mp_tree_callstack,&coords);
	m_callstack_old_height=coords.bottom-coords.top;
	m_callstack_old_width=coords.right-coords.left;

	get_window_position(mp_edit,&coords);
	m_edit_old_height=coords.bottom-coords.top;
	m_edit_old_width=coords.right-coords.left;
}

void CMonitorView::OnDestroy() 
{
	CView::OnDestroy();
	
	// Add to the list of scripts to be ignored, so that any script-info packets 
	// for that script that arrive late get ignored.
	gIgnoredScriptsIdList.AddId(m_id);

	// This unpauses the script if it was in single-step mode when the window was closed.
	OnGo(0,0);

	// Tell the PS2 to stop watching this script.
	Net::MsgDesc msg;
	msg.m_Id = GameNet::vMSG_ID_DBG_STOP_WATCHING_THIS_CSCRIPT;
	msg.m_Length=4;
	msg.m_Data=&m_id;
	gClient->EnqueueMessageToServer(&msg);
}

BOOL CMonitorView::OnEraseBkgnd(CDC* pDC)
{
	// TODO: Add your message handler code here and/or call default
	// Set brush to desired background color
	CBrush backBrush(RGB(0xc0, 0xc0, 0xc0));
	// Save old brush
	CBrush* pOldBrush = pDC->SelectObject(&backBrush);
	CRect rect;
	pDC->GetClipBox(&rect); // Erase the area needed
	pDC->PatBlt(rect.left, rect.top, rect.Width(), 	rect.Height(), PATCOPY);
	pDC->SelectObject(pOldBrush);

	return TRUE;


	//return CView::OnEraseBkgnd(pDC);
}
