/*
	MipMapGen.h
	Mip Map Generator class
*/

#ifndef __MIPMAPGEN__
#define __MIPMAPGEN__

#include "ImageData.h"
#include "Quantizer.h"
#include "llist.h"

class MipMapGen
{
	LinkList<char*> srcList;		// List of source files to use in conversion/generation operation
	char*           destFile;		// The file to output
	char*           outPath;		// The output file path
	char*           outPalFile;		// Output palette file
	char*           inPalFile;		// Input palette file
	int             outDepth;		// Output bit depth of files
	int             curOutDepth;	// Current output bit depth
	bool            bGenMips;		// True if mipmaps should be generated
	unsigned int    mipTargetWidth;	// Smallest width of mipmap
	unsigned int    mipTargetHeight;// Smallest height of mipmap
	unsigned int    scaleWidth;		// Width that the image should be scaled to
	unsigned int    scaleHeight;	// Height that the image should be scaled to
	bool            bDumpTarga;		// Dump a targa image as output
	bool            bUTransparent;	// Makes the horizontal edges transparent
	bool            bVTransparent;	// Makes the vertical edges transparent
	int             quantColors;	// Number of colors to quantize palette down to (0 to not quantize)
	bool            bReduceCN;		// Reduce using closest neighbor method (false uses box filter)
	bool            bDither;		// True if the image should be dithered after quantization
	bool            bUseTransColor;	// True if the user defined transparency color should be used
	color24         transColor;		// A user-defined transparency color to override for alpha 0
	int             limitMips;      // Limits number of mips (-1 don't limit)
	bool            bDelOldMips;	// True if mips greater than the generated amount should be deleted
	bool            bReport;		// True if to just generate info on given png

	void Proc8BitBoxReduce(ImageData* destImage, ImageData* srcImage);
	void Proc16BitBoxReduce(ImageData* destImage, ImageData* srcImage);
	void Proc24BitBoxReduce(ImageData* destImage, ImageData* srcImage);
	void Proc32BitBoxReduce(ImageData* destImage, ImageData* srcImage);

	void Proc32BitBoxReduceACN(ImageData* destImage,
		                       ImageData* srcImage,
							   unsigned int destWidth,
							   unsigned int destHeight);

	void Dither8(ImageData* ditheredImage,
		         ImageData* quantizedImage,
				 ImageData* origImage);

	void Dither16(ImageData* ditheredImage,
		          ImageData* quantizedImage,
				  ImageData* origImage);

	void Dither24(ImageData* ditheredImage,
		          ImageData* quantizedImage,
				  ImageData* origImage);

	void Dither32(ImageData* ditheredImage,
		          ImageData* quantizedImage,
				  ImageData* origImage);

	void EdgeTransU(ImageData* image);
	void EdgeTransV(ImageData* image);

	bool IsTransU(ImageData* image);
	bool IsTransUTop(ImageData* image);
	bool IsTransUBottom(ImageData* image);

	bool IsTransV(ImageData* image);
	bool IsTransVLeft(ImageData* image);
	bool IsTransVRight(ImageData* image);
	bool IsCutout(ImageData* image);

	void AssignTrans(ImageData* destimage, color24 transColor);

public:
	ImageData  origImage;
	ImageData  destImage;
	Quantizer  quant;

	MipMapGen();
	~MipMapGen();

	void MakeUTransparentTop(ImageData* destImage,
		                     ImageData* srcImage);

	void MakeUTransparentBottom(ImageData* destImage,
		                        ImageData* srcImage);

	void MakeUTransparent(ImageData* destImage,
		                  ImageData* srcImage);

	void MakeVTransparent(ImageData* destImage,
		                  ImageData* srcImage);

	void MakeVTransparentLeft(ImageData* destImage,
		                      ImageData* srcImage);

	void MakeVTransparentRight(ImageData* destImage,
		                       ImageData* srcImage);

	void ResizeImageCN(ImageData* destImage,
		               ImageData* srcImage, 
					   unsigned int destWidth, 
					   unsigned int destHeight);

	void ResizeImageBox(ImageData* destImage,
		                ImageData* srcImage, 
					    unsigned int destWidth, 
					    unsigned int destHeight);

	void Dither(ImageData* ditheredImage,
		        ImageData* quantizedImage,
				ImageData* origImage);

	void DumpHelp();

	// Parses the command line options
	void Init(int srgc,char** argv);

	// Processes the command line options and return number of mip levels processed
	int  Proc();
	void ProcScale(ImageData* img);

	void DeleteWildcard(char* path,char* wildcard);
};

#endif
