#include "FuncEnter.h"

/*
	PropList.cpp
	Property List Window Class
	
	Adam Lippmann
	adam@neversoft.com
*/

#include "PropList.h"
#include "PropListInt.h"
#include "../Resource.h"
#include "SpinEdit.h"
#include "ScriptSelect.h"
#include "NodeSelect.h"
#include "../UI/SlideNum.h"
#include "../UI/MultiList.h"
#include "../UI/FlagSelect.h"
#include "../UI/MtlSelect.h"
#include "../UI/Static.h"
#include "../UI/ColorPopup.h"
#include "../UI/OptFile.h"
#include <stdio.h>
#include "../PropEdit/ParseFuncs.h"
#include "../MaxScriptInt.h"
#include "expr.h"
#include "path.h"
#include <string.h>

#define PROP_SPACING          4     // The vertical spacing in pixels between object property fields
#define PROP_DROPDOWNHEIGHT 300		// The drop down height 
#define STATIC_ID           100		// Base offset for static IDs
#define PROP_HEIGHT          20		// Height of the property and text fields
#define PROP_HSPACING         5		// Horizontal spacing in pixels between field desc and field
#define PROP_LEFTSPACING      2		// Number of pixels from the left of the window edge to display fields
#define PROP_RIGHTSPACING     4		// Number of pixels from the right of the window edge to display fields
#define PROP_TOP              2		// Number of pixels from the top of the window to the first field
#define SCROLL_WIDTH         10		// Width of scrollbar
#define SCROLL_SPACE          5

#define TXT_HIGHLIGHT        RGB(215,215,215)
#define TXT_HIGHLIGHT_RED    RGB(250,200,200)
#define TXT_HIGHLIGHT_GREEN  RGB(200,250,200)
#define TXT_HIGHLIGHT_BLUE   RGB(200,200,250)
#define TXT_HIGHLIGHT_PURPLE RGB(250,200,250)
#define TXT_HIGHLIGHT_WHITE  RGB(255,255,255)

extern Interface* gInterface;

/////////////////////////////// PropList

PropList::PropList(HINSTANCE hInstance,int cols) :
	ttip(hInstance)
{ FUNC_ENTER("PropList::PropList"); 
	this->hInstance = hInstance;

	hwndParent=NULL;
	hwndScroll=NULL;
	x=y=0;
	numCols=cols;

	// Clear out callbacks
	fpFocus=NULL;
	pFocusData=NULL;
	
	fpEscape=NULL;
	pEscapeData=NULL;

	fpApply=NULL;
	pApplyData=NULL;

	fpDblClick=NULL;
	pDblClickData=NULL;

	fpChange=NULL;
	pChangeData=NULL;

	fpExtList=NULL;
	pExtListData=NULL;

	fpSProc=NULL;
	pSProcData=NULL;
	OldStaticProc=NULL;

	bHasSliders=FALSE;
	bHasApply=true;

	bSetValLocked=true;
	bManualRedraw=false;

	lastModID=-1;
	curIndex=-1;
	linkAddPos = props.GetTail();

	// Initialize the property list window class
	wndclass.style         = CS_HREDRAW|CS_VREDRAW;
	wndclass.lpfnWndProc   = RedirectWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "PropList";
	
	RegisterClass(&wndclass);
	
	// Register our own custom control interfaces to be used in the list
	SpinEdit::Register(hInstance);

	// Initialize the brush used to paint the background in the property name child windows
	hBGBrush=CreateSolidBrush(TXT_HIGHLIGHT);
	hBGBrushR=CreateSolidBrush(TXT_HIGHLIGHT_RED);
	hBGBrushG=CreateSolidBrush(TXT_HIGHLIGHT_GREEN);
	hBGBrushB=CreateSolidBrush(TXT_HIGHLIGHT_BLUE);
	hBGBrushP=CreateSolidBrush(TXT_HIGHLIGHT_PURPLE);
	hBGBrushW=CreateSolidBrush(TXT_HIGHLIGHT_WHITE);

	// Create the custom font for the dropdown comboboxes so we can fit more onscreen
	hFont = CreateFont(14,0,0,0,0,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, "");

	bFloatingWindow=false;
	bAttachedWindow=false;
}

PropList::PropList(HINSTANCE hInstance,HWND hwdParent,int x,int y,int width,int height,CStr DlgText) :
	ttip(hInstance)
{ FUNC_ENTER("PropList::PropList"); 
	this->hInstance = hInstance;

	numCols=1;
	hwndParent=hwdParent;
	hwndScroll=NULL;
	this->x=x;
	this->y=y;
	this->width=width;
	origWidth=width;
	this->height=height;

	// Clear out callbacks
	fpFocus=NULL;
	pFocusData=NULL;
	
	fpEscape=NULL;
	pEscapeData=NULL;

	fpApply=NULL;
	pApplyData=NULL;

	fpChange=NULL;
	pChangeData=NULL;

	fpExtList=NULL;
	pExtListData=NULL;

	fpDblClick=NULL;
	pDblClickData=NULL;

	fpSProc=NULL;
	pSProcData=NULL;
	OldStaticProc=NULL;

	bHasSliders=FALSE;
	bHasApply=true;

	bSetValLocked=true;

	lastModID=-1;
	curIndex=-1;
	linkAddPos = props.GetTail();

	// Initialize the property list window class
	wndclass.style         = CS_HREDRAW|CS_VREDRAW;
	wndclass.lpfnWndProc   = RedirectWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = LoadIcon(NULL,IDI_APPLICATION);
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(LTGRAY_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "PropList";
	
	RegisterClass(&wndclass);

	if (DlgText==CStr(""))
	{
		// Create the property list floating window
		hwnd=CreateWindow("PropList",
						  NULL,
						  WS_POPUP|WS_DLGFRAME,
						  x,y,
						  width,height,
						  hwdParent,
						  NULL,
						  hInstance,
						  NULL);

		bFloatingWindow=false;
	}
	else
	{
		// Create a floating property list window
		hwnd=CreateWindow("PropList",
						  DlgText,
						  WS_DLGFRAME,
						  x,y,
						  width,height,
						  hwdParent,
						  NULL,
						  hInstance,
						  NULL);

		bFloatingWindow=true;
	}

	SetWindowLong(hwnd,GWL_USERDATA,(LONG)this);

	// Set window as topmost popup
	SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,height,0);

	if (DlgText!=CStr(""))
		ShowWindow(hwnd,SW_SHOW);

	RECT winRect;
	GetClientRect(hwnd,&winRect);

	winRect.left+=PROP_LEFTSPACING;
	winRect.top +=PROP_TOP;

	rectRefTXT.left=winRect.left;
	rectRefTXT.top=winRect.top;
	rectRefTXT.right=winRect.left+(width/2)-PROP_HSPACING;
	rectRefTXT.bottom=rectRefTXT.top+PROP_HEIGHT;

	rectRefFLD.left=winRect.left+width/2;
	rectRefFLD.top =winRect.top;
	rectRefFLD.right=rectRefFLD.left+(width/2)-PROP_HSPACING;
	rectRefFLD.bottom=rectRefFLD.top+PROP_HEIGHT;

	rectRefTXT.right-=(SCROLL_WIDTH+SCROLL_SPACE)/2;
	rectRefFLD.right-=(SCROLL_WIDTH+SCROLL_SPACE)/2;

	//width+SCROLL_WIDTH+SCROLL_SPACE

	TXTWidth=rectRefTXT.right-rectRefTXT.left;
	FLDWidth=rectRefFLD.right-rectRefFLD.left;

	TXTHeight=rectRefTXT.bottom-rectRefTXT.top;
	FLDHeight=rectRefFLD.bottom-rectRefFLD.top;
	
	// Initialize the brush used to paint the background in the property name child windows
	hBGBrush=CreateSolidBrush(TXT_HIGHLIGHT);
	hBGBrushR=CreateSolidBrush(TXT_HIGHLIGHT_RED);
	hBGBrushG=CreateSolidBrush(TXT_HIGHLIGHT_GREEN);
	hBGBrushB=CreateSolidBrush(TXT_HIGHLIGHT_BLUE);
	hBGBrushP=CreateSolidBrush(TXT_HIGHLIGHT_PURPLE);
	hBGBrushW=CreateSolidBrush(TXT_HIGHLIGHT_WHITE);

	// Create the custom font for the dropdown combo boxes
	hFont = CreateFont(14,0,0,0,0,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, "");

	bAttachedWindow=false;
}

PropList::~PropList()
{ FUNC_ENTER("PropList::~PropList"); 
	ttip.Close();
	DeleteObject(hBGBrush);
	DeleteObject(hBGBrushR);
	DeleteObject(hBGBrushG);
	DeleteObject(hBGBrushB);
	DeleteObject(hBGBrushP);
	DeleteObject(hBGBrushW);
	
	if (hFont)
		DeleteObject(hFont);

	DestroyWindow(hwnd);
}

void PropList::SaveValues()
{ FUNC_ENTER("PropList::SaveValues"); 
	savedValues.Clear();

	int numProps=props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		CStr value;
		GetValue(i,value);
		savedValues.Add(&value);
	}
}

void PropList::SaveModStates()
{ FUNC_ENTER("PropList::SaveModStates"); 
	modStates.Clear();

	int numProps=props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		bool bModState = WasUpdated(i);
		modStates.Add(&bModState);
	}
}

void PropList::RestoreModStates()
{ FUNC_ENTER("PropList::RestoreModStates"); 
	int nProps = modStates.GetSize();

	for(int i = 0; i < nProps; i++)
		SetMod(i, modStates[i]);
}

void PropList::SaveProperties()
{ FUNC_ENTER("PropList::SaveProperties"); 
	savedRestoreValues.Clear();
	RestoreProp     rprop;

	int numProps = props.GetSize();

	for(int i = 0; i < numProps; i++)
	{
		rprop.name = GetName(i);
		GetValue(i, rprop.value);
		
		savedRestoreValues.Add(&rprop);
	}
}

void PropList::RestoreProperties()
{ FUNC_ENTER("PropList::RestoreProperties"); 
	Link<RestoreProp>* curprop = savedRestoreValues.GetHead();

	while(curprop)
	{
		SetValue(curprop->data.name, curprop->data.value);
		curprop = curprop->next;
	}
}

void PropList::GetValues(LinkList<CStr>* list)
{ FUNC_ENTER("PropList::GetValues"); 
	int numProps = props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		CStr value;
		GetValue(i,value);
		list->Add(&value);
	}
}

void PropList::SetValues(LinkList<CStr>* list)
{ FUNC_ENTER("PropList::SetValues"); 
	int numProps = list->GetSize();

	for(int i=0;i<numProps;i++)
		SetValue(i,(*list)[i]);
}

void PropList::RestoreValues()
{ FUNC_ENTER("PropList::RestoreValues"); 
	Link<CStr>* curNode=savedValues.GetHead();
	int  index=0;
	bool bUpdated;

	while(curNode)
	{
		// Set values and retain modification status
		bUpdated=WasUpdated(index);
		SetValue(index,curNode->data);
		SetMod(index++,bUpdated);
		curNode=curNode->next;
	}
}

void PropList::DestroyUI()
{ FUNC_ENTER("PropList::DestroyUI"); 
	bSetValLocked=true;

	// Destroy all windows
	Link<Property>* plink=props.GetHead();

	while(plink)
	{
		// Destroy UI child windows
		
		// Remove subclass on the property text window
		SetWindowLong(plink->data.hwndTxt,GWL_WNDPROC,(LONG)OldStaticProc);

		DestroyWindow(plink->data.hwndTxt);
		DestroyWindow(plink->data.hwndFld);
		
		plink=plink->next;
	}

	if (hwndApply)
		DestroyWindow(hwndApply);

	if (hwndScroll)
		DestroyWindow(hwndScroll);
}

void PropList::Clear()
{ FUNC_ENTER("PropList::Clear"); 
	bSetValLocked=true;

	// Destroy all windows
	Link<Property>* plink=props.GetHead();
	ColorData*      cdata;

	DestroyUI();

	while(plink)
	{
		// Handle any special cleanup
		switch(plink->data.ptype)
		{
		case PROPTYPE_COLOR:
			cdata=(ColorData*)plink->data.pdata;
			ReleaseIColorSwatch(cdata->IColor);
			cdata->IColor=NULL;
			break;
		}

		plink=plink->next;
	}

	// Reset property management list
	props.Clear();
	
	// Reset temporary property value buffer
	propVals.Clear();

	width=origWidth;

	// Reset last modified flag
	lastModID=-1;
	curIndex=-1;
	linkAddPos = props.GetTail();
}

void PropList::AddStatic(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddStatic"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_STATIC;
	newProp.pdata = 0;

	StaticCtl* sc = new StaticCtl(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = sc;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
}

void PropList::AddEdit(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddEdit"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_EDIT;
	newProp.pdata      =NULL;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = props.AddAfterLink(linkAddPos,&newProp);
}

SpinEdit* PropList::AddSpinEdit(CStr propName,float min,float max,float incr,CStr ToolTip)
{ FUNC_ENTER("PropList::AddSpinEdit"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_SPINEDIT;
	newProp.pdata = 0;

	SpinEdit* spe = new SpinEdit;

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = spe;
	
	spe->SetLimits(min,max);
	spe->SetIncr(incr);

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return spe;
}

SliderNum* PropList::AddSliderNum(CStr propName,int min,int max,CStr ToolTip)
{ FUNC_ENTER("PropList::AddSliderNum"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_SLIDERNUM;
	newProp.pdata = 0;

	SliderNum* sn = new SliderNum(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = sn;
	
	sn->SetLimits(min,max);

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return sn;
}

ColorPopup* PropList::AddColorPopup(CStr propName,int min,int max,CStr ToolTip)
{ FUNC_ENTER("PropList::AddColorPopup"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_COLORPOPUP;
	newProp.pdata = 0;

	ColorPopup* cp = new ColorPopup(hInstance);

	cp->SetLimits(min,max);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = cp;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return cp;
}

ListData* PropList::AddList(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddList"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;
	
	newProp.ptype      =PROPTYPE_LIST;
	newProp.pdata      =0;

	// List data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link=props.AddAfterLink(linkAddPos,&newProp);;
	link->data.pdata    =new ListData;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;

	return (ListData*)link->data.pdata;
}

ListData* PropList::AddExtList(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddExtList"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_EXTLIST;
	newProp.pdata      =0;

	// List data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link=props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata    =new ListData;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return (ListData*)link->data.pdata;
}

void PropList::AddSlider(CStr propName,int min,int max,CStr ToolTip)
{ FUNC_ENTER("PropList::AddSlider"); 
	Property    newProp;
	SliderData* sdata;

	bHasSliders=TRUE;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_SLIDER;
	newProp.pdata      =0;

	// Slider data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link  =props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata=sdata=new SliderData;

	sdata->min=min;
	sdata->max=max;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
}

FileData* PropList::AddFile(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddFile"); 
	Property   newProp;
	FileData*  fdata;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_FILE;
	newProp.pdata      =0;

	// File data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link  =props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata=fdata=new FileData;

	fdata->Fltr[0]='\0';
	fdata->Fltr[1]='\0';

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return fdata;
}

ColorData* PropList::AddColor(CStr propName,CStr ToolTip,COLORREF Color,CStr DlgText)
{ FUNC_ENTER("PropList::AddColor"); 
	Property   newProp;
	ColorData* cdata;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_COLOR;
	newProp.pdata      =0;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	// Color data must be added after the property is created
	// so it doesn't get deleted by the init property destructor
	Link<Property>* link  =props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata=cdata=new ColorData;

	cdata->IColor  =NULL;
	cdata->DefColor=Color;
	cdata->DlgText =DlgText;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return cdata;
}

void PropList::AddCheck(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddCheck"); 
	Property   newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip =ToolTip;

	newProp.ptype      =PROPTYPE_CHECK;
	newProp.pdata      =0;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = props.AddAfterLink(linkAddPos,&newProp);
}

ScriptSelect* PropList::AddScript(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddScript"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_SCRIPT;
	newProp.pdata = 0;

	ScriptSelect* ss = new ScriptSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ss;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return ss;
}

NodeSelect* PropList::AddNode(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddNode"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_NODE;
	newProp.pdata = 0;

	NodeSelect* ns = new NodeSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ns;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return ns;
}

MultiList* PropList::AddMulti(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddMulti"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_MULTI;
	newProp.pdata = 0;

	MultiList* ml = new MultiList(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ml;
	
	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return ml;
}

FlagSelect* PropList::AddFlags(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddFlags"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_FLAGS;
	newProp.pdata = 0;

	FlagSelect* fs = new FlagSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = fs;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
	return fs;
}

void PropList::AddMtl(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddMtl"); 
	Property newProp;

	newProp.hwndTxt=NULL;
	newProp.hwndFld=NULL;

	newProp.strPropName=propName;

	if (ToolTip!=CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_MTL;
	newProp.pdata = 0;

	MtlSelect* ms = new MtlSelect(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos,&newProp);
	link->data.pdata = ms;

	if (linkAddPos)
		curIndex++;
	else
		curIndex=0;

	linkAddPos = link;
}

OptFile* PropList::AddOptFile(CStr propName,CStr ToolTip)
{ FUNC_ENTER("PropList::AddOptFile"); 
	Property newProp;

	newProp.hwndTxt = NULL;
	newProp.hwndFld = NULL;

	newProp.strPropName = propName;

	if (ToolTip != CStr(""))
		newProp.strToolTip = ToolTip;

	newProp.ptype = PROPTYPE_OPTFILE;
	newProp.pdata = 0;

	OptFile* of = new OptFile(hInstance);

	Link<Property>* link = props.AddAfterLink(linkAddPos, &newProp);
	link->data.pdata = of;

	if (linkAddPos)
		curIndex++;
	else
		curIndex = 0;

	linkAddPos = link;
	return of;
}

LRESULT CALLBACK PropList::RedirectWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropList::RedirectWndProc"); 
	PropList* thisclass=(PropList*)GetWindowLong(hwnd,GWL_USERDATA);

	if (msg == WM_CREATE)
		return TRUE;

	if (thisclass)
		return thisclass->WndProc(hwnd,msg,wParam,lParam);

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

LRESULT PropList::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropList::WndProc"); 
	int ID=LOWORD(wParam);
	CStr FileTitle;
	FileData* fdata;
	RECT winRectPos;
	RECT winRectSize;
	char buf[128];
	int  index;

	//DisableAccelerators();

	switch(msg)
	{
	case WM_CREATE:
		return TRUE;

	case WM_PAINT:
		{
			// In some scenarios the property list backgrounds were not always showing
			// up gray.  This will force the control's background to paint gray first
			if (bManualRedraw)
			{
				PAINTSTRUCT ps;
				HDC         hdc = (HDC)GetDC(hwnd);
				RECT        rect;
				GetClientRect(hwnd, &rect);

				BeginPaint(hwnd, &ps);

					FillRect(hdc, &rect, (HBRUSH)GetStockObject(LTGRAY_BRUSH));

				EndPaint(hwnd, &ps);
				
				ReleaseDC(hwnd, hdc);

				// Force all controls to redraw themselves
				Link<Property>* curprop = props.GetHead();

				while(curprop)
				{
					InvalidateRect(curprop->data.hwndTxt, NULL, FALSE);
					InvalidateRect(curprop->data.hwndFld, NULL, FALSE);
					curprop = curprop->next;
				}

				if (hwndApply)
					InvalidateRect(hwndApply, NULL, FALSE);

				if (hwndScroll)
					InvalidateRect(hwndScroll, NULL, FALSE);
			}
			else
				break;
		}
		return TRUE;

	case WM_SHOWWINDOW:
		if ((BOOL)wParam)
			DisableAccelerators();

		return TRUE;

	case WM_CHILDUPDATED:
		if ((int)wParam <= props.GetSize())
		{
			props[(int)wParam].bWasUpdated=true;
			lastModID=(int)wParam;

			if (fpChange)
				fpChange(this, pChangeData);
		}

		return TRUE;

	case WM_CHILDGOTFOCUS:
		if (fpFocus)
			fpFocus(this,(int)wParam,pFocusData);

		return TRUE;

	case WM_CTLCOLORSTATIC:
		{
			return ProcColor((HWND)lParam, (HDC)wParam);
		}

	case WM_VSCROLL:
		if (lParam)
		{
			UpdateUI(UpdateScroll((HWND)lParam,LOWORD(wParam),HIWORD(wParam)));
		}
		return TRUE;

	case CC_COLOR_CHANGE:
	case CC_SLIDER_CHANGE:
		props[ID].bWasUpdated=true;
		lastModID=ID;

		if (fpChange)
			fpChange(this, pChangeData);

		return TRUE;

	case WM_COMMAND:
		// Check if the apply button was pressed and if so,
		// call the attached callback function
		if (fpApply && (ID==IDC_ACCEPT || (HWND)lParam==hwndApply))
		{
			ProcExtLists();

			fpApply(this,pApplyData);
			return TRUE;
		}

		switch(HIWORD(wParam))
		{
		case CBN_SELENDOK:
		case CBN_SELCHANGE:		// This is the same as STN_DBLCLK  (have to process both)

			// Handle STN_DBLCLK
			if (ID>=STATIC_ID)
			{
				if (fpDblClick)
				{
					ttip.Close();
					fpDblClick(this,ID-STATIC_ID,pDblClickData);
				}

				return TRUE;
			}

			props[ID].bWasUpdated=true;
			lastModID=ID;

			// Force the combobox to change it's text now
			index=SendMessage((HWND)lParam,CB_GETCURSEL,0,0);
			if (index != CB_ERR)
			{
				SendMessage((HWND)lParam,CB_GETLBTEXT,(WPARAM)index,(LPARAM)buf);
				SetWindowText((HWND)lParam,buf);
			}

			if (fpChange)
				fpChange(this, pChangeData);

			return TRUE;

		case CBN_EDITCHANGE:
		case EN_CHANGE:
			props[ID].bWasUpdated=true;
			lastModID=ID;

			if (fpChange)
				fpChange(this, pChangeData);

			return TRUE;

		case CBN_SETFOCUS:
		case EN_SETFOCUS:
			if (fpFocus)
				fpFocus(this, ID, pFocusData);

			return TRUE;

		case STN_CLICKED:	// Static label clicked (Tooltip request)
			// Check if actually file property
			if (ID<STATIC_ID)
			{
				if (props[ID].ptype==PROPTYPE_FILE)
				{
					fdata=(FileData*)props[ID].pdata;
					OpenFileDlg(hwnd,&fdata->FileName,&FileTitle,CStr("Select file for ")+props[ID].strPropName,fdata->Fltr,fdata->path);

					if (FileTitle!=CStr(""))
						SetDlgItemText(hwnd,ID,(char*)FileTitle);

					props[ID].bWasUpdated=true;
					lastModID=ID;
				}
			}
			
			if (ID>=STATIC_ID)
			{
				GetWindowRect(props[ID-STATIC_ID].hwndTxt,&winRectPos);
				GetClientRect(hwnd,&winRectSize);

				ttip.CreateTip(props[ID-STATIC_ID].strToolTip,
					           winRectPos.left+15,winRectPos.top+15,
							   winRectSize.right,100);

				return TRUE;
			}
			else
			{
				props[ID].bWasUpdated=true;
				lastModID=ID;

				if (fpChange)
					fpChange(this, pChangeData);
			}

			return TRUE;
		};
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

void PropList::OpenFileDlg(HWND hwnd,CStr* File,CStr* FileTitle,char* Title,char* Filter,char* initialDir)
{ FUNC_ENTER("PropList::OpenFileDlg"); 
	OPENFILENAME ofn;
	char filename[256]="";
	char filetitle[128]="";

	if (initialDir && *initialDir == 0)
		initialDir = 0;

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter=Filter;
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=filetitle;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir=initialDir;
	ofn.lpstrTitle=Title;
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".q");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	*File=CStr(filename);
	*FileTitle=CStr(filetitle);
}

int PropList::GetVisiblePropCount()
{ FUNC_ENTER("PropList::GetVisiblePropCount"); 
	Link<Property>* curprop = props.GetHead();
	int count = 0;

	while(curprop)
	{
		if (curprop->data.bVisible)
			count++;

		curprop = curprop->next;
	}

	return count;
}

void PropList::UpdateUI(int scrollpos)
{ FUNC_ENTER("PropList::UpdateUI"); 
	// We now have to reorder the UI windows based on where the scroll bar is
	RECT rect;
	GetWindowRect(hwnd,&rect);
	
	if (scrollpos==-1)
		return;

	//int pos=SendMessage(hwndScroll,SBM_GETPOS,0,0);
	int pos=scrollpos;

	int width=rect.right-rect.left+PROP_RIGHTSPACING;
	int height=rect.bottom-rect.top;
	
	int numItems=((height)/((FLDHeight+PROP_SPACING)/numCols));
	
	int disppos=0;	// Display position

	//int listCount=props.GetSize();
	int listCount=GetVisiblePropCount();

	// Force full rows unless last item
	if (numItems%numCols>0 &&
		numItems<listCount)
		numItems-=numItems%numCols;

	int nProps = props.GetSize();

	for(int c=0;c<nProps;c++)
	{
		ShowWindow(props[c].hwndTxt,SW_HIDE);
		ShowWindow(props[c].hwndFld,SW_HIDE);
	}

	for(int i=0,curprop=0;curprop<nProps;curprop++)
	{
		if (props[curprop].bVisible)
		{
			// Should be added to display
			if (i>=scrollpos && i<scrollpos+numItems)
			{
				MoveWindow(props[curprop].hwndTxt,
						   GetXTXTPos(disppos),
						   GetYTXTPos(disppos),
						   TXTWidth,TXTHeight,TRUE);

				// Ensure proper height if dropdown listbox
				if (props[curprop].ptype==PROPTYPE_LIST ||
					props[curprop].ptype==PROPTYPE_EXTLIST)
				{
					// Expand the field width if there is no scroll bar
						MoveWindow(props[curprop].hwndFld,
								   GetXFLDPos(disppos),
								   GetYFLDPos(disppos),
								   FLDWidth,FLDHeight+PROP_DROPDOWNHEIGHT,TRUE);
				}
				else
				{
					MoveWindow(props[curprop].hwndFld,
							   GetXFLDPos(disppos),
							   GetYFLDPos(disppos),
							   FLDWidth,FLDHeight,TRUE);
				}

				// Advance to next display position
				disppos++;

				// Windows with disabled display properties won't be displayed (ever)
				if (props[curprop].strDispName == CStr(" "))
				{
					ShowWindow(props[curprop].hwndTxt,SW_HIDE);
					ShowWindow(props[curprop].hwndFld,SW_HIDE);
				}
				else
				{
					ShowWindow(props[curprop].hwndTxt,SW_SHOW);
					ShowWindow(props[curprop].hwndFld,SW_SHOW);
				}
			}

			i++;
		}
	}

	if (nProps>curprop)
		for(i=curprop;i<nProps;i++)
		{
			ShowWindow(props[i].hwndTxt,SW_HIDE);
			ShowWindow(props[i].hwndFld,SW_HIDE);
		}

	// Reposition the apply button
	if (bHasApply)
	{
		MoveWindow(hwndApply,
			       rectRefTXT.left,(FLDHeight+PROP_SPACING)*disppos,
				   TXTWidth*2+10,TXTHeight,TRUE);
	}

	if (hwndScroll)
		InvalidateRect(hwndScroll, NULL, TRUE);
}

void PropList::SetScrollRange()
{ FUNC_ENTER("PropList::SetScrollRange"); 
	if (hwndScroll)
	{
		int listSize=GetVisiblePropCount();

		SendMessage(hwndScroll,SBM_SETRANGE,(WPARAM)0,(LPARAM)listSize-((listSize%numCols==0) ? numCols : 1));
	}
}

void PropList::BuildUI()
{ FUNC_ENTER("PropList::BuildUI"); 
	// Calc maximum properties that can fit on the property list window
	//RECT rect;
	//GetWindowRect(hwnd,&rect);

	//int width=rect.right-rect.left+PROP_RIGHTSPACING;
	//int height=rect.bottom-rect.top;

	if (!bAttachedWindow)
		width+=PROP_RIGHTSPACING;

	int listSize=props.GetSize();

	if (listSize==0)
		return;

	// Compute the required height
	int reqHeight = ((bHasApply) ? listSize+1 : listSize) * (FLDHeight+PROP_SPACING)+9;

	FLDWidthS = FLDWidth;

	// Construct a scroll bar if the desired height is greater than the UI window
	if (reqHeight>height*numCols)
	{
		int swidth,sheight;

		if (bAttachedWindow)
		{
			swidth=width-SCROLL_WIDTH+SCROLL_SPACE;
			sheight=height;
		}
		else
		{
			swidth=width+SCROLL_WIDTH;
			
			if (bFloatingWindow)
				sheight=height-25;
			else
				sheight=height;
		}

		hwndScroll=CreateWindow("scrollbar",
			                    "scroll",
								WS_CHILD|WS_VISIBLE|SBS_VERT|SBS_RIGHTALIGN,
								//rect.right-rect.left-20,rect.bottom-rect.top,
								0,0,
								swidth,sheight,
								hwnd,
								(HMENU)0,
								hInstance,
								NULL);

		SetScrollRange();

		EnableWindow(hwndScroll,TRUE);
	}
//	else
//		if (numCols==1 && bAttachedWindow)
//			FLDWidthS -= SCROLL_WIDTH+SCROLL_SPACE+12;

	int numItems = ((height)/((FLDHeight+PROP_SPACING)/numCols));

	// Force full rows unless last item
	if (numItems%numCols>0 &&
		numItems<listSize)
		numItems-=numItems%numCols;

	for(int i=0;i<listSize;i++)
	{
		int theID=STATIC_ID+i;

		// Any properties that start with '%' will not have their property names displayed
		if ((props[i].strPropName.Length() > 0 && props[i].strPropName[0] == '%') ||
			(props[i].strDispName.Length() > 0 && props[i].strDispName[0] == '%'))
			props[i].strDispName = " ";

		// Create the property text window
		props[i].hwndTxt=CreateWindow("static",
			(props[i].strDispName.Length() == 0) ? props[i].strPropName : props[i].strDispName,
									  WS_CHILD|WS_BORDER|SS_NOTIFY,
									  GetXTXTPos(i),
									  GetYTXTPos(i),
									  TXTWidth,
									  TXTHeight,
									  hwnd,
									  (HMENU)theID,
									  hInstance,
									  NULL);

		// Subclass the property text window
		OldStaticProc=(WNDPROC)GetWindowLong(props[i].hwndTxt,GWL_WNDPROC);
		SetWindowLong(props[i].hwndTxt,GWL_WNDPROC,(LONG)StaticProc);

		switch(props[i].ptype)
		{
		case PROPTYPE_EDIT:
			props[i].hwndFld=BuildEditUI(i);
			break;

		case PROPTYPE_LIST:
			props[i].hwndFld=BuildListUI(i,(ListData*)props[i].pdata);
			break;

		case PROPTYPE_EXTLIST:
			props[i].hwndFld=BuildExtListUI(i,(ListData*)props[i].pdata);
			break;

		case PROPTYPE_SLIDER:
			props[i].hwndFld=BuildSliderUI(i,(SliderData*)props[i].pdata);
			break;

		case PROPTYPE_FILE:
			props[i].hwndFld=BuildFileUI(i);
			break;

		case PROPTYPE_SCRIPT:
			props[i].hwndFld=BuildScriptUI(i);
			break;

		case PROPTYPE_COLOR:
			props[i].hwndFld=BuildColorUI(i,(ColorData*)props[i].pdata);
			break;

		case PROPTYPE_CHECK:
			props[i].hwndFld=BuildCheckUI(i);
			break;

		case PROPTYPE_SPINEDIT:
			props[i].hwndFld=BuildSpinEditUI(i);
			break;

		case PROPTYPE_STATIC:
			props[i].hwndFld=BuildStaticUI(i);
			break;

		case PROPTYPE_SLIDERNUM:
			props[i].hwndFld=BuildSliderNumUI(i);
			break;

		case PROPTYPE_NODE:
			props[i].hwndFld=BuildNodeUI(i);
			break;

		case PROPTYPE_MULTI:
			props[i].hwndFld=BuildMultiUI(i);
			break;

		case PROPTYPE_FLAGS:
			props[i].hwndFld=BuildFlagsUI(i);
			break;

		case PROPTYPE_MTL:
			props[i].hwndFld=BuildMtlUI(i);
			break;

		case PROPTYPE_COLORPOPUP:
			props[i].hwndFld=BuildColorPopup(i);
			break;

		case PROPTYPE_OPTFILE:
			props[i].hwndFld=BuildOptFile(i);
			break;
		}

		// Subclass the control such that we can intercept TAB messages
		// so we can allow the user to cycle through controls
		SubclassTabChange(props[i].hwndFld);

		ShowWindow(props[i].hwndTxt,SW_HIDE);
		ShowWindow(props[i].hwndFld,SW_HIDE);

		/*
		if (i<numItems)
		{
			ShowWindow(props[i].hwndTxt,SW_SHOW);
			ShowWindow(props[i].hwndFld,SW_SHOW);
		}

		if (!props[i].bVisible)
		{
			ShowWindow(props[i].hwndTxt,SW_HIDE);
			ShowWindow(props[i].hwndFld,SW_HIDE);
		}
		*/
	}

	// Calculate the proper window height
	int propHeight=(listSize+1)*(FLDHeight+PROP_SPACING)+9;
	//SetWindowPos(hwnd,HWND_TOPMOST,0,0,width,propHeight,SWP_NOMOVE|SWP_SHOWWINDOW);
	
	//SetWindowPos(hwnd,HWND_TOPMOST,x,y,width+SCROLL_WIDTH+SCROLL_SPACE,propHeight,SWP_SHOWWINDOW);
	
	if (!bAttachedWindow)
	{
		if (reqHeight>height*numCols)
			//SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,height,SWP_SHOWWINDOW);
			SetWindowPos(hwnd,HWND_TOPMOST,x,y,width+SCROLL_WIDTH+SCROLL_SPACE,height,SWP_SHOWWINDOW);
		else
			SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,propHeight,SWP_SHOWWINDOW);
	}
	
	// Create the accept button at the bottom of the property list
	if (bHasApply)
	{
		hwndApply=CreateWindow("button",
			                    "&Apply",
								WS_CHILD|WS_VISIBLE,
								rectRefTXT.left,(FLDHeight+PROP_SPACING)*i+4,
								TXTWidth*2+10,TXTHeight,
								hwnd,
								(HMENU)IDC_ACCEPT,
								hInstance,
								NULL);
	}

	if (!fpApply)
		EnableWindow(hwndApply,FALSE);

	bSetValLocked=false;
	RestoreInternalValues();
	UpdateUI(0);
	ResetUpdateStatus();
}

int PropList::GetXTXTPos(int index)
{ FUNC_ENTER("PropList::GetXTXTPos"); 
	int col=index%numCols;
	return rectRefTXT.left+((TXTWidth+PROP_HSPACING)+(FLDWidth+PROP_HSPACING))*col;
}

int PropList::GetXFLDPos(int index)
{ FUNC_ENTER("PropList::GetXFLDPos"); 
	int col=index%numCols;
	return rectRefFLD.left+((TXTWidth+PROP_HSPACING)+(FLDWidth+PROP_HSPACING))*col;
}

int PropList::GetYTXTPos(int index)
{ FUNC_ENTER("PropList::GetYTXTPos"); 
	//int col=index%numCols;
	return (FLDHeight+PROP_SPACING)*(index/numCols);
	//(FLDHeight+PROP_SPACING)*i
}

int PropList::GetYFLDPos(int index)
{ FUNC_ENTER("PropList::GetYFLDPos"); 
	//int col=index%numCols;
	return (FLDHeight+PROP_SPACING)*(index/numCols);
}

HWND PropList::BuildEditUI(int index)
{ FUNC_ENTER("PropList::BuildEditUI"); 
	HWND hwndEdit;

	hwndEdit=CreateWindow("CustEdit",
				          props[index].strPropName,
						  WS_CHILD,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	ICustEdit* IEdit = GetICustEdit(hwndEdit);
	IEdit->WantDlgNextCtl(TRUE);
	ReleaseICustEdit(IEdit);

	return hwndEdit;
}

HWND PropList::BuildStaticUI(int index)
{ FUNC_ENTER("PropList::BuildStaticUI"); 
	HWND hwndStatic;

	StaticCtl* sc = (StaticCtl*)props[index].pdata;

	hwndStatic=CreateWindow("StaticCtl",
                            "",
							WS_CHILD|WS_BORDER,
							GetXFLDPos(index),
							GetYFLDPos(index),
							FLDWidthS,
							FLDHeight,
							hwnd,
							(HMENU)index,
							hInstance,
							sc);

	sc->Attach(hwndStatic);

	return hwndStatic;
}

HWND PropList::BuildSpinEditUI(int index)
{ FUNC_ENTER("PropList::BuildSpinEditUI"); 
	HWND hwndSpinEdit;

	SpinEdit* spe = (SpinEdit*)props[index].pdata;

	hwndSpinEdit=CreateWindow(SPIN_EDIT_NAME,
							  props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,
							  FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  spe);

	return hwndSpinEdit;
}

HWND PropList::BuildSliderNumUI(int index)
{ FUNC_ENTER("PropList::BuildSliderNumUI"); 
	HWND hwndSliderNum;

	SliderNum* sn = (SliderNum*)props[index].pdata;

	hwndSliderNum = CreateWindow(SLIDENUM_CLASSNAME,
								 props[index].strPropName,
								 WS_CHILD,
								 GetXFLDPos(index),
								 GetYFLDPos(index),
								 FLDWidthS,
								 FLDHeight,
								 hwnd,
								 (HMENU)index,
								 hInstance,
								 sn);

	return hwndSliderNum;
}

HWND PropList::BuildListUI(int index,ListData* data)
{ FUNC_ENTER("PropList::BuildListUI"); 
	HWND hwndEdit;

	hwndEdit=CreateWindow("combobox",
				          props[index].strPropName,
						  WS_CHILD|CBS_DROPDOWNLIST|CBS_SORT|WS_VSCROLL,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight+PROP_DROPDOWNHEIGHT,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	SendMessage(hwndEdit,WM_SETFONT,(WPARAM)hFont,MAKELPARAM(TRUE,0));
	SendMessage(hwndEdit, CB_SETDROPPEDWIDTH,(WPARAM)200,0);

	int numEntries=data->list.GetSize();

	// Add all the entries to the list
	for(int j=0;j<numEntries;j++)
		SendMessage(hwndEdit,CB_ADDSTRING,0,(LPARAM)(char*)data->list[j]);

	// Set the starting selection for the combo box
	SendMessage(hwndEdit,CB_SETCURSEL,data->curSel,0);

	return hwndEdit;
}

HWND PropList::BuildExtListUI(int index,ListData* data)
{ FUNC_ENTER("PropList::BuildExtListUI"); 
	HWND hwndEdit;

	hwndEdit=CreateWindow("combobox",
				          props[index].strPropName,
						  WS_CHILD|CBS_DROPDOWN|CBS_AUTOHSCROLL|CBS_SORT|WS_VSCROLL,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight+PROP_DROPDOWNHEIGHT,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	SendMessage(hwndEdit, WM_SETFONT,(WPARAM)hFont,MAKELPARAM(TRUE,0));
	SendMessage(hwndEdit, CB_SETDROPPEDWIDTH,(WPARAM)200,0);

	int numEntries=data->list.GetSize();

	// Add all the entries to the list
	for(int j=0;j<numEntries;j++)
		SendMessage(hwndEdit,CB_ADDSTRING,0,(LPARAM)(char*)data->list[j]);

	// Set the starting selection for the combo box
	SendMessage(hwndEdit,CB_SETCURSEL,data->curSel,0);

	return hwndEdit;
}

HWND PropList::BuildSliderUI(int index,SliderData* data)
{ FUNC_ENTER("PropList::BuildSliderUI"); 
	HWND            hwndSlider;
	ISliderControl* ISlider;

	hwndSlider=CreateWindow("SliderControl",
				            props[index].strPropName,
						    WS_CHILD,
						    GetXFLDPos(index),
							GetYFLDPos(index),
						    FLDWidthS,
							FLDHeight,
						    hwnd,
						    (HMENU)index,
						    hInstance,
						    NULL);

	ISlider=GetISlider(hwndSlider);
	ISlider->SetLimits(data->min,data->max);
	ReleaseISlider(ISlider);

	return hwndSlider;
}

HWND PropList::BuildFileUI(int index)
{ FUNC_ENTER("PropList::BuildFileUI"); 
	HWND            hwndFile;

	hwndFile=CreateWindow("button",
		                  "None",
						  WS_CHILD,
						  GetXFLDPos(index),
						  GetYFLDPos(index),
						  FLDWidthS,
						  FLDHeight,
						  hwnd,
						  (HMENU)index,
						  hInstance,
						  NULL);

	// Store the location of the name so the file dialog will know later on
	// when the button is pushed
	//LONG val=SetWindowLong(hwndFile,GWL_ID,100);
	//DWORD err=GetLastError();

	return hwndFile;						  
}

HWND PropList::BuildColorUI(int index,ColorData* cdata)
{ FUNC_ENTER("PropList::BuildColorUI"); 
	HWND            hwndColor;

	hwndColor=CreateWindow("ColorSwatch",
		                   props[index].strPropName,
						   WS_CHILD,
						   GetXFLDPos(index),
						   GetYFLDPos(index),
						   FLDWidthS,
						   FLDHeight,
						   hwnd,
						   (HMENU)index,
						   hInstance,
						   NULL);

	cdata->IColor=GetIColorSwatch(hwndColor,cdata->DefColor,cdata->DlgText);

	return hwndColor;
}

HWND PropList::BuildCheckUI(int index)
{ FUNC_ENTER("PropList::BuildCheckUI"); 
	HWND            hwndCheck;

	// The name of the button should not include anything before a slash
	char* name = NULL;
	
	if (props[index].strPropName.Length() > 0)
		name = strrchr(props[index].strPropName, '/');
	
	if (!name)
		name = props[index].strPropName;
	else
		name++;

	hwndCheck=CreateWindow("button",
		                   name,
						   WS_CHILD|BS_AUTOCHECKBOX,
						   GetXFLDPos(index),
						   GetYFLDPos(index),
						   FLDWidthS,FLDHeight,
						   hwnd,
						   (HMENU)index,
						   hInstance,
						   NULL);

	return hwndCheck;
}

HWND PropList::BuildScriptUI(int index)
{ FUNC_ENTER("PropList::BuildScriptUI"); 
	HWND           hwndScript;

	ScriptSelect* ss = (ScriptSelect*)props[index].pdata;

	hwndScript = CreateWindow("ScriptSelect",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  ss);

	return hwndScript;
}

HWND PropList::BuildNodeUI(int index)
{ FUNC_ENTER("PropList::BuildNodeUI"); 
	HWND           hwndNode;

	NodeSelect* ns = (NodeSelect*)props[index].pdata;

	hwndNode = CreateWindow("NodeSelector",
		                    props[index].strPropName,
							WS_CHILD,
							GetXFLDPos(index),
							GetYFLDPos(index),
							FLDWidthS,FLDHeight,
							hwnd,
							(HMENU)index,
							hInstance,
							ns);

	return hwndNode;
}

HWND PropList::BuildMultiUI(int index)
{ FUNC_ENTER("PropList::BuildMultiUI"); 
	HWND           hwndMulti;

	MultiList* ml = (MultiList*)props[index].pdata;

	hwndMulti = CreateWindow("MultiList",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  ml);

	return hwndMulti;
}

HWND PropList::BuildFlagsUI(int index)
{ FUNC_ENTER("PropList::BuildFlagsUI"); 
	HWND           hwndFlags;

	FlagSelect* fs = (FlagSelect*)props[index].pdata;

	hwndFlags = CreateWindow("FlagSelect",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  fs);

	return hwndFlags;
}

HWND PropList::BuildMtlUI(int index)
{ FUNC_ENTER("PropList::BuildMtlUI"); 
	HWND           hwndFlags;

	FlagSelect* fs = (FlagSelect*)props[index].pdata;

	hwndFlags = CreateWindow("MtlSelect",
		                      props[index].strPropName,
							  WS_CHILD,
							  GetXFLDPos(index),
							  GetYFLDPos(index),
							  FLDWidthS,FLDHeight,
							  hwnd,
							  (HMENU)index,
							  hInstance,
							  fs);

	return hwndFlags;
}

HWND PropList::BuildColorPopup(int index)
{ FUNC_ENTER("PropList::BuildColorPopup"); 
	HWND           hwndColorPopup;

	ColorPopup* cp = (ColorPopup*)props[index].pdata;

	hwndColorPopup = CreateWindow("ColorPopup",
		                          props[index].strPropName,
								  WS_CHILD,
								  GetXFLDPos(index),
								  GetYFLDPos(index),
								  FLDWidthS,FLDHeight,
								  hwnd,
								  (HMENU)index,
								  hInstance,
								  cp);

	return hwndColorPopup;
}

HWND PropList::BuildOptFile(int index)
{ FUNC_ENTER("PropList::BuildOptFile"); 
	HWND           hwndOptFile;

	OptFile* of = (OptFile*)props[index].pdata;

	hwndOptFile = CreateWindow("OptFile",
		                       props[index].strPropName,
							   WS_CHILD,
							   GetXFLDPos(index),
							   GetYFLDPos(index),
							   FLDWidthS,FLDHeight,
							   hwnd,
							   (HMENU)index,
							   hInstance,
							   of);

	return hwndOptFile;
}

CStr PropList::GetName(int index)
{ FUNC_ENTER("PropList::GetName"); 
	return props[index].strPropName;
}

PropType PropList::GetType(int index)
{ FUNC_ENTER("PropList::GetType"); 
	return props[index].ptype;
}

bool PropList::RulesMet(int index, INode* node)
{ FUNC_ENTER("PropList::RulesMet"); 
	if (RulesMetOr(index, node) || RulesMetAnd(index, node))
		return true;

	return false;
}

bool PropList::RulesMetOr(int index, INode* node)
{ FUNC_ENTER("PropList::RulesMetOr"); 
	Link<PropRule>* currule = props[index].ruleList.GetHead();
	bool bRuleMet = true;

	while(currule)
	{
		CStr valResult;
		bRuleMet = false;

		// Evaluate this rule and update visibility for the property
		switch(currule->data.ruleType)
		{
		case RULETYPE_OR_MAXSCRIPT:
			{
				CStr strMAXScript;
				
				if (node)
				{
					if (HasInvalidCharacters(node->GetName()))
					{
						char buf[256];
						sprintf(buf, "The node '%s' contains invalid characters and cannot be processed.\n", (char*)node->GetName());
						MessageBox(gInterface->GetMAXHWnd(), buf, "Invalid Node name", MB_ICONWARNING|MB_OK);
						break;
					}

					strMAXScript = ReplaceStr(currule->data.strRuleValue, "$OBJECT", CStr("$") + node->GetName(), false);
					strMAXScript = ReplaceStr(strMAXScript, " ", "");
					
					if (RunScriptStringResult(strMAXScript))
						return true;
				}
			}
			break;

		case RULETYPE_OR_PROP:
			{
				CStr strRule;
				GetValue(currule->data.strRuleValue, valResult);

				Link<CStr>* exprlink = currule->data.exprList.GetHead();

				while(exprlink)
				{
					if (strstr(exprlink->data,"#"))
					{
						strRule = ReplaceStr(exprlink->data, "#", valResult);

						// Now compare expression value to determine vis
						Expr expr;
						int  err;

						err = expr.load(valResult + exprlink->data);

						float   ans[3];
						float*  sRegs;
						Point3* vRegs;
						int     sCount = expr.getVarCount(SCALAR_VAR);
						int     vCount = expr.getVarCount(VECTOR_VAR);

						sRegs = new float[sCount];
						vRegs = new Point3[vCount];

						err = expr.eval(ans, sCount, sRegs, vCount, vRegs);

						delete [] sRegs;
						delete [] vRegs;

						if (expr.getExprType() != SCALAR_EXPR)
							break;

						// If we don't meet the requirement the prop is not visible
						if (ans[0]) 
							return true;
					}
					else
					{
						if (valResult == exprlink->data)
							return true;
					}

					exprlink = exprlink->next;
				}
			}
			break;
		}

		currule = currule->next;
	}

	return false;
}

bool PropList::RulesMetAnd(int index, INode* node)
{ FUNC_ENTER("PropList::RulesMetAnd"); 
	Link<PropRule>* currule = props[index].ruleList.GetHead();
	bool bRuleMet = true;

	while(currule)
	{
		CStr valResult;
		bRuleMet = false;

		// Evaluate this rule and update visibility for the property
		switch(currule->data.ruleType)
		{
		case RULETYPE_MAXSCRIPT:
			{
				CStr strMAXScript;
				
				if (node)
				{
					if (HasInvalidCharacters(node->GetName()))
					{
						char buf[256];
						sprintf(buf, "The node '%s' contains invalid characters and cannot be processed.\n", (char*)node->GetName());
						MessageBox(gInterface->GetMAXHWnd(), buf, "Invalid Node name", MB_ICONWARNING|MB_OK);
						break;
					}

					strMAXScript = ReplaceStr(currule->data.strRuleValue, "$OBJECT", CStr("$") + node->GetName(), false);
					strMAXScript = ReplaceStr(strMAXScript, " ", "");
					
					if (RunScriptStringResult(strMAXScript))
					{
						bRuleMet = true;
						break;
					}
				}
			}
			break;

		case RULETYPE_PROP:
			{
				CStr strRule;
				GetValue(currule->data.strRuleValue, valResult);

				Link<CStr>* exprlink = currule->data.exprList.GetHead();

				while(exprlink)
				{
					if (strstr(exprlink->data,"#"))
					{
						strRule = ReplaceStr(exprlink->data, "#", valResult);

						// Now compare expression value to determine vis
						Expr expr;
						int  err;

						err = expr.load(valResult + exprlink->data);

						float   ans[3];
						float*  sRegs;
						Point3* vRegs;
						int     sCount = expr.getVarCount(SCALAR_VAR);
						int     vCount = expr.getVarCount(VECTOR_VAR);

						sRegs = new float[sCount];
						vRegs = new Point3[vCount];

						err = expr.eval(ans, sCount, sRegs, vCount, vRegs);

						delete [] sRegs;
						delete [] vRegs;

						if (expr.getExprType() != SCALAR_EXPR)
							break;

						// If we don't meet the requirement the prop is not visible
						if (ans[0]) 
						{
							bRuleMet = true;
							break;
						}
					}
					else
					{
						if (valResult == exprlink->data)
							bRuleMet = true;
					}

					exprlink = exprlink->next;
				}
			}
			break;
		}

		// All rules must match
		if (!bRuleMet)
			break;

		currule = currule->next;
	}

	return bRuleMet;
}

bool PropList::RulesMet(CStr strName, INode* node)
{ FUNC_ENTER("PropList::RulesMet"); 
	Link<Property>* link = props.GetHead();

	while(link)
	{
		if (link->data.strPropName == strName)
			return RulesMet(props.GetIndex(link));

		link = link->next;
	}

	return false;
}

bool PropList::GetValue(CStr name,CStr& propVal)
{ FUNC_ENTER("PropList::GetValue"); 
	name.toLower();

	CStr val;
	bool bFound = false;

	Link<Property>* prop = props.GetHead();

	while(prop)
	{
		val = prop->data.strPropName;
		val.toLower();

		if (name == val)
		{
			bFound = true;
			break;
		}

		prop = prop->next;
	}

	if (bFound)
	{
		int index = props.GetIndex(prop);
		return GetValue(index,propVal);
	}

	return false;
}

bool PropList::GetValue(int index,CStr& strVal)
{ FUNC_ENTER("PropList::GetValue"); 
	// If there are unadded properties return their value
	if (propVals.GetSize()>0)
	{
		Link<PropVal>* curNode=propVals.GetHead();

		while(curNode)
		{
			if (curNode->data.index==index)
			{
				strVal=curNode->data.value;
				return true;
			}

			curNode=curNode->next;
		}
	}
		

	Property* propval=&props[index];

	if (!propval)
		return false;

	ICustEdit*      IEdit;
	ISliderControl* ISlider;
	IColorSwatch*   IColor;
	SpinEdit*       sedit;

	char buf[512];
	int  val;

	COLORREF cval;

	if (!propval->hwndFld)
		return false;

	switch(propval->ptype)
	{
	case PROPTYPE_EDIT:
		IEdit=GetICustEdit(propval->hwndFld);

		if (IEdit)
		{
			IEdit->GetText(buf,512);
			strVal=buf;
			ReleaseICustEdit(IEdit);
		}
		else
			return false;

		break;

	case PROPTYPE_STATIC:
		{
			StaticCtl* sc = (StaticCtl*)propval->pdata;
			if (sc)
				strVal = sc->GetValue();
		}
		break;

	case PROPTYPE_SPINEDIT:
		sedit = (SpinEdit*)propval->pdata;
		if (sedit)
		{
			sedit->GetValue(strVal);
		}
		break;

	case PROPTYPE_MULTI:
		{
			MultiList* ml = (MultiList*)propval->pdata;
			if (ml)
				strVal = ml->GetValue();
		}
		break;

	case PROPTYPE_FLAGS:
		{
			FlagSelect* fs = (FlagSelect*)propval->pdata;
			if (fs)
				strVal = fs->GetValue();
		}
		break;

	case PROPTYPE_SLIDERNUM:
		{
			SliderNum* snum = (SliderNum*)propval->pdata;
			if (snum)
			{
				sprintf(buf,"%i",snum->GetValue());
				strVal=buf;
			}
		}
		break;

	case PROPTYPE_LIST:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;
		break;

	case PROPTYPE_EXTLIST:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;
		break;

	case PROPTYPE_SLIDER:
		ISlider=GetISlider(propval->hwndFld);

		if (ISlider)
		{
			val=ISlider->GetIVal();
			ReleaseISlider(ISlider);
			_itoa(val,buf,10);
			strVal=buf;
		}
		else
			return false;

		break;

	case PROPTYPE_SCRIPT:
		GetWindowText(propval->hwndFld,buf,512);
		strVal=buf;		
		break;

	case PROPTYPE_NODE:
		{
			NodeSelect* ns = (NodeSelect*)propval->pdata;
			strVal = ns->GetValue();
		}
		break;

	case PROPTYPE_FILE:
		{
			//GetWindowText(propval->hwndFld,buf,512);
			FileData* fd = (FileData*)propval->pdata;
	
			// Starting path should be relative to start
			char* appenv = getenv(APP_ENV);

			if (appenv)
			{
				CStr replaceStr;
				
				if (fd->relPath.Length() > 0)
					replaceStr = fd->relPath + CStr("\\");
				else
					replaceStr = CStr(appenv) + CStr(FILE_PATH);

				CStr fileName = fd->FileName;
				replaceStr.toLower();
				fileName.toLower();

				strVal = ReplaceStr(fileName, replaceStr, "");
			}
			else
				strVal = fd->FileName;
		}
		break;

	case PROPTYPE_COLOR:
		IColor=((ColorData*)propval->pdata)->IColor;

		if (IColor)
		{
			cval=IColor->GetColor();
			sprintf(buf,"[%i,%i,%i]",GetRValue(cval),GetGValue(cval),GetBValue(cval));
			strVal=buf;
		}
		else
			return false;

		break;

	case PROPTYPE_CHECK:
		if (SendMessage(propval->hwndFld,BM_GETCHECK,0,0)==BST_CHECKED)
			strVal="TRUE";
		else
			strVal="FALSE";

		break;

	case PROPTYPE_MTL:
		{
			MtlSelect* ms = (MtlSelect*)propval->pdata;
			strVal = ms->GetValue();
		}
		break;

	case PROPTYPE_COLORPOPUP:
		{
			ColorPopup* cp = (ColorPopup*)propval->pdata;
			strVal = cp->GetValue();
		}
		break;

	case PROPTYPE_OPTFILE:
		{
			OptFile* of = (OptFile*)propval->pdata;
			strVal = of->GetValue();
		}
		break;
	}

	if (strVal == CStr("[none]"))
		strVal = "";

	return true;
}

void PropList::RestoreInternalValues()
{ FUNC_ENTER("PropList::RestoreInternalValues"); 
	Link<PropVal>* curNode=propVals.GetHead();
	bool bMod;

	while(curNode)
	{
		// Preserve mod status
		bMod = WasUpdated(curNode->data.index);
		SetValue(curNode->data.index,curNode->data.value);
		SetMod(curNode->data.index, bMod);
		curNode=curNode->next;
	}

	propVals.Clear();
}

bool PropList::SetValue(CStr strName,CStr strValue)
{ FUNC_ENTER("PropList::SetValue"); 
	Property srchProp;
	srchProp.strPropName = strName;

	Link<Property>* prop = props.Find(&srchProp);

	if (prop)
		return SetValue(props.GetIndex(prop),strValue);

	return false;
}

bool PropList::SetValue(int index,CStr strValue)
{ FUNC_ENTER("PropList::SetValue"); 
	if (index>props.GetSize()-1)
		return false;

	if (bSetValLocked)
	{
		PropVal newPropVal;

		newPropVal.index=index;
		newPropVal.value=strValue;

		Link<PropVal>* link = propVals.Find(&newPropVal);

		if (link)
			link->data = newPropVal;
		else
			propVals.Add(&newPropVal);

		return false;
	}

	// Must be a reference so the property doesn't get copied
	// (if it's copied there will be 2 pointers to the property data, since the property
	//  destroys it's own data, the data would get destroyed twice and cause a crash down the line)
	Property& propval=props[index];

	ICustEdit*      IEdit;
	ISliderControl* ISlider;
	IColorSwatch*   IColor;
	SpinEdit*       sedit;

	int  val;

	int red,green,blue;

	switch(propval.ptype)
	{
	case PROPTYPE_EDIT:
		IEdit=GetICustEdit(propval.hwndFld);

		if (!IEdit)
			return false;

		IEdit->SetText(strValue);
		ReleaseICustEdit(IEdit);
		break;

	case PROPTYPE_STATIC:
		{
			StaticCtl* sc = (StaticCtl*)propval.pdata;

			if (sc)
				sc->SetValue(strValue);
		}
		break;

	case PROPTYPE_SPINEDIT:
		sedit=(SpinEdit*)propval.pdata;

		if (sedit)
		{
			sedit->SetValue(atof(strValue));
		}
		break;

	case PROPTYPE_MULTI:
		{
			MultiList* ml = (MultiList*)propval.pdata;
			
			if (ml)
				ml->SetValue(strValue);
		}
		break;

	case PROPTYPE_FLAGS:
		{
			FlagSelect* fs = (FlagSelect*)propval.pdata;

			if (fs)
				fs->SetValue(strValue);
		}
		break;

	case PROPTYPE_SLIDERNUM:
		{
			SliderNum* snum = (SliderNum*)propval.pdata;

			if (snum)
			{
				snum->SetValue(atoi(strValue));
			}
		}
		break;

	case PROPTYPE_LIST:
		val=SendMessage(propval.hwndFld,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)strValue);
		if (val==CB_ERR)
			return false;

		SendMessage(propval.hwndFld,CB_SETCURSEL,(WPARAM)val,0);
		break;

	case PROPTYPE_EXTLIST:
		val=SendMessage(propval.hwndFld,CB_FINDSTRING,(WPARAM)-1,(LPARAM)(char*)strValue);
		
		if (val!=CB_ERR)
			SendMessage(propval.hwndFld,CB_SETCURSEL,(WPARAM)val,0);
		else
			SetWindowText(propval.hwndFld,(char*)strValue);
		break;

	case PROPTYPE_SLIDER:
		ISlider=GetISlider(propval.hwndFld);
		ISlider->SetValue(atoi(strValue),FALSE);
		ReleaseISlider(ISlider);
		break;

	case PROPTYPE_SCRIPT:
		SetWindowText(propval.hwndFld,(char*)strValue);
		break;

	case PROPTYPE_NODE:
		{
			NodeSelect* ns = (NodeSelect*)propval.pdata;
			ns->SetValue(strValue);
		}
		break;

	case PROPTYPE_FILE:
		{
			FileData* fd = (FileData*)propval.pdata;
			if (strstr(strValue, ":"))
				fd->FileName = strValue;
			else
				fd->FileName = fd->relPath + CStr("\\") + strValue;

			char* pos;
			CStr  name;

			if (pos = strrchr(fd->FileName, '\\'))
				name = pos + 1;	
			else
				name = fd->FileName;

			SetWindowText(propval.hwndFld,(char*)name);
		}
		break;

	case PROPTYPE_COLOR:
		IColor=((ColorData*)propval.pdata)->IColor;
		
		if (!IColor)
			return false;

		sscanf((char*)strValue,"[%i,%i,%i]",&red,&green,&blue);
		IColor->SetColor(RGB(red,green,blue));
		break;

	case PROPTYPE_CHECK:
		if (strValue==CStr("TRUE"))
		{
			SendMessage(propval.hwndFld,BM_SETCHECK,(WPARAM)BST_CHECKED,0);
			return true;
		}
		
		if (strValue==CStr("FALSE"))
		{
			SendMessage(propval.hwndFld,BM_SETCHECK,(WPARAM)BST_UNCHECKED,0);
			return true;
		}
		
		return false;

	case PROPTYPE_MTL:
		{
			MtlSelect* ms = (MtlSelect*)propval.pdata;
			ms->SetValue(strValue);
			return true;
		}

	case PROPTYPE_COLORPOPUP:
		{
			ColorPopup* cp = (ColorPopup*)propval.pdata;
			cp->SetValue(strValue);
			return true;
		}

	case PROPTYPE_OPTFILE:
		{
			OptFile* of = (OptFile*)propval.pdata;
			of->SetValue(strValue);
			return true;
		}
	}

	return true;
}

void PropList::SetApplyCB(void(*Func)(PropList*,void*),void* pData)
{ FUNC_ENTER("PropList::SetApplyCB"); 
	fpApply=Func;
	pApplyData=pData;

	if (Func)
		EnableWindow(hwndApply,TRUE);
	else
		EnableWindow(hwndApply,FALSE);
}

void PropList::SetChangeCB(void(*Func)(PropList*,void*),void* pData)
{ FUNC_ENTER("PropList::SetChangeCB"); 
	fpChange=Func;
	pChangeData=pData;
}

void PropList::SetExtListCB(void(*Func)(CStr,CStr,void*),void* pData)
{ FUNC_ENTER("PropList::SetExtListCB"); 
	fpExtList=Func;
	pExtListData=pData;
}

void PropList::SetDblClickCB(void(*Func)(PropList*,int,void*),void* pData)
{ FUNC_ENTER("PropList::SetDblClickCB"); 
	fpDblClick=Func;
	pDblClickData=pData;
}

void PropList::SetFocusCB(void(*Func)(PropList*,int,void*),void* pData)
{ FUNC_ENTER("PropList::SetFocusCB"); 
	fpFocus=Func;
	pFocusData=pData;
}

void PropList::SetEscapeCB(void(*Func)(PropList*,void*),void* pData)
{ FUNC_ENTER("PropList::SetEscapeCB"); 
	fpEscape=Func;
	pEscapeData=pData;
}

void PropList::SetStaticCB(LRESULT(*Func)(PropList*,HWND,UINT,WPARAM,LPARAM,void*),void* pData)
{ FUNC_ENTER("PropList::SetStaticCB"); 
	fpSProc=Func;
	pSProcData=pData;
}

// Get position functions 
int PropList::GetX()
{ FUNC_ENTER("PropList::GetX"); 
	RECT rect;
	GetWindowRect(hwnd,&rect);
	return rect.left;
}

int PropList::GetY()
{ FUNC_ENTER("PropList::GetY"); 
	RECT rect;
	GetWindowRect(hwnd,&rect);
	return rect.top;
}
///

void PropList::ProcExtLists()
{ FUNC_ENTER("PropList::ProcExtLists"); 
	char curval[128];
	char buf[128];

	if (!fpExtList)
		return;

	// Scan through the property list for extensible lists
	int numProps=props.GetSize();

	for(int i=0;i<numProps;i++)
	{
		if (props[i].ptype==PROPTYPE_EXTLIST)
		{
			HWND propHwnd=props[i].hwndFld;			
			GetWindowText(propHwnd,curval,128);

			// See if this value already exists
			int numItems=SendMessage(propHwnd,CB_GETCOUNT,0,0);

			for(int j=0;j<numItems;j++)
			{
				SendMessage(propHwnd,CB_GETLBTEXT,(WPARAM)j,(LPARAM)buf);
				
				if (strcmp(curval,buf)==0)
					return;
			}

			// The value doesn't exist, call the callback to add it
			fpExtList(props[i].strPropName,CStr(curval),pExtListData);
		}
	}
}

int PropList::UpdateScroll(HWND scrollBar,int code,int pos)
{ FUNC_ENTER("PropList::UpdateScroll"); 
	RECT rect;
	GetWindowRect(hwnd,&rect);

	int width=rect.right-rect.left+PROP_RIGHTSPACING;
	int height=rect.bottom-rect.top;
	
	int scrollpos=SendMessage(scrollBar,SBM_GETPOS,0,0);
	int numItems=((height)/((FLDHeight+PROP_SPACING)/numCols));

	int listCount=props.GetSize();

	switch(code)
	{
	case SB_LINEUP:
		scrollpos-=numCols;
		break;

	case SB_LINEDOWN:
		scrollpos+=numCols;
		break;

	case SB_PAGEUP:
		scrollpos-=numItems;

		if (scrollpos%numCols>0 &&
			scrollpos<=listCount)
			scrollpos-=scrollpos%numCols;
		break;

	case SB_PAGEDOWN:
		scrollpos+=numItems;

		if (scrollpos%numCols>0 &&
			scrollpos<=listCount)
			scrollpos-=scrollpos%numCols;
		break;

	case SB_THUMBPOSITION:
		scrollpos=pos;

		if (scrollpos%numCols>0)
			scrollpos-=(scrollpos%numCols);		
		break;

	default:
		return scrollpos;
	};

	if (scrollpos<0)
		scrollpos=0;
	
	if (scrollpos>listCount-numCols)
	{
		if (listCount%numCols==0)
			scrollpos=listCount-numCols;
		else
			scrollpos=listCount-listCount%numCols;
	}

	SendMessage(scrollBar,SBM_SETPOS,(WPARAM)scrollpos,(LPARAM)TRUE);
	return scrollpos;
}

void PropList::Attach(HWND hwndThis)
{ FUNC_ENTER("PropList::Attach"); 
	hwnd=hwndThis;
	SetWindowLong(hwnd,GWL_USERDATA,(LONG)this);

	RECT winRect;
	GetClientRect(hwnd,&winRect);

	width =winRect.right-winRect.left;
	origWidth = width;
	height=winRect.bottom-winRect.top;

	winRect.left+=PROP_LEFTSPACING;
	winRect.top +=PROP_TOP;

	rectRefTXT.left=winRect.left;
	rectRefTXT.top=winRect.top;
	rectRefTXT.right=winRect.left+(width/2)-PROP_HSPACING;
	rectRefTXT.bottom=rectRefTXT.top+PROP_HEIGHT;

	//rectRefFLD.left=winRect.left+width/numCols;
	rectRefFLD.left=rectRefTXT.right+PROP_HSPACING;
	rectRefFLD.top =winRect.top;
	rectRefFLD.right=rectRefFLD.left+(width/2)-PROP_HSPACING;
	rectRefFLD.bottom=rectRefFLD.top+PROP_HEIGHT;

	rectRefTXT.right-=(SCROLL_WIDTH+SCROLL_SPACE);
	rectRefFLD.right-=(SCROLL_WIDTH+SCROLL_SPACE);

	TXTWidth=rectRefTXT.right-rectRefTXT.left;
	FLDWidth=rectRefFLD.right-rectRefFLD.left;

	TXTHeight=rectRefTXT.bottom-rectRefTXT.top;
	FLDHeight=rectRefFLD.bottom-rectRefFLD.top;

	TXTWidth/=numCols;
	FLDWidth/=numCols;

	rectRefFLD.left=rectRefTXT.left+TXTWidth+PROP_HSPACING;
	rectRefFLD.right=rectRefFLD.left+FLDWidth;

	bAttachedWindow=true;
}

BOOL PropList::ProcColor(HWND hwndCtl, HDC hdc)
{ FUNC_ENTER("PropList::ProcColor"); 
	int ID=GetWindowLong(hwndCtl,GWL_ID);

	if (ID>=STATIC_ID)
	{
		switch(props[ID-STATIC_ID].color)
		{
		case PROPCOLOR_GRAY:
			SetBkColor(hdc, TXT_HIGHLIGHT);
			return (BOOL)hBGBrush;

		case PROPCOLOR_RED:
			SetBkColor(hdc, TXT_HIGHLIGHT_RED);
			return (BOOL)hBGBrushR;

		case PROPCOLOR_GREEN:
			SetBkColor(hdc, TXT_HIGHLIGHT_GREEN);
			return (BOOL)hBGBrushG;

		case PROPCOLOR_BLUE:
			SetBkColor(hdc, TXT_HIGHLIGHT_GREEN);
			return (BOOL)hBGBrushB;

		case PROPCOLOR_PURPLE:
			SetBkColor(hdc, TXT_HIGHLIGHT_PURPLE);
			return (BOOL)hBGBrushP;

		case PROPCOLOR_WHITE:
			SetBkColor(hdc, TXT_HIGHLIGHT_WHITE);
			return (BOOL)hBGBrushW;
		}
	}

	return (BOOL)hBGBrush;
}

PropColor PropList::GetColor(int index)
{ FUNC_ENTER("PropList::GetColor"); 
	return props[index].color;
}

bool PropList::SetColor(int index,PropColor color)
{ FUNC_ENTER("PropList::SetColor"); 
	if (index>props.GetSize()-1)
		return false;

	props[index].color=color;

	// Refresh if changed after UI built
	if (props[index].hwndTxt)
		InvalidateRect(props[index].hwndTxt,NULL,TRUE);

	return true;
}

void PropList::HasApply(bool bValue)
{ FUNC_ENTER("PropList::HasApply"); 
	bHasApply=bValue;
}

bool PropList::WasUpdated(int index)
{ FUNC_ENTER("PropList::WasUpdated"); 
	return props[index].bWasUpdated;
}

bool PropList::WasUpdated()
{ FUNC_ENTER("PropList::WasUpdated"); 
	Link<Property>* curProp=props.GetHead();

	while(curProp)
	{
		if (curProp->data.bWasUpdated)
			return true;

		curProp=curProp->next;
	}

	return false;
}

void PropList::ResetUpdateStatus()
{ FUNC_ENTER("PropList::ResetUpdateStatus"); 
	Link<Property>* curProp=props.GetHead();

	while(curProp)
	{
		curProp->data.bWasUpdated=false;
		curProp=curProp->next;
	}

	lastModID = -1;
}

void PropList::SetFlags(int index,int flags)
{ FUNC_ENTER("PropList::SetFlags"); 
	props[index].flags=flags;
}

DWORD PropList::GetFlags(int index)
{ FUNC_ENTER("PropList::GetFlags"); 
	if (index >= props.GetSize())
		__asm int 3;

	return props[index].flags;
}

void PropList::SetMod(int index,bool bVal)
{ FUNC_ENTER("PropList::SetMod"); 
	if (index > props.GetSize() - 1)
		return;

	props[index].bWasUpdated=bVal;
	lastModID = index;					// TEST:
}

int PropList::GetPropFromHWnd(HWND hwndScan)
{ FUNC_ENTER("PropList::GetPropFromHWnd"); 
	int index=0;
	Link<Property>* curProp=props.GetHead();

	while(curProp)
	{
		if (curProp->data.hwndTxt==hwndScan ||
			curProp->data.hwndFld==hwndScan)
			return index;

		index++;
		curProp=curProp->next;
	}

	return -1;
}

HWND PropList::GetHWndFromProp(CStr name)
{ FUNC_ENTER("PropList::GetHWndFromProp"); 
	Link<Property>* curProp = props.GetHead();

	while(curProp)
	{
		if (curProp->data.strPropName == name)
			return curProp->data.hwndFld;

		curProp = curProp->next;
	}

	return NULL;
}

HWND PropList::GetHWndFromProp(int idx)
{ FUNC_ENTER("PropList::GetHWndFromProp"); 
	return props[idx].hwndFld;
}

HWND PropList::GetStaticHWndFromProp(CStr name)
{ FUNC_ENTER("PropList::GetStaticHWndFromProp"); 
	Link<Property>* curProp = props.GetHead();

	while(curProp)
	{
		if (curProp->data.strPropName == name)
			return curProp->data.hwndTxt;

		curProp = curProp->next;
	}

	return NULL;
}

HWND PropList::GetStaticHWndFromProp(int index)
{ FUNC_ENTER("PropList::GetStaticHWndFromProp"); 
	return props[index].hwndTxt;
}

void PropList::SetPropFocus(CStr name)
{ FUNC_ENTER("PropList::SetPropFocus"); 
	Link<Property>* curProp = props.GetHead();

	while(curProp)
	{
		if (curProp->data.strPropName == name)
		{
			SetFocus(curProp->data.hwndFld);
			return;
		}

		curProp = curProp->next;
	}
}

LRESULT CALLBACK PropList::StaticProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("PropList::StaticProc"); 
	HWND hwndParent=GetParent(hwnd);
	PropList* pthis=(PropList*)GetWindowLong(hwndParent,GWL_USERDATA);

	if (pthis->fpSProc)
	{
		LRESULT rVal=pthis->fpSProc(pthis,hwnd,msg,wParam,lParam,pthis->pSProcData);

		if (rVal!=FALSE)
			return rVal;
	}

	return CallWindowProc(pthis->OldStaticProc,hwnd,msg,wParam,lParam);
}

void PropList::DeleteProp(int index, bool bDelSaved)
{ FUNC_ENTER("PropList::DeleteProp"); 
	int curindex=0;
	Link<Property>* curProp=props.GetHead();

	// Delete property
	while(curProp)
	{
		if (index==curindex)
		{
			props.Remove(curProp);
			break;
		}

		curindex++;
		curProp=curProp->next;
	}

	// Delete saved value
	if (bDelSaved)
	{
		Link<CStr>* curVal=savedValues.GetHead();

		curindex=0;
		while(curVal)
		{
			if (index==curindex)
			{
				savedValues.Remove(curVal);
				break;
			}

			curindex++;
			curVal=curVal->next;
		}
	}

	curIndex = props.GetSize()-1;
	linkAddPos = props.GetTail();
}

CStr PropList::GetDesc(int index)
{ FUNC_ENTER("PropList::GetDesc"); 
	return props[index].strToolTip;
}

void PropList::SetDesc(int index,CStr str)
{ FUNC_ENTER("PropList::SetDesc"); 
	props[index].strToolTip=str;
}

void PropList::SetUserData(int index,CStr str)
{ FUNC_ENTER("PropList::SetUserData"); 
	props[index].strExtData=str;
}

void PropList::SetUserVal(int index,DWORD val)
{ FUNC_ENTER("PropList::SetUserVal"); 
	props[index].userData = val;
}

CStr PropList::GetUserData(int index)
{ FUNC_ENTER("PropList::GetUserData"); 
	return props[index].strExtData;
}

DWORD PropList::GetUserVal(int index)
{ FUNC_ENTER("PropList::GetUserVal"); 
	return props[index].userData;
}

int PropList::GetReqHeight(int numProps)
{ FUNC_ENTER("PropList::GetReqHeight"); 
	return (numProps+1)*(FLDHeight+PROP_SPACING)+9;
}

void PropList::SetAddPos(int i)
{ FUNC_ENTER("PropList::SetAddPos"); 
	Link<Property>* curProp = props.GetHead();

	for(int c=0;c<i;c++)
		curProp = curProp->next;

	if (curProp!=NULL)
		linkAddPos = curProp;
	else
		linkAddPos = props.GetTail();
}

void PropList::RemoveProp(int ID)
{ FUNC_ENTER("PropList::RemovePropA"); 
	Link<Property>* tmpAddlink;
	int             tmpPosIndex;
	int             scrollVal;

	Link<Property>* curProp = props.GetHead();

	for(int c=0;c<ID;c++)
		curProp = curProp->next;

	if (curProp)
	{
		tmpAddlink = curProp->prev;
		
		if (tmpAddlink)
			tmpPosIndex = FindProp(tmpAddlink);
		else
			tmpPosIndex = 0;

		//SaveValues();
		scrollVal = GetScrollVal();	
		DestroyUI();
		//DeleteProp(ID,false);
		DeleteProp(ID);
		//BuildUI();
		//RestoreValues();
		
		if (scrollVal!=-1)
			SetScrollVal(scrollVal);

		curIndex   = tmpPosIndex;
		linkAddPos = tmpAddlink;
	}
}

int PropList::FindProp(Link<Property>* prop)
{ FUNC_ENTER("PropList::FindProp"); 
	Link<Property>* curProp = props.GetHead();
	int c=0;

	while(curProp)
	{
		if (curProp==prop)
			return c;

		curProp=curProp->next;
		c++;
	}

	return -1;
}

int PropList::GetScrollVal()
{ FUNC_ENTER("PropList::GetScrollVal"); 
	if (hwndScroll)
		return SendMessage(hwndScroll,SBM_GETPOS,0,0);

	return -1;
}

void PropList::SetScrollVal(int val)
{ FUNC_ENTER("PropList::SetScrollVal"); 
	if (hwndScroll && val != -1)
		SendMessage(hwndScroll,SBM_SETPOS,(WPARAM)val,(LPARAM)TRUE);	

	UpdateUI(val);
}

int PropList::GetCurIndex()
{ FUNC_ENTER("PropList::GetCurIndex"); 
	return curIndex;
}

int PropList::ResetIndexToEnd()
{ FUNC_ENTER("PropList::ResetIndexToEnd"); 
	curIndex = props.GetSize()-1;
	linkAddPos = props.GetTail();
	return curIndex;
}

bool PropList::PropExists(CStr name)
{ FUNC_ENTER("PropList::PropExists"); 
	Link<Property>* curprop = props.GetHead();

	while(curprop)
	{
		if (curprop->data.strPropName==name)
			return true;

		curprop=curprop->next;
	}

	return false;
}

void PropList::SetPos(int x,int y)
{ FUNC_ENTER("PropList::SetPos"); 
	this->x=x;
	this->y=y;
	SetWindowPos(hwnd,HWND_TOPMOST,x,y,width,height,0);
}

void PropList::SetName(int i,CStr name)
{ FUNC_ENTER("PropList::SetName"); 
	props[i].strPropName = name;

	if (props[i].hwndTxt)
		SetWindowText(props[i].hwndTxt,name);
}

PropList& PropList::operator = (PropList& right)
{ FUNC_ENTER("PropList::operator="); 
	props       = right.props;
	savedValues = right.savedValues;
	propVals    = right.propVals;

	return *this;
}

void PropList::ClearValues()
{ FUNC_ENTER("PropList::ClearValues"); 
	int numProps = props.GetSize();

	for(int i=0;i<numProps;i++)
		SetValue(i,"");
}

void PropList::AddProperty(Property* prop)
{ FUNC_ENTER("PropList::AddProperty"); 
	Property newProp;

	newProp.hwndTxt     = NULL;
	newProp.hwndFld     = NULL;
	newProp.strPropName = prop->strPropName;
	newProp.strToolTip  = prop->strToolTip;
	newProp.strExtData  = prop->strExtData;
	newProp.ptype       = prop->ptype;
	newProp.color       = prop->color;
	newProp.bWasUpdated = prop->bWasUpdated;
	newProp.flags       = prop->flags;
	newProp.userData    = prop->userData;

	switch(prop->ptype)
	{
	case PROPTYPE_UNDEFINED:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_EDIT:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_LIST:
		newProp.pdata = new ListData(*((ListData*)prop->pdata));
		break;

	case PROPTYPE_EXTLIST:
		newProp.pdata = new ListData(*((ListData*)prop->pdata));
		break;

	case PROPTYPE_SLIDER:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_SCRIPT:
		newProp.pdata = new ScriptSelect(*((ScriptSelect*)prop->pdata));
		break;

	case PROPTYPE_FILE:
		newProp.pdata = new FileData(*((FileData*)prop->pdata));
		break;

	case PROPTYPE_COLOR:
		newProp.pdata = new ColorData(*((ColorData*)prop->pdata));
		break;

	case PROPTYPE_CHECK:
		newProp.pdata = NULL;		
		break;

	case PROPTYPE_SPINEDIT:
		newProp.pdata = new SpinEdit(*((SpinEdit*)prop->pdata));
		break;

	case PROPTYPE_STATIC:
		newProp.pdata = NULL;
		break;

	case PROPTYPE_SLIDERNUM:
		newProp.pdata = new SliderNum(*((SliderNum*)prop->pdata));
		break;

	case PROPTYPE_NODE:
		newProp.pdata = new NodeSelect(*((NodeSelect*)prop->pdata));
		break;

	case PROPTYPE_MULTI:
		newProp.pdata = new MultiList(*((MultiList*)prop->pdata));
		break;

	case PROPTYPE_FLAGS:
		newProp.pdata = new FlagSelect(*((FlagSelect*)prop->pdata));
		break;
		
	case PROPTYPE_MTL:
		newProp.pdata = new MtlSelect(*((MtlSelect*)prop->pdata));
		break;

	case PROPTYPE_COLORPOPUP:
		newProp.pdata = new ColorPopup(*((ColorPopup*)prop->pdata));
		break;
	}
}

void PropList::AddRule(int propidx, RuleType type, char* value, int nExpr, CStr* expr)
{ FUNC_ENTER("PropList::AddRule"); 
	CStr* curexpr = expr;

	Link<Property>* proplink = props.GetLink(propidx);
	Link<PropRule>* rulelink = proplink->data.ruleList.Add();

	rulelink->data.ruleType     = type;
	rulelink->data.strRuleValue = value;

	for(int i = 0; i < nExpr; i++)
	{
		rulelink->data.exprList.Add(curexpr);
		curexpr++;
	}
}

bool PropList::AddRule(char* propname, RuleType type, char* value, int nExpr, CStr* expr)
{ FUNC_ENTER("PropList::AddRule"); 
	CStr* curexpr = expr;

	Property searchProp;
	searchProp.strPropName = propname;

	Link<Property>* link = props.Find(&searchProp);
	
	if (!link)
		return false;

	Link<PropRule>* rulelink = link->data.ruleList.Add();

	rulelink->data.ruleType     = type;
	rulelink->data.strRuleValue = value;

	for(int i = 0; i < nExpr; i++)
	{
		rulelink->data.exprList.Add(curexpr);
		curexpr++;
	}

	return true;
}

void PropList::AddCmd(int propidx, int id, int nParams, CStr* param, ...)
{ FUNC_ENTER("PropList::AddCmd"); 
	CStr*   curcmd = param;

	Link<PropCmd>* link = props[propidx].cmdList.Add();
	link->data.id = id;

	for(int i = 0; i < nParams; i++)
	{
		link->data.params.Add(curcmd);
		curcmd++;
	}
}

bool PropList::AddCmd(char* propname, int id, int nParams, CStr* param, ...)
{ FUNC_ENTER("PropList::AddCmd"); 
	Property searchProp;
	searchProp.strPropName = propname;

	Link<Property>* link = props.Find(&searchProp);

	if (link)
	{
		Link<PropCmd>* linkCmd = link->data.cmdList.Add();
		linkCmd->data.id = id;

		CStr* curcmd = param;

		for(int i = 0; i < nParams; i++)
		{
			linkCmd->data.params.Add(curcmd);
			curcmd++;
		}

		return true;
	}

	return false;
}

LinkList<PropCmd>* PropList::GetCmdList(char* propname)
{ FUNC_ENTER("PropList::GetCmdList"); 
	Property searchProp;
	searchProp.strPropName = propname;

	Link<Property>* link = props.Find(&searchProp);

	if (link)
		return &link->data.cmdList;

	return NULL;
}

LinkList<PropCmd>* PropList::GetCmdList(int idx)
{ FUNC_ENTER("PropList::GetCmdList"); 
	if (idx < 0 || idx > props.GetSize())
		return NULL;

	return &props[idx].cmdList;
}

void PropList::ProcTabFwd(HWND hwnd)
{ FUNC_ENTER("PropList::ProcTabFwd"); 
	// Process the tab (move to the next property in the property list)
	int idx = GetPropFromHWnd(hwnd);
	
	if (idx == -1)
		return;

	// Advance to the next property and handle wrapping
	idx++;
	if (idx > NumProps())
		idx = 0;
		
	while(idx < NumProps() && !IsWindowVisible(props[idx].hwndFld))
		idx++;

	if (idx >= NumProps())
	{
		idx = 0;
	
		while(idx < NumProps() && !IsWindowVisible(props[idx].hwndFld))
			idx++;
	}

	HWND hwndProp = GetHWndFromProp(idx);
	SetFocus(hwndProp);
	InvalidateRect(hwndProp, NULL, TRUE);
}

void PropList::ProcTabBkwd(HWND hwnd)
{ FUNC_ENTER("PropList::ProcTabBkwd"); 
	// Process the tab (move to the next property in the property list)
	int idx = GetPropFromHWnd(hwnd);
	
	if (idx == -1)
		return;

	// Backup to the previous property and handle wrapping
	idx--;
	if (idx < 0)
		idx = NumProps() - 1;
		
	while(idx > -1 && !IsWindowVisible(props[idx].hwndFld))
		idx--;

	if (idx == -1)
	{
		idx = NumProps() - 1;
	
		while(idx < NumProps() && !IsWindowVisible(props[idx].hwndFld))
			idx--;
	}

	HWND hwndProp = GetHWndFromProp(idx);
	SetFocus(hwndProp);
	InvalidateRect(hwndProp, NULL, TRUE);
}

// This structure is maintained for each subclassed window
// We will need to do a search in order to locate the current window
struct TabProcData
{
	WNDPROC wndproc;	// The original window proceedure getting called on this window that
						// we're intercepting

	HWND    hwnd;		// The window that this specific record represents
	HWND    hwndParent;	// Original parent window as it exists in the property list

	PropList* plist;	// This property list

	BOOL    bLock;		// True when the overridden window proceedure is processing

	bool operator ==(TabProcData& right)
	{ FUNC_ENTER("TabProcData::operator=="); 
		if (hwnd == right.hwnd)
			return true;

		return false;
	}
};

LinkList<TabProcData> tabSubclassDB;

LRESULT PropList::TABProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("PropList::TABProc"); 
	TabProcData tpd;
	tpd.hwnd = hwnd;
	Link<TabProcData>* linkTabProcData = tabSubclassDB.Find(&tpd);

	if (!linkTabProcData)
		return DefWindowProc(hwnd, msg, wParam, lParam);

	switch(msg)
	{
	case WM_DESTROY:
		{
			// Need to unhook the TabProc
			WNDPROC wndproc = linkTabProcData->data.wndproc;
			SetWindowLong(hwnd, GWL_WNDPROC, (LONG)wndproc);
			tabSubclassDB.Remove(linkTabProcData);
			return CallWindowProc(wndproc, hwnd, msg, wParam, lParam);
		}

	case WM_KEYDOWN:
		switch((int)wParam)
		{
		case VK_TAB:
			if (GetKeyState(VK_SHIFT) & 0x8000)
				linkTabProcData->data.plist->ProcTabBkwd(linkTabProcData->data.hwndParent);
			else
				linkTabProcData->data.plist->ProcTabFwd(linkTabProcData->data.hwndParent);
			
			return TRUE;

		case VK_ESCAPE:
			if (linkTabProcData->data.plist->fpEscape)
				linkTabProcData->data.plist->fpEscape(linkTabProcData->data.plist, linkTabProcData->data.plist->pEscapeData);

			return TRUE;
		}
	}
	
	return CallWindowProc(linkTabProcData->data.wndproc, hwnd, msg, wParam, lParam);

		/*
	if (!ptpd)
		return 0;

	if (ptpd->bLock)
		if (ptpd->wndproc)
		{
			SetWindowLong(hwnd, GWL_USERDATA, ptpd->userdata);
			rVal = CallWindowProc(ptpd->wndproc, hwnd, msg, wParam, lParam);

			WNDPROC wndproc = (WNDPROC)GetWindowLong(hwnd, GWL_WNDPROC); 
			
			if (wndproc != TABProc)
				ptpd->wndproc = wndproc;

			ptpd->userdata = GetWindowLong(hwnd, GWL_USERDATA);
			SetWindowLong(hwnd, GWL_WNDPROC, (LONG)TABProc);
			SetWindowLong(hwnd, GWL_USERDATA, (LONG)ptpd);
			return rVal;
		}
		else
			return 0;

	ptpd->bLock = true;

/*
	switch(msg)
	{
	case WM_DESTROY:
		{
			// Need to unhook the TabProc
			TabProcData temp = *ptpd;
			
			delete ptpd;
			SetWindowLong(hwnd, GWL_USERDATA, temp.userdata);
			SetWindowLong(hwnd, GWL_WNDPROC, (LONG)temp.wndproc);
			
			return CallWindowProc(temp.wndproc, hwnd, msg, wParam, lParam);
		}

	case WM_KEYDOWN:
		switch((int)wParam)
		{
		case VK_TAB:
			ptpd->plist->ProcTab(ptpd->hwnd);
			return TRUE;
		}
	}
*//*
	SetWindowLong(hwnd, GWL_USERDATA, ptpd->userdata);
	rVal = CallWindowProc(ptpd->wndproc, hwnd, msg, wParam, lParam);

	WNDPROC wndproc = (WNDPROC)GetWindowLong(hwnd, GWL_WNDPROC); 
	
	if (wndproc != TABProc)
		ptpd->wndproc = wndproc;

	ptpd->userdata = GetWindowLong(hwnd, GWL_USERDATA);
	SetWindowLong(hwnd, GWL_WNDPROC, (LONG)TABProc);
	SetWindowLong(hwnd, GWL_USERDATA, (LONG)ptpd);

	ptpd->bLock = false;

	return rVal;
	*/
}

struct SubClassTabEnum
{
	PropList* plist;
	HWND      hwndParent;
	WNDPROC   fpTABProc;
};

BOOL CALLBACK SubclassTabChildProc(HWND hwnd, LPARAM lParam)
{ FUNC_ENTER("SubclassTabChildProc"); 
	SubClassTabEnum* pscte = (SubClassTabEnum*)lParam;

	WNDPROC wndproc  = (WNDPROC)GetWindowLong(hwnd, GWL_WNDPROC);
	TabProcData ptpd;
	
	ptpd.plist      = pscte->plist;
	ptpd.hwnd       = hwnd;
	ptpd.hwndParent = pscte->hwndParent;
	ptpd.wndproc    = wndproc;
	ptpd.bLock      = false;

	tabSubclassDB.AddToTailUnique(&ptpd);
	SetWindowLong(hwnd, GWL_WNDPROC, (LONG)pscte->fpTABProc);

	// Recursively catch all subwindows
	EnumChildWindows(hwnd, SubclassTabChildProc, lParam);

	return TRUE;
}

void PropList::SubclassTabChange(HWND hwnd)
{ FUNC_ENTER("PropList::SubclassTabChange"); 
	SubClassTabEnum scte;
	scte.plist      = this;
	scte.hwndParent = hwnd;
	scte.fpTABProc  = TABProc;

	// Subclass this control and the first layer of child windows
	SubclassTabChildProc(hwnd, (LPARAM)&scte);
	EnumChildWindows(hwnd, SubclassTabChildProc, (LPARAM)&scte);
}

void PropList::EnableProp(int index)
{ FUNC_ENTER("PropList::EnableProp"); 
	HWND hwndProp   = GetHWndFromProp(index);
	HWND hwndStatic = GetStaticHWndFromProp(index);

	EnableWindow(hwndStatic, TRUE);
	EnableWindow(hwndProp, TRUE);
}

void PropList::EnableProp(CStr name)
{ FUNC_ENTER("PropList::EnableProp"); 
	HWND hwndProp   = GetHWndFromProp(name);
	HWND hwndStatic = GetStaticHWndFromProp(name);

	EnableWindow(hwndStatic, TRUE);
	EnableWindow(hwndProp, TRUE);
}

void PropList::DisableProp(int index)
{ FUNC_ENTER("PropList::DisableProp"); 
	HWND hwndProp   = GetHWndFromProp(index);
	HWND hwndStatic = GetStaticHWndFromProp(index);

	EnableWindow(hwndStatic, FALSE);
	EnableWindow(hwndProp, FALSE);
}

void PropList::DisableProp(CStr name)
{ FUNC_ENTER("PropList::DisableProp"); 
	HWND hwndProp   = GetHWndFromProp(name);
	HWND hwndStatic = GetStaticHWndFromProp(name);

	EnableWindow(hwndStatic, FALSE);
	EnableWindow(hwndProp, FALSE);
}
