#include "FuncEnter.h"

/*
	MultiPropList.cpp
	Multi-Property List.  This is a property list that has categorical control.
	Visually it's a property list with a MultiList on top that lists out the categories and
	will update the property list to correspond to the selected category, it behaves like
	a standard property list though.  This works functionally the same, but categories are
	defined by /'s between levels for each property name
*/

#include "../MaxScriptInt.h"
#include "expr.h"
#include "MultiPropList.h"
#include "../PropEdit/ParseFuncs.h"

extern Interface* gInterface;

#define CUSTIDC_MULTILIST  1
#define CUSTIDC_PROPLIST   2
#define CUSTIDC_TABLIST    3

MultiPropList::MultiPropList(HINSTANCE hInstance,HWND hwdParent,int x,int y,int width,int height,CStr DlgText) :
	PropList(hInstance,hwdParent,x,y,width,height,DlgText), UIControl(hInstance, CLASS)
{ FUNC_ENTER("MultiPropList::MultiPropList"); 
	// Copy creation parameters for use in creation of future prop lists
	hInstance = hInstance;
	hwdParent = hwdParent;
	x         = x;
	y         = y;
	width     = width;
	height    = height;
	DlgText   = DlgText;
	
	fpChange    = NULL;
	pChangeData = NULL;
	
	bExtCreateMethod = true;

	mlist = NULL;
	ruleNode = NULL;

	bPropChangeLock = FALSE;

	HasApply(false);
	UIControl::hwnd = PropList::hwnd;
	BuildControlUI();
}

MultiPropList::MultiPropList(HINSTANCE hInstance,int cols) :
	PropList(hInstance,cols), UIControl(hInstance, CLASS)
{ FUNC_ENTER("MultiPropList::MultiPropList"); 
	// Copy creation parameters for use in creation of future prop lists
	hInstance = hInstance;
	cols      = cols;
	bExtCreateMethod= false;

	fpChange    = NULL;
	pChangeData = NULL;

	mlist = NULL;
	ruleNode = NULL;

	bPropChangeLock = FALSE;
}

MultiPropList::~MultiPropList()
{ FUNC_ENTER("MultiPropList::~MultiPropList"); 

}

void MultiPropList::BuildNameList()
{ FUNC_ENTER("MultiPropList::BuildNameList"); 
	Link<Property>* curprop = props.GetHead();

	names.Clear();

	while(curprop)
	{
		names.AddToTail(&curprop->data.strPropName);
		curprop = curprop->next;
	}
}

LRESULT MultiPropList::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("MultiPropList::WndProc"); 
	//InvalidateRect(hwndMultiList,NULL,FALSE);

	switch(msg)
	{
	case WM_CREATE:

		return TRUE;

	case WM_NOTIFY:
		{
			NMHDR* nmhdr = (NMHDR*)lParam;
			switch(nmhdr->code)
			{
			case TCN_SELCHANGE:
				{
					TCITEM item;
					char   buf[256];
					
					item.mask       = TCIF_TEXT;
					item.pszText    = buf;
					item.cchTextMax = 255;
					int iPage       = TabCtrl_GetCurSel(hwndTab);	

					TabCtrl_GetItem(hwndTab, iPage, &item);
					mlist->SetValue(buf);
					CategoryChange();
					return TRUE;
				}
			}
		}
		return FALSE;


	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case CBN_SELCHANGE:
			switch(LOWORD(wParam))
			{
			case CUSTIDC_MULTILIST:
				CategoryChange();
				return TRUE;
			}
		}
	}

	return PropList::WndProc(hwnd,msg,wParam,lParam);

	//return DefWindowProc(hwnd,msg,wParam,lParam);
}

void MultiPropList::BreakPathRoot(CStr& src, CStr& path, CStr& root)
{ FUNC_ENTER("MultiPropList::BreakPathRoot"); 
	path = src;
	int  pos  = RInstr(path,"/");

	if (pos == -1)
		path = CStr("");
	else
		path[pos] = 0;

	root = src.Substr(pos+1,src.Length());
}

void MultiPropList::AssignVisibility()
{ FUNC_ENTER("MultiPropList::AssignVisibility"); 
	// Scan through the property list and flag any properties that don't exist
	// within this category as invisible
	CStr category = mlist->GetValue();

	bool bAllVisible = true;

	if (category == CStr("All"))
	{
		Link<Property>* curprop = props.GetHead();

		while(curprop)
		{
			curprop->data.bVisible = true;
			curprop = curprop->next;
		}
	}
	else
	{
		if (category == CStr("Default"))
			category = CStr("");

		Link<Property>* curprop = props.GetHead();
		Link<CStr>* curname = names.GetHead();

		assert(props.GetSize() == names.GetSize());

		while(curprop)
		{
			CStr path,root;

			BreakPathRoot(curname->data,path,root);

			if (path == category)
				curprop->data.bVisible = true;
			else
			{
				bAllVisible = false;
				curprop->data.bVisible = false;
			}

			ShowWindow(curprop->data.hwndTxt,SW_HIDE);
			ShowWindow(curprop->data.hwndFld,SW_HIDE);

			curprop = curprop->next;
			curname = curname->next;
		}
	}

	mlist->SetGroupText(!bAllVisible);
}

void MultiPropList::CategoryChange()
{ FUNC_ENTER("MultiPropList::CategoryChange"); 
	AssignVisibility();
	UpdateInitialVisibility();
	UpdateRules();

	//RestoreValues();
	bPropChangeLock = TRUE;

	SaveValues();
	SaveModStates();
	DestroyUI();
	PropList::BuildUI();
	RestoreValues();
	RestoreModStates();

	bPropChangeLock = FALSE;
}

void MultiPropList::BuildControlUI()
{ FUNC_ENTER("MultiPropList::BuildControlUI"); 
	RECT wndRect;
	GetClientRect(UIControl::hwnd,&wndRect);

	mlist = new MultiList(UIControl::hInstance);

	// Create the tab sheet
	hwndTab = CreateWindow(WC_TABCONTROL,
		                   "Default",
						   WS_CHILD|WS_VISIBLE,
						   0,
						   0,
						   wndRect.right  - wndRect.left,
						   wndRect.bottom - wndRect.top,
						   UIControl::hwnd,
						   (HMENU)CUSTIDC_TABLIST,
						   UIControl::hInstance,
						   NULL);

	// Create the multilist
	hwndMultiList = CreateWindow("MultiList",
		                         "Default",
								 WS_CHILD,
								 0,
								 0,
								 wndRect.right - wndRect.left,
								 20,
								 UIControl::hwnd,
								 (HMENU)CUSTIDC_MULTILIST,
								 UIControl::hInstance,
								 mlist);

	ShowWindow(hwndMultiList, SW_HIDE);

	// Create the window that will contain the active property list
	hwndPropList = CreateWindow("PropList",
		                        "",
								WS_CHILD|WS_VISIBLE,
								0,
								30,
								wndRect.right - wndRect.left,
								wndRect.bottom - wndRect.top - 30,
								hwndTab,
								(HMENU)CUSTIDC_PROPLIST,
								UIControl::hInstance,
								NULL);

	DWORD err = GetLastError();

	mlist->Attach(hwndMultiList);
	mlist->DispFullPath(true);
	mlist->EvalFullPath(true);
	mlist->AddItem("Default");
	mlist->AddItem("All");

	// Create the Default and All tabs
	TCITEM tci1, tci2;
	tci1.mask    = TCIF_TEXT;
	tci1.pszText = "Default";
	tci2.mask    = TCIF_TEXT;
	tci2.pszText = "All";

	TabCtrl_DeleteAllItems(hwndTab);
	TabCtrl_InsertItem(hwndTab, 0, &tci1);
	TabCtrl_InsertItem(hwndTab, 1, &tci2);

	//SetWindowLong(PropList::hwnd,GWL_USERDATA,(LONG)this);
	PropList::Attach(hwndPropList);

	PropList::SetChangeCB(PropChange,this);

	//InvalidateRect(UIControl::hwnd,NULL,FALSE);
}

void MultiPropList::BuildUI()
{ FUNC_ENTER("MultiPropList::BuildUI"); 
	// Scan through the list and build up our own list of names within the list
	// categorized according to any sub levels that may be defined with '/'s
	// we will also change the stored property name to the final name in the hierarchy
	names.Clear();
	mlist->ResetKeepText();
	mlist->AddItem("Default");
	mlist->AddItem("All");

	// Create the Default and All tabs
	TCITEM tci1, tci2;
	tci1.mask    = TCIF_TEXT;
	tci1.pszText = "Default";
	tci2.mask    = TCIF_TEXT;
	tci2.pszText = "All";

	TabCtrl_DeleteAllItems(hwndTab);
	TabCtrl_InsertItem(hwndTab, 0, &tci1);
	TabCtrl_InsertItem(hwndTab, 1, &tci2);
	int curTabID = 1;

	Link<Property>* curprop = props.GetHead();

	while(curprop)
	{
		CStr name = curprop->data.strPropName;
		CStr path, root;
		BreakPathRoot(curprop->data.strPropName, path, root);

		if (path.Length()>0)
			if (mlist->AddItemUnique(path) != -1)
			{
				TCITEM tci;
				tci.mask = TCIF_TEXT;
				tci.pszText = path;
				
				TabCtrl_InsertItem(hwndTab, ++curTabID, &tci);
			}

		curprop->data.strDispName = root;
		names.AddToTail(&name);

		curprop = curprop->next;
	}

	PropList::HasApply(false);
	mlist->SetValue("Default", false);
	AssignVisibility();
	UpdateInitialVisibility();
	UpdateRules();
	//CategoryChange();

	//PropList::DestroyUI();
	PropList::BuildUI();
}

void MultiPropList::DestroyUI()
{ FUNC_ENTER("MultiPropList::DestroyUI"); 
	PropList::DestroyUI();
}

void MultiPropList::SetCategory(char* name)
{ FUNC_ENTER("MultiPropList::SetCategory"); 
	// Scan through the tabs.  If one of the tabs has the given name, select it
	TCITEM item;
	char   buf[256];
					
	item.mask       = TCIF_TEXT;
	item.pszText    = buf;
	item.cchTextMax = 255;

	int nTabs       = TabCtrl_GetItemCount(hwndTab);
	
	for(int i = 0; i < nTabs; i++)
	{
		TabCtrl_GetItem(hwndTab, i, &item);
		
		if (strcmp(buf, name) == 0)
		{
			TabCtrl_SetCurSel(hwndTab, i);
			mlist->SetValue(buf);
			CategoryChange();
		}
	}
}

void MultiPropList::GetCategory(CStr& category)
{ FUNC_ENTER("MultiPropList::GetCategory"); 
	category = mlist->GetValue();
}

void MultiPropList::ResetInitialVisibility()
{ FUNC_ENTER("MultiPropList::ResetInitialVisibility"); 
	Link<Property>* curprop = props.GetHead();

	while(curprop)
	{
		curprop->data.bVisible = curprop->data.bInitialVisible;
		curprop = curprop->next;
	}
}

bool MultiPropList::UpdateRules()
{ FUNC_ENTER("MultiPropList::UpdateRules"); 
	// Clear initial visibility
	bool* bVisArray = new bool[props.GetSize()];
	int curVis = 0;

	Link<Property>* curprop = props.GetHead();
	while(curprop)
	{
		bVisArray[curVis] = curprop->data.bVisible;
		curprop->data.bVisible = curprop->data.bInitialVisible;

		curVis++;
		curprop = curprop->next;
	}

	UpdateRulesAnd();
	UpdateRulesOr();

//	if (!bRebuild)
//		bRebuild = UpdateRulesOr();

	//return UpdateRulesOr();

	// Determine if a rebuild of the list is necessary (i.e. some visibility property has changed)
	curprop = props.GetHead();
	curVis = 0;
	while(curprop)
	{
		if (curprop->data.bVisible != bVisArray[curVis])
		{
			delete [] bVisArray;
			return true;
		}

		curVis++;
		curprop = curprop->next;
	}

	delete [] bVisArray;
	return false;
}

bool MultiPropList::UpdateRulesOr()
{ FUNC_ENTER("MultiPropList::UpdateRulesOr"); 
	Link<Property>* curprop = props.GetHead();
	bool bRebuild = false;

	CStr strCategory = mlist->GetValue();

	//ResetInitialVisibility();

	while(curprop)
	{
		Link<PropRule>* currule = curprop->data.ruleList.GetHead();
		bool bOldVis = curprop->data.bVisible;

		// Visibility is only processed on properties visible on current page
		if (curprop->data.bInitialVisible && currule)
		{	
			//curprop->data.bVisible = false;

			while(currule)
			{
				CStr valResult;

				// Evaluate this rule and update visibility for the property
				switch(currule->data.ruleType)
				{
				case RULETYPE_OR_MAXSCRIPT:
					{
						CStr strMAXScript;
						
						if (ruleNode)
						{
							strMAXScript = ReplaceStr(currule->data.strRuleValue, "$OBJECT", CStr("$") + ruleNode->GetName(), false);
							strMAXScript = ReplaceStr(strMAXScript, " ", "");
							
							if (RunScriptStringResult(strMAXScript))
							{
								curprop->data.bVisible = true;
								break;
							}
						}
					}
					break;

				case RULETYPE_OR_PROP:
					{
						CStr strRule;
						GetValue(currule->data.strRuleValue, valResult);

						Link<CStr>* exprlink = currule->data.exprList.GetHead();

						while(exprlink)
						{
							if (strstr(exprlink->data,"#"))
							{
								strRule = ReplaceStr(exprlink->data, "#", valResult);

								// Now compare expression value to determine vis
								Expr expr;
								int  err;

								err = expr.load(valResult + exprlink->data);

								float   ans[3];
								float*  sRegs;
								Point3* vRegs;
								int     sCount = expr.getVarCount(SCALAR_VAR);
								int     vCount = expr.getVarCount(VECTOR_VAR);

								sRegs = new float[sCount];
								vRegs = new Point3[vCount];

								err = expr.eval(ans, sCount, sRegs, vCount, vRegs);

								delete [] sRegs;
								delete [] vRegs;

								if (expr.getExprType() != SCALAR_EXPR)
									break;

								// If we don't meet the requirement the prop is not visible
								if (ans[0] && curprop->data.bInitialVisible) 
								{
									curprop->data.bVisible = true;
									break;
								}
							}
							else
							{
								if (valResult == exprlink->data)
									curprop->data.bVisible = true;
							}

							exprlink = exprlink->next;
						}

						//if (bOldVis != curprop->data.bVisible)
						//	bRebuild = true;
					}
					break;
				}

				// All rules must match
				//if (!curprop->data.bVisible)
				//	break;

				currule = currule->next;
			}

			if (bOldVis != curprop->data.bVisible)
				bRebuild = true;
		}

		curprop = curprop->next;
	}

	return bRebuild;
}

// Scan through all the rules and update the visibility of the properties
// Returns true/false depending if the list needs rebuilt
bool MultiPropList::UpdateRulesAnd()
{ FUNC_ENTER("MultiPropList::UpdateRulesAnd"); 
	Link<Property>* curprop = props.GetHead();
	bool bRebuild = false;

	CStr strCategory = mlist->GetValue();

	//ResetInitialVisibility();

	while(curprop)
	{
		Link<PropRule>* currule = curprop->data.ruleList.GetHead();
		bool bOldVis = curprop->data.bVisible;

		// Visibility is only processed on properties visible on current page
		if (curprop->data.bInitialVisible && currule)
		{	
			while(currule)
			{
				CStr valResult;

				curprop->data.bVisible = false;

				// Evaluate this rule and update visibility for the property
				switch(currule->data.ruleType)
				{
				case RULETYPE_MAXSCRIPT:
					{
						CStr strMAXScript;
						
						if (ruleNode)
						{
							strMAXScript = ReplaceStr(currule->data.strRuleValue, "$OBJECT", CStr("$") + ruleNode->GetName(), false);
							strMAXScript = ReplaceStr(strMAXScript, " ", "");
							
							if (RunScriptStringResult(strMAXScript))
							{
								curprop->data.bVisible = true;
								break;
							}
						}
					}
					break;

				case RULETYPE_PROP:
					{
						CStr strRule;
						GetValue(currule->data.strRuleValue, valResult);

						Link<CStr>* exprlink = currule->data.exprList.GetHead();

						while(exprlink)
						{
							if (strstr(exprlink->data,"#"))
							{
								strRule = ReplaceStr(exprlink->data, "#", valResult);

								// Now compare expression value to determine vis
								Expr expr;
								int  err;

								err = expr.load(valResult + exprlink->data);

								float   ans[3];
								float*  sRegs;
								Point3* vRegs;
								int     sCount = expr.getVarCount(SCALAR_VAR);
								int     vCount = expr.getVarCount(VECTOR_VAR);

								sRegs = new float[sCount];
								vRegs = new Point3[vCount];

								err = expr.eval(ans, sCount, sRegs, vCount, vRegs);

								delete [] sRegs;
								delete [] vRegs;

								if (expr.getExprType() != SCALAR_EXPR)
									break;

								// If we don't meet the requirement the prop is not visible
								if (ans[0] && curprop->data.bInitialVisible) 
								{
									curprop->data.bVisible = true;
									break;
								}
							}
							else
							{
								if (valResult == exprlink->data)
									curprop->data.bVisible = true;
							}

							exprlink = exprlink->next;
						}

						//if (bOldVis != curprop->data.bVisible)
						//	bRebuild = true;
					}
					break;
				}

				// All rules must match
				if (!curprop->data.bVisible)
					break;

				currule = currule->next;
			}

			if (bOldVis != curprop->data.bVisible)
				bRebuild = true;
		}

		curprop = curprop->next;
	}

	return bRebuild;
}

// Check the rule lists whenever a property changes
void MultiPropList::PropChange(PropList* plist, void* pData)
{ FUNC_ENTER("MultiPropList::PropChange"); 
	MultiPropList* pthis = (MultiPropList*)pData;

	if (pthis->bPropChangeLock)
		return;

	pthis->bPropChangeLock = true;

	// Call change prop of MultiPropList user
	if (pthis->fpChange)
		pthis->fpChange(plist, pthis->pChangeData);

	// Process any attached rules for determining the visibility of the properties
	if (pthis->UpdateRules())
	{
		CStr strPropName;

		int idx          = pthis->GetLastMod();

		if (idx != -1)
			strPropName = pthis->GetName(idx);

		pthis->SaveProperties();
		pthis->DestroyUI();
		pthis->PropList::BuildUI();
		pthis->RestoreProperties();

		if (idx != -1)
			pthis->SetPropFocus(strPropName);
	}

	pthis->bPropChangeLock = false;
}


void MultiPropList::SetChangeCB(void(*Func)(PropList*,void*),void* pData)
{ FUNC_ENTER("MultiPropList::SetChangeCB"); 
	fpChange    = Func;
	pChangeData = pData;
}

void MultiPropList::UpdateInitialVisibility()
{ FUNC_ENTER("MultiPropList::UpdateInitialVisibility"); 
	Link<Property>* curprop = props.GetHead();

	while(curprop)
	{
		curprop->data.bInitialVisible = curprop->data.bVisible;
		curprop = curprop->next;
	}
}
