#include "FuncEnter.h"

#include "Trigger.h"
#include <next.h>
#include <resource.h>
#include "../PropEdit/PropEdit.h"
#include "../misc/gwRender.h"
#include "../appdata.h"
#include "../Link/LinkUI.h"

TriggerCreateCallBack TriggerCreateCB;

static	TriggerClassDesc trigger_class_desc;
ClassDesc*	GetTriggerClassDesc( void ) { FUNC_ENTER("GetTriggerClassDesc");  return &trigger_class_desc; }

extern PropEditor* pPropEdit;

struct NodeRenderDesc
{
	char* className;
	char* typeName;			// Null string denotes should be used for all types
	int (*proc)(TimeValue t, INode* inode, ViewExp *vpt, int flags, char* className, char* typeName);
};

// This object is responsible for forcing the user to not be able to change the pivot position of the trigger
// This is normally not possible but we can make the TriggerResetter reference all the trigger nodes and then
// reset the pivots whenever they actually change
class TriggerResetter: public ReferenceMaker
{
public:
	void AssignReference(INode* node);

	// from ReferenceMaker
	RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID,RefMessage message);
};

static TriggerResetter gTriggerResetter;

void TriggerResetter::AssignReference(INode* node)
{ FUNC_ENTER("TriggerResetter::AssignReference"); 
	if (MakeRefByID(FOREVER, 0, node) != REF_SUCCEED)
	{
		char strErr[256];
		sprintf(strErr, "Failed to make reference to node '%s'", (char*)node->GetName());
		MessageBox(gInterface->GetMAXHWnd(), strErr, "MakeRefID Failed", MB_ICONSTOP|MB_OK);
	}
}

RefResult TriggerResetter::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message)
{ FUNC_ENTER("TriggerResetter::NotifyRefChanged"); 
	if (message == REFMSG_CHANGE)
	{
		// Determine if the pivot position has changed from (0,0,0) if it has, reset it
		Point3 ptOffset;
		INode* node = (INode*)hTarget;
		
		ptOffset = node->GetObjOffsetPos();

		if (ptOffset.x != 0.0f ||
			ptOffset.y != 0.0f ||
			ptOffset.z != 0.0f)
		{
			ptOffset.x = ptOffset.y = ptOffset.z = 0.0f;
			node->SetObjOffsetPos(ptOffset);
		}
	}

	return REF_SUCCEED;
}

//////////////////////// Class specific node render routines ///////////////////////////////////////////////////
int RenderProximNode(TimeValue t, INode* inode, ViewExp *vpt, int flags, char* className, char* typeName)
{ FUNC_ENTER("RenderProximNode"); 
	// The radius should only be rendered if RenderToViewport is on
	CStr value;

	GraphicsWindow *gw = vpt->getGW();

	CStr  strRadius;
	float radius;
	
	inode->GetUserPropString("radius",strRadius);

	// Its possible that the radius property was defaulted.  We should check if this
	// is the case and if so, set it to the default value in scripts.ini
	inode->GetUserPropString("radius", strRadius);

	if (strRadius == CStr("!"))
		radius = (float)atof(pPropEdit->GetDefault(className, typeName, "radius"));
	else
		radius = (float)atof(strRadius);

	DrawCircleSphere(gw, radius);
	return 0;
}

////////////////////////////////////  Render Proc List /////////////////////////////////////////////////////////

static NodeRenderDesc gNodeRenderList[] = 
{
	{ "ProximNode", "", RenderProximNode },
};

////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Point3 GetCenter(Mesh& mesh,int numVerts=-1)
{ FUNC_ENTER("GetCenter"); 
	if (numVerts==-1)
		numVerts=mesh.numVerts;

	if (numVerts<2 || mesh.numFaces<1)
		return Point3(0.0f,0.0f,0.0f);

	float Xmin,Xmax,Ymin,Ymax,Zmin,Zmax;

	Xmax=Xmin=mesh.verts[mesh.faces[0].v[0]].x;
	Ymax=Ymin=mesh.verts[mesh.faces[0].v[0]].y;
	Zmax=Zmin=mesh.verts[mesh.faces[0].v[0]].z;

	// Determine the minimum and maximum extents of the mesh
	for(UINT i=0;i<(UINT)numVerts;i++)
	{
		bool bInFace=false;

		// only count the vert if it is included in a face
		for(int j=0;j<mesh.numFaces/2;j++)
		{
			if (mesh.faces[j].v[0]==i ||
				mesh.faces[j].v[1]==i ||
				mesh.faces[j].v[2]==i)
			{
				bInFace=true;
				break;
			}
		}

		if (bInFace)
		{
			if (mesh.verts[i].x<Xmin)
				Xmin=mesh.verts[i].x;

			if (mesh.verts[i].y<Ymin)
				Ymin=mesh.verts[i].y;

			if (mesh.verts[i].z<Zmin)
				Zmin=mesh.verts[i].z;

			if (mesh.verts[i].x>Xmax)
				Xmax=mesh.verts[i].x;

			if (mesh.verts[i].y>Ymax)
				Ymax=mesh.verts[i].y;

			if (mesh.verts[i].z>Zmax)
				Zmax=mesh.verts[i].z;
		}
	}

	// Determine center
	Point3 center;
	center.x=Xmin+((Xmax-Xmin)/2.0f);
	center.y=Ymin+((Ymax-Ymin)/2.0f);
	center.z=Zmin+((Zmax-Zmin)/2.0f);

	return center;
}

void BuildCylinderMesh(Mesh &mesh,
		int segs, int smooth, int llsegs, int capsegs, int doPie,
		float radius1, float radius2, float height, float pie1, float pie2,
		int genUVs)
{ FUNC_ENTER("BuildCylinderMesh"); 
	Point3 p;
	int ix,na,nb,nc,nd,jx,kx, ic = 1;
	int nf=0,nv=0, lsegs;
	float delta,ang, u;	
	float totalPie, startAng = 0.0f;	

	if (doPie) doPie = 1;
	else doPie = 0; 

	lsegs = llsegs-1 + 2*capsegs;

	// Make pie2 < pie1 and pie1-pie2 < TWOPI
	while (pie1 < pie2) pie1 += TWOPI;
	while (pie1 > pie2+TWOPI) pie1 -= TWOPI;
	if (pie1==pie2) totalPie = TWOPI;
	else totalPie = pie1-pie2;		
	
	if (doPie) {
		delta    = totalPie/(float)(segs-1);
		startAng = pie2;
	} else {
		delta = (float)2.0*PI/(float)segs;
		}

	if (height<0) delta = -delta;

	int nverts;
	int nfaces;
	if (doPie) {
		nverts = (segs+1)*(lsegs);		
		nfaces = 2*(segs+1)*(lsegs-1) + 2*(segs-1);		
	} else {
		nverts = 2+segs*(lsegs);
		nfaces = 2*segs*(lsegs);	
		}
	mesh.setNumVerts(nverts+1);	// plus one for center vert
	mesh.setNumFaces(nfaces+1);	// plus one for face connecting center vert
	mesh.setSmoothFlags((smooth != 0) | ((doPie != 0) << 1));
	if (0/*genUVs*/) {
		mesh.setNumTVerts(nverts);
		mesh.setNumTVFaces(nfaces);
	} else {
		mesh.setNumTVerts(0);
		mesh.setNumTVFaces(0);
		}
	
	// bottom vertex 
	mesh.setVert(nv, Point3(0.0,0.0,0.0));
	//if (genUVs) mesh.setTVert(nv,0.5f,1.0f,0.0f);
	nv++;
		
	// Bottom cap vertices	
	for(ix=0; ix<capsegs; ix++) {
		
		// Put center vertices all the way up
		if (ix && doPie) {
			p.z = height*((float)ic/float(lsegs-1));
			p.x = p.y = 0.0f;
			mesh.setVert(nv, p);
			//if (genUVs) mesh.setTVert(nv,0.5f, (float)ic/float(lsegs-1), 0.0f);
			nv++;
			ic++;
			}
		
		p.z = 0.0f;
		u   = float(ix+1)/float(capsegs);
		ang = startAng;
		for (jx = 0; jx<segs; jx++) {			
			p.x = (float)cos(ang)*radius1*u;
			p.y = (float)sin(ang)*radius1*u;	
			mesh.setVert(nv, p);
			//if (genUVs) mesh.setTVert(nv,float(jx)/float(segs),1.0f-u,0.0f);
			nv++;
			ang += delta;
			}	
		}

	// Middle vertices 
	for(ix=1; ix<llsegs; ix++) {
			
		// Put center vertices all the way up
		if (doPie) {
			p.z = height*((float)ic/float(lsegs-1));
			p.x = p.y = 0.0f;
			mesh.setVert(nv, p);
			//if (genUVs) mesh.setTVert(nv,0.5f, (float)ic/float(lsegs-1), 0.0f);
			nv++;
			ic++;
			}
		
		float   u = float(ix)/float(llsegs);
		float rad = (radius1*(1.0f-u) + radius2*u);
		p.z = height*((float)ix/float(llsegs));
		ang = startAng;
		for (jx = 0; jx<segs; jx++) {
			p.x = (float)cos(ang)*rad;
			p.y = (float)sin(ang)*rad;
			mesh.setVert(nv, p);
			//if (genUVs) mesh.setTVert(nv,float(jx)/float(segs),(float)ix/float(llsegs),0.0f);
			nv++;
			ang += delta;
			}	
		}

	// Top cap vertices	
	for(ix=0; ix<capsegs; ix++) {
		
		// Put center vertices all the way up
		if (doPie) {
			p.z = height*((float)ic/float(lsegs-1));
			p.x = p.y = 0.0f;
			mesh.setVert(nv, p);
			//if (genUVs) mesh.setTVert(nv,0.5f, (float)ic/float(lsegs-1), 0.0f);
			ic++;
			nv++;
			}
		
		p.z = height;
		u   = 1.0f-float(ix)/float(capsegs);
		ang = startAng;
		for (jx = 0; jx<segs; jx++) {			
			p.x = (float)cos(ang)*radius2*u;		
			p.y = (float)sin(ang)*radius2*u;	
			mesh.setVert(nv, p);
			//if (genUVs) mesh.setTVert(nv,float(jx)/float(segs),u,0.0f);
			nv++;
			ang += delta;
			}	
		}

	/* top vertex */
	if (!doPie) {
		mesh.setVert(nv, (float)0.0, (float)0.0, height);
		//if (genUVs) mesh.setTVert(nv,0.5f,0.0f,0.0f);
		nv++;
		}	

	// Now make faces ---

	// Make bottom cap		

	for(ix=1; ix<=segs - doPie; ++ix) {
		nc=(ix==segs)?1:ix+1;
		if (doPie && ix==1) 
			 mesh.faces[nf].setEdgeVisFlags(capsegs>1,1,1);
		else if (doPie && ix==segs - doPie) 
			 mesh.faces[nf].setEdgeVisFlags(1,1,0);
		else mesh.faces[nf].setEdgeVisFlags(capsegs>1,1,0);
		mesh.faces[nf].setSmGroup(1);
		mesh.faces[nf].setVerts(0,nc,ix);
		mesh.faces[nf].setMatID(1);
		//if (genUVs) mesh.tvFace[nf].setTVerts(0,nc,ix);
		nf++;
		}

	/* Make midsection */
	for(ix=0; ix<lsegs-1; ++ix) {
		if (doPie) {
			jx = ix*(segs+1);
		} else {
			jx=ix*segs+1;
			}
				
		for(kx=0; kx<segs+doPie; ++kx) {			
			DWORD grp = 0;
			int mtlid;
			BOOL inSlice = FALSE;

			if (kx==0 && doPie) {
				mtlid = 3;
				grp = (1<<1);
				inSlice = TRUE;
			} else 
			if (kx==segs) {
				mtlid = 4;
				grp = (1<<2);
				inSlice = TRUE;
			} else
			if (ix < capsegs-1 || ix >= capsegs+llsegs-1) {
				grp = 1;
				mtlid = (ix<capsegs-1)?0:1;
			} else	{		
				mtlid = 2;
				if (smooth) {				
					grp = (1<<3);	
					}			
				}

			na = jx+kx;
			nb = na+segs+doPie;
			nc = (kx==(segs+doPie-1))? jx+segs+doPie: nb+1;
			nd = (kx==(segs+doPie-1))? jx : na+1;			
			mesh.faces[nf].setEdgeVisFlags(0,!inSlice,1);
			mesh.faces[nf].setSmGroup(grp);
			mesh.faces[nf].setVerts(na,nc,nb);
			mesh.faces[nf].setMatID(mtlid);
			//if (genUVs) mesh.tvFace[nf].setTVerts(na,nc,nb);
			nf++;
			mesh.faces[nf].setEdgeVisFlags(!inSlice,1,0);
			mesh.faces[nf].setSmGroup(grp);
			mesh.faces[nf].setVerts(na,nd,nc);
			mesh.faces[nf].setMatID(mtlid);
			//if (genUVs) mesh.tvFace[nf].setTVerts(na,nd,nc);
			nf++;
			}
	 	}

	//Make Top cap 			
	if (doPie) 
	{		
		na = (lsegs-1)*(segs+1);	
		jx = na + 1;
	} 
	else 
	{
		na = mesh.getNumVerts()-1;	
		jx = (lsegs-1)*segs+1;
	}
	for(ix=0; ix<segs-doPie; ++ix) 
	{
		nb = jx+ix;
		nc = (ix==segs-1)? jx: nb+1;		
		if (doPie && ix==0) 
			 mesh.faces[nf].setEdgeVisFlags(1,1,0);
		else if (doPie && ix==segs-doPie-1) 
			 mesh.faces[nf].setEdgeVisFlags(capsegs>1,1,1);
		else mesh.faces[nf].setEdgeVisFlags(capsegs>1,1,0);		
		mesh.faces[nf].setSmGroup( 1);
		mesh.faces[nf].setVerts(na,nb,nc);
		mesh.faces[nf].setMatID(0);
		//if (genUVs) mesh.tvFace[nf].setTVerts(na,nb,nc);
		nf++;
	}

	/* center vert */
	// The mesh actually contains 8 verticies at this point rather than the
	// expected 4 due to the fact we're constructing a degenerate cylinder
	mesh.setVert(nv,GetCenter(mesh,nverts-1));
	nv++;

	// Make a face that connects to the center vert so it can be used
	mesh.faces[nf].setEdgeVisFlags(1,1,1);
	mesh.faces[nf].setSmGroup(1);
	mesh.faces[nf].setVerts(1,nv-1,1);
	mesh.faces[nf].setMatID(0);
	nf++;

	if (genUVs) 
	{
		Matrix3 tm(1);
		float r = fabs(radius1) > fabs(radius2) ? (float)fabs(radius1) : (float)fabs(radius2);
		float h = height;
		if (r==0.0f) r = 1.0f;
		else r = 1.0f/r;
		if (h==0.0f) h = 1.0f;
		else h = 1.0f/h;
		tm.Scale(Point3(r,r,h));
		tm.RotateZ(HALFPI);
		tm.SetTrans(Point3(0.0f,0.0f,-0.5f));
		mesh.ApplyUVWMap(MAP_CYLINDRICAL,
			1.0f, 1.0f, 1.0f,
			0, 0, 0, 0,
			tm);
	}

	assert(nf==mesh.numFaces);
	assert(nv==mesh.numVerts);
	mesh.InvalidateGeomCache();
	mesh.BuildStripsAndEdges();

}

void BuildSimpleMesh(Mesh& mesh,float size)
{ FUNC_ENTER("BuildSimpleMesh"); 
   mesh.setNumVerts(5);
   mesh.setNumFaces(5);
   

   mesh.setVert(0, size * Point3(0.0f, 1.0f, 0.0f));
   mesh.setVert(1, size * Point3(0.0f, 0.25f, 0.75f));
   mesh.setVert(2, size * Point3(0.5f, 0.0f, 0.0f));
   mesh.setVert(3, size * Point3(-0.5f, 0.0f, 0.0f));
   mesh.setVert(4, size * Point3(0.0f, 0.5f, 0.375f));	// Add Center
   

   /*
	mesh.setVert(0, size * Point3(0.0f, 1.0f, 0.0f));
	mesh.setVert(1, size * Point3(0.0f, -1.0f,1.0f));
	mesh.setVert(2, size * Point3(1.0f, 0.0f, 0.0f));
	mesh.setVert(3, size * Point3(-1.0f, 0.0f, 0.0f));
	mesh.setVert(4, size * Point3(0.0f, 0.0f, 0.0f));
   */

   mesh.faces[0].setVerts(0, 1, 2);
   mesh.faces[0].setEdgeVisFlags(1, 1, 1);
   mesh.faces[0].setSmGroup(0);
   mesh.faces[1].setVerts(0, 3, 1);
   mesh.faces[1].setEdgeVisFlags(1, 1, 1);
   mesh.faces[1].setSmGroup(0);
   mesh.faces[2].setVerts(0, 2, 3);
   mesh.faces[2].setEdgeVisFlags(1, 1, 1);
   mesh.faces[2].setSmGroup(0);
   mesh.faces[3].setVerts(3, 2, 1);
   mesh.faces[3].setEdgeVisFlags(1, 1, 1);
   mesh.faces[3].setSmGroup(0);
   mesh.faces[4].setVerts(0,4,0);					// Fake face to visually show center
   mesh.faces[4].setEdgeVisFlags(1, 1, 1);
   mesh.faces[4].setSmGroup(0);
}

Trigger::Trigger(void) :
	suspendSnap(FALSE),
	TriggerUI(gInterface)
{ FUNC_ENTER("Trigger::Trigger");  
	BuildSimpleMesh(mesh,19);

	/*	
	BuildCylinderMesh(mesh, 
	3, 0, 1, 1, 0, 
	10.0, 0.0, 19, 0.0,0.0, 
	0);
	*/

	int i;

	//AddCenterVert(mesh);

	// rotate the mesh a bit...

	Matrix3 tm(1);
	tm.RotateZ(PI);

	//tm.RotateZ(PI/6);
	//tm.RotateX(-PI/2);
	//tm.RotateX(-PI/12);
	//tm.RotateZ(PI);

	for( i = 0; i < mesh.numVerts; i++ )
	{
		mesh.verts[i]=mesh.verts[i]*tm;		
	}

	/*
	for( i = 0; i < mesh.numVerts; i++ ) 
	{
		mesh.verts[i].z += TRIGGER_OBJ_OFFSET_Z;
	}
	*/
}

bool IsNodeSelected(INode* node)
{
	int nNodes = gInterface->GetSelNodeCount();

	for(int i = 0; i < nNodes; i++)
	{
		if (gInterface->GetSelNode(i) == node)
			return true;
	}

	return false;
}

RefTargetHandle Trigger::Clone(RemapDir &remap)
{ FUNC_ENTER("Trigger::Clone"); 
	// Determine if I have any trigger links that reference me
	RefList& refList = GetRefList();
	RefListItem* refItem = refList.FirstItem();

	while(refItem)
	{
		char buf[256];
		CStr className;
		refItem->maker->GetClassName(className);

		sprintf(buf, "Trigger has ref: %s\n", (char*)className);
		OutputDebugString(buf);

		if (refItem->maker->ClassID() == Class_ID(BASENODE_CLASS_ID, 0))
		{
			INode* node = (INode*)refItem->maker;
			
			if (IsNodeSelected(node))
				OutputDebugString("Node IS selected!\n");
			else
				OutputDebugString("Node IS NOT selected!\n");
		}

		if (refItem->maker->ClassID() == vLINK_OBJ_CLASS_ID)
		{
			int zzz;
			zzz = 0;
		}

		refItem = refItem->next;
	}

	return new Trigger;
}

void Trigger::RefAdded(RefMakerHandle rm)
{ FUNC_ENTER("Trigger::RefAdded"); 
	// When a Trigger Node is created, is creates a reference to this object.
	// This callback will occur as a result of that. Check if this is our Trigger Node.
	// If so, set its Object Offset (pivot pt) to TRIGGER_OBJ_OFFSET_Z
	/*if( rm->ClassID() == Class_ID( BASENODE_CLASS_ID, 0 ))
	{
		Object *obj;

		INode *node = (INode *) rm;
		obj = node->EvalWorldState(0).obj;
		if( obj->ClassID() == vTRIGGER_CLASS_ID )
		{
			Matrix3 tm = Inverse( node->GetObjTMAfterWSM( 0 ));
			Point3 offset( 0, 0, TRIGGER_OBJ_OFFSET_Z );
			Point3 final_offset;

			final_offset = tm * offset;
			node->SetObjOffsetPos( final_offset );
			node->SetObjOffsetRot( IdentQuat() );				
		}
	}*/

	TriggerCreated(rm);

	if (rm->ClassID() == Class_ID(BASENODE_CLASS_ID, 0))
	{
		node = (INode*)rm;
		CStr name = node->GetName();
		pivotOffset = node->GetObjOffsetPos();

		gTriggerResetter.AssignReference(node);

		/*
		if (MakeRefByID(FOREVER, 0, node) != REF_SUCCEED)
		{
			MessageBox(gInterface->GetMAXHWnd(), "Failed to make reference to new node", "MakeRefID Failed", MB_ICONSTOP|MB_OK);
		}
		*/
	}
}

int Trigger::Display(TimeValue t, INode* inode,
	ViewExp *vpt, int flags)
{ FUNC_ENTER("Trigger::Display"); 
	GraphicsWindow *gw = vpt->getGW();
	Matrix3 mat = inode->GetObjectTM(t);

#ifdef NO_SCALE
	mat.NoScale();
	float scaleFactor = vpt->NonScalingObjectSize()*vpt->GetVPWorldWidth( mat.GetTrans()) / 360.0f;
	if( scaleFactor != 1.0f )
	{
		mat.Scale( Point3( scaleFactor, scaleFactor, scaleFactor ));
	}
#endif
	DWORD rlim = gw->getRndLimits();
	gw->setTransform(mat);	
	mesh.render(gw, inode->Mtls(),
		(flags&USE_DAMAGE_RECT) ? &vpt->GetDammageRect() : NULL,
		COMP_ALL, inode->NumMtls());
	gw->setRndLimits(rlim);

	// Output the classes and types of the trigger object to the viewport
	AppDataChunk* appdata;
	ReferenceTarget* scene=gInterface->GetScenePointer();
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_TRIGGERSETTINGS_ID);

	CStr strClass, strType;

	if (!inode->GetUserPropString("Class",strClass))
		inode->GetUserPropString("class",strClass);

	if (!inode->GetUserPropString("Type",strType))
		inode->GetUserPropString("type",strType);

	if (appdata)
	{
		TriggerUIData* pData=(TriggerUIData*)appdata->data;
		
		if (pData->bShowNames)
		{
			CStr strOut,strClass,strType;

			strOut=strClass+CStr(" (")+strType+CStr(")");
			gw->text(&Point3(0.0f, 3.0f, 1.0f), strOut);
		}
	}

	// Now do any specific rendering for this node based on its type
	int nRenderTypes = sizeof(gNodeRenderList) / sizeof(NodeRenderDesc);

	// No external render routines should be called if viewport render is disabled
	if (ShouldRenderVPData())
	{
		for(int i = 0; i < nRenderTypes; i++)
		{
			if (strcmp(gNodeRenderList[i].className, strClass) == 0)
			{
				if (gNodeRenderList[i].typeName[0] == 0 ||
					strcmp(gNodeRenderList[i].typeName, strType) == 0)
				{
					// Check the render properties of the node to see if we should actually execute
					// the drawing function for this specific trigger

					// Check the SelectRenderOnly property
					bool bSelectRenderOnly = false;
					CStr value;

					// If SelectRenderOnly is checked then render this only if it appears within the selection set
					if (!inode->GetUserPropString("SelectRenderOnly", value))
					{
						if (!node->GetUserPropString("%SelectRenderOnly", value))
						{
							if (value == CStr("!"))
								value = pPropEdit->GetDefault(strClass, strType, "SelectRenderOnly");

							if (value == CStr("TRUE"))
								bSelectRenderOnly = true;
							else
								bSelectRenderOnly = false;
						}
					}
					else
					{
						if (value == CStr("!"))
							value = pPropEdit->GetDefault(strClass, strType, "SelectRenderOnly");

						if (value == CStr("TRUE"))
							bSelectRenderOnly = true;
						else
							bSelectRenderOnly = false;
					}

					if (bSelectRenderOnly && inode->Selected() ||
						!bSelectRenderOnly)
					{
						// If we don't find the normal RenderToViewport property we should look for an invisible version
						// the user may have defined a type specifically within scripts.ini that isn't supposed to ever
						// render to the viewport
						if (!inode->GetUserPropString("RenderToViewport", value))
							inode->GetUserPropString("%RenderToViewport", value);
						
						if (value == CStr("!"))
							value = pPropEdit->GetDefault(strClass, strType, "RenderToViewport");

						if (value == CStr("TRUE"))
						{
							// Call the specific rendering routine for this node type
							return gNodeRenderList[i].proc(t, inode, vpt, flags, strClass, strType);
						}
					}
				}
			}
		}	// end for
	}

	return 0;
}

RefResult Trigger::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID,RefMessage message)
{ FUNC_ENTER("Trigger::NotifyRefChanged");  
	// If we get any changes verify that the pivot hasn't moved
	if (message == REFMSG_CHANGE)
	{
		OutputDebugString("Trigger: got REFMSG_CHANGE\n");
	}

	return REF_SUCCEED; 
}


int TriggerCreateCallBack::proc(ViewExp *vpt,int msg, int point, int flags, IPoint2 m, Matrix3& mat )
{ FUNC_ENTER("TriggerCreateCallBack::proc"); 	
	if (msg == MOUSE_FREEMOVE)
	{
		// Suspend property editor changes until properties are assigned
		// after node creation (MOUSE_INIT message doesn't occur for some reason ???)
		//pPropEdit->LockSelChange();
		pPropEdit->DisableAutoPropAssign();

		#ifdef _3D_CREATE
			vpt->SnapPreview(m,m,NULL, SNAP_IN_3D);
		#else
			vpt->SnapPreview(m,m,NULL, SNAP_IN_PLANE);
		#endif
	}

	if (msg==MOUSE_POINT||msg==MOUSE_MOVE) {
		switch(point) {
			case 0:
				ob->suspendSnap = TRUE;

				#ifdef _3D_CREATE	
					mat.SetTrans(vpt->SnapPoint(m,m,NULL,SNAP_IN_3D));
				#else	
					mat.SetTrans(vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE));
				#endif
				break;
			case 1:
				#ifdef _3D_CREATE	
					mat.SetTrans(vpt->SnapPoint(m,m,NULL,SNAP_IN_3D));
				#else	
					mat.SetTrans(vpt->SnapPoint(m,m,NULL,SNAP_IN_PLANE));
				#endif
				if (msg==MOUSE_POINT) {
					ob->suspendSnap = FALSE;

					// Holding Shift will cyclically link
					BOOL bCyclicLink;

					if (flags & MOUSE_SHIFT)
						bCyclicLink = TRUE;
					else
						bCyclicLink = FALSE;

					// Holding CTRL will turn off link last trigger
					if (flags & MOUSE_CTRL)
						ob->TriggerLink(TRUE, bCyclicLink);		// End of link sequence
					else
						ob->TriggerLink(FALSE, bCyclicLink);	// Continuing link sequence

					return 0;
					}
				break;			
			}
	} else 
	if (msg == MOUSE_ABORT) 
	{		
		// Suspend property editor changes until properties are assigned
		// after node creation
		//pPropEdit->UnLockSelChange();
		pPropEdit->EnableAutoPropAssign();

		return CREATE_ABORT;
	}
	else if (msg == MOUSE_UNINIT)
	{
		// Suspend property editor changes until properties are assigned
		// after node creation
		//pPropEdit->UnLockSelChange();
		pPropEdit->EnableAutoPropAssign();
	}

	return 1;
}

