#include "FuncEnter.h"

/*
	FlagReassigner.cpp
	A utility plugin to store and reassign face channel data
*/

#include "FlagReassigner.h"
#include "NExt.h"
#include "Resource.h"
#include "Appdata.h"

static FlagReassignerClassDesc theFlagReassignerDesc;
ClassDesc2* GetFlagReassignerDesc() { FUNC_ENTER("GetFlagReassignerDesc");  return &theFlagReassignerDesc; }

FlagReassigner* gpthis;				// only 1 instance of a tool may be instantiated at once so this is okay

FlagReassigner::FlagReassigner()
{ FUNC_ENTER("FlagReassigner::FlagReassigner"); 
	mtlIDs   = NULL;
	numFaces = NULL;
	numSel   = 0;
}

FlagReassigner::~FlagReassigner()
{ FUNC_ENTER("FlagReassigner::~FlagReassigner"); 
	ClearData();
}

void FlagReassigner::ClearData()
{ FUNC_ENTER("FlagReassigner::ClearData"); 
	if (mtlIDs)
	{
		delete [] mtlIDs;
		mtlIDs = NULL;
	}
}

BOOL FlagReassigner::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("FlagReassigner::DlgProc"); 
	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_STORE:
			//EnableWindow(GetDlgItem(hwnd,IDC_APPLY),TRUE);
			//EnableWindow(GetDlgItem(hwnd,IDC_STORE),FALSE);
			gpthis->StoreFlagData();
			return TRUE;

		case IDC_APPLY:
			//EnableWindow(GetDlgItem(hwnd,IDC_STORE),TRUE);
			//EnableWindow(GetDlgItem(hwnd,IDC_APPLY),FALSE);
			gpthis->ApplyFlagData();
			return TRUE;

		case IDC_STORETOFILE:
			gpthis->StoreFlagsToFile();
			return TRUE;

		case IDC_ASSIGNFROMFILE:
			gpthis->AssignFlagsFromFile();
			return TRUE;
		}
	}

	return FALSE;
}

void FlagReassigner::BeginEditParams(Interface* ip, IUtil* iu)
{ FUNC_ENTER("FlagReassigner::BeginEditParams"); 
	this->ip=ip;
	this->iu=iu;

	gpthis = this;
	hwndRollup = ip->AddRollupPage(hInstance,MAKEINTRESOURCE(IDD_FLAGREASSIGNER),DlgProc,"Face Flag Reassigner");
}

void FlagReassigner::EndEditParams(Interface* ip, IUtil* iu)
{ FUNC_ENTER("FlagReassigner::EndEditParams"); 
	ip->DeleteRollupPage(hwndRollup);
}

void FlagReassigner::SelectionSetChanged(Interface* ip, IUtil* iu)
{ FUNC_ENTER("FlagReassigner::SelectionSetChanged"); 

}

void FlagReassigner::StoreFlagData()
{ FUNC_ENTER("FlagReassigner::StoreFlagData"); 
	OutputDebugString("Pre -------------------------------------------------\n");
	DebugFaces();

	numSel = ip->GetSelNodeCount();

	if (numSel==0)
	{
		MessageBox(ip->GetMAXHWnd(),"You must have an object selected to use this tool.","No objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	if (numSel>1)
	{
		MessageBox(ip->GetMAXHWnd(),"This tool may only be used with one object at a time.","Multiple objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	// Allocate data
	ClearData();

	for(int i=0;i<numSel;i++)
	{
		INode* node = ip->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			// Now scan through all the faces of the object and pick out the flag data
			numFaces = mesh.numFaces;

			// Acquire access to face flags data
			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));

			if (!pFDMgr)
			{
				char strErr[256];
				sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
				MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
				return;
			}

			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			CASFaceFlagsData* CASfdc = dynamic_cast<CASFaceFlagsData*>(pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

			// Allocate memory for mtlIDs
			mtlIDs = new MtlID[numFaces];
			
			if (fdc)
				fdata.FacesCreated(0,numFaces);
			
			if (CASfdc)
				fdataCAS.FacesCreated(0,numFaces);

			int mtlindex  = 0;

			for(int face=0;face<numFaces;face++)
			{
				if (fdc)
				{
					FlagType flagtype;

					fdc->GetValue( face, flagtype );
					fdata.SetValue( face, flagtype );
				}

				if (CASfdc)
				{
					CASFlagType casflagtype;

					CASfdc->GetValue( face, casflagtype );
					fdataCAS.SetValue( face, casflagtype );
				}

				// Store mtlIDs
				mtlIDs[face] = mesh.getFaceMtlIndex(face);

				// Assign the material to a sequential value
				mesh.setFaceMtlIndex(face,++mtlindex);
			}

			if (triObj != obj)
				triObj->DeleteThis();
		}

		SaveFlagData(node);
	}

	OutputDebugString("Post -------------------------------------------------\n");
	DebugFaces();

	ip->ForceCompleteRedraw();
}

void FlagReassigner::ApplyFlagData()
{ FUNC_ENTER("FlagReassigner::ApplyFlagData"); 
	OutputDebugString("Pre -------------------------------------------------\n");
	DebugFaces();

	int numSel = ip->GetSelNodeCount();

	if (numSel==0)
	{
		MessageBox(ip->GetMAXHWnd(),"You must have an object selected to use this tool.","No objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	if (numSel>1)
	{
		MessageBox(ip->GetMAXHWnd(),"This tool may only be used with one object at a time.","Multiple objects selected",MB_ICONWARNING|MB_OK);
		return;
	}

	for(int i=0;i<numSel;i++)
	{
		INode* node = ip->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		LoadFlagData(node);

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);

			// Now scan through all the faces of the object and attach the flag data
			Mesh& mesh = triObj->GetMesh();
			numFaces = mesh.numFaces;

			// Acquire access to face flags data
			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));

			if (!pFDMgr)
			{
				char strErr[256];
				sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
				MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
				return;
			}

			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			CASFaceFlagsData* CASfdc = dynamic_cast<CASFaceFlagsData*>(pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

			if (fdc == NULL)
			{
				// The Face data channel doesn't exist, add it
				fdc = new FaceFlagsData();
				pFDMgr->AddFaceDataChan( fdc );
				fdc->FacesCreated( 0, numFaces );
			}

			if (CASfdc == NULL)
			{
				// The CAS Face data channel doesn't exist, add it
				CASfdc = new CASFaceFlagsData();
				pFDMgr->AddFaceDataChan( fdc );
				CASfdc->FacesCreated( 0, numFaces );
			}

			int mtlindex  = 0;

			for(int face=0;face<numFaces;face++)
			{
				// We can tell the appropriate flags that should be applied
				// by the material ID - 1 since we reassigned them sequentially before

				MtlID matID = mesh.getFaceMtlIndex(face);
				int   faceID = matID - 1;

				FlagType flagtype;

				fdata.GetValue( faceID, flagtype );
				fdc->SetValue( face, flagtype );

				CASFlagType casflagtype;

				fdataCAS.GetValue( faceID, casflagtype );
				CASfdc->SetValue( face, casflagtype );

			// Assign the face material back to its original value
				mesh.setFaceMtlIndex(face,mtlIDs[faceID]);
			}

			if (triObj != obj)
				triObj->DeleteThis();			
		}

		DeleteFlagData(node);
	}

	OutputDebugString("Post -------------------------------------------------\n");
	DebugFaces();

	ip->ForceCompleteRedraw();
}

void FlagReassigner::LoadFlagData(INode* node)
{ FUNC_ENTER("FlagReassigner::LoadFlagData"); 
	AppDataChunk* appdata;
	appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID);

	ClearData();

	if (appdata && appdata->data)
	{
		char* pos          = (char*)appdata->data;
		int   numFlags;
		int   numCASFlags;

		numFaces = *((int*)pos);
		pos += sizeof(int);
		numFlags = *((int*)pos);
		pos += sizeof(int);
		numCASFlags = *((int*)pos);
		pos += sizeof(int);

		fdata.AllFacesDeleted();
		fdata.FacesCreated(0,numFlags);
		
		int i;

		for(i=0;i<numFlags;i++)
		{
			FlagType ftype = *((FlagType*)pos);
			pos += sizeof(FlagType);

			fdata.SetValue(i,ftype);
		}

		fdataCAS.AllFacesDeleted();
		fdataCAS.FacesCreated(0,numCASFlags);

		for(i=0;i<numCASFlags;i++)
		{
			CASFlagType ftype = *((CASFlagType*)pos);
			pos += sizeof(CASFlagType);

			fdataCAS.SetValue(i,ftype);
		}

		mtlIDs = new MtlID[numFaces];

		for(i=0;i<numFaces;i++)
		{
			mtlIDs[i] = *((MtlID*)pos);
			pos += sizeof(MtlID);
		}
	}
}

void FlagReassigner::SaveFlagData(INode* node)
{ FUNC_ENTER("FlagReassigner::SaveFlagData"); 
	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID);

	int size = sizeof(int) +  // numFaces
			   sizeof(int) +  // numFlags
			   sizeof(int) +  // numCASFlags
		       sizeof(FlagType) * fdata.Count() +
			   sizeof(CASFlagType) * fdataCAS.Count() +
			   sizeof(MtlID) * numFaces;

	int i;
	int numFlags    = fdata.Count();
	int numCASFlags = fdataCAS.Count();

	char* data = (char*)malloc(size);
	char* pos  = data;
	memcpy(pos,&numFaces,sizeof(int));
	pos += sizeof(int);
	memcpy(pos,&numFlags,sizeof(int));
	pos += sizeof(int);
	memcpy(pos,&numCASFlags,sizeof(int));
	pos += sizeof(int);

	for(i=0;i<fdata.Count();i++)
	{
		memcpy(pos,&fdata.m_data[i],sizeof(FlagType));
		pos += sizeof(FlagType);
	}

	for(i=0;i<fdataCAS.Count();i++)
	{
		memcpy(pos,&fdataCAS.m_data[i],sizeof(CASFlagType));
		pos += sizeof(CASFlagType);
	}

	for(i=0;i<numFaces;i++)
	{
		memcpy(pos,&mtlIDs[i],sizeof(MtlID));
		pos += sizeof(MtlID);
	}

	node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID,size,data);
}

void FlagReassigner::DeleteFlagData(INode* node)
{ FUNC_ENTER("FlagReassigner::DeleteFlagData"); 
	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_FLAG_ASSIGNMENT_ID);
}

void FlagReassigner::DebugFaces()
{ FUNC_ENTER("FlagReassigner::DebugFaces"); 
	numSel = ip->GetSelNodeCount();

	// Allocate data
	ClearData();

	for(int i=0;i<numSel;i++)
	{
		INode* node = ip->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			for(int j=0;j<mesh.numFaces;j++)
			{
				char buf[256];
				sprintf(buf,"Face %i:  Material: %i\n",j,mesh.getFaceMtlIndex(j));
				OutputDebugString(buf);
			}
		}
	}
}

void FlagReassigner::StoreFlagsToFile()
{ FUNC_ENTER("FlagReassigner::StoreFlagsToFile"); 
	int nSel = ip->GetSelNodeCount();

	if (nSel != 1)
	{
		MessageBox(hwndRollup, "A single node must be selected in which flag info should be stored from", "Flag Reassigner", MB_ICONWARNING|MB_OK);
		return;
	}

	INode* node = ip->GetSelNode(0);
	Object* obj = node->EvalWorldState(0).obj;

	if (obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
		Mesh& mesh = triObj->GetMesh();

		IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));

		if (!pFDMgr)
		{
			char strErr[256];
			sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
			MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
			return;
		}

		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

		if (!fdc)
		{
			char strErr[256];
			sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
			MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
			return;
		}

		OPENFILENAME ofn;
		char filename[256]="";

		ofn.lStructSize=sizeof(ofn);
		ofn.hwndOwner=gInterface->GetMAXHWnd();
		ofn.hInstance=hInstance;
		ofn.lpstrFilter="Flag Data Files (*.flg)\0*.flg\0\0";
		ofn.lpstrCustomFilter=NULL;
		ofn.nMaxCustFilter=0;
		ofn.nFilterIndex=0;
		ofn.lpstrFile=filename;
		ofn.nMaxFile=256;
		ofn.lpstrFileTitle=NULL;
		ofn.nMaxFileTitle=128;
		ofn.lpstrInitialDir=NULL;
		ofn.lpstrTitle="Save captured face flags as...";
		ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
		ofn.nFileOffset=0;
		ofn.nFileExtension=0;
		ofn.lpstrDefExt=TEXT(".flg");
		ofn.lCustData=0;
		ofn.lpfnHook=NULL;
		ofn.lpTemplateName=NULL;

		GetSaveFileName(&ofn);

		if (strlen(filename) == 0)
			return;

		FILE* fp = fopen(filename, "wb");

		if (!fp)
		{
			MessageBox(hwndRollup, "Failed to save file.", "Face Flag Reassigner", MB_ICONWARNING|MB_OK);
			return;
		}

		// Write out the face flag data to the file
		unsigned int size = fdc->m_data.Count();
		fwrite(&size, sizeof(unsigned int), 1, fp);

		for(unsigned int i = 0; i < size; i++)
			fwrite(&fdc->m_data[i], sizeof(FlagType), 1, fp);

		fclose(fp);

		MessageBox(hwndRollup, "Face Flag save completed!", "Face Flag Reassigner", MB_ICONINFORMATION|MB_OK);
		return;
	}

	MessageBox(hwndRollup, "Object failed to convert to TriObject?", "Face Flag Reassigner", MB_ICONWARNING|MB_OK);
}

void FlagReassigner::AssignFlagsFromFile()
{ FUNC_ENTER("FlagReassigner::AssignFlagsFromFile"); 
	int nSel = ip->GetSelNodeCount();

	if (nSel != 1)
	{
		MessageBox(hwndRollup, "A single node must be selected in which flag info should be stored from", "Flag Reassigner", MB_ICONWARNING|MB_OK);
		return;
	}

	INode* node = ip->GetSelNode(0);
	Object* obj = node->EvalWorldState(0).obj;

	int numFaces = 0;

	if (obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
		Mesh& mesh = triObj->GetMesh();

		numFaces = mesh.numFaces;

		IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));

		if (!pFDMgr)
		{
			char strErr[256];
			sprintf(strErr,"The object '%s' is not compatible with face flag data.",(char*)node->GetName());
			MessageBox(ip->GetMAXHWnd(),strErr,"Incompatible object in selection",MB_ICONWARNING|MB_OK);
			return;
		}

		// Prompt use for the flag file that they want to apply to this node
		OPENFILENAME ofn;
		char filename[256]="";

		ofn.lStructSize=sizeof(ofn);
		ofn.hwndOwner=gInterface->GetMAXHWnd();
		ofn.hInstance=hInstance;
		ofn.lpstrFilter="Flag files (*.flg)\0*.flg\0\0";
		ofn.lpstrCustomFilter=NULL;
		ofn.nMaxCustFilter=0;
		ofn.nFilterIndex=0;
		ofn.lpstrFile=filename;
		ofn.nMaxFile=256;
		ofn.lpstrFileTitle=NULL;
		ofn.nMaxFileTitle=128;
		ofn.lpstrInitialDir=NULL;
		ofn.lpstrTitle="Apply Flag File to Node...";
		ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
		ofn.nFileOffset=0;
		ofn.nFileExtension=0;
		ofn.lpstrDefExt=TEXT(".flg");
		ofn.lCustData=0;
		ofn.lpfnHook=NULL;
		ofn.lpTemplateName=NULL;

		GetOpenFileName(&ofn);

		if (strlen(filename) == 0)
			return;

		FILE* fp = fopen(filename, "rb");

		if (!fp)
		{
			MessageBox(hwndRollup, "Failed to open file!", "Flag Reassigner", MB_ICONWARNING|MB_OK);
			return;
		}

		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

		if (!fdc)
		{
			// Add a new face data channel if one doesn't exists (usually the case)
			fdc = new FaceFlagsData();
			pFDMgr->AddFaceDataChan( fdc );
			fdc->FacesCreated( 0, numFaces );
		}

		unsigned int size;
		fread(&size, sizeof(unsigned int), 1, fp);

		if (size > numFaces)
		{
			MessageBox(hwndRollup, "This file contains more faces than exist on the current object.  Cannot continue.", "Flag Reassigner", MB_ICONWARNING|MB_OK);
			fclose(fp);
			return;
		}

		for(int i = 0; i < size; i++)
			fread(&fdc->m_data[i], sizeof(FlagType), 1, fp);

		fclose(fp);

		MessageBox(hwndRollup, "Face Flag load completed!", "Face Flag Reassigner", MB_ICONINFORMATION|MB_OK);
		return;
	}

	MessageBox(hwndRollup, "Object failed to convert to TriObject?", "Face Flag Reassigner", MB_ICONWARNING|MB_OK);
}
