#include <Export/AnimOptionsDlg.h>
#include <next.h>
#include <appdata.h>
#include <Export/ExportOptions.h>
#include <Export/SkinExportOptions.h>
#include "path.h"
#include "../UI/PropList.h"

static PropList* pSkinLODPropList = NULL;

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

CStr GetModelName()
{ FUNC_ENTER("GetModelName"); 
	AppDataChunk* appdata;

	ReferenceMaker* scene=gInterface->GetScenePointer();
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_ANIM_EXPORT_OPTIONS_ID);

	if (!appdata)
		return CStr("default");

	// Version doesn't matter since modelname resided at same location in every version
	AnimOptionData* opts=(AnimOptionData*)appdata->data;
	
	if (strlen(opts->modelname)>0)
		return CStr(opts->modelname);
	
	return CStr("default");
}

static INT_PTR CALLBACK SkinExportOptionsDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("SkinExportOptionsDlgProc"); 
	static SkinOptionsDlg *dlg = NULL;
	
	switch(msg) 
	{
		case CC_SPINNER_CHANGE:
		{
			// If the number of LOD levels has changed we need to add another field
			// to the LOD property list
			
			switch(LOWORD(wParam))
			{
			case IDC_LODLEVELSSPIN:
				ISpinnerControl* iLODLevelSpin = GetISpinner( GetDlgItem( hWnd, IDC_LODLEVELSSPIN));
				
				if (iLODLevelSpin)
				{
					int nLODLevels = iLODLevelSpin->GetIVal();

					pSkinLODPropList->SaveProperties();
					pSkinLODPropList->DestroyUI();
					pSkinLODPropList->Clear();

					for(int i = 0; i < nLODLevels; i++)
					{
						char buf[80];
						_itoa(i + 1, buf, 10);

						CStr fieldName = CStr("Distance_") + buf;
						pSkinLODPropList->AddSpinEdit(fieldName, 0.0f, 99999.0f, 0.1f, "Distance at which this level should be activated");
					}

					pSkinLODPropList->BuildUI();
					pSkinLODPropList->RestoreProperties();
				}
				return TRUE;
			}
			break;
		}

		case WM_INITDIALOG:
		{
			SkinExportOptions options;
			int i;

			// Initialize the LOD Distance Property List
			if (pSkinLODPropList)
				delete pSkinLODPropList;

			pSkinLODPropList = new PropList(hInstance, 1);
			pSkinLODPropList->Attach(GetDlgItem(hWnd, IDC_LODLEVELLIST));
			pSkinLODPropList->HasApply(false);

			dlg = (SkinOptionsDlg*) lParam;
			
			// Fill the selection set combo box
			for( i = 0; i < gInterface->GetNumNamedSelSets(); i++ )
			{
				SendDlgItemMessage( hWnd, IDC_SELECTION_COMBO, CB_ADDSTRING, 0, 
					(long) gInterface->GetNamedSelSetName( i ));
			}

			GetSkinExportOptions( &options );

			ICustEdit*       model_lod_lower_cap       = GetICustEdit( GetDlgItem( hWnd, IDC_LODCAPB ) );
			ICustEdit*       model_lod_num_levels      = GetICustEdit( GetDlgItem( hWnd, IDC_LODLEVELS ) );
			ISpinnerControl* model_lod_lower_cap_spin  = GetISpinner( GetDlgItem( hWnd, IDC_LODCAPBSPIN ) );
			ISpinnerControl* model_lod_num_levels_spin = GetISpinner( GetDlgItem( hWnd, IDC_LODLEVELSSPIN ) );

			model_lod_lower_cap_spin->LinkToEdit( GetDlgItem( hWnd, IDC_LODCAPB ), EDITTYPE_FLOAT);
			model_lod_num_levels_spin->LinkToEdit( GetDlgItem( hWnd, IDC_LODLEVELS ), EDITTYPE_INT);

			model_lod_lower_cap_spin->SetLimits(0.0f, 100.0f);
			model_lod_num_levels_spin->SetLimits(0,100);

			model_lod_lower_cap_spin->SetValue(options.m_LODLowerLimit, TRUE);
			model_lod_num_levels_spin->SetValue(options.m_LODNumLevels, TRUE);

			ReleaseICustEdit( model_lod_lower_cap );
			ReleaseICustEdit( model_lod_num_levels );
			ReleaseISpinner( model_lod_lower_cap_spin );
			ReleaseISpinner( model_lod_num_levels_spin );

			if( options.m_AlwaysLOD )
			{
				SendDlgItemMessage( hWnd, IDC_ALWAYSLOD, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_LODProgressive )
			{
				SendDlgItemMessage( hWnd, IDC_LODPROGRESSIVE, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_WarnOnLitVerts )
			{
				SendDlgItemMessage( hWnd, IDC_LITVERTWARNING, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_ViewUsgFile )
			{
				SendDlgItemMessage( hWnd, IDC_OPEN_USG, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_DirWarn )
			{
				SendDlgItemMessage( hWnd, IDC_DIRWARN, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_ExportVisibleOnly )
			{
				SendDlgItemMessage( hWnd, IDC_EXPORT_VISIBLE, BM_SETCHECK, BST_CHECKED, 0 );				
			}
			if( options.m_ExportSelected )
			{
				SendDlgItemMessage( hWnd, IDC_EXPORT_SELECTED, BM_SETCHECK, BST_CHECKED, 0 );				
			}
			if( options.m_ExportBySkinName )
			{
				SendDlgItemMessage( hWnd, IDC_EXPORTBYNAME, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_AllowPropOverrides )
			{
				SendDlgItemMessage( hWnd, IDC_ALLOWOVERRIDES, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_IgnoreSkelLeaves )
			{
				SendDlgItemMessage( hWnd, IDC_IGNORESKELLEAVES, BM_SETCHECK, BST_CHECKED, 0 );
			}
			if( options.m_ExportSelectionSet )
			{
				HWND sel_combo;
				int index;

				SendDlgItemMessage( hWnd, IDC_EXPORT_SELECTION_SET, BM_SETCHECK, BST_CHECKED, 0 );

				index = SendDlgItemMessage( hWnd, IDC_SELECTION_COMBO, CB_FINDSTRING, 0, (long) (char *) options.m_ExportSet );
				if( index != CB_ERR )
				{
					SendDlgItemMessage( hWnd, IDC_SELECTION_COMBO, CB_SETCURSEL, index, 0 );
				}		

				sel_combo = GetDlgItem( hWnd, IDC_SELECTION_COMBO );
				EnableWindow( sel_combo, TRUE );
			}			
			else
			{
				HWND sel_combo;

				sel_combo = GetDlgItem( hWnd, IDC_SELECTION_COMBO );
				EnableWindow( sel_combo, FALSE );
			}

			// Assign LOD Distances
			for(i = 0; i < options.m_LODNumLevels; i++)
			{
				char buf[80];
				_ltoa(options.m_LODDistances[i], buf, 10);
				pSkinLODPropList->SetValue(i, buf);
			}


			{
				// Initialize the export filename edit box
				ICustEdit* edit = GetICustEdit( GetDlgItem( hWnd, IDC_EDITFILENAME) );
				edit->SetText(options.m_DirPath);
				ReleaseICustEdit( edit );

				edit = GetICustEdit( GetDlgItem( hWnd, IDC_SCALE ));
				edit->SetText( options.m_Scale ); 
				ReleaseICustEdit( edit );

				ShowWindow( hWnd, SW_SHOW );
			}
			break;
		}

		case WM_COMMAND:
		{
			switch (LOWORD(wParam)) 
			{
				case IDC_GETFILENAME:
				{
					OPENFILENAME ofn;
					char         filename[256]="";

					// Pop open a file dialog
					ofn.lStructSize=sizeof(ofn);
					ofn.hwndOwner=hWnd;
					ofn.hInstance=hInstance;
					ofn.lpstrFilter="All Files (*.*)\0*.*\0\0";
					ofn.lpstrCustomFilter=NULL;
					ofn.nMaxCustFilter=0;
					ofn.nFilterIndex=0;
					ofn.lpstrFile=filename;
					ofn.nMaxFile=256;
					ofn.lpstrFileTitle=NULL;
					ofn.nMaxFileTitle=0;
					ofn.lpstrInitialDir=NULL;
					ofn.lpstrTitle="Select directory for file to be saved as";
					ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
					ofn.nFileOffset=0;
					ofn.nFileExtension=0;
					ofn.lpstrDefExt=TEXT(".skin");
					ofn.lCustData=0;
					ofn.lpfnHook=NULL;
					ofn.lpTemplateName=NULL;

					GetSaveFileName(&ofn);

					if (strlen(filename)>0)
					{
						// End the filename at the last backslash so we only
						// get the directory
						char* lstbs=strrchr(filename,'\\');
						if (lstbs)
							*lstbs='\0';

						ICustEdit* edit = GetICustEdit( GetDlgItem( hWnd, IDC_EDITFILENAME ));
						edit->SetText(filename);
						ReleaseICustEdit( edit );
					}
					break;
				}

				case IDOK:
				{
					SkinExportOptions options;
					
					options.m_ExportSelected     = SendDlgItemMessage( hWnd, IDC_EXPORT_SELECTED, BM_GETCHECK, 0, 0 ) == BST_CHECKED;					
					options.m_ExportVisibleOnly  = SendDlgItemMessage( hWnd, IDC_EXPORT_VISIBLE, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_ExportSelectionSet = SendDlgItemMessage( hWnd, IDC_EXPORT_SELECTION_SET, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_WarnOnLitVerts     = SendDlgItemMessage( hWnd, IDC_LITVERTWARNING, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_ViewUsgFile        = SendDlgItemMessage( hWnd, IDC_OPEN_USG, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_DirWarn            = SendDlgItemMessage( hWnd, IDC_DIRWARN, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_AlwaysLOD          = SendDlgItemMessage( hWnd, IDC_ALWAYSLOD, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_LODProgressive     = SendDlgItemMessage( hWnd, IDC_LODPROGRESSIVE, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_ExportBySkinName   = SendDlgItemMessage( hWnd, IDC_EXPORTBYNAME, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_AllowPropOverrides = SendDlgItemMessage( hWnd, IDC_ALLOWOVERRIDES, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_IgnoreSkelLeaves   = SendDlgItemMessage( hWnd, IDC_IGNORESKELLEAVES, BM_GETCHECK, 0, 0 ) == BST_CHECKED;

					ICustEdit* model_lod_lower_cap  = GetICustEdit( GetDlgItem( hWnd, IDC_LODCAPB ) );
					ICustEdit* model_lod_num_levels = GetICustEdit( GetDlgItem( hWnd, IDC_LODLEVELS ) );

					options.m_LODLowerLimit = model_lod_lower_cap->GetFloat();
					options.m_LODNumLevels  = model_lod_num_levels->GetInt();

					ReleaseICustEdit( model_lod_lower_cap );
					ReleaseICustEdit( model_lod_num_levels );

					if( options.m_ExportSelectionSet )
					{
						int index;

						index = SendDlgItemMessage( hWnd, IDC_SELECTION_COMBO, CB_GETCURSEL, 0, 0 );
						if( index != CB_ERR )
						{
							char selection_set[128];

							SendDlgItemMessage( hWnd, IDC_SELECTION_COMBO, CB_GETLBTEXT, index, (long) selection_set );
							options.m_ExportSet = selection_set;
						}
					}

					ICustEdit* edit = GetICustEdit( GetDlgItem( hWnd, IDC_SCALE ));
					options.m_Scale = edit->GetFloat();
					ReleaseICustEdit( edit );

					// Assign directory path
					edit = GetICustEdit( GetDlgItem( hWnd, IDC_EDITFILENAME ));
					
					// Ironically, MAXs own GetText method does not work properly with MAXs CStr
					// class a buffer must be pre-allocated
					//edit->GetText(options.m_DirPath, SkinExportOptions::vMAX_LENGTH_DIRPATH);
					char tmpBuf[1024];
					edit->GetText(tmpBuf,1023);
					options.m_DirPath = tmpBuf;

					ReleaseICustEdit( edit );

					// Assign LOD level distances
					options.SetNumLODLevels(options.m_LODNumLevels);

					for(int i = 0; i < options.m_LODNumLevels; i++)
					{
						CStr value;
						pSkinLODPropList->GetValue(i, value);

						options.m_LODDistances[i] = atof(value);
					}

					SetSkinExportOptions( &options );
					DestroySkinOptionsDialog();
					break;
				}
				case IDCANCEL:
				{					
					DestroySkinOptionsDialog();
					break;
				}				
				case IDC_EXPORT_VISIBLE:					
				case IDC_EXPORT_SELECTED:
				{
					HWND sel_combo;

					sel_combo = GetDlgItem( hWnd, IDC_SELECTION_COMBO );
					EnableWindow( sel_combo, FALSE );
					break;
				}
				case IDC_EXPORT_SELECTION_SET:
				{
					HWND sel_combo;

					sel_combo = GetDlgItem( hWnd, IDC_SELECTION_COMBO );
					EnableWindow( sel_combo, TRUE );
					break;
				}
				default:
					break;
			}
			break;
		}		
		default:
			return FALSE;
	}
	return TRUE;
}

void	SkinOptionsDlg::Show( void )
{ FUNC_ENTER("SkinOptionsDlg::Show"); 
	m_wnd = CreateDialogParam(
		hInstance,
		MAKEINTRESOURCE(IDD_SKIN_EXPORT_OPTIONS),
		gInterface->GetMAXHWnd(),
		SkinExportOptionsDlgProc,
		(LPARAM)this);	
	TSTR title = "Skin Options";
	SetWindowText( m_wnd, title );
	gInterface->RegisterDlgWnd( m_wnd );
}

void	SkinOptionsDlg::Hide( void )
{ FUNC_ENTER("SkinOptionsDlg::Hide"); 
	gInterface->UnRegisterDlgWnd( m_wnd );
	DestroyWindow( m_wnd );	
}

void GetSkinExportOptions( SkinExportOptions* options )
{ FUNC_ENTER("GetSkinExportOptions"); 
	char* options_data;
	ReferenceTarget *scene = gInterface->GetScenePointer();
	AppDataChunk* pad = scene->GetAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SKIN_EXPORT_OPTIONS_ID );

	// Return the export options if they exist
	if( pad && pad->data )
	{
		int version;

		options_data = (char *) pad->data;
		version = *options_data;
		
		// Remain backward compatible with version 4

		//if( version != SkinExportOptions::vVERSION_NUMBER )
		if ( version < 0x0004 /*SkinExportOptions::vVERSION_NUMBER*/ )
		{
			// for now, don't be backward compatible. This will force the dialog
			// to use the default values
			scene->RemoveAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SKIN_EXPORT_OPTIONS_ID );			
		}
		else
		{
			// Versions later than 10 use Saveable system
			if (version < 0x000A)
			{
				char set_name[SkinExportOptions::vMAX_LENGTH_EXPORT_SET];
				char dirpath[SkinExportOptions::vMAX_LENGTH_DIRPATH];

				memcpy( &options->m_Version, options_data, sizeof( int ));			
				options_data += sizeof( int );
				memcpy( &options->m_ExportVisibleOnly, options_data, sizeof( bool ));
				options_data += sizeof( bool );
				memcpy( &options->m_ExportSelected, options_data, sizeof( bool ));
				options_data += sizeof( bool );
				memcpy( &options->m_ExportSelectionSet, options_data, sizeof( bool ));
				options_data += sizeof( bool );
				memcpy( set_name, options_data, SkinExportOptions::vMAX_LENGTH_EXPORT_SET );
				options_data += SkinExportOptions::vMAX_LENGTH_EXPORT_SET;
				options->m_ExportSet = set_name;
				memcpy( &options->m_Scale, options_data, sizeof( float ));
				options_data += sizeof( float );
				memcpy( dirpath, options_data, SkinExportOptions::vMAX_LENGTH_DIRPATH );
				options->m_DirPath=dirpath;
				options_data += SkinExportOptions::vMAX_LENGTH_DIRPATH;

				// Remain backward compatible with v4
				if ( version >= 0x0005 )
				{
					memcpy( &options->m_WarnOnLitVerts, options_data, sizeof( bool ));
					options_data += sizeof( bool );
				}

				if ( version >= 0x0006 )
				{
					memcpy( &options->m_DirWarn, options_data, sizeof( bool ));
					options_data += sizeof( bool );
				}

				if ( version >= 0x0007 )
				{
					memcpy( &options->m_AlwaysLOD, options_data, sizeof( bool ));
					options_data += sizeof( bool );

					// Temporarily disabling.  All MultiRes mods should be manually applied for now
					options->m_AlwaysLOD = false;

					memcpy( &options->m_LODLowerLimit, options_data, sizeof( float ));
					options_data += sizeof( float );

					memcpy( &options->m_LODNumLevels, options_data, sizeof( int ));
					options_data += sizeof( int );

					memcpy( &options->m_LODProgressive, options_data, sizeof( bool ));
					options_data += sizeof( bool );
				}

				if ( version >= 0x0008 )
				{
					memcpy( &options->m_ExportBySkinName, options_data, sizeof( bool ));
					options_data += sizeof( bool );
				}

				if ( version >= 0x0009 )
				{
					memcpy( &options->m_ViewUsgFile, options_data, sizeof( bool ));
					options_data += sizeof( bool );
				}
			}
			else
				options->Retrieve(options_data);
		}
	}	
}

void SetSkinExportOptions( SkinExportOptions* options )
{ FUNC_ENTER("SetSkinExportOptions"); 
	char* options_data;	
	AppDataChunk* pad;
	ReferenceTarget *scene;
	//int version;
	//char set_name[SkinExportOptions::vMAX_LENGTH_EXPORT_SET];
	//char dirpath[SkinExportOptions::vMAX_LENGTH_DIRPATH];

	assert( options );

	scene = gInterface->GetScenePointer();	
	pad = scene->GetAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SKIN_EXPORT_OPTIONS_ID );
	
	if( !pad )
	{
		options_data = new char[GetSizeSkinExportOptions()];		
		scene->AddAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SKIN_EXPORT_OPTIONS_ID, 
			GetSizeSkinExportOptions(), (void*) options_data );	
	}
	else
	{
		options_data = (char*) pad->data;		
	}

	/*
	version = SkinExportOptions::vVERSION_NUMBER;
	memcpy( options_data, &version, sizeof( int ));	
	options_data += sizeof( int );
	memcpy( options_data, &options->m_ExportVisibleOnly, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ExportSelected, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ExportSelectionSet, sizeof( bool ));
	options_data += sizeof( bool );
	strcpy( set_name, options->m_ExportSet );
	memcpy( options_data, set_name, SkinExportOptions::vMAX_LENGTH_EXPORT_SET );
	options_data += SkinExportOptions::vMAX_LENGTH_EXPORT_SET;	
	memcpy( options_data, &options->m_Scale, sizeof( float ));
	options_data += sizeof( float );
	strcpy( dirpath, options->m_DirPath );
	memcpy( options_data, dirpath, SkinExportOptions::vMAX_LENGTH_DIRPATH );
	options_data += SkinExportOptions::vMAX_LENGTH_DIRPATH;
	memcpy( options_data, &options->m_WarnOnLitVerts, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_DirWarn, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_AlwaysLOD,sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_LODLowerLimit, sizeof( float ));
	options_data += sizeof( float );
	memcpy( options_data, &options->m_LODNumLevels, sizeof( int ));
	options_data += sizeof( int );
	memcpy( options_data, &options->m_LODProgressive, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ExportBySkinName, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ViewUsgFile, sizeof( bool ));
	options_data += sizeof( bool );
	*/
	options->Store(options_data);
}

int	GetSizeSkinExportOptions( void )
{ FUNC_ENTER("GetSizeSkinExportOptions"); 
	int size;

	size = 0;
	
	size += sizeof( int );	//	int		m_Version;	
	size += sizeof( bool );
	size += sizeof( bool );
	size += sizeof( bool );
	size += SkinExportOptions::vMAX_LENGTH_EXPORT_SET;		
	size += sizeof( float );	// float	m_Scale;
	size += SkinExportOptions::vMAX_LENGTH_DIRPATH;
	size += sizeof( bool );
	size += sizeof( bool );		// m_DirWarn   aml
	size += sizeof( bool );     // m_AlwaysLOD
	size += sizeof( float );    // m_LODLowerLimit
	size += sizeof( int );      // m_LODNumLevels
	size += sizeof( bool );     // m_LODProgressive
	size += sizeof( bool );     // m_ExportBySkinName
	size += sizeof( bool );     // m_ViewUsgFile  aml

	return size;
}

SkinExportOptions::SkinExportOptions( void )
{ FUNC_ENTER("SkinExportOptions::SkinExportOptions"); 	
	m_Version = vVERSION_NUMBER;
	m_ExportVisibleOnly = true;
	m_ExportSelected = false;
	m_ExportSelectionSet = false;
	m_SkinName = "";
	m_ExportSet = "";
	m_Scale = 1.0f;
	m_DirWarn = true;

	m_AlwaysLOD          = false;
	m_LODLowerLimit      = 50.0f;
	m_LODNumLevels       = 5;
	m_LODProgressive     = false;	
	m_ExportBySkinName   = false;
	m_ViewUsgFile        = false;
	m_AllowPropOverrides = true;
	m_IgnoreSkelLeaves   = true;

	m_LODDistances = (float*)malloc(sizeof(float) * m_LODNumLevels);
}

SkinExportOptions::~SkinExportOptions()
{ FUNC_ENTER("SkinExportOptions::~SkinExportOptions");

	free(m_LODDistances);
}

// From Saveable
int  SkinExportOptions::GetSize()
{ FUNC_ENTER("SkinExportOptions::GetSize");
	int size;
	size = 0;
	
	size += sizeof( int );		// m_Version;	
	size += sizeof( bool );		// m_ExportVisibleOnly
	size += sizeof( bool );		// m_ExportSelected
	size += sizeof( bool );		// m_ExportSelectionSet
	size += GetStringSize(m_SkinName);
	//size += SkinExportOptions::vMAX_LENGTH_EXPORT_SET;		
	size += sizeof( float );	// m_Scale;
	size += GetStringSize(m_DirPath);
	//size += SkinExportOptions::vMAX_LENGTH_DIRPATH;
	size += sizeof( bool );		// m_WarnOnLitVerts
	size += sizeof( bool );		// m_DirWarn
	size += sizeof( bool );     // m_AlwaysLOD
	size += sizeof( float );    // m_LODLowerLimit
	size += sizeof( int );      // m_LODNumLevels
	size += sizeof( bool );     // m_LODProgressive
	size += sizeof( bool );     // m_ExportBySkinName
	size += sizeof( bool );     // m_ViewUsgFile
	size += sizeof( bool );		// m_AllowPropOverrides
	size += sizeof( float ) * m_LODNumLevels;

	return size;
}

void SkinExportOptions::Store(void* data)
{ FUNC_ENTER("SkinExportOptions::Store");
	unsigned char* pos = (unsigned char*)data;

	m_Version = vVERSION_NUMBER;
	Write(&pos, sizeof(int),  &m_Version);
	Write(&pos, sizeof(bool), &m_ExportVisibleOnly);
	Write(&pos, sizeof(bool), &m_ExportSelected);
	Write(&pos, sizeof(bool), &m_ExportSelectionSet);
	WriteString(&pos, m_SkinName);
	Write(&pos, sizeof(float), &m_Scale);
	WriteString(&pos, m_DirPath);
	Write(&pos, sizeof(bool), &m_WarnOnLitVerts);
	Write(&pos, sizeof(bool), &m_DirWarn);
	Write(&pos, sizeof(bool), &m_AlwaysLOD);
	Write(&pos, sizeof(float), &m_LODLowerLimit);
	Write(&pos, sizeof(int), &m_LODNumLevels);
	Write(&pos, sizeof(bool), &m_LODProgressive);
	Write(&pos, sizeof(bool), &m_ExportBySkinName);
	Write(&pos, sizeof(bool), &m_ViewUsgFile);
	Write(&pos, sizeof(bool), &m_AllowPropOverrides);
	
	for(int i = 0; i < m_LODNumLevels; i++)
		Write(&pos, sizeof(float), &m_LODDistances[i]);

	Write(&pos, sizeof(bool), &m_IgnoreSkelLeaves);
}

void SkinExportOptions::Retrieve(void* data)
{ FUNC_ENTER("SkinExportOptions::Retrieve");
	unsigned char* pos = (unsigned char*)data;

	Read(&pos, sizeof(int), &m_Version);
	Read(&pos, sizeof(bool), &m_ExportVisibleOnly);
	Read(&pos, sizeof(bool), &m_ExportSelected);
	Read(&pos, sizeof(bool), &m_ExportSelectionSet);
	GetString(&pos, m_SkinName);
	Read(&pos, sizeof(float), &m_Scale);
	GetString(&pos, m_DirPath);
	Read(&pos, sizeof(bool), &m_WarnOnLitVerts);
	Read(&pos, sizeof(bool), &m_DirWarn);
	Read(&pos, sizeof(bool), &m_AlwaysLOD);
	Read(&pos, sizeof(float), &m_LODLowerLimit);
	Read(&pos, sizeof(int), &m_LODNumLevels);
	Read(&pos, sizeof(bool), &m_LODProgressive);
	Read(&pos, sizeof(bool), &m_ExportBySkinName);
	Read(&pos, sizeof(bool), &m_ViewUsgFile);
	Read(&pos, sizeof(bool), &m_AllowPropOverrides);

	for(int i = 0; i < m_LODNumLevels; i++)
		Read(&pos, sizeof(float), &m_LODDistances[i]);

	if (m_Version >= 0x000B)
		Read(&pos, sizeof(bool), &m_IgnoreSkelLeaves);
}

void SkinExportOptions::SetNumLODLevels(unsigned int nLevels)
{ FUNC_ENTER("SkinExportOptions::SetNumLODLevels");
	m_LODDistances = (float*)realloc(m_LODDistances, sizeof(float) * nLevels);
	m_LODNumLevels = nLevels;	
}
